package Geo::Coordinates::OSGB;
use base qw(Exporter);
use strict;
use warnings;
use Carp;
use 5.008; # at least Perl 5.8 please

our $VERSION = '2.17';

our %EXPORT_TAGS = (all => [qw( 
        ll_to_grid 
        grid_to_ll 

        ll_to_grid_helmert 
        grid_to_ll_helmert
        
        is_grid_in_ostn02
        set_default_shape
    )]);

our @EXPORT_OK = ( @{ $EXPORT_TAGS{all} } );

my %ellipsoid_shapes = (
    WGS84  => [ 6_378_137.000, 6_356_752.31424518, 298.257223563,  0.006694379990141316996137233540 ],
    ETRS89 => [ 6_378_137.000, 6_356_752.314140,   298.257222101,  0.006694380022900787625359114703 ],
    GRS80  => [ 6_378_137.000, 6_356_752.314140,   298.257222101,  0.006694380022900787625359114703 ],
    OSGB36 => [ 6_377_563.396, 6_356_256.909,      299.3249612665, 0.0066705400741492318211148938735613129751683486352306 ], 
); 

my $default_shape = 'WGS84';

sub set_default_shape {
    my $s = shift;
    croak "Unknown shape: $s" if !exists $ellipsoid_shapes{$s};
    $default_shape = $s;
    return;
}

sub is_grid_in_ostn02 {
    my ($e, $n) = @_;
    return _find_OSTN02_shifts_at($e, $n)
        && _find_OSTN02_shifts_at($e-5000, $n)
        && _find_OSTN02_shifts_at($e+5000, $n)
        && _find_OSTN02_shifts_at($e, $n-5000)
        && _find_OSTN02_shifts_at($e, $n+5000);
}

# constants for OSGB mercator projection
use constant ORIGIN_LONGITUDE    => -2 / 57.29577951308232087679815481410517;
use constant ORIGIN_LATITUDE     => 49 / 57.29577951308232087679815481410517;
use constant ORIGIN_EASTING      =>  400_000;
use constant ORIGIN_NORTHING     => -100_000;
use constant CONVERGENCE_FACTOR  => 0.9996012717;

# constants for OSTN02 data
use constant MIN_X_SHIFT =>  86000;
use constant MIN_Y_SHIFT => -82000;

use constant TENTH_MM     => 0.0001;
use constant HUNDREDTH_MM => 0.00001;

# load all the data from below
my @ostn_data; 
while (<DATA>) {
    push @ostn_data, $_
}
close DATA; # closing the DATA file handle supresses annoying additions to any error messages
my %ostn_shifts_for;

sub _llh_to_cartesian {
    my ($lat, $lon, $H, $shape) = @_;

    my ($a, $b, $f, $ee) = @{$ellipsoid_shapes{$shape}};

    my $phi = $lat / 57.29577951308232087679815481410517;
    my $sp = sin $phi; 
    my $cp = cos $phi;
    my $lam = $lon / 57.29577951308232087679815481410517;
    my $sl = sin $lam; 
    my $cl = cos $lam;

    my $nu = $a / sqrt(1 - $ee*$sp*$sp);

    my $x = ($nu+$H) * $cp * $cl;
    my $y = ($nu+$H) * $cp * $sl;
    my $z = ((1-$ee)*$nu+$H)*$sp;

    return ($x,$y,$z);
}

sub _cartesian_to_llh {
    my ($x, $y, $z, $shape) = @_;

    my ($a, $b, $f, $ee) = @{$ellipsoid_shapes{$shape}};

    my $p = sqrt($x*$x+$y*$y);
    my $lam = atan2 $y, $x;
    my $phi = atan2 $z, $p*(1-$ee);

    my ($nu, $oldphi, $sp);
    while (1) {
        $sp = sin $phi;
        $nu = $a / sqrt(1 - $ee*$sp*$sp);
        $oldphi = $phi;
        $phi = atan2 $z+$ee*$nu*$sp, $p;
        last if abs($oldphi-$phi) < 1E-12;
    }        
     
    my $lat = $phi * 57.29577951308232087679815481410517;
    my $lon = $lam * 57.29577951308232087679815481410517;
    my $H = $p/cos($phi) - $nu;
    return ($lat, $lon, $H);

}

sub _small_Helmert_transform_for_OSGB {
    my ($direction, $xa, $ya, $za) = @_;
    my $tx = $direction * -446.448;
    my $ty = $direction * +125.157;
    my $tz = $direction * -542.060;
    my $sp = $direction * 0.0000204894 + 1;
    my $rx = ($direction * -0.1502/3600) / 57.29577951308232087679815481410517;
    my $ry = ($direction * -0.2470/3600) / 57.29577951308232087679815481410517;
    my $rz = ($direction * -0.8421/3600) / 57.29577951308232087679815481410517;
    my $xb = $tx + $sp*$xa - $rz*$ya + $ry*$za;
    my $yb = $ty + $rz*$xa + $sp*$ya - $rx*$za;
    my $zb = $tz - $ry*$xa + $rx*$ya + $sp*$za;
    return ($xb, $yb, $zb);
}

sub _shift_ll_from_osgb36_to_wgs84 {
    my ($lat, $lon) = @_;
    my ($xa, $ya, $za) = _llh_to_cartesian($lat, $lon, 0, 'OSGB36' );
    my ($xb, $yb, $zb) = _small_Helmert_transform_for_OSGB(-1,$xa, $ya, $za);
    my ($latx, $lonx, $junk) = _cartesian_to_llh($xb, $yb, $zb, 'WGS84');
    return ($latx, $lonx);
}

sub _shift_ll_from_wgs84_to_osgb36 {
    my ($lat, $lon) = @_;
    my ($xa, $ya, $za) = _llh_to_cartesian($lat, $lon, 0, 'WGS84');
    my ($xb, $yb, $zb) = _small_Helmert_transform_for_OSGB(+1,$xa, $ya, $za);
    my ($latx, $lonx, $junk) = _cartesian_to_llh($xb, $yb, $zb, 'OSGB36');
    return ($latx, $lonx);
}

sub ll_to_grid {

    my ($lat, $lon, $options) = @_;

    if (ref $lat && defined $lat->{lat} && defined $lat->{lon}) {
        $options = $lat;
        $lat = $options->{lat};
        $lon = $options->{lon};
    }

    # correct reversed arguments 
    if ($lat < $lon) { 
        ($lat,$lon) = ($lon,$lat)
    }

    my $shape = exists $options->{shape} ? $options->{shape} : $default_shape;

    croak "Unknown shape: $shape" if !exists $ellipsoid_shapes{$shape};

    my ($e,$n) = _project_onto_grid($lat, $lon, $shape);
    
    my @out;

    # We are done if we were using LL from OS maps
    if ($shape eq 'OSGB36') {
        @out = map { sprintf '%.3f', $_ } ($e, $n);
        return wantarray ? @out : "@out";
    }

    # now shape is WGS84 etc so we must adjust
    my ($dx, $dy) = _find_OSTN02_shifts_at($e,$n);
    if ($dx) {
        @out = map { sprintf '%.3f', $_ } ($e + $dx, $n + $dy);
        return wantarray ? @out : "@out";
    }

    # still here? Then do Helmert shift into OSGB36 and re-project
    return ll_to_grid_helmert($lat, $lon)
}

sub ll_to_grid_helmert {
    my ($lat, $lon) = @_;
    my @out = map { sprintf '%.0f', $_ } # round to metres
              _project_onto_grid( _shift_ll_from_wgs84_to_osgb36($lat, $lon), 'OSGB36');
    return wantarray ? @out : "@out";
}


sub _project_onto_grid {

    my ($lat, $lon, $shape) = @_;

    my ($a,$b,$f,$e2) = @{$ellipsoid_shapes{$shape}};

    my $n = ($a-$b)/($a+$b);
    my $af = $a * CONVERGENCE_FACTOR;

    my $phi = $lat / 57.29577951308232087679815481410517;
    my $lam = $lon / 57.29577951308232087679815481410517;

    my $cp = cos $phi; my $sp = sin $phi; 
    my $sp2 = $sp*$sp;
    my $tp  = $sp/$cp; # cos phi cannot be zero in GB
    my $tp2 = $tp*$tp; 
    my $tp4 = $tp2*$tp2;

    my $splat = 1 - $e2 * $sp2; 
    my $sqrtsplat = sqrt($splat);
    my $nu  = $af / $sqrtsplat;
    my $rho = $af * (1 - $e2) / ($splat*$sqrtsplat);
    my $eta2 = $nu/$rho - 1;

    my $p_plus  = $phi + ORIGIN_LATITUDE;
    my $p_minus = $phi - ORIGIN_LATITUDE;
    my $M = $b * CONVERGENCE_FACTOR * (
           (1 + $n * (1 + 5/4*$n*(1 + $n)))*$p_minus
         - 3*$n*(1+$n*(1+7/8*$n))  * sin(  $p_minus) * cos(  $p_plus)
         + (15/8*$n * ($n*(1+$n))) * sin(2*$p_minus) * cos(2*$p_plus)
         - 35/24*$n**3             * sin(3*$p_minus) * cos(3*$p_plus)
           );

    my $I    = $M + ORIGIN_NORTHING;
    my $II   = $nu/2  * $sp * $cp;
    my $III  = $nu/24 * $sp * $cp**3 * (5-$tp2+9*$eta2);
    my $IIIA = $nu/720* $sp * $cp**5 *(61-58*$tp2+$tp4);

    my $IV   = $nu*$cp;
    my $V    = $nu/6   * $cp**3 * ($nu/$rho-$tp2);
    my $VI   = $nu/120 * $cp**5 * (5-18*$tp2+$tp4+14*$eta2-58*$tp2*$eta2);

    my $dl = $lam - ORIGIN_LONGITUDE;
    my $north = $I + $II*$dl**2 + $III*$dl**4 + $IIIA*$dl**6;
    my $east = ORIGIN_EASTING + $IV*$dl + $V*$dl**3 + $VI*$dl**5;

    return ($east, $north);
}

sub _find_OSTN02_shifts_at {

    my ($x, $y) = @_;

    return if $x < 0;
    return if $y < 0;

    my $e_index = int $x/1000;
    my $n_index = int $y/1000;

    return if $n_index+1 > $#ostn_data;

    my $lo_pair_ref = _get_ostn_pair_reference($e_index, $n_index)   or return;
    my $hi_pair_ref = _get_ostn_pair_reference($e_index, $n_index+1) or return;

    my ($se0,$sn0,$se1,$sn1) = @$lo_pair_ref;
    my ($se2,$sn2,$se3,$sn3) = @$hi_pair_ref;

    my $t = $x/1000 - $e_index; # offset within square
    my $u = $y/1000 - $n_index;

    my $f0 = (1-$t)*(1-$u);
    my $f1 =    $t *(1-$u);
    my $f2 = (1-$t)*   $u ;
    my $f3 =    $t *   $u ;

    my $se = $f0*$se0 + $f1*$se1 + $f2*$se2 + $f3*$se3;
    my $sn = $f0*$sn0 + $f1*$sn1 + $f2*$sn2 + $f3*$sn3;

    return ($se, $sn);
}

sub _d32 {
    my ($a, $b, $c) = unpack 'CCC', $_;
    return ($a << 10) + ($b << 5) + $c - 50736; 
    # the magic number here is (48<<10)+(48<<5)+48 to allow for us
    # using a character set based at chr(48) = 0.
}

sub _get_ostn_pair_reference {
    my $x = shift;
    my $y = shift;
    my $k = $x+701*$y;

    if ( exists $ostn_shifts_for{$k} ) {
        return $ostn_shifts_for{$k}
    }

    my $leading_zeros = substr $ostn_data[$y], 0, 3;

    return if $x < $leading_zeros;

    my $index = 3 + 6*($x-$leading_zeros);
    return if $index + 12 > length $ostn_data[$y];
    
    my @shifts = map { _d32 } unpack "x[$index]A3A3A3A3", $ostn_data[$y];

    return if 0 == $shifts[0];
    return if 0 == $shifts[1];
    return if 0 == $shifts[2];
    return if 0 == $shifts[3];

    $shifts[0] += MIN_X_SHIFT;
    $shifts[1] += MIN_Y_SHIFT;
    $shifts[2] += MIN_X_SHIFT;
    $shifts[3] += MIN_Y_SHIFT;

    return $ostn_shifts_for{$k} = [ map { $_ / 1000 } @shifts ]; 
}

sub grid_to_ll {

    my ($e, $n, $options) = @_;

    if (ref $e && defined $e->{e} && defined $e->{n}) {
        $options = $e;
        $e = $options->{e};
        $n = $options->{n};
    }

    my $shape = exists $options->{shape} ? $options->{shape} : $default_shape;

    croak "Unknown shape: $shape" if !exists $ellipsoid_shapes{$shape};

    my ($os_lat, $os_lon) = _reverse_project_onto_ellipsoid($e, $n, 'OSGB36');

    # if we want OS map LL we are done
    if ($shape eq 'OSGB36') {
        return ($os_lat, $os_lon)
    }

    # If we want WGS84 LL, we must adjust to pseudo grid if we can
    my ($dx, $dy) = _find_OSTN02_shifts_at($e,$n);
    if ($dx) {
        my $in_ostn02_polygon = 1;
        my ($x,$y) = ($e-$dx, $n-$dy);
        my ($last_dx, $last_dy) = ($dx, $dy);
        APPROX:
        for (1..20) { 
            ($dx, $dy) = _find_OSTN02_shifts_at($x,$y);
            
            if (!$dx) {
                # we have been shifted off the edge
                $in_ostn02_polygon = 0;
                last APPROX
            }
                
            ($x,$y) = ($e-$dx, $n-$dy);
            last APPROX if abs($dx-$last_dx) < TENTH_MM
                        && abs($dy-$last_dy) < TENTH_MM;
            ($last_dx, $last_dy) = ($dx, $dy);
        }
        if ($in_ostn02_polygon ) {
            return _reverse_project_onto_ellipsoid($e-$dx, $n-$dy, 'WGS84')
        }
    }

    # If we get here, we must use the Helmert approx
    return _shift_ll_from_osgb36_to_wgs84($os_lat, $os_lon)
}

sub grid_to_ll_helmert {
    my ($e, $n) = @_;
    my ($os_lat, $os_lon) = _reverse_project_onto_ellipsoid($e, $n, 'OSGB36');
    return _shift_ll_from_osgb36_to_wgs84($os_lat, $os_lon)
}

sub _reverse_project_onto_ellipsoid {
   
    my ($easting, $northing, $shape) = @_;

    my ($a,$b,$f,$e2) = @{$ellipsoid_shapes{$shape}};

    my $n = ($a-$b)/($a+$b);
    my $af = $a * CONVERGENCE_FACTOR;

    my $dn = $northing - ORIGIN_NORTHING;
    my $de = $easting - ORIGIN_EASTING;
    
    my ($phi, $lam, $M, $p_plus, $p_minus);
    $phi = ORIGIN_LATITUDE + $dn/$af;

    while (1) {
        $p_plus  = $phi + ORIGIN_LATITUDE;
        $p_minus = $phi - ORIGIN_LATITUDE;
        $M = $b * CONVERGENCE_FACTOR * (
               (1 + $n * (1 + 5/4*$n*(1 + $n)))*$p_minus
             - 3*$n*(1+$n*(1+7/8*$n))  * sin(  $p_minus) * cos(  $p_plus)
             + (15/8*$n * ($n*(1+$n))) * sin(2*$p_minus) * cos(2*$p_plus)
             - 35/24*$n**3             * sin(3*$p_minus) * cos(3*$p_plus)
               );
        last if abs($dn-$M) < HUNDREDTH_MM;
        $phi = $phi + ($dn-$M)/$af;
    }

    my $cp = cos $phi; my $sp = sin $phi; 
    my $sp2 = $sp*$sp;
    my $tp  = $sp/$cp; # cos phi cannot be zero in GB
    my $tp2 = $tp*$tp; 
    my $tp4 = $tp2*$tp2;
    my $tp6 = $tp4*$tp2 ;

    my $splat = 1 - $e2 * $sp2; 
    my $sqrtsplat = sqrt($splat);
    my $nu  = $af / $sqrtsplat;
    my $rho = $af * (1 - $e2) / ($splat*$sqrtsplat);
    my $eta2 = $nu/$rho - 1;

    my $VII  = $tp /   (2*$rho*$nu);
    my $VIII = $tp /  (24*$rho*$nu**3) *  (5 +  3*$tp2 + $eta2 - 9*$tp2*$eta2);
    my $IX   = $tp / (720*$rho*$nu**5) * (61 + 90*$tp2 + 45*$tp4);

    my $secp = 1/$cp; 

    my $X    = $secp/$nu;
    my $XI   = $secp/(   6*$nu**3)*($nu/$rho + 2*$tp2);
    my $XII  = $secp/( 120*$nu**5)*(      5 + 28*$tp2 +   24*$tp4);
    my $XIIA = $secp/(5040*$nu**7)*(    61 + 662*$tp2 + 1320*$tp4 + 720*$tp6);

    $phi = $phi 
         - $VII *$de*$de
         + $VIII*$de*$de*$de*$de 
         - $IX  *$de*$de*$de*$de*$de*$de;

    $lam = ORIGIN_LONGITUDE 
         + $X   *$de 
         - $XI  *$de*$de*$de 
         + $XII *$de*$de*$de*$de*$de
         - $XIIA*$de*$de*$de*$de*$de*$de*$de;

    # now put into degrees & return
    return ($phi * 57.29577951308232087679815481410517,
            $lam * 57.29577951308232087679815481410517);
}

1;

=pod

=head1 NAME

Geo::Coordinates::OSGB - Convert coordinates between Lat/Lon and the British National Grid

An implementation of co-ordinate conversion for England, Wales, and
Scotland based on formulae and data published by the Ordnance Survey of
Great Britain.

=head1 VERSION

2.17

=for HTML <a href="https://travis-ci.org/thruston/perl-geo-coordinates-osgb">
<img src="https://travis-ci.org/thruston/perl-geo-coordinates-osgb.svg?branch=master"></a>

=head1 SYNOPSIS

  use Geo::Coordinates::OSGB qw(ll_to_grid grid_to_ll);

  ($easting,$northing) = ll_to_grid($lat,$lon); 
  ($lat,$lon) = grid_to_ll($easting,$northing);

=head1 DESCRIPTION

These modules convert accurately between an OSGB national grid reference
and coordinates given in latitude and longitude.  

In Version 2.10 and above, the default ellipsoid model used is the I<de
facto> international standard WGS84.  This means that you can take
latitude and longitude readings from your GPS receiver, or read them
from Wikipedia, or Google Earth, or your car's sat-nav, and use this
module to convert them to accurate British National grid references for
use with one of the Ordnance Survey's paper maps.  And I<vice versa>, of
course.

The module is implemented purely in Perl, and should run on any platform
with Perl version 5.8 or better. 

In this description, the abbreviations `OS' and `OSGB' mean `the
Ordnance Survey of Great Britain': the British government agency that
produces the standard maps of England, Wales, and Scotland.  Any mention
of `sheets' or `maps' refers to one or more of the map sheets defined in
the accompanying maps module.

This code is fine tuned to the British national grid system.  It is of
no use outside Britain.  In fact it's only really useful in the areas
covered by the OS's main series of maps, which excludes the Channel
Islands and Northern Ireland.

=head1 SUBROUTINES/METHODS

The following functions can be exported from the
C<Geo::Coordinates::OSGB> module:

    grid_to_ll ll_to_grid

Neither of these is exported by default.

=head2 Main subroutines

=head3 C<ll_to_grid(lat,lon)>

C<ll_to_grid> translates a latitude and longitude pair into a grid
easting and northing pair.

When called in a list context, C<ll_to_grid> returns the easting and
northing as a list of two.  When called in a scalar context, it returns
a single string with the numbers separated by spaces.

The arguments should be supplied as real numbers representing
decimal degrees, like this

    my ($e,$n) = ll_to_grid(51.5, -2.1); # (393154.801, 177900.605)

Following the normal convention, positive arguments mean North or
East, negative South or West.  

If you have data with degrees, minutes and seconds, you can convert them
to decimals like this:

    my ($e,$n) = ll_to_grid(51+25/60, 0-5/60-2/3600);

If you have trouble remembering the order of the arguments, or the
returned values, note that latitude comes before longitude in the
alphabet too, as easting comes before northing.  

However, since reasonable latitudes for the OSGB are in the range 49 to
61, and reasonable longitudes in the range -9 to +2, C<ll_to_grid>
accepts the arguments in either order; if your longitude is larger than your
latitude, then the values of the arguments will be silently swapped.  

You can also supply the arguments as named keywords (but be sure to use
the curly braces so that you pass them as a reference):

    my ($e,$n) = ll_to_grid( { lat => 51.5, lon => -2.1 } );

The easting and northing will be returned as the orthogonal distances in metres
from the `false point of origin' of the British Grid (which is a point some way
to the south-west of the Scilly Isles).  The returned pair refers to a point on 
the usual OSGB grid, which extends from the Scilly Isles in the south west to 
the Shetlands in the north.  

    my ($e,$n) = ll_to_grid(51.5, -2.1); # (393154.801, 177900.605)
    my $s      = ll_to_grid(51.5, -2.1); # "393154.801 177900.605"

If the coordinates you supply are in the area covered by the OSTN02
transformation data, then the results will be rounded to 3 decimal places,
which corresponds to the nearest millimetre.  If they are outside the coverage
(which normally means more than a few km off shore) then the conversion is
automagically done using a Helmert transformation instead of the OSTN02 data.
The results will be rounded to the nearest metre in this case, although you
probably should not rely on the results being more accurate than about 5m.

   # A point in the sea, to the north-west of Coll
   my $s = ll_to_grid(56.75,-7); # returns "94471 773206" 

The numbers returned may be negative if your latitude and longitude are
far enough south and west, but beware that the transformation is less
and less accurate or useful the further you get from the British Isles.

If you want the result presented in a more traditional grid reference
format you should pass the results to one of the grid formatting
routines from L<Grid.pm|Geo::Coordinates::OSGB::Grid>.  Like this.

    my $s = ll_to_grid(51.5, -2.1);              # "393154.801 177900.605"
    $s = format_grid(ll_to_grid(51.5,-2.1));     # "ST 931 779"
    $s = format_grid_GPS(ll_to_grid(51.5,-2.1)); # "ST 93154 77900"
    $s = format_grid_map(ll_to_grid(51.5,-2.1)); # "ST 931 779 on A:173, B:156, C:157"

C<ll_to_grid()> also takes an optional argument that sets the ellipsoid
model to use.  This defaults to `WGS84', the name of the normal model
for working with normal GPS coordinates, but if you want to work with
the traditional latitude and longitude values printed around the edges of
OS maps before 2015 then you should add an optional shape parameter like this:

    my ($e, $n) = ll_to_grid(49,-2, {shape => 'OSGB36'});

Incidentally, if you make this call above you will get back
(400000,-100000) which are the coordinates of the `true point of origin'
of the British grid.  You should get back an easting of 400000 for any
point with longitude 2W since this is the central meridian used for the
OSGB projection.  However you will get a slightly different value unless
you specify C<< {shape => 'OSGB36'} >> because the WGS84 meridians are not
quite the same as OSGB36.

=head3 C<grid_to_ll(e,n)>

The routine C<grid_to_ll()> takes an easting and northing pair
representing the distance in metres from the `false point of origin' of
the OSGB grid and returns a pair of real numbers representing the
equivalent longitude and latitude coordinates in the WGS84 model.  

Following convention, positive results are North of the equator and East
of the prime meridian, negative numbers are South and West.  The
fractional parts of the results represent decimal fractions of degrees. 

No special processing is done in scalar context because there is no
obvious assumption about how to round the results.  You will just get
the length of the list returned, which is 2.

The arguments must be an (easting, northing) pair representing the
absolute grid reference in metres from the point of origin.  You can get
these from a traditional grid reference string by calling
C<parse_grid()> first.

    my ($lat, $lon) = grid_to_ll(parse_grid('SM 349 231'))

An optional last argument defines the ellipsoid model to use just as it
does for C<ll_to_grid()>.  This is only necessary is you are working
with an ellipsoid model other than WGS84.  Pass the argument as a hash
ref with a `shape' key.

    my ($lat, $lon) = grid_to_ll(400000, 300000, {shape => 'OSGB36'});

If you like named arguments then you can use a single hash ref for all
of them (this is strictly optional):

    my ($lat, $lon) = grid_to_ll({ e => 400000, n => 300000, shape => 'OSGB36'});

The results returned will be floating point numbers with the default
Perl precision.  Unless you are running with long double precision
floats you will get 13 decimal places for latitude and 14 places for
longitude;  but this does not mean that the calculations are accurate to
that many places.  The OS online conversion tools return decimal degrees
to only 6 places.  A difference of 1 in the sixth decimal place
represents a distance on the ground of about 10 cm.  This is probably a
good rule of thumb for the reliability of these calculations, but all
the available decimal places are returned so that you can choose the
rounding that is appropriate for your application.  Here's one way to do
that:

    my ($lat, $lon) = map { sprintf "%.6f", $_ } grid_to_ll(431234, 312653);


=head2 Additional subroutines

=head3 C<set_default_shape(shape)>

The default ellipsoid shape used for conversion to and from latitude and
longitude is `WGS84' as used in the international GPS system.  This
default it set every time that  you load the module.  If you want to
process or produce a large number latitude and longitude coordinates in
the British Ordnance Survey system (as printed round the edges of OS
Landranger and Explorer maps before 2015) you can use C<< set_default_shape('OSGB36'); >> to
set the default shape to OSGB36.  This saves you having to add C<< {
shape => 'OSGB36' } >> to every call of C<ll_to_grid> or C<grid_to_ll>.

You can use C<< set_default_shape('WGS84'); >> to set the default shape back
to WGS84 again when finished with OSGB36 coordinates.

=head3 C<ll_to_grid_helmert(lat, lon)>

You can use this function to do a quicker conversion from WGS84 lat/lon
to the OS grid without using the whole OSTN02 data set.  The algorithm
used is known as a Helmert transformation.  This is the usual coordinate
conversion algorithm implemented in most consumer-level GPS devices.  It
is based on parameters supplied by the OS; they suggest that in most of
the UK this conversion is accurate to within about 5m.   

    my ($e, $n) = ll_to_grid_helmert(51.477811, -0.001475);  # RO Greenwich

The input must be decimal degrees in the WGS84 model, with latitude
first and longitude second.  The results are rounded to the nearest
whole metre.  They can be used with C<format_grid> in the same way as
the results from C<ll_to_grid>.  

This function is called automatically by C<ll_to_grid> if your
coordinates are WGS84 and lie outside the OSTN02 polygon.

=head3 C<grid_to_ll_helmert(e,n)>

You can use this function to do a quicker conversion from OS grid
references to WGS84 latitude and longitude coordinates without using the
whole OSTN02 data set.  The algorithm used is known as a Helmert
transformation.  This is the usual coordinate conversion algorithm
implemented in most consumer-level GPS devices.  It is based on
parameters supplied by the OS; they suggest that in most of the UK this
conversion is accurate to within about 5m.   

    my ($lat, $lon) = grid_to_ll_helmert(538885, 177322);

The input must be in metres from false point of origin (as produced by
C<parse_grid>) and the results are in decimal degrees using the WGS84
model.

The results are returned with the full Perl precision in the same way as
C<grid_to_ll> so that you can choose an appropriate rounding for your
needs.  Four or five decimal places is probably appropriate in most
cases.  This represents somewhere between 1 and 10 m on the ground.

This function is called automatically by C<grid_to_ll> if the grid
reference you supply lies outside the OSTN02 polygon.  (Generally such
spots are in the sea).  The results are only reliable close to mainland
Britain.

=head3 C<is_grid_in_ostn02(e,n)>

This function can be used to check whether a grid point is covered
by the OSTN02 data set.  The input is an easting, northing pair as usual, 
and the output will be either true or false.  If you get a true value you can
expect an accurate result from C<grid_to_ll>, if false then it's likely that
C<grid_to_ll> will fall back on using the approximate Helmert transformation. 
This function is used by C<random_grid> from L<Geo::Coordinates::OSGB::Grid.pm>.

=head3 Importing all the functions

You can import all the functions defined in C<OSGB.pm> with an C<:all> tag.

    use Geo::Coordinates::OSGB ':all';

=head1 EXAMPLES

  use Geo::Coordinates::OSGB qw/ll_to_grid grid_to_ll/;
  
  # Latitude and longitude according to the WGS84 model
  ($lat, $lon) = grid_to_ll($e, $n);
  
  # and to go the other way
  ($e, $n) = ll_to_grid($lat,$lon);

See the test files for more examples of usage.

=head1 BUGS AND LIMITATIONS

The formulae supplied by the OS and used for the conversion routines are
specifically designed to be close floating-point approximations rather
than exact mathematical equivalences.  So after round-trips like these:

  ($lat1,$lon1) = grid_to_ll(ll_to_grid($lat0,$lon0));
  ($e1,$n1)     = ll_to_grid(grid_to_ll($e0,$n0));

neither C<$lat1 == $lat0> nor C<$lon1 == $lon0> nor C<$e1 == $e0> nor
C<$n1 == $n0> exactly.  However the differences should be very small.

The OS formulae were designed to give an accuracy of about 1 mm of
error.  This means that you can rely on the third decimal place for grid
references and about the seventh or eighth for latitude and longitude
(although the OS themselves only provide six decimal places in their
results).

For all of England, Wales, Scotland, and the Isle of Man the error will
be tiny.  All other areas, like Northern Ireland, the Channel Islands or
Rockall, and any areas of sea more than a few miles off shore, are
outside the coverage of OSTN02, so the simpler, less accurate
transformation is used.  The OS state that this is accurate to about 5m
but that the parameters used are only valid in the reasonably close
vicinity of the British Isles.

Not enough testing has been done.  I am always grateful for the feedback
I get from users, but especially for problem reports that help me to
make this a better module.

=head1 DIAGNOSTICS

The only error message you will get from this module is about the
ellipsoid shape used for the transformation.  If you try to set C<<
{shape => 'blah'} >> the module will croak with a message saying
C<Unknown shape: blah>.  The shape should be one of the shapes defined:
WGS84 or OSGB36.

Should this software not do what you expect, then please first read this
documentation, secondly verify that you have installed it correctly and
that it passes all the installation tests on your set up, thirdly study
the source code to see what it's supposed to be doing, fourthly get in
touch to ask me about it.

=head1 CONFIGURATION AND ENVIRONMENT

There is no configuration required either of these modules or your
environment.  It should work on any recent version of Perl, on any
platform.

=head1 DEPENDENCIES

Perl 5.08 or better.

=head1 INCOMPATIBILITIES

None known.

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2002-2016 Toby Thurston

OSTN02 transformation data included in this module is freely available
from the Ordnance Survey but remains Crown Copyright (C) 2002

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

=head1 AUTHOR

Toby Thurston -- 05 Feb 2016       

toby@cpan.org

=head1 SEE ALSO 

See L<Geo::Coordinates::OSGB::Grid> for routines to format grid references.

The UK Ordnance Survey's explanations on their web pages.

See L<Geo::Coordinates::Convert> for a general approach (not based on the OSGB).

=cut

# leave blank line after cut so that pod2html works properly

__DATA__
0795H00<=5H?0<C5HM0<H5I@0=25J40=?5JH0=K5K<0>75L00>B5LC0>N5M80?:5ML0?G5N@0@35O40@?5OK0@J60C0A661<0AA6240AL62L0B863E0BC64=0BN6560C:
0785GA0=55GO0=:5H>0=?5HL0=E5I<0=K5IO0>75JC0>C5K80>O5KL0?;5L?0?G5M40@45MH0@?5N;0@K5O30A65OL0AC60D0AN61=0B96250BE62M0C063F0C;64>0CG6570D26600D=
0775G20=L5G@0>25GO0>75H=0><5HL0>A5I:0>G5IK0?05J?0?<5K40?H5KG0@45L;0@@5M00@L5MD0A85N<0AC5O50AO5OM0B:60E0BE61>0C16260C<62O0CH63G0D464@0D?6580DJ6610E666D0E=0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007:;17=7:I17G7;01817;818;
0775G10>I5G@0>N5GN0?45H<0?95HK0?>5I90?D5IH0?I5J;0@45K00@@5KC0@M5L70A95LL0AE5ME0B05N=0B<5O60BG5ON0C260F0C>61?0CI6270D46300D@63H0DK64A0E76590EB65M0EJ66?0F26710F90000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006BC0K26C60K96CH0KB6D:0KI6DM0L100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000079F17G7:61827:F18=7:N18F7;51907;=19:7;D19E7;K19O7<21:87<91:B7<@1:L7<G1;6
0765FC0?A5G10?G5G?0?L5GM0@15H<0@75HK0@<5I90@@5IG0@F5J70@M5JK0A95K?0AF5L50B15LM0B<5MF0BH5N>0C35O70C?5OO0CK60G0D661@0DA6290DM6310E863J0EC64B0EO6550F765H0F?66:0FF66M0FN0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006AL0K?6B>0KF6C10KN6CC0L56D60L=6DH0LE6E;0LM6EM0M46F@0M<6G20MD6GE0MK00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000079118279A18<7:118F7:A1917:L19<7;319E7;;19O7;B1:97;I1:D7<01:N7<71;87<>1;A7<E1;K7<K1<67=21<@
0755F30@95FB0@>5G00@C5G?0@I5GM0@N5H;0A35HJ0A95I80A>5IG0AC5J50AI5JG0B25K=0B>5L60BI5LO0C55MG0C@5N?0CK5O80D76000DB60I0DM61A0E962:0ED6320F063K0F<64>0FD6510FL65C0G36660G;66H0GB67;0GJ0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006A50KL6AG0L36B:0L;6BL0LB6C?0LJ6D10M16DD0M96E60MA6EI0MH6F;0N16FM0N86G@0N@6H30NG6HE0NO00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000078;18<78K18F79<19179L19<7:<19F7:J1:17;11:;7;91:E7;@1:N7;G1;97;N1;C7<51;M7<<1<77<C1<@7<I1<K7=01=57=71=?
0755F30A65FA0A;5G00A@5G>0AF5GL0AK5H;0B05HI0B55I80B;5IF0B@5J40BE5JF0BJ5K?0C65L70CA5M00CM5MH0D85N@0DD5O90DO6010E:60J0EF61C0F162;0F<6330FH63G0G164:0G864L0G@65?0GH6610GO66D0H76760H>67H0HF0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006A00L@6AC0LH6B50LO6BG0M76C:0M>6CM0MF6D?0MM6E10N56ED0N=6F70ND6FI0NL6G<0O46GN0O<6H@0OC00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000077F18F78619178F19<79719F79G1:17:71:;7:H1:F7:O1;07;61;:7;>1;D7;E1;N7;L1<87<31<B7<:1<L7<A1=67<G1=@7<N1=J7=51>47=91>9
0755F20B25FA0B85FO0B=5G=0BB5GL0BH5H:0BM5HI0C25I70C85ID0C>5J20CC5J@0CI5K@0CM5L80D95M10DD5MI0E05NA0E<5O:0EG6030F360K0F>61D0FI62<0G56300G=63C0GE6450GL64H0H465:0H<65M0HD66?0HK6710I367D0I:0000000000000000000000000000000000000000000000000000000000000000000000000000000000006@90LM6@L0M46A>0M<6B10MD6BC0MK6C60N36CH0N:6D:0NB6DM0NJ6E@0O16F20O96FD0O@6G70OH6GI0OO6H<1086HN10?00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000077019177A19<78119F78A1:17921:;79B1:F7:21;07:C1;;7:M1;E7;41;O7;<1<97;C1<C7;J1<M7<11=77<81=A7<?1=K7<E1>57<L1>>7<O1>B7=21>H7=61>L7=91?1
0745EC0BJ5F10BO5F@0C55FO0C:5G=0C?5GK0CE5H80CK5HG0D15I50D65IC0D;5J10DA5J>0DG5K@0DE5L:0E05M20E=5MJ0EH5NC0F35O;0F?6040FJ60L0G561E0GA6290GJ62L0H163>0H96410HA64C0HH6550I065H0I866;0I@66M0IG67?0IO0000000000000000000000000000000000000000000000000000000000000000000000000000006?B0M:6@40MA6@G0MI6A:0N16AL0N86B>0N@6C10NG6CD0NO6D60O66DH0O>6E;0OF6EM0OM6F@1056G210<6GE10D6H710K6HJ11300000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000076;19<76K19E77<1:077L1:;78<1:E78L1;079=1;;79M1;E7:>1<07:K1<:7;21<D7;:1<N7;A1=87;H1=C7;O1=L7<61>67<=1>@7<B1>H7<E1>L7<H1?17<L1?67<O1?;7=21??7=61?D
0745EC0CG5F10CL5F?0D25FM0D75G;0D=5GI0DC5H70DI5HE0DN5I30E35IA0E:5IO0E?5J=0EE5K10EH5L<0EF5M40F35ML0F?5ND0FL5O=0G76050GB60M0GN61B0H76250H>62G0HF6390HN63L0I564>0I=6510ID65C0IL6660J366H0J<67;0JC67N0JK0000000000000000000000000000000000000000000000000000000000000000000000006?>0MM6@00N56@B0N=6A50NE6AG0NM6B:0O46BL0O<6C?0OC6D10OK6DD1026E610:6EI10B6F;10I6FN1116G@1186H311@6HE11H6I711O0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007661:076F1:;7771:E77G1;07871;;78G1;E7981<079H1<:7:91<E7:I1=07;01=97;81=C7;?1=M7;F1>87;M1>B7<41>L7<81?27<;1?67<?1?;7<B1??7<E1?E7<H1?I7<L1?N7<O1@27=21@7
0745EA0DD5EO0DJ5F=0E05FK0E65G90E;5GG0EA5H50EF5HC0EL5I10F25I?0F75IM0F=5J;0FB5JI0FI5L80FA5M60FI5MN0G55NF0GB5O>0GN6060H;60K0HD61>0HK6200I362B0I:6350IB63H0IJ64:0J164L0J965?0J@6610JH66D0JO6760K767H0K?0000000000000000000000000000000000000000000000000000000000000000000000006?90NB6?K0NJ6@>0O16A10O96AC0OA6B50OI6BH1006C:1086CM10?6D?10G6E110O6ED1166F711>6FI11E6G<11M6GN1246H@12<6I312D00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000075A1:;7611:E76A1;07721;;77B1;E7821<078B1<:7931<E79C1=07:41=:7:D1=D7:N1=O7;51>97;=1>B7;D1>M7;K1?77;N1?;7<11?@7<51?E7<81?I7<;1?N7<?1@37<B1@77<E1@<7<I1@A7<L1@E7<O1@J
0745E?0EC5EM0EH5F;0EN5FI0F35G70F95GE0F?5H30FE5HA0FJ5HO0FO5I=0G55IK0G;5J90GA5JG0GF5KI0GD5M80G?5MO0GL5NG0H85O@0HD6030I160E0I96190I@61K0IG62>0IO6300J663C0J>6450JF64H0JM65:0K565M0K<66?0KD6720KL67D0L30000000000000000000000000000000000000000000000000000000000000000000000006?40O66?G0O>6@90OF6@L0OM6A>1056B110<6BC10E6C610L6CH1146D;11;6DM11C6E@11K6F21226FD12:6G712A6GJ12I6H<1306HN1386IA13@00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000074K1:E75;1;075L1;;76<1;E76M1<077=1<:77M1<D78=1<O78N1=979>1=D79O1=O7:?1>97:L1>D7;31>N7;;1?87;A1?A7;D1?E7;G1?J7;K1?N7;N1@37<11@77<51@<7<81@A7<;1@F7<?1@J7<B1@O7<E1A47<I1A87<L1A=
0735DO0F;5E=0F@5EK0FF5F:0FL5FG0G25G50G75GC0G=5H10GB5H@0GH5HM0GN5I;0H35II0H95J70H>5JF0HD5K:0HG5M00H=5N10HB5NI0HO5O>0I:5ON0IE60>0J06110J661E0J=6280JD62L0JK63>0K36410K:64C0KB6550KI65H0L166;0L866M0L@67?0LH0000000000000000000000000000000000000000000000000000000000000000000000006?00OK6?B1026@510:6@G10B6A:10I6AL1116B>1186C111@6CD11H6D61206DH1276E;12?6EM12G6F@12N6G21366GE13=6H713E6HJ13M6I<1446IO14<6J?14F00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000074F1;:7561;D75G1;O7671<:76G1<D7781<O77H1=97881=D78I1=O7991>979J1>D7::1>O7:J1?97;11?C7;61?K7;:1?O7;=1@47;A1@87;D1@=7;G1@A7;K1@F7;N1@J7<11A07<51A47<81A97<;1A=7<?1AB7<B1AF7<E1AK7<I1B0
0735DN0G95E;0G?5EI0GD5F70GJ5FF0GO5G40H55GA0H;5GO0HA5H=0HF5HL0HL5I:0I15IG0I75J50I=5JC0IB5K20IH5L@0I@5N30I95NH0ID5O80J05OH0J:6080JE60J0JN61>0K56210K<62D0KB6380KH63L0KO64?0L66510L>65C0LE6660LM66H0M467;0M<0000000000000000000000000000000000000000000000000000000000000000000000006>K10?6?>10G6@010O6@B1166A511>6AH11E6B:11M6BL1246C?12<6D112D6DD12L6E71336EI13;6F;13C6FN13J6G@1426H31496HE14A6I714I6IJ1516J:15<6JJ15F6K:1616KK16;6L;16F6LL17100000000000000000000000000000000000000000000000000000000000000000000000000000000000074A1;O7511<:75B1<D7621<O76B1=97731=D77C1=O7831>978D1>D7941>O79E1?97:51?D7:E1?N7:L1@47;01@97;31@>7;61@B7;:1@G7;=1@K7;A1A07;D1A47;G1A97;K1A>7;N1AC7<11AG7<51AL7<81B07<;1B57<?1B97<B1B>7<F1BC
0735DL0H75E:0H=5EG0HB5F50HG5FD0HN5G20I35G@0I95GM0I>5H;0ID5HJ0IJ5I80IO5IF0J55J30J:5JA0J@5K00JF5L20JC5MC0J85N;0J;5NO0JE5OC0JO6030K:60C0KE6160KL61J0L262=0L96300L@63D0LF6470LM64K0M465>0M;6620MA66D0MI6760N10000000000000000000000000000000000000000000000000000000000000000000000006>G1146?911;6?K11C6@>11K6A11226AC12:6B512A6BH12I6C:1306CM1386D?13@6E213G6ED1406F71476FI14?6G<14F6GN14N6HA1556I315=6IE15F6J51616JE16;6K516F6KF1716L617;6LG17F6M71806MG18:6N818E6NH19000000000000000000000000000000000000000000000000000000073L1<:74<1<D74L1<O75=1=975M1=D76=1=O76N1>977>1>D77N1>N78?1?878O1?C79<1?N79I1@77:91@>7:E1@C7:I1@G7:L1@L7;01A17;31A57;61A:7;:1A>7;>1AC7;A1AG7;D1AM7;G1B17;K1B67;N1B:7<11B?7<51BC7<81BH7<<1BM7<F1C5
0735DJ0I55E80I:5EF0I@5F40IF5FB0IK5G00J15G>0J65GL0J=5H:0JB5HH0JH5I60JM5ID0K35J20K95J@0K>5JN0KD5KC0KG5LF0K75MB0K35N60K=5NK0KH5O?0L26060L;60N0LC61B0LJ6250M162I0M763<0M>6400MD64C0MK6570N165K0N866>0N?6710NE0000000000000000000000000000000000000000000000000000000000000000000000006>B11H6?51206?G1276@912?6@L12G6A>12N6B11366BC13=6C613E6CH13M6D;1446DM14<6E@14C6F214K6FD1536G715;6GJ15B6H<15J6HN1626I?16;6J016F6J@1706K017:6KA17E6L11806LB18:6M218E6MB1906N319:6NC19E6O319O6OD1::00000000000000000000000000000000000000000073F1<O7471=974G1=C7581=N75H1>97681>C76H1>N7791?877F1?D7821?O78?1@:78L1@?79<1@E79M1@M7::1A27:B1A67:F1A:7:I1A?7:L1AC7;01AH7;31AM7;61B17;:1B67;>1B;7;A1B?7;D1BD7;G1BI7;K1BM7;N1C27<21C77<<1C?7<F1CH7=01D1
0745E60J95ED0J>5F20JC5F@0JI5FN0JO5G<0K55GJ0K:5H80K@5HF0KE5I40KL5IB0L15J00L65J=0L85JK0L95K80L95KJ0L65LH0KL5M=0L65N20L@5O00LI5OO0M260D0M96180MA61M0MG62@0MN6340N563I0N<64<0NB64O0NI65C0O06660O666J0O<0000000000000000000000000000000000000000000000000000000000000000000000000000006?012D6?B12L6@51336@G13;6A:13C6AL13J6B?1426C11496CD14A6D614I6DH1506E;1586EN15?6F@15G6G215N6GE1676H716>6HJ16F6I:1706IK17:6J;17E6JK1806K<18:6KL18E6L=1906LM19:6M=19E6MM19O6N>1::6NN1:E6O?1:O6OO1;:00000000000000000000000000000000000073A1=C7421=N74B1>97531>C75C1>N7601?976<1?D76I1?O7761@;77B1@D7801@G78@1@M7901A479A1A<79N1AA7:81AD7:?1AI7:C1AM7:F1B27:I1B67:L1B;7;01B?7;31BD7;61BI7;:1BN7;>1C27;A1C77;D1C<7;H1C@7<21CI7<<1D27<F1D;7=01DC7=:1DK
0745E40K65EB0K<5F00KB5F>0KG5FL0KM5G:0L25GH0L85H60L>5HD0L?5I10L@5I>0LA5IK0LB5J90LB5JF0LC5K30LD5KA0LE5KO0LE5LH0LM5MA0M55N:0M=5O30ME5OL0MM60E0N561>0N=6270ND62K0NK63?0O26430O964G0O@65;0OF65O0OM66B1040000000000000000000000000000000000000000000000000000000000000000000000000000000000006?>13@6@013H6@B1406A51476AH14?6B:14F6BL14N6C?1556D115=6DD15E6E715L6EI1646F;16;6FN16C6GA16K6H31726HE17:6I517E6IF1806J618:6JF18E6K71906KG19:6L819E6LH19O6M81::6MH1:E6N91:N6NI1;96O:1;D6OJ1;N70;1<970K1<C00000000000000000000000073<1>973M1>C74:1>N74G1?97531?E75?1@075L1@;7681@F76F1@K7741@O77D1A57841A<78D1AC7941AJ79C1B079M1B47:61B77:<1B<7:?1B@7:C1BE7:F1BI7:I1BN7:M1C27;01C77;31C<7;61CA7;:1CE7;>1CJ7;H1D37<21D;7<<1DD7<F1DM7=01E57=:1E>
0745E20L55E@0L:5EN0L?5F<0LE5FJ0LG5G70LH5GD0LH5H10LI5H?0LJ5HL0LK5I:0LK5IG0LL5J40LM5JB0LM5K10LN5K?0LN5KM0M35L;0M<5LJ0MF5M<0MN5N20N75NK0N>5OD0NG60=0NN6150O661O0O>62H0OF63B0ON64<10565210=65E10D66:10K0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006?L14<6@>14D6A114L6AC1536B515;6BH15B6C;15J6CM1626D?1696E216A6ED16H6F71706FI1776G<17?6GN17G6H@1806I018:6I@18E6J11906JA19:6K219D6KB19O6L31:96LC1:D6M31:N6MC1;96N41;D6ND1;N6O51<96OE1<C7061<N70F1=97161=C71F1=N7261>972D1>C7311>O73=1?:73I1?E7461@074B1@;74O1@G75<1@O75I1A37671A776G1A=7781AD77H1AK7881B378H1B97971B?79A1BC79K1BF7:51BJ7:91BO7:<1C37:?1C87:C1C<7:F1CA7:I1CE7:M1CK7;01CO7;31D47;>1D<7;H1DE7<21DN7<<1E77<F1E?7=01EH7=:1F0
0745E00LN5E=0LO5EJ0M05F80M15FE0M15G20M25G@0M35GM0M45H:0M45HI0M45I70M55IF0M65J40M65JB0M75K10M75K?0M<5KL0MA5L:0MF5LH0N25M60N>5MF0NG5N60NO5NJ0O75O?0O@6050OH60M10061G10862?10@63810H64211064K1180000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006@91586@L15@6A>15H6B115O6BC1676C616>6CH16F6D;16N6DM1756E@17=6F217D6FE17L6G71836GJ18;6H;18D6HK18O6I;1996IL19D6J<19O6JM1:96K=1:D6KN1:N6L>1;96LN1;D6M>1;N6MO1<96N?1<C6O01<N6O@1=97001=C70A1=N70N1>971;1>D71G1>O7231?:72@1?E72L1@17391@<73E1@G7421A274?1A774M1A:75;1A>75K1AE76;1AL76L1B377<1B:77L1BA78<1BI78K1BO7961C279@1C679I1C97:21C=7:51CB7:91CF7:<1CK7:?1CO7:C1D47:F1D97:I1D>7;31DF7;>1DO7;H1E77<21E@7<<1EI7<F1F17=01F:7=:1FC7=D1FL
0755E90M:5EF0M;5F30M;5FA0M<5G00M<5G>0M=5GL0M=5H:0M>5HI0M>5I70M>5IF0M?5J40M@5JB0MA5K00ME5K>0MJ5KK0MO5L90N45LG0N=5M50NI5MC0O55N20O@5NB0OH5O21005OC10960810A60M10I61B11262711:63011A63I11J0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006@H1646A:16;6AL16D6B?16K6C11736CD17:6D617B6DH17J6E;1816EN1896F@18@6G218H6GE1906H61996HF19D6I619O6IG1:96J71:D6JH1:N6K81;96KH1;D6L91;N6LI1<96M91<C6MJ1<N6N:1=96NK1=C6O81=N6OD1>97011>D70>1>O70J1?;7161?F71C1@171O1@<72;1@G72H1A37351A;73C1A?7411AC74>1AF74O1AM75?1B375O1B;76?1BB7701BI77@1C07801C878@1C?78J1CB7941CE79>1CH79H1CL79O1D07:21D57:51D97:91D>7:<1DB7:?1DH7:I1E07;31E87;>1EA7;H1EJ7<21F37<<1F;7<F1FD7=01FM7=:1G57=D1G>
0755E60MD5ED0MD5F20MD5FA0ME5G00MF5G>0MF5GL0MG5H:0MG5HI0MG5I70MH5IF0MH5J40MJ5JA0MN5K00N45K=0N85KK0N=5L80NA5LF0NG5M40O35MB0O@5N00OL5N>1085NM10A5O>10I5ON11160>11961111A61F11J62;12262O12:0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006A51706AH1776B:17?6BL17G6C?17O6D21866DD18>6E718F6EI18M6F;1956FN19<6GA19D6H119O6HA1:96I11:D6IB1:N6J21;96JC1;D6K31;N6KC1<86L41<C6LD1<M6M41=86MB1=C6MN1=O6N;1>96NH1>E6O41?06O@1?;6OM1?F70:1@170F1@=7121@G71?1A371K1A>7281AC72G1AF7341AJ73B1AN7421B474B1B<7531BC75C1BI7631C176C1C87741C?77D1CF7841CN78>1D178H1D57921D879<1D;79F1D>79K1DC79O1DH7:21DL7:51E17:?1E:7:I1EB7;31EK7;>1F37;H1F<7<21FE7<<1FM7<F1G67=01G?7=:1GH7=D1H0
0755E60MM5ED0MM5F20MN5FA0MN5G00MO5G>0N05GL0N05H:0N15HI0N15I70N15IF0N35J30N75JA0N=5JO0NA5K<0NF5KJ0NK5L70O05LE0O45M30O=5MA0OI5MO1055N=10B5NK10O5O:11:5OJ11B60:11J60J12261:12:61J12B0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006AC17M6B51836BH18;6C;18C6CM18J6D?1936E219:6EE19B6F719I6FI1:16G<1:96GL1:D6H<1:N6HL1;86I=1;C6IM1;N6J>1<86JN1<C6K>1<M6KL1=96L91=C6LE1=O6M21>:6M>1>E6MJ1?06N71?;6NC1?G6O01@26O<1@=6OH1@H7051A470B1A>70N1AG71<1AK71J1AO7281B272F1B67361B<73F1BC7461BJ74G1C27571C975G1C@7671CG76G1CO7781D577H1D<7831DA78=1DD78F1DH7911DK79;1DN79E1E179H1E679K1E:7:51ED7:?1EL7:I1F47;31F=7;>1FF7;H1FO7<21G77<<1G@7<F1GI7=01H17=:1H:7=D1HB
0765ED0N75F20N75FA0N85G00N85G>0N95GL0N:5H:0N:5HI0N:5I70N<5IE0N@5J20NF5J@0NJ5JN0NO5K<0O45KI0O95L70O=5LD0OB5M20OH5M@1045MN10@5N<10L5NJ1185O911D5OG12160612;60E12C61512K61F1330000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006AO18J6BB1916C61986CH19?6D;19F6DM19N6E@1:66F21:>6FE1:E6G61:N6GG1;86H71;C6HG1;N6I81<86IH1<C6J61<M6JC1=96JO1=D6K;1=O6KH1>:6L51>E6LA1?16LM1?<6M:1?G6MF1@26N31@>6N?1@I6NL1A46O81A?6OD1AJ7021AO70@1B370N1B671<1B:71I1B=7291BD72J1BK73:1C273J1C974:1C@74K1CH75;1CO75K1D676;1D=76L1DD77<1DK77G1E07811E378;1E778E1E:78O1E>7991EA79B1ED79K1EM7:51F67:?1F>7:I1FG7;31G07;>1G87;H1GA7<21GI7<<1H27<F1H;7=01HD7=:1HL7=D1I57=N1I>
0775F20N@5FA0NA5G00NB5G>0NB5GL0NC5H:0NC5HI0NE5I60NJ5ID0NO5J20O35J?0O85JM0O=5K;0OB5KH0OF5L60OL5LC1005M21055M?10>5MM10K5N;1175NI11C5O811O5OF12;60412G60B13361113<0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006A119>6AK19E6BE19M6C@1:56D51:<6DI1:B6E;1:J6EN1;16F@1;:6G11;C6GB1;N6H21<86H@1<C6HM1<N6I91=96IF1=D6J21=O6J?1>;6JK1>F6K71?16KD1?<6L01?H6L=1@36LI1@>6M51@I6MB1A46MN1A@6N;1AJ6NH1B36O61B76OC1B:7011B>70?1BB70M1BF71=1BL71M1C372>1C:72N1CA73>1CH73N1CO74>1D774O1D>75?1DD75O1DL76?1E37701E:77<1E?77F1EB7801EF7891EI78C1EM78M1F07971F@79K1F@7:51FH7:?1G07:I1G97;31GB7;>1GK7;H1H37<21H<7<<1HE7<F1HM7=01I67=:1I>7=D1IG7=N1J00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007IF1OD7IO1OF7J71OG7J@1OI7JI1OJ7K11OL7K:1OM7KB1OO7KK200
0775F20NJ5FA0NK5G00NK5G>0NL5GL0NL5H:0NN5HH0O35I60O85IC0O<5J10OA5J>0OF5JL0OK5K:0OO5KH1055L51095LC10>5M110B5M>10H5ML1155N:11A5NH11M5O71295OE12F60313260A13>0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006?K1:16@E1:86A@1:@6B:1:H6C51:O6CO1;76DI1;>6ED1;G6F<1;N6FJ1<86G71<C6GC1<O6H01=96H<1=E6HH1>06I51>;6IA1>F6IN1?26J:1?=6JG1?H6K31@36K?1@>6KL1@J6L91A46LE1A@6M11AK6M>1B66MK1B;6N91B?6NG1BC6O51BF6OC1BI7001BM70A1C47111C;71A1CB7211CI72B1D07321D773B1D>7421DE74B1DM7531E475C1E;7631EB76C1EJ7701EN77:1F277D1F577N1F97881F<78B1FM78K1G?7991G>7:01G=7:?1GC7:I1GL7;31H47;>1H=7;H1HE7<21HN7<<1I77<F1I@7=01IH7=:1J17=D1J:7=N1JB7>81JK0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007I=2017IE2037IN2047J62067J?2077JH2097K020:7K920<7KA20=7KJ20>7L220@
0785FA0O45G00O55G>0O55GL0O75H90O<5HG0OA5I50OF5IC0OJ5J00OO5J>1045JL1095K910>5KG10B5L410G5LB10L5M01115M>1155MK11?5N911K5NG1275O612D5OD13060213<0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006>H1:B6?A1:K6@:1;26A51;;6AO1;B6BI1;J6CD1<16D>1<96E81<@6F21<H6FD1=66G21=E6G?1>06GL1><6H81>G6HD1?26I11?=6I=1?H6IJ1@46J61@>6JC1@J6JO1A56K;1A@6KH1AK6L41B66LA1B?6LO1BC6M=1BF6MJ1BJ6N91BN6NF1C16O41C56OD1C<7041CB70E1CJ7151D171E1D87251D?72F1DF7361DN73F1E47461E;74G1EC7571EJ75G1F17671F876D1F=76O1FA7791FD77B1FH77L1G77861GM78?1H>78E1H<79>1H<7:51H<7:I1H>7;31HG7;>1HO7;H1I87<21IA7<<1II7<F1J27=01J:7=:1JD7=D1JL7=N1K57>81K=7>B1KF0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007I320>7I<20@7ID20A7IM20C7J520D7J>20F7JG20G7JO20I7K820J7K@20K7KI20L7L120N7L:20O0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008172;=81D2<18212<D82=2=882J2=L8372>@83D2?38402?G84A2?18512>185B2=28642<386E2;4
0795G00O>5G=0O@5GK0OF5H90OJ5HG0OO5I41045IB1095IO10=5J=10B5JK10G5K910K5KF1105L41155LA11:5LO11>5M=11C5MJ11I5N81255NF12B5O512N5OC13:0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006>61;36>K1;;6?@1;D6@51;L6@K1<46AD1<<6B>1<D6C81<L6D21=36DL1=;6EG1=C6F;1=O6FI1>>6G81>L6GF1?;6H41?I6H@1@46HL1@?6I91@J6IF1A66J21A@6J>1AL6JK1B76K71BB6KE1BG6L31BK6LA1BN6LN1C26M<1C56MJ1C96N81C<6NH1CC6O81CK6OH1D27091D870I1D@7191DG71I1DN7291E572J1E<73:1ED73J1EJ74:1F274K1F975;1F@75K1FG7691FM76C1G076M1G37771GB77A1H777K1HM7831I=7891I;78J1I;79C1I;7::1I:7;21I:7;>1IA7;H1IK7<21J37<<1J<7<F1JD7=01JM7=:1K67=D1K>7=N1KG7>81L07>B1L97>L1LA0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007HJ20K7I220L7I;20N7IC20O7IL2117J42127J=2147JF2167JN2177K72187K?2197KH21;7L021<7L921>00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000081K2<881A2<18152;K81A2<>81N2=182:2=E82G2>98342>L83A2?@83M2@484<2@184M2?185>2>285O2=386@2<38712;487B2:5
0805G=0OO5GJ1035H81085HF10=5I310B5IA10F5IO10K5J<1105JJ1155K81195KE11>5L311C5L@11H5LO11M5M<1215MJ1265N712@5NE12L5O41380000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006=E1;C6>:1;K6>O1<46?D1<<6@91<D6@N1<M6AC1=56B71==6BM1=F6CG1=M6DA1>66E<1>=6F21>G6F@1?66FO1?E6G>1@46GL1@D6H:1A36HH1AA6I51AM6IA1B76IN1BC6J;1BK6JH1BO6K61C26KD1C66L21C:6L@1C=6LN1CA6M;1CE6MK1CK6N<1D26NL1D96O<1DA6OL1DH70=1DN70M1E671=1E=71M1ED72>1EK72N1F273>1F:73N1F@74?1FH74O1FO75?1G675M1G<7671G?76A1GL76K1HB7751I777?1IL77H1J<77M1J:7881J:78O1J979G1J97:?1J87;71J87;H1J=7<21JE7<<1JN7<F1K67=01K@7=:1KH7=D1L17=N1L97>81LB7>B1LK7>L1M37?61M<0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007HA2187HI2197I221;7I:21<7IC21>7IK21?7J421A7J<21B7JE21D7JN21E7K621F7K?21H7KG21I7L021K7L821L7LA21N0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008292=181O2<K81D2<F8182<@81>2<K81K2=?8272>382D2>F8312?:83=2?N83K2@B8482A084I2@18592?285K2>286<2=386M2<487=2;587O2:5
0825H710F5HE10K5I310O5I@1155IN1195J;11>5JI11B5K711H5KE11L5L21215L@1265LN12:5M;12?5MI12D5N712J5NE1370000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006=51<46=J1<<6>>1<D6?41<M6?H1=56@=1==6A21=F6AG1=N6B<1>66C11>?6CE1>G6D:1>O6E01?76EI1?@6F71?O6FF1@>6G41@M6GC1A=6H11AL6HB1B<6I21BM6IB1C36J11C46J@1C66JO1C96K>1C<6KN1C?6L=1CB6LJ1CH6M61CO6MC1D86N11DA6N@1DH6O01DO6O@1E67011E>70A1ED7111EL71A1F37221F:72B1FA7321FI73B1G07421G774C1G>7531GF75B1GL75L1H67651HK76?1IA76J1J77731JL77<1K;77B1K:77G1K878<1K87951K779L1K77:D1K67;<1K67<21K87<<1K@7<E1KI7<O1L27=91L:7=D1LC7=N1LL7>81M57>B1M=7>L1MF7?61MN7?@1N70000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007GN21C7H721E7H@21F7HH21H7I121I7I921K7IB21L7IJ21N7J321O7J;2217JD2227JM2237K52257K>2267KF2287KO2297L722;7L@22<7LH22>00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000082<2=882=2=E8222=@81G2=:81<2=481;2=981H2=M8242>@82A2?482N2?H83:2@;83H2@O8442AC84D2A18552@185F2?28672>386H2=48792<487J2;588;2:688L296
0845I211>5I@11B5IM11G5J;11K5JI1215K61255KD12:5L112?5L?12D5LM12H5M;12M5MH1320000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006<D1<C6=91<L6=N1=46>C1==6?81=E6?L1=M6@A1>66A61>>6AK1>F6B@1>O6C51?76CJ1??6D>1?H6E31@06EH1@86F?1A06G31AH6GE1B96H51BJ6HF1C;6I71CK6IF1D06J51CM6JC1CJ6K11CF6K?1CC6KL1CE6L81CK6LD1D26M01D:6M=1DB6MK1DK6N91E46NF1E=6O41EE6OE1EM7051F370E1F;7151FB71E1FI7261G072G1G77371G?73H1GF7481GM74I1H47561H@75@1I675J1IK7641JA76=1K676G1KL7701L;7761L977<1L777J1L678A1L679:1L67:11L67:I1L57;A1L57<91L47<E1L<7<O1LD7=91LM7=D1M57=N1M>7>81MG7>B1MO7>L1N87?61NA7?@1NJ7?J1O27@41O;7@>1OC7@H1OL7A22057A<20=7AF20F7B120O0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007GM2227H62237H?2257HG2267I02287I82297IA22;7II22<7J222>7J:22>7JC22@7JL22A7K422C7K=22E7KE22F7KN22H7L622I7L?22K7LG22L0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008232<K82;2=F82@2>:8252>381J2=M81?2=H8182=F81E2>:8222>M82>2?A82K2@58382@H83E2A<8412B084@2B08512A185A2@28632?386D2>38752=487E2<58872;588H2:689529089=27E
0875J:12:5JH12>5K612C5KC12H5L112M0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006<51=46<I1=<6==1=D6>21=L6>G1>56?<1>=6@11>E6@E1>N6A:1?66AO1?>6BD1?G6C91?O6CN1@76DB1@@6E81@H6EL1A16FB1AA6G91B<6H01C86HG1D36I;1DK6IK1DN6J71DI6JB1DD6JN1D?6K:1D:6KF1D56L21D66L>1D=6LK1DD6M81DN6ME1E66N31E?6N?1EI6O01F26OI1F:70:1FB70J1FI71;1G071K1G772<1G?72M1GF73=1GM73N1H474>1H<74H1HF74M1I;7551J475>1JK75H1K@7621L676<1LL76D1M:76J1M87701M67771M577O1M578F1M579?1M47:61M47:N1M37;F1M37<>1M27<O1M67=91M?7=D1MH7=N1N17>81N97>B1NB7>L1NJ7?61O37?@1O<7?J1OD7@41OM7@>2067@H20?7A220G7A<2107AF2197B121@7B921B0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007GM22@7H522A7H>22C7HG22E7HO22F7I822H7I@22I7II22K7J122K7J:22M7JB22N7JK2307K32317K<2337KD2347KM2367L52377L>2397LF23;7LO23<00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000081A2;E81I2<?8222=982:2>382C2>L8282>H81M2>B81C2><8172>681B2>G81O2?;82;2?O82H2@B8352A683B2AJ83N2B=84;2C084L2B185=2A185N2@286?2?38702>387A2=48822<588A2;288J29G89328<89;271
1266<11=E6<B1=M6=21>56=B1>=6>61>E6>K1>N6?@1?66@51?>6@J1?G6A?1?O6B31@76BH1@@6C=1@H6D21A16DG1A96E<1AA6F01AJ6FE1B26G<1BM6H31CH6HJ1DD6IA1E?6J01E@6J=1E:6JI1E56K51E16K@1DK6KM1DF6L91DH6LE1DO6M21E86M?1EA6MJ1EK6N51F66NK1F?6OD1FG70=1G07101G771@1G?7211GF72B1GL7321H473C1H;7431HB74=1HN74;1I>74C1J774L1JO7541KG75=1L?75F1M57601MK7681N976>1N876D1N576J1N477<1N37841N378K1N379D1N27:;1N27;31N17;K1N17<C1N17=91N17=D1N;7=N1NC7>81NK7>B1O47>L1O=7?61OF7?@1ON7?J2077@420@7@>20H7@H2117A22197A<21B7AF21K7B021O7B821O7BA2210000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007H42307H=2317HF2337HN2347I72367I?2377IH2387J023:7J923;7JA23=7JJ23>7K323@7K;23A7KD23C7KL23D7L523F7L=23G7LF23I7LO23J0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008172;881@2<381H2<M8212=G8292>@82B2?:82;2?<8212?681F2?081;2>J81?2?581L2?I8282@<82E2A08322AD83>2B783L2BK8482C?84H2C08592B185J2A186;2@286L2?387=2>387M2=48862;H88?2:=88H29289127G89926<89B250
1266;M1>66<>1>>6<N1>F6==1>N6=M1?66>=1?>6>O1?F6?D1?N6@91@76@N1@?6AC1@G6B71A06BL1A86CA1A@6D61AI6DK1B16E@1B96F51BB6FI1BJ6G@1C?6H71D:6HM1E56IE1F06J31F26J@1EM6JM1EG6K91EB6KF1E=6L31E86L?1E:6LJ1ED6M51EN6M@1F96MM1FC6NF1FK6O?1G37081G<7111GE71F1GL7271H472H1H;7381HB73I1HI7411I57401IF7411J:7491K274B1KJ74K1LB7531M;75<1N375D1NJ75M1O87621O776:1O276E1NH7731NL77A1O17891O17911O179I1O17:@1O17;91O07<01O07<H1NO7=@1NO7=N1O57>81O>7>B1OG7>L1OO7?62087?@20@7?J20I7@42127@>21;7@H21C7A221L7A<2257AF22;7AO22<7B722>7B@22?0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007H323>7H<23@7HE23A7HM23C7I623D7I>23E7IG23G7IO23H7J823J7J@23K7JI23M7K223N7K:2407KC2417KK2437L42447L<2467LE2477LN2487M624:00000000000000000000000000000000000000000000000000000000000000000000000000000080N2:L8162;F81?2<@81G2=:8202>38282>N82A2?H82?2@08242?J81I2?D81>2??81<2?B81I2@58252@I82B2A>82O2B183;2BE83I2C98452CL84C2CO8542C085E2B18662A286G2@28782?387B2=J87K2<?8842;488=29I88F28=88O27289725F
1266;I1>G6<91>O6<I1?76=91??6=I1?G6>91?O6>J1@76?:1@?6?I1@G6@=1@O6A21A76AG1A@6B<1AH6C11B06CE1B96D:1BA6DO1BI6ED1C26F91C:6FM1CB6GC1CO6H:1DK6I11EF6IH1FA6J71FD6JD1F?6K01F:6K=1F46KJ1EO6L71EJ6LA1F16LK1F;6M61FF6MH1FO6NA1G86O:1G@7031GI70L1H171E1H972=1HB72N1HI73>1I073F1I<73D1IN73C1J?73H1K47401KM7481LE74A1M>74J1N67521NN75;1OD75J1OJ7651O@76?1O776K1NM77<1O:77J1O@7871OF78D1OL7961OO79M1OO7:E1OO7;=1ON7<51ON7<M1OM7=E1OM7>82007>B2097>L20B7?620J7?@2137?J21<7@421D7@>21M7@H2257A222>7A<22G7AE22I7AN22K7B722L7B?22N7BH22O0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007H223M7H;23N7HD2407HL2417I52427I=2447IF2457IN2477J72487J?24:7JH24;7K124=7K924>7KB24@7KJ24A7L324C7L;24D7LD24E7LL24F7M524H7M>24J00000000000000000000000000000000000000000000000000000000000080<29F80D2:@80M2;:8152<481>2<N81G2=G81O2>A8282?;82@2@582B2@D8272@>81L2@881A2@28192?O81F2@C8232A782?2AJ82L2B>8392C283F2CF8422D984?2DM8502D085A2C18622B186C2A286O2?L8782>@87@2=587I2;J8822:?88;29388D27H88M26=
1266;E1?86<51?@6<E1?H6=51@06=E1@76>51@@6>F1@H6?51A06?E1A86@51A@6@E1AH6A61B06AK1B96B@1BA6C51BI6CJ1C26D>1C:6E31CB6EH1CK6F=1D36G21D;6GG1DD6H=1E<6I41F76IK1G36J;1G66JG1G16K41FL6KA1FF6KN1FA6L81F@6LB1FH6LL1G36MC1G<6N<1GD6O51GL6ON1H570G1H>71@1HF7291HN7321I773;1ID7391J57371JF7361K773>1L073G1LH73O1M@7481N874A1NJ74J1O=7561OK75N20076:1OF76E1O=7701O377F1OG7831OM78@20378M20979;20@79H20F7:520L7:J20M7;B20L7<:20L7=220L7=J20L7>B20L7>L2147?621<7?@21E7?J21N7@42277@>22?7@H22H7A22317A<2367AD2377AM2397B623;7B>23<7BG23>0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007H224;7H:24=7HC24=7HL24?7I424@7I=24B7IE24C7IN24E7J624F7J?24H7JH24J7K024K7K924M7KA24N7KJ2507L22517L;2527LC2537LL2557M52567M=2587ME25900000000000000000000000000000000000000000000000080229:80;2:380C2:M80L2;G8142<A81=2=;81F2>581N2>O8272?I82?2@B82E2A882:2A281O2@L81E2@F81:2@A81C2A18202AD82<2B882I2BL8362C@83C2D383O2DG84<2E;84K2E085<2D085N2C186;2AM86D2@B86M2?78762=L87>2<A87G2;588029I88928>88B273
1266;A1?H6<11@16<A1@96=11@A6=A1@I6>11A16>A1A96?11AA6?A1AI6@11B16@A1B96A21BA6AA1BI6B11C16BD1C96C91CB6CN1CJ6DB1D26E81D;6EL1DC6FA1DK6G61E46GK1E<6HA1EM6I71FH6IO1GC6J>1GH6JK1GB6K81G>6KD1G96L11G46L81G76LE1G?6M>1GH6N71H06NO1H96OE1HC70<1HN7121I871H1IA72?1IL72M1J772L1JM72J1K?72L1L37351LK73>1M>73G1N17401ND7491O774E1OG75320075H20576@1OK76K1OB7761O877M20378<20<78I20B79720H79D20M7:12137:>2197:L21?7;921E7;G21K7<?21K7=721J7=O21J7>G21I7?621O7?@2287?J22@7@422I7@>2317@H23;7A223C7A;23D7AC23F7AL23G7B523I7B=23J7BF23L7BN23M0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007H124J7H924J7HB24L7HK24M7I324O7I<2507ID2527IM2537J52557J>2567JG2587JO2597K825;7K@25<7KI25>7L125?7L:25@7LB25B7LK25C7M425E7M<25F7ME25H7MM25I0000000000000000000000007O82797OA2847OI28M80229G80:2:A80B2;;80K2<58132<N81<2=H81E2>B81M2?<8262@682>2A082G2AJ82=2AF8232AA81H2A;81=2A481@2A>81M2B28292BF82F2C98332CM83?2DA83L2E58492EH84G2EO8582E085H2CO8602BC8692A886B2?M86K2>B8732=787<2;K87E2:@87N29588727J
1276;M1@A6<=1@J6<M1A26==1A:6=M1AB6>=1AJ6>M1B26?=1B:6?M1BB6@=1BJ6@M1C26A=1C:6AM1CB6B=1CJ6BM1D26C>1D:6D21DB6DG1DK6E<1E36F11E;6FE1ED6G:1EL6GO1F46HD1F=6I;1G96J21H46JB1H:6JO1H56K;1GO6KH1GJ6L21GK6L:1H06LL1H<6MB1HF6N81I06NO1I:6OE1ID70<1IN7121J871H1JC72@1JH72?1K=72=1L272D1LD72M1M77361MJ73?1N>73H1O17421OA74A1OJ75020375C20:76E2007711OG77;1O=78420<78E20I79320O79@21679M21<7::21B7:H21H7;521M7;B2237<02297<=22?7<J22E7=<22H7>422H7>L22H7?@22J7?J2337@423;7@>23D7@H23M7A12417A:2437AB2447AK2467B42477B<2497BE24:7BM24<7C624=0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007H02577H82597HA25:7HJ25<7I225=7I;25?7IC25@7IL25B7J425C7J=25E7JF25F7JN25H7K725I7K?25K7KH25L7L025M7L925O7LA2607LJ2627M32637M;2657MD2667ML2687N52697N=26;7NF26<7NN26M7O727G7O@28A7OH29:8012:48092:O80B2;H80J2<B8132=<81<2>681D2>O81L2?I8252@C82=2A=82F2B782A2B:8262B481K2AN81@2AI81=2AK81J2B?8262C382C2CF8302D:83<2DN83J2EA8462F584C2FI8532F085<2DE85E2C:85N2AO8672@D86@2?986I2=M8712<A87:2;687C29K87L28@
1276;I1A36<91A:6<I1AC6=91AK6=I1B36>91B;6>I1BC6?91BK6?I1C36@91C;6@I1CC6A91CK6AI1D36B91D;6BI1DC6C91DK6CI1E36D91E;6DK1ED6E@1EL6F51F46FI1F=6G?1FE6H31FM6HH1G66I>1GJ6J51HE6JF1HL6K21HG6K?1HB6KK1H?6L21HB6L@1HO6LN1I<6MA1IG6N81J16NN1J;6OD1JE70;1JO7111K971I1K@7211KM71O1LB72<1M172E1MD72N1N77371NJ73A1O=73O1OF74>1OO74M20875=20@76K2067761OM77B1OG78;20F78N21779<21=79I21C7:621I7:D21O7;12257;>22;7;L22A7<922H7<F22N7=32337=A2397=N23?7>;23E7?123F7?I23F7@423M7@>2477@H24>7A124@7A924A7AB24C7AK24D7B324F7B<24G7BD24I7BM24J7C524L0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007GO25F7H725G7H@25I7HI25J7I125L7I:25M7IB25O7IK2607J32627J<2637JD2657JM2667K62687K>2697KG26:7KO26<7L826=7L@26?7LI26@7M226B7M:26C7MC26E7MK26F7N426H7N<26I7NE26K7NM27;7O62847O?28N7OG29H8002:B8082;<80A2<580I2<O8122=J81;2>C81C2?=81L2@78242A182=2AJ82E2BD82D2BN8292BH8292BG82F2BH8322BJ83>2BL83I2C683O2CL8462D@84<2E584A2EJ84C2F?84D2G484I2FG8522E;85:2D085C2BE85L2A:8652?O86>2>C86G2=886O2;M8782:B87A297
1286<51AL6<E1B36=51B<6=E1BD6>51BL6>E1C46?51C<6?E1CD6@51CL6@E1D46A51D<6AE1DD6B51DL6BE1E46C51E<6CE1ED6D51EL6DE1F46E51F=6EF1FD6F91FM6FN1G56GC1G=6H71GF6HL1GN6IA1H;6J91I66JI1I>6K61I96KC1I46KK1I56L81IA6LF1IN6M31J:6MA1JG6N71K16NN1K<6OD1KF70:1L07121L771C1L>71F1M17241M=72>1N172G1ND7301O773>1O@73L1OJ74;20374J20<75920E76J20<77<20277I20078B20O79821E79E21L7:22227:?2287:M22=7;:22C7;H22I7<522O7<B2357<O23;7=<23A7=J23G7>723M7>D2447?22497??24?7?N24D7@>24I7@G24M7A024N7A82507AA2517AI2537B22547B;2567BC2577BL2597C425:7C=25<0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007GN2647H72657H@2677HH2697I126:7I926<7IB26=7IJ26?7J326@7J;26B7JD26C7JM26E7K526E7K>26G7KF26H7KO26J7L726K7L@26M7LH26N7M12707M:2727MB2737MK2757N32767N<2787ND2787NM27H7O528B7O>29<7OF2:67OO2:O8072;I80@2<C80H2==8112>681:2?081B2?J81K2@E8232A>82<2B882D2C282N2CC8352CC83=2CB83E2CB83M2CB8452CB8492C>8492CL84;2DB84<2E784>2EK84?2FA84A2G684>2G784G2F28502DG8582C;85A2B085J2@E8632?986<2=N86E2<C86M2;887629L
1286<01B<6<A1BE6=11BL6=A1C56>11C=6>A1CE6?11CM6?A1D56@11D=6@A1DE6A11DM6AA1E56B11E=6BA1EE6C11EM6CA1F56D11F=6DA1FE6E11FM6EA1G66F11G=6FA1GF6G21GM6GG1H56H<1H>6I01HF6IE1HN6J<1IG6JM1J06K:1IK6KD1II6L01J36L>1J@6LK1JM6M91K:6MF1KF6N71L36NM1L<6OC1LF70;1LO7161LO71=1M>71K1MK7251N=72?1O072L1O;73;1OD73I1OM74820674G20@75620I76D20A77A20778020978I21879A22379N2297:;22?7:I22E7;622K7;D2317<12377<>23>7<K23C7=823I7=F23O7>32457>@24;7>N24A7?;24G7?H24M7@42557@>25?7@L25F7A325D7A925A7A@25@7AH25A7B125C7B:25D7BB25F7BK25G7C325I7C<25J0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007GM26B7H626D7H?26E7HG26G7I026H7I826J7IA26K7II26M7J226N7J:2707JC2727JL2727K42747K=2757KE2777KN2787L627:7L?27;7LG27=7M027>7M927@7MA27A7MJ27C7N227D7N;27E7NC27G7NL2857O528O7O=29I7OF2:B7ON2;<8072<780?2=180G2=J8102>D8182?>81A2@881J2A18222AK82:2BF82B2C@8302CO8382CO83A2CO83H2CO8402CO8482CO8462CF8462CO8472DC8492E884;2EN84<2FC84;2G28432FO84<2FH84E2E=84N2D28562BG85?2A;85H2@08612>E86:2=:86C2;N86K2:C
1296<=1C56<M1C>6==1CE6=M1CN6>=1D66>M1D>6?=1DF6?M1DN6@=1E66@M1E>6A=1EF6AM1EN6B=1F66BM1F>6C=1FF6CM1FN6D=1G66DM1G>6E=1GF6EM1GO6F=1H66FM1H?6G>1HF6GM1HN6H@1I76I41I?6IJ1IG6J?1J86K11JB6K=1J=6KG1JD6L61K16LC1K>6M11KK6M>1L86ML1LE6N:1M26NM1M=6OC1MG70?1MF7141MM71C1N971N1NJ72:1O772I1O@7381OI73F20274520;74D20D75320M76>20F77G20=78720C79121B79J22A7:722H7:E22M7;22337;?2397;M23?7<:23E7<G23K7=42417=B2477=O24=7><24C7>I24I7?52517??25;7?I25E7@325O7@=2697@J26A7A926G7AF26K7AM26I7B426G7B:26D7BA26B7BG26?7BN26=7C426:7C;2687CC2690000007D526<7D=26>7DF26?7DN26A7E726B7E?26D7EH26E7F026G7F926H7FB26J0000000000000000000000007GL2717H52727H=2747HF2757HO2777I72787I@27:7IH27;7J127=7J927>7JB27?7JK27A7K327B7K<27D7KD27E7KM27G7L527H7L>27J7LF27K7LO27M7M827N7M@2807MI2817N12827N:2847NB2857NK28C7O429=7O<2:67OE2;07OM2;J8062<D80>2==80G2>780O2?28182?L81A2@E81I2A?8212B:8282C482A2CN8312D883<2D<83D2D<83L2D<8432D<8472D98442CM8432D08442DE8462E:8472EO8492FD8432FM83G2FG8412FM84:2F384C2DH84L2C<8542B185=2@F85F2?;85O2>08682<D86A2;986I29N
1296<:1CC6<I1CO6=91D76=I1D>6>81DG6>I1DO6?91E76?I1E?6@91EG6@I1EO6A91F76AI1F?6B91FG6BI1FO6C91G76CI1G?6D91GG6DI1GO6E91H76EI1H?6F91HH6FI1HO6G91I86GI1I?6H91IG6HI1J06I91J76IN1J@6JC1JI6K41K46K?1K66KM1KC6L;1L06LI1L=6M61LJ6MD1M76N21MD6N?1N16NM1N>7041MO70F1N;7181NG71H1O17271O:72F1OC7351OM73C20674220?74A20H75021176920L77J20B78>20L79821K7:122J7:A2357:N23;7;;23A7;I23G7<623M7<C2437=02497==24?7=K24E7>724M7>@2577>J25A7?425K7?>2657?H26?7@226I7@<2737@H27;7A727B7AF27H7B627N7BA2817BG27N7BN27L7C427J7C;27G7CA27D7CH27B7CN27?7D527=7D<27;7DB2787DI2757E02737E62717E?2727EG2747F02757F82777FA2787FJ2797G227;7G;27<7GC27>7GL27?7H427A7H=27B7HF27D7HN27E7I727G7I?27H7IH27J7J027K7J927L7JA27N7JJ27O7K32817K;2827KC2847KL2857L42877L=2887LE28:7LN28;7M728=7M?28>7MH28?7N028A7N928B7NA28D7NJ2907O329J7O;2:D7OD2;>7OL2<88052=180=2=K80F2>E80N2??8172@981?2A381G2AM81N2BH8262CC82A2D78312DB83?2DJ83G2DI83O2DI8472DI8452D@8412D583O2D28412DG8422E<8442F18452FF83L2FG83>2FA83F2FF83O2FJ8482E>84A2D384J2BH8522A=85;2@285D2>F85M2=;8662;O86>2:D86G29987027N
1306<H1D96=51DE6=D1E06>41E76>E1E@6?51EH6?E1F06@51F86@D1F@6A51FH6AE1G06B51G86BE1G@6C51GH6CE1H06D51H86DE1H@6E51HH6EE1I06F51I86FE1IA6G51IH6GE1J16H51J86HE1JA6I51JI6IE1K06J51K96JG1KA6K71KH6KE1L56L31LB6LC1LN6M;1M86N41MB6NL1ML6OE1N770>1NA7101NJ71;1O171F1O97211O@72>1OH72K1OO73820673E20>74120E74>20L74M21676321277E20G78E21679?2257:82337:J23C7;723I7;E23O7<22457<?24;7<L24A7=824I7=B2537=L25=7>625G7>@2617>I26;7?326E7?=26O7?G2797@127C7@;27M7@F2867A528=7AD28C7B428J7BC2907C22977C;2977CA2937CH2917CN28O7D528L7D<28J7DB28G7DI28E7E028B7E628@7E=28=7EC28:7EJ2887F02867F72847F>2817FD27N7FK27L7G127I7G:27K7GB27L7GK27N7H327O7H<2817HE2827HM2847I62857I>2877IG2887IO2897J828:7J@28<7JI28=7K228?7K:28A7KC28B7KK28D7L428E7L<28G7LE28H7LN28J7M628K7M?28L7MG28M7N028O7N82907NA2927NI29=7O22:77O;2;17OC2;K7OK2<E8042=?80<2>980E2?380M2?M8142@H81=2AB81E2B=81L2C68242D182B2DA8312DK83B2E583J2E78422E78462E28422DH83O2D<83L2D483M2DH83O2E>8402F38412FG83D2FA8362F;83:2F>83D2FD83M2F48462DI84?2C>84G2B38502@H8592?<85B2>185K2<F8642;;86<2:086E28D86N27987725N
1326=B1E<6>01EH6>A1F06?11F96?A1FA6@01FI6@@1G16A11G96AA1GA6B11GI6BA1H16C11H96CA1HA6D11HI6DA1I26E11I96EA1IA6F11II6FA1J16G11J:6GA1JA6H11JJ6HA1K16I11K:6IA1KB6J11KI6JA1L26K11L96K?1LB6L41LL6LH1M76M=1MB6N21ML6NF1N76O;1NA6OO1NL70D1O771=1O@71O1OH72;20072H20773520?73A20F73O20N74<21674J21>75721F75O21I77>21B78B21O79F22>7:?23=7;32417;A2477;N24=7<924E7<C24O7<M2597=725C7=A25M7=K2677>526A7>?26K7>H2757?227?7?<27I7?F2837@028=7@:28G7@D2907A32977AB29=7B229D7BA29J7C02:17C?2:77CN2:>7D52:<7D<2:97DB2:67DI2:47E02:27E629O7E=29L7EC29J7EJ29H7F029E7F729C7F>29@7FD29=7FK29;7G12997G82977G>2937GE2917GK28O7H228L7H828J7H?28G7HF28E7HM28B7I528D7I=28E7IF28F7IN28G7J728I7J?28J7JH28L7K128M7K928O7KB2907KJ2927L32937L;2957LD2977LL2987M52997M>29:7MF29<7MO29=7N729?7N@29@7NH29K7O12:E7O:2;?7OB2<87OK2=28032=L80;2>F80C2?A80K2@;8122A681;2B181B2BK81J2CF8222D?82B2DJ8322E583B2E?83N2ED8452ED8432E:8402DO83L2DD83I2D983J2DK83K2E?83M2F483J2FA83<2F;82O2F582N2F68382F=83B2FD83K2E@8442D584=2BJ84E2A>84N2@28572>G85@2=<85I2<18622:F86:29:86C27O86L26D878260
1346>=1F?6>M1FI6?<1G26?L1G96@<1GB6@M1GJ6A=1H26AM1H:6B<1HB6BL1HJ6C=1I26CM1I:6D=1IB6DM1IK6E=1J26EM1J:6F=1JB6FM1JJ6G=1K36GM1K:6H=1KC6HM1KJ6I=1L36IM1L;6J=1LB6JM1LK6K=1M26KL1M<6L71MB6LL1MM6M@1N86N51NB6NI1NM6O>1O77021OB70J1OL71C20572920>72H20F73720O73F21774521?74C21H75222075A22876022A77122M78423:79923G7:<2447;;24A7;E24K7;N2557<825?7<B25I7<L2637=626=7=@26G7=J2717>427;7>>27E7>H27O7?12897?;28C7?F28M7?O2977@929A7@C29K7A12:27A@2:87B02:?7B?2:E7BN2:L7C=2;27CL2;87D<2;?7DI2;C7E02;A7E62;?7E=2;<7EC2;97EJ2;77F02;57F72;27F>2:O7FD2:M7FK2:J7G12:H7G82:F7G>2:C7GE2:@7GK2:>7H22:<7H82:97H?2:67HF2:47HM2:27I329O7I:29L7I@29J7IG29G7IM29E7J429C7J:29@7JA29=7JH29;7K029<7K829=7KA29?7KI29@7L229B7L:29C7LC29E7LK29F7M429G7M=29I7ME29J7MN29L7N629M7N?29O7NG2:87O02;27O92;L7OA2<F7OI2=@8012>:8092?580A2?O80I2@J8102AD8182B?81@2C981H2D48232DI82B2E48322E>83C2EH8412F18442EM8412EA83M2E783J2DK83F2D@83F2DL83H2EA83J2F683B2F;8352F582G2EO82C2EN82M2F48362F;83@2F683I2DK8422C@84;2B484C2@I84L2?>8552>385>2<G85F2;<8602:186828F86C27D87027887=26J87J26=
1366?91GB6?H1GK6@81H26@I1H;6A91HC6AH1HK6B81I36BH1I;6C91IC6CI1IK6D91J36DH1J;6E91JD6EI1JK6F91K36FI1K;6G91KC6GI1KL6H91L36HI1L<6I91LC6II1LL6J91M46JI1M;6K91MD6KI1MK6L91N86L@1N;6LO1NC6MC1NM6N81O86NM1OC6OA1OM70620870O20A71F20J72521272D21;73321C73B21L74022474?22<75222E75I22O76C23:77G23F78K24379O24@7:O24M7;>25>7;M25N7<72697<A26C7<K26M7=52777=?27A7=I27K7>32857>=28?7>G28I7?02937?:29=7?E29F7?O2:17@82:;7@B2:E7A02:L7A?2;37AN2;97B=2;?7BL2;F7C;2;L7CJ2<37D:2<97DI2<@7E82<F7EC2<H7EJ2<F7F02<D7F72<A7F>2<>7FD2<<7FK2<:7G12<77G82<57G>2<27GE2<07GK2;M7H22;K7H82;H7H?2;E7HF2;C7HM2;A7I32;?7I:2;<7I@2;97IG2;77IM2;47J42;27J:2:O7JA2:M7JH2:J7JN2:H7K52:F7K;2:B7KB2:@7KH2:>7KO2:<7L52:97L<2:67LC2:47LK2:47M42:67M<2:77ME2:97MM2::7N62:<7N>2:=7NF2:F7NO2;?7O82<:7O?2=47OG2=N7OO2>I8072?C80?2@>80F2A880N2B38162BM81>2CH81F2DB8232E282C2E=8332EH83C2F28432F=8422F483N2EI83K2E>83G2E383D2DH83C2DN83E2EC83F2F883:2F582M2EO82@2EJ8282EF82A2EM82K2F48352F:83>2EA83G2D68402BJ8492A?84A2@484J2>I8532==85<2<285D2:G85N29<86:28L86G28>87428187A27D87O278
1376?F1H86@41HD6@D1HK6A41I46AD1I<6B41ID6BD1IL6C51J46CE1J<6D41JD6DD1JL6E41K46EE1K<6F51KD6FE1KL6G51L46GE1L<6H51LE6HE1LL6I51M56IE1M<6J51ME6JE1MM6K51N46KE1N=6L61NF6LG1O36LN1O76M41O:6MG1OC6N<1ON6O02096OD20C70<20M71321671B21?72121G72@21O72O22873=22@74522J74L23475C23=76:23G77:24378>24@79B24M7:B2597;125J7;A26:7;O26J7<>27:7<J27G7=42817=>28:7=H28E7>228O7><2997>F29C7?029L7?92:77?C2:A7?N2:K7@72;57@A2;?7@N2;G7A=2;N7AL2<47B;2<;7BJ2<A7C92<G7CH2<N7D82=47DG2=:7E62=@7EE2=G7F42=M7F>2=N7FD2=K7FK2=I7G12=G7G82=D7G>2=A7GE2=?7GK2==7H22=:7H82=77H?2=57HF2=37HM2=07I32<N7I:2<K7I@2<H7IG2<F7IM2<D7J42<A7J:2<>7JA2<<7JH2<:7JN2<77K52<57K;2<27KB2;O7KH2;M7KO2;K7L52;H7L<2;E7LB2;C7LI2;A7M02;?7M72;<7M=2;97MD2;77MJ2;47N12;27N72:O7N>2:L7NF2;37NM2;N7O62<H7O=2=C7OE2>=7OM2?88052@280=2@L80D2AG80L2BA8142C<81<2D681D2E18242E<82C2EG8332F183<2F483D2F683M2F983L2F183H2EF83E2E;83A2DO83?2E083A2EE83A2F68332F082E2EJ8282ED81L2E>8262EE82@2EK82I2F28332F783<2DL83E2CA83N2B68472@K84?2??84H2>48512<I85:2;>85D2:@8622:386?29F86L29987928L87F28?
1396@A1I;6A01ID6A@1IM6B01J56B@1J=6C11JE6C@1JM6D01K56D@1K=6E01KE6EA1KM6F11L56FA1L=6G01LE6GA1LM6H11M56HA1M=6I11ME6IA1MN6J11N56JA1N>6K11NF6KA1NM6L11O66LB1O@6M41ON6M;2026MB2066MJ2096N?20D6O320O6OH21970@21B70O21K71>22371L22;72@22E73722N73O23874F23A75=23K76424576M24?78124L7952597:625F7:E2667;426F7;C2767<227F7<A2877=028G7==2957=G29?7>129I7>;2:37>E2:<7>O2:G7?92;17?B2;;7?M2;E7@72;N7@@2<97@L2<A7A;2<H7AJ2<N7B92=57BH2=;7C72=B7CF2=H7D62=O7DE2>57E42><7EC2>B7F22>I7FB2>O7G12?67G82?37G>2?17GE2>N7GK2>L7H22>I7H82>F7H?2>D7HF2>B7HM2>@7I32>=7I:2>:7I@2>87IG2>57IM2>37J42>07J:2=N7JA2=K7JH2=I7JN2=G7K52=D7K;2=A7KB2=?7KH2==7KO2=:7L52=77L<2=57LB2=27LI2=07M02<N7M72<K7M92<F7M:2<A7M;2<;7M=2<67M>2<07MA2;K7MJ2;J7N72;M7NB2<:7NJ2<B7O22=<7O:2>57OB2>N7OK2?H8032@A80:2A;80B2B680J2C08122CJ81:2DE81D2E;8242EE82<2EH82D2EJ82L2EM8352EO83=2F183E2F483F2EN83B2EB83>2E783<2E283=2EG8382F082K2EJ82>2ED8212E>81C2E881K2E=8242ED82>2EK82H2F18312EB83:2D783C2BL83L2AA8452@584=2>J84F2=?84O2<585<2;G85I2;:8672:M86D2:@8712:387>29F
1416A<1J=6AL1JF6B<1JN6BL1K66C<1K>6CL1KF6D<1KN6DL1L66E=1L>6EM1LF6F<1LN6FL1M66G=1M>6GM1MF6H=1MN6HM1N66I<1N>6IM1NG6J=1NN6JM1O76K=1O>6KM1OG6L=1OO6LN20:6M@20H6MI20M6MO2116N72156NB21;6O721E6OM21O70<22770K22@71C22I72:23373123=73H23F74@24075724:75N24C76F24M77D25878G25E79J2627:926B7:H2727;727B7;F2827<528C7<D2937=229C7=B2:37>02:B7>:2:L7>D2;77>N2;A7?82;K7?A2<57?L2<>7@62<I7@?2=37@J2=<7A92=C7AH2=I7B72>07BG2>67C62>=7CE2>C7D42>I7DC2>O7E22?67EA2?<7F02?C7F@2?I7FO2@07G>2@67GK2@;7H22@87H82@67H?2@37HF2@17HM2?O7I32?L7I:2?I7I@2?G7IG2?E7IM2?B7J42??7J:2?=7JA2?;7JH2?87JN2?67K52?37K;2?07KB2>N7KH2>L7KK2>G7KL2>A7KM2><7KN2>67KO2>07L02=K7L12=F7L32=@7L42=:7L52=57L62=07L;2<L7LD2<J7LM2<H7M62<F7M@2<D7N02=M7N?2=N7NE2=N7NN2>87O72>L7O?2?E7OG2@>7OO2A88072B180?2BJ80G2CC80O2D=8172E681>2E=81E2E>81M2EA8252EC82=2EE82E2EH82M2EJ8362EM83>2EO83?2EJ83<2E>8382E483:2EH8312EK82D2EE8262E>81I2E881<2E281?2E581I2E<8232EB82<2EI82F2F082O2DN8382CC83A2B783J2@L8422?A84;2>584F2=;8542<N85A2<A85N2<586;2;H86H2;;8752:N87B2:@
1446BH1KG6C81KO6CH1L76D81L?6DH1LG6E91LO6EH1M76F81M?6FH1MG6G81MO6GI1N86H91N?6HH1NG6I81NO6II1O76J91O@6JI1OG6K92006KH2076L920@6LI20H6M:2146ML21B6N721H6N=21L6ND2206NK2246O:22;6ON22D70E22N71=23772423A72K23J73B24474:24>75124G75H25176?25;77725D78:26179>26>79M26N7:<27>7:K27N7;:28>7;I28O7<729?7<F29O7=52:?7=D2:O7>32;?7>B2;O7>M2<;7?72<D7?A2<N7?K2=97@52=C7@?2=M7@H2>67A72>=7AF2>D7B62>J7BE2?07C42?77CC2?=7D22?D7DA2?J7E02@17E?2@77EN2@>7F>2@D7FM2@K7G<2A17GK2A87H:2A>7HF2A@7HM2A>7I32A;7I:2A97I@2A67IG2A47IM2A17J42@N7J:2@L7J=2@H7J>2@B7J?2@<7J@2@77JA2@27JB2?L7JC2?F7JD2?A7JF2?<7JG2?67JH2?07JI2>K7JJ2>E7JK2>@7JL2>:7JM2>47JN2=O7K52=L7K>2=J7KG2=H7L02=G7L92=D7LC2=C7LL2=A7N02>J7N;2@87N=2?C7NA2?97NJ2?C7O32?M7O;2@;7OC2A57OK2AN8032BH80;2CA80C2D:80K2E48132EG81:2E=81?2E>81E2E>81J2E>81O2E?8262EA82>2EC82F2EE82N2EH8372EJ8392EF8362E;8362EJ82I2EE82<2E>81O2E881B2E28142DL8132DM81>2E481G2E;8212EA82;2EH82D2ED82M2D98362BM83>2AB83H2@78412>O84>2>C84K2>68582=I85E2=<8632<O86@2<B86M2<587:2;H
1476D41M06DD1M86E41M@6ED1MH6F41N06FD1N86G41N@6GE1NH6H41O16HD1O86I41O@6IE1OH6J52006JE2096K420@6KD20I6L52106LE2196M521A6MG21O6N922<6ND22D6NK22H6O222K6O822O6OH23570@23=71723E71N23O72E24973<24C74424L74K25675B25@76925I77126277M26>79126J79A27;7:027K7:>28:7:M28J7;=29;7;K29K7<:2:;7<I2:K7=82;<7=G2;K7>62<;7>E2<K7?32=<7?@2=I7?J2>37@42>=7@>2>F7@H2?07A52?87AD2??7B42?E7BC2?L7C22@27CA2@87D02@>7D?2@E7DN2@K7E>2A17EM2A87F<2A>7FK2AE7G:2AK7GI2B27H82B87HH2B=7I32B@7I12B=7I22B77I32B27I42AM7I52AG7I72AA7I82A<7I92A67I:2A17I;2@K7I<2@F7I=2@@7I>2@;7I?2@57I@2?O7IA2?J7IB2?E7IC2??7ID2?97IE2?47IG2>N7IO2>L7J82>J7JA2>I7JK2>F7K42>E7K=2>C7KF2>A7KO2>@7L82>=7MI2?I7M32@=7N82AM7N:2A87N=2@E7NE2@O7NN2A97O72AB7O?2AL7OG2BE7OO2C?8072D880?2E180G2EK80O2EK8162EA81<2E>81A2E>81F2E?81K2E?8212E@8262E@82;2EA82@2EA82G2EC82O2EE8332EC82O2EE82B2E>8252E881G2E281:2DL80L2DF80H2DE8122DL81<2E281F2E981O2E@8292EG82B2DO82K2CD8342B883<2@M83I2@78462?J84C2?=8502?085=2>C85J2>68682=I86E2=<8722<O87?2<B
1496E01N16E@1N96F01NA6F@1NI6G01O16G@1O96H01OA6H@1OJ6I02016I@2096J120A6J@20I6K02126K@2196L121B6LA21I6M12226MA22;6N422I6NF2376O223?6O823C6O?23G6OF23J70;24171224:71H24A72@24J73725173N25:74E25D75<25M76326776K26A77B26J78D27779527G79D2877:228G7:A2977;029G7;?2:77;N2:G7<=2;77<L2;H7=:2<87=J2<G7>92=77>G2=H7?62>87?E2>H7@32?67@=2?@7@G2?K7A32@27AB2@97B22@?7BA2@F7C02@L7C?2A37CN2A97D>2A@7DM2AF7E<2AL7EJ2AN7F:2B17FH2B47G72B77GF2B:7H52B=7H:2B@7H?2BC7H:2BD7H22BC7GO2BA7H02B;7H12B67H22B17H32AJ7H42AE7H52A@7H62A:7H72A47H82@O7H92@I7H:2@D7H;2@>7H=2@87H>2@37H@2?N7HI2?L7I22?J7I;2?I7ID2?G7IN2?E7J72?C7J@2?A7JI2??7K32?>7K<2?<7KE2?:7M02@?7M;2AB7M=2BO7N62CB7N82BM7N:2B77NA2B:7NI2BD7O22BN7O;2C87OC2CB7OK2D68032DO80;2EH80C2F:80J2EO8112EE8182E>81=2E?81C2E?81H2E@81M2E@8222EA8272EA82=2EA82B2EB82G2EB82L2EC82M2EC82;2E981M2E381@2DM8122DG80E2DA80=2D>80F2DD8102DK81:2E281D2E881N2E>8272EE82@2D:82I2BO8332AK83@2A>83M2A184:2@E84H2@88552?J85B2?=85O2?086<2>C86I2>68762=J
1546G<1OJ6GL2026H<20:6HL20C6I<20J6IL2126J<21:6JL21B6K<21K6KM2226L=22;6LL22B6M<22K6MM2356N@23C6O22416O?24:6OF24>6OM24B70624F70M24N71D25672:25>73225F73I25N74?26575726>75N26E76E26N77<27878727C78H28379728C79F2937:529C7:D2:37;32:C7;B2;37<12;C7<@2<47<N2<D7==2=47=L2=D7>;2>47>J2>D7?92?47?H2?D7@72@57@F2@E7A12@M7A@2A47B02A:7B?2A=7BM2A?7C<2AB7CK2AE7D:2AG7DI2AJ7E82AM7EF2B17F52B37FD2B67G32B97G>2B<7GB2B@7GF2BC7GK2BG7GI2BH7GA2BH7G92BG7G12BG7FM2BD7FN2B?7FO2B:7G02B47G12AN7G22AI7G32AC7G42A>7G52A87G62A27G:2@N7GC2@L7GL2@K7H52@H7H?2@G7HH2@E7I12@C7I:2@B7IC2@?7IL2@>7J62@<7J?2@:7JH2@87K12@67L72A67MC2BG7LD2BN7LE2BN7MD2C@7N52DA7N72CL7N<2CF7NE2CO7NM2D97O62DC7O?2DM7OG2E77OO2EE8082F>80?2F>80F2F480M2EI8142E?81:2E?81?2E@81D2E@81I2EA81N2EA8242EA8292EB82>2EB82C2EC82I2EE8312EJ8392EN82F2EE8242E;81B2E280O2DH80<2D>80>2D?80E2DC80N2DI8182E081B2E781L2E=8252E082>2CE82J2C38372BF83E2B98422AL84?2A>84L2A18592@E85F2@88642?K86A2?>86N2?1
1586I821<6IH21C6J821L6JH2236K822;6KH22D6L822K6LH2346M823;6MH23D6N:2406NL24=6O>24K6OM25670325970;25=70H25C71?25J72626272M26:73D26B74;26J75227275I27:76@27B77727J77N28278:28;78H28K79729:79E29J7:42::7:B2:J7;12;:7;@2;I7;N2<97<=2<I7<K2=97=:2=I7=I2>87>72>H7>F2?87?42?H7?C2@87@22@G7@K2A47A=2A97AL2A;7B;2A>7BI2AA7C82AD7CG2AG7D62AJ7DE2AL7E42AO7EB2B27F12B47F@2B77FE2B;7FJ2B?7FN2BB7G32BF7G72BJ7G82BL7G02BL7FH2BK7F@2BJ7F82BJ7F02BJ7EJ2BH7EK2BC7EL2B=7EM2B77EN2B27F32AN7F=2AM7FF2AJ7FO2AI7G82AG7GB2AE7GK2AD7H42AA7H=2A@7HG2A>7I02A<7I92A;7IB2A87IK2A77J42A47J>2A37K?2AM7M52C@7M=2CC7LN2C<7L@2C67LN2C:7ME2DC7N52EA7N82E27N@2E;7NI2EE7O12EO7O:2F97OC2FC7OK2FM8032FM80:2FB80A2F880I2EN8102ED8152E@81;2E@81@2EA81F2EA81K2EA8202EB8252EB82:2EC82@2ED82H2EI82O2EN8372F383?2F78322F182@2EG81M2E=81;2E48122DO8152E08172E281:2E281=2E481A2E581J2E<8252DG82B2D:82O2CM83<2C@83J2C38472BF84D2B98512AL85>2A?85L2A28692@E86F2@8
1606J422<6JD22E6K422L6KD2346L423=6LD23D6M423M6MD2446N524=6NF24J6O82576OJ25F70;26170A26570H26971:26?72126G72H26O73?27774627>74M27G75D27N76;28777228>77H28E78<28M78D29778L29C79529M79=2:979J2:H7:92;87:H2;H7;62<77;E2<G7<42=77<B2=G7=12>67=@2>F7=N2?67>=2?E7>K2@57?A2@C7@=2@N7A92A;7AH2A=7B72A@7BF2AC7C42AE7CC2AH7D22AK7DA2AM7E02B17E?2B47EH2B77EL2B:7F12B>7F52BB7F:2BE7F?2BI7FC2BM7FF2C07F>2C07F72BO7EO2BO7EH2BN7EA2BN7E92BN7E22BM7E02BN7E82BM7E@2BK7EI2BI7F22BG7F<2BE7FE2BC7FN2BB7G72B@7GA2B>7GJ2B<7H32B:7H<2B97HF2B77HO2B57I82B37IA2B17IJ2B07JG2BC7KE2C67LB2C97LG2CJ7LG2CI7LJ2CC7L<2C<7M02D97ME2EH7N42F@7N<2FG7ND2G07NM2G47O52FJ7O>2FA7OD2FD7OH2FJ8062FG80=2F<80D2F280K2EH8122E@8172EA81=2EA81B2EA81G2EB81L2EB8222EC8272EC82>2EH82F2EL82N2F18352F783=2F;83F2FA83>2F=82K2F38292EI81G2E?81I2EA81L2EA81O2EC8212ED8242EE8222EB8242EE82=2E>82G2E58342DG83A2D:83N2CM84;2C@84I2C38562BG85C2B:8602AL86=2A?
1616J@2356K023>6K@23E6L023M6L@2466M024=6M@24F6N124M6NA2566O325D6OE26270726?70H26L70O27071627471L27;72C27D73:27K74128474H28;75?28D76628K76M29377C29978929@78G29H78O2:37972:>79?2:I79G2;479O2;?7:72;J7:?2<57:M2<D7;<2=47;J2=D7<92>47<G2>C7=62?37=E2?C7>72@27?32@>7?O2@J7@K2A57AD2A?7B32AA7BB2AD7C02AG7C?2AJ7CN2AM7D=2B07DK2B27DO2B67E42B:7E82B=7E=2BA7EA2BD7EI2BJ7F52C17FB2C97FH2C<7FG2C;7FF2C:7F@2C97F82C97F12C97EI2C97EB2C97ED2C77EG2C57EN2C87F62C<7F?2C>7FF2C<7FN2C;7G62C97G>2C77GE2C67GM2C47H52C37H=2C17HE2C07HM2BN7I42BM7I<2BK7IC2BE7J?2BI7K<2BM7L92C07L42CG7KK2D67KN2CO7L12CI7L>2CH7M42E57MJ2FC7N72FN7N?2FD7NG2F;7NO2F27O62F;7O=2FC7O?2FF7OI2FE8092FA80@2F680G2EL80N2EA8132EA8182EA81>2EB81C2EB81I2EC81N2EC8242EF82<2EK82D2F082L2F58342F:83;2F?83D2FD83L2FI83J2FH8372F>82D2F582=2F182@2F282C2F482E2F482H2F681N2EA8262EK82E2F182M2EK8362EC83>2E;83G2E48432DH84@2D;84M2CN85:2C@85H2C38652BG86B2B:
1626JL23N6K<2476KL24>6L<24F6LL24O6M<2566ML25?6N<25F6NM2606O?26>70126L70C27:71627H71<27K71G28072>28873528@73L28H74C29075:29876129@76G29F77=29L7832:378J2:97922:D79:2:O79B2;:79J2;E7:22<07::2<;7:B2<F7:J2=17;12=<7;:2=G7;B2>27<02>B7<>2?17<M2?A7=I2?M7>E2@87?A2@E7@=2A07A92A<7AO2AC7B>2AF7BL2AI7C;2AK7CJ2AN7D22B17D72B57D;2B97D@2B=7DK2BC7E72BK7ED2C27F02C97F=2CA7FI2CI7FJ2CI7FH2CH7FG2CF7FF2CF7FE2CE7FA2CD7F:2CC7F52CC7F82CA7F<2C@7F@2C>7FE2C?7FM2CC7G52CF7G=2CI7GE2CM7GM2CN7H52CL7H<2CK7HE2CI7HL2CG7I42CF7I<2CD7ID2CC7J02C?7J62C27J62B@7K32BC7KO2BG7K@2CC7JO2DB7K22D<7KC2D;7L72D;7LK2D<7MA2E;7N12F57N92EK7NA2EH7NG2F17NN2F97O42F@7O52FB7O<2FC7OM2F@80;2F;80B2F180J2EF8102EA8152EB81;2EB81@2EC81E2EC81K2EE8232EJ82:2EO82B2F482J2F98322F>83:2FC83B2FG83J2FM8422G18452G483C2FJ8322FA8342FC8372FD8392FE82:2EG81O2EE8242EM82J2F:8362F>83>2F783G2F083O2EH8472EA84?2E984H2E28522DH85?2D;85M2CN86:2CA
1636K824G6KH2506L82576LH25@6M825H6MH25O6N82686NH26?6O926K6OK27870>27F71028471B28B71J28G72928M73029473G29=74>29D75529L75L2:376B2:97782:@77N2:F78D2:L7952;579=2;@79E2;K79M2<67:52<A7:<2<L7:D2=77:L2=B7;42=M7;=2>87;D2>C7;M2>M7<72?57<D2?97=<2?F7>72@47?32@?7?O2@K7@K2A77AG2AC7B:2AG7BH2AJ7C52AM7C:2B17C?2B57CL2B=7D92BD7DE2BL7E22C37E>2C:7EK2CB7F82CI7FD2D17FM2D67FK2D57FJ2D47FI2D37FG2D27FF2D17FE2D07FB2CM7FH2CL7FN2CL7G12CI7G42CH7G82CF7G<2CE7GD2CI7GL2CL7H42D07H<2D47HD2D87HL2D<7I42D?7I<2D>7IC2D<7IK2D:7J92D57JI2CO7JJ2C?7JA2BG7JJ2B:7K?2BB7JM2C@7J=2D?7JJ2DI7K=2DL7L02DO7LD2DO7M82DO7ML2E<7N32E>7N92EG7N?2EO7NF2F77NJ2F=7NL2F?7NN2FA7O?2F>8002F:80>2F480E2EK80L2EB8112EB8162EC81<2EC81A2ED81I2EI8212EN8292F382@2F782H2F=8302FA8382FG83@2FK83H2G08402G58482G:84?2G?83N2G783I2G383K2G48382FA8252EC81O2EH8222F082M2FA83;2FG83G2FJ83O2FD8472F=84?2F584H2EN8502EF8582E>85@2E785I2E08612DH
1646KD25@6L425I6LD2606M42696MD26@6N426H6ND2716O42786OE27D70828370J28A71<28N71N29<72829B72K29I73B2:17492:97502:@75F2:G76<2:M7732;377I2;:78?2;@7952;G79@2<179H2<<79O2<G7:72=27:?2==7:G2=H7;12>07;;2>77;E2>?7;O2>F7<92>N7<C2?57=32?87=D2?;7>52??7>O2?O7?H2@>7@B2@M7A;2A<7B62AI7BA2AO7BN2B77C:2B=7CG2BE7D32BM7D@2C47DL2C<7E92CC7EE2CJ7F22D27F?2D97FK2DA7FN2DB7FM2DB7FJ2D?7FG2D<7FC2D97F@2D57F=2D27F@2D27FH2D37G12D27G=2D27GI2D27GM2D07H02CO7H32CM7H:2CO7HC2D37HK2D67I32D:7I:2D>7IB2DB7IJ2DE7J52DF7JC2DB7K22D?7K?2D<7K=2CM7K62C37JM2B;7JL2B?7JC2C@7J;2D@7JJ2DG7K>2DJ7L12DL7LD2DO7M82E27M@2E<7MK2E=7N12EE7N82EN7N>2F57N@2F97NB2F;7ND2F>7O12F;7OC2F88042F480A2EN80H2EE80M2EC8132EC8182ED81@2EH81G2EM81O2F18272F782?2F;82F2FA82N2FE8362FJ83>2FO83F2G483N2G98462G>84=2GC84E2GH84=2GC8472G=82B2EK8252EG8202EL81O2F382J2FD83A2G083L2G48472G784@2G084H2FJ8502FB8582F;85@2F385I2EK8612ED86:2E<0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008N>1MD8O21M>8OJ1M=90A1M;9191M:9201M892H1M793@1M5
1656L02696L@26B6M026I6M@2726N02796N@27A6O027J6O@28170228?70D28M71629:71H29I72:2:672G2:>73=2:F7442:N74K2;475A2;:7672;A76M2;G77C2;N7892<478O2<:79B2<A79L2<K7:52=27:?2=:7:I2=A7;32=I7;=2>07;G2>87<12>@7<;2>G7<E2>O7=32?37=C2?67>32?97>C2?<7?32??7?I2?I7@B2@87A;2@F7B12A47B>2A<7BK2AE7C82AM7CE2B77D12B?7D?2BG7DL2C07E82C97EE2CB7F12D27F=2DA7FI2DJ7FI2DG7FE2DC7FB2D@7F>2D=7F;2D97F82D67F62D47FA2D97FI2D;7FO2D97G92D97GF2D97H22D97H?2D87HI2D77HL2D67I12D67I92D97IA2D=7II2DA7J12DE7J@2DA7JN2D>7K02D=7K22D<7K42D<7K62D<7K42CL7JN2C17JH2B@7J@2CA7J82DB7JK2DE7K>2DH7L12DJ7LD2DL7LK2E87M12EE7MC2E;7MI2EC7N02EK7N52F37N72F57N92F77N:2F:7ND2F:7O52F67OF2F38072EO80D2EI80J2EC80O2ED8152EF81>2EK81F2F081M2F58252F:82=2F?82E2FD82L2FI8342FN83<2G383D2G883L2G=8442GB84<2GF84C2GK84I2GM83A2FF82;2EF8262EK8212F081M2F582H2FF83C2G78412G=84<2G@84H2GD8502G=8582G785A2FO85I2FG8622F@86:2F886B2F10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008MH1O88N61NK8NH1ND8O@1NC9081NA90O1N@91G1N>92>1N=9361N;93M1N:94E1N8
1666L<2726LL27;6M<27B6ML27K6N<2826NL28:6O<28C6OL28K70?29971129G71C2:57252:B72G2;07382;:73O2;A74E2;H75;2;N7612<476F2<:77;2<?7812<E78F2<K79;2=179K2=47:;2=77:K2=:7;52=B7;@2=J7;J2>17<32>97<=2>@7<G2>H7=22>O7=B2?27>22?47>B2?77?22?:7?B2?=7@22??7@B2?B7A;2@17AM2@;7B:2@E7BF2@M7C32A57C?2AE7CJ2B57D62BD7DB2C47DN2CD7E92D37EE2DC7F22DJ7F@2DK7F@2DH7F=2DE7F92DA7F62D>7F22D;7EO2D77F82D;7FC2D@7FK2DB7G02D@7G62D?7GB2D?7GN2D?7H;2D?7HH2D>7I42D>7IA2D=7IH2D<7J02D@7J;2D?7J>2D?7J@2D>7JB2D=7JD2D=7JG2D<7JI2D<7JK2D;7JM2D;7JJ2CJ7JD2C07J=2CC7J82D@7JK2DC7K?2DE7L22DH7L72E57L=2EA7LB2EM7M42EE7MA2EA7MH2EJ7ML2EO7MM2F17MO2F47N12F77N62F77NG2F47O82F17OI2EM80:2EJ80F2ED80L2EE8142EJ81<2EO81D2F481K2F98232F>82;2FC82A2FF82H2FJ82N2FM8352G083;2G483B2G883I2G;83O2G?8412G=83M2G382J2EO82<2EJ8272EN8222F481L2F982F2FI83A2G:8472GF84B2GJ84M2GM8582H185A2GJ85I2GC8622G<86:2G486B2FM86J2FE0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008M320L8M@20@8MM2038N?1OJ8O61OI8ON1OG90E1OF91=1OD9241OC92L1OA93D1O@94;1O>9531O=
1666L827C6LH27K6M82846MG28;6N828D6NH28K6O82936OH29<70929F70K2:371=2:A71O2:O72A2;=7332;K73F2<274;2<87512<>75F2<D76;2<J7702=077E2=578;2=;7902=@79B2=E7:22=H7:A2=L7;12=O7;A2>27<12>57<?2>:7<I2>A7=32>I7=A2>M7>12>O7>A2?37?12?67?A2?97@12?<7@>2?@7@K2?E7A82?J7AD2@87B02@H7B<2A87BG2AG7C32B77C?2BG7CJ2C67D62CF7DB2D67DM2DE7E;2DJ7EI2DK7F72DK7F82DI7F42DE7F12DB7EM2D?7EJ2D<7EN2D=7F92DB7FD2DH7FL2DH7G22DH7G72DF7G>2DE7GJ2DE7H72DE7HD2DD7I02DD7I=2DD7IJ2DD7J42DD7J32DA7J32D?7J42D>7J72D=7J92D<7J;2D<7J=2D<7J?2D;7JA2D:7JC2D:7JA2CI7J:2CD7J82D>7JL2DA7K>2DD7KC2E07KI2E=7KO2EI7L42F67LD2EO7M72EE7M@2EH7MB2EL7MD2EN7MF2F17MH2F37MI2F57N92F27NJ2EN7O<2EK7OM2EH80@2EF80F2EE80M2EH8132EL81:2F081A2F381G2F781N2F;8242F>82;2FA82A2FE82H2FH82N2FL8342G083;2G38392FK8352FA8322F782B2EH82=2EN8282F38222F781M2F=82C2FL83>2G=84:2GM84G2H38522H685=2H:85I2H>8622H786:2H086B2GH86J2GA8732G:0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008L>22A8LK2248M821G8ME21:8N520O8NL20N8OD20L90<20K91320J91K20I92B20G93:20F94120D94I20C
1676LD28<6M328D6MC28M6N42946ND29=6O429D6OD29L7032:570E2:?7172:N71I2;<72;2;I72N2<773@2<B7422<H74F2<N75;2=47602=:76E2=@77:2=E77O2=K78E2>17992>679H2>:7:82>=7:H2>@7;82>C7;H2>F7<72>I7<G2>L7=42?07=C2?57>02?:7><2??7>I2?D7?62?I7?C2?M7@02@27@=2@77@K2@>7A82@J7AD2A:7B02AJ7B<2B97BG2BI7C32C87C>2CH7CJ2D87D62DG7DC2DK7E12DK7E?2DK7EM2DL7EO2DJ7EK2DG7EH2DC7EE2D@7ED2D?7EO2DD7F;2DI7FF2DO7FN2E07G32DO7G92DM7G?2DL7GF2DK7H32DK7H@2DK7HL2DJ7I<2DK7IK2DM7J42DM7J42DJ7J32DH7J32DE7J32DB7J22D@7J22D>7J22D<7J42D;7J62D:7J82D:7J:2D97J72CH7J92D<7JJ2D@7JO2DL7K52E97K;2EE7K@2F17KF2F>7L52F97LG2EN7M72EF7M92EH7M;2EK7M=2EN7M@2F17MB2F57N12F57ND2F47O72F27OJ2F18052EK80:2E?80@2E@80F2ED80M2EH8132EK81:2EN81A2F281G2F681M2F98242F=82:2FA82A2FD82E2FD82A2F:82>2F082:2EG82<2EM82C2EL82>2F18282F78232F;81N2FA82A2FN83<2G?8472H084L2H<8582H?85C2HC85O2HG86:2HJ86B2HD86J2H<8732H587;2GN0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008KH2458L523H8LB23;8LO22N8M<22A8MK2258NC2248O:22290222190I21O91A21N92821L93021K93H21I94?21H95721G
1686LO2956M?29=6MO29F6N@29M6O02:66O@2:=6OO2:F70@2:N7112;:71C2;H7252<572H2<D73:2=173L2=774?2==7522=D75E2=J76:2>076O2>677D2>;7892>A78O2>F79?2>J79O2>M7:>2?07:N2?47;>2?77;N2?:7<82??7<B2?C7=22?H7=B2?L7=O2@17><2@57>H2@;7?52@?7?B2@E7@02@I7@>2A17@L2A<7A82AK7AD2B;7B02BK7B;2C:7BG2CJ7C22D97C>2DI7CL2DK7D:2DK7DH2DL7E62DM7ED2DN7EF2DK7EC2DH7E@2DE7E<2DB7EF2DE7F12DK7F=2E17FH2E67FO2E77G52E57G:2E57G@2E37GF2E27GO2E17H=2E27HM2E37I=2E57IL2E67J52E67J52E37J42E17J42DO7J32DK7J32DI7J22DG7J22DD7J22DB7J12D?7J12D<7J02D:7J12D97J62D<7J;2DH7JA2E57JG2EA7JL2EN7K22F;7K82FG7KF2FC7L82F87M22EH7M52EJ7M82EN7M;2F37M>2F87MA2F>7MM2F>7N@2F=7O42F;7OE2F97OI2EN7ON2EB8032E88092E;80?2E?80F2EC80L2EF8132EJ81:2EN81@2F181G2F481M2F881M2F281I2EI81F2E?81H2EH81K2F48222FE82D2G:82>2F58292F;8242F?81O2FD82>2G183:2GB8452H38502HC85=2HI85I2HL8642I086?2I386J2I78732I187;2HI87C2HB0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008K225I8K?25<8KM2508L:24C8LG2468M423H8MA23;8N923:8O02388OH23790@23591723491O23292F23193>22O94522N94M22L95E22K
1696M;29N6MK2:66N<2:?6NL2:F6O;2:O6OK2;670<2;?70L2;G71>2<57202<B72B2=07342==73F2=G7492=M74L2>375>2>:7602>@76D2>F7792>K77N2?178D2?77962?;79E2?>7:52?B7:E2?E7;12?I7;<2?M7;E2@27<02@67<@2@;7=12@>7=A2@C7=N2@H7>;2@M7>H2A17?52A77?B2A;7@12AF7@?2B07@L2B=7A82BM7AC2C<7AO2CL7B;2D<7BG2DK7C52DK7CC2DL7D12DL7D?2DM7DM2DN7E;2DO7E>2DL7E;2DI7E72DE7E<2DH7EH2DM7F32E27F>2E87FI2E=7G12E>7G62E=7G<2E;7GB2E:7GG2E97H02E:7H?2E;7HN2E<7I>2E>7IM2E?7J62E>7J52E<7J52E:7J42E77J42E57J42E27J32DO7J32DM7J22DK7J22DH7J12DE7J12DC7J32DD7J42DH7J22E27J22E>7J82EJ7J>2F77JC2FC7JI2FO7K72FM7KM2F?7LE2F27M32F17M62F67M92F;7M<2FA7M?2FF7MJ2FG7N=2FF7O02FD7O92F<7O>2F17OB2EE7OG2E;7OL2E38022E78092E;80?2E>80F2EA80L2EE8132EH8182EK8152EA8112E88142EA8172EO81;2F<81>2FJ81M2GA82A2H?82?2FF82:2F>8252FD81O2FH82<2G48372GE8432H684N2HF85B2I285N2I58692I986D2I<8702I@87;2ID87C2I>87L2I60000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008J=27>8JJ2718K726D8KD2678L225I8L?25<8LL2508M924C8MO24@8NG24>8O>24=90624;90M24:91E24892=24793424593L24494C24295;241
1696M72:?6MG2:G6N82:O6NG2;86O72;?6OG2;H7082;O70H2<87182<A71J2<N72<2=<72O2=J73A2>67432>=74F2>C7582>I75K2>O76=2?57702?;77C2?A7882?G78L2?L79<2?O79L2@27:52@77:?2@;7:I2@@7;32@E7;>2@I7;O2@M7<?2A17=02A67=@2A:7=M2A?7>:2AD7>G2AI7?42B07?C2B:7@22BC7@@2BO7@L2C?7A82CN7AC2D>7B02DK7B>2DK7BK2DL7C92DM7CG2DN7D62DN7DD2DO7E22DO7E62DN7E22DJ7E32DI7E>2DO7EI2E47F42E:7F@2E?7FK2EE7G22EE7G82ED7G=2EB7GC2EA7GK2EA7H52EC7H@2ED7HO2EE7I?2EG7IN2EH7J72EH7J62EE7J62EC7J52E@7J52E>7J42E;7J42E87J42E67J32E47J32E17J42E17J72E27J:2E37J;2E77J82EA7J62EK7J42F57J12F@7J52FK7J:2G87JI2G67K?2FL7L12FG7LC2FB7M42F>7M72FD7M:2FI7M=2FN7MG2G07N:2FO7NI2FJ7NM2F@7O22F47O72EI7O;2E>7O@2E27OE2DO7OL2E28022E68082E980?2E=80A2E:80=2E080@2E<80C2EI80G2F780J2FD80M2G18122G@81G2H>82<2I<82A2H=82;2FB8252FG8202FL82:2G78352GG8402H884K2HI85F2I:8632I?86?2IB86J2IF8752II87@2IM87K2J18842IK0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008J428E8JB2888JO27K8K<27>8KI2718L626D8LC2678M025J8ME25F8N=25D8O425C8OL25A90D25@91;25>92225<92J25;93B25994925895125695I255000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000:C2226:CH222:D>21M
1706MC2;86N32;@6NC2;H6O32<16OC2<87032<A70D2<H7132=171D2=;7262=I72I2>673;2>D73M2>L74?2?37522?975E2??7672?E76J2?K77<2@177O2@778B2@=78O2@A7992@E79C2@J79M2@N7:72A37:@2A87:L2A<7;=2A@7;M2AD7<>2AH7<N2AM7=?2B17=L2B67>92B:7>G2BC7?62BM7?E2C77@42CB7@@2D07@L2D@7A82DK7AF2DL7B42DL7BB2DM7C02DN7C>2DO7CL2DO7D;2E07DI2E07DM2DO7DJ2DK7E52E17E@2E67EK2E;7F62EA7FA2EF7FL2EK7G32EL7G92EK7G?2EJ7GG2EJ7H02EK7H:2EL7HD2EN7I02EN7I@2F07IO2F17J72F17J72EN7J62EK7J62EI7J62EG7J52ED7J52EA7J42E?7J52E>7J82E@7J;2EA7J=2EA7J@2EC7JB2EE7J?2EO7J=2F:7J:2FD7J82FN7J62G87J32GA7J82GE7JJ2G@7K<2G<7KN2G77L@2G37M32FM7M92G17M;2G77MD2G:7N62G87N=2FN7NA2FC7NF2F77NJ2EL7NO2EA7O42E57O82DJ7O>2DJ7OE2DN7OK2E17OI2DI7OL2E68002ED8032F18062F>80:2FK80=2G980@2GF80M2H<81B2I:8262J882B2J482<2GJ8262FK8212G08272G:8322GJ83N2H;84I2HL85D2I<8692IH86D2IK86O2IO87:2J287E2J68812J:88<2J>0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008IL29L8J929@8JF2928K328E8K@2888KN27K8L;27>8LH2728M;26K8N326J8NK26H8OB26G90:26E91126D91I26B92A26A93826?93O26>94G26<95?26;966269000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000:B722M:BN22H:CD22D:D;22>:E122::EH225:F>221:GE221:I3224:JA226:KO228
1716N02;O6N?2<96NO2<A6O?2<J6OO2=170?2=:70O2=A71?2=J7212>572C2>C7352?173G2?<74:2?B74L2?H75?2?N7612@476D2@:7762@@77I2@F78:2@L78C2A178K2A67942A;79<2AA79D2AF79N2AJ7::2AO7:K2B37;<2B77;L2B;7<=2B@7<M2BC7=>2BH7=L2BM7>:2C77>I2CA7?82CL7?G2D67@42DB7@A2DK7@O2DL7A=2DM7AK2DN7B92DO7BG2DO7C52DO7CC2E07D12E17D?2E27DE2E07DK2E27E62E87EA2E=7EL2EB7F82EH7FC2EN7FN2F37G52F37G;2F27GB2F17GK2F37H52F47H?2F57HI2F77I32F77IA2F97J02F;7J82F:7J82F77J72F47J72F27J62F07J62EM7J62EL7J92EM7J<2EN7J>2EO7JA2F07JC2F17JG2F27JI2F47JG2F>7JC2FD7J?2FL7J<2G37J92G:7J52G@7J:2GE7JF2GF7K22GF7K>2GH7KL2GG7L=2GC7M02G>7M:2G?7M@2GC7MM2G=7N12G17N62FF7N:2F;7N?2EO7NC2ED7NH2E87NL2DN7O12DB7O52DC7O92E17O<2E>7O?2EK7OC2F87OF2FF7OI2G37OM2G@8002GM8032H;80H2I981=2J681M2JK82;2JO82=2I@8272G68222G48252G<8302GM83L2H>84G2HO85B2I?86=2J086I2J58742J887@2J;87K2J?8862JC88B2JG0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008ID2;48J12:F8J>2:98JK29L8K829@8KE2938L328F8L@2898M22818MI2808NA27N8O827M90027K90H27J91?27H92627G92N27E93F27D94=27B95527A95L27?0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000009CK2;C9DB2;69E82:I9EO2:<9FF29O9G=29B9H42959HK28G9IA28A000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000:A=23D:B423?:BJ23::CA235:D7231:DN22L:ED22H:F;22C:GA22C:HO22F:J>22H:KM22K:KL22?:KK224:KK21G:KN20J:L21ON
1726N=2<F6NK2=26O;2=:6OK2=B70;2=J70K2>371;2>:71K2>C72=2>O72O2?=73A2?K7442@274G2@87592@>75K2@D76>2@J7712A077C2A67822A;78=2A@78E2AE78M2AJ7962AO79>2B479F2B:7:12B?7:A2BC7;12BH7;B2BM7<22C07<B2C57=22C97=B2C?7>12CI7>?2D37>N2D>7?<2DH7?K2E07@92E17@F2E17A42E17AB2E17B02E17B>2E17BL2E17C:2E17CH2E27D62E27DB2E57DM2E:7E82E?7EC2EE7EN2EJ7F92EO7FD2F47FO2F:7G62F;7G=2F:7GG2F;7H02F<7H:2F>7HD2F>7HN2F@7I72FA7IB2FB7J12FD7J92FC7J82F@7J82F>7J72F;7J72F97J:2F;7J=2F;7J?2F=7JB2F>7JD2F>7JG2F@7JG2F@7JG2FA7JG2FA7JI2F>7JE2FD7JB2FK7J>2G27J;2G:7J72G@7J;2GD7JH2GF7K42GF7K@2GH7KK2GI7L72GJ7LC2GK7M02GL7M=2GK7MA2G@7MF2G47MK2FI7N02F>7N52F17N:2EF7N?2E;7ND2DO7NF2DG7ND2DH7NG2E77NK2EG7NO2F77O42FG7O82G77O<2GF7O@2H57OC2HC7OM2I680<2IE80K2J581:2JD81G2JD8232J28292HL8222G88232G?82N2H083I2H@84D2I185?2IB86;2J386O2J>87:2JA87E2JD8812JH88<2JL88G2K00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008I;2<;8IH2;N8J62;A8JC2;48K02:G8K=2::8KJ29M8L729@8LH2978M?2958N72938NO2928OF29090>29091528N91M28M92E28K93<28J94328H94K28G95C28E96:28D0000000000000000000000000000000000000000000000009<N2869=F2849>>2839>M28G9?<29=9?K2:29@:2:G9@I2;<9A82<29AG2<G9B:2<L9C02<?9CG2<29D>2;E9E52;89EL2:K9FC2:>9G92:19H029C9HG2989I>2939J428O000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000:@C24::A:246:B0241:BG23L:C=23G:D423C:DJ23>:EA23::F7235:G>235:HL237:J;23;:KI23=:KJ232:KI22G:KH22;:KL21>:KO20A:L31OD
1736NJ2==6O82=I6OG2>37072>;70G2>C7172>L71G2?37272?<72J2?I73<2@873N2@A74@2@G7532@M75E2A37682A976J2A?77<2AE77K2AJ7872AN78?2B478G2B978O2B>7972BC79?2BH79M2BN7:<2C37:L2C77;<2C<7;L2C@7<<2CE7<M2CI7=<2CN7=K2D87>:2DB7>H2DL7?72E77?E2E=7@32EA7@A2EA7@O2EA7A=2EA7AK2EA7B92EA7BG2EA7C42EA7CB2EA7CO2E@7D<2EA7DI2EE7E62EH7EC2EL7EO2F17F;2F77FF2F<7G12FA7G82FA7GB2FC7GL2FD7H52FE7H?2FG7HI2FG7I32FI7I<2FJ7IF2FK7J22FM7J92FL7J92FI7J92FG7J;2FG7J>2FI7J@2FJ7JA2FJ7JA2FK7JA2FL7JA2FL7JA2FM7JA2FM7JA2FM7JA2FN7JC2FJ7JH2FD7JD2FK7JA2G27J=2G97J:2G@7J=2GD7JI2GF7K52GF7KA2GG7KM2GI7L92GJ7LE2GK7LN2GM7M42GL7M;2G@7M@2G57ME2FI7MJ2F>7MO2F27N42EG7N92E;7N:2E47N:2DN7N82E07N52E37N72E?7N;2EO7N?2F?7NC2FO7NG2G?7NK2H07O02H?7O:2I07OI2I?8072IO80F2J>8132J881?2IF81K2I58242H98202GB82K2H383G2HC84B2I485=2IE8692J58742JG87?2JK87J2JN8862K188A2K588L2K98982K=0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008I32=B8I@2=58IM2<H8J:2<;8JG2;N8K42;B8KA2;48KO2:G8L>2:<8M62:;8MM2:98NE2:88O<2:69042:590L2:391C2:292:2:093229O93J29N94A29L95929K96029I96H29H97@29F98729E98O29C99F29B9:>29@9;529?9;M29=9<E29<9=<29:9>42999>F29F9?52:;9?D2;09@32;E9@B2<;9A12=09A@2=E9B62=;9BM2<N9CD2<A9D;2<49E22;G9EH2;:9F?2:M9G62:?9GM2:39HD29J9I:29E9J029@000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000:?I251:@@24L:A624G:AM24C:BC24>:C:24::D0245:DF241:E=23L:F423G:G:23G:HI23J:J723M:KE23O:KG23D:KF239:KE22N:KI221:KM215:L0208:L41O;:L71N>
1736NI2=G6O72>36OE2>?7032>K70C2?47132?<71C2?E7232?L72D2@67362@D73H2A174:2A774M2A=75@2AC7622AI76D2AO7742B477C2B87802B=7882BB78@2BG78H2BM7912C279:2C779I2C<7:72CB7:F2CG7;62CL7;G2CO7<72D47<G2D97=72D<7=F2DG7>52E17>C2E;7?12EE7??2EJ7?N2EO7@<2F27@J2F27A72F27AE2F27B32F27BA2F27BO2F27C=2F27CG2EK7D52EN7DB2F27DO2F67E<2F:7EI2F>7F62FB7FD2FF7G12FJ7G;2FK7GE2FL7H02FM7H:2FN7HD2G07HN2G07I72G27IA2G37IK2G47J42G67J:2G57J<2G57J=2G67J=2G67J=2G77J=2G77J=2G87J=2G87J=2G97J=2G:7J<2G:7J<2G:7J<2G;7J<2G;7J>2G:7JB2G37JF2FL7JC2G17J?2G97J<2G@7J>2GD7JK2GE7K72GF7KC2GG7KO2GI7L:2GJ7L@2GM7LE2H07LL2GM7M32G@7M:2G57M?2FJ7MD2F>7MI2F37MN2EG7MN2EA7MN2E;7MN2E57ML2E87MI2E;7MG2E>7MG2EH7MK2F77MO2FH7N32G87N72GH7N?2H97NI2HI7O52I:7OD2II8022J880?2IK80K2I:8172HI81C2H781O2GE82I2H583E2HF84@2I785;2IH8662J88712JI87D2K48802K788;2K:88F2K>8922KB89=2KF89H2KJ0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008HJ2>I8I72>=8IE2>08J22=C8J?2=68JL2<H8K92<;8KF2;N8L42;B8LL2;A8MC2;?8N;2;>8O32;<8OJ2;;90B2;99192;89212;692I2;593@2;39472;294O2;095G2:O96>2:M9762:L97M2:K98E2:I99=2:H9:42:F9:L2:E9;C2:C9<;2:B9=22:@9=J2:?9>?2:D9>N2;99?=2;N9?L2<D9@;2=99@J2=N9A<2>79B32=J9BJ2==9C@2=09D72<C9DN2<69EE2;I9F<2;<9G32:O9GI2:B9H@2:<9I62:79IM2:39JC29N000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000:>O25G:?F25C:@<25>:A3259:AI255:B?250:C624L:CM24G:DC24B:E:24=:F0249:G624::HE24<:J424?:KB24A:KE247:KD23K:KC23@:KF22E:KJ21H:KN20K:L11OO:L51O2:L81N6
1746O62>=6OD2>I7022?670A2?B70O2?M71?2@571O2@>72?2@E7302A073B2A>7442AG74G2AM75:2B275L2B876=2B>76L2BC77;2BG77J2BK7822C078:2C678B2C;78J2C@7962CF79D2CK7:32D07:B2D67;12D;7;B2D?7<22DC7<B2DH7=22DL7=A2E67=O2E@7>>2EJ7>L2F17?:2F77?H2F<7@62FA7@D2FC7A22FC7A@2FC7AN2FC7B<2FC7BJ2FC7C42F;7C@2F87CM2F<7D;2F@7DH2FD7E52FH7EB2FL7EO2G07F=2G37FK2G37FK2G67G62G77GA2G77GK2G97H62G:7HB2G;7HL2G<7I72G=7IB2G>7IM2G@7J72G=7J82G@7J82GC7J82GC7J82GD7J82GD7J82GE7J82GF7J82GF7J82GF7J82GG7J82GG7J82GH7J82GI7J82GI7J<2GB7J@2G;7JD2G47JB2G87J>2G?7J@2GD7JL2GE7K82GF7KD2GG7KL2GJ7L22GM7L72H07L=2H37LD2GM7LK2GA7M22G67M92FJ7M>2F>7MB2F47MB2EN7MB2EH7MB2EC7MB2E>7M?2EA7M=2ED7M:2EG7M82EJ7M72F17M;2FA7M?2G07ME2G@7MO2H27N92HC7NC2I37O02IC7O?2J27OK2I@8072HO80C2H<81;2H58242H982H2H=83B2HI84=2I98582IJ8642J;86O2JL87J2K=8852K@88A2KD88L2KG8972KK89B2KO89M2L38:92L78:D2L:0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008GH2@K8H52@>8HB2@18HO2?D8I<2?78II2>J8J72>=8JD2>08K12=C8K>2=68KK2<I8LB2<G8M:2<E8N12<D8NI2<B8O@2<A9082<?9102<>91G2<<92?2<;9362<993N2<894F2<695=2<59642<396L2<197D2<098;2;N9932;M99J2;K9:B2;J9;:2;I9<12;H9<I2;F9=@2;E9>82;C9>G2<79?62<L9?E2=B9@42>79@C2>L9A82>F9B02>99BF2=L9C=2=?9D42=29DK2<D9EA2<89F82;K9FO2;>9GF2;29H<2:N9I32:I9II2:E9J?2:@9K62:<000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000:>526>:>L269:?B265:@8260:@O25L:AF25G:B<25C:C325=:CI259:D?254:E6250:EL24K:G324K:HB24N:J0250:K>253:KC24J:KB24=:KA242:KD237:KG22;:KK21?:KO20B:L21OE:L91O9:L@1O1
1746O52>G6OC2?37012?@70@2?L70N2@871<2@E71K2@N72;2A772K2A>73<2AJ73O2B674A2B<7542BB75F2BH7652BM76D2C17732C677B2C:77L2C@7842CD78<2CI78D2CO7912D479@2D979O2D?7:>2DE7:L2DJ7;<2DO7;L2E37<<2E87<L2E;7=;2EE7=J2EO7>92F97>F2F>7?42FD7?B2FI7@12FN7@?2G47@M2G47A:2G47AH2G47B72G47BB2FM7BL2FD7C92FF7CF2FJ7D32FM7D@2G17DM2G67E:2G:7EH2G<7F62G<7FE2G<7FB2G?7FF2GA7G12GB7G<2GC7GF2GD7H12GF7H<2GF7HH2GH7I22GI7IF2GH7IM2GE7IO2GA7J12GB7J22GH7J32GN7J32H27J32H27J32H37J32H37J32H37J32H47J32H57J32H67J32H67J32H67J62H17J:2GJ7J>2GC7JB2G=7JA2G?7JB2GD7JN2GE7K92GF7K>2GJ7KD2GM7KI2H07KO2H37L42H77L;2GM7LC2GA7LJ2G67M12FJ7M62FA7M62F;7M62F67M62F07M62EJ7M62EF7M32EI7M12EL7LN2EO7LK2F27LI2F57LF2F97LK2FI7M52G:7M?2GJ7MI2H<7N22HL7N<2I=7NJ2IE7O72I47OL2HJ80H2HE81C2H?82:2HO82K2IH83B2I584;2I<8562IM8622J>86M2JN87H2K@88;2KI88F2KM8912L089<2L489H2L88:32L<8:>2L@8:J2LC8;52LF0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008G52B;8G?2B28GL2AE8H92A88HG2@K8I42@>8IA2@18IN2?D8J;2?78JH2>J8K52>=8KB2>08L82=M8M02=K8MG2=J8N?2=H8O72=G8ON2=E90F2=D91=2=B9252=@92M2=?93D2==94;2=<9532=:95K2=996B2=797:2=69812=498I2=399A2=19:82=09;02<N9;G2<M9<?2<K9=62<J9=N2<H9>@2=59>O2=J9?>2>@9?N2?59@>2?B9A52?59AL2>H9BC2>;9C92=N9D02=@9DG2=49E>2<G9F52<:9FL2;M9GB2;E9H82;@9HO2;<9IF2;79J<2;29K32:M000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000:=;275:>1270:>H26K:??26G:@526B:@L26>:AB269:B8264:BO25O:CE25K:D<25F:E225B:EI25=:G025=:H>25@:IM25C:K;25F:K@25<:K@250:K?24E:KB23K:KE22N:KI222:KL215:L220C:L920;:L@203:LG1OL:LN1OD
1756OB2?>7002?J70?2@670M2@B71;2@N71I2A;7272AG72G2B07372B773I2BD74;2BL74N2C275>2C675M2C;76<2C@76K2CD77:2CI77E2CN77M2D37852D978>2D=78M2DC79<2DH79K2DN7:92E37:H2E87;72E>7;G2EB7<72EG7<G2EK7=62F47=E2F>7>32FE7>A2FJ7>O2G07?=2G67?K2G;7@92G@7@G2GD7A52GD7AC2GD7AO2G@7B92G77BD2G07C12G47C>2G77CK2G;7D92G?7DF2GC7E42GD7EB2GE7F12GE7F@2GE7F<2GH7F62GL7FA2GM7FL2GN7G72GO7GB2H07GL2H17H72H37HJ2H27I=2H07ID2GN7IF2GJ7IH2GE7IJ2GC7IK2GJ7IL2H07IM2H67IM2H<7IN2H@7IN2HA7IN2HB7IN2HB7IN2HC7IN2HC7IN2HC7J02H?7J42H97J82H27J<2GK7J@2GD7JC2GD7JK2GF7K02GJ7K62GM7K;2H07KA2H47KF2H77KL2H97L32GM7L:2GB7LB2G77LF2FN7LI2FG7LI2FB7LI2F=7LI2F77LJ2F17LI2EN7LF2F17LD2F47LB2F77L?2F;7L=2F>7L=2FA7LA2FI7LJ2G97M32GI7M<2H97ME2HI7MN2I87N>2I@7O:2I:8052I58102HO81L2HI82=2IE82J2JA83D2J=84<2IN8542J085O2JA86J2K187E2KB88@2L388K2L68962L:89B2L=89M2LA8:92LE8:D2LI8:O2LL8;:2M08;E2M38<12M70000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008FM2C?8G52C78G>2C08GF2BH8GN2B@8H72B:8H?2B28HK2AE8I82A88IF2@K8J32@>8J@2@28JM2?E8K:2?88KO2?38LF2?18M=2>O8N52>N8NM2>L8OD2>K90<2>I9142>H91K2>F92C2>E93:2>C9422>B94J2>@95A2>?9682>=9702><97H2>:98?2>99972>799N2>69:F2>49;>2>39<52>19<M2>09=D2=N9>92>39>H2>I9?72?>9?G2@39@;2@19A12?D9AH2?79B?2>J9C62>=9CM2>09DD2=C9E:2=69F12<I9FH2<;9G?2<79H52<29HL2;N9IB2;H9J82;D9JO2;?9KF2;;000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000::>289:;4284:;J280:<A27K:=827G:=N27B:>E27>:?;279:@1275:@H26O:A>26K:B526F:BK26B:CB26=:D8269:DO264:EE25O:FL25O:H;262:II265:K8268:K>25N:K=25C:K=257:K?24>:KC23A:KF22E:KK21M:L221E:L921>:L@216:LG20O:LN20G:M520@
1756OA2?I6OO2@570>2@A70L2@M71:2A971H2AE7262B172D2B=7332BI73C2C17452C;74H2CA7572CF75F2CJ7642CO76C2D37732D877?2D<77G2DB77O2DG78:2DL78H2E27982E779F2E<7:52EA7:D2EG7;32EL7;B2F17<12F67<B2F;7=12FB7=@2FM7=N2G27><2G77>J2G=7?72GB7?F2GG7@42GM7@B2H27A02H67A<2H27AF2GI7B12G@7B=2G=7BJ2GA7C72GE7CD2GI7D12GM7D@2GM7DN2GM7E=2GN7EK2GN7F:2GN7F72H17EN2H57F12H77F<2H97FG2H97G22H;7G=2H<7GN2H<7HC2H:7I22H97I:2H67I=2H37I?2GN7IA2GJ7IC2GF7ID2GJ7IE2H07IF2H77IG2H=7IG2HC7IH2HI7II2HO7II2HO7II2I07II2I17IJ2HO7IN2HH7J22HA7J62H:7J;2H07J@2GG7JC2GJ7JH2GM7JM2H07K32H47K82H77K>2H:7KD2H97KK2GM7L12GC7L52G<7L92G47L<2FM7L<2FH7L=2FD7L=2F>7L>2F97L=2F77L:2F:7L92F=7L92FA7L:2FD7L;2FG7L<2FK7L?2G37LH2GC7M12H27M:2HB7MC2I27MO2ID7NF2J87OB2IM80=2IB8182I98202I?82=2J;82I2K783A2K:84<2K28562JH85M2JD86H2K487C2KE88?2L68912L?89<2LC89G2LF8:22LJ8:>2LN8:I2M28;42M58;?2M98;K2M<8<62M@8<B2MD0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008FM2D:8G52D38G>2CL8GF2CD8GN2C<8H72C58H?2BM8HG2BF8HO2B?8I72B78I@2B08IK2AF8J82A98JE2@L8K22@>8KD2@88L<2@78M42@58MK2@48NC2@28O:2@19022?O90J2?N91A2?L9292?K9312?I93H2?H94?2?F9572?E95O2?C96F2?B97>2?@9852??98M2?=99E2?<9:<2?:9;42?99;K2?79<C2?69=:2?49>22?39>B2?G9?02@<9?@2@M9@72@@9@N2@39AE2?F9B<2?99C32>L9CI2>?9D@2>29E72=D9EN2=79FE2<N9G;2<I9H12<D9HH2<?9I?2<;9J52<69JL2<19KB2;M000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000:7@29>:8729::8M295:9C290:::28K:;028G:;G28B:<>28=:=4289:=J284:>A280:?727K:?N27F:@D27A:A;27=:B1278:BH274:C>26O:D526K:DK26E:EB26A:FI26B:H726D:IF26G:K426I:K<26A:K;265:K:25I:K=251:K@245:KD238:KK22O:L222H:L922@:L@228:LG221:LN21I:M521B:M<21:
1756OA2@26ON2@?70=2@K70K2A87192AD71G2B07252B<72C2BH7312C473?2C@73O2CK74@2CO74O2D475>2D975M2D=76;2DB76J2DF7792DK77A2E077I2E57852E;78D2E@7932EE79B2EK7:12F07:@2F67:N2F;7;=2FA7;L2FF7<<2FJ7<K2G17=:2G:7=H2G?7>62GD7>D2GJ7?22GO7?@2H47?N2H97@<2H>7@I2HA7A42H;7A>2H27AH2GJ7B52GK7BB2GO7BO2H37C=2H67CK2H67D:2H67DI2H67E72H77EF2H77F42H77F12H:7EH2H>7EA2HB7EL2HC7F72HD7FB2HF7G22HF7GF2HD7H:2HC7HH2HA7HO2H>7I42H;7I62H67I82H37I:2GN7I<2GJ7I=2GL7I>2H27I?2H87I@2H>7IA2HD7IB2HK7IB2I17IC2I77ID2I=7IE2I>7II2I57IN2HL7J32HB7J82H87J=2GN7JA2GO7JD2H27JH2H57JL2H87K02H:7K52H>7K<2H97K@2H17KD2GJ7KH2GB7KL2G:7KO2G37L02FN7L02FI7L12FD7L12F?7L62F?7L62FC7L72FG7L82FJ7L92FN7L92G17L:2G57L=2G<7LF2GL7LO2H<7M82HL7MB2I<7N72J57NM2JN7OI2JE80D2J;81@2J08212J882=2K182I2KM83=2L88482KO8532KG85O2K>86F2K787A2KG88<2L98962LH89A2LL89L2M08:82M38:C2M78:N2M;8;:2M>8;E2MB8<02MF8<<2MI8<G2MM8=22N18=;2MK0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008FM2E68G52DO8G>2DG8GF2D@8GN2D98H72D18H?2CI8HG2CB8HO2C:8I82C38I@2BK8II2BD8J12B=8J92B58JA2AM8JI2AF8K;2A>8L32A<8LJ2A;8MA2A98N92A88O12A68OH2A590@2A49172A291O2A192G2@O93>2@N9462@L94N2@K95E2@I96<2@H9742@F97L2@E98C2@C99;2@B9:22@@9:J2@?9;B2@=9<92@;9=12@:9=H2@89>;2@E9>J2A:9?=2A<9@42@O9@K2@B9AA2@59B82?H9BO2?;9CF2>N9D=2>@9E42>39EJ2=G9FA2=?9G72=:9GN2=69HE2=19I;2<M9J12<H9JH2<D9K>2<?000000000000000000000000000000000000000000000000000000000000000000000000000000000000:602:9:6F2:5:7<2:0:8329K:8I29F:9@29B::729=::M299:;C294:<:290:=028K:=G28F:>=28A:?428=:?J288:@@284:A727O:AN27K:BD27F:C;27A:D127<:DG278:E>273:FE274:H4276:IB279:K127;:K:273:K926H:K826<:K:25E:K>24H:KD24::KK242:L223J:L923C:L@23;:LG234:LN22L:M522E:M<22=:MC225
1766ON2@I70<2A570J2AA7182AM71F2B:7242BF72B2C37302C?73>2CK73L2D77482D?74G2DC7562DH75E2DL7642E176C2E57722E:77:2E>77B2ED7812EI78@2EN78O2F479>2F:79M2F?7:;2FD7:J2FJ7;92FO7;H2G47<72G:7<F2G@7=42GF7=C2GL7>12H17>>2H67>K2H97?82H<7?F2H?7@32HC7@@2HF7@K2HD7A62H<7A@2H57AN2H97B;2H<7BI2H?7C72H?7CF2H?7D52H?7DC2H?7E22H@7E@2H@7EN2H@7EL2HC7EC2HG7E:2HK7E<2HN7EG2HO7F62HO7FJ2HN7G?2HL7H32HL7H=2HI7HE2HF7HK2HC7HM2H?7HO2H;7I12H67I32H37I52GN7I62GM7I72H37I82H97I92H?7I:2HF7I;2HL7I<2I27I<2I87I@2I=7IF2I=7IK2I37J02HI7J52H?7J:2H67J>2H37JA2H67JE2H97JI2H<7JM2H?7K02HB7JO2H?7K32H77K72H07K;2GH7K?2G@7KB2G97KC2G47KC2FO7KD2FJ7KK2FI7L62FJ7L62FM7L62G07L62G47L72G87L82G<7L82G?7L;2GF7LD2H57LM2HE7M62I57MJ2IM7N@2JF7O62K@8002K>80L2K481H2JI8252K882?2KM82I2LC83:2M58452LL8502LD85L2L<86G2L387>2KJ88:2L;8952LL89F2M58:22M98:=2M<8:I2M@8;42MD8;?2MG8;J2MK8<52MO8<A2N28<L2N68=72N:8=C2N>8=K2N80000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008FE2F:8FM2F28G52EK8G>2EC8GF2E;8GO2E48H72DL8H?2DE8HG2D>8HO2D68I82CO8I@2CG8II2C@8J12C88J92C18JA2BJ8JL2B:8K<2AG8KJ2AG8LA2AC8M@2A>8N<2A>8NM2AB8O>2AG8OO2AK90@2B09112B491E2B692=2B49342B493L2B294D2B195;2AO9632AM96K2AL97B2AJ9892AI9912AG99I2AF9:@2AD9;82AC9;O2AA9<G2A@9=?2A>9>42AC9>C2B89?92AK9@12A>9@G2A19A>2@D9B52@79BL2?J9CB2?<9D92>O9E02>C9EG2>69F>2>19G42=M9GJ2=H9HA2=D9I72=?9IN2=:9JD2=69K;2=19L12<L000000000000000000000000000000000000000000000000000000000000000000000000:552;0:5L2:L:6B2:F:792:B:802:=:8F2:9:9<2:4::32:0::I29K:;@29F:<629A:<M29=:=C298:>9293:?028O:?G28J:@=28F:A428A:AJ28<:B@287:C7283:CN27N:DD27J:E;27E:FB27F:H127H:I?27K:JO27N:K727E:K727::K626N:K8268:K=25D:KD25<:KK255:L224M:L924E:L@24=:LG246:LN23N:M523G:M<23?:MD237:MK230
1766OM2A470;2A@70I2AL7172B871E2BD7232C072A2C<72O2CI73=2D573K2DA7472DF74B2DM74N2E675=2E;75L2E?76;2ED76J2EH7742EN77>2F377M2F878<2F>78J2FC79:2FH79H2FM7:72G37:F2G87;42G=7;D2GC7<22GI7<A2GM7<N2H07=;2H47=H2H77>52H:7>B2H>7>O2HA7?<2HD7?I2HG7@62HK7@C2HN7@M2HE7A92HC7AF2HF7B52HH7BC2HH7C22HI7C@2HI7CN2HI7D=2HI7DL2HI7E:2HI7EI2HI7EF2HL7E=2I07E52I47DM2I97E:2I97EN2I87FC2I67G72I57GK2I47H32I27H:2HO7HB2HL7HD2HH7HF2HD7HH2H?7HJ2H;7HL2H77HN2H37I02GN7I02H47I12H:7I22HA7I32HG7I42HM7I52I37I;2I77IA2I:7IG2I;7IL2I17J12HG7J62H=7J;2H87J>2H;7JB2H>7JF2H@7JJ2HC7JJ2HB7JI2H?7JH2H;7JJ2H67JN2GN7K22GF7K62G>7K62G:7K72G47K<2G37KG2G47L22G47L72G77L62G:7L52G>7L52GB7L62GF7L72GI7L92GO7LB2H?7LK2HO7M=2IE7N32J>7NI2K77O?2L18082L78132KL81N2KD8282L882B2LM82L2MB8392MM8422MJ84M2MB85I2M986D2M087?2LH8872L>8922LO89L2M>8:82MB8:C2MF8:N2MI8;92MM8;D2N18;O2N48<;2N88<F2N;8=22N?8==2NC8=H2NG8>32NJ8><2NE8>E2N=8>M2N58?52MN0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008FE2G58FM2FM8G52FG8G>2F?8GF2F78GO2F08H72EH8H?2EA8HG2E98I02E28I82DK8I@2DC8II2D<8J12D48J;2CG8JG2C58K42BC8K@2B18LG2A88L=2AJ8M52AG8ML2AC8ND2A?8O?2A;90E2A591I2A092<2A492M2A893>2A=93O2AA94@2AF9522AJ95C2AO9642B496E2B89772B=97H2BA9892BF98J2BJ99?2BL9:62BJ9:N2BI9;F2BG9<=2BF9=42BD9=L2BC9>?2BG9?62B:9?M2AM9@D2A@9A:2A39B22@F9BH2@89C?2?L9D62??9DM2?29EC2>H9F:2>C9G02>?9GG2>:9H=2>69I42>19IJ2=L9J@2=G9K72=C9KN2=>000000000000000000000000000000000000000000000000000000000000:3E2;K:4;2;G:522;B:5I2;=:6?2;8:752;4:7L2:O:8B2:K:992:F:9O2:B::F2:<:;<2:8:<22:3:<J2:0:=B29L:>929H:?229D:?I29@:@A29=:A9299:B1294:BI290:CA28M:D828I:E028E:E?28;:FL28=:H=28A:IN28D:JJ28H:K@28L:K;286:K427A:K626N:K=26F:KD26?:KK267:L225O:L925H:L@25@:LH259:LO251:M624J:M=24B:MD24::MK242:N223J
1766OL2A>70:2AJ70H2B67162BB71D2BO7222C;72@2CG72N2D373<2D?73J2DK7482E174A2E374L2E;7582EE75D2EN7632F276B2F776N2F<77:2FA77I2FG7872FL78F2G17952G779D2G<7:32GA7:A2GF7;02GL7;>2H07;M2H57<;2H:7<H2H<7=42H>7=@2H?7=L2HA7>92HC7>F2HF7?22HI7??2HL7?M2HO7@:2I27@E2HN7A22I07A@2I17AO2I17B=2I17BL2I27C:2I27CI2I27D82I27DF2I27E52I27EC2I27EA2I57E82I97DO2I=7DH2IA7E22IB7EG2I@7F;2I?7G02I>7GA2I<7GH2I:7H02I77H72I57H:2I17H<2HL7H?2HH7HA2HD7HC2H?7HE2H;7HF2H77HH2H37HJ2H67HK2H<7HK2HB7HL2HH7I02HL7I62I07I=2I47IC2I77II2I97IN2HN7J32HD7J82H<7J<2H?7J?2HB7JC2HE7JD2HE7JC2HB7JC2H?7JB2H;7JA2H87J@2H47JD2GL7JI2GD7JI2G@7JN2G=7K92G=7KD2G?7KO2G@7L82GA7L72GE7L62GH7L42GL7L52H07L52H37L72H87L@2HH7M02I<7MF2J67N;2JO7O12KI7OG2LB80?2LO81;2LD8212LC82;2M982E2MM8312N;83E2N18492MH84O2MK85F2N286@2MN87;2MF8872M=8902M289K2MB8:=2MK8:H2MO8;32N28;>2N68;J2N:8<52N=8<A2NA8<L2NE8=72NH8=B2NL8=M2O08>82O38>D2O78>M2O18?52NJ8?=2NB8?F2N;8?N2N38@62ML8@>2ME8@F2M=8@O2M68A72LN0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008FE2H18FM2GJ8G62GB8G>2G:8GG2G38GO2FL8H72FD8H?2F=8HG2F58I02EN8I82EF8I@2E>8IJ2E48J62DB8JB2CO8JO2C=8K;2BK8KG2B98LK2AA8M?2A88M02AN8MH2AJ8N@2AG8O82AC8OO2A?90G2A;91?2A792D2A193I2@L94N2@G95K2@E96<2@J96N2@N97?2A39802A898A2A<9922AA99C2AE9:42AJ9:F2AN9;72B39;H2B79<92B<9<J2B@9=:2BF9=D2BN9=M2C69>62C49>K2BH9?F2B<9@@2AO9A72AB9AN2A49BE2@H9C;2@;9D32?N9DI2?@9E@2?:9F62?69FM2?19GC2>L9H92>G9I02>C9IG2>>9J=2>:9K42>59KJ2>0000000000000000000000000000000000000000000000000000000:342<G:3M2<D:4D2<@:5<2<<:642<8:6L2<5:7D2<1:8<2;M:942;I:9K2;E::C2;B:;;2;>:<32;::<K2;6:=C2;2:>:2:O:?32:K:?J2:G:@B2:C:A:2:?:B22:<:BJ2:8:CB2:4:D:2:0:DM29I:E829=:EK293:G;297:HL29;:IL29?:JB29C:K729F:KN29J:KJ295:K<286:K=27I:KD27A:KL27::L3272:L:26K:LA26B:LH26;:LO263:M625L:M=25D:MD25<:MK255:N224M:N924F
1777092B470G2B@7152BM71C2C97212CE72?2D172M2D=73;2DJ73I2E67482E;74A2E>74K2EA7562EI75B2F475M2F>76:2FF76G2FJ7762G077D2G57832G:78A2G@7902GD79?2GJ79M2GN7:;2H37:J2H87;92H<7;G2HB7<52HF7<C2HJ7<O2HL7=;2HM7=H2HO7>42I07>@2I27>L2I37?82I57?E2I67@12I87@=2I97@L2I:7A:2I:7AI2I:7B82I:7BF2I;7C52I;7CC2I;7D22I;7DA2I;7DO2I<7E>2I<7E;2I?7E32IB7DJ2IF7DM2IH7E22II7E?2IH7F32IH7FH2IF7G72IE7G>2IB7GE2I?7GM2I=7H12I97H32I57H52I17H72HL7H92HH7H;2HD7H=2H?7H?2H;7HA2H77HC2H67HD2H=7HE2HC7HK2HF7I22HJ7I82HM7I?2I17IE2I57IK2I57J02HL7J52HB7J92HD7J<2HF7J?2HI7J>2HE7J=2HB7J<2H>7J;2H;7J;2H87J:2H47J92H17J;2GJ7J?2GF7JJ2GG7K52GI7K@2GJ7KK2GJ7L62GL7L82GO7L72H27L52H67L42H97L42H=7L52HB7LB2I57M82IN7MN2JH7ND2KA7O:2L:8002M380F2MH81B2M=8242MC82>2N882I2NI83=2N?8422N584E2ML8592MB85O2MB86F2MJ87=2N28842N988O2N289J2MI8:B2N48:M2N88;92N;8;D2N?8;O2NC8<:2NF8<F2NJ8=12NN8=<2O18=G2O58>32O98>>2O<8>J2O@8?52OD8?=2O>8?F2O78?N2NO8@62NH8@>2N@8@G2N88@O2N18A72MJ8A@2MB8AH2M;8B02M38B82LL8BA2LD0000000000000000000000000000000000000000000000000000000000000000000000000000008F52I<8F=2I48FE2HL8FM2HF8G62H>8G>2H68GG2GO8GO2GG8H72G@8H?2G88HG2G18I02FJ8I92FA8IE2EN8J12E<8J>2DJ8JJ2D88K62CF8KB2C38KO2BA8LO2AJ8M02B28M62B68MA2B48N:2B08O32AJ8OK2AG90C2AC91;2A?9222A;92J2A793B2A394D2@N95I2@I96N2@C9832@>9982@899L2@<9:=2@A9:N2@E9;?2@J9;I2A19<32A:9<<2AB9<E2AJ9<O2B39=82B;9=B2BC9=K2BK9>52C39>>2C89>G2BN9?02BC9?92B99@22AM9@L2AA9AG2A49BB2@G9C=2@;9DE2@69EB2?O9F52?D9FM2??9GF2?<9H?2?:9I82?89J12?69JJ2?39KC2?09L<2>N9M62>L000000000000000000000000000000000000:2>2>3:362=O:3N2=K:4E2=G:5=2=D:652=@:6M2=<:7E2=8:8=2=4:952=1:9L2<M::E2<I:;<2<E:<42<A:<L2<>:=D2<::><2<6:?42<1:?K2;N:@C2;J:A;2;F:B32;B:BK2;?:CC2;;:D;2;7:DF2:K:E12:?:E<2:2:F:29M:GK2:1:HN2:5:ID2:9:J:2:<:JO2:@:KE2:D:L;2:H:LF2:@:L=29I:KD28D:KL28<:L3284:L:27M:LA27E:LH27>:LO276:M626N:M=26G:MD26?:MK268:N3260:N:25H:NA25@
1777082B>70F2BJ7142C771B2CC7202CO72?2D<72L2DH73:2E473H2E@7482EG74B2EI74L2EK7552EN75A2F775L2FB7672FL76C2G77712G=77?2GB77M2GG78<2GL78J2H17992H679G2H:7:62H?7:D2HD7;22HI7;A2HN7;O2I37<>2I87<J2I:7=62I<7=C2I=7=O2I?7>;2I@7>G2IB7?32IC7??2IE7?L2IE7@72IB7@F2IB7A42IC7AC2IC7B22IC7BA2ID7BO2ID7C>2ID7CL2ID7D;2ID7DJ2IE7E82IE7E62IH7DM2IK7DN2IM7E32IN7E72IO7E<2J17EL2J07FA2IO7FL2IM7G42IJ7G;2IH7GB2IE7GH2IB7GJ2I>7GL2I97GN2I57H02I17H22HL7H42HI7H62HD7H82H?7H:2H<7H<2H87HA2H<7HG2H@7HM2HC7I42HG7I:2HK7I@2HN7IG2I27IM2I37J22HI7J62HI7J:2HK7J92HH7J82HE7J72HB7J62H>7J52H;7J42H77J32H47J32H27J32H07J;2H17JF2H37K12H37K<2H47KG2H67L22H67L92H97L82H<7L62H?7L52HC7L42HG7L52HM7LK2IE7MA2J?7N72K87NL2L27OC2LK8082ME80N2N>81J2N58272ND82B2O78362NM83J2NC84>2N:8522N085E2MF86:2M<86O2M987E2MB88=2MI8942N189J2N88:B2N?8;32NA8;>2NE8;I2NH8<42NL8<@2NO8<K2O38=62O78=B2O;8=M2O>8>82OB8>D2OE8>O2OI8?:2OM8?E3018?N2OK8@62OD8@>2O<8@G2O58@O2NM8A82NE8A@2N>8AH2N78B02MO8B82MH8BA2M@8BI2M98C22M18C:2LI8CB2LB8CJ2L;8D32L30000000000000000000000000000008ED2JG8EM2J?8F52J88F=2J08FE2IH8FN2IA8G62I98G>2I28GG2HK8GO2HC8H72H<8H?2H48HH2GM8I42G;8I@2FI8IL2F78J92EE8JE2E28K12D@8K=2CM8KJ2C;8L22BM8L@2BC8LL2BB8M72B@8MC2B?8MO2B=8NG2B:8O?2B49082B09112AK91J2AF92C2AA93;2A=9442A894M2A395E2@O96C2@K97I2@E98O2@A9:62@<9;22@?9;<2@E9;F2@K9<02A09<:2A79<D2A?9<M2AG9=62B09=@2B89=I2B@9>32BH9><2C09>F2C99>O2C29?82BG9?A2B=9?J2B39@42AI9@=2A>9A82A19BJ2A49D@2A79F52A;9FI2A09G=2@E9H02@:9HD2?O9I72?D9IK2?99J>2>N9K42>F9KL2>C9LE2>@9M?2>>9N82><9O12>:9OJ2>7:0C2>5:1=2>3:262>0:2N2=M:3H2=K:4A2=I:5:2=G:632=D:6L2=B:7E2=?:8?2==:982=:::02=8::J2=6:;C2=4:<<2=1:=52<N:=N2<L:>H2<J:?A2<G:@:2<E:A32<C:AL2<A:BE2<>:C>2<;:D42<9:D?2;M:DJ2;@:E52;4:E@2:G:FI2:H:H02:L:HF2:O:I;2;2:J12;6:JG2;::K?2;?:L;2;E:M62<0:N22<;:M=2;;:LC2:4:L3297:L:290:LA28H:LH28A:LO288:M6281:M=27I:MD27A:MK27::N3272:N:26K:NA26C:NH26<
1777072BI70E2C57132CA71A2CM71O2D972>2DE72L2E27392E>73H2EK7482F174B2F474L2F77562F975?2F;75K2FF7662G076A2G:76M2GD7742GF77G2H37862H978D2H=7932HB79A2HG7:02HL7:>2I17:L2I57;;2I;7;J2I?7<82IE7<E2IH7=12II7=>2IK7=J2IL7>62IN7>B2IO7>N2J17?:2J27?B2IM7?O2IK7@?2IK7@N2IK7A=2IK7AL2IL7B;2IL7BI2IM7C82IM7CF2IM7D52IM7DD2IN7E22IN7E02J17DO2J37E42J47E82J57E=2J67EA2J87EF2J87F92J87FB2J57FI2J27G12J07G82IN7G?2IK7GA2IF7GC2IB7GE2I>7GG2I97GI2I57GK2I17GM2HL7GO2HJ7H32HH7H62HE7H:2H?7HA2H@7HH2HA7HO2HC7I52HD7I<2HH7IB2HK7IH2HO7IO2I17J32HL7J42HL7J32HH7J22HE7J12HB7J12H@7J12H?7J22H>7J22H<7J22H;7J22H97J:2H;7JC2H=7JN2H?7K92H?7KD2H@7KO2HB7L:2HC7L92HF7L82HJ7L62HM7L52I17LB2IE7M52J>7MJ2K47N?2KJ7O52LC7OK2M<80A2N58172NO8212NO82;2OD82N2O;83B2O18462NH84J2N>85>2N48622MJ86F2MA87:2M787O2M188E2M989<2MA8:22ML8:G2NF8;92NE8;C2NL8;N2O18<:2O58<E2O88=12O<8=<2O@8=G2OD8>22OG8>=2OK8>I2ON8?43028??3068?K30:8@630=8@>3078@G3018@O2OI8A82OA8A@2O:8AH2O28B02NK8B92NC8BA2N<8BI2N58C22MM8C:2MF8CB2M>8CJ2M68D32LO8D;2LH8DC2L@8DK2L98E42L18E<2KJ8ED2KB8EM2K:8F52K38F=2JL8FE2JD8FN2J=8G62J58G?2IN8GG2IF8GO2I?8H72I88HC2HH8HO2H68I;2GC8IH2G18J42F?8J@2EM8JL2E:8K62DJ8K=2D=8KC2D18KJ2CD8L82C78LJ2BO8M92BL8ME2BJ8N12BI8N<2BG8O32BC8OL2B>90E2B:91=2B59262B092O2AK93H2AG94A2AB95:2A=9622A896F2A897E2A098L2@K9:42@H9;12@L9;?2A49;I2A99<32A>9<=2AD9<G2AJ9=12AO9=<2B49=F2B:9>02B@9>:2BE9>D2BM9>M2C69?62C69?@2BK9?I2BA9@32B79@=2AM9@G2AC9A>2A=9C32AA9DG2AF9F<2AJ9G92AN9GM2AF9HB2A>9I62A69IJ2@N9J>2@E9K22@99KE2?O9L92?C9LM2?99M@2>M9N42>C9NG2>79O;2=M:032=J:0L2=G:1F2=E:2?2=C:382=@:412==:4J2=;:5C2=9:6<2=7:752=4:7O2=2:8H2<O:9A2<M:::2<K:;32<H:;L2<F:<E2<D:=>2<A:>72<>:?12<<:?J2<::@C2<8:A<2<5:B52<3:C32<5:CM2<::DA2<>:DN2<5:E92;H:EH2;>:G22;B:GH2;E:H=2;I:I52;N:J12<4:JL2<;:KG2<A:LC2<H:M>2=4:MO2=6:N>2=0:N=2<K:MC2;F:LI2:@:LA29J:LH29C:LO29;:M6293:M=28L:MD28D:MK28=:N3285:N:27N:NA27E:NH27>:NO276
17870D2C@7122CL71@2D871N2DD72=2E072K2E<7392EH73G2F47492F=74B2F?74L2FA7562FD75@2FG75I2FJ7652G476@2G>76L2GI7762H277>2H677I2H978<2HF78M2HO79<2I379J2I97:82I=7:G2IB7;52IG7;D2IK7<22J17<A2J57<M2J77=92J87=E2J:7>12J;7>>2J=7>J2J>7?12J:7?92J57?G2J37@82J47@G2J47A62J47AE2J57B42J57BC2J57C12J57C@2J67CO2J67D>2J77DM2J77E02J87E52J:7E92J;7E>2J<7EB2J=7EG2J>7EK2J?7F02J@7F82J>7F?2J;7FF2J87FN2J67G52J37G82IO7G:2IK7G;2IF7G>2IC7GA2IA7GD2I?7GH2I<7GK2I:7GN2I87H12I57H52HM7H:2HH7HA2HI7HH2HJ7HN2HK7I52HL7I<2HN7IC2HO7IJ2I07J02I17J02HO7J02HN7J02HL7J02HK7J02HI7J02HI7J02HG7J12HF7J12HD7J22HC7J:2HE7JC2HG7JL2HI7K52HJ7K@2HL7KK2HL7L62HM7L:2I07L92I47L72I87L<2IB7LO2J<7MA2K77N52KO7NJ2LE7O@2M;8052N180J2NG81@2O@8242OO82F2OI83:2O?83O2O684B2NL8562NB85J2N886>2MO8722ME87F2M;88:2M188N2LJ89?2N88:42NH8:J2O18;<2NO8;J2NN8<52O58<@2O;8<K2OA8=62OE8=A2OI8=L2OM8>73018>C3048>N3088?:30;8?E30?8@030C8@;30G8@F30J8A030D8A830=8A@3068AH2ON8B02OG8B92O?8BA2O88BI2O08C22NH8C:2NB8CB2N:8CK2N28D32MK8D;2MC8DC2M<8DK2M48E42LM8E<2LF8EE2L>8EM2L78F52KO8F=2KG8FE2K@8FN2K98G62K18G?2JJ8GG2JB8H12J58H>2IB8HJ2I08I62H>8IC2GL8IO2G98J:2FG8JA2F;8JG2EN8JN2EA8K42E58K:2DH8KB2D<8L12CO8L?2CC8M22C;8MF2C68N22C48N>2C38NJ2C18O?2BM9092BH9112BC91J2B?92C2B:93<2B59442B194M2AK95F2AG9682AL96G2AD97B2A;98I2A79:12A29:N2A89;A2AB9;K2AH9<62AM9<@2B39<J2B89=42B>9=?2BC9=I2BI9>32BN9>=2C49>G2C99?12C?9?A2C<9?L2C09@62BF9@@2B;9@J2B19A42AG9AM2AB9CA2AG9E62AK9F32B@9FF2C09H52B:9HN2B19IB2AH9J62A@9JK2A89K?2A09L42@H9LH2@?9M=2@89N12?O9NE2?G9O:2??9ON2?7:0B2>M:152>C:1I2>7:2<2=M:302=A:3D2=7:4:2=1:532<N:5M2<K:6F2<I:7>2<G:882<D:912<B:9J2<@::C2<=:;<2<;:<52<8:<O2<6:=H2<4:>@2<1:?:2;O:@32;L:@L2;J:AK2;M:BL2<2:CE2<7:D82<;:DL2<?:E=2<>:F42<8:FJ2<<:GF2<B:HA2<H:I=2<O:J92=5:K42=;:L02=B:LK2=M:M?2>2:ML2=O:N82=L:NG2=C:NF2=>:ND2=9:MI2<2:LO2:L:LO2:>:M62:6:M=29N:MD29F:MK29?:N3297:N:290:NA28H:NH28A:NO289:O6281000000000000000000000000000000000000000000000000000000000000000000000000000000;2924F
1797112D671?2DB71M2DO72;2E;72J2EG7382F373F2F?7492FG74B2FJ74M2FM7562FO75@2G175I2G47632G876?2GB76K2GM7762H777?2H?77H2HF77N2HC78>2HL7912I979D2IE7:22IJ7:A2IN7:O2J37;>2J87;L2J=7<;2JB7<H2JE7=42JG7=A2JH7=M2JJ7>92JK7>A2JG7>H2JB7?02J=7??2J<7@12J<7@A2J<7@O2J=7A>2J=7AM2J>7B<2J>7BK2J>7C:2J>7CH2J>7D82J?7DF2JA7DM2JC7E12JD7E52JE7E92JG7E=2JH7E@2JJ7ED2JK7EL2JN7F32JL7F:2JH7F@2JD7FF2JA7FL2J>7G02J;7G32J87G62J67G:2J47G=2J27G@2IN7GC2IL7GF2IJ7GI2IH7GM2IE7H02I<7H42I47H:2I07HA2I27HH2I27HN2I47I52I57I<2I67II2I<7J02I=7J02I;7IO2I97IO2I67IO2I47IO2I37IO2I27IO2I07IO2HO7J02HM7J22HL7J:2HO7JC2I07JL2I27K42I47K=2I67KH2I87L32I97L;2I:7L:2I>7L82IB7LH2J97M;2K47MN2KO7N@2LI7O52M?7OJ2N580?2NK8142OA81J30782:30A82M30983@3018442OG84G2O?85:2O785N2NN86A2NE8742N=87G2N588;2ML88L2NE89B2NK8:82O48:N2O;8;?2O98;O2O48<<2O78<G2O=8=12OD8=<2OJ8=G3018>23068>=30:8>H30=8?430A8??30D8?J30H8@530L8@@3108@L3138A73178A@3118AH30J8B130C8B930;8BA3048BI2OL8C22OD8C:2O=8CB2O58CK2NN8D32NG8D;2N?8DC2N88DK2N08E42MI8E<2MA8EE2M98EM2M38F52LK8F=2LC8FF2L<8FN2L48G62KM8G@2KA8GM2JO8H92J=8HE2IK8I12I98I=2HF8IE2H88IL2GL8J22G?8J82G28J?2FE8JE2F98JK2EL8K22E?8K;2E38KJ2DF8L82D98LG2CM8M;2CG8MO2CB8N@2C=8NK2C<8O72C;8OL2C690E2C291>2BM9262BI92O2BC93H2B?94A2B:9582B895J2B>96;2B@96H2B197@2AF98G2AA99N2A>9:K2AC9;B2B19;N2B79<82B<9<B2BB9<L2BG9=72BM9=A2C29=K2C89>62C=9>@2CC9>J2CH9?42CN9?K2CK9@82C49@B2BJ9@M2B@9A72B69AA2AK9B<2AG9D02AL9EE2B19EH2C79F22D29GB2C<9I22BG9IN2B:9JB2B39K72AJ9KK2AB9L@2A:9M42A19MI2@J9N=2@B9O12@99OF2@1:0:2?I:0N2?A:1C2?9:272?0:2K2>I:3@2>@:442>8:4I2>0:5=2=H:622=@:6E2=6:792<K:7M2<@:8A2<7:9:2<5::32<2::L2<0:;E2;M:<?2;K:=82;H:>12;F:>J2;D:?C2;B:@B2;E:AD2;J:BE2<0:C<2<4:D02<8:DD2<>:E=2<I:F72=7:G22==:GN2=D:HI2=J:IE2>0:JA2>6:K<2>=:L82>E:M02>M:M<2>K:MH2>H:N52>E:NA2>B:O02>6:NO2>1:NN2=L:NJ2=D:N02<>:M62;8:M>2;1:ME2:I:ML2:B:N32:::N:2:3:NA29K:NH29C:NO29;:O6293:O=28L:OD28D:OK28=;02285;0:27N;0A27F;0H27>;0O277;1626O;1=26H;1D26@;1K269;22260;2925I;2A25A
18071>2DL71L2E872:2EE72I2F17372F=73F2FI7492G274C2G574M2G77562G:75@2G<75J2G?7632GA76=2GF76I2H07752H;77@2HE77I2HL7812I47882I478?2I27932I>79F2IK7:92J87:I2J@7;82JD7;G2JJ7<52JN7<C2K37<O2K47=;2K67=H2K77>02K47>82JN7>?2JJ7>G2JE7?82JD7?J2JD7@:2JD7@I2JE7A72JE7AF2JF7B52JF7BD2JG7C32JG7CB2JH7D02JI7D?2JM7DF2JN7DJ2K07DN2K17E12K37E52K47E92K57E@2K77EH2K:7F02K=7F62K97F=2K57FC2K17FI2JN7FL2JK7FO2JH7G22JF7G52JD7G82JA7G;2J?7G>2J=7GA2J:7GE2J87GH2J47GL2IK7H02ID7H42I<7H92I97HA2I:7HG2I<7HN2I<7I=2IB7IN2IJ7J12II7J12IG7J02IE7J02IB7IO2I?7IO2I=7IN2I;7IN2I97IN2I87IN2I67J22I67J:2I87JC2I:7JL2I<7K42I>7K=2I@7KE2IB7KO2IC7L:2IE7L;2IH7LB2J67M52K17MH2KL7N:2LF7NL2MA7O@2N:8052O080J2OF81@30<82531282B31182M31583@30L84430D84G30;85:30285N2OJ86A2OA8742O987G2O088:2O688N2OB89E2O88:<2O>8;22OG8;B2OB8<22O>8<C2O88<M2O?8=82OE8=C2OL8=M3028>83098>C30?8>N30F8?930J8?D30M8?O3118@;3158@F3198A131=8A=31@8AH31C8B131>8B93168BA30O8BI30H8C230@8C:3098CB3018CK2OJ8D32OB8D;2O;8DC2O48DL2NL8E42NE8E=2N=8EE2N58EM2MN8F52MF8F=2M?8FF2M88FO2LN8G;2L<8GH2KJ8H42K78H@2JD8HI2J58HO2II8I62I<8I<2HO8IC2HC8II2H68J02GJ8J62G=8J=2G08JC2FD8JI2F78K42EJ8KB2E>8L12E18L?2DE8LO2D98MC2D48N82CN8NL2CH8O82CF8OD2CD9092C@9112C<91J2C792C2C393<2BM9442BI94J2BJ95<2C095M2C696<2BM96I2B>97=2B198D2AM99K2AH9:H2AN9;?2B;9<12BF9<;2BK9<E2C09<O2C69=92C<9=D2CA9=N2CF9>82CL9>B2D29>L2D79?72D<9@22D79@F2CC9@O2BN9A92BD9AC2B:9AN2B09BK2AL9D@2B19EA2BC9E=2CO9E?2E39FO2D>9H?2CI9IN2C39JO2BE9KC2B=9L82B49LL2AL9M@2AD9N52A<9NI2A49O>2@K:022@D:0F2@;:1;2@3:1O2?K:2C2?C:382?;:3L2?3:4@2>J:552>C:5I2>::6>2>2:722=J:7F2=B:8;2=::8O2=1:9C2<I::82<A::L2<9:;@2<1:<52;H:<I2;?:=A2;;:>:2;8:?:2;<:@;2;B:A<2;K:B=2<5:C;2<=:D32<H:DK2=4:ED2=?:F=2=M:G:2>>:H62>E:I22>K:IM2?2:JI2?8:KD2?>:L@2?I:LL2?F:M82?D:MD2?A:N12?>:N=2?;:NO2?3:O92>I:O82>D:O72>@:O52>::O02>0:N62<J:ME2;K:ML2;D:N32;<:N:2;5:NA2:M:NH2:F:NO2:>:O62:6:O=29O:OE29G:OL29@;03298;0:290;0A28H;0H28A;0O289;16281;1=27J;1D27B;1K27;;22273;2926K;2A26D;2H26<
1827292EO72H2F;7362FG73E2G47492G=74C2G@74M2GB7572GE75@2GG75J2GJ7642GL76=2GO76G2H47732H?77?2HI77J2I37822I:78:2IA78B2IE78G2I@7942ID79H2J17:;2J>7:N2JK7;A2K67;O2K;7<=2K@7<J2KC7=62KD7=?2KA7=G2K;7=O2K77>62K17>>2JM7?02JM7?B2JM7@32JN7@B2JN7A12JN7A?2JN7AO2JO7B=2JO7BL2JO7C;2K17CI2K57D92K97D?2K:7DC2K<7DF2K=7DJ2K>7DN2K@7E42KB7E<2KD7ED2KG7EL2KJ7F32KI7F92KE7F@2KA7FF2K=7FH2K;7FJ2K87FM2K67G02K47G32K17G72JO7G:2JM7G=2JK7G@2JH7GC2JC7GG2J;7GK2J37GO2IK7H32IB7H92IB7HA2IC7I12IJ7IB2J17J22J87J22J57J12J37J12J17J02IN7J02IK7IO2II7IO2IG7IN2ID7IN2IB7IM2I?7J22I@7J:2IB7JC2ID7JL2IE7K42IH7K=2II7KE2IK7KN2IN7L72IO7L<2J27LO2JN7MB2KH7N42LC7NF2M>7O92N87OL2O480@2OK81530A81J31782=31D82J31B83631>83@31H84431@84G31685:30N85N30F86A30=87430487G2OL88:30@89130789H2OM8:?2OI8;53018;E2OL8<52OG8<F2OB8=42OA8=?2OG8=I2ON8>43048>?30:8>J30A8?530H8??30N8?J3158@531:8@@31>8@L31B8A731F8AB31I8AM31L8B83208BA31K8BI31C8C231<8C:3158CC30M8CK30F8D330>8D;3078DC2OO8DL2OG8E42O@8E=2O98EE2O18EM2NJ8F52NB8F>2N;8FJ2MI8G62M68GC2LD8GM2L38H42KG8H:2K:8H@2JM8HG2J@8HM2J48I42IG8I:2I:8I@2HM8IG2HA8IN2H48J42GG8J:2G;8JA2FN8JM2FA8K;2F58KJ2EH8L82E<8LF2DO8M72DF8ML2D@8N@2D:8O42D48OF2CO9012CN90E2CJ91>2CF9262CA9302C<93H2C794;2C<94M2CB95?2CH9602CJ96=2C:96J2BJ97:2B<98A2B899H2B49:F2B:9;<2BG9<32C49<=2C99<H2C@9=22CE9=<2CJ9=F2CO9>12D69>;2D;9>E2D@9>O2DE9?92DK9@92DC9A32D49A<2C89AF2BI9B02B>9B:2B49C:2B19DN2B69E62C:9E22DF9DN2F29F<2E?9GK2DJ9I;2D59JK2C?9KO2BO9LD2BG9M82B?9MM2B79NA2AN9O62AF9OJ2A>:0>2A6:122@N:1G2@E:2;2@=:2O2@5:3D2?M:482?E:4M2?<:5A2?5:662>L:6J2>D:7>2><:832>4:8G2=L:9;2=D:9O2=;::D2=4:;82<K:;M2<C:<A2<;:=62<2:=I2;H:>;2;H:>O2<=:@02<G:A12=1:B22=;:C32=E:D;2=>:E32=J:EL2>6:FD2>A:GB2?9:H>2?F:I:2?L:J62@2:K12@9:KM2@B:L<2@B:LI2@?:M52@=:MA2@::MN2@5:N?2@1:O02?J:O@2?B:OB2?<:OA2?8:O@2?3:O?2>M:O=2>I:O72><:N<2=5:N32<?:N:2<8:NA2<0:NI2;H:O02;A:O72;8:O>2;1:OE2:I:OL2:B;032::;0:2:3;0A29K;0H29C;0O29<;16294;1=28M;1D28E;1L28=;23286;2:27N;2A27F;2H27>;3;26O000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000<4M1<7<5@1;H<671;A<701;?
1847352G273E2G>74:2GH74C2GJ74M2GM7572H075A2H275J2H47642H776=2H976G2H<7722HC77=2HM77I2I77832IA78<2IH78D2IN78L2J57922J17982IK79I2J77:<2JD7:O2K07;C2K=7<62KJ7<F2L07<N2KM7=62KH7=>2KD7=F2K>7=M2K:7>72K67>H2K67?:2K67?L2K67@;2K67@J2K77A92K77AH2K77B72K77BE2K:7C42K=7CC2KA7D12KD7D72KG7D;2KG7D?2KI7DC2KJ7DH2KL7E02KO7E82L27E@2L47EH2L77EO2L97F62L57F<2L27FC2KN7FE2KK7FF2KI7FH2KG7FL2KD7FO2KA7G22K?7G52K=7G82K:7G;2K87G?2K37GC2JK7GG2JB7GK2J:7GO2J27H32IK7HE2J17I62J87IG2J?7J32JD7J32JA7J22J?7J22J<7J12J:7J12J87J02J57J02J37IO2J07J02J37J12J67J62J87J>2J:7JG2J;7K02J=7K82J>7K@2J@7KI2JA7L12JB7L:2JD7LF2JI7M;2KD7MN2L@7NA2M;7O32N57OF2O18082OK80K30F81@31<82532282E32683232383?32083J31M84432;84G32385:31J85N31A86A31987431087F31@88=31689430K89K30A8:C3078;830:8;H3068<83018<I2OL8=92OG8=E2OI8>03008>;3068>F30=8?030C8?;30J8?F3108@13168@;31=8@F31C8A131J8A<31O8AG3238B33268B>3298BI32=8C23288C:3208CC31I8CK31A8D331:8D;3138DD30K8DL30D8E430<8E=3048EE2OM8EM2OF8F92O58FE2NC8G12N08G82MD8G>2M78GE2LJ8GK2L=8H22L18H82KD8H>2K78HE2JK8HK2J>8I12J28I82IE8I>2I98IE2HL8IH2HC8IK2H:8J12GG8JD2G=8K32FN8KC2F@8L12F38L?2EG8LN2E:8M@2E28N42DL8NI2DF8O=2D@9012D:90?2D89122D491J2CO92C2CK93;2CG93M2CM94?2D39512D995C2D?9612D696>2CG96K2C79782BG98>2BC99F2B>9:C2BE9;92C29<02C@9<@2CI9<J2CN9=42D39=?2D99=I2D?9>32DD9>=2DI9>G2DO9?22E59?=2E99@?2DO9A@2DE9AI2CI9B32BM9B=2BC9BG2B99CI2B79DO2BE9DK2D19DG2E=9DC2FI9EH2FA9G82EK9HH2E69J72DA9KG2CK9M02C99MD2C19N92BI9NM2B@9OB2B8:062B1:0J2AH:1?2A@:232A8:2G2A0:3;2@H:402@?:4D2@7:592?O:5M2?G:6B2??:762?6:7J2>O:8?2>F:932>>:9G2>6::<2=N:;02=F:;E2==:<92=2:<N2<H:=>2<C:=L2<D:>C2<A:?62=5:?H2=J:@E2>7:AF2>A:BG2>L:CK2?0:E82>C:F32>L:FK2?8:GH2?L:HF2@F:IB2@N:J>2A4:K92A;:KM2A>:L92A;:LE2A9:M22A4:M@2A0:MN2@K:ND2@H:O32@@:OG2@8:OM2?O:OI2?J:OJ2?K:OI2?E:OH2?A:OF2?<:OE2?6:O=2>H:NB2=A:NB2=2:NI2<K:O02<C:O72<;:O>2<4:OE2;L:OL2;E;032;=;0:2;5;0A2:N;0H2:F;102:>;172:6;1>29O;1E29G;1L29@;23298;2:290;2A28I;2N28=;3A27N;4427?000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000<3L1>6<4?1=F<521=6<5E1<G<6<1<@<751<><7N1<;<8G1<9<9?1<7<:81<4<;11<1<;I1;O<<B1;L<=;1;I<>51;D<>O1;A<?H1;=<@A1;9<A;1;4<B31:M
18573D2GH74:2H374C2H674N2H87572H:75A2H=75J2H?7642HB76>2HD76G2HG7712HI77<2I177G2I;7832IE78=2IN78E2J578M2J<7962JC79<2JB79B2J=79K2J<7:>2JI7;12K67;E2KC7<82L07<E2L37<M2L07=42KK7=<2KG7=D2KA7=O2K>7>A2K>7?22K>7?D2K>7@52K?7@C2K?7A22K?7AA2K@7B02KC7B?2KF7BM2KJ7C<2KM7CJ2L07D02L27D42L37D82L57D<2L77DD2L97DL2L<7E42L?7E<2LA7ED2LD7EL2LG7F22LF7F92LB7F?2L>7FB2L;7FC2L97FD2L77FG2L47FJ2L27FM2KO7G02KM7G42KJ7G72KH7G:2KB7G>2K:7GB2K17GF2JJ7GJ2JA7H02JA7HJ2J?7I;2JF7IK2JN7J42K07J32JN7J32JK7J22JH7J22JF7J12JD7J22JE7J32JI7J42JL7J52K07J62K37J<2K57JD2K77JM2K87K52K:7K>2K;7KF2K=7KN2K>7L72K@7L@2KA7LO2KI7M@2L37N42LO7NI2MM7O?2NK8022OH80E30C81731>81K32682?32I82M32G83:32E83H32B84332=84:32H84G32N85:32E85N32=86A32587432387I32488A31J89731@89N3168:F30L8;;30D8;K30@8<<30:8<L3058=<3018=L2OK8>73018>A3088>L30>8?730E8?B30K8?L3128@73198@B31?8@M31F8A831L8AB3238AM3298B832?8BC32B8BO32F8C:32J8CC32E8CK32=8D33268D;31N8DD31F8DL31@8E43188E=3108EH30B8F42OO8F<2OA8FB2O58FI2NH8FO2N;8G52MO8G<2MB8GC2M58GI2LI8GO2L<8H32L38H52KJ8H82KB8H:2K:8H=2K18H?2JI8HB2JA8HC2J38H?2HJ8I?2HC8J52H;8JE2H28K52GC8KD2G58L42FG8LD2F78M32EI8MH2E?8N=2E88O12E28OE2DL90:2DF90L2DA91>2D>9272D992M2D993?2D?9402DE94B2DK9542E195E2E39622DC96?2D396L2CD9792C498;2BN99C2BJ9:@2C09;62C=9;M2CJ9<C2D79<M2D=9=72DB9=B2DH9=L2DM9>62E39>@2E89>J2E>9?42EC9?D2EE9@F2E<9AG2E29B72D;9B@2C?9BJ2BG9C42B=9D82B<9DD2C<9D@2DH9D<2F49D82G@9E52GC9FE2FM9H42F79ID2EB9K42DM9LC2D79N12CC9NE2C;9O92C39ON2BK:0B2BC:162B::1K2B2:2?2AJ:332AB:3H2A::4<2A1:502@J:5E2@B:692@9:6N2@1:7B2?I:872?A:8K2?9:9?2?0::42>G::I2>=:;=2>3:<22=H:<G2==:=22==:==2=?:>42==:>K2=::?>2=M:@02>B:@C2?7:A;2?H:B;2@2:C=2@<:DC2@;:F12?N:G32?N:GM2@=:HL2A8:IJ2AO:JF2B5:K<2B::KI2B7:L72B3:LD2AN:M22AJ:M@2AF:N32AA:NG2A=:O72A5;042@N;0?2@E;082@;;002@2:OM2?O;022@8;012@4:OO2?O:ON2?I:OM2?E:OC2?3:NI2=M:O02=F:O72=>:O>2=7:OE2<N:OL2<G;032<?;0;2<8;0B2<0;0I2;H;102;A;172;9;1>2;2;1E2:J;1L2:B;232:;;2:2:3;2A29K;3329=;3F28M;4928>000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000<2K1@3<3>1?D<411?5<4D1>E<571>6<5K1=F<6B1=@<7:1==<831=;<8L1=8<9E1=6<:>1=3<;61=0<;O1<N<<H1<K<=A1<H<>:1<D<?41<@<?M1<<<@G1<8<A@1<3<B71;H
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
19175K2I@7652IB76>2IE76H2IH7722IJ77<2IL77E2IN77O2J278:2J;78F2JF7912K07992K779B2K=79J2KD7:22KK7::2L37:A2L07:F2KK7:K2KF7;62KG7;J2L47<92L<7<A2L77<I2L37=:2L47=L2L57>=2L67>N2L77?@2L:7@02L=7@>2L@7@L2LC7A;2LF7AI2LJ7B82LM7BG2M17C62M47C:2M67C?2M87CG2M:7CO2M=7D82M@7D@2MC7DH2MF7DO2MI7E72ML7E?2MN7EG2N17EO2N27F52MO7F82ML7F92MI7F:2MG7F;2ME7F<2MB7F?2M?7FB2M=7FF2M;7FI2M87FM2M07G12LH7G42LA7G22LD7G12LG7GF2LF7H@2LC7I:2LA7IM2L=7J82L<7J:2LB7J<2LH7J>2LN7J@2M47JA2M97JB2M<7JC2M?7JD2MB7JF2MF7JG2MI7JL2ML7K52MM7K>2MO7KG2N17L02N37L92N57LB2N67LL2N87M92N?7MI2NH7N:2O17NJ2O;7O;2OE7OL30180A30O81631M81K32L82@33K83134083A34384134584?34684L34385833N85D33I85L34185N35186G34;87A33B88<32J89032:89>32289M31I8:=31C8:O31?8;B31;8<53168<H3118=;30M8=N30H8>A30C8?530>8?@30E8?J30M8@53148@?31<8@J31C8A531K8A?3238AJ32:8B432B8B>32I8BI3318C43388C>33@8CE33D8CI33E8D233D8DI33B8DG33A8DA33B8D;33B8D433B8D133?8D33378D632O8D832F8D:32?8D=3268D?31N8DB31F8DD31=8DG3158DI30M8DK30D8DF2O:8D@2MO8D=2LH8DF2KM8EI2KG8FJ2KA8GJ2K:8HF2K38I62JI8IH2J@8J82J68JI2IL8K92IC8KI2I68L92HG8LI2H98M>2H08N42GF8NI2G=8O?2G49062FL90O2FG91I2FD92@2F?92E2F;9372FA93I2FG94;2FM94M2G395;2FI95H2F99652EI96B2E:96O2DJ97<2D:9832CO99;2CK9:72D29:M2D?9;D2DL9<:2E99=12EG9=?2EN9=I2F49>32F:9>>2F?9>H2FD9?22FJ9?<2G09@82FJ9A:2FA9B<2F79C=2EN9CH2E29D22D69D:2C<9D52D29CC2EB9C>2FN9C;2H:9C72IF9C;2JH9DK2J29F;2I<9GJ2HG9I:2H29JJ2G<9L92FG9MI2F19O92E<:0H2DF:1G2D9:2;2D2:2O2CI:3D2CA:482C9:4L2C1:5A2BG:662B=:6K2B3:7?2AH:842A>:8I2A4:9>2@I::22@>::G2@5:;<2?J:;F2?L:;O2?M:<72?O:<G2?O:==2?M:>42?K:>K2?H:?B2?E:@62@3:@I2@H:A;2A=:AN2B2:B@2BG:C32C<:CF2D1:D<2DC:E@2DG:FN2D;:GO2E3:I@2DB:JF2D0:K52CO:KD2CO:L32CN:LB2CM:M62CL:ML2CH:NB2CD:ND2BJ:N92AK:O>2B=;0@2BG;102BC;142B6;182AJ;162A>;0O2A4;0H2@K;0A2@B;0:2@8;0:2@<;0@2@I;0E2A5;0H2A9;0F2A4;0E2@O;052@6;032?G;0;2??;0B2?8;0I2?0;102>I;172>A;1>2>:;1E2>1;1L2=J;232=B;2=2=9;302<J;3C2<;;462;K;4J2;<;5=2:M;602:>;6C29N;7629?;7I290;8<28A;8O281000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;O51F<;OH1EM<0;1E=<0N1DN<1B1D><251CO<2H1C@<3;1C1<3N1BB<4A1B2<541AC<5H1A4<6;1@D<721@><7J1@;<8D1@9<9<1@6<:51@4<:N1@1<;F1?N<<?1?L<=81?I<>11?F<>K1?B<?D1?><@=1?:<A51>O<AL1>E<BC1>:<C;1=O<D21=E<DI1=;
19376?2J076H2J27722J577<2J877F2J:77O2J<7892J?78D2JJ7902K479:2K=79C2KD79K2KK7:32L27:<2L97:D2L@7:K2LA7;02L<7;62L77;<2L27;K2L:7<82L=7<B2L:7=32L;7=E2L<7>82L@7>K2LF7?=2LJ7?K2LL7@92LO7@G2M17A42M37AB2M67B22M97B@2M=7BO2M@7C32MB7C;2ME7CC2MH7CK2MJ7D42MM7D<2N07DC2N27DK2N57E32N87E;2N;7EC2N>7EK2NA7F22N?7F42N<7F52N97F72N77F82N57F92N27F;2N07F>2MM7FA2MK7FD2MG7FH2M?7FL2M77FK2M97FJ2M<7FI2M?7GC2M<7H=2M:7I02M57IA2M07J32LK7J92LM7J;2M37J=2M97J?2M?7JA2MF7JC2ML7JF2N27JH2N87JJ2N>7JL2ND7K22NI7K<2NK7KE2NM7KN2NN7L72O17L@2O27LI2O47M32O67MB2O>7N22OG7NB3017O330;7OD30E80430O80E31;81;32982033882D34483434783D34;84434>84D34@85434C85@34>85K34786733N86;34>87333I87M33288;32I88K32C89>32?8:032;8:C3278;63238;H31O8<;31J8<O31F8=B31@8>531<8>G3188?53178?B31;8?L31C8@731J8@A3228@L3298A632A8AA32I8AK3308B63388B@33?8BE33A8BI33B8BM33B8C133C8C533C8CK33A8DB33>8E:33;8E733;8E033;8DJ33;8DC33;8D=33<8D733<8D033<8CJ33<8CD33<8C>33<8CA3348CC32L8CA31N8CH3118D03058D82O98D@2N=8DH2MA8DO2LE8F02L;8G72L78H62L08HG2KG8I82K=8II2K38J92JJ8JJ2J@8K:2J78KK2IK8L:2I=8LM2I18MB2HH8N82H?8NN2H68OG2H190A2GM91;2GI9242GD92N2G@9322G:93;2G893M2G>94>2GD9502GE95<2G595I2FF9662F696C2EF9702E797=2DG9802D:9982D69:42D=9:J2DK9;A2E89<72EE9<N2F29=B2F>9=L2FC9>62FH9>@2FN9>J2G49?42G99??2G>9@?2G69A@2FM9BB2FD9CD2F:9D62ED9D?2DH9DC2D?9D>2E29D52F19CC2G@9C12I19BL2J>9BH2KI9D82K49EG2J>9G72IH9HG2I39J62H>9KF2GH9M62G39NE2F>:052EH:1E2E2:2G2DD:3;2D<:402D2:4E2CH:592C=:5O2C3:6C2BI:782B>:7M2B4:8B2AJ:962A?:9K2A4::@2@J:;12@D:;:2@F:;C2@H:;L2@J:<72@K:<N2@H:=E2@F:><2@D:?32@A:?I2@>:@=2@K:A02A?:AC2B4:B52BJ:BH2C?:C;2D3:CM2DH:D@2E>:E22F3:F;2F3:GH2EI:IB2DI:JC2D<:K12D;:K@2D::KO2D9:LE2D8:M;2D6:MM2D5:N52CO:M32BI:M<2BE:N?2BO:OA2C9;0B2CA;0F2C4;0J2BH;0N2B;;122AN;162AA;122A7;0K2@M;0D2@D;0=2@:;0?2@A;0D2@N;0J2A:;0O2AG;0N2AB;0M2A=;0<2@B;0B2@:;0I2@2;102?K;172?C;1>2?<;1E2?4;1L2>L;232>E;2B2>9;352=I;3H2=:;4;2<K;4O2<;;5B2;L;652;=;6H2:N;7<2:>;7O29O;8B29@;95290;9H28A;:;282;:N27B;;B273000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;N41H9;NG1GJ;O:1G;;ON1FL<0A1F=<141EM<1G1E><2:1DN<2M1D?<3A1D0<441CA<4G1C2<5:1BB<5M1B3<6@1AC<771A=<801A:<8I1A8<9A1A5<:;1A3<;31A1<;L1@M<<E1@K<==1@H<>61@F<?01@A<?I1@><@A1@5<A91?K<B01?@<BG1?5<C?1>K<D61>@<DM1>6<EE1=L
19476I2J>7732J@77<2JB77F2JD7802JH7892JJ78C2JN78N2K879:2KB79D2KJ79L2L17:42L97:=2L@7:E2LF7:M2LM7;52M27;;2LM7;@2LH7;F2LC7;M2L@7<<2LA7<O2LF7=B2LL7>52M07>H2M67?82M97?G2M<7@42M>7@B2M@7A02MB7A>2ME7AL2MG7B:2MI7BH2ML7BO2MO7C72N27C?2N57CG2N77CO2N:7D72N=7D?2N?7DG2NB7DO2NE7E72NH7E?2NK7EG2NN7EO2NO7F12NL7F22NJ7F32NH7F42NE7F52NB7F62N@7F92N>7F<2N;7F@2N77FD2MO7FD2MN7FB2N17FA2N47FF2N57G@2N37H22MN7HD2MI7I52MD7IG2M?7J92M97J;2M?7J=2MF7J?2MK7JA2N17JC2N77JF2N=7JG2NC7JK2NK7K52O57K=2O@7KE2OD7KL2OJ7L42OL7L>2ON7LG3007M03017M:3047MJ30=7N;30G7NK3117O<31;7OL31E80=31N80M32881?32E82433E82H34<83834?83H34B84834D84G34D85734C85F34H86234B86=34986I33K87:33C87M33?88?33;89233789D3338:732O8:J32K8;<32G8;O32B8<B32?8=53298=H3258>83238>G3228?63218?C3228?N3298@832A8@C32I8@N3308A83388AA33>8AE33>8AI33>8AM33?8B133@8B533A8B933B8B=33B8BL33@8CD33>8D;33;8E23388EJ3358EF3358E@3358E93368E03368DI3378DA3378D93388D13388CI3388CA3398C@3388CK32F8D331J8D;30N8DB3028DJ2O68E22N:8E:2M>8F22LN8G92LI8H82LC8HI2L:8I92L08IJ2KG8J:2K=8JK2K48K;2JK8KL2JA8L<2J38M02IH8MF2I?8N@2I;8O92I69032I290M2HN91F2HI92@2HE9392HA93A2H;93D2H59402H694B2H<9512H295=2GB95J2G29672FC96D2F39712EC97>2E497N2DE9952DA9:12DH9:G2E59;>2EC9<42F09<K2F>9=A2FJ9=O2G29>92G79>C2G=9>M2GB9?72GH9?D2GL9@F2GB9AG2G99BI2G09CJ2FG9DC2F59DL2E99DK2EB9DF2F69DA2FJ9D52GO9CC2I?9C22JO9B@2L@9CH2KN9E:2K69FL2J>9H=2IE9IN2HL9K@2H49M12G<9NB2FC:042EK:1F2E2:2E2F=:3D2DM:4=2D>:522D3:5G2CI:6<2C?:712C4:7E2BJ:8:2B?:8O2B5:9D2AJ::92A@::E2A>::N2AA:;72AB:;@2AD:;I2AG:<?2AD:=62AA:=M2A?:>D2A=:?:2A::@12A8:@E2AA:A82B7:AK2BL:B=2CA:C02D6:CC2DK:D52E@:DH2F5:E;2FJ:F62G0:GJ2F1:ID2E1:J?2DG:JN2DF:K=2DE:L42DD:LG2DB:M52DB:MD2DB:MM2D>:LK2C6:LL2C8:M?2CA:NB2CK:OD2D5;072D2;0;2CF;0?2C9;0C2BM;0G2B@;0K2B3;0O2AG;132A:;0N2@O;0G2@F;0@2@<;0C2@E;0H2A1;0N2A>;142AK;162B0;152AK;0I2A5;102@N;172@F;1>2@>;1F2@7;1M2?O;242?G;2H2?8;3;2>I;3N2>9;4A2=J;542=:;5G2<K;6:2<<;6N2;M;7A2;>;842:N;8G2:?;9;2:0;9N29@;:A291;;428B;;G282;<:27C;<M274;=A26E;>4265;>G25F;?:256;?M24G000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;K:1KE;KM1K6;L@1JG;M31J7;MF1IH;N91I8;NL1HI;O@1H:<031GK<0F1G<<191FM<1L1F=<2?1EM<331E><3F1DO<491D@<4L1D1<5?1CB<631C2<6F1BC<7=1B<<861B9<8N1B7<9G1B5<:@1B2<;91B0<<11AM<<J1AJ<=C1AH<>;1AE<?51AA<?N1A;<@E1A1<A=1@F<B41@;<BK1@1<CC1?F<D:1?;<E11?1<EI1>G<F@1><
19677=2JM77F2K07802K27892K478C2K778M2K<7992KF79D2L079M2L87:52L?7:>2LF7:F2LM7:N2M47;72M;7;?2MB7;E2M>7;K2M97<02M47<92M27<L2M77=?2M<7>22MA7>E2MF7?42MI7?B2MK7@02MM7@>2MO7@K2N27A92N47AG2N67B72N97BC2N;7BJ2N;7C12N;7C92N>7CA2NB7CI2NE7D22NH7D:2NK7DB2NN7DJ2O17E32O47E;2O87EC2O;7EK2O>7EN2O<7EO2O:7F02O87F12O57F22O27F32O07F42NN7F82NK7F;2NF7F=2NB7F;2NE7F:2NH7F82NK7FC2NK7G42NG7GF2NB7H82N<7HJ2N87I;2N27IM2MM7J;2MK7J=2N17J?2N77JA2N=7JC2NC7JE2NI7JM2O37K62O>7K?2OG7KH3027L030:7L830?7L?30D7LF30I7LN30M7M730O7MC3137N331=7NC31G7O43217OE32:80532D80F32N81633881G33B82834182K34D83;34G83K34F84;34E84J34D85:34C85J34B86934M86D34D87334<87E34788734488H33O89:33K89K33G8:>33C8;033>8;C33;8<53368<H3328=;32N8=J32M8>832L8>G32K8?632J8?D32I8?O32O8@:3378@A33;8@E33;8@I33;8@M33<8A133=8A533>8A933>8A=33?8AA33?8AE33@8AN33@8BF33=8C=33;8D43388DL3358EI33;8EM3328EE3338E>3338E63348DM3348DE3358D=3358D63358CN3368CF3378CA3378D13348D=32C8DE31G8DM30K8E42OO8E=2O38EE2N78F42MA8G;2M<8H;2M68HK2LL8I;2LC8IL2L:8J<2L08JL2KG8K=2K=8KM2K48L?2JJ8M82JD8N22J@8NK2J<8OE2J890?2J39182IN9222IK92K2IF93E2IB9412I<9432I69462HO94E2HN9522H?95>2GO95K2G?9682FO96E2F@9722F097?2E@97L2E19922DK99N2E49:D2EA9;;2EN9<12F;9<H2FH9=>2G69>12G@9>;2GF9>E2GL9?02H19?:2H69?K2H89@L2GO9AN2GE9C02G<9D12G39E02FG9E72F19E32FE9DN2G99DI2GM9DE2HA9D72IF9CI2JK9C<2KO9D;2KC9EM2JA9GA2I@9I32H?9JD2GG9L62FO9MH2F79O92E>:0<2F9:192GL:262I?:352H8:442F=:532DK:5O2D9:6I2CJ:7>2C@:832C5:8H2BK:9<2B@::02B7:::2B9::C2B;::L2B=:;42B?:;=2BA:<02B@:<G2B=:=>2B;:>42B8:>K2B6:?B2B4:@92B1:@N2B9:A@2BN:B32CC:BE2D9:C82DM:CK2EB:D=2F7:E02FM:EF2G=:FK2G2:GL2F8:IF2E8:J;2E2:JL2E2:K@2E0:KO2E0:L>2DO:LM2DO:M;2DO:MF2DL:LD2CE:LL2CM:LN2D4:MB2D<:NE2DG:OG2E1:OL2DE;002D8;042CK;082C>;0<2C1;0@2BE;0D2B8;0H2AK;0L2A>;102A1;0J2@H;0C2@?;0G2@J;0M2A6;122AC;182B0;1=2B=;1=2B9;182AH;1?2AA;1F2A9;1M2A1;292@F;2M2@7;3@2?H;432?9;4G2>I;5:2>:;5M2=J;6@2=;;732<L;7F2<=;892;N;8L2;>;9@2:O;:32:?;:F2:0;;929A;;M292;<@28B;=3283;=F27D;>9275;>L26E;??266;@325F;@F257;A924H;AL249;B?23J;C223;;CE22K;D922;;DL21L;E?21=;F220N;FF20?;G9200;GL1O@;H?1O1;I21NA;IE1N2;J81MC;JL1M4;K?1LD;L21L5;LE1KF;M91K6;ML1JG;N?1J8;O21IH;OE1I9<081HJ<0K1H;<1?1GL<221G<<2E1FM<381F=<3K1EN<4>1E?<511E0<5E1DA<681D2<6K1CB<7B1C<<8;1C9<941C6<9L1C4<:E1C2<;>1BO<<61BL<=01BI<=H1BG<>A1BD<?;1B@<@21B6<@J1AL<AA1AA<B81A7<C01@L<CG1@A<D>1@7<E61?L<EM1?B<FD1?8<G;1>M
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
19978:2KJ78D2KM78M2KO7972L279B2L879M2LC7:82LL7:@2M37:H2M:7;12MA7;92MH7;B2MO7;J2N57<22N=7<:2N@7<?2N;7<F2N77=92N<7=L2NB7>=2NE7>K2NH7?82NJ7?G2NL7@52NN7@E2O17A62O37AG2O57B12O77B;2O87BC2O87BI2O87C02O87C62O87C=2O87CD2O87CK2O;7D42O>7D<2OA7DD2OD7DM2OG7E62OO7EA30:7EC3087ED3067EF3047EG3027EI3017EK2OO7EL2OM7EN2OK7EJ2OO7EG3037ED3067EA3097E?30<7F03067FA3017G22OK7GC2OF7H42O@7HE2O;7I62O57IG2NO7J82NK7J>2NN7JH2O87K12OC7K:2ON7KC3087KL30C7L530M7L>3187LG31B7M031I7M731O7M>3237ME3297MM32=7N;32@7NK32J7O<3347OM33>80=33H80M34281>34<81N34F82?35082J35583435783D35584335384C35185234N85A34M86134J86@34K87035;87A35788335388D34N89634J89G34F8:934A8:J34=8;;3498;M3448<A33O8=<33H8>633B8>E33A8?433?8?C33>8@233=8@>33;8@<3328@832K8?B3358?=3398?A33:8?E33;8?I33;8?M33;8@133<8@533=8@933>8@I33<8AA3398B93388C833@8D733H8E233N8EE3428EK33I8EJ33:8EG32N8E?32O8E732O8DO32O8DG3308D?3308D73318CO3328CG3328D332O8DD32L8E432I8EB32=8EJ31A8F130D8F92OH8FA2NL8G@2NB8H@2N;8I02N28I@2MI8J12M?8JA2M68K12LL8KD2LD8LB2L=8M?2L78N:2L18O32KO8OK2KM90D2KJ91<2KH9252KF92M2KD93F2KA94>2K?94O2K89522JH9542J89562IH9592I895@2HH95M2H996:2GI96G2G99742FJ97A2F:97N2EJ98L2EB99H2EK9:>2F79;42FD9;K2G29<B2G?9=82GM9=O2H99>@2HC9>K2HI9?52HO9??2I49@82I09A92HG9B;2H>9C=2H49D?2GK9E;2GC9EH2G>9ED2GG9E?2H09E;2H99E72HB9E52HK9EA2IH9E42JM9E=2K99G02J79HC2I59J62H39KI2G29M72FA9N32H69NO2IK9OK2KA:0G2M3:1I2M@:2G2LC:3D2K5:4@2IH:5<2H9:692FL:752E>:802DH:8K2D9:9B2CN:9K2D0::42D2::=2D4::F2D6:;22D7:;H2D5:<@2D2:=62CO:=M2CM:>D2CJ:?;2CH:@12CF:@H2CC:A>2CH:B02D=:BC2E2:C62EG:CH2F<:D;2G1:DM2GF:EK2GJ:G02G?:H52G5:I:2FJ:J42FD:J22EK:J@2EK:JO2EK:K=2EK:KM2EJ:L;2EJ:LJ2EJ:M72EH:L?2DK:LM2E7:LO2E>:M02EE:M22EL:MI2F4:NL2F?:O32F4:O72EH:O;2E;:O?2DN:OC2DB:OG2D5:OK2CH:OO2C;;032BN;072BB;0;2B5;0?2AH;0C2A;;0G2@N;0J2@F;102A2;162A@;1=2AM;1E2B<;1M2BJ;212C3;2E2BE;382B6;3K2AF;4>2A7;512@H;5D2@8;672?I;6K2?9;7>2>J;812>;;8D2=L;982==;9K2<N;:>2<>;;12;N;;D2;?;<72;0;<J2:A;==2:2;>129B;>D293;?728D;?J284;@=27E;A1276;AD26G;B7267;BJ25H;C=259;D024J;DD24:;E723K;EJ23;;F=22L;G022=;GC21N;H621?;HJ20O;I=20@;J0200;JC1OA;K71O2;KJ1NC;L=1N4;M01MD;MC1M5;N61LE;NI1L6;O=1KG<001K8<0C1JI<161J9<1I1IJ<2=1I;<301HK<3C1H<<461GM<4I1G=<5<1FN<601F?<6C1F0<761EA<7M1E:<8F1E7<9>1E5<:81E2<;01E0<;I1DN<<B1DK<=:1DH<>31DF<>J1DA<?C1D8<@:1CM<A21CB<AI1C8<B@1BM<C81BC<CO1B8<DF1AM<E>1AC<F51A8<FL1@M<GC1@D<H;1@9<I21?N<II1?D
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
20178N2LE7982LG79A2LJ79K2LL7:62M57:A2M?7:J2MG7;32MN7;<2N57;D2N;7;L2NC7<42NJ7<=2O17<E2O87<M2O>7=42O>7=F2OB7>42OD7>C2OG7?42OI7?E2OK7@62ON7@G3007A63027A?3037AI3047B33057B;3067BB3067BH3067BO3067C53067C<3067CC3067CI3067D03067D73077DC30D7DN30O7E53157E63137E83117E:3107E;30M7E=30L7E<30M7E83137E33177DO31=7DJ31B7DF31G7DC31K7DO31G7E@31B7F131=7FE31:7G93187GN3167HB3137I63117IK30O7J?30M7K130M7K;3107KE3147KO3187L931=7LB31H7LK3237M432=7M=32H7MF3327MO3397N633>7N=33C7NJ33F7O933H7OH33N80834880H34B81934L81I35682:35@82H35H83035J83935L83B35N83M35O84=35M84L35J85<35I85K35G86935C86F35=87535K87F35I88835D88I35@89;35<89L3578:>3538:O34N8;A34J8<<34C8=734<8>23458>N33N8?I33H8@=33D8@L33C8A;33B8A;3398A83318A632H8A432?8A13268@>32=8?I32H8?33328>E33:8>I33;8>M33;8?F33;8@E33C8AD33K8BC3438CB34;8D<34A8DN34E8EA34H8EH34A8EG3418EF33B8ED3328E@32J8E832J8E032K8DH32L8D@32L8D832L8D132M8D532L8DE32I8E632E8EG32B8F832?8FF3268FN31:8G630>8GE2OG8HE2OA8I52O88IF2NN8J62NE8JJ2N<8KG2N58LE2MO8MB2MH8N@2MA8O=2M:9092M69112M391I2M192B2LO93;2LL9432LJ94L2LH9592LC95B2L=95H2KO95L2K;9612JG9662J29682IB96;2I296=2HB96I2H39762GC97C2G39802FD98G2F899B2FA9:82FN9:N2G;9;E2GI9<<2H69=22HC9=I2I09>?2I=9?02IG9?:2IL9?D2J29@E2IH9AG2I?9BI2I69CB2I09CN2HO9D;2HO9DG2HN9E32HM9EE2H:9EM2H59EH2H=9ED2HF9EO2I29FK2IK9F?2JN9H22IL9IE2HK9K82GJ9L62I59M22JK9N02L09O22L=:052LI:172M6:292MC:372LD:442K6:502IH:5M2H;:6I2FM:7=2F<:7K2F;:8:2F;:8M2F5:9E2EI:9N2EK::82EM::J2EL:;A2EI:<82EG:<O2EE:=F2EA:><2E?:?32E=:?J2E::@A2E8:A82E5:AN2E7:B@2EL:C32FA:CF2G6:D82GK:DK2H@:F02H6:G52GL:H:2GB:H=2G3:I@2H5:J92G8:J52G7:J=2G7:J=2FE:JL2FE:K;2FE:KJ2FD:L92FD:LG2FD:L=2F4:LN2FA:M02FG:M12FO:M32G6:M52G=:M62GC:MO2GM:N:2GD:N>2G7:NB2FJ:NE2F>:NI2F1:NM2EE:O12E8:O52DK:OC2D:;012CI;0@2C9;0N2BG;1<2B7;112AD;1F2AL;2I2B:;3>2BG;3K2C3;482C?;4D2CK;4J2D7;4J2D5;4<2C5;3O2B6;592BC;652B=;6H2AN;7<2A>;7O2@O;8A2@@;952@0;9H2?A;:;2?2;:N2>B;;A2>3;<42=D;<G2=4;=:2<D;=M2<5;>@2;F;?32;6;?G2:G;@:2:8;@M29H;A?299;B328J;BF28:;C927K;CL27;;D?26K;E226<;EE25M;F825>;FK24N;G>24?;H1240;HE23@;I8231;IJ22A;J=222;K121B;KD213;L720C;LJ204;M=1OE;N01O6;NC1NF;O61N7;OI1MH<0<1M8<0O1LI<1C1L9<251KI<2H1K:<3;1JK<3O1J;<4B1IL<551I=<5H1HN<6;1H><6L1GM<7>1G=<861G7<8J1G3<9?1FN<:51FI<:K1FE<;@1FA<<61F=<<L1F8<=A1F3<>61EM<>J1EB<?B1E:<A41E;<B11DN<BH1DD<C@1D9<D71CO<DN1CD<EE1C9<F=1BO<G41BD<GK1B9<HC1B0<I:1AE<J11A:<JH1A0<K@1@E<L71@;<LN1@0<MF1?E
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
20379B2M?79L2MB7:52ME7:?2MG7:J2N17;52N;7;>2NB7;F2NI7;N2O07<72O77<?2O>7<G2OD7<O2OK7=83037=I30=7>430D7>E30F7?630H7?G30J7@830M7@D30N7@M3107A73107AA3127AK3137B33137B:3137BA3137BG3137BN3137C53137C;3167CB3187CI31:7D031=7D:31C7DE31O7DI3207DJ31N7DL31L7DN31K7DJ31O7DF3247DA3297D=32?7D832D7D932I7D>32N7DD3327DM3377E<33=7EL33D7F?33E7G333C7GG33B7H;33?7I033=7ID33;7J93387JM3367KA3347KK3387L533;7L?33@7LJ33D7M433H7M>33K7MH33O7N23437N<3477NE34B7NM34I7O;34L7OH34O80635180D35481435=81D35G82536082E36:82N36<83636?83?36A83H36C84036F84936H84G36F85636E85E36C86336>86@36986N36587;36387K36:88=36688O36189@35M8:135I8:C35D8;=35=8<83578=33508=N34I8>J34B8?E34;8@@3448A;33N8B533G8B933B8B73388B33368B03358AL3348AH3328AD3328A133;8@;33E8?E3408?433O8?733M8@13478A034?8AO34G8BN34O8CG3548D93588DK35;8E=35>8EE3588ED34I8EC3498EB33J8EA33:8E?32K8E:32E8E232F8DJ32G8DB32G8D:32H8D632H8DG32E8E832B8EI32?8F932<8FJ3298G;3268GK3208H33148HJ30G8I:30=8J03058JN2ON8KK2OG8LH2OA8MF2O:8NC2O38O@2NL90>2NE91;2N?9272N992O2N793H2N594@2N29592N095C2MK95K2ME9642M?96<2M996A2LF96E2L296J2K=96O2JJ9742J59782IB97=2HM97@2H<97E2GM9822G=98A2FN99<2G79:22GE9:H2H29;?2H?9<62HL9<L2I99=C2IF9>92J49?02JA9??2JJ9@12JK9A32JA9AH2J<9B52J;9BA2J;9BM2J:9C:2J99CF2J89D22J89D?2J79DN2IN9EA2I99F52HC9F12HK9FI2I99H52IN9I42IB9J92J59K:2JM9L<2K99M>2KF9N@2L39OC2L@:0E2LL:1G2M9:2I2MF:3G2LE:4D2K7:5@2IJ:6=2H<:722GF:7@2GE:802GE:8>2GD:8M2GD:9<2GC:9J2GC::C2G@:;:2G>:<02G<:<H2G9:=>2G7:>52G4:>L2G1:?C2FO:@92FM:A02FJ:AG2FG:B>2FF:C12G;:CC2H0:D62HE:E02HM:F52HC:FL2H0:F@2G4:GD2H7:HG2I<:J02H?:J?2GL:J;2GK:J62GJ:J>2GJ:JG2GI:K02GI:K82G?:KG2G?:L62G?:L;2G=:LO2GJ:M12H2:M32H9:M42H?:M62HF:M72HM:M92I5:M;2I<:MB2I3:N12HC:N?2H1:NM2G@:O;2G0:OI2F>;082EN;0F2E=;142DL;1B2D;;202CJ;1:2BJ;182BC;1G2BJ;2C2C6;3=2CC;3I2CO;462D:;4D2DF;542DK;622C2;6C2AD;6J2A;;6F2A8;642A;;6?2AB;782AJ;802B1;8G2B8;9F2B6;:92AF;:L2A7;;?2@H;<22@8;<E2?I;=72?9;=H2>G;>92>6;>J2=E;?;2=4;?L2<C;@=2<1;@N2;A;A?2:O;B02:?;BA29M;C229<;CC28K;D428:;DE27I;E6278;EG26G;F8265;FI25E;G:253;GK24B;H;241;HM23@;I>230;IO22@;J@220;K221?;KC20O;L420@;LE200;M71O@;MH1O0;N91N@;NK1N0;O<1M@;OM1M0<0?1L@<101L0<1F1KA<2>1K3<351JE<3N1J7<511JC<601J;<6K1IL<7F1I><8B1HN<991HH<9M1HC<:C1H?<;81H:<<01H7<<H1H5<=@1H2<>91GK<?41G@<?M1G6<@31G0<@51FM<@E1FC<AJ1F:<CE1F1<D?1EE<E61E:<EM1E0<FE1DE<G<1D;<H31D0<HJ1CE<IB1C<<J91C1<K01BF<KH1B<<L?1B1<M61AF<MN1A<<NE1A1<O<1@G=031@<=0N1@3=251?O
20479L2MM7:52MO7:?2N27:I2N57;42N?7;?2NH7;G2NO7;O2O67<82O>7<@2OD7<H2OK7=13027=93097=B30@7>63137>F3157?63177?H31:7@631<7@@31=7@I31>7A331@7A=31@7AF31B7B031C7B631C7B=31C7BD31C7BJ31C7C131F7C831H7C?31J7CF31L7CM31O7D53237DA32?7DC32>7DD32<7DE32;7DA32@7D=32E7D832K7D63307D;3357D@3387DF33=7DK33B7E033F7E<33K7EL3427F<3487FL34>7G<34D7GN34F7HC34D7I734A7IK34?7J?34=7K434;7KH3487L434<7L>34@7LH34D7M234G7M<34K7MF34O7N03537N:3577NE35:7NO35?7O=35>7OI35@80735C80E35F81235H81B35O82236982C36C82L36F83536H83=36J83F36L83O36O84837184@37384K37385;37285J36O86736J86D36E87136@87>36<87O36C88@36?89136:89C3668:43618:M35K8;H35D8<D35=8=?3578>:3508?534I8@034A8@L34;8AC3458B53448B@3458B?3488B;3478B73468B33448B03448AL3428A;34;8@D34E8@134K8?E34:8?G3478@:34C8A534M8B43558C335=8CE35@8D735C8DI35G8E;35J8ED35C8EC3548EB34E8E@3458E?33F8E>3368E=32G8E732C8DN32D8DF32E8D?32E8D732E8DH32B8E932?8EJ32<8F:3298FK3268G;3238GM3208H=31L8HM31:8ID3118JA30J8K>30D8L<30=8M93068N62OO8O42OH9012OA90N2O;91L2O492I2NM93B2NK94;2NH9532NF95C2NB95L2N;9642N696=2N096E2MJ96K2M<9702LG9752L39792K?97>2JK97B2J797G2IC97K2HO9802H;9832GJ98@2G:9992GC99O2H09:E2H=9;<2HJ9<32I89<I2IE9=@2J29>62J?9>M2JL9?B2K89@82K79@L2K39A82K29AD2K19B12K19B=2K09BI2JO9C62JN9CB2JN9CN2JM9D;2JL9DK2J>9E?2IH9F32I39F:2I29H22IE9I=2J59J?2JA9KB2JN9LD2K:9MF2KG9NH2L49OK2LA:0M2LN:1O2M::312MG:3O2LF:4L2K8:5H2IJ:6E2H<:742H::7B2H::822H9:8@2H9:8O2H9:9@2H4::82GI::=2H3:;82H6:<82H5:<O2H2:=F2H0:>=2GM:?42GJ:?J2GH:@A2GF:A82GC:AO2G@:BE2G>:C92H2:CK2HG:D@2I9:EE2HO:F12H6:EB2G4:FE2H8:GI2I<:HM2J2:J72HE:JB2H6:J=2H5:J92H4:J92H3:JC2H3:JL2H3:K52H2:K>2GM:KM2GM:L:2H0:M02H?:M12HF:M32HM:M52I5:M62I7:M72I8:M82I9:M92I::MK2I>:NB2I4:O02HC:O>2H2:OL2GA;0;2G0;0I2F?;172EN;1E2E>;232DL;2B2D<;102C0;1<2C4;1K2C:;2@2CC;3<2D1;3I2D<;452DH;4C2E3;5>2DH;6=2B8;6B2B1;6H2AI;712A@;782A8;7A2A>;872AG;8G2B3;962B?;9:2BC;9?2B9;9N2A:;:?2@I;;02@8;;A2?G;<22?6;<C2>E;=42>3;=E2=B;>62=1;>G2<@;?82;O;?I2;>;@:2:M;@K2:<;A<29K;AM29:;B>28I;BO289;C@27I;D2279;DC26I;E5269;EF25I;F7259;FI24J;G:24:;GK23J;H<23:;HN22I;I?229;J021I;JA219;K320I;KD209;L51OJ;LF1O:;M81NJ;MI1N:;N>1MK;O61M<;ON1LN<0E1L?<1=1L2<251KC<311K><4=1L9<581KI<641K:<6O1JL<7J1J=<8F1IN<9@1IB<:71I?<;01I<<;H1I9<<@1I7<=81I4<>11HN<>K1HE<?E1H:<@:1GO<@C1GN<@F1GL<A21GC<AB1G8<B51FM<CE1FM<E:1F6<F11EK<FI1EA<G@1E6<H71DK<HN1DA<IF1D6<J=1CL<K41CB<KL1C7<LC1BL<M:1BB<N21B7<NI1AM<O@1AB=081A7=191A1=2@1@M=3F1@J=4M1@F
2057:62N;7:?2N=7:I2N?7;32NC7;?2NN7;H2O67<02O=7<92OD7<A2OK7<I3017=23097=:30@7=C30G7?731A7>F31E7?731G7?H31I7@231K7@<31L7@E31L7@O31N7A831O7AB3207AL3217B23217B93217B@3237BG3267BN3297C532;7C<32=7CC32?7CJ32B7D132D7D;32M7D=32L7D<32L7D83317D43367D833;7D=33?7DB33D7DG33H7DM33M7E23417E73457E<34:7EK34@7F<34G7FK34M7G;3537GK35:7H;35@7HK35F7I>35F7J235D7JG35B7K;35@7KO35=7L<35@7LF35C7M135G7M;35K7ME35O7MO3637N93677NA3667NE35O7O936680436280>36580H36781336:81B36<82036@82@36K82K36O83337183<37383E37683N37884637:84?37<84G37?85037A85@37?85N37:86;37686H37187536L87B36G88136L88C36G89436C89E36?8:>3698;93628<435J8<O35C8=J35=8>F3568??3508@134M8@C34L8A534J8AG34G8B234E8B:34J8B@3548B?3578B;3568B73548B43548AD35:8@N35E8@A3568@634E8@734A8@J34M8A=35:8B935C8C135G8CC35K8D535N8DG3618E93658EC3608EA35@8E@3518E?34A8E>3418E=33B8E<3328E:32C8E332B8DK32B8DC32B8D;32C8DI32@8E:32=8EJ32:8F;3278FL3248G=3218GM31N8H>31K8I631L8J431F8K131@8KO31:8LL3138MI30L8NG30E8OD30>90B30791?30192<2OJ93:2OC9452O>94N2O;95D2O895L2O29652NM96=2NG96E2NA96N2N;9762N197:2M=97?2LI97C2L597H2KA97L2JM9812J89862IE98:2I098?2H<98C2GH9972GM99M2H;9:D2HH9;:2I59<12IB9<G2J09==2J=9>42JJ9>J2K79?@2KD9@02KI9@<2KH9@H2KG9A42KG9A@2KF9AM2KE9B92KD9BE2KD9C12KC9C>2KB9CJ2KA9D62KA9DI2JN9E=2J89F12IB9FK2J>9IC2J>9JH2JC9KJ2JO9LL2K<9MN2KI9O02L6:032LC:152M0:272M<:392MI:472LF:542K9:602IK:6G2HO:762HO:7D2HO:832HN:8B2HN:942HI:9L2H<::D2H1:;12H0::O2HB:;@2HI:<E2HI:=I2HH:>E2HF:?;2HC:@22HA:@I2H?:A@2H<:B62H9:BN2H7:CA2HI:D32I?:E42I::E72H;:DC2G4:EG2H8:FJ2I<:GN2JA:I42J8:J?2HJ:JE2H?:J@2H?:J<2H>:J82H=:J>2H<:JH2H<:K12H<:K:2H;:KD2H9:L92HE:LL2I1:M32I::M32I9:M42I9:M52I9:M52I8:M62I7:M72I4:MA2I9:NG2I;:OA2I4:OO2HC;0>2H3;0L2GA;1:2G0;1H2F@;262EN;2E2E>;2=2DD;102C>;1@2CD;1O2CK;2?2D1;3;2D?;3H2DJ;452E6;4M2DO;5I2D>;6<2BF;6B2B=;6G2B6;6N2AM;772AD;7F2AL;862B8;8@2BA;8D2BF;8H2BJ;8K2BO;8O2C3;972@2;9G2?:;:<2?D;:M2?3;;>2>C;<02>3;<A2=C;=22=2;=D2<B;>52<2;>F2;B;?72;2;?I2:B;@:2:3;@K29C;A<293;AN28C;B?283;C027C;CB273;D326C;DD263;E625C;EG253;F824C;FJ243;G;23C;GL233;H=22C;HO223;I@21C;J1213;JB20C;K6204;KN1OF;LE1O7;M=1NI;N51N;;NM1ML;OD1M><0>1M1<1>1M2<1J1M0<331M?<4B1M4<5>1LE<691L6<751KG<801K8<8L1JI<9H1JA<:F1J><;?1J;<<71J9<=01J6<=H1J2<>B1IH<?81I=<?E1HO<@81HD<@O1HE<A71HJ<A?1HD<AO1H8<B@1GM<C01GB<D11G9<F51FF<FM1F<<GD1F1<H;1EG<I21E<<IJ1E1<JA1DH<K81D=<L01D2<LG1CH<M>1C=<N61C2<NM1BH<OD1B==0>1B3=1D1B0=2K1AL=411AH=581AC=6>1A@=7E1A<
2067:@2NH7:I2NK7;32NM7;=2O17;I2O;7<22OD7<:2OK7<B3017<J3087=330?7=;30F7=D30M7>K31B7?<3247?83267?D3287?N3297@832:7@A32<7@K32<7A432>7A>32?7AH32@7AO32B7B632D7B=32F7BD32I7BK32K7C232M7C932O7C@3327CG3357CN3377D533;7D333=7D433B7D:33F7D?33K7DD33O7DI3447DO3477E434<7E934A7E>34E7ED34J7EK34N7F;3547FK35;7G;35A7GJ35G7H:35N7HK3647I:36:7IJ36A7J:36G7JN36F7KB36D7L636B7LD36C7LO36G7M936K7MC36O7MM3737N236M7N636E7N;36=7NI37280=36E80K36F81536I81?36L81I36N82337182>37382I37883237:83;37<83D37?83L37A84537C84>37E84F37H84O37J85737L85D37K86137E86>37A86K37<87837887E37288437488E37089636L89M36F8:I36?8;D3688<?3618=:35J8=L35H8>>35G8?035D8?B35B8@435@8@F35=8A935;8AC3598AN3578B535>8B;35H8B@3638B?3668B;3648AM36:8A>3628A335A8@G3518@F34J8A93578AL35C8B=3608BO3638CA3668D33698DE36<8E836@8EA36;8E@35L8E?35<8E>34M8E=34>8E;33N8E:33>8E932O8E832?8E032?8DH32@8D@32@8DI32?8E;32<8EK3298F<3268FL3238G>31O8GN31L8HK3218IH3268JE3268KB31O8L?31J8M<31D8N:31?8O731990431391230M91O30F92M30?93J30894H30195A2ON95M2OI9652OD96=2O>96F2O896O2O29772NL97@2NF97D2N297I2M?97M2LJ9822L79872KB98;2JN98@2J:98D2IF98I2I29972HC99F2H69:<2HB9;22HO9;I2I<9<?2II9=52J69=L2JD9>B2K19?92K>9?M2KJ9@62KM9@=2L09@C2L49@I2L89A02L;9A<2L:9AH2L:9B52L99BA2L89BM2L79C:2L79CF2L69D32L29DG2K=9E;2JK9FE2K49IE2K09JM2JM9L22K19M42K=9N62KJ9O92L7:0;2LD:1=2M0:2@2M=:3B2MJ:4?2LG:5<2K9:682IK:6I2ID:782ID:7F2IC:852IB:8H2I=:9@2I1::82HF::O2H9:;F2GO:;D2H?:;B2I0:;I2I>:<M2I=:>22I<:?62I;:@:2I::A12I8:AG2I5:B>2I2:C52I0:CH2IA:DE2IE:D<2H@:CE2G4:DH2H9:EL2I<:FO2JA:H32KE:I<2J=:JF2I0:JG2HJ:JC2HI:J?2HH:J:2HG:J:2HF:JC2HF:JL2HE:K02H=:K42H::KD2HG:L92I6:M02II:M@2FM:M=2IH:M<2JF:M=2JE:M22J8:M02I4:M:2I5:N=2I6:OB2I8;0A2I5;0O2HC;1=2H3;1K2GB;292G1;2H2F@;362EO;222DJ;142CN;1C2D5;232D;;2C2DB;3:2DL;3G2E8;4<2E7;582DK;652C@;6<2C3;6A2BK;6G2BC;6M2B;;752B2;7D2B=;7J2BC;7N2BH;822BM;852C1;892C6;8=2C:;8A2C?;8G2@N;9>2A8;:12@5;:E2?3;;42==;;C2=<;<12=;;<B2<L;=32<<;=E2;L;>62;<;>G2:L;?82:<;?J29L;@;29=;@L28M;A=28=;AO27M;B@27=;C126M;CC26=;D425M;DE25=;E724M;EH24=;F923M;FK23=;G<22M;GM22=;HF21O;I=21A;J5212;JM20D;KD205;L;1OG;M41O:;MN1NM;NM1NI;O:1NG;OF1NF<021ND<0O1N><2C1N4<471MK<5C1M@<6?1M1<7:1LB<861L3<921KE<9O1KB<:M1K><;K1K;<<H1K8<=@1K5<>61JK<>D1J=<?11J0<?>1IC<@71I8<A11I1<AA1IA<AK1ID<B<1I9<BL1HN<C=1HB<CN1H7<D>1GL<F31GE<GH1FM<H?1FB<I61F7<IN1EM<JE1EB<K<1E8<L41DN<LK1DC<MB1D8<N:1CN<O11CC<OH1C9=0H1C2=1O1BN=351BJ=4<1BF=5C1BB=6I1B>=7L1B9=7O1AB=821@L=861@6
2077:I2O57;32O87;=2O;7;G2O@7<32OJ7<;3017<C3077<L30?7=430F7=<30M7=E3137>?31C7@A32C7?B32G7?>32L7?H32O7@23307@;3327@D3347@N3357A93327AD32I7AK32L7B232O7B93327B@3357BG3377BN33:7C533=7C<33@7CC33B7CJ33E7D633M7D<3417DA3467DF34;7DK34>7E134C7E634G7E;34L7E@3517EF3557EK3597F035=7F;35C7FJ35J7G:3607GJ3667H:36<7HJ36C7I:36I7IJ36O7J93767JJ37<7K937B7KI37H7L=37G7LL37G7M737K7M?37K7MC37C7MH37<7ML3747N036L7N@37D7OM37B81837881B37:81L37=82637?82@37B82I37E82J37@83237C83;37E83D37H83L37J84537L84=37O84F38184O38385738585@38885L38786638286B37L86O37H87<37C87I37>88737:88H37989>3738:936L8;436F8;G36C8<936@8<K36?8==36<8=O36:8>A3688?43668?F3638@83618@J35O8A435M8A?35K8AJ35I8B03638B536<8B;36G8B@3728B;36N8AO36=8AD35M8A835<8A63548AI35A8B:35M8BC3668BL36>8C?36A8D136E8DC36H8E636K8E@36G8E?3678E>35H8E<3588E;34I8E:34:8E933J8E833;8E732K8E432=8DL32=8DE32>8DJ32<8E;3298EL3268F=3238FM3208GC3208H?3258I<3298J932>8K532B8L332?8M03298MM3238NK31M8OH31H90E31B91B31=92@31693=31194:30K95830F95M30@96630:96>30496F2ON96O2OI9782OC97@2O=97I2O897O2NH9832N49882M@98<2LL98A2L898E2KD98J2K098N2J;99<2IM99K2I?9::2I29:J2HH9;A2I59<72IB9<M2J09=D2J=9>:2JJ9?12K79?G2KD9@52KK9@>2KO9@F2L39@L2L79A32L:9A92L=9A@2LA9AF2LE9AM2LI9B32LL9B=2LM9BI2LL9C62LL9CB2LK9D12LB9DE2L=9FG2KK9I>2KA9JO2K?9L62K<9M<2K?9N>2KL9OA2L9:0C2LE:1F2M2:2H2M?:3J2ML:4G2LG:5C2K::6<2J9:6K2J8:7:2J8:7H2J8:8<2J1:942IE:9K2I9::C2HN:;;2HB:<22H6:<92H>:<72HO:<52I?:<32J0:=62J1:>:2J0:??2IO:@C2IN:AH2IM:BF2IK:C=2II:D52J2:CA2HF:BH2G7:CJ2H9:DM2I=:F12JA:G42KE:H:2KO:ID2JB:JN2I5:JJ2I4:JF2I2:JB2I2:J=2I1:J22I6:J42HA:J72H9:J:2H0:J@2H;:K02HI:KF2I<:L=2J2:M32II:MG2F4:ML2F=:MM2I7:M22J1:L=2I?:M82HO:N32I2:O82I4;0>2I5;1@2I5;1N2HD;2=2H3;2K2GB;392G1;3?2F=;1H2E0;182D?;1G2DE;262DK;2F2E2;392E:;3K2E>;4F2E3;5B2DG;662CH;6;2C@;6A2C8;6F2C0;6L2BH;732B@;772BD;7;2BH;7?2BM;7C2C1;7G2C6;7K2C:;7O2C>;832CC;872CG;8D2C2;952B4;9H2A1;:;2?O;:N2>L;;B2=@;<02:I;<>2:I;<L2:G;=;2:F;=I2:E;>72:D;>H2:6;?929F;?K297;@<28G;@M287;A>27G;B0277;BA26G;C2266;CD25F;D5256;DF24F;E=248;F523J;FM23;;GD22M;H;22>;I221O;IJ21B;JD216;K?20I;L;20A;LH20@;M720=;MN204;NF1OM;O>1OD<071O?<141O7<2:1NK<3F1NA<5:1N7<6E1MK<7@1M<<8<1LM<991LE<:61LB<;41L?<<11L<<<O1L7<=C1KM<=O1K><><1K0<>I1JD<?;1J7<@51IM<A11ID<AM1IO<B;1J=<BI1J9<C91IN<CJ1IC<D:1I8<DK1HL<E;1HA<F91H9<HC1G=<I:1G3<J21FH<JI1F=<K@1F3<L81EI<LO1E><MF1E4<N>1DI<O51D><OL1D4=131D0=291CL=3@1CH=4G1CD=5M1C@=741C<=7I1BO=7L1B9=7O1AC=821@M=851@7=891?A
2087;32OC7;=2OE7;G2OG7<12ON7<<3077<D30>7<M30E7=530L7==3137=F31:7>331C7@832E7@G3367@C33<7@?33B7@;33H7@733N7@>33K7@K33>7A83327A@3317AG3337AN3367B53397B<33;7BC33>7BJ33B7C133E7C833G7C?33J7CO3437D534;7DC34C7DM34J7E334N7E83537E=3577EB35;7EG35@7EM35D7F235I7F835M7F=3627FJ3677G:36>7GJ36D7H936J7HJ3717I93777II37=7J937C7JI37J7K93807KI3867L938=7LH38C7M138A7M538:7M93827M=37J7MB37B7MF37:7N73877O>38>81138181O37K82937O82C38182M38483638683838283037F83537G83D38183M38384538584>38884G38:84O38<85838>85@38A85I38C86738D86C38>86M38:87838587C37O87M37J88:37E88N37@89B37=8:437;8:F3798;83778;J3758<=3728<O3708=A36N8>336L8>E36I8?736H8?I36F8@:36C8@E36A8A036?8A;36<8AE36<8AJ36F8B03718B53788B836O8B<36F8B53688AI35G8AF35>8B835K8BA3638BJ36<8C336D8C=36L8CO36O8DA3738E33768E?3728E=36C8E<3638E;35D8E:3548E934E8E83468E633F8E53378E332G8E132;8DI32;8DK32:8E<3278EM3248F>3218G73248H43298I032=8IM32B8JJ32F8KF32K8LC32M8MA32H8N>32B8O;32<90932691632192331K93131F93N31@94J31:95=30M95O30G96=30I96G30E97030@97830:97A30497I2ON9812OH9892O>98=2NJ98B2N598F2MB98K2LM98O2L:9952KE99A2K69:02JI9:?2J;9:N2IN9;>2IB9;O2I<9<E2II9=<2J69>22JD9>I2K19??2K>9@32KJ9@<2KM9@E2L19@N2L59A62L89A<2L<9AC2L@9AI2LC9B02LG9B62LJ9B=2LN9BC2M29BI2M69C02M99C62M<9C>2M?9DE2M59FH2LC9I12L39K12L19L82KN9M@2KL9NF2KM9OI2L::0K2LF:1N2M3:302M@:422MM:4O2LH:5L2K::6>2JN:6M2JM:7<2JM:802JE:8H2J::9?2IN::72IB::N2I6:;F2HJ:<>2H?:<N2H<:<L2HL:<J2I>:<H2IN:<F2J?:=>2JD:>C2JD:?G2JC:@L2JB:B02JA:C?2J>:BF2HK:AM2G<:BL2H9:CO2I=:E22JA:F62KF:G92LJ:HA2L4:IL2JH:K12I?:JM2I>:JI2I<:JD2I<:ID2J5:IL2H3:IH2GN:ID2GJ:ID2GC:IL2H<:J<2HL:K32IB:KJ2J7:LA2JL:M62IJ:M52FC:NI2F:;0=2F1:O12GL:LM2I3:MI2HN:NO2HO;032I1;192I2;2>2I4;2N2HE;3<2H3;3J2GC;352FC;1?2E6;1;2DO;1K2E5;2:2E;;2J2EB;342EF;452E;;512DO;5L2DC;652D5;642CJ;5N2C=;672C6;6E2C0;732BJ;7D2B?;822B7;862B;;8:2B@;8>2BD;8B2BI;8F2BM;8J2C1;8N2C6;9<2B>;9K2AB;::2@G;:H2?K;;82>O;;G2>3;;O2=G;<:2;2;<H29@;=429:;=B292;=N28K;>;28D;>H28=;?4285;?A27O;@027N;@>27M;@N27@;A?270;B526B;BL262;CC25D;D;256;E224G;EI248;FA23J;G;23>;H5231;I022E;IJ229;JA222;K921J;L021B;LH219;M@211;N720I;O020C;OJ20=<0C208<1?201<2E1OE<3K1O:<511NM<6=1NC<7F1N7<8B1MH<9?1ME<:=1MB<;:1M=<<71M8<=41M3<=?1LD<=K1L6<>71KF<>C1K7<?B1K4<@E1K4<AI1K5<BB1JK<C01JJ<C91JL<CI1JG<D81JA<DG1J8<E81IM<EH1IB<F91I7<FI1HK<HC1H;<J61GC<JM1G9<KD1FN<L<1FD<M31F:<MJ1EO<NB1ED<O91E:=081E2=1>1DN=2E1DK=3K1DF=521DB=681D>=7?1D;=7E1CF=7H1C0=7K1B9=7O1AC=821@M=851@7=881?B
2097;>3007;G3037<13057<;30<7<E30E7<N30L7=63137=?31:7=G31@7=O31G7?M32F7AH33D7A=33G7A233K7@H33N7@=3417@=33L7@J33?7A53387A<3387AC33;7AJ33>7B133@7B833C7B?33F7BF33H7BM33K7C433N7CC3467D134A7D534J7DB3517E035:7E>35B7ED35G7EI35K7EO3607F43647F:3697F?36<7FD36A7FJ36F7G936L7GI3727H93797HI37?7I937E7II37K7J93827JI3887K938>7KH38E7L538H7L?38J7LI38K7LK38H7LO38A7M33887M73817M;37I7MN38I7NN39:80A38M82538@82F38@83038B83:38E83D38H83F38D83>38883637L83737H83H38584638>84?38A84G38C85038E85838H85A38J85J38L86338N86@38N86O38L87938G87D38A87O38<88:38788D38089237E89A37E89O37E8:?37F8:N37G8;=37H8;N37F8<@37D8=237B8=D37@8>637>8>I37<8?;3798?K3778@63758@@3728@L3708A636N8A?3708AE37;8AH3728AL36J8AO36B8B33698B63608B535H8B?3618BH3698C136C8C:36K8CC3738CM37;8D?37>8E137B8E=37>8E<36O8E;36@8E:3608E935@8E73518E634A8E43428E333B8E23338E132C8DN3298DL3288E=3258EO3238FK3288GH32<8HE32A8IB32E8J>32J8K;32O8L83338M43388N23368NO3318OL32K90J32E91G32?92D32:93A32494631I94J31>95>31295O30J96=30M96K30O97931097A30J97J30E98230?98:30998C30398G2O?98L2NK9912N79962MC99:2LO99F2L@9:52L39:D2KE9;32K89;C2JK9<22J>9<A2J09=42J09=K2J=9>A2JK9?72K79?N2KD9@;2KK9@D2KO9@M2L39A62L79A?2L:9AF2L>9AM2LB9B32LE9B92LI9B@2LL9BF2M09BM2M39C32M79C92M;9C@2M?9CK2MB9E02MF9FJ2M:9HL2LI9K32LC9L;2L@9MC2L>9NJ2L;:012L;:132LH:262M5:382MB:4:2MO:572LI:622KC:6@2KB:6O2KB:7D2K::8<2JN:932JB:9J2J6::B2IK:;:2I?:<12I3:<I2HG:=A2H<:=@2HK:=?2I<:==2IM:=;2J>:=92JN:=G2K8:>L2K7:@@2K3:B;2JN:D52JI:B@2HA:AN2H9:C12I>:D42JA:E82KF:F;2LJ:G?2MG:HI2L::J32JM:K42IH:K02IH:J=2J5:I:2K1:H<2KJ:IB2G?:I>2G::I:2G6:I92GF:I92H=:II2I2:IJ2I<:J42IF:J?2J2:JI2J=:K?2JH:M12IO:NC2I6;042H=:NE2HJ:MC2HJ:NE2HK:OJ2HL;0O2HN;242HO;392I1;3M2HE;4;2H4;2K2FJ;152E<;1?2E?;1O2EE;232EH;222EF;2J2EE;422EA;4D2E1;502D<;4O2D1;5<2CK;5J2CF;672C@;6E2C:;732C4;7B2BL;842B>;8G2B0;902AN;952B3;992B7;9=2B;;9A2B@;9E2BD;:32AJ;:B2@N;;12@2;;@2?7;;M2>@;<32>:;<92>5;<D2;F;=32:9;=@2:2;=M29J;>929C;>F29=;?3295;??28N;?M28G;@928@;@F288;A327O;AD27C;B:275;C026F;CF267;D<25H;E5253;F7246;G423G;H0237;HJ230;I>234;J3233;JK22I;KB22?;L9226;M121O;MJ21H;NC21B;O=21<<06217<0O211<1J20J<3020?<46203<5;1OG<6A1O<<7G1O0<8H1NH<9F1ND<:C1N><;@1N:<<>1N5<=01MK<=<1M;<=G1LL<>71L?<?11L7<?O1L3<A21L3<B21L0<BG1KG<C>1K=<CI1K><D91K:<DI1K4<E91K0<EH1JH<F71J@<FE1J7<G61IL<GG1IA<HA1I9<K11H4<KI1GI<L@1G?<M71G5<MN1FJ<NF1F@<O=1F5=0B1F0=1I1EM=2O1EI=461ED=5=1EA=6C1E==7>1E3=7B1D==7E1CG=7H1C1=7K1B:=7N1AD=821@N=851@8=881?B
2107;H30=7<130@7<;30B7<F30J7<O3127=731:7=@31@7=H31G7>031N7?@32F7A=33F7A;33H7A233K7@G33N7@<3417@<33M7@I33@7A133?7A833@7A?33B7AF33E7AM33H7B433K7B;33M7BB3407BI3437C53497CN34G7D23507D63587DA35@7DO35H7E=3607EK3687F636?7F<36C7FA36H7FF36L7FK3717G13767G937:7GI37A7H837G7HI37M7I83837IH38:7J838@7JH38F7K738K7K@38M7KJ38N7L43917L>3927LH3947LH38N7LH38G7LL38?7M13887ME39;7N>3:780239J81F39<83339183=39483G39784139984439683L38J83E38>83<38283937K83J38784:38C84H38L85038N85939185B39385K39486339786<39986J39:87;39987F39488138N88;38I88F38E89037M89=37@89L37A8:<37B8:K37B8;937B8;H37C8<737D8<G37E8=637E8=E37F8>337G8>C37H8?237H8?>37I8?I37G8@437E8@>37B8@J37@8A437>8A<3798AB3718AH36I8AM36B8B336:8B83638B?3638BG36<8BO36@8C736I8C@3718CI3798D237B8D=37I8DO37L8E<37J8E;37:8E:36K8E836<8E735L8E635=8E434M8E334>8E233N8E133>8E032O8DN32?8DM3268EC3268F@32<8G=32@8H932E8I632I8J332N8JO3328KL3378LI33;8ME33@8NB33E8O@33?90=33:91:33492732O93032E93C32:94731N94J31C95>31696030N96>31096L31397:31597H31898231698;31098C30J98L30D9923059972OA99;2NM99@2N899K2MI9::2M<9:J2LO9;92LA9;H2L49<72KG9<F2K:9=52JL9=E2J?9>92JD9>O2K19?F2K>9@92KJ9@B2KM9@K2L19A52L59A>2L99AG2L<9B02L@9B62LC9B=2LG9BC2LK9BI2LO9C02M29C62M69C<2M99CC2M=9CI2M@9DJ2MD9EN2MI9G32ML9HN2M@9K52M69L=2M39ME2M09NL2LM:042LJ:1<2LI:2>2M6:3@2MC:4B2N0:5?2LI:632L7:6B2L7:772KN:7O2KB:8G2K6:9>2JK::62J>::N2J3:;E2IG:<=2I<:=52HO:=L2HD:>52HI:>32I::>12IK:=O2J;:>02JN:>22KA:?O2LA:A72K=:C12K8:DL2K3:CL2IE:C52IK:CL2JK:DD2KJ:E=2LJ:F@2MO:GG2ML:I12L?:J;2K2:K62J3:J32K0:I02KL:GM2LH:H32L8:I92FJ:I52FF:I32G5:I42GI:I:2H2:IB2H<:IM2HG:J82I2:JC2I>:JL2IL:K02JC:KD2K2:LJ2K9:N72JL:NH2J@:N>2IN:NO2HI:ON2HF;102HI;212HJ;332HL;452HO;472H<;2A2G0;142EI;1=2EI;1H2EI;232EH;2?2EH;3=2EE;4<2E>;4:2DE;4B2D?;4O2D;;5<2D5;5J2CO;672CI;6E2CD;722C>;7@2C9;822BK;8E2B=;982AO;9K2AA;9O2AE;:32AJ;:72AN;:;2B1;:J2A6;;:2@:;;H2??;<12?3;<72>N;<=2>I;<C2>C;<M2<;;==2;;;=K2:J;>82:C;>E2:<;?12:5;?>29M;?K29F;@829?;@D298;A228N;AA28C;B1287;BF27J;C;27;;D226L;DG26>;EL256;G323I;GK23E;H?23I;I423M;IH241;J>23L;K523B;KM238;LD22L;M=22F;N622A;NO22;;OI225<0B220<1;21J<2521D<3;219<4A20L<5G20A<6L205<7N1OL<8N1OE<9K1O@<:I1O;<;F1O7<<A1O1<<M1NA<=81N2<=M1MH<>G1M?<?@1M7<@<1M1<A?1M1<B:1LK<BO1LB<CD1L8<D:1KN<DH1KK<E91KF<EI1KB<F91K=<FI1K8<G71K0<GF1JG<H51J?<HD1J6<I41IK<K41I0<LD1H:<M;1H0<N31GF<NJ1G;<OF1G3=0M1FO=231FK=3:1FF=4A1FC=5G1F?=6N1F;=7;1EJ=7>1E4=7A1D>=7D1CH=7H1C1=7K1B;=7N1AE=811@O=841@9=881?B=8;1>L=8>1>6=8A1=@=8D1<J=8H1<4=8K1;>=8N1:H=921:2=9519;=9818E=9;17O=9?179=:016H
2107;H30H7<130K7<;30M7<E3107=03197=831@7=A31F7=I31N7>23257?432G7A533H7A033J7@N33K7@G33N7@<3417@;33N7@F33G7@M33G7A433H7A;33I7AB33L7AI33O7B03417B73447B>3477BH34<7CA34J7D03557D435>7D735G7D@35O7DN3667E<36?7EJ36F7F836O7FF3777FM37<7G237@7G837E7G=37I7GH37O7H83857HH38;7I838A7IH38H7J838N7JB3907JL3917K63947K@3957KJ3977L33987L=39:7LE39:7LD3947LC38N7LH38J7M<39M7N03;27OC3:F8163:982J39K83J39E84439H84>39J84B39H84:39<84239083K38D83C38883;37L83L38884<38E84M39185:39985B39;85K39>86439@86<39B86E39D87339E87E39D88239A88=39<88H39789238L89<37K89I37<8:837<8:G37=8;637>8;E37>8<437?8<D37?8=237@8=A37A8>037B8>=37E8>I37H8?537L8?@37O8?K37M8@637K8@@37H8@L37F8A837D8AF37?8AL3778B136O8B736H8B<36@8BB3698BI36<8C336I8C<3768CC3798CI37<8D037@8D937H8DA3818DN3888E;3858E937F8E83768E736G8E63688E435H8E33598E234I8E134:8E333L8E433>8E732O8E<32D8F032B8FN32F8GL32J8HJ32N8IG3328JD3368K@33;8L=33?8M:33D8N733H8O333M90033N90N33H91I33B92=33693032J93C32?94732394K31G95>31;96031396>31596L31797:31:97H31<98631>98D31@98L31;99531599<30J99A30799E2OB9:12O29:@2NE9:O2N89;>2MK9;M2M=9<<2M09<K2LC9=;2L69=J2KH9>92K;9>H2JN9?>2K79@42KD9@A2KK9@J2KO9A32L39A<2L79AE2L:9AO2L>9B82LB9B@2LF9BF2LI9BL2LL9C32M09C92M49C@2M89CF2M;9CL2M?9DE2MB9EI2MF9FN2MK9H22MO9I72N29K72MH9L?2ME9MG2MB9NN2M?:062M<:1>2M::2F2M8:3H2ME:4J2N1:5F2LL:652LL:6K2LB:7C2L7:8;2KJ:922K?:9J2K3::B2JH:;92J;:<12J0:<I2ID:=@2I9:>82HL:>J2HG:>L2I::>N2IM:?02J@:?22K3:?42KF:A42LH:C>2ML:E32MG:F52L;:EJ2KD:DO2KG:E32L;:EK2M<:F>2N9:FJ2O2:G22N@:G32N5:H42LN:HJ2L0:G?2LO:G32MB:G02N5:GO2JN:HO2F7:HN2FD:I32FM:I92G5:I?2G>:IG2GG:J22H3:J=2H>:JG2HI:JM2I=:K22J3:K62JJ:KK2K::M02KA:N52KH:NN2JJ:OF2IK;0;2HI;172HB;262HG;362HL;452I2;4?2HF;492H2;2E2FE;1J2EK;252EK;2@2EK;2K2EJ;402EG;472E3;4<2DH;4F2DE;512DB;5<2D?;5J2D9;672D3;6D2CN;722CH;7@2CC;802C8;8C2BJ;962B<;9I2AN;:<2A@;:I2A8;:M2A=;;32A>;;B2@B;;O2?L;<52?F;<:2?A;<@2?<;<F2?6;<L2?0;=72=;;=G2<<;>72;B;>C2;<;?02;4;?=2:M;?J2:F;@62:?;@C2:8;A129M;A?29A;AN296;B=28J;C228=;CG280;D=27B;E;26E;EI26@;F326C;FL268;GN25J;I025<;J124N;JI24C;K@247;L723K;M023A;MI23=;NB238;O;234<0522N<0N22I<1G22C<2@22=<3F222<4K21F<6121:<73212<8320K<9220C<:120=<:O208<;L203<<=1OG<<J1O9<=C1O1<><1NH<?61N@<?O1N8<@I1MO<AL1MN<BA1ME<C71M<<CL1M4<D@1M5<E11M1<EE1LC<F81L5<FH1KO<G81KJ<GI1KE<H91K@<HG1K7<I51JO<ID1JG<J31J><K11IO<M?1HK<N71HA<NN1H6=011H1=181GM=2?1GI=3E1GE=4L1GA=621G==741G7=771F@=7;1EJ=7>1E4=7A1D>=7D1CH=7G1C2=7K1B<=7N1AF=811A0=841@9=871?C=8;1>M=8>1>7=8A1=A=8D1<J=8G1<4=8K1;>=8N1:H=911:2=9519<=9818F=9;180=:817C=;H17===8178=>H172
2107;H3137<23167<;3187<E31;7<N31=7=931F7=B31M7=J3247>332;7>I32H7@M33J7@D33K7@B33N7@A33O7@<3417@:33N7@B33N7@I33N7@O33O7A63417A>3437AE3467AL3497B334<7B;34?7C334M7CL35;7D135C7D535M7D83657D?36=7DM36E7E;36M7EI3757F737=7FE37E7G337M7G?3857GD3897GI38=7H738C7HH38J7I83907IC3937IM3947J73977JA3987JK39:7K539;7K?39=7KI39>7L339A7L=39B7LB39A7LA39:7L?3947M33:@7MG3;D7O33;C80F3;582:3:H83N3::84A3::84K3:<84O3::84H39N84@39B84839684138J83I38>83A38283N38;84>38G84O39385?39@85L39G86439I86=39K86E39M86N3:087<3:087O3:088>39N88I39I89439D89>38J89H37H8:53778:D3788;33798;B3798<137:8<@37;8<O37;8=<37?8=G37B8>337F8>?37J8>K37N8?73828?B3858?M3838@83818@C37O8@N37L8A?37I8B037E8B537=8B;3768B@36O8BF36G8BK36@8C436F8C=3738CF37@8D037M8D73828D=3868DD3898DJ38=8E138A8E938A8E83828E937C8E;3768E=36G8E?3698EA35K8EB35=8ED34O8EF34A8EH3438EJ33E8F433=8FB3388G63378H433:8I133>8J033B8JM33E8KK33I8LI33L8MG3408NE3448OB34790A34;91634291I33F92=33;93032O93D32C94732794K31L95>31@96131696>31996M31;97;31=97H31@98631B98D31D99231G99>31F99F31@99K30L9:630<9:E2ON9;42OA9;C2O49<22NG9<B2N99=12ML9=@2M?9=O2M29>>2LE9>M2L79?<2KJ9?L2K>9@@2KI9@I2KM9A22L09A;2L49AD2L89AM2L<9B62L@9B?2LC9BH2LG9C02LK9C62LO9C<2M39CC2M69CI2M99CO2M=9D?2MA9EC2ME9FH2MI9GL2ML9I12N19J52N59K:2N89LA2N79MI2N59O02N2:082MO:1@2ML:2H2MI:402MG:522N2:5I2MA:6?2M6:772LJ:7O2L?:8F2L3:9>2KG::62K;::H2K2:;;2JH:;M2J?:<@2J5:=22IL:=E2IB:>C2I>:?G2IB:@02J2:@22JE:@42K8:@62KK:B72LM:DA2N2:FL2O6:G=2O;:FE2N?:EN2MB:EG2MB:F22N;:F>2O5:FI2OA:G12NM:G92MO:GA2M3:G72M5:F?2MB:G02N5:GB2N=:I52I<:HM2EH:I32F1:I82F9:I>2FA:IE2FJ:IL2G3:J62G>:JA2GJ:JK2H7:JO2HN:K32ID:K82J;:K<2K1:L12KA:M12K::MI2J;:NM2IB;042HL;152HI;242HO;342I4;3I2I1;3O2HC;462H5;4<2GG;322F@;2B2EM;2M2EL;3;2EK;452EA;482E4;4?2DN;4I2DJ;542DH;5?2DE;5J2DB;672D=;6D2D8;722D2;7@2CL;7N2CF;8A2C7;942BJ;9G2B;;::2AM;:L2A?;;@2A1;;J2@J;<22@?;<82@9;<>2@4;<D2?O;<J2?I;=02?D;=62??;=A2>=;>12==;>A2<>;>O2<4;?;2;M;?H2;E;@52;?;@B2;7;A02:L;A>2:@;AM2:5;B<29I;BJ29=;C>290;D428D;DH287;E7281;ED27K;F527C;F=26K;GJ26I;I2268;J925C;K4256;KK24J;LB24=;M<247;N5244;NN23O;OG23K<0@23G<1:23B<2323<<2L236<4122K<5622@<67228<77220<8721I<9621B<:621;<;4215<;N20N<<@20A<=9209<>2200<>L1OH<?E1O@<@>1O7<A81NO<B31NI<BH1ND<C=1ND<D11NF<DF1NG<E41NL<EG1N><F:1N0<FM1MB<G@1M4<H31LE<HF1L8<I81KM<IH1KH<J71K?<LI1J@<LN1J@<M31J?<MK1IG<O61I3=0<1HO=1C1HK=2J1HG=401HC=571H?=6=1H<=711GN=741G8=771FA=7:1EK=7>1E5=7A1D?=7D1CH=7G1C2=7J1B<=7N1AF=811A1=841@:=871?D=8:1>N=8=1>8=8A1=B=8D1<K=8G1<5=8J1;?=8M1:I=911:2=9419<=9818F=:@18>=<1189==A183=?017M=@@17H=B017B=C@17=
2107;H31>7<231@7<<31C7<E31F7<O31H7=831J7=C3247=K32;7>432B7>=32I7@B33K7@833N7@733O7@53417@43437@73457@>3457@E3467@K3477A23477A934;7A@34>7AH34A7AO34C7BF3507C?35>7CO35J7D33627D636;7D936C7D>36L7DL3737E:37<7EH37D7F637K7FD3847G238;7G@38D7GN38K7H73917HE3967HO3977I93997IC39;7IL39=7J639>7J@39@7JK39A7K439D7K>39E7KH39G7L239H7L<39J7L?39G7L=39@7L<3:27M93;I7NC3<?8073<281K3;D83>3;684N3:J8583:M85=3:L8553:@84N3:484F39G84>39<84739083O38D83H38884038<84@38I85139585A39A86239M86=3:486F3:786O3:98773:;87E3:<8883:;88J3::8953:689@39H89J38G8:537E8:A3728;03738;?3748;N3758<:3788<F37<8=237?8=>37C8=I37H8>537K8>A37O8>M3838?93878?D38;8?O3898@:3878@E3858A33828AE37O8B837K8B?37D8BD37<8BJ3758C036M8C636F8C>3708CH37=8D137J8D:3878DC38D8DK38K8E138O8E83938E?3968EF3928EK38G8EL3888EN37K8F037<8F236O8F336@8F53638F735D8F93568F;34H8F=34;8FK3468G:3418GI33M8H<33K8I933N8J73418K53458L33498M134=8MO34A8NM34D8OJ34G90@34J91434791I33K92=33?93033393D32H94732<94K32095>31D96131:96?31=96M31?97;31B97I31D98731F98D31I99331K99@31M99N31O9:;31E9:J3189;930J9;H30=9<63009<D2OC9=12O59=@2NH9=N2N;9>;2MN9>I2M@9?72M39?E2LF9@32L89@C2L09A02KO9A92L39AB2L79AL2L:9B52L>9B>2LB9BG2LF9C02LI9C92LM9C?2M19CF2M49CL2M89D32M<9D:2M?9E>2MC9FC2MG9GG2MK9HL2MO9J02N29JH2N89K?2N?9LA2N@9MI2N?9O12N=:092N;:1B2N::2I2N8:422N7:592N5:5M2MM:6?2MD:722M::7D2M1:872LG:8J2L>:9<2L4:9O2KK::B2KA:;42K8:;G2JN:<92JD:<H2J>:=O2J;:??2J9:@<2JA:A42JK:A62K=:A82L0:C;2M3:EA2N8:FA2OF:G52O6:F92N;:EB2M?:EF2N=:F22O6:F=300:FG2OK:FO2O8:G72N<:G?2M?:GG2LC:G82MF:FG2NN:GN2ND:H62JD:HE2FD:I82E=:I>2EE:ID2EN:IJ2F6:J02F>:J;2FJ:JE2G5:JL2GH:K12H>:K52I5:K:2IK:K>2JB:KC2K9:L>2K4:ME2J>:NL2IG;032I0;132I1;222I6;322I<;392HN;3?2H?;3F2H2;3L2GC;432G6;3?2F;;392EN;3O2EM;442E?;4;2E9;4B2E2;4M2E0;582DM;5C2DJ;5N2DH;692DE;6D2DB;722D<;7@2D6;7M2D0;8?2CD;922C6;9E2BH;:82B:;:J2AL;;>2A?;<02A8;<=2A1;<C2@K;<H2@E;<N2@@;=42@:;=:2@4;=@2?N;=K2??;>;2>@;>K2=@;?:2<L;?G2<E;@42<>;@A2<6;@O2;K;A=2;?;AL2;3;B;2:H;BI2:<;C72:0;CJ29D;D?298;E428M;ED28E;F028A;F=28<;G0283;H327A;I:26L;J@268;K?25I;L625<;LN253;MG24O;NA24J;O:24F<0324B<0L24=<1F249<2?245<3823O<4;23E<5<23=<6<236<7;22N<8;22H<9:22@<::229<;:222<<321I<<L21A<=D218<>=20O<?520F<?N20><@F205<A?1OM<B61OL<BL200<CB205<D7209<DL20:<E620F<EI209<F=1OH<G01O7<GD1NF<H71N4<HJ1MD<I=1M2<J11LB<JK1L3<LK1K:<N01JG<N71JB<NC1JB<OA1IJ=0I1II=1M1II=341IE=4;1IA=5A1I>=6H1I:=6M1HE=701GO=741G9=771FB=7:1EL=7=1E6=7@1D@=7D1CI=7G1C3=7J1B==7M1AG=801A1=841@;=871?E=8:1>O=8=1>9=8@1=B=8D1<L=8G1<6=8J1;@=8M1:J=901:3=9919?=:I199=<9194==I18N=?818I=@I18C=B918==CH188=E8182=FH17M
2107;I31I7<231L7<<31N7<E3207<O3237=93267=C3297=M32A7>532H7>=32O7@633K7?L33O7?K3417?J3437?N3457@334:7@:34=7@A34>7@G34>7@N34?7A534B7A<34D7AC34G7B93537C135A7CJ35O7D03687D436@7D736I7D;3727D>37:7DK37B7E937J7EG3827F538:7FC38B7G138J7G?3927GM39:7H939E7HD39?7HN39@7I839A7IB39C7IL39D7J639G7J@39H7JJ39J7K439K7K>39M7KG39N7L13:07L;3:27L;39M7L939H7LA3;97MC3<F7O@3<D81;3<A82O3<384C3;F85E3;?85K3;>85C3;285;3:F8543::84L39M84D39A84=39684538J83N38>84238>84C38K85339785C39D8643:086D3:<86O3:B8783:D87@3:F87O3:G88A3:F8933:F89A3:C89L39F8:738E8:A37C8:M36N8;93728;D3768<03798<<37=8<H37A8=437E8=@37H8=K37L8>73808>C3848>O3888?;38<8?F38@8@138?8@<38=8@G38;8A93888AL3848B>3808BH37J8BN37B8C437;8C:3738C?36N8CI37;8D237H8D;3858DD38A8DM38N8E739;8E?39E8EF39H8EM39L8F339B8F:3998F?38N8FA38@8FC3828FE37D8FF3768FH36G8FJ3698FL35K8FN35=8G53538GD34O8H234J8HA34F8I034A8IB34>8J?34B8K=34F8L;34I8M934M8N73518O53549023588OO35790L34D91I34092=33D93133893D32L94732@94K32595?31I96131?96?31A96N31C97;31F97I31H98731J98E31L99331O99A32199I3299:932<9:?3219:J31E9;83189;F30J9<430=9<A3009<O2OB9==2O59=K2NH9>92N;9>G2MM9?52M@9?C2M39@22LI9@C2LF9A42LB9AD2L>9B42L<9B=2L@9BF2LC9BO2LG9C82LK9CA2LO9CI2M39CO2M69D62M:9D=2M>9E92MB9F=2MF9GB2MI9HE2MM9I<2N39J32N99JK2N@9KB2NF9LA2NH9MI2NG9O12NE:092ND:1A2NB:2I2NA:412N?:542NE:5@2NE:5F2N=:692N3:6L2MJ:7>2M@:812M6:8D2LM:962LC:9I2L:::<2L0::N2KG:;=2K@:;L2K9:=<2K7:>M2K5:@42K7:A12K@:AN2KG:B:2L6:CO2M=:DO2NM:EI2O=:F=2NM:F22N7:E<2ML:EF2O7:F1301:F<30G:FE304:FN2OB:G62NI:G>2ML:G>2MF:FN2NL:FJ2NI:F12K::GL2L9:H;2GH:HM2FA:I@2E;:II2E::IO2EB:J52EK:J?2F6:JI2FB:JN2G9:K22GO:K72HF:K;2I<:KD2J;:KM2K::L=2K9:MD2JB:NK2IK;012I5;112I9;202I>;2B2I8;2I2HJ;2O2H<;362GN;3<2G@;3C2G2;3I2FD;3L2F4;412EK;472EE;4>2E>;4F2E8;512E5;5<2E2;5G2E0;612DM;6<2DK;6G2DH;732DE;7@2D@;7M2D:;8<2D2;8O2CC;9C2C6;:52BG;:H2B=;;;2B6;;N2AN;<@2AG;<M2AA;=32A;;=92A4;=?2@N;=G2@F;>22@7;>=2?H;>H2?9;?52>B;?F2=E;@22==;@?2=5;@N2<J;A<2<>;AK2<2;B92;G;BH2;;;C62:O;CD2:C;D52:7;DJ29L;E>29A;F5297;FH28M;G<28D;GO28:;HC281;IB27A;JI26L;KJ26;;LA25O;M:25I;N325F;NL25A;OF25<<0?259<19254<22250<2K24L<3E24F<4C24=<5D244<6E23J<7E23A<8F238<9F22O<:G22E<;F22><<722@<<L228<=A221<>721H<?021@<?I217<@A20O<A:20F<B120E<BG20I<C=20N<D4212<DM211<E=216<F020J<FC209<G61OH<GJ1O7<H=1NF<I11N5<ID1MD<J81M3<K51LE<L51L9<MK1KC<NH1K3<NM1JJ<OI1JA=191J7=2D1J2=3A1J@=4F1J@=5L1J<=6F1J1=6I1I;=6M1HE=701GO=731G9=771FC=7:1EM=7=1E7=7@1DA=7C1CJ=7G1C4=7J1B>=7M1AH=801A1=831@;=871?E=8:1>O=8=1>9=8@1=C=8C1<M=8G1<7=8J1;A=8M1:J=9A1::=;11:5=<A19O=>119I=?A19D=A119>=BA199=D0193=EA18M=G118H=HA18C
2117<23267<<3297<F32<7<O32>7=932@7=C32B7=M32G7>632O7>>3357?J33L7?@3417?A3447?F3467?K3497@034>7@634B7@<34E7@C34E7@J34G7A134I7A834L7AK3567BD35D7C<3627CN36>7D136F7D536O7D83777D<37@7D?37H7DJ3817E83887EF38A7F438H7FB3917G03987G>39B7GJ39O7H73:<7HB3:77HM3:07I739J7IA39K7IK39M7J539N7J?3:07JI3:17K33:47K=3:57KG3:77L13:87L93:97L83:27L13:J7LE3<?7N<3<H8093<E8263<C8433<B85F3<58693<08613;D85J3;885A3:L85:3:@8523:484K39G84C39;84;39084438D84438@84E38L85539985E39E8663:186F3:>8763:J87A3;087J3;18883;288J3;289<3;189J3:L8:339M8:<38N8:E37O8:N37<8;:37<8;F37=8<237?8<>37B8<J37F8=637J8=B37N8=M3828>93858>E3898?138=8?=38A8?H38E8@438E8@>38C8@M38@8A?38=8B23898BE3858C23808C837H8C>37A8CC37:8CJ3788D337E8D<3828DE38>8DN38K8E83988EA39E8EJ3:28F33:>8F:3:<8FA3:38FH39I8FO39@8G43958G638G8G73888G937K8G;37<8G=36O8G?36@8GA3638GM35L8H<35G8HJ35C8I935?8IH35:8J73568JG3528KE3568LC35:8MA35=8N?35A8O<35D90;35H8OG35C90D34O91A34;92>33H93133<93D33194832E94K32995?31N96231C96@31E96N31G97<31I97J31L98731N98E32099432399<32:99@32E99O32I9:B32E9:G32:9:M3209;531E9;D3179<130J9<?30=9<M3009=;2OB9=I2O59>72NH9>E2N:9?32MM9?B2MB9@32M?9@C2M;9A42M79AE2M39B62M09BG2LL9C62LI9C?2LM9CH2M19D22M59D92M99D@2M<9E32M?9F72MC9G92MH9H02MN9HG2N59I>2N;9J52NA9JM2NG9KD2NM9LA2O09MI2NN9O12NM:092NK:1A2NJ:2I2NH:3M2NL:4N2O4:5H2O9:5C2O3:5@2NL:632NB:6F2N9:782MO:7K2MF:8>2M<:902M3:9C2LI::22LB::@2L;:;92L6:<I2L4:>92L2:?I2L0:@J2L7:AF2L>:BL2LD:C=2N2:D<2OA:E12O4:EE2ND:EN2N2:EK2MO:EF2OO:F130M:F;311:FC30>:FL2OL:G42O6:G<2N9:G32NL:FM2O1:FM2NB:FM2N2:GE2M@:H02HM:HC2GF:I62F?:II2E8:J42DO:J:2E7:JD2EA:JK2F3:JO2FI:K42G@:K<2H=:KE2I<:KN2J;:L72K::LF2K7:MB2JG:NJ2J0;002I;;0O2I@;1K2IB;222I5;282HF;2?2H9;2F2GJ;2L2G=;332FO;3;2FA;3H2F8;432F0;4:2EI;4A2EC;4I2E>;542E;;5?2E8;5J2E5;652E2;6@2E0;6K2DM;762DK;7A2DH;7M2DD;8:2D?;8M2D1;9@2CC;:32C:;:F2C3;;92BL;;K2BE;<>2B=;=12B6;=>2B0;=D2AJ;=N2A=;>92@N;>D2@>;>O2?O;?:2?@;?F2?1;@12>B;@>2>4;@L2=G;A92=9;AG2<K;B52<>;BB2;O;BO2;B;C=2;4;CI2:H;D72:<;DK2:1;E@29F;F729<;G4293;H028J;HM28A;IC287;J627N;K127A;L526N;LM26E;MF26@;N?26<;O9268<02263<0L25M<1G25F<2A25@<3;259<46253<5124H<6224?<73246<8423L<9423C<:523:<;6231<;N22N<<=234<=222N<=G22F<><22><?1228<?F220<@<21H<A521?<AL21><BC21@<C<21@<D621?<DO21?<EE21@<F621;<FJ20J<G>209<H11OH<HD1O7<I71NF<IK1N5<J?1ME<K?1M8<L?1LL<M?1L?<NL1KM<O@1K@<OG1K>=151K4=2L1JF=4>1J;=5<1JI=691K6=6C1JH=6F1J2=6I1I<=6M1HF=701H0=731G9=761FC=791EM=7=1E7=7@1DA=7C1CK=7F1C5=7I1B?=7M1AI=801A2=831@<=861?F=891?0=8=1>9=8@1=C=8C1<M=8F1<7=8I1;A=9I1;5=;:1;0=<I1:K=>91:E=?I1:?=A91:9=BI1:4=D919O=EI19I=G919C=HI19=
2117<332B7<<32D7<F32F7=032I7=932L7=C32N7=M3307>73357>?33<7?>33M7?:3447?>3467?B3497?H34<7?M34A7@334F7@934J7@?34M7@F34M7@M3517A>3597B735G7BO3657CH36C7CO36L7D33757D637=7D:37E7D=37O7D@3877DI38?7E738G7EE38O7F33977FA39?7FO39J7G;3:77GH3:D7H43;17H@3:M7HK3:G7I53:@7I@3::7IJ3:47J43:77J>3:87JH3::7K23:;7K<3:=7KF3:>7L03:@7L63:?7L33:B7KN3;N7M93<K7O63<H8123<G82O3<E84L3<B86@3<G86?3<58673;J8603;>85H3;285@3:F8583::85139M84I39A84A39584:38J84638A84G38N85739:85H39G8683:486H3:@8783:L87I3;98833;<88A3;=8933;=89?3;689K3:N8:63:>8:?39@8:H38A8;037O8;;37M8;G37N8<337O8<?37O8<L37O8=83808=D3838=O3878>;38;8>G38>8?338B8??38F8?J38J8@638K8@@38I8A338E8AF38B8B938>8BK38;8C<3878CB37O8CG37H8CM37@8D437B8D=37O8DF38;8E038H8E93958EB39B8EL39O8F53:<8FG3:E8G13:F8FO3:C8G63::8G=3:08GD39G8GI39<8GK38N8GM38@8GN3828H037D8H23768H636I8HE36E8I336@8IC36<8J13688J@3638JO35O8K=35J8KM35G8LK35J8MH35M8NG3618OD36590:3678O?35N90<35:91934G92634393133A93D33594832J94L32>95?32396231F96@31I96N31L97<31N97J32098832398F32598N32<99332G99633299E3379:<3359:J32N9;032D9;63299;<31O9;B31D9;O3179<=30J9<K30=9=92OO9=G2OB9>52O59>C2NH9?12N<9?B2N89@32N59@D2N19A52MM9AE2MI9B72MF9BG2MB9C82M>9CI2M:9D:2M69DC2M:9DN2M>9EL2MB9FD2MI9G;2MO9H22N59HI2N;9I@2NB9J72NH9JO2NN9KF2O49LA2O89MI2O79O12O5:092O4:1A2O2:2F2O4:3G2O<:4H2OD:5I2OK:5K2OG:5F2OA:5A2O;:5M2O1:6?2NH:722N?:7D2N5:872ML:8G2MD:952M=:9C2M6::F2M3:<62M0:=F2LO:?62LL:@E2LJ:AF2L7:B92KO:BM2ML:CG2O>:D<2NO:E02N@:EC2N6:EI2NH:EE30G:F031H:F931;:FA30H:FJ305:G32OC:G82NK:G02O9:G02NI:G02N9:G02MI:FN2M0:GG2JA:H92HI:HL2GC:I?2F=:J22E6:J@2DC:JH2DM:JM2ED:K42F@:K=2G?:KF2H>:L02I=:L92J;:LB2K::M02K4:MA2JK:NH2J5:ON2IB;0M2IG;1;2I?;1B2I1;1H2HC;1O2H5;252GG;2?2G;;2M2G1;3;2FH;3I2F>;462F4;4=2EN;4D2EH;4M2EC;582EA;5C2E>;5N2E;;692E8;6D2E5;6O2E3;7:2E0;7E2DM;802DK;8;2DH;8K2D?;9>2D8;:12D1;:C2CI;;82CA;;K2C9;<?2C0;=32BG;=D2B@;>12B2;>;2AB;>E2A3;?02@D;?:2@5;?D2?E;?O2?6;@92>F;@E2>7;A22=J;A?2=<;AM2<N;B;2<A;BH2<2;C52;E;CA2;8;CM2:L;D92:A;DM2:6;EA29K;F:29A;G6298;H228O;HO28F;IK28=;JH284;KD27K;L?27A;M927;;N3274;NM26N;OH26G<0B26A<1=26:<27264<3125M<3L25F<4F25><5A253<6@24J<7A24@<8B247<9B23N<:C23D<;D23;<<623><<E23E<=723D<=L23<<>B234<?722M<?L22E<@A22><A6226<AL21O<BE21N<C?21N<D821M<E121L<EK21L<F:21K<G021<<GD20K<H7209<HK1OI<I>1O7<J21NG<JI1N8<KJ1MK<LI1M><MI1M2<NI1LE<OL1L7=0:1L3=0H1L6=251KI=3C1K==501K1=5N1K9=6F1KF=6F1JO=6F1J3=6I1I==6L1HG=701H0=731G:=761FD=791EN=7<1E8=7@1DA=7C1CK=7F1C5=7I1B?=7L1AI=801A3=831@==861?G=891?1=8<1>:=8@1=D=8C1<N=8F1<8=:21<1=;B1;K==11;E=>A1;@=@21;;=AA1;4=C11:O=DA1:I=F11:D=GA1:?=I11:8
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
2117<33377<=33:7<F33<7=033>7=:33A7=D33D7=M33F7>733H7>A33K7>J3407?33487?734;7?;34?7?A34D7?H34H7?M34M7@33517@93567@C35=7A<35M7B536;7BM36I7CF3777CN37A7D137I7D53827D938:7D<38B7D?38K7DC3947DH39<7E539D7EC39M7F03::7F=3:F7FJ3;37G63;@7GC3;M7GO3<:7H;3<<7HF3<57I13;O7I;3;I7IF3;C7J13;<7J;3;67JG3:O7K13:K7K;3:M7KE3:O7KN3;07KL3;;7KB3<E7LJ3=37N73=<7OD3=E8103=O82=3>883I3>B8553>J85=3<785L3<086;3;I8643;=85M3;285D3:F85=3::85539M84N39A84F39584>38I84K39285;39>85L39J86<3:786L3:C87<3;087M3;<88<3;I88G3;H8943;B89A3;:89L3;38:83:K8:C3:B8:L39D8;43988;<38O8;H38O8<43908<@3918<M3918=93918=E3928>13938>=3938>J3948?63948?B3958?N3958@:3978@M3948A@3908B338M8BE38I8C938E8CK38B8D538;8D:3848D@37L8DI3868E238C8E;3908EE39=8EN39J8F73:78G03:=8H53:=8I:3:?8IF3:@8I>3:@8I63:A8HO3:A8HG3:B8HF3:>8HM3:58I239J8I439<8I638N8I838@8IF38;8J53878JD3828K237N8KA37I8L037E8L?37A8LN37<8M<3788MK3738N:36O8O63729043768O83728NN36F8OK36290I35>91F34J92C34793@33C94833394L32G95?32<96331O96A32196O32397=32697K32898432?98732J98;33598?33A98C33L99034199H3409:?33N9;533K9;;33A9;A3369;G32L9;M32A9<33279<931L9<>31B9<G3169=530J9=C30<9>12OO9>B2OK9?22OG9?D2OD9@42O@9@E2O<9A82O89B02O29BH2NM9C?2NH9D82NB9DC2N49E62N19EN2N59FG2N99G?2N=9H62NC9HN2NI9IE2NO9J<2O69K32O<9KJ2OB9LA2OH9MH2OG9O02OE:092OD:1:2OK:2;303:3;30;:4530D:4N30M:5>310:5?30J:5@30D:5B30=:5C307:5B300:612OH:6?2OA:6M2O;:7;2O4:7O2NM:9?2NK::M2N6:<82N6:=C2N>:>M2NE:@72NN:AB2O5:BG2O;:C72OD:CB2OA:D92OL:DE302:E1309:E=30B:E?327:E431O:E<31>:EK30J:FG304:G:2O?:G92NN:G82N<:G82ML:G72M::G62LI:G62L::GE2K2:H92IG:HN2H8:IC2FJ:J82E;:JL2E;:K?2EB:KI2FA:L22G@:L;2H?:LD2I>:LM2J=:M72K6:ME2JN:N52JG:NF2J=:OJ2J1;042IF;0;2I9;0H2HN;162HD;1D2H;;212H1;2@2GG;2M2G>;3;2G4;3I2FJ;462FA;4C2F8;4J2F1;542EN;5?2EK;5J2EI;652EF;6@2EC;6K2EA;782E>;7G2E;;852E8;8D2E5;932E2;9G2DI;:;2DA;:O2D8;;C2CO;<62CG;<J2C?;=22C<;=22C<;=22C<;=B2BK;>52B7;>G2AB;?92@N;?H2@;;@22?L;@=2?=;@G2>M;A12>>;A>2>0;AL2=B;B92=4;BF2<G;C22<:;C>2;N;CI2;B;D52;7;DA2:K;E02:?;ED2:5;F>29K;G:29B;H6299;I3290;IO28G;JL28>;L028F;M528O;N728:;O527N;OO27H<0I27A<1D27;<2>274<3826N<4226G<4M26?<5H265<6C25J<7>25@<8>256<9>24M<:?24C<;@24:<<8248<<G24?<=524F<=C24M<>724G<>M24@<?B248<@7240<@L23I<AA23A<B7239<BM231<CC22I<D<22H<E522H<EO22G<FA22I<G922;<H121M<HE21<<I820K<IL20:<J?1OJ<K>1O=<L>1O0<M=1ND<N=1N7<O:1MM=011MH=0I1MD=1@1M>=271M:=2N1M5=3B1M2=3O1M4=541LK=6A1L?=7>1LI=7@1L3=781JM=701IH=6L1HH=6O1H2=721G<=751FF=791F0=7<1E9=7?1DC=7B1CM=7E1C7=7I1BA=7L1AJ=7O1A4=821@>=851?H=891?2=8<1><=931=L=:B1=G=<21=B==B1=<=?21=6=@B1=0=B21<K=CB1<F=E21<@=FB1<:=H11<4=IB1;O
2117<433B7<=33E7<G33G7=033I7=:33L7=D33N7=N3417>73437>A3467>J3487>O34;7?234@7?634D7??34F7?E34J7?K34O7@03547@83597A035J7AH3697B@36J7C837:7CL37F7CO37O7D33887D638@7D:38H7D=3917DA39:7DD39B7DG39J7E43:47EA3:A7EM3:N7F:3;;7FG3;H7G43<57G@3<B7GM3<N7H93=37HD3<M7HO3<F7I93<@7ID3<97IO3<37J93;L7JD3;F7JO3;?7K:3;97KD3;67KL3;77KG3<97KL3=17L:3>07MB3>>7NN3>H80;3?181H3?:8343?D84@3?K84G3=18533;O85B3;H8613;A8633;785K3:L85C3:@85;3:485439G84L39;84D38O84M39485=39@85N39M86>3:986N3:E87>3;287M3;?8853;M88B3;G88O3;@89=3;:89J3;38:83:M8:D3:F8:N3:68;639M8;>39D8;H39A8<439A8<A39A8<M39B8=939C8=E39C8>239D8>>39D8>J39D8?739D8?D39A8@139@8@>39>8A43998AF3968B93928BL38N8C?38K8D238H8D>38B8DD38;8DJ3838E338@8E<38M8EF39:8EO39G8F83:48G43:88H:3::8I?3::8JD3:<8K03:=8JH3:=8JA3:=8J93:>8J13:?8IJ3:@8IB3:@8IB3:<8IH3:18II39D8IO3988J>3948JM3908K<38K8KK38G8L938B8LH38>8M73898MF3858N53818NC37L8O237H8OA37C90337E8NH37?8NF3718OC36=90A35J91>35692:34B93833N94533;94L32L95@32@96332496A32697032897=32:97F32A97J32L97N33898233C98633N98:34:98F34?99=34=9:534;9:L3499;=3449;C33J9;I33?9;O3359<532K9<;32@9<A3269<G31K9<M31A9=33169=A30I9>130E9>C30@9?;30;9@33069@K3019AC2OK9B;2OF9C32OA9CJ2O;9D62NM9DB2N>9E62N<9EN2NA9FG2NE9G?2NI9H72NM9I02O19IG2O79J>2O=9K52OC9KL2OI9LC2OO9MH2OO9O02ON:03303:1430::2030D:2I30M:3B316:4;31?:4N31C:4O31=:51317:52310:5330J:5630B:5A307:5K304:65301:6>2ON:7:2OI:8F2ON::32ON:;@2ON:<N2ON:>;2ON:?H2ON:A62ON:B@301:C3305:C=306:CG302:D<30<:DH30C:E430J:E3320:DI31I:DK31B:EJ30M:FF307:GC2OA:GD2NO:GC2N>:GB2MM:GA2M<:G>2LL:G72L=:GB2K;:H72IM:HL2H>:IA2G0:J62F3:JJ2F8:K>2F>:L32FC:L<2G@:LE2H?:LN2I?:M72J=:MA2K3:N12JK:N?2JD:NN2J;:O?2J1:OL2IH;0;2I>;0H2I4;162HK;1D2HA;212H7;2@2GM;2M2GD;3;2G:;3I2G0;472FG;4E2F>;4M2F7;582F4;5C2F1;5N2EN;692EK;6H2EH;772EF;7F2EC;842E@;8C2E=;922E:;9C2E5;:72DL;:J2DD;;>2D<;<22D3;<F2CJ;<J2CI;<J2CI;<J2CJ;<J2CJ;=:2C8;=M2BC;>?2AO;?12A;;?D2@F;@62@2;@D2?@;@N2?1;A82>B;AD2>3;B22=E;B?2=7;BK2<K;C62<?;CB2<4;CM2;H;D92;<;DE2;0;E12:E;EF2:9;F@2:0;G<29G;H829>;I5295;J128M;K6295;L;29>;M@29G;NA293;OE28;<0?284<1927N<2427H<2N27A<3I27;<4C274<5>26J<6926?<74265<8025K<8L25A<9L258<:M24N<;N24E<<@24I<<O250<==256<=K25=<>=25<<?2255<?G24N<@<24F<A124?<AG247<B=23O<C323G<CH23><D>237<E8235<F1234<FG235<G=22K<H522=<HK21N<I>21<<J220L<JH20<<KH1OO<LH1OC<MG1O6<NB1NN<O91NJ=001ND=0H1N@=1?1N;=261N7=2N1N1=3E1MM=4<1MH=531MD=5E1MB=6C1M<=7J1M9=7M1LE=7D1K@=7<1J:=741I5=6O1H2=721G<=751FF=781F0=7<1E:=7?1DD=7B1CN=7E1C8=7H1BA=7L1AK=7O1A5=821@?=851?I=881?2=9;1>H=:J1>B=<:1><==K1>7=?:1>2=@J1=L=B:1=F=CJ1=A=E:1=;=FJ1=6=H<1<K=IO1<>
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
2127<>34E7<H34G7=134J7=;34M7=E34O7=N3517>83547>B3577>C35>7>A35G7>@3517>D3557>M3567?73577?C35C7@535N7@H36@7A;3707AO37B7BB3827C538C7C?38O7CB3977CE39A7CI39I7CL3:17CO3:<7D33:J7D53;87D83;F7D@3;J7E23;E7E?3<27EL3<>7F83<K7FE3=77G43=D7H@3>67I23>67I33=L7I;3=D7IE3=?7J03=97J:3=47JD3<N7JO3<H7K;3<E7KJ3<G7L83<H7LG3<J7M63<K7M?3=97ME3>17N43?07NG3@47O;3A87ON3B;8153BJ82?3BN82F3@382M3=98373;D83G3;=8453;684D3;08533:I85B3:C85F3:985>39M85739A85339985C39F8643:286D3:>86K3:M8723;<87;3;K87I3;M8893;I88H3;D8983;@89G3;<8:73;78:G3;38;63:O8;C3:K8;I3:J8;O3:G8<93:I8<G3:E8=43:B8=A3:@8=N3:>8>;3:<8>H3::8?53:88?B3:68?O3:48@<3:28@I3:08A>39L8B339G8BH39C8C>39?8D239:8DE3978E83948EA38M8EI3918F239>8F=39J8GB39K8HG39M8IM39M8J639L8J:39J8J?39H8J:39I8J539J8IO39J8II39K8ID39L8I>39M8I839M8I>3:98IM3:K8J;3;<8JN3;?8KB3;:8KN3;68L93;38LD3;08LN3:L8M93:H8MC3:D8MN3:@8N93:=8ND3::8NN3:68O93:28OH39L90I39:91J38J91K38@92937K92G37693536@93C35K94135694>34A94M33L95;33796432B96>32E96B33196E33<96I33H96N34397234>97534J97935597=35@97G35H98>35F99635C99M35B9:D35@9;<35>9<035:9<63509<<34E9<B34;9<H3419<N33F9=533=9=B3389=M3359>D32L9?A32?9@<3269A43219AL31L9BD31F9C03189C;30I9CJ30D9D<30B9DK30>9E830;9F030>9FI30A9GA30E9H930H9I130K9II30N9JA3129K:3159L23189LJ31;9M<3109ML31=9ND31F9O=31O:06329:0O32B:1I32K:2A334:3:33<:3@338:3A332:3D32K:3O329:4:31H:4E316:5030L:5:30I:5C30G:6G30F:8430F:9A30F::O30G:<<30G:=H30G:?130M:@9311:AB316:BH31;:C331=:C=31@:CG31C:D131F:D;31I:DO31J:ED31N:EJ31D:E=30G:F:2O8:G62MO:GC2NH:GG2NN:G@2N@:G:2N1:G32MB:FM2M3:G;2KM:H02J?:HE2I0:I92HH:IN2HL:JB2I2:K62I7:KK2I<:L?2IB:M32IG:MG2IK:N62J>:NB2JJ:NK2JE:O22JC:O82J@:OB2J:;002J1;0>2IH;0L2I>;192I5;1G2HL;252HC;2D2H9;332H1;3B2GH;412G@;4@2G7;4K2G8;572G9;5L2FO;6B2FD;742F>;7C2F;;822F8;8@2F5;8O2F2;9>2EO;9L2EL;:=2EG;;12E>;;E2E6;<02E2;<02E2;<02E2;<02E2;<02E3;<02E4;<02E4;<12E2;<C2D?;=52CJ;=H2C6;>:2BA;>L2AM;?>2A8;@12@D;@C2?O;A52?;;B02>H;B>2><;BJ2=M;C62==;CB2<O;CN2<C;D:2<8;DE2;K;E12;@;E<2;4;EJ2:I;FE2:?;GC2:9;HH2:B;IM2:K;K22;4;L72;=;M=2;F;N?2<1;O@2;?<0D2:F<1H29M<2K295<3F28N<4@28H<5:28A<66287<7027L<7K27A<8G278<9B26M<:=26B<;8269<<825O<<J263<=826:<=F26@<>526G<>C26N<?2275<?B276<@726N<@M26G<AC26?<B8267<BN25O<CD25G<D:25?<DO256<EE24O<F;24F<G124><GI249<H@23K<I823=<J122O<JF22@<KB227<L:222<M121M<MH21H<N@21D<O721?<ON218=0F212=1>20L=2620F=2N20?=3E209=4=203=551OM=5L1OF=6D1O@=7<1O:=821O4=8@1O7=8>1N;=8=1M>=8<1LB=8;1KF=891JJ=821IE=7I1H@=7A1G:=791F4=7;1E<=7>1DF=7A1D0=7D1C9=7G1BC=7K1AM=8C1A?=:31A:=;C1A4==31@N=>C1@I=@61@>=AI1@1=C<1?D=E01?7=FC1>J=H61>==II1>1=K=1=D=M01=7
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
2067:M3@?7;73@B7;93@I7;83A27;63A;7;63AE7;A3B57;K3BD7<<3C47=23CC7=I3D27>43D:7>@3DC7>K3DK7?73E37?B3E;0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007GK3I47H83I87HE3I<7I23IA7I23IC7HO3IC7HN3IC7I13I87I53HN7I83HC7I;3H:7I?3GO7IB3GD7IF3G;7II3G07IL3FF7J03F;7J33F27J73EG7J:3E<7J=3E37JE3DJ7K43DF7K:3E07L23E97LK3EB7ME3EL7N=3F57NK3F87O63F;7O@3F>7OK3FA8053FD80@3FF80K3FK8193G281G3G:8243GA82B3GI8303H183=3H883K3H@8493HG84G3HO8543I685B3I<8613IC86=3II86?3IN86A3J28753J<87J3JF88?3K18933K;89H3KD8:93KG8;73K;8<=3JK8=B3J;8>A3IO8>M3IN8?93IN8?C3J28?K3J;8@33JC8@;3JL8@C3K58@E3K28@D3JJ8@L3JH8A83JK8AC3JL8AO3JG8B;3JB8BG3J=8C43J98CB3J38D:3IL8E33IE8EL3I>8F>3IO8G03J@8GC3K28H53KB8HG3L48I93LE8IK3M78J=3MG8JO3N98KB3NJ8L03O98L>3OG8M53OC8MM3O?8NE3O<8O<3O89043O590M3O291E3O192>3NO9363NL93K3NK9423NL94<3NI94B3ND94E3NB94G3N@94I3N>94M3N<9533N995:3N795@3N595G3N395M3N09633MN96:3ML96E3MC97>3M:98=3ME99<3N09:;3N;9;;3NF9<:3NM9=93O19>:3O59>N3O;9>F3O;9>=3N:9>?3M59>:3M79@63NF9>K3NG9>G3N39>C3M?9>?3LK9><3L79>A3L49?03LA9?G3L@9@B3L79A<3KN9B73KD9C23K;9CM3K29DH3JH9EB3J?9EB3IL9E53I39DM3H@9DN3H?9E03H=9E23H:9ED3H>9F?3HD9G93HJ9GI3HL9H93HM9HI3HM9I93HO9IG3I09II3I39IJ3I69IL3I89IM3I;9J93I89K23HO9KJ3HE9LB3H=9M23H89MA3H39N03GM9N?3GG9NM3G=9O=3G09OL3FD:0:3F7:0I3EJ:173E<:1F3DM:243D@:2C3D2:313CD:3@3C6:3N3BH:4>3BA:4N3B;:5>3B5:6<3B6:7;3B7:893B8:983B8:9<3B0:9?3AG:9C3A?:9I3A5::33@L::=3@C::G3@;::G3@9::J3@9:;43?L:;93?5:;?3>=:;D3=F:;J3<N:<33<C:<=3<::<E3;O:<@3;9:<;3:D:<539N:<0398:;J38B:<:37E:<M36F:=@35G:>B35::?E350:@I34E:AL34;:BM340:D:33@:E<32O:F=32>:G?31L:I@30?:I230D:I;316:KG30G:MA30H:NL310;07318;1B31@;2M31H;44322;5632=;6932I;7<334;8>33?;9A33K;:D345;;K346;=A348;=932?;=8321;=:326;=:31H;=1320;>D320;@7320;A<32B;B:335;C>339;D434D;DH35O;E936=;EI36F;F936O;EH37?;E737O;DD387;D@37B;D736B;CL37E;DK38=;EJ37O;E935G;E9354;E934?;F134:;FO347;GL344;HJ341;IG33N;JD33K;KA33H;L733:;LH333;LN33>;M333K;M:33D;MB335;MI32E;N1326;N831G;O130=;OE2N;<0;2M0<172M3<242M6<302M9<3F2NA<3:2N5<442N?<562O5<5?2OH<6;2OA<732O;<7J2O4<8@2NN<962NG<9L2N@<:C2N9<;92N3<;O2ML<<E2MF<=;2M?<>12M9<>H2M2<?>2LK<@42LE<@J2L><AA2L7<B72L1<DE2JC<E<2K0<F52K7<FM2K=<GF2KD<H>2KJ<I62L0<IO2L7<JG2L<<K@2L:<L42L4<L=2KJ<M62K?<N02K3<NC2JG<O02J8<O42IM<O72ID<O92I;<O<2I2<O>2HI<OK2GF=0;2FA=162FE=362G0=5F2G==4O2G@=4N2G3=5E2F7=6;2E==722DB=7I2CF=8@2BK=992BB=:32B==:M2B9=;G2B4=<A2B0==:2AK=>52AF=>N2AA=?L2A2=@M2@2=B02?0=C22>0=D42<O=E62;N=F72:G=G;29@=HB297=II28M=K028D=L728:=M>281=N:278=O0262=OG24K>0=23D>1322=>1J216>2D209>3J205>4N200>631OK>781OF>8>1OA>9C1O>>:H1O?>;M1O@>=01NH>=O1MG>?01LE>@01KC>A21JA>B21I@>C41H>>D41G<>E51F:>F61E8>G71D7>H81C5>I91B3>J:1A1>K;1@0>LC1?>>ML1>L>NL1><>OL1=L?0L1=;?1L1<K?2M1<;?3L1;K?4L1;;?5M1:K?6M1:;?7M19J?8M19:?9M18J
2077;63@M7;53A67;43A?7;33AH7;83B57;B3BD7;L3C47<A3CD7=83D37=J3D?7>53DG7>A3DO7>L3E87?83E@7?D3EH0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007H93IC7HF3IH7I33IL7I23IN7I03IO7HM3J07I13IF7I43I;7I73I17I;3HF7I>3H=7IB3H27IE3GG7IH3G>7IL3G37IO3FI7J33F>7J63F47J93EJ7J=3E?7JH3E97K53E77K;3EA7L43EJ7LM3F37MF3F<7N?3FE7O53FN7O@3G07OJ3G38053G680?3G880J3G;8143G>81B3GE8203GM82=3H482K3H<8393HC83G3HK8443I384B3I:8503IB85=3II85L3IO86:3J686H3J=86K3JB86L3JF8783JM87M3K788A3KA8963KK89J3L58:<3L:8;33L48<83KD8=>3K48>@3JE8>L3JE8?73JE8?C3JE8?K3JM8@33K68@;3K?8@C3KH8@E3KD8@E3K>8@J3K:8A53K<8AA3K>8AM3K88B93K48BE3JO8C23JJ8CE3JD8D=3J<8E63J58EN3J08F@3JB8G23K28GE3KD8H73L58HI3LF8I;3M78IN3MI8J@3N:8K23NK8KA3O;8KJ3OH8L34058LI40A8MA40=8N94088O04058OG40190?3OM9173OI91N3OE92F3OA93>3O=93O3O99453O794<3O594B3O294I3O094O3NN9563NL95<3NI95B3NF95I3ND95O3NB9663N?96<3N=96B3N;96M3N497C3M>98>3M;99=3MF9:<3N19;;3N<9<:3NH9=93O39>83O>9?73OI9?93O=9>@3N49=F3LK9>73LM9?J3N89>J3NO9>G3N<9>C3MG9>?3M49>;3L@9>H3LH9?;3LN9@63LE9A03L<9AK3L39BF3KJ9C@3KA9D<3K89E23K59ED3JM9EA3J69E83I<9E=3I;9E>3I99E@3I79EB3I59F23I89FK3I=9G;3I>9GK3I@9H;3I@9HL3IA9I<3IC9IL3IC9J:3IE9J<3IG9J=3IJ9J?3IM9JH3IL9K@3IB9L93I99LK3HM9M:3HB9MH3H<9N73H79NF3H19O63GH9OE3G<:033G0:0B3FC:113F7:1?3EM:1M3EB:2=3E7:2M3DL:3<3D?:3K3D0:493CC:4I3C::593C4:5I3BN:6:3BI:783BJ:863BJ:8I3BG:8L3B>:903B5:9;3AL:9E3AC:9O3A:::93A1:::3@O:::3@O::B3@O::J3@>:;03?F:;53>N:;;3>7:;@3=?:;F3<H:;N3<;:<63;O:<03;9:;K3:D:;F39N:;A398:;;38B:;837K:;K36L:<L36?:=O365:?535J:@=35@:AF354:BJ34G:CG345:DE33D:EB332:F@32@:G=31O:H;31=:J1313:L630M:MA314:NL31=;0731E;1B31L;2L325;3O32@;5132L;64337;7733B;8:33K;9A33L;:G33M;;N33N;=C342;=;329;=430@;<7316;<032:;=B32<;?532;;@H32;;A=33B;B534A;C:34E;D>34J;EB34N;F7367;FI36G;FO372;F;37?;E:37@;EA37J;E;36O;DO36=;DA37B;DL38?;F@391;EL36@;EL35K;EM357;F334G;G034D;GM34A;HK34>;IH34;;JE347;K?33O;L433A;LG335;LM33A;M333M;M9348;M?34;;MF33K;MN33<;N832L;O6310;OI2NO<0>2M9<152N6<1J2OA<2>30L<2230?<1@2OH<0N2O1<262OE<402ON<5L2O<<6C2O9<7:2O7<812O5<8H2O1<9?2NO<:62NI<:L2NB<;B2N<<<82N5<<N2MO<=D2MI<>:2MB<?12M;<?G2M4<@=2LN<A42LG<BC2KJ<D:2L3<E=2L><F52LE<FM2LL<GF2M1<H?2M0<I92LM<J22LL<JK2LJ<KC2LG<L12L=<L?2L2<LF2KH<LL2K><M12K4<NG2JN<NA2J;<ND2J2<NG2II<NJ2IA<OD2IA=052HA=0C2G:=2H2G5=592GB=7J2GO=6E2H0=572GK=5D2G4=6:2F9=712E>=7H2DB=8?2CG=972C3=:02BN=:J2BJ=;D2BD=<>2B@==82B;=>22B7=>L2B2=?M2A4=@O2@2=B22?2=C32>1=D62=1=E72<0=F92:J=G=29G=HD29==IK293=K228J=L928A=M@287=N@27E=O826>>00257>0F240>1=22J>2321B>2J20?>3O209>54205>6:206>7?206>8E207>9J208>:O209><4209>=71O@>>81N<>?91M9>@61L9>A61K8>B71J6>C81I5>D81H2>E:1G1>F:1EO>G<1DN>H<1CK>I=1BI>J>1AH>KB1A4>M01@G>N=1@2>O=1?B?0>1?2?1=1>B?2>1>2?3>1=A?4>1=1?5>1<A?6>1<1?7>1;A?8>1;1?9>1:@?:>1:0
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
2647M03G47MI3G>7NB3GG7O;3H18043H:80?3H=80I3H?8133HB81>3HE81I3HG8243HL82B3I382O3I;83=3IC83K3IJ8493J284F3J98543JA85B3JH8603JO86>3K686L3K<87:3KC87D3KI87E3KM87F3L18823L988F3LC89;3LM89O3M78:D3M@8;33M?8<03M58=53LE8>;3L48>I3L38?43L28?@3L18?K3L28@33L;8@;3LD8@C3LM8@F3LL8@F3LE8@F3L>8A03L?8A<3LA8AH3L;8B43L78B@3L18C23KL8CJ3KD8DC3K>8E<3K68F23K58FE3KF8G73L78GI3LI8H;3M:8HM3MK8I?3N<8J23NM8JC3O?8JL3OK8K54098K>40F8KF4148L041A8LH4278M@4238N74208NO41K8OF41G90>41D91D40291F3OH92;3OE93@3OL94=40294K40295440195:3ON95@3OL95G3OJ95M3OH9643OE96:3OC96@3O@96G3O>96N3O<9743O997=3O49833N>98H3MH99?3M39:>3M>9;=3MI9<73N79<J3NI9==3O;9>03OL9>C40>9>03O=9=63N49=;3MF9=K3MA9>L3N59>F3NL9>C3N99>?3ME9>F3MD9?=3MC9@83M:9A33M09AN3LG9BE3LD9C;3LA9D03L>9DG3L<9E63L29E?3KA9E<3JJ9E73J69E>3J99EE3J:9EK3J<9F23J>9F@3J>9G03JA9G@3JD9H03JG9H@3JH9I03JI9I@3JJ9J03JK9JA3JL9K13JM9K@3JO9KA3K19KE3K39L=3JI9LO3J>9M=3J09MK3IC9N93I69NG3HJ9O73HC9OE3H>:043H4:0C3GG:123G;:1?3G1:1K3FH:283F>:2D3F5:323EK:3C3E?:463E3:4I3DH:5<3D@:5M3D<:6=3D9:6M3D5:7>3D1:7J3CK:843CC:8>3C::8H3C1:923BG:9<3B>:9@3B;:9?3B::9?3B8:9C3B8::23B7::83A?::=3@H::C3@0::H3?9::N3>A:;33=I:;;3=5:;E3<>:;E3;L:;B3;9:;?3:F:;<3:2:;939>:;738K:;M38=:<L382:>537F:?>37<:@F371:AO36F:C4369:D235G:DO356:EM34D:FJ342:GE33G:H@33<:IL332:K=32G:M032=:NL326;0732=;1?32F;2B332;3G338;4M338;6433:;7;33;;8A33;;9H33<;:N33=;<533>;=<33>;=@31M;<J31O;<L32:;=H323;>C331;@4332;@I34;;A>35B;B336I;BN37>;CA378;D7372;DO36K;EG36E;FB36G;G136I;G>37=;G836C;G135I;F@363;EJ37;;F5388;FC394;G5382;G437<;G436G;G5363;G535?;H035:;HM358;IJ355;JC34L;K934=;KN33N;LC33?;LM33E;M3340;M834<;M>34H;MC354;MI35?;N;34:;N433;;NI340;O=34J<0035C<0C34L<1533O<1H332<2;325<2N317<3731=<2H318<2M301<4N2N2<612MI<6H2MF<7?2MD<862MA<8M2M?<9D2M<<:;2M9<;22M7<;I2M4<<@2M2<=72LO<=N2LL<>E2LJ<?M2LN<A12MB<B>2N4<D62N;<EC2N<<F<2N5<G52MO<GO2ML<HH2MK<IB2MI<J;2MG<JO2MB<K=2M7<KK2LL<L82LB<LE2L7<M12KM<ME2KL<N82KK<NJ2KJ<O22K9<OM2K6=112K9=1O2K7=372JN=4F2J?=642J0=7B2IB=942HI=8O2HK=762HI=6E2H<=6K2GI=712G5=7F2F;=8=2E@=942DE=9K2CO=:E2CJ=;?2CF=<92CA==32C<==M2C7=>O2B7=@12A6=A32@5=B52?5=C72>4=D92=3=E;2<2=F=2:O=GA2:3=HI29J=J029@=K7297=L828M=M:28C=N;289=O5277=OM260>0F24K>1C23J>2?22J>2D21O>4=21F>5B21G>6H21H>7N21H>9321I>:821J>;=21K><C21L>=D20N>>E1OK>?G1NG>@H1ME>AI1LB>BJ1K?>CK1J;>DD1I>>EB1H>>FC1G<>GD1F:>HE1E8>I?1D>>JN1D2>L<1CF>MK1BO>O@1B??0@1AN?1@1A>?2@1@N?3@1@>?4@1?N?5@1?>?6@1>N?7A1>=?8@1=M?9A1==?:A1<M?;A1<=
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
28487L3N588:3N<88H3NC8963NI89D3NO89E3O489F3O889G3O<89I3OA8:<3OJ8;14058;E40>8<440=8<D40=8=E4008>A3OE8>M3OE8?93OD8?D3OC8@03OB8@;3OB8@D3OJ8@J3ON8@J3OG8@J3O?8@I3O88A03O78A<3O28AN3NL8BF3NF8C?3N>8D83N78E03N08EI3MI8F=3N18G03NC8GB3O38H43OE8HA4048HJ40A8I240N8I;41;8IE41I8IM4268J642C8J@42N8JM4358K:43<8KF43C8L543I8LO43I8M@4378MD42=8M04208N140=8O340B90940J91>41192C41993H41A94M41H95B41K96041L96=41M96J41M97841N97E41O97L41M98241J98941H98?41F98E41D99;40O99N40899<3O498M3N49:93O89:<3O@9:?3OI9:B4029:E40:9;640K9;I41=9;N41;9;N40H9<=40D9<M40>9>640G9?D4149?M40A9@23OM9@H3OK9A=3OH9B43OE9BJ3OC9C?3OA9D03O89D93NG9DB3N69DK3MD9E33M39E03L;9E73L=9E>3L>9EI3L>9F63L=9FC3L<9G03L:9G<3L:9GK3L<9H;3L?9HL3LB9I<3LE9IL3LH9J<3LK9JM3LN9K=3M19KI3LN9L53LJ9L@3LD9LO3L>9M=3L69ML3KN9N;3KC9NJ3K79O:3JK9OI3J?:0:3J::0L3J6:1C3IN:283IF:2N3I=:3D3I5:3L3HM:3H3HF:4;3H::4N3GN:5A3GB:653G6:6H3FK:773FE:7<3F>:7@3F8:7B3F2:7D3EL:7G3EG:7I3EB:7K3E<:7K3E::7I3E::7G3E::7G3E::873E::8I3E6:973DO:9<3DK:9B3DF:9K3D8::53CH::>3C9::G3BJ:;03B::;F3?>:;N3>O:<33>B:<83>4:<>3=E:<@3=8:=83<K:>73<?:?53<3:@33;G:A23;;:B03;0:BO3:C:CL3:8:DG39O:EB39G:F=39;:G8391:H438E:HO38::J737O:KA37C:KL36O:L536::L?35E:M735::N?35::OG35:;10359;28359;3@358;4L35:;6G35I;8C367;:?36F;<:374;=I36I;?836=;@>369;@K36;;A@379;AD373;AH36M;AO36H;BD36J;C:36L;CO36N;DD36O;E:371;EO373;FH36N;GA36K;GN36A;F@379;G7375;GO372;H@372;HF380;IM398;K@39O;LH38G;LK38=;L@38@;L038B;L938;;LA380;LJ37E;LE37K;L=384;L438=;L839:;LK39D;N6390;N>38;;NI37E;O536O;OA36:;OL35D<0?35C<1835N<2036:<2I36F<3B36=<4;357<5133N<5=32A<5C32D<5I32K<5N332<6K330<8832I<9D32A<9@30H<6<2M2<672K<<7B2K2<8N2JK<:;2JN<;H2K4<=52K9<>C2K?<@02KD<A>2KJ<B02M7<BB2NE<C4304<CF31C<E0328<F6326<G031N<GJ31F<H@31=<HN310<I=30E<J030?<JC30:<K4304<KB2ON<L02OG<L?2OA<M52NN<ML2N;<ND2MJ<O@2M<=0<2LL=182L?=1D2LI=202M2=2=2M<=3:2LB=562JL=6I2IN=7N2J6=932J==:72JD=;72JK=:12JC=:72J0=:=2I==:B2HI=:H2H6=:N2GC=;42G0=;:2F==<72E<==@2D8=>H2C3=@12AN=A92@I=BB2?D=C<2>N=CN2>:=DA2=F=E32=2=FB2<A=H72<G=HH2<4=I>2;B=J72;2=K12:A=KJ2:0=LC29@=M=28O>1;2:3>27292>26288>1D27A>1H26N>3A26E>5;26<>74262>8N25I>:625I>;<25I><A25J>=F25F>>G24C>?H23?>@I22<>AJ219>BK206>CM1O3>DN1N1>F71M:>G@1LB>HH1KJ>J11K3>K91J;>LB1ID>M71J>>N>1J4>OO1H9?261H>?361GN?461G>?561FM?661F=?771EM?871E=?971DM?:71D=?;71CL?<71C<?=71BL?>71B<
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
27786M40K87841087F41788441?88A41G88O41M89=42389L42:8::42A8:H42G8;642N8;D4358<243;8<?43B8<@43E8<A43J8<B43O8<C4428<D4478<L44=8==44A8=L44A8>;44>8>I4488?64438?B4408?O43M8@<43J8@I43G8A:43>8AM4388BK43<8BK42F8B<4108C541A8D=41K8F?40N8E94048ED40?8EO40J8F:4158FF41@8G241H8G?4208GL42:8H842B8HE42K8I342O8ID4308J34328JC4338K24358KA43<8KK4408L844<8LH44?8M844B8MI44F8N944I8NJ44L8O:4508OJ45390>45290J44D91?44A92644?92M44>93D44<94:44;95244995H44896D44:97444;97<44<97:44E97944K97=45097A4549824599;543>9>G41A9=440@9:B40G98H41399O4249:M42K9;D42N9<<4319=44349>143E9>243I9>G43G9?;43E9@043C9@E43A9A943?9AI43?9AK43E9B043E9B743=9BA4369C242O9C@42C9CI4239D341A9D;4109DD40?9DM3ON9E=3O?9F23O19FH3NC9G43NC9G?3NC9GK3NC9H73NC9HC3NC9HN3N@9I63ND9I>3NI9IK3NF9J93NC9JF3N?9K43N<9KA3N99KN3N69L:3N49LA3N;9LH3NA9M63NB9MG3NA9N73N?9NG3N?9O83N>9OH3N<:093MM:0H3M@:193M2:1J3LE:2:3L7:2J3KJ:3;3K<:3L3JO:4;3J@:4?3IH:4B3I0:4F3H7:4L3G@:5>3G4:623FH:643FJ:663FM:673FO:693G2:6;3G5:6<3G8:6J3GA:773GJ:7D3H3:843GL:8D3GC:953G;:9G3G2:::3FK::G3FD::M3F?::N3F1::L3E>::L3DO:;43DL:;73DC:;C3D=:<43D9:<93CM:<M3CG:=>3BK:>?3B?:?C3B7:@H3AO:A<3A::B03@E:BD3@0:C;3?=:D53>K:DO3>::EI3=I:FB3=9:G;3<I:H53<::HN3;K:I33;<:HN3:M:I03:@:I;3:?:ID3:C:K13:::MB3;E;093;I;3K3;E;5F3;M;6<3;=;6D3;6;6M3;0;7B3:H;:43:8;<539I;=B3:=;?43:<;@G3:8;@F39K;BE3:0;D13:C;BO3<5;CF3;L;DB3;A;E=3;6;F:3:J;G53:?;H13:4;HM39H;IH39=;I239M;I73:E;I=3;>;IF3<5;K73<9;KJ3<:;KK3<5;KO3<7;L53<<;L=3<B;LD3<H;LK3<?;M03:1;MA38K;N638H;NL38E;OB38C<0<38I<17394<2339><2N39I<303:0<303:5<303:9<303:=<303:@<303:D<303:H<313:K<3B3:=<3E39O<3G3:0<3N38H<49376<5737E<64384<7;387<7M393<8D39:<9<39A<:138M<:G387<;H385<=737@<>E36L<@3366<AB35B<C034M<C@354<C335J<C636;<D<36D<E736@<F236:<FL366<FH342<FC33C<F6340<FD33H<G333@<G?344<GH35><H:355<HE353<HL354<I9351<IJ34M<J@34D<L:33=<ME32D<NB328<O?31L=0<31@=19314=2D31F=4032;=5=32O=51338=5@32L=5G32D=6E32J=7L32<=8H314=9H2OI=:G2N>=;G2M3=<?2L5==32K>==I2JH=>@2J2=?72I==?N2HG=@E2H2=A<2G<=B42FG=A>2EJ=@H2DL=A52DB=BL2D0=CI2C0=DA2B7=F72AC=GM2@N=IB2@:=J@2?E=K02?2=KO2>L=M62>L=N<2>L=O<2>I>072>C>122>=>1N2>8>2I2>3>3D2=M>4?2=G>5D2=>>6K2=4>822<I>982<?>:?2<5>;E2;J><L2;@>>22;2>?:2:A>@A2:0>AH29@>BO28O>D628>>E>27N>FE27=>GL26L>I326<>J:25K>KA25:>LI24J>N0249>O723H?0>237?1E22G?2M222?5>20H?6O209?7M1OI?8G1O7?9B1NE?:=1N4?;71MB?<21M1?<L1L??=G1KM?>B1K<
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
2367G340J7G?40N7GL4127H94167HF41:7I341?7I?41C7I=41D7I:41E7I841G7I641G7I341I7I141J7HN41K7I141A7I54177I840L7I?40D7IN40?7J>40;7JN4077K>4027KN3ON7KO4067L040>7L140F7L240N7M441=7MM41F00000000000000000000000081I43382B43<83;43F84443O84M44885F44B86?44K87845488145=88J45F89C45O8:<4698:J46?8;946F8;G46L8<54738<J47;8=?47D8>447L8>I4858?>48>8@348F8?24858>?4888>?4868>?4848>;4838>74818>447O8>647H8>947B8>>47;8>H4748?946O8?O46H8@D46A8A;46:8B14648BF45M8BF46<8BB46O8C;4708D546L8E145M8CO4538DJ44>8E:4488E;44?8EI44E8F>44J8FK44K8G>44E8FJ44?8J842J8JA43?8JK4428K544G8K?45;8KN45B8L>45E8LO45I8M?45L8N345K8NF45K8O:45J8OM45J90@45J90F45N90K46291146591G46A92@46M93947:94147G94947O94@48794H48?94O48H95748O95G49E9744::98?4:J99>4:;9:<49M9;:49>9;648K9;;48:9<14829;H4829;?4829;54829;I47N9<B47J9<I4759=74779=G47=9>847C9>F47H9>I47J9?247I9?:47E9?A47>9?I4769@04709@846H9@?46A9@F46:9@N4639A645L9AH45D9B945=9BK4559C<44N9CM44G9D?44?9E64449EM43K9FD43B9G94399GM42L9HB42@9I64239IJ41G9J=41:9K240N9KA40G9KG40J9KN40M9L54109L;4129LE4129LN4179M441=9M;41?9MB41A9MI41A9N041B9N741C9NF4199OA40@:0<3OG:173NN:263MI:353LE:3D3L8:3G3L;:3J3L>:3M3LB:403LE:443LH:473LL:4F3L?:513KL:543K4:5:3JA:5D3J7:5N3IM:673IC:6A3I9:6G3I=:6J3IM:6M3J=:723JK:7C3JL:843JM:8E3JN:973JO:9H3JO::@3JM:;83J?:;?3IL:;K3IC:<73I;:<D3I3:<H3HJ:<K3H@:=83H::=M3H9:>;3H0:>I3GF:?73G;:?E3G1:@23FG:@@3F=:@L3EO:A63E<:A?3DI:AI3D6:B23CD:BE3C7:C@3BK:D;3B?:E63B4:F13AH:FK3A<:GF3A1:HA3@E:I<3@8:J73?J:K23?<:KM3>N:LJ3>@:MN3>6:O83>1;0B3>:;1K3>J;373?:;563?<;6@3?;;783?6;7O3?1;8J3>K;:43>?;;?3>2;<I3=F;>33=:;?G3<E;AB3;I;C>3:M;DJ3:3;E73:B;ED3;0;F13;?;F63<?;G=3<=;HG3<?;IO3<@;K43<L;LD3=E;M>3=K;M?3=;;M@3<M;M:3=0;M83=5;ME3=1;N23<M;N43<N;N53=2;N@3=5;OG3=:<023>A<0:3??<0=3?5<0@3?3<143?E<1G3?D<293?B<2K3?@<383>D<3E3>7<423=J<4>3==<4K3=0<583<C<5D3<6<613<0<6<3;I<6F3;C<703;<<7:3;6<7E3:O<7O3:H<8:3:B<8D3:;<8N3:4<9I3:<<:<3:H<:J3;4<;73;@<;O3;N<=@3;?<?13;0<@33:I<?F3:O<??3;5<@23;4<@=3:N<@I3:H<A43:B<A?3:<<B;3:8<CB3:7<F:3:K<FJ3:F<F83:<<EF3:3<EF39?<GA3:1<H?3:4<I339N<IF39I<JK39L<LA3:5<K6397<K6375<K5353<JF34><KC34M<LN35D<N2360<O3366=0336<=1D35O=3;35<=5234J=6H347=8?33D=:5332=;032J=;732K=<531C==4308==N2O8=>B2NA=?62MJ=?J2M3=@>2L==A22KF=AF2JO=B:2J:=BG2J0=C42IF=CA2I<=CI2I;=DL2I4=F32HG=G;2H:=GJ2GE=H22FM=I02F;=IM2EH=JK2E6=KH2DD=LC2D>=M?2D9=N:2D3=O52CM>002CH>0K2CC>1G2C=>2B2C7>3H2BM>4N2BC>652B9>7;2AN>8B2AC>9I2A2>;02@B><72@1>=?2?@>>F2>O>?M2>?>A42=N>B;2==>CC2<M>DJ2<<>F12;K>G52;:>H62:G>I62:3>J729@>K728M>L828:>M827H>N9274>OM26<?2:25D?3N25=?55254?5M24??6N241?7?23K?7O23E?8@23@?9123:?9B234?:322O?:C22I
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
1837033G@70K3GK71B3H77293HC7313HO73I3I;74@3IG7583J27603J?76G3JK7793K277B3K577L3K77863K97873KD78B3L478L3LD7963M379A3MB79K3N27:;3N?7:F3NG7;23NO7;>3O87;I3O@7<53OH7<@4017<L4097=740A7=C40J7=O4127>:41:7>F41B7?441J7?C4217@24287@?42=7@L42@0000000000000000000000000000000000000000000000007D?43G7DL43L7E943O7EF4447F24487F@44<7FL44A7G944E7GF44I7H344N7H@4517HM4567I:45;7IE45>7IE45A7IE45D7IE45G7IE45J7IE45N7IE4617IE4647IE4677IE46:7IJ46:7J84657JG4617K545K7KD45F7L245B7LC45J7LJ4617LK46:7LK46A7MI47=7ML4787N94777NG4767O54757OB47480047380=47280K47181847081F46O82446N82A46M82O46L83:48:84349>85849O86<4:@87A4;188E4;A89D4;L8984;188M4:588J49B89549C89I4848:@4698;744?8;M42C8<D40H8=;3NN8>23M38>B3LF8?03M68?23NO8?440H8?642B8?844;8?:4658?<47O8?>49H8?F4:G8@44;18@B4;:8A54;:8DE4;:8FK4<F8G64=G8G?4=;8GA4>58H14=G8HF4<H8I:4;I8I?4;78I94:O8IO4:?8HO4;78HL4;<8HI4;@8HF4;E8HB4;J8I74:G8J;49=8KM4838HO4618KG47B8L@4828LI48B8M14928MA4978N?48<8O;47L90048;90E48I91=49592749A93149L93J4:794D4:B95>4:M9674;99774;:9874;998G4;B9974;J99F4<39:64<<9:E4<F9;74=69<44=89=G4<K9?;4<>9@O4<19C04;09B@4:A9AC4:;9@F4:69?J4:09>M49K9><4989>@4959>C4929>G4909>J48M9>N48J9?148H9?548E9?848B9?<48?9??48=9?E48;9?O48:9@948:9@C4899@M4899A74899A@4889B04839BI47F9CB47<9D94739DO46K9EF46B9F=46:9G34619GJ45I9HE45@9I@4569J;44M9K644D9L044:9L>4409LH43F9M343<9M<4339ME42E9MJ41N9MO4179N440A9N93OJ9NE3O;9NE3O@9ND3OH9NC3OO9NB4079NC40>9NF40C9NH40H9NK40N9NM4139NO4189O241=9O141B9O<41F:0041J:0E41O:1:423:1N428:2D425:39420:3O41K:4D41B:59415:5O40I:6D40=:79400:7N3OD:8E3O8:9@3O2:9H3O3:9O3O5::83O7::A3O6::L3O5:;73O3:;A3O2:;L3O1:<73NO:<B3NN:<M3NJ:=;3N9:=J3MH:>83M7:>I3LF:?E3L7:@@3KH:A;3K9:BF3JH:C63J;:C53IO:C63IB:C<3I0:CB3H?:D13H3:DI3GK:E93GE:EH3G>:F63G8:FF3G0:G83FG:GJ3F=:H;3F4:HL3EJ:I=3E@:IN3E3:J?3DG:JO3D::LD3D5:MM3D0:O63CK;0?3CF;1H3CA;313C<;4:3C7;5C3C2;6M3BN;853BH;9?3BD;:G3B>;<13B:;=93B5;>C3B0;??3AN;?I3AN;@23AM;@;3AL;@E3AK;A<3B1;B;3B8;C93B@;D83BG;E63BO;F43C7;G53C@;GD3C9;GM3C0;H83BJ;I43C3;IO3C=;JK3CF;K;3CG;K73C<;K<3C1;KO3BC;LL3BK;MK3C7;NI3CC;OH3CN<0;3B;<0G3AM<0O3B7<173BA<1@3BJ<1H3C4<203C><2<3C9<2L3B5<2=3B7<1J3@B<173>N<0D3=;<1@3=1<343=F<4I3>;<5N3>M<613?9<6<3>K<6G3>:<7L3>7<913>G<:53?=<;93@4<<=3@K<<I3A8<;F3@?<:F3@0<;73@8<<L3@O<>@3AH<@13B;<@D3AN<A73AA<B93AJ<C?3B:<DD3BI<E<3B><EL3A0<F<3?C<FM3>5<GB3=0<I23=@<JB3=O<L23>><LJ3>B<KK3=G<KA3=7<L53=2<LH3<L<M<3<H<MO3<B<N:3<6<N=3;C<N@3;0<NC3:<<NF39H<NI394<NL38A<OB38:=1K38J=4339:=6638K=8437B=:236:=;M354==B33H=?832E=AJ33>=AO335=AF32E=A<322=@N315=@F2OI=A32OD=AA2O?=B32O<=BG2O:=C;2O7=CO2O4=DC2O1=DM2O==E3300=EC301=F=2OE=G62O9=H02NM=HK2N>=IH2ML=JF2M9=KC2LG=LA2L5=M>2KC=N;2K4=O72JN>032JH>1:2J>>2@2IN>3H2I>>4O2HM>622H;>722GG>832G4
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
1686HO3F46I<3F96IJ3F>6J83FB6JE3FH6K33FL6K@3G16KN3G66L<3G;6LJ3G?6M83GK6MF3H76N43HC6NB3I06O73I<6ON3IH70F3J371=3J?7253JK72M3K773D3KC74<3KO7543L:75K3LG76C3M377;3M>77C3MD77M3N37873NC78A3O278K3OB79640279A40A79L40I7:84117:C41:7:O41B7;;41J7;F4237<242;7<=42C7<I42L7=44347=@43<7=L43D7>743L7>C4457>N44:7?944?7?C44D7?N44H7@944N7@D4527@N4587A945<7AD45A7AO45F0000000000000000000000000000007CO46C7D:46H7DE46M7DO4727E:4767EE47<7EO47@7F;47E7FH47I7G647M7GD4817H14867H@4897HM48=7I248<7I748:7I<4897I@4887IE4867J148>7J@48K7K04987KC49C7L749N7LH4:97LN4:A7M34:J7M44;07M34;57M14;:7LO4;?7L>4;77L84:L7L=4:A7N?4;17N;4:G7O:4;77OG4;88004;780=4;480C4:M7OI4:K80H4;@81J4<081?4;I8134;A80G4;:81@4;@82C4;L83G4<78554<F86D4=38834=B89A4>08994>:8924>C88K4>M88D4?788<4?A8854?J87M4?187I4=O88D4=68994<M89I4=38::4=98:K4=?8;<4=E8;L4=K8<=4>18=14>08><4<F8?M4<38A=4;@8BM4:N8D>4:;8EN49I8G?4978HO48D8J048<8JD48<8IN4908I849E8I@4;88HC4=68GM4>;8EN4=98EB4=78G84>28IG4?E8L54A88L34@H8KF4?O8K:4?68JM4><8JF4=F8K84=<8LB4<L8N;4<88O84<08OO4;M90E4;K90M4<>9144=091<4=B91L4=H92B4=H9384=H93N4=G94L4=G95L4=G96M4=F97L4=F98L4=F99M4=E9:D4=J9;34>39;C4><9<@4>;9>44=I9?I4=69@64<G9@54<99@44;J9@44;<9@34:M9@24:?9@24:09@149A9?<49:9>?4949>?48M9>C48J9>F48H9>O48G9?948F9?C48E9?M48E9@848E9@F48G9A548H9AC48K9B148L9B?48N9BM48O9C<4919CJ4929D84959DF4969E44989EC4919F348E9FC48;9G247O9GI47F9H347=9H=4729HG46H9I146>9I;4649IE45J9IN45@9J84569JB44L9JL44B9K64489K@43N9KM43B9L<4379LJ42L9M942@9MG4249N541I9NC41=9O14129O14199O041A9NO41H9NN4209O54249O=4279OE42:9OM42=:0;42A:1042F:1F42I:2942L:2@42G:2F42B:2L42=:32427:3D41L:4641@:4G414:5940G:5L40;:6A400:7=3OH:8:3OB:963O::9L3O7::53O7::;3O9::@3O<::H3O>::O3O?:;73OA:;?3OC:;F3OE:;N3OF:<63OH:<>3OJ:<E3OL:<O3OK:=D3O>:?93NK:@O3N9:BC3MF:D93M4:DC3LG:DB3L::DC3KK:DD3K=:DE3JO:DF3J@:E<3J9:F33J2:F:3J::FG3J=:G;3J9:H03J3:HD3IL:I43I@:IE3I4:J63HH:JF3H<:K73H0:KG3GD:L<3GB:M83H5:N43HI:NO3I<:OD3II;093II;1:3IC;2:3I<;3;3I6;4<3I0;5=3HJ;6=3HD;7>3H>;8?3H7;9@3H0;:A3GH;;D3GC;<L3GA;>53G?;?=3G>;@<3FO;@O3FB;A>3F?;AM3F;;BB3FF;CC3H1;D73H;;DG3H@;E83HC;EG3HB;FB3H=;FJ3G3;G03EC;GL3E>;HH3E8;IN3E1;K63DI;L=3DC;M;3CF;MK3AK;N;3@0;N>3>O;N33?1;N43?>;N53?K;N73@8;NO3@B;OG3@K<0>3A4<163A><1N3AG<3B3B;<3@3BN<3>3CA<3B3D7<403D8<4>3D9<4L3D9<5:3D9<5H3D:<663D;<6D3D<<723D<<7=3DC<8:3D;<993D5<:?3D5<;<3E4<<?3G8<=G3I<<>:3HA<?53H1<@03G@<@J3FO<AE3F9<B?3E?<C93DF<CK3D7<D=3C?<DO3BG<EB3AN<F73A6<FN3@><GD3?F<H;3>N<HO3>F<ID3>F<JH3>M<KM3?5<M13?=<N53?D<O03?J<OL3@0=0G3@6=1C3@==2>3@C=3:3@I=453@O=4H3@D=563@1=6A3>J=7G3=@=8M3<6=:33;;=;93:F=<>3:2==D39>=?538A=?<38:=?B382=?I37K=@037C=AB36<=BM35<=CC34M=D734>=DG33O=E733A=EG332=F632C=F531I=F6318=F;31K=F@32==FG32O=G@32B=H9326=I331J=IM31>=JG312=K@30F=LG307=N52OH=O42O;=O>2O1=OH2N@>022MM>0@2M?>1>2LM>2>2L:>3>2KG>4?2K4>5?2JA>6@2IN>7@2I;>8A2HH>9A2H5
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
1636E43H@6EE3HF6F53HL6FF3I16G63I66GG3I<6H73IA6HH3IF6I83IL6II3J26J93J76JJ3J<6K:3JB6KK3JG6L;3JL6LL3K26M<3K86ML3K=6N:3KB6O03L16OD3L97083L?70L3LE71@3LL7243M272H3M873<3M?7403ME74G3N675E40L75;40A75140674G3OK7473NN7453O974<3OH75B40;76J40N77<41B77F42678142I78;43=78F44279044E79;45979H45B7:645J7:D4617;246:7;@46A7;O46I7<=4737<L47<7=;47F7=J47O7>94897>B48=7>J48A7?248D7?<48H7?G48M7@24927@<4977@G49;7A249A7A<49E7AG49K7B249O7B=4:47BI4:87C74:=7CD4:A7D34:D7D@4:I7DN4:M7E<4;17EJ4;57F84;97FF4;=7G44;A7GC4;J7H34<<7H;4<N7HB4=?7HM4>07I:4>87J44>87JN4>87KG4>77LA4>77M:4>67N44>67NM4>57OC4>780<4>881=4>881E4>>81G4>D81H4>L81J4?481K4?;81L4?B81N4?J81O4@18214@980K4@D80C4?G80F4>I80H4=O80I4=L80H4>080B4>E80<4?:8194?:82>4?683C4?184G4>M85L4>H8694?18654?C8624@485N4@F85J4A785F4AH85B4B985?4BK85;4C<85D4CG8674CL86M4C@87C4C488:4BH8914B;89G4AO8:>4AC8;34A:8;G4A98=M4A?8@64AJ8A74A58AI4?08BJ4>N8CL4?18DN4?58F14?98FL4??8G?4?G8H24?O8HE4@78I84@?8IK4@G8IK4AB8I74BE8I94CG8IK4CH8J<4CK8JM4CL8K>4CN8KO4D08L@4D28LI4D48M64D38M94C;8M;4BB8M>4AI8MF4A=8N64AC8NG4AJ8O74B08OG4B790@4B891;4B69264B59334B39414AO9504AL95N4AH96M4AD97L4A?98J4A<99I4A89:G4A49;D4@F9<>4@09<E4?H9<G4?D9<I4?@9<K4?<9<M4?89<O4?49=04?19=34>J9=64>A9=:4>:9==4>29=A4=I9=D4=B9>84<O9>O4<<9?G4;I9@>4;59A64:A9AN49N9BD49<9C349>9CB49A9D249C9DA49E9E049H9E?49I9EN49K9F=49N9FM4:09G<4:29GF4:19GI49I9GM49A9H049;9H048H9GO4839H947A9HE4709I046=9I=4669II45N9J645G9JD4619K546H9K:46I9K:46@9K;4679K;45N9K?45D9K?45;9KK4519L744H9LC44?9LO4459M;43L9N043F9NK43B9O:43>9OI43::07436:0E432:1442M:1C42I:2042F:2842J:2D42H:2J42C:3042>:37429:3=423:3D41N:3J41I:4041D:4741>:4=419:4L40N:5F40E:6:40D:6N40D:7B40C:8740B:8K40A:9?40A::340A::C40J::O41::;<41J:;J424:<=41M:=041F:=B41>:>4417:>G410:?:40H:?L40A:@>40::A1403:AC3OK:B23OE:BB3O>:C23O7:CA3NO:CJ3NF:D33N?:D<3N8:DD3N1:DM3MJ:E63MC:EA3M?:F73M@:FN3MA:GF3MB:H:3M=:HN3M9:IB3M4:J73M0:JK3LJ:K?3LF:L43L@:LI3KN:M73KK:MA3KN:MK3L1:N53L4:N?3L7:NI3L::O33L=:OI3LH;0A3LI;163L<;1K3KN;2B3KD;3=3KD;493KD;563KD;623KD;6>3K:;773JB;7N3II;8F3I1;9>3H8;:63G@;:K3FM;;;3FJ;;I3FF;<83FC;<G3F@;>03F6;??3EK;@M3E?;B<3E4;CJ3DI;E93D>;FH3D3;H63CH;I@3C@;JJ3C:;L33C3;M=3BK;NE3BD;O83B3;NK3BO;NC3CJ;NO3D@;O;3E5;OF3EI;O23EG<043EE<1L3E@<4:3DO<563D9<3E3C<<473C:<5;3D><733EH<9:3F@<:73FF<:E3FG<:O3G8<;93GI<;C3H:<;M3HJ<<73I<<<B3IL<<L3J><=63JO<=?3K?<=D3I?<=G3G8<>43EF<>O3DO<?I3D9<@C3CC<A=3BN<B73B;<C83AN<D53AK<DM3AL<EF3AN<F?3AN<G83B0<H13B1<HJ3B2<I<3B1<IK3AN<J:3AK<JM3AG<KM3A?<LN3A7<MN3@O<NO3@G<OO3@?=0O3@6=203?O=303?F=413??=513?6=6=3>M=823>A=9H3>5=;=3=J=<@3=G==?3=E=>@3<N=?@3<9=@93;C=@O3:M=A=3:N=AF3:E=B03:<=B83:4=B?39M=BF39G=BM39@=C4399=C;393=CC38L=D638D=E638;=F5381=G537G=GM37==HH36O=J636>=K<35K=LB359=MH34G=NH346=O;33K=OH33@>03335>0=32L>0G32B>12328>1?31F>1J313>2430?>2>2OL>2I2O8>332NE>3>2N2>3N2MF>4>2M<>4N2M0>5L2L?>6L2KL>7L2K8>8M2JE>9M2J2>:N2I?>;N2HL
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
1616B83L76BJ3L46C<3L16CO3KN6DA3KK6E33KH6EF3KE6F83KB6FJ3K?6G=3K<6GO3K:6HA3K76I23K86IC3K>6J43KC6JD3KI6K43KN6KE3L46L63L96LI3L96M=3L76N13L46NF3L16O:3KN6ON3KK70B3KI7163KH71J3KI72>3KJ73=3M17423N?7413NL7403O;73O3OI73N40873M40G73L41573J41D74342274B42@75342L75M43=77144177<44E77F45878045L78;46@78I46H79747079E4777:447?7:D47F7;347N7;C4877<248@7<A48I7=04927=@49<7=N49E7>;49N7>C4:17>K4:47?24:77?:4:;7?E4:?7@04:D7@;4:I7@E4:N7A04;37A<4;77AK4;;7B84;@7BF4;C7C44;G7CB4;L7D14<37D@4<;7DO4<C7E>4<L7EN4=:7F=4=J7FM4>;7G?4?17H14?F7H=4@97HE4@I7HE4?M7I94?D7J54?B7K14?A7KM4?@7LI4??7MF4?>7NA4?=7O54?67O;4?37O84?37O94?:7O:4?A7O;4?I7O<4@07O>4@87O?4@@7OF4@G8074@O80I4A681;4A=8194A581:4@=81<4?G81=4?E81>4?D81?4?A8194@681J4@=82O4@98444@484F4@584N4@98564@D85?4A485I4AD8624B38624BE8614C486C4C:8764C?87H4CD88:4CJ88M4CO89?4D58:14D:8:E4D;8;;4CN8<24CB8<D4C?8=44C>8=E4C=8>54C;8>F4C;8?64C:8@=4C;8AG4C=8BA4C78C94B28D;4B78E44B=8EG4BE8F:4BN8FM4C58G@4C>8H24C:8HC4C38I54BL8IB4C58II4C58J04C78J74C88J=4CC8JD4D78JK4DL8K14E@8KK4E18LF4D@8M@4CN8MC4C58ME4B=8N24B98NB4B?8O24BF8OC4BL9034C390D4C99144C?91D4CF9254CL92F4D393>4D49494D29574CN9664CK9754CG9834CC9904C899J4BA9:D4AJ9;?4A39<94@=9=34?F9=64?A9=84?>9=:4?:9=<4?69=>4?29=@4>N9=B4>J9=D4>F9=J4>?9>94>39>G4=H9?:4=?9?M4=89@A4=09A44<H9AJ4<99BA4;F9C94;29D04:>9DH49K9E949G9EH49I9F749K9FG49N9G64:09GC4:19GG49I9GJ49B9GM49;9H14939H448L9H848E9H84839HD47B9HO4709I;46@9IJ46;9J:4669JG46D9K44749KB47F9KF47>9KO4769L846M9L?46E9KN4699K=45L9KC45B9KO4599L:4509M444K9MO44H9NH44D9O744?9OE44;:03447:0B443:1043O:1=43J:1F43A:1N439:29433:2K434:3=435:3O436:4A436:4N434:5342L:5742E:5;42=:5@426:5C41M:5:419:5H415:6<414:70414:7D413:89412:8M411:9A411::5410::G414:;341E:;@426:;M42F:<9436:<K433:==42L:>042E:>C42>:?5426:?H41O:@:41H:@J41A:A941::AK411:BA40D:C8408:CM3OL:D83OD:DA3O<:DJ3O5:E33NN:E=3NF:EF3NC:EN3NI:F63O0:FL3O2:GD3O2:H93NO:HM3NJ:IB3NF:J63NA:JJ3N::K?3MI:L43M8:LI3LG:M=3L6:ML3L0:N73L3:NA3L6:NK3L9:O@3LD;093M1;113M?;1G3M9;2=3LK;323L>;3G3L1;4<3KC;513K6;5F3JH;6>3J@;6L3J<;6E3J0;6>3IE;6<3I7;743H?;7L3GG;8B3G1;913FN;9@3FK;:K3F@;<:3F5;=I3EI;?73E?;@F3E3;B53DH;CC3D=;E23D2;F@3CG;GO3C;;I=3C1;JL3BE;L63B>;M43B5;M:3AK;MI3B6;N=3BD;O03C0;OD3C;<083CG<0L3D6<1B3DI<273E9<2L3EE<2?3F><393FA<4C3FD<5M3FF<773FI<8A3FL<9J3FO<:L3G3<:M3G><:O3GH<;13H2<;33H=<;43HF<;63I1<;83I<<;73IG<;63J3<;F3IL<<N3HG<>63GC<?>3F=<@13E9<@:3DF<@B3D3<@J3C@<AJ3C@<BJ3C@<CB3C4<D:3BG<E23B:<EL3B;<FD3B;<G=3B=<H43B=<HC3B;<I>3B4<J?3AL<K?3AD<L?3A;<M@3A4<N@3@L<OA3@C=0A3@<=1B3@3=2B3?L=3B3?C=4C3?;=5C3?3=6D3>K=7D3>D=8D3>;=9G3>3=;=3=G==13=;=>73<F=?13;K=@:3:F=A;3;3=AN3;>=B53;7=B<3;0=BC3:J=BJ3:C=C13:==C83:6=C?39O=CF39H=CM39A=D439:=D@394=EC38I=G438;=HE37L=IL37;=K236H=L8366=M>35C=N7355=NI34J=O<34>=ON342>0@33G>0O33;>1:332>1D32H>1N32=>2=31J>2L316>3;30D>3F301>402O=>4;2NJ>4E2N6>552MK>5E2M?>662M5>6F2LI>7E2L7>8F2KD>9F2K1>:G2J>>;G2IJ><H2I7>=H2HD>>I2H1>@72G<>AL2G6>C@2G0
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
1626AB3NL6B43NI6BF3NF6C93NC6CK3N@6D>3N=6E03N:6EB3N86F53N56FG3N26G93MO6GL3ML6H>3MI6I03MF6IC3MB6J73M:6JK3M46K=3M46L23M46LE3M66M93M76ML3M76NA3M86O43M:6OH3M:70<3M;7103M=71D3M?72H3O67353OH73@40573I40C73H41173G41@73F41O73E42=73D42L73C43;74243H74A44575244C75A45076545>77146277;46F77F47978447A78B47I79248079B4887:248?7:A48G7;148O7;A4987<049A7<?49K7<M4:47==4:>7>34:J7>:4:I7>=4:G7>I4;57?04;87?84;<7?@4;@7?L4;D7@64;I7@D4;M7A14<17A@4<77AO4<?7B>4<G7BM4<O7C<4=87CK4=?7D;4=K7DJ4>;7E:4>J7EI4?:7F:4?L7FL4@A7G>4A67H04AL7H?4B@7HB4B07HB4A37HO4@C7IJ4@A7JG4@@7KC4@?7L?4@>7M34@87MF4@17N84?J7N74?J7N44?J7N14?J7N04?M7N?4@47O04@<7OB4@C8044@K80E4A28174A981I4AA81N4A>81L4A581J4@N81J4@M81K4@J81L4@H81M4@F81J4A682N4A183E4A083L4A48444A884<4AA84E4B184N4B@8584BO85E4C;8654CE86K4CJ87B4CN8884D188O4D589E4D:8:<4D=8;24DA8;H4DE8<?4DJ8=34DM8==4DM8=L4DL8>;4DK8>J4DJ8?84DJ8?G4DH8@64DG8@D4DG8A24D:8AM4D78BJ4D88CH4CO8D34CF8DF4C;8E94C08EL4BE8F>4B<8G04B58GB4AN8H34AI8H<4B88H>4C08HO4CG8I64D<8I<4E08ID4EE8IJ4F98JD4EJ8K>4E98L94DG8M34D68MH4CC8MK4BJ8N94BK8NJ4C28O:4C88OJ4C?90;4CE90K4CL91<4D291L4D892=4D?92M4DE93>4DL93N4E294>4E89534E<9604E996O4E597J4D>98D4CG99>4C19:84B;9;34AD9;M4@M9<G4@79=@4?A9=B4?=9=D4?99=F4?59=H4?19=J4>N9>44>D9>C4>89?14=L9?@4=@9?O4=49@A4<L9A44<D9AG4<<9B;4<49BN4;M9CA4;E9D54;=9DH4;69E=4:I9F44:69FH49M9G749O9GB49N9GF49G9GI49?9GM4989H04919H348J9H748B9H:48;9H?4859HO4809I?47K9IN47F9J>47@9JN47=9K;47L9KD4889KA47N9KG47F9L647@9LE4799LB4709L546F9KD4699K=45N9KO45G9LJ45C9MF45?9N@45;9NO4579O>4539OL44O:0944I:0A44A:0I448:11440:1:43H:1B43?:2043<:2A43<:3343<:3E43=:4743>:4I43?:5;43?:5M43?:6>43@:7143A:6K430:6?429:6;41J:6E41C:7241>:7F41=:8:41=:8N41<:9B41;::641:::J41;:;641L:;C42=:;O42M:<=43>:<I43O:=;43K:=M43D:>?43<:?2435:?D42N:@342G:@F42=:A=420:B241D:BH418:C?40K:D540?:DE405:DN3ON:E73OF:E@3O?:EI3O8:F13O=:F:3OD:FB3OK:FK402:GC403:H8400:HL3OK:I@3OG:J53O<:JJ3NJ:K?3N9:L43MH:LI3M7:M=3LF:N23L5:N>3L4:NJ3L9:OB3LF;0;3M4;133M@;1K3MN;2A3MC;363M6;3K3LI;4@3L;;563KN;5K3KA;6A3K2;783J@;723J2;6M3IC;6H3I3;6D3HC;6=3H7;6:3GJ;723G4;7I3FN;983FC;:F3F8;<53EL;=D3EB;?23E6;@A3DK;B03D@;C@3D4;E23CI;FC3C=;H43C2;I:3BL;J>3BG;K43BG;KA3C1;L>3C<;M33CG;MH3D3;N>3D>;O33DJ;OH3E6<0=3EA<133EL<1H3F8<1=3FO<2;3FN<393FL<473FK<4K3FF<4@3F3<5E3F4<6O3F7<893F:<993F><9;3FH<9<3G3<9>3G><9@3GG<9?3H4<9>3H@<9=3HM<9<3I9<:93HA<;A3G<<<I3F:<=@3EO<>83EC<>O3E8<?<3DI<@<3DJ<A<3DK<B;3DL<C83DE<D03D8<DH3CJ<E@3C=<F83C1<G03BD<GI3BB<HJ3B:<IJ3B1<JK3AJ<KK3AB<LL3A:<ML3A2<NL3@I<OM3@B=0M3@9=1N3@1=2N3?I=3N3?A=4O3?:=5O3?1=703>I=803>A=903>9=:13>1=;13=I=<23=A==:3<F=>D3;>=@13:5=A33;@=B>3;E=BE3;>=BL3;8=C33;1=C93:J=CB3:C=CK3:==D53:7=D>3:0=DG39I=E039B=E939;=F7390=GH38B=I8384=J?37B=KE36O=LJ36<=MF35N=N835B=NJ357=O<34K=ON34?>0@344>1233H>1D33<>20332>2:32H>2G32:>3631G>3E313>4430@>4C2OM>4M2O9>582NF>5D2N5>642MJ>6E2M?>752M3>7E2LI>8A2L8>9A2KD>:B2K1>;B2J>><C2IK>=C2I9>>N2HC>@B2H=>B62H7>CJ2H1>E>2GJ>G22GD>HF2G>>J:2G8
1626@N4086A@4056B34026BE3OO6C83OL6CJ3OI6D<3OF6DN3OD6EA3OA6F43O;6FH3O36G<3NL6H03NE6HD3N=6I83N66IL3MO6J@3MG6K43MH6KD3MM6L83N06LL3N06M?3N16N33N26NG3N36O;3N46ON3N570B3N67193NA7273OO72C40=72N40K73941873D41E73D42473C42C73B43173A43@73@43O73I44<74944J74I45775945D75I46276946@77147377;47F77I47N78948678I48=79948E79I48L7:84947:H49;7;849E7;G49N7<64:77<I4:B7=?4:N7>54;;7>;4;97><4;47>>4:N7>D4;37?04;C7?;4;N7?C4<27@04<67@?4<=7@N4<E7A=4<M7AM4=57B<4=>7BK4=E7C:4=N7CI4>;7D94>K7DH4?97E84?D7EH4?N7F54@87FI4A57G<4B27GO4BO7H@4CA7H@4BD7H@4AF7HI4A27IE4A17JA4A07K=4@O7L24@I7LE4@C7M84@<7MH4@67ME4@57MB4@57M?4@57N34@:7NI4@>7O=4@D8014@I80C4A08154A881G4A?8284AG8274A?8264A>8254A=8234A=8224A<8234A;8244A982<4A?8344A?83;4AC83C4AF83J4AJ8434B784<4BG84I4C48564C@85B4CM8644D686J4D:87@4D=8874DA88M4DF89D4DJ8::4DM8;14E18;G4E68<=4E:8=14E>8=<4E?8=E4E@8=O4E@8>94E@8>D4EA8>M4EB8?:4EA8?I4E@8@>4E=8A?4E38BA4DJ8C=4D@8CH4D88D34D08D>4CG8DN4C=8EA4C28F44BG8FG4B<8G?4BJ8G14BH8G34C?8GE4D78H:4DO8HL4EF8I34F;8IC4F?8J=4EN8K74E=8L24DK8LM4D:8MG4CH8MM4C18N=4C58NM4C;8O>4CB8ON4CH90?4CN90O4D591@4D;9204DB92A4DH9314DN93A4E59424E;94B4EB9534EH95C4EO96<4EI9764E29814D;98K4CD99E4BN9:?4B79;:4A@9<44@I9<N4@39=F4??9=H4?;9=J4?79=L4?39>24>L9>A4>A9>O4>59?>4=H9?M4=<9@;4=09@K4<F9A>4<>9B24<69BE4;N9C84;F9CK4;?9D?4;79E24;09EE4:H9F84:A9FL4:89G?4:09GD49I9GG49B9GL49>9H04999H44949H848O9H=48K9HA48E9HI48B9I948A9IH48=9J84889JH4839K84809KD48@9KC48;9K@4819K?47G9KN47A9L=47;9L54719KK46G9KA46=9K<4649L84619M445M9MO45I9ND45E9O245A9OA45=9OK455:0344M:0;44D:0C44;:0K443:1443K:1<43B:1K43?:2<43@:2N43A:3@43A:4243B:4D43B:5643C:5I43D:6C43E:7743E:7>43@:7242J:6G424:7141N:7:41G:7F41B:8;41B:8O41A:9C41A::741@::K41?:;8420:;D42@:<1431:<>43A:<J442:=944::=K443:>>43L:?143D:?@43>:@4433:@J42F:A@42::B641N:BL41A:CB415:D840H:DK40>:E4406:E=3ON:EF3OG:EO3OC:F73OK:F@402:FH408:G040@:GB40C:H840A:HL40<:I@405:J53OC:JJ3O2:K?3NA:L43N0:LI3M?:M=3LN:N23L=:NB3L5:O73L@:OO3LM;0H3M;;1@3MH;283N6;2N3MI;3C3M;;483LM;4N3LA;5C3L3;6:3KA;713K0;7=3JA;723J3;6M3IC;6H3I3;6D3HD;6?3H4;6;3GD;693G5;7H3FJ;993F>;:J3F3;<;3EH;=M3E<;?=3E0;@O3DE;B@3D9;D13CN;E93CG;F>3CB;GB3C=;HF3C8;IC3C1;J43C7;JJ3C@;L63CK;LK3D6;MA3DC;N63DN;NK3E9;OA3EE<063F0<0K3F<<0I3G1<1>3G5<2<3G4<393G2<473G1<553FO<613FN<5F3F;<5@3EK<6K3EN<853F1<8A3F9<8C3FC<8D3FN<8C3G:<8B3GF<8A3H2<8@3H?<8?3HK<9C3GK<:K3FG<;C3F;<<;3F0<=33EE<>63E<<?>3E8<@;3E:<A83E=<B53E@<C23E=<CM3DO<DH3D@<EC3D0<F>3CA<G93C3<GJ3BM<HI3BF<II3B><JI3B6<KJ3AN<LJ3AF<MJ3A?<NK3A7<OK3@O=0K3@G=1K3@?=2L3@7=3L3@0=4M3?G=5M3?@=6N3?8=7N3?0=8N3>H=9N3>@=:N3>8=;M3=O=<?3<K=>13;>=?B3:>=@M3;6=B>3;I=C33;@=C<3;9=CE3;3=CN3:L=D83:E=DA3:>=DI3:8=E23:1=E;39J=ED39D=EM39==FA394=H138E=IB387=JH37E=KN372=M436@=MF364=N835I=NJ35==O=351=OO34F>0@34:>1233N>1D33C>26337>2@32M>2K32B>3:31O>3J31<>4930H>4H304>572OA>5A2NN>5L2N;>6<2MO>6L2MD>7<2M9>7L2LM>8>2LB>9>2KO>:?2K<>;?2JH><@2J5>=@2IB>?32I4>@G2HN>B;2HH>CO2HA>EC2H;>G72H5>HK2GO>J?2GI>L42GB>MG2G<
1626@:41D6@M41A6A?41>6B141:6BE4146C940L6CM40D6DA40>6E54066EI3ON6F=3OH6G13O@6GE3O86H93O26HM3NJ6IA3NC6J53N;6JJ3N=6K;3NC6KL3NH6L>3NJ6M23NK6MF3NL6N:3NM6NM3NN6OA3NO7053O070O3OD71E40E72041272;41@72F41M73242;73=42H73@43673?43E73>44473=44B73A45174045>74@45K75046975@46G76047376@47A77148377@48;78048B78@48I78O49179@49879O49@7:?49H7:N4:17;?4::7<54:G7<K4;37=A4;?7>74;L7>74;F7>14;<7>14;77><4;B7>F4;L7?14<67?94<>7?H4<G7@84=07@I4=:7A:4=C7AJ4=K7B94>47BH4><7C84>I7CG4?47D84?>7DH4?I7E74@37EG4@>7F04@?7FD4A<7G74B87GJ4C57H94CE7H>4C77H?4B:7HD4AB7I@4A@7J<4A?7K24A:7KD4A37L74@M7LJ4@F7M64@A7M34@A7M14@@7M54@K7N24@L7NN4@M7OB4A38064A980I4A>81=4AD8214AI82A4AO82A4B082@4AO82?4AN82>4AL82=4AL82<4AL82;4AK82C4AL82J4B08324B48394B883A4B>83L4BK8494C884F4CD8534D185@4D=8624DF86H4DJ87?4DM8854E288L4E689B4E:8:94E=8:O4EB8;E4EF8<94EJ8<N4EO8=;4F18=D4F18=N4F28>84F38>C4F38>L4F38?64F48?@4F48@44EO8A64EF8B74E<8C34E38C>4DJ8CI4DB8D34D:8D>4D18DI4CH8E74C?8EJ4C58FG4D78F54CG8EG4CB8FB4D?8G74E78GM4EO8HD4FL8I=4FA8J74F28K14E@8KL4DO8LF4D=8M@4CL8N14C88NA4C>8O14CE8OB4CK9024D190C4D89134D>91C4DE9244DK92D4E29344E993D4E@9444EF94D4EM9544F395D4F:9674F096L4E?97C4DJ98:4D69924CA99L4BK9:G4B49;A4A=9<;4@F9=54@09=L4?=9=N4?99>04?49>>4>H9>M4>=9?;4>19?J4=E9@94=99@G4<L9A64<@9AH4<79B;4<09BO4;H9CB4;@9D54;99DH4;19E<4:J9EO4:B9FB4:;9G14:79GA4:39GH4:19GO49N9H349J9H749E9H;49@9H@49;9HD4979HI4929I348O9IC48N9J248L9JB48K9K248E9KA48E9KF48H9KB48>9K?4839K;47I9KE47C9L347<9KI4729K?46H9K;46@9KM46=9LA46:9M=4679N84639NG45O9O445I9O=45A9OE4589OM450:0544G:0=44?:0F446:0N43N:1643E:1F43C:2743D:2I43E:3;43E:3M43E:4@43F:5:43I:6543K:7043L:7;43I:7D43E:7F43;:7942E:7<42::7F423:8041L:8;41G:8O41F:9C41E::841E::L41D:;9423:;F42C:<2434:<?43E:<L445:=844F:=J44B:><44;:>M444:?B43I:@743<:@N430:AD42D:B:427:C041K:CF41>:D<412:E140F:E;40>:ED407:EM400:F5401:F=408:FF40?:FN40F:G640M:GA413:H7411:HK40M:I@40<:J53OK:JJ3O::K?3NI:L43N8:LH3MG:M=3M6:N23LG:NJ3LA:OD3LG;0=3M5;153MB;1M3N0;2F3N;;3;3MN;403M@;4E3M3;5<3LC;633L1;6J3K@;7A3JO;7<3JB;723J5;6M3IE;6H3I5;6D3HE;6?3H5;6N3GD;7?3G3;883FD;9J3F8;;;3EM;<L3EA;>=3E6;?N3DJ;A83DC;B=3D=;C@3D8;DE3D3;EI3CN;FM3CJ;GK3CC;HG3C<;I43CE;J?3D0;KO3D:;LD3DF;M:3E2;MO3E=;ND3EH;O93F4;ON3F@<0@3FM<0@3G=<1>3G;<2<3G:<3:3G8<483G7<553G4<643G3<773G6<6L3FC<6@3F1<6F3EE<7F3EI<7H3F4<7G3F@<7F3FL<7E3G8<7D3GE<7C3H1<7D3H:<8M3G6<9F3FH<:>3F=<;E3F5<=13EN<><3EG<?G3E?<@L3E=<AI3E@<BF3EB<CD3EE<D?3E9<E:3DI<F53D:<G03CL<GK3C=<H23C9<HH3C6<ID3C1<JE3BH<KE3BA<LE3B9<ME3B1<NF3AI<OF3AB=0G3A9=1G3A2=2G3@I=3H3@B=4H3@:=5H3@2=6H3?J=7I3?B=8I3?:=9J3?3=:J3>J=;03>8=;93=6=<K3;I=>;3;<=?F3<3=B13<5=BO3;F=CJ3;<=D:3;4=DC3:M=DL3:F=E53:@=E>3::=EG3:3=F039L=F939E=FB39>=FK397=H;38I=IK38;=K137H=L7376=M536F=MG36;=N9360=NJ35C=O=358=OO34L>0A34A>13345>1E33I>2733>>2F332>3132I>3?327>3N31C>4=310>4L30=>5;2OI>5J2O6>662NB>6D2N4>742MI>7D2M>>842M3>8E2LG>9<2L9>:<2KF>;=2K3><>2J?>=D2IK>?82IE>@L2I?>B@2I9>D42I2>EI2HL>G=2HF>I02H?>JE2H9>L92H3>ML2GM>NJ2G@
1636@>42=6A24266AF41O6B:41G6BN41@6CB4196D64116DJ40J6E>40C6F240;6FF4046G:3OM6GN3OE6HB3O>6I63O76IJ3NO6J@3O26K13O76KB3O<6L33OB6LE3OE6M93OE6ML3OG6N@3OH6O43OH6OH3OJ70D40G71341:71>41H71I42672442C72?43072J43>73543L73<44973;44H73:45673945E73H46374846@74H46M75847:75H47H76748676F48@77648H77F48O78649778F49>79649F79F49M7:64:47:E4:;7;64:C7;H4:L7<94;67<K4;?7==4;H7=E4;J7=?4;?7=H4;F7>34<07>;4<97>C4<A7>K4<I7?44=27?F4=<7@74=E7@H4=O7A94>87AL4>A7BE4>N7C74?97CG4?D7D74?N7DG4@97E64@C7EF4@N7EK4@E7F?4AC7G24B?7GE4C<7H04CA7H74C:7H=4BM7H?4B17I;4B07J14AL7JC4AE7K74A>7KI4A77L<4A07LD4@M7LA4@M7LD4A47M44A<7N14A=7NO4A>7OG4AA80;4AG80N4AL81C4B28264B882F4B>82G4BE82G4BF82F4BE82E4BD82D4BC82C4BB82B4BA82E4BB82K4BE8314BH8374BL83<4C083J4C=8474CI84D4D68514DB85>4DN8614E686G4E:87=4E>8844EB88J4EF89A4EI8:74EN8:N4F28;B4F68<64F;8<J4F?8=:4FB8=D4FC8=M4FC8>74FD8>A4FE8>K4FE8?54FE8?B4FC8?O4F?8@L4F78AN4EN8BH4EE8C44E=8C>4E48CI4DK8D44DC8D?4D:8DI4D28E44CJ8EE4CH8EI4D68EE4E88F44DF8FN4E08GE4EK8HI4G28I@4F<8J94EL8K24E=8KK4DM8LD4D=8M<4CN8N44CA8ND4CH8O44CO8OD4D69034D=90D4DD9144DJ91D4E19234E892D4E?9344EF93C4EL9434F394D4F:9534FA95F4F?96;4F196M4EB97@4E19884D=98O4CI99F4C59:=4B@9;44AL9;L4A79<C4@C9==4?M9>14?:9><4?19>K4>E9?94>99?H4=M9@74=A9@E4=59A44<I9AB4<<9B24<19BE4;J9C84;B9CK4;:9D?4;39E24:L9EA4:H9F14:D9FA4:@9G04:<9G@4:89GJ4:59H04:49H74:19H=4:09HC49L9HG49H9HL49C9I049>9I>49<9IM49;9J=4999JL4989K<4969KH4959KE48J9KA48@9K=4869K:47L9K=47C9KG47=9K=4739K:46J9KM46H9L?46F9M346C9MF46@9N=46=9NF4659NN45L9O745D9O?45;9OG4539OO44K:0844B:0@449:0H441:1043I:1A43G:2243H:2D43I:3743I:4243L:4L43M:5G43O:6B441:77441:7?43M:7H43I:8143E:7M436:7H42E:8242>:8;427:8E421:9041K:9D41J::841J::L41I:;:426:;G42G:<4438:<A43I:<M449:=:44I:=H451:>:44K:>O44?:?E442:@;43F:A243::AG42M:B=42A:C4424:CJ41H:D@41;:E640O:EA40G:EJ40@:F3408:F;40>:FC40E:FK40L:G4414:G<41::GE41A:H741B:HK416:I@40D:J5404:JJ3OB:K>3O6:L23NJ:LF3N?:M:3N3:N23ML:NO3MI:OK3MF;0H3MC;1B3MI;2:3N7;323N@;3G3N3;4=3MD;553M3;5L3LA;6C3L0;7:3K>;7O3JN;7;3JD;723J5;6N3IF;6H3I6;763HE;7G3H4;883GC;8I3G1;9:3FA;::3F2;;K3EF;=73E>;>;3E9;??3E3;@D3DO;AH3DI;BL3DE;D03D@;E43D:;F33D4;FO3CM;GG3CJ;H63D4;J53D?;KG3DJ;L<3E6;M23EA;MG3EL;N<3F8;O13FD;OG3FO;OC3GD<0A3GC<1?3GA<2=3G@<3:3G><483G<<563G;<643G9<723G8<8=3G><823FK<7F3F8<7;3EE<713EJ<6J3F2<6I3F><6H3FK<6G3G8<6F3GD<6N3GE<7J3G5<953FN<:A3FG<;L3F?<=73F8<>B3F2<?M3EJ<A93EC<B:3EB<C83EE<D53EH<E03EB<EL3E3<FG3DD<GB3D6<H=3CG<H93CE<HO3CC<IF3C@<J@3C;<K@3C3<LA3BK<MA3BD<NA3B;<OA3B4=0B3AK=1B3AD=2C3A<=3C3A4=4C3@L=5D3@E=6D3@<=7D3@5=8D3?L=9E3?E=:B3?<=:43>A=:33=A=;F3<4==33<:=?03<E=AC3<?=BB3;M=C=3;C=D93;9=E43;0=EA3:H=EJ3:B=F33:;=F<3:4=FE39M=FN39G=G739@=G@399=HE38M=J538>=K;37L=L@379=M536M=MG36B=N9366=NK35J=O=35>=OO353>0A34G>1334;>1E340>2733D>2I338>3732N>3D32?>4331L>4B318>5130E>5@301>5O2O>>6>2NK>6K2N9>7;2MN>7K2MC>8<2M8>8L2LL>9<2LA>::2L0>;:2K=><;2JJ>=I2J;>?=2J5>A22IO>BE2IH>D:2IB>EN2I<>GB2I6>I62I0>JJ2HI>L>2HC>N12H=>NJ2H0
1636@34326@H42J6A;42C6AO42<6BC4246C741M6CK41E6D?41>6E34176EG40O6F;40H6FO40A6GC4096H74026HK3OK6I>3OC6J63OF6JG3OK6K84016KH4066L:40;6LK40?6M?40A6N340A6NG40B6O:40D70541C70@42070K42=71642K71A43971L43E72744372C44A72N44N73845;73745J73646973@46G73O47474?47B74O47O75?48<75M48F76;48O76K49777;49>77K49F78;49N78J4:479;4:<79K4:D7::4:M7:H4;17;54;47;B4;77;O4;;7<@4;C7=14;L7=44;L7==4<47=E4<<7=M4<E7>54<L7>>4=57>F4==7?34=F7?D4=O7@54>97@J4>D7AD4?17B>4?=7C64?J7CF4@47D64@?7DF4@I7E54A47EE4A?7EH4A07F:4AI7FM4BF7G@4CC7GG4C<7GN4C57H64BN7H<4BC7I04B<7IC4B57J64AO7JI4AH7K;4AA7KN4A:7L24A87L44A>7L:4AK7M34AN7N04AO7NN4AO7OL4B080@4B58134B;81H4BA82;4BF82K4BL82L4C382M4C:82M4C<82L4C;82K4C;82J4C982I4C882K4C98314C<8384C@83=4CC83=4CD83G4CN8444D:84A4DG84N4E385;4E@85O4EF86E4EJ87<4EN8824F288I4F689?4F:8:64F>8:J4FB8;>4FG8<24FL8<F4G08=94G48=C4G58=L4G58>64G58>A4G68>K4G78?64G58?C4G28@04FN8@B4FI8AD4F@8B>4F68BI4EN8C44EF8C?4E=8CJ4E58D44DM8DD4DE8E74D>8EI4D;8F94D@8FI4DF8G34DM8G94E68GI4EL8HL4FL8ID4F68J<4EF8K54E78KN4DG8LF4D@8M>4D:8N54D58NC4D68O34D=8OC4DD9034DJ90C4E19134E791C4E>9234EE92C4EL9334F393C4F99434F@94C4FG9534FN95I4F@96>4F29704EC97B4E49854DD98L4D099C4C;9:;4BG9;24B29;I4A>9<@4@J9=84@69=O4?A9>@4?09?04>B9?A4>69@14=I9@A4==9A14=19A@4<E9AO4<99B>4;M9BO4;C9CA4;=9D24;99DA4;59E14;19E@4:M9F04:I9F@4:E9G04:A9G?4:=9GK4::9H24:89H84:69H?4:49HF4:29HL4:19I349O9I849K9IH49I9J649H9JD49G9K349F9KA49E9KF49:9KC48O9K@48C9K<4899K947O9K547E9K;47=9K94769KL4739L?4709M246N9MB46M9N146M9N>46E9NE46;9NK4619O145G9O945>9OA4569OJ44N:0244E:0:44<:0B444:0J43L:1<43L:1N43L:2I43N:3D440:4>442:59444:64445:6N448:7;445:7C441:7M43L:8543I:8>43D:85430:8=42I:8G42C:9042=:9:426:9E420::941O::M41N:;<42::;H42K:<543<:<B43L:<O44<:=;44M:=H45>:><455:?344H:?I44<:@?43O:A543C:AK437:BA42J:C742>:CM421:DC41E:E:418:EG410:F040H:F940D:FA40K:FI413:G141::G:41A:GB41H:GK41O:H6420:HJ41D:I>419:J240M:JF40A:K:405:KN3OJ:LB3O>:M:3O7:N73O4:O33O2;003NO;0M3NL;1I3NI;2F3NF;3?3NF;463N4;4N3MC;5E3M1;6<3L@;733KO;7G3K>;7M3JO;7:3JE;723J6;7?3IF;803I4;8A3HC;923H2;9?3GD;9L3G5;::3FG;:C3F<;;?3EO;<C3EJ;=G3EE;>K3E@;?O3E;;A33E6;B83E1;C;3DL;D;3DF;E73D?;F33D8;FH3D9;GK3DC;IK3DN;K@3E9;L53EE;LJ3F0;M@3F<;N53FH;NJ3G3;O43GD;OD3GJ<0A3GI<1?3GG<2=3GE<3;3GD<493GB<563GA<653G?<723G><803G<<9C3GF<983G3<8M3FH<8C3FO<893G6<7O3G<<7E3GC<703GD<6C3GH<6@3GG<803G@<9;3G8<:G3G1<<23FJ<==3FC<>I3F;<@43F5<A?3EM<BJ3EF<CI3EH<DF3EK<EC3EL<F>3E=<G83DN<H43D@<HM3D1<HF3D1<I73CO<IN3CL<JD3CI<K;3CF<L<3C=<M<3C6<N=3BN<O=3BF=0=3B>=1=3B6=2>3AN=3>3AG=4>3A?=5?3A7=6?3@O=7@3@G=8@3@?=9@3@7=9F3?D=983>I=8N3=M=:@3<@=;O3=5=>C3=0=A43<J=B53<3=C13;J=CL3;@=DG3;6=EB3:M=F>3:C=FH3:==G13:6=G:39O=GC39H=GL39A=H539:=HO391=J>38A=KD37O=LC37?=M5374=MG36H=N936<=NK360=O=35E=OO35:>0A34M>1334B>1F346>2833K>2J33?>3;333>3I32G>48323>4G31@>5530M>5D30:>632OF>6C2O3>722NA>7C2N3>832MH>8D2M=>942M2>9D2LF>:72L:>;82KG><92K4>=N2JL>?C2JF>A72J@>BJ2J9>D?2J3>F32IM>GG2IG>I;2I@>JO2I:>LC2I4>N02HL>NI2H?>OA2H1
1636?H43E6@<43>6A04376AD4306B842H6BL42A6C@42:6D44226DH41J6E<41D6F041<6FD4146G840N6GL40F6H@40>6I44086IL40;6J=40A6JN40E6K?40K6L04106LA4156M241:6ME41;6N:41<6NM41=6OB4286OM42F70943370D43@70O43N71:44;71E44H72045672;45D72G46173246>73346M73747;73G47I74748674F48B75448L75B49575O49>76?49F77049N77?4:577O4:=78?4:D78O4:L79?4;479O4;=7:=4;C7:K4;F7;84;J7;E4;M7<24<17<?4<47<K4<:7=24<B7=84<I7=@4=07=H4=97>04=@7>84=H7>@4>17?14>:7?I4>F7@C4?37A<4??7B64?L7C04@87CE4@D7D54@O7DE4A97E54AD7EC4AL7EE4A=7F44B07FH4BM7G74C>7G>4C87GF4C17GM4BK7H54BJ7HE4BF7I64B?7IH4B97J:4B27JM4AL7K@4AE7KC4AG7KI4B57L44B?7M24B@7N04B@7NM4BA7OK4BB80E4BD8184BI81M4BO82@4C58304C;8314CA8324CI8334D08334D38324D18314D18304D08324D08384D483>4D783@4D983A4D:83E4D>8424DK84?4E784L4ED8594F085N4F686D4F:87:4F>8814FB88G4FF89=4FJ8:24FO8:G4G38;:4G88;N4G<8<C4GA8=74GE8=B4GF8=L4GG8>54GH8>?4GH8>J4GH8?74GD8?E4GA8@24G=8@?4G98A:4G28B44FI8B>4F@8BI4F88C44EO8CB4EH8D54EA8DH4E:8E;4E38EM4DN8F>4E38FN4E98G>4E>8GN4EC8H<4EI8HN4FD8IF4F18J?4EE8K74E@8KO4E:8LG4E48M?4DN8N74DI8ND4DH8O34DJ8OC4E09024E790C4E>9134EE91B4EL9224F392C4F99334F@93B4FG9424FM94B4G49564G095L4FA96A4F39734EC97E4E59874DG98I4D799A4CB9:84BN9:O4B:9;G4AF9<>4A19=54@=9=L4?H9>>4?79>N4>F9?>4>59?O4=H9@?4=;9@O4<O9A?4<C9B04<69B@4;J9C04;C9C@4;@9D04;=9D@4;:9E04;69E?4;29EO4:M9F?4:J9FO4:F9G>4:A9GL4:>9H34:<9H:4:;9HA4:99HG4:79HN4:59I64:59IA4:69IK4:79J94:79JH4:59K64:49KD4:<9KD49D9K@4959K=48J9K:48?9K74839K447H9K947@9KH47@9L747@9LE47@9M547@9MC47@9N247=9N@4719NM46H9O846?9O=4659OC45J9OI45A9ON457:0444M:0:44C:0?448:0E43O:1@441:2;442:35445:40446:4K448:5E44;:6@44<:7644=:7?448:7H445:81440:8:43L:8B43H:8H43B:8I435:9242N:9<42G:9F42A:9O42::::424::N423:;=42>:;J42O:<643?:<C43O:=144>:=@44J:=O456:>@45;:?7451:?L44E:@B449:A943M:AO43@:BE434:C;42G:D142;:DG41N:E=41B:EM418:F6411:F?412:FG419:FO41@:G741G:G@41N:GI424:H4426:H?428:HL428:I:428:IN41M:JB41A:K6415:KJ40I:LC40B:M?40@:N<40=:O840:;05407;143OK;233O?;303O7;3E3O7;413NK;4A3N9;523MF;5B3M4;623LA;6>3KN;6D3L3;713KH;7E3JC;823J5;8@3IF;8M3I8;9;3HJ;9H3H;;:63GM;:C3G>;:C3G<;;23FH;;N3F6;=23F1;>63EL;?:3EG;@?3EB;AC3E=;BC3E7;C?3E0;D<3DI;E83DB;EH3DH;GA3E3;I@3E>;K83EH;KN3F4;LC3F@;M83FK;MM3G7;NB3GB;NF3H1;OD3H0<0B3GN<1@3GM<2>3GK<3;3GJ<493GH<573GG<653GE<733GD<813GB<8O3GA<:I3GN<:?3GN<903GC<8C3GG<863GK<7I3GO<7<3H4<6O3H8<6N3H2<7I3GK<9>3GC<:M3G;<<83G4<=C3FM<>O3FF<@:3F><AE3F8<C13F1<D93EK<E73EN<F43F1<FO3EG<GK3E8<HE3DI<I;3D=<I43D=<I>3D;<J53D8<JK3D5<KA3D3<L83D0<M73CH<N83C@<O83C8=093C1=193BI=293BA=393B9=4:3B1=5:3AI=6:3AB=7;3A9=8;3A2=973@H=8I3?L=8<3?1=7O3>6=993=>=;B3=A=>63=;=@F3=4=AF3<8=BD3<1=C?3;G=D;3;==E63;4=F13:J=FL3:@=GE3:7=GN3:0=H739J=HA39C=HJ39<=I8394=JH38E=KM383=LC37F=M637:=MH36O=N936C=NK367=O>35L>0035@>0B354>1434H>1F34=>28341>2J33E>3<33:>3N32N>4<32<>4L31I>5<318>5M30G>6=307>6M2OE>782O0>7<2N5>7L2M<>8A2LN>9;2M7>9K2LL>:<2L@>;52L1><@2KC>>42K=>?H2K7>A<2K0>C02JJ>DD2JD>F82J>>GL2J8>I@2J1>K42IK>LH2IE>MO2I;>NH2HM>O@2H@?092H3
1636?=4496@14426@E43K6A943C6AM43<6BA4356C542M6CI42F6D=42?6E14276EE4206F941I6FM41A6GA41:6H54136HI40K6IA40O6J34156JD41:6K541@6KF41E6L741J6LH4206M84246ML4266N@4276O042M6O;43;6OF43I70144570=44C70H45171345>71>45K71I46972446G72@47472K47A72O47O73?48=73L48G74:49174H49;75649D75D49N7644:576D4:=7744:D77D4:L7844;478D4;=7944;E79D4;M7:34<67:@4<97:M4<<7;;4<@7;H4<C7<64<G7<G4<L7<O4=37=64=;7=<4=B7=C4=I7=I4>07>04>87>64>>7>H4>H7?A4?57@;4?A7A54?N7AO4@:7BH4@G7CB4A47D44A?7DD4AI7E44B47E@4B87EB4AI7EO4B77FC4C47FN4C:7G64C37G=4BM7GE4BL7GN4C27H=4BN7HO4BH7I@4BB7J14B<7JB4B57K34B17K94B>7K?4BK7L34C17M14C27MO4C27NL4C27OJ4C380G4C481>4C88224C>82E4CD8354CI8364D18374D78374D>8384DE8394DI8384DH8374DG8384DG83>4DJ83C4DM83D4DO83D4E083E4E283O4E<84<4EI84I4F58564FA85L4FF86B4FJ8794FN87O4G288F4G689;4G;89O4G?8:C4GD8;74GH8;K4GM8<?4H18=34H68=A4H88=K4H88>44H98>>4H:8>L4H78?94H38?F4GO8@34GK8@@4GH8A14GD8AI4G;8B44G28B@4FJ8C34FC8CF4F<8D94F68DL4EO8E?4EI8F24EB8FB4EG8G24EL8GB4F28H24F78HB4F<8I24FB8IH4F?8J@4F98K84F38L04EN8LH4EH8M@4EC8N84E=8NF4E<8O34E:8OB4E=9024ED90B4EK9124F291B4F99224F?92B4FF9324FM93B4G49424G;94B4G?9594G195O4FB96E4F39774ED97H4E698:4DG98L4D899>4CJ9:54C59:L4BA9;D4AL9<;4A89=24@C9=J4?O9><4?>9>L4>M9?<4><9?M4=K9@=4=;9@M4<L9A=4<@9AN4<49B>4;H9BN4;@9C>4;?9CN4;=9D?4;:9DO4;89E?4;69EN4;39F>4:N9FN4:J9G>4:G9GM4:C9H44:A9H;4:?9HB4:=9HK4:>9I54:>9I@4:@9IJ4:A9J44:C9J?4:D9JJ4:D9K94:K9KH4;39KB4:09K>49;9K;4909K848E9K54899K948A9K;4839KJ4839L84829LG4829M64829ME4829N347J9NA47?9NO4749O<46K9OH46B9OO469:0545N:0:45D:0@45;:0F451:0K44G:1<44@:2044;:2G449:3B44;:4=44=:5744?:6244A:6M44C:7;44A:7C44<:7L448:85444:8>440:8F43L:8O43H:9443@:9>439:9G433::142L::;42F::E42@::N429:;>42A:;K431:<;43=:<J43I:=9445:=G44B:>644N:>F45::?:454:@044J:@F44?:A<445:B243J:BH43=:C?431:D542D:DK428:EA41L:F341A:F<419:FE41?:FM41G:G541M:G>424:GI426:H4428:H>42::HJ42<:I742=:IE42=:J242=:J?42=:K2424:KK41M:LI41G:MH41;:NG40O:OG40C;0F407;1E3OK;2@3OF;373OH;3M3OH;453OC;4B3O5;513NB;5=3MO;5I3M;;653LG;6A3LC;743L?;7I3KJ;8<3JL;8I3J>;973IO;9D3I@;:13I2;:?3HC;:D3H=;:C3H<;:N3GN;;D3G2;<>3F=;=B3F8;>F3F3;?J3EO;@K3EH;AG3EB;BD3E;;C@3E3;D<3DM;E83E7;G73EB;I63EM;K13F8;KF3FD;L;3FO;M13G;;MF3GF;MI3H9;NG3H7;OE3H6<0B3H4<1A3H3<2>3H1<3<3H0<4:3GN<583GM<663GK<733GJ<813GH<903GH<:13H0<9H3GM<9<3H2<8O3H7<8B3H:<853H?<7H3HC<7>3HE<7?3H=<7@3H7<933GO<:H3GG<<<3G?<=I3G8<?53G1<@@3FI<AK3FB<C73F;<DB3F4<EH3F0<FE3F3<GB3F1<H<3EB<I73E3<II3DH<IB3DH<IF3DG<J=3DD<K33DA<KI3D?<L@3D<<M63D9<N33D3<O33CK=033CC=143C;=243C3=353BK=453BD=563B;=663B4=763AL=863AD=8;3A0=7M3@5=7@3?:=733>?=8@3>1=;43=K==I3=F=@83=>=A73<B=B73<8=C23;N=CM3;D=DI3;;=ED3;1=F?3:G=G;3:==H63:4=HL39K=I539D=I>39>=IG397=K138I=L2388=LD37L=M637A=MH375=N:36I=NL36>=O>362>0035G>0B35:>1434O>1F34D>29347>2K33L>3>33A>42335>4F32L>5632<>5F31K>6731:>6G30I>77308>7=2O>>7B2NE>7G2MJ>872M1>8I2L7>9<2KD>:82L0>;42L:><E2L3>>92KM>?M2KG>AA2KA>C52K:>DI2K4>F=2JN>H12JH>IE2JA>K92J;>LM2J5>MN2IJ>NG2I<>O?2HO?082HB
1636?244N6?F44F6@:44?6@N4486AB4406B643I6BJ43B6C>43:6D24336DF42L6E:42D6EN42=6FB4266G641N6GJ41G6H>41?6I641C6IJ41J6J:41O6JK4246K<42:6KM42?6L>42D6LO42I6M?42O6N34366N>43C6NI4406O444>6O?44K6OJ45970645F70A46470L46@71846M71C47971O47F72:48372F48@73348L73A49673O49@74=49I74K4:37594:=75I4:D7684:L76H4;47794;<77H4;D7884;M78H4<57984<=79H4<F7:64<L7:C4<O7;04=27;>4=57;L4=:7<>4=?7<L4=E7=34=L7=94>47=@4>;7=F4>B7=M4>I7>:4?17>I4?:7?>4?E7@44@07@M4@=7AG4@I7BA4A67C:4AC7D34AO7DC4B97E34BD7E=4BE7E?4B67EJ4B>7F>4C;7FE4C57FM4BO7G54BN7G>4C57GG4C:7H64C77HG4C17I84BJ7II4BD7J:4B=7JG4BH7JO4C57K54CB7L34CB7M04CC7MN4CD7NK4CD7OI4CD80F4CE81C4CG8274CL82J4D283:4D883:4D?83;4DF83<4DM83=4E383>4E:83?4E@83>4E>83>4E>83E4EB83F4EC83G4EE83H4EF83H4EH83M4EM84:4F984G4FE8544G285J4G686A4G:8774G?87M4GB88C4GG8974GK89K4H08:?4H48;34H98;G4H>8<<4HB8=04HG8=@4HJ8=J4HJ8>34HJ8>?4HI8>M4HE8?:4HA8?G4H>8@54H;8@B4H78@O4H38A?4GM8B14GF8BD4G?8C74G88CJ4G28D=4FK8E04FD8EC4F=8F64F68FF4F:8G64F?8GG4FE8H74FJ8HG4FO8I84G28IN4FO8JB4FL8K94FG8L14FB8LI4F<8MA4F68N94F18NG4EO8O54EN8OB4EL9014F190B4F89124F?91A4FE9214FL92B4G39314G:93A4GA9414GH94F4G@95<4G29624FB96H4F497:4EE97L4E698=4DH98O4D999A4CK9:34C;9:I4BH9;A4B39<84A?9<O4@K9=G4@79>94?D9>I4?49?:4>C9?J4>29@:4=B9@K4=19A;4<@9AL4<29B<4;F9BL4;?9C<4;=9CL4;;9D<4;99DL4;79E<4;49EL4;29F=4;09FM4:M9G=4:K9GL4:G9H64:E9H@4:F9HJ4:G9I44:G9I=4:G9IF4:H9IO4:H9J94:H9JB4:H9JL4:J9K<4;B9KG4;B9K@4:<9K;49B9K84979K;49:9K@49B9KB49?9K>48F9KK48E9L:48E9LI48D9M848D9MG48C9N44889NB47L9O047A9O>4779OJ46N:0746F:0A46<:0G462:0L45H:1245>:1?457:23451:2F44L:3:44G:3O44B:4I44D:5C44E:6?44H:7644H:7?44E:7G44@:8144<:89448:8B444:8K440:9643L:9?43J:9H43D::243>::<437::E430::O42J:;842C:;D42?:<342I:<B436:=143B:=@43N:=O44::>>44F:>M452:?>458:@444N:@J44C:A@449:B643O:BL43E:CB43::D842N:DN42B:EE425:F:41I:FB41F:FK41M:G3423:G>426:GI428:H342::H>42=:HI42?:I442@:IB42@:IO42A:J=42A:JJ42B:K;42?:L;423:M:41G:N941::O940N;0840B;17406;1O406;2F407;3=408;44409;4>405;4G3OJ;543O6;5@3NC;5M3MN;6:3M<;6I3LN;7:3LD;7M3LA;8B3KK;933K5;9@3JF;9M3J8;:;3II;:E3I=;:D3I<;:D3I<;:I3I3;;?3H8;<53G<;<M3FE;>13F?;?33F:;@03F3;@L3EL;AH3EE;BD3E>;C<3E<;DM3EF;FM3F2;HL3F<;JI3FH;K?3G3;L43G>;LI3GJ;M>3H6;MI3H?;NG3H=;OE3H<<0C3H:<1A3H9<2?3H7<3<3H6<4:3H4<583H3<663H1<773H6<8<3HD<9@3I3<:93HG<:53H=<9H3HB<9;3HF<8N3HJ<8A3HO<843I3<803I0<813HI<823HB<8G3H:<:<3H3<<13GJ<=E3GC<?:3G;<@F3G4<B23FL<C=3FE<DH3F><F33F7<G63F6<H33F8<HN3EL<II3E=<J73E4<J03E3<IN3E3<JD3E0<K:3DM<L13DK<LG3DH<M=3DE<N43DC<NO3D=<OO3D6=0O3CM=203CF=303C==403C6=513BN=613BF=723B>=7L3B4=7?3A8=713@>=6D3?C=6C3>J=833><=:G3>7==;3>1=?I3=I=@I3<M=AJ3<?=BE3<5=CA3;L=D<3;B=E73;8=F23:N=FN3:D=GI3:;=HE3:1=I@39G=J339?=J<398=K:38L=L238>=LD383=M637G=MH37<=N:370=NL36D=O?369>0235M>0E35B>18357>1K34K>2>34?>31344>3D33H>4733>>4L337>5@32O>6032=>6A31L>7131<>7>30H>7C2ON>7H2O3>7L2N9>822M?>8B2LF>942KL>9J2KD>:H2L7><D2L=>>>2L>>@22L7>AF2L1>C:2KK>DN2KE>FB2K?>H62K8>IJ2K2>K>2JL>M22JF>MM2J9>NF2IK>O>2I>?072I0?0O2HC
1636>G45A6?;45;6?O4536@C44K6A744D6AK44=6B?4456C343N6CG43G6D;43?6DO4386EC4316F742I6FK42B6G?42;6H34236HK4266I?42>6J142C6JA42I6K242N6KC4336L44396LE43>6M643C6MD43L6N14426N<44>6NH44K6O34586O?45D6OJ46170646=70B46J70N47771947D71E48172048>72=48J72J49673749C73D49O7414:874?4:B74N4:L75=4;475M4;;76=4;C76M4;L77=4<477M4<=78=4<F78L4<N79=4=679L4=>7:94=A7:F4=E7;34=H7;C4=M7<54>27<F4>87=04>>7=64>F7==4>M7=D4?47=O4?<7>>4?E7>M4?N7??4@77@54@C7@J4@M7A@4A97B94AE7C34B27CL4B>7DB4BJ7E24C57E:4C27E<4BB7EE4BD7F54C87F=4C27FE4C17FN4C77G74C<7G@4CA7GN4C?7H@4C97I14C37IB4BL7J04C27J;4C>7JD4CK7K44D47L24D47M04D47MM4D57NJ4D67OH4D780F4D781C4D782<4D:82O4D@83>4DG83?4DM83A4E483B4E;83C4EB83C4EI83D4F083E4F683E4F683H4F883I4F983J4F;83K4F<83L4F>83M4F?8484FJ84D4G78524GB85I4GF86?4GK8764GN87K4H288@4H78934H<89G4HA8:<4HE8;04HJ8;D4HN8<84I38<L4I78=?4I;8=I4I<8>34I;8>A4I78>N4I48?;4I18?I4HM8@64HI8@B4HG8@O4HD8AE4HA8B54H;8BH4H48C;4GN8CN4GG8DA4G@8E34G98EG4G28F:4FL8FJ4FM8G:4G28GK4G88H;4G>8HK4GC8I?4GB8J44G?8JI4G;8K=4G88L34G58LJ4G08MB4FK8N:4FE8NI4FC8O64FB8OD4FA9014F?90A4FE9114FL91A4G39214G:92A4GA9314GH93A4GN9424H094I4GA95?4G29654FC96K4F597=4EF97O4E798A4DI9934D:99D4CK9:64C<9:H4BN9;>4B;9<54AF9<L4A29=D4@=9>74?K9>G4?:9?74>J9?H4>99@84=H9@H4=89A94<G9AI4<69B94;F9BJ4;=9C;4;<9CK4;:9D<4;89DM4;69E=4;49EN4;29F?4;09G04:N9G@4:L9GO4:L9H94:J9HB4:J9HL4:J9I54:J9I>4:J9IG4:J9J14:J9J:4:J9JC4:J9JN4:M9K>4;I9KI4<99KB4;19K=4:29KA4:;9KF4:C9KK4:K9KI4::9KE49A9KM4979L<4979LJ4979M:4969MH4909N648E9NC4899O147N9O?47C9OL47::09471:0E46I:1246@:19466:1A45M:2545G:2I45C:3=45>:41458:4E453:5944N:6044L:6K44N:7:44L:7C44H:7L44D:8644@:8B44=:8M449:98446:9B443:9I442::143O::;43H::D43A::N43;:;8434:;A42N:;L42I:<;42F:<J42M:=8439:=G43E:>6442:>F44>:?444J:?C456:@7451:@N44G:AD44=:B:442:C043I:CF43>:D<433:E242I:EH42?:F>422:FH423:G3426:G>428:GI42::H342=:H>42>:HI42@:I442C:I?42D:IM42E:J:42E:JG42F:K942C:KM42>:LL423:MK41G:NK41::OJ40N;0H40D;1?40E;2640F;2M40G;3D40I;4;40J;4G40F;5140:;5=3OF;5J3O2;673N?;6F3N1;763MC;7E3M7;843LI;8F3LB;9;3KN;9I3K=;:73JO;:D3J@;:E3J=;:E3J<;:D3J<;:G3IL;;=3I3;<23H:;<G3GA;=;3FK;>83FE;?43F>;@03F6;@L3EO;AH3EH;BD3EK;DC3F6;FB3FA;HB3FL;J@3G7;K73GB;KL3GN;LA3H:;LL3HF;MJ3HE;NH3HC;OF3HB<0C3H@<1B3H?<2?3H=<3=3H<<4;3H:<5=3HC<6B3I3<7G3IA<8L3IO<:03J><:@3I?<:A3HM<:43I1<9G3I6<9:3I:<8M3I?<8A3IC<8B3I<<8B3I4<8C3HM<8D3HF<:03H><;E3H7<=:3GN<>O3GG<@C3G><B83G7<CC3G0<DN3FH<F:3FB<GE3F:<HD3F;<I@3F6<J;3EG<JE3E?<J>3E?<J83E?<JL3E<<KB3E9<L83E7<LO3E4<ME3E1<N;3DO<O23DL<OJ3DH=0K3D@=1K3D8=2K3D0=3L3CH=4L3C@=5L3C9=6M3C1=703B<=6B3AB=653@F=6=3?O=7;3?<=873>I=::3>B=<N3><=?:3>4=@;3=7=A;3<?=B93<<=C43<2=CO3;I=DK3;?=EF3;5=FA3:L=G<3:B=H83:8=I339N=IN39D=JJ39;=K@391=L238E=LE38:=M937O=MK37C=N>377=O136L=OD36@>07365>0J35J>1=35>>20353>2C34G>3634<>3I340>4=33G>5233@>5G339>6:330>6K32?>7;31N>7C316>7H30=>7N2OB>822NH>872MN>8<2M3>8O2L@>9H2LC>:A2LF>;=2L:>>92LC>@72LH>AK2LB>C@2L<>E32L6>FG2KO>H<2KI>J02KC>KC2K=>M42K5>ML2JH>NE2J;>O=2IM?052I??0N2I2?1D2H?
1626=H46=6><4666?045N6?D45G6@845@6@L4586A@4516B444I6BH44B6C<44;6D04436DD43L6E843E6EL43=6F@4366G342O6GH42G6H@42K6I64336IG4396J843>6JH43B6K943H6KJ43M6L;4426LJ44:6M744C6MD44K6N14516N=4586NI45A6O545N6O@46;6OL46H70847570C47B70O47N71:48;71F48H72349472@49A72M49M73:4:973G4:E7444;174B4;;7524;C75B4;K7624<476A4<<7724<E77A4<M7814=578A4=>7914=F79A4=O79O4>47:<4>77:I4>;7;:4>?7;K4>E7<=4>K7<M4?17=34?77=:4??7=D4?G7>34@07>C4@87?24@A7?A4@J7@54A57@K4A?7AA4AK7B74B57BM4BA7CD4BM7D>4C97E14CD7E74C>7E94BO7E@4BK7EL4C47F44C37F>4C97FG4C>7G04CD7G94CJ7GG4CH7H84CA7HI4C;7I94C<7ID4CH7J04D57J:4DB7K34DE7L14DF7LO4DG7ML4DG7NI4DG7OG4DH80E4DI81B4DI82@4DJ8344DO83D4E583E4E<83F4EC83F4EI83G4F083H4F883I4F?83J4FE83K4FL83L4FN83M4FO83M4G183N4G283O4G48404G58454G;84B4GH8514H285G4H786>4H;8744H>87G4HC88<4HG8904HL89D4I18:84I58:L4I:8;@4I?8<44IC8<I4IH8==4IM8=H4IM8>54IJ8>B4IG8>O4IC8?<4I@8?H4I>8@54I<8@A4I:8@N4I78AJ4I38B<4HN8BL4HI8C?4HB8D24H;8DE4H48E84GN8EK4GH8F=4GA8FO4GA8G?4GF8GO4GK8H?4H18I04H48IE4H18J:4GN8JO4GK8KC4GH8L94GE8LM4GB8MC4G?8N;4G98NJ4G78O74G58OE4G59024G390A4G29104G991@4G@9204GG92A4GN9304H493@4H;9454H194L4GB95B4G39684FD96N4F697A4EF9824E898D4DJ9964D:99H4CL9:94C=9:K4BN9;=4B@9<24AM9<J4A99=A4@D9>54@29>E4?A9?54?19?F4>@9@64=O9@F4=?9A64<N9AG4<A9B94<59BN4;I9CC4;?9D44;=9DD4;:9E54;99EF4;79F64;59FG4;39G84;19GG4;19H24;79H<4;59HD4:M9HM4:L9I64:L9I@4:L9II4:L9J24:M9J;4:M9JE4:M9JO4;09K@4;L9KJ4<@9L24<O9KN4<39KM4;C9L14;L9L44;M9L14;49KM4:;9KO49I9L=49I9LM49I9M;49I9MI49>9N74929NE48G9O248<9OA4809ON47F:0:47=:0G475:1446L:1A46C:2646<:2J467:3>462:4345M:4G45H:5;45C:5O45>:6C459:77454:7B451:7M44N:8944J:8D44G:8O44C:9;44?:9E44<:9L44;::344::::448::C442::M43L:;743E:;@43?:;J438:<5433:<C430:=142N:=@431:=O43=:>>43I:>M445:?<44B:?K44N:@;455:A144K:AG44A:B=446:C443L:CJ43B:D@437:E642M:EL42C:FB429:G1429:G>42::GI42=:H342>:H>42@:HI42C:I442E:I>42F:IJ42I:J742I:JE42I:K642F:KK42B:L?42=:M=422:N=41F:O<41:;08414;0O414;1E416;2=417;33418;3J419;4A41:;50417;5:40J;5G405;643OB;6C3O4;723NG;7B3N9;813ML;8@3M>;8O3M0;9?3LD;:23L4;:@3KF;:F3K>;:F3K=;:F3JN;:G3J=;:J3IK;;?3I2;<43H9;<I3G@;=G3GC;>I3GN;>83FH;?43FA;@13F:;@M3F3;B:3F;;D93FE;F83G1;H73G;;J63GG;K03H2;KE3H=;L23HL;LM3HM;MK3HK;NH3HJ;OF3HH<0D3HF<1B3HE<2@3HC<3E3I2<4I3I@<5N3IO<723J=<873JK<9<3K;<::3K;<:G3J7<:M3I=<:@3IA<:33IF<9F3IJ<993IN<923IN<933IF<943I@<953I9<953I1<9E3HJ<;:3HB<<O3H:<>D3H2<@83GJ<AM3GB<CA3G;<E43G3<F@3FK<GK3FE<I53F><J23F@<JM3F1<K33EK<JL3EK<JF3EK<K33EH<KJ3EF<L@3EC<M63E@<ML3E><NC3E;<O93E8<OO3E6=0F3E3=1F3DJ=2G3DC=3G3D;=4G3D3=5H3CK=6A3C@=643BE=5F3AJ=683A5=753@A=823?N=8O3?;=9M3>M=<A3>G=>L3>==?L3=A=@L3<E=AL3<C=BG3<9=CB3;O=D>3;F=E93;<=F43;3=G03:I=GK3:?=HF3:5=IB39K=JB39>=KD38N=L838I=LJ38A=M>386=N137K=NC37?=O7373=OJ36H>0<36<>0O361>1C35F>2535:>2H34O>3;34D>3N348>4C340>5733I>5L33B>6A33;>75332>7D32@>7I31F>7N30K>83301>882O7>8>2N=>8D2MB>9;2M6>:32M:>:K2M=>;C2MA><J2K5>?N2LH>B12M3>CE2LL>E82LF>FM2L@>HA2L:>J52L3>KI2KM>M32KD>MK2K7>ND2JI>O<2J<?042IN?0M2IA?1C2HL?232GJ
1616<I4786==4716>146J6>E46B6?946;6?M4646@A45L6A545E6AI45>6B=4566C144O6CE44H6D944@6DM4486EA4426F543J6FI43B6G=43<6H543?6HL43H6I=43M6IN4426J?4486JO44=6K@44B6L044I6L=4526LJ45;6M845D6MD45J6N14616N=4676NJ46=6O646F6OA4726OM47?70947L70D48971048E71<49271I49>72649K72C4:77304:C73=4;073J4;=7474;J74G4<37564<<75F4<D7664<L76F4=57764==77F4=F7864=O78E4>77964>?79E4>F7:24>J7:?4>M7;04?27;B4?87<44?>7<E4?D7=14?J7=94@17=H4@:7>84@C7>G4@L7?64A57?F4A>7@64AF7@K4B27AA4B=7B74BH7BM4C37CC4C>7D94CI7E04D47E54CK7E74C;7E;4C27ED4C57EM4C;7F64CA7F@4CG7FI4CL7G24D17G?4D17H04CJ7HA4CF7HM4D37I94D@7IE4DL7J44E67K34E67L04E77LN4E87MK4E97NH4E97OF4E:80D4E:81B4E;82?4E;83:4E=83I4EC83I4EJ83J4F183K4F883L4F?83L4FG83J4G283H4G=83N4G>83O4GA8404GE8414GG8414GH8424GJ8434GK8444GM84@4H884O4HC85E4HG86<4HJ8704HO87D4I48884I888L4I=89@4IA8:54IF8:I4IJ8;=4IO8<14J38<E4J88=94J=8=H4J=8>54J;8>A4J88>N4J68?;4J38?G4J18@44IO8@@4IM8@M4IJ8AK4IE8BK4I@8C44I=8CC4I78D64I18DH4HJ8E<4HC8EO4H<8FA4H58G34H48GC4H98H34H>8HC4HD8I64HD8IK4HA8J@4H>8K54H;8KJ4H88L?4H58M44H28MH4GO8N=4GK8NK4GK8O94GJ8OF4GH9044GG90B4GE9104GF91@4GM9204H492@4H;9304HA93B4H@9484H194O4GB95E4G496;4FE9714F697D4EG9864E998G4DJ9994D;99K4CM9:=4C>9:N4BO9;@4BA9<24B29<G4A@9=>4@L9>24@99>B4?H9?34?79?C4>G9@44>69@D4=J9A54==9AF4=19B;4<E9C04<99CE4;M9D;4;B9DM4;=9E=4;;9EN4;:9F?4;79FO4;69G?4;59GJ4;;9H54;@9H@4;@9HG4;99HO4;19I84:N9IA4:N9IJ4:N9J44:O9J=4:O9JG4:O9K14;29KA4;O9KJ4<@9L14<O9L94=?9L:4=69L=4=59L<4<H9L84;O9L54;59L14:<9L?4:;9LN4:;9M=4:79MJ49K9N849@9NF4959O448I9OB48>:00483:0<47I:0I47@:16478:1J46N:2B46D:3846=:3M468:4A463:5545N:5I45I:6=45D:7145?:7D45;:80458:8;454:8G450:9244M:9=44I:9H44F:9O44E::644C::=44B::D44@::L44<:;6446:;?43O:;I43I:<243B:<>43<:<L43::=:438:=H436:>6435:>E43A:?443M:?C449:@244E:@A451:A544O:AK44E:BA44::C7440:CM43E:DC43;:E:431:F042F:FF42A:G742A:GD42B:H142C:H=42C:HI42E:I442F:I>42I:IJ42K:J542L:JB42M:K442J:KH42F:L=42A:M142=:MN422:NN41F:OG41C;0>41D;1541E;1L41G;2C41G;3:41I;4141J;4H41K;5941I;5D419;6040E;6?408;6O3OJ;7>3O<;7M3NO;8=3NA;8L3N3;9;3MF;9K3M8;::3LK;:G3L>;:H3KN;:I3K=;:J3JK;:K3J:;:M3II;;B3I0;<73H7;<L3GF;=O3H7;?23HC;@53HN;?D3GJ;?53FE;@03F?;B03FJ;CN3G4;EM3G@;GM3GJ;IL3H6;JH3HA;K=3HM;KO3I4;LM3I3;MK3I1;NI3I0;OG3HN<0F3I1<1K3I@<303IN<453J<<593JK<6>3K:<7C3KH<8H3L7<9L3LE<:A3L2<:N3JO<;93IL<:L3J1<:@3J5<:23J9<9E3J><9C3J9<9D3J2<9E3IK<9F3ID<9G3I<<9G3I6<:O3HM<<C3HF<>83H><?M3H6<AA3GN<C63GG<DK3G><F?3G7<H13FO<I<3FH<JC3FD<K?3F;<KA3F7<K:3F6<K43F6<K;3F5<L13F2<LH3EO<M>3EL<N43EJ<NJ3EG<OA3ED=073EB=0M3E?=1D3E<=2B3E5=3B3DM=4B3DE=5C3D==5E3CI=583BN=633B:=703AG=7M3A4=8I3@@=9G3?M=:C3?:=<43?2=>>3>H=?=3=L=@>3=0=A?3<J=B:3<@=C53<6=D13;L=DL3;C=EG3;9=FB3;0=G>3:F=H?3:7=IA39H=JB39:=KC38K=LD38<=M138;=MC388=N6382=NI37G=O<37<=OO370>0B36D>14369>1H35M>2;35B>2M357>3A34K>4434@>4H349>5=341>6233K>6G33C>7;33<>7K32O>81325>8731:>8>30@>8D2OG>8J2NL>952N5>:12MK>:K2MF>;C2MJ><;2MM>=22K>>>?2K:>AC2LM>CJ2M=>E>2M6>G22M0>HF2LJ>J:2LD>KN2L>>M22L3>MJ2KF>NC2K8>O;2JK?032J>?0L2J0?1A2I9?212H7
1616<>47L6=247E6=F47=6>:4766>N46O6?B46G6@646@6@J4696A>4616B245J6BF45C6C:45;6CN4546DB44M6E544E6EJ44>6F>4476G143O6GJ4426HB44<6I344B6ID44G6J544L6JF4516K54586KB45A6L045J6L>4636LK46<6M846C6MD46J6N14706N=4766NJ47=6O647C6OB47I6ON48670:48C70F48O71349<71@49H71L4:47284:C72D4;172O4;@73:4;O73O4<974B4<A7504<J75>4=375K4=<76:4=E76K4=N77;4>677J4>>78:4>G78J4>O79:4?779H4?<7:54?@7:G4?E7;94?K7;J4@07<<4@67<N4@<7==4@D7=M4@M7><4A67>K4A?7?;4AH7?J4B17@:4B:7@L4BD7AB4BN7B84C:7BN4CE7CC4D07D94D;7DJ4E07DO4EA7E24DJ7E74CM7E>4CA7EF4CC7EO4CI7F84CN7FB4D47FK4D:7G84D97GI4D37H54D=7HB4DJ7HO4E67I?4E@7J04EH7JI4EJ7KH4EJ7LG4EK7MF4EK7NE4EK7OD4EL80C4EL81A4EL82>4EM83;4EM83M4F283N4F983N4FA83L4FL83J4G783G4GB83E4GM83J4GO8414GO8434H28444H68444H:8454H>8454H@8464HA8474HC84=4HJ84N4I385D4I78684I;86L4I@87@4ID8854II88I4IM89=4J28:14J68:F4J;8;;4J@8<04JC8<E4JG8=<4JL8=H4JO8>44JM8>A4JK8>M4JI8?:4JF8?F4JD8@34JB8@?4J@8@L4J=8AL4J88C44J18CB4IM8CJ4IK8D:4IF8DL4I?8E@4I88F24I18FE4HJ8G74HG8GH4HL8H84I28HH4I78I<4I48J14I18JF4HN8K;4HK8L04HH8LE4HE8M:4HB8MN4H>8NB4H:8O24H88O=4H;8OH4H<9064H;90C4H:9114H891?4H;91O4HA92?4HH92O4HN93E4HA94;4H29524GC95H4G596>4FE9744F797G4EH9894E:98J4DK99=4D<99N4CN9:@4C>9;24C09;D4BB9<54B29<G4AD9=;4A39>04@@9>@4?O9?04??9?A4?39@24>F9@C4>:9A44=N9AF4=A9B<4=59C14<I9CG4<>9D<4<29E24;F9EE4;>9F64;=9FG4;:9G64;:9GB4;@9GM4;E9H84;K9HC4;L9HK4;D9I24;<9I:4;49IB4;19IL4;19J54;19J>4;19JH4;19K24;59KB4<09KI4<@9L14<O9L94=?9L@4=O9LG4>;9LC4=B9L@4<I9L<4<09L94;79LA4:M9M04:M9M>4:D9ML4:89N949M9NG49A9O54969OC48K:0148?:0>486:0K47L:1:47C:23479:2L46O:3F46D:4;46>:4O469:5C464:6845O:6L45J:7@45E:8345A:8>45>:8I45::94457:9@453:9K44O::244N::944L::@44K::G44J::N44H:;544G:;>44@:;H449:<2443:<<43L:<F43F:=443D:=B43B:>043?:>>43=:>M43;:?<43E:?K442:@944>:@I44J:A9453:AO44H:BE44>:C;444:D143I:DG43?:E=435:F342K:FJ42J:G=42I:GI42J:H642K:HC42L:HO42L:I<42L:II42M:J442O:J?431:K142N:KF42I:L:42E:LN42@:MB42<:N@421:O7422:ON423;0E424;1<426;22426;2J428;3A429;4742:;4O42;;5B42;;5M41I;6<41:;6L40M;7;40?;7J402;893OD;8I3O6;983NH;9H3N9;:73MI;:G3M:;:N3LE;:L3KL;:M3K;;:N3JI;:O3J8;;03IF;;E3HN;<:3H5;=43H@;>83HL;?:3I7;@>3IC;A@3IN;A53HO;@F3GG;AE3G9;CD3GD;EC3GO;GB3H:;IA3HE;J@3I0;K23I<;L03I:;LN3I9;ML3I6;NM3I?<023IM<173J<<2<3JK<3A3K9<4E3KH<5J3L6<6N3LD<833M4<983MB<:<3MN<:H3LJ<;53KF<;B3JB<;83JA<:L3JE<:?3JI<:53JK<:53JE<:63J=<:73J6<:83IO<:83IH<:93IA<:C3I9<<83I2<=M3HJ<?A3HB<A63H:<BJ3H2<D?3GJ<F43GB<GI3G;<I=3G2<JN3FK<L13FE<KO3FB<KI3FB<KA3FB<KB3FA<L93F><LO3F;<ME3F8<N<3F6<O23F3<OH3F0=0?3EN=153EK=1K3EH=2B3EE=3>3E?=4>3E8=563DL=513D3=5N3C@=6K3BL=7H3B9=8D3AF=9A3A2=:>3@?=;;3?L=;K3?===O3?3=>O3>7=?O3=:=A13<M=AN3<G=BI3<==CD3<3=D?3;J=E;3;@=F<3;1=G=3:C=H>3:4=I@39E=JA396=KC38H=LD388=M>37O=MI37O=N<37K=NN37H=OA37C>04378>0G36L>1:36@>1M365>2@35J>3335>>3F353>4934I>4N34A>5B34;>68343>6L33L>7A33E>8733>>8=32D>8C31J>8I310>8O306>952O<>9J2NL>:F2NA>;B2N6><<2N7>=32N:>>12L@>?=2L:>@I2L4>C92M3>EC2MG>G72MA>HK2M;>J?2M5>L32LN>M12LB>MI2L5>NB2KG>O:2K:?022JL?0K2J??1?2IF?202HD
1616<348@6<G4896=;4826=O47J6>C47C6?747<6?K4746@?46M6A346F6AG46>6B;4676BO4606CC45H6D745A6DJ45:6E?4526F344K6FF44C6G?44F6H84516HI4566I:45;6IK45A6J;45G6JH4616K64696KC46B6L146K6L>4736LL47<6M847C6MD47I6N147O6N=4866NJ48<6O648B6OC48I6OO48O70;49;70F49J7124:871=4:G71H4;57244;D72?4<372O4<?73G4<G74>4<O74L4=975:4=B75G4=K7644>476B4>=76O4>F77?4>N77O4?778?4?@78N4?H79=4?N79L4@37:=4@87:O4@=7;A4@C7<44@I7<F4A07=;4A87>04AA7>@4AI7?04B27??4B<7?O4BE7@>4BM7@M4C67AB4CA7B84CL7BN4D77CD4DB7D94DO7D?4F47DJ4FF7E14F97E74EA7E>4E67ED4DJ7EK4D@7F14D67F;4D<7FD4DA7G04DA7G>4DG7GK4E37H;4E=7HL4EG7I;4F17IM4F97JE4F<7K>4F>7L=4F>7M<4F?7N;4F?7O:4F?8094F?8184F?8274F?8364F@83K4FE83K4G183I4G<83G4GG83E4H283B4H=83E4HA83M4HA8444HA8474HC8484HG8484HJ8494HN8494I384:4I784:4I984;4I;84L4IC85@4IG8654IL86I4J087=4J58834J988H4J=89=4JA8:24JF8:G4JI8;<4JN8<14K28<G4K68=?4K:8=G4K?8>34K?8>?4K>8>L4K<8?84K98?E4K78@14K58@>4K28@K4K08AM4JJ8C54JC8D14J?8D94J<8DA4J98E04J38EC4IM8F64IG8FI4I@8G;4I:8GL4I@8H<4IE8HN4IG8IC4ID8J84IA8JL4I>8KA4I;8L64I88LK4I48M?4I18N34HL8NG4HG8O:4HD8OE4HF9004HH90;4HJ90E4HM9124HL91@4HJ91O4HN92?4I59324I193H4HA94>4H39554GD95K4G596A4FF9774F897J4EI98<4E:98N4DL99@4D=9:24CN9:C4C?9;54C19;G4BB9<94B39<J4AE9=<4A69=N4@H9>?4@<9>O4?O9?@4?C9@14?79@B4>J9A34>>9AH4>29B>4=F9C34=:9CH4<N9D>4<C9E34<69EI4;K9F>4;?9FN4;?9G94;D9GE4;J9H04<09H;4<59HF4<79HN4<09I64;G9I=4;@9IE4;79IM4;39J64;49J@4;49JI4;49K44;89KA4<09KI4<@9L04<O9L84=?9L@4=K9LH4=O9LK4=M9LG4=B9LD4<J9L@4<19LB4;@9M14;=9M?4;19MM4:F9N;4:;9NH49O9O749D9OD498:0248M:0@48B:0M489:1D47O:2=47D:36479:40470:4I46E:5=46?:6246::6F466:7:461:7O45L:8@45G:8L45D:9745@:9B45=:9M459::6458::=456::C454::J453:;1451:;8450:;?44O:;G44K:<144D:<;44=:<D447:<N441:=<43N:=J43L:>943I:>G43G:?543E:?C43B:@243I:@A445:A044B:A?44N:B244L:BH44B:C?448:D543M:DK43B:EA439:F8434:FO433:GB433:GO433:H;433:HH434:I5435:IB436:IN436:J;436:JG439:K9438:KN433:LC42N:M842I:ML42C:ND42C:O;42D;0342E;0J42F;1B42F;2942G;3042I;3G42I;4>42K;5542L;5K42L;6942=;6H420;7841A;7H411;8740A;8F402;963OC;9F3O4;:53NE;:E3N6;;43MF;;73LN;;33L2;;13K8;;23JG;;33J5;;43ID;;H3HL;<<3H;;==3I5;>@3I@;?C3IL;@F3J7;AI3JB;BL3JO;BL3J6;CD3IC;D=3I0;E93H?;G83HJ;I73I5;J93I@;K33IB;L03I@;M53IL;N93J<;O>3JJ<0B3K8<1G3KG<2L3L5<413LD<553M3<6:3MA<7?3N0<8C3N><9H3NL<:C3NF<:O3MA<;<3L><;I3K:<;D3K1<;83K5<:K3K9<:F3K7<:G3K0<:G3JH<:H3JB<:I3J:<:J3J3<:K3IL<:K3IE<;L3I=<=A3I6<?63HM<@K3HF<B?3H=<D43H6<EI3GN<G>3GF<I23G><JG3G6<L93FN<L=3FN<L73FN<L03FN<KJ3FM<L@3FK<M73FH<MM3FE<NC3FB<O:3F?=003F<=0F3F:=1<3F7=233F4=2I3F2=3?3EO=493EJ=4L3E8=5I3DE=6E3D2=7B3C>=8?3BK=9<3B8=:93AE=;53A2=<23@?=<O3?K==A3?==>A3>A=?@3=D=@A3<L=A@3<N=B<3<D=C73<:=D93;L=E:3;<=F<3:N=G=3:?=H>3:0=I@39A=JA393=KB38D=LC385=ME37F=N037D=NB37A=O537>=OG37;>0:378>0L374>1?36I>2236=>2E361>3835F>3K35:>4?351>5334J>5I34C>6=34<>72345>7G33N>8=33H>8H334>8N329>9431@>9:30F>9C2ON>:?2OB>;<2O8><82NL>=42NC>=I2N6>?12MB>@<2M<>AH2M6>C42M0>DN2M9>G=2N2>I02ML>JD2MF>L82M?>M02M1>MH2LC>NA2L7>O92KI?012K;?0J2JN?1>2J3?1N2I1?2J2H5
1616;H4956<<48M6=048E6=D48?6>84876>L47O6?@47H6@447A6@H4796A<4726B046K6BD46D6C846<6CL4656D@45N6E345F6EH45?6F;4586G445:6GO45F6H?45K6I04616I@4676IN46?6J;46H6JI4716K647:6KD47C6L247L6L?4856LL48<6M848B6MD48H6N148O6N=4956NI4986O549;6O@49>6OK49M7064:<70B4:J70M4;97194;H71D4<671O4<E72G4<M73?4=57474=>74H4=G7564>075C4>97604>B76>4>K76K4?47794?>77F4?G7834@078C4@87934@@79C4@E7:44@K7:G4A17;:4A77;L4A<7<?4AC7=44AL7=I4B57>=4B=7?24BF7?C4BN7@34C87@B4CA7A24CJ7AC4D37B94D>7BO4DI7CD4E47CN4F37D:4FA7DF4FN7E14GB7E74G57E>4FJ7ED4F?7EK4F37F14EI7F84E=7F>4E27FH4E07G84E:7GH4ED7H84EO7HH4F97I84FB7IJ4FK7JB4FM7K:4FO7L34G27M24G27N14G27O04G27OO4G380N4G481N4G582M4G58394G?83>4GK83C4H783B4HB83?4HM83A4I483H4I383O4I38474I284;4I484;4I784<4I<84<4I@84=4IC84=4IG84>4IK84?4IO84J4J385?4J78654J;86J4J@87?4JC8844JG88I4JL89>4JO8:44K48:I4K88;>4K<8<34K@8<M4KD8=@4KH8=H4KN8>34KO8>>4KO8>K4KO8?74KL8?D4KJ8@04KH8@=4KE8@J4KC8AN4K=8C64K68D>4JO8DH4JL8E04JJ8E94JG8EG4JB8F:4J;8FM4J48G@4IM8H04J38H@4J88I44J78II4J48J>4J18K24IN8KG4IK8L<4IG8M04IC8MD4I>8N84I:8NL4I58O@4I18OM4I19084I490C4I690M4I89184I:91C4I=91O4I=92?4I@9354I193K4HB94A4H49584GE95N4G696D4FG97:4F997N4EI98?4E;9924DM99C4D=9:54CO9:F4C@9;94C29;J4BC9<<4B49<K4AI9=:4A?9=J4A49><4@I9>M4@>9??4@39@04?G9@A4?:9A44>N9AI4>B9B?4>69C44=K9CI4=?9D>4=59E24<J9EG4<@9F;4<59G04;M9G<4;N9GH4<49H34<99H?4<?9HJ4<C9I14<;9I94<39I@4;K9IH4;C9J04;;9J84;69JA4;69JK4;69K54;;9KA4<09KH4<@9L04<O9L84=89L@4=<9LG4=A9M04=E9LO4==9LK4=29LH4<G9LD4<29M34;J9M@4;?9MN4;49N<4:H9NJ4:=9O84:19OF49F:0449::0B48N:1448D:1N489:2G47O:3@47E:4947::52470:5L46F:6@46@:7446<:7H467:8=462:8N45N:9945J:9E45G::045C::945A::@45?::G45>::M45<:;445;:;;45::;B458:;I457:<0454:<944N:<C44H:<M44A:=744;:=E448:>3445:>A443:?0441:?>43O:?L43L:@:43I:@I43M:A8449:AF44E:B6450:BL44E:CB44;:D8441:DN43F:EE43=:F<43<:G343<:GH43;:H443<:HA43<:HM43<:I:43=:IG43>:J343=:J>439:JK43B:K943J:KK43J:L@43E:M443?:MI43::NA43;:O843<;0043<;0H43=;1?43>;2643?;2N43?;3F43@;4=43A;5;439;69430;6F42K;7642<;7E41M;8541=;8D40N;9440?;9D400;:33O@;:C3O1;;23NB;;B3N3;;?3M7;;<3L;;;93K@;;73JE;;83J3;;83IB;;J3HJ;<>3I2;=F3II;>I3J5;?L3J@;@N3JK;B23K8;C43KC;D43L4;E43KI;F43K9;G53JH;H43J7;HM3ID;J63IL;K;3J;;L@3JI;ME3K8;NI3KF;ON3L4<133LD<273M2<3<3M@<4A3MO<5E3N=<6J3NL<7O3O;<943OI<:9408<:J3O=<;63N9<;C3M5<<03L1<<13K@<;D3KD<;73KI<;73KB<;83K;<;93K4<;:3JM<;:3JE<;;3J?<;<3J7<;=3J0<;A3II<=63IA<>K3I9<@?3I1<B43HJ<CI3HA<E=3H:<G23H2<HG3GJ<J;3GB<K43G><K93G=<K?3G<<KD3G;<KO3G:<LG3G;<M=3G9<N43G7<NJ3G4<O@3G2=063G0=0L3FN=1B3FK=283FI=2O3FG=3E3FE=4<3F?=5:3EL=673E9=743DF=813D2=8M3C?=9J3BL=:G3B8=;D3AE=<A3A2==;3@8=>93?==?03>3=?C3=@=@83=4=A43=5=B63<E=C73<7=D83;H=E:3;9=F;3:J=G<3:<=H=39M=I?39>=J@38O=KB38A=LC382=ME37C=NF374=NH377=O;374=OM371>0@36M>1236J>1D36G>2736D>2J369>3=35N>4035B>4E35:>59353>5O34L>6D34E>7934>>7N347>8C340>9333C>9932I>9?31O>9E315>:930D>;5309><12ON><M2OC>=I2O8>>E2NI>@02ND>A<2N>>BH2N8>D32N2>E?2ML>FK2MF>I52N<>JJ2N5>L72MN>LO2M@>MG2M3>N@2LE>O82L8?002KJ?0I2K=?1<2J@?1L2I>?2J2HC
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
1616;24:<6;F4:56<:49N6<N49F6=B49?6>64986>J4906?>48I6@248B6@F48:6A:4836AN47L6BB47D6C647=6CJ4766D=46N6E146G6EF46@6F>46B6G:46O6GL4746H947=6HG47F6I447O6IA4886J048A6J>48H6JL4906K:4986KH49?6L249C6L=49F6LH49J6M449M6M?4:06MK4:46N74:76NC4:=6NM4:N6O84;?6OC4<06ON4<>70:4<L70G4=:71?4=B7274=K7304>273H4>;74@4>C74N4>L75;4?575H4?>7654?G76C4@07714@:77>4@C77K4@L7874A778A4AE7944AL79F4B27:84B87:K4B>7;>4BD7<14BK7<F4C47=:4C=7=O4CE7>D4CN7?84D77?M4D?7@B4DH7A64E07AJ4E97B:4EB7BM4F27C94F<7CF4FE7D34FM7D>4G>7DA4GD7DC4GI7DE4GN7DI4GI7DO4G<7E54G07E;4FC7EL4F>7FI4F?7G44F@7GA4FH7H14G27HB4G;7I24GE7ID4GM7J<4H07K44H27KL4H57LC4H97M>4H<7N=4H=7O=4H>80<4H>81<4H?8214HD8264I182;4I=82@4II82H4J18354J483D4J683N4J68454J684=4J584C4J684C4J984D4J=84D4JA84E4JF84D4JG84B4JI84B4JK8544JO85F4K48694K886M4K<87B4K@8874KD88L4KH89A4KL8:74L18:L4L58;A4L98<=4L=8=94L@8=B4LF8=J4LK8>24LO8>>4M08>J4M08?64M18?A4M08?N4LN8@:4LK8@H4LH8B04LB8C84L;8D@4L58EF4KO8EN4KL8F64KI8F>4KF8FF4KC8G54K?8GH4K88H94K98HK4K:8I@4K78J54K48JI4JO8K=4JL8L24JG8LF4JC8M:4J>8MN4J98NB4J68O64J18OJ4IM90>4II90I4IJ9134IM91>4IO91I4J19234J392D4II9364I;93H4HN94:4HA94M4H495>4GF9644G896J4FI97@4F:9844EK98F4E=9984DN99I4D@9:84D69:F4CK9;54CA9;E4C69<34BK9<B4B@9=14B59=C4AK9>54A?9>G4A69?;4A09?O4@I9@C4@C9@K4@<9A@4@19B44?G9BI4?<9C=4?19D24>G9DF4><9E:4>29F04=G9G04==9G=4=99GK4=49H94<O9HE4=39I04=99I84=29I?4<I9IG4<B9IO4<99J64<29J>4;J9JF4;B9JM4;:9K84;@9K@4;M9KG4<19KO4<69L74<:9L?4<?9LG4<C9LO4<H9M74<L9M94<I9M;4<@9M?4<79MD4;O9MN4;C9N=4;59NK4:H9O:4:<9OG4:1:0349L:0>49H:0I49B:1:49;:26491:3348F:4048<:4L480:5F47F:6@47<:79471:8146H:8E46C:9946>:9I46:::5467::?464::F462::M461:;445O:;:45N:;A45M:;H45K:;O45J:<645H:<=45G:<D45E:<K45C:=545<:=>455:=H44O:>544L:>D44J:?244G:?@44E:?N44B:@<44@:@K44>:A944;:AG448:B5446:BD44B:C444M:CI44C:D?448:E6440:EM43O:FD43N:G;43L:H343L:H?43L:HL43M:I843M:ID43I:IO43F:J943B:JG43K:K4443:KA44<:KO44E:L<44M:LN44M:MC44I:N;44J:O344K:OJ44K;0G44E;1E44<;2D444;3B43L;4@43B;5>43:;6<432;7;42I;7H42N;81435;8@42F;90426;9@41G;9O417;:?40H;:N409;;>3OJ;;M3O;;<23ND;;O3MI;;L3LM;;I3L1;;G3K6;;D3J:;;A3I?;;O3IG;=43JG;>73K2;?83KA;@73L2;A63LD;B63M4;C93MD;D<3N3;E93N<;E;3N6;F13M@;FG3LK;G=3L6;H23KA;HL3L;;K>3M3;N13MI;OJ3N<<0N3NK<233O9<383OH<4=406<5A40D<6F414<7K41B<90420<:442?<:K420<;840L<;E3OH<<13NE<<>3M@<<I3L@<<<3LD<<93L@<<:3L9<<;3L1<<<3KK<<=3KC<<=3K<<<>3K5<<?3JN<<@3JG<<A3J?<<A3J9<>43J0<?H3II<A=3IA<C23I9<DF3I1<F;3HI<H03HA<HD3H><HJ3H=<HO3H<<I@3H;<K83H:<LM3H:<M?3H=<N23H@<ND3HD<O63HG<OI3HJ=0?3HI=153HF=1K3HD=2A3HB=373H@=3N3H==4L3GI=5J3G5=6H3FB=7F3EO=8B3E<=9?3DH=:<3D5=;93C@=<73BG==83B6=>:3AD=?;3A2=@=3@@=A>3?O=A93?:=A13>E=A<3>0=B@3=;=CD3<F=DI3<2=EM3;==G23:H=H63:4=I:39?=J>38J=KA389=LB37K=MD37<=NE36M=OF36>>0A363>0:36<>0L369>1>366>21363>2D360>3535N>3:36>>3L369>5235N>5F35E>6;35>>70357>7E351>8:34I>8O34B>9D34;>:033H>:632O>:H32<>;D321><@31F>=<31:>=M30I>>?307>?A2ON>@K2OG>B:2OD>CI2OA>E82O>>FH2O<>H72O9>IG2O7>K62O4>L52NK>LM2N?>ME2N1>N>2MC>O62M6>OO2LH?0G2L;?182K:?1I2J8?2I2I?
1616:G4;16;;4:I6;O4:A6<C4:;6=74:36=K49K6>?49E6?349=6?G4956@;48O6@O48G6AC48?6B74896BJ4816C?47I6D347B6DF47;6E;4736F34766FO47C6G@47K6GN4836H<48:6HJ48B6I848I6IG4916J54986JC49@6K149H6K<49L6KD4:06KN4:46L94:76LE4:;6M14:>6M<4:A6MH4:D6N44:M6N>4;>6NI4<06O34<@6O>4=06OJ4=?70?4=H7174>171O4>972H4>A73@4>I7484?174I4?:7574?D75D4?M7614@676?4@?76M4@H77:4A177D4A?77L4AO78;4B878M4B>79?4BE7:14BK7:C4C27;64C77;J4C?7<?4CH7=34D17=H4D97>=4DB7?14DJ7?F4E37@:4E<7@O4EE7AD4EN7B:4F77BH4F;7C64FB7CB4FK7CO4G37D44G;7D64GA7D84GF7D:4GK7D<4H17D@4GN7DF4GB7DM4G67EJ4G77FG4G87G74G:7G@4G;7GN4GC7H>4GM7HN4H77IA4H>7J94HA7K14HD7KH4HG7L@4HL7M84I07N44I27O44I38034I48124I481?4I=81D4IJ81I4J68224J=82?4J@82L4JB83;4JD83K4JF8484JG84?4JG84F4JF84G4JJ84G4JO84G4K284F4K384D4K584C4K784G4K98594K>85L4KB86>4KF8704KJ87C4KO8894L288N4L789C4L;8:94L?8:N4LC8;G4LG8<C4LK8=;4LO8=C4M48=J4M:8>24M?8>>4M@8>J4M@8?54MA8?A4MA8?M4M@8@94M>8@I4M;8B14M48C94LN8DA4LH8EI4LA8F<4L=8FE4L:8FM4L78G54L58G>4L28GL4KL8H=4KL8I14KJ8IF4KF8J:4KB8JN4K>8KB4K98L74K58LK4K08M?4JL8N34JH8NG4JC8O;4J?8OO4J:90C4J69114J691<4J991F4J;9214J=92B4J39344IE93F4I89484HJ94J4H>95<4H095N4GB96@4G59724FH97D4F:9884EL98F4EA9964E699E4DL9:34DA9:B4D79;14CL9;@4CA9;O4C79<>4BL9<N4BA9=B4B:9>64B49>I4AM9?=4AF9@14A@9@E4A99@L4A39A:4@K9AN4@@9BC4@69C74?K9CK4?A9D@4?69E44>K9EL4>A9FM4>69G=4>19GK4=L9H94=H9HG4=B9I34=C9I;4==9IC4=59IJ4<M9J24<E9J94<=9JA4<59JF4<09JK4;K9K04;J9K74<19K>4<89KE4<>9KL4<E9L34<L9L:4=29LA4=99LK4=49M>4<J9MI4<?9MN4<59N34;M9N74;D9NC4;79O34:J9O?4:B9OK4:>:064:8:0A4:3:0L49N:1<49H:2849=:35492:4248G:4N48<:5K482:6H47G:7B47=:8<472:9446I:9H46D::846@::B46=::I46;:;046::;7469:;>467:;D466:;K464:<2463:<9461:<@461:<G45O:<N45N:=445L:=>45G:=G45@:>1459:>>456:>L454:?:451:?I44N:@744L:@E44J:A344H:AA44E:B044B:B>44@:BL44>:C;44E:CM44G:DC44<:E:448:F1448:FH446:G@445:H7444:HE445:I0443:I;43O:IG43L:J243I:J>43K:JK444:K844<:KF44E:L344M:LA456:LN45>:M@45A:N=45;:O;452;0944J;1844B;26448;34440;4243H;5143?;5O436;6M42N;7K42F;8:42I;8C431;8N432;9>42C;9M423;:=41D;:L415;;<40F;;K406;<;3OG;<:3NM;<73N1;<43M6;<23L:;;O3K>;;L3JC;<83J6;<;3JE;=?3K=;>B3KL;?D3L;;@H3LJ;AK3M:;BN3MI;D23N8;E53NG;EC3NA;EH3N<;EC3N8;ED3N0;F;3M;;G03LF;FN3M2;HG3MJ;K:3NB;ML3O8<0?400<2D40D<3H413<4M41A<62420<7642><8;42L<9@43<<:E43J<;242H<;?41D<;L40@<<83O<<<F3N8<=23M4<<J3M2<<K3LK<<L3LD<<L3L=<<M3L6<<N3KN<<O3KH<=03K@<=03K9<=13K2<=23JK<=33JD<=H3J<<?=3J5<A23IL<BF3IE<D;3I<<F03I5<G:3HO<G@3HM<GE3HM<H93HL<J13HK<KG3HJ<M93HK<MK3HN<N>3I1<O13I4<OC3I8=053I;=0H3I>=1:3IA=1O3IA=2E3I?=3;3I<=453I2=543H>=623GJ=703G7=7N3FC=8K3F0=9H3E==:E3DF=;G3D4=<H3CB==H3C0=>D3B==?@3AI=@<3A5=A23@E=A<3@4=A43??=@M3>J=B13>5=C53=@=D:3<K=E>3<7=FB3;B=GG3:M=HK3:9=IO39D=K438O=L838;=M<37E=N@371=OE36<>0G35L>1335I>1235O>1E35L>1K36:>1O36I>24379>2937H>3636L>4F36@>61365>6D35K>7735B>7K359>8A352>9634K>9J34D>:<348>:B33>>;=332><932G><M327>=?31F>>1314>>D30D>?K30<>A2304>B@300>CO2ON>E>2OK>FN2OG>H=2OE>IM2OB>K32O;>L42O:>LL2NM>MD2N@>N=2N2>O52ME>ON2M7?0F2LI?172KG?1G2JD?2H2IL?422I9
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
1636974?D69J4??6:>4?96;14?46;D4>N6<74>H6<K4>C6=>4>=6>14>86>D4>26?74=L6?K4=E6@B4=:6A:4<O6B04<D6BG4<96C?4;N6CN4;B6D54;56DG4;26EE4;76F<4;@6FM4;L6GD4<66HA4<<6HM4<A6I64<E6I@4<I6IK4<L6J44=06J>4=46JH4=86K24=<6K<4=@6KK4=O6L;4>>6LK4>O6M:4??6MJ4@06N;4@@6NJ4A06O:4A@6OJ4B170:4BA70J4C27144C@71>4CN71M4D772>4D?72O4DF73A4DK7424E074D4E67554E<75G4EA7684EF76J4EL77<4F177M4F678B4F@79:4FN7:14G;7:H4GH7;@4H67<74HC7<N4I17=F4I>7>>4IH7?34IN7?@4IK7@34J37@B4J07A04IM7A?4IJ7AM4IG7B:4IF7BB4IG7BJ4IH7C24II7C;4IJ7CB4IM7CG4J07CM4J37D34J67D94J97DL4J@7ED4JI7F54K17FE4K67G44K97GC4K>7H24KB7HD4KF7I<4KK7J24L27JH4L87K=4L>7L44LD7LL4LH7MD4LM7N34M07N@4M27NM4M57O:4M77OG4M;8034M>80?4MA80N4MC81>4MD81M4MG82=4MJ82L4MM8394N183F4N58424N984<4N=84=4N=84@4N;84K4N;85<4N=85N4N@86?4ND8704NI87A4NM8824O188C4O58934O889B4O:8:14O<8:@4O>8;04O@8;E4OD8;N4OJ8<84OO8<C5058=35038=H4OL8>=4OD8?84OC8@94OJ8A;5018B<5088C?50?8DA50H8ED50O8F150O8F650N8F@50N8FK5118G55128G@5148GK5158H55178H@5198HJ51;8I751:8IL5148JA50N8K650H8KL50B8LA50<8M65058MK4OO8NA4OI8O64OC9044O99194NM92=4N@92O4MN92>4M192@4L?92N4L693;4KL93I4KC9474K:94E4K09534JG95A4J>9624J296B4IG9744I:97E4HO9854HC98F4H89984GH99J4G89:<4FG9:M4F99;<4EO9<14F29<E4F49=94F69=N4F89>B4F:9?74F<9?K4F>9@@4F>9@N4F89A54F29A<4EL9AC4EF9AK4E@9B24E:9B@4E19C44DG9D24D<9E34D19F44CG9G54C<9GJ4C59GL4C69GF4C:9G?4C<9G74C59G04BL9FH4BE9F@4B<9F74B59EO4AL9EN4AD9F:4A99FF4@O9G74@E9GL4@<9H@4@29I44?I9II4??9J=4?69K14>L9KF4>C9L:4>99LN4=O9MC4=G9N84=?9NL4=89OA4=6:054=5:0I4=5:1>4=4:224=3:2D4=2:364=2:3I4=1:4;4=0:4M4=0:5C4<L:6@4<C:714;M:7B4;7:844:B:8E49L:96497:9G48B::847L::I476:;746O:;C471:;N473:<;475:<G476:=2478:=>47::=J47<:>647>:>A47@:>N47B:?947C:?A47C:?H47A:?O47@:@947<:@G479:A5476:AG474:B<471:C246N:CG46K:D<46H:E146F:EG46C:F<46@:G146=:GG46::H;468:HI465:HO460:I545K:I<45F:IB45A:II45<:J2456:JN44N:KI44E:LC44H:M<44K:N644N:NO451:OH454;0B458;1;45;;2445>;2L454;3C44K;4;44A;53447;5J43M;6A43C;79439;80430;8H42F;9D42=;:B424;;@41L;<<41E;<E41M;<N425;=642=;=@42E;=H42M;>1436;>:43<;>J430;?<42F;?N42<;@@422;A241H;AD41>;B641C;BI41B;C=41B;D141A;DE41@;DN412;E;40L;EL40K;F:40E;FD40;;FO402;G:3OI;GG3OC;H43O<;HB3O5;HO3NO;HO3NK;JA3ON;LM41B;M8414;M=408;MC3OL;MJ402;N1406;N840;;N?40?;NE40D;NL40I;OL41=<13423<2A41G<3N41:<5:413<6340G<6O40:<7=403<7K3OL<8:3OE<8H3O><963O7<9D3NO<:33NI<:F3NA<;83N9<;K3N3<<>3MK<=03MC<=C3M<<>53M5<>H3LM<?;3LF<?M3L><A@3L5<CB3L2<DO3L:<FB3L><H:3L=<J13L=<KH3L<<M?3L<<O63L;=013L>=0D3LA=163LD=1F3LD=203L>=2:3L9=2E3L4=2O3KN=383KH=3A3KA=3I3K9=423K2=4:3JK=4C3JC=5<3J;=6;3J1=7;3IG=8:3I>=9<3I1=:B3H>=;@3GJ=<43G:=<E3FK==53F<==G3EM=>73E>=>H3DO=?93D@=?J3D1=@;3CB=@L3C3=A03B?=B53AK=C93A6=D=3@A=E?3?M=E:3?D=E53?;=E03?3=DK3>J=DF3>A=E13>1=F53=A=GH3=3=I;3<E=JM3<7=L@3;I=N03;;=O53;1>0:3:G>1>3:>>2C3:4>3K39J>5839>>6E393>8138H>9>38=>:D38A>;=388><5380><N37H>=F37?>>>377>?:36L>@836?>A635N>B435=>C334M>D134<>DO33L>EN33;>FK339>G=32A>GH31@>I4312>J=30G>KC30>>LJ304>MB30A>N7315>NO30M>OG30@?0:2OD?0K2NB?1<2M??2F2LL?402L:?5:2KF?6D2K3?7O2J??992IL?9K2IF?:;2IA
16368D4@L6974@F69J4@@6:=4@;6;04@66;C4@06<64?J6<J4?E6==4??6>04?86>H4>N6??4>B6@64>86@M4=L6AD4=A6B;4=66C24<K6CG4<@6CN4<36D54;F6DD4;@6EB4;F6F74<06FH4<;6GD4<C6HA4<J6I>4=16IL4=56J54=96J?4==6JI4=A6K34=E6K=4=I6KG4=N6L74>>6LG4>N6M74?>6MG4?O6N74@?6NG4A06O64A@6OF4B07004B?70;4BM70D4C;70N4CI7184D771G4DA7284DI72J4E073:4E773K4E?74<4EF74N4EK75@4F07614F676C4F<7754F@77F4FE78=4G37954G@79L4GN7:C4H;7;;4HH7<34I37<J4I=7=C4IG7>:4J07>G4IJ7>O4I>7?B4IF7@64IM7@H4J47A;4J:7AI4J77B24J87B;4J97BC4J:7BK4J<7C34J=7C;4J>7CC4J?7CK4J@7D44JA7D94JE7DF4JJ7E94K27EI4K97F94K>7FH4KB7G74KF7GL4KK7HE4KO7I<4L47J24L;7JH4LA7K>4LH7L34LN7LI4M47M=4M;7MI4M>7N64MA7NC4MD7O04MF7O>4MH7OJ4MK8064MN80F4N18154N381E4N68244N882D4N;8304N?83<4NC83I4NH8454NL8494NM84>4NJ84J4NJ85;4NM85M4O086>4O386O4O687@4O;8814O?88B4OC8914OE89@4OG89O4OI8:>4OK8:M4OL8;:5008;D5058;O50;8<950@8<M50:8=B5028>B5088?C50?8@D50E8AF50L8BG5148CJ51;8DL51C8EF51H8EK51E8F651G8F@51I8FK51K8G551L8G@51N8GK51O8H55218H?5238HJ5248I45268IH5218J=51K8K251D8KH51>8L=5188M25128MG50K8N=50E8O650>90;50291@4OF92E4O993:4NH92I4MJ92;4LN92I4LE9374L<93E4L29424KI94@4K@94O4K695>4JL95O4J@96@4J59714II97B4I=9834I198C4HF9954H899G4GH9:94G79:K4FF9;>4F?9<24FB9<G4FC9=;4FE9>04FH9>D4FI9?84FL9?M4FN9@B4FO9A04FO9A74FI9A>4FB9AE4F<9AL4F69B34F09B:4EJ9BN4E@9CN4E59DO4DJ9F04D@9G14D69GI4CN9GC4D19G;4CJ9G34CA9FL4C:9FD4C29F<4BJ9F34BB9EK4B:9EC4B29E@4AI9EL4A?9F>4A69G24@L9GF4@C9H;4@99HO4@09ID4?G9J84?>9JN4?79KB4>O9L74>H9LL4>@9MA4>89N64>09NJ4=I9O?4=H:034=H:0H4=G:1<4=F:204=E:2E4=E:394=D:3M4=C:4B4=B:564=B:5L4=>:6;4<K:6L4<6:7=4;@:7N4:J:8?4:5:9049?:9A48J::2485::D47@:;446O:;?470:;K472:<7474:<C476:<O478:=;479:=G47<:>247=:>>47?:>J47A:?647C:?B47E:?N47F:@947H:@A47F:A247C:AG47@:B=47=:C147;:CE479:D9477:DM475:EB473:F6471:FJ470:G>46M:H246L:HF46J:I246G:I846C:I>46=:IE468:IK464:J245N:JH45F:KD45>:L@455:M;44M:N644K:NO44N:OH451;0B454;1;457;2445:;2N45<;3E452;4=44H;5444?;5L445;6C43K;7;43A;82437;8J42M;9A42D;:942:;;2420;<141H;<N41@;=741H;=@420;=I428;>242@;>:42J;>;43E;>>44=;??43F;@843A;@N439;A@42O;B242E;BB42?;C742@;CK42@;D7424;D@41G;DJ419;E:418;EK417;F<417;FL414;G740K;GA40A;GK407;H63OM;H@3OC;HL3O;;I93O4;I?3NO;K7402;KH403;L@40;;M@40I;MH40O;MO414;N6419;N<41>;NC41D;NI41I;O041N;O7423;OJ423<1641J<2C41?<3O414<5<40I<6340H<6F40A<78409<7J402<8=3OK<903OC<9B3O<<:53O5<:H3NM<;:3NF<;M3N><<@3N7<=23N0<=E3MH<>73MA<>J3M:<?=3M2<@63LJ<AM3LA<D>3L5<F33L:<G@3LB<HL3LJ<JA3LM<L83LL<MO3LL<OF3LK=0>3LN=113M1=1>3LO=1H3LJ=223LD=2<3L>=2E3L7=2M3KO=363KH=3>3KA=3G3K9=3O3K2=483JK=4A3JD=5>3J:=6=3J1=7=3IG=8<3I==9<3I4=:;3HJ=;63H?=;G3H0=<83GA=<I3G2==93FC==K3F4=>;3EE=>L3E6=?=3DG=?N3D8=@?3CI=@M3C9=AE3BD=BJ3B0=CD3A>=C?3A5=C:3@L=C53@C=CH3@1=D<3??=E13>M=EE3>:=FK3=K=H=3=<=J03<N=KC3<?=M43<2=N93;H=O>3;>>0B3;5>1G3:K>2L3:A>413:7>5<39M>6I39A>86396>9B38K>:D395>;<38M><538E><M38<>=F384>>A37I>?>37<>@;370>A936A>B8360>C635@>D4350>E234C>EO34A>FL34>>GB33M>GD326>HA31@>J0313>K?30D>LG30;>MC30B>N8315>NN31=>OF30O?09301?0I2NO?1;2MN?2F2M:?402LG?5:2L3?6D2KA?7N2JN?982J:?9L2J3?:<2IN
16468C4AN6964AI69I4AC6:<4A=6:O4A76;B4A26<64@L6<M4@@6=D4@66>;4?J6?24?@6?I4?46@@4>I6A74>>6AN4>36BE4=H6C=4==6CG4=16CN4<D6D54<86DA4;O6E?4<46F34<?6FH4<I6GE4=06HA4=76I>4=?6J;4=E6JJ4=J6K44=N6K>4>26KG4>66L34>=6LC4>N6M34?>6MC4?N6N34@?6NC4@O6NM4A=6O74AL6OA4B:6OJ4BH7044C670?4CD70H4D37124D@71B4DK7234E372D4E:7354EA73F4EI7474F074H4F87594F?75J4FE76;4FK76M4G077A4G87884GE78O4H379H4H>7:?4HG7;84I17;O4I;7<G4IE7=?4IO7>74J97>;4IF7>?4I27?24I97?G4ID7@<4J07A04J<7AA4JC7AH4JL7B04JN7B84JO7B@4K07BI4K17C14K27C94K37CA4K57CI4K67D14K67D<4K67DM4K:7E>4KB7EM4KF7F<4KK7G44KO7GM4L37HF4L77I<4L>7J24LD7JH4LJ7K>4M17L34M77LF4M@7M74MJ7MC4ML7MO4MO7N:4N17NG4N47O44N77OA4N:7ON4N<80=4N?80L4NA81<4ND81L4NF82;4NJ82G4NM8334O183@4O683L4O98484O<84<4O984H4O:85:4O<85K4O?86=4OB86N4OD87?4OI8804OL88@4OO88O50289>50389M5058:<5088:K50:8;950<8;;50A8;E50F8<150H8<K50E8=L50L8>N5138?O51;8A051A8B251H8C351O8D55278E852>8EA52>8EK52@8F652A8F@52C8FK52D8G552F8G@52G8GJ52J8H552K8H?52M8HJ52N8I45308ID52M8J952G8JN52A8KD52;8L95258LN51O8MC51H8N951B8O>51690B50J91G50>92L50293E4OB9344ND92C4MG92E4M49334LK93@4LB93N4L994<4KO94J4KF95;4K:95L4JO96=4JC96N4J797?4IK9804I@98A4I49924HH99D4H89:64GH9:G4G79;E4G09<94G29<M4G49=B4G69>74G89>K4G:9??4G<9@44G>9@>4G:9@G4FM9@N4F@9A74FA9AA4FB9AJ4FD9B44FE9B;4FA9BJ4F99CK4EO9DL4ED9EL4E49FL4DD9GC4DA9G:4D99G24D19FI4CH9FA4C@9F84C89EO4BO9EG4BG9E?4B?9E84B79E14AO9ED4AF9F:4A>9FN4A69GC4@O9H84@F9HM4@?9IB4@79J74@09JL4?H9KA4?@9L54?89LJ4?19M?4>I9N44>A9NI4>;9O=4>::014>9:0F4>8:1:4>8:1N4>8:2C4>7:374>6:3L4>5:4@4>4:584=M:5N4=C:6?4=3:6J4<>:784;I:7H4;3:8:4:>:8K49H:9<492:9M48=::>47H::O472:;<470:;H472:<4473:<@476:<L477:=8479:=C47;:=O47=:>;47?:>G47@:?347C:??47D:?J47F:@947I:A147E:A@483:B4480:BH47O:C=47L:D147J:DE47I:E947F:EM47E:FA47C:G547A:GJ47?:H>47=:I247;:I;47::IA475:IH470:IN46K:J446F:JC46?:K?467:L:45N:M645E:N245=:NM454:OH44M;0B450;1;453;25456;2N459;3G45:;4?450;5644F;5N44<;6E442;7=43I;8443?;8K436;9C42L;:;42B;;2428;;J41N;<A41D;=@41<;=J41C;>341K;>742>;>9439;>?441;>I44H;?4453;?L446;@E441;A>43K;B743E;C043@;C@437;CI42I;D342<;D<41N;DH41E;E941D;EJ41D;F;41C;FL41B;G=41A;GI41:;H3410;H>40F;HH40<;I3402;I=3OH;IG3O?;J63O7;K33OA;L23OO;M240>;N140K;N?414;NF418;NL41=;O441C;O:41H;OA41M;OG423<03426<0M424<2:41J<3F41?<52414<63414<66414<6G40M<7:40E<7M40><8?407<923OO<9D3OH<:73OA<:J3O9<;<3O2<;O3NJ<<B3NC<=43N<<=G3N4<>93MM<>L3MF<??3M><@B3M7<B:3LM<DK3LA<G63L9<HC3LA<J03LJ<K=3M2<LJ3M:<N?3M<=053M;=0K3M>=163M:=1@3M4=1H3LM=213LE=293L>=2B3L7=2K3L0=333KH=3<3KA=3D3K:=3M3K2=463JK=4@3JD=5@3J:=6?3J1=7?3IG=8>3I==9>3I4=:<3HJ=;:3H<=;M3GO=<;3GD=<I3G8==73FL==E3FA=>33F5=>A3EJ=?03E<=?A3DN=@23D?=@C3D0=A63C>=AI3BN=AK3BC=B?3B1=C33A?=CG3@L=D<3@9=E03?G=ED3?5=F93>C=GA3>3=I43=D=JF3=6=L83<H=M=3<?=NA3<5=OF3;L>0K3;B>203;8>343:N>493:D>5@3::>6M39O>8:39D>9F398>:C39J>;<39A><4399><M391>=H38F>>E38:>?B37M>@?37A>A=374>B;36C>C9363>D735J>E335H>F135E>FM35C>GG358>GI33B>GM31O>I=31A>JM313>L?30G>MC30B>N9315>NM31H>OE31>?0730>?0G2O<?1;2N;?2E2MH?3O2M5?5:2LB?6D2KN?7N2K;?982JH?9L2JA?:=2J;?:N2J5
16467O4C568B4C06954BJ69H4BE6:;4B?6;24B46;I4AI6<A4A>6=74A36=N4@H6>F4@=6?=4@16@44?G6@K4?;6AB4?06B94>E6C04>:6CA4=O6CH4=B6CO4=56D64<I6D>4<=6E<4<C6EN4<N6FH4=66GE4==6HB4=E6I>4=K6J;4>36K84>:6KH4>?6L34>C6L?4>M6LO4?>6M?4?M6MI4@<6N44@J6N=4A96NG4AF6O14B56O;4BC6OE4C26ON4C@7094CN70B4D<70L4DJ71<4E471M4E=72?4EC7304EK73@4F37414F:74B4FA7534FI75D4G07654G876F4G?77=4GH7844H278L4H<79D4HF7:<4I07;44I:7;L4IC7<D4IM7=<4J77=L4J57>04IA7>74I:7>L4IF7?A4J27@64J=7@K4JI7A74JL7AC4K17AG4K<7AL4KE7B44KG7B<4KH7BD4KH7BM4KJ7C54KK7C=4KL7CE4KM7D04KL7DA4KH7E24KJ7EB4KO7F<4L37G54L87GN4L;7HG4L@7I=4LG7J24LM7JH4M37K>4M:7L04MD7LA4MN7M24N87M=4N:7MI4N=7N54N@7N@4NB7NL4NE7O84NG7OE4NJ8054NM80F4O08164O381F4O58244O882@4O:82L4O=8384O@83E4OD8424OH84:4OI84G4OI8584OL85J4OO86;50186M50487>50687O50;88>50<88M50?89<50A89K50C8::50E8:I50H8;750J8;A50M8;A5128<25188=551@8>751H8?951O8@:5268A<52=8B=52D8C?52K8D@5328E65368EA5388EK53:8F653;8F@53=8FK53>8G553@8G@53B8GJ53C8H553E8H?53F8HJ53H8I453J8I@53J8J553D8JK53>8K@5388L55328LJ52L8M?52F8N@52:8OE51N90J51B91N51693350J94050<93?4O>92N4N@92@4MD92N4M;93<4M193J4LH9474L?94G4L59584KH95I4K=96:4K196K4JF97<4J997M4IN98>4IB98O4I799@4HH9:24H89:F4GH9;K4G@9<@4GB9=44GD9=I4GF9>=4GH9?24GJ9?@4GH9?L4GE9@64GB9@?4G59@D4F=9@K4F09A54F29A>4F39AH4F49B14F69BB4F19CB4EA9DC4E19EC4DA9F@4D@9G94DK9G94DH9G14D@9FI4D79F@4CO9F74CG9EO4C>9EF4C59E>4BM9E54BE9E64B;9EF4B59F74AO9FM4AG9GA4A?9H64A89HK4A09I@4@H9J54@@9JJ4@99K?4@19L34?I9LH4?A9M=4?:9N24?29NG4>M9O;4>L9OO4>K:0D4>K:184>J:1M4>I:2A4>H:354>H:3M4>B:4E4>;:5=4>5:604=H:6A4=9:704<H:7:4<2:7E4;<:854:F:8F4:1:9749;:9H48E::9480::J47;:;9470:;E471:<0473:<=475:<H476:=4479:=@47::=L47=:>847>:>D47@:?047B:?>47D:@347H:@I47L:AD47A:B647H:BB48;:C448@:CH48>:D<48<:E048::EE488:F9486:FM484:GA483:H5480:HI47O:I<47M:ID47L:IJ47G:J047C:J747=:J>478:K9470:L546G:M146>:ML466:NH45M:OC45D;0?45<;1;453;25452;2N455;3H458;4@458;5844N;6044E;6G44;;7?441;8643G;8M43=;9E433;:=42I;;442@;;L426;<C41L;=:41B;>141:;>@41M;>K42E;?543<;?@443;?K44K;@645A;@@45;;@H44B;BM44>;CL448;C<443;C043H;CC432;CN42C;D8425;DG421;E8420;EI420;F:41O;FK41N;G<41M;GM41M;H;41I;HF41?;I0415;I;40K;IE40A;IO408;J@402;K33OK;KF3OE;LC402;MC40@;NB40N;O6417;O=41=;OD41B;OJ41G<0141M<08422<0C426<13428<21424<3=41J<4I41@<65415<6B41:<6E41:<6I419<7<411<7O40J<8A40C<9440;<9F404<:93OM<:L3OE<;>3O><<13O6<<D3NO<=63NH<=I3N@<>;3N9<>N3N2<?D3MJ<@N3MB<BF3M9<E83LM<GI3LA<IG3LA<K43LI<LA3M1<MM3M:<O:3MB=0B3MJ=0L3MC=153M<=1=3M4=1F3LM=1N3LF=273L>=2@3L7=2H3L0=313KI=3:3KA=3B3K;=3K3K3=433JK=4C3JD=5B3J:=6B3J1=7A3IG=8A3I==9?3I3=::3HI=;93H:=<73GJ=<E3G>==33G3==A3FH==O3F;=>=3F0=>K3EE=?83E9=?G3DM=@53DB=@C3D6=A23CJ=AI3C==B>3BK=C33B9=CG3AG=D;3A5=E03@B=ED3@0=F83?==FM3>K=H63>;=II3=N=K<3=?=L@3=5=ME3<K=NJ3<B=OO3<9>133;O>283;E>3=3;;>4B3;1>5G3:H>713:=>8>3:1>9J39G>:C3:>>;;3:6><439M><O39D>=L397>>I38K>?F38>>@C382>A@37E>B>377>C;372>D836O>E536L>F236I>FN36G>GL36D>GN34N>HG33H>IO32L>K=32:>LO31O>N331F>N>31B>NN31I>OD31M?0530K?0F2OI?1;2NI?2E2N6?3O2MB?592M0?6C2L<?7N2KI?982K6?9M2JN?:>2JH?:N2JB?;@2J=
16567N4D768A4D16974CG69N4C<6:F4C16;=4BF6<44B;6<K4B06=B4AE6>94A96?04@O6?G4@C6@>4@86A54?M6AL4?B6BC4?76C:4>L6CA4>?6CH4>36CO4=F6D64=:6D=4<M6E84=26EK4=<6FH4=C6GE4=K6HB4>26I>4>96J<4>@6K84>G6L54>N6LF4?:6M04?I6M:4@76MC4@F6MM4A36N84AB6NA4B06NK4B?6O54BL6O?4C;6OI4CI7024D770=4DE70F4E37164E=71H4EF7294EM72J4F573;4F<73L4FC74=4FK74N4G275?4G:7604GA76:4GK7764H27814H;78I4HD79A4HN7:94I87;14IB7;I4IL7<A4J67=84J@7=@4J07=F4IE7>24IH7>G4J37?;4J@7@14JK7@A4K27@M4K67A:4K97AB4KA7AF4KL7AK4L67B04L>7B84L?7B@4LA7BI4LB7C14LB7C94LD7CE4LB7D64L>7DI4L97ED4L87F=4L;7G74L@7H04LD7HG4LI7I=4M07J34M67JH4M=7K94MG7KJ4N37L<4N=7LK4NF7M84NI7MC4NK7MO4NN7N;4O07NF4O37O34O67OA4O98014O<80A4O>8114OA81B4OD81N4OF82:4OI82F4OL8324OO83>50183K50484750784E50985750<85I50>86:50A86K50C87<50E87M50H88<50J88K50L89:50N89H5118:85138:F5158;55188;D51:8;F51=8;>51:8<B51B8=D51K8>G5238?J52;8@M52D8B052L8C35338D>53B8E453J8E@5408EK5438F65458F@5468FJ5498G554:8G@54<8GJ54=8H554?8H?54@8HJ54B8I454D8I>54F8J154A8JG54;8K<5458L153O8LF53I8MC53@8NH5338OL52G91152;92551O93:51C94;51593J5089394O:92H4N<92J4MJ9374MA93E4M79434LN94E4LB9554L795F4KK9674K?96H4K39794JH97J4J=98;4J098L4IE99=4I999O4HH9:I4H;9<04H29<G4H29=;4H49>04H69>C4H89>N4H49?84H09?D4GL9?N4GH9@74G;9@<4FE9@B4EM9@I4EA9A24EC9A<4EC9AE4EE9B94E>9C94DN9D:4D>9E64D;9F04DG9FI4E19G94E69G04DN9FH4DF9F?4D=9F64D59EN4CM9EE4CD9E=4C<9E54C49E>4BH9EO4BB9F?4B<9FO4B69G@4B09H44AI9HI4AA9I>4A99J34A19JH4@I9K=4@B9L24@:9LF4@29M<4?J9N04?C9NE4??9O94?>9ON4?>:0B4?=:164?<:1K4?;:2@4?8:394?1:414>K:4J4>C:5B4>;:624=L:6C4==:734<M:7A4<<:7K4;F:864:O:8@4:9:9149D:9B48O::4489::E47C:;646O:;A470:;M473:<9474:<E476:=1478:==479:=I47<:>447=:>C47@:?847D:?M47H:@B47L:A847O:B647=:BL479:C947L:CF48@:D448O:DH48M:E<48L:F048I:FD48H:G848E:GM48D:HA48B:I548@:IF48?:IM48?:J348::J9485:J?47O:K447H:L047@:LK477:MG46N:NC46F:O>46=;0:464;1545L;2145D;2M45;;3H454;4A457;5:456;6144L;6I44B;7A448;8843O;8O43E;9G43;;:=433;;042L;;D42E;<842>;<K427;=?420;>@41G;?;420;?L42O;@<43N;@L44L;A745D;AA46;;AM45K;AF44O;CI44K;C044E;B;44>;BN43H;CA431;D542>;DF42=;E742=;EH42<;F942;;FJ42:;G;42:;GK429;H<428;HN427;I841N;IC41D;IM41:;J9411;JK40K;K>40F;L040@;LB40;;M5405;N440B;O4410;OM41<<0441A<0:41G<0A41L<0H421<13425<1C427<2442:<33425<4@41J<5L41@<6G41<<7141@<7441@<7841@<7>41=<81416<8C40O<9640G<9H40@<:;408<:N402<;@3OJ<<33OB<<F3O<<=83O4<=K3NL<>=3NE<?03N><@03N6<A:3MN<C33MD<ED3M9<H63LM<JG3LA<L23LH<M13M0<N13M:<O03MB<OO3MJ=0E3MC=123M<=1;3M5=1C3LM=1L3LF=243L?=2>3L7=2F3L1=2O3KI=373KB=3@3K;=3H3K4=413JL=4E3JD=5D3J:=6D3J1=7C3IG=8B3I==9=3I3=:93HF=;73H7=<63GG=<N3G9==<3FN==K3FB=>83F6=>F3EK=?43E?=?B3E3=@13DH=@>3D==@L3D1=A?3CC=B43C6=BL3BJ=CD3B>=D;3AO=DO3A==EC3@K=F73@8=FL3?F=G@3?3=HM3>D=J?3>6=KD3=L=LI3=B=MN3=8=O33<O>073<E>1<3<<>2A3<2>3F3;H>4J3;?>5O3;5>753:K>8B3:@>9N3:4>:B3;3>;;3:K><53:@>=33:4>=O39G>>M39;>?J38O>@G38B>AC382>BB37H>C>37K>D:37N>E6381>F337O>G037K>GM37I>H@36C>IG35G>JO34K>L733O>M?335>NB32L>NN32I>O932E>OC329?03318?0C306?1:2O7?2E2ND?3O2N0?592M=?6C2LJ?7M2L7?982KD?9N2K:?:?2K5?:O2K0?;@2JJ?<12JD
16768K4DD69B4D96:94CN6;04CC6;G4C86<>4BM6=54BB6=L4B76>D4AL6?;4A@6@14A66@I4@J6A@4@@6B74@46BN4?I6C:4?=6CA4?16CH4>D6CO4>86D64=K6D=4=?6E34=A6EL4=I6FH4>16GE4>86HB4>?6I?4>F6J94?16K24?<6KL4?G6LE4@36M44@@6M>4@O6MG4A=6N24AL6N;4B96NE4BH6NO4C66O94CD6OC4D26OM4D@7064DO70A4E=7114EG71C4EO7234F772E4F?7364FF73F4FM7474G574H4G<7584GD75B4GN75L4H876J4H>77I4HE78F4HM79>4I77:64I@7:M4IJ7;F4J47<=4J>7=04J@7=44J07=?4J17=L4J67>A4JA7?64JM7?J4K87@74K<7@C4K?7A04KB7A=4KF7AA4L17AE4L;7AI4LF7AN4M07B44M77B<4M87BE4M97BM4M;7C:4M87CL4M47DH4LK7ED4LD7F?4LD7G84LH7H14LL7HH4M27I=4M97J14MA7JB4MK7K44N77KE4NA7L64NL7LF4O47M24O67M>4O97MI4O<7N54O?7NA4OA7NO4OD7O=4OG7OM4OJ80=4OM80M4OO81;50281H50582450882@50:82L50=83850@83D50B83J50A84B50G85350J85D50L86650N86G51287851487J51688;51988J51;89851>89G51?8:651A8:E51D8;351F8;B51H8;H51E8;?51A8;M51D8=151M8>35258?652>8@952E8A<52M8BL53?8D<5408E05488E=54>8EH54D8F454J8F@5508FJ5528G55548G@5558GJ5578H55588H?55;8HI55<8I455>8I>55?8IN55>8JC5588K85528KM54L8LE54D8MJ5488NO53L90353@91853392=52G93A52;94F51O94551293D5039334O692E4N:9334N193A4MG9404M=94A4M19534LE95C4L99644KN96E4KB9764K697G4JJ9884J?98I4J399:4IG99L4I89:L4HN9<24HD9<N4HC9=C4HD9>04HC9>;4H?9>F4H;9?04H89?;4H49?F4H09?N4GC9@44FL9@:4F59@?4E=9@F4E29A04E39A94E49B04DL9C04D<9CM4D79DG4DB9E@4DM9F:4E79G34EC9FO4E=9FG4E49F>4DL9F64DD9EM4D;9EE4D39E<4CK9E;4CA9EG4C59F74BN9FH4BH9G84BC9GH4B=9H84B79HI4B19I<4AJ9J14AB9JF4A:9K;4A39L04@K9LD4@C9M:4@;9MN4@39NC4@19O74@09OL4@0:0@4?O:144?N:1M4?G:2E4?@:3>4?9:464?2:4O4>K:5D4>@:644>1:6E4=B:754=2:7E4<C:824;O:8<4;9:8G4:C:9149M:9=497:9N48B::?47L:;0476:;>470:;J472:<6473:<B476:<N477:=9479:=H47<:>=47@:?247C:?G47H:@<47L:A247O:AJ47J:BH478:CC46K:D047>:D=481:DI48E:E6498:EH49=:F<49;:G0499:GD497:H8495:HL493:I@492:IN492:J6492:J<48L:JB48G:JO48A:KK489:LF47O:MB47G:N>47?:O9476;0446M;1046E;1L46<;2H464;3C45K;4>45B;5:45:;63454;6I44K;7=44E;8044>;8D447;98440;9K43I;:?43C;;243<;;F436;<:42O;<M42H;=J42?;?0423;@341O;@D42M;A443L;AD44K;B545J;AO46C;AO45G;AE44J;B<44N;AK452;B>44>;C@43H;DC433;D>42I;DN42H;EA42H;F442G;FG42F;G942F;GJ42E;H;42D;HL42C;I=42C;IK42=;J5423;JC41K;K641F;KH41@;L;41:;LN414;M@40O;N240I;NE40D;OE412<0D41A<0K41F<1141K<18420<1C424<23426<2D429<3442;<47425<5C41J<6J41B<7941D<7@41F<7C41F<7F41F<7J41E<8341B<8E41;<98414<9J40L<:=40D<;040><;B406<<53ON<<H3OG<=:3O@<=M3O8<>?3O1<?23NJ<@<3NB<AF3N9<C@3N0<F13MD<HB3M9<JH3LO<KA3LL<L>3M1<M=3M:<N=3MC<O<3MK=043MC=0H3M==153M5=1A3LN=1J3LG=223L?=2;3L8=2D3L1=2L3KJ=353KB=3=3K;=3F3K4=3O3JM=4G3JD=5F3J:=6F3J0=7E3IG=8@3I<=9;3I1=:83HD=;73H4=<53GD==43G5==F3FH=>43F==>B3F1=?03EE=?>3E:=?L3DO=@:3DC=@H3D7=A93CJ=AM3C==B@3BN=C73BB=CN3B6=DG3AJ=E>3A>=F63A1=FL3@@=G@3?N=H43?<=IB3>M=JH3>C=KM3>9=M23=O=N73=F=O<3=<>0@3=2>1E3<H>2J3<?>3O3<5>533;L>683;B>7=3;8>8F3:M>:13:F>:B3;H>;<3;>><93;1>=63:E>>33:9>?039I>?L397>@I38G>AE385>BE380>CC382>D?385>E;388>F738:>G338=>H038?>I637N>J=371>KD364>LL357>N334;>NJ33A>O332O>O<32D>OE329?0131E?0B30C?1:2OD?2D2O1?3N2N>?592MK?6C2M8?7M2LD?972L1?9O2KH?:?2KB?;02K=?;A2K7?<22K1?<B2JL
1706:D4D@6;;4D56<14CJ6<I4C?6=@4C46>74BH6>N4B>6?E4B26@<4AH6A34A<6AJ4A16BA4@F6C34@:6C:4?N6CB4?A6CI4?56D04>H6D74><6D>4=O6DO4=O6EL4>76FH4>>6GC4>H6H<4?46I64??6IO4?J6JI4@66KB4@B6L<4@M6M54A96MB4AF6MK4B56N64BC6N?4C16NI4C?6O34CN6O=4D<6OG4DJ7004E870;4EF70K4F071=4F971N4FA72?4FI7304FO73A4G77424G?74A4GG74K4H17554H;75?4HE76=4HK77>4I178;4I97994I@7:24II7:J4J37;B4J=7<:4JF7<D4J<7<N4J<7=84J=7=G4JC7><4JO7?14K;7?@4KA7?M4KE7@94KH7@F4KL7A34KO7A<4L67A@4LA7AD4LK7AI4M57AM4M@7B24MJ7B84N07B@4N17BN4MN7CK4MF7DG4M>7EC4M67F@4LN7G94M17H24M57HH4M;7I;4MD7IL4N07J=4N:7JO4NE7K@4O07L14O;7L@4OC7LL4OE7M84OG7MC4OJ7MO4OL7N=4OO7NK5037O95067OH50980850<80H50?81550A81A50D81M50F82950I82E50K82L50K83250I83?50K84=51584O51885@51;86251=86C51?87451B87F51D88851H88I51J89751L89F51N8:55218:D5238;25248;A5278;A5208;A51J8;:51G8<=51O8=@5278>C52?8?F52H8A:53<8BM5408D954>8DL54E8E954K8ED5518F05578F<55=8FH55C8G355J8G@55O8GJ5618H45638H?5658HI5668I45688I>5698IJ56;8J?5658K455N8KI55H8LM55<8N15508O654D90;54891?53L92C53>93E52L94452894351C93O50M93>4OO92M4O292O4N@93=4N793N4MJ94>4M?9504M395A4LH9614L<96B4L09744KE97D4K99854JM98F4JA9974J699I4IH9:O4I@9<54I79=34I19==4HN9=H4HJ9>34HG9>>4HC9>H4H?9?34H;9?>4H89?F4GJ9?L4G39@24F<9@74ED9@=4DM9@D4DB9@N4DD9AF4D99BE4D29C>4D=9D74DH9E14E39EJ4E>9FC4EI9FO4EL9FF4EC9F>4E;9F54E39EL4DJ9ED4DB9E;4D:9ED4CN9EO4CA9F@4C;9G04C59G@4C09H04BJ9HA4BD9I14B>9IA4B89J24B29JD4AK9K94AC9KN4A<9LC4A49M84@M9MM4@D9NA4@C9O54@C9OJ4@B:0A4@=:1:4@6:224?O:2J4?G:3C4?A:4;4?::544?2:5F4>D:664>5:6G4=F:774=7:7G4<H:884<8:8C4;C:8M4:M:974:7:9B49@:9L48J:::485::K47?:;;470:;G471:<2473:<?475:<M477:=B47<:>747@:>L47C:?A47G:@647L:@L47O:AA483:B<47F:C:473:D846A:DG470:E347C:E@486:EM48I:F:49=:FG49L:G;49K:H049H:HD49G:I849E:IL49C:J749D:J>49D:JE49>:JK499:KE491:LA48H:M<48@:N8487:O447O:OO47F;0K47=;1F475;2B46M;3>46D;4846;;4O461;5G45G;6<45?;6L45;;7?455;8244N;8F44H;9:44A;9M44:;:A443;;443L;;H43E;<<43?;=4437;>;42L;?A42@;@H424;A;42L;AL43K;B<44J;B>45D;A8465;A645A;A>44F;B044B;B644C;C>44;;DM43J;ED438;DM434;E5433;EH433;F;432;FN431;GA430;H4430;HG42O;I:42O;IM42N;J=42L;JN42F;KA42@;L342:;LE424;M841O;MK41I;N=41D;NO41=;OB417<06415<1641C<1B41J<1I41O<23423<2C426<33428<3D42:<4442=<5:426<6F41K<7<41I<7L41M<7O41M<8341L<8641L<8941K<8<41K<8G41G<9:41@<9L418<:?411<;240J<;D40B<<740;<<J403<=<3OL<=O3OE<>B3O=<?>3O5<@H3NN<B23NF<CL3N<<F>3N0<HO3ME<J=3M><K63M;<KO3M8<LJ3M:<MJ3MC<NI3MK<OA3MD=093M==0K3M6=173LN=1D3LG=203L@=283L8=2A3L1=2J3KJ=333KC=3;3K;=3D3K5=3L3JM=4I3JD=5I3J:=6G3J0=7C3IF=8>3I;=993I0=:73HA=;53H1=<43GB==33G2=>03FC=>>3F8=>L3EL=?:3E@=?H3E5=@63DI=@D3D==A33D1=AF3CD=B:3C6=BM3BH=CA3B;=D93AN=E13AB=EI3A6=FA3@J=G93@==H03@1=HH3?D=IL3?:=K13?0=L63>F=M;3><=N?3>3=OD3=I>0I3=?>1N3=5>323<K>473<B>5<3<9>6A3;O>7E3;E>8J3;;>9I3<E>:C3<;>;@3;O><=3;@>=93:O>>53:>>?239M>?N39<>@K38K>AG38:>BG388>CH389>DD38;>EA38>>F=38A>G838D>H538I>I?38?>JG37B>KN36F>M535I>N<34L>NK343>O533;>O?32D>OH328?0131L?0@310?19302?2D2O??3N2NL?582N8?6B2MF?7M2M2?972L??:02L5?:@2L0?;12KJ?;B2KD?<32K>?<C2K9?=42K4
1726;E4DG6<<4D<6=34D16=J4CF6>A4C;6?84C06?O4BE6@F4B96A>4AO6B54AC6BK4A86C44@L6C;4@?6CB4@36CI4?F6D04?:6D74>M6D>4>A6DM4>?6EG4>K6F@4?76G94?B6H34?M6HL4@96IF4@E6J?4A06K84A<6L24AG6LK4B26ME4B?6MO4BL6N:4C;6NC4CI6NM4D76O74DE6OA4E36OK4EB7044EO70E4F97174FB71I4FK7294G272K4G973<4GA73I4GI7434H474=4H>74H4HH7524I375O4I87734I=7804ID78M4IL79K4J37:G4J;7;?4JE7<44JK7<<4JF7<G4JH7=14JI7=B4K17>64K<7>I4KF7?64KJ7?B4KN7?O4L17@<4L57@I4L87A54L<7A;4LE7A?4M07AD4M;7AH4ME7AM4N07B14N:7B54NE7BM4N@7CI4N87DF4N17EB4MH7F?4M@7G;4M97H24M>7HD4MH7I54N47IG4N>7J84NI7JI4O37K:4O?7KK4OI7L:5007LF5037M25057M>5097MK50<7N950?7NG50B7O550E7OD50H80450J80B50M80N51081;51281G51581O51582551482;51282A51183751784451@84K51F85<51I85N51K86?51N87052087B52388552688F52989652;89E52=8:452?8:B52A8;152D8;952B8;:52;8;:5248;:51N8;I5218<L5298>452E8?H5398A;53M8BO54@8D554K8DI5528E45588EA55?8EL55E8F855K8FD5618G05678G;56=8GH56C8H356I8H?56N8HI5708I45718I>5748II5758J;5718K056K8KO56A8M45658N955I8O=55<90?54J91@54792B53E93=53193;52=93:51H93951393750?9364OJ92J4O093:4NE93K4N994<4MM94M4MA95>4M695O4LK96?4LA9714L697B4KL9824KA98C4K79944JL99J4JC9;14J89<34J<9<H4I=9=54I59=@4I19=K4HN9>64HJ9>@4HF9>K4HB9?64H?9?>4H19?D4G:9?J4FB9?O4EL9@54E49@:4D=9@B4D39A;4CN9B54D89BN4DD9CG4DN9DA4E:9E:4ED9F34EO9FM4F:9FE4F29F=4EI9F44EA9EK4E99EC4E09E@4DG9EL4D:9F84CN9FI4CH9G94CB9GI4C<9H94C79HJ4C19I:4BK9IJ4BE9J:4B?9JK4B89K;4B39KL4AL9LA4AE9M64A=9MK4A69N?4A69O54A39OM4@L:0F4@E:1>4@=:274@7:2O4@0:3H4?H:4@4?A:574?9:5H4>I:684>::6I4=K:794=<:7I4<L:8:4<=:8I4;M:934;6:9>4:@:9H49J::2494::=48>::G47H:;7472:;C470:<2473:<G477:=<47;:>147?:>F47C:?;47G:@047K:@E47O:A;483:B1483:BO47A:CL46O:DJ46=:E>46A:EJ474:F747H:FD48;:G048O:G=49A:GJ4:4:H;4:::HO4:8:ID4:6:J84:4:J@4:6:JG4:6:JM4:1:K@49J:L<49A:M7498:N3490:NN48H:OJ48?;0F486;1A47N;2947D;3147:;3H470;4?46G;5746=;5N464;6F45J;7045J;7A45E;8445>;8H458;9<451;9O44K;:C44D;;744=;;J446;<>43O;=E43C;>K438;@242L;A842@;B442K;BD43J;BL44G;AF457;@A45G;@845;;@K44F;AK449;BK43M;C543L;E243?;DE43O;CL442;EO43D;FB43B;G543B;GH43@;H;43?;HO43?;IB43=;J543<;JH43;;K4439;K?433;L4430;LI42K;M?42G;N442C;NG42=;O:428;OM422<0?41L<1141G<1G41E<2941N<2C423<33425<3D427<4442:<4D42<<5442><6=426<7@41O<7O420<8;423<8>423<8B422<8E422<8H421<8K421<8N420<9<41L<9N41D<:A41=<;4416<;F40N<<940G<<L40?<=>408<>1401<>D3OI<?J3OB<A43O9<B>3O2<D93NH<FJ3N<<I93N0<J33MM<JK3MJ<KD3MG<L=3MD<M73MC<N63ML<NN3ME<OF3M==0>3M7=0M3LO=1:3LG=1G3LA=243L9=2?3L1=2H3KK=303KC=393K<=3A3K5=3L3JM=4K3JD=5J3J:=6E3IO=7A3ID=8<3I9=973HN=:63H?=;43GO=<33G?==13G0=>03F@=>H3F2=?53EF=?C3E;=@23DO=@@3DD=@N3D8=A@3CK=B33C==BF3BO=C:3BA=CM3B4=DD3AG=E<3A;=F33@O=FK3@B=GC3@6=H;3?I=I33?E=IM3?A=K33?8=L83>O=M>3>E=NC3><=OH3>3>0M3=J>233=@>383=7>4>3<N>5C3<E>6H3<;>7N3<2>933;I>9I3=8>:F3<F>;B3<5><?3;D>=;3;3>>73:B>?33:1>@039@>@L38O>AI38?>BJ38@>CL38A>DI38B>EF38E>FB38H>G>38K>H:394>IF38H>K0384>L7377>M>369>NC35=>NM34F>O733N>O@337>OJ32??0431L?0?31=?1930@?2D2OM?3M2O:?582NF?6B2N3?7L2M@?972LL?:02LC?:A2L=?;22L7?;C2L1?<32KL?<D2KF?=52KA?=F2K;
1746<G4DN6=>4DC6>54D86>L4CM6?C4CA6@:4C76A14BK6AH4BA6B?4B56BM4AI6C44A=6C;4A06CB4@C6CI4@76D04?J6D74?>6DA4?66E54??6EJ4?G6F=4@26G14@=6GI4@G6HB4A36I<4A?6J54AJ6JO4B56KH4BA6LB4BM6M;4C86N44CD6N>4D26NG4D@6O14DO6O;4E=6OE4EK6OO4F970?4FC7114FL71C4G47244G;72E4GC7324GL73<4H773F4HA7404HK74:4I574E4I@75B4IF76E4IJ77E4J078B4J879@4J?7:>4JF7;<4JM7;K4K17<64K27<@4K47<J4K67=<4K?7>14KJ7>?4KO7>L4L37?84L77?E4L:7@24L>7@?4LB7@L4LE7A64LK7A:4M57A>4M@7AC4MJ7AG4N57AL4N@7B34O07BG4O27CD4NJ7DA4NB7E>4N:7F;4N27G74MJ7GM4MM7H?4N77I04NB7IA4NM7J24O87JC4OC7K54OM7KF5087L450?7L@50A7LL50D7M:50G7MH50J7N550M7NC5107O15137O?5167OO51980<51;80H51>81251?81751>81>51<81D51;81J51:82151982N51B83K51K84G52485852785J52:86;52<86M52>87>52A88152D88C52G89552J89C52L8:252N8:A5308;05338;252M8;352F8;352?8;45288;55238<B52B8>55368?I53I8A=54=8BM54O8D25588DE55?8E055E8E=55K8EH5618F45688F@56>8FL56D8G756K8GD5718GO5778H;57=8HG57C8I357I8I>57M8II57O8J757N8K257F8L757:8M956I8N:5668O<55C90=55091>54=92@53K92D53792C52A92A51M92@51892>50D92=4OO92F4OA9364O993H4O39494NI94J4N>95;4N495K4MJ96<4M?96N4M597>4LK97O4L@98@4L59914KL99G4KB9:I4L69;I4LB9<34JJ9<F4J09=94I49=C4I59=N4I19>84HN9>C4HJ9>N4HF9?64H89?;4GB9?A4FJ9?G4F39?M4E<9@24DD9@;4D49@L4D49AE4D?9B?4DJ9C84E49D14E@9DK4EJ9ED4F69F=4F@9FD4FA9F<4F99F34F09EK4EH9EB4E?9EI4E39F44DG9FA4D;9G14D59GA4CO9H14CI9HB4CC9I24C>9IB4C89J24C29JB4BK9K34BE9KC4B@9L34B:9LD4B49M44AN9MI4AI9NA4AB9O:4A;:024A3:0K4@M:1C4@E:2<4@>:344@7:3M4@0:4E4?I:594?>:5J4>N:6:4>>:6K4=O:7;4=@:7K4=1:8<4<B:8L4<3:9:4;@:9D4:J:9N4:4::949>::D48H::N481:;847;:;L473:<A477:=647;:=K47?:>@47C:?547G:?J47K:@?47O:A5483:AJ487:BC47O:CA47=:D>46K:E<469:F4463:FA46F:FN479:G:47L:GG48?:H4493:HA49F:HM4:::I;4:J:IO4:G:JA4:G:JI4:I:K04:H:K;4:C:L74:::M24:1:MM49I:NI49@:OC498;0;48N;1248D;1J48:;2A480;3947F;4047=;4G473;5?46I;6646?;6N465;7@460;7D465;8645O;8J45H;9>45A;:145;;:E454;;944M;;L44F;<O44;;>5440;?<43E;@B439;AI42M;BK42I;C:43I;B5448;A044I;?K45:;?:455;@744F;A7449;B743M;C943E;D6435;D=43?;DD44@;EK44;;GB440;H543O;HH43N;I;43L;IN43L;JA43J;K543I;KH43I;KK43D;L043?;L>43:;M3436;MI432;N>42M;O442I;OI42E<0>42@<1442=<1J428<2?424<33422<3D424<44426<4D429<5442;<5D42><6542@<7@426<83426<8A426<8J429<8M429<91428<94428<97427<9:427<9=426<9A426<:0420<:C41I<;641B<;H41:<<;413<<N40K<=@40D<>340=<>L405<@63OM<A@3OE<BJ3O=<DF3O3<G73NH<HO3N@<IH3N=<J@3N:<K93N7<L23N4<LK3N1<MC3MM<N;3ME<O33M><OK3M7=0C3M0=103LH=1=3LA=1J3L:=273L2=2D3KK=2N3KD=363K<=3?3K5=3N3JM=4M3JD=5H3J9=6C3IN=7?3IC=893I8=963HL=:53H<=;33GM=<23G===03FM==O3F>=>N3EN=??3EB=?N3E6=@;3DJ=@I3D?=A93D2=AM3CD=B@3C7=C33BI=CG3B;=D:3AM=DN3A?=EF3A3=F>3@G=G63@;=GN3?O=HF3?J=I>3?G=J63?D=K33??=L83?6=M>3>M=NC3>C=OH3>:>0N3>1>233=G>383=>>4>3=5>5C3<K>6H3<B>7N3<9>933;O>:I3;H>:J3<K>;D3<9><@3;I>=<3;7>>83:F>?53:5>@139D>@N393>AK38G>BM38H>CN38H>DN38I>EK38L>FG38N>GC391>H?39?>IL391>K938D>LA37H>MG36K>ND360>NN359>O834A>OB33I>OL332?0632:?0@31C?1930N?2C30:?3M2OG?582O5?6B2NA?7L2MN?962M:?:12M0?:B2LI?;22LD?;D2L??<42L9?<E2L3?=52KM?=G2KH?>72KC
1766=H4E56>?4DJ6?64D?6?M4D46@D4CH6A<4C>6B24C26BF4BG6BM4B;6C44AN6C;4AB6CC4A56CJ4@H6D14@<6D84?O6DK4@76E@4@@6F44@H6FH4A16G<4A:6H04AE6HC4AO6I64B96IL4BD6JE4C06K>4C;6L84CG6M14D26MK4D=6NA4DJ6NK4E86O54EF6O?4F56OI4FB70:4FL70K4G571=4G>71O4GE72:4GO72E4H972O4HD7394HN73C4I873M4IB7474IM7544J37674J777:4J=7874JD7954JK7:34K27;04K97;D4K=7;O4K?7<94K@7<C4KB7=64KL7=H4L57>54L87>B4L<7>N4L@7?;4LC7?H4LG7@54LK7@B4LN7@N4M17A54M;7A94ME7A>4N07AC4N<7AK4NM7B44O;7BF4O>7C>4O<7D;4O37E74NL7F44NC7G14N;7GH4N<7H94NF7HK4O17I<4O<7IM4OF7J>5017K050<7KA50G7KO50L7L:50O7LH5127M65157MD5187N151;7N?51>7NM51A7O;51D7OJ51H80551J80;51H80@51G80F51E80L51D81251B81851A81H51D82E51N83C52784?52A85552E85F52G86752J86I52M87;53087M53388?53589153889B53;8:153=8:@53?8:J53>8:K5388:K5318:L52J8:L52E8;252C8<D5338>753F8?J54:8A>54N8BI55<8CN55E8DA55M8DM5638E95698EE56?8F056E8F<56K8FH5718G45778G@57=8GL57D8H757J8HC5808HO5868I;58<8IG58B8J458F8K45848L657A8M756O8N956<8O:55I90<55691=54D91L54091K53;91I52G91H52291G51=91M50N92750@92B5039304OF93A4OA9444O?94G4O=9584O395H4NI9694N>96J4N497;4MI97L4M?98=4M498N4LJ99D4LA9:C4MD9;64N29;>4L89<14K<9<D4JA9=74IF9=F4I99>04I59>;4I19>F4HN9>N4H@9?34GH9?94G19?>4F:9?F4EF9@34E=9@@4E39@M4DJ9A<4DK9AO4E09BH4E;9CA4EF9D;4F19E44F<9EM4FF9FD4FO9F;4FG9F24F?9EJ4F69EF4EM9F14E@9F=4E39FI4DH9G94DB9GJ4D<9H:4D69HJ4D09I:4CK9IK4CD9J;4C>9JK4C89K<4C29KL4BL9L<4BE9LN4B;9M?4B09N24AK9NF4AI9O?4AC:074A;:0O4A4:1H4@M:2A4@F:394@?:424@7:4J4@0:5;4?B:5L4?3:6<4>D:6M4>4:7=4=E:7M4=5:8>4<F:8N4<7:9?4;H:9K4;4::54:>::@49G::J491:;448;:;B47I:<947?:=047::=E47?:>:47C:>O47F:?D47K:@947O:@O483:AD486:B948;:C647K:D3479:E046G:EN464:FK45D:G8467:GE46J:H147>:H>481:HK48E:I7498:ID49K:J14:>:J=4;1:JJ4;9:K24;;:K94;;:L14;3:LM4:J:ME4:A:N=4:7:O449M:OK49C;0C49:;1:490;2148F;2I48<;3@483;4747I;4O47?;5F475;6>46K;7546A;7L467;8346:;8846?;8L468;9@462;:345K;:G45D;;:45>;<9454;=@44H;>F44=;?M441;A343E;B:439;C@42N;BD43;;A>43K;@944;;?444L;>C453;?L454;A4458;B=45;;CF45=;DF451;E@444;F?43?;F@44J;GI44D;I544<;IH44;;J;44:;JN448;KA448;L4446;L@444;LA43O;LD43J;LN43E;M=43@;N343<;NH438;O>434<03430<0H42L<1>42G<2342C<2J42@<3E42?<4;42=<5242:<5D42:<6542=<6E42@<7742A<8642<<8E42=<9342><9942?<9<42?<9?42><9C42><9F42=<9I42=<9L42=<:042=<:342=<:E425<;841M<;J41G<<=41?<=0417<=B411<>540I<?840A<@B409<AL401<C63OI<E23O?<GD3O3<HD3NO<I=3NL<J63NI<JN3NF<KG3NC<L?3N;<M73N2<ML3MI<NA3M?<O83M7<OO3M0=0E3LI=123LA=1?3L:=1L3L3=293KK=2G3KD=343K==3=3K5=403JM=4K3JB=5F3J8=6A3IL=7=3IB=873I7=953HI=:43H:=;23GJ=<13G:=<O3FK==N3F;=>M3EK=?I3E<=@73E0=@E3DE=A33D9=AF3CK=B:3C==BM3C0=C@3BB=D43B5=DG3AF=E:3A8=F13@L=FI3@?=GA3@3=H93?O=I13?L=II3?I=JA3?F=K:3?C=L83?==M>3?3=NC3>J=OI3>A>0N3>7>233=N>383=E>4>3=;>5C3=2>6I3<I>7N3<@>983<7>;23;O><L3;H><<3<>><B3;L>=>3;;>>:3:J>?73:9>@339H>@O397>AN38N>C038O>D1390>E2391>F0393>FL395>GH398>HD39I>J239:>K?38M>LJ389>N137<>NF36C>NO35K>O9354>OC34<>OM33E?0832L?0B322?1;31<?2C30H?3M305?572OB?6A2NN?7L2N;?962MI?:22M<?:C2M7?;32M1?;D2LL?<52LF?<F2L@?=62L;?=G2L5?>82L0?>I2KJ
1776>24EG6>J4E<6?A4E16@84DF6@O4D:6AF4D06B=4CD6BF4C86BM4BK6C54B?6C<4B26CC4AF6CJ4A96D14@M6DA4@N6E54A76EI4A?6F>4AI6G24B16GF4B:6H;4BC6HN4BM6IA4C86J54CB6JI4CL6K<4D76L04DA6LH4DL6MA4E86N;4EC6NO4EO6O94F>6OC4FL7034G670F4G?7184GH71C4H171M4H<7274HF72B4I172L4I;7364IE73@4J073J4J974G4J@75J4JE76M4JI77L4JO78J4K779H4K>7:A4KD7;54KH7;H4KK7<24KM7<<4KO7=14L:7=>4L>7=K4LB7>84LE7>D4LI7?14LL7?>4M07?K4M47@84M77@D4M;7A04M@7A44MJ7A:4N87AA4NI7AI4O97B44OE7BG4OH7C94OL7D54OE7E14O=7EN4O57FK4NM7GD4NK7H44O67HF4O@7I74OK7IH5057J950?7JJ50K7K;5157KI51;7L651>7LD51A7M251D7M@51G7MM51J7N;51M7NI5207O65227O>5227OC5217OI5207OO51N80551M80;51K80A51J80G51I81@52082=52983952C84752L85153385B53686353886E53;87753>87I53@88;53C88M53F89?53I89O53L8:?53N8:C53I8:C53C8:D53=8:E53:8:E5378;45378<E53C8>85478?L54K8A?55>8BF55I8CJ5638D=56:8DI56@8E556F8EA56L8EM5728F85788FD57>8G057D8G<57K8GH5818H45878H?58=8HK58C8I758I8IC58L8J<58B8K55888L457H8M55758N756B8O855O90:55<91554I91354591253@91052K91552:91?51N91H51@92251292=50E92L50993:4OL93M4OI94?4OH9524OF95E4OE9664O=96G4O39784NH97I4N=98:4N498J4MI99B4M@9:=4NK9:@4N;9:I4MC9;<4LI9;O4KO9<B4K39=54J89=H4I=9>34I99>>4I59>F4HG9>K4GO9?14G89?>4FN9?K4FD9@84F;9@E4F19A24EG9A@4EC9B04ED9B?4EF9C24EL9CK4F79DE4FB9E>4FM9F74G89F:4G59F14FM9EI4FE9EN4F:9F:4EM9FF4E@9G24E49GB4DO9H24DI9HC4DC9I34D=9IC4D79J34D19JD4CI9K54C?9KF4C59L74BK9LH4B@9M:4B69MK4AL9N?4AI9O44AG9OH4AE:0<4AC:144A<:1M4A5:2E4@M:3>4@F:474@?:4M4@6:5=4?G:5N4?7:6>4>H:6O4>9:7?4=I:804=::8@4<K:904<<:9A4;M::14;=::<4:G::F4:1:;049;:;;48E:;O489:<G47O:=>47F:>447C:>J47F:?>47J:@347N:@I483:A>486:B348::BJ489:CH47F:DE474:EC46B:F@460:G>45>:GO45I:H;46<:HH470:I447C:I@486:IE48L:II49A:IN4:7:J24:L:J74;A:JG4;I:KG4;I:L>4;@:M54;7:MM4:M:ND4:C:O;4:9;0349O;0J49E;1B49;;29492;3148H;3H48?;4?485;5747K;5N47A;6E477;7=46M;8446D;8C46@;8G46F;8N46I;9B46B;:546;;:H464;;C45L;<I45A;>0454;?744I;@=44>;AC442;BH43E;CL43;;CA43H;B:448;A444H;?M458;?7461;@846M;AA470;BJ473;D3475;E546N;F0461;G3453;H5446;H=450;IF44M;JH44H;K;44G;KN44F;LA44E;M444D;M744?;M944;;M;445;MC441;ML43L;N<43G;O243B;OH43?<0=43:<12436<1H432<2=42M<3>430<4542N<4K42K<5A42I<6842F<6O42C<7E42A<8:42A<8H42C<9742C<9E42F<9H42E<9K42E<9N42D<:242D<:542C<:942C<:<42C<:?42C<:B42C<:G42A<;:42:<;L423<<?41K<=241D<=D41<<>:415<?D40M<@N40D<B840=<CB405<E?3OK<GA3OB<H93O?<I23O<<IK3O8<JC3O2<K;3NI<L33NA<LK3N8<MA3MN<N53ME<NJ3M;<O>3M2=043LJ=0H3LB=153L:=1B3L4=1O3KL=2<3KD=2I3K>=373K7=3O3JM=4I3JA=5D3J6=6?3IL=7:3I@=853I6=943HF=:33H7=;13GG=<03G8=<N3FH==M3F8=>L3EH=?K3E9=@A3DK=@O3D@=A@3D3=B43CE=BG3C7=C:3BI=CM3B;=DA3AN=E43A@=EG3A2=F<3@E=G33@8=GK3@3=HD3@0=I<3?M=J43?J=JM3?G=KE3?D=L=3?A=M>3?:=NC3?1=OI3>H>0N3>>>233>5>383=L>4>3=C>5C3=9>6I3=0>7N3<G>9A3<?>;;3<7>=63;O>>;3<D>=M3<2>=@3;?>><3:N>?83:=>@539L>A139;>B1396>C2397>D4397>E5398>F539:>G139<>GM39C>HJ3:2>J839D>KE396>M238H>N:37M>NG376>O136>>O;35F>OE34L?00342?0:338?0M32B?1G31L?2B316?3L30C?57300?6A2O<?7K2NI?952N6?:32MJ?:C2MD?;42M??;E2M9?<62M3?<F2LM?=72LH?=H2LC?>92L=?>I2L7??:2L1
1796?44EN6?K4EC6@B4E86A94DM6B04DA6B@4D66BG4CI6BN4C=6C54C06C<4BD6CC4B76CJ4AJ6D64AF6DJ4AO6E?4B86F34B@6FG4BI6G<4C26H04C;6HD4CC6I94CL6IM4D66J@4D@6K44DJ6KG4E56L;4E?6LN4EI6MB4F46N54F>6NJ4FI6O=4G56OM4G@70?4GM7114H:71=4HD71A4HJ71J4I47244I=72>4IH72H4J27324J=73<4JF7494JM75<4K276?4K677A4K<78?4KC79=4KJ7:14KN7:F4L27;;4L67;K4L97<64L;7<G4LD7=44LG7=A4LK7=N4LN7>:4M27>G4M57?44M97?@4M=7?N4MA7@:4MD7@G4MI7A14N47A94ND7A@4O57AG4OF7B54OO7BG5027C95057CN5067DK4OO7EH4OF7FE4O>7G?4O:7H04OE7HA4OO7I250:7IC50E7J450O7JE5197K651D7KD51I7L151K7L=51M7LJ51O7M75217MC5227MO5247N<5267NI5287O;52;7O25287O85277O>5267OD5247OJ52380052180:52281652;82452D83152N83N53784K53@85>53D86053F86A53I87353M87E53O88854288I54589;54789M54:8:;54;8:<5468:<5438:=5408:=53L8:>53I8;753J8<F5448>954H8?M55;8A>55M8BB5678CG56@8D956G8DE56N8E15748E=57:8EI57@8F457F8F@57L8FL5828G85888GD58>8H058D8H;58J8HD5908I05928IJ58H8JC58>8K<5848L557K8M457;8N656H8O756590855C90<54O90:54990=53H90G53:91052M91:52?91C52291L51D92851892H50K93650>93F50394850294K50095>4OO9604OM96C4OL9754OG97F4O<9874O298G4NH99?4N?99N4O69:74N=9:F4MJ9;54M79;E4LD9<44L09<C4K=9=24JI9=F4IO9>64I=9>>4HN9>H4H@9?54H69?B4GL9@04GB9@=4G99@J4FO9A74FE9AD4F<9B34F<9BB4F>9C14F?9C@4FA9D54FH9DN4G39EG4G>9F:4GE9F14G<9EK4G39F74FF9FC4F:9FN4EM9G:4EA9GJ4E<9H:4E69HK4DM9I=4DD9IN4D99J?4CO9K04CE9KA4C;9L24C19LC4BF9M54B<9MF4B29N74AI9NL4AF9O@4AE:054AC:0I4A@:1>4A>:224A<:2J4A5:3C4@N:4;4@F:4O4@::5?4?K:604?<:6@4>M:714>>:7A4=O:824=?:8B4<O:924<@:9C4<1::44;B::B4;1::M4:;:;749E:;E492:<=48H:=448?:=K486:>B47L:?947J:?M47N:@B482:A8486:AM48::BB48>:C=484:D:47B:E8470:F546=:G345K:H0459:H?45D:HE46::HK470:I047G:I648=:I<494:IB49J:IH4:A:IN4;7:J74;C:JG4;M:L04;O:LH4;G:M>4;?:N54;7:NL4:O:OC4:E;0;4:;;124:1;1I49H;2A49>;38494;3O48K;4G48A;5>487;6547M;6M47C;7D479;8<470;9346F;9746K;9;470;9D473;:746L;:M46D;<4469;=:45M;>@45A;?F455;@K44I;B044=;C5441;D643J;D>445;C744E;B1454;@J45D;@246=;@K483;AN48H;C748K;D@48N;EE48K;F@47M;GC470;HE463;IH455;J>451;KD456;L:454;LN453;MA451;ML44O;MN44K;N044F;N244A;N744<;NA447;NJ442;O<43M<0143I<0G43E<1<43A<2243<<2O43><3N43?<4D43<<5;43:<62437<6H435<7?432<86430<8H42K<9;42I<9I42J<:442L<:742K<:;42K<:>42J<:A42J<:D42I<:G42I<:K42I<:N42I<;142I<;442H<;<42F<;N42?<<A427<=4420<=F41H<>F41A<@0419<A:411<BD40I<CN40A<EL407<G5401<GO3ON<HG3OJ<I?3OA<J73O8<JN3NO<KF3NF<L>3N=<M53N4<MJ3MJ<N>3MA<O33M7<OG3LM=0<3LD=0L3L<=1:3L5=1H3KN=263KH=2E3K@=333K9=3O3K4=4O3JJ=5F3J:=6=3IJ=783I@=843I3=933HC=:13H4=;03GD=;O3G4=<M3FE==L3F5=>K3EE=?J3E6=@H3DF=A:3D9=AM3CL=B@3C>=C43C1=CG3BC=D:3B5=DN3AG=EA3A9=F43@L=FG3@>=G>3@9=H63@6=HO3@3=IG3@0=J?3?L=K83?I=L03?F=LH3?C=MA3?@=NC3?8=OI3>O>0N3>E>233><>383>3>4>3=J>5C3=@>6I3=7>803<N>9J3<F>;E3<?>=?3<8>?93<0>?L3<7>?>3;E>?03;3>?:3:A>@63:0>A239?>B339>>C539>>D639?>E839@>F939A>G639C>H239N>I03:;>J>39M>KK39?>M8391>N>38?>NI37F>O336L>O=363>OH35:?0234@?0@33G?19331?2332<?2M31F?3L310?5730=?6@2OK?7K2O7?952ND?:32N7?:D2N2?;52ML?;F2MF?<72M@?<G2M;?=82M5?=I2M0?>:2LJ?>J2LD??;2L???L2L9
1806??4F@6@54F56@M4EI6AD4E?6B94E36B@4DG6BG4D:6BN4CN6C54CA6C<4C56CD4BH6CL4B>6D@4BF6E44BO6EI4C86F=4CA6G14CI6GF4D26H:4D;6HN4DD6IC4DM6J74E56JK4E>6K?4EI6L24F36LF4F=6M94FH6MM4G26N@4G<6O44GG6OG4H17084H>70J4HK7174I771;4I=71@4IC71F4IJ7204J572;4J@72E4JI72O4K473K4K:74O4K?7624KC7764KH7844KO78M4L579B4L87:74L<7:L4LA7;A4LE7<14LI7<=4LM7<J4M07=74M47=D4M97>14M>7>=4MC7>J4MH7?74MM7?C4N37@04N77@>4N<7A04N@7A74O17A>4OB7AF5027B55097BG50<7C:50@7CL50C7DE50@7EA5087F>4OO7G:4OJ7GK5047H<50?7HM50I7I>5147IO51>7J@51I7K15207K>5227KD5217L05237L=5247LI5267M65287MB52:7MO52;7N?52>7O352B7O352B7NG52>7NL52>7O352<7O952:7O?52980152>80N52G81K53082H53:83E53C84B53L85:54385L54586>54886O54:87A54=88454@88F54C89754F89I54I8:354J8:454H8:554D8:654A8:654=8:754:8;:54>8<G54E8>;5588?N55K8A;56;8B>56D8CC56N8D55748DA57:8DM57@8E957G8EE57M8F15838F<5898FH58?8G458F8G@58L8GK5928H15988H>5988I758N8J158D8JJ58:8KC5818L=57G8M657=8N456N8O556;8OD55H8OE5558ON54G90854990A53M90K53?91453191>52D91G52792451I92C51<93250O93@50B94150;94D50995650895I50696;50596N50497A50298350198D4OF9994OF99D4O89:34ND9:C4N19;24M>9;B4LK9<14L89<@4KE9=04K29=@4J@9=O4IM9>>4IB9>M4I=9?:4I49?H4HJ9@54HA9@B4H89@O4GN9A<4GD9AI4G:9B74G39BF4G59C54G79CD4G89D34G;9DB4G<9E84GD9F14GO9F04GK9F44G@9F@4G39FK4FG9G64F;9GH4EO9H64E@9HF4E39I74DJ9IH4D?9J94D59JK4CK9K<4CA9KM4C69L>4BL9M04BB9MA4B89N24AM9ND4AF9O94AD9OM4AC:0B4A@:164A>:1K4A<:2?4A9:344A7:3H4A5:4@4@O:514@@:5A4@0:624?A:6B4?1:734>B:7C4>3:844=D:8D4=4:944<E:9E4<6::64;F::F4;7:;34:D:;=49N:<249B:<I499:=A48O:>848F:>O48=:?G483:@<482:A2486:AG48::B<48>:C148B:CO47O:DL47=:EJ46K:FG46::GI45I:HD45E:HJ46<:HO472:I547I:I;48?:IA495:IG49L:IN4:=:J74:H:J@4;4:JJ4;?:K14;N:L44<2:M24;M:MI4;F:N@4;=:O64;5:OL4:M;0C4:E;1:4:=;214:4;2I49J;3@49@;47496;4O48L;5F48C;6=489;7547O;7L47F;8C47<;9;472;9G471;9K476;9O47<;:947<;;>470;<C46E;=H469;>M45M;@345A;A8454;B=44H;CB44<;D@449;E144C;D5452;BN45B;AG461;@L46I;AE48?;B?4:6;CD4:C;DM4:F;F54:G;G049J;H348L;I547O;J8472;K:465;L=458;M:45A;MM45@;N@45?;NC45;;NE456;NG451;NI44L;NL44H;O644B;O?44=;OI448<0;444<1043O<1F43L<2?43I<3F440<4>43N<5443L<5K43I<6A43F<7843C<8043D<8H43E<9643:<9?42I<:042E<:C432<:F431<:I431<:M430<;0430<;3430<;6430<;:430<;=42O<;@42O<;C42N<;F42N<<042K<<C42C<=642<<=H424<?241M<@<41D<AF41=<C0414<D:40M<F240D<FK40A<GC408<H;3OO<I23OG<IJ3O><JB3O5<K:3NL<L23NC<LI3N9<M>3N0<N33MF<NG3M=<O>3M4=0E3LN=0J3LA=163L7=1D3L1=223KJ=2@3KC=2N3K<=3H3K7=543K4=5O3JF=6F3J6=7=3IG=7O3I9=8D3HJ=9C3H;=:A3GK=;@3G<=<>3FL==<3F==>;3EM=?93E>=@83DO=AB3D8=B53CJ=BG3C==C:3C0=CM3BB=D@3B5=E23AH=EE3A;=F<3@O=G83@H=G@3@==H>3@9=I:3@4=J23@1=JK3?N=KC3?K=L;3?H=M33?E=ML3?B=ND3??=OI3?6>0N3>M>243>C>393>:>4>3>1>5C3=G>6I3=>>8:3=5>:43<N>;N3<G>=H3<?>?C3<8>A=3<0>A=3;K>@O3;9>@A3:F>@83:4>A539D>B639E>C839F>D939G>E;39G>F<39H>G;39J>H73::>I63:D>JC3:7>L139H>MF39:>N@38K>NK383>O637:>O@36@>OJ35G?0534M?0L346?1E33A?2?32K?39326?4231@?5630K?6@308?7K2OD?952O1?:42ND?:E2N??;62N8?;G2N3?<72MN?<H2MH?=92MB?=J2M<?>:2M7?>K2M2??<2LL??M2LF?@=2L@
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
1826@34GD6@K4G96AB4FN6B24FB6B:4F66BA4EI6BH4E=6BO4E06C64DD6C=4D76CK4D66D?4D>6E44DG6EH4E06F<4E96G14EA6GE4EJ6H:4F36HN4F<6IB4FE6J74FM6JK4G66K?4G?6L44GH6LH4H16M<4H:6MO4HD6NC4HN6O64I96OJ4IC70<4J070L4J;7104JB7144JH7194JN71=4K471A4K:71J4KD7244KN7304L57434L97574L>76<4LC77<4LJ77K4M178?4M77934M=79G4MC7:;4MH7;14MI7;H4MI7<:4MM7<E4N77<O4NA7=<4NF7=I4NK7>64O07>B4O67>O4O;7?<4O?7@24O?7@I4O>7A44OJ7A<50:7AC50K7B650N7BH5117C:5157CM5187D?51;7E551;7F15137FO50K7GA5137H251>7HC51H7I45227IE52;7J652?7JG52C7K852F7KF52I7KJ52G7KM52D7L152C7L;52D7LG52E7M752G7MJ52K7N>5307O65357NO5337NC52O7N752K7N652J7NB52K7O?53480<53=81:53G82654083354984154B84N54L85C55286555586F55887755:87H55=88:55@88K55C89<55F89?55F89B55E89E55E89G55E89G55A8:255?8;?55F8<M55L8>=5698?O56K8A35758B757>8C<57H8CN57O8D95858DE58;8E158A8E=58G8EI58M8F55938F?5998FD59@8FJ59F8G:59D8H459:8HM5908IF58F8J?58=8K85838L257J8LK57@8MD5778MK5758N=5718O156F8O:5688OD55K8OM55>90755090A54B90J54691353H91=53:91L52M92:52A92I52493851G93G51:94550M94H50K95:50I95M50H96?50G97250E97E50D98750B98F50?99@4OM9:04O;9:?4NH9:O4N69;?4MC9;N4M09<>4L>9<M4KK9=<4K99=L4JF9>;4J=9>J4J89?94J39?H4IN9@74II9@E4IC9A44I>9AC4I99B24I49B@4HM9BM4HD9C<4HF9CK4HG9D=4HG9DO4HF9EB4HD9F44HC9F;4HC9F:4HA9FB4H79GA4GH9I04G89I;4G29IH4FK9J54F:9JC4EI9K24E99K?4DH9KM4D79L;4CF9LI4C59M74BE9MG4B99N84AO9NI4AE9O>4AB:024A?:0G4A?:1<4A@:214AB:2E4AD:3:4AF:3O4AG:4D4AI:584AK:5H4A<:684@L:6G4@;:774?K:7F4?;:864>K:8E4>;:954=J:9D4=:::44<J::C4<9:;34;J:;C4;::<74;0:<M4:D:=D4:::>;4:0:?249G:?I49=:@@494:A848K:AO48A:BE48A:C:48E:D5489:E347G:F6478:G:46I:H?46::I346@:I9476:I?47L:IG48<:J048H:J9493:JB49>:JK49J:K44:5:K=4:A:KF4:M:KO4;::L64;J:L>4<9:MF4<::N=4<3:O44;J:OJ4;B;0A4;:;184;2;1N4:J;2E4:A;3<4::;424:1;4I49I;5?49A;66499;6M491;7D48G;8;48=;93484;9J47J;:A47@;:J47B;:N47G;;B47H;<847G;==47;;>B470;?H46D;@M467;B345K;C845?;D=453;E3458;EE45A;EO45L;DH46<;CB46L;BB47@;C;497;D44:M;DM4<C;EG4>6;G04>9;H04=C;I34<F;J54;I;K84:K;L:49N;M=491;N>483;O>46N;ON467<00461<0345M<0545H<0745C<0945><0?45:<0H454<1244O<1<44J<1E44E<2E44H<3K44N<4F44N<5=44K<6444J<6L44K<7D44K<8=44L<9544N<9I44I<:2449<:=43J<;143E<;K441<<143I<<443F<<643E<<743C<<943B<<;43@<<=43?<<>43=<<@43<<<B43:<<E43:<<H43:<<K439<=9434<>@42L<?J42D<A342=<B;425<CD41M<DI41F<EH41A<F@418<G440L<GI40@<H=405<I13OH<II3O?<JA3O6<K93NN<L13NE<LG3N<<M@3N3<NG3MM<ON3MI=0M3MA=123M4=173LF=1=3L7=1I3KO=273KH=2E3KA=3:3K;=4F3K8=623K5=793JO=7I3JE=8<3J7=933IF=9>3I9=9H3HM=:63H@=:D3H1=;:3GC=<93G3=>13F7=?L3E9=AF3D==B<3CM=BO3C@=CA3C3=D53BF=DH3B9=EH3B3=FI3AL=GI3AF=H23A;=H73@O=H<3@D=I;3@?=J83@:=K63@5=L43@0=M13?L=MI3?I=NA3?F=O:3?C>023?@>0N3?:>243?1>393>H>4>3>>>5C3>5>723=M>8L3=E>:G3=>><A3=6>>;3<N>@53<H>A;3<@>B;3<9>C13;O>B<3;=>AH3:M>A83<2>B33<D>BO3<E>CA3<E>DM3;K>F:3;0>G@3:F>H=3;6>IL3;8>KG3:H>M23:9>MG39K>N<39=>O038N>O;384>OE37:?0036A?0J35K?1D356?2>34@?3733J?41335?4K32??5D31I?6?313?7J30@?942OM?:62O>?:F2O9?;82O3?;H2NN?<92NH?<J2NB?=:2N<?=K2N7?><2N2?>M2ML??=2MF??N2M@?@?2M;?A02M6
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
1846@I4HH6A@4H=6AL4H16B34GE6B:4G86BA4FL6BH4F?6C04F26C74EF6CK4EN6D?4F66E34F?6EH4FH6F<4G16G04G96GE4GB6H94GK6HM4H46IA4H=6J64HE6JJ4HN6K>4I76L34I@6LG4II6M;4J16N04J:6ND4JC6O94JL6OM4K570@4K@70E4KF70I4KM70M4L37124L97194L?71A4LE71I4LK72;4M173C4M774K4M=7624MB76O4MJ7764N477<4N?77I4NF78=4NM7934O079J4O07:@4O07;74NO7;N4NO7<E4NO7=24O67==4OA7=G4OL7>25067><50A7?050C7?F50B7@=50B7A150C7A95137AD51?7B651B7BI51F7C;51I7CM51L7D@51O7E35227EE5267FB51N7G75217GH52<7H952F7HJ52J7I;52M7IL5317J=5357JN5397K?53=7KO53@7L253=7L653;7L953:7L=5377M853>7N253E7NC53I7NL53M7O453O7NH53J7N=53F7N153A7N<53D7O753M80454681254?82054I82M55283K55<84H55E85855K85I56186:56586K56887<56;87M56>88?56A88I56B88L56B88N56A89156A89456A89856B8:956G8;E56N8=25748>@57:8?G57F8@L5808B05898C458C8CF58I8D25908D=5968DI59<8E259B8E859I8E>59O8EC5:58F65:08FO59F8GI59<8HB5938I;58I8J458@8JK5878K25858K:5838KA5828KI5808L157O8LB57K8MG57D8NM57<8O956O8OC56A8OL56490655G90@55990I54L91454?91C54292253E92A53892O52K93>52>93M52194;51D94L51<95>51;96151:96C51897651797D51597O51298G50B99A5029:;4O@9:L4NN9;<4N<9;K4MI9<;4M69<J4LD9=:4L19=J4K?9>84K99>G4K49?64JO9?D4JI9@44JD9@B4J?9A14J99A@4J49B04IM9BC4IE9C74I=9D;4I59E>4HN9F04HL9FC4HJ9G64HI9GC4HH9G<4HK9GK4HK9I84HG9J74HA9JB4H;9JO4H59K;4GO9KG4GE9L54G59LC4FD9M14F39M?4EC9MM4E29N;4DA9NI4D09O24C>9O64BI9OG4BA:0:4B<:0N4B;:1C4B<:284B?:2L4B@:3A4BB:464BD:4K4BE:5?4BG:644BI:6F4B?:754AO:7D4A>:844@N:8D4@>:934?M:9B4?>::24>M::B4>=:;14=L:;B4=?:<44=4:<G4<J:=:4<@:=O4<5:>F4;J:?<4;?:@34;4:@J4:I:A@4:>:B64:3:BM49H:CB499:D8492:E<48C:F@485:GE47F:HI477:IH46G:J2473:J;47>:JD47I:JM485:K648@:K?48L:KI498:L249C:L;49N:LD4:::LL4:G:M44;6:M;4;E:MB4<4:MO4<@:O24<?:OH4<7;0?4;O;164;F;1L4;?;2C4;7;394:N;404:G;4G4:>;5=4:6;6449N;6K49F;7A49>;88496;8N48N;9E48E;:;48>;;2486;;I47M;<?47N;=247O;=F47O;>:480;?=47F;@B47:;AG46N;BM46B;D2466;E545L;EF466;F846@;FJ46J;FC476;E<47F;D7487;E049N;EI4;D;FB4=;;G;4?1;H44@H;HL4A3;IK4?O;JK4>K;KK4=F;LJ4<B;MJ4;=;NI4:9;OI495<0H480<1>473<1946I<1946A<1;46=<1>467<1@463<2@45K<3245@<2945<<3045C<4345K<50461<5I461<6A462<79463<82464<8J464<9B466<:;467<:E45H<;545><;I458<<>453<=2458<=8450<=?44H<=E44B<=I44=<=K44;<=L44:<=N448<=O447<>1445<>3444<>4443<>6442<>743O<>843L<>M43F<@543?<A=437<BE430<CG42I<DF42E<ED42@<FB42<<GA427<H741M<HK41A<I@415<J440I<JI40><K=402<L23OE<LN3O6<N13NH<O73NC=0>3N==153N3=1:3MF=1?3M8=1D3LJ=1J3L==1O3KN=2;3KG=2L3K@=483K==5?3K:=6@3K8=7K3K2=8I3J<=9B3IG=9M3I;=:93HO=:D3HC=:O3H7=;G3H:=<L3GN=>83FJ=?C3EF=A:3DG=BD3D1=C63CD=D53C==E53C7=F53C0=G63BJ=H63BD=HF3B:=HL3AN=I13AB=I63A6=I;3@K=J73@E=K53@@=L23@<=M13@6=MN3@2=NL3?L=OJ3?H>0H3?C>1A3?@>293?=>393?6>4?3>M>5J3>D>7E3><>9?3>5>:H3=N>;I3=F><J3=>>=K3=7>>L3<O>?M3<H>@O3<@>B63<9>B?3<C>B43=:>AJ3>1>B?3=L>CK3=1>E83<5>FC3;<>GA3;L>HN3<=>JI3;O>L63;?>LL3:O>MB3:?>N839O>NN39@>O@38N>OO386?0I37@?1B36K?2<365?3635??3O34J?4I344?5B33>?6<32I?76323?7O31=?9430I?:7309?:H303?;92ON?;J2OG?<:2OB?<K2O=?=<2O7?=M2O1?>>2NK?>N2NF???2NA?@02N;?@A2N5?A12MO
1846@<4IE6A34I:6AE4HN6AL4HB6B34H56B:4GI6BB4G<6BI4G06C04FC6C@4FE6D44FN6DI4G76E=4G@6F14GH6FF4H16G:4H:6GO4HC6HC4HK6I74I46IL4I=6J@4IF6K44IO6KH4J76L=4J@6M14JI6ME4K26N:4K:6NN4KC6OB4KL7074L570J4L87174L>71?4LC71G4LI71O4LO7284M472@4M;72H4MA73>4MG74F4ML75M4N276I4N:7704ND7764NO77=4O977I4OB78?4OC7964OC79M4OC7:C4OC7;:4OB7<14OB7<H4OB7=94OF7=D5017>050=7><50J7>K5137?@5147@75137@M5127A751@7AD51J7B751M7BI5207C;5237CN5277DA52:7E352=7EE52@7F<52?7G252A7GC52L7H45307HE5337I65377IG53:7J853>7JI53B7K:53F7KK53I7L553J7L853H7L<53F7LE53F7M@53M7N55437N?5467NH5497O154;7NJ5497N?5447N35407NA5437O654980254B80O54L81M55582K55>83I55H84C56085356785C56>86456D86E56G87656J87G56M88956O88?57088B57088E57088G56O88M5718975768:@57<8;L57B8=857H8>B57M8?C5838@H58=8AL58F8C15908CB5968CN59=8D959C8D?59I8DD59O8DJ5:58DO5:;8ED5:68F=59L8G659B8H05998HI5908I>58G8IF58F8IN58D8J558C8J=58@8JD58?8JL58=8K358<8KE5888LJ5808MO57I8O457A8O>5748OG56F90156890;55K90D55>91055191?54D91M54792<53J92K53=93:53093H52D94752794F51J95751D95J51C96<51A96O51@97:51>97D51;98;50M99550<99O4OL9:I4O:9;;4NH9;J4N59<94MC9<I4M09=84L>9=H4KL9>74KG9>F4KB9?44K<9?D4K79@24K29@A4JL9A04JG9AB4J?9B54J69BG4IN9C?4IF9DB4I>9EE4I79FI4HO9G=4HM9GO4HK9H44HL9HE4HL9IL4HJ9J<4HJ9JL4HL9K<4HL9KH4HF9L44HA9L@4H;9LN4GJ9M<4G99MJ4FI9N84F89NF4EG9O24E69O64DA9O:4CM9OB4C=:064C8:0I4C3:1=4BN:214BK:2F4BN:3:4BO:3O4C1:4D4C2:594C5:5N4C6:6B4C8:744C0:7D4B@:834AO:8C4A?:924@O:9B4@?::14?O::A4?>:;04>N:;A4>A:<44>6:<G4=L:=94=B:=L4=8:>@4<M:?74<B:?M4<7:@D4;L:A;4;A:B14;6:BD4:E:C64:1:CE49L:D:49J:E@49<:FE48N:GI48?:HM481:IN47=:JE46O:JN47::K747F:K@482:KI48=:L248H:L<494:LE49?:LN49K:M74:7:M?4:D:MF4;4:MM4;C:N54<2:N<4<A:O<4<F;034<=;0I4<5;1@4;M;264;E;2M4;=;3D4;5;4:4:M;514:D;5G4:=;6>4:4;7549L;7K49E;8B49<;99494;9O48L;:E48C;;8487;<6483;=8480;=J482;>>483;?1484;?J481;@O47F;B5479;C:46M;D?46A;E?46;;F046E;FB470;G3479;G@47C;F9483;E348C;EK4::;FD4<0;G=4=G;H64?=;HM4@C;IA4@C;J54@C;K14?N;L04>J;M04=E;MO4<A;NN4;=;ON4:8<0M494<1M47O<2447=<20473<1M46I<1O46D<2246@<2<46<<2I464<2M460<3D462<4F46:<5C46@<6;46D<7346G<7L46G<8D46H<9<46I<:546J<:K46H<;646:<;K465<<@45O<=545J<=E45K<=L45D<>245<<>9454<>?44N<>D44H<>E44G<>G44E<>H44D<>J44C<>L44B<>M44@<>O44><?044;<?1447<?1444<@943L<AB43E<BG43><CE439<DD435<EB430<F@42L<G?42G<H=42B<I2427<IG41K<J<41@<K0414<KD40G<L940<<M93OJ<N;3O8<O@3NL=0G3NG=183N<=1>3MO=1C3MA=1H3M4=1M3LF=233L7=283KJ=2E3KB=403K>=553K;=6?3K5=7J3JN=8K3J<=9D3II=9O3I<=::3I0=:E3HD=;93HC=<83I0==?3H>=>J3G;=@63F7=AA3E3=BK3D5=CK3CN=DL3CH=EL3CB=FL3C;=GL3C5=HM3BO=I33BC=I83B8=I=3AK=IB3A@=IG3A4=J63@K=K43@F=L23@A=M03@<=MM3@7=NK3@3=OI3?M>0G3?I>1D3?C>2C3??>3<3?;>4?3?3>643>K>7?3>D>8@3>=>9A3>5>:G3=N>;O3=G>=73=A>>?3=:>?G3=4>@O3<M>B03<G>BJ3<B>BI3=1>B?3=H>BD3>1>D03=6>E=3<<>FD3;O>GG3<?>I<3<J>K73<:>L33;J>LI3;:>M?3:J>N53::>NK39J>OA39:?0;38E?15380?1N37:?2H36E?3B35O?4;359?5534D?5N33N?6H338?7B32B?8;31L?95316?:830F?:I30@?;:30:?;K305?<;2OO?<L2OJ?==2OD?=N2O>?>>2O9?>O2O3??@2NN?@12NH?@A2NB?A22N=
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
2016G94KJ6GM4L36HB4L;6I64LE6IJ4LM6J?4M66K34M?6KG4MG6L<4N16M04N96MD4NB6N94NJ6NM4O36O@4O87034O;70G4O>71:4OA71M4OE72A4OH7344OK73G4OO74:50274N50575A50976450<76G50?77750E77E50L78251478@51;78M51C79=51I7:351I7:J51I7;A51I7<551K7<F51O7=75237=H5277>952<7>J52@7?852K7?J52O7@=5327@O53:7A>53A7AO53F7BA53K7C25407CD5457D65497DG54>7E954B7EJ54H7F;54I7FJ54I7G954H7GK54I7H=54K7HO54L7I@54N7J254O7JD5517K65527KG5547L95557LH5587M255<7M;55?7MD55B7MN55D7N755G7N@55J7NI55M7NI55L7NB55I7O755O7OL56580@56;81556A81J56G82B56O83<57883M57?84=57F84M57M85=58385N58:86>58@86N58G87858J87B58O87M59488759988A59>88K59B89M59I8;959O8<C5:28=C5:18>D59O8?85:38?I5:88@:5:>8A05:C8B65:L8B<5;28BC5;88C<5:N8CM5:H8D55:G8D<5:E8DD5:C8DK5:B8E35:@8E;5:>8EB5:=8EJ5:;8F15:98F95:88F@5:68FH5:58FO5:38G75:28G?59O8H159L8I659D8J;59<8K@5958LE58M8MJ58F8NO58>8OE58390057F90?57990N56L91<56?91K56292955E92I55893754L93F54=94453O94B53A95153395M52G95E52D96052B96:52@96K52897E51G98>51699850E9:25049:L4OC9;F4O29<@4NA9=34MO9=B4MC9>14M>9>D4M49?64LL9?I4LD9@;4L;9@N4L39A@4KJ9B34KB9BF4K99C94K19D<4JI9E@4JB9FC4J:9GF4J69HF4J;9I@4J99IN4IJ9J<4I;9JJ4HM9K<4HJ9KN4HN9LA4I09M44I29MG4I59N94I79NL4I:9O@4ID9OD4I09OG4H;9OK4GF9OO4G2:094FC:0L4F?:1@4F9:234F4:2G4EO:3:4EJ:3N4EF:4B4E@:554E;:5I4E6:6<4E1:714E0:7E4E3:8:4E4:8O4E6:9>4DE:9N4D5::>4CD::M4C5:;?4BI:<24B?:<E4B5:=74AK:=J4A@:>;4A5:>K4@G:?;4@::?K4?M:@64?0:@D4>2:A64=>:AE4=9:B54=4:BD4<O:C44<I:CC4<D:D24<@:DB4<::F24<B:G64<3:H:4;D:I?4;6:J=4:>:K:49F:L748M:M4484:ML47C:N647O:N?48::NH48E:O1492:O:49=:OB49K:OI4:;;004:J;084;9;0?4;H;0F4<7;0M4<F;184=4;224<O;2I4<F;3@4<>;464<6;4M4;N;5C4;F;6:4;>;6N4;3;7A4:G;844::;8G49N;9;49A;9N495;:C48L;;>48N;<=48N;=K48H;?848A;@F48;;AL487;B;48;;BJ48?;C948C;CL48D;DO48;;F447O;FE489;G748B;GH48L;H:496;HL49@;I=49J;HG4:8;I64;H;IN4=?;JC4=E;K74=E;KK4=E;L?4=F;M44=F;MH4=G;N<4=G;O04=H;OD4=H<084=H<0L4=I<1C4=B<2B4<=<3B4;9<4B4:4<54499<5048O<4L48D<4H48:<4K489<5;48:<6148;<6F48<<7?48@<8948E<9248L<9L491<:C48M<;848H<;N48B<<D48=<=9488<=N483<>D47N<?:484<?G483<?O47L<@647F<@>47?<@E478<@M470<A546J<A<46C<AB46=<AH464<AG460<AF45L<AE45H<AD45D<AC45A<AB45=<AI459<B4456<BM44N<D244B<E4449<EM44:<FG44;<G@44;<H:44;<I344;<J2447<K0442<KM43L<LF43?<MG42L<NI420<OJ414=0B40;=1:3OC=223NL=2D3N6=2K3MF=323M6=393LF=3H3L:=4>3L0=543KG=5I3K>=6?3K4=753JK=7J3JA=903J:=9I3IL=:B3J2=;A3J@=<?3JN==>3K;=>=3KD=?I3J@=A33I@=B83I0=C<3HB=D@3H3=ED3GD=FE3FM=GE3EO=HE3E1=IF3DG=JC3DG=JH3D6=JL3CF=K13C6=K63BF=K;3B:=K@3AN=KE3AB=KN3A8=LL3A3=MJ3@N=NH3@I=OF3@E>0E3@D>1D3@C>2C3@B>3C3@A>4F3@?>5L3@<>733@6>8;3@0>9C3?I>:L3?C><33?=>=<3?6>>A3?0>?;3>J>@53>D>A03>>>AJ3>9>BD3>3>C?3=N>D93=G>DE3=M>F13=K>GE3>=>I:3>H>J:3>8>JO3=H>KI3=7>LC3<F>M=3<5>N83;D>O23;3>OL3:A?0F3:0?1@39:?2938B?3437K?3M376?4H36@?5B35K?6=356?7734A?8033F?8G32E?9>31D?:530D?:A30C?:K30J?;<30E?;M30??<=30:?<N304?=?2ON?>02OI?>A2OC??12O>??B2O8?@32O2?@D2NM?A42NG
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
2056I54N=6IJ4NE6J>4NN6K24O76KF4O?6L;4OI6LO5016MD50:6N850B6NL50G6O?50J70250N70E51171851471K51872?51;73251>73E51A74851D74L51H75?51K76351N76F52177:52477M52778@52:79452=79E52A7:352I7:E52M7;;52M7;L5327<=5367<N53:7=?53?7>153C7>B53H7?353M7?E5417@75497@I54@7A;54I7AH54O7B75567BF55<7C755A7CI55E7D;55J7DL55N7E>5647EN5657F>5647FM5627G<5617GL5617H=5637HO5657IA5667J35687JD5697K656;7KH56<7L656;7LE56;7M156;7M;56<7MD56=7MN56>7N856A7NA56D7NJ56G7NM56G7OB56N80657480K57:81>57@82057G82@57M82O58383>58983N58@84>58G84N58N85=59685N59=86B59H8755:287A5:887K5:=8865:B88@5:F88N5:K8::5;18;C5;58<D5;48=D5;28>D5;18?95;48?J5;:8@;5;>8@M5;D8A>5;H8A@5;G8AC5;E8AK5;D8B25;B8B:5;A8BA5;?8BI5;=8C15;;8C85;:8C@5;88CG5;78CO5;58D65;48D>5;18DE5;08DM5:N8E45:M8E<5:K8ED5:J8F65:F8G;5:>8HA5:68IF59O8JK59G8L059?8M55988N>58O8OG58C90558690D57G91257991A56K91O56=92>55N92L55@93;55293I54C94854694M53H96653>96I53796853495F53296252M96L52<97F51K98@51;99:50I9:45089:N4OF9;I4O59<C4NC9==4N49>64MM9>L4MF9??4M>9@24M59@D4LM9A64LE9AI4L;9B<4L39BN4KK9CG4KB9DK4K;9EN4K39FN4K99GN4K=9HN4KB9IF4K?9J44JO9JB4J@9K04J19K>4IC9KL4I49L:4HD9LM4HF9M?4HH9N24HJ9NE4HM9O:4I:9OO4IG:064I9:094HD:0D4H6:174H1:1K4GL:2>4GH:324GB:3F4G>:494G8:4M4G3:5@4FO:644FI:6H4FE:7;4F?:7O4F;:8B4F6:964F3:9K4F4::<4EH::K4E8:;>4DM:<04DD:<A4D7:=14CJ:=A4C<:>14BN:>@4BB:?14B4:?@4AF:@04A9:@>4@C:BD4?D:AB4?2:A=4>H:AL4>B:B;4>>:BK4>8:C:4>3:CJ4=N:D94=I:DH4=D:EK4=G:G?4=E:HC4=7:IG4<I:JD4<0:KA4;7:L>4:>:M;49E:N948L:O5483:OE483:ON48?;0848J;0@498;0G49G;0N4:6;154:E;1=4;4;1D4;C;1K4<3;224<B;294=1;2G4=;;3>4=3;444<K;4I4<A;5<4<4;5O4;H;6B4;<;764:O;7I4:C;8;4:7;8N49J;9B49>;:9499;;449;;<049<;=7498;>E492;@348L;A@48E;BN48?;D648;;DD48?;E348B;EB48G;F148K;FG48M;G9497;GJ49A;H<49K;HM4:4;I?4:>;J04:I;JA4;3;JJ4<6;K>4<6;L24<6;LF4<6;M;4<7;MO4<7;NC4<8;O74<9;OK4<9<0?4<9<134<9<1H4<:<2<4<:<304<;<3D4<;<484<<<4L4<<<5L4;7<6E4:7<6A49M<6A49H<6C49H<6E49H<6M49H<7D49K<8=49N<9349H<9I49B<:?49=<;5498<;K492<<A48M<=648H<=L48B<>A48><?948H<?K48M<@848L<@F48L<@N48E<A548><A=487<A<483<A6481<@O47O<A:47C<B1470<BA46@<B@46=<B?469<B>465<B>461<BE45N<C045J<C:45G<CE45D<D045A<E4455<F:44H<G@44<<HE43O<IH43E<JB43F<K;43G<L543G<LF43N<M<43B<N;42G<O341O<OJ417=0B40?=1:3OG=213NO=2I3N7=3A3M?=453LM=4K3LD=5?3L<=643L4=6I3KK=7>3KC=833K;=8H3K2=9=3JI=:33JC=:M3K1=;H3K>=<C3KL==A3L9=>@3LG=?B3LE=@I3KN=AN3K>=BN3JB=CM3ID=DM3HF=EL3GH=FL3FK=GL3F4=HM3EK=IN3EB=K33EC=KD3EB=KH3E2=KM3DA=L13D1=L63C@=L;3C0=L?3B?=LD3AO=LK3A@=MJ3A?=NI3A>=OH3A=>0G3A<>1F3A;>2E3A:>3E3A9>4K3A5>603A2>763@O>8>3@K>9E3@D>:N3@=><63@7>=63@1>>13?K>>K3?F>?E3?@>@@3?:>A:3?4>B43>O>C:3>H>DA3>B>EA3>;>FO3>N>H?3?D>I?3?>>J:3>M>K43><>KN3=K>LH3=9>MB3<H>N=3<7>O73;F?013;5?0L3:C?1E39O?2@398?3938A?4337K?4M374?5G36=?6?35A?7634A?7M33??8D32??9;31=?:230=?:C2OM?:J30:?;230G?;C30A?<330<?<D307?=5301?=F2OK?>72OE?>G2O@??82O;??I2O5?@:2NO?@J2NI
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
2076II50=6J>50F6K250O6KF5186L;51A6LO51I6MC5226N75276NJ52:6O=52=70052A70D52D71752G71J52J72=52M73053073D53374753774J53:75>53=76153@76E53C77853F77L53I78?53L79353O79F5427:;5457:N5437;C5497<754@7<H54E7=:54J7=L54O7>=5537>N5587?@55<7@255A7@D55H7A655O7AE5677B356=7BA56D7BN56K7C=5717CN5767D@57:7E257@7EB57@7F157?7F@57=7FO57=7G?57;7GN57:7H>57:7I057<7IB57=7J357?7JE57@7K457@7KB57@7L157?7L?57>7LN57=7M957>7MC57@7ML57@7N657B7N@57C7NM57G7O?57M80058380A58:81258@81B58F82058L82?59282M59883<59=83L59D84<59L84N5:585B5:?8665:J86J5;487=5;?87J5;E8845;J88?5;O89;5<48:D5<98;D5<78<E5<68=E5<48>E5<38?;5<68?L5<:8@=5<@8@O5<C8AB5<G8AM5<H8AH5<E8AD5<B8A?5<@8A;5<=8A65<:8A25<78A65<68A>5<48AE5<28AM5<18B45;O8B<5;M8BC5;L8BK5;J8C35;H8C:5;G8CB5;E8CI5;D8D<5;@8EA5;88FF5;08GK5:H8I05:A8J>5:68L159I8MC59<8NK5948OJ58J90958<90G57M91657?91D57192356C92B56593055F93>55893M54J95454?96>54697H53L97F53G97553D96C53A96K52O97252=97L51K98F51999@50H9:;5069;54OE9;O4O39<J4NB9=C4N79><4N19?54MJ9?O4MD9@H4M=9A?4M69B24LM9BD4LE9C74L<9D64L59E74L69F74L;9G74L@9H64LE9I74LK9IK4LC9J94L59JH4KE9K64K69KD4JG9L24J99L@4IJ9LN4I:9M<4HK9MK4H>9N=4HA9O24HN9OH4I;:0=4II:0O4IJ:1C4ID:264I@:2H4I9:394I2:3J4HK:4;4HD:4L4H>:5=4H7:5N4H0:6?4GI:704GB:7A4G;:824G5:8C4FN:9;4FL:::4FL:;24FI:;A4F<:<14EO:<A4EB:=04E4:=@4DG:=O4D9:>>4CL:>N4C>:?>4C1:?M4BC:@J4B6:E04AK:E?4AF:E24A7:D04@E:BN4@3:BB4?G:C24?B:CA4?=:D14?7:D@4?3:DO4>N:EB4>J:GG4?8:HK4>J:IO4>9:JL4=A:KI4<H:LF4;O:MC4;7:N@4:>:O=49E;0:48L;14488;1=48C;1E492;1L49A;234:1;2:4:@;2B4:O;2H4;@;2M4<2;324<D;374=6;3=4=E;3J4=5;4=4<I;514<=;5C4<1;664;D;6I4;8;7=4:L;804:?;8C4:3;9649G;:049E;:K49H;;F49I;<D49J;>249C;??49=;@M496;B:490;CH48I;E548C;F?48>;FN48B;G>48G;H348N;HH495;I949>;IG49J;J04:7;J94:D;JA4;1;JK4;?;K44;N;KH4<6;LM4;N;MA4;M;N54;L;NI4;J;O>4;I<024;G<0F4;F<1:4;E<1N4;C<2B4;B<364;@<3K4;?<4?4;=<534;<<5G4;;<6;4;:<6O4;8<7C4;7<834;2<834:J<824:D<824:=<844:6<8G4:3<9:4:0<9M49M<:@49J<;349F<;G49C<<=49><=3498<=I493<>>48N<?749;<?M49F<@;49F<@J49F<A849E<AD49B<A@49?<A949><A249<<@K49;<@O495<A<48C<B447O<BL47<<C;46M<C;46J<C:46F<CA46C<CK46@<D646=<D@469<DK466<E6463<EA460<F645I<G<45<<HB450<IG44C<JM447<L143E<L=43E<LK44:<M<43L<N3435<NJ42=<OA41G=08410=0O409=1E3OB=2I3NN=3O3N:=563MF=673M6=6L3LM=7A3LE=863L==8K3L4=9@3KL=:53KD=:K3K@=;F3KN=<A3L;==;3LI=>63M7=?13MD=?E3MA=@73M7=A53L<=B43K>=C43J@=D43IB=E33HD=F33GF=G33G8=H43FO=I53FG=J63F>=K73F5=L?3FN=LD3F==LI3EM=LN3E<=M23DL=M73D<=M<3CK=MA3C:=MH3BJ=N03B:=NK3B6=OJ3B5>0I3B4>1I3B3>2G3B2>3J3B0>503AM>663AJ>7;3AF>8A3AC>9H3A?>;03A8>;L3A2><F3@L>=C3@F>>K3@@>@23@9>A93@3>B@3?L>CH3?E>DO3??>EN3?8>FI3?1>H83?F>I:3?O>IJ3?G>J93??>K43>O>L23>>>LN3=M>MH3=;>NB3<K>O<3<9?073;H?113;7?1K3:E?2E39N?3?397?4838A?5137H?5H36M?6?35M?7634L?7M33K?8D32K?9;31J?9L319?:7316?:A313?:L310?;A30@?<330:?<F304?=82ON?=J2OG?>=2OA?>O2O;??A2O5?@32NN?@F2NH?A82NB?AJ2N<
2086J351>6JG51G6K<51O6L05286LD52A6M952J6ML52N6N?5316O35356OF53870953;70L53>71?53B72253E72F53H73953L73L53O74@54275454575G54876:54;76N54>77A54A78554D78H54F79<54I7:454M7:F54K7;954H7;M54L7<B5557=755?7=I55D7>;55H7>L55N7?=5627?O5677@A56;7A356C7AC56K7B15717B?5777BM57>7C:57E7CH57L7D:5807DL5857E<5867EK5847F:5837FI5827G85817GG5807H757O7HH5807I95817IK5837J;5837JJ5837K85827KG5817L55807LD5807M257O7M=5807MG5817N05837N:5837NH5877O958=7OJ58D80;58J80K59081959681H59<82659B82E59G83359M83C5:38455:;84H5:F85<5;18605;;86D5;F8785<187I5<:8845<>88>5<C89A5<I8:E5<J8;E5<H8<E5<G8=E5<E8>F5<C8?<5<F8?M5<K8@>5=08A15=38AD5=68B35=88AO5=58AJ5=38AF5=08AA5<M8A=5<J8A85<G8A35<E8@O5<B8@J5<@8A05<=8A75<<8A?5<:8AF5<98AN5<78B55<58B=5<48BD5<28BL5<18C?5;M8DD5;E8EI5;>8FN5;58H=5:J8J05:=8KC5:08M459D8N<59<8OC59490358F90B58891057J91?57;91M56M92<56?92J56193955B93N55595854L96A54B97K54898>54097L53N97;53J97B53897I52F98052498<51A9975109:150?9:K4OM9;F4O<9<@4NJ9=94N<9>34N69>L4MO9?E4MH9@>4MA9A74M;9B14M49BI4LN9C;4LE9D<4LE9E;4LJ9F;4LO9G;4M49H;4M99I;4M>9IN4M69J<4LG9JK4L89K94KI9KG4K:9L54JK9LC4J<9M14IM9M?4I=9ML4HN9N;4HA9N>4H>9NK4HG9O94I09OF4I8:074I1:0H4HJ:194HD:1J4H>:2;4H7:2L4H0:3=4GI:3N4GB:4?4G;:504G5:5A4FN:6?4FN:7=4FO:8<4FO:9:4G0::94G0::N4FL:;C4FH:<44F<:<C4EO:=34EB:=B4E4:>24DG:>A4D::?14CL:?@4C>:@04C1:C64BC:D?4B>:DM4B9:E<4B3:EI4AH:DF4A6:CD4@C:C54@7:CE4@1:D44?M:DC4?G:E24?B:EB4?>:GJ4@0:I04?C:J34?2:JO4>9:KM4=@:LJ4<H:MF4;O:ND4;6:O@4:>;0>49E;1;48L;1N48C;23495;2849H;2>4:9;2C4:K;2H4;=;2N4;O;334<A;384=3;3=4=E;3F4=G;434=5;4A4<H;544<<;5G4;O;6:4;C;6M4;7;7@4:J;834:>;8F4:1;9=49K;:849M;;349N;;O4:1;=149O;>?49H;?M49B;A:49;;BH495;D548O;EC48H;G048B;GL48B;H?48H;HM48L;I;49;;ID49H;IM4:5;J64:B;J?4;0;JH4;=;K24;M;K;4<<;KD4<K;M04<C;N94<9;NM4<8;OA4<7<054<6<0I4<4<1>4<3<224<1<2F4<0<3:4;O<3N4;M<4B4;L<564;J<5K4;I<6?4;G<734;F<7G4;E<854;J<8;4;7<8B4:D<8A4:=<8@4:6<8J4:1<9=49N<:049K<:C49H<;649E<;I49A<<=49><=049;<=C498<>849;<>J49><?B49K<@64:3<@D4:2<A44:1<AC4:0<AN49L<B649E<B649A<AO49@<AH49><B4496<BE48M<B748?<BM47L<CE479<CH472<CI46N<D446K<D>46H<DI46E<E446B<E?46?<EI46<<F4469<F>466<G:45L<H@45@<IE451<JF444<K?43B<L3448<L?44L<M5447<MK43?<NC42I<O9422=0041;=0H40D=1N401=353O==4;3NI=5B3N6=6H3MC=7C3M7=883LN=8L3LF=9B3L>=:73L4=:K3KL=;E3L5=<@3LC==:3M0=>53M>=>N3MG=??3M==@13M4=@C3LJ=A83L:=B83K==C73J?=D73IA=E63HC=F63GM=G73GE=H83G==I93G4=J:3FK=K;3FC=LN3GD=M23G3=M73FB=M<3F2=M@3EB=ME3E1=MJ3DA=N03D0=N73C?=N=3BN=NN3BC=OK3BA>0J3B@>1J3B?>2I3B>>3L3B;>523B8>683B5>7=3B2>8D3AN>9I3AK>:J3AE>;O3A?>=73A8>>>3A2>?E3@K>@L3@E>B43@>>C<3@7>DC3@0>EF3?I>FA3?C>G=3?<>HL3@2>IC3@0>J23?H>JB3?@>K13?8>KM3>G>LK3>7>MH3=E>NE3=4>O?3<C?0:3<2?143;A?1N3;0?2H3::?3B39C?4:38K?5237O?5H374?6?364?77353?7M342?8D332?9:325?9E322?:031O?::31L?;231>?<930D?<F302?=82OL?=J2OF?>=2O??>O2O9??A2O3?@42NM?@F2NG?A82N@?AJ2N:?B=2N4
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
22874159@74E59C75959F75L59I76D59M77C5:178;5:278N59O79@59L7:259I7:E59F7;759C7;J59@7<<59=7<N59:7=A5977>35957>G59;7?<59D7@259N7@G5:87A<5:B7B05:H7AJ5;;7A95<67B45;H7BO5;;7CJ5:M7D<5:J7DK5:L7E;5;17EK5;77F<5;;7FL5;A7G;5;E7G?5;H7GC5;K7GK5;O7H15;M7H25;G7H=5;C7HI5;@7I55;=7IK5;>7JG5;D7KC5;I7L?5;N7M<5<37N75<87NE5<=7O45<@7OD5<G8045<M80E5=48155=:81I5=<82=5==8305=?83D5=@8485=B84K5=C85?5=E8635=G86E5>28755>?87F5>M88;5?58955?389O5?18:J5>O8;D5>N8<>5>L8=75>L8=K5>N8>?5>O8>L5>H8?<5>I8?L5>L8@=5>N8@N5?08A75?18A35>O8@O5>L8@K5>J8@H5>H8@C5>E8@?5>C8@;5>A8@75>?8@35><8?O5>:8?L5>88?H5>68?M5>28@65=O8@?5=K8@H5=F8A15=C8A:5=?8AC5=;8B:5=48CM5<G8E@5<:8G35;M8H@5;C8IH5;;8JO5;28L65:K8M=5:B8ND5::8OJ5:290959C90G59691658G92358;93=58194G57G96157=97;57498D56J99N56@9;85669<?55K9<;55<9<854L9<454<9<053L9;M53<9;I52L9;F52<9;B51L9;@51=9<;51;9=75189>35159>N5129?J5109@E50N9AB50N9BA5119C;5109D550N9DN50N9EG50L9FA50K9G:50J9H350H9HL50F9I95079IG4OH9J:4O<9K44O39KM4NJ9LF4NB9M?4N99N84N19N34M?9MI4LM9M?4L:9M84KH9MB4K;9MK4JM9N54J@9N>4J3:024JL:1C4K@:214K2:2?4JE:2M4J7:3;4IJ:3K4I>:4I4I>:5H4I?:6F4I?:7D4I@:8C4IG:9?4J4:9O4I:::?4HA:;24H5:;F4GN:<:4GH:<N4GB:=B4G;:>64G4:>J4FO:?>4FJ:?N4FE:@84FB:@B4F?:@L4F<:A64F8:A@4F5:AN4F0:B=4EL:BK4EG:C94EC:CH4E>:D64E9:DD4DM:E34DC:F64DC:H54DG:IH4D>:K64CN:KG4CF:L84C>:LJ4C2:MB4B?:N;4AL:O34A9:OK4@E;0D4@2;1<4?>;254>K;2M4>8;3F4=D;3L4>3;414>E;474?7;4=4?C;4K4?1;584>>;5E4=M;624=;;6?4<I;6M4<7;7:4;E;7>4<9;7O4<8;8J4<6;9F4<3;:B4<2;;=4<0;<94;M;=44;K;>34<6;?74<9;@<4<=;AA4<@;BG4<D;CL4<H;E14<K;F64<O;G;4=2;H@4=5;IE4=9;J84=>;JB4=B;JN4=F;K84=J;KC4=O;KM4>3;L74>@;L@4>O;LI4?>;M34?M;M=4@=;MF4@L;N34A8;OE4@L<164@@<2H4@3<4:4?G<5K4?:<7<4>N<8?4>H<934>F<9=4>N<8I4@5<914?B<9>4?1<9K4>A<:94>1<:F4=@<;34=0<;A4<@<;L4<D<<74<I<<C4<L<<O4=0<=:4=4<=F4=9<>14=<<>B4=@<?54=E<?H4=H<@<4=M<@O4>1<A?4=O<AN4=H<B=4=B<B<4=7<B54<L<AN4<A<AG4<6<C54;M<ED4;G<F84;?<FL4;7<G@4:N<GI4:D<GH4:7<GG49I<GE49<<GD48O<GB48A<FM48<<G248<<G;48<<GF48<<H048<<H:48<<HC48<<HM489<I6486<I?481<IH47M<J747J<K547K<L347L<M147K<N447:<O546J=0746:=1945I=2;458=3=44H=4?448=5A43G=6C437=7E42F=8G426=9I41E=:K415=;M40D==340@==H405=>=3OH=?13O==?G3O1=@;3NF=A03N9=AE3MN=B:3MB=BN3M7=CC3LJ=D83L?=E23L7=F03L1=FN3KL=GL3KH=HK3KB=II3K==JH3K8=KF3K2=LD3JN=MB3JH=N@3JC=NK3J3=OC3J9=OF3IC=ON3I3>083HE>0A3H7>0J3GI>143G;>1>3FM>1G3F>>203F1>2:3EB>303EE>3K3E?>543E>>6>3E>>7H3E>>963E9>:E3E1><33DI>=B3DA>?13D9>@=3D2>AC3CL>BI3CF>CO3C?>E53C9>F;3C3>GB3BL>HE3BF>I@3B?>J:3B8>JJ3AN>K:3AB>KJ3A8>L:3@M>LI3@B>M93@7>MI3?L>N63?;>NC3>J>O33>8>OM3=D?0G3<N?1A3<:?2;3;F?383:O?463:8?5339A?5J38E?6<381?7>37;?8E36B?9L35H?;234O?<9345?=@33<?>F32C??M31I?A4310?B;307?A;2N2?AN2ML?B@2MF?C22M@?CD2M:?D72M3?DI2LM?E;2LG?EM2LA?F@2L;?G22L4
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
2487;:5G>7;M5G;7<?5G87=15G57=D5G27>65FO7>H5FL7?:5FH7?M5FE7@=5F=7@A5E;7@E5D97AI5DH7BI5E47CI5EB7DJ5EO7E>5F77F05F?7FB5FF7FK5FM7FN5G37G@5G97H55G>7HJ5GC7I@5GH7J>5GF7K=5GD7L<5GB7M;5G@7N95G?7O65G?8035G>80I5GC81=5GG81N5GJ82@5GL8325GO83D5H28465H484H5H785:5H985L5H<86>5H?8705HA87B5HC8815HC88=5HB88H5HB8945H@89?5H?89L5H<8::5H58:H5GN8;65GG8;D5G?8<25G98<B5G18=75FE8=L5F:8>A5EN8>M5EH8?85EC8?C5E>8?N5E98@95E48@E5DN8A05DJ8A;5DD8AF5D?8B15D:8B<5D58BG5D08C35CK8C>5CF8CI5CA8D45C<8D?5C68DJ5C28E55BL8EA5BH8EN5BD8F:5B@8FG5B=8G35B98G@5B68G45A@8FG5@J8G55@@8H95@>8IJ5@B8K;5@G8LM5@L8N?5A09005A591A5A:9335A?94E5AC95@5A@9675A;96N5A797E5A298<5@M9945@I99K5@D9:B5@?9;95@;9<25@89<K5@79=E5@59>>5@49?85@29@25@19A05?79AJ5?49BD5?19C?5>N9D85>M9DN5>L9ED5>H9F:5>E9G05>B9GF5>?9H<5><9I25>;9IG5>;9J75>69JD5=N9K35=I9KG5=I9L:5=I9LN5=J9M=5=I9MJ5=F9N85=C9NE5=@9O95=:9OO5=3:0E5<L:1;5<E:1L5<@:205<@:255<A:2<5<A:2K5<?:3M5;I:565:K:6?59M:7D592:7L58L:8D58E:9O58=::@58I:;958I:<258F:<O58E:>=58M:?J596:A859>:B=59D:C959H:D459L:DO5:0:EJ59;:FB58J:G458H:GF58F:H758C:HI58A:I:58>:I5587:HN580:IN57<:J=56?:JO55F:L555M:LG55>:M354I:M?543:MK53>:N752I:NC524:NO51>:O;50I:OG504;1F4O>;3G4NI;5L4N@;704M;;7E4L0;7M4K@;854K0;8<4J@;8D4J0;8M4IA;9B4IC;:74IG;:G4IC;:I4J1;:I4JC;:L4JI;;=4JA;<34J=;<I4J7;=@4J2;>74IM;>N4IG;?E4IA;@<4I;;A34I6;AK4I0;BC4I4;C;4IC;D54J4;E24J?;F14JI;FO4K2;GI4K6;H=4K0;I14JK;IE4JF;J94J@;JM4J;;KA4J6;L54J0;LI4IJ;M=4IE;N54IO;NO4J<;OH4JG<0A4JG<194JG<224JI<2J4JO<3B4K4<4:4K9<524K?<5J4KE<6B4KK<7:4L0<824L6<8E4L8<8>4KE<874K4<814JC<7I4J1<7C4I@<894I9<9J4I2<;<4HJ<<N4HD<>@4H=<@24H5<AD4GN<B34GA<BA4G3<BO4FE<C=4F8<CK4EJ<D94E=<DG4DN<E64DD<ED4D9<F34CN<FA4CC<G04C8<G?4BM<GM4BB<H<4B7<HI4B2<I54AL<IB4AH<IO4AC<J<4A><JI4A9<K44A3<K?4@M<KJ4@H<L44@B<L?4@=<LJ4@7<M44@1<M?4?K<MI4?F<N94?A<O94?==094?8=184?4=284?0=374>L=474>H=554>E=5H4>I=6<4>M=704?1=7C4?5=864?9=8J4?>=9>4?A=:14?E=:E4?J=;94?M=;H4@3=<34@<=<E4@>==G4@0=>K4?C=@84?7=AD4>J=C14>==D>4>0=EK4=C=G84=7=H74<J=HE4<?=I24<3=I?4;H=IL4;==J:4;1=JJ4:F=K94:;=KH49O=L849D=LJ49==M=495=MO48N=NB48>=O?47O>0>47A>1>473>2>46F>3846=>3O468>4E463>5<45N>6345H>6I45C>7?45>>86459>9044F>9H44<>:D43B>;?42H><<41N>=441=>=J411>>@40E>?7409>?M3OM>@C3OA>A:3O5>B03NI>BF3N=>C=3N1>D33MF>DI3M:>E@3LM>F63LA>FM3L:>GE3L7>H<3L5>HO3L4>I43L4>J23KE>K83K1>L?3J>>MC3IM>NH3I<>OL3HL?103H;?253GJ?2O3G=?3@3G2?413FH?4B3F<?533F2?5D3EF?653E<?6K3DO?7C3DB?8;3D5?943CG?9L3C:?:E3BN?;=3B@?<63B3?<N3AE?=G3A8?>?3@K??73@>?@03@1?@H3?C?AA3?6?B93>I?C23><?D@3=D?EM3<N?G:3<7?HH3;??J53:I?KB3:2?LO39:?N=38D?OJ37L@17376@2D36:@3>360@4235O@4E35M@5835L@5I35J@6935G@6H35E@7735C@7G35@@8735>@8F35<@9535:
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
2497;45IA7;G5I>7<95I;7<K5I87==5I57>05I17>B5HN7?45HK7?F5HE7?J5GC7@55FL7A55G27B?5GJ7C15H17CC5H87D65H?7DI5HG7E;5HN7F25I57G05I:7GD5I?7H95ID7HN5IJ7IH5IL7JF5IK7KE5IH7LC5IG7MB5ID7NA5IC7O=5ID8095IH80N5IJ81@5IK8225IL82C5IN8355IO83G5J08485J184J5J385<5J485N5J786@5J:8725J<87C5J?8855J@88B5J>8905J789=5J189K5IJ8::5IC8:H5I;8;65I58;D5HN8<25HG8<@5H@8<N5H98=<5H28>05GF8>D5G<8>O5G78?:5G28?E5FM8@05FH8@;5FC8@F5F=8A25F98A=5F38AH5EN8B35EI8B>5ED8BI5E?8C45E:8C?5E58CJ5E08D55DK8D@5DE8DK5DA8E75D;8EB5D68EM5D18F85CL8FC5CG8FN5CB8G:5C>8G25BK8FM5B98G55AM8GO5AE8I@5AJ8K15AN8LC5B38N55B88OF5B<9175BA92I5BF94:5BK95L5BO96C5BK97;5BF9815BA98H5B<99@5B89:85B49;15B39;K5B19<D5B09==5AN9>75AM9?15AM9?M5B09@F5B09A=5AL9B55AF9BO5AA9CJ5A<9DD5A79E<5A19F25@L9FG5@G9G=5@A9H25@<9HH5@79I=5@29J05@09JC5@09K75@19KK5@19L>5@19M25@29MD5@29N15?O9N>5?L9NL5?I9O95?F9OG5?D:045?A:0H5?;:175?7:1<5?7:1G5?2:285>I:2I5>?:3:5>6:4N5=F:6:5<J:7:5;F:8;5:B:8H5:A:985:N::05:N::K5:L:;E5:I:<?5:G:=95:D:>D5:K:@15;3:A?5;;:BL5;D:D:5;L:EH5<4:FN5<;:GD5<5:GE5;O:G?5;H:G95;A:G35;::FL5;2:FF5:K:G45:>:I;59<:IJ58@:J957C:JI56G:K855J:KG54N:LN554:N455;:O:55A;01558;0=54B;0I53M;15538;1A52C;2H51N;4I518;6?50B;6M504;764OJ;7>4O@;7G4O6;7O4NL;874N<;8?4MK;8G4M;;8O4LK;9B4LK;:64LN;;24LL;<94LB;=@4L:;>74L5;>N4KO;?E4KI;@=4KC;A44K>;AK4K8;BB4K2;C94JL;D04JG;DH4JE;EA4K6;F:4KE;G34L6;GK4LF;HF4M4;I<4M4;J04LN;JE4LI;K84LD;KL4L@;L@4L=;M44L9;MH4L5;N;4L2;NO4KN;OC4KK<074KF<0I4K6<1?4K3<274K9<2O4K><3H4KD<4@4KI<574KO<604L5<6H4L;<7@4L@<884LF<904LK<9H4M1<9J4LF<9C4L5<9<4KC<964K1<9;4JE<:=4JC<;=4J@<<E4J=<>74J5<?I4IN<A;4IG<BH4I?<C74I1<CE4HD<D24H6<D@4GH<DN4G;<E<4FM<EJ4F?<F84F1<FF4EC<G44E8<GC4DM<H24DB<H@4D7<HO4CM<I<4CE<II4C@<J64C;<JC4C6<JO4C2<K<4BL<KI4BG<L64BB<LC4B=<LO4B8<M<4B3<MG4AN<N34AI<N=4AC<NL4A=<OK4A9=0K4A5=1K4A1=2K4@M=3J4@I=4J4@D=5I4@@=6I4@==7I4@8=8H4@4=9H4@0=:H4?L=;G4?G=<=4?J==04?N==<4@7==H4@>=>44@F=>@4@O=>L4A6=?84A?=?D4AF=@;4AD=A=4A6=B?4@I=CA4@;=DC4?M=EE4??=FG4?1=H14>D=I>4>8=JE4=K=K34=@=KA4=5=L04<I=LA4<@=M44<9=MF4<2=N84;K=NK4;C=O>4;=>004;5>0C4:G>154:7>1H49G>2G498>3G48K>4A48B>5748<>5N487>6E482>7;47L>8247F>8I47=>9?472>:646H>:M46=>;G45M><B45<>==44K>>744:>?243I>?M439>@H42C>AD41H>B@40N>C:409>D03OM>DF3OA>E=3O5>F33NI>FI3N=>G@3N1>H63ME>HM3M9>IC3LM>J:3LI>K23LG>K=3LF>KB3LG>KH3LG>L?3L=>ME3KH>NL3K4?033J??1:3IL?2?3I9?3C3HI?4H3H8?5=3GL?5N3GA?6?3G6?733FK?7K3F=?8C3F0?9<3EC?:43E6?:M3DI?;F3D;?<>3CN?=53CA?=J3C6?>>3BJ??23B???G3B3?@;3AH?A03A<?AD3A1?B83@E?C53@:?D73@0?E=3?@?FF3>K?GO3>6?I83=@?J=3<K?K<3<4?L;3;>?M@3:H?NO3:1@0>39:@1M38E@3<37N@5737B@5G37?@6637=@6E37;@75379@7D376@84374@8C372@9336O@9B36M@:136K@:A36I@;136F
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
2507:N5KD7;A5KA7<35K=7<E5K:7=75K77=J5K37><5K07>N5JM7?25IK7?E5I>7@F5IE7AJ5J17B=5J87BO5J?7CA5JF7D35JM7DE5K37E75K:7F55K@7G65KF7H85KK7I35L07IO5L07JL5L17KH5L57LC5L87M?5L<7N;5L?7O75LB8035LE80N5LH81G5LL8295LM82J5LN83<5LO83N5M184@5M28515M385C5M58655M586G5M587:5M287M5LO88>5LK88M5LB89;5L889I5KO8:85KE8:F5K;8;45K28;B5JH8<05J?8<?5J58<M5IL8=;5IB8=I5I98>75I28>E5HK8?05HF8?;5HA8?F5H=8@15H98@<5H58@H5H28A35GO8A>5GK8AI5GH8B45GD8B?5G@8BJ5G<8C55G98C@5G68CK5G28D65FN8DA5FK8DL5FG8E85FC8EC5F@8EN5F=8F95F98FD5F58FO5F28G:5EN8HJ5EM8JI5EO8L:5EL8LB5EA8LI5E48M@5DK8N=5DD8O95D;9065D49135CK92?5CM9405D295B5D79705D:97G5D598>5D19975CN9:05CM9:K5CN9;F5D19<A5D49=<5D79>85D:9?25D=9?N5D?9@E5D;9A;5D79B35D29BI5CN9C@5CJ9D75CE9DO5CA9EJ5C;9FA5C59G65C19GK5BL9HA5BH9I65BF9IK5BD9J@5BB9K65B?9KK5B>9L@5B<9M65B:9MI5B:9N85B99NE5B69O25B39O?5B09OL5AL:095AI:0E5AF:115A>:1=5A7:235@K:2K5@=:3C5?O:4;5?A:4M5?7:5>5>M:6M5>?:7K5>9:8I5>2:9F5=L::75=A::D5=6:;<5=2:<55<N:<N5<J:=H5<F:?25<C:@D5<G:B75<K:CJ5<O:E75=3:E=5=<:EC5=D:EI5=L:E35>::DJ5>5:DD5=N:D>5=F:D85=@:FO5<=:HG5;;:I75:?:IF59C:J658F:JE57J:K456N:KC561:L3555:LB549:MF54<:NL54B;0254I;1954O;2?555;3;552;3J54<;5653H;5H536;6952D;6B52:;6K51O;7351E;7<51;;7D512;7M50G;8550=;8F502;9>4OF;:54O;;;24O0;<94NF;=@4N;;>G4N3;@04MJ;A84MC;B@4M;;C94M4;D04LO;DH4LI;E?4LD;F64L>;FN4L8;GF4LG;H?4M7;I84MG;IN4N3;J@4N:;K34NA;KE4NI;L84O1;LJ4O9;M=4OA;N14O>;ND4O:;O84O6;OL4O3<0@4NO<144NL<1G4ND<2:4N4<2L4MD<3>4M4<414LD<4E4L9<5=4L?<654LE<6M4LK<7E4M0<8>4M6<964M;<9M4MA<:F4MG<;64MG<:O4M5<:I4LC<:B4L2<;?4KN<<@4KL<=A4KJ<>B4KH<?C4KE<A14K?<BC4K9<CL4JO<D:4JB<DG4J4<E64IG<EC4I9<F14HK<F?4H><FM4GO<G;4GB<GI4G4<H74FF<HD4F9<I34EM<IB4EB<J04E9<J<4E3<JI4DO<K64DJ<KC4DE<L04D@<L<4D;<LI4D6<M64D1<ME4CN<N?4D4<O94D5<OO4CF=044C:<OF4C@=044C@=124C<=1O4C9=2M4C5=3J4C2=4H4BN=5E4BK=6B4BH=7@4BE=8=4BA=9@4B==;34B6=<F4AO==E4B2=>C4B6=?E4AL=@>4A>=@M4AA=A94AI=AE4B2=B14B9=B=4BB=BI4BI=C54C2=D24BI=E44B<=F54AN=G74A@=H94A3=I;4@E=J=4@7=K?4?J=L@4?<=LK4?6=M=4>N=N04>G=NC4>@=O74>8=OJ4>2>0=4=J>104=C>1C4=<>2:4<N>324<?>3I4<0>4<4;@>4O4;0>5J4:F>6A4:?>774:5>7N49K>8E49A>9;496>:248L>:I48B>;?488><647M><N47A>=I470>>D46?>?>45N>@945>>A444L>AN44;>BI43K>CD43:>D>42I>E9428>F441G>FN417>GI40E>HD3ON>I@3O5>J63NI>JM3N<>KC3N1>L93ME>M03M:>MF3M8>ML3M8>N13M9>N63M:>N<3M:>NK3M4?023L@?193KK?2@3K7?3F3JB?4M3IO?643I:?7:3HF?7N3H;?8B3H0?973GD?9K3G8?:?3FM?;33FB?;H3F6?<<3EK?=13E??=E3E3?>:3DH?>N3D=??B3D1?@73CF?@K3C:?A?3BO?B53BD?C73B9?D:3AN?E=3AD?F@3A9?GH3@E?I13?O?J:3?:?KC3>D?LL3=O?N53=:?O?3<E@0G3;O@1H3;9@2F3:C@3E39M@4E397@6?391@7238N@7A38L@8138J@8A38H@9038E@9?38C@9O38A@:?38>@;438<@;N389@<H387@=A384
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
24775N65<76@65977365677E65378764O78I64L79<64I79N64E7:@64B7;264?7;F64<7<F64A7=C64F7>C64M7?C6537@D6597AD65?7BD65E7CE65K7DE6617EF6677FF66>7GF66D7HA66K7I;6717J56757JO67;7KG67<7L?67>7M867@7N067A7NH67D7O@67D80867B81067@81H67>82@67=83867;84167:84J67:85=67885L67486966N86E66H87266B87>66<87J66688766088D65K89065E89A65>8:56588:G6518;:64J8;M64D8<A64=8=864<8>164=8>K64>8?E64?8@564@8?L64=8?C64:8?F6478?O6448@76428@@63O8@J63M8A263J8A;63G8AD63D8AM63B8B563?8B>63=8BG63:8C06378C96348CB6318CK62O8D362L8D<62J8DE62G8DN62D8EM62>8FK6288GH6228HA61O8IB61E8JD61;8KF60O8M=60M8O760L91060J92I60I94B60G96<60F98560D99A60L99M61H9:96249:D61D9:O6149;;60C9;F6039<15OC9<<5O39<F5NI9=05NB9=:5N;9=D5N59=M5MN9>85MH9>N5MC9?D5M>9@C5M<9AL5M=9C45M=9D<5M=9EI5M=9G;5M=9HE5M;9IE5M79JH5M;9LE5M?9M15M;9M=5M89MI5M49N55M09NA5LL9NM5LH9O:5LD9OC5LB9OK5L?:0@5L7:155KL:1L5KA:2B5K6:385JK:3N5J@:4D5J4:5:5IJ:605I>:6F5I4:795HG:7=5H4:7A5GK:7B5GA:7B5G7:7J5FM:955F@::A5F1:;I5EJ:<I5EJ:=F5EK:>C5EM:?@5EN:@=5EO:A;5F0:B85F2:C05EJ:CC5E=:D75DO:DJ5DA:E>5D4:F15CF:FC5C9:G65BJ:GI5B<:H;5AN:HN5A@:J85A0:KF5@D:M55@7:NC5?K;015?>:L75?O:M85?K:NB5?E:OM5??;155>N;2>5><;3E5=I;4M5=;;5M5=I;6N5>6;7G5>9;875>0;8G5=G;975=>;9<5=1;9@5<C;9E5<7;9I5;I;:65;<;:M5;1;;E5:E;<=5:8;=559M;=M59A;>D595;?<58I;@458=;B958:;D>586;FC583;HH57O;JM57K;M257H;MH57I;ML57F;MO57C;N257@;N657=;N@576;O656H;OK56;<0@55N<1A56A<2=56><3956<<4656;<52568<5O566<6K565<7H562<8C560<9855K<9N55E<:C55?<;:55E<;L55I<<955=<<E551<=254E<=>549<=J53M<>653@<>E53:<?3533<?A52K<?O52D<@=52><@J52L<A653;<AB53J<AN549<BB54@<CH548<E4544<FB540<H053M<I>53@<JM532<L;52C<LO52C<M@52:<N151N<NB51B<O3516<OD50I=0550==0G501=174OE=1H4O9=294NL=2K4N@=3<4N4=3M4MG=4=4M;=4O4LN=5I4LG=724LE=824L<=8C4KE=8M4KB=974KE=9@4KH=9I4KK=:24KN=:;4L1=:D4L4=:M4L6=;64L9=;?4L<=;H4L>=<14L@=<:4LB=<D4LE==@4LM=>?4M6=?>4M?=@>4MH=A=4N1=BC4N2=CG4N7=DK4N<=F04N==G54N==H:4N==I?4N==J?4ML=K?4M5=LA4KB=M=4JG=N34J<=NG4IM=O<4I>>004I0>0B4H>>144GM>1>4G?>1G4G2>204FE>294F9>2B4EK>2K4E>>344E1>3>4DD>3K4D7>4@4CF>554C5>5J4BC>6?4B2>734AA>7H4A0>8=4@>>924?M>9G4?;>:=4>I>;44>5>;K4=B><G4=5>=E4<O>>D4<K>?:4<H>@04<F>@F4<C>A=4<A>B34<?>C54;@>D84:C>E;49E>F4492>FN48I>GH48@>HB487>I<47N>J647E>K047<>KJ473>LD46J>M>46@>N8467>O245N>OL45E?0I455?1F44E?2C445?3?43D?4:435?5342E?5L426?6E41G?7>417?8640H?8O409?9H3OJ?:A3O;?;:3NL?=J3NG?@03NE?@63N;?@<3N1?@B3MG?@H3M=?@N3M4?A43LJ?A:3L@?A23KN?@E3KH?B33K;?C63JO?D93JD?E<3J9?F?3IN?GB3IC?HE3I8?IH3HM?JK3HA?KM3H6?M13GK?N33GA?O63G6@043G2@123G1@213G1@303G0@3N3FO@4M3FN@5L3FN@6J3FM@7H3FL@8G3FK@9F3FK@:E3FH@;I3EO@<N3E5@>33DA@?83D3@@=3CF@AB3C8@BG3BJ@CL3B=@E03AO@F63AB@G:3A4@H?3@F@IE3@9@JI3?J
24775F66376866076J65M77=65I77O65F78A65C79465?79F65<7:86597:M6597;E6597<J65?7=O65H7>L65M7?J6627@G6677AE66<7BC66B7C@66G7D@66M7E@6737FA6797GA67?7HB67E7I<67L7J66817JN6827KF6847L>6867M66887MN68:7NF68;7O?68=80768;80O68981G68782?68683768483O68284G68185:67N85F67G86367A86?67<86K67687867087D66J88166E88=66?88J66889766289J65L8:=65F8;065?8;F65<8<@65=8=965>8>365?8>M65@8?F65A8@865A8?N65?8?E65<8?<6598?>6568?F6548?O6518@864O8@A64L8@J64J8A364F8A<64C8AD64A8AM64>8B664<8B?6498BH6478C16448C:6418CB63N8CK63K8D963H8E763B8F563<8G36368GO6318HH62N8IH62E8JJ62;8KL6208LN61E8NB61A90;61?92461>93M61<95F61;97@61999861999D6269:06329:;62L9:F62<9;161L9;=61<9;H60L9<360;9<>5OK9<J5O;9=35O39==5NM9=G5NF9>15N?9>E5N:9?;5N69@C5N69AL5N69C45N69DG5N69F85N59GC5N49HD5N09IE5ML9JH5N39M75N;9MC5N79MO5N39N;5MN9NG5MK9O45MG9O<5ME9O?5ME9OD5MC:095M=:0M5M5:1C5LK:295L?:2O5L4:3E5KI:4;5K>:515K3:5H5JH:6>5J=:735J1:7C5IC:7M5I3:7M5H?:855H4:855GJ:865GA:8O5G5:::5FF:;C5F>:<D5F?:=A5F@:>>5FA:?;5FC:@85FD:A65FE:B05FC:BE5F7:C85EI:CL5E;:D?5DN:E25D@:EF5D2:F95CE:FM5C7:GA5BI:H85B?:I45B9:J65B1:KD5AD:M25A7:NA5@K:OE5@A:LB5@N:ML5@H:O75@C;0B5@=;1L5@6;345?D;4<5?2;5D5>A;6D5>O;7E5?;;865?4;8F5>K;975>B;9F5>9;9K5=L;9O5=>;:35=1;:85<C;:<5<7;;35;J;;K5;>;<C5;3;=:5:G;>25:;;>J59O;?B59C;@K59;;C0597;E5593;G:590;I?58L;KD58I;MC58C;N<586;NF587;NM586;O1583;O5580;O857M;O<57J<0157<<0F56O<1F57:<2C577<3?576<4;574<58571<64570<7156N<7M56K<8J56I<9F56G<:C56E<;:56A<;O56;<<B56><<N563<=;55G<=G55;<>354O<>?54B<>L546<?853J<?F53B<@453;<@B534<@O53:<A;53I<AG549<B454G<B@556<BL55E<CN55A<E=552<FK54C<H9545<IH53F<K6537<LE52I<M;52G<MM52G<N?52G<O052><OA521=0251E=0D519=1550M=1F50@=26504=2H4OH=394O<=3J4O0=4;4NC=4L4N7=5=4MJ=6B4MG=7G4MC=884LL=8I4L8=924L;=9;4L>=9D4LA=9M4LD=:64LG=:?4LJ=:H4LL=;14LO=;:4M2=;C4M4=;L4M7=<54M;=<>4M==<G4M?==C4MG=>C4N1=?H4N2=@N4N2=B54N2=C;4N2=DB4N2=EF4N6=FJ4N==GM4NC=I14NJ=J44O1=K74N0=L:4L;=M54K?=ML4K2=NC4JI=O74J9=OL4IK>0@4I<>114HJ>1D4H8>224GI>2;4G<>2D4FO>2M4FB>364F5>3?4EH>3H4E:>414DM>4>4D?>524CM>5F4C;>6:4BI>6N4B7>7C4AF>884A4>8M4@C>9B4@2>:74?A>:L4>O>;A4>>><84=J>=24=;>>04=5>>H4=2>?>4=0>@44<N>@J4<L>A@4<I>B64<G>C84;K>D;4:M>E@4:8>FC49H>G=49>>H7495>I048L>IJ48B>JD489>K=480>L847G>M247>>ML475>NF46L>O@46C?0<466?1845E?24455?2M44F?3F447?4?43H?58439?6142I?6J42:?7C41J?8;41;?9440L?9M40=?:F3ON?;?3O??>43O;?A43O<?A:3O2?AA3NH?AG3N>?AM3N4?B33MJ?B63M??AI3LJ?@M3LD?A53L;?BJ3KM?CM3KB?E03K7?F33JK?G63J@?H93J5?I<3IJ?J?3I??KB3I4?LD3HI?ME3H<?NB3GO?O<3GA@0;3GB@1;3GH@2;3GN@3:3H4@4:3H:@593HB@653HM@733I2@813I1@903I0@9N3I0@;03HC@;N3G9@=23DO@><3DK@?F3DL@@N3DH@B33D:@C83CM@D=3C?@EB3C1@FF3BD@GL3B6@I03AH@J53A;@K:3@M
24674K66N75>66K76066H76B66D77566A77G66>78966:78K66779>6647:56657:L6657;C6677<F6687>966H7?766N7@56737A26787B067>7BN67C7CK67H7DI67N7EG6827FE6877GB68=7H@68B7I<68G7J468H7JM68J7KE68L7L=68N7M56907MM6917NE6937O=69580569380M69281F69082>68O83668M83N68K84G68J85:68G85E68@86168986<68386H67L87367E87?67>87K67888767288D66L89066G89@66A8:566<8:N66=8;G66>8<A66?8=;66@8>466A8>N66B8?G66C8@:66C8@166A8?G66>8?>66<8?56688?56668?>6638?G6608@065N8@965K8@A65I8@K65F8A365C8A<65@8AE65>8AN65;8B76598B?6568BH6538C16508CC64L8DA64F8E?64@8F=64:8G;6448H66408HO63M8IM63F8JO63;8L16308M362E8N662:8OE62591?62393862295162096K61O98C61M99:62C99F63@9:26449:=63D9:I6349;462D9;?6249;J61C9<66139<A60C9<L6039=75OD9=A5O=9=K5O79><5O19?:5NO9@C5NO9B25NO9CD5NN9E65NN9FC5NM9GC5NI9HA5NG9I95NJ9JK5NL9MI5O59N55O19NA5NM9NM5NJ9O55NG9O85NG9O;5NG9O>5NG:025NA:0F5N::1;5N3:205MI:2F5M>:3<5M2:435LH:4I5L<:5?5L1:655KF:6K5K;:7=5JN:7M5J@:8=5J1:8=5I=:8<5HH:8G5H=:8H5H3:8I5GJ::45G<:;=5G3:<?5G3:=<5G5:>95G6:?65G7:@25G7:@M5G5:AF5G0:B:5FC:BM5F5:CA5EG:D45E9:DG5DK:E;5D>:EO5D0:FI5CH:GE5CB:H@5C<:I<5C5:J85BO:KA5BD:M05B8:N?5AK:O35AC:M75AL:NB5AF:OL5A@;175A:;2B5A4;3K5@L;545@:;6<5?H;7<5@4;865@8;8F5?O;965?F;9F5?=;:55?3;:95>F;:>5>9;:B5=L;:G5=>;:K5=1;;95<D;<05<8;<H5;M;=@5;A;>85;4;>O5:I;?H5:=;AC5:8;CH5:4;EM5:1;H259M;J759I;L359C;M>596;NI58=;NI58:;O358;;O=58<;OG58<<0058=<0:58=<0=58:<0N583<1L583<2H580<3D57O<4A57M<5=57J<6:57I<7657G<8357D<8O57C<9K57@<:H57><;D57=<<A57:<=8574<=D56H<>056=<><561<>I55E<?5558<?A54L<?M54@<@:544<@G53K<A553I<AA548<AM54F<B9552<BE55<<C255G<C>561<D755H<EF558<G454I<HC54:<J253L<K@53=<LN52N<MH52J<N:52J<NL52J<O>52K<OO52K=0A52A=12525=1C51H=2451<=2E510=3650D=3G508=484OK=4I4O?=5:4O3=634NJ=7<4NH=7N4N4=8?4M==8M4M1=964M4=9?4M7=9H4M:=:14M==::4M@=:C4MB=:L4ME=;54MH=;?4MJ=;H4MM=<14N1=<:4N4=<B4N6==04N7=>14N2=?94N2=@@4N2=AF4N2=BM4N2=D34N2=E:4N2=F@4N2=GE4N4=HI4N;=IL4NB=K04NI=L24M4=LN4L7=ME4KH=N<4K?=O34K5=OG4JF>0<4J8>0O4IH>1A4I6>234HD>2E4H2>2N4GE>374G8>3@4FL>3I4F>>424F1>4;4ED>4D4E6>534DG>5G4D6>6;4CD>6O4C2>7C4B@>884AN>8L4A<>9@4@J>:44@8>:H4?F>;<4?4><14>C><F4>2>==4=A>>54==>>K4=;>?A4=9>@74=6>@N4=4>AD4=2>B:4<O>C<4<7>DB4;F>EH4;4>FO4:B>GN4:4>HG49K>IA49B>J:498>K448O>KN48E>LG48<>M@483>N947I>O347@>OM476?0G46G?1@467?2945H?32459?3K44J?4C44;?5=43L?6543<?6O42L?7G42=?8@41N?9941??:2410?:K40A?;D402?>?3OO?B:403?B@3OI?BF3O??BL3O5?C23NL?BL3N;?B@3MF?AE3M>?@I3M:?AE3LN?C=3LA?DD3L4?EG3KI?FJ3K>?GM3K3?I03JH?J33J<?K63J2?L53ID?M33I6?N13HI?NM3H??OC3H:@0@3GD@1>3G7@2?3G=@3>3GC@4>3GJ@5>3H0@6>3H6@7=3H<@8=3HB@9=3HI@:=3HO@:L3I@@;?3H@@=93DL@>C3DM@?M3DO@A73E0@B@3E3@CI3DO@DN3DB@F33D4@G73CF@H<3C8@IA3BJ@JF3B=@KK3AO
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
24774M68975@68676268376D67O77767L77L67L78D67M79;67N7:26807:I6817;A6827<C6847=N68:7?C68J7@K6967AI69:7BG69@7CD69E7DB69J7E?6:07F=6:57G76:87GM6::7HD6:<7I:6:?7J26:?7JJ6:?7KB6:?7L:6:?7M36:?7MK6:A7NC6:C7O;6:E8036:F80K6:D81C6:B82;6:@8356:@83O6:?84H6:?85>6:=85J6:68656:086@69I86L69C87769<87C69588069088>68M88L68I89968E89G68A8:968@8;168A8;J68A8<D68C8==68D8>668D8?068F8?J68G8@?68G8@568D8?L68B8?C68?8?968=8?068:8>G6878>E6848>N6828?767O8??67M8?H67J8@167G8@:67D8@B67A8@L67?8A567<8A=67:8B76758C566N8D366H8E166B8F066<8FN6658GJ6618HC65N8I=65J8J865F8K:65;8L<6508M>64E8N@64:8OB64090D63D91M63<93F63;95@63:97963898G63>99364:99@65799L6649:765D9:B6549:M64D9;96449;D63D9;O6349<:62C9<E6239=061C9==6129><60C9?J60@9A>60@9C160@9DA60?9EE60C9FK60F9H060K9I660O9J<6129L36139N66109NJ60M9NM60M9NO60M9O260M9O560M9OE60H:0960A:0N60;:1B604:265OM:2K5OE:3A5O::475NO:4M5ND:5C5N9:6:5MN:6O5MB:7A5M5:825LG:8B5L9:925KK:9<5K::9;5JF:9;5J1:9:5I=:9D5HN:9M5HF:;15H<:<45H=:=25H?:=L5H<:>G5H9:?B5H7:@<5H5:A15GH:AD5G::B75FL:BN5FA:CJ5F::DE5F4:EB5EN:F=5EG:G95EA:H55E;:I15E4:IM5DN:JI5DH:KE5DA:LK5D8:N:5CK:N15CH:NA5CF:OL5C@;165C;;2A5C5;3K5BO;565BI;6@5BC;7J5B8;8E5B6;955AM;9E5AE;:55A<;:E5A3;;25@H;;75@;;;<5?N;;@5?A;;D5?3;;I5>F;;M5>9;<<5=L;=35=@;=K5=4;>C5<I;?;5<=;@M5<6;C25<2;E75;N;G<5;K;I25;C;J>5;8;KI5:L;M55:5;N@59=;OK58D;O@58@;OC58B;OM58C<0:58F<0O58L<1C593<2=58I<3458D<3I58@<4:58=<5058:<5M588<6I585<7E583<8B580<9?57M<:;57J<;857G<<457E<=157B<=B578<=H56I<=O569<>655I<>=55:<>K54O<?954E<?J54><@C54A<A=54E<B754H<BD552<C155<<C=55G<CI561<D556<<DD56@<E8567<F855H<GF557<I654F<JD546<L353H<MA539<NA532<O3532<OE533=07533=0I533=1;533=1L533=2?533=30534=3A52G=4252;=4D51O=5551C=5E517=6=50M=7:50A=7K4OK=8<4O4=8L4N@=954NC=9>4NF=9G4NI=:04NL=:94NN=:B4O1=:K4O4=;44O6=;>4O9=;G4O<=<04O?=<B4O>==74O;==M4O9=>C4O6=?B4NO=@B4NG=AA4N@=BA4N:=CJ4N8=E44N7=F>4N7=GG4N5=I14N4=J;4N4=KC4N3=L?4MG=M64M8=MM4LJ=ND4L>=O=4L9>034KO>0G4K?>1;4K1>1M4J?>2?4IM>314I<>3C4HJ>454H8>4>4GK>4G4G?>504G1>594FD>5B4F7>5K4EI>6<4E:>704DG>7D4D5>894CD>8M4C2>9A4B?>:54AN>:I4A<>;=4@J><14@8><F4?F>=;4?4>>64>J>?14>?>?F4>5>@;4=L>A14=J>AG4=H>B<4=E>C24=C>CH4=A>D@4=<>E=4=4>F94<L>G@4<9>HF4;H>II4;8>JC4:N>K<4:E>L64:;>LO4:2>MH49H>N;49=>N?48O>NB48?>O3481?0247C?11476?1O46H?2N46;?3M45N?4L45@?5F451?6?44B?78443?8143C?8J434?9C42E?:<426?;441G?;M417??3417?C141<?DJ418?DF40G?D9402?CL3O>?C63O2?B93NO?A<3NJ?@M3ND?BF3N6?D=3MG?F23M:?G53LN?H73LC?I53L5?J43KH?K23K:?L03JL?LO3J??MM3J1?NK3IC?OC3I=@093I9@0O3I6@1L3H?@2J3GH@3G3G1@4F3FG@5E3FM@6E3G3@7E3G:@8E3G@@9E3GF@:E3GL@;93H?@<43FH@=F3E0@?03E2@@:3E3@AD3E5@BM3E7@D83E9@EB3E:@FK3E<@H53E=@I>3E=@JC3DO@KH3DB
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
24875069D75D69C76;69D77369F77J69G78A69H79969I7:069K7:G69L7;?69M7<A69O7=H6:07?86:<7@M6:M7B?6;=7C=6;B7D;6;H7E16;J7EH6;K7F>6;N7G46<07GK6<27HA6<47I86<67J06<67JH6<67K@6<67L86<67M06<77MH6<77N@6<77O86<78006<780I6<781C6<682=6<68366<58406<584I6<585B6<48606;O86=6;K86K6;H8796;D87F6;@8846;<88B6;888O6;589=6;189K6:M8:96:I8:I6:H8;;6:F8;L6:F8<F6:F8=@6:G8>96:H8?36:I8?L6:J8@C6:K8@:6:I8@16:F8?G6:C8?>6:@8?56:>8>K6:;8>A6:98>86:68>46:38>=6:08>F69N8>O69K8?869I8?@69F8?M69C8@K69=8AI6978BG6908CF68J8DD68D8EB68=8F@6878G>6828H867N8I167K8IK67G8JE67D8KD67;8LG6718MI66E8NK66;8OM65O90O65E92165:93364O94564D95N64C97G64A98@65598L66299966N99E67K9:067D9:;6759:G66D9;26649;C65@9<A64D9<M63:9=96279>C6289?M62:9A762;9AF6229C:6229DN6219E<61H9F<61J9GG6229I262:9J=62D9KL6359M:63F9NH63G9NN6379NM6339O86309OM62I:0A62B:1562;:1J625:2>61O:3261H:3G61A:4<617:5260L:5H60@:6>606:745OJ:7E5O=:865NN:8F5NA:965N2:9F5MD::65M6::;5LC:::5KO:::5K::::5JF::N5J7:=35J7:><5J?:>B5J?:>L5J4:?65II:?@5I>:?O5I3:@K5HM:AF5HF:BB5H@:C>5H::D:5H3:E65GM:F25GG:FN5GA:GJ5G::HF5G4:IB5FN:J>5FG:K95FA:L65F;:M15F4:N55EL:NA5EG:OK5EA;165E;;2@5E5;3K5DO;525DG;685D>;7>5D5;8<5D4;9=5CM;:45CC;:D5C:;;55C1;;E5BI;<05B=;<55B0;<95AC;<=5A6;<B5@H;<F5@;;<K5?N;<O5?A;=?5?4;>65>H;>N5><;@85>3;B=5>0;DB5=L;F25=C;G=5=7;HI5<L;J45<@;K@5;N;LK5;6;N65:=;OI59K;O:59I<0059B<0O598<1M59F<2A59M<3?599<46595<4M590<5C58K<6:58F<6O58C<7A58@<8358=<8E58:<97587<9N585<:K582<;H57O<<D57M<=A57J<=N57=<>556N<><56><>D560<?055K<?=55E<?K55@<@@55=<A955><B255B<BL55E<CF55H<D4561<D@56;<DL56F<E9571<EK56L<F>56C<G9567<H955I<IE558<K554F<LF543<N553D<O:53:<OL53:=0>53:=1053:=1B53:=2453;=2F53;=3853;=3I53<=4;53<=4M53<=5?53==6153:=6E52N=76528=7G51B=8850K=8I505=95502=9>505=9G508=:050:=:950==:B50@=:K50B=;450E=;>50H=<450E=<I50B==?50?=>450==>J50:=??508=@8503=A74OL=B64OF=C64O?=D54O7=E44O0=F44NJ=G34NC=H94N?=IC4N>=JM4N==L64N<=M>4N:=N54MK=NK4M==OD4M7>0=4M4>124LH>1G4L9>2:4KH>2K4K6>3=4JE>404J3>4B4IA>534I0>5E4H>>5N4H1>674GD>6@4G7>6I4FJ>724F<>7F4EK>8:4E9>8N4DG>9B4D5>:64CC>:K4C2>;?4B?><34AN><G4A<>=?4@M>>:4@C>?64@8>@14?M>@L4?C>AB4?8>B74>N>BK4>C>C@4>9>D64>7>DK4>4>EA4>2>F74=O>G34=H>GO4=@>HJ4=8>IG4=0>JB4<I>KC4<<>L>4<1>LG4;C>LK4;5>LO4:F>M34:7>M649H>M:498>N448K>O348=?0147O?1047B?1O475?2N46G?3L46:?4K45L?5J45??6H451?7G44D?8F446?9E43I?:C43<?;>42L?<742=??542E?AN42L?BE428?C?41=?D940D?CN40G?D440D?D940J?DE40O?E:40F?E43ON?EB3OC?FA3O3?H23MJ?I13M=?IO3LO?JN3LA?KL3L4?LJ3KF?MI3K8?NG3JK?OD3J?@093J<@0O3J8@1E3J4@2;3J0@383I:@463HC@533GM@613G7@6N3F@@7L3F7@8M3F=@9L3FC@:L3FJ@;K3G1@<J3E3@>33E5@?=3E6@@G3E8@B13E9@C;3E<@DE3E=@EO3E?@G83E@@HB3EB@IL3EC@K63EE@L@3EG
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
2016286@F62O6@;63F6@064=6?E6546?:65L6>O66C6>D6796>96816=M68H6=C6926=769F6==6::6=F6:N6=M6;A6>16<56>46<K6>76=J6><6>I6>@6?G6>D6@F6>H6AE6>L6BD6?06CC6?56DB6?96EA6?=6F?6?A6G>6?E6H=6?I6I56?I6IG6?G6J:6?D6JL6?A6K>6?>6L16?:6LC6?76M56?400000000000000000000000000000000000000000000000000000000000074F6?375=6?47646?676L6?777C6?878:6?97926?;79I6?<7:@6?=7;86?>7<;6?@7=B6?B7>I6?C7@06?E7A@6@27B@6@87C76@;7CO6@>7DG6@B7EA6@E7F<6@J7G66@N7H06A17HH6A:7I:6AJ7J16AJ7JI6AK7K@6AK7L76AL7M16AJ7MJ6AG7NC6AG7O<6AE8076A;8126A181N6@G82H6@>83C6@784<6@:8546@@85L6@F86C6@M87;6A38836A988E6A;8966A;89G6A<8:86A=8:I6A>8;:6A>8;K6A>8<<6A?8<M6A>8=?6A<8>06A;8>B6A:8?46A88?E6A78@66A68@M6A68@I6A48@?6A18@76@N8@06@H8?I6@B8?C6@=8?<6@88?66@28>O6?M8>I6?H8>O6?C8?@6?A8@26?>8@D6?;8A?6?68B=6>O8C;6>I8D96>C8E86>=8F66>78G46>18GN6=N8HG6=J8IA6=G8J;6=D8K46=@8KN6==8LH6=:8MG6=18NI6<G8OK6<;90M6<191O6;E9316;;9436;09566:E95L6:896A69J97669=97K68O98?68A99568499I67F9:>6789;<6759<A6799=D67=9>H67A9?K67E9@N67I9B267N9C56819D96869E<68:9F?68=9GB68B9HH68C9IM68B9JH68@9J@6889JG6819L26809M=67N9NH67M:0367K:1967E:1L67@:2=67<:2O679:3A675:46672:4L66O:5A66L:6766I:6M66F:7A66A:82663:8C65F:93659:9C64M::464?::E642:;563E:;F638:<662K:<A628:<L61D:=7610:=H60@:>?600:?55OA:?K5O2:@B5NC:A85N4:AO5ME:BE5M6:CM5LL:DE5LF:E=5L@:F55L::FM5L4:GF5KN:H>5KH:I65KB:IN5K=:JF5K6:K>5K0:L75JJ:LO5JD:MG5J>:N?5J8:O75J3:OO5IM;0H5IG;1A5IA;2@5I?;3>5I=;4=5I;;5;5I9;6:5I8;785I5;875I4;955I1;:65HH;;85H?;;I5H2;:O5GA;:;5G0;:=5FC;:@5F5;:K5EH;;G5E<;<C5E0;=?5DC;>:5D7;?65CK;@25C?;A<5C2;BJ5BE;D85B8;E?5AM;FE5A@;GJ5A5;H?5@G;J55@C;KN5@6;M05?M;MO5?A;NM5?2;OK5>B<0H5>2<1F5=B<2C5=2<3N5<><4O5<5<605;L<705;C<815;:<915;0<:25:G<;35:><<35:5<=459L<>459E<?559=<?J59:<@;599<@F59<<@O59@<A759E<A?59I<AG59M<AO5:1<B95:2<BF59M<C359H<C@59B<D459?<DI59<<E?599<F5596<FJ593<G@591<H4590<HG58L<I:58F<IM58><J@587<K:585<L9585<M657K<N557@<O3576=0156K=0O56@=1M565=2L55K=3I55?=50555=5M551=6?552=71552=7:550=7C54O=7L54L=8554K=8>54I=8G54G=9054E=9>54?=:454==:I54:=;?548=<4545=<J542==?540=>553M=>J53J=??53G=@553D=@K53B=A@53?=B653==BK53:=C@537=D6534=E552M=F452G=G352@=H3528=I2521=J151K=K151D=L051<=M0515=MO50N=NE515=O351D=OA524>0:51N>1=50O>2@501>3=4OA>414O1>4C4N@>554MN>5G4M<>694LK>6K4L9>7=4KG>7O4K6>8A4JD>934J2>9D4I@>:54I0>:>4HC>:O4H2>;C4G@><74FO>=24FC>=M4F9>>H4EM>?C4EC>@>4E8>A:4DM>B54DC>C04D7>CK4CM>DF4CB>EA4C8>F=4BM>G74BB>GL4B8>H@4AM>I54AC>IJ4A8>J>4@N>K24@C>KB4@2>KM4?9>L14>I>L24><>L44=O>L54=B>L64=5>L84<I>L94<<>L<4<0>LN4;N>M?4;L>N14;J>NC4;G>O44;E>O54:K?0:4:>?1=4:1?2B49C?3F495?4J48H?5N48;?7547O?8@47E?9K47;?;6471?<A46G?=L46=??2462?@745F?A;45;?AJ44E?B:443?B<43<?BK42L?CG42@?DD426?EA423?F>420?G:41N?H741L?I441J?IK41;?JC40L?K:40>?L13ON?M;3O0?N93NC?O83N5@063MG@103M>@1F3M:@2<3M7@313M4@3H3M0@4=3LL@533LH@5I3LD@6?3LA@7;3KK@893K5@963J?@:43IH@;33I9@<13HM@=73HA@>>3H7@?F3GK@@N3G@@B63G5@C>3FJ@DF3F>@EN3F3@G63EL@H@3EN@IJ3EO@K43F2
20161L6AD62C6A86396@N6416@B64H6@765?6?L6666?A66M6?667D6>K68;6>@68L6>469;6>569O6>>6:D6>F6;76>I6;J6>L6<D6>O6=C6?36>A6?86?@6?<6@?6?@6A>6?D6B=6?H6C<6?L6D:6@16E:6@56F86@96G76@=6H66@B6HM6@A6I?6@>6J26@;6JD6@86K66@56KH6@26L;6?O6LM6?K6M?6?H00000000000000000000000000000000000000000000000000000074E6@175<6@27636@376K6@577B6@57896@77916@879H6@97:?6@;7;66@;7<:6@>7=A6@?7>H6@A7?O6@C7A;6@K7B96A17C06A47CI6A77DD6A;7E>6A?7F86AC7G36AG7GM6AJ7HD6B47I66BE7IK6BK7JB6BK7K<6BJ7L56BG7M06BD7MI6BB7NB6BB7O:6BD8036BB80N6B881I6AM82D6AD83?6A:8496A58516A785I6A>86A6AD8786AJ8806B188G6B68986B789I6B78::6B88:K6B88;<6B98;L6B:8<:6B:8<H6B98=;6B98=N6B98>A6B88?46B88?H6B78@;6B78@N6B78@N6B38@H6AM8@A6AG8@:6AB8@36A=8?M6A78?F6A28?@6@M8?96@H8?36@B8?A6@>8@36@<8@D6@98A76@68AH6@38BE6?N8CD6?H8DB6?B8E@6?<8F>6?68G:6?08H56>L8HN6>I8IH6>F8JB6>C8K;6>?8L56><8LO6>88ML6>18NN6=G9006=<9136=19246<F9376<;9496<09526;C95G6;596<6:H9716::97E69M98:69@98O69299E68E9:=68:9;968=9<768C9=:68G9>>68J9?A68O9@D6939AH6979BK69;9CN69?9E269C9F569G9G969K9H>69K9IC69J9JI69J9K@69G9K869?9L>69=9MI69;9O469::0=693:1A68G:2668>:2J685:3=67O:3O67L:4B67H:5767E:5M67B:6B67?:7767<:7J679:8=672:8O66H:9?66<::065O::A65B:;1654:;A64G:<264::<>63H:<I634:=362A:=>61M:=I619:><60I:?360::?I5OK:@?5O<:A65NM:AL5N>:BB5MO:D=5MI:E<5MC:F45M=:FL5M8:GE5M2:H=5LL:I55LE:IM5L?:JE5L9:K=5L3:L65KN:LM5KH:ME5KB:N>5K<:O65K6:ON5K0;0F5JJ;1;5JC;225J9;3:5J7;3H5J7;4<5J6;5:5J4;695J2;775J1;865IN;945IM;:55ID;;55I:;:<5HH;9C5H7;8J5GE;9D5G9;:@5FL;;<5F@;<85F3;=35EG;=O5E;;>K5DO;?G5DC;@C5D6;AM5CJ;C;5C=;DI5C0;F65BC;G@5B7;HF5AL;I85A=;J05A3;L65A1;M45@C;N15@3;NN5?C;OL5?3<0J5>C<1G5>3<2E5=C<3L5=3<4L5<I<5M5<@<6M5<7<7N5;M<8O5;D<9O5;<<;05;5<<05:N<=05:F<>05:><?15:8<?G5:5<@85:4<@J5:2<A;5:2<AL5:1<B65:5<B>5:8<BF5:=<BO5:A<C75:D<CD5:><D15:9<DC5:5<E95:2<EO59O<FD59L<G959K<GM59J<HA59I<I459I<IH59B<J;59<<JM595<KH593<LF590<MD58F<NC58:<OA580=0?57E=1=57:=2;56O=3956D=47569=5655N=6455D=7255<=7<55:=7F558=81556=8=555=8H553=94552=9@550=9K54O=::54N=:M54L=;C54J=<854G=<M54D==C54B=>854?=>N54<=?C549=@9546=@N544=AD541=B953O=BO53L=CD53I=D:53F=DO53C=EO53==FN536=GM52O=HM52G=IL52A=JK52:=KK522=LJ51K=MJ51D=N9524=NG52D=O6533=OD53C>0@533>1D524>2G516>3H507>4@4O?>524NL>5D4N:>664MH>6H4M6>7:4LE>7L4L3>8>4KA>904JO>9B4J>>:44IL>:F4I:>;14HL>;C4H;><84GJ>=44G?>=O4G5>>J4FJ>?E4F?>@@4F5>A<4EI>B74E?>C24E4>CM4DJ>DH4D?>ED4D4>F?4CI>G:4C>>H54C4>HJ4BI>I>4B?>J34B4>JG4AJ>K<4A?>KG4@G>L24?N>L=4?5>LA4>F>LB4>9>LC4=L>LD4=?>LF4=2>LG4<F>M14<>>MC4<<>N54<9>NF4<7>O84<5>OI4<3?0;4<1?0M4;O?0I4:K?1M4:>?324:1?4549C?5:495?6C48K?7N48A?99487?:D47M?;I47A?<M476?>246J??646>?@:462?A>45F?AL451?B:44;?BB43G?C?43<?D;431?E742F?F442??G142=?GM42:?HJ428?IG425?J>41F?K6417?KM40H?LD40:?MH3O=?NF3NO?OE3NB@0C3N4@1;3MN@213MJ@2G3MF@3=3MC@433M?@4H3M;@5>3M7@643M4@6J3M0@7B3LD@8@3KN@9=3K8@:<3JJ@;;3J>@<:3J2@=B3IF@>J3I;@@23I0@A:3HE@BB3H:@CJ3GN@E13GC@F:3G8@GB3FM@HI3FB@J13F7@K:3F4
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
2066216O362I6NH63B6N=64:6N26526MH65K6M=66C6M267;6LG6846L=68L6L269A6L369O6LH6:M6M06;M6M36<O6M66>06M:6?16M>6@26MA6A26MC6B26MF6C26MF6CN6M86DJ6LI6EH6M96FF6N46GF6N>6HG6N;6II6N86JK6N46KJ6N26LD6MO6M=6MM6N76ML6O16MI6OK6MG70E6ME71=6MF7246MN72K6N573B6N;7496NB7506NI75G6O176>6O87756O>77L6OE78C6OL79:7047:170:7:G70A7;?70H7<570O7=170O7=N70M7>K70J7?H70H7@D70F7AD70=7BE7007CF6OC7DG6O67ED6NN7F56O17FG6O57G96O87GK6O;7H<6O>7I56O?7IN6OA7JI6OD7KF6OJ7LA7017M=7087N770@7O270G7OM70N80B70M81870J81M70I82B70G83870F83N70G84C70H85970J85O70K86C70L87070F87=70@87J70:88770488E7008966OL89G6OI8:96OF8:N6OD8;A6OD8<46OD8<H6OD8=;6OD8=N6OD8>A6OD8?46OD8@06O<8@L6O48AH6NK8BD6NC8C@6N;8D<6N28E86MI8F46MA8FI6M:8G86M58GH6LO8H76LI8HG6LD8I66L>8I86LC8I26LL8HL6M68HF6MA8HG6MH8IG6MA8JG6M:8KG6M38LG6LM8MG6LF8NG6L@8OF6L:90@6L791:6L09256KE9316K:93M6JN94H6JD95D6J896?6IN96M6IN97;6IN97J6IO9886IO98F6J09956J199C6J19:46J59:E6J<9;E6J09<N6IJ9>76ID9??6I?9@H6I89B26I29C:6HM9DC6HG9EL6HA9G36H;9H66H39I:6GJ9J>6GB9KB6G:9LE6G19ME6FO9ND6FM9OD6FK:0C6FI:1C6FG:2B6FD:3B6FB:4?6F@:5;6F=:666F::726F7:7H6EN:876E>:8H6DM:9:6D<:9L6CL::@6C>:;66C9:;M6C4:<D6BO:==6BE:>66B::>O6AO:?H6AD:@A6A9:A;6@N:AM6@=:B:6?H:BH6?3:C66>>:CL6>0:DD6=D:E<6=7:F46<J:FL6<=:GD6<0:H<6;C:I46;6:IO6:J:L06:F:MA6:A:N96:;:O26:6:OJ6:0;0B69I;1:69C;2369>;2D692;3568G;3G68;;4867O;4I67D;5@675;6866G;7866D;8866A;9866>;:766;;;8668;<7665;=7662;>765O;>O65K;?H65G;@A65C;A;65=;B<651;C664F;CI64;;D=640;E063E;EB63<;F:638;G=62O;GN62E;HA62:;I6620;IJ61E;J=616;JO60I;KB60:;L45OM;M45O=;NB5NJ<005N8<1:5MG<2B5M7<3J5LG<525L7<6:5KG<7B5K7<885JF<8D5J4<915IA<:95GH<;A5EN<<E5E8<=D5E;<>@5E=<?:5E=<@35E><@M5E@<AG5EA<B@5EA<C:5EC<D45ED<DM5ED<EG5EF<F@5EG<G:5EG<H45EI<HM5EH<IH5EF<J@5E><K85E4<L15DH<LI5D><MA5D3<N95CH<O15C=<OI5C2=0B5BG=1:5B<=225B1=2=5B2=2E5B5=2N5B8=3D5B9=4:5B9=4J5B8=595B6=5H5B3=675B1=6F5AO=765AM=7E5AJ=845AH=925A>=9O5A4=:K5@J=;?5@==<95?M==95?==>95>L=?95><=@95=L=A95=;=B95<K=BO5<F=CC5<B=D75<?=DK5<:=E=5;M=F05;A=FB5;3=G45:F=GG5:9=H959L=HL59?=I>592=J558F=K;58A=LA58<=M7580=MH57D=N9577=NJ56K=O>575>0157B>0C57N>1658:>1J58<>2=584>3157L>3E57D>4857<>4K574>5?56M>6356D>6F566>7955C>8<55@>9I55E>;655K>;M55B><E558>==54O>>554F>>M54=>?E544>@=53K>A553B>AO537>BK52M>CG52C>DC528>E>51N>F551B>FD516>G350K>GB50?>H0502>H?4OF>HN4O:>I<4NN>IK4NB>J:4N7>JI4MJ>K74M>>KF4M2>L>4L?>M;4KH>N84K2>O54J<?024IF?0O4HO?1>4HC?1M4H6?2;4GJ?314GG?3G4GG?4>4GG?554GG?5L4GG?6@4GC?734G<?7E4G6?884FO?8J4FI?9=4FB?:04F<?:B4F6?;44EO?;G4EH?<:4EB?<M4E6?=A4DH?>54D:?>H4CM??<4C>?@44BF?@K4AN?AB4A6?B94@??C14?G?CH4>O?D=4>8?E04=B?ED4<K?FE4<9?GC4<0?H74;H?HL4;??I@4;6?J44:M?JI4:D?K=4:;?L14:3?LO49H?N049<?O0490?OA48<@0047I@0J47D@1D47@@2=47<@36478@40473@4J470@5E46K@6J46F@7N46@@9246;@:7466@;;461@<?45K@=D45F@>H45@@?L45;@A1455@B5450@C944K@D=44E@EB44@@FF44;@GK445@HO440
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
20964E74565=73J66673?66N73567F72J68?72?6977246:071I6:H71?6;77216;O72=6<J72F6=D72O6>>7396?<73B6@;73H6A<73K6B=73N6C>7426D@7426EB73O6FC73K6GE73H6HG73E6IH73>6JI7386KK7326LL72M6MM72G6NN72A6OO72<70G72E71>72O72573972M73C73D73N74;74775374A75J74K76A75577975?78075I78G76379?76=7:676G7:M7717;E77;7<<77E7=377M7>277E7?37797@376L7A276@7B37637C275F7D275:7E274M7F174A7G17447H173H7HJ73C7I;73E7IL73H7J<73L7JL7417K<7477KK74<7L;74A7LK74F7M;74K7MK7517N;7567NO7557OD75480975480O75681F75482=75283475083L74M84A74K85474I85H74G86;74E86M74C87>74?88074<88A74989474789I7448:?7428;47408;I73N8<>73L8=173L8=H73I8>E73C8?A73=8@=7378A97318B572K8C272E8CN72?8DJ7298EF7228FB71L8G?71F8H;71@8I771:8J37148JE7118J?71;8J971E8J471O8IN7298IH72C8IB72M8II7318JI72J8KI72C8LK7238ML71D8NN7148OO70E9117059226OF9306O:93L6NO94H6NE95C6N;96?6N097;6ME9856M@98M6MA99G6MB9:?6MC9;?6N89<?6O29=@6OL9>870<9>J70C9?<70I9?N7109@@7169A47199B=70?9CE6OE9DN6NK9F66N29G?6M89HE6LI9ID6LC9JC6L=9KD6L69LC6L09MC6KJ9NC6KC9OC6K=:0B6K7:1B6K0:2B6JJ:3B6JC:4A6J=:5A6J6:6A6J0:756IE:7D6I7:836HI:8E6H;:996GM:9N6G>::C6G0:;=6FF:<;6F@:=:6F::>86F5:?76EO:@56EJ:A46EE:AM6E9:BF6DO:C@6DC:D96D8:E26CM:EL6CB:ED6@A:F>6@>:G86@;:H26@8:HL6@5:IG6@2:JA6?O:K;6?L:L56?I:LO6?F:MI6?C:NC6?>:O<6?3:OM6>J;0>6>B;0O6>:;1@6>2;216=J;2B6=A;336=:;3C6=1;446<I;4E6<A;5<6<:;6;6<3;766;M;7O6;G;8I6;@;9@6;<;:76;8;:M6;4;;E6;0;<;6:L;=26:H;=I6:D;>@6:@;?:6:=;A;6:I;C46:L;D;6:<;E869D;F168J;FL681;H:67H;IH67@;K7678;L366N;LD66C;M6669;MG65O;N865D;NI65:;O;650<0364E<1764:<2;63N<3@63C<4962D<4M61=<5A606<665O0<6J5MI<;B5O1<;@5OJ<;K5OD<<75O?<<C5O:<=65NG<>55M=<?E5MC<A45N4<AM5N6<BF5N8<C?5N:<D85N<<E15N<<EB5ML<F35M;<FD5LL<G55L;<GG5KK<H85K:<HI5JJ<I:5J:<J65IN<K15IB<KM5I7<LH5HK<MD5H?<NA5H0<O=5GB=0:5G3=165FE=225F6=305EL=3N5EG=4L5EB=5J5E==6B5E9=735E7=7B5E4=825E1=8A5DO=915DL=9A5DJ=:15DG=:@5DD=;05DB=;@5D?=<65D;==45D7==M5CN=>=5C9=>L5BF=?;5B1=?K5A?=@B5A0=A95@J=AM5@F=BA5@C=C55@?=CI5@;=D>5@8=E25@4=EE5?K=FK5?E=H15?@=I75?:=J=5?5=KC5>O=LH5>J=MN5>D=O45>?=OL5>4>0=5=H>0N5=;>1@5<N>215<A>2B5<5>335;H>3D5;;>455:N>4F5:B>575:5>5I59I>6:59<>6M596>7@58C>83580>8F57=>9956K>:<56G>;I56M>=6572>>B578>?O57=>A<57D>B>57B>C6578>CN56O>DF56F>E>56=>F6564>FN55K>GI55@>HD555>I@54L>J<54A>K0546>K?53I>KN53=>L=531>LK52E>M:529>MI51M>N851A>NF515>O550I>OD50=?02501?0A4OE?104O9?1?4NM?1M4NA?2<4N5?2K4MI?3:4M=?3H4M1?474LE?4I4L??5?4L??654L??6K4L??7@4L??864L??8M4L??9D4L>?:94L=?:M4KO?;A4KB?<44K4?<H4JF?=<4J9?>04IK?>C4I=??74I0??K4HB?@>4H4?A24GG?AF4G9?B:4FK?C14F3?CH4E;?D@4DD?E74CK?EN4C4?FD4B@?G:4AL?GO4A9?HE4@F?I;4@2?J04???JD4>O?K64>D?KG4>:?L94=O?LJ4=E?M<4=9?MN4<O?N?4<D?O04<9?OE4<0@0<4;E@164;@@2:4;:@3?4;4@4C4:O@5H4:J@6L4:D@804:?@944::@:94:4@;=49O@<B49J@=F49E@>J49?@?N49:@A2495@B748O@C;48J@D?48E@ED48?
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
21468H77A69@7766:976L6;176A6;J7666<<76@6=276M6=L7766>G77@6?C77I6@A7816A?7896B=78A6C<78J6D978C6E978A6F878@6G778>6H87886I97826J;77M6K<77G6L=77A6M>77<6N@7756OA76O70C76I71D76D72C76A73;76K74277574I77?75A77H76878276O78<77G78G78>79179579:79M79D7:D79N7;;7:77<07:;7<F7:@7=;7:C7>17:H7>F7:M7?;7;07@17;57@M7:M7AM7:A7BM7:57CL79H7DL79<7EL78O7FL78C7GK7867HB7847I87827IN77O7JB77O7K777O7KJ77O7L;7827LL7847M=7867MN7897N?78;7NO78>7OB78?80878>80M78>81C78=82978<82O78=83C78;84678984J78885=78786178686E78587878487L78288B77K89777F89L77D8:A77A8;677?8;I77:8<;7728<L76J8=@76C8>;76<8?67668@07608@L75J8AF75C8BA75=8C<7578D57528DL74M8ED74G8F<74C8G374>8GH74;8GG74F8GF7518GE75;8GD75F8GI7608H@76<8I776F8J476F8K57658L875E8M77548N374@8O474190673A91773292972B93:72294:71D95671:96171096M70E97G70A98@70B99970C9:270D9:J70E9;D70F9<<70G9=570H9=N70I9>M71>9?N7289@N7329AO73L9C074G9D075@9E176;9F376O9G775O9H;74K9I?73H9JC72D9KG71A9LK70=9MM6OG9NK6O<9OJ6O1:0I6NG:1H6N=:2F6N2:3E6MH:4D6M=:5=6M2:666LG:6N6L<:7E6KJ:8<6K4:936J@:9J6IK::B6I6:;86HC:;M6H4:<B6GF:=66G7:=J6FI:>E6F@:?D6F::@B6F5:AA6EO:B;6F0:C36F5:CJ6F::DA6FA:E96FF:F06FK:FG6G0:F>6CG:G66CO:GO6D9:HG6DA:I@6DI:J96E2:JO6DO:KD6DG:L:6D@:LO6D9:ME6D2:N:6CJ:O06CC:OG6C9;0@6C0;186BG;1I6B@;296B7;2K6AO;3<6AG;3L6A?;4=6A7;4N6@O;5M6@H;6L6@B;7J6@;;8H6@5;9B6?O;:;6?H;;56?B;;O6?<;<H6?5;=B6>O;>:6>J;?06>F;@;6>H;AB6>9;BJ6=J;D16=;;E86<L;F@6<=;GA6;I;HE6;B;I=6;;;J66;4;K06:M;KH6:F;LA6:?;M36:1;M?69>;MK68K;NH68><0:68G<2769;<4369=<5N697<7I690<9D68I<9A67<<9566J<8E66G<8@66B<9H668<;165O<<O65:<>364M<>?64G<>L64A<?O64K<A665;<AM65F<BC65E<C564M<CF644<D763;<DH62A<E:61H<EK610<F<607<FM5O><G>5NG<H05N7<HA5MF<I25M5<ID5LE<J75L5<K45KF<L15K8<LM5JJ<MJ5J@<NH5J;<OF5J7=0D5J1=1B5IM=2@5IH=3>5IC=4<5I>=5:5I:=685I5=765I0=845HK=8I5HH=995HE=9I5HB=:85H@=;25H<=<05H7=<N5H2==K5GN=>I5GI=?G5GD=@D5G?=A85G0=AI5F:=B=5EC=C05DK=CC5D4=D75C<=DN5C5=EG5C9=F?5C<=G85C?=H15CB=HJ5CE=IB5CI=J?5CK=KH5CH=M05CF=N85CC=OA5CA>0I5C?>1K5C1>2<5B6>2L5A>>3<5@G>3L5?O>4<5?8>4L5>@>5<5=I>5L5=1>6>5<@>745<7>7I5<0>8?5;G>915;:>995:8>9B597>:658E>:H582>;B57F><O57L>><581>?I587>A558=>BB58B>CO58H>E;58M>FH593>H0596>HH58L>I@58C>J858:>K0580>KH57G>LC57=>M?572>N;56G>NO56<>O>561>OM55E?0<558?0J54L?1954@?1H544?2653H?2E53=?34530?3C52D?41528?4@51L?4O51@?5>514?5L50H?6;50<?7050;?7F50;?8;50;?9150:?9G50:?:<50:?;1509?;B4ON?<24OC?<B4O8?=24NM?=C4NA?>34N6?>D4MK??44M@??E4M4?@54LJ?@E4L>?A54L3?AF4KH?B64K=?BG4K2?C74JF?CG4J;?D74J0?DE4IG?E;4I4?F04H@?FG4GL?G<4G8?H24FE?HH4F1?I>4E=?J44DJ?JJ4D7?K?4CC?L54BO?LK4B<?M@4AN?N64AL?NK4AK?O@4AI@034AH@0G4AF@1;4AD@1O4AC@2B4A@@364A?@3J4A<@4>4A;@514A9@5E4A7@694A6@744A1@8:4@I@9@4@B@:G4@:@;L4@2@=34?J@>84?C@??4?;@@D4?3@AJ4>K
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
2196<K7:M6=@7:O6>57;=6?07;G6?J7<06@G7<86AE7<A6BC7<I6CB7=16D@7<E6E>7<26F<7;L6G;7;J6H:7;I6I97;H6J87;G6K77;F6L67;E6M57;C6N57;C6O47;B7047;>7157;87267;27387:M7497:F7577:F75N7;076E7;:77=7;D7847;N78K7<879B7<>7:77<B7:L7<F7;B7<K7<77<N7<L7=37=B7=87>77=;7>M7=@7?B7=D7@87=H7@M7=M7AB7>17B87>57BM7>97CG7>77DG7=J7EF7=>7FF7=17GC7<H7H97<E7HO7<C7IE7<A7J;7<?7K17<<7KG7<:7L;7<:7M07<:7MD7<:7N87<;7NJ7<<7O?7<<8037<<80H7<<81=7<<8227<;82G7<;83<7<:8417<:84E7<98587<885L7<786@7<68747<187J7;J88@7;C8957;>89N7;?8:F7;A8;97;:8;J7;18<:7:G8<K7:=8=<7:48=L79J8><79G8?079:8?D78N8@878B8@O78<8AG7888B>7828C677N8CN77H8DD77E8DC77O8E778:8EN78E8FF7908G=79:8H479E8HK7:08IC7:;8JD79L8KF79<8LH78L8MJ78<8NL77K8ON77<91076L92076:92I75D93A74N94B74?95D73O96@73E97:73B98273C98K73E99D73E9:=73F9;573G9;O73H9<G73I9=@73J9>973K9?273K9?K73M9@D73N9A<73N9B<74D9C?74J9DF74?9EL7449G373J9H973A9I>73<9JD7389KI7339LO72O9N472J9O972E:0>72@:0K727:1<71L:2571C:2N718:3H70N:4B70D:5;70::65701:7A706:8<6OO:966OH::06OA::M6O6:;I6NK:<E6NA:=B6N6:>>6ML:?:6MB:@76M7:A26LJ:AL6L8:BJ6JK:C;6JF:CH6JG:D66JJ:DD6JK:E26JN:E@6K0:EM6K1:F36JN:F=6JE:G46JC:GJ6J<:H?6J5:I56IN:IJ6IF:J@6I?:K56I8:KK6I0:L@6HI:M66HB:MK6H::NA6H3:O76GK:OO6GB;0H6G9;1@6G0;286FG;306F>;3D6F5;456EM;4F6EE;5?6E>;6=6E7;7<6E1;8;6DJ;996DC;:86D=;;46D6;;O6D0;=16CI;>36CB;?46C;;@66C3;A76BM;B96BF;CH6B;;E@6AO;G76AC;H@6A7;IA6@L;JC6@B;KD6@7;L<6?L;LI6?C;M76?:;MD6?0;N26>G;N?6>>;OC6>;<136><<2D6>=<426>0<4G6>@<5J6?1<776?A<8D6@0<8N6?B<8B6>C<876=D<836<J<8N6<H<9M6<E<;56<><<<6<8<=C6<1<>K6;J<@26;B<@M6;J<AD6<5<B<6<?<BN6;H<C?6:O<D06:6<DA69=<E268D<ED67K<F5672<FF669<G765@<GI64G<H?63F<I562E<IL61B<JF60I<KK616<M061D<N5621<O261K<OO61E=0L61?=1I619=2F613=3C60M=4@60G=5;60==635OM=6L5O<=7D5NL=8;5N;=945MK=9M5M9=:G5LI=;>5KL=<<5K<==E5K==>F5K9=?C5K3=@A5JN=A?5JJ=B=5JD=C65J9=CN5IL=DF5I?=E>5I3=F55HC=FI5GL=G<5G6=GO5F?=HE5F0=IH5F:=JJ5FE=KL5G0=LO5G:=MH5G>=NA5GA=O:5GE>0=5GD>1F5GA>2N5G?>3?5F6>425DN>4C5CE>555B<>5K5A9>6B5@8>795?7>805>N>8F5>G>9<5>?>:15>7>:G5=O>;<5=F>;D5<E>;L5;D><55:B><=59@><K58F>>558J>?B590>@O596>B<59<>CH59A>E559F>FB59L>GO5:2>I;5:8>JH5:=>L55:C>MA5:H>N:5:?>O25:6>OJ59M?0B59D?1>599?2:58O?2N58C?3=587?3L57L?4:57@?4I574?5856G?5G56;?6555O?6D55C?73558?7B54L?8054??8@546?96546?9L546?:A546?;7546?;L543?<<53H?<L53=?=<532?=M52G?>=52;?>N521??>51E??O51:?@?50N?@O50D?A?509?B04OM?B@4OC?BE4O@?B@4OF?BH4OC?BO4O@?C74O=?C?4O;?CO4NM?DE4N:?E;4MG?F04M3?FG4L??G<4KL?H24K8?HH4JD?I>4J0?J44I=?JJ4HJ?K?4H8?L04H8?LB4H8?M34H7?MJ4H6?NA4H4?O84H4?OO4H3@0F4H2@1=4H1@244H0@2K4GO@3B4GN@494GM@504GL@5G4GK@6=4GJ@714GH@7D4GF@894GE@8L4GB@9@4GA@:44G?@:H4G=
2206=D7;F6>67<86?17<B6?L7<K6@I7=46AG7=<6BE7=D6CC7=L6DA7=@6E?7<M6F=7<A6G<7<?6H;7<?6I:7<=6J97<<6K87<<6L77<:6M67<96N67<86O47<77037<57137<57237<07357;J7467;D7577;>7617;E76H7;O77@7<87877<B78M7<K79C7<N7:87=37:N7=77;C7=;7<97=?7<N7=D7=D7=H7>97=L7>N7>17?C7>47@97>97@N7>=7AD7>A7B97>F7BO7>J7CD7>N7D?7>I7E?7>=7F?7>07G=7=E7H37=C7HI7=A7I?7=>7J57=<7JK7=:7KA7=87L77=57LK7=67M?7=67N37=67NG7=67O=7=68027=580G7=581<7=58217=582E7=483:7=483O7=384D7=38577=285K7=186?7<O8747<I87I7<B88@7<=8987<?8:07<@8:I7<B8;=7<>8;N7<48<?7;J8<O7;@8=@7;68>07:N8>?7:N8?27:E8?F7:98@:79M8@N79A8AA7948B578H8BK78A8CC78;8D078>8DG78H8E>7948F579>8FL79I8GD7:48H;7:?8I27:I8IM7:M8JN7:=8L179M8M279>8N578N8O678=90977M91;77=92=76M93976994175C94J74N95L74>96G74597A74698974799374899K7499:D74:9;=74;9<674<9<N74=9=H74>9>@74>9?974@9@274A9@K74A9AC74B9B?74B9CD74>9DJ7489EO7449G573O9H:73K9I?73F9JD73B9KJ73=9LO7389N47339O:72O:0?72J:0J72@:0J724:1D71J:2=71@:37716:4070L:4J70B:5C708:7E710:8?70I:9970A::470:::M704:;G6OM:<C6OC:=@6O8:><6NN:?86NC:@56N8:A16MO:AM6MD:C;6LL:D66KK:DD6KM:E26KO:E?6L0:EF6KM:EL6KK:F26KH:F96KE:FE6K<:FO6K1:GA6JK:H86JI:HO6JF:IG6JC:J=6J<:K26J5:KG6IM:L=6IF:M36I?:MH6I8:N>6I0:O46HI:OL6H?;0D6H6;1<6GM;256GC;2M6G:;3E6G1;476FI;4H6FA;5B6F:;6A6F4;7?6EM;8>6EG;9=6E@;:;6E9;;=6E2;<@6DK;=C6DD;>D6D=;?F6D6;@G6CO;AI6CH;BJ6CA;D16C9;EH6BM;G76B@;H96B6;I;6AK;J<6A@;K>6A5;L?6@J;M86@@;ME6@7;N36?M;N@6?D;NM6?;;OO6?7<1A6?8<316?7<4:6>E<4M6?8<5B6?K<6O6@:<8<6@I<8N6@B<8B6?B<866>C<8A6>4<9<6>3<:86>2<;46>0<<86=K<=?6=D<>F6==<?M6=7<A06=5<AG6=@<B>6=G<C06<N<CA6<5<D26;<<DC6:C<E469K<EF692<F7689<FI67><G@66=<H665;<HM649<ID637<J;626<K@62D<LE631<MJ63><NO63K<ON63K=0L63E=1I63?=2F63:=3@62N=4862==5061L=5H61<=6@60L=7860;=805OJ=8H5O:=9B5NI=:<5N8=;65MF=;M5LK=<C5KM==G5KF=>O5KF=@55KF=A35K@=B15K;=BM5K5=CE5JH=D>5J;=E65IN=EN5IB=FF5I5=G=5HE=H05GN=HC5G7=I65F@=J75FI=K:5G4=L<5G?=M?5GI=NA5H4=O<5H8>045H<>135H=>2;5H;>3<5GK>3N5FB>4@5E9>525D2>5J5C1>6@5B0>775@N>7O5?M>8E5?6>9<5?0>:25>I>:I5>B>;>5>:>;N5=I><75<G><?5;E><G5:D><O59B>=B58M>>O593>@<599>AI59>>C559D>DB59I>EO59O>G<5:5>HH5::>J55:@>KB5:E>LN5:K>N;5;1>O95:K?015:B?0I5:9?1A5:0?2=59E?3959;?3J590?4958C?4H587?5757K?5E57??64574?6C56G?7256;?7@55O?7O55C?8>557?8N54O?9D54O?::54O?:O54O?;E54N?<854G?<H54<?=8541?=I53F?>953:?>I530??952D??J529?@:51N?@K51C?A;518?AL50L?B<50B?B750F?B550K?B<50H?BD50E?BK50B?C350??C;50=?CO4OK?DE4O7?E;4NC?F04N0?FG4M=?G<4LI?H24L5?HH4KB?I>4JN?J44J:?JJ4IG?K<4I@?KN4I@?L?4I@?M14I??MG4I>?N>4I=?O54I<?OL4I;@0C4I:@1;4I9@214I8@2H4I7@3?4I6@464I5@4M4I4@5D4I4@6;4I2@724I1@7I4I1@8=4HN@914HM@9D4HK@:94HI
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
2236@K7II6A?7II6B:7J26C87J;6D67J:6E37IG6F17I46FO7HB6GM7GO6HJ7G<6IH7FJ6JF7F>6KE7F=6LD7F<6MC7F;6ND7FC6OE7G870G7GM71H7HA72>7H:7337H273H7GJ74=7GB7527G=75F7G876D7G677A7G97877GF78O7GJ79H7GN7:@7H37;97H77<17H;7<I7H?7=B7HC7>:7HG7?27HK7?K7I07@C7I37A;7I77B47I<7C97I67CO7I07D;7I07E07HN7EE7HL7F;7HJ7G07HF7GE7HC7H;7H@7I17H>7IF7H;7J;7H87K17H77KN7H67LI7H77MC7H97N>7H:7O87H<8047H=8137H>8227H?8317H@83B7H98427H284B7GL8527GE85?7GA85H7GG86B7GI87=7GL8877GO8917H389J7H88:D7H=8;=7HC8<47I28<I7I<8==7IF8>17IO8>E7J98?57J<8?D7J:8@37J98@B7J78A17J58A@7J48AO7J28BD7IM8D37I<8DM7HF8E87GK8EB7G18EL7F68F>7F28G37F88GI7F=8H>7FB8I:7FB8J;7F<8K=7F78L?7F28M@7EL8NB7EF8OC7E@90E7E;91B7DN92@7DG93A7DB94J7DC9637DC97=7DE98F7DF9:07DF9;:7DG9<C7DH9=M7DI9?67DJ9@67DF9A37D?9B17D99BD7D89BM7D;9C77D?9CA7DC9CJ7DF9D57DI9D>7DL9DH7E09E27E49E;7E79F17DF9FI7D39GA7C?9H;7C39I=7C59J>7C79K?7C99L@7C;9MA7C=9O<7BJ9OC7B99OG7AH9OK7A5:0<7A9:1?7@F:2A7@3:3D7?A:4F7>N:5H7>;:6J7=H:7M7=5:8O7<B::17;O:;47;<:<27:O:<K7:G:=D7:@:><7:9:?67:1:@279D:A0791:AN78?:BC781:C:77B:CO774:DD76M:E976H:F2768:FC75L:G475@:GE754:H674G:HG74<:I8740:IJ73D:J;738:JK730:K;72H:KL72B:L<72::LL722:M<71L:NC71N:OM722;18726;2C72:;3M72?;5872C;6B72G;80731;9?73>;:M73K;<<748;=L74F;>O74C;@073O;A073<;AK72O;BF72@;C@725;D771M;DL71I;EA71C;F571?;FJ719;G>715;H3710;I370:;J16OD;K16NN;L06N;;LO6MN;MN6MA;NM6M3;OL6LF<0K6L:<1K6KL<2K6K=<436JK<506KI<5M6LF<6H6MB<7;6N8<7O6NN<8I6NK<9D6N?<:@6N5<;;6MI<<66M?<=26M3<=M6LI<>I6L=<?D6L2<@?6KG<A;6KB<B56K4<BO6J6<CG6IC<D@6I0<E96H><F16GL<FJ6G:<GC6FG<H;6F5<I46EC<IM6E0<JM6D=<L26CI<M56C6<N96BC<O86B<=066B8=146B4=226B0=316AL=3I6AD=4B6A<=5;6A4=686@N=7<6@K=8@6@G=9D6@D=:H6@A=;L6@==<O6@:=>36@6=?36?O=?@6?4=?N6>9=@;6===A16<K=AK6<8=BI6;?=CD6:O=D@6:N=EO6:F=G?6:?=H56:<=H>6:;=HG6:;=HO6:6=I469E=IC697=JA68O=K?68G=L<687=M867A=N366K=NO665=OK65?>0G64I>1C643>2?63=>3:62G>4761O>53613>60609>6L5OH>845OD>945O1>:55N>>;55MJ><55M6>=65LE>>75L3>?85KB>@95K0>A45JB>AF5J7>B75IM>BH5IB>C;5I7>CN5HL>D?5HC>E85H<>FB5H0>GL5GE>I55G9>J?5FN>KI5FC>M35F7>N<5EL>OF5EA?105E5?295DJ?3C5D??4L5D4?665CH?7@5C=?8I5C2?:45BG?;35B=?;E5AC?<;5B3?<85B8?<?5B5?<G5B2?<O5B0?=65AM?=>5AJ?=F5AG?=N5AD?>55AB?>=5A?
2226@47K56@L7JI6AC7JB6B<7JN6C97K66D77K46E57JA6F37IO6G07I<6GN7HI6HK7H76II7GD6JG7G46KF7G26LE7G16ME7G66NG7GK6OH7H@70J7I471E7I;72:7I372O7HJ73D7HD7487H?74M7H:75D7H776D7H477E7H478;7HA7927HN79J7I37:B7I67;:7I;7<37I?7<K7IC7=D7IG7><7IK7?47IO7?M7J37@E7J77A=7J<7BO7IL7CB7IG7CL7IF7D;7IF7E07IE7ED7IC7F:7I@7G07I>7GE7I;7H:7I87I07I67IF7I37J;7I17K57I07L37HO7M07I07MJ7I17ND7I37O?7I480=7I581=7I682<7I683;7I78467I684F7HO8567HI85F7HB8637H>8617HE86A7HJ87;7HM8867I08907I289J7I58:E7I88;?7I98<57I<8<I7IF8==7IO8>07J98>E7JB8?77JJ8?F7JH8@57JG8@D7JE8A37JD8AB7JB8B17JA8BG7J<8D67IK8EA7I98EK7H>8F57GD8F?7FI8G07FA8GF7FG8H;7FL8I07G28IF7G78J;7G=8K77G;8L97G68M;7G18N<7FK8O>7FE90?7F?9167E@9287E693A7E794K7E89647E997>7E:98G7E;9:17E<9;:7E=9<D7E>9=M7E?9?67E?9@@7E@9A>7E;9B77E79BA7E;9BK7E>9C47EB9C>7EE9CH7EH9D27EL9D<7EO9DE7F39DO7F79E97F:9EK7EO9FC7E;9G;7DG9H:7DF9I<7DI9J<7DJ9K>7DL9L?7DO9M@7E19O17DF9O>7D09OB7C>9OF7BM:007B=:127AJ:247A8:377@E:497@2:5;7??:6>7>L:7@7>9:8B7=F:9D7=3::G7<@:;I7;M:<H7;?:=A7;7:>:7;0:?37:I:@07:9:@O79F:AE797:B:78J:C078?:CE78::D9785:DN77O:EK77;:F<76O:FM76C:G?766:H075K:HA75?:I2753:ID74G:J574;:JF73N:K773B:KH736:L972K:LJ72?:M;722:N@728:OO72E;1>732;2M73?;4<73M;5K74:;7:74G;8H754;:875B;;G75O;=576<;>9768;?375H;@2756;A774>;B<73H;C8739;CL735;DA72O;E672K;EJ72F;F?72A;G472<;GH727;H=722;I<71=;J;70G;K:701;L96O;;M86NE;N76N8;O76MK<066M=<156M0<246LC<366L3<496L6<566M3<646N0<6N6NK<7B6OA<87701<936OF<9N6O;<:I6O0<;E6NE<<@6N9<=<6MO<>76MC<?36M9<?N6LM<@I6LC<AE6L9<B>6KD<C76K2<CO6J@<DH6IN<EA6I<<F96HI<G26H6<GK6GD<HC6G2<I<6F@<J56EN<K86E:<L<6DF<M?6D3<ND6C@<OD6C3=0C6C0=1A6BL=2?6BG=3<6BC=446B:=4M6B2=5H6AK=6L6AG=806AD=946AA=:76A>=;<6A;=<?6A7==C6A4=>G6A0=?K6@M=@E6@?=A26?D=A?6>I=AO6=I=BJ6<K=CH6<2=DA6;K=E76;M=FE6;F=H56;>=HH6;;=I16;;=I96;;=IB6;:=IJ6;5=J06:D=JN6:<=KL6:2=LH69;=MD68E=N@67O=O<679>0766C>1365L>20652>2M647>3J63<>4F62A>5C61F>6@60K>7;60E>8D60L>9J60?>:J5OL>;K5O8><K5ND>=L5N1>>L5M=>?M5LI>@M5L6>AE5KF>B85K:>BK5JO>C>5JD>D15J9>DA5J1>E05IK>E?5ID>FF5I:>H05HN>I95HC>JC5H8>KM5GM>M65GB>N@5G6>OI5FK?135F@?2=5F4?3G5EI?515E=?6:5E2?7E5DH?915D>?:35D4?:M5CJ?;L5C=?<45C9?<<5C7?<C5C4?<K5C2?=25BO?=:5BL?=B5BI?=J5BF?>15BD?>95BA?>@5B?
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
2206>G7MK6??7M@6@77M56A07LJ6AH7L@6B?7LD6C<7LM6D:7LH6E87L56F57KC6G27K06H07J=6HN7IK6IL7I86JI7HE6KH7H=6LI7I16MK7IF6NL7J:6ON7JO70H7K371=7JN7217JH72F7JD73:7J?73O7J:74D7J575E7J476F7J277G7J178B7J77997JD79O7K17:E7K>7;>7KB7<67KG7<N7KK7=G7KN7>?7L37?77L77@A7KN7B:7K87BJ7K57C47K57C>7K47CH7K47D;7K37E07K27EE7K07F:7JO7FO7JL7GD7JK7H:7JH7I07JG7IE7JE7J=7JC7K<7JB7L;7JB7M:7JA7N77JA7O37JB8027JC8117JD8207JE82O7JE83O7JF84N7JG85M7JH86?7JB86O7J<87;7J88797J?8767JD87=7JE8897JF8947JH89O7JI8:K7JK8;C7JC8<87J88<J7J:8==7JB8>07JK8>D7K58?87K>8?K7KE8@97KD8@I7KB8A87KA8AG7K?8B57K>8BL7K88D;7JG8EJ7J68G17IC8G;7HH8GF7GM8H57GA8HJ7GF8I?7GK8J57H08JJ7H68K@7H;8L57H@8LK7HF8M@7HK8N67I08O37HO9047HI90<7G691C7G192K7FK9437FG95>7FH96H7FH9837FI99=7FJ9:H7FK9<37FK9==7FL9>H7FM9@37FN9A67G09AC7G59B27G;9B?7GA9BM7GG9C87GJ9CB7GN9CL7H29D67H59D@7H99DI7H<9E37H@9E?7H?9F77GK9G77GK9H87GM9I97H09J97GJ9K77G>9L47G19M27FE9N07F89NN7F29ON7F5:0O7F4:137DO:157CE:247C4:337BA:427B0:517A?:617@M:707@<:7O7?J:8N7?9:9M7>G:;07>4:<27=A:=57<N:>47<?:>N7<5:?G7;F:@<7;?:A17;::AE7;4:B:7:O:BO7:I:CD7:D:DC7:4:E?79A:F0795:FA78I:G278=:GD781:H577E:HF779:I776L:IH76@:J:765:JK75I:KD75E:LG75K:MJ761:NN768;0276?;1A76L;30779;4?77G;5N784;7=78A;8K78N;::79<;;J79I;<J79B;=F792;>@78B;?;782;@677B;A:76L;B?765;C=75F;D475@;DK75:;EA754;F774N;FN74H;GE74B;H;74<;I0747;IO73B;JN72L;KM726;LM71@;MK70J;NJ704;OJ6O><0H6NK<1H6N?<2G6N2<3I6ML<4G6NI<5D6OG<6B70D<7;71><7O722<8K71G<9F71<<:B711<;=70F<<870:<=4700<=O6OD<>J6O5<?C6N:<@<6M?<A56LC<B;6M<<C36M<<CG6LL<D@6L:<E96KG<F16K5<FJ6JC<GC6J1<H;6I?<I46HL<IM6H:<JH6GG<KL6G4<M06F@<N46EM<O86E9=0<6DE=1;6D>=296D:=386D6=416CO=4I6CF=5K6CB=706C?=836C<=976C8=:;6C5=;?6C1=<B6BN==F6BJ=>J6BG=?N6BC=A16B@=B66B==C96B:=CI6A<=D>6@0=DI6>@=EE6=D=F;6=D=G16=G=H26=D=IA6===IL6=<=J56=;=J=6=;=JG6=:=JO6=:=K86=:=KI6<N=LF6<3=MC6;8=N?6:==O<69B>0968G>1567L>23672>2O666>3L65<>4I64@>5E63F>6B62K>7?623>8962>>9962H>:J62L><662H>=7625>>761A>?760L>@6602>A55O9>B45N?>C15MI>CD5M>>D75M2>DE5LL>E45LF>EC5L@>F25L:>FA5L4>G05KN>H75KC>I@5K7>JJ5JL>L45J@>M>5J6>NG5IK?015I??1;5I4?2G5HJ?425H@?5>5H5?6I5GL?825GB?8M5G8?9M5FI?;F5EC?<45E;?<;5E8?<C5E6?<K5E3?=25E0?=:5DN?=A5DK?=I5DH?>15DE?>95DC?>@5D@?>H5D=
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
2186=:80A6>28066>K7OK6?C7OA6@;7O66A47NL6AL7N@6BC7N96C>7NB6D<7N<6E:7MI6F77M76G57LD6H37L16I07K>6IN7JN6JK7JH6KG7JB6L>7JE6M87JL6N;7KC6O=7L970?7LM7137LI71G7LD72<7L?7317L:73E7L574F7L475F7L276G7L177H7KO78I7KM79@7L:7:67LG7:L7M37;C7M@7<97MM7=17N37>27N07?L7M:7AE7LD7B27LC7B<7LB7BF7LB7C07LB7C:7LA7CF7LA7D;7LA7E07L@7EE7L?7F:7L>7FO7L>7GD7L<7H:7L:7I07L87IE7L77JD7L67KC7L57LB7L47MB7L57NB7L77O@7L080A7L081B7L182C7L483C7L484B7L585A7L686A7L787@7L78887L588D7L288@7L788:7L78857L788?7L989:7L:8:67L<8:O7L68;E7KH8<;7K98<M7K98=?7K;8>07K>8>D7KH8?87L18?L7L:8@>7LA8@M7L?8A<7L>8AK7L<8B:7L:8C27L48DA7KC8F07K28G?7JA8HB7IM8HL7I38I97H@8IO7HE8JD7HJ8K97I08KO7I58LD7I:8M:7I?8MO7IE8NE7IJ8O:7IO8NL7HK8O37I69017I49127I09277HK93>7HF94F7HA95N7H<9777H998A7H:99L7H:9;67H;9<A7H<9=L7H=9?67H=9?J7HB9@87HH9@F7HM9A47I39AB7I99B07I>9B>7ID9BK7II9C97IO9CG7J59D57J:9DC7J@9E97JE9F47J;9FM7IN9GJ7IA9HH7I59IF7HI9JD7H<9KB7H09L?7GD9M>7G79N;7FK9O:7FF:0;7FC:1<7F=:2=7F7:2=7DK:3;7D9:4;7CH:5:7C7:697BE:787B4:887AB:967A1::67@?:;57?N:<47??:=47?1:>37>D:>L7>9:?87>4:?F7=O:@;7=J:A07=D:AD7=>:B97=9:BN7=4:D:7<8:E27;G:ED7;;:F57:O:FF7:C:G77:7:GI79J:H:79>:HM795:J179<:K479B:L879I:M;79O:N>7:6:OA7:<;0E7:C;1H7:I;337;4;4A7;A;617;O;7@7<<;8N7<I;:=7=6;;<7<L;<77<<;=27;L;=M7;<;>H7:M;?C7:=;@>79M;A=79:;BB78B;C@783;D777N;DM77H;ED77B;F;77<;G1776;GG770;HD76F;IE767;JF75J;KF75;;LG74M;MF747;ND73>;OB72E<0@71M<1?715<2=70=<2L6OF<4770?<5471=<6272:<6O738<7G741<8C73H<9>73=<::732<;372:<;L71?<<E70D<=>6OH<>76NM<?06N2<?I6M6<@B6L<<AH6M3<C46N@<D26NO<DC6NA<E46N2<EI6MB<FB6M0<G:6L=<H36KK<HL6K9<IE6JF<J=6J4<K=6IA<LA6HM<ME6H:<NI6GF<OL6G3=116F?=246EM=326EI=3M6EB=4K6E<=5O6E9=736E6=866E2=9:6DO=:>6DK=;B6DH=<E6DE==J6DB=><6D6=>M6CI=?>6C<=@76C4=A66C0=B56BL=C86BB=E16B8=FI6B2=H=6AA=HM6@B=HK6?@=IA6?B=JH6?<=K06?<=K96?<=KB6?<=L56>6=LF6=G=M76=:=MG6<N=N76<B=NG6<6=O?6;B>0;6:G>1869L>25691>31686>3N67;>4K66@>5H65F>6E64J>7A640>8=63M>97649>:164D>;?64I>=064M>>>64H>?>63O>@=635>A<62;>B;61B>C;60H>D9600>DJ5OI>E95OB>EH5O;>F65O5>FE5NO>G45NI>GC5NC>H25N=>H@5N7>IH5MK>K25M@>L=5M6>MH5LL>O45LB?0?5L8?1J5KN?365KC?4A5K9?5M5K0?6M5JF?7N5J5?9G5HN?;@5GI?<45G<?<;5G9?<C5G7?<J5G4?=25G2?=:5FO?=A5FL?=I5FI?>15FF?>85FD?>@5FA?>H5F?
2186=;81A6>48176>L80L6?E80A6@=8076A57OK6AN7OA6BF7O66C@7O>6D=7O66E;7ND6F97N16G67M>6H47LL6I17L>6IN7L76JK7L16KG7KK6L@7KK6M77KN6MN7L16NM7LA6OO7M770N7MF71C7MA7277M=72L7M873E7M474F7M375G7M176H7M077I7LM78J7LL79C7M57::7MB7;07MO7;F7N<7<=7NI7=N7NA7?A7N07A;7M:7AE7M:7AO7M:7B87M:7BA7M:7BJ7M:7C47M:7CF7M97D<7M87E07M77EE7M77F;7M77G07M57GD7M47H:7M37I07M17II7M07JH7LO7KG7LO7LG7M17MG7M37NF7M57O?7M680E7LC81F7LE82G7LG83H7LI84H7LJ85J7LM86J7LN87J7LO88I7M08987LO8927M088L7M088F7M188B7M289=7M38:97M58;07LH8;F7L:8<<7KL8<O7KH8=A7KK8>27KM8>D7L18?87L:8?K7LD8@?7LM8@O7LM8A>7LL8AM7LJ8B<7LI8C57LC8DD7L18F27KA8GA7JO8I07J?8I@7IF8IL7HO8JA7I48K67I98KL7I?8LA7ID8M77II8ML7IO8NB7J48NN7IO8NF7I@8NM7IL8O87J59087J19197IM9297II93:7IE94<7IA95D7I<96K7I69837I299>7I39:H7I39<37I49==7I59>G7I69?47I;9?B7IA9@07IF9@>7IL9@L7J29A:7J79AH7J=9B67JB9BD7JH9C<7JM9D67K29DO7K89EN7K=9F=7K;9G>7JG9H;7J59I37IH9IK7I:9JC7HM9K<7H@9L67H39M47GF9N27G:9O07FN:017FK:127FE:247F>:317F4:3@7E6:4?7DD:5>7D3:6=7CA:7<7C0:8<7B>:9;7AO::;7AB:;:7A4:<:7@F:=:7@8:>97?J:?27?@:?>7?::?J7?6:@67?1:@E7>L:A:7>G:AN7>A:BC7><:D67=::DL7<J:E>7<>:EO7<2:F@7;F:G17;9:GC7:N:HF7;4:II7;;:JL7;A:L07;H:M37;N:N67<5:O97<;;0=7<B;1@7<H;2D7<N;3L7=8;5;7=D;6I7>2;887>?;9G7>M;:E7>@;;@7>1;<;7=A;=67=1;>17<A;>L7<1;?G7;A;@B7;1;A>7:@;BD79I;CB79:;D8794;DO78N;EE78H;F<78C;G278=;GM783;HO77E;IO777;K076I;L176;;M275M;N375?;O4751<02748<0O73@<1N72G<0@745<0A71<<156OI<3=71@<5D738<7674A<8174N<8K74:<9D73><:=72C<;671H<;O70M<<H701<=A6O7<>:6N;<?36M@<?L6LE<@I6L7<B56ME<CA6O2<DB6OI<E36O;<ED6NL<F56N><FJ6ML<GC6M:<H<6LH<I46L6<IM6KD<JF6K1<KG6J><LK6IJ<MO6I7<O36HC=076H0=1;6G<=2?6FH=3?6F@=4:6F:=5?6F7=6B6F3=7F6F0=8J6EL=9E6EE=:76E8=:O6E0=;N6DK=<M6DH==L6DC=>K6D?=?K6D;=@I6D6=AH6D2=BH6CN=CM6CA=E@6BE=G86B:=I56B==J:6B:=J=6A>=J>6@>=K:6@==KB6@<=L36??=LF6>==M76>0=MG6=D=N86=8=NH6<L=O96<@=OI6<3>096;G>0K6;9>1H6:?>2D69C>3A68I>4>67N>5:673>68668>7465=>8164B>8K64J>9F656>:@65B>;I65H>=;65J>>L65I>?N653>@N64:>AM63@>BL62F>CK61L>DJ61<>E?612>EM60L>F<60F>FJ60?>G8609>GG602>H65OL>HD5OE>I25O?>IM5O7>K95NM>LE5NC>N05N9>O;5MN?0G5ME?225M;?3>5M1?4I5LF?5L5L<?6O5KK?8G5JE?:@5I>?<05H??<75H<?<?5H9?<G5H6?<N5H4?=65H1?=>5GN?=E5GL?=M5GI?>55GF?><5GC?>D5GA?>K5G>
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
2166;N8486<G83M6=?83B6>78386?082L6?H82B6@@8276A981M6B181B6BJ8176CB8146D@80K6E>8086F;7OE6G87O=6H57O66I17O06IN7NJ6JK7NC6KG7N=6LC7N96M:7N;6N27N>6NI7NA6O@7ND7077NF7147O471M7O872F7O473F7O274H7O175H7NO76J7NM77J7NL78K7NJ79K7NK7:A7O87;77OE7=27O77>;7O07?:7NO7@77NL7@O7NI7A97NI7AB7NI7AK7NI7B57NI7B>7NI7C27NI7CF7NH7D<7NG7E17NG7EF7NF7F;7NF7G07NE7GE7ND7H:7NC7I07NF7J27NG7K27NI7L27NK7M27NL7N27NO7NN7O07OG7O180?7O281D7N?82M7MM83O7N084O7N18617N38717N58827N788N7N98937N<8987N>89>7NA89C7NC89E7NF89D7NF8:<7N<8;27MM8;H7M?8<>7M08=37LG8=D7LJ8>67LL8>G7LN8?97M08?K7M78@?7M@8A37MI8AB7MI8B17MG8B@7MF8C;7M?8DI7LM8F87L=8GG7KL8I57K;8JD7JJ8K17IO8KE7J38L;7J98M07J>8ME7JC8MJ7J48N27J?8N97JK8N@7K68NH7KA8NO7KM8OF7KN90G7KK91G7KG92H7KB93H7K>94I7K;95I7K796I7K397J7JO98J7JK99N7JG9;67JC9<A7JD9=;7JH9=I7JM9>77K39>I7K89?B7K>9@<7KC9A57KH9AN7KM9BH7L29CA7L79D:7L=9E37LA9EM7LG9G@7LN9FN7M=9GL7LJ9HN7L69IN7KA9K07JM9L07J89LI7IJ9MB7I<9N:7HO9O27HA9OE7H6:0A7GM:1=7GD:287G;:347G1:3J7FH:4I7FB:5H7F5:6H7EG:7G7E9:8G7DK:9G7D=::F7D0:;F7CB:<F7C4:=E7BF:>E7B8:??7AM:?K7AH:@67AC:@B7A?:@O7A::A;7A5:AG7A0:B67@F:CL7?>:D@7?0:E17>D:F07>E:G37>L:H67?2:I97?9:J=7??:K@7?F:LC7?L:MF7@3:NJ7@9:OM7@?;117@E;247@L;377A2;4:7A8;5>7A?;6M7AL;8;7B:;977AK;:27A;;:M7@K;;H7@;;<C7?K;=>7?;;>97>K;?47>;;?O7=K;@J7=;;AD7<K;BG7<6;CD7;H;D;7;C;E27;<;EH7;6;FA7:O;GB7:@;HC7:3;IC79D;JD797;KE78H;LF78:;MG77L;NG77>;OG770<0E76B<18765<1I769<4976L<1A74:;NG71H;LC6OD<4<73C<9H704<9H70M<9H71F<9J72;<:C71@<;<70E<<56OJ<<N6NN<=G6N3<>@6M8<?96L=<@66KN<AB6M<<BN6NH<D9706<EB71<<F370M<FC70?<G4701<GE6OB<H66O4<HL6NB<IE6N0<J>6M><K76LK<L<6L7<M?6KD<ND6K1<OG6J==0L6II=1O6I5=336H@=486H3=5:6H2=6:6GN=7:6GJ=886GF=986GB=:76G==;56G:=<56G5==46G1=>36FM=?26FH=@16FD=A06F@=AO6F<=BN6F7=CM6F4=E86EA=FK6DD=H=6CH=IO6BL=K26AF=KJ6A4=LA6@C=M76?G=MH6?0=N96>D=NI6>8=O:6=L=OJ6=?>0:6=3>0J6<G>1;6<;>1K6;N>2<6;B>2L6;5>3<6:I>3M6:=>4@69M>5=693>6:687>7667=>8366B>8O668>9I66C>:D66O>;>67;><?67D>>067C>?A67A>A067<>AO66B>BN65I>CM650>DL64A>EJ644>FG63G>G663@>GD63:>H3633>HA62M>HO62F>I>62>>IL625>J:61L>K061F>KO61C>M461<>N?612>OK60I?1660>?2B604?3L5OJ?505O7?6H5N1?8A5LK?::5KD?<05J@?<75J=?<?5J:?<F5J8?<N5J5?=65J2?=>5J0?=E5IM?=M5IK?>45IG?><5ID?>D5IB?>K5I???65I8
2156;885C6<08586<I84N6=A84C6>98486?283M6?J83B6@B8386A;82M6B382B6BK8276CD81O6DA81D6E?8136F;80L6G880F6H580@6I180:6IN8036JK7OM6KG7OG6LD7O@6M<7OB6N47OE6NK7OH6OA7OJ7097OM71080071G80372F80373G80174H7OO75I7ON76J7OL77K7OK78L7OI79M7OH7:D8037<47OK7=37OI7>27OH7?17OG7?O7OE7@I7OB7A27OB7A;7OB7AE7OA7AN7OA7B=7OA7C27O@7CG7O?7D<7O?7E17O?7EF7O>7F;7O=7G07O<7GE7O=7H;7OA7I17OE7J07OG7K77OF7L77OH7M77OJ7N77OL7O37ON7OK7ON80C80081<80182L7NL8427NB8537NE8647NF8757NI8867NJ88?7NL88D7NO88I7O288N7O58947O789;7O889J7O48:=7NN8;37N?8;I7N18<?7MC8=57M78=F7M98>77M;8>I7M=8?:7M@8?L7MB8@?7MI8A37N38AD7N88B37N68BB7N48C=7MM8DL7M<8F;7LK8GJ7L:8I87KI8JG7K88KD7JB8L87JB8LM7JH8MB7JM8MC7JI8MK7K58N37K@8N:7KK8NA7L78NH7LB8O07LM8OM7LK90N7LG91N7LD92O7L?93O7L;9507L79607L49717L09817KL9917KH9:27KD9;27K@9<37K<9<L7KA9=F7KF9>?7KK9?87L09@17L59@K7L:9AD7L@9B=7LD9C67LJ9D07LO9DI7M49EB7M99F>7M>9H37MG9GA7N69H37ML9I57M79J57LC9K77KN9L87K:9M97JE9N97J29O17ID9OI7I6:097HM:147HC:207H::2E7H1:357GG:3O7GA:4K7G;:5N7G;:6N7FN:7M7F@:8M7F2:9L7EE::L7E6:;L7DI:<K7D;:=K7CM:>K7C?:?E7C4:@17BO:@<7BJ:@I7BE:A57BA:AA7B;:AM7B7:BH7A<:CH7@?:DD7@>:EH7@D:FK7@K:GO7A1:I27A8:J57A>:K87AD:L<7AK:M?7B1:NB7B8:OE7B>;0I7BD;1L7BJ;2O7C1;427C7;567C>;697CD;7E7D0;8@7C@;9;7C0;:67B@;;17B0;;L7A@;<G7A0;=B7@@;>=7@0;?87?@;@37?0;@N7>@;AH7>1;BH7==;CF7<O;D=7<I;E37<C;EJ7<=;FK7;O;GL7;@;HM7;3;IN7:D;JO7:6;KO79H;M079:;MN78L;NL78>;OK77O<0I77B<19775<0C76O<1A776<14760;N573>;NJ71C<5D73F<;06M4<;06MN<;06NH<;06OA<:O70;<;?703<<86O8<=16N<<=J6MB<>D6LF<?<6KK<@B6L@<AN6MM<C:6O;<DF70G<F2725<FB71G<G3719<GD70J<H570;<HE6OM<I66O><IM6NM<JF6N;<KB6MH<LF6M4<MJ6LA<NN6KM=016K:=156JD=276IN=3<6I==4B6I2=5F6HL=6H6HL=7K6HK=8K6HI=9J6HD=:I6H@=;I6H<=<H6H8==G6H3=>F6H0=?E6GK=@D6GG=AC6GC=BB6G?=CA6G:=D@6G6=EM6FA=G@6ED=I26DH=J96CE=K;6B?=L=6A9=M?6@3=N76?==NI6>N=O:6>B=OJ6>5>0;6=I>0K6==>1;6=1>1L6<D>2<6<8>2L6;K>3=6;?>3N6;3>4>6:G>4N6:;>5>69N>5O69B>6I68J>7F67O>8C674>9>675>:867A>;267L>;M688><M68A>>:68@>?L68=>A=68<>B@67G>C?66M>D>66<>E<65M>F;65?>G9652>GJ64J>H964C>HG64=>I5647>ID63M>J263D>J?63;>JM633>KC62L>LB62I>MA62F>NG62??02625?1>61L?2I61A?4160M?5I5OG?7B5NA?9;5M:?;35L4?<35K@?<;5K=?<C5K9?<J5K7?=25K4?=95K2?=A5JO?=I5JM?>15JJ?>85JG?>@5JD?>G5JA??65J6??M5IB
2146:186L6:L86A6;F8676<A85L6=;85B6>58586?084M6?J84B6@D8486A=83M6B583B6BM8386CF82M6DB82B6E?82<6F;8266G88206H581I6I181C6IN81=6JK8166KG8106LD80J6M>80H6N580K6NL80N6OC81170;81471281771K81672K80N73L80M74L80K75M80K76M80I77N80H78N80G79O80E7:L80D7;K80B7<J80A7=J80@7>I80>7?H80=7@B8097@L8097A58097A>8097AG8097B=8087C28087CG8077D<8077E18067EF8057F;8057G18087GG80<7H=80A7I380D7II80H7K080H7L<80F7M<80H7N<80J7O780K80080L80H80N81@80N82@80C8457O88567O88677O:8787O<87J7O>8807O@8857OC88:7OE88E7OE8947O@89C7O<8:27O88:A7O68;47O28;J7NC8<@7N68=67MG8=H7MH8>97MJ8>K7MM8?<7MO8?N7N18@?7N38A37N<8AG7NF8B57NE8BD7NB8C@7N;8DO7MJ8F=7M98GL7LH8I;7L78JJ7KF8L87K58LJ7K28M67K38M=7K>8MD7KJ8ML7L58N37LA8N;7LL8NB7M78NI7MC8O47ML9047MH9157MD9257M@9367M<9467M89577M49677M09787LM9887LI9997LE9:97LA9;97L:9<87L29=87L89>47L>9>M7LC9?G7LG9@@7LM9A97M29B37M79BL7M<9CE7MA9D?7MF9E87ML9F17N19G87N79HG7N@9H57NO9H:7NN9I<7N99J<7ME9K>7M09L?7L<9M@7KH9NA7K29OB7J>:0@7IL:0M7IC:1A7I::217I0:2@7HF:3:7H@:447H9:4O7H2:607H0:727H0:837GG:937G9::27FK:;27F>:<17EO:=17EB:>17E4:?17DF:?K7D;:@77D6:@C7D1:@O7CM:A;7CG:AG7CC:B?7BL:C:7B2:D:7A@:E=7AC:F@7AK:GD7B1:HG7B9:IK7BA:JN7BG:L17BO:M57C7:N87C=:O;7CE;0?7CM;1B7D3;2F7D;;3I7DB;4L7DI;607E1;6N7E;;7G7E3;8D7DE;9?7D5;::7CE;;57C5;<07BE;<K7B5;=F7AE;>A7A5;?;7@E;@77@6;A17?F;AM7?6;BJ7>D;CG7>6;D>7>0;E47=J;F57=<;G67<N;H67<@;I77<2;J87;D;K77;6;L57:H;M47::;N279K;O079=;OO78O<0M78A<19785<0D77O;OO77H<0@77F;NJ75=<1>73N<4172><9F6N?<;76LF<<86KI<<76LB<<76M<<<76N5<<<6NF<=56MK<=M6M0<>G6L4<?C6KE<@O6M2<B:6N?<CF6OM<E2719<F>72G<G272@<GC722<H471C<HE715<I570G<IF708<J76OJ<JN6O8<KL6NE<M06N1<N46M><O76LH=0:6L2=1=6K<=2@6JG=3F6J<=4M6J2=626IH=756IG=876IF=996IE=:;6IE=;=6IC=<;6I?==:6I:=>:6I6=?96I2=@86HN=A76HI=B66HF=C56HA=D46H==E36H9=FB6G@=H56FD=IB6ED=JB6D>=KD6C8=LF6B2=MG6A0=NC6@@=O66@0=OI6??>0;6>O>0N6>?>1A6=O>236=>>2F6<N>386<>>3L6;N>4>6;=>4O6;1>5?6:D>606:8>6@69L>7069@>7@693>8668A>9267G>9M681>:H68=>;C68J><?696>=?69>>>K69<>@769:>AH699>C168K>CO685>DN67G>EM679>FK66K>GJ66=>H>664>HM65M>I;65F>II65<>J7654>JE64K>K364B>KA649>L7643>M5640>N563M>O363J?0:63B?1E638?3262C?4J61=?6B606?8;5O1?:45MJ?;L5LD?<75L??<?5L<?<F5L9?<N5L6?=55L4?==5L1?=E5KN?=M5KL?>45KI?><5KF?>C5KC??65K3??M5J@?@B5IL
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
21266A8:767;89M68589B69089869J88M6:F88C6;E8896<D87N6=B87D6>A8796??86O6@>86E6A=86:6B;8606C:85F6DD85;6EO8506FG8576G>85@6H585E6I185>6IN8586JK8526KG84L6LD84E6MA84?6N:84>6O184B6OH84E70@84H71>84@72?84773>83N74>83E75?83<76?83977@83978@83779A8367:68337:L82O7;B82L7<882I7<O82F7=O82E7>N82C7?K82B7@982A7@B82A7A382@7AH82@7B=82?7C282?7CH82B7D>82F7E482J7EJ82N7F@8337G68367GL83;7HB83?7I883B7IN83G7JD83K7K:83O7LD83M7MH8427NL8477OI84580E84281B84082>83N83;83K84783>85782C86881H87880M88080>88?80988N80589=80189L7ON8:;7OJ8:J7OE8;97OB8;H7O>8<67OA8<C7OL8=97O>8=M7O68>?7O88?17O;8?B7O=8@37O?8@D7OB8A67OD8AH7OF8B:7OK8BK7ON8CH7OE8E77O58FF7NC8H57N38I77MI8J:7MA8K<7M98L>7M08M77N28M@7N@8MG7NL8MN7O78N67OB8N>7ON8NE8098NL80D8OJ80B90J80>91J80;92K80793K80294L7ON95L7OK96M7OG97M7O<98L7NN99M7N@9:L7N29;J7MM9<E7N99=A7NC9>C7NG9?F7NL9@H7NO9AK7O49BF7O89C@7O>9D97OC9E27OH9EK7OM9FE8029GH8089IG80A9JB80K9J081:9I?81I9J181>9JN80D9KJ7OK9LG7O19MD7N79NA7M=9O>7LD:027KN:0B7KD:197K<:247K5:2N7JO:3I7JH:4D7JA:5>7J::697J3:787J1:8;7J1:9>7J1::@7J1:;C7J1:<C7IE:=B7I6:>B7HI:?B7H;:@=7GO:@I7GJ:A57GF:AD7G=:B?7FC:C@7F3:DA7EC:EB7E3:FB7DC:GD7D6:HG7D=:IJ7DD:JM7DL:L17E3:M47E::N87EB:O;7EH;0?7F0;1B7F8;2D7F@;3:7FN;417G=;4H7GK;5?7GL;667GF;6N7G@;7F7G9;8=7G3;957FM;9N7FE;:K7F7;;H7EH;<E7E9;=C7DK;>@7D=;?=7CN;@:7C@;A77C1;B57BB;C27B4;D07AE;DN7A8;EM7@I;FK7@;;GI7?M;HH7??;IF7?0;JE7>C;KC7>4;LA7=F;M@7=8;N>7<J;O<7<<<0:7;N<197;?<1:7;6<0E7;0<007:I<0M7:0<32792<5?793<75784<8C761<9H73J<:D72G<;@71C<<;70@<=76O=<>26N9<>G6MA<?16M6<?>6LO<@I6ME<B36NH<C@705<DK71B<F772O<GC74=<HB74M<I274><ID740<J473B<JE733<K572E<KF726<L771H<LG71:<ME70B<NL6OF<OO6O0=126N;=286N1=3?6ME=4E6M:=5K6M0=726LD=886L9=9;6L7=:=6L6=;?6L6=<A6L5==D6L3=>F6L3=?H6L2=@J6L1=AL6L0=BN6L0=CN6KM=DN6KI=EM6KE=G06K;=H;6J;=I<6I5=J=6GO=K>6FI=L?6EC=M>6DH=N:6D8=O66CH>026C8>0L6BH>1?6B8>226AG>2D6A7>376@G>3J6@7>4<6?F>4O6?6>5A6>F>656>6>6G6=E>7:6=5>7L6<D>8?6<5>926;D>9J6;8>:F6;0>;A6;5><=6;;>=86;B>>36;N>?76<4>@C6<1>B06<3>C=6<7>DI6<2>F76;2>GE6:2>HE69B>IB694>II68L>J168D>JC68N>K6699>KD690>L268G>L@68>>LN686>M<67M>N067F>NO67C>ON67@?0M67<?2566<?3D659?5=643?7662L?8N61F?:F60@?<35OB?<;5O@?<B5O=?<J5O:?=15O7?=95O4?=A5O2?=I5NO?>05NM?>A5N???65MK??M5M8?@@5M0?A15LO?AB5LO?B45LO
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
21164B8=965@8<N66?8<D67>8<968<8;O69;8;E6::8;:6;88;06<78:F6=68:;6>48:16?389G6@289<6A08926AO88H6C088=6D;8826EF87G6G087=6H;8716I58776IK87@6JA87H6K88806KN8896LD88B6MA88;6N>8856O688670188371287J72187A73287874286O75286F76286=77286478285J78O85F79E85C7:;85@7;185<7;G8597<<8567=28537=H8507>>84M7?384I7?I84H7@?84L7A58507AJ8527B?8547C58587CJ85:7D?85<7E485?7EI85A7F?85D7G485G7GI85I7H>85L7I385N7IH8617J=8647K38667KO8687M48697N08657NM8627OF85G80>85=81585281N84G82F84<83=84284683G84N83<85E83186>82F87582<87M82188F81F89=81;8:68118:N80F8;F80<8<@8058=97OM8=G7OK8>08018><80>8?B80K8@480M8@F8108A88128AJ8148B<8178BL8178C;8178CI8178DA8148ED80K8FF80C8GH80:8HJ8018IM7OI8JO7OA8L17O88LL8018MC81=8N28298N982D8N@82O8NH83;8O?83<90?83991@83592@83193@82L94@82>95@82096@81A97@81398@80E99@8079:@7OH9;=7OH9<78059=280A9=L80L9>M8129?O8169A181:9B481>9C681C9D981G9E;81K9F=81O9G88249H782:9J782B9L682J9L>82I9KN82F9K=82C9KC82:9KK8209L381F9LJ80N9MD8039N>7O89O;7N@:087N7:0N7ML:1H7MD:2C7M=:3=7M7:487M0:537LI:5N7LB:6H7L;:7B7L4:8A7L1:9D7L1::G7L1:;I7L1:<L7L1:=N7L1:?17L1:@37L0:@O7KD:AF7K6:BF7JF:CG7J5:DH7IF:EI7I6:FI7HE:GJ7H5:HK7GE:IL7G4:JM7FO:L17G7:M47G>:N87GE:O07H3:OF7HA;0=7I0;147I>;1J7IL;2A7J;;387JD;407J=;4H7J7;5?7J1;677IJ;6O7ID;7F7I>;8>7I7;967I1;9M7HK;:A7HE;;A7H@;<@7H3;=<7GC;><7G3;?;7FC;@;7F4;A:7ED;B:7E4;C97DE;D87D5;E77CJ;F67C@;G67C5;H57BJ;I47B>;J27AO;K17AB;KO7A3;LM7@E;MK7@7;NJ7?I;OH7?:<0F7>M<1E7>><1;7>6<0F7>0<287=0<487;K<697:F<8C7:4<:O7;<<<07;K<=27<5<>474J<?274><?>744<?I73K<@373@<@>736<@I72L<A372B<B7732<CA73H<DK74><F5755<G@75K<HI76A<J1779<JC76J<K6769<KI75H<L<757<LO74G<MB745<N573E<NH733<O=72A=0I71A=1E71:=2A713=3=70M=4>70D=5D708=6K6ON=816OB=976O7=:=6NL=;A6NH=<C6NF==F6NE=>H6NE=?J6ND=@L6NC=AN6NB=C06NB=D36NA=E46N@=F66N?=G86N1=H96MB=I;6LG=J;6KA=K=6J;=L=6I5=M;6H9=N46GM=NM6G@=OJ6G0>0E6FA>1A6F1>2=6EA>356E0>3H6D@>4;6CO>4M6C@>5@6BO>636B?>6E6AN>786A>>7J6@N>8>6@>>906?M>9H6?A>:D6?6>;?6>L><;6>E>=66>I>>16>O>>M6?4>?M6?9>A86?;>BC6?>>CN6?@>E96?C>FE6?5>GE6>3>HE6=0>ID6;O>J96;:>JN6:E>K>6:3>KK6:3>L=6:=>M06:H>MC6;3>N66;>>NI6;H>O76;@>OJ6;9?0L6:L?2469L?3<68J?4D67J?6066H?7H65B?9A64<?;:635?<762C?<>62@?<F62>?<M62;?=5628?=<625?=K61K?>A617??560J??G60I?@860I?@I60I?A;60H?AM60G?B>60F
21164>8>565<8=K66;8=A6798=76888<L6978<B6:68<86;48;M6<38;C6=28;96>08:N6>O8:D6?M8::6@L89O6AK89E6BL89;6D788O6EC88E6FM8896H887O6I@87F6J687O6JL8886KC88@6L988H6LO8916MF8996N>89>6O889<70889371888J72888A73888874987M75:87B76;87677<86K78<86@79686:79L8677:B8647;88607;M85M7<C85J7=885G7=O85D7>D85A7?985?7?M85A7@A85D7A585F7AI85H7B=85J7C185M7CF8607D;8627E08647EE8677F;86:7G086<7GE86?7H:86A7I086C7IE86G7J:86I7JO86K7KK86M7LJ86L7MF86F7N>86;7O68617ON85E80F85;81>85082684E82N84:83F83O84>83E85783<86183586J82M87D82F88>82?8988278:18208:K81H8;E81A8<?81:8=88128>280K8>980L8>@80M8>G80N8?A8128@M81>8B181D8B@81D8BO81D8C>81D8CL81D8D:81D8DI81D8E781D8F681=8G88158H:80L8I<80D8J?80;8KA8038LA8068M881D8MN8308N:83E8NA8408NI84;8OF84990G84591G84292G83H93G83:94G82L95G82=96F81O97G81@98F81299G80D9:F8069;A80A9<<80L9=68189>181D9?081K9@281O9A48239B78279C982<9D<82@9E>82D9F@82H9GB82L9I682A9K>81F9KO8259L@82C9LB82K9L382G9L;82>9LC8249LJ81J9M381@9MI80I9NC7ON:007OE:0F7O9:1=7NM:227NB:2H7N6:3C7MO:4=7MH:587MA:637M::6M7M3:7G7LM:8D7LG:9F7LG::I7LG:;K7LG:<N7LG:>07LG:?37LG:@57LG:A77LF:B97L6:C=7KG:D?7K7:E@7JF:FA7J6:GB7IF:HB7I6:IB7HF:JC7H5:KE7GJ:LH7H2:MB7H?:N97HM:NO7I<:OF7IJ;0=7J9;137JG;1I7K6;2A7KA;387K;;407K4;4H7JN;5?7JH;677JA;6O7J;;7F7J5;8=7IN;907II;9D7IC;:?7I=;;H7I9;<L7I0;=G7H@;>A7GN;?>7G>;@=7FN;A=7F>;B<7EO;C;7E?;D;7DO;E:7DD;F97D9;G87CO;H77CD;I77C:;J67BO;K47BA;L37B3;M17AE;MO7A6;NN7@I;OL7@:<0J7?L<1I7?=<1<7?6<1;7>H<3;7=C<5;7<><7<7;9<8K7:I<:?7<D<;?7=1<=37<?<>076K<>M768<?K760<@;75G<@F75=<A1753<A;74I<AF74><BF74K<D175B<E;768<FE76O<GO77E<I8789<J9785<K177@<KI76L<LA766<M975B<ML751<N@74A<O373O<OE73>=0A72H=1@72<=2;725=3871N=4371G=5071@=5N718=7570M=8;70A=9B707=:H6OK=;M6OB=<O6OA=>26OA=?46O@=@66O?=A86O>=B:6O>=C<6O==D?6O<=E@6O;=FB6NN=GC6N?=HE6MO=IF6M@=JG6L:=KH6K4=LI6IN=ME6I8=N>6HL=O76H?>006H3>0L6GC>1H6G3>2D6FC>3@6F3>446EC>4G6E2>596DB>5L6D2>6?6CB>716C1>7D6BA>876B1>8I6AA>9?6A2>::6@H>;56@=><16@2><L6?L>=H6@0>>C6@5>??6@;>@F6@>>B16@A>C<6@C>DF6@F>EM6@D>FE6@2>GF6?0>HF6=M>IC6<N>J86<9>JM6;D>KB6:N>L66:D>LJ6:L>M=6;5>N06;?>NC6;J>O66<5>OI6<??0?6<<?1G6;;?2O6:;?4669:?5>68:?6I678?8A662?::64K?<363E?<:63B?<B63@?<I63=?=163:?=9637?=K62H?>A625??2622??D622?@6621?@G621?A8620?AJ61O?B;61O?BM61O
2116498?36588>H6678>>6758>46848=I6938=?6:18=56;08<J6;N8<@6<M8<66=L8;L6>K8;A6?I8;76@H8:M6AG8:B6BI8:76D489M6E?89A6FJ8976H588L6I@88A6JA88?6K788G6KN88O6LD8986M:89A6N189I6NG8:16OB8:470B89H71C89=72D89273E88F74F88;75G87O76H87D77I87978J86M79@86J7:586G7:K86D7;A86@7<786=7<L86:7=B8687>68667>K8657??8677@386:7@G86<7A;86>7B086@7BD86C7C786E7CK86G7D@86J7E486L7EG86N7F<8707G08737GD8757H88777HL8797IA87<7J687>7JL87A7KE87C7L>8797M686N7MN86C7NH86<7OB86580;85N81585F81O85>82I85883C85084<84H85684A86084:86I84287C83K88=83D89683<8:08358:J82M8;D82F8<>82?8=78278>18208>D81L8>K81M8?281M8?;81N8@781O8AB82;8BF82@8C58238CM8208D=8208DK8208E:8218EH8218F78218FH8208GK81G8HM81?8IO8168K180N8L480E8LL81I8MC8368N:84C8NB8518NM85:8OM85690N85291M84D92N84593M83G94M83995M82K96M82=97M81N98M81@99M8129:K80L9;F8189<@81D9=:8209>582<9?382D9@582H9A782L9B:8309C<8349D?8399EA83<9FG83:9HH81J9J<80I9JM8179K>81F9KO8249L@82C9LJ82L9M382B9M:8299MB81O9MJ81E9N281;9NI80E:0?80G:1580;:1K7OO:2A7OD:377O8:3M7NL:4C7N@:5=7N9:687N3:727ML:7M7ME:8G7M>:9H7M=::K7M=:;M7M=:=07M=:>27M=:?57M=:@67M@:A77M>:B47M1:C67LB:D:7L3:E>7KC:FA7K3:GE7JC:HH7J4:IL7ID:JO7I4:L27HL:LK7I9:MA7IH:N87J6:NN7JE:OE7K3;0<7KB;127L1;1I7L>;2A7L7;387L1;407KK;4H7KD;5?7K>;677K8;6M7K2;7@7JL;837JG;8G7JA;9?7J<;:F7J6;;N7J1;=67IL;>27I<;>L7HL;?G7H:;@A7GJ;A?7G:;B?7FJ;C>7F:;D>7EK;E=7E?;F<7E3;G;7DI;H:7D>;I97D3;J87CI;K87C>;L77C2;M57BD;N37B6;O17AH<007A9<0N7@L<1M7@=<1<7@6<2>7?<<4?7>7<6?7=1<7@7<=<8G7<J<9N7=K<;97=J<=47<7<=M78M<>G782<?F77J<@D77B<A8779<AC76O<AM76E<B876;<C676E<D@77<<EJ782<G478H<H<798<I<794<J=78N<K678<<KO77H<LG773<M?76><N775J<NO755<OG74A=0@73K=1:73>=26736=3272O=3N72H=4J72B=5F72:=6B723=7?71L=8E71A=9L716=;270J=<870@==;70==>>70<=?@70;=@B70:=AD70:=BF709=CH707=DK707=EL6OJ=FM6O;=GO6NL=I06N==J16MN=K36M4=L46KN=M56JH=MO6J7=NH6IK=OA6I>>0:6I2>126HF>1N6H6>2J6GF>3F6G6>4@6FF>526F5>5E6EE>686E5>6J6DE>7=6D4>806CD>8C6C3>956BC>:06B9>:L6AN>;G6AD><C6A9>=>6A2>>96A7>?56A<>@46AA>A?6AD>BI6AF>D46AI>E:6AG>F26A7>FJ6@G>GF6?N>HF6>K>IC6=M>J86=8>JL6<C>K>6<<>KO6<F>LC6<N>M76=6>MK6=>>N?6=F>O86=D?0:6=5?1;6<E?2A6;K?3I6:J?5169I?6868I?7B67G?9;66B?;365;?<664D?<>64B?<F64??<M64<?=564:?=K63F?>?63;??063;??A63:?@3639?@E639?A6638?AG637?B9637?BJ637
2116458@06548?F6638?;6718?16808>G68N8><69M8>26:L8=H6;J8==6<I8=36=H8<I6>G8<>6?E8<46@D8;J6AB8;@6BF8;56D18:J6E<8:?6FG8:46H289I6I<89>6JH8936KB8976L989?6LO89H6ME8:06N<8:96O28:A6OK8:F7128:972289M73289@74289575388H76488=77588278687F79387>79I87;7:?8787;48757;J8727<?8707=486M7=I86K7>>86K7?186M7?F8707@:8727@N8747AB8767B68797BJ87:7C>87=7D287?7DF87A7E:87C7EN87F7FB87H7G687J7GJ87L7H>87O7I28817IF8837J:8877JM88<7K@88A7L98877M48807MM87I7NG87A7OA87:80:87381486K81N86C82G86=83A86584;85N85585F85O85?86H85887B85088<84I89584B89O84:8:I8428;B83L8<<83D8=683<8>08358>J82N8?682L8?=82M8?D82O8@282L8@L82L8B78378C<83>8CJ8318D982C8E082@8EI82=8F982=8FH82=8G682=8GE82>8H=82:8I?8218JA81I8KC81@8LA81O8M883<8MN84I8NC8618O486790485N91485@92485193484C94484595483F96483997482J98382<99481M9:381?9:O81D9;J8209<D82<9=?82G9>98339?683<9@883A9A:83E9B=83I9C?83M9DB8429F<82L9H28249I48209J281D9JK81I9K;8269KK82D9L<8329LM83@9M083?9M783<9M?8399MG8359N182F9OA81K:0M81I:1C81=:29811:2O80E:3E80::4;7ON:517OB:5G7O7:6=7NK:777NC:827N<:8L7N6:9J7N2::M7N2:;O7N2:=27N2:>47N3:?67N6:@77N9:A77N6:B47MI:C17M=:D37LN:E77L>:F:7KO:G>7K?:HA7JO:IE7J?:JH7J0:KI7J8:LI7JC:M@7K1:N77K?:NN7KN:OD7L=;0;7LK;117M:;1I7M4;2A7LN;387LH;407LA;4H7L;;5<7L6;5O7L0;6B7KK;767KD;7I7K?;8=7K9;9E7K5;:M7JO;<57JJ;==7JE;>>7J:;?87II;@37I8;@M7HF;AG7H6;BB7GD;C@7G5;D@7FE;E?7F8;F?7EN;G>7EB;H=7E8;I<7DM;J;7DC;K:7D8;L97CM;M97CC;N77C5;O67BG<047B9<127AK<217A=<1A7A5<3B7@0<5B7>J<6:7>:<7H7>A<8D7>K<9@7?0<;;7>9<=37<<<=I7:N<>B79K<?@79D<@?79<<A=794<B578L<B@78A<BJ788<CF78><DO795<F:79K<G@7:7<H@7:3<I@79M<JA79I<K;799<L478D<LL77O<MD77;<N<76F<O4761<OL75==0F74H=1M74;=2K742=3H73J=4E73C=5A73<=6<735=7972N=8572G=9072@=:6725=;=71J=<C71?==H717=>J716=?L716=@N715=B0714=C2713=D5713=E670F=F7707=G96OH=H:6O9=I<6NJ=J=6N;=K>6ML=L?6LH=M@6KB=N96K5=O26JI=OK6J=>0D6J0>1=6ID>256I8>306HI>3M6H9>4I6GI>5>6G8>616FH>6D6F7>766EH>7I6E7>8<6DG>8N6D6>9G6CJ>:B6C@>;=6C5><96BJ>=46B@>=O6B9>>K6B>>?F6BC>@L6BF>B76BJ>CB6BL>DG6BJ>E@6B9>F86AI>G06A9>GH6@I>HG6?I>IB6>L>J76>7>JH6>3>K76>>>KI6>H>L<6>O>M36?3>N56>C>O66>4?086=D?1:6=4?2;6<E?3=6<5?4C6;:?5K6:9?73698?8;687?:4671?;L65K?<:65D?<B65A?<I65>?=5656?=J64C?><64C?>N64C???64B?@164B?@B64A?A364@?AE64@?B764@?BH64??C964>
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
2176878IM6968IC6:58I96;38HO6<18HD6=08H:6=O8H06>N8GE6@18G;6A<8FO6BG8FE6D28F96E=8EO6FH8EC6H38E96I=8DM6JI8DC6L48D86M>8CM6NI8CB7048C771?8BL72J8BA73E8B:74=8B47568AN75N8AH76G8AB77?8A<78;8A979A8AA7::8AG7;48AL7;N8B27<H8B87=B8B>7>88BC7>K8BH7?>8BM7@18C20000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008F>8?O8G38?J8GG8?F8H<8?A8I08?=8IE8?98J:8?48JN8?08L58?98LG8>N8M58>A8MC8>48N28=F8NA8=98NO8<L8O=8<?90>8=491J8>A9378?E94I8?D96:8?C97L8?A98O8?C99K8?D9:G8?G9;C8?I9<?8?J9=;8?M9>78?N9?38@09@78?O9A=8?D9BC8?89CI8>L9E08>@9F58>59FJ8>49G?8>49H48>39HI8>29IB8=N9J=8=J9K88=E9L28=A9LM8=<9MG8=99N98==9OB8=<:108=9:2?8=8:318=9:3C8=;:458=<:4F8==:578=?:5I8=@:6=8=;:778<D:828;M:8M8;7:9F8;4::?8;0:;C8:E:<O8:4:>@89>:@088E:A?87K:BO871:C;871:C<879:C<87C:C9881:CI895:E189H:F?89K:GH89>:HK890:IN88B:K0883:LH87@:N887@:O>87E;1G87L;1;88;;1B87C;1786J;0L86@;0=867;0585N;1<861;2B865;3I868;4O86<;6686?;7<86C;8C86F;9J86F;;286>;<:867;=C861;>K85J;@385C;A;85<;BC855;CK84N;E484G;F:83O;G@83?;HF83E;IJ83I;JO83B;L383:;M8833;N<82L;O@82E<0D82=<1?81A<1A80><1C7O9<1E7N6<1G7M6<1L7M;<4O7L?<6N7L7<8N7KN<9B7KA<:F7K5<;J7JI<<N7J><>17J2<?57IF<@L7I<<A67HO<A97H7<A<7GA<AJ7FN<BO7FD<D57F9<E;7EO<F@7EE<GH7E9<I17DL<J17E2<JL7DL<KF7DE<L@7D?<M;7D9<ME7D0<MJ7CF<N67C<<OA7C0=0M7BD=287B8=3C7AL=4O7AA=6:7A5=7E7@I=907@==:<7@1=;O7?F=<I7?@==H7?:=>G7?3=?F7>L=@D7>E=AC7>==BB7>7=C@7>0=D?7=I=E=7=B=F<7=;=G;7=4=H97<M=I47<?=II7;J=J?7;5=K57:@=KJ79K=L@796=M678A=N478==O3788>02785>11781>2077M>2N77I>3M77=>4L771>5J76E>6I769>7G75M>8F75@>9E754>:D74H>;B74<><A740>=?73D>>>738>?=72L>@@728>A671L>AI71C>BD713>C@70C>D;702>E66OA>F26O1>G06N?>H16MJ>I26M6>J16LG>JN6L;>KK6KO>LG6KC>MD6K7>NH6JD>ON6IM?156I6?276HC?396H1?4:6G??5<6FL?6>6F:?7?6EG?8A6E5?9C6DB?:D6CO?;D6CG?<A6CL?=>6D1?><6D6??96D;??B6DL?@36DK?@E6DK?A66DK?AG6DJ?B96DI?BJ6DI?C<6DH?CM6DH?D?6DH?E06DG?EB6DF
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
21868I8L;69H8L16:G8KF6;E8K<6<D8K26=B8JG6>A8J=6?H8J26A38IG6B=8I<6CH8I16E38HF6F>8H;6GI8H06I48GE6J?8G:6KJ8FO6M58FD6N@8F96OJ8EN7158EC7248E;72M8E573E8DO74=8DI7568DC75N8D=76G8D77848D@7928DG79L8DM7:E8E27;@8E87<98E=7=38EC7=M8EI7>F8EO0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008JC8B=8K78B88KL8B48L@8B08M58AL8N58AO8NH8AF8O78A98OE8@L9038@?90B8@191>8@=92J8AJ9498B995K8B89798B798=8B999B8B;9:?8B=9;<8B?9<88BA9=38BC9>08BE9>O8BB9@58B69A;8AK9BA8A?9CK8AA9E68AG9FA8AM9GL8B39I48AM9J98AB9JO8AA9KG8A>9LA8A99M<8A?9N68AI9O08B39OB8B79OH8B49OO8B2:0N8AO:1@8B1:288AB:328@L:3M8@5:4H8?>:5B8>H:6>8>2:788=;:838<D:8N8;O:9G8;L::@8;H:;98;D:;@8<9:<=8<@:=E8;4:?38:9:@@89>:AL88K:A:8:>:@@8<9:@68=C:A98=?:BJ8=5:D;8<J:EL8<?:G;8<5:HB8;J:II8;=:K08;1:L68:E:M=8:::MM8:9:N68:>:NG89H:O?890;07887:ON87L:OH87D;0O87G;2587K;3<87N;4B882;5I885;6O889;8688<;9<88?;:C88B;;I88F;=088I;>688M;?=890;@C893;AJ894;C288J;D9882;E?87<;FE86D;GL85M;I2855;J884?;K>84E;LC84I;MH84N;NM853<03858<12858<1:845<1B830<1J81L<2280G<2@7OG<3>7O0<4G7ND<6E7N<<8D7N3<:77MI<:=7M:<:M7LJ<<07L?<=47L4<@37JK<A07K:<AJ7KD<AM7JL<B07J6<C67IL<D<7IB<EB7I7<FG7HM<GM7HC<I27H9<J87GO<K>7GD<LG7G7<MH7F><N47EI<N27EF<N27EA<N77E7<N>7DM<OI7DB=1A7D8=3>7D0=5;7CG=787C>=957C6=:K7BN=;B7BG=<87B?=<O7B8==F7B1=>>7AK=?67AH=?N7AC=@F7A?=AA7A:=B@7A3=C>7@L=D=7@E=E<7@>=F;7@7=G97@0=H77?I=I67?B=J07?4=JF7>?=K<7=K=L27=5=LG7<@=ME7<:=ND7<6=OC7<2>0B7;O>1A7;K>2?7;G>3>7;C>4=7;?>5<7;<>6;7;4>797:H>887:<>977:0>:679D>;4797><278K>=178?>>0783>>O77G>@>76D>B6754>C674>>CO740>DJ73?>EF72O>FA72>>G<71M>H871=>I370L>IN70;>JJ6OJ>KE6O:>LF6NF>MD6N7>NA6MK>OA6M;?0H6LE?1O6KN?366K7?4:6JC?5<6J0?6=6I>?7?6HK?8A6H9?9C6GF?:D6G4?;C6G2?<@6G7?==6G<?>:6GA??76GF?@56GK?@=6HD?@O6HC?A@6HC?B26HC?BC6HB?C46HA?CF6HA?D76H@?DI6H??E:6H??EL6H??F=6H>
21868E8M869C8LN6:B8LD6;A8L96<@8KO6=>8KE6>=8K:6?E8JO6A08JD6B:8J96CE8IN6E08IC6F;8I86GF8HM6I18HB6J<8H86KF8GL6M18GB6N=8G66OG8FL7118FA71L8F972E8F373=8EM7458EG74N8EA75F8E<76B8E87828EC78M8EI79G8EO7:A8F47;;8F97<48F?7<N8FE7=H8FK0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008JF8C@8K:8C;8KO8C78LC8C38M88BN8ML8BJ8NJ8BL8O>8BD8OM8B790;8AI90J8A<91:8A292F8B>9448C595F8C49738C49878C599<8C89:=8C:9;98C<9<58C>9=18C@9=M8CA9>O8C;9@68BO9A;8BD9BB8B89CG8AM9E28B19F=8B89GH8B>9I38BD9J>8BJ9KF8BF9L>8BC9M88BL9N28C69NL8C@9OG8CJ:018CJ:088CG:0?8CD:0M8C4:1F8B9:2A8AB:3;8@L:468@5:518??:5L8>H:6G8>2:7A8=;:8<8<D:968<6:9O8<3::H8<0::I8<O:;?8=O:<;8=?:=?8;F:>M8:L:@:8:1:A>8:7:A=8;=:@C8=8:@C8>7:AB8>5:BD8>2:D58=G:EF8=<:G78=2:H>8<F:IE8<::JK8;N:L28;B:LN8;;:MB8;8:MK8;8:N>8:::O389?:OK88F:OD88:;00887;1788:;2>88>;3D88A;4K88E;6188H;7888L;8>88O;9E892;:K895;<2899;=889<;>?89@;?F89E;@L8:1;B38:=;C>89O;D<88O;EC888;FI87@;GO86I;I6862;J<85;;KB84L;LG851;MM856;O285;<0785A<1>85N<1J84O<2283L<2:82G<2B81C<3480D<3O7OK<577O?<6B7O2<8A7NJ<:@7NB<:G7N2<:M7MB<;F7M5<=E7L9<@57K?<A27KN<AN7L=<B37KI<B?7K5<CE7JK<DJ7JA<F07J7<G67IL<H;7IC<IA7I9<JG7HN<KM7HD<M37H9<MN7G8<NG7F2<O67E9<O47EB<O37F1<O:7EB=177E:=347E2=517DI=6N7D@=8J7D8=:>7D0=;47CI=;L7CA=<B7C:==97C4=>07BL=>G7BF=??7BA=@87B>=A07B9=AH7B5=BA7B1=C>7AK=D=7AD=E;7A<=F97A5=G87@O=H77@H=I67@A=J37@8=JH7?C=K>7>M=L47>9=LK7=G=MJ7=C=NI7=?=OH7=;>0G7=8>1F7=4>2E7=0>3D7<L>4B7<H>5A7<E>6@7<A>7?7<8>8>7;L>9<7;?>:;7;3>;97:G><87:;>=779O>>579C>?7792>A077B>BH763>CK75:>DF74I>EA748>F=73H>G8737>H472F>HO726>IJ71F>JF715>KA70D>L<704>M76OC>N86NO>O46NC?076N1?1>6M:?2E6LD?3K6KM?516K7?626JE?746J3?866I@?986HN?:96H;?;96H4?<66H9?=36H>?>06HC?>M6HH??K6HM?@H6I2?@L6IK?A=6IK?AO6IK?BA6IJ?C26II?CC6II?D56IH?DF6IH?E86IH?EI6IG?F;6IF
21969?8MK6:>8MA6;=8M66<<8LL6=:8LA6>98L86?A8KL6@L8KB6B78K66CB8JL6DM8J@6F88J66GC8IJ6HM8I@6J98I46KC8HJ6LN8H>6N98H46OD8GH70J8G>71D8G772=8G27358FL73M8FE74F8F?75>8F:76@8F;77N8FE78H8FK79B8G07:<8G67;58G;7;O8GB7<I8GG0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008JI8DC8K=8D>8L28D:8LF8D68M;8D18MO8CM8ND8CH8O?8CI9048CA90C8C49118BG91?8B992B8C293O8D295A8D096L8D19818D29968D49::8D79;68D99<38D:9<N8D=9=K8D>9>O8D49@68CI9A;8C>9BB8C19CG8BF9DN8B;9F98BB9GD8BH9HO8BN9J:8C59KE8C;9LH8CI9MH8D:9NH8DM9OC8E6:0:8E?:0<8E6:098DG:068D7:108CF:2<8C5:368B6:3J8A3:4@8@6:5:8??:658>H:708>2:7J8=;:8E8<D:9?8<<:9O8<E::08=D::78>C:;?8?4:<98>>:=98<9:>G8;>:?N8;7:@O8;C:A@8<<:@F8>7:A18>K:B08>H:BO8>F:CO8>C:E@8>9:G08=N:H:8=C:I@8=7:JG8<K:KD8<C:L@8<<:M78<8:MD8;K:N68:M:NH89N:O?895:O:88H;0988J;1@88M;2F891;3M894;53898;6:89;;7@89>;8G89A;9M89E;;489N;<;8::;=B8:G;>H8;4;?O8;@;A68;M;B?8;?;CJ8;1;E88:F;F689F;FO88?;H387F;I:86N;J@867;KF85?;LL858;N185=;O785B<0=85J<1D868<2:85J<2B84F<2J83B<3282><3I81B<4C80I<5H809<717OL<8>7OA<:=7O8<;07NJ<;77N:<;=7MJ<>B7L7<@67L4<A37LC<B07M1<B:7LE<BN7L4<D47KK<E:7KA<F?7K6<GE7JL<HJ7JB<J07J8<K57IN<L<7IC<M87I6<MM7H8<NH7G;=157GL=0;7H0=0D7G;=0O7F==2J7F4=4G7EL=6C7EB=8@7E:=:17E2=:H7DK=;?7DD=<67D==<L7D6==C7CO=>:7CG=?17C@=?I7C<=@A7C8=A97C4=B27BO=BJ7BK=CB7BG=D<7BB=E:7B;=F97B4=G87AM=H67AF=I57A?=J47A8=JK7@F=KA7@1=L67?<=M17?0=MO7>L=NN7>H=OM7>D>0L7>@>1K7>=>2J7>9>3I7>5>4H7>1>5G7=N>6F7=J>7D7=F>8C7=;>9B7<O>:@7<C>;?7<7><>7;K>=<7;?>>;7;3>?I7:0>AB78A>C;771>DB763>E>75B>F9751>G474A>H0741>HK73@>IF72O>JB72?>K=71N>L871=>M370M>MO70<>NJ6OK>OG6O;?0M6NG?236N0?3:6M9?4@6LC?5G6KM?6I6K:?7K6JG?8L6J5?9N6IB?:N6I6?;L6I<?<I6I@?=F6IE?>C6IJ??@6IO?@>6J4?A76J=?A;6K4?AM6K3?B>6K2?C06K2?CA6K1?D36K1?DD6K0?E56K0?EG6JO?F86JN?FJ6JN
21969;8NH6::8N>6;98N46<78MI6=68M?6>48M46?>8LJ6@I8L>6B48L46C?8KH6DI8K>6F58K26G@8JH6HJ8J=6J58J26K@8IG6LK8I<6N68I16OA8HF70B8H<71<8H57248H072M8GJ73E8GD74>8G>7568G876>8G=77J8GG78C8GL79=8H27:78H87;08H>7;K8HC7<D8HI0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008JL8EF8K@8EA8L58E=8LI8E98M>8E48N38E08NG8DK8O;8DG9048DF90J8D?9188D191G8CD92>8CG93J8DN95<8DM96F8DM97J8DO98O8E19:48E39;48E69<08E79<L8E:9=J8E:9>O8DN9@68DB9A;8D79BB8CK9CG8C@9DN8C49F48BL9G@8C29HK8C89J58C>9K@8CE9L@8D79M@8DI9N@8E:9O@8EO:078FC:038EH:008E89OM8DH:118D7:2=8CE:3I8C4:4K8B;:5>8A9:628@6:6E8?4:798>2:848=;:8N8<G:978=::988>8:998?7::=8@9:;=8@4:<78?<:=58=1:>?8<7:?@8<C:@A8<O:AB8=<:@J8?5:A>8?>:B=8?<:C<8?::D;8?7:E;8?5:FJ8>K:H68>@:I<8>4:J:8=K:K58=E:L18=>:LK8=7:M<8<>:MM8;?:N?8:A:O489E:O;89:;0B89=;1H89A;2O89D;4589H;5<89K;6B8:7;7I8:D;908;0;:78;=;;>8;I;<D8<6;=K8<B;?28<O;@98=<;AA8=0;BK8<@;D78<3;ED8;G;G28;=;H28:?;HK898;IC881;JD873;KJ86<;M185D;N685D;O;85J<0B865<1I86B<2J86F<3185A<3984><3C839<4=82@<5881G<69813<7B80G<8K80;<::7OO<;97OC<;@7O3<=07MK<?;7L9<@77LH<A47M7<B17MF<B@7MB<C=7M4<DB7LJ<EH7L@<FN7L5<H47KK<I@7K=<JC7K4<KA7JM<L@7JE<M87IO<MM7I1<NI7H9=007H3=1?7HN=297H5=2E7G8=4=7FN=6:7FE=877F<=9D7F5=:;7EM=;27EF=;I7E?=<?7E8==67E1==M7DI=>D7DC=?;7D<=@37D6=@K7D2=AC7CN=B;7CJ=C37CF=CL7CA=DD7C==E<7C9=F97C3=G77BL=H67BE=I57B>=J37B7=JN7AI=KC7A5=L97@?=M67@8=N57@4=O47@0>037?L>117?I>207?E>2O7?A>3N7?=>4M7?:>5L7?6>6K7?2>7I7>N>8I7>J>9G7>?>:F7>3>;D7=G><C7=;>=B7<N>>B7<?>@;7;0>B479@>CM77O>E776O>F176?>FL75O>GG75?>HB74N>I=74>>J873N>K373>>KN72N>LI72>>MD71M>N?71=>O:70M?0470<?1B6O<?2I6NF?3O6MO?566M9?6=6LC?7?6L0?8A6K=?9C6JK?:D6J8?;B6J>?<?6JB?=<6JG?>96JL??66K1?@46K6?A16K:?A?6L0?AJ6L;?B<6L:?BM6L:?C>6L:?D06L9?DB6L8?E36L8?ED6L7?F66L6?FG6L6
2206:68O;6;48O16<38NF6=28N<6>08N26?;8MG6@F8M<6B18M16C<8LF6DF8L;6F18L06G<8KE6HG8K:6J28JO6K=8JD6LH8J96N38IN6O=8IC70;8I:7148I471L8HN72E8HH73=8HB7468H<74N8H776;8H@77E8HI78>8HN7988I47:28I:0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008JO8FI8KC8FD8L88F?8LL8F<8MA8F78N68F38NJ8EN8O>8EI9038EF90J8EC91@8E<91N8DO92=8DA93F8EG9578EI96?8EI97D8EL98I8EN99M8F09;18F39;N8F49<J8F69=J8F39>O8EG9@68E<9A<8E09BB8DD9CH8D99DN8CN9F48CB9G;8C<9HG8CC9J18CI9K88D49L88DE9M98E79N68F19O18G09ON8GN9OJ8F>9OF8EI9OG8E::128DH:2>8D6:3J8CD:568C3:6@8B@:738A>:7G8@;:858?2:8C8=I:8A8>2:8@8>N:8A8?L:958@M::@8AI:;;8A4:<68@;:=38>0:>18=C:?28=O:@38>;:A48>H:@M8@3:AK8@2:BJ8@0:CI8?M:DH8?K:EH8?I:FG8?G:H28?=:I08?4:IL8>M:JF8>H:K@8>C:LB8>3:M58=4:MF8<3:N88;4:NI8:4:OD89M;0J8:3;218:@;388:M;4>8;9;5F8;F;6M8<2;838<?;9:8<L;:A8=8;;H8=E;<N8>1;>58>>;?<8>J;@D8>B;AN8>2;C78=B;DC8=5;F08<I;G>8<>;HL8<3;IM8;8;JF8:1;K?88J;L787C;M486@;N;85L;O@862<0G86?<1N86M<34879<3B86=<3J858<47848<5283><5M82E<6J81M<8381A<9<815<:E80I<<1809<=D7OH<?87O7<@67NB<@B7N0<A67ML<B27N:<BG7N><CL7N3<E17MI<F>7M:<GK7LM<HI7LE<IH7L><JG7L7<KF7KO<LD7KF<M97JH<N17IM<O97I?=0B7I1=1G7IA=3O7I0=4:7H3=607GG=7M7G>=987G7=9N7G0=:E7FH=;<7FA=<37F:=<I7F3==@7EL=>77EE=>N7E>=?D7E7=@<7E0=A47DL=AM7DH=BE7DD=C=7D@=D57D<=DN7D7=EF7D3=F>7CO=G77CJ=H57CC=I47C==J27C6=K07BM=KF7B8=L<7AE=M;7AA=N:7A==O97A9>087A5>167A2>267@N>347@J>437@F>527@B>617@?>707@;>7O7@7>8N7@3>9L7?O>:K7?C>;J7?6><I7>J>=G7>>>?57=>>@N7;N>BF7:>>CM79A>E378E>F777K>G277;>GM76J>HH76:>IC75J>J>75:>K974J>L474:>LO73I>MI739>NE72I>O?729?0;71C?1570@?3>6O;?4E6NE?5L6MO?736M8?866LD?986L2?:96K??;76K@?<46KD?=16KJ?=O6KN?>L6L3??I6L8?@F6L=?AC6LB?AG6MC?B96MB?BK6MB?C<6MB?CM6MA?D?6MA?E06M@?EB6M??F36M??FE6M??G66M>
2216;08ON6;O8OC6<N8O96=M8NO6?88ND6@B8N96AM8MN6C98MC6DC8M86EN8LM6G98LB6HD8L76IO8KL6K:8KA6LE8K66MO8JK6O78J@7038J870L8J271D8IM72=8IG7358I@73N8I:74J8I77698IB0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008K28GK8KF8GG8L;8GB8LO8G>8MD8G:8N98G58NM8G18OA8FL9068FI90K8FD91?8F?9268F992D8EL93B8F<9528FE9698FF97>8FI98B8FK99G8FM9:L8FO9;K8G19<G8G39=J8FL9?08FA9@68F59A<8EI9BB8E>9CH8E39DN8DG9F48D;9G:8D09HC8CM9IM8D39K08DB9L08E59LL8F39MH8G29ND8H19O@8HO9OC8GO9O<8F:9OH8EJ:138E9:2?8DG:3N8D5:5?8CC:718C0:7=8AH:7J8@?:888?5:878??:828@3:7M8@H:7M8A@:9;8BB::>8BH:;98B4:<48A9:<68A::<88A<:=E8@L:?;8@7:@F8@3:A98@I:B88@F:C88@C:D78@A:E68@@:F58@?:G38@>:GJ8@>:H>8@=:I88@7:J38@3:K38?D:L:8>M:M08=N:MA8<O:N28;N:O68;7:OM8;B;148;O;2;8<;;3B8<H;4I8=5;5O8=A;768=N;8=8>:;9D8>G;:J8?4;<18?@;=88?M;>?8@9;?G8@4;A08?D;B:8?4;CB8>C;DO8>6;F<8=K;GJ8=@;I88=5;JE8<I;KI8<1;LA8:J;M:89C;N388<;NK874;OE86=<0L86J<23876<3:87C<41878<49864<4K855<5F84<<6A83B<7<82I<8C82<<9L820<;981I<=H833<?<82A<@M820<A7819<AA80B<AM7ON<BA7OK<C?7O:<DJ7NI<F17N><G07N6<GO7N0<HN7MH<IM7M@<JL7M:<KK7M2<LE7L?<M97KA<NA7K3<OJ7JE=137J8=2;7II=4>7IE=607HM=7C7H@=8J7H9=9A7H2=:87GJ=:O7GD=;F7G==<=7G5==37FN==J7FH=>A7F@=?87F9=?O7F2=@E7EK=A>7EF=B67EB=BN7E>=CF7E:=D?7E6=E77E2=EO7DM=FH7DI=G@7DE=H87DA=I37D;=J27D4=K17CM=KH7C;=LA7BN=M@7BJ=N?7BF=O>7BB>0=7B>>1<7B;>2;7B6>3:7B2>497AN>577AK>677AG>757AC>847A?>937A;>:27A8>;17A2>;O7@F><N7@:>=N7?L>?G7><>A<7=0>BB7<5>CH7;9>DM7:=>F379A>G878F>H3786>HN77F>II775>JD76F>K>766>L:75E>M4755>N074E>NJ744>OE73E?0@72K?1;71I?2570F?4L6O??6A6NE?7H6MN?8L6M9?9N6LG?:M6LB?;J6LF?<G6LL?=D6M0?>B6M6???6M:?@<6M??A96MD?B76MI?B76NK?BH6NK?C96NJ?CK6NI?D=6NI?DN6NH?E?6NH?F16NH?FB6NG?G46NF
2236<J9066=I8OL6?58OA6@?8O66AJ8NK6C58N@6D@8N56EK8MJ6G68M?6HA8M46IL8LI6K68L>6LA8L36ML8KH6O08K?6OK8K670D8K071<8JK7248JE72M8J?0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008J@8I38K58HN8KJ8HJ8L>8HE8M28HA8MG8H=8N<8H88O08H48OD8GO9098GK90N8GG91B8GB9278G>92L8G793>8G094M8GB9638GC9788GE98<8GH99A8GJ9:E8GK9;I8GN9<E8H09=J8GE9?08G:9@68FO9A<8FC9BB8F79CH8EL9DN8E@9F48E59G:8DJ9H@8D=9IH8D>9JF8E79KB8F69L>8G49M:8H29N68I19O28J09O<8I@9O68GK:058F@:1E8EG:368E4:4G8DB:698D0:7;8C6:7@8AK:7M8@A:7K8@L:7F8A@:7B8B2:7?8B<:8<8C7:9E8D5::H8D8:;G8CH:<E8C8:<I8C<:<<8CN:<M8CM:>L8C1:@L8B5:A?8AC:BC8AB:CC8AB:DA8AA:E@8A@:F?8A?:G:8A?:GI8A@:H88AA:HG8AA:IF8A4:JK8@=:L28?G:LJ8>J:M;8=I:NA8=2:OM8=2;0;8=>;1>8=J;2E8>7;3K8>C;528?0;698?=;7@8?I;8G8@6;9N8@B;;48@O;<;8A<;=B8AH;>J8AF;@38A6;A<8@E;BE8@5;CN8?D;E;8?8;FI8>M;H68>A;ID8>6;K18=K;L?8=@;ME8<J;N=8;C;O68:;;ON895<0C887<1687=<2887@<3?87N<4A883<4I86O<5@864<6;85:<7584A<8083G<94836<:=82I<<O84><?@85K<A385:<AO84E<B983O<BC838<BM82A<D781H<F6810<E87OM<F67OA<G57O9<H47O2<I37NK<J27NC<K17N<<L07N5<LE7M8<MI7LG<O27L9=0;7KK=1C7K>=2L7K0=4O7J<=787IF=8>7I;=957I4=9K7HM=:B7HF=;97H?=<07H8=<F7H0==>7GJ=>47GB=>K7G;=?B7G4=@97FM=@O7FF=AG7FA=B@7F==C87F8=D07F5=DH7F0=E@7EL=F97EH=G17EC=GI7E?=HB7E;=I:7E7=J27E3=K07DL=KK7D?=LG7D6=MF7D2=ND7CN=OD7CJ>0B7CG>1A7CC>2@7C?>3?7C;>4>7C7>5=7C4>6<7C0>7:7BL>8:7BH>987BD>:77BA>;67B=><57B6>=47AJ>>@7@K>@07?D>A67>H>B<7=M>CB7=1>DG7<5>EM7;9>G37:=>H979B>I4791>IO78A>JI782>KD77A>L?771>M:76A>N5760>O075@>OK751?0F743?1A730?2;71M?3670J?686OC?8=6ND?9C6MO?:C6MD?;@6MI?<=6MN?=:6N2?>86N8??56N<?@26NA?@O6NF?AM6NK?BG6O3?BF703?C7702?CH701?D:701?DK701?E=700?EN700?F@6OO?G16ON?GC6ON
3188JC8J68K88J18KM8IM8LA8IH8M58ID8MJ8I@8N?8I;8O38I78OG8I290<8HN9118HJ91E8HE92:8HA92N8H<93B8H494H8H>95L8H@9718HB9868HD99:8HF9:?8HH9;D8HJ9<D8HJ9=J8H?9?08H49@68GH9A<8G<9BB8G19CH8FE9DO8F;9F68F09G=8EF9HD8E=9I?8E09J98EM9K58G29L18H49LM8I39MI8J19NE8K09O58K29ON8IF:1;8H?:2G8G8:438F2:5@8DO:728D=:7=8C8:7B8AM:7@8B8:7=8BF:7:8C1:788C;:7B8CM:8L8DO:9J8DM::F8DG:<18E3:=88E5:>68DD:=I8E5:?38DF:@C8CL:@N8C7:A<8BD:BA8BC:CG8BB:DL8BB:EK8BB:FI8BA:G98BA:GI8BA:H>8B>:I28B;:IJ8AN:JD8A6:KJ8@@:LE8?D:ML8>O:OM8>F;0C8>N;118?:;1H8?F;2O8@2;458@?;5<8@K;6C8A8;7J8AE;908B1;:78B>;;>8BJ;<E8C7;=M8C8;?68BG;@?8B7;AH8AF;C18A6;D:8@F;EG8@:;G58?N;HB8?C;J08?8;K=8>M;LK8>A;N98>6;O@8=C<098<<<118;4<1E8::<2889A<2L88G<3D888<4K88E<5:87K<65871<6O868<7J85><8E84E<9E840<;I84O<>E874<A788D<BF882<C087<<C:86E<C:85G<BE84:<E383;<FE82N<F2820<F?816<G:80?<H9805<I77ON<J77OF<K57O?<L17NO<M17N;<N:7MM<OC7M@=177LO=3;7L:=4L7KH=667K8=7@7JH=8H7J?=9@7J1=:57II=:L7IA=;C7I:=<:7I3==07HL==G7HE=>>7H==?57H7=?L7GO=@C7GH=A97GA=B17G;=BI7G7=CA7G3=D:7FN=E27FK=EJ7FF=FB7FB=G;7F>=H37F:=HK7F5=IC7F1=J;7EM=K47EH=KM7EB=LL7E?=MK7E;=NJ7E7=OI7E3>0G7DO>1G7DL>2E7DH>3D7DD>4C7D@>5B7D=>6A7D9>7@7D5>8?7D1>9=7CM>:<7CJ>;;7CF><;7CB>=:7C9>>D7B8>?J7A<>A07@@>B67?E>C<7>I>DA7=M>EH7=1>FM7<5>H37;:>I97:>>J579M>JO79=>KK78M>LE78=>M@77L>N;77<>O676L?0176<?0L75:?1F747?2A735?3<722?4@70O?7C6OG?:86NF?;56NK?<36O0?=06O5?=N6O:?>K6O>??H6OC?@E6OH?AB6OM?B@702?BO70G?C571:?CF71:?D7719?DI719?E:718?EL718?F>717?FO716?G@716
3188JI8K78K;8K48L08JO8LD8JK8M88JG8MM8JC8NB8J>8O68J98OK8J590?8J19148IM91H8IH92=8IC9318I?93F8I;94A8I:95F8I=96K8I?97O8I@9948IC9::8ID9;B8IF9<F8IG9=M8I=9?48I39@<8HI9AB8H>9BJ8H49D08GJ9E88G@9F?8G59GF8FL9HG8F@9I98F39IN8EO9JJ8G29KE8H69LA8I:9M=8J>9N88KB9O58L>:0?8K=:1J8J6:358HO:4A8GH:5L8FA:778E::7=8CD:788C;:758CE:738D0:708D::6M8DE:838ED:938EL:9O8EE::K8E>:;O8EB:=;8EN:>F8F9:?:8F<:@78ED:@A8DO:@L8D::A88CE:B=8CD:CD8CD:DI8CC:EO8CB:FO8C>:GC8C;:H88C8:HL8C5:IA8C2:J78BI:K18B2:KK8A::M68@J:O88@B;0K8@>;198@J;1G8A6;258AA;388AN;4?8B:;5F8BG;6M8C3;848C@;9;8CM;:A8D9;;H8DF;<O8DJ;>98D9;?A8CI;@K8C8;B48BH;C=8B8;DF8AH;F38A<;GA8A0;HN8@E;J<8@:;KI8?N;M78?C;NE8?8<028>M<1<8><<248=6<2G8<=<3:8;C<3N8:J<4A8:1<54898<5N88H<6I87O<7D876<8?86=<9;85E<:C85A<=?87E<@;89J<BO8;=<CH8:H<CI89K<C588?<B@872<AL85E<DF852<F884E<GI849<FJ836<GF82?<HA81H<I<811<J;80I<K:80A<L:7OO<MB7OB<OB7NN=0K7N==257MM=3?7M==4I7LM=627L==7<7KM=8D7KA=9L7K<=:L7K3=;97J?=;M7J5=<D7IN==:7IG=>17I@=>H7I9=??7I2=@67HJ=@L7HC=AC7H==B:7H5=C27H1=CK7GM=DC7GI=E;7GD=F37GA=FL7G<=GD7G8=H<7G4=I57G0=IM7FK=JE7FH=KB7FA=LB7F;=MA7F7=N?7F3=O>7F0>0=7EL>1;7EH>2:7EE>397EB>487E>>567E:>657E7>737E3>827DO>917DL>9O7DI>:N7DE>;M7DA><K7D?>=G7D9>>E7CI>?D7C4>@J7B9>B07A=>C67@A>D<7?F>EA7>J>FG7=N>GM7=2>I37<6>J97;:>K57:I>L07:9>LK79H>MF798>NA78H>O<788?0777E?1176B?1L75??2G74=?3B73:?6G722?95712?:C70<?;I702?<F707?=C70<?>@70@??=70E?@;70J?A870O?B5714?C2719?C772:?CC72B?D572B?DG72A?E872A?EI72@?F;72??FL72??G>72??GO72>
3188K48L78K>8L78L38L28LG8KN8M<8KJ8N08KE8NE8KA8O98K<8ON8K890B8K49178JO91K8JK92@8JF9348JB93I8J>94E8J69628J29798J398@8J499H8J59:O8J69<68J79=18JH9>68JB9?=8J89@E8IN9AL8IC9C38I:9D:8HO9EA8HE9FH8H;9GN8H19HA8GC9I48G59IG8FH9J>8G39K:8H89L58I;9M18J?9MM8KB9NI8LB9OK8L>:148KA:2?8J::3K8I3:568GL:6A8FE:738E4:6M8DD:6K8DO:6H8E::6E8ED:798F::8=8FK:998FD::58F=:;08F6:;M8F1:=98F<:>D8FI:?K8G2:@58FF:@@8F1:@K8E<:A58DG:B:8DE:C@8DD:DE8DD:EO8D?:G68D8:H28D3:HF8D0:I;8CM:J08CJ:JE8CD:K>8BM:L98B5:MK8B3:ON8AL;1:8B5;1O8BF;2=8C1;2K8C=;3B8CI;4I8D6;608DB;778DO;8=8E;;9D8EH;:K8F5;<28F;;=;8EK;>D8E:;?N8DJ;A78D:;B@8CJ;CI8C9;E28BI;F?8B=;GM8B2;I:8AF;JH8A<;L58A0;MC8@E;O18@:<0>8?N<1L8?C<338?4<3N8>=<4C8=B<588<H<5M8;M<6B8;3<768:8<8189@<8N88H<9>87I<9D86J<<?88H<?98:G<B38<F<D88>1<CD8<D<C08;7<B;89J<AG88=<B=876<D886I<EJ86=<G;861<GF85:<H184?<HL83I<IH831<JC82:<KA81C<LJ812<N480B<O>802=0H7OB=217O2=3;7NB=4E7N2=5N7MB=787M1=8A7LE=9I7L?=;07L:=<77L4=<E7KA==17JN==E7JB=>;7J;=?27J4=?I7IM=@@7IF=A67I?=AN7I7=BD7I0=C<7HL=D47HG=DM7HC=EE7H?=F=7H:=G57H7=GM7H2=HF7GN=I>7GJ=J97GE=K97G>=L97G7=M97G1=N87FL=O77FH>067FE>147FA>237F>>327F:>417F7>4O7F3>5N7EO>6M7EL>7K7EH>8J7EE>9I7EA>:H7E>>;E7E;><B7E9>=>7E8>>:7E0>?87D@>@57D0>A27C?>AO7BN>C07B9>D67A>>E;7@B>FA7?F>GG7>J>HM7=N>J37=2>K97<7>L67;D>M17;4>ML7:D>NG7:4>OB79D?0=78L?1777I?2276G?2N75D?6474<?8073B?9=72L?:J726?<771@?=971>?>671C??371G?@171L?@N721?AK726?BH72;?CE72@?CA73K?D373J?DD73I?E573I?EG73H?F973H?FJ73G?G;73G?GM73F?H>73E
3188K>8M78KE8M88L68M58LJ8M18M?8LM8N38LH8NH8LD8O<8L?9018L;90E8L791:8L291N8KN92C8KI9388KE93L8KA94I8K996B8JH97M8JE9958JF9:<8JG9;C8JH9<H8JL9=@8KB9>?8KG9?G8K=9@N8K39B58JI9C<8J?9DC8J49EJ8IJ9G18I@9GI8I49H<8HF9HO8H89IA8GK9J48G=9JN8H89KJ8I<9LE8J@9MA8KD9N=8LF9O?8LB:0A8L?:1I8KE:348J>:4@8I7:5L8H0:6I8FC:6C8EN:6@8F9:6>8FC:6@8G0:7F8GI:8B8GB:9>8G;:::8G5:;58FN:<18FG:=78FL:>B8G8:?I8GB:@48G>:@>8G4:@I8F?:A48EI:B68EF:C<8EE:DE8EA:F18E;:G:8E4:H88DM:I58DG:IJ8DD:J>8DA:K28D>:KM8CG:LJ8C4:NE8C8;0B8C8;1B8CG;2=8D<;338DM;3A8E9;3O8EE;538F1;698F>;7A8FJ;8G8G7;9N8GC;;58GM;<>8G<;=G8FL;?08F<;@:8EL;AB8E;;BL8DK;D58D;;E>8CJ;FK8C?;H98C4;IF8BH;K48B=;LA8B2;MO8AF;O=8A<<0J8A0<268@L<398@9<458?A<508>I<5K8>1<6G8=9<7B8<B<8=8;J<988;2<8J89=<8E883<;<89O<=F8;?<?O8<O<B98>><C68>H<BI8=M<B68<C<AB8;6<@M89H<B988L<CK88@<E<884<FN87H<H?87;<H<86@<I785H<J3851<JN84:<KH836<LG827<N181G<O:817=0D80G=1M807=377OF=4A7O6=5K7NF=757N6=8>7MH=9E7MC=:L7M==<47M7==;7M2=>27LD=>>7L0=>J7K<=?<7JO=@37JH=@J7JA=A@7J:=B77J2=BN7IL=CE7IF=D>7IB=E67I==EN7I9=FG7I5=G?7I1=H77HM=I17HH=J17HA=K17H;=L17H4=M17GM=N17GG=O07GA=OO7G>>0N7G;>1M7G7>2K7G3>3J7FO>4H7FL>5G7FH>6F7FE>7D7FA>8C7F>>9B7F:>:?7F8>;<7F6><87F5>=57F3>>17F2>>N7EH>?K7E7>@I7DG>AE7D6>BB7CF>C@7C5>D=7BE>E:7B4>F;7A?>GB7@B>HG7?F>IM7>K>K37=O>L97=3>M77<@>N27;O>NM7;?>OG7;0?0B7:4?1=791?2;77O?5=76H?6J762?8775<?9E74F?;2740?<?73:?=L72E?>I72I??G72O?@D733?AA738?B>73=?C;73B?D773H?D0752?DB751?E3751?ED751?F6750?FG750?G974O?GJ74N?H;74M?I274M
3188K?8N;8L18N78L98N88LM8N48MB8N08N68MK8NK8MG8O?8MB9048M>90H8M:91=8M59218M192F8LL93;8LH93O8LD94N8L;96F8KJ98@8K999I8K89;08K99<78K:9=78KE9=O8L;9>H8LL9@08LB9A68L89B>8KN9CE8KD9DL8K99F38JO9G18JC9GD8J69H68II9HI8I;9I<8HM9IO8H@9JC8H99K?8I=9L:8JA9M68KE9N18LH9O48LF:068LC:178L?:2?8KH:3J8JB:558I::6?8H3:698G8:668GB:638GM:708HG:7K8HA:8G8H;:9C8H3::?8GL:;:8GE:<68G>:=58G;:>@8GG:?G8H1:@28GM:@=8GI:@H8GB:A38FL:B28FG:C:8FD:DF8F=:F28F6:G>8EO:H<8EI:I:8EB:J88E<:JL8E9:KA8E6:L:8DC:MC8D8:O=8D<;0K8DG;1J8E:;2E8EO;3?8FB;478G5;4E8GA;5=8GL;6C8H9;7J8HF;918I2;:88I?;;A8HN;<J8H>;>38GN;?<8G>;@E8FM;AO8F=;C88EL;DA8E=;EJ8DL;G78D@;HE8D5;J28CJ;K@8C?;LN8C4;N;8BH;OI8B=<148B;<2?8B8<3@8A@<4;8@D<568?L<618?4<6L8>=<7H8=E<8C8<M<998<6<8K8:;<8M89M<;68;=<=@8<L<?I8><<B38?L<B28?;<AF8>A<A98=G<@H8<A<@38;4<AK8:D<C=8:7<DN89K<F@89><H2892<I:88D<IB87I<J>871<K9866<L284N<LK83F<ML82L<O682<=0@81K=1J81;=3480K=4=80;=5G7OK=717O;=8:7NK=9B7NF=:I7N?=<07N:==87N5=>?7N0=?>7MF=?K7M2=@77L>=@C7KK=A47K<=AK7K5=BA7JN=C87JG=CO7J@=DG7J<=E?7J8=F87J3=G07IO=GH7IK=HH7IE=IH7I>=JH7I7=KH7I1=LH7HJ=MH7HC=NH7H==OH7H7>0G7H3>1F7H0>2D7GL>3C7GH>4B7GD>5@7GA>6?7G>>7>7G:>8=7G6>9:7G4>:67G2>;37G1>;O7FO><L7FN>=H7FL>>E7FK>?A7F?>@>7EO>A<7E?>B97DN>C67D=>D37CM>E07C=>EM7BL>FJ7B;>GH7AK>HE7A;>IG7@C>JM7?G>L37>K>M97>0>N87=;>O27<K>OM7<;?0H7;<?1I7:8?47798?5D78B?7277L?8?776?9L76@?;:75J?<F754?>474>??<741?@:745?A774;?B474??C174D?CN74I?D?75<?D?769?E0769?EB769?F3768?FE768?G6767?GH766?H9766?HO765
3188K>8O@8L88O98LB8O98M08O68ME8O38N98NN8NN8NI8OB8NE9078N@90L8N=91@8N89248N492I8MO93>8MJ9428MG9528M>96K8LL98D8L;9:=8KJ9;E8KK9<L8KL9=F8L>9>>8M49?58MI9@98MG9A?8M>9BG8M39CN8LI9E58L>9F88L49FL8KF9G>8K99H18JK9HD8J=9I78J09II8IB9J<8I49K38I>9KO8JB9LJ8KF9MF8LJ9NG8LJ9OJ8LF:0L8LC:1N8L?:348KM:4?8JF:5K8I?:5N8HA:5K8HL:668I=:758I@:818I9:8M8I2:9H8HK::D8HD:;@8H>:<<8H6:=78GO:>>8H6:?E8H@:@08H<:@;8H8:@F8H4:A18GN:B08GF:C<8G?:DG8G8:F38G2:G?8FK:H@8FD:I>8F=:J=8F7:K;8F0:L08EM:LH8E=:N<8E=:OF8EI;0G8F<;1H8FO;2I8GB;3H8H5;4B8HJ;5;8I=;5I8IH;6M8J4;848JA;9;8JN;:D8J@;;M8J0;=68I@;>?8HO;?H8H?;A18GN;B;8G>;CC8FN;DM8F>;F58EM;GC8EB;I18E7;J>8DL;KL8D@;M:8D5;NG8CM<028CJ<1=8CG<2H8CD<3F8BH<4@8AH<5<8A0<678@8<728?A<7M8>H<8I8>1<988=7<918;<<8F8;:<;08<J<=98>:<?C8?J<A;8@J<@N8@0<@B8?5<@58>;<?I8=A<?L8<G<A>8<;<C08;O<DA8;C<F28;6<GD8:J<I58:><IN89I<JI892<KB87M<L<86F<M485><MM846<O383@=0<830=1F82@=30820=4981@=5C810=6M80?=867OO=9>7OH=:E7OC=;M7O===47O8=><7O2=?C7NM=@K7NG=A77N4=AC7M@=B07LL=B;7L8=BK7KI=CB7KB=D97K;=E17K6=EI7K2=FA7JN=G@7JH=H@7JA=I@7J;=J?7J4=K@7IM=L?7IG=M@7I@=N?7I:=O@7I3>0@7HM>1>7HH>2=7HE>3<7HA>4;7H=>597H:>687H7>777H3>847H0>907GO>9M7GM>:J7GL>;F7GJ><C7GI>=?7GG>>;7GF>?87GD>@57G7>A27FF>AO7F6>BL7EE>CI7E5>DF7DD>ED7D4>FA7CC>G>7C3>H;7BB>I87B2>J57AB>K27A1>L07@@>M37?H>N97>L>O87>7?037=F?167<B?327;H?4>7;3?5L7:=?7979G?8F791?:478;?;A77E?<N76O?>;769??I75C?@L75=?AI75A?BG75F?CD75K?DA760?DG770?DN77B?E?77A?F177@?FC77@?G477??GE77??H777??HM77=?IC77<
3188K=90E8L790>8LM9088M49098MH9058N=9018O18OL8OE8OH90:8OC90O8O?91C8O;9278O692L8O293A8NM9458NI9568N@9708MO98H8M>9:B8LK9<98L=9=<8LB9>48M79>L8MM9?D8NC9@B8NM9AH8NC9C08N89D78MN9E>8MD9F38M79FF8LI9G98L;9GL8KN9H?8K@9I18K29ID8JE9J78J79JJ8IJ9KC8JC9L?8KG9M;8LK9N<8LN9O>8LJ:0@8LG:1B8LC:2D8L?:3I8L1:548JI:5C8IJ:5A8J5:6?8J?:7:8J8:868J0:928IJ:9M8IC::I8I<:;E8I5:<A8HN:=<8HG:>;8HE:?D8HO:?O8HK:@98HG:@D8HD:A98HA:B88H@:C=8H;:DI8H4:F58GM:GA8GF:HC8G?:IB8G9:JA8G2:K?8FL:L>8FE:M:8F=:NB8FL:OB8G?;0C8H2;1D8HE;2E8I7;3E8IJ;4F8J=;5E8K1;6?8KD;778L0;8>8L<;9G8L2;:O8KB;<98K1;=B8JA;>K8J0;@48I@;A=8I0;BF8H@;CO8H0;E88G?;FB8FO;GO8FD;I=8F9;JK8EM;L88EB;MD8E<;NO8E9<0:8E6<1F8E3<308E1<3M8CO<4F8BK<5A8B3<6=8A<<788@D<848?L<8N8?4<988>8<978<><8J8<?<:J8>8<=38?G<?=8A7<@78A><?J8@D<?>8?J<?28>O<?48>4<@88=5<A@8<6<BL8<4<D:8<8<EI8<<<G68<@<HH8<4<J98;H<K38:M<KM89E<LE88=<M>876<N785N<O084F=09845=1B83E=2L834=4582D=5?824=6I81D=83814=9;80K=:B80F=;J80A==180;=>8805=?@800=@G7OK=AO7OE=BD7O5=C07NB=C<7MN=CH7M:=D47LG=DC7L6=E:7L1=F77KK=G77KD=H77K>=I77K7=J77K1=K77JJ=L77JC=M77J==N77J6=O77J0>077II>177IC>277I=>357I:>447I6>527I3>617HO>6N7HM>7K7HL>8G7HJ>9D7HI>:@7HG>;=7HF><97HC>=67HB>>27H@>>O7H?>?K7H=>@I7GN>AE7G>>BB7FM>C@7F=>D=7EL>E:7E<>F77DK>G47D;>H17CJ>HN7C:>IL7BI>JI7B9>KE7AH>LC7A8>M@7@G>N=7@7>O:7?F?0?7>N?1L7>8?397=B?4F7<L?637<6?7A7;A?8N7:K?:;7:5?;H79??=578I?>C783?@077=?A=76G?B=76I?C:76M?D7772?E4777?DO78D?E=78I?EN78I?F@78H?G178H?GC78G?H478G?HJ78F?IA78E?J878D
3188K<91K8L691C8LO91<8M?9198MK9188N@9148O490O8OH90K90=90F91290B91F90>92:90992O90593D9009488OL95;8OC9748O198M8N@9:F8MN9<?8M=9=K8M<9>C8N19?;8NF9@38O<9@K9019B28OH9C98O=9D@8O39E;8NG9EN8N:9FA8ML9G48M>9GF8M19H98LC9HL8L59I?8KH9J18K:9JD8JL9K88JD9L38KH9LO8LL9N08M19O28LN:048LK:168LG:288LC:3:8L@:4>8L5:598K4:5H8K=:6D8K6:7?8JO:8;8JI:978JB::28J;::N8J3:;J8IM:<F8IF:=A8I?:>=8I8:?B8I>:?M8I::@88I7:A58I5:B48I4:C48I2:D48I1:E28HM:F68HI:GB8HB:HG8H;:IF8H4:JD8GN:KC8GG:LF8GE:MF8GC:NE8H8:OE8HL;0E8I@;1D8J5;2D8JI;3C8K>;4B8L2;5C8LF;6C8M8;7C8MK;8I8MD;:28M4;;<8LC;<E8L3;=N8KB;?78K2;@@8JB;AI8J2;C28IA;D;8I1;ED8HA;FN8H1;H;8GE;II8G;;K78FO;LB8FK;MM8FI;O88FE<0C8FC<1N8F@<398F=<438E7<4L8D0<5H8C7<6C8B?<7>8AG<8:8A0<8N8@8<978?8<9=8=?<908=A<:C8?E<<M8A5<?68BE<?38B2<>G8A8<>:8@=<>>8?B<?M8>C<@I8=E<B18<F<C88;F<DB8;0<F08;5<G>8;9<HK8;=<J98;A<K<8<3<L68;=<LO8:5<MH88M<NA87F<O:86>=05859=1>84I=2H849=4283I=5<839=6E82I=7O829=9881N=:?81I=;F81D=<M81>=>5818=?<813=@D80N=AK80G=C380B=D0808=D=7OD=DI7O0=E57N<=EA7MI=EO7M5=FK7LH=GL7L@=HM7L7=IN7KN=JN7KG=KO7K@=LO7K:=MO7K3=NN7JL=OO7JF>0N7J?>1O7J9>2N7J2>3M7IO>4L7IL>5I7II>6E7IH>7A7IF>8>7IE>9;7IC>:77IB>;47I@><07I?><M7I=>=I7I<>>E7I:>?B7I9>@>7I6>A<7HF>B97H5>C67GD>D37G4>E07FD>EM7F3>FJ7EB>GH7E3>HE7DB>IA7D1>J?7CB>K<7C3>L97BD>M67B4>N37AE>O07A6>OM7@G?0L7@6?247?F?3A7?6?4N7>G?6;7>1?7H7=;?957<E?:C7;O?<07;9?==7:C?>J79M?@8797?AE78A?C077O?CM785?DJ789?EG78>?E:7:1?EL7:1?F>7:0?FO7:0?G@7:0?H279O?HH79N?I>79M?J579L?JM79K
3198L592H8LN92A8MH92:8N192:8NC9278O79228OK91N90@91I91591E91I91A92>91<93291893G91394;90O95?90E9799039928OB9:K8O09<D8N?9=H8NI9>K8O29?J8O?9@B9059A:90J9B;90M9CB90B9DC9079E68OJ9EH8O<9F<8NO9FN8NA9GA8N49H38MF9HG8M89I98LK9IL8L=9J?8KO9K28KB9KH8KI9LC8LM9MD8M59NF8M19OH8LN:0J8LK:1L8LG:2N8LD:418L@:528L<:5N8L5:6I8KN:7E8KG:8A8K@:9<8K9::88K2:;48JL:;O8JE:<K8J=:=F8J6:>B8J0:?@8IM:@28IJ:A28II:B18IG:C18IF:D08ID:E08IC:F08IA:FN8I>:GL8I;:HL8I8:J08I6:K48I4:L88I3:M;8I1:N:8HI:O58I3;058IG;158J<;248K0;348KE;438L9;538LN;628MB;728N6;828NG;988N?;:=8N4;;G8MD;=08M4;>:8LD;?B8L4;@L8KC;B58K3;C>8JC;DG8J3;F08IB;G:8I3;HH8HG;J58H=;K@8H;;LK8H8;N68H4;OA8H2<0L8GO<278GM<3@8GE<498F?<538E7<5N8D:<6I8CC<7D8BK<8@8B3<8M8A<<978@:<9B8>A<968>C<:=8A3<<G8BB<>;8CA<=O8BF<=C8AL<=I8A0<?98@1<@G8?2<A:8>5<BB8=5<CI8<6<E18;7<F88:7<GE8:1<I28:6<JA8::<KB8:M<L>8;J<M98;M<N28:E<NK89=<OD886=0=86N=1;85N=2E85>=3N84N=5884==6A83M=7K83==94832=:;82L=;C82F=<J82A=>282<=?9826=@@820=AH81K=BO81E=D781@=E=81:=EI80F=F5802=FA7O?=G27NK=GC7N7=H;7MH=I<7M?=J=7M7=K>7LM=L?7LE=M@7L==NA7L4=OB7KK>0C7KB>1D7K:>2E7K2>3F7JH>4C7JF>5?7JE>6<7JC>787JB>857J@>917J?>9N7J=>:K7J<>;G7J:><D7J9>=@7J7>><7J5>?97J3>@57J2>A27IM>AO7I=>BL7HN>CI7H?>DF7H0>ED7GA>FA7G2>G>7FC>H;7F4>I87EE>J57E6>K27DF>KO7D7>LL7CH>MI7C9>NG7BJ>OD7B;?0A7AK?1@7A8?2?7@E?3K7@6?587?G?6D7?9?817>J?9>7>;?:J7=I?<77=4?=D7<>??27;H?@?7;2?AM7:<?C979F?D@79;?E=79A?EO7:2?EI7;9?F;7;9?FL7;8?G>7;8?GO7;7?HE7;6?I<7;5?J37;4?JJ7;3?KA7;3
3208LM93F8MG93?8N<93:8NF93:8O:9358ON93090C92L91892H91L92D92A92?93592:93J92694>92295D91H97=91699690E9:O9039<C8OI9=D9069>G90?9?J90H9@M9129AH91C9BD9229CK91H9D>91:9E090L9EC90?9F69019FI8OC9G;8O69GN8NH9HA8N;9I48MM9IG8M@9J98M29JL8LE9K?8L79L88LN9M88M99N:8M59O<8M1:0>8LN:1A8LK:2E8LN:3J8M4:4O8M::5L8M5:6H8LN:7D8LG:8@8L?:9<8L8::88L1:;48KI:<08KB:<L8K;:=H8K3:>D8JL:?G8JI:@F8JI:AG8JE:BH8JB:CI8J?:DI8J<:EJ8J8:FJ8J5:GK8J2:HH8J7:IB8JC:JE8JB:KI8J@:LM8J>:N18J=:O08J6:OE8IM;0E8JB;1E8K6;2D8KK;3D8L?;4C8M4;5C8MH;6B8N=;7B8O1;8D8O9;9J8O1;:O8NI;;O8ND;=88N5;>B8MD;?L8M4;A68LE;B@8L5;CJ8KE;E38K4;F<8JD;GF8J4;I38IM;J=8IJ;KI8IG;M48ID;N>8IA;OJ8I?<148I<<2@8I9<3F8HM<4@8GF<598F?<648E><6O8DF<7J8CN<8C8C7<8M8B?<968A:<9A8?B<9<8?E<:78B@<<@8D0<=88D5<<L8C;<=48B?<>D8A?<@28@@<A68?B<AK8>D<C38=E<D:8<F<EB8;F<FI8:G<H189G<I:88O<JH892<KH89H<LD8:D<M@8;@<N<8<<<O58;5<OM89N=0G88E=1?87>=2A86B=3J862=5485B=6>851=7H84B=90845=:883O=;@83I=<G83D==N83?=?5839=@=833=AD82N=BL82I=D382C=E:82==FB828=G681H=GB814=H480A=HF7OM=I77O9=IK7NG=JL7N?=KM7N6=LN7MM=MO7MD=O07M<>017M3>127LJ>237LB>347L9>427L4>4E7L>>5?7LA>6<7L>>787L=>857L;>917L9>9N7L7>:K7L6>;G7L4><D7L2>=@7L1>>=7KN>?:7KM>@67KK>A=7K1>BB7JB>C@7J2>D=7IC>E:7I4>F77HE>G47H6>H17GG>HN7G7>IK7FH>JH7F9>KE7EJ>LC7E;>M@7DL>N=7D=>O:7CM?077C>?157BM?247B:?347AG?467A5?5B7@F?6O7@8?8;7?I?9H7?:?;47>K?<A7>=?=M7=N??97=??@G7<L?B47<6?CA7;@?DN7:J?F07:G?F77;F?F97<A?FJ7<@?G;7<@?GM7<??HC7<??I:7<>?J07<<?JH7<<?K?7<;?L57<9
3218ME94D8N@94<8NM94:8O=94890194390F93O91;93K91O93F92D93B93893=93M93994A93595H92J97A92899;91G9;39159<>9189=@91C9>C91M9?F9269@I92?9AL92I9BO9329CE92J9D892=9DK91O9E>91A9F09149FC90F9G69089GI8OK9H;8O=9HO8O09IA8NB9J48N49JG8MG9K:8M99KM8LO9LL8M<9MN8M99O18M9:068M?:1<8ME:2@8MK:3F8N2:4J8N8:5J8N6:6F8MO:7B8MG:8>8M@:9:8M9::68M1:;28LJ:;N8LC:<J8L;:=F8L4:>O8L2:@28L2:A18L7:B18L4:C28L0:D38KM:E38KJ:F48KG:G58KC:H68K@:IB8K=:J<8KH:K:8KN:L>8KL:MB8KJ:NF8KH:OE8KB;088JN;158K=;258L2;348LF;448M:;538MO;638NC;728O8;828OL;968OK;:<8OC;;B8O;;<G8O3;=H8NM;>J8NE;@48N5;A>8ME;BH8M5;D28LE;E<8L5;FF8KE;GN8K5;I;8K9;JG8K6;L18K3;M<8K0;NG8JN<028JK<1>8JI<2H8JF<3M8J5<4F8HM<5?8GG<698FB<758EI<808E2<8C8D:<8L8CC<968B;<9A8@C<9B8@G<:08CN<<:8E><<48DI<<?8CM<=N8BN<?>8AO<@M8@O<AD8@2<B<8?4<CD8>5<DK8=5<F38<6<G:8;7<HB8:7<IJ898<K1888<KN88C<LJ89><ME8:;<NB8;7<O=8<2=088;E=108:>=1I895=2B87N=3G876=5086G=6:867=7D85F=8M858=:5852=;<84M=<C84G==K84B=?284<=@:846=AA841=BH83L=CO83F=E783@=F?83;=GF836=HE82J=I6826=IH81C=J980O=JK80;=K=7OH=L<7O>=M=7O5=N>7NL=O?7ND>0@7N;>1A7N3>2B7MJ>3C7MA>4<7MC>4O7MM>5B7N7>6;7N:>777N9>847N7>917N6>9M7N3>:J7N2>;F7N0><C7MN>=@7MM>><7MK>?97MI>@57MG>AE7L5>BK7KF>D17K7>E07JH>EM7J9>FJ7II>GG7I:>HD7HK>IA7H<>J?7GM>K;7G>>L97FN>M67F?>N37F0>O07EA>OM7E2?0K7DB?1I7D0?2I7C=?3H7BJ?4F7B7?5M7AE?797A7?8F7@I?:27@9?;?7?K?<K7?<?>87>M??D7>>?A07>0?B=7=A?CI7=2?E67<C?FC7;O?F?7=9?FH7=H?G97=H?GJ7=G?H@7=G?I77=F?IN7=D?JE7=D?K<7=C?L37=A?LJ7=A
3218MD95I8N>95B8O895:8O@95;90495690I95291>94N92294I92G94E93;94@94094<94E94895L93L97F93:99?92I9;89279<:92F9=<9309>?93:9?B93C9@E93L9AH9469BI9489C>93D9CJ9379D=92J9E092=9ED9209F791B9FJ9159G>90H9H190;9HD8ON9I78OA9IK8O39J>8NF9K18N:9KD8ML9LC8MJ9MH8N09NM8N7:028N=:178NC:2<8NI:3A8NO:4F8O5:5H8O7:6D8O0:7@8NH:8<8NA:988N:::48N2:;08MK:;L8MD:<H8M<:>28M::?=8M::@>8M;:A=8M?:B;8MB:C<8M?:D=8M;:E>8M8:F>8M5:G?8M2:H@8LN:IH8LL:K58LN:L08M9:M48M8:N88M6:O;8M4;0<8LO;0N8L;;1E8L8;2E8LM;3D8MA;4D8N6;5C8NJ;6C8O?;7B903;8B90E;9H90=;:N905;<48OL;=98OD;>?8O<;?A8O6;@?8O3;AF8NF;C08N5;D:8ME;ED8M5;FM8LE;H48L5;I;8KE;JH8L<;L:8L@;ME8L=;O08L;<0;8L8<1F8L5<318L2<438K<<4L8J6<5F8HN<6@8GG<7:8FM<868F6<8B8E><8K8DG<958C<<9@8AC<9H8AI<:08E6<;=8F8<;J8E;<=98D<<>I8C=<@78B><AG8A><B38@B<BM8?D<D58>D<E=8=E<FD8<F<GL8;F<I38:G<J08:1<K589I<L9893<M188L<MK895<NG8:1<OC8:M=0?8;I=1:8<5=238:M=2L89E=3E88>=4M87K=6687;=7@86K=8I86:=:1865=;9860=<@85J==G85D=>O85?=@685:=A=854=BE84N=CL84I=E484D=F;84>=G=844=HB83L=IG83B=JJ839=K;82E=KM821=L?81==M180J=ML80==NL805=ON7OL>0N7OC>207O;>307O2>427NI>4F7O2>597O<>5M7OE>6@7OO>77805>84803>90801>9L800>:I7ON>;F7OL><C7OK>=?7OH>>;7OG>?87OE>@57OC>AM7M9>C37LJ>D97L:>E?7KK>F@7K<>G>7JM>H;7J>>I77IO>J57I?>K27I0>KO7HA>LL7H2>MI7GC>NF7G4>OC7FE?0@7F5?1?7EE?2>7E2?3=7D??4<7CL?5;7C9?6:7BF?7D7B6?907AH?:=7A8?;I7@J?=57@;?>B7?L??O7?=?A;7>O?BG7>@?D47>1?E@7=B?FK7=<?FG7>M?G67?0?GH7?0?H=7>O?I57>N?IK7>M?JB7>L?K:7>K?L07>J?LG7>I?M?7>H
3228N=96G8O796?8OJ96;90896990L96591A96192595L92J95H93>95C94395?94H95;96194O97J94=99C93L9;:93>9<69449=894>9><94G9?>9509@B95:9AD95C9BB9559C?94<9CG93O9CO93C9D@9359E392H9EF92;9F:91N9FM91A9G@9149H390F9HG9099I:8OL9IM8O?9J@8O29KA8O39LC8O19ME8NO9NI8O39ON8O9:138O?:288OF:3=8OL:4B902:5F908:6B900:7>8OI:8:8OB:968O:::28O3::N8NL:;J8ND:=58NB:>H8NC:?I8NC:@J8NE:AI8NH:BG8NM:CG8NM:DG8NJ:EH8NF:FH8NC:GI8N@:HJ8N=:IO8N::KG8N7:LJ8N?:MI8NC:NM8NB;008N@;128N<;1C8MG;258M4;358MH;448N=;548O0;638OE;73909;8290N;95917;::90N;;@90G;<F90?;=L906;?18ON;@78OF;A:8O?;B88O<;C88O6;DB8NF;EK8N6;G28MF;H98M6;IA8LF;JH8L7;K88LF;MN8MJ;O98MG<0D8MD<1O8MA<3:8M><498LD<538K=<5M8J6<6F8HO<7@8H1<888G9<8B8FB<8K8EK<958D=<9@8C<<9K8DJ<:78F8<;58FI<<D8EJ<>38DK<?C8CL<A28BL<B@8AM<BA8A2<C>8@3<DF8?4<EM8>5<G58=5<H=8<6<HN8;L<I28;M<K:8;8<L@8:?<M>8:0<N789I<NO89A<OI89H=0E8:C=1@8;?=2=8<;=368;==3O8:5=4I88O=6288?=7<87O=8F87@=9N878=;5873=<<86M==D86G=>K86B=@386==A:867=BB861=CI85L=DM85D=EO85:=G2850=H784H=I<84>=JA846=KF83N=LK83E=MA833=N282@=ND81L=O<81=>0=814>1>80K>2?80C>3?80:>4=806>5080@>5C80K>67814>6J81>>7=81H>8381O>8O81M>9L81L>:H81J>;E81H><B81G>=>81E>>;81C>?881A>@481@>B57N=>C;7MN>DA7M?>EG7M0>FM7LA>H17L1>HN7KB>IK7K3>JH7JD>KE7J5>LB7IF>M?7I6>N=7HG>O97H8?077GI?147G:?237FG?327F4?417EA?507DN?5O7D;?6N7CH?7N7C5?9;7BG?:G7B8?<47AI?=@7A:?>L7@L?@97@=?AE7?N?C27?@?D>7?0?EK7>B?FN7?2?G47@9?GE7@8?H;7@7?I27@6?II7@6?J@7@4?K77@3?KN7@3?LE7@1?M<7@0?N37@0
3238O697D8OO97=90<97<90O97791D97492896O92M96K93A96F94696A94K96>96596197O95?99H94N9;594M9<295B9=595K9>89659?;96>9@>96G9A?96J9B;9619C89599CJ94F9D194:9D993M9DC93@9E69339EI92F9F<9299G091L9GC91>9H69129HJ90E9I=9079J?90:9KB90<9LC90;9ME9089NG9069OJ907:0O90=:2490C:3990I:4>90O:5C915:6A911:7<90J:8990B:9490;::0904::L8OL:<78OJ:=J8OK:?48OL:@58OL:A68ON:B5901:C3905:D190::E1908:F2905:G3902:H48OO:I48OL:J68OI:KO8OF:MC8OE:N>8OO:OB8ON;0F8OL;1G8OH;298O3;2K8N?;3E8NC;4E8O7;5D8OL;6C90@;7C915;8B91I;9G91I;:L91A;<2918;=8910;>>90H;?C90@;@I908;AO8OO;C38OI;D18OE;DO8O@;F08O6;G88NF;H?8N7;IF8MG;JN8M7;L28LJ;L>8M7;NE8N:<0L8O0<278NN<3C8NK<4@8ML<598LE<638K=<6L8J6<7F8I4<888H=<8A8GE<8J8FO<958FB<9B8G@<:28GJ<:C8H4<;B8G5<=>8F9<>N8E:<@<8D;<AL8C;<BO8B><C08AB<CO8@C<E78?D<F>8>D<GF8=E<GL8=E<H08=F<H@8=F<K?8<F<LE8;N<MJ8;5<ND8:M<O<8:F=048:?=0M8:7=1F8::=2B8;5=3>8<1=488;M=528:E=5O89D=79894=8B88D=9J88;=;1886=<9880==A87K=>H87E=?O87@=A787:=B<873=C>86I=DA86?=EC865=FG85L=GL85D=I185;=J6852=K;84J=L@84A=ME848=NJ840=OF83B>0882N>0K82<>1M823>2M81J>3O81B>4G81E>5:81O>5M829>6A82C>7482L>7G836>8;83@>8N83I>9K83H>:H83F>;D83D><A83B>==83A>>:83?>?783=>@383<>B=7OB>CD7O2>DI7NC>EO7N3>G57MD>H;7M5>IA7LF>J>7L7>K;7KH>L97K8>M57JI>N37J:>O07IK>OM7I<?0J7HM?1H7H<?2G7GI?3F7G6?4E7FC?5D7F0?6C7E=?7B7DJ?8A7D7?9E7CF?;17C7?<>7BH?=J7B:??77AK?@C7A<?B07@M?C<7@??DI7@0?F27?M?G27@I?GC7A@?H87A??I07A??IF7A>?J=7A<?K57A;?KK7A;?LB7A9?M:7A8?N17A8?NG7A7
3248ON98C90G98<91298:91G98692;98293097M93D97I94997D94N97@96:97498396B99L9609;096;9;N96O9=19789>497B9?797K9@:9849A797G9B396N9C09669CL95>9D49529D;94E9DB9489DI93L9E893?9EK9319F?92D9G29279GE91J9H:91>9I=91A9J?91D9KB91F9LD91D9MF91A9NG91?9OI91=:0K91;:2091A:3491G:4:91M:5>923:6?921:7:91K:8791C:9391;:9N915:;:912:<M913:>>914:?@915:@@916:AB917:BA91::C?91>:D=91B:E<91G:F<91D:G=91A:H>91>:I>91::J?917:L5915:MM912:O7916;0791:;1;918;2>915;2O90@;3A8OL;458O>;55902;6490G;7491;;83920;9392C;:992:;;?923;<D91K;=J91B;?091:;@5912;A;90J;BA90B;CF909;DL902;EJ8OL;FJ8OC;GI8O<;HI8O3;IL8NH;K38N8;L:8MH;M=8M=;MI8MJ;O58NC<228OM<3K908<4F8O3<5?8ML<698LE<738K><7M8J8<888I@<8E8HG<938GO<9C8H1<:38H;<:D8HE<;58HL<<38H2<=98F><?H8EI<A78DJ<BF8CJ<C=8BN<C>8B2<D@8A3<EH8@4<FE8?><FJ8??<FN8?A<G38?A<J?8>N<KE8>5<LJ8=<<N08<D<O18<1<OI8;J=0B8;C=1:8;<=228;4=2K8:M=3D8:L=4?8;H=5;8<==648;6=758:8=8>89H=9G89>=:N898=<6893===88N=>D88H=?K88B=@M888=B087O=C287E=D587;=E7871=F<86I=GA86@=HG867=IK85N=K185F=L585>=M;854=N?84L=OE84C>0I84;>1L840>2=83<>3<833>4=82J>51833>5D83=>6883H>6K842>7>84;>8184E>8E84O>98859>9K85C>:G85B>;C85A><@85>>==85=>>985;>?685:>@084N>BE80E>CL806>E17OG>F87O8>G>7NI>HC7N9>IJ7MJ>JO7M;>KO7LL>LL7L=>MI7KN>NF7K>>OC7JO?0@7J@?1=7J1?2<7I>?3;7HK?4:7H8?597GE?687G2?787F??867EL?957E9?:57DF?;<7D6?<H7CG?>57C9??A7BJ?@N7B;?B:7AL?CG7A>?E37@O?F57AC?G67B>?H47BG?HM7BG?ID7BF?J;7BD?K27BD?KI7BC?L@7BA?M77BA?MN7B@?NE7B??O<7B>
32590G99@91899<91J99992?99593399093G98L94<98H95198C96>98698797D9:09749:L97J9;K98=9<M98F9>198O9?39999@599<9@O98D9AL97L9BI9739CF96:9D695I9D=95<9DD94O9DL94C9E39469E;93I9EN93=9FB9309G892F9H;92I9I>92L9J@92N9KC9309LD92M9MF92K9NH92I9OJ92G:0K92D:1M92B:3092D:4592J:5:930:6=933:7992K:8592D:9192=::=92;:<092;:=B92<:>J92=:?K92=:@M92?:AN92@:BN92C:CK92G:DI92K:EH930:FF932:GG92O:HH92L:II92I:JI92E:L<92C:N592@:OM92=;0M92F;2092D;3392A;3E91M;47918;4I90D;5E90N;6D91B;7D927;8C92K;9E935;:K92M;<192D;=692<;><924;?A91L;@G91D;AM91;;C3914;D990K;E;90J;F?90@;G@905;HB8OK;IC8O?;JC8O6;KC8NN;LB8NF;MG8N9;NH8N0;O48N=<288OJ<3O90L<4I909<5B8O3<6<8MK<758LD<7O8K=<8H8J?<958IF<9C8HM<:48HL<:D8I6<;48I@<;H8IB<<E8I0<=K8G;<?18EF<B18E9<C@8D:<CL8C><CM8BB<E18AC<ED8A8<EH8A9<EM8A:<F18A<<F68A<<JD8@=<KJ8?D<LO8>K<N58>2<O:8=:=078<N=0O8<G=1G8<?=2@8<9=388<1=408;J=4I8;C=5A8;>=6=8<:=778;F=8;8:M=9D8:A=:K8:<=<28:6==98:1=>=89H=??89>=@B894=AD88J=BG88@=CI886=DM87M=F187E=G787<=H;873=IA86K=JE86B=KK86:=M0861=N585H=O:85@>0?857>1D84N>2H84E>3N84=>4H848>5;84B>5N84L>6B856>7585@>7H85J>8;864>8O86=>9B86H>:5872>:H87<>;C87=><@87;>=<879>>8877>?5876>@9860>BM81J>D481:>E:80K>F@80<>GF7OM>HK7O>>J27NN>K77N?>L>7N0>M?7M@>N<7M1>O97LB?077L3?137KD?217K3?307J@?3O7IM?4O7I:?5N7HG?6M7H4?7L7GB?8K7FO?9J7F<?:I7EI?;H7E6?=37DF?>?7D8??L7CI?A87C:?BE7BL?D17B=?E97B>?F:7C:?G@7CM?H;7CL?I47CK?IM7CJ?JF7CJ?K?7CJ?L77CI?M07CH?MI7CG?NB7CG?O:7CF@017CE
3248OL9:M90F9:E91?9:>91M9:<92B9:89369:393J99O94?99K95499F96C99898<98F99L98B9:H9999;G99I9<J9:39=M9:<9?09:E9?M9:99@G99A9AD98I9BB9809C?9779D896A9D?9649DG95G9DN95;9E594N9E<94A9ED9459F693M9G99409H;9439I>9469JA9499KC9499LE9469MG9449NH9429OJ940:0L93M:1N93K:2O93I:4193G:5693M:6;943:7793L:8393E:9@93C:;393C:<E93D:>593E:?693F:@793G:A893H:B993I:C:93K:D7940:E6944:F4949:G294=:H194=:I294::J3947:K3944:LC941:N;93O;0493L;1E93L;2F940;3I93N;4;93:;4M92E;5?921;6591H;7592=;84931;9493F;:793F;;=93?;<C937;=I92N;>N92F;@492=;A9926;B?91N;CA91K;DC91K;EE91I;FN91D;H0912;I190G;J290<;K4902;L58OG;M68O<;N78O1;O98NF<0A8NM<288ON<3N90O<4M913<5C908<6=8O0<768MJ<808LB<988K=<9E8JD<:28IL<:E8IG<;58J1<;E8J<<<<8J9<=68IM<><8H8<?B8FC<@H8DN<D98DI<D:8CN<D;8C2<DB8C1<DF8C3<DK8C4<E18C5<E88C5<GK8BK<JI8AL<KO8A3<M58@:<N:8?A<O@8>H=0D8>2=1<8=K=248=D=2M8=<=3E8=5=4=8<N=568<G=5N8<?=6F8<9=7?8<2=8:8<6=9@8;D=:G8;?=;L8;7=<N8:M=>18:C=?38:9=@68:0=A889F=B;89<=C=892=DB88I=EG88A=FL889=H187O=I687G=J;87>=K@876=LE86M=MJ86D=NO86<>06863>1>85J>2F85A>3N858>4K851>5@85A>66864>6L86D>7?86O>82879>8E87C>9987L>9L886>:?88@>;288J>;F894><?897>=;895>>8893>?5892>@C871>C682N>D<82?>EA81O>FH81@>GN811>I480A>J:802>K?7OC>LF7O4>MK7NE>NO7N6>OL7MF?0J7M7?1G7LH?2F7L6?3E7KC?4C7K0?5C7J=?6B7IJ?7A7I7?8@7HD?9?7H1?:>7G>?;=7FK?<<7F8?=>7EF?>J7E7?@67DI?AB7D9?BO7CK?D<7C<?E=7D5?FC7E9?GJ7E1?HC7E0?I<7E0?J57DO?JN7DO?KF7DN?L?7DM?M87DL?N17DL?NI7DL?OB7DK@0;7DJ
3248OK9<290E9;J91>9;C9259;=92E9;;9399;693N9;194B9:N9579:I96G9:;98@99I99G9:19:C9:G9;C9;79<F9;@9=I9;J9>K9;N9?E9;69@@9:>9A=99E9B:98L9C79849D597;9DB96K9DI96>9E09629E795E9E?9589EF94L9F?94O9GA9529HD9559IF9589JJ95=9L095B9M095A9MO95?9NN95;9ON959:0M957:1N955:2O952:41950:5394N:67951:7594L:8C94L::694L:;H94L:=;94M:>A94N:?B94O:@C950:AD951:BE952:CF954:DD958:EB95=:F@95B:G>95E:H<95J:I<95H:J=95E:K=95B:LI95?:NB95=;0;95:;23958;3;95;;4>959;5194F;5C942;6593=;6F92I;7E938;8D93L;9D94A;:I949;;O940;=593H;>;93@;?@938;@F930;AH92M;BJ92L;CL92K;DN92J;EO92H;G892D;HD92=;IA91D;JB919;KC90N;LD90C;MF908;NG8ON;OH8OC<0G8O:<27902<3N913<5391?<5G914<6=906<778NO<818MG<9;8LA<:58KC<:C8JJ<;68JC<;F8JL<<68K6<<O8JO<=H8JH<>M8I5<@38GA<A:8EL<B>8D:<CA8D8<DC8D7<EF8D5<F;8D5<FA8D6<G58D5<I48D1<J68CF<JO8C:<L48BB<M:8AI<N?8A0<OE8@7=0J8?>=1I8>O=2B8>H=3:8>A=428>9=4K8>2=5C8=K=6;8=D=748===7L8=5=8D8<O=9>8<E=:=8<<=;@8<3=<B8;I==E8;?=>G8;5=?J8:K=@L8:A=AO8:7=C289N=D789F=E<89==FA895=GF88L=HK88C=J388:=K;881=LC87I=MK87@=O3876>0;86M>1C86E>2J86<>42863>4O85K>5?85L>6586?>6K872>7A87F>88889>8N88L>9B89;>:689E>:I89O>;<8:9>;O8:C><C8:M>=:8;1>>78;0>?48:M>@L882>C>842>DD83B>EJ833>G082D>H6825>I<81F>JB817>KG80G>LN808>N37OI>O:7O9?0@7NJ?1=7N;?2:7MK?3:7M8?497LE?587L2?677K??767JL?857J9?947IF?:37I3?;27H@?<17GM?=07G;?>07FH??47F6?@A7EH?AM7E9?C97DJ?D@7E0?EE7F??G07F;?H27F5?HK7F5?ID7F5?J=7F4?K67F3?KN7F2?LG7F2?M@7F2?N87F1?O17F0?OJ7EO@0C7EO
3238NO9=E8OI9=790C9=091=9<H9279<@92H9<=93<9<99419<494E9<195:9;L96L9;=98E9:K99C9;?9:?9<59;?9<D9<B9<M9=E9=69>C9<L9?=9<49@89;;9A59:B9B399I9C09919CM9889DD97C9DK9769E296J9E:96=9EA9609EH95F9FH9619GJ9659HM9689J096;9K696A9L;96G9MA96L9N@96I9O?96G:0>96D:1>96A:2=96?:3<96=:4<96::5;968:6:965:7E964:98965::K965:<>965:=K966:>M967:?N968:@O969:B096::C196;:D296=:E096A:EN96F:FL96J:GJ96N:HH973:IG977:JG974:KH971:M096N:NI96K;0A96H;2:96F;4296D;5496E;5G962;6995>;6K94I;7=945;85943;9494H;:6952;;<94K;<A94B;=G94:;>M942;@093O;A193N;B393L;C593L;D793J;E893I;F:93H;GC93C;HO93<;J:935;K2925;L391K;M491@;N5915;O790K<0690A<15908<27905<3N917<5:91L<5M91@<6@915<77904<828NM<9<8MG<:E8LA<;38KH<;F8K?<<78KH<<J8KL<=C8KE<>;8K><??8J3<@D8H><AK8FI<BE8F0<CG8EJ<DI8EG<EM8ED<FO8E@<H68E=<I<8E9<J<8DH<K28D><KI8D4<L>8CK<M?8C8<NE8B?<OJ8AF=108@M=258@4=2O8?L=3G8?E=4?8?>=588?7=608>O=6H8>H=7A8>A=898>:=938>0=9O8=>=:M8=1=;K8<I=<I8<?==H8<7=>F8;N=?E8;E=@C8;==AC8;4=BI8:J=D08:A=E88:9=F@8:0=GH89G=I089>=J8895=K@88L=LG88C=MO88:=O8881>0?87I>1G87@>2O876>4786M>5286G>5A86@>6486J>6J87=>7@880>8788D>8M897>9C89J>:98:>>:O8;0>;E8;C><98<1><M8<;>=@8<F>>68<L>?38<J>A5895>CF856>DL84G>F2848>G883I>H>839>ID82J>JJ82:>L081K>M681<>N<80M>OB80>?0H7ON?1M7O??2O7NM?3N7N:?4M7MG?5L7M4?6K7LA?7J7KN?8I7K;?9H7JH?:G7J5?;G7IC?<E7I0?=D7H=?>D7GJ??B7G7?@K7FG?B87F8?CC7EK?DG7G5?F27GG?G=7G>?H:7G;?I37G:?IL7G9?JD7G8?K=7G8?L67G7?LO7G7?MH7G6?N@7G5?O97G4@027G4
3228N39?A8NM9?28OH9>D90B9>691<9=M9259=F92O9=?93?9=<9449=794H9=395=9<O9709<?98B9<799?9<M9:;9=D9;<9>19<?9>;9=A9>A9>;9=I9?59=09@19<89@N9;?9AL9:F9BH99N9CE9959DC98=9DM97M9E597A9E<9749EC96H9EN9709G19739H39769I69799J;97?9KA97D9LG97J9MM9809O1984:00982:0O97O:1O97M:2N97J:3M97G:4M97E:5L97C:6J97D:8;97=:9M97=:;@97=:=397>:>797?:?997@:@997A:A;97B:B<97C:C=97D:D>97F:E<97J:F:97N:G8983:H6987:I498<:J398@:K198C:L298?:M698<:NO989;0H987;2A984;49982;5I981;6=97?;6O96K;7A966;8395B;8E94O;9D95B;:H95E;;N95<;=3954;>6951;?894O;@:94O;A;94M;B>94L;C?94K;DA94I;EC94I;FE94G;GM94B;I994<;JE945;L193N;LB92G;MC92=;ND921;OF91G<0E91><1D915<2B90M<3N91;<5A929<6491M<6G91A<79915<83902<9=8NM<:H8MF<;C8LG<<68L><<H8LC<==8LB<>68L;<>O8L4<@08JO<A68I;<B48H9<BM8GF<CL8G=<DO8G:<F28G7<G68G4<H=8G1<I?8FB<JA8F1<KB8E?<LD8DM<M:8DD<N18D:<NJ8CM<OO8C5=158B;=2:8AC=3<8A0=448@I=4L8@B=5E8@:=6=8@3=768?L=7N8?E=8G8?;=9D8>I=:@8>8=;=8=F=<<8====:8=5=>88<L=?78<C=@58<:=A48<1=B28;I=C28;C=D58;==E=8;4=FE8:J=GM8:A=I58:9=J<8:0=KD89G=LL89>=N4895=O<88L>0D88C>1L88:>34882>4<87I>5687A>5D87;>63875>6I87I>7?88<>8688O>8L89B>9B8:5>:88:H>:N8;<>;D8;O><;8<B>=18=5>=G8=H>>=8><>?28>F>A>8:6>CN86:>E485K>F:85;>G@84L>HF84=>IL83N>K183?>L682O>M;82?>N@820>OD81@?0J810?2080D?3780;?4:7OI?5;7O4?6;7N??7<7MJ?8<7M5?9<7L@?:<7KK?;<7K8?<;7JE?=97J2?>97I???87HL?@77H9?A67GF?BB7G7?CI7GL?E47I3?F?7HI?GI7H@?HB7H??I;7H>?J37H=?JM7H=?KE7H=?L>7H<?M77H;?MO7H:?NH7H:?OA7H:@0:7H9
3228N19@M8NK9@?8OF9@190@9?C91;9?59249>K92N9>D93C9>?9479>:94K9>695@9>29759=B98>9=F99:9><9:69?29;89?>9<;9?H9=89?>9>39>E9>M9=M9?J9=59@G9<<9AD9;D9BA9:K9C>9:29D<9999E098E9E79899E>97L9EE97@9F79819G:9849H<9879IA98=9JG98C9KM98I9M398O9N89949O>99::0A99<:1@99::2@998:3?995:4>993:5=991:6;992:79993:8H98L::C98F:<698F:=B98G:>C98H:?D98I:@E98J:AG98K:BG98L:CI98M:DJ98N:EH993:FF997:GD99<:HB99@:I@99E:J?99H:K=99M:L<99N:M=99J:O699H;0O99E;2G99B;4@99@;6399;;7198K;7E987;8797C;8H96N;9;96:;:496>;;:967;<=962;=?961;>A960;?B95O;@D95N;AF95L;BH95K;CI95J;DL95I;EM95H;FO95F;H795B;IC95;;JO954;L;94M;M293D;N3939;O492N<0492D<1392<<22923<3091K<3O91A<5C92@<6;92:<6N91N<7@91B<87915<9@900<:I8NL<<28ME<<G8M=<=98M><>18M8<>I8M1<?B8LK<@A8KM<AC8JC<B;8IO<C48I;<D28I1<E48HM<F88HJ<G=8HG<HB8H><IC8GL<JE8G;<KF8FI<LH8F7<MI8EE<NK8E3<OC8DH=098D?=1:8CJ=2@8C1=3E8B9=4A8AM=598AF=628A?=6J8A7=7C8A0=8<8@F=998@4=:58?C=;18?1=;N8>>=<L8>2==J8=I=>I8=A=?G8=8=@F8<O=AD8<F=BC8<>=CB8<9=DB8<3=EA8;N=FJ8;E=H18;==I98;4=JA8:J=KI8:A=M18:9=N98:0=OA89G>0H89>>21895>3888L>4@88C>5:88<>5I886>67880>6H883>7>88G>8589:>8K89M>9A8:A>:78;4>:M8;F>;C8<:><98<M>=08=@>=F8>4>>48>K>>?8>M>E0875>D587=>E:86N>F>86>>GD85N>HH85?>IM84O>K284?>L7840>M<83@>NA830>OE82A?0L825?2381J?3:81@?4B817?5E80G?6F801?7F7O<?8F7NG?9G7N2?:G7M=?;H7LH?<H7L3?=I7K>?>I7JH??I7J3?@I7I>?AJ7HI?BL7HB?D67J>?EA7J5?FL7IK?H17ID?HJ7IC?IC7IC?J;7IC?K47IB?KM7IA?LE7I@?M?7I@?N77I??O07I??OI7I>
3218M59BI8MO9B;8NJ9AM8OD9A>90>9A09199@A9239@392M9?I93G9?A94:9?=94O9?995C9?59799>D9899?59969?K9:29@A9;49@L9<69A39=09@:9=K9?B9>E9>K9?B9>29@?9=99A<9<@9B:9;G9C79:N9D49:69E199=9E99909E@98C9EH9879FA9939GC9969HH99;9IM99A9K399G9L899L9M>9:29ND9:89OJ9:>:0O9:D:219:E:309:B:409:@:4N9:?:5M9:@:6K9:B:7I9:B:959:=:;999O:<K99O:=N9:0:>O9:1:@09:2:A19:3:B29:4:C49:5:D59:6:E69:7:F49:;:G29:@:H09:E:HN9:I:IM9:M:JK9;1:KI9;6:LG9;;:MG9;9:O=9;6;159;4;2N9;1;3M9:B;4J9:2;5H99A;6E990;7B98@;8@980;9@97@;:D976;;F974;<H972;=J971;>K970;?M96N;@O96M;B196L;C296K;D496J;E696H;F796G;G:96F;HB96A;IN96;;K:964;LE95M;N295F;NB946;OD93K<0B93B<1A93:<2@930<3>92H<4=92?<5C92D<6B92F<7492:<7K922<8G91K<9O90F<;88OA<<88NL<=48N=<=K8N6<>D8MN<?=8MG<@58MA<A28LL<AJ8L8<BC8KE<C;8K1<D78JC<E:8J@<F=8J=<GD8J9<HE8IH<IG8I6<JI8HD<KJ8H2<LL8GA<MM8FO<NO8F==008EK=128E9=1K8DM=2E8D@=3J8CG=4N8C1=5F8BJ=6?8BB=778B;=818B2=8M8A?=9J8@N=:F8@<=;C8?J=<?8?8==<8>G=>;8>>=?98>5=@88=M=A68=D=B58=;=C38=3=D38<O=E28<I=F28<D=G28<?=H68<7=I>8;N=JE8;F=KN8;==M68;4=N=8:J=OE8:A>0M8:9>258:0>3=89G>4E89>>5>897>5M891>6;88K>6I88E>7=892>8489E>8J8:8>9@8:K>:68;?>:L8<2>;B8<E><88=9><O8=K>=38>;>>58=N>?78=B>BG88E>D287L>E887F>F?87@>GD870>HI86@>IN861>K385A>L8851>M<84B>NA842>OG83E?0O83;?26831?3=82F?4E82=?5L823?7181D?8180O?9180:?:27OE?;27NO?<27N:?=27ME?>37M0??37L:?@47KF?A47JO?B17J2?C87KC?DC7K@?EN7K7?G97JM?H97JI?I27JH?IJ7JH?JC7JG?K<7JF?L57JE?LN7JE?MF7JE?N?7JD?O87JC
3218M39D68MM9CH8NH9C98OB9BK90=9B<9179AO9219A@92L9A293F9@G94?9@?9529@<95F9@79799?M9859@C9929A999N9B09;09B99;O9B09<H9A89=C9@@9>>9?H9?;9>O9@89>69A59==9B29<D9BO9;L9CL9;39DJ9::9E;99H9EC99;9EJ99H9FI9:59GM9:99I39:?9J89:E9K>9:K9LD9;19MJ9;79NO9;<:059;B:1;9;H:2A9;N:3B9;M:4@9;M:5>9;N:6<9<0:7:9<1:899<2:9B9;M:;H9;=:=99;8:>:9;9:?;9;;:@<9;;:A=9;<:B>9;>:C?9;>:D@9;?:EB9;A:F@9;D:G?9;I:H<9;M:I:9<1:J99<6:K79<::L59<?:M39<C:N29<G:OC9<D;0I9<8;1F9;H;2G9;8;3O9:H;569:8;6=99H;7E997;8L98G;:098@;;198=;<098;;<O988;>1986;?3985;@5983;A7981;B9980;C;97N;D=97L;E?97J;FA97I;GC97G;HG97@;J3979;K@972;LM96L;N:96E;OH96><0294H<1194?<20946<2N93N<3M93E<4L93=<5K933<6H933<7A92K<8<92F<9792@<:@91;<;B90A<<>903<=>8OC<=M8O<<>F8O5<??8NM<@88NF<A?8N:<B18MN<BJ8M:<CC8LG<D=8L6<E@8L3<FD8L0<GH8KC<HI8K2<IK8J@<JL8IN<KN8I<<LO8HJ<N18H8<O38GG=048G5=168FB=278F1=398E?=438E3=558D==638CN=6L8CG=7E8C==8B8BK=9?8B9=:;8AG=;88A6=<48@C==18@1==N8?@=>K8?3=?J8>K=@H8>A=AF8>8=BE8>0=CD8=I=DD8=D=EC8=?=FB8=9=GB8=4=HB8<O=IB8<I=JK8<@=L28<7=M:8;N=NB8;F=OJ8;=>128;4>2:8:J>3B8:B>4I8:9>5B8:1>6089K>6?89F>6M89@>7<89=>838:0>8I8:C>9?8;7>:58;J>:K8<=>;A8=0>;G8=I><I8==>=J8=0>>L8<D>@I8;1>BH88B>CO88;>E5885>F<87N>GC87G>HI87@>IO872>K386B>L8862>M=85B>NC855>OJ84K?1284A?29847?3@83M?4H83B?5O839?7782O?8;82A?9<81L?:<817?;=80A?<=7OM?=>7O8?>>7NB??=7MK?@;7LN?A87L1?B;7L9?CE7LL?E07LB?F<7L8?GF7KO?H@7KN?I:7KM?J27KL?JK7KK?KD7KK?L<7KK?M57KJ?MN7KI?NG7KH
3218M19ED8MK9E58NF9DG8O@9D890;9CJ9159C<91O9BM92J9B?93D9B094>9AD9559A?95I9A:9759A;9819B198M9BH99J9C=9:L9CE9;G9BM9<A9B59=:9A=9>69@D9?39?K9@19?39@N9>:9AK9=A9BH9<I9CE9<09DB9;79E>9:?9EE9:29F09;49G39;89H99;=9I?9;C9JD9;I9KJ9;O9M09<49N59<:9O;9<@:0A9<G:1G9<N:2M9=6:419=<:509=@:5O9=C:6M9=D:7K9=E:8J9=F:9H9=H:;@9=>:=<9=3:>L9<G:?G9<C:@H9<D:AI9<F:BJ9<G:CK9<G:DL9<I:EM9<J:FL9<M:GK9=1:HH9=6:IG9=::JE9=?:KC9=C:LA9=H:M?9=L:N@9=L:OE9=@;0M9=0;249<?;3;9;O;4C9;?;5J9:O;729:>;8999O;9<99J;:<99H;;>99E;<>99C;=<99A;>;99>;?:99<;@<999;A>998;BA996;CB994;DE993;EF991;FI98O;GJ98M;HN98F;J198;;K5981;L997G;MC97@;O0979<0=972<10961<2095G<3095<<40952<5294F<67946<6M93F<8293@<8L93:<9G934<:L927<;K91H<<N917<=E90K<=I90H<>B90@<?;908<@?8ON<B38O?<BA8O9<C18O0<CJ8N=<DB8MI<EE8MF<FJ8M?<GL8LM<HM8L;<IO8KI<K08K8<L28JF<M38J4<N58IB<O68I0=088H>=1:8GL=2;8G;=3=8FI=4>8F7=5@8EE=6?8E3=7:8DH=878D6=938CD=:08C2=:L8BA=;I8AO=<E8A<==B8@K=>?8@9=?;8?H=@:8??=A88?7=B78>N=C58>D=D58>>=E48>9=F38>4=G38=N=H38=J=I28=E=J28=?=K18=:=L78=2=M?8<I=NG8<@=OO8<7>168;N>2>8;F>3F8;=>4N8;4>5E8:M>648:G>6C8:A>718:;>7?8:4>828:;>8H8:N>9>8;A>:48<5>:J8<H>;<8<L><>8<?>=@8<3>>B8;F>?C8;:>B788H>BE890>CL88J>E388C>F988<>G@886>HG87O>IM87H>K487A>L9873>M?86F>NF86;>OM861?1585G?2<85=?3C853?4K84I?6284??7:845?8A83K?9G83??:G82I?;G824?<H81??=G80G?>D7OJ??B7NM?@@7N1?A=7M4?BG7N7?D37MN?E=7MD?FI7M:?H07M3?HI7M2?IA7M1?J:7M1?K37M0?KL7M0?LD7LO?M=7LN?N67LN
3218LO9G08MI9FB8ND9F38O>9EE9099E79139DH91M9D;92H9CL93B9C>94<9BO9579BB95M9B<9709BI97L9C@98H9D699F9DK9:D9DB9;?9CJ9<99C29=39B99=O9AA9>L9@H9?I9@09@F9?79AC9>>9BA9=F9C>9<M9D;9<49E89;;9EG9:J9F99<59G?9<;9HE9<A9IJ9<H9K19=09L79=79M=9=?9NC9=F9OI9=N:0O9>6:259>>:3;9>E:4>9>K:5=9>N:6=9?2:7<9?5:8;9?8:9:9?;:::9?>:;99?A:<L9?::>E9?2:@=9>H:AC9>9:B59=O:C69=O:D79>0:E89>2:F99>2:G89>6:H79>::I49>?:J39>C:K19>H:L09>K:M39>H:N49>G:O29>G;0:9>7;1A9=G;2H9=6;3O9<G;579<7;6>9;F;7F9;6;8G9;4;9H9;1;:I9;0;;J9:N;<K9:K;=K9:I;>I9:F;?G9:C;@E9:A;AE9:>;BH9:<;CJ9:;;DL9:9;EN9:7;G09:5;H29:4;I599K;J999A;K=997;L@98M;MD98C;NH989;OL97O<1097B<26972<3;96B<4@962<5E95B<6K952<6B94><7G947<8M940<:293I<;;93:<<>92J<=A929<=D925<=F922<>>91K<?791D<A1913<BG90D<C590><CC908<D1902<DI8O?<EL8O9<FM8NI<GO8N7<I18ME<J28M3<K48LA<L68KO<M78K><N98JL<O:8J9=0<8IH=1=8I5=2?8HA=3A8GM=4C8G8=5E8FE=6H8F4=7J8EC=8H8E0=9E8D==:A8CL=;>8C:=<:8BH==78B6=>48AD=?08A3=?M8@@=@J8@4=AH8?K=BG8?B=CE8?:=DE8?4=ED8>O=FD8>J=GD8>D=HD8>?=IC8>:=JB8>5=KB8=O=LB8=J=MD8=D=NK8=;>038=2>1;8<I>2C8<@>3K8<7>538;N>5J8;G>688;A>6F8;;>758;6>7C8;0>828:J>8G8;:>9=8;M>:08<9>;28;M><38;@>=58;4>>78:G>?98:;>@:89N>A=89H>BE89B>CM89>>E4898>F<894>GD88O>HL88J>J488E>K;88@>LC88;>MI880>O087F?0787;?1=870?2C86F?3I86;?4O861?6685F?7=85;?8D851?9K84G?;184;?<183B?<N82F?=L81I?>I80L??G800?@E7O3?AI7OC?C47O9?D@7NO?EJ7NF?G67N<?H77N7?I07N7?II7N6?JB7N6?K;7N5?L37N4?LL7N3?ME7N3?N>7N3
3218LM9H>8MG9GO8NB9GA8O<9G39069FD9119F691K9EG92F9E993@9DK94:9D<9559CN95O9CB96L9D897H9DN98D9ED99B9F79:<9E?9;69DG9<19CN9<K9C79=H9B>9>E9AE9?A9@M9@?9@49A<9?;9B99>B9C69=I9D49=19E19<89EK9<29FF9=@9GL9=H9I39=O9J99>79K?9>?9LE9>F9MK9>O9O19?6:079?>:1>9?E:2D9?M:3J9@4:4K9@9:5K9@=:6J9@@:7I9@D:8I9@G:9H9@J::G9@M:;E9@K:<N9@F:>>9@I:@09@L:AF9@G:C?9@>:D:9?L:DC9?::ED9?;:FE9?;:GE9?>:HC9?C:I@9?H:JB9?H:KE9?E:LH9?B:MG9?C:NE9?C:OF9?>;0N9>O;259>>;3<9=N;4D9=>;5K9<M;729<@;829<=;949<<;:59<:;;59<7;<79<6;=89<4;>89<1;?79;N;@59;L;A39;I;B29;F;C09;C;D19;A;E39;?;F59;>;G79;<;H99;:;I=9;1;J@9:F;KD9:<;LH9:2;ML99H;O399<<0;98N<1D98@<2I980<3O97@<54970<6996@<7>960<66955<7;94O<8A94G<:294B<;L94<<=193L<=@93C<=B93@<=E93=<>:937<?F92J<AB929<C<91H<CI91C<D791><DD918<E2912<F090D<G1902<H38O><I58NJ<J78N6<K98MB<L;8LN<M<8L9<N>8KE<O@8K1=0B8J==1D8II=2F8I4=3G8H@=4I8GL=5M8G;=758G4=888FF=948EN=:48E:=;28DE=;O8D4=<K8CA==H8BO=>E8B>=?A8AL=@>8A:=A:8@I=B98@@=C78@7=D68?O=E68?J=F58?D=G58??=H48?:=I48?4=J38>O=K38>J=L38>E=M38>@=N28>;=O18>5>088=M>1@8=D>2H8=;>408=2>578<I>5M8<C>6<8<<>6J8<6>788<0>7G8;J>868;D>8F8;D>9E8;<>:G8;>>;G8;A><I8;C>=I8;F>>J8;I>?K8;L>@N8;H>B48;:>C:8;6>DA8;0>EI8:L>G18:G>H98:B>I@8:=>JH8:8>L08:3>M789L>N>89A>OD897?0K88L?2188B?37887?4=87L?5D87B?6K877?8186L?9786A?:=865?;985;?<684B?=383H?>182L?>N81O??L812?@L80G?B780E?CA80;?DM801?F77OH?GC7O??H@7O<?I87O<?J17O;?JJ7O:?KB7O9?L;7O9?M47O8?MM7O8
3218LK9IJ8ME9I<8N@9HN8O:9H?9049H190O9GB91I9G492D9FF93>9F89489EJ9539E;95@9D>95L9E:96N9F097O9FB9959FH9:49F<9:N9ED9;I9DL9<C9D49=@9C;9>=9BC9?:9AI9@89A09A59@89B29?A9C09>K9CM9>69DK9=@9F19=<9FL9>H9H;9?79IA9?>9JG9?G9KM9?N9M49@69N99@>9O@9@E:0F9@M:1L9A4:329A<:489AD:589AH:689AK:779AO:879B2:969B6::59B9:;49B::<09AK:=79AD:>I9AG:@99AJ:AJ9AM:C<9B0:DL9B3:F@9B3:G19A>:G19@E:H19@G:I39@E:J79@B:K:9@@:L=9@>:M:9@>:N99@?:O69@?;0:9@6;1B9?E;2I9?5;409>E;589>5;6=9=J;7>9=H;8?9=F;9@9=D;:A9=B;;B9=@;<C9=>;=E9=<;>E9=9;?D9=7;@C9=4;AA9=1;B@9<N;C>9<L;D=9<I;E;9<F;F=9<D;G>9<B;HA9<@;ID9<6;JH9;L;KL9;B;M49;4;N<9:G;OE9:9<0N99K<2699<<3<98L<4C98=<5H97M<6M97=<8396M<5J95M<7095E<8:95?<:395:<;L955<==950<=?94N<=A94K<=C94H<>;94@<@7940<B393><CO92M<D=92F<DH92<<E3921<EA91G<F691:<G790E<H:902<I;8O=<J=8NI<K?8N5<LA8MA<MB8LM<NE8L8<OF8KD=0H8K0=1J8J<=2L8IG=3N8I4=528HD=698H<=7A8H4=8H8GM=9?8FM=:?8F9=;>8ED=<>8E0===8D;=>98CI=?68C7=@38BE=@O8B3=AL8AB=BI8A5=CH8@L=DG8@D=EF8@?=FE8@:=GE8@5=HE8@0=IE8?J=JD8?E=KD8?@=LC8?:=MC8?5=NB8?0=OB8>K>0B8>E>1D8>>>2L8>6>448=M>5<8=D>618==>6@8=7>6N8=1>7<8<K>7K8<F>8A8=0>988=>>:78=B>;88=E><98=H>=:8=K>>;8=M>?<8>0>@?8=K>AG8=:>BK8<N>CN8<I>E68<D>F>8<?>GF8<:>HM8<5>J58<0>K=8;K>LE8;F>MK8;=>O28;3?098:H?1?8:>?2E8:3?3K89H?5289>?68892?7?88H?8E88=?9F87K?:C870?;?866?<<85>?=984E?>683L??3831?@1824?A9820?BD81G?CO81=?E:814?FE80K?GN80A?HG80A?I@80@?J980??K280??KJ80>?LC80>?M<80=
3218LI9K88MD9JJ8N>9J;8O89IM9039I>90M9I091G9HB92B9H393<9GE9479G69519FH9509EA94F9E695F9EF96L9EJ9829EK9989EM9:>9F09;A9EI9<;9E19=89D89>69CA9?39BL9@19B69@N9A@9AL9@J9BJ9@49CG9?>9DH9?59F59>L9G09?O9HB9@@9IO9@N9K69A69L;9A=9MB9AE9NH9AM9ON9B4:149B<:2:9BD:3A9BL:4F9C3:5F9C6:6E9C::7D9C=:8D9CA:9C9CD::C9CH:;?9C9:<<9BJ:=A9BC:?29BF:@C9BI:B49BL:CE9BO:E69C2:FG9C5:GO9C8:GL9B?:H99AI:IE9AA:JG9A?:KJ9A=:LM9A::ML9A::NJ9A;:OH9A;;0N9@M;269@=;3=9?M;4E9?<;5I9?4;6I9?2;7K9?0;8K9>N;9M9>K;:N9>J;;O9>H;=09>E;>19>D;?29>B;@39>?;A19><;B09>:;BN9>7;CL9>4;DJ9>1;EI9=N;FG9=L;GF9=I;HH9=F;IL9=;;K59<M;L=9<@;MF9<1;NN9;D<079;6<1@9:H<2H9::<3O99J<5599:<6<98J<7A98:<8F97J<5>96D<6D96=<8:968<:4962<<695N<=F95K<=E95L<=E95M<=E95N<>L95?<@G94I<BC944<D:93@<DE936<E092K<E<92A<F0925<FD91H<G>919<H@90E<IB900<JC8O<<KE8NH<LG8N4<MI8M@<NK8LK<OL8L8=0N8KC=208JO=328J;=468IK=5>8ID=6E8I==7M8I5=958HM=9O8H3=:J8G8=;I8FC=<I8EO==H8E:=>G8DF=?G8D1=@D8C>=A@8BM=B=8B;=C:8AI=D88AA=E78A:=F78A5=G68A0=H68@J=I68@E=J58@@=K58@:=L48@5=M48@0=N38?J=O38?E>038?A>128?;>228?6>318?1>498>H>5A8>>>658>8>6C8>2>728=L>7H8>7>8?8>D>968?2>9M8?A>:I8?J>;J8?L><K8?O>=K8@2>>M8@4>@08?N>A88?>>B?8>L>CD8>A>DD8><>EK8>7>G38>2>H;8=M>IC8=H>JJ8=C>L28=?>M:8=9>N@8<O>OF8<D?0M8<9?238;O?398;D?4?8;9?5F8:N?6M8:D?838:9?9089@?9M88E?:I87J?;F871?<B869?=?85@?><84H??983O?@;83<?AF833?C182I?D<82@?EG826?G281L?H681F?HO81E?IH81E?J@81D?K981D?L281C?LK81B
3218LG9LE8MB9L68N<9KH8O79K99019JK90K9J=91F9IN92@9I@93:9I29459HD94E9H294?9FE94;9F294J9EG9609EJ9779EM98=9F099C9F39:K9F69<59F59=29E@9>09DJ9>M9D49?K9C?9@H9BI9AF9B39BD9A=9C@9@K9DL9@E9F99@<9G99A29HG9AG9J79B99KD9BE9LJ9BL9N09C59O69C<:0<9CD:1C9CK:2I9D3:3O9D;:549DA:639DE:729DH:829DL:919DO::09E3::N9DH:;J9D9:<G9CJ:=K9CB:?<9CE:@M9CG:B>9CJ:CO9CM:E@9D0:G19D3:GK9D7:H19CH:H79C1:IJ9BE:K79B<:L:9B9:M=9B7:N>9B6:O;9B6;0;9B4;1B9AD;2J9A4;419@D;549@>;659@<;769@:;879@7;999@6;:99@4;;;9@1;<;9@0;=<9?N;>>9?K;?>9?J;@@9?H;A?9?E;B=9?B;C<9??;D:9?<;E89?:;F69?7;G59?4;H39?1;H@9>?;J;9>7;KG9=J;M09=<;N89<O;OA9<@<0I9<3<229;E<3:9;8<4A9:G<5H9:7<6N99G<85997<8I98D<5397;<6A975<8?971<:C96M<<F96H<=N96G<=N96G<=N96H<>B96=<??95L<A7957<C294B<DB940<DM93E<E993;<EL92O<F@92C<G3927<GH91K<HF918<IH90D<JJ8OO<KL8O<<LM8NG<MO8N3<O18M?=038LK=158L6=268KB=3;8K3=4B8JL=5J8JD=728J==898J6=9A8IN=:A8I;=;58H8=<48GB==48FN=>38F9=?28EE=@28E0=A18D<=B18CG=BN8C5=CK8BB=DI8B5=EH8AO=FG8AJ=GG8AE=HG8A?=IF8A:=JE8A5=KE8A0=LE8@K=ME8@F=ND8@@=OD8@;>0C8@6>1C8@0>2B8?K>3B8?F>4B8?A>5E8?:>698?3>6O8?>>7F8?L>8=8@:>958@G>9L8A5>:C8AB>;:8B0><;8B3>=<8B6>>=8B8>?A8B2>@I8AA>B08A0>C88@?>D;8@4>E<8?O>F@8?J>GH8?E>HO8?A>J78?;>K?8?7>LG8?1>MN8>K>O48>@?0;8>5?1A8=K?2G8=@?3N8=5?548<J?6:8<??7>8<0?8:8;5?978:;?:389??:O88E?;L87L?<I873?=F86;?>B85C???84K?@H84>?B3845?C>83K?DI83B?F4838?G?82N?H>82J?I782J?J082I?JI82I?KA82H?L:82G?M282G
3218LF9N18M@9MD8N:9M58O59LG8OO9L990J9KJ91D9K<92>9JM9399J?9439J193N9I;93N9GI9409FN9459FF95<9FI96B9FL97H9FO9959G29:I9G59<19G89<L9FI9=J9F39>G9E=9?E9DG9@C9D29A@9C<9B>9BF9CB9B=9DO9B59F=9AL9GC9B29HL9BO9J<9C@9KN9D19M89D<9N>9DD9OE9DK:0K9E3:219E::379EB:4=9EJ:5A9F0:6@9F3:7@9F7:8?9F::9>9F>::<9F7:;99EH:<59E8:=29DI:>49D@:?F9DC:A79DE:BG9DH:D99DK:EJ9DN:G39E1:GH9E6:H59E1:H79D8:IA9D1:K:9CA:LJ9C6:MM9C4:NO9C2:OM9C2;0O9BL;269B<;3>9AK;4?9AH;5A9AF;6B9AC;7C9AB;8D9A@;9E9A=;:F9A<;;G9A:;<H9A7;=I9A6;>J9A4;?K9A1;@L9A0;AM9@M;BK9@J;CI9@G;DH9@E;EF9@B;FE9@?;G89@4;G:9?>;G?9>I;I:9>B;K69>;;M19>4;NJ9=L<039=><1<9=0<2D9<B<3M9<4<539;D<6:9;4<7@9:E<8G9:4<8>99?<4M986<6L97O<8O97K<;397G<=697C<>797A<>797C<>G97:<?D96H<@A968<AF95E<CA950<DK94?<E6945<EH93I<F;93=<FO930<GC92D<H7928<HL91K<IN917<K090B<L28OO<M48O:<N68NF<O78N2=098M>=1;8LJ=2@8L;=3G8L4=4O8KL=668KE=7>8K==8E8K6=9M8JN=;28JC=;G8I@=<@8HA==?8GM=>>8G8=?>8FD=@=8EO=A<8E;=B<8DG=C;8D2=D;8C==E98BK=F98BE=G88B@=H78B;=I78B5=J78B0=K68AL=L58AG=M58AC=N48A?=O48A;>038A7>128A3>228A0>318@K>418@G>508@C>5O8@@>6L8@G>7D8A<>8<8AO>938B<>9J8BJ>:B8C8>;98CE><08D3><M8D:>=N8D=>?28D5>@98CD>AA8C3>BH8BB>D08B1>E38AG>F48AB>G58A=>H<8A9>ID8A3>JL8@O>L48@I>M;8@E>NB8@<>OH8@1?0O8?G?258?;?3<8?1?4B8>F?5H8>;?6G8=E?7D8<J?8@8<0?9=8;4?:98::?;589??<288G?<O87O?=L876?>I86=??L85K?A585@?B@857?CK84M?E684C?FA84:?GL840?HF83O?I?83O?J783N?K083M?KI83M?LB83L
3218LD9O?8M>9O18N99NB8O39N48OM9ME90H9M791B9LH92=9L:9379KL9419K>9389JB93=9HL93D9GJ93@9GD94G9GG95M9GJ97?9GN9939H19:G9H49;N9H79<F9H19=D9G<9>A9FF9??9F09@=9E:9A:9DD9B89D69CE9CM9E39CD9FA9C<9GN9C29I09D79JA9DG9L29E89MD9EI9NL9F3:039F::199FC:2?9FJ:3F9G2:4K9G9:5O9G?:6N9GB:7M9GF:8L9GI:9J9GF::G9G6:;D9FG:<@9F8:==9EI:>>9E>:?O9EA:A@9ED:C29EG:DB9EJ:F39EM:G09F0:GE9F5:H99F9:H79E>:I>9E9:K19DM:LK9D=:N=9D1:O@9CN;0?9CN;1C9CC;2J9C4;3K9C2;4L9C0;5M9BN;6O9BL;7O9BJ;909BH;:29BF;;29BD;<49BB;=49B?;>69B>;?79B<;@89B9;A99B8;B:9B6;C99B3;D89B0;E69AM;F49AJ;F;9A7;F=9@B;F@9?L;G59?;;H@9?4;J89>J;L19>?;MK9>8;OG9>0<1B9=I<369=@<4?9=1<5E9<B<6I9<3<7H9;G<8G9;<<6H9:=<6<99D<7L993<9?98F<;C98B<=F98=<>@98=<>M986<?J97E<@G974<AD96C<BA961<D195?<E394O<EC94C<F7947<FK93K<G?93?<H3933<HF92F<I:92:<J491J<K6916<L890B<M:8ON<N<8O9<O=8NE=0@8N1=1D8MC=2L8M;=438M4=5;8LL=6B8LE=7J8L>=928L6=:98KO=;A8KG=<88JH=<M8IC==J8HM=>I8H8=?I8GC=@H8FO=AG8F:=BG8EF=CF8E1=DF8D==EE8CK=FE8D1=GD8CM=HD8CH=IC8CD=JB8CA=KB8C==LA8C8=M@8C5=N@8C1=O?8BM>0>8BH>1>8BE>2=8BA>3<8B=>4<8B8>5;8B5>6<8AL>7=8AI>858B>>8M8C2>9E8CG>:>8D=>;68E2>;N8EF><F8F6>=>8FA>>D8F8>?K8EG>A28E6>B98DE>CA8D4>DH8CC>EK8C:>FL8C5>GM8C1>I18BK>J98BG>K@8BB>LH8B=>N08B8>O68AM?0=8AB?1C8A7?2J8@M?408@B?558@5?618?:?6N8>??7J8=E?8F8<J?9C8;O?:?8;4?;<8:;?<889B?=588J?>2881??087:?@886M?AB86B?BM869?D985O?EC85E?FO85<?H5854?HN854?IF853?J?852?K8852?L1851
3218LB:0L8M<:0=8N7:008O19OA8OK9O390F9ND91@9N692:9MH9359M993>9LF92B9KJ92M9J09359HJ92L9HC9459HF95I9HI97=9HM9929I09:E9I39;K9I69<A9I99=>9HE9>;9GO9?99G99@79FC9A49EM9B<9EF9CI9E=9E79E49FD9DK9H29DB9I59E>9JG9EO9L79F@9MI9G09O:9GA:0A9GJ:1G9H2:2M9H9:449HA:5:9HH:6<9HN:7;9I1:8;9I4:999I4::69HE:;29H6:;N9GG:<K9G8:=G9FH:>H9F=:@99F@:AJ9FC:C;9FF:DL9FH:F89FL:FL9G0:GA9G4:H59G8:H89FE:I>9F@:JI9F9:LB9EI:N;9E9:OO9DK;109DK;269D>;379D<;489D:;599D8;6:9D6;7;9D4;8<9D2;9=9D0;:>9CN;;?9CK;<@9CJ;=A9CH;>B9CE;?D9CD;@D9CB;AE9C?;BG9C>;CG9C<;DF9C8;E<9C1;E>9B;;EL9AH;F<9A6;FO9@G;H19@7;HJ9@2;J29?M;KK9?C;MC9?9;O;9>O<139>E<2L9>;<4?9>2<5I9=H<6G9=<<7F9=1<8E9<E<6<9;B<7K9;2<9;9:B<:K9:1<<;99A<>6998<?2993<?O98A<@M980<AI97?<BF96N<CD96=<DA95L<E?95><F3951<FG94E<G:949<GN93L<HB93@<I6934<IJ92H<J>92;<K<91I<L>915<M@90A<NB8OL<OD8O9=0I8NK=208NC=388N<=4?8N4=5G8MM=6O8ME=868M>=9>8M7=:E8LO=;M8LH=<J8L0==>8JL=>58IL=?58I7=@48HC=A38GN=B38G9=C28FE=D28F0=E18E<=F08E0=G08E:=GO8E>=HO8E:=IN8E6=JN8E2=KM8DN=LM8DJ=ML8DF=NK8DB=OK8D>>0J8D:>1I8D6>2H8D2>3H8CN>4G8CJ>5G8CF>6J8C9>7O8BI>8G8C>>9?8D4>:78DI>:O8E=>;G8F3><@8FH>=88G=>>48GK>?<8GK>@C8G9>AJ8FI>C28F7>D98EF>EA8E5>FC8DM>GD8DH>HD8DD>IF8D?>JM8D:>L58D5>M=8D0>ND8CH>OJ8C>?118C3?278BH?3>8B>?4>8AJ?5;8A0?678@5?748?:?808>??8L8=D?9I8<I?:E8;N?;B8;6?<?8:=?=;89E?>888L??=88<?@E87N?B087D?C:87:?DF871?F086G?G<86>?H<869?I6868?IN868?JG867?K@867
3218L@:298M::1K8N5:1<8NO:0N8OI:0@90D:0191>9OC9289O49339NF92H9MN91K9M292;9K392E9IJ92@9ID9439IE95G9IH97;9IL98O9IO9:C9J29;H9J59<>9J89=89IM9>69I89?39HB9@19GL9A29G>9B@9G69CM9FL9E;9FC9FH9F:9H59F19I:9FF9JK9G69L<9GG9MN9H89O?9HH:0O9I9:269IA:3<9II:4B9J1:5H9J8:6I9J=:7H9J@:8H9JC:9D9J4::A9IE:;=9I5:<99HF:=69H7:>29GH:?29G;:@C9G>:B49GA:CE9GD:E69GG:F49GJ:FI9GO:G>9H2:H29H7:H<9GN:I>9GF:JE9GB:L99G5:N39FE:OK9F5;1@9EF;2B9ED;3C9EB;4D9E@;5E9E>;6F9E<;7G9E:;8H9E8;9I9E7;:J9E4;;K9E2;<L9E1;=M9DN;>N9DM;@09DK;A19DI;B29DG;C39DE;D49DC;DJ9D6;E99CE;EI9C2;F99B@;FH9AN;H29B6;HN9A2;IF9@L;J?9@K;KE9@F;M=9@<;O59@2<0M9?H<2=9??<3K9?8<5=9>O<6E9>E<7D9>:<709=D<7K9=0<9;9<@<:J9<0<<:9;?<=F9:N<>F9:><?E99L<@A99<<A>98L<B:98;<C797K<D497:<E696G<EL965<FA95C<G6953<GJ94G<H>94:<I293N<IF93B<J:936<JM92J<KC92<<LE91H<MF914<NH90@<OM902=158OK=2=8OC=3D8O<=4L8O5=638NM=7;8NF=8B8N>=9J8N7=;18MO=<98MH==;8M8=>08L4=>D8K0=?@8J6=@?8IB=A>8HM=B>8H9=C=8GD=D=8FO=E<8F;=F<8F6=G;8F?=H;8FI=I:8FL=J:8FG=K98FC=L88F?=M88F<=N78F7=O78F3>068EO>158EL>258EG>348EC>438E@>538E;>628E7>778DE>8?8CN>988D@>:08E4>:H8EI>;@8F?><88G4>=18GH>=I8H>>>M8HB>@48HB>A<8HB>BC8H;>CJ8GI>E18G8>F98FH>G;8F@>H<8F<>I<8F6>J=8F2>KB8EM>LJ8EH>N28EC>O88E:?0?8DO?1E8DD?2L8D:?3H8C@?4E8BE?5A8AJ?6=8@O?7:8@4?868?9?928>>?9O8=D?:K8<I?;H8<0?<E8;8?=B8:@?>B89J??J89=?A2890?B=88F?CG88<?E3883?F>87I?GI87@?HE87>?I=87=?J687=?JO87<
3218L>:3F8M8:378N3:2I8NM:2;8OG:1M90B:1?91<:10926:0B92M:039219O69159N:91K9L792=9K492C9K39439JJ95E9JH9799JK98M9JN9:A9K19;D9K49<:9K79=29K69>09J@9>M9IJ9?K9I49A69HN9BD9HE9D19H<9E>9H39FL9GJ9H99GA9I?9GM9K09H>9LA9HN9N29I?9OD9J1:159JA:2D9K0:3J9K8:509K@:679KG:769KK:869KO:939KC:9O9K3::L9JD:;H9J5:<E9IF:=A9I7:>=9HH:?<9H9:@M9H<:B=9H?:CO9HB:E=9HE:F19HI:FE9HN:G:9I1:GO9I6:HA9I8:I>9HM:JE9HH:L19HA:MI9H1:OE9G>;1E9FG;2L9F?;3I9F=;4J9F:;5K9F9;6L9F7;7M9F5;8N9F3;9O9F1;;09EO;<19EM;=29EK;>39EI;?49EG;@69EE;A79ED;B89EA;C99E@;D:9E=;E19DN;EE9D<;F59CJ;FE9C8;GI9C;;I59CB;IJ9AM;JB9AE;K;9AD;L39AC;M79A@;NO9A5<0;9@M<1H9@E<369@><4I9@5<6<9?N<7B9?C<7J9>O<9:9>?<:C9=N<;C9=<<<D9<K<=D9<:<>E9;I<?E9;8<@C9:G<A@9:6<B=99E<C@993<DA98?<EA97L<F697:<FK96H<G@966<H695E<HK952<I@94A<J5940<JI93C<K=937<L192K<LK92;<ML91H<O291;=0:913=1A90K=2I90D=4090<=58905=6?8ON=7G8OF=8N8O?=:68O7=;>8O0=<E8NH==M8N@=>A8M<=?68L8=?K8K6=@J8JA=AI8IM=BI8I8=CH8HD=DH8GN=EG8G:=FG8G<=GF8GE=HF8GN=IF8H8=JE8H8=KE8H4=LD8H1=MC8GM=NB8GH=OB8GE>0A8GA>1A8G=>2@8G8>3?8G5>4?8G1>5>8FM>6>8FI>7E8F2>8L8E;>9I8E@>:B8F6>;:8FK><28G?><J8H4>=B8HJ>>>8I9>?F8I9>@L8I9>B48I9>C;8I9>DB8I9>EJ8HJ>G18H:>H38H3>I38GN>J48GJ>K58GE>L78G@>M?8G;>NF8G5>OL8FK?138F@?268EO?328E5?3N8D:?4K8C??5G8BD?6C8AI?7@8@O?8<8@3?988?9?:58>>?;18=D?;O8<L?<K8<3?=H8;;?>N8:L?@68:>?A>8:1?BJ89H?D489>?E@895?FK88L?H388C?HM88B?IE88B?J>88A?K788@
3218L<:538M6:4E8N1:478NK:3H8OE:3:90@:2K91::2=924:1O927:1:91;:0>90?9OB91:9M;9299LC92E9LA9459L895E9KN9779KJ98K9KM9:?9L09;A9L39<79L69<M9L99=J9KI9>G9K39?M9JG9A:9J=9BG9J49D59IK9EB9IC9G09I:9H=9I19IF9I29K59IE9LF9J69N79JG9OI9K8:1:9KI:2K9L9:499LG:5>9LO:6D9M6:7C9M::8A9M2:9>9LB:::9L3:;79KD:<39K4:=09JE:=L9J6:>I9IG:?E9I8:A69I;:BG9I>:D89IA:E99ID:EN9IH:FB9IM:G69J1:GK9J5:H?9J9:I>9J4:JE9IO:KL9IJ:MC9I;:OC9HC;1B9GL;3<9G9;479G7;539G5;619G3;729G1;839G0;949FM;:59FL;;69FI;<79FG;=89FF;>99FC;?:9FB;@<9F@;A=9F>;B>9F<;C?9F:;D=9F4;E49ED;EK9E3;FA9DA;G@9D?;HM9DK;IO9D>;J=9BL;K49BB;KL9BA;LD9B@;M>9B>;N:9B;;OF9B3<129AK<2B9AC<459A;<5H9A4<7@9@H<8D9@:<9D9?I<:E9?8<;E9>F<<F9>5<=F9=D<>G9=3<?G9<A<@H9<1<AJ9;=<BK9:J<CM9:7<DO99C<EJ991<F?98?<G597M<GJ97;<H?96I<I4968<IJ95E<J?954<K494B<KJ940<L?93><M492M<N792B<O>92:=0F923=1M91L=3591D=4<91==5D915=6L90N=8390F=9;90?=:B908=;J900==28OI=>98OA=?38ND=?G8MA=@<8L==A58K@=B58JL=C48J7=D48IC=E38HN=F28H:=G28HA=H28HJ=I18I4=J18I==K18IG=L08IF=M08IB=MO8I>=NN8I:=OM8I6>0M8I2>1L8HN>2K8HJ>3K8HF>4J8HB>5J8H>>6K8H5>828G>>998FG>:;8FB>;38G6>;K8GK><C8HA>=;8I6>>38IJ>?78J0>@>8J0>AE8J0>BL8J0>D48J0>E;8J0>FB8J0>GJ8IL>HK8IG>IL8IA>JL8I=>KM8I8>LN8I3>N38HN>O:8HG?0A8H<?1?8GE?2<8FJ?388EO?458E4?518D:?5M8C>?6J8BD?7F8AI?8B8@N?9?8@3?:;8?8?;88>??<58=G?=18<N?>38<:??;8;M?@C8;@?AL8;4?C78:J?DB8:A?EM8:7?G889N?H;89H?I489G?IM89F?JF89F
3218L::6@8M4:628MO:5C8NI:558OC:4G90>:49918:3J923:3<91A:2B90E:1E8OI:0I90F9NM9259N392H9MO9489MF95H9M<9789M398I9LL9:=9LO9;>9M29<49M59<J9M89=D9M29>C9L?9@09L69A=9KM9BK9KD9D99K;9EF9K29G39JJ9HA9J@9IN9J79K:9JM9LK9K=9N<9KO9ON9L?:1?9M0:309MA:4A9N1:5M9N>:729NE:809NA:8L9N1:9I9MB::E9M2:;B9LC:<>9L4:=;9KE:>79K6:?49JG:@09J7:A@9J9:C19J<:DA9J?:E69JD:EJ9JG:F>9JK:G39K0:GH9K3:HD9K6:IM9K0:K59JJ:L=9JD:MA9J@:O@9IH;1@9I1;3?9H;;4G9H2;5B9GO;6>9GN;799GL;899GJ;9:9GH;:;9GF;;<9GD;<=9GB;=?9G@;>?9G>;?@9G<;@B9G:;AC9G9;BD9G6;CE9G5;D@9FJ;E79F:;EN9EI;FF9E;;G@9E4;HH9E;;J09EB;J=9D0;K49CE;KL9CD;LD9CC;MJ9C3;NH9BN;O>9C2<0=9C1<1N9BI<469B6<6>9AC<7B9AA<8I9A=<:29@M<;39@;<<39?K<=49?9<>49>H<?59>6<@59=F<A69=4<B79<A<C99;N<D;9;;<E=9:H<F49:5<FI99D<G?992<H498@<HI97N<I>97<<J496K<JI968<K>95G<L4955<LI94C<M?943<NI93F<OJ93;=12933=2992L=3A92D=4I92==60926=7891N=8?91G=9G91?=:O918=<6910==>90I=>F90B=?D8OL=@98NI=@M8ME=AA8LA=B@8KK=C?8K6=D?8JB=E>8IM=F>8I>=G=8IG=H=8J0=I=8J9=J<8JC=K<8JL=L;8K5=M;8K3=N:8JO=O:8JL>098JG>188JC>288J@>378J<>468J7>558J3>658J0>788IA>8@8HJ>9G8H4>:L8GB>;D8H8><=8HM>=58IA>=M8J6>>H8JG>?O8JG>A78JG>B>8JG>CE8JG>DL8JG>F48JG>G88JC>H58J3>I88J7>J=8J@>KB8JI>LE8JK>ME8JF>NH8JA>OM8J5?0I8I:?1F8H??2B8GE?3>8FI?4;8EO?578E3?638D9?708C>?7L8BC?8I8AI?9E8@M?:A8@3?;>8?:?<;8>A?=88=I?>@8=<??H8<O?A08<B?B98<6?CD8;M?DO8;C?F:8;9?GE8;0?HC8:L?I<8:L?J58:K
3218L8:7N8M3:7?8MM:718NG:6B8OB:6490<:5F917:5791F:4F90J:3J8ON:2N8O2:22901:0E91A9OJ92J9O=94:9O495K9NJ97;9NA98K9N79:<9MN9;;9N19<19N49<G9N79==9N:9>G9MN9@49ME9AA9M=9BO9M49D<9LK9EJ9LB9G79L99HE9L09J29KG9K?9L49M09LE9NA9M6:029MG:1D9N7:359NH:4F9O9:679OJ:7>9OO:8;9O@:979O1::49NA:;09N2:;L9MC:<I9M4:=F9LD:>B9L5:??9KF:@;9K6:AJ9K7:C;9K::D=9K>:E29KB:EG9KF:F;9KJ:FO9KO:H59KN:I<9KL:JF9KF:KO9K@:M29K;:N59K6:O>9JM;1>9J7;3=9I@;579HL;629HJ;6N9HH;7I9HF;8D9HE;9@9HB;:A9HA;;B9H?;<C9H<;=E9H;;>E9H8;?G9H7;@H9H5;AH9H3;BJ9H1;CK9GO;DC9G@;E:9G0;F19F@;FK9F8;GD9F0;H=9EG;IE9EO;J>9E4;K39DG;KK9DF;LO9D9;N69CH;O79C@;OM9CE<0D9CH<329C2<5:9B?<6@9B<<7D9B:<8J9B8<:39AN<;?9A?<<B9@N<=B9@=<>C9?K<?C9?;<@C9>I<AD9>8<BE9=F<CG9=3<DI9<?<EH9;L<F>9;;<G39:H<GH9:7<H=99E<I3993<IH98A<J=97O<K397=<KI96L<L?96;<M595K<MK95:<O794N=0F94B=1D946=2F93L=3M93E=5593==6<936=7D92O=8L92G=:392@=;;928=<C921==J91I=?291A=@6915=@J901=A?8NM=B38MI=BK8LK=CJ8L5=DJ8KA=EI8JL=FI8JD=GH8JM=HH8K6=IH8K?=JH8KH=KG8L2=LF8L;=MF8LD=NE8LA=OE8L=>0D8L9>1D8L5>2C8L1>3C8KM>4B8KH>5A8KE>6@8KA>7F8JN>8M8J6>:58I@>;<8HI><58I9><N8IM>=F8JC>>>8K8>?A8K>>@H8K>>AO8K>>C78K>>D>8K>>EE8K>>FD8K2>GA8JC>H=8J3>I>8J1>JC8J9>KH8JC>LM8JK>N28K4>O;8KA?038JO?108J4?1L8I:?2H8H>?3E8GD?4A8FI?5=8EN?6:8E3?768D8?828C>?8O8BB?9K8AH?:H8@N?;E8@5?<A8?=?=D8>K?>L8>>?@58>0?A<8=C?BF8=8?D18<N?E<8<E?FG8<;?H28<2?HK8<1?ID8<0
3218L6:9:8M1:8L8MK:8=8NF:7O8O@:7A90::73915:6E910:5N904:528O8:468N<:398O@:2>914:1G92I:12948:0I95I:0?979:0698I9OL9::9OC9;@9O99<69O:9<M9O=9=C9O@9>E9O=9@39O59AB9NL9C09NC9D?9N;9EM9N19G;9MH9HH9M?9J69M79KD9M;9M59MM9NF9N>:079NO:1E9OC:32:09:4@:0M:67:19:7H:14:8F:0B:9B:00::?9OA:;;9O1:<79NB:=49N3:>09MD:>M9M5:?J9LE:@F9L6:B49L6:CE9L9:D:9L=:DN9LA:EC9LE:F>9LH:GF9LF:HN9LE:J69LB:K?9L;:LC9L6:MF9L2:NJ9KL:OM9KH;1;9K<;3;9JE;5:9IN;6B9ID;7>9IC;899IA;959I?;:09I=;:K9I;;;H9I9;<I9I7;=K9I5;>K9I3;?M9I1;@N9HO;AO9HN;C09HK;CO9HG;DF9H6;E=9GF;F69G=;G09G5;GI9FL;HB9FD;I;9F<;J>9F:;K39EI;L49E@;M<9DO;NC9D>;OF9D3<0;9D7<1N9CN<479C:<5<9C8<6B9C5<7F9C3<8L9C1<:19BO<;=9B@<<I9B0<>09A@<?19A0<@19@><A29?M<B29?;<C39>J<D59>7<E79=C<F29=1<FG9<?<G=9;M<H29;;<HG9:J<I<9:8<J299F<JG995<K>98D<L4983<LJ97C<MA972<N796A<OE965=1495I=2C95>=3@952=4;94F=5A94>=6I946=8093O=9893G=:@93@=;G939=<O931=>792J=?>92B=@F92:=A<919=B0905=BE8O1=C98MM=D68M5=E58LA=F48KL=G48KI=H38L2=I38L;=J38LE=K38LN=L28M7=M28MA=N18MJ=O18N2>008MN>108MJ>1O8MF>2N8MB>3N8M>>4M8M:>5M8M6>6L8M1>848L:>9;8KC>:B8JL>;I8J5><G8J9>=?8JO>>78KD>?28L5>@98L5>AA8L5>BH8L5>CO8L5>E38L1>F08KA>FL8K2>GI8JB>HE8J3>ID8IK>JI8J3>KN8J=>M38JE>N=8K2>OL8KE?0>8K??168JO?228J4?2O8I9?3K8H>?4G8GC?5D8FI?6@8EM?7<8E3?898D8?958C=?:18BB?:N8AI?;K8A0?<I8@:?>18?L??98???@A8?2?AI8>D?C38>:?D>8>0?EI8=G?G48==?H:8=6?I38=5?IL8=5
3218L5::H8LO::98MI:9K8ND:9=8O>:8N909:8@913:8190::768O>:6:8NB:5=8N<:4D8O::4>90N:3H92C:31945:2?95E:26976:1L98F:1C9:6:199;B:109<=:0F9=5:0F9=L:0I9>B:0L9?O:0D9A=:0;9BL:029D:9OJ9EI9O@9G79O79HF9NO9J49NF9K>9O19LL9OF9N::0:9OH:0O:15:1D:2G:25:4B:2<:6>:2C:7O:2;:92:1K::0:19::K:0E:;F:01:<C9OB:=?9O3:>;9ND:?89N4:@49ME:A19M5:B=9M5:CB9M8:D79M<:DK9M@:F09M?:G79M>:H?9M<:IG9M;:JO9M7:L49M2:M79LL:N;9LH:O>9LC;0A9L?;1D9L9;389KJ;589K3;729J?;7M9J=;8I9J;;9D9J:;:@9J7;;<9J6;<79J4;=29J1;>19J0;?19IN;@39IL;A49IJ;B59IH;C69IF;D29I=;DI9HL;EB9HC;F;9H;;G49H2;GM9GJ;HG9GB;I@9G9;J99G1;K99FF;L@9F5;MH9ED;NO9E4<049DE<0J9DJ<349D5<499D3<5>9D1<6D9CN<7H9CL<8N9CJ<:39CH<;:9CA<<G9C2<>29BC<?>9B3<@?9AB<A@9A0<B@9@@<CA9?N<DC9?;<EE9>H<F<9>6<G19=D<GG9=2<H<9<@<I19;N<IG9;=<J=9:L<K39:;<KJ99K<L@99:<M698I<ML989<ND97I=0397==1B971=3196E=4?96:=5<95N=6795B=75957=8=94O=9D94H=:L94@=<4949==;942=>C93J=?K93C=A293;=AN92A=BB91>=C6909=CJ8O6=DA8N4=E@8M@=F?8LK=G?8LO=H?8M8=I>8MA=J>8MJ=K>8N4=L=8N==M=8NF=N<8O0=O<8O9>0<8O?>1;8O<>2;8O8>3:8O3>498O0>588NL>688NG>7:8N=>8A8MG>9H8M0>:O8L8><68KB>=88K;>>08KO>>I8LE>?K8LL>A28LL>B98LL>C@8LL>D?8L@>E;8L0>F88K@>G48K1>H18JA>HM8J2>IJ8IE>JO8IM>L48J6>M>8JC>O08K8?0>8KI?0O8KD?1A8K>?288JN?358J3?418I9?4M8H>?5J8GC?6F8FH?7C8EM?8?8E2?9;8D7?:88C=?;48BD?<18AL?=68A;?>>8@N??F8@@?@N8@3?B68?G?C@8?<?DL8?2?F68>I?GB8>??HB8>;?I;8>:
3218L3:<58LM:;F8MG:;88NB::I8O<::;906:9L90@:9:8OC:8=8NH:7A8MK:6E8NJ:628OO:5@90G:5I92<:51940:4:95B:3M972:3D98B:3:9:3:319;C:2G9<@:2=9=;:249>4:229>J:259?J:239A8:1J9BG:1A9D5:199ED:0O9G2:0F9H@:0?9ID:0C9JO:119L=:1F9MK:2;9O9:30:12:37:2N:3>:4I:3F:6D:3M:86:3C:99:32::<:2A:;9:1O:<4:1<:=0:0H:=K:04:>G9OC:?C9O4:@?9ND:A<9N5:BG9N3:C?9N7:D99N9:EA9N7:FH9N6:H09N4:I79N3:J?9N1:KE9MM:LI9MH:ML9MC:NO9M>;029M9;169M5;299M0;3<9LK;559L9;759KA;8=9K8;999K6;:49K4;:O9K2;;K9K0;<G9JN;=B9JL;>>9JJ;?99JH;@99JG;A:9JD;B;9JC;C<9J@;D59J3;DN9IH;EG9I@;F@9I8;G99HO;H29HG;HL9H?;IG9H6;JB9GN;KC9G@;LL9FL;N49F:;O<9EI<0C9E8<219E1<369DN<4;9DL<5@9DJ<6F9DG<7K9DE<909DC<:59DA<;:9D><<D9D3<>09CD<?<9C4<@H9BE<AN9B5<BO9AC<CO9A2<E19@@<F09?M<FE9?;<G;9>I<H09>7<HF9=F<I<9=5<J29<D<JI9<4<K?9;C<L59;2<LL9:B<MB9:1<N999@<O2990=0A98D=20989=3?97M=4N97A=6<975=7996J=8396>=8N961=:195H=;895A=<@959==H952=>O94J=@794C=A>94<=B?93I=C392F=CH91A=D<90>=E18O:=EK8N?=FK8MK=GJ8N5=HJ8N>=IJ8NG=JJ8O0=KI8O9=LH8OC=MH8OL=NH905=OH90>>0G90G>1G90M>2F90I>3F90E>4E90A>5D90=>6C909>7G8OJ>8O8O3>:68N<>;=8ME><D8LN>=J8L;>>B8M1>?<8MC>@C8MC>AJ8MC>BO8M>>CK8LN>DG8L?>ED8KO>F@8K@>G=8K0>H98J@>I58J1>J28IA>K58IG>L?8J4>N08JI>OB8K>?0O8KN?1@8KH?228KB?2D8K=?3;8JN?478J3?548I8?608H=?6L8GB?7I8FH?8E8EL?9A8E2?:>8D7?;;8C>?<;8BJ?=C8B<?>K8AO?@38AB?A;8A5?BC8@H?CM8@>?E98@4?FD8?J?GO8?A?HI8??
3218L1:=B8LK:=48ME:<E8N@:<78O::;H904:;:8OI::B8NM:9F8N2:8J8N5:828O9:7@90?:6N91@:6B925:7193J:6:95>:5C96O:5:98?:5099O:4G9;@:4=9<B:449=>:3J9>9:3A9?2:3=9?I:3@9A3:399BB:319D0:2H9E?:2>9FK:289GO:2<9I4:2?9J@:319KN:3F9MB:439O>:4::19:4A:34:4I:4O:50:6K:56:8<:4J:9?:4:::B:3I:;E:39:<C:2F:=>:22:>9:1?:?4:0K:?O:07:@K9OC:AG9O4:BF9O1:CJ9O1:E29O0:F:9NN:GA9NM:HI9NK:J09NK:K69NI:L:9NC:M=9N>:NA9N::OD9N4;0G9N0;1J9MK;2N9MF;419MB;549M=;739LF;8M9L3;9I9L0;:D9KN;;@9KL;<;9KJ;=69KI;>29KF;>M9KE;?I9KC;@D9KA;A@9K?;BA9K=;CA9K9;D99JN;E29JF;EK9J=;FG9J4;GA9IL;H<9ID;I89I;;J29I3;JN9HK;KK9HB;M49GM;N=9G9;OF9FD<0N9F0<239EJ<389EG<4>9EE<5B9EC<6H9EA<7M9E><929E<<:79E:<;<9E7<<B9E4<=N9DD<?:9D5<@F9CF<B29C7<C=9BG<D=9B6<E?9AD<F:9A1<FO9@@<GE9?N<H;9?><I19>M<IH9><<J>9=L<K49=;<KK9<J<LA9<:<M79;I<MN9;8<ND9:H<O@9:8=0O99L=2>99@=3L995=5<98I=6K98==89981=9597E=:097:=:J96N=;E96A=<L96:=>4962=?;95K=@C95C=AK95;=C0952=CE93N=D992J=DN91F=EB90B=F68O>=G68O1=H58O:=I58OC=J58OL=K5905=L490?=M490H=N3912=O391;>0391D>1291M>22926>3192:>41926>50922>6091N>6O91K>85916>9<90?>:C8OI>;J8O1>=28N:>>98MD>?38N1>@48N:>A<8N:>B:8MM>C78M>>D38LN>DO8L>>EL8KO>FH8K?>GE8JO>HA8J@>I=8J0>J:8I@>K?8IE>M18J:>NC8JO?058KC?1@8L2?218KL?2D8KG?358KA?3G8K<?4=8JM?5:8J3?668I7?738H=?7O8GB?8K8FG?9H8EL?:D8E2?;A8D:?<G8CK?>08C>??78C1?@@8BD?AG8B6?C08AI?D:8A@?EF8A6?G18@M?H98@E
3218KO:>O8LI:>@8MC:>28N>:=D8O8:=58OO:<F8O3:;J8N7::N8M>::28ND:9@8OH:8N90M:8<922:7J92J:8093C:8;958:7D96K:7198;:6H99L:6>9;<:659<E:5K9=A:5B9><:589?7:4N9@1:4I9@N:4H9B=:4@9CK:479E6:429F::469G>:499HC:4=9J2:509KM:569MI:5=9OD:5D:1@:5L:3::62:56:69:74:6?:8C:61:9F:5A::I:50:;L:4@:<O:40:=L:3=:>G:2I:?B:25:@=:1B:A9:0N:B4:0::C7:03:D=:04:ED:04:FJ:05:H1:05:I7:06:J<:05:K@9OO:LH9OC:MO9O8:O59O0;099NK;1<9NG;2?9NA;3B9N=;4E9N8;5H9N3;709MK;8O9M5;:89LK;;49LH;;O9LG;<K9LE;=G9LC;>B9LA;?=9L?;@99L=;A49L<;B09L9;BK9L8;CF9L3;DA9KK;E<9KC;F79K:;G29K2;GM9JJ;HH9J@;IC9J8;J>9J0;K:9IH;L59IA;M=9HN;NE9H9;ON9GE<169G0<2<9FB<3<9F@<4@9F><5D9F<<6J9F:<7O9F7<949F5<:99F3<;>9F0<<C9EN<=L9EF<?89E7<@C9DG<B09D8<C<9CH<DG9C9<EM9BI<FD9B6<G:9AF<H19A5<HG9@D<I=9@4<J49?C<JJ9?2<K@9>B<L79>1<LM9=@<MC9=0<N:9<?<O09;N<ON9;?=1=9;4=2L9:H=4:9:<=5I9:0=7899E=8G999=:698M=;198A=;L985=<G97I==B97==>@973=?H96K=@O96D=B796<=C>965=D7956=DK943=E?92N=F391K=FH90F=GA907=HA90@=I@90I=J@912=K@91;=L?91D=M?91N=N>927=O>92A>0>92J>1>933>2=93<>3<93E>4<93H>5;93C>6;93@>7;93:>8B92C>9I91L>;1915><890>>=?8OG>>F8O0>?F8O1>@J8NL>AF8N<>BB8MM>C?8M=>D;8LM>E78L>>F48L1>G18KE>GN8K9>HJ8JL>IG8J@>JG8J4>L28IK>MD8J@>O68K4?0H8KI?218L6?2C8L1?358KK?3F8KE?488K@?4J8K:?5@8JL?6<8J2?798I7?858H<?928GB?9N8FF?:J8EM?;L8E:?=48DM?><8D@??E8D2?@L8CE?B58C8?C=8BK?DH8BB?F38B8?G>8AO?H@8AI
3218KM:@<8LG:?N8MA:??8N<:?18O6:>C8O9:=N8N=:=18MA:<58MM:;?8O3::N908::<91<:9I92B:9793F:8F944:9>951:9D96F:8M988:8>99H:859;9:7K9<G:7B9=C:789>>:6N9?9:6E9@5:6<9@O:649B8:5O9CA:5L9DE:5O9EI:629G2:659H>:659J9:699L4:6@9MO:6G9OK:6N:1F:75:3A:7<:5<:7C:7B:7H:8I:78:9L:6H::O:68:<2:5G:=5:56:>8:4F:?5:43:@0:3@:@L:2L:AG:28:BB:1E:CE:1?:DK:1?:F2:1?:G9:1@:H?:1@:ID:1?:JE:1<:KJ:14:M2:0I:N::0=:OA:02;0I9OF;209O=;339O8;469O3;5:9NN;6=9NJ;7@9NE;8M9N:;:H9ME;;D9MC;<?9MB;=;9M?;>69M>;?29M;;?M9M:;@I9M8;AD9M6;B?9M4;C89M3;D29M0;DL9LH;EG9L?;FC9L7;G=9KO;H89KF;I39K>;IN9K5;JJ9JM;KE9JE;LA9J=;ME9IN;NM9I:<069HE<1?9H1<2H9G9<3H9G8<4H9G6<5H9G5<6L9G3<819G0<969FN<:;9FL<;@9FI<<E9FG<=J9FE<?69F7<@A9EH<AM9E9<C:9DJ<DE9D;<EO9CK<FO9C><GF9BM<H<9B<<I29AL<II9A;<J?9@J<K69@:<KL9?I<LB9?8<M99>H<MO9>7<NE9=F<O<9=6=0<9<G=1J9<;=3:9<0=4I9;D=679;8=7F9:L=959:A=:D9:5=<399H=<M99===H991=>C98E=?>989=@997M=A;97D=BC97==CJ975=DH96>=E<95;=F1946=FE933=G8923=H191C=I191A=IO91L=JN926=KL92@=LJ92J=MJ933=NJ93==OI93F>0I93O>1I948>2H94A>3H94K>4G954>5G955>6F951>7H94F>9093O>:7938>;>92A><F91K>=L913>?390<>@68OE>A38O9>B08NM>BL8N@>CJ8N4>DF8MG>EC8M;>F@8LN>G=8LB>H:8L6>I68KI>J68K>>K@8K5>LK8JK>N78JE>OI8K:?1;8KO?2B8L;?348L5?3F8KO?478KJ?4I8KD?5;8K??5M8K9?6B8JL?7?8J1?8;8I7?988H;?:48GA?;18FI?<98F<?=A8EN?>I8EA?@18E4?A98DG?BA8D:?CJ8CN?E58CD?F@8C;?GK8C1
3218KK:AI8LE:A;8M@:@L8N::@>8O4:?O8NC:?68MF:>98M8:=?8N=:<N8OA:<;90G:;I91K:;8931::E945::3950::495>::L96?::M984::699E:9L9;5:9B9<E:999=F:8O9>A:8F9?<:8<9@8:829A3:7I9B4:7D9C<:7G9DE:7I9EN:7K9G7:7N9HH:7I9J?:7C9L::7J9N6:81:01:88:1M:8?:3H:8F:5C:8M:7M:90:90:8@::3:7O:;6:7?:<9:6N:=<:6>:>?:5M:?B:5=:@>:4J:A::47:B5:3C:C0:2O:D3:2J:E::2J:FA:2K:GH:2K:HK:2J:IL:2G:JM:2E:L4:2::M<:1O:ND:1C:OK:17;13:0L;2;:0@;3C:05;4K9OJ;5N9OE;729O@;859O;;989O7;:J9NH;<49N>;<O9N<;=K9N:;>F9N8;?A9N6;@=9N4;A89N2;B29N1;BK9N0;CD9MN;D=9MM;E89ME;F39M<;FN9M4;GI9LL;HD9LC;I?9L;;J99L3;K59KJ;L19KC;LM9K;;MM9K0;O69J;<0>9IG<1G9I2<309H:<449GO<549GN<649GL<749GK<849GI<989GG<:=9GE<;B9GB<<H9G@<=L9G><?39G9<@?9FI<AK9F:<C79EJ<DC9E;<EH9DN<FJ9D@<GL9D2<HH9CD<I>9C3<J59BC<JK9B2<KA9AA<L89A0<LN9@@<MD9?O<N;9?><O19>N<OG9>==0J9=O=289=C=3G9=7=569<K=6E9<@=849<4=9C9;H=;19;<=<A9;1==O9:E=>J9:8=?D99M=@?99A=A:995=B598I=C098==D7986=E997G=EN96C=FB95@=G594A=GJ93I=HD93:=IB92F=JA92L=K?937=L=93@=M<93K=N:945=O994?>0794I>16953>2495>>3395G>43960>53969>6296B>7296B>86962>9=95;>:E94E>;J93L><O933>>492:>?:91A>@?90H>A?906>B<8OJ>C98O=>D58O1>E28NE>EO8N9>FL8ML>GI8M@>HE8M4>ID8LH>JN8L>>L:8L5>MD8KL>NO8KB?0<8K??1M8L4?338L??3E8L9?478L4?4I8KN?5;8KH?5L8KC?6>8K=?708K8?7E8JL?8A8J1?9>8I6?::8H<?;>8GJ?<F8G=?=N8G0??68FB?@>8F6?AF8EI?BN8E;?D78E0?EB8DF?FM8D=?H78D3
3218KI:C68LC:BH8M>:B:8N8:AK8NH:A:8ML:@>8M0:?B8MG:>M8NL:>;900:=I916:=792::<E93?:<394E:;A95I::O96::;B96H:<;97N:<699A:;B9;1:;99<B::O9=H::F9>C::<9?>::39@::9I9A5:9@9B8:9=9CA:9@9DJ:9B9F2:9D9G;:9G9I1:9>9JH:949LA:939N<:9::08:9B:23:9I:3N::0:5J::6:83::7:96:9G::::96:;<:8F:<@:86:=B:7E:>F:75:?H:6D:@L:64:AH:5A:BC:4M:C>:4::DB:44:EI:45:FO:45:H3:44:I3:41:J5:3N:K6:3J:L>:3?:MF:34:NN:2H;06:2=;1=:22;2E:1F;3M:1;;55:0O;6=:0D;7E:09;8I:02;9L9OM;:O9OH;<C9O8;=?9O7;>:9O4;?69O3;@29O1;@K9NO;AD9NN;B>9NM;C69NK;CO9NJ;DI9NI;EC9NA;F>9N9;G99N1;H49MH;HO9M@;IJ9M8;JE9LO;KA9LH;L=9L?;M99L8;N59L0;O>9K<<0F9JG<1O9J3<389I;<4@9HG<5@9HE<6@9HD<7@9HB<8@9HA<9@9H?<:@9H><;D9H;<<J9H9<=N9H7<?49H5<@=9GJ<AI9G;<C59FL<D?9F=<EA9EO<FC9EB<GE9E4<HG9DG<II9D:<JA9CI<K79C8<KM9BH<LD9B7<M:9AF<N09A6<NG9@E<O=9@4=039?D=189?6=2G9>K=459>?=5D9>3=739=G=8B9=<=:19=0=;@9<C=<N9<8=>=9;L=?L9;A=@F9;4=AA9:H=B<9:==C69:1=D199E=DL999=EJ98N=F?97N=G296O=GE960=H=95?=I694O=IN94@=K293L=L1947=LO94A=MN94L=NL955=OJ95?>0I95J>1G964>2F96>>3D96H>4B972>5A97=>6?97G>7>981>8A97=>9F96D>:L95J><1951>=6948>>;93?>?A92F>@F91M>AK914>BH90H>CE90;>DA8OO>E>8OC>F;8O6>G88NI>H58N=>I28N1>J=8MG>KG8M>>M28M5>N=8LL>OH8LC?138L9?2@8L:?3D8LC?468L>?4H8L8?5:8L2?5L8KM?6=8KG?6O8KB?7A8K<?838K6?8G8JK?9D8J0?:B8I9?;K8HL?=28H>?>;8H1??B8GE?@K8G7?B38FJ?C;8F=?DD8F2?EO8EH?G:8E>?H@8E8
3218KG:DC8LB:D58M<:CF8N6:C88N2:BB8M6:AF8M1:@M8N6:@;8O;:?I90@:?791E:>E92J:>393N:=A954:<O968:<=976:<897D:=1983:=H99<:=@9:N:=09<>:<G9=K:<=9>F:<39?A:;J9@=:;@9A::;89B=:;59CF:;89DO:;;9F7:;=9GD:;<9I;:;29K2::I9LH::?9NC::E:0>::L:2::;2:45:;9:60:;@:8::;?:9=::O::@::>:;C:9M:<F:9=:=I:8M:>L:8<:?O:7K:A2:7;:B5:6K:C1:67:CL:5D:E0:5?:F7:5@:G::5?:H;:5<:I<:59:J=:56:K@:50:LH:4D:N0:49:O8:3M;0@:3B;1G:37;2O:2K;47:2@;5?:25;6G:1I;7O:1>;97:12;:>:0G;;D:0?;<G:09;=O:01;>J9OO;?E9OM;@>9OL;A79OK;B09OJ;BI9OH;CB9OG;D;9OF;E49OE;EN9O>;FJ9O6;GD9NM;H@9NE;I;9N=;J59N4;K19ML;KM9ME;LH9M=;ME9M5;N@9LM;OF9L<<0N9KH<279K3<3A9J<<4J9ID<5L9I=<6L9I;<7L9I:<8L9I8<9L9I7<:L9I5<;L9I4<<L9I2<>09I0<?69HN<@;9HK<AG9H<<C39GM<D89G?<E;9G1<F<9FD<G>9F7<H@9EI<IB9E;<JC9DN<KB9D@<L99CO<LO9C?<ME9BN<N<9B=<O29AM<OI9A<=0?9@K=1E9@>=349@2=4C9?F=629?;=7A9>O=909>C=:?9>7=;N9=L==<9=@=>K9=4=@:9<H=AG9<==BC9<1=C=9;D=D89;8=DM9:K=E?9:?=F;9:;=FN99==GA98>=H697E=I0975=IH96F=JA966=K@95B=LB958=MA95B=N?95K=O>966>0<96@>1:96K>29974>3797>>4697I>54983>6298=>7198G>8098I>90986>:097D>;3970><8967>==95=>>B94D>?G93K>@L932>B2929>C491E>D1918>DM90L>EJ90@>FG903>GD8OG>HA8O;>IJ8O1>K58NH>LA8N?>MK8N6>O68ML?0A8MC?1L8M:?378M1?468LH?4H8LB?598L<?5K8L7?6=8L1?6O8KL?7@8KF?828K@?8D8K;?958K5?9J8JK?:O8J:?<78IM?=@8I@?>G8I3?@08HF?A78H8?B?8GK?CG8G>?E18G4?F=8FJ?GG8F@
3198J1:FM8JK:F>8KE:F08L@:EA8M::E48N5:DE8M<:CI8L@:BM8M@:B:8NE:AI8OJ:A790O:@E923:@2939:?A94=:>O95C:>=96G:=J97L:=998@:=G98N:>>99=:?69:J:>I9<;:>=9=K:>39>H:=J9?D:=@9@?:=79A?:=19BB:<N9CK:=19E4:=49F=:=69GN:=19ID:<G9K;:<>9M2:<49NI:;N:0E:<5:2@:<<:4;:<C:67:<J:8@:<F:9C:<5::G:;E:;I:;5:<M::D:=O::3:?3:9C:@5:93:A9:8B:B;:82:C?:7A:D::6N:E?:6K:FB:6H:GC:6F:HC:6C:IE:6A:JE:6>:KK:65:M2:5J:N::5?:OB:53;0J:4H;21:4=;39:41;4A:3F;5I:3:;71:2O;89:2D;9A:28;:H:1L;<0:1A;=8:16;>?:0L;?7:0J;@1:0I;@J:0H;AC:0F;B<:0F;C5:0D;CN:0C;DG:0B;E@:0@;F::0<;G5:03;H09OJ;HK9OB;IF9O:;JA9O1;K=9NJ;L89NB;M59N:;N09N2;NL9MK;ON9M=<169LH<2?9L4<3I9K=<529JD<689J4<789J3<889J1<989J0<:89IN<;89IM<<89IK<=89IJ<>89IH<?89IG<@=9ID<AE9I=<C09HN<D29HA<E39H3<F59GF<G79G9<H99FK<I;9F=<J=9F0<K>9EC<L@9E5<M;9DF<N29D5<NH9CE<O>9C4=049BC=0K9B3=239AF=3B9A:=519@N=6@9@B=7O9@7=9>9?K=:L9?>=<;9?3==J9>G=?99><=@H9=O=B79=D=C:9=6=CK9<H=D<9<:=E29<7=EJ9<5=FK9;K=G>9:L=H199M=HI99<=IB98L=J:98==K397M=KM97==M296D=N296B=O096L=OO977>0M97@>1L97J>2J985>3I98?>4G98I>5E993>6D99=>7B99H>8B99>>9B98L>:C989>;C97G><C976>=D96C>>I95I>?N950>A4947>B993>>C>92E>D<926>E:91J>F691>>G3911>H090E>I990;>JD902>KO8OI>M98O?>ND8O6>OO8NM?188NE?2?8N=?3F8N7?4I8ML?5=8MD?5N8M>?6@8M7?728M1?7D8LJ?858LD?8G8L=?988L6?9J8L0?:<8KI?;<8K<?<D8JO?=L8JB??48J4?@<8IG?AD8I:?BL8HM?D48H@?E>8H5?FJ8GL?H48GB
3168G@:IE8H::I78I4:HI8IO:H:8JI:GL8KC:G=8L>:FO8M8:FA8MA:EN8LF:E28LJ:D:8MO:CI8O4:C6909:BD91>:B392C:AA93H:@N94M:@<961:?K977:?998;:>F99<:><99K:?49:9:?L9:G:@E9<7:@49=H:?K9>K:?A9?G:?89@B:>N9AD:>K9BG:>G9CO:>I9E8:>L9FA:>O9H7:>F9IN:><9KE:>29M<:=I9O2:=?:0K:=?:2G:=F:4B:=M:6=:>3:8G:=M:9J:==::M:<L:<0:<<:=3:;K:>6:;;:?9::K:@<::::A?:9I:BB:99:CE:8I:DH:88:EI:83:FJ:81:GK:7N:HL:7K:IM:7H:JN:7E:L5:7;:M<:6O:ND:6D:OL:68;14:5M;2;:5B;3C:56;4K:4K;63:4@;7;:44;8C:3I;9K:3>;;2:32;<7:2F;=::2:;>>:1N;?B:1F;@<:1E;A5:1C;AO:1B;BH:1A;C@:1?;D::1?;E3:1=;EL:1<;FF:19;G@:10;H;:0H;I6:0?;J1:06;JM9OO;KH9OF;LD9O?;M@9O7;N<9NO;O89NG<069N><1?9MJ<2G9M5<419L><5:9KF<6D9JM<7D9JJ<8D9JI<9D9JG<:D9JF<;D9JD<<D9JC<=D9JA<>D9J@<?D9J><@D9J=<AD9J;<BI9J0<CK9IB<DL9I5<EO9HH<G09H:<H29GM<I49G?<J69G2<K89FE<L:9F7<M;9EJ<N=9E<<O49DK<OJ9D;=0@9CJ=179C9=2A9BM=409BA=5?9B6=6N9AI=8=9A>=9L9A2=;;9@G=<I9@:=>69?N=??9?@=@I9?2=B29>E=C99>7=CM9=N=DE9=M=E=9=L=F59=J=G;9=:=GN9<;=HC9;A=I<9;1=J49:B=JM9:2=KF99C=L>993=M>98?=NC97G=OB97M>0@987>1>98A>2=98K>3;996>4:99?>5899J>679:4>759:>>849:E>949:2>:599A>;598O><598=>=597J>>5979>?696G>@6965>A:95<>B?94C>CE93J>DH933>EE92G>FB92;>G?91N>HG91D>J291;>K=912>LH90I>N090A>O7909?0?902?1E8OK?2L8OD?438O=?568O3?628NF?6D8N@?768N:?7G8N3?898MM?8J8MF?9<8M??9N8M9?:@8M2?;08LK?;I8L>?=18L0?>98KC??A8K6?@I8JI?B18J<?C98IO?DA8IA?EL8I7?G78HN?H>8HG
3138ED:JD8F7:JJ8FJ:JO8G>:K28H8:JD8I2:J58IM:IG8JG:I88KA:HJ8L<:H<8M6:GM8LK:G68L4:F:8M9:EH8N>:E68OC:DD90H:D291M:C@932:BN947:B<95<:AJ96@:A897F:@F98J:@499O:?B9:F:?J9;4:@B9;C:A;9<1:B39=D:AA9>M:A89?I:@N9@G:@H9AJ:@D9BM:@A9D5:@B9E=:@E9FJ:@D9HA:@:9J7:@19KN:?G9ME:?=9O<:?3:12:>J:2M:>O:4H:?6:6E:?<:8M:?5::0:>D:;4:>3:<6:=C:=::=3:><:<B:?@:<1:@B:;B:AF:;1:BI::@:CN:9N:E1:9B:F2:9;:G2:98:H3:95:I4:93:J5:90:K7:8L:L?:8A:MF:85:NN:7J;06:7>;1>:73;2F:6H;3M:6<;55:61;6=:5F;7E:59;8J:4N;9N:4B;;1:46;<5:3J;=8:3>;><:31;?@:2I;@E:2B;AA:2?;B::2>;C3:2=;CL:2;;DE:2:;E>:29;F7:28;G1:25;GL:1M;HG:1E;IB:1<;J<:13;K8:0L;L4:0D;M0:0<;ML:04;NH9OM;OD9OE<0@9O=<1G9NJ<2O9N6<499M><5B9LF<6L9KN<809KB<909K@<:09K?<;09K=<<09K<<=09K:<>09K9<?09K7<@09K6<A09K6<AO9K7<BL9K0<CJ9JE<DJ9J7<EK9IJ<FK9I<<GK9HN<HM9HB<IO9H4<K19GF<L39G9<M49FL<N69F><O89F0=059EB=0L9E1=1B9DA=2O9D5=4>9CI=5M9C==7<9C2=8J9BF=:49B8=;>9AJ=<G9A==>19@O=?:9@A=@C9@4=AL9?G=C69?E=D89?D=E09?B=EH9?A=FB9?>=GJ9>I=H=9=J=I59=7=IN9<H=JG9<8=K?9;I=L89;9=M19:J=MK9:9=NO99A>0398M>12997>2099B>2N99L>3M9:6>4K9:@>5J9:J>6H9;4>7F9;>>8F9;:>9F9:H>:G9:5>;G99C><G992>=G98@>>H97M>?H97<>@H96J>AH968>BH95F>CL94O>E1946>F193E>FN938>H592N>I@92E>JI92=>L0925>M791N>N>91H>OE91@?0L919?22912?3990K?4@90D?5C908?6C8OL?7:8OC?7K8O<?8=8O6?8N8NO?9@8NH?:28NB?:C8N;?;58N4?;F8ML?<68M??=>8M2?>F8LE??N8L8?A68KK?B>8K=?CE8K0?DN8JC?F98J9?GD8J0
3128DL:KA8EF:KG8F9:KM8FL:L28G?:L68H6:L18I0:KC8IK:K48JE:JF8K?:J88L::II8M1:I:8L5:H=8LC:GH8MH:G68NM:FD902:F2917:E@92<:DN93A:D<94F:CJ95K:C8970:BF984:B499::AB9:>:A09;B:@@9<1:A89<?:B19<M:BI9=?:C<9?0:BO9?L:BE9@M:B@9B0:B=9C2:B:9D::B;9EB:B=9G4:B89HK:AO9JA:AE9L8:A;9MO:A29OE:@H:1<:@>:33:@9:4O:@@:72:@E:94:@;::7:?L:;::?;:<=:>J:=@:>::>C:=J:?F:=9:@K:<G:B0:<5:C5:;C:D9:;5:E:::N:F:::F:G;::?:H<::<:I<::::J=::7:KA::1:LI:9F:N1:9::O8:8O;0@:8D;1H:88;30:7M;47:7A;5>:76;6A:6J;7E:6>;8H:62;9L:5F;:O:59;<3:4M;=6:4A;>::45;?>:3L;@C:3E;AG:3>;BF:3:;C?:38;D8:37;E1:36;EJ:34;FC:34;G<:32;H7:2J;I2:2A;IM:29;JH:21;KD:1I;L@:1A;M<:19;N8:12;O4:0I<00:0B<0K:0:<1O9OK<379O7<4A9N@<5K9MG<749LO<8<9L9<9<9L8<:<9L6<;<9L5<<<9L3<=<9L2<><9L1<?<9L3<@;9L4<A;9L5<B:9L6<C99L3<D39KC<DN9K:<EO9JM<FO9J?<GO9J1<HO9ID<J09I6<K09HI<L09H;<M19GN<N19G@<O19G3=039FE=159F8=1N9EH=3=9E<=4I9E0=639DB=7<9D5=8E9CG=9N9C:=;79BL=<@9B?==I9B2=?29AE=@<9A<=AF9A;=C19A:=D;9A9=E;9A8=F39A7=G39@O=H:9@8=HO9?==IG9>N=J@9>>=K99=O=L29=?=LJ9=0=MC9<@=N<9<1=O<9;;>0A9:D>1C9:8>2B9:B>3@9:L>4>9;6>5=9;A>6;9;J>7:9<5>889<?>989;N>:99;=>;99:K><99:8>=999F>>9995>?:98C>@:980>A:97?>B:96M>C;96;>D;95I>E;957>F=94B>GB949>HI941>J093J>K693C>L>93<>MD935>NK92N?0292G?1992@?2@928?3F921?4M91K?5O91??70912?7O90E?8@90>?92908?9D901?:68OJ?:G8OD?;98O=?;K8O6?<:8NM?<H8N@?=J8N4??28MG?@:8M9?AB8LL?BJ8L>?D38L2?E;8KE?FF8K<?H18K2
3128DG:LC8EA:LI8F;:LO8FN:M48GA:M98H4:M>8HN:M08II:LA8JC:L38K=:KD8L8:K68L;:JA8KM:IH8M2:I58N7:HD8O<:H290A:G@91F:FM92K:F<940:EJ955:E896::DE97?:D498C:CB99I:C09:M:B=9<2:AL9<L:AN9=::BF9=I:C?9>7:D69>M:DE9@0:D=9A2:D:9B5:D79C7:D39D>:D49EG:D69G>:CM9I4:CC9JK:C99LA:C09N8:BF9OO:B=:1F:B3:3<:AJ:55:AJ:7?:AN:9::AC::>:A2:;A:@B:<D:@2:=G:?A:>M:>O:@2:>=:A7:=J:B<:=7:CA:<G:DA:<A:EB:<9:FC:<2:GD:;K:HD:;E:IE:;B:JF:;?:KK:;7:M3::L:N;::@:OB::5;0J:9I;22:9>;35:92;49:8F;5<:8:;6@:7N;7C:7A;8F:75;9J:6I;:M:6=;<1:61;=4:5E;>8:59;?<:4O;@@:4G;AE:4@;BI:4:;CJ:44;DD:43;E=:41;F6:40;FO:3O;GH:3N;HC:3G;I>:3>;J8:36;K4:2N;L0:2F;LL:2>;MH:27;ND:1O;O@:1G<0;:1?<17:17<27:0L<3@:08<4I9O@<639NH<7<9MO<8F9M7<9H9LO<:H9LN<;H9LL<<H9LM<=G9LO<>G9M0<?G9M2<@F9M3<AF9M4<BF9M5<CE9M5<D?9LE<E:9L6<F39KO<G39KB<H39K4<I49JG<J49J9<K49IK<L59I><M59I1<N59HC<O59H5=059GH=169G:=249FM=399F@=4B9F3=5K9EE=749E8=8=9DK=9F9D>=:O9D0=<89CC==B9C5=>K9C2=@69C1=A@9C0=BK9BO=D59BN=E?9BM=F>9BL=GE9B?=HJ9AG=IA9A4=J:9@D=K39@4=KK9?E=LD9?5=M=9>F=N69>6=NN9=G=OI9=6>0N9<>>229;F>339;C>429;M>509<7>5O9<A>6M9<K>7K9=6>8J9=6>9J9<C>:K9<1>;K9;@><K9:N>=K9:;>>L99I>?L998>@L98F>AL983>BL97B>CM970>DM96>>EL965>FK95L>GO95?>I6958>J=951>KD94J>LK94B>N294<>O8944?0?93M?1F93F?2M93??44938?5;931?6<92E?7=928?8<91K?9691A?9H91:?:9913?:K90M?;=90F?;N90??<@909?<O8OO?==8OB?>78O5???8NH?@G8N:?AO8MM?C78MA?D?8M3?EH8LH?G38L>
3118CI:M?8DB:ME8E<:MK8F6:N18G0:N68GC:N;8H6:NA8HL:N=8IG:MO8JA:M@8K<:M28L6:LC8KD:KJ8L<:K58MA:JD8NF:J18OK:I?910:HN925:H<93::GI94?:G795D:FF96H:F497N:EA993:DO9:8:D>9;=:CL9<A:C99=G:BG9>7:C<9>E:D49?3:DL9?A:ED9@5:F79A7:F49B::F09C=:EM9DC:EM9F0:EK9GG:EA9I>:E89K5:DO9LK:DE9NB:D;:08:D2:1O:CH:3F:C>:5<:C4:7M:C7:9A:BJ::D:B::;I:AH:<N:A6:>3:@C:?9:@1:@>:??:AC:>L:BH:>::CI:>3:DJ:=L:EK:=E:FK:==:GL:=7:HL:<O:IM:<I:JN:<G:L5:<<:M=:<1:NE:;E:OI:;9;0M::M;20::A;33::5;47:9I;5::9=;6>:91;7A:8E;8E:89;9H:7M;:K:7A;;O:75;=2:6H;>6:6<;?::61;@>:5J;AC:5C;BG:5<;CL:55;DO:4N;EH:4M;FB:4L;G;:4K;H3:4I;HN:4C;II:4;;JD:43;K@:3K;L<:3D;M8:3;;N4:34;NO:2L;OK:2D<0G:2<<1C:25<2?:1M<3H:19<51:0A<6;9OI<7D9O1<8N9N8<:49MH<;49MJ<<39MK<=39ML<>29MN<?29MO<@29N1<A29N2<B19N3<C19N4<D19N6<DK9MG<EF9M8<FA9LH<G;9L9<H89L7<I89KI<J89K<<K89JN<L99J@<M99J3<N99IE<O99I8=0:9HK=1:9H==269H0=329GC=4:9G6=5C9FI=6L9F<=859EN=9?9EA=:G9E4=<19DI==;9DH=>F9DG=@09DF=A:9DE=BE9DD=CO9DC=E:9DB=FI9D?=H69CN=I;9C9=J39BJ=JM9B:=KE9AK=L>9A;=M79@L=MO9@<=NH9?M=OA9?=>0:9>N>1;9>9>2?9=@>3D9<H>4C9<M>5A9=7>6@9=B>7>9=L>8=9>6>9<9=J>:<9=9>;=9<G><=9<4>==9;C>>=9;1>?>9:>>@>99L>A>99;>B>98I>C>98=>D=984>E;97L>F:97C>G;978>H>96J>ID96>>JJ968>L1961>M895I>N?95B>OE95;?0L954?2394M?3:94F?4A94??5H948?6I93L?7I93>?8I931?9J92D?:=92<?:O926?;A91O?<291H?<D91B?=591;?=D910?>290C?>D906??L8OI?A48O<?B<8NO?CD8NB?DL8N5?F58MI?G@8M@
3118CD:NB8D=:NG8E7:NM8F1:O28FK:O88GE:O>8H8:OC8HK:OH8IE:O;8J?:NM8K::N?8KJ:MN8KF:M68LK:LC8N0:L18O5:K?90::JN91?:J;92D:II93I:I794N:HF963:H3978:GA98=:FO99B:F>9:F:EK9;L:E99=0:DG9>6:D59?2:D29?@:DJ9?O:EB9@C:F?9A=:G@9A@:GK9B@:GI9CB:GF9DH:GF9F::G@9H1:G79IG:FM9K>:FC9M5:F:9NK:F0:0B:EF:29:E<:3O:E2:5E:DC:7I:D9:9K:CO:;1:C<:<6:BJ:=;:B8:>@:AF:?E:A4:@K:@A:B0:?O:C1:?E:D2:?>:E2:?7:F3:?0:G4:>I:H4:>B:I5:>;:J6:>3:K7:=M:L=:=A:M@:=5:ND:<I:OG:<=;0K:<1;1N:;E;32:;8;45::L;58::@;6<::4;7?:9H;8C:9<;9F:90;:I:8D;;M:88;=0:7L;>4:7@;?7:74;@<:6M;A@:6F;BE:6?;CJ:68;DN:61;F3:5I;FM:5H;GF:5F;H@:5F;I9:5@;J4:58;K0:50;KL:4H;LG:4@;MD:49;N?:40;O;:3I<07:3A<13:3:<1O:31<2K:2J<40:29<59:1B<6C:0I<7L:01<969O=<:=9O4<;>9NM<<?9NJ<=>9NK<>>9NM<?>9NN<@=9O0<A=9O1<B=9O2<C<9O4<D<9O5<E89NI<F29N:<FM9MK<GH9M;<HB9LL<I<9LL<J<9L?<K<9L1<L=9KC<M=9K6<N=9JH<O=9J:=0>9IN=1<9I@=279I3=339HG=429H:=5<9GL=6D9G?=7N9G2=979FE=:A9F@=;K9F>==59F==>@9F==?J9F<=A89F:=BJ9F6=D<9F2=EN9EO=G<9EC=HI9E6=IK9DK=JF9D@=K?9D1=L79CA=M09C2=MI9BB=NB9B3=O;9AC>039A4>0L9@D>1G9@3>2L9?;>419>B>549=N>639>8>719>B>809>L>8N9?2>9N9>?>:O9=M>;O9=<><O9<J>=O9<7>?09;F>@09;4>@O9:E>AO9:=>BM9:4>CL99L>DK99B>EJ99:>FI991>GK98E>HN986>J297G>K797>>L>977>ME970>NL96I?0396A?1:96;?2A964?3G95L?4N95E?6595??76952?8694E?96947?:693J?;293??;D938?<6931?<H92K?=992D?=K92=?>9921?>G91D??5918?@990K?A@90>?BI901?D08OC?E98O6?FC8NK?GM8NB
3108BE:O>8C?:OD8D8:OI8E2:OO8EL;048FF;0:8G@;0@8H:;0F8HM;0K8IC;0I8J>;0:8K8:OL8K4:O58L6:NC8M::N18N?:M?8OD:LM90I:L;91N:KI933:K7948:JE95=:J396B:IA97G:HO98L:H=9:1:GK9;5:G99<;:FG9=@:F59>D:EC9?J:E19@=:E@9@N:F:9AI:G=9BE:H?9BO:I29C3:I<9CH:I?9DM:I?9FD:I59H::HK9J1:HA9KH:H89M>:GI9O5:G;:0K:FM:2A:F>:48:EO:5N:EA:7E:E4::3:DM:;@:D=:<F:CK:=K:C9:?0:BF:@5:B4:A;:AB:B<:A1:C::@G:D;:@@:E<:@9:F<:@2:G=:?L:H>:?E:I>:?>:J@:?8:KE:?4:LB:>F:M>:>9:NB:=M:OE:=@;0I:=4;1L:<H;30:<<;43:<0;56:;D;6::;8;7=::L;8A::?;9D::3;:H:9G;;K:9;;<N:8O;>2:8C;?5:87;@::80;A>:7H;BC:7A;CG:7;;DL:73;F0:6L;G5:6E;H2:6B;HK:6A;IE:6=;J?:65;K<:5M;L7:5F;M4:5>;MO:56;NK:4N;OG:4F<0C:4><1?:47<2;:3O<36:3G<48:3:<5A:2C<6K:1K<84:13<9?:0K<:F:0C<;G:0<<<H:06<=I9OO<>I9OL<?I9OM<@I9OO<AH:00<BH:01<CH:03<DG:04<ED9OJ<F>9O;<G99NL<H49N=<HN9MN<II9M><J@9MA<KA9M4<LA9LF<MA9L9<NB9KK<OB9K==0B9K0=1=9JD=299J7=359IJ=409I==549I0=6=9HC=7F9H6=909H5=:;9H5=;G9H3==99GO=>K9GL=@=9GI=AO9GE=CA9GA=E39G==FB9G3=GO9FF=I59F:=J79EN=K39EF=KN9E==LI9E6=MC9DH=N;9D8=O49CI=OM9C9>0F9BJ>1>9B:>279AK>389A5>4=9@=>5B9?E>6D9?8>7C9?B>8A9?M>9@9?F>:A9?5>;A9>B><A9>0>=A9=?>>B9<M>?A9<E>@?9<<>A>9<4>B=9;J>C<9;B>D;9;9>E:9;1>F99:H>G79:?>H:9:2>I>99B>JA994>KE98F>LK98=>N3986>O9980?0@97H?1G97A?2N97:?44973?5;96L?6B96E?7C968?8C95K?9C95>?:C950?;D94C?<:94:?<K943?==93M?=O93F?>@93??>N933??<92F??J92:?@E91M?AN91??C5912?D=90E?EE908?G08OM?H:8OD?I48OC
2095H7:FM5I6:FC5J4:F95K3:EN5L1:ED5M0:E:5MO:DO5NM:DE5OL:D;60K:D061J:CF62M:C<646:C1651:BI65M:B@66H:B867D:B068?:AG69;:A?6:7:A76