package Padre::DocBrowser;
use strict;
use warnings;

use Class::Autouse;
use Padre::DocBrowser::POD;
use Carp qw( confess );
use Scalar::Util qw( blessed );
use Class::XSAccessor
  getters => {
    get_providers => 'providers',
    get_viewers   => 'viewers',
    get_schemes    => 'schemes',
    
  },
  setters => {
    set_providers => 'providers',
    set_viewers   => 'viewers',
    set_schemes    => 'schemes',
  };


our $VERSION = '0.24';

=pod

=head1 NAME

Padre::DocBrowser

=head1 DESCRIPTION

Provide an interface for retrieving / generating documentation, resolving terms
to documentation (search?) and formatting documentation.

Allow new packages to be loaded and interrogated for the mimetypes they can
generate documentation for. Provide similar mechanism for registering new
documentation viewers and uri schemes accepted for resolving. 

=head1 SYNOPSIS

  # Does perlish things by default via 'Padre::DocBrowser::POD'
  my $browser = Padre::DocBrowser->new();
  my $source = Padre::Document->new( filename=>'source/Package.pm' );
  
  my $docs = $browser->docs( $source );
  # $docs provided by DocBrowser::POD->generate
  #  should be Padre::Document , application/x-pod
  
  my $output = $browser->browse( $docs );
  # $output provided by DocBrowser::POD->render
  #  should be Padre::Document , text/x-html
  
  $browser->load_viewer( 'Padre::DocBrowser::PodAdvanced' );
  # PodAdvanced->render might add an html TOC in addition to
  #  just pod2html
  
  my $new_output = $browser->browse( $docs );
  # $new_output now with a table of contents
  
  
=head1 METHODS

=head2 new

Boring constructor , pass nothing. yet.

=head2 load_provider

Accepts a single class name ,  will attempt to Autouse the class and 
interrogate it's ->provider_for method. Any mimetypes returned will be
associated with the class for dispatch to ->generate

Additionally , interrogate class for ->accept_schemes and associate the class
with uri schemes for dispatch to ->resolve

=head2 load_viewer

Accepts a single class name , will attempt to Autouse the class and
interrogate it's ->viewer_for method. Any mimetypes returned will be
assiciated with the class for dispatch to ->render

=head2 resolve

Accepts a URI or scalar 

=head2 browse

=head2 accept


=head1 EXTENDING



  package My::DocBrowser::Doxygen;
  
  # URI of doxygen:$string or doxygen://path?query
  sub accept_schemes {
      'doxygen',	  
  }
  
  sub provider_for {
      'text/x-c++src'      
  }
  
  sub viewer_for {
      'text/x-doxygen',
  }
  
  sub generate {
	    my ($self,$doc) = @_;
	    # $doc will be Padre::Document of any type specified 
	    # by ->provider_for
	    
	    # push $doc through doxygen
	    # ...
	    # that was easy :)
	    
	    # You know your own output type, be explicit
	    my $response = Padre::Document->new();	  
	    $response->{original_content} = $doxygen->output;
	    $response->set_mimetype( 'text/x-doxygen' );
	    return $response;
  }

  sub render {
      my ($self,$docs) = @_;
      # $docs will be of any type specified 
      # by ->viewer_for;
      
      ## turn $docs into doxygen(y) html document
      #  ...
      #
      
      my $response = Padre::Document->new();
      $response->{original_content} = $doxy2html->output;
      $response->set_mimetype( 'text/x-html' );
      return $response;
      	  
  }

=cut

sub new {
	my ($class,%args) = @_;

	my $self = bless \%args, ref ($class) || $class;
	$self->set_providers({}) unless $args{providers};
	$self->set_viewers({}) unless $args{viewers};
	$self->set_schemes({}) unless $args{schemes};
	
	# Provides pod from perl, pod: perldoc: schemes
	$self->load_provider( 'Padre::DocBrowser::POD' );
	
	# Produces html view of POD
	$self->load_viewer( 'Padre::DocBrowser::POD' );
	
	

	return $self;
}

sub load_provider {
	    my ($self,$class) = @_;

	    Class::Autouse->autouse( $class );
	    if ( $class->can( 'provider_for' ) ) {
			$self->register_providers(
			    $_ => $class
			) for $class->provider_for;
	    }
	    else {
			confess "$class is not a provider for anything.";
	    }
	    
	    if ( $class->can( 'accept_schemes' ) ) {
		    $self->register_schemes( 
			$_ => $class
			) for $class->accept_schemes;			
	    }
	    else {
			    confess "$class accepts no uri schemes";
	    }
	    
	$self;
}

sub load_viewer {
	    my ($self,$class) = @_;
	    Class::Autouse->autouse( $class );
	    if ( $class->can( 'viewer_for' ) ) {
			$self->register_viewers(
			    $_ => $class
			) for $class->viewer_for;
	    }
	    $self;
}


sub register_providers {
  my ($self,%provides) = @_;
  while ( my ($type,$class) = each %provides ) {
	  # TODO - handle collisions, ie multi providers
    $self->get_providers->{$type} = $class;	      
  }
  $self;
}

sub register_viewers {
  my ($self,%viewers) = @_;
  while ( my ($type,$class) = each %viewers ) {
    $self->get_viewers->{$type} = $class;
    Class::Autouse->autouse( $class );
  }
  $self;
}

sub register_schemes { 
	    my ($self,%schemes) = @_;
	    while ( my ($scheme,$class) = each %schemes ) {
			    $self->get_schemes->{$scheme} = $class;			    
	    }
	    $self;
}

sub provider_for {
  my ($self,$type) = @_;
  my $p;
eval {
  if ( exists $self->get_providers->{$type} ) {
	$p =  $self->get_providers->{$type}->new();
  } 
};
confess $@ if $@;
return $p;
}

sub accept {
	    my ($self,$scheme) = @_;
	    if ( defined $self->get_schemes->{$scheme} ) {
			return 1;
	    }
	    return;
}

sub viewer_for {
  my ($self,$type) = @_;
  my $v;
eval {
  if ( exists $self->get_viewers->{$type} ) {
	   $v = $self->get_viewers->{$type}->new;
  }	
};
  confess $@ if $@;
  return $v;
}

sub docs {
  my ($self,$doc) = @_;
  if ( my $provider = $self->provider_for( $doc->get_mimetype ) ) {
	  my $docs = $provider->generate( $doc );
	  return $docs;	  
  }
  warn "No provider for " . $doc->get_mimetype;	
  return;
}

sub resolve {
  my ($self,$ref) = @_;
  my @refs;
  if ( blessed $ref and $ref->isa( 'URI' ) ) {
	    return $self->resolve_uri( $ref );
  }
  # TODO this doubles up if a provider subscribes to multi
  # mimetypes . 
  foreach my $class ( values %{$self->get_providers} ) {
	    my $resp = $class->resolve( $ref );
	    push @refs ,$resp if $resp;
  }	
  #warn "Returning first of " . scalar @refs . " resolved\n";
  return $refs[0];
}

sub resolve_uri { 
	    my ($self,$uri) = @_;
	    my $resolver = $self->get_schemes->{ $uri->scheme };
	    my $doc = $resolver->resolve( $uri->opaque );
	    return $doc;
}

sub browse {
  my ($self,$docs) = @_;

  if ( my $viewer = $self->viewer_for( $docs->get_mimetype ) ) {
    return $viewer->render( $docs );
  }
  return;
}
  

1;

