package Calendar::Indonesia::Holiday;
BEGIN {
  $Calendar::Indonesia::Holiday::VERSION = '0.02';
}
# ABSTRACT: List Indonesian public holidays

use 5.010;
use strict;
use warnings;
use Log::Any '$log';

use Data::Clone;
use DateTime;
use Sub::Spec::Gen::ReadTable qw(gen_read_table_func);

use Exporter;
our @ISA = qw(Exporter);
our @EXPORT_OK = qw(
                       list_id_holidays
                       enum_id_workdays
                       count_id_workdays
               );

our %SPEC;

my @fixed_holidays = (
    my $newyear = {
        day        =>  1, month =>  1,
        id_name    => "Tahun Baru",
        en_name    => "New Year",
        tags       => [qw/international/],
    },
    {
        day        => 17, month =>  8,
        id_name    => "Proklamasi",
        en_name    => "Declaration Of Independence",
        tags       => [],
    },
    my $christmas = {
        day        => 25, month => 12,
        id_name    => "Natal",
        en_name    => "Christmas",
        tags       => [qw/international religious religion=christianity/],
    },
);

sub _h_chnewyear {
    my ($r, $opts) = @_;
    $r->{id_name}    = "Tahun Baru Imlek".
        ($opts->{hyear} ? " $opts->{hyear}":"");
    $r->{en_name}    = "Chinese New Year".
        ($opts->{hyear} ? " $opts->{hyear}":"");
    $r->{id_aliases} = [];
    $r->{en_aliases} = [];
    $r->{is_holiday} = 1;
    $r->{tags}       = [qw/international calendar=lunar/];
    ($r);
}

sub _h_mawlid {
    my ($r, $opts) = @_;
    $r->{id_name}    = "Maulid Nabi Muhammad";
    $r->{en_name}    = "Mawlid";
    $r->{id_aliases} = [qw/Maulud/];
    $r->{en_aliases} = ["Mawlid An-Nabi"];
    $r->{is_holiday} = 1;
    $r->{tags}       = [qw/religious religion=islam calendar=lunar/];
    ($r);
}

sub _h_nyepi {
    my ($r, $opts) = @_;
    $r->{id_name}    = "Nyepi".
        ($opts->{hyear} ? " $opts->{hyear}":"");
    $r->{en_name}    = "Nyepi".
        ($opts->{hyear} ? " $opts->{hyear}":"");
    $r->{id_aliases} = ["Tahun Baru Saka"];
    $r->{en_aliases} = ["Bali New Year", "Bali Day Of Silence"];
    $r->{is_holiday} = 1;
    $r->{tags}       = [qw/religious religion=hinduism calendar=saka/];
    ($r);
}

sub _h_goodfri {
    my ($r, $opts) = @_;
    $r->{id_name}    = "Jum'at Agung";
    $r->{en_name}    = "Good Friday";
    $r->{id_aliases} = ["Wafat Isa Al-Masih"];
    $r->{en_aliases} = [];
    $r->{is_holiday} = 1;
    $r->{tags}       = [qw/religious religion=christianity/];
    ($r);
}

sub _h_vesakha {
    my ($r, $opts) = @_;
    $r->{id_name}    = "Waisyak".
        ($opts->{hyear} ? " $opts->{hyear}":"");
    $r->{en_name}    = "Vesakha".
        ($opts->{hyear} ? " $opts->{hyear}":"");
    $r->{id_aliases} = [];
    $r->{en_aliases} = ["Vesak"];
    $r->{is_holiday} = 1;
    $r->{tags}       = [qw/religious religion=buddhism/];
    ($r);
}

sub _h_ascension {
    my ($r, $opts) = @_;
    $r->{id_name}    = "Kenaikan Isa Al-Masih";
    $r->{en_name}    = "Ascension Day";
    $r->{id_aliases} = [];
    $r->{en_aliases} = [];
    $r->{is_holiday} = 1;
    $r->{tags}       = [qw/religious religion=christianity/];
    ($r);
}

sub _h_isramiraj {
    my ($r, $opts) = @_;
    $r->{id_name}    = "Isra Miraj";
    $r->{en_name}    = "Isra And Miraj";
    $r->{id_aliases} = [];
    $r->{en_aliases} = [];
    $r->{is_holiday} = 1;
    $r->{tags}       = [qw/religious religion=islam calendar=lunar/];
    ($r);
}

sub _h_eidulf {
    my ($r, $opts) = @_;
    $opts //= {};
    $r->{id_name0}   = "Idul Fitri".
        ($opts->{hyear} ? " $opts->{hyear}H":"");
    $r->{en_name0}   = "Eid Ul-Fitr".
        ($opts->{hyear} ? " $opts->{hyear}H":"");
    $r->{id_name}    = $r->{id_name0}.($opts->{day} ? ", Hari $opts->{day}":"");
    $r->{en_name}    = $r->{en_name0}.($opts->{day} ? ", Day $opts->{day}":"");
    $r->{id_aliases} = ["Lebaran"];
    $r->{en_aliases} = [];
    $r->{is_holiday} = 1;
    $r->{tags}       = [qw/religious religion=islam calendar=lunar/];
    ($r);
}

sub _h_eidula {
    my ($r, $opts) = @_;
    $r->{id_name}    = "Idul Adha";
    $r->{en_name}    = "Eid Al-Adha";
    $r->{id_aliases} = ["Idul Kurban"];
    $r->{en_aliases} = [];
    $r->{is_holiday} = 1;
    $r->{tags}       = [qw/religious religion=islam calendar=lunar/];
    ($r);
}

sub _h_hijra {
    my ($r, $opts) = @_;
    $opts //= {};
    $r->{id_name}    = "Tahun Baru Hijriyah".
        ($opts->{hyear} ? " $opts->{hyear}H":"");
    $r->{en_name}    = "Hijra".
        ($opts->{hyear} ? " $opts->{hyear}H":"");
    $r->{id_aliases} = ["1 Muharam"];
    $r->{en_aliases} = [];
    $r->{is_holiday} = 1;
    $r->{tags}       = [qw/calendar=lunar/];
    ($r);
}

sub _jointlv {
    my ($r, $opts) = @_;
    $opts //= {};
    my $h = $opts->{holiday};
    $r->{id_name}        = "Cuti Bersama".
        ($h ? " (".($h->{id_name0} // $h->{id_name}).")": "");
    $r->{en_name}        = "Joint Leave".
        ($h ? " (".($h->{en_name0} // $h->{en_name}).")": "");
    $r->{id_aliases}     = [];
    $r->{en_aliases}     = [];
    $r->{is_joint_leave} = 1;
    $r->{tags}           = [];
    ($r);
}

sub _make_jl_tentative {
    my ($holidays) = @_;
    for (@$holidays) {
        push @{$_->{tags}}, "tentative" if $_->{is_joint_leave}
    }
    $holidays;
}

my %year_holidays;

# decreed feb 5, 2008
my $hijra2008a;
my $eidulf2008;
$year_holidays{2008} = [
    _h_hijra     ({day => 10, month =>  1}, {hyear=>1929}),
    _h_chnewyear ({day =>  7, month =>  2}, {hyear=>2559}),
    _h_nyepi     ({day =>  7, month =>  3}, {hyear=>1930}),
    _h_mawlid    ({day => 20, month =>  3}),
    _h_goodfri   ({day => 21, month =>  3}),
    _h_ascension ({day =>  1, month =>  5}),
    _h_vesakha   ({day => 20, month =>  5}, {hyear=>2552}),
    _h_isramiraj ({day => 30, month =>  7}),
    ($eidulf2008 =
    _h_eidulf    ({day =>  1, month => 10}, {hyear=>1929, day=>1})),
    _h_eidulf    ({day =>  2, month => 10}, {hyear=>1929, day=>2}),
    _h_eidula    ({day =>  8, month => 12}),
    _h_hijra     ({day => 29, month => 12}, {hyear=>1930}),

    _jointlv     ({day => 11, month =>  1}, {holiday=>$hijra2008a}),
    _jointlv     ({day => 29, month =>  9}, {holiday=>$eidulf2008}),
    _jointlv     ({day => 30, month =>  9}, {holiday=>$eidulf2008}),
    _jointlv     ({day =>  3, month => 10}, {holiday=>$eidulf2008}),
    _jointlv     ({day => 26, month => 12}, {holiday=>$christmas}),
];

# decreed juni 9, 2008
my $eidulf2009;
$year_holidays{2009} = [
    _h_chnewyear ({day => 26, month =>  1}, {hyear=>2560}),
    _h_mawlid    ({day =>  9, month =>  3}),
    _h_nyepi     ({day => 26, month =>  3}, {hyear=>1931}),
    _h_goodfri   ({day => 10, month =>  4}),
    _h_vesakha   ({day =>  9, month =>  5}, {hyear=>2553}),
    _h_ascension ({day => 21, month =>  5}),
    _h_isramiraj ({day => 20, month =>  7}),
    ($eidulf2009 =
    _h_eidulf    ({day => 21, month =>  9}, {hyear=>1930, day=>1})),
    _h_eidulf    ({day => 22, month =>  9}, {hyear=>1930, day=>2}),
    _h_eidula    ({day => 27, month => 11}),
    _h_hijra     ({day => 18, month => 12}, {hyear=>1931}),

    _jointlv     ({day =>  2, month =>  1}, {holiday=>$newyear}),
    _jointlv     ({day => 18, month =>  9}, {holiday=>$eidulf2009}),
    _jointlv     ({day => 23, month =>  9}, {holiday=>$eidulf2009}),
    _jointlv     ({day => 24, month => 12}, {holiday=>$christmas}),
];

# decreed aug 7, 2009
my $eidulf2010;
$year_holidays{2010} = [
    _h_chnewyear ({day => 14, month =>  2}, {hyear=>2561}),
    _h_mawlid    ({day => 26, month =>  2}),
    _h_nyepi     ({day => 16, month =>  3}, {hyear=>1932}),
    _h_goodfri   ({day =>  2, month =>  4}),
    _h_vesakha   ({day => 28, month =>  5}, {hyear=>2554}),
    _h_ascension ({day =>  2, month =>  6}),
    _h_isramiraj ({day => 10, month =>  7}),
    ($eidulf2010 =
    _h_eidulf    ({day => 10, month =>  9}, {hyear=>1931, day=>1})),
    _h_eidulf    ({day => 11, month =>  9}, {hyear=>1931, day=>2}),
    _h_eidula    ({day => 17, month => 11}),
    _h_hijra     ({day =>  7, month => 12}, {hyear=>1932}),

    _jointlv     ({day =>  9, month =>  9}, {holiday=>$eidulf2010}),
    _jointlv     ({day => 13, month =>  9}, {holiday=>$eidulf2010}),
    _jointlv     ({day => 24, month => 12}, {holiday=>$christmas}),
];

# decreed jun 15, 2010
my $eidulf2011;
$year_holidays{2011} = [
    _h_chnewyear ({day =>  3, month =>  2}, {hyear=>2562}),
    _h_mawlid    ({day => 16, month =>  2}),
    _h_nyepi     ({day =>  5, month =>  3}, {hyear=>1933}),
    _h_goodfri   ({day => 22, month =>  4}),
    _h_vesakha   ({day => 17, month =>  5}, {hyear=>2555}),
    _h_ascension ({day =>  2, month =>  6}),
    _h_isramiraj ({day => 29, month =>  6}),
    ($eidulf2011 =
    _h_eidulf    ({day => 30, month =>  8}, {hyear=>1932, day=>1})),
    _h_eidulf    ({day => 31, month =>  8}, {hyear=>1932, day=>2}),
    _h_eidula    ({day =>  7, month => 11}),
    _h_hijra     ({day => 27, month => 11}, {hyear=>1933}),

    _jointlv     ({day => 29, month =>  8}, {holiday=>$eidulf2011}),
    _jointlv     ({day =>  1, month =>  9}, {holiday=>$eidulf2011}),
    _jointlv     ({day =>  2, month =>  9}, {holiday=>$eidulf2011}),
    _jointlv     ({day => 26, month => 12}, {holiday=>$christmas}),
];

# decreed may 16, 2011
my $eidulf2012;
$year_holidays{2012} = _make_jl_tentative [
    _h_chnewyear ({day => 23, month =>  1}, {hyear=>2563}),
    _h_mawlid    ({day =>  4, month =>  2}),
    _h_nyepi     ({day => 23, month =>  3}, {hyear=>1934}),
    _h_goodfri   ({day =>  6, month =>  4}),
    _h_vesakha   ({day =>  6, month =>  5}, {hyear=>2556}),
    _h_ascension ({day => 17, month =>  5}),
    _h_isramiraj ({day => 16, month =>  6}),
    ($eidulf2012 =
    _h_eidulf    ({day => 19, month =>  8}, {hyear=>1933, day=>1})),
    _h_eidulf    ({day => 20, month =>  8}, {hyear=>1933, day=>2}),
    _h_eidula    ({day => 26, month => 10}),
    _h_hijra     ({day => 15, month => 11}, {hyear=>1934}),

    _jointlv     ({day => 21, month =>  8}, {holiday=>$eidulf2012}),
    _jointlv     ({day => 22, month =>  8}, {holiday=>$eidulf2012}),
    _jointlv     ({day => 26, month => 12}, {holiday=>$christmas}),
];

my @years     = sort keys %year_holidays;
our $min_year = $years[0];
our $max_year = $years[-1];
our $max_joint_leave_year;
for my $y (reverse @years) {
    if (grep {$_->{is_joint_leave}} @{$year_holidays{$y}}) {
        $max_joint_leave_year = $y;
        last;
    }
}

my @holidays;
for my $year ($min_year .. $max_year) {
    my @hf;
    for my $h0 (@fixed_holidays) {
        my $h = clone $h0;
        push @{$h->{tags}}, "fixed-date";
        $h->{is_holiday}     = 1;
        $h->{is_joint_leave} = 0;
        push @hf, $h;
    }

    my @hy;
    for my $h0 (@{$year_holidays{$year}}) {
        my $h = clone $h0;
        $h->{is_holiday}     //= 0;
        $h->{is_joint_leave} //= 0;
        delete $h->{id_name0};
        delete $h->{en_name0};
        push @hy, $h;
    }

    for my $h (@hf, @hy) {
        $h->{year} = $year;
        my $dt = DateTime->new(year=>$year, month=>$h->{month}, day=>$h->{day});
        $h->{date} = $dt->ymd;
        $h->{dow}  = $dt->day_of_week;
    }

    push @holidays, (sort {$a->{date} cmp $b->{date}} @hf, @hy);
}

my $res = gen_read_table_func(
    table_data => \@holidays,
    table_spec => {
        columns => {
            date => ['str*'=>{
                column_index => 0,
                column_searchable => 0,
            }],
            day => ['int*'=>{
                column_index => 1,
            }],
            month => ['int*'=>{
                column_index => 2,
            }],
            year => ['int*'=>{
                column_index => 3,
            }],
            dow => ['int*'=>{
                summary => 'Day of week (1-7, Monday is 1)',
                column_index => 4,
            }],
            en_name => ['str*'=>{
                summary => 'English name',
                column_index => 5,
                column_filterable => 0,
                column_sortable => 0,
            }],
            id_name => ['str*'=>{
                summary => 'Indonesian name',
                column_index => 6,
                column_filterable => 0,
                column_sortable => 0,
            }],
            en_aliases => ['array*'=>{
                summary => 'English other names, if any',
                column_index => 7,
                column_filterable => 0,
                column_sortable => 0,
            }],
            id_aliases => ['array*'=>{
                summary => 'Indonesian other names, if any',
                column_index => 8,
                column_filterable => 0,
                column_sortable => 0,
            }],
            is_holiday => ['bool*'=>{
                column_index => 9,
            }],
            is_joint_leave => ['bool*'=>{
                summary => 'Whether this date is a joint leave day '.
                    '("cuti bersama")',
                column_index => 10,
            }],
            tags => ['array*'=>{
                column_index => 11,
                column_sortable => 0,
            }],
        },
        pk => 'date',
    },
);

die "BUG: Can't generate func: $res->[0] - $res->[1]"
    unless $res->[0] == 200;

my $spec = $res->[2]{spec};
$spec->{summary} = "List Indonesian holidays in calendar";
$spec->{description} = <<"_";

List holidays and joint leave days ("cuti bersama").

Contains data from years $min_year to $max_year (joint leave days until
$max_joint_leave_year).

_
$SPEC{list_id_holidays} = $spec;
no warnings;
*list_id_holidays = $res->[2]{code};
use warnings;

$SPEC{enum_id_workdays} = {
    summary => 'Enumerate working days in a certain period (month/year)',
    description => <<'_',

Working day is defined as day that is not Saturday*/Sunday/holiday/joint leave
days*. If work_saturdays is set to true, Saturdays are also counted as working
days. If observe_joint_leaves is set to false, joint leave days are also counted
as working days.

_
    args => {
        month => ['int*' => {
            # defaults to current month
        }],
        year => ['int*' => {
            # defaults to current year
        }],
        work_saturdays => ['bool' => {
            summary => 'If set to 1, Saturday is a working day',
            default => 0,
        }],
        observe_joint_leaves => ['bool' => {
            summary => 'If set to 0, do not observe joint leave as holidays',
            default => 1,
        }],
    },
};
sub enum_id_workdays {
    my %args = @_;

    # XXX args
    my $now = DateTime->now;
    my $year  = $args{year}  // $now->year;
    my $month = $args{month} // $now->month;
    my $work_saturdays = $args{work_saturdays} // 0;
    my $observe_joint_leaves = $args{observe_joint_leaves} // 1;
    my $eom = DateTime->new(year=>$year, month=>$month, day=>1)->
        add(months=>1)->subtract(days=>1);

    my @args = (year=>$year, month=>$month);
    push @args, (is_holiday=>1) if !$observe_joint_leaves;
    my $res = list_id_holidays(@args);
    return [500, "Can't list holidays: $res->[0] - $res->[1]"]
        unless $res->[0] == 200;
    #use Data::Dump; dd $res;

    my @wd;
    for my $day (1..$eom->day) {
        my $dt = DateTime->new(day=>$day, month=>$month, year=>$year);
        next if $dt->day_of_week == 7;
        next if $dt->day_of_week == 6 && !$work_saturdays;
        my $ymd = sprintf "%04d-%02d-%02d", $year, $month, $day;
        next if $ymd ~~ @{$res->[2]};
        push @wd, $ymd;
    }

    [200, "OK", \@wd];
}

$spec = clone($SPEC{enum_id_workdays});
$spec->{summary} = "Count working days in a certain period (month/year)";
$SPEC{count_id_workdays} = $spec;
sub count_id_workdays {
    my $res = enum_id_workdays(@_);
    return $res unless $res->[0] == 200;
    $res->[2] = @{$res->[2]};
    $res;
}

1;


=pod

=head1 NAME

Calendar::Indonesia::Holiday - List Indonesian public holidays

=head1 VERSION

version 0.02

=head1 SYNOPSIS

 use Calendar::Indonesia::Holiday qw(
     list_id_holidays
     enum_id_workdays
     count_id_workdays
 );

 # list Indonesian holidays for the year 2011, without the joint leave days
 # ("cuti bersama"), show only the dates

 my $res = list_id_holidays(year => 2011, is_joint_leave=>0);
 # sample result
 [200, "OK", [
   '2011-01-01',
   '2011-02-03',
   '2011-02-16',
   '2011-03-05',
   '2011-04-22',
   '2011-05-17',
   '2011-06-02',
   '2011-06-29',
   '2011-08-17',
   '2011-08-31',
   '2011-09-01',
   '2011-11-07',
   '2011-11-27',
   '2011-12-25',
 ]];

 # list religious Indonesian holidays, show full details
 my $res = list_id_holidays(year => 2011,
                            has_tags => ['religious'], detail=>1);

 # sample result
 [200, "OK", [
   {date       => '2011-02-16',
    day        => 16,
    month      => 2,
    year       => 2011,
    id_name    => 'Maulid Nabi Muhammad',
    en_name    => 'Mawlid',
    en_aliases => ['Mawlid An-Nabi'],
    id_aliases => ['Maulud'],
    is_holiday => 1,
    tags       => [qw/religious religion=islam calendar=lunar/],
   },
   ...
 ]];

 # check whether 2011-02-16 is a holiday
 my $res = list_id_holidays(date => '2011-02-16');
 print "2011-02-16 is a holiday\n" if @{$res->[2]};

 # enumerate working days for a certain period
 my $res = enum_id_workdays(year=>2011, month=>7);

 # idem, but returns a number instead. year/month defaults to current
 # year/month.
 my $res = count_id_workdays();

=head1 DESCRIPTION

This module provides one function: B<list_id_holidays>.

This module uses L<Log::Any> logging framework.

This module's functions has L<Sub::Spec> specs.

=head1 FUNCTIONS

None are exported by default, but they are exportable.

=head2 count_id_workdays(%args) -> [STATUS_CODE, ERR_MSG, RESULT]


Count working days in a certain period (month/year).

Working day is defined as day that is not Saturday*/Sunday/holiday/joint leave
days*. If work_saturdays is set to true, Saturdays are also counted as working
days. If observe_joint_leaves is set to false, joint leave days are also counted
as working days.

Returns a 3-element arrayref. STATUS_CODE is 200 on success, or an error code
between 3xx-5xx (just like in HTTP). ERR_MSG is a string containing error
message, RESULT is the actual result.

Arguments (C<*> denotes required arguments):

=over 4

=item * B<month>* => I<int>

=item * B<observe_joint_leaves> => I<bool> (default C<1>)

If set to 0, do not observe joint leave as holidays.

=item * B<work_saturdays> => I<bool> (default C<0>)

If set to 1, Saturday is a working day.

=item * B<year>* => I<int>

=back

=head2 enum_id_workdays(%args) -> [STATUS_CODE, ERR_MSG, RESULT]


Enumerate working days in a certain period (month/year).

Working day is defined as day that is not Saturday*/Sunday/holiday/joint leave
days*. If work_saturdays is set to true, Saturdays are also counted as working
days. If observe_joint_leaves is set to false, joint leave days are also counted
as working days.

Returns a 3-element arrayref. STATUS_CODE is 200 on success, or an error code
between 3xx-5xx (just like in HTTP). ERR_MSG is a string containing error
message, RESULT is the actual result.

Arguments (C<*> denotes required arguments):

=over 4

=item * B<month>* => I<int>

=item * B<observe_joint_leaves> => I<bool> (default C<1>)

If set to 0, do not observe joint leave as holidays.

=item * B<work_saturdays> => I<bool> (default C<0>)

If set to 1, Saturday is a working day.

=item * B<year>* => I<int>

=back

=head2 list_id_holidays(%args) -> [STATUS_CODE, ERR_MSG, RESULT]


List Indonesian holidays in calendar.

List holidays and joint leave days ("cuti bersama").

Contains data from years 2008 to 2012 (joint leave days until
2012).

Returns a 3-element arrayref. STATUS_CODE is 200 on success, or an error code
between 3xx-5xx (just like in HTTP). ERR_MSG is a string containing error
message, RESULT is the actual result.

Field selection arguments (C<*> denotes required arguments):

=over 4

=item * B<detail> => I<bool> (default C<0>)

Return detailed data (all fields).

=item * B<fields> => I<array>

Select fields to return.

When off, will return an array of values without field names (array/list).

Default is off, will be turned on by default when 'fields' or 'detail' options
are specified.

=item * B<show_field_names> => I<bool>

Show field names in result (as hash/assoc).

When off, will return an array of values without field names (array/list). When
on, will return an array of field names and values (hash/associative array).

=back

Filter arguments:

=over 4

=item * B<q> => I<str>

Search.

=back

Filter for date arguments:

=over 4

=item * B<date> => I<str>

Only return results having certain value of date.

=item * B<date_contain> => I<str>

Only return results with date containing certain text.

=item * B<date_match> => I<str>

Only return results with date matching specified regex.

=item * B<date_not_contain> => I<str>

Only return results with date not containing certain text.

=item * B<date_not_match> => I<str>

Only return results with date matching specified regex.

=item * B<max_date> => I<str>

Only return results having a certain maximum value of date.

=item * B<min_date> => I<str>

Only return results having a certain minimum value of date.

=back

Filter for day arguments:

=over 4

=item * B<day> => I<int>

Only return results having certain value of day.

=item * B<max_day> => I<int>

Only return results having a certain maximum value of day.

=item * B<min_day> => I<int>

Only return results having a certain minimum value of day.

=back

Filter for dow arguments:

=over 4

=item * B<dow> => I<int>

Only return results having certain value of dow.

=item * B<max_dow> => I<int>

Only return results having a certain maximum value of dow.

=item * B<min_dow> => I<int>

Only return results having a certain minimum value of dow.

=back

Filter for is_holiday arguments:

=over 4

=item * B<is_holiday> => I<bool> (default C<0>)

Only return results having a true is_holiday value.

=back

Filter for is_joint_leave arguments:

=over 4

=item * B<is_joint_leave> => I<bool> (default C<0>)

Only return results having a true is_joint_leave value.

=back

Filter for month arguments:

=over 4

=item * B<max_month> => I<int>

Only return results having a certain maximum value of month.

=item * B<min_month> => I<int>

Only return results having a certain minimum value of month.

=item * B<month> => I<int>

Only return results having certain value of month.

=back

Filter for tags arguments:

=over 4

=item * B<has_tags> => I<array>

Only return results having specified values in tags.

=item * B<lacks_tags> => I<array>

Only return results not having specified values in tags.

=back

Filter for year arguments:

=over 4

=item * B<max_year> => I<int>

Only return results having a certain maximum value of year.

=item * B<min_year> => I<int>

Only return results having a certain minimum value of year.

=item * B<year> => I<int>

Only return results having certain value of year.

=back

Order arguments:

=over 4

=item * B<random> => I<bool> (default C<0>)

If on, return result in random order.

=item * B<sort> => I<str>

Order data according to certain fields.

A list of field names separated by comma. Each field can be prefixed with '-' to
specify descending order instead of the default ascending.

=back

Paging arguments:

=over 4

=item * B<result_limit> => I<int>

Only return a certain number of results.

=item * B<result_start> => I<int> (default C<1>)

Only return results from a certain position.

=back

=head1 FAQ

=head2 What is "joint leave"?

Workers are normally granted 12 days of paid leave per year. They are free to
spend it on whichever days they want. The joint leave ("cuti bersama") is a
government program of recent years (since 2008) to recommend that some of these
leave days be spent together nationally on certain days, especially during
Lebaran (Eid Ul-Fitr). It is not mandated, but many do follow it anyway, e.g.
government civil workers, banks, etc. I am marking joint leave days with
is_joint_leave=1 and is_holiday=0, while the holidays themselves with
is_holiday=1, so you can differentiate/select both/either one.

=head2 Holidays before 2008?

Will be provided if there is demand and data source.

=head2 Why only provide data up to (current year)+1?

Some religious holidays, especially Vesakha, are not determined yet. Joint leave
days are also usually decreed by the government in May/June of the preceding
year.

=head1 SEE ALSO

This API is available on GudangAPI, http://www.gudangapi.com/ , under
"calendar/id" module. To use GudangAPI, you can use L<WWW::GudangAPI>.

=head1 AUTHOR

Steven Haryanto <stevenharyanto@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2011 by Steven Haryanto.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut


__END__

