package Business::Tax::ID::PPH21;

our $DATE = '2015-12-22'; # DATE
our $VERSION = '0.01'; # VERSION

use 5.010001;
use strict;
use warnings;

use Exporter::Rinci qw(import);

our %SPEC;

our %arg_tp_status = (
    tp_status => {
        summary => 'Taxypayer status',
        description => <<'_',

Taypayer status reflects his/her marital status and affect the amount of
his/her non-taxable income.

_
        schema => ['str*', in=>[
            'TK/0', 'TK/1', 'TK/2', 'TK/3',
            'K/0' , 'K/1',  'K/2',  'K/3',
        ]],
        req => 1,
    },
);

our %arg_year = (
    year => {
        schema => ['int*', min=>1983],
        req => 1,
        pos => 0,
    },
);

our %arg_net_income = (
    net_income => {
        summary => 'Yearly net income',
        schema => ['float*', min=>0],
        req => 1,
    },
);

$SPEC{':package'} = {
    v => 1.1,
    summary => 'Routines to help calculating Indonesian income tax article 21 (PPh pasal 21)',
    description => <<'_',

The law ("undang-undang") for income tax ("pajak penghasilan") in Indonesia is
UU 7/1983 (along with its several amendments, the latest of which is UU
36/2008). This law is comprised of several articles ("pasal"). Article 21
("pasal 21") regulates earned income, which is income generated by
individual/statutory bodies from performed work/services, including: employment
(salary and benefits, as well as severance pay), freelance work, business,
pension, and "jaminan hari tua" (life insurance paid to dependents when a worker
is deceased). Article 21 also regulates some types of passive income, which is
income generated from capital or other non-work sources, including: savings
interests, gifts/lotteries, royalties.

Some other passive income like rent and dividends are regulated in article 23
("pasal 23") instead of article 21. And some articles regulate other aspects or
special cases, e.g. income tax for sales to government agencies/import/specific
industries ("pasal 22"), rules regarding monthly tax payments ("pasal 25"), or
rules regarding work earned in Indonesia by non-citizens ("pasal 26").

This module contains several routines to help calculate income tax article 21.

_

};

$SPEC{get_pph21_op_rates} = {
    v => 1.1,
    summary => 'Get tax rates for PPh21 for individuals ("OP", "orang pribadi")',
    description => <<'_',

PPh21 differentiates rates between individuals ("OP", "orang pribadi") and
statutory bodies ("badan"). Both are progressive. This routine returns the tax
rates for individuals.

Keywords: tax rates, tax brackets.

_
    'description.alt.lang.id_ID' => <<'_',

Kata kunci: tarif pajak, lapisan pajak.

_
    args => {
        year => {
            schema => ['int*', min=>1983],
            req => 1,
            pos => 0,
        },
    },
    examples => [
        {args=>{year=>2015}},
    ],
};
sub get_pph21_op_rates {
    my %args = @_;
    my $year = $args{year};
    if ($year >= 2009 && $year <= 2015) {
        state $res = [
            200, "OK",
            [
                {                   max=> 50_000_000, rate=>0.05},
                {xmin=> 50_000_000, max=>250_000_000, rate=>0.15},
                {xmin=>250_000_000, max=>500_000_000, rate=>0.25},
                {xmin=>500_000_000,                   rate=>0.30},
            ],
            {'table.fields' => [qw/xmin max rate/]},
        ];
        return $res;
    } elsif ($year >= 2000 && $year <= 2008) {
        state $res = [
            200, "OK",
            [
                {                   max=> 25_000_000, rate=>0.05},
                {xmin=> 25_000_000, max=> 50_000_000, rate=>0.10},
                {xmin=> 50_000_000, max=>100_000_000, rate=>0.15},
                {xmin=>100_000_000, max=>200_000_000, rate=>0.25},
                {xmin=>200_000_000,                   rate=>0.35},
            ],
            {'table.fields' => [qw/xmin max rate/]},
        ];
        return $res;
    } else {
        return [412, "Year unknown or unsupported"];
    }
}

# TODO: get_pph21_badan_rates

$SPEC{get_pph21_op_ptkp} = {
    v => 1.1,
    summary => 'Get PPh21 non-taxable income amount ("PTKP") for individuals',
    description => <<'_',

When calculating individual income tax, the net income is subtracted by this
amount first. This means that if a person has income below this amount, he/she
does not need to pay income tax.

_
    'description.alt.lang.id_ID' => <<'_',

Kata kunci: penghasilan tidak kena pajak.

_
    args => {
        %arg_year,
    },
    examples => [
        {args=>{year=>2015}},
    ],
};
sub get_pph21_op_ptkp {
    my %args = @_;

    my $tp_status = $args{tp_status};
    my $year = $args{year};

    my $code_make = sub {
        my ($base, $add) = @_;
        return {
            map {
                ("TK/$_" => $base + $add*$_,
                 "K/$_"  => $base + $add + $add*$_)
            } 0..3
        };
    };

    if ($year >= 2015 && $year <= 2015) {
        state $res = [200, "OK", $code_make->( 36_000_000, 3_000_000)];
        return $res;
    } elsif ($year >= 2013 && $year <= 2014) {
        state $res = [200, "OK", $code_make->( 24_300_000, 2_025_000)];
        return $res;
    } elsif ($year >= 2009 && $year <= 2012) {
        state $res = [200, "OK", $code_make->( 15_840_000, 1_320_000)];
        return $res;
    } elsif ($year >= 2006 && $year <= 2008) {
        state $res = [200, "OK", $code_make->( 13_200_000, 1_200_000)];
        return $res;
    } elsif ($year >= 2005 && $year <= 2005) {
        state $res = [200, "OK", $code_make->( 12_000_000, 1_200_000)];
        return $res;
    } elsif ($year >= 2001 && $year <= 2004) {
        state $res = [200, "OK", $code_make->(  2_880_000, 1_440_000)];
        return $res;
    } elsif ($year >= 1994 && $year <= 2000) {
        state $res = [200, "OK", $code_make->(  1_728_000,   864_000)];
        return $res;
    } elsif ($year >= 1983 && $year <= 1994) {
        state $res = [200, "OK", $code_make->(    960_000,   480_000)];
        return $res;
    } else {
        return [412, "Year unknown or unsupported"];
    }
}

sub _min { $_[0] < $_[1] ? $_[0] : $_[1] }

$SPEC{calc_pph21_op} = {
    v => 1.1,
    summary => 'Calculate PPh 21 for individuals ("OP", "orang pribadi")',
    args => {
        %arg_year,
        %arg_tp_status,
        %arg_net_income,
    },
};
sub calc_pph21_op {
    my %args = @_;

    my $year       = $args{year};
    my $tp_status  = $args{tp_status};
    my $net_income = $args{net_income};

    my $res;

    $res = get_pph21_op_ptkp(year => $year);
    return $res unless $res->[0] == 200;
    my $ptkps = $res->[2];
    my $ptkp = $ptkps->{$tp_status}
        or die "BUG: Can't get PTKP for '$tp_status'";

    my $pkp = $net_income - $ptkp;
    return [200, "OK", 0] if $pkp <= 0;

    $res = get_pph21_op_rates(year => $year);
    return $res unless $res->[0] == 200;
    my $brackets = $res->[2];

    my $tax = 0;
    for my $bracket (@$brackets) {
        if (defined $bracket->{max}) {
            $tax += (_min($pkp, $bracket->{max}) -
                         ($bracket->{xmin} // 0)) * $bracket->{rate};
            last if $pkp <= $bracket->{max};
        } else {
            $tax += ($pkp - $bracket->{xmin}) * $bracket->{rate};
            last;
        }
    }

    [200, "OK", $tax];
}

1;
# ABSTRACT: Routines to help calculating Indonesian income tax article 21 (PPh pasal 21)

__END__

=pod

=encoding UTF-8

=head1 NAME

Business::Tax::ID::PPH21 - Routines to help calculating Indonesian income tax article 21 (PPh pasal 21)

=head1 VERSION

This document describes version 0.01 of Business::Tax::ID::PPH21 (from Perl distribution Business-Tax-ID-PPH21), released on 2015-12-22.

=head1 SYNOPSIS

=head1 DESCRIPTION


The law ("undang-undang") for income tax ("pajak penghasilan") in Indonesia is
UU 7/1983 (along with its several amendments, the latest of which is UU
36/2008). This law is comprised of several articles ("pasal"). Article 21
("pasal 21") regulates earned income, which is income generated by
individual/statutory bodies from performed work/services, including: employment
(salary and benefits, as well as severance pay), freelance work, business,
pension, and "jaminan hari tua" (life insurance paid to dependents when a worker
is deceased). Article 21 also regulates some types of passive income, which is
income generated from capital or other non-work sources, including: savings
interests, gifts/lotteries, royalties.

Some other passive income like rent and dividends are regulated in article 23
("pasal 23") instead of article 21. And some articles regulate other aspects or
special cases, e.g. income tax for sales to government agencies/import/specific
industries ("pasal 22"), rules regarding monthly tax payments ("pasal 25"), or
rules regarding work earned in Indonesia by non-citizens ("pasal 26").

This module contains several routines to help calculate income tax article 21.

=head1 FUNCTIONS


=head2 calc_pph21_op(%args) -> [status, msg, result, meta]

Calculate PPh 21 for individuals ("OP", "orang pribadi").

This function is not exportable.

Arguments ('*' denotes required arguments):

=over 4

=item * B<net_income>* => I<float>

Yearly net income.

=item * B<tp_status>* => I<str>

Taxypayer status.

Taypayer status reflects his/her marital status and affect the amount of
his/her non-taxable income.

=item * B<year>* => I<int>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)


=head2 get_pph21_op_ptkp(%args) -> [status, msg, result, meta]

Get PPh21 non-taxable income amount ("PTKP") for individuals.

Examples:

=over

=item * Example #1:

 get_pph21_op_ptkp( year => 2015);

Result:

 [
   200,
   "OK",
   {
     "K/0"  => 39000000,
     "K/1"  => 42000000,
     "K/2"  => 45000000,
     "K/3"  => 48000000,
     "TK/0" => 36000000,
     "TK/1" => 39000000,
     "TK/2" => 42000000,
     "TK/3" => 45000000,
   },
 ]

=back

When calculating individual income tax, the net income is subtracted by this
amount first. This means that if a person has income below this amount, he/she
does not need to pay income tax.

This function is not exportable.

Arguments ('*' denotes required arguments):

=over 4

=item * B<year>* => I<int>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)


=head2 get_pph21_op_rates(%args) -> [status, msg, result, meta]

Get tax rates for PPh21 for individuals ("OP", "orang pribadi").

Examples:

=over

=item * Example #1:

 get_pph21_op_rates( year => 2015);

Result:

 [
   200,
   "OK",
   [
     { max => 50000000, rate => 0.05 },
     { max => 250000000, rate => 0.15, xmin => 50000000 },
     { max => 500000000, rate => 0.25, xmin => 250000000 },
     { rate => 0.3, xmin => 500000000 },
   ],
   { "table.fields" => ["xmin", "max", "rate"] },
 ]

=back

PPh21 differentiates rates between individuals ("OP", "orang pribadi") and
statutory bodies ("badan"). Both are progressive. This routine returns the tax
rates for individuals.

Keywords: tax rates, tax brackets.

This function is not exportable.

Arguments ('*' denotes required arguments):

=over 4

=item * B<year>* => I<int>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)

=head1 SEE ALSO

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Business-Tax-ID-PPH21>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Business-Tax-ID-PPH21>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Business-Tax-ID-PPH21>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
