package Bencher::Scenario::LevenshteinModules;

our $DATE = '2016-10-14'; # DATE
our $VERSION = '0.11'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

our $scenario = {
    summary => 'Benchmark various modules calculating the Levenshtein edit distance',
    modules => {
        'Text::Levenshtein' => {version => 0.11},
    },
    participants => [
        {
            fcall_template => "PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)",
            tags => ['no_unicode_support'],
        },
        {
            fcall_template => "Text::Levenshtein::fastdistance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::XS::distance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::Flexible::levenshtein(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::LevenshteinXS::distance(<word1>, <word2>)",
            tags => ['no_unicode_support'],
        },
    ],
    datasets => [
        { name=>"a",       args => {word1=>"a"      , word2=>"aa"},      result => 1 },
        { name=>"foo",     args => {word1=>"foo"    , word2=>"bar"},     result => 3 },
        { name=>"program", args => {word1=>"program", word2=>"porgram"}, result => 2 },
        { name=>"reve"   , args => {word1=>"reve"   , word2=>"rêves"},   result => 2, tags=>['unicode'], exclude_participant_tags=>['no_unicode_support'] },
        { name=>"euro"   , args => {word1=>"Euro"   , word2=>"€uro"},    result => 1, tags=>['unicode'], exclude_participant_tags=>['no_unicode_support'] },
    ],
    on_result_failure => 'warn',
};

1;
# ABSTRACT: Benchmark various modules calculating the Levenshtein edit distance

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::LevenshteinModules - Benchmark various modules calculating the Levenshtein edit distance

=head1 VERSION

This document describes version 0.11 of Bencher::Scenario::LevenshteinModules (from Perl distribution Bencher-Scenarios-LevenshteinModules), released on 2016-10-14.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m LevenshteinModules

To run module startup overhead benchmark:

 % bencher --module-startup -m LevenshteinModules

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<PERLANCAR::Text::Levenshtein> 0.02

L<Text::Levenshtein> 0.13

L<Text::Levenshtein::Flexible> 0.09

L<Text::Levenshtein::XS> 0.503

L<Text::LevenshteinXS> 0.03

=head1 BENCHMARK PARTICIPANTS

=over

=item * PERLANCAR::Text::Levenshtein::editdist (perl_code) [no_unicode_support]

Function call template:

 PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)



=item * Text::Levenshtein::fastdistance (perl_code)

Function call template:

 Text::Levenshtein::fastdistance(<word1>, <word2>)



=item * Text::Levenshtein::XS::distance (perl_code)

Function call template:

 Text::Levenshtein::XS::distance(<word1>, <word2>)



=item * Text::Levenshtein::Flexible::levenshtein (perl_code)

Function call template:

 Text::Levenshtein::Flexible::levenshtein(<word1>, <word2>)



=item * Text::LevenshteinXS::distance (perl_code) [no_unicode_support]

Function call template:

 Text::LevenshteinXS::distance(<word1>, <word2>)



=back

=head1 BENCHMARK DATASETS

=over

=item * a

=item * foo

=item * program

=item * reve [unicode]

=item * euro [unicode]

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.24.0 >>, CPU: I<< Intel(R) Core(TM) i5-2400 CPU @ 3.10GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default options (C<< bencher -m LevenshteinModules >>):

 #table1#
 +------------------------------------------+---------+-----------+-----------+------------+---------+---------+
 | participant                              | dataset | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +------------------------------------------+---------+-----------+-----------+------------+---------+---------+
 | Text::Levenshtein::fastdistance          | program |   11000   | 88        |    1       | 2.1e-07 |      20 |
 | PERLANCAR::Text::Levenshtein::editdist   | program |   22766.4 | 43.9243   |    2.00492 | 1.2e-11 |      20 |
 | Text::Levenshtein::fastdistance          | reve    |   26000   | 38        |    2.3     | 5.3e-08 |      20 |
 | Text::Levenshtein::fastdistance          | euro    |   29000   | 34        |    2.6     | 5.1e-08 |      22 |
 | Text::Levenshtein::fastdistance          | foo     |   52000   | 19        |    4.6     | 8.7e-08 |      20 |
 | PERLANCAR::Text::Levenshtein::editdist   | foo     |   88055   | 11.357    |    7.7546  | 1.1e-11 |      20 |
 | Text::Levenshtein::fastdistance          | a       |  170000   |  6.1      |   15       | 6.5e-09 |      21 |
 | PERLANCAR::Text::Levenshtein::editdist   | a       |  348500   |  2.869    |   30.69    | 5.4e-11 |      20 |
 | Text::Levenshtein::XS::distance          | program |  480000   |  2.09     |   42.2     | 8.3e-10 |      20 |
 | Text::Levenshtein::XS::distance          | reve    |  614650   |  1.6269   |   54.129   | 1.1e-11 |      30 |
 | Text::Levenshtein::XS::distance          | euro    |  660600   |  1.514    |   58.18    | 4.7e-11 |      25 |
 | Text::Levenshtein::XS::distance          | foo     |  732018   |  1.36609  |   64.465   |   0     |      20 |
 | Text::Levenshtein::XS::distance          | a       |  898180   |  1.1134   |   79.098   | 9.1e-12 |      22 |
 | Text::LevenshteinXS::distance            | program | 3180000   |  0.315    |  280       | 2.1e-10 |      20 |
 | Text::Levenshtein::Flexible::levenshtein | program | 4170000   |  0.24     |  367       | 9.2e-11 |      26 |
 | Text::Levenshtein::Flexible::levenshtein | euro    | 4360000   |  0.229    |  384       | 4.2e-11 |      20 |
 | Text::Levenshtein::Flexible::levenshtein | reve    | 6200000   |  0.16     |  540       | 4.1e-10 |      23 |
 | Text::LevenshteinXS::distance            | foo     | 7250000   |  0.138    |  639       | 5.2e-11 |      20 |
 | Text::Levenshtein::Flexible::levenshtein | foo     | 7520000   |  0.133    |  662       |   1e-10 |      21 |
 | Text::Levenshtein::Flexible::levenshtein | a       | 8358530   |  0.119638 |  736.092   |   0     |      32 |
 | Text::LevenshteinXS::distance            | a       | 8880000   |  0.113    |  782       | 5.7e-11 |      22 |
 +------------------------------------------+---------+-----------+-----------+------------+---------+---------+


Benchmark module startup overhead (C<< bencher -m LevenshteinModules --module-startup >>):

 #table2#
 +------------------------------+-----------+------------------------+------------+---------+---------+
 | participant                  | time (ms) | mod_overhead_time (ms) | vs_slowest |  errors | samples |
 +------------------------------+-----------+------------------------+------------+---------+---------+
 | Text::LevenshteinXS          |      14   |      9.3               |        1   | 4.3e-05 |      21 |
 | Text::Levenshtein            |      13   |      8.3               |        1.1 | 5.9e-05 |      20 |
 | Text::Levenshtein::Flexible  |      12   |      7.3               |        1.2 | 7.1e-05 |      20 |
 | Text::Levenshtein::XS        |       8.8 |      4.1               |        1.7 | 4.3e-05 |      20 |
 | PERLANCAR::Text::Levenshtein |       5.1 |      0.399999999999999 |        2.8 | 2.1e-05 |      23 |
 | perl -e1 (baseline)          |       4.7 |      0                 |        3.1 | 2.7e-05 |      20 |
 +------------------------------+-----------+------------------------+------------+---------+---------+

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-LevenshteinModules>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-LevenshteinModules>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-LevenshteinModules>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<Bencher::Scenario::DamerauLevenshteinModules>

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
