package Bencher::Scenario::URIEscaping;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2020-06-18'; # DATE
our $DIST = 'Bencher-Scenario-URIEscaping'; # DIST
our $VERSION = '0.005'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

# we do this so URI::XSEscape does not override URI::Escape's functions, because
# obviously we want to test both
$ENV{PERL_URI_XSESCAPE} = 0;

our $scenario = {
    summary => 'Benchmark URI escaping using various modules',

    #precision => 0.001,

    participants => [
        {fcall_template => 'URI::Escape::uri_escape(<str>)', tags=>['escape']},
        {fcall_template => 'URI::Escape::uri_escape_utf8(<str>)', tags=>['escape', 'utf8']},
        {fcall_template => 'URI::Escape::uri_unescape(<str>)', tags=>['unescape']},
        {fcall_template => 'URI::XSEscape::uri_escape(<str>)', tags=>['escape']},
        {fcall_template => 'URI::XSEscape::uri_escape_utf8(<str>)', tags=>['escape', 'utf8']},
        {fcall_template => 'URI::XSEscape::uri_unescape(<str>)', tags=>['unescape']},
        {fcall_template => 'URI::Escape::XS::uri_escape(<str>)', tags=>['escape']},
        #{fcall_template => 'URI::Escape::XS::uri_escape_utf8(<str>)', tags=>['escape', 'utf8']},
        {fcall_template => 'URI::Escape::XS::uri_unescape(<str>)', tags=>['unescape']},

        {fcall_template => 'URI::Encode::uri_encode(<str>)', tags=>['escape']},
        {fcall_template => 'URI::Encode::uri_decode(<str>)', tags=>['unescape']},
        {fcall_template => 'URI::Encode::XS::uri_encode(<str>)', tags=>['escape']},
        {fcall_template => 'URI::Encode::XS::uri_decode(<str>)', tags=>['unescape']},
    ],

    datasets => [
        {
            name => 'empty',
            tags => ['escape'],
            include_participant_tags => ['escape'],
            args => {str=>''},
        },
        # sample data from URI-XSEscape distribution
        {
            name => 'ascii53',
            tags => ['escape'],
            include_participant_tags => ['escape'],
            args => {str=>'I said this: you / them ~ us & me _will_ "do-it" NOW!'},
        },
        # sample data from URI-XSEscape distribution
        {
            name => 'utf36',
            tags => ['escape', 'utf8'],
            include_participant_tags => ['escape & utf8'],
            args => {str=>'http://www.google.co.jp/search?q=小飼弾'},
        },
        # sample data from URI-XSEscape distribution
        {
            name => 'u_ascii53',
            tags => ['unescape'],
            include_participant_tags => ['unescape'],
            args => {str=>'I%20said%20this%3a%20you%20%2f%20them%20~%20us%20%26%20me%20_will_%20%22do-it%22%20NOW%21'},
        },

        # sample data from URI-Escape-XS distribution
        {
            name => 'ascii66',
            tags => ['escape'],
            include_participant_tags => ['escape'],
            args => {str=>'https://stackoverflow.com/questions/3629212/how can perls xsub die'},
        },
        # sample data from URI-Escape-XS distribution
        {
            name => 'u_ascii66',
            tags => ['unescape'],
            include_participant_tags => ['unescape'],
            args => {str=>'https%3A%2F%2Fstackoverflow.com%2Fquestions%2F3629212%2Fhow%20can%20perls%20xsub%20die'},
        },
    ],
};

1;
# ABSTRACT: Benchmark URI escaping using various modules

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::URIEscaping - Benchmark URI escaping using various modules

=head1 VERSION

This document describes version 0.005 of Bencher::Scenario::URIEscaping (from Perl distribution Bencher-Scenario-URIEscaping), released on 2020-06-18.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m URIEscaping

To run module startup overhead benchmark:

 % bencher --module-startup -m URIEscaping

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<URI::Escape> 3.31

L<URI::XSEscape> 0.001000

L<URI::Escape::XS> 0.14

L<URI::Encode> 1.1.1

L<URI::Encode::XS> 0.11

=head1 BENCHMARK PARTICIPANTS

=over

=item * URI::Escape::uri_escape (perl_code) [escape]

Function call template:

 URI::Escape::uri_escape(<str>)



=item * URI::Escape::uri_escape_utf8 (perl_code) [escape, utf8]

Function call template:

 URI::Escape::uri_escape_utf8(<str>)



=item * URI::Escape::uri_unescape (perl_code) [unescape]

Function call template:

 URI::Escape::uri_unescape(<str>)



=item * URI::XSEscape::uri_escape (perl_code) [escape]

Function call template:

 URI::XSEscape::uri_escape(<str>)



=item * URI::XSEscape::uri_escape_utf8 (perl_code) [escape, utf8]

Function call template:

 URI::XSEscape::uri_escape_utf8(<str>)



=item * URI::XSEscape::uri_unescape (perl_code) [unescape]

Function call template:

 URI::XSEscape::uri_unescape(<str>)



=item * URI::Escape::XS::uri_escape (perl_code) [escape]

Function call template:

 URI::Escape::XS::uri_escape(<str>)



=item * URI::Escape::XS::uri_unescape (perl_code) [unescape]

Function call template:

 URI::Escape::XS::uri_unescape(<str>)



=item * URI::Encode::uri_encode (perl_code) [escape]

Function call template:

 URI::Encode::uri_encode(<str>)



=item * URI::Encode::uri_decode (perl_code) [unescape]

Function call template:

 URI::Encode::uri_decode(<str>)



=item * URI::Encode::XS::uri_encode (perl_code) [escape]

Function call template:

 URI::Encode::XS::uri_encode(<str>)



=item * URI::Encode::XS::uri_decode (perl_code) [unescape]

Function call template:

 URI::Encode::XS::uri_decode(<str>)



=back

=head1 BENCHMARK DATASETS

=over

=item * empty [escape]

=item * ascii53 [escape]

=item * utf36 [escape, utf8]

=item * u_ascii53 [unescape]

=item * ascii66 [escape]

=item * u_ascii66 [unescape]

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.30.2 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux LinuxMint version 19 >>, OS kernel: I<< Linux version 4.15.0-91-generic >>.

Benchmark with default options (C<< bencher -m URIEscaping >>):

 #table1#
 +--------------------------------+-----------+--------------+--------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | dataset   | ds_tags      | p_tags       | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-----------+--------------+--------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | URI::Encode::uri_decode        | u_ascii53 | unescape     | unescape     |      3600 |  280      |                 0.00% |            834128.52% | 4.5e-07 |      23 |
 | URI::Encode::uri_encode        | ascii53   | escape       | escape       |      3600 |  280      |                 0.32% |            831493.00% | 4.2e-07 |      21 |
 | URI::Encode::uri_encode        | ascii66   | escape       | escape       |      3600 |  280      |                 1.28% |            823568.11% | 6.6e-07 |      22 |
 | URI::Encode::uri_decode        | u_ascii66 | unescape     | unescape     |      3700 |  270      |                 2.58% |            813173.26% | 4.8e-07 |      20 |
 | URI::Encode::uri_encode        | empty     | escape       | escape       |      3800 |  260      |                 6.14% |            785883.33% | 4.8e-07 |      20 |
 | URI::Escape::uri_escape_utf8   | ascii53   | escape       | escape, utf8 |    140000 |    7.15   |              3816.41% |             21200.83% | 2.7e-09 |      30 |
 | URI::Escape::uri_escape        | ascii53   | escape       | escape       |    144000 |    6.93   |              3936.82% |             20565.50% | 3.3e-09 |      20 |
 | URI::Escape::uri_unescape      | u_ascii53 | unescape     | unescape     |    164390 |    6.083  |              4501.06% |             18031.23% | 2.5e-11 |      24 |
 | URI::Escape::uri_escape_utf8   | utf36     | escape, utf8 | escape, utf8 |    168000 |    5.97   |              4589.58% |             17688.97% | 1.5e-09 |      26 |
 | URI::Escape::uri_escape_utf8   | ascii66   | escape       | escape, utf8 |    219000 |    4.57   |              6026.25% |             13517.27% | 1.5e-09 |      24 |
 | URI::Escape::uri_escape        | ascii66   | escape       | escape       |    230000 |    4.35   |              6332.56% |             12868.85% | 1.6e-09 |      23 |
 | URI::Escape::uri_unescape      | u_ascii66 | unescape     | unescape     |    280000 |    3.6    |              7604.67% |             10727.56% |   5e-09 |      20 |
 | URI::Escape::uri_escape_utf8   | empty     | escape       | escape, utf8 |   1700000 |    0.59   |             47092.42% |              1667.72% | 8.3e-10 |      20 |
 | URI::Escape::XS::uri_escape    | ascii53   | escape       | escape       |   2500000 |    0.41   |             68577.90% |              1114.70% | 5.5e-10 |      26 |
 | URI::Escape::XS::uri_escape    | ascii66   | escape       | escape       |   2462000 |    0.4062 |             68799.80% |              1110.79% | 2.4e-11 |      31 |
 | URI::Escape::uri_escape        | empty     | escape       | escape       |   2508000 |    0.3987 |             70104.43% |              1088.28% | 2.2e-11 |      20 |
 | URI::XSEscape::uri_escape_utf8 | ascii66   | escape       | escape, utf8 |   2690000 |    0.372  |             75194.07% |              1007.96% | 2.1e-10 |      20 |
 | URI::XSEscape::uri_escape_utf8 | ascii53   | escape       | escape, utf8 |   2730000 |    0.366  |             76397.44% |               990.53% | 1.7e-10 |      31 |
 | URI::Escape::XS::uri_unescape  | u_ascii53 | unescape     | unescape     |   2862000 |    0.3494 |             79994.19% |               941.56% | 1.7e-11 |      20 |
 | URI::XSEscape::uri_escape_utf8 | utf36     | escape, utf8 | escape, utf8 |   2976000 |    0.336  |             83197.91% |               901.50% | 2.4e-11 |      20 |
 | URI::Escape::XS::uri_escape    | empty     | escape       | escape       |   3051000 |    0.3278 |             85291.12% |               876.95% | 5.8e-12 |      23 |
 | URI::Escape::XS::uri_unescape  | u_ascii66 | unescape     | unescape     |   3150000 |    0.317  |             88177.67% |               845.01% | 1.1e-10 |      20 |
 | URI::XSEscape::uri_escape_utf8 | empty     | escape       | escape, utf8 |   3750000 |    0.267  |            104754.54% |               695.61% | 8.6e-11 |      32 |
 | URI::XSEscape::uri_unescape    | u_ascii53 | unescape     | unescape     |   4920000 |    0.203  |            137523.90% |               506.17% | 2.4e-11 |      20 |
 | URI::XSEscape::uri_unescape    | u_ascii66 | unescape     | unescape     |   4983000 |    0.2007 |            139356.66% |               498.20% | 1.7e-11 |      20 |
 | URI::XSEscape::uri_escape      | ascii66   | escape       | escape       |   5500000 |    0.18   |            152651.86% |               446.13% | 2.7e-10 |      27 |
 | URI::XSEscape::uri_escape      | ascii53   | escape       | escape       |   5700000 |    0.18   |            159472.86% |               422.79% | 2.6e-10 |      29 |
 | URI::XSEscape::uri_escape      | empty     | escape       | escape       |  10000000 |    0.1    |            278783.59% |               199.13% | 1.6e-10 |      20 |
 | URI::Encode::XS::uri_encode    | ascii53   | escape       | escape       |  12000000 |    0.083  |            335845.34% |               148.32% | 2.1e-10 |      20 |
 | URI::Encode::XS::uri_encode    | ascii66   | escape       | escape       |  12000000 |    0.0831 |            336757.14% |               147.65% | 2.4e-11 |      24 |
 | URI::Encode::XS::uri_decode    | u_ascii66 | unescape     | unescape     |  12000000 |    0.083  |            336990.50% |               147.48% | 2.1e-10 |      20 |
 | URI::Encode::XS::uri_decode    | u_ascii53 | unescape     | unescape     |  12700000 |    0.0789 |            354725.57% |               135.11% |   6e-11 |      23 |
 | URI::Encode::XS::uri_encode    | empty     | escape       | escape       |  30000000 |    0.034  |            834128.52% |                 0.00% | 7.8e-11 |      20 |
 +--------------------------------+-----------+--------------+--------------+-----------+-----------+-----------------------+-----------------------+---------+---------+


Benchmark module startup overhead (C<< bencher -m URIEscaping --module-startup >>):

 #table2#
 +---------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | participant         | time (ms) | mod_overhead_time | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +---------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | URI::Encode         |      20   |              15   |                 0.00% |               241.33% |   0.00021 |      20 |
 | URI::Escape         |      10   |               5   |                71.67% |                98.83% |   0.00024 |      20 |
 | URI::Encode::XS     |       8   |               3   |               118.60% |                56.14% |   0.00012 |      20 |
 | URI::Escape::XS     |       8   |               3   |               128.19% |                49.58% |   0.00011 |      21 |
 | URI::XSEscape       |       5.8 |               0.8 |               198.63% |                14.30% | 2.2e-05   |      20 |
 | perl -e1 (baseline) |       5   |               0   |               241.33% |                 0.00% |   0.00014 |      20 |
 +---------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+


To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 BENCHMARK NOTES

L<URI::Encode::XS> is the fastest, but it does not support custom list of safe
characters. If you don't want to encode C</> as C<%2F>, for example, you're out
of luck. For URI escaping with custom character list support, the fastest is
L<URI::XSEscape> followed by L<URI::Escape::XS>.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-URIEscaping>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-URIEscaping>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-URIEscaping>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2020, 2017, 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
