package Bencher::Scenario::LevenshteinModules;

our $DATE = '2016-01-13'; # DATE
our $VERSION = '0.08'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

our $scenario = {
    summary => 'Benchmark various modules calculating the Levenshtein edit distance',
    modules => {
        'Text::Levenshtein' => 0.11,
    },
    participants => [
        {
            fcall_template => "PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)",
            tags => ['no_unicode_support'],
        },
        {
            fcall_template => "Text::Levenshtein::fastdistance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::XS::distance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::LevenshteinXS::distance(<word1>, <word2>)",
            tags => ['no_unicode_support'],
        },
        {
            fcall_template => "Text::Levenshtein::Damerau::PP::pp_edistance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::Damerau::XS::xs_edistance(<word1>, <word2>)",
        },
    ],
    datasets => [
        { name=>"a",       args => {word1=>"a"      , word2=>"aa"},      result => 1 },
        { name=>"foo",     args => {word1=>"foo"    , word2=>"bar"},     result => 3 },
        { name=>"program", args => {word1=>"program", word2=>"porgram"}, result => 2 },
        { name=>"reve"   , args => {word1=>"reve"   , word2=>"rêves"},   result => 2, tags=>['unicode'], exclude_participant_tags=>['no_unicode_support'] },
        { name=>"euro"   , args => {word1=>"Euro"   , word2=>"€uro"},    result => 1, tags=>['unicode'], exclude_participant_tags=>['no_unicode_support'] },
    ],
    on_result_failure => 'warn',
};

1;
# ABSTRACT: Benchmark various modules calculating the Levenshtein edit distance


__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::LevenshteinModules - Benchmark various modules calculating the Levenshtein edit distance

=head1 VERSION

This document describes version 0.08 of Bencher::Scenario::LevenshteinModules (from Perl distribution Bencher-Scenario-LevenshteinModules), released on 2016-01-13.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m LevenshteinModules

To run module startup overhead benchmark:

 % bencher --module-startup -m LevenshteinModules

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

L<PERLANCAR::Text::Levenshtein> 0.02

L<Text::Levenshtein> 0.13

L<Text::Levenshtein::Damerau::PP> 0.25

L<Text::Levenshtein::Damerau::XS> 3.0

L<Text::Levenshtein::XS> 0.502

L<Text::LevenshteinXS> 0.03

=head1 BENCHMARK PARTICIPANTS

=over

=item * PERLANCAR::Text::Levenshtein::editdist (perl_code)

Function call template:

 PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)



=item * Text::Levenshtein::fastdistance (perl_code)

Function call template:

 Text::Levenshtein::fastdistance(<word1>, <word2>)



=item * Text::Levenshtein::XS::distance (perl_code)

Function call template:

 Text::Levenshtein::XS::distance(<word1>, <word2>)



=item * Text::LevenshteinXS::distance (perl_code)

Function call template:

 Text::LevenshteinXS::distance(<word1>, <word2>)



=item * Text::Levenshtein::Damerau::PP::pp_edistance (perl_code)

Function call template:

 Text::Levenshtein::Damerau::PP::pp_edistance(<word1>, <word2>)



=item * Text::Levenshtein::Damerau::XS::xs_edistance (perl_code)

Function call template:

 Text::Levenshtein::Damerau::XS::xs_edistance(<word1>, <word2>)



=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default option:

 +-----+----------------------------------------------+---------+-------------+-------------+---------+---------+-----------------------------------+
 | seq | participant                                  | dataset | rate        | time        | errors  | samples | notes                             |
 +-----+----------------------------------------------+---------+-------------+-------------+---------+---------+-----------------------------------+
 | 5   | Text::Levenshtein::fastdistance              | program | 1.87e+04    | 53.6μs      | 2.2e-08 | 30      |                                   |
 | 18  | Text::Levenshtein::Damerau::PP::pp_edistance | program | 22490.6     | 44.4629μs   | 0       | 20      | Result (1) is not as expected (2) |
 | 2   | PERLANCAR::Text::Levenshtein::editdist       | program | 3.02e+04    | 33.1μs      | 1.1e-08 | 28      |                                   |
 | 6   | Text::Levenshtein::fastdistance              | reve    | 4.37e+04    | 22.88μs     | 6.5e-09 | 21      |                                   |
 | 19  | Text::Levenshtein::Damerau::PP::pp_edistance | reve    | 4.973e+04   | 20.11μs     | 5.8e-09 | 26      |                                   |
 | 7   | Text::Levenshtein::fastdistance              | euro    | 50488.4     | 19.8065μs   | 0       | 26      |                                   |
 | 20  | Text::Levenshtein::Damerau::PP::pp_edistance | euro    | 5.12e+04    | 19.5μs      | 2e-08   | 20      |                                   |
 | 4   | Text::Levenshtein::fastdistance              | foo     | 8.812e+04   | 11.35μs     | 3e-09   | 24      |                                   |
 | 17  | Text::Levenshtein::Damerau::PP::pp_edistance | foo     | 93597.4     | 10.6841μs   | 0       | 20      |                                   |
 | 1   | PERLANCAR::Text::Levenshtein::editdist       | foo     | 1.18e+05    | 8.49μs      | 8.9e-09 | 25      |                                   |
 | 16  | Text::Levenshtein::Damerau::PP::pp_edistance | a       | 288770      | 3.46296μs   | 0       | 20      |                                   |
 | 3   | Text::Levenshtein::fastdistance              | a       | 2.89e+05    | 3.46μs      | 1.3e-09 | 32      |                                   |
 | 0   | PERLANCAR::Text::Levenshtein::editdist       | a       | 451280      | 2.21592μs   | 0       | 20      |                                   |
 | 23  | Text::Levenshtein::Damerau::XS::xs_edistance | program | 5.86e+05    | 1.71μs      | 3.1e-09 | 23      | Result (1) is not as expected (2) |
 | 10  | Text::Levenshtein::XS::distance              | program | 648738      | 1.54146μs   | 0       | 20      |                                   |
 | 24  | Text::Levenshtein::Damerau::XS::xs_edistance | reve    | 765250      | 1.30676μs   | 0       | 20      |                                   |
 | 25  | Text::Levenshtein::Damerau::XS::xs_edistance | euro    | 819046      | 1.22093μs   | 0       | 23      |                                   |
 | 11  | Text::Levenshtein::XS::distance              | reve    | 846049      | 1.18196μs   | 0       | 20      |                                   |
 | 12  | Text::Levenshtein::XS::distance              | euro    | 9.15e+05    | 1.09μs      | 1.6e-09 | 21      |                                   |
 | 22  | Text::Levenshtein::Damerau::XS::xs_edistance | foo     | 939304      | 1.06462μs   | 0       | 20      |                                   |
 | 9   | Text::Levenshtein::XS::distance              | foo     | 1.01139e+06 | 0.988739μs  | 0       | 20      |                                   |
 | 21  | Text::Levenshtein::Damerau::XS::xs_edistance | a       | 1.27e+06    | 0.789μs     | 1.2e-09 | 21      |                                   |
 | 8   | Text::Levenshtein::XS::distance              | a       | 1.27894e+06 | 0.781895μs  | 0       | 33      |                                   |
 | 15  | Text::LevenshteinXS::distance                | program | 4.47707e+06 | 0.22336μs   | 0       | 20      |                                   |
 | 14  | Text::LevenshteinXS::distance                | foo     | 9.42e+06    | 0.106μs     | 4.6e-11 | 20      |                                   |
 | 13  | Text::LevenshteinXS::distance                | a       | 1.18368e+07 | 0.0844821μs | 0       | 23      |                                   |
 +-----+----------------------------------------------+---------+-------------+-------------+---------+---------+-----------------------------------+


Benchmark module startup overhead:

 +-----+--------------------------------+-------+-------------------+---------+---------+
 | seq | participant                    | time  | mod_overhead_time | errors  | samples |
 +-----+--------------------------------+-------+-------------------+---------+---------+
 | 4   | Text::LevenshteinXS            | 8.9ms | 6.4ms             | 3.3e-05 | 20      |
 | 2   | Text::Levenshtein              | 7.4ms | 4.9ms             | 3.1e-05 | 20      |
 | 3   | Text::Levenshtein::XS          | 5.2ms | 2.7ms             | 1.9e-05 | 20      |
 | 5   | Text::Levenshtein::Damerau::PP | 4.3ms | 1.8ms             | 2.7e-05 | 20      |
 | 6   | Text::Levenshtein::Damerau::XS | 3.9ms | 1.4ms             | 3e-05   | 20      |
 | 1   | PERLANCAR::Text::Levenshtein   | 2.8ms | 0.28ms            | 1.9e-05 | 21      |
 | 0   | perl -e1 (baseline)            | 2.5ms | 0ms               | 1.7e-05 | 20      |
 +-----+--------------------------------+-------+-------------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-LevenshteinModules>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-LevenshteinModules>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-LevenshteinModules>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut

