=head1 B<P6C::IMCC>

IMCC.pm is still a "ball of mud" at the moment, but context
propagation has been moved to Context.pm.  Next for refactoring is
symbol handling.

Context should not be propagated during code generation, as the
context propagation pass handles this.  This rule is broken for
hyper-operators, since I can't think of a good way to handle them
using context.

Code is generated by a depth-first recursive traversal of the op tree.
Each node type should define a C<val> function to be called by its
parent node.  This function should gather values from child nodes (by
calling their C<val> functions), then emit the code for the node's
operation (using C<P6C::Compiler::code>).  Code is appended to the
current function in the order in which it is generated, so subnodes
must be evaluated in the proper order.

C<val> should return one of the following:

=over

=item * undef if the node has no rvalue.

=item * a reference to an array of registers if called in tuple
context.

=item * the name of a PMC register holding the rvalue otherwise.

=back

Node types that can act as lvalues should define an C<assign> function
that takes an unevaluated rvalue tree and a context structure.  This
function should return a PMC register or array ref (like C<val>) if
(like an assignment) it serves as both an lvalue and an rvalue.

=head2 External interface

If C<P6C::IMCC> is imported with the ":external" flag, it will define
the following interface, used by the driver:

=over

=item B<init()>

Initialize or reset compiler state.  This should be called before
generating any code.  C<init> destroys all functions and globals,
resets the current function, and reinitializes builtins.

=item B<compile($top)>

Compile a tree based at $top, but do not emit any code.

=item B<emit()>

Emit IMCC code on standard output, including a header that calls
C<main>, and the code for any builtin functions (see
C<P6C::Builtins>).  C<emit> will fail if you have not defined C<main>.

=back

=cut

package P6C::IMCC;
use strict;
use P6C::Builtins;
use P6C::Util;
use P6C::Context;

# Map Perl types to IMCC parameter types:
my %paramtype = (int => 'int',
		 str => 'string',
		 num => 'float');
# Map Perl types to Parrot register types:
my %regtype   = (int => 'I',
		 str => 'S',
		 num => 'N');
sub regtype(*) {
    $regtype{$_[0]} || 'P';
}

sub paramtype(*) {
    $paramtype{$_[0]} || $_[0];
}

sub import {
    my ($class, $type) = @_;
    my @syms = qw(globalvar localvar paramvar findvar
		  add_localvar add_param
		  push_scope pop_scope
		  gentmp genlabel newtmp
		  code
		  add_function set_function exists_function_def
		  exists_function_decl set_function_params
		  gen_counted_loop do_scalar_to_array do_flatten_array);
    my @external = qw(init compile emit);
    my $caller = caller;
    no strict 'refs';
    if ($type eq ':all') {
	foreach (@syms) {
	    *{$caller . '::' . $_} = \&$_;
	}
    } elsif ($type eq ':external') {
	foreach (@external) {
	    *{$caller . '::' . $_} = \&$_;
	}
    }
    1;
}

our $curfunc;			# currently compiling function
our %funcs;			# all known functions
our %globals;			# global variables

sub init {			# reset state
    %funcs = ();
    %globals = ();
    undef $curfunc;
    P6C::Builtins::declare(\%funcs);
}

sub compile {			# compile input (don't emit)
    my $x = shift;
    my $ctx = new P6C::Context type => 'void';
    use P6C::Addcontext;
    if (ref $x eq 'ARRAY') {
	# propagate context:
	foreach my $stmt (@$x) {
	    $stmt->ctx_right($ctx);
	}
	# generate code:
	foreach my $stmt (@$x) {
	    $stmt->val;
	}
    } else {
	# probably single stmt.
	$x->ctx_right($ctx);
	$x->val;
    }
}

sub emit {			# emit all code
    die "Must define main" unless $funcs{main};
    print <<'END';
.sub _main
	call main
	end
	ret
END
    P6C::Builtins::add_code(\%funcs);
    while (my ($name, $sub) = each %funcs) {
	unless($sub->code) {
	    print STDERR "Skipping empty sub $name (builtin or external?)\n";
	    next;
	}
	print ".sub $name\n";
	$sub->emit;
    }
    P6C::Builtins::emit;
}

=head2 Internals

If C<P6C::IMCC> is imported with the ":all" flag, it exports an
internal interface.

The compiler maintains a "current function" (could be generalized to
"current scope") in which code is emitted, locals are declared, and
symbol lookups begin.  The following functions manipulate the current
function context.

=over

=item B<code($x)>

Append IMCC code C<$x> to the current function.

=item B<add_function($name)>

Create a new function stub for C<$name>.  If C<$name> exists, it will
be overwritten.

=item B<exists_function_def($name)>

Return true if function C<$name> is defined (i.e. not just "declared").

=item B<exists_function_decl($name)>

Return true if a stub exists for C<$name>, even if it has no code.

=item B<$oldfunc = set_function($name)>

Set the code insertion point to the end of function C<$name>,
returning the name of the previously active function.  Function
C<$name> should exist before this is called.

=item B<set_function_params(@params)>

Set the parameter list for the current function.  The arguments should
be a list of C<P6C::param> objects.  XXX: there is currently no way to
handle variable parameter lists.  This is a limitation of the current
parameter-passing scheme, not just this interface.

=back

=cut

sub code {			# add code to current function
    die "Code must live within a function" unless defined $curfunc;
    $funcs{$curfunc}->code .= join "\n", @_;
}

sub add_function($) {
    my $f = shift;
    if (exists $funcs{$f}) {
	diag "Redefining function $f";
    }
    $funcs{$f} = new P6C::IMCC::Sub;
    # NOTE: top-level closure will push a scope.
    1;
}

sub exists_function_def($) {
    my $f = $funcs{+shift};
    return $f && $f->code;
}

sub exists_function_decl($) {
    return $funcs{+shift} ? 1 : 0;
}

sub set_function($) {	       # switch to function, returning old one
    my $func = shift;
    my $ofunc = $curfunc;
    $curfunc = $func;
    return $ofunc;
}

sub set_function_params {
    for my $p (@_) {
	push @{$funcs{$curfunc}->args()},
	    [$p->var->type, $p->var->mangled_name];
    }
}

=head2 Name lookup

This is a primitive symbol table.  Which is okay, since Parrot doesn't
have stashes yet.  Hopefully the interface will be useful when things
get more complicated.

=over

=item B<$name = findvar($var)>

=item B<($name, $isglobal) = findvar($var)>

Find variable C<$var>, a C<P6C::variable>, returning a PMC register
containing its value.  Currently C<findvar> looks at the active
function's parameters, then locals, then globals (which don't exist,
so it won't find anything there).  Returns undef if the variable is
not found.  C<$isglobal> is currently unused.

=item B<$name = globalvar($var)>

Lookup a global variable.

=item B<$name = localvar($var)>

Find local variable C<$var>, returning its IMCC name.

=item B<$name = paramvar($var)>

Find parameter C<$var>.

=item B<add_localvar($var)>

Declare local variable C<$var>.  Warns if C<$var> is already defined.
If C<$var-E<gt>type> is a PMC type, C<$var> will automatically be
initialized.

=item B<push_scope()>

Push a scope within the current function.

=item B<pop_scope()>

Pop a scope from the current function.

=back

=cut

sub globalvar($) {
    # Globals always exist...
    # except when they don't
    unimp 'No globals yet';
    my $name = shift->mangled_name;
    $globals{$name} ||= 1;
    $name;
}

sub localvar($) {
    my $var = shift;
    
    die "Local variable outside function" unless defined $curfunc;
    return $funcs{$curfunc}->localvar($var);
}

sub add_localvar($) {
    return $funcs{$curfunc}->add_localvar(@_);
}

sub paramvar($) {
    my $var = shift->mangled_name;
    return $var if grep {
	$_->[1] eq $var;
    } @{$funcs{$curfunc}->args};
    return undef;
}

sub findvar($) {
    my ($var) = @_;
    my $name;
    $name = paramvar($var) || localvar($var) || globalvar($var);
    return wantarray ? ($name, 0) : $name;
}

sub push_scope {
    $funcs{$curfunc}->push_scope;
}

sub pop_scope {
    $funcs{$curfunc}->pop_scope;
}

=head2 Temporary names

=over

=item C<gensym([$str])>

Generate a unique identifier.  If C<$str> is given, include it as part
of the identifier.

=item C<genlabel([$str])>

Generate a unique label containing C<$str>.

=item C<newtmp([$type])>

Create a new temporary register to hold a value of type C<$type>,
which should be "int", "num", "str", or some PMC type.  If C<$type> is
a PMC type, the register will be initialized with a new value.  If
C<$type> is omitted, it default to C<PerlUndef>.

=item C<gentmp([$type])>

Generate an uninitialized temporary register.

=back

=cut

my $lastsym = 0;
my $lasttmp = 0;
my $lastlabel = 0;
sub gensym(;*) {
    'S'.$_[0] . ++$lastsym
} 

sub _gentmp(*) {		# uninitialized temporary (internal)
    '$' . $_[0] . ++$lasttmp
}

sub gentmp(;*) {			# uninitialized temporary
    _gentmp(regtype($_[0] || 'PerlUndef'));
}

sub genlabel(;$) {		# new label (optionally annotated)
    'L_'.$_[0]. ++$lastlabel
}

sub newtmp(;*) {			# initialized temporary
    my $type = shift || 'PerlUndef';
    my $reg = regtype $type;
    my $name;
    if ($reg eq 'S') {
 	$name = _gentmp S;
	code(<<END);
	$name = ""
END
    } elsif ($reg eq 'I' || $reg eq 'N') {
	$name = _gentmp $reg;
	code(<<END);
	$name = 0
END
    } else {
	$name = _gentmp P;
	die unless $type;
	use Carp;
	code(<<END);
	$name = new $type
END
    }
    return $name;
}

=head2 Code generation functions

=over

The following functions generate useful and common pieces of code.

=item B<gen_counted_loop($counter, $body)>

Generate a counted loop using C<$counter> as the repetition count.
The loop will iterate over values between 0 and $counter - 1,
inclusive.  C<$counter> will be used as the iteration variable, so it
can be used in indexing expressions in the loop body.

=item B<do_scalar_to_array($val)>

Emit the code to turn a scalar into a one-element array, returning the
array's name.

=item B<do_flatten_array($vals)>

Emit code to evaluate each item in C<@$vals>, which are assumed to be
in list context.  The results are concatenated into a single array,
whose name is returned.

=back

=cut

sub gen_counted_loop {
    my ($count, $body) = @_;
    my $start = genlabel;
    my $end = genlabel;
    return <<END;
$start:	if $count == 0 goto $end
	dec $count
	$body
	goto $start
$end:
END
}

sub do_scalar_to_array {
    my $x = shift;
    my $a = newtmp 'PerlArray';
    code(<<END);
	$a = 1
	$a\[0] = $x
END
    return $a;
}

sub do_flatten_array {
    my $vals = shift;
    my $tmp = newtmp 'PerlArray';
    my $len = gentmp 'int';
    my $offset = gentmp 'int';
    my $tmpindex = gentmp 'int';
    my $ptmp = newtmp 'PerlUndef';
    code(<<END);
# START array flattening.
	$offset = 0
END
    for my $i (0..$#{$vals}) {
	my $item = $vals->[$i]->val;
	code(<<END);
	$len = $item
END
	code(gen_counted_loop($len, <<END));
	$ptmp = $item\[$len]
	$tmpindex = $offset + $len
	$tmp\[$tmpindex] = $ptmp
END
	code(<<END);
	$len = $item
	$offset = $offset + $len
END
    }
    code("# END array flattening\n");
    return $tmp;
}

=head2 P6C::IMCC::Sub

Stores IMCC code for a subroutine.

XXX: the fact that e.g. C<P6C::prefix> relies on this for argument
information is just wrong.  This information should be retrieved from
the parse tree structures instead.

=over

=item B<code($sub)>

The code (not including C<.local> definitions, etc).  Can be appended
to like C<$func->code .= $thing>.

=item B<emit($sub)>

Emit a complete function body, minus the C<.sub> directive.

=back

=cut

package P6C::IMCC::Sub;
use Class::Struct 'P6C::IMCC::Sub'
    => { scopes => '@',		# scope stack
	 args => '@'		# arguments, in order passed
       };
#	{scopelevel}		# current scope number
#	{oldscopes}		# other closed scopes in this sub.

use P6C::Util 'diag';

sub localvar {
    my ($x, $var) = @_;
    my $name = $var->mangled_name;
    for (@{$x->scopes}) {
	if (exists $_->{$name}) {
	    return $_->{$name}[0];
	}
    }
    return undef;
}

sub add_localvar {
    my ($x, $var) = @_;
    my $name = $var->mangled_name;
    my $scopename = $name.$x->{scopelevel};
    if ($x->scopes->[0]{$name}) {
	diag 'Redeclaring lexical '.$var->name." in $curfunc";
    }
    $x->scopes->[0]{$name} ||= [$scopename, $var->type];
    return $scopename;
}

sub push_scope {
    my $x = shift;
    $x->{scopelevel}++;
    unshift @{$x->scopes}, { };
}

sub pop_scope {
    my $x = shift;
    push @{$x->{oldscopes}}, shift @{$x->scopes};
}

sub code : lvalue {
    my $x = shift;
    $x->{code};
}

sub emit {
    my $x = shift;
    print <<END;
	saveall
# Parameters:
END
    foreach (@{$x->args}) {
	my ($t, $pname) = @$_;
	my $ptype = P6C::IMCC::paramtype($t);
	print <<END;
	.param $ptype	$pname
END
    }
    print "# Named locals:\n";
    for (@{$x->scopes}, @{$x->{oldscopes}}) {
	for my $v (values %$_) {
	    my ($n, $t) = @$v;
	    print "\t.local $t $n\n";
	}
    }
    # Maybe constructors for locals:
    for (@{$x->scopes}, @{$x->{oldscopes}}) {
	for my $v (values %$_) {
	    my ($n, $t) = @$v;
	    next if $t eq '1';	# uninitialized locals
	    print "\t$n = new $t\n"
		if P6C::IMCC::regtype($t) eq 'P';
	}
    }
    print $x->code;
    print <<END;
	restoreall
	ret
END
}

######################################################################
# Node-type code generation functions

##############################
package P6C::Register;

sub val {
    shift->reg;
}

##############################
package P6C::ValueList;
use P6C::IMCC ':all';
use P6C::Util ':all';

# XXX: ValueList::val returns an array-ref in tuple context.  This is
# inconsistent with other C<val> functions, which all return single
# values.  However, if you're creating a tuple context, you should
# know what to expect.

sub val {
    my $x = shift;
    my $ctx = $x->{ctx};
    
    if ($ctx->flatten) {
	# XXX: flatten has to come first.
	# In flattening context, we have to build a new array out of
	# the values.  All the values should have been evaluated in
	# array context, so they will all be PerlArrays.
	return do_flatten_array($x->vals);

    } elsif ($ctx->is_array) {
	# In array context, the list's value is an array of all its
	# elements.
	my $tmp = newtmp 'PerlArray';
	code("\t$tmp = ".@{$x->vals}."\n");
	for my $i (0..$#{$x->vals}) {
	    my $item = $x->vals($i)->val;
	    code(<<END);
	$tmp\[$i] = $item
END
	}
	return $tmp;

    } elsif ($ctx->is_scalar || $ctx->type eq 'void') {
	# The value of a list in scalar context is its last value, but
	# we need to evaluate intermediate expressions for possible
	# side-effects.
	for (@{$x->vals}[0..$#{$x->vals} - 1]) {
	    $_->val;
	}
	return $x->vals($#{$x->vals})->val;

    } elsif ($ctx->is_tuple) {
	# In N-tuple context, the list's value is its first N elements.
	my @ret;
	my $min = @{$x->vals} < $ctx->nelem ? @{$x->vals} : $ctx->nelem;
	for my $i (0..$min - 1) {
	    $ret[$i] = $x->vals($i)->val;
	}
	for my $i ($min .. $#{$x->vals}) {
	    $x->vals($i)->val;
	}
	return [@ret];

    } else {
	use Data::Dumper;
	unimp "Can't handle context ".Dumper($ctx);
    }
}

##############################
package P6C::Binop;
use P6C::IMCC ':all';
use P6C::Util ':all';
use P6C::Context;

# Create generic code for $a op $b.
sub simple_binary {
    my $x = shift;
    my $ltmp = $x->l->val;
    my $rtmp = $x->r->val;
    my $dest = newtmp;
    my $op = $x->op;
    code("\t$dest = $ltmp $op $rtmp\n");
    return $dest;
}

# XXX: exponentiation in a loop.  Will be replaced once IMCC allows more ops.
sub slow_pow {
    my $x = shift;
    my $dest = newtmp;
    my $lv = $x->l->val;
    my $rv = $x->r->val;
    my $cnt = gentmp 'int';
    code(<<END);
# POW
	$dest = 1
	$cnt = $rv
END
    code(gen_counted_loop($cnt, "$dest = $dest * $lv\n"));
    return $dest;
}

# '=' assignment op.
sub do_assign {
    my $x = shift;
    return $x->l->assign($x->r);
}

# short-circuit logical '&&' operator
sub do_logand {
    my $x = shift;
    my $dest = newtmp;
    my $thenlab = genlabel 'logical_and';
    my $endlab = genlabel 'logical_and';
    my $res = $x->l->val;
    code(<<END);
	if $res goto $thenlab
	goto $endlab
$thenlab:
END
    $res = $x->r->val;
    code(<<END);
	$dest = $res
$endlab:
END
    return $dest;		# will be undef if first failed.
}

# Short-circuit logical or.
sub do_logor {
    my $x = shift;
    my $dest = newtmp;
    my $endlab = genlabel 'logical_or';
    my $res = $x->l->val;
    code(<<END);
# LOGICAL OR
	$dest = $res
	if $dest goto $endlab
END
    $res = $x->r->val;
    code(<<END);
	$dest = $res
# END_LOGICAL_OR
$endlab:
END
    return $dest;
}

# Definedness test.  Result is the first defined value, or undef.
sub do_defined {
    my $x = shift;
    my $val = $x->l->val;
    my $itmp = gentmp 'int';
    my $res = newtmp;
    my $endlab = genlabel 'defined';
    code(<<END);
	$res = $val
	$itmp = defined $res
	if $itmp goto $endlab
END
    $val = $x->r->val;
    code(<<END);
	$res = $val
$endlab:
END
    return $res;
}

# String concatenation 
sub do_concat {
    # XXX: The PMC concat doesn't seem to work, so we have to go
    # through strings.
    my $x = shift;
    my $lt = gentmp 'str';
    my $rt = gentmp 'str';
    my $restmp = gentmp 'str';
    my $res = newtmp;
    my $lval = $x->l->val;
    my $rval = $x->r->val;
    code(<<END);
	$lt = $lval
	$rt = $rval
	$restmp = $lt . $rt
	$res = $restmp
END
    return $res;
}

# Handle a comma operator sequence.  Just flattens and calls off to
# C<P6C::ValueList>.
sub do_array {
    my $x = shift;
    use Carp 'cluck';
    cluck "Should provide context to comma operator" unless $x->{ctx};
    my @things = flatten_leftop($x, ',');
    my $vallist = P6C::ValueList->new(vals => \@things);
    $vallist->{ctx} = $x->{ctx};
    return $vallist->val;
}

# 'x' operator.  Waiting for IMCC development, since it's just a
# simple opcode.
sub do_repeat {
    unimp 'repeat';
}

# Binary infix operators.
our %ops =
(
 '+'	=> \&simple_binary,
 '-'	=> \&simple_binary,
 '*'	=> \&simple_binary,
 '/'	=> \&simple_binary,
 '%'	=> \&simple_binary,
 '**'	=> \&slow_pow,

 '>>'	=> \&simple_binary,
 '<<'	=> \&simple_binary,
 '|'	=> \&simple_binary,
 '&'	=> \&simple_binary,
 '~'	=> \&simple_binary,

# '_' => \&simple_binary, # PMC concat broken.
 '_'	=> \&do_concat,
 '='	=> \&do_assign,
 '||'	=> \&do_logor,
 '&&'	=> \&do_logand,
 '~~'	=> \&simple_binary,
 '//'	=> \&do_defined,
 ','	=> \&do_array,
 'x'	=> \&do_repeat,
 '..'	=> \&do_range,
);

sub val {
    my $x = shift;
    if (ref($x->op) eq 'P6C::hype') {
	return do_hyped($x->op->op, $x->l, $x->r);
    }
    if ($ops{$x->op}) {
	return $ops{$x->op}->($x);
    } elsif($x->op =~ /^([^=]+)=$/ && $ops{$1}) {
	# Translate assignment operation into a binary operation.
	# XXX: Context propagation is broken for these, so we won't
	# ever do this.
	return $ops{'='}->(new P6C::Binop op => '=', l => $x->l,
			   r => P6C::Binop->new(op => $1, l => $x->l,
						r => $x->r));
    } else {
	unimp $x->op;
    }
}

# XXX: We go through typed registers instead of PMC registers for some
# hyped operators.  Not sure if this is a good idea.

our %optype;
BEGIN {
    my %opmap = (int => [ qw(>> << | & ~ ~~)],
		 num => [ qw(+ - * / % **)]);
    while (my ($t, $ops) = each %opmap) {
	@optype{@$ops} = ($t) x @$ops;
    }
}

# Generate the loop body to compute "$targ = $lindex $op $rindex".
#
# XXX: may need to re-map op symbols if IMCC and Perl 6 don't agree on
# them.
#
sub simple_hyped {
    my ($op, $targ, $lindex, $rindex) = @_;
    my $optype = $optype{$op} or unimp "Can't hype $op yet";
    my $ltmp = gentmp $optype;
    my $rtmp = gentmp $optype;
    my $dest = gentmp $optype;
    return <<END;
	$ltmp = $lindex
	$rtmp = $rindex
	$dest = $ltmp $op $rtmp
	$targ = $dest
END
}

sub hype_and {
    my ($op, $targ, $lindex, $rindex) = @_;
    my $tmp = newtmp;
    my $middle = genlabel;
    my $end = genlabel;
    return <<END;
	$tmp = $lindex
	if $tmp goto $middle
	goto $end
$middle:
	$tmp = $rindex
	$targ = $tmp
$end:
END
}

sub hype_or {
    my ($op, $targ, $lindex, $rindex) = @_;
    my $tmp = newtmp;
    my $end = genlabel;
    # XXX: targ, lindex, and rindex may be subscripted, so we can't
    # use them directly in the test.
    return <<END;
	$tmp = $lindex
	if $tmp goto $end
	$tmp = $rindex
$end:
	$targ = $tmp
END
}

our %hype_body = ('||' => \&hype_or, '&&' => \&hype_and);

sub hype_body {
    my $op = $_[0];
    if (exists $hype_body{$op}) {
	return $hype_body{$op}->(@_);
    } else {
	return simple_hyped(@_);
    }
}

# Hyped operations promote a scalar left- or right-hand side to an
# array.  XXX: should probably do context, since a hyper-operator in
# tuple or scalar context can do less work.
sub do_hyped {
    my ($op, $l, $r) = @_;
    if (is_array_expr($l) && is_array_expr($r)) {
	return hype_array_array(@_);
    } elsif (is_array_expr($l)) {
	return hype_array_scalar(@_);
    } elsif (is_array_expr($r)) {
	return hype_scalar_array(@_);
    } else {
	diag "Tried to hyper-operate two scalars";
	return simple_binary(@_);
    }
}

# @xs ^op $y
sub hype_array_scalar {
    my ($op, $l, $r) = @_;
    my $lval = $l->val;
    my $rval = $r->val;
    my $len = gentmp 'int';
    my $dest = newtmp 'PerlArray';

    # Initialization code:
    code(<<END);
	$len = $lval
	$dest = $len
END
    my $code = hype_body($op, "$dest\[$len]", "$lval\[$len]", $rval);
    code(gen_counted_loop($len, $code));
    return $dest;
}

# $x ^op @ys
sub hype_scalar_array {
    my ($op, $l, $r) = @_;
    my $lval = $l->val;
    my $rval = $r->val;
    my $len = gentmp 'int';
    my $dest = newtmp 'PerlArray';

    # Initializers:
    code(<<END);
	$len = $rval
	$dest = $len
END
    my $code = hype_body($op, "$dest\[$len]", $lval, "$rval\[$len]");
    code(gen_counted_loop($len, $code));
    return $dest;
}

# @xs ^op @ys
#
# Currently iterates over the number of elements in the _shorter_ of
# the two arrays, rather than the longer.  This is useful for working
# with infinite lists, but may not be the behavior in the Apocalypses
# (XXX: check this).
#
sub hype_array_array {
    my ($op, $l, $r) = @_;
    my $lval = $l->val;
    my $rval = $r->val;
    my $llen = gentmp 'int';
    my $rlen = gentmp 'int';

    my $cntlabel = genlabel;
    my $dest = newtmp 'PerlArray';
    my $looptop = genlabel 'hyper';
    my $loopend = genlabel 'hyper_end';

    # Header to figure out appropriate length.
    code(<<END);
	$llen = $lval
	$rlen = $rval
	if $llen > $rlen goto $cntlabel
	$llen = $rlen
$cntlabel:
	$dest = $llen
END
    my $code
	= hype_body($op, "$dest\[$rlen]", "$lval\[$rlen]", "$rval\[$rlen]");
    code(gen_counted_loop($rlen, $code));
    return $dest;
}

sub do_range {
    my $x = shift;
    my $ctx = $x->{ctx};

    if ($ctx->is_array) {
	# XXX: no way to clone PMC's so we have to go through
	# temporaries to create new values.
	my $ret = newtmp 'PerlArray';
	my $itmp = gentmp 'int';
	my $vtmp = gentmp 'int';
	my $lval = $x->l->val;
	my $rval = $x->r->val;
	my $val = newtmp;
	my $start = genlabel 'range_start';
	my $end = genlabel 'range_end';
	code(<<END);
	$val = $lval
	$itmp = 0
$start:
	if $val > $rval goto $end
	$vtmp = $val
	$ret\[$itmp] = $vtmp
	inc $val
	inc $itmp
	goto $start
$end:
END
	return $ret;

    } elsif ($ctx->is_scalar) {
	# Probably an iterator.  Or maybe that's its own context.  Not
	# sure.  We lose in any case.
	unimp "Range in scalar context.";

    } elsif ($ctx->is_tuple) {
	# generate enough undef's:
	my @ret;
	for (@{$ctx->type}) {
	    push @ret, newtmp;
	}
	# Figure out endpoints, and jump to the end if we go past the end.
	my $lval = $x->l->val;
	my $rval = $x->r->val;
	my $end = genlabel 'range_end';
	my $vtmp = gentmp 'int';
	for my $i (0 .. $#{$ctx->type}) {
	    # XXX: promoting everything to PMC registers.
	    code(<<END);
	if $lval > $rval goto $end
	$vtmp = $lval
	$ret[$i] = $vtmp
	inc $lval
END
	}
	code(<<END);
$end:
END
	return [@ret];

    } else {
	use Data::Dumper;
	unimp "Unsupported range context ".Dumper($ctx->type);
    }
}

######################################################################
package P6C::incr;
use P6C::IMCC ':all';

our %inplace_op = ('++' => 'inc', '--' => 'dec');
our %outaplace_op = ('++' => '+ 1', '--' => '- 1');

sub val {
    my $x = shift;
    my $ret;

    # XXX: I'm extra-cautious here because we may be incrementing a
    # temporary, in which case we have to copy it back.  If this can
    # never happen, then the assigns can be removed here.

    # Optimize post-increment in void context to a pre-increment.
    if ($x->post && !$x->{ctx}->type eq 'void') {
	my $op = $outaplace_op{$x->op}
	    or die $x->op().' increment not understood';
	my $val = $x->thing->val;
	my $tmp = newtmp;
	my $tmp2 = newtmp;
	code(<<END);
	$tmp = $val
	$tmp2 = $val $op
END
	$x->thing->assign(new P6C::Register reg => $tmp2);
	return $tmp;

    } else {
	my $op = $inplace_op{$x->op}
	    or die $x->op().' increment not understood';
	$ret = $x->thing->val;
	code("\t$op $ret\n");
	$x->thing->assign(new P6C::Register reg => $ret);
	return $ret;
    }
}

######################################################################
package P6C::ternary;
use P6C::IMCC ':all';

# Ternary operator as an r-value.  Context-aware.
sub val {
    my $x = shift;
    my $tmp = newtmp;
    my ($thenlab, $endlab) = (genlabel("ternary_then"),
			      genlabel("ternary_end"));
    code(<<END);
# START TERNARY
END
    my $ifval = $x->if->val;
    code(<<END);
	if $ifval goto $thenlab
END
    my $elseval = $x->else->val;
    code(<<END);
	$tmp = $elseval
	goto $endlab
$thenlab:
END
    my $thenval = $x->then->val;
    code(<<END);
	$tmp = $thenval
$endlab:
# END TERNARY
END
    return $tmp;
}

# Ternary operator as an l-value.  Ignores incoming context.  However,
# the r-value to be assigned will be evaluated in the proper context
# for each branch.

# REMEMBER: since the two branches may have different contexts, they
# have different op-trees.

# REMEMBER: we haven't always been able to propagate context, so we
# fall back to just using the same op-tree for both sides.
sub assign {
    my ($x, $thing) = @_;
    my $tmp = newtmp;
    my ($thenlab, $endlab) = (genlabel("ternary_then"),
			      genlabel("ternary_end"));
    code(<<END);
# START TERNARY
END
    my $ifval = $x->if->val;
    code(<<END);
	if $ifval goto $thenlab
END
    my $elseval = $x->else->assign($x->{else_right} || $thing);
    code(<<END);
	$tmp = $elseval
	goto $endlab
$thenlab:
END
    my $thenval = $x->then->assign($x->{then_right} || $thing);
    code(<<END);
	$tmp = $thenval
$endlab:
# END TERNARY
END
    return $tmp;
}

######################################################################
sub P6C::sv_literal::val {
    use P6C::Util ':all';

    my $x = shift;
    return undef if $x->{ctx}->type && $x->{ctx}->type eq 'void';
    my $type = $x->type;
    my $ctx = $x->{ctx};

    # XXX: these are actually _references_.  But we don't support them
    # anyways.
    die "Don't support ".$type if $type =~ /Perl(Hash|Array)/;
    my $val = $x->lval;
    my $ret;
    if (!$ctx->type
	|| $ctx->type eq 'void'
	|| same_type($ctx->type, $type)
	|| (is_scalar($ctx->type) && is_scalar($type))) {
	warn "literal in void context" if $ctx->type eq 'void';
	$ret = newtmp;
	code(<<END);
	$ret = $val
END

    } elsif ($ctx->is_array) {
	$ret = do_scalar_to_array($val);

    } elsif ($ctx->is_tuple) {
	$ret = newtmp;
	code(<<END);
	$ret = $val
END

    } else {
# 	use Data::Dumper;
# 	unimp "Context ", Dumper($ctx);
	# XXX: bogus
	$ret = newtmp;
	code(<<END);
	$ret = $val
END
    }
    return $ret;
}

######################################################################
# Prefix operators (i.e. functions and control structures)
package P6C::prefix;
use P6C::IMCC ':all';
use P6C::Util ':all';
use P6C::Context;

sub val_noarg {
    my $block = shift;
    # XXX: pretend that the block has a no-argument prototype, since
    # otherwise it will complain.  This is the wrong behavior for the
    # topicalizing control structures, but we don't support them yet,
    # anyways.

    my $saveparam = $block->params;
    $block->params(new P6C::params req => [], opt => [], rest => undef);
    $block->val;
    $block->params($saveparam);
}

# if/elsif/elsunless/else sequence
sub prefix_if {
    my $x = shift;
    my $end = genlabel "endif";
    my $tmp = newtmp;
    my $nextlab;
    foreach (@{$x->args}) {
	my ($sense, $test, $block) = @$_;
	$sense ||= $x->name;
	if ($nextlab) {
	    code("$nextlab:\n");
	}
	$nextlab = genlabel 'if';
	if (!ref $test) {
	    val_noarg($block);
	} else {
	    my $v = $test->val;
	    if ($sense =~ /if$/) { # (els)?if
		code(<<END);
	$tmp = ! $v
	if $tmp goto $nextlab
END
	    } else {		# (els)?unless
		code(<<END);
	if $v goto $nextlab
END
	    }
	    val_noarg($block);
	    code(<<END);
	goto $end
END
	}
    }
    code(<<END);
$nextlab:
$end:		# END OF @{[$x->name]}
END
    return undef;
}

sub common_while {
    my ($name, $gentest, $genbody) = @_;
    my $start = genlabel 'start_while';
    my $end = genlabel 'endwhile';
    code(<<END);
$start:
END
    my $testval = $gentest->();
    if ($name eq 'while') {
	my $startbody = genlabel 'while_body';
	code(<<END);
	if $testval goto $startbody
	goto $end
$startbody:
END
    } else {
	code(<<END);
	if $testval goto $end
END
    }
    $genbody->();
    code(<<END);
	goto $start
$end:
END
}

sub prefix_while {
    my $x = shift;
    my ($test, $body) = ($x->args->vals(0), $x->args->vals(1));
    common_while($x->name, sub { $test->val }, sub { val_noarg($body) });
}

# Do a subroutine call.
#
# XXX: currently ignores context.  We don't have a way of
# communicating context to functions anyways, so this isn't a problem.
sub gen_sub_call {
    my ($x) = @_;

    my $func = $P6C::IMCC::funcs{$x->name};
    my $args = $x->args->val;

    # Sometimes function arguments are a tuple, sometimes not.  Make
    # things consistent.
    if (ref($args) ne 'ARRAY') {
	$args = [$args];
    }
    if (@$args != @{$func->args}) {
	# internal error.
	die "Wrong number of arguments for ".$x->name.": got ".@$args
	    .", expected ".@{$func->args};
    }

    foreach (reverse @$args) {
	code("\t.arg	$_\n");
    }
    my $name = $x->name;
    code("\tcall	$name\n");
    return newtmp;		# XXX: return values not implemented.
}

sub prefix_for {
    my ($x) = @_;
    # XXX: apo 4 explicitly says this is lazy, but we take a greedy
    # approach here.
    my ($streams, $body) = @{$x->args->vals};
    unless (ref $streams eq 'ARRAY') {
	use Data::Dumper;
	die Dumper($streams);
    }
    my @bindings = map { [flatten_leftop($_, ',')] }
	flatten_leftop($body->params, ';');
    die "for: internal error" unless @bindings == 1 || @bindings == @$streams;
    
    # XXX: body closure should take care of params, but since we're
    # faking the scope, we need to handle the params here.

    # XXX: we iterate over the shortest length.  Apo 4 doesn't say
    # anything about this, but it's consistent with what we're doing
    # for hyperoperators, and all but necessary if we deal with
    # infinite streams.

    # XXX: There should be a "clean" version for the common case where
    # we're iterating over one stream.

    push_scope;

    my @vars;			# variables to be bound for each iter.
    print STDERR "for: bindings";
    for (@bindings) {
	my @l;
	for my $v (@$_) {
	    push @l, add_localvar($v);
	}
	print STDERR ' '.@l;
	push @vars, [@l];
    }
    print STDERR "\n";

    my @streamvals = map { $_->val } @$streams;
    print STDERR "for ".@streamvals." streams\n";

    ##############################
    if (@bindings == 1) {
	# No semicolons on RHS => alternate across streams:
	@vars = @{$vars[0]};
	my $nstreams = @$streams;
	my $valsrc = newtmp 'PerlArray'; # value streams.
	my $tmpsrc = newtmp 'PerlUndef'; # temp for stream.
	my $stream = gentmp 'int';	# index into streams.
	my $streamoff = gentmp 'int'; # offset within streams.
	my $streamlen = gentmp 'int'; # length of shortest stream.
	my $niters = gentmp 'int';	# number of iterations.
	my $itmp = gentmp 'int';
	my $loopstart = genlabel 'start_for';

	# Initialization:
	code(<<END);
	$stream = 0
	$streamoff = 0
	$streamlen = 2000000000
	$niters = 0
	$valsrc = $nstreams
END
	for my $i (0..$#{$streams}) {
	    my $streamval = $streamvals[$i];
	    my $notless = genlabel;
	    code(<<END);
	$valsrc\[$i] = $streamval
	$itmp = $streamval
	if $itmp > $streamlen goto $notless
	$streamlen = $itmp
$notless:
END
	}

	# Figure out number of iterations:
	my $nvars = @vars;
	code(<<END);
	$niters = $streamlen * $nstreams
	$niters = $niters / $nvars
$loopstart:
END

	# bind variables:
	for my $v (@vars) {
	    my $notnext = genlabel;
	    code(<<END);
	$tmpsrc = $valsrc\[$stream]
	$v = $tmpsrc\[$streamoff]
	inc $stream
	if $stream < $nstreams goto $notnext
	$stream = 0
	inc $streamoff
$notnext:
END
	}

	# Loop body:
	val_noarg($body);
	code(<<END);
	dec $niters
	if $niters > 0 goto $loopstart
END

    ##############################
    } else {
	# Semicolon on RHS => parallel iteration.
	my $niters = gentmp 'int'; # number of iterations
	my @streamoff;		# offset within each stream.
	push(@streamoff, gentmp 'int') for @streamvals;
	my $itmp = gentmp 'int';
	my $loopstart = genlabel 'start_for';
	my $notless = genlabel;
	
	code(<<END);
	$niters = 2000000000
END
	# Figure out how many iterations:
	for my $i (0 .. $#streamvals) {
	    my $nvars = @{$vars[$i]};
	    code(<<END);
	$streamoff[$i] = 0
	$itmp = $streamvals[$i]
	$itmp = $itmp / $nvars
	if $itmp > $niters goto $notless
	$niters = $itmp
$notless:
END
	}
	code(<<END);
$loopstart:
END
	# Bind variables:
	for my $i (0 .. $#streamvals) {
	    for my $j (0 .. $#{$vars[$i]}) {
		code(<<END);
	$vars[$i][$j] = $streamvals[$i]\[$streamoff[$i]]
	inc $streamoff[$i]
END
	    }
	}
	
	# Generate loop body:
	val_noarg($body);
	code(<<END);
	dec $niters
	if $niters > 0 goto $loopstart
END
    }
    pop_scope;
    return undef;
}

# unary minus.
sub prefix_neg {
    my $x = shift;
    my $tmp = $x->args->val;
    my $res = newtmp;
    code(<<END);
	$res = - $tmp
END
    return $res;
}

our %prefix_ops =
(
 'if' => \&prefix_if,
 'unless' => \&prefix_if,
 'while' => \&prefix_while,
 'until' => \&prefix_while,
 'for' => \&prefix_for,
 '-' => \&prefix_neg,
);

sub val {
    my $x = shift;
    # XXX: temporary hack.
    if (exists_function_decl($x->name)) {
	return gen_sub_call($x, @_);
    } elsif (exists $prefix_ops{$x->name}) {
	return $prefix_ops{$x->name}->($x, @_);
    } else {
	unimp "Prefix operator ".$x->name();
    }
}

######################################################################
# Guards
package P6C::guard;
use P6C::IMCC ':all';
use P6C::Util 'unimp';

sub guard_if {
    my $x = shift;
    my $test = $x->test->val;
    my $end = genlabel $x->name;
    if ($x->name eq 'unless') {
	code(<<END);
	if $test goto $end
END
    } else {
	my $foo = genlabel;
	code(<<END);
	if $test goto $foo
	goto $end
$foo:
END
    }
    $x->expr->val;
    code(<<END);
$end:
END
    return undef;
}

sub guard_while {
    my $x = shift;
    P6C::prefix::common_while($x->name,
			      sub { $x->test->val },
			      sub { $x->expr->val });
}

our %guards =
(
 'if' => \&guard_if,
 'unless' => \&guard_if,
 'while' => \&guard_while,
 'until' => \&guard_while,
);

sub val {
    my $x = shift;
    # XXX: temporary hack.
    if (exists $guards{$x->name}) {
	return $guards{$x->name}->($x);
    } else {
	unimp "Guard ".$x->name();
    }
}

######################################################################
# Chained comparisons
package P6C::compare;
use P6C::IMCC ':all';

# XXX: since IMCC doesn't give us access to cmp, cmp_num, and
# cmp_string separately, we need to go through num and str temporaries
# to get the right kind of comparison.
our %type;
BEGIN {
    $type{$_} = 'num' for qw(<= == >= < > !=);
    $type{$_} = 'str' for qw(eq ne ge le lt gt);
}

# remap operator names.
our %imccop;
BEGIN {
    @imccop{qw(<= == >= < > !=)} = qw(<= == >= < > !=);
    @imccop{qw(le eq ge lt gt ne)} = qw(<= == >= < > !=);
}

# Generate one element of a comparison.  Unlike other places, $l and
# $r are already evaluated here (i.e. they may be recycled
# temporaries).  $fail is a label (defined elsewhere) to be branched
# to if the comparison fails.
sub gen_compare {
    my ($op, $l, $r, $fail) = @_;
    my $label = genlabel;
    $op = $imccop{$op};
    code(<<END);
	if $l $op $r goto $label
	goto $fail
$label:
END
}
our %ops;
BEGIN {
    for my $op (qw(<= == >= < > != le eq ge lt gt ne)) {
	$ops{$op} = \&gen_compare;
    }
}

# Evaluate a comparison sequence from left to right.  If any
# comparison fails, it will branch to the end of the sequence.  Since
# only two values are active at once, we only need two PMC
# temporaries.  For each operator, look up the operator type, and put
# the operand values in appropriate N or S temporaries.  As an
# optimization, the type of the previous comparison is kept around,
# and the PMC value of the left operand is not re-fetched if it was
# used as a right operand for the same type of operator.

# NOTE: the typing we do here duplicates what's done during the
# context pass.  Eventually these should be combined.
sub val {
    my $self = shift;
    die "Nothing to compare" if $self->size == 0;
    my $x = $self->seq;
    my $result = gentmp 'int';
    my %tmps;
    $tmps{0} = newtmp;		# PMC temporary for odd terms
    $tmps{2} = newtmp;		# PMC temporary for even terms
    my $res = newtmp;
    my $lasttype;
    my $fail = genlabel 'comparison';
    code("\t$result = 0\n");
    $tmps{0} = $x->[0]->val;
    for (my $i = 1; $i < $#{$x}; $i += 2) {
	my $op = $x->[$i];
	my $type = $type{$op} or die "No such op: $op";
	unless ($tmps{"$type 0"}) {
	    # Initialize temporaries for this type if we haven't already.
	    $tmps{"$type 0"} = gentmp $type;
	    $tmps{"$type 2"} = gentmp $type;
	}

	# Only recompute expression values if necessary
	my $ltmp = $tmps{$type.' '.(($i - 1) % 4)};
	my $rtmp = $tmps{$type.' '.(($i + 1) % 4)};
	if ($lasttype ne $type) {
	    code(<<END);
	$ltmp = $tmps{($i - 1) % 4}
END
	}
	$lasttype = $type;	# store type of right operand
	$tmps{(($i + 1) % 4)} = $x->[$i + 1]->val;
	code(<<END);
	$rtmp = $tmps{(($i + 1) % 4)}
END
	$ops{$op}->($op, $ltmp, $rtmp, $fail);
    }
    code(<<END);
	$result = 1
$fail:
    $res = $result
END
    return $res;
}

######################################################################
sub P6C::sub_def::val {
    use P6C::IMCC ':all';
    use P6C::Util 'diag';
    my $x = shift;

    if (exists_function_def($x->name)) {
	diag "Redefining function ".$x->name;
    }
    add_function($x->name);
    my $ofunc = set_function($x->name);
    $x->closure->val;
    set_function($ofunc);
}

######################################################################
package P6C::closure;
use P6C::Util 'unimp';
use P6C::IMCC ':all';

# A sub with no explicit parameter list gets @_.

# NOTE: This parallels what's done in Addcontext.pm.  These things
# should be integrated.
our $default_params;
BEGIN {
    my $underscore = P6C::variable->new(name => '@_', type => 'PerlArray');
    $default_params = new P6C::params req => [], opt => [],
	rest => P6C::param->new(var => $underscore);
}

sub val {
    my $x = shift;
    unless ($x->params) {
	$x->params($default_params);
    }
    push_scope;
    if ($x->params->isa('P6C::Binop')) {
	# Closure arguments are different, since ';' means "dimension"
	my @params;
	foreach ($x->params->flatten_leftop(';')) {
	    push @params, $_->flatten_leftop(',');
	}
	set_function_params(@params);
    } elsif (!defined $x->params->max) {
	set_function_params($x->params->rest);
    } elsif ($x->params->min != $x->params->max) {
	# Only support variable number of params if it's zero - Inf.
	unimp "Unsupported parameter arity: ",
	    $x->params->min . ' - ' . $x->params->max;
    } else {
	set_function_params(@{$x->params->req});
    }
    if (defined($x->block)) {
	# Real definition, not just declaration.
	foreach my $stmt (@{$x->block}) {
	    $stmt->val;
	}
    }
    pop_scope;
}

######################################################################
package P6C::variable;
use P6C::IMCC ':all';
use P6C::Context;
use P6C::Util qw(is_scalar same_type unimp);

# Name mangling to protect IMCC from Perl's wacky variable names.
sub mangled_name {
    my %mangle = (qw(! _BANG_
		     $ SV_
		     @ AV_
		     % HV_
		     & CV_));
    my $x = shift;
    return $x->{mangled} if exists $x->{mangled};
    my $name = $x->name;
    $name =~ s/([\!\$\@\%\&])/$mangle{$1}/eg;
    return $x->{mangled} = $name;
}

# XXX: need to redo this when we get globals.
sub val {
    my $x = shift;
    my $ctx = $x->{ctx};
    return undef if $ctx->type eq 'void';
    my $v = findvar($x);
    if (!$ctx->type
	|| same_type($x->type, $ctx->type)
	|| (is_scalar($x->type) && is_scalar($ctx->type))
	|| $ctx->is_tuple) {
	return $v;
    } elsif (is_scalar($x->type) && $ctx->is_array) {
	return do_scalar_to_array($v);
    } else {
#	unimp "Variable of type ", $x->type, " in context ", $ctx->type;
	# XXX: bogus.
	return $v;
    }
}

sub assign {
    my ($x, $thing) = @_;
    my $name = findvar($x);
    my $tmpv = $thing->val;
    code("\t$name = $tmpv\n");
    return $name;
}

######################################################################
# Variable declarations, which may have initializers
package P6C::decl;
use P6C::IMCC ':all';
use P6C::Util 'unimp';
use P6C::Context;

sub val {
    my $x = shift;
    if ($x->qual && $x->qual->scope ne 'my') {
	unimp 'global variables';
    }
    if (@{$x->props}) {
	unimp 'variable properties';
    }
    add_localvar($_) for @{$x->vars};
}

# A declaration with initializers shows up as assigning to a decl.
#
# XXX: The shenanigans with temporaries should go away, as they are
# the Wrong Way to make sure that the declared variable is not defined
# within its own initializer.
sub assign {
    my ($x, $thing) = @_;

    my $tmpv = $thing->val;

    # once again, we are evaluating an expression in tuple context, so
    # the val function may return an array ref.
    if (@{$x->vars} == 1) {
	if (ref($tmpv) eq 'ARRAY') {
	    $tmpv = $tmpv->[-1];
	}
	add_localvar($x->vars(0));
	$x->vars(0)->assign(new P6C::Register reg => $tmpv);
    } else {
	if (ref $tmpv ne 'ARRAY') {
	    $tmpv = [$tmpv];
	}
	my $min = @$tmpv < @{$x->vars} ? @$tmpv : @{$x->vars};
	for my $i (0.. $min - 1) {
	    add_localvar($x->vars($i));
	    $x->vars($i)->assign(new P6C::Register reg => $tmpv->[$i]);
	}

	# In case we had more variables than values (tuple rvalues can
	# do this), declare the rest of the variables.
	for my $i ($min .. $#{$x->vars}) {
	    add_localvar($x->vars($i));
	}
	return undef;
    }
}

######################################################################

sub P6C::indices::val {
    my $x = shift;
    my $ctx = $x->{ctx};
    use Data::Dumper;
    return $x->indices->val;
}

##############################
package P6C::subscript_exp;
use P6C::Util 'unimp';
use P6C::IMCC ':all';

# Temporary types for different slices:
our %temptype = qw(PerlArray int PerlHash str);

# Slice value.
#
# XXX: ignores context in bad ways.  For example, "@x = @y[1]" will
# set @x's size, but "@x = @y[1,2,3]" will assign a slice array to @x.
# This should be fixable.
#
# XXX: doesn't handle subscripting by an array, e.g. @x[@y].
#
# XXX: completely broken.  Need to fix this.
sub val {
    my ($x) = @_;
    if (@{$x->subscripts} > 1) {
	# XXX: shouldn't be too hard -- just evaluate subscripts
	# recursively on temporaries.  Not sure how context would work.
	unimp "multi-level subscripting";
    }
    code("# Base for indexing\n");
    my $thing = $x->thing->val;
    my $type = $x->subscripts(0)->type;
    code("# done; result in $thing\n");

    code("# Indexing expression\n");
    my $indexval = $x->subscripts(0)->val;
    code("# done; result in $indexval\n");

    my $ret;
    my $ctx = $x->{ctx};
    if ($ctx->is_scalar) {
	# Scalar context or single item => return last/only item.
	$ret = newtmp;
	my $itmp = gentmp $temptype{$type};
	$indexval = $indexval->[-1] if ref $indexval;
	code(<<END);
	$itmp = $indexval
	$ret = $thing\[$itmp]
END

    } elsif ($ctx->is_array) {
	# Slice in array context.
	$ret = newtmp 'PerlArray';
	my $tmp = gentmp;
	my $itmp = gentmp $temptype{$type};
	my $ptmp = newtmp;
	my $ret_index = gentmp 'int';
	code(<<END);
	$ret_index = $indexval
	$ret = $ret_index
END
	code(gen_counted_loop($ret_index, <<END));
$itmp = $indexval\[$ret_index]
	$ptmp = $thing\[$itmp]
	$ret\[$ret_index] = $ptmp
END

    } elsif ($ctx->is_tuple) {
	my $itmp = gentmp $temptype{$type};
	my @ret;
	die unless ref $indexval eq 'ARRAY';

	for (@$indexval) {
	    my $rettmp = newtmp;
	    push @ret, $rettmp;
	    code(<<END);
	$itmp = $_
	$rettmp = $thing\[$itmp]
END
	}
	$ret = [@ret];

    } else {
	use Data::Dumper;
	confess "slice in unsupported context ".Dumper($ctx);
    }
    return $ret;
}

sub assign {
    my ($x, $thing) = @_;
    if (@{$x->subscripts} > 1) {
	unimp "multi-level subscripting";
    }
    my $indexval = $x->subscripts(0)->val;
    my $type = $x->subscripts(0)->type;
    my $ret;
    my $rhs = $thing->val;
    my $lhs = $x->thing->val;
    my $lctx = $thing->{ctx};

    if ($lctx->is_scalar) {
	# XXX: This isn't quite right, since we're taking lhs's
	# C<val>.  But it works for simple @arrays.
	$ret = newtmp;
	my $itmp = gentmp $temptype{$type};
	$indexval = $indexval->[-1] if ref $indexval;
	code(<<END);
	$itmp = $indexval
	$lhs\[$itmp] = $rhs
END
	return $rhs;	# XXX: should return $lhs[$itmp] ?
    } else {
	use Data::Dumper;
	unimp 'Assignment to multi-element slice: '.Dumper($lctx);
    }
}

1;
