package Crypt::Sodium::XS::generichash;
use strict;
use warnings;

use Crypt::Sodium::XS;
use Exporter 'import';

_define_constants();

my @constant_bases = qw(
  BYTES
  BYTES_MIN
  BYTES_MAX
  KEYBYTES
  KEYBYTES_MIN
  KEYBYTES_MAX
);

my @bases = qw(
  init
  keygen
);

my $default = [
  "generichash",
  (map { "generichash_$_" } @bases),
  (map { "generichash_$_" } @constant_bases, "PRIMITIVE"),
];
my $blake2b = [
  "generichash_blake2b",
  "generichash_blake2b_init_salt_personal",
  "generichash_blake2b_salt_personal",
  (map { "generichash_blake2b_$_" } @bases),
  (map { "generichash_blake2b_$_" } @constant_bases),
  "generichash_blake2b_PERSONALBYTES",
  "generichash_blake2b_SALTBYTES",
];

our %EXPORT_TAGS = (
  all => [ @$default, @$blake2b ],
  default => $default,
  blake2b => $blake2b,
);

our @EXPORT_OK = @{$EXPORT_TAGS{all}};

1;

__END__

=encoding utf8

=head1 NAME

Crypt::Sodium::XS::generichash - Cryptographic hashing

=head1 SYNOPSIS

  use Crypt::Sodium::XS::generichash ":default";

  my $msg = "hello, world!";
  my $hash = generichash($msg);

  my $output_len = 64;
  my $key = generichash_keygen;
  $hash = generichash($msg, $output_len, $key);

  my $multipart = generichash_init;
  $multipart->update($msg);
  $hash = $multipart->final;

=head1 DESCRIPTION

L<Crypt::Sodium::XS::generichash> computes a fixed-length fingerprint for an
arbitrary long message.

Sample use cases:

=over 4

=item * File integrity checking

=item * Creating unique identifiers to index arbitrary long data

=back

=head1 FUNCTIONS

Nothing is exported by default. A C<:default> tag imports the functions and
constants as documented below. A separate import tag is provided for each of
the primitives listed in L</PRIMITIVES>. For example, C<:blake2b> imports
C<generichash_blake2b>. You should use at least one import tag.

=head2 generichash

  my $hash = generichash($message, $hash_length, $key);
  my $hash = generichash($message, $hash_length);
  my $hash = generichash($message);
  my $hash = generichash($message, undef, $key);

C<$hash_length> is the desired length of the hashed output. It is optional. If
C<$hash_length> is omitted or numifies to zero (undef, 0, ""), the default hash
length (L</generichash_BYTES>) will be used.

C<$key> is optional.

=head2 generichash_init

  my $multipart = generichash_init($hash_length, $key);
  my $multipart = generichash_init($hash_length);
  my $multipart = generichash_init(undef, $key);

C<$hash_length> is the desired length of the hashed output. It is optional. If
it is omitted or numifies to zero (undef, 0, ""), the default hash length
(L</generichash_BYTES>) will be used.

C<$key> is optional. If provided, it must be L</generichash_KEYBYTES> in
length.

See L</MULTI-PART INTERFACE>.

=head2 generichash_keygen()

  my $key = generichash_keygen();
  my $key = generichash_keygen($key_length);

C<$key_length> is the desired length of the generated key. It is optional. If
it is omitted or numifies to zero (undef, 0, ""), the default key length
(L</generichash_KEYBYTES>) will be used. The length of C<$key_length>, if
given, must be from L</generichash_KEYBYTES_MIN> to
L</generichash_KEYBYTES_MAX>, inclusive.

=head2 generichash_keygen

  my $key = generichash_keygen();
  my $key = generichash_keygen($key_length);

=head1 MULTI-PART INTERFACE

A multipart hashing object is created by calling the L</generichash_init>
function. Data to be hashed is added by calling the L</update> method of that
object as many times as desired. An output hash is generated by calling its
L</final> method. Do not use the object after calling L</final>.

The multipart hashing object is an opaque object which provides the following
methods:

=head2 clone

  my $multipart_copy = $multipart->clone;

=head2 final

  my $hash = $multipart->final;

Note there is a difference to the sodium API. Finalizing the hash does not
require, nor accept, a new output length. The output hash length will be the
original C<$hash_length> given to L</generichash_init>.

Once C<final> has been called, the hashing object must not be used further.

=head2 update

  $multipart->update($message);
  $multipart->update(@messages);

=head1 CONSTANTS

=head2 generichash_PRIMITIVE

  my $default_primitive = generichash_PRIMITIVE();

=head2 generichash_BYTES

  my $hash_default_length = generichash_BYTES();

=head2 generichash_BYTES_MIN

  my $hash_min_length = generichash_BYTES_MIN();

=head2 generichash_BYTES_MAX

  my $hash_max_length = generichash_BYTES_MAX();

=head2 generichash_KEYBYTES

  my $key_default_length = generichash_KEYBYTES();

=head2 generichash_KEYBYTES_MIN

  my $key_min_length = generichash_KEYBYTES_MIN();

=head2 generichash_KEYBYTES_MAX

  my $key_max_length = generichash_KEYBYTES_MAX();

=head1 PRIMITIVES

All constants (except _PRIMITIVE) and functions have
C<generichash_E<lt>primitive<gt>>-prefixed counterparts (e.g.
generichash_blake2b, generichash_blake2b_BYTES).

=over 4

=item * blake2b

=back

=head2 blake2b FUNCTIONS

L<Crypt::Sodium::XS::generichash> has the following functions available only in
their primitive-specific form.

=head3 generichash_blake2b_salt_personal

  my $hash = generichash_blake2b_salt_personal($message, $salt, $personal, $hash_length, $key);
  my $hash = generichash_blake2b_salt_personal($message, $salt, $personal, $hash_length);
  my $hash = generichash_blake2b_salt_personal($message, $salt, $personal);
  my $hash = generichash_blake2b_salt_personal($message, $salt, $personal, undef, $key);

C<$salt> as an arbitrary string which is at least
C<generichash_blake2b_SALTBYTES> in length (see warnings below).

C<$personal> as an arbitrary string which is at least
C<generichash_blake2b_PERSONALBYTES> in length (see warnings below).

C<$hash_length> is the desired length of the hashed output. It is optional. If
C<$hash_length> is omitted or numifies to zero (undef, 0, ""), the default hash
length (C<generichash_blake2b_BYTES>) will be used.

C<$key> is optional.

B<WARNING>: C<$salt> and C<$personal> must be at least
C<generichash_blake2b_SALTBYTES> and C<generichash_blake2b_PERSONALBYTES> in
length, respectively. If they are longer than the required length, only the
required length of initial bytes will be used. If these values are not being
randomly chosen, it is recommended to use an arbitrary-length string as the
input to a hash function (e.g., L<Crypt::Sodium::XS::generichash/generichash>
or L<Crypt::Sodium::XS::shorthash/shorthash>) and use the hash output rather
than the strings.

=head3 generichash_blake2b_init_salt_personal

  my $multipart = generichash_blake2b_init_salt_personal($hash_length, $key);
  my $multipart = generichash_blake2b_init_salt_personal($hash_length);
  my $multipart = generichash_blake2b_init_salt_personal($hash_length, $key);

C<$hash_length> is the desired length of the hashed output. It is optional. If
it is omitted or numifies to zero (undef, 0, ""), the default hash length
(C<generichash_blake2b_BYTES>) will be used.

C<$key> is optional. If provided, it must be C<generichash_blake2b_KEYBYTES> in
length.

=head2 blake2b CONSTANTS

L<Crypt::Sodium::XS::generichash> has the following constants available only in
their primitive-specific form.

=head3 generichash_blake2b_PERSONALBYTES

=head3 generichash_blake2b_SALTBYTES

=head1 SEE ALSO

=over 4

=item L<Crypt::Sodium::XS>

=item L<https://doc.libsodium.org/hashing/generic_hashing>

=back

=head1 FEEDBACK

For reporting bugs, giving feedback, submitting patches, etc. please use the
following:

=over 4

=item *

RT queue at L<https://rt.cpan.org/Dist/Display.html?Name=Crypt-Sodium-XS>

=item *

IRC channel C<#sodium> on C<irc.perl.org>.

=item *

Email the author directly.

=back

=head1 AUTHOR

Brad Barden E<lt>perlmodules@5c30.orgE<gt>

=head1 COPYRIGHT & LICENSE

Copyright (c) 2022 Brad Barden. All rights reserved.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=cut
