package ParseUtil::Domain::ConfigData;


use strict;
## no critic
my $arrayref = eval do {local $/; <DATA>}
  or die "Couldn't load ConfigData data: $@";
close DATA;
my ($config, $features, $auto_features) = @$arrayref;

sub config { $config->{$_[1]} }

sub set_config { $config->{$_[1]} = $_[2] }
sub set_feature { $features->{$_[1]} = 0+!!$_[2] }  # Constrain to 1 or 0

sub auto_feature_names { grep !exists $features->{$_}, keys %$auto_features }

sub feature_names {
  my @features = (keys %$features, auto_feature_names());
  @features;
}

sub config_names  { keys %$config }

sub write {
  my $me = __FILE__;

  # Can't use Module::Build::Dumper here because M::B is only a
  # build-time prereq of this module
  require Data::Dumper;

  my $mode_orig = (stat $me)[2] & 07777;
  chmod($mode_orig | 0222, $me); # Make it writeable
  open(my $fh, '+<', $me) or die "Can't rewrite $me: $!";
  seek($fh, 0, 0);
  while (<$fh>) {
    last if /^__DATA__$/;
  }
  die "Couldn't find __DATA__ token in $me" if eof($fh);

  seek($fh, tell($fh), 0);
  my $data = [$config, $features, $auto_features];
  print($fh 'do{ my '
	      . Data::Dumper->new([$data],['x'])->Purity(1)->Dump()
	      . '$x; }' );
  truncate($fh, tell($fh));
  close $fh;

  chmod($mode_orig, $me)
    or warn "Couldn't restore permissions on $me: $!";
}

sub feature {
  my ($package, $key) = @_;
  return $features->{$key} if exists $features->{$key};

  my $info = $auto_features->{$key} or return 0;

  # Under perl 5.005, each(%$foo) isn't working correctly when $foo
  # was reanimated with Data::Dumper and eval().  Not sure why, but
  # copying to a new hash seems to solve it.
  my %info = %$info;

  require Module::Build;  # XXX should get rid of this
  while (my ($type, $prereqs) = each %info) {
    next if $type eq 'description' || $type eq 'recommends';

    my %p = %$prereqs;  # Ditto here.
    while (my ($modname, $spec) = each %p) {
      my $status = Module::Build->check_installed_status($modname, $spec);
      if ((!$status->{ok}) xor ($type =~ /conflicts$/)) { return 0; }
      if ( ! eval "require $modname; 1" ) { return 0; }
    }
  }
  return 1;
}
## use critic

=head1 NAME

ParseUtil::Domain::ConfigData - Configuration for ParseUtil::Domain

=head1 SYNOPSIS

  use ParseUtil::Domain::ConfigData;
  $value = ParseUtil::Domain::ConfigData->config('foo');
  $value = ParseUtil::Domain::ConfigData->feature('bar');

  @names = ParseUtil::Domain::ConfigData->config_names;
  @names = ParseUtil::Domain::ConfigData->feature_names;

  ParseUtil::Domain::ConfigData->set_config(foo => $new_value);
  ParseUtil::Domain::ConfigData->set_feature(bar => $new_value);
  ParseUtil::Domain::ConfigData->write;  # Save changes


=head1 DESCRIPTION

This module holds the configuration data for the C<ParseUtil::Domain>
module.  It also provides a programmatic interface for getting or
setting that configuration data.  Note that in order to actually make
changes, you'll have to have write access to the C<ParseUtil::Domain::ConfigData>
module, and you should attempt to understand the repercussions of your
actions.


=head1 METHODS

=over 4

=item config($name)

Given a string argument, returns the value of the configuration item
by that name, or C<undef> if no such item exists.

=item feature($name)

Given a string argument, returns the value of the feature by that
name, or C<undef> if no such feature exists.

=item set_config($name, $value)

Sets the configuration item with the given name to the given value.
The value may be any Perl scalar that will serialize correctly using
C<Data::Dumper>.  This includes references, objects (usually), and
complex data structures.  It probably does not include transient
things like filehandles or sockets.

=item set_feature($name, $value)

Sets the feature with the given name to the given boolean value.  The
value will be converted to 0 or 1 automatically.

=item config_names()

Returns a list of all the names of config items currently defined in
C<ParseUtil::Domain::ConfigData>, or in scalar context the number of items.

=item feature_names()

Returns a list of all the names of features currently defined in
C<ParseUtil::Domain::ConfigData>, or in scalar context the number of features.

=item auto_feature_names()

Returns a list of all the names of features whose availability is
dynamically determined, or in scalar context the number of such
features.  Does not include such features that have later been set to
a fixed value.

=item write()

Commits any changes from C<set_config()> and C<set_feature()> to disk.
Requires write access to the C<ParseUtil::Domain::ConfigData> module.

=back


=head1 AUTHOR

C<ParseUtil::Domain::ConfigData> was automatically created using C<Module::Build>.
C<Module::Build> was written by Ken Williams, but he holds no
authorship claim or copyright claim to the contents of C<ParseUtil::Domain::ConfigData>.

=cut


__DATA__
do{ my $x = [
       {
         'tld_regex' => qr/(?:j(?:p(?:\.(?:h(?:okkaido(?:\.(?:k(?:a(?:m(?:i(?:s(?:unagawa|hihoro)|furano|kawa)|oenai)|yabe)|i(?:ta(?:hiroshima|mi)|mobetsu|yosato|konai)|u(?:r(?:omatsunai|iyama)|nneppu|shiro|tchan)|oshimizu|embuchi|yowa)|s(?:h(?:i(?:r(?:a(?:nuka|oi)|iuchi)|m(?:amaki|okawa|izu)|n(?:shinots|tok)u|be(?:cha|tsu)|ka(?:be|oi))|a(?:kotan|ri))|ar(?:ufutsu|oma)|unagawa|obetsu)|t(?:o(?:y(?:o(?:tomi|ura)|a(?:ko)?)|ma(?:koma|r)i|betsu|hma)|a(?:k(?:i(?:kawa|noue)|asu)|iki)|su(?:kigata|betsu)|eshikaga)|h(?:o(?:ro(?:kanai|nobe)|ku(?:ryu|to)|nbetsu)|i(?:(?:gashika(?:gur|w)|dak)a|roo)|a(?:matonbetsu|kodate|boro))|n(?:a(?:ka(?:satsunai|tombetsu|gawa)|n(?:poro|ae)|yoro|ie)|i(?:shiokoppe|ikappu|ki)|oboribetsu|emuro|umata)|a(?:s(?:h(?:ibetsu|oro)|ahikawa|sabu)|k(?:abira|keshi)|b(?:ashiri|ira)|ibetsu|tsuma)|m(?:i(?:namifurano|kasa)|o(?:mbetsu|seushi)|a(?:tsuma|shik)e|u(?:roran|kawa))|o(?:t(?:o(?:(?:fuk|b)e|ineppu)|aru)|bi(?:hiro|ra)|k(?:oppe|eto)|zora|umu)|r(?:i(?:shir(?:ifuj)?i|kubetsu)|ankoshi|ebun)|i(?:wa(?:mizawa|nai)|shikari|makane|keda)|fu(?:k(?:ushim|agaw)a|r(?:ubira|ano))|u(?:r(?:a(?:kawa|usu)|yu)|tashinai)|e(?:sa(?:shi|n)|m?betsu|niwa|rimo)|bi(?:(?:rator|ba|e)i|fuka|horo)|chi(?:ppubetsu|tose)|wa(?:kkanai|ssamu)|y(?:akumo|oichi)|pippu|date))?|yogo(?:\.(?:s(?:a(?:n(?:nan|da)|sayama|yo)|hi(?:n(?:onsen|gu)|so)|umoto)|t(?:a(?:k(?:a(?:razuka|sago)?|ino)|tsuno|ishi|mba)|oyooka)|a(?:(?:(?:magas|og)ak|waj|io)i|s(?:hiya|ago)|k(?:ashi|o))|ka(?:mi(?:gori|kawa)|s(?:uga|ai)|wanishi|kogawa)|(?:(?:fukusa|goshi)k|mi(?:namiawaj|k))i|y(?:a(?:shiro|bu)|ok(?:aw)?a)|i(?:(?:chik|nag)awa|tami)|nishi(?:nomiya|waki)|h(?:arima|imeji)|ono))?|iroshima(?:\.(?:s(?:h(?:inichi|obara)|era(?:nishi)?|aka)|h(?:igashihiroshima|atsukaichi|ongo)|o(?:sakikamijima|nomichi|take)|(?:takehar|etajim|daiw|nak)a|k(?:u(?:mano|re|i)|aita)|fu(?:kuyama|chu)|mi(?:yoshi|hara)|jinsekikogen|asaminami))?)|s(?:a(?:itama(?:\.(?:k(?:a(?:wa(?:g(?:uchi|oe)|jima)|mi(?:izumi|kawa|sato)|sukabe|zo)|o(?:shigaya|unosu)|u(?:magaya|ki)|itamoto)|h(?:a(?:(?:to(?:gay|yam)|sud)a|n(?:no|yu))|i(?:gashi(?:matsuyama|chichibu)|daka)|onjo)|(?:t(?:o(?:k(?:oroz|ig)aw|d)|surugashim)|a(?:rakaw|sak)|uraw)a|m(?:i(?:y(?:ashiro|oshi)|(?:nan|sat)o)|atsubushi|oroyama)|s(?:a(?:(?:it|y)ama|kado|tte)|hi(?:raoka|ki)|ugito|oka)|y(?:o(?:shi(?:(?:kaw|d)a|mi)|koze|rii|no)|ashio)|o(?:g(?:a(?:no|wa)|ose)|(?:kegaw|miy)a|taki)|n(?:a(?:gatoro|megawa)|iiza)|i(?:(?:rum|n)a|watsuki)|fu(?:jimi(?:no)?|kaya)|r(?:yokami|anzan)|chichibu|warabi))?|ga(?:\.(?:k(?:a(?:mimine|nzaki|ratsu|shima)|i(?:ta[gh]at|yam)a|ouhoku|yuragi)|(?:(?:yoshinog|im)ar|o(?:(?:ma|u)ch|g)|fukudom|genka)i|t(?:a(?:ku|ra)|osu)|h(?:amatama|izen)|s(?:hiroishi|aga)|ari(?:ake|ta)|nishiarita))?|pporo(?:\.(?!city$)[^\.]+)?)|hi(?:zuoka(?:\.(?:m(?:a(?:kinohara|tsuzaki)|i(?:namiizu|shima)|orimachi)|k(?:a(?:wa(?:nehon|zu)|kegawa|nnami)|ikugawa|osai)|s(?:hi(?:m(?:[ao]da|izu)|zuoka)|usono)|fu(?:ji(?:(?:nomiy|kaw|ed)a)?|kuroi)|h(?:a(?:mamatsu|ibara)|igashiizu)|i(?:zu(?:nokuni)?|wata|to)|(?:a(?:tam|ra)|omaezak)i|n(?:ishii|uma)zu|y(?:oshida|aizu)|gotemba))?|ga(?:\.(?:t(?:aka(?:shima|tsuki)|o(?:rahime|yosato))|n(?:(?:agaham|otogaw)a|ishiazai)|k(?:o(?:nan|sei|ka|to)|usatsu)|hi(?:gashiomi|kone)|o(?:mihachiman|tsu)|m(?:oriyam|aibar)a|r(?:itto|yuoh)|(?:aish|gam)o|yasu))?|mane(?:\.(?:h(?:i(?:k(?:awa|imi)|gashiizumo)|amada)|o(?:k(?:inoshima|uizumo)|hda)|m(?:a(?:suda|tsue)|isato)|ya(?:tsuka|kumo|sugi)|t(?:suwano|amayu)|a(?:kagi|ma)|nishinoshima|kakinoki|shimane|gotsu|izumo|unnan))?)|endai(?:\.(?!city$)[^\.]+)?)|k(?:a(?:g(?:oshima(?:\.(?:k(?:a(?:(?:goshim|noy)a|wanabe)|ouyama|inko)|m(?:a(?:kurazaki|tsumoto)|inamitane)|n(?:ishinoomot|akatan)e|s(?:atsumasendai|oo)|i(?:s(?:en|a)|zumi)|(?:hiok|yusu)i|a(?:kune|mami)|tarumizu))?|awa(?:\.(?:(?:(?:higashik|ay)agaw|naoshim)a|t(?:a(?:kama|do)tsu|onosho)|m(?:a(?:rugame|nno)|itoyo)|k(?:otohira|anonji)|(?:zentsuj|sanuk)i|u(?:chinomi|tazu)))?)|nagawa(?:\.(?:(?:m(?:i(?:namiashiga|u)r|atsud)|sa(?:gamihar|mukaw)|fujisaw|isehar|ebin)a|y(?:(?:okosuk|ugawar)a|ama(?:kita|to))|k(?:a(?:makura|isei)|iyokawa)|h(?:a(?:dano|kone)|iratsuka)|a(?:ikawa|tsugi|yase)|o(?:i(?:so)?|dawara)|(?:chigasak|tsuku)i|n(?:inomiya|akai)|z(?:ushi|ama)))?|wasaki(?:\.(?!city$)[^\.]+)?)|o(?:chi(?:\.(?:n(?:a(?:kamura|nkoku|hari)|i(?:yodogaw|shitos)a)|t(?:o(?:sa(?:shimizu)?|yo)|suno)|k(?:(?:(?:ag)?am|och)i|itagawa)|m(?:(?:otoyam|ihar)a|uroto)|o(?:t(?:suki|oyo)|kawa|chi)|s(?:u(?:kumo|saki)|akawa)|hi(?:gashitsuno|daka)|(?:geise|umaj|ak)i|y(?:usuhar|asud)a|ino))?|be(?:\.(?!city$)[^\.]+)?)|yoto(?:\.(?:k(?:yota(?:n(?:abe|go)|mba)|am(?:eoka|o)|i(?:ta|zu)|umiyama)|(?:(?:fukuc|higas)hiyam|ya(?:mashin|wat)|wazuk)a|m(?:i(?:nami(?:yamashiro)?|yazu)|aizuru|uko)|na(?:(?:gaokak|kag)yo|ntan)|(?:(?:tan|ay)ab|i[dn])e|s(?:akyo|eika)|uji(?:tawara)?|oyamazaki|joyo))?|umamoto(?:\.(?:k(?:(?:a(?:miamakus|shim)|os)a|umamoto|ikuchi)|m(?:i(?:nam(?:ioguni|ata)|fune)|ashiki)|ya(?:ma(?:ga|to)|tsushiro)|(?:(?:gyoku|sumo)t|choy)o|(?:hitoyosh|takamor)i|a(?:(?:ra|s)o|makusa)|n(?:ishihara|agasu)|o(?:guni|zu)|u(?:ki|to)))?|itakyushu(?:\.(?!city$)[^\.]+)?)|n(?:a(?:g(?:a(?:no(?:\.(?:m(?:i(?:n(?:ami(?:m(?:inowa|aki)|aiki)|owa)|(?:y(?:ad|ot)|as)a)|atsu(?:kawa|moto)|ochizuki)|(?:i(?:i(?:(?:ji|ya)m|zun|d)|k(?:usak|ed)|n)|h(?:a(?:kub|r)|iray)|(?:ue|wa)d)a|s(?:hi(?:(?:nanomach|ojir)i|mosuwa)|ak(?:a(?:ki|e)|u(?:ho)?)|u(?:zak|w)a)|t(?:a(?:ka(?:(?:mor|g)i|yama)|t(?:eshina|suno))|o(?:g(?:akushi|ura)|mi))|k(?:i(?:so(?:fukushima)?|taaiki)|a(?:ruizawa|wakami)|om(?:agane|oro))|o(?:(?:o(?:shik|kuw)|gaw|kay)a|(?:m(?:ach)?|ta[kr])i|buse)|n(?:a(?:g(?:a(?:no|wa)|iso)|ka(?:gawa|no))|ozawaonsen)|a(?:(?:(?:sa|c)h|ok)i|gematsu|zumino|nan)|ya(?:ma(?:nouchi|gata)|s(?:uo|a)ka)|chi(?:ku(?:hoku|ma)|no)|fujimi))?|saki(?:\.(?:s(?:hi(?:nkamigoto|mabara)|a(?:ikai|sebo)|eihi)|k(?:uchinotsu|awatana)|o(?:(?:bam|mur)a|seto)|(?:matsuur|chijiw)a|t(?:sushima|ogitsu)|h(?:asami|irado)|i(?:sahaya|ki)|nagasaki|futsu|unzen|goto))?)|oya(?:\.(?!city$)[^\.]+)?)|ra(?:\.(?:k(?:a(?:(?:wa(?:nish|kam)?|tsurag|nmak)i|(?:shi(?:har|b)|mikitayam)a)|urotaki|oryo)|s(?:hi(?:mo(?:kitayama|ichi)|njo)|a(?:kurai|ngo)|oni)|y(?:ama(?:to(?:koriyam|takad)a|zoe)|oshino)|t(?:a(?:waramoto|katori)|en(?:kawa|ri))|(?:n(?:osegaw|ar)|ik(?:arug|om)|ud)a|h(?:igashiyoshino|eguri)|(?:mi(?:tsu|yak)|gos)e|o(?:yodo|uda|ji)|ando))?)|iigata(?:\.(?:t(?:o(?:kamachi|chio)|su(?:bame|nan)|a(?:gam|ina)i)|m(?:i(?:namiuonuma|tsuke)|u(?:rakami|ika)|yoko)|s(?:e(?:kikawa|irou?)|a(?:nj|d)o|hibata)|(?:n(?:agaok|iigat)|uonum)a|y(?:(?:uzaw|oit)a|ahiko)|ka(?:shiwazaki|riwa|mo)|i(?:zumozaki|toigawa)|o(?:jiya|mi)|aga(?:no)?|joetsu|gosen))?|e)|to(?:k(?:yo(?:\.(?:k(?:o(?:(?:uzushim|dair)a|(?:kubunj|gane)i|mae|to)|i(?:yose|ta)|atsushika|unitachi)|h(?:i(?:gashi(?:murayama|kurume|yamato)|no(?:hara|de)?)|a(?:chi(?:oji|jo)|mura))|m(?:i(?:(?:nat|zuh)o|taka)|usashi(?:murayama|no)|achida|eguro)|s(?:hi(?:n(?:agawa|juku)|buya)|u(?:ginami|mida)|etagaya)|a(?:(?:ogashim|rakaw)a|ki(?:shima|runo)|dachi)|o(?:(?:(?:kuta|shi)m|gasawar|t)a|me)|t(?:a(?:(?:chikaw|m)a|ito)|oshima)|ch(?:iyoda|ofu|uo)|i(?:tabash|nag)i|n(?:akano|erima)|fu(?:chu|ssa)|edogawa|bunkyo))?|ushima(?:\.(?:m(?:i(?:(?:yosh|nam)i|ma)|atsushige|ugi)|(?:s(?:anagoch|hishiku)|wajik)i|k(?:omatsushima|ainan)|na(?:kagawa|ruto)|i(?:chiba|tano)|a(?:izumi|nan)|tokushima))?)|chigi(?:\.(?:(?:o(?:h(?:tawa|i)r|yam)|(?:ashik|h)ag|yait)a|n(?:asu(?:shiobara)?|i(?:shikata|kko)|ogi)|k(?:a(?:(?:rasuya|nu)m|minokaw)a|uroiso)|s(?:hi(?:motsuke|oya)|a(?:kura|no))|t(?:(?:akanezaw|sug)a|ochigi)|m(?:o(?:tegi|ka)|ashiko|ibu)|u(?:tsunomiya|jiie)|i(?:chikai|wafune)|bato))?|yama(?:\.(?:t(?:o(?:(?:yam|g)a|nami)|a(?:teyam|kaok|ir)a)|n(?:a(?:(?:kani|mer)ikawa|nto)|yuzen)|fu(?:(?:kumits|ch)u|nahashi)|k(?:amiichi|urobe)|(?:johan|yamad)a|u(?:nazuki|ozu)|i(?:mizu|nami)|(?:asah|him)i|oyabe))?|ttori(?:\.(?:k(?:o(?:toura|ge)|awahara)|(?:sakaiminat|hin)o|n(?:ichinan|anbu)|(?:mis|wak)asa|y(?:onago|azu)|tottori|chizu))?)|o(?:k(?:inawa(?:\.(?:k(?:i(?:ta(?:nakagusuku|daito)|n)|u(?:mejima|nigami)|adena)|t(?:o(?:(?:kashi|na)ki|migusuku)|a(?:ketomi|rama))|n(?:a(?:k(?:agusuku|ijin)|(?:nj|g)o|ha)|ishihara)|i(?:shi(?:gaki|kawa)|(?:hey|zen)a|toman)|y(?:o(?:na(?:baru|guni)|mitan)|aese)|g(?:ino(?:wan|za)|ushikami)|h(?:i(?:gashi|rara)|aebaru)|o(?:(?:kinaw|nn)a|gimi)|(?:shimoj|zamam|agun)i|m(?:inamidaito|otobu)|ur(?:asoe|uma)))?|ayama(?:\.(?:k(?:u(?:rashiki|menan)|a(?:gamino|saoka)|ibichuo)|s(?:(?:h(?:inj|o)|atosh)o|etouchi|oja)|t(?:a(?:kahashi|mano)|suyama)|(?:(?:hayashi|okaya)m|ibar)a|n(?:i(?:shiawakura|imi)|agi)|a(?:sakuchi|kaiwa)|m(?:aniwa|isaki)|(?:yakag|wak)e|bizen))?)|saka(?:\.(?:k(?:a(?:(?:shiwar|izuk|dom)a|(?:wachinag|t)ano|nan)|i(?:shiwad|t)a|umatori)|t(?:a(?:(?:ka(?:tsuk|ish)|ish|jir)i|daoka)|o(?:yon(?:aka|o)|ndabayashi))|h(?:i(?:gashi(?:(?:yodogaw|osak)a|sumiyoshi)|rakata)|a(?:bikino|nnan))|s(?:hi(?:jonawate|mamoto)|e(?:nnan|ttsu)|a(?:yama|kai)|uita)|m(?:i(?:n(?:ato|oh)|saki)|atsubara|origuchi)|i(?:zumi(?:otsu|sano)?|baraki|keda)|(?:osakasayam|fujiider)a|ch(?:ihayaakasaka|uo)|n(?:eyagawa|ishi|ose)|(?:aben|dait|ya)o))?|ita(?:\.(?:k(?:(?:amitsu|okono)e|u(?:nisaki|[js]u))|h(?:i(?:(?:meshim|t)a|ji)|asama)|b(?:ungo(?:takada|ono)|eppu)|t(?:sukumi|aketa)|us(?:uki|a)|saiki|oita|yufu))?|r)|fuku(?:oka(?:\.(?:k(?:a(?:(?:su[gy]|war)a|ho)|ur(?:(?:at|um)e|ogi)|eisen|oga)|s(?:h(?:in(?:yoshitomi|gu)|onai)|a(?:saguri|igawa)|oeda|ue)|m(?:i(?:ya(?:(?:wak|m)a|ko)|(?:zumak|nam)i)|unakata)|t(?:a(?:(?:gaw|kat)a|chiarai)|o(?:yotsu|ho)|suiki)|o(?:k(?:a(?:gaki|wa)|i)|n(?:ojo|ga)|gori|muta|to)|h(?:i(?:(?:rokaw|sayam)a|gashi)|akata)|y(?:a(?:m(?:ada|e)|nagawa)|ukuhashi)|ch(?:iku(?:(?:[ghj]|shin)o|zen)|uo)|n(?:(?:aka(?:gaw|m)|ogat)a|ishi)|u(?:(?:su|m)i|kiha)|i(?:natsuki|izuka)|dazaifu|fukuchi|ashiya|buzen))?|shima(?:\.(?:k(?:a(?:(?:neyam|wamat)a|gamiishi)|ita(?:shiobar|kat)a|o(?:riyama|ori)|unimi)|s(?:h(?:i(?:rakawa|mogo)|owa)|(?:(?:ame|uka)gaw|om)a)|i(?:(?:zumiz|w)aki|nawashiro|shikawa|itate)|a(?:izu(?:wakamatsu|misato|bange)|sakawa)|y(?:a(?:mat(?:suri|o)|naizu|buki)|ugawa)|t(?:a(?:(?:makaw|nagur)a|ishin)|enei)|h(?:i(?:r(?:ata|ono)|gashi)|anawa)|n(?:ishi(?:aizu|go)|a(?:mie|ngo))|o(?:(?:moteg|n)o|(?:ku|ta)ma)|fu(?:(?:kushim|tab)a|rudono)|mi(?:shima|haru)|bandai|date))?|i(?:\.(?:(?:t(?:akaham|surug)|katsuyam|wakas|iked)a|mi(?:namiechizen|hama)|e(?:chizen|iheiji)|o(?:bama|hi|no)|sa(?:bae|kai)|fukui))?)|i(?:baraki(?:\.(?:t(?:o(?:(?:mob|rid|n)e|kai)|a(?:matsukur|kahag)i|su(?:chiur|kub)a)|s(?:hi(?:mo(?:tsuma|date)|rosato)|ak(?:uragawa|ai)|uifu|owa)|k(?:a(?:s(?:(?:hi|a)m|umigaur)a|misu)|oga)|i(?:na(?:shiki)?|baraki|tako|wama)|y(?:a(?:(?:magat|war)a|chiyo)|uki)|(?:a(?:sah|m)|ryugasak|chikuse)i|hitachi(?:(?:o(?:miy|t)|nak)a)?|(?:fujishir|band|daig|jos)o|o(?:(?:mitam|gaw)a|arai)|u(?:chihara|shiku)|m(?:i[ht]o|oriya)|na(?:megat|k)a))?|wate(?:\.(?:k(?:(?:a(?:(?:rum|w)a|negasak|maish)|itakam)i|u(?:(?:zumak|j)i|nohe))|i(?:chino(?:seki|he)|wa(?:izumi|te))|(?:ya(?:hab|mad)|rikuzentakat)a|h(?:ir(?:aizumi|ono)|anamaki)|m(?:i(?:zusawa|yako)|orioka)|s(?:hi(?:zukuishi|wa)|umita)|t(?:a(?:kizaw|nohat)a|ono)|o(?:funato|tsuchi|shu)|fu(?:jisawa|dai)|n(?:inohe|oda)|joboji))?|shikawa(?:\.(?:k(?:a(?:(?:nazaw|wakit|g)a|hoku)|omatsu)|n(?:o(?:(?:noich|m)i|to)|a(?:kanot|na)o)|(?:uchinad|wajim)a|tsu(?:bata|rugi)|haku(?:san|i)|s(?:hika|uzu)|anamizu))?)|a(?:[cd]|ichi(?:\.(?:t(?:o(?:yo(?:(?:kaw|t)a|(?:ak|n)e|hashi)|k(?:oname|ai)|bishima|ei|go)|(?:a(?:kaham|har)|sushim)a)|k(?:a(?:sugai|riya|nie)|o(?:maki|nan|ta)|i(?:yosu|ra))|i(?:(?:n(?:azaw|uyam)|chinomiy|wakur)a|sshiki)|o(?:(?:(?:wariasa|guc)h|kazak)i|(?:har|b)u)|h(?:a(?:nda|zu)|igashiura|ekinan)|s(?:hi(?:nshiro|katsu|tara)|eto)|n(?:is(?:shin|hio)|agakute)|a(?:isai|suke|njo|ma)|(?:gamagor|yatom)i|mi(?:yoshi|hama)|chi(?:ryu|ta)|fuso))?|kita(?:\.(?:k(?:a(?:mi(?:koani|oka)|tagami|zuno)|(?:itaakit|osak|yow)a)|h(?:a(?:chirogata|ppou)|igashinaruse|onjy?o)|(?:n(?:oshir|ikah)|fujisat)o|m(?:i(?:sato|tane)|oriyoshi)|o(?:g(?:at)?a|date)|y(?:urihonjo|okote)|(?:akit|ikaw)a|semboku|daisen|gojome))?|omori(?:\.(?:(?:n(?:akadomar|ohej)|itayanag|kuroish|aomor)i|h(?:a(?:chinohe|shikami)|ir(?:osak|ana)i)|t(?:su(?:garu|ruta)|owada|akko)|s(?:hi(?:chinohe|ngo)|annohe)|(?:roku|go)nohe|m(?:isawa|utsu)|o(?:irase|wani)))?)|mi(?:ya(?:gi(?:\.(?:s(?:hi(?:(?:(?:og|k)am|bat)a|chikashuku|roishi)|emine)|m(?:i(?:namisanriku|sato)|a(?:tsushima|rumori)|urata)|k(?:a(?:(?:wasak|m)i|kuda)|esennuma)|o(?:(?:(?:gawa|hi)r|nagaw)a|saki)|t(?:a(?:gajo|iwa)|om(?:iya|e))|(?:higashimatsushim|furukaw)a|i(?:shinomaki|wanuma)|(?:yamamot|za)o|wa(?:kuya|tari)|natori|rifu))?|zaki(?:\.(?:k(?:i(?:ta(?:gaw|kat|ur)a|jo)|a(?:waminami|dogawa)|u(?:nitomi|shima)|obayashi)|m(?:i(?:ya(?:konojo|zaki)|mata)|orotsuka)|n(?:i(?:shimera|chinan)|obeoka)|t(?:aka(?:haru|nabe|zaki)|suno)|s(?:hi(?:ntomi|iba)|aito)|(?:hyug|ay)a|gokase|ebino))?)|e(?:\.(?:k(?:a(?:meyama|wagoe)|i(?:sosaki|ho|wa)|u(?:mano|wana)|omono)|m(?:i(?:[hy]ama|namiise|sugi)|(?:atsusak|eiw)a)|(?:(?:yokkaic|asa)h|watara|nabar)i|t(?:a(?:(?:ma|i)?ki|do)|oba|su)|u(?:reshi|do)no|s(?:uzuk|him)a|i(?:nab|s)e))?)|y(?:ama(?:g(?:ata(?:\.(?:s(?:h(?:i(?:rataka|njo)|onai)|a(?:k(?:egaw|at)a|gae))|(?:m(?:(?:amurog|ik)aw|urayam)|funagat)a|y(?:ama(?:gata|nobe)|(?:onezaw|uz)a)|o(?:(?:banazaw|ishid|hkur)a|guni|e)|ka(?:(?:mino|ne)yama|wanishi|hoku)|n(?:a(?:kayama|gai|nyo)|ishikawa)|t(?:(?:akahat|suruok|ozaw)a|endo)|(?:higashin|iid)e|asahi))?|uchi(?:\.(?:t(?:o(?:kuyam|yot)a|abuse)|(?:kudamats|mito|ab|yu)u|h(?:(?:ikar|ag)i|ofu)|sh(?:imonoseki|unan)|iwakuni|nagato|oshima|ube))?)|nashi(?:\.(?:n(?:a(?:kamichi|rusawa|nbu)|i(?:shikatsura|rasaki))|fu(?:ji(?:kawa(?:guchiko)?|yoshida)|efuki)|k(?:o(?:s(?:uge|hu)|fu)|ai)|(?:ichikawamisat|chu)o|min(?:ami-alps|obu)|h(?:ayakawa|okuto)|yamana(?:kako|shi)|(?:uenohar|show)a|t(?:abayama|suru)|o(?:shino|tsuki)|doshi))?)|okohama(?:\.(?!city$)[^\.]+)?)|g(?:[or]|ifu(?:\.(?:k(?:a(?:sa(?:matsu|hara)|kamigahara|waue|ni)|itagata)|h(?:i(?:(?:gashishirakaw|d)a|chiso)|ashima)|t(?:a(?:(?:jim|ru)i|kayama)|o(?:mika|ki))|m(?:i(?:n(?:okam)?o|zunami|take)|otosu)|s(?:eki(?:gahara)?|hirakawa|akahogi)|(?:i(?:bigaw|ked)|nakatsugaw|en)a|g(?:i(?:nan|fu)|(?:od|uj)o)|(?:(?:wanou|anpa)ch|ogak)i|y(?:a(?:magata|otsu)|oro)))?|unma(?:\.(?:t(?:a(?:ka(?:saki|yama)|tebayashi|mamura)|su(?:kiyono|magoi)|omioka)|(?:(?:(?:yosh|fuj)io|anna)k|higashiagatsum|chiyod)a|k(?:a(?:tashin|n[nr]|wab)a|(?:usats|iry)u)|n(?:a(?:ganohara|kanojo|nmoku)|umata)|m(?:(?:i(?:nakam|dor)|aebash)i|eiwa)|sh(?:i(?:(?:bukaw|monit)a|nto)|owa)|i(?:sesaki|takura)|o(?:[rt]a|izumi)|ueno))?)|c(?:hiba(?:\.(?:k(?:a(?:(?:m(?:agay|ogaw)|shiw)a|t(?:suura|ori))|(?:ujukur|ozak)i|i(?:saraz|mits)u|yonan)|s(?:hi(?:r(?:ako|oi)|mofusa|sui)|o(?:degaur|s)a|ak(?:ura|ae))|y(?:o(?:k(?:oshibahikari|aichiba)|tsukaido)|achi(?:mata|yo))|m(?:i(?:namiboso|dori|hama)|(?:utsuzaw|obar)a|atsudo)|t(?:o(?:(?:hnosh|misat)o|gane)|a(?:teyama|ko))|n(?:a(?:r(?:ashino|ita)|gar(?:eyam)?a)|oda)|i(?:chi(?:nomiy|har|kaw)a|(?:nza|sum)i)|o(?:amishirasato|migawa|njuku|taki)|ch(?:o(?:s[eh]i|nan)|uo)|fu(?:nabashi|ttsu)|a(?:biko|sahi)|hanamigawa|urayasu))?|o)|wakayama(?:\.(?:k(?:i(?:(?:nokaw|tayam)a|mino)|a(?:mitonda|tsuragi|inan)|u(?:shimoto|doyama)|o(?:z(?:agaw)?|y)a)|(?:arid(?:agaw)?|nachikatsuur|yu(?:as|r)|wakayam)a|h(?:i(?:rogaw|dak)a|ashimoto)|shi(?:rahama|ngu)|mi(?:hama|sato)|i(?:nami|wade)|ta(?:nabe|iji)|gobo))?|e(?:hime(?:\.(?:(?:n(?:amikat|iiham)|yawataham)a|k(?:umakogen|amijima|ihoku)|ma(?:tsu(?:yama|no)|saki)|s(?:hikokuchu|aij|eiy)o|i(?:mabari|kata|yo)|u(?:wajima|chiko)|to(?:be|on)|ainan|honai|ozu))?|d)|blogspot|lg)|morgan(?:chase)?|rs)?|o(?:[ty]|\.(?:n(?:ame|et)|com|edu|gov|mil|org|sch)|b(?:urg|s))?|e(?:\.(?:gov|net|org|sch|co)|welry|tzt|ep)?|u(?:stforu|niper|egos)|m(?:\.[^\.]+|p)?|a(?:guar|va)|c[bp]|l[cl]|io|nj)|n(?:o(?:\.(?:xn--(?:s(?:(?:k(?:n(?:land-fx|it-yq)|j(?:ervy-v1|k-so)|ierv-ut)|(?:m(?:la-h|na-g)|eral-l|gne-g)r|-1f)a|r(?:-(?:(?:(?:fron|odal)-q1|aurdal-l8)a|varanger-ggb)|(?:reisa-q1|fold-by|um-gr)a)|t(?:jrdal(?:shalsen-sqb|-s1a)|fold-9xa\.xn--vler-qoa|re-toten-zcb)|n(?:(?:(?:es-p|sa-r)o|ase-nr)a|dre-land-0cb)|and(?:nessjen-ogb|y-yua)|l(?:at-5na|t-elab))|(?:t(?:r(?:(?:any-y|oms-z)u|gstad-r1|na-wo)|(?:ysvr-v|jme-h)r|nsberg-q1)|f(?:r(?:(?:de-g|ya-h)r|na-wo)|l(?:or-jr|-zi)|inny-yu|jord-lr)|d(?:(?:nna-g|yry-i)r|avvenjrga-y4|rbak-wu)|o(?:s(?:tery-fy|yro-wu)|ppegrd-ix)|j(?:rpeland-54|lster-by)|eveni-0qa01g|indery-fy|unjrga-rt)a|b(?:(?:hc(?:cavuotna-k7|avuotna-s4)|e(?:aralvhki-y4|rlevg-jx)|j(?:arky-fy|ddar-pt)|tsfjord-9z|mlo-gr|od-2n)a|r(?:nny(?:sund-m8|-wu)ac|um-voa)|i(?:dr-5nac|evt-0qa)|dddj-mrabd|lt-elab)|r(?:(?:s(?:t(?:a-fr|-0n)|kog-uu)|l(?:ingen-mx|and-uu)|y(?:rvik-by|ken-vu)|(?:ady-i|ros-g)r|is(?:a-5n|r-ir)|ennesy-v1|mskog-by)a|d(?:(?:al-po|e-ul)a|y-0nab)|h(?:kkervju-01af|olt-mra))|m(?:(?:l(?:atvuopmi-s4|selv-iu|i-tl)|o(?:reke-ju|sjen-ey|t-tl)|e(?:rker-ku|ly-ir)|jndalen-64|uost-0q)a|re-og-romsdal-qqb\.(?:xn--hery-ira|sande)|tta-vrjjat-k7af|sy-ula0h)|l(?:(?:a(?:headju-7y|ngevg-jx)|r(?:enskog-54|dal-sr)|(?:hppi-x|oabt-0)q|(?:inds-p|ury-i)r|dingen-q1|-1f)a|e(?:agaviika-52b|sund-hua)|t(?:en-gra|-liac)|grd-poac)|k(?:(?:v(?:nangen-k0|fjord-nx|itsy-fy)|(?:fjord-i|army-y|snes-u)u|lbu-wo)a|r(?:(?:a(?:nghke-b0|ger-gy)|dsherad-m8|ehamn-dx)a|johka-hwab49j)|oluokta-7ya57h)|h(?:(?:y(?:landet-54|anger-q1)|o(?:ltlen-hx|bl-ir)|(?:bme|pmi)r-xq|gebostad-g3|nefoss-q1|-2f)a|cesuolo-7ya35b|mmrfeasta-s4ac)|v(?:r(?:e-eiker-k8a|ggt-xqad|y-yla5g)|e(?:stvgy-ixa6o|grshei-c0a)|g(?:sy-qoa0j|an-qoa|-yiab)|a(?:ds|rd)-jra)|g(?:(?:i(?:ldeskl-g0|vuotna-8y)|jvik-wu)a|gaviika-8ya47h|ls-elac)|n(?:(?:mesjevuemie-tcb|vuotna-hw)a|ttery-byae|ry-yla5g)|a(?:(?:(?:nd|sk)y-ir|very-yu)a|urskog-hland-jnb)|y(?:(?:garden-p1|er-zn)a|stre-slidre-ujb)|porsgu-sta26f)|s(?:t(?:o(?:r(?:d(?:al)?|-elvdal|fjord)|kke)|a(?:v(?:anger|ern)|t(?:helle)?|nge)|jordal(?:shalsen)?|ei(?:nkjer|gen)|r(?:anda?|yn)|\.gs)?|o(?:r(?:-(?:(?:aur|o)dal|varanger|fron)|(?:tlan|fol)d|reisa|um)|n(?:dre-land|gdalen)|gn(?:dal|e)|l(?:und|a)|kndal|mna)|a(?:nd(?:nes(?:sjoen)?|efjord|oy)|l(?:a(?:ngen|t)|tdal)|u(?:herad|da)|mnanger|rpsborg)|k(?:i(?:e(?:rva|n)|ptvet)?|a(?:n(?:land|it)|un)|edsmo(?:korset)?|j(?:ervoy|ak)|odje)|u(?:n(?:ndal|d)|l(?:dal|a)|rnadal)|v(?:albard(?:\.gs)?|e(?:lvik|io))|n(?:a(?:ase|sa)|illfjord|oasa)|i(?:[gr]dal|ellak|ljan)|p(?:jelkavik|ydeberg)|el(?:j(?:ord|e)|bu)?|f(?:\.gs)?|ykkylven|lattum|mola)|h(?:a(?:m(?:m(?:arfeasta|erfest)|ar(?:oy)?)|(?:gebosta|ugesun)d|r(?:(?:sta|ei)d|am)|(?:ttfjellda|dse)l|(?:bme|pmi)r|l(?:den|sa)|svik)?|o(?:r(?:daland\.os|nindal|ten)|l(?:mestrand|talen|e)?|y(?:landet|anger)|kksund|nefoss|bol|f)|e(?:dmark\.(?:xn--vler-qoa|valer|os)|m(?:sedal|nes?)|rad)|j(?:elmeland|artdal)|ur(?:dal|um)|l(?:\.gs)?|m(?:\.gs)?|yllestad|valer|itra)|n(?:o(?:r(?:d(?:land\.(?:xn--(?:hery-ir|b-5g)a|heroy|bo)|-(?:(?:aur|o)dal|fron)|re(?:-land|isa)|kapp|dal)|e-og-uvdal)|t(?:odden|teroy))|a(?:m(?:s(?:skogan|os)|dalseid)|r(?:vi(?:ika|k)|oy)|amesjevuemie|nnestad|ustdal|vuotna)|e(?:s(?:odd(?:tang)?en|se(?:by|t)|na)|dre-eiker)|i(?:ss|tt)edal|l(?:\.gs)?|t(?:\.gs)?)|b(?:a(?:l(?:(?:estran|sfjor)d|langen|at)|(?:hc?cavuotn|daddj)a|(?:jd|i)dar|r(?:du|um)|tsfjord|mble)|e(?:r(?:g(?:en)?|levag)|ar(?:alvahki|du)|iarn)|r(?:onnoy(?:sund)?|umunddal|emanger|yne)|u(?:(?:skerud\.ne|\.g)s|dejju)?|j(?:erkreim|arkoy|ugn)|i(?:rkenes|evat|ndal)|o(?:(?:ml|d)o|kn)|y(?:gland|kle)|logspot)|a(?:u(?:r(?:(?:skog-ho)?land|e)|st(?:evoll|rheim)|dnedaln|kra)|l(?:(?:esun|gar)d|aheadju|stahaug|vdal|ta)?|k(?:ershus\.nes|noluokta|rehamn)|s(?:k(?:voll|er|im|oy)|eral|nes)|r(?:e(?:mark|ndal)|dal|na)|nd(?:asuolo|ebu|oy)|a(?:rborte|\.gs)?|h(?:\.gs)?|m(?:li|ot)|gdenes|ejrie|fjord|veroy)|m(?:o(?:re-og-romsdal\.(?:heroy|sande)|s(?:(?:kene)?s|joen|vik)|(?:arek|ld)e|d(?:alen|um)|-i-rana)|a(?:l(?:atvuopmi|selv|vik)|r(?:nardal|ker)|s(?:fjorden|oy)|tta-varjjat|ndal)|e(?:l(?:and|dal|hus|oy)|raker)|i(?:d(?:tre-gauldal|sund)|l)|u(?:osat|seum)|r(?:\.gs)?|jondalen)|l(?:e(?:i(?:r(?:fjord|vik)|kanger)|(?:angaviik|sj)a|k(?:svik|a)|vanger|besby|nvik|rdal)|a(?:va(?:ngen|gis)|r(?:dal|vik)|akesvuemie|ngevag|hppi)|i(?:lle(?:hammer|sand)|nd(?:esne|a)s|er(?:ne)?)|o(?:(?:ding|t)en|renskog|abat|ppa|m)|u(?:n(?:ner|d)|ster|roy)|yng(?:dal|en))|f(?:l(?:e(?:kkefjord|sberg)|a(?:tanger|kstad)?|or[ao])|r(?:o(?:(?:st|y)a|land|gn)|e(?:drikstad|i)|ana)|o(?:l(?:kebib|lda)l|r(?:sand|de)|snes)|u(?:o(?:isku|ssko)|sa)|e(?:t(?:sund)?|dje)|y(?:lkesbib|resda)l|a(?:rsund|uske)|i(?:nnoy|tjar)|j(?:aler|ell)|m(?:\.gs)?|hs)|v(?:e(?:st(?:re-(?:slidre|toten)|(?:vago|b)y|fold\.sande|nes)|ga(?:rshei)?|r(?:dal|ran)|velstad|nnesla|fsn)|a(?:r(?:ggat|do|oy)|(?:apst|ll)e|g(?:an?|soy)|n(?:ylven|g)|ksdal|\.gs|dso)?|o(?:ss(?:evangen)?|agat|lda)|i(?:k(?:na)?|ndafjord)|f(?:\.gs)?|gs)|k(?:v(?:i(?:n(?:nherad|esdal)|t(?:eseid|soy))|a(?:(?:fjor|lsun)d|nangen|m))|r(?:o(?:kstadelva|dsherad)|a(?:anghke|gero)|istians[au]nd)|a(?:r(?:asjo(?:hka|k)|(?:ls|m)oy)|utokeino|fjord)|o(?:ngs(?:vinger|berg)|pervik|mmune)|l(?:abu|epp)|irkenes)|r(?:a(?:h(?:kkeravju|olt)|n(?:daberg|a)|(?:is|um)a|d(?:oy|e)|kkestad|lingen)|o(?:y(?:rvik|ken)|ms(?:kog|a)|llag|doy|ros|an|st)|i(?:n(?:g(?:e(?:rike|bu)|saker)|dal)|s(?:or|sa))|en(?:ne(?:soy|bu)|dalen)|l(?:\.gs)?|uovat|ygge)|t(?:r(?:o(?:ms[ao]|ndheim|andin|gstad)|an(?:[bo]y|a)|\.gs|ysil)?|y(?:s(?:fjord|nes|var)|nset|dal)|elemark\.(?:xn--b-5ga|bo)|o(?:nsberg|rsken|kke|lga)|i(?:n(?:gvoll|n)|me)|j(?:eldsund|ome)|ana(?:nger)?|m(?:\.gs)?|vedestrand)|o(?:s(?:t(?:fold\.valer|re-toten|eroy)|lo(?:\.gs)?|oyro|en)|r(?:k(?:anger|dal)|s(?:kog|ta)|land)|y(?:stre-slidre|garden|er)|v(?:re-eiker|erhalla)|(?:masvuotn|dd)a|pp(?:egard|dal)|f(?:\.gs)?|l(?:\.gs)?|ksnes)|g(?:a(?:(?:ngaviik|ivuotn|ls)a|u(?:sdal|lar)|mvik)|r(?:a(?:n(?:vin|e)?|tangen)|imstad|ong|ue)|j(?:e(?:r(?:drum|stad)|mnes|sdal)|ovik)|i(?:ehtavuoatna|ldeskal|ske)|u(?:ovdageaidnu|len)|loppen|ol)|e(?:i(?:d(?:s(?:(?:ber|ko)g|voll)|fjord)?|gersund)|v(?:en(?:assi|es)|je-og-hornnes)|n(?:gerdal|ebakk)|tne(?:dal)?|gersund|lverum)|d(?:i(?:vt(?:tasvuot|asvuod)na|elddanuorri)|r(?:a(?:ngedal|mmen)|obak)|avve(?:njarg|siid)a|o(?:nna|vre)|e(?:atnu|p)|yroy)|j(?:o(?:rpeland|lster|ndal)|e(?:ssheim|vnaker)|an-mayen(?:\.gs)?)|u(?:l(?:lens(?:aker|vang)|vik)|(?:njarg|tsir)a)|i(?:v(?:eland|gu)|bestad|nderoy|drett)|p(?:ors(?:ang(?:er|u)|grunn)|riv)|c(?:ahcesuol)?o)|rt(?:h(?:landinsurance|westernmutual)|on)|w(?:ruz|tv)?|kia)?|e(?:t(?:\.(?:d(?:o(?:ntexis|es-i)t|yna(?:thome|lias)|ns(?:alias|dojo))|s(?:e(?:l(?:ls-it|fip)|rve(?:bbs|ftp))?|crapper-site)|h(?:ome(?:(?:linu|uni)x|(?:ft|i)p)|am-radio-op|u)|i(?:s(?:-a-(?:chef|geek)|a-geek)|n-the-band)|(?:office-on-th|thruher|podzon)e|b(?:(?:uyshouse|logdn)s|roke-it)|(?:endofinterne|cloudfron)t|(?:at-band-cam|webho|j)p|from-(?:az|co|la|ny)|g(?:ets-it|b)|kicks-ass|uk|za)|(?:ban|wor)k|aporter|flix)?|x(?:(?:tdirec)?t|us)|w(?:holland|s)?|ustar|c)?|a(?:\.(?:c(?:[ac]|om?)|m(?:obi|x)|in(?:fo)?|school|[uw]s|name|org?|pro|dr|tv)|me(?:\.h(?:er|is)\.forgot)?|t(?:ionwide|ura)|(?:sper|ple)s|goya|dex|vy|b)?|f(?:\.(?:(?:fir|co)m|(?:oth|p)er|store|arts|info|net|rec|web)|l)?|r(?:[aw]|\.(?:info|biz|com|edu|gov|net|org))?|i(?:k(?:on|e)|\.[^\.]+|ssa[ny]|nja|co)?|g(?:\.(?:com|edu|gov|net|org|ac)|o)|u(?:\.(?:m(?:erse)?ine|shacknet))?|z(?:\.(?:co\.blogspot|[^\.]+))?|l(?:\.(?:b(?:logspot|v)|co))?|p(?:\.[^\.]+)?|c(?:\.asso)?|ba|hk|tt|yc)|m(?:[dhpq]|u(?:s(?:eum(?:\.(?:c(?:o(?:l(?:o(?:nialwilliamsburg|radoplateau)|umb(?:ia|us)|lection|dwar)|m(?:muni(?:cations?|ty)|puter(?:history)?)|n(?:temporary(?:art)?|vent)|r(?:poration|vette)|a(?:staldefence|l)|unt(?:ryestate|y)|penhagen|stume|dy)|h(?:i(?:ldren(?:s(?:garden)?)?|ropractic|cago)|e(?:sapeakebay|ltenham)|ristiansburg|attanooga|ocolate)|a(?:s(?:adelamoneda|t(?:res|le))|r(?:toonart|rier)|n(?:ada)?|lifornia|pebreton|mbridge|daques)|i(?:vil(?:i[sz]ation|war)|n(?:cinnati|ema)|rcus)|r(?:a(?:nbrook|fts)|eation)|ultur(?:al(?:center)?|e)|e(?:ltic|nter)|l(?:inton|ock)|y(?:ber|mru))|s(?:c(?:ience(?:(?:and(?:indust|histo)|histo)ry|(?:snaturelle)?s|-fiction|centers?)?|h(?:o(?:(?:enbrun|kolade)n|ol)|lesisches|weiz)|otland)|a(?:n(?:ta(?:barbara|cruz|fe)|(?:francisc|dieg)o)|l(?:vadordali|zburg|em)|skatchewan|intlouis|vannahga|tx)|t(?:a(?:t(?:(?:eofdelawar)?e|ion)|rnberg|lbans|dt)|e(?:iermark|am)|petersburg|ockholm|uttgart|john)|o(?:u(?:th(?:carolina|west)|ndandvision)|ciety|logne)|u(?:r(?:geonshall|rey)|isse)|e(?:ttle(?:ment|rs)|aport)|he(?:rbrooke|ll)|i(?:beni|l)k|k(?:ole|i)|p(?:ace|y)|vizzera|quare|weden|ydney)|a(?:r(?:t(?:(?:anddesig|educatio)n|(?:sandcraft)?s|gallery|center|deco)?|ch(?:aeolog(?:ical|y)|itecture)|boretum)|m(?:erican(?:a(?:ntiques|rt)?)?|b(?:ulance|er)|sterdam)|s(?:s(?:(?:assin|oci)ation|isi)|tronomy|matart)|n(?:t(?:hro(?:pology)?|iques)|nefrank|d)|u(?:st(?:ralia|in)|tomotive)|(?:la(?:bam|sk)|tlant)a|ir(?:guard)?|griculture|quarium|viation|cademy|xis)|m(?:a(?:r(?:yl(?:hurst|and)|itim[eo]|burg)|n(?:chester|sions?|x)|(?:dri)?d|llorca)|u(?:s(?:e(?:um(?:vereniging|center)|et)|ic)|en(?:chen|ster)|(?:lhous|nci)e)|i(?:n(?:nesota|ers|ing)|ss(?:oula|ile)|l(?:itary|l)|datlantic|chigan)|o(?:n(?:t(?:icello|real)|mouth|ey)|torcycle|dern|scow|ma)|e(?:di(?:zinhistorisches|cal|a)|saverde|morial|eres))|b(?:r(?:i(?:tish(?:columbia)?|stol)|a(?:ndywinevalley|sil)|u(?:ssels?|xelles|nel)|oadcast)|a(?:l(?:timor)?e|se(?:bal)?l|(?:uer|h)n|rcelona|dajoz|ghdad|ths)|e(?:r(?:(?:li)?n|keley|gbau)|eldengeluid|auxarts|llevue)|o(?:tan(?:ic(?:al(?:garden)?|garden)|y)|(?:sto|n)n)|i(?:r(?:thplace|dart)|l(?:bao|l)|ble)|u(?:s(?:hey)?|ilding|rghof))|p(?:h(?:ila(?:delphi(?:aare)?a|tely)|o(?:tography|enix)|armacy)|a(?:l(?:msprings|ace|eo)|(?:saden|nam)a|derborn|cific|ris)|o(?:sts-and-telecommunications|rt(?:l(?:ligat|and)|al))|la(?:n(?:t(?:ation|s)|etarium)|za)|r(?:es(?:ervation|idio|s)|oject)|i(?:ttsburgh|lots)|ub(?:lic|ol))|n(?:a(?:t(?:u(?:r(?:al(?:history(?:museum)?|sciences)|historisches|e)|urwetenschappen)|i(?:onal(?:firearms|heritage)?|veamerican))|umburg|val)|e(?:w(?:hampshire|jersey|mexico|spaper|port|york)|braska|ues)|u(?:ern|rem)berg|or(?:folk|th)|y(?:ny|c)|iepce|rw)|f(?:r(?:an(?:ziskaner|caise|kfurt)|e(?:emasonry|iburg)|(?:ibour|o)g)|i(?:n(?:earts?|land)|l(?:atelia|m)|gueres|eld)|o(?:r(?:t(?:missoula|worth)|ce)|undation)|a(?:rm(?:e(?:quipment|rs)|stead)?|mily)|u(?:rniture|ndacio)|l(?:anders|orida))|h(?:isto(?:r(?:i(?:c(?:al(?:society)?|houses)|sch(?:es)?)|y(?:ofscience)?)|ire)|e(?:l(?:sinki|las)|mbygdsforbund|imatunduhren|ritage|alth)|a(?:(?:rvestcelebrati|nds)on|lloffame|mburg|waii)|o(?:rology|use)|umanities)|e(?:n(?:vironment(?:alconservation)?|cyclopedic|tomology|gland)|(?:mbroider|thnolog|pileps)y|(?:isenbah|gyptia)n|ast(?:africa|coast)|l(?:vendrell|burg)|x(?:hibition|eter)|ducation(?:al)?|s(?:tate|sex))|l(?:a(?:n(?:(?:de)?s|cashire)|bou?r|jolla|rsson)|o(?:calhistory|sangeles|yalist|ndon|uvre)|i(?:ving(?:history)?|n(?:coln|z))|u(?:xembourg|cerne|zern)|ewismiller)|u(?:s(?:c(?:ountryestat|ultur)e|a(?:(?:ntique|rt)s)?|h(?:istory|uaia)|decorativearts|livinghistory|garden)|n(?:iversity|dersea)|hren|tah|vic|lm)|t(?:e(?:le(?:kommunikat|vis)ion|x(?:tile|as)|chnology)|r(?:ust(?:ee)?|ansport|olley|ee)|o(?:pology|rino|uch|wn)|ime(?:keeping)?|heater|ank|cm)|d(?:e(?:l(?:menhorst|aware)|(?:troi|po)t|corativearts|nmark|sign)|a(?:l(?:las|i)|tabase)|i(?:scovery|nosaur)|o(?:nostia|lls)|urham|dr)|w(?:a(?:tch(?:-and-|and)clock|l(?:lonie|es)|shingtondc|r)|i(?:l(?:liamsburg|dlife)|ndmill)|est(?:fale|er)n|orkshop|haling)|g(?:e(?:o(?:logy|rgia)|mological|elvinck)|a(?:(?:ller|tewa)y|rden)|ra(?:ndrapids|z)|uernsey|iessen|las?s|orge)|i(?:n(?:dian(?:a(?:polis)?|market)?|te(?:lligenc|ractiv)e)|(?:llustratio|sleofma)n|mageandsound|r(?:aq|on))|j(?:e(?:w(?:ish(?:art)?|elry)|fferson|rusalem)|u(?:d(?:ygarland|aica)|edisches|if)|ournalism|amison|fk)|v(?:i(?:r(?:tu[ae]l|ginia)|llage|king)|a(?:lley|ntaa)|olkenkunde|ersailles|laanderen)|xn--(?:co(?:rreios-e-telecomunicaes-ghc29a|municaes-v6a2o)|9dbhblg6di|lns-qla|h1aegh)|r(?:o(?:c(?:hester|kart)|ma)|es(?:istance|earch)|ail(?:road|way)|iodejaneiro|ussia)|o(?:ceanographi(?:que|c)|n(?:tario|line)|regon(?:trail)?|penair|xford|maha|tago)|k(?:unst(?:unddesign|sammlung)?|ar(?:ikatur|ate)|oe(?:benhav|l)n|ids)|yo(?:rk(?:shire)?|semite|uth)|zoolog(?:ical|y)|quebec))?|ic)|\.(?:com?|org?|gov|net|ac)|tu(?:al(?:funds)?|elle)|ltichoice)?|a(?:r(?:ket(?:ing|s)?|shalls|riott)|\.(?:press|gov|net|org|ac|co)|n(?:agement|go)?|i(?:[fl]|son)|t(?:rix|tel)|(?:keu)?p|ybelline|serati|drid|cys)?|o(?:[eim]|\.(?:com|edu|gov|net|org)|n(?:tblanc|ster|ash|ey)|v(?:i(?:star|e))?|r(?:tgage|mon)|to(?:rcycles)?|bi(?:l[ey])?|scow|zaic|par|da)?|e(?:\.(?:(?:pri|go)v|edu|its|net|org|ac|co)|(?:lbourn|tlif)e|d(?:i(?:cal|a))?|m(?:orial|e)|rck(?:msd)?|nu?|et|o)?|v(?:\.(?:m(?:useum|il)|n(?:ame|et)|(?:ae|p)ro|co(?:op|m)|in(?:fo|t)|biz|edu|gov|org))?|w(?:\.(?:co(?:op|m)?|(?:in|ne)t|museum|biz|edu|gov|org|ac))?|i(?:[hl]|t(?:subishi|ek)?|(?:am)?i|crosoft|n[it])|l(?:[bs]|\.(?:presse|gou?v|com|edu|net|org))?|g(?:\.(?:(?:[cn]o|t)m|edu|gov|mil|org|prd))?|k(?:\.(?:n(?:ame|et)|com|edu|gov|inf|org))?|x(?:\.(?:(?:blogspo|ne)t|com|edu|gob|org))?|y(?:\.(?:n(?:ame|et)|com|edu|gov|mil|org))?|c(?:\.(?:asso|tm)|d(?:onalds)?|kinsey)?|r(?:\.(?:blogspot|gov)|muscle|porter)?|z(?:\.(?!teledata$)[^\.]+|ansimagic)?|n(?:\.(?:edu|gov|nyc|org)|et)?|t(?:[nr]|\.[^\.]+|pc)?|m(?:\.[^\.]+|a)?|sd?|ba)|c(?:o(?:m(?:\.(?:i(?:s(?:-(?:a(?:-(?:(?:(?:p(?:ersonaltrain|hotograph|aint|lay)|h(?:ard-work|unt))e|financialadviso)r|c(?:(?:ubicle-sla|onservati)ve|aterer|hef|pa)|l(?:iber(?:tarian|al)|a(?:ndscap|wy)er|lama)|(?:s(?:ocialis|tuden)|anarchis)t|b(?:(?:ookkeep|logg)er|ulls-fan)|d(?:e(?:mocrat|signer)|octor)|t(?:e(?:acher|chie)|herapist)|r(?:epublican|ockstar)|n(?:ascarfan|urse)|g(?:reen|eek|uru)|musician)|n-(?:a(?:c(?:t(?:ress|or)|countant)|(?:narch|rt)ist)|en(?:tertain|gine)er))|into-(?:(?:car(?:toon)?|game)s|anime)|(?:(?:not-)?certifie|with-theban)d|(?:uber)?leet|slick|gone)|a-(?:hockeynut|geek)|smarterthanyou)|amallama)|a(?:mazonaws\.(?:s3(?:-(?:website-(?:us-(?:(?:gov-we|ea)st-1|west-[12])|ap-(?:southeast-[12]|northeast-1)|(?:eu-we|sa-ea)st-1)|(?:(?:fips-us-gov|eu)-we|sa-ea)st-1|ap-(?:southeast-[12]|northeast-1)|us-(?:gov-west-1|west-[12])))?|elb)|ppspot|r)|d(?:yn(?:dns-(?:(?:offic|remot|fre|hom)e|at-(?:home|work)|w(?:iki|ork|eb)|server|blog|mail|pics|ip)|-o-saur|alias)|o(?:(?:es)?ntexist|omdns)|ns(?:alias|dojo)|reamhosters|e)|from-(?:m[adinost]|n[c-ehjmv]|i[adln]|w[aivy]|a[klr]|o[hkr]|[hr]i|c[at]|d[ce]|k[sy]|p[ar]|s[cd]|t[nx]|v[at]|fl|ga|ut)|s(?:e(?:l(?:ls-for-(?:less|u)|fip)|rvebbs)?|a(?:ves-the-whales)?|pace-to-rent|imple-url)|e(?:st-(?:(?:a-la-ma(?:is|si)|le-patr)on|mon-blogueur)|lasticbeanstalk|u)|g(?:[br]|o(?:ogle(?:apis|code)|tdns)|etmyip)|h(?:o(?:me(?:linu|uni)x|bby-site)|u)|b(?:log(?:spot|dns)|etainabox|r)|c(?:odespot|echire|n)|likes(?:candy|-pie)|(?:teaches-yog|z)a|n(?:eat-url|o)|writesthisblog|r(?:hcloud|u)|operaunite|u[ksy]|jpn|kr|qc)|p(?:a(?:ny|re)|uter)|m(?:unity|bank)|cast|sec)?|\.(?:(?:fir|co)m|in(?:fo|t)|n(?:et|om)|arts|edu|gov|mil|org|rec|web)|n(?:s(?:truction|ulting)|t(?:ractors|act)|(?:nector|do)s)|o(?:[lp]|king(?:channel)?)|(?:l(?:leg|ogn)|ffe)e|u(?:pons?|ntry|rses)|r(?:sica|p)|ach|des)?|a(?:\.(?:b(?:logspot|c)|n[bfls-u]|[am]b|[gq]c|[sy]k|co|on|pe)|r(?:e(?:ers?|more)?|insurance|avan|tier|d?s)?|s(?:h(?:backbonus)?|e(?:ih)?|ino|a)|n(?:cerresearch|alplus|on)|l(?:vinklein|ifornia|l)?|t(?:alonia|ering|holic)?|p(?:ital(?:one)?|etown)|m(?:era|p)?|dillac|fe|b)?|n(?:\.(?:x(?:[jz]|n--(?:55qx5d|io0a7i|od0alg))|g(?:[dsxz]|ov)|n(?:[mx]|et)|h[abeikln]|c(?:om|q)|m(?:il|o)|s[cdhnx]|[bfz]j|j[lsx]|[ly]n|a[ch]|t[jw]|edu|org|qh))?|i(?:\.(?:xn--aroport-bya|(?:in|ne)t|a(?:sso|c)|go(?:uv)?|presse|com?|edu?|org?|md)|t(?:y(?:eats)?|adel|ic?)|priani|alis|rcle|sco|mb)?|h(?:a(?:n(?:giairport|n?el)|r(?:ity|tis)|tr?|se)|e(?:v(?:rolet|y)|sapeake|ap)|r(?:istmas|ysler|ome)|\.blogspot|intai|urch|loe|k)?|l(?:\.(?:go[bv]|mil|co)|i(?:ni(?:que|c)|ck)|o(?:thing|ud)|ub(?:med)?|eaning|aims)?|r(?:\.(?:[cg]o|ac|ed|fi|or|sa)|edit(?:union|card)?|uises?|icket|own|s)?|c(?:\.(?:(?:ftpacces|myphoto)s|game-server|scrapping))?|u(?:\.(?:com|edu|gov|inf|net|org)|isinella)?|w(?:\.(?:com|edu|net|org))?|y(?:(?:mr|o)u|\.[^\.]+)?|k(?:\.(?!www$)[^\.]+)?|f(?:[ad]|\.blogspot)?|x(?:\.(?:ath|gov))?|e(?:[bo]|nter|rn)|v(?:\.blogspot)?|z(?:\.blogspot)?|b(?:[ans]|re)|d(?:\.gov)?|m(?:\.gov)?|pa|sc|g)|i(?:t(?:\.(?:a(?:[pt]|n(?:dria(?:-(?:barletta-trani|trani-barletta)|barlettatrani|tranibarletta)|cona)?|l(?:essandria|to-?adige)?|s(?:coli-?piceno|ti)|g(?:rigento)?|o(?:st[ae])?|v(?:ellino)?|q(?:uila)?|r(?:ezzo)?)|c(?:[bilnstz]|a(?:r(?:bonia-?iglesias|rara-?massa)|mp(?:idano-?medi|obass)o|(?:ltanisset|ser)ta|tan(?:zaro|ia)|gliari)?|e(?:sena-?forli)?|r(?:emona|otone)?|o(?:senza|mo)?|h(?:ieti)?|uneo)|t(?:[psv]|r(?:a(?:ni(?:-(?:andria-barlett|barletta-andri)|andriabarlett|barlettaandri)a|pani)|e(?:nt(?:in)?|vis)o|ieste)?|e(?:mpio-?olbia|r(?:amo|ni))?|a(?:ranto)?|(?:orin)?o|(?:uri)?n)|b(?:[gnstz]|a(?:r(?:letta(?:-trani-|trani)andria|i)|lsan)?|o(?:l(?:ogna|zano)|zen)?|e(?:nevent|llun|rgam)o|r(?:indisi|escia)?|l(?:ogspot)?|i(?:ella)?)|m(?:[bcnst]|o(?:nza(?:(?:(?:-e-della)?-|e(?:della)?)?brianza)?|dena)?|a(?:(?:ssa-?carra|te)r|cerat|ntov)a|e(?:dio-?campidano|ssina)?|i(?:lano?)?)|p(?:[cdgnt-vz]|e(?:s(?:aro-?urbino|cara)|rugia)?|a(?:(?:d(?:ov|u)|rm|vi)a|lermo)?|i(?:(?:s(?:toi)?|acenz)a)?|o(?:rdenone|tenza)?|r(?:ato)?)|v(?:[bcr-tv]|e(?:r(?:(?:bani|on)a|celli)|n(?:ezia|ice))?|i(?:(?:bo-?valenti|cenz)a|terbo)?|a(?:rese)?)|r(?:[cgmn]|e(?:ggio-?(?:calabr|emil)ia)?|a(?:(?:venn|gus)a)?|i(?:(?:min|et)i)?|o(?:m[ae]|vigo)?)|f(?:[cgm]|o(?:rli-?cesen|ggi)a|e(?:r(?:rara|mo))?|r(?:osinone)?|i(?:renze)?|lorence)|l(?:[ct]|a(?:-?spezi|quil|tin)a|e(?:cc[eo])?|i(?:vorno)?|u(?:cca)?|o(?:di)?)|s(?:[prsv]|a(?:lerno|ssari|vona)?|i(?:(?:racus|en)a)?|(?:ondri)?o|uedtirol)|i(?:glesias-?carbonia|m(?:peria)?|s(?:ernia)?)|o(?:g(?:liastra)?|lbia-?tempio|r(?:istano)?|t)|n(?:a(?:p(?:les|oli))?|o(?:vara)?|u(?:oro)?)|g(?:o(?:rizia|v)?|r(?:osseto)?|e(?:nov?a)?)|u(?:rbino-?pesaro|d(?:ine)?)|e(?:n(?:na)?|du)|dell-?ogliastra|kr)|au|v)?|n(?:[cgk]|f(?:o(?:\.(?:(?:barrel?l-of-knowledg|here-for-mor)e|(?:selfi|webho)p|groks-th(?:is|e)|knowsitall|for-our|dyndns)|sys)?|initi|y)|\.(?:g(?:en|ov)|n(?:et|ic)|blogspot|firm|edu|ind|mil|org|res|ac|co)|t(?:e(?:rnationa)?l|\.eu|uit)?|(?:d(?:ustrie|ian)|vestment)s|s(?:ur(?:anc)?|titut)e)?|m(?:\.(?:co(?:\.(?:ltd|plc))?|n(?:et|ic)|gov|org|ac)|mo(?:bilien)?|amat|db)?|s(?:\.(?:(?:in|ne)t|com|edu|gov|org)|t(?:anbul)?|elect|maili|lam)?|r(?:\.(?:xn--mgba3a4f(?:16|r)a|gov|net|org|sch|ac|co|id)|ish|a)?|d(?:\.(?:m(?:il|y)|[cg]o|biz|net|sch|web|ac|or)|n)?|q(?:\.(?:com|edu|gov|mil|net|org))?|l(?:\.(?:co\.blogspot|[^\.]+))?|e(?:\.(?:blogspot|gov)|ee)?|o(?:\.(?:github|com))?|(?:kan|vec)o|c(?:[eu]|bc)|piranga|[bf]m|inet|wc)|p(?:l(?:\.(?:s(?:ta(?:r(?:achowice|gard)|lowa-wola)|wi(?:noujscie|ebodzin|dnica)|z(?:cz(?:ecin|ytno)|kola)|o(?:s(?:nowiec)?|pot)|(?:l(?:up|a)s|ano)k|k(?:oczow|lep)|e(?:jny|x)|iedlce|uwalki|hop)|g(?:o(?:v(?:\.(?:s(?:(?:tarostw)?o|r)|u(?:[gm]|(?:po)?w)|p[ao]))?|rlice)|d(?:a(?:nsk)?|ynia)|(?:niezn|rajew)o|l(?:iwice|ogow)|mina|sm)|k(?:a(?:(?:zimierz-doln|szub)y|r(?:pacz|tuzy)|towice|lisz)|o(?:n(?:skowola|in)|bierzyce|lobrzeg)|(?:lodzk|utn)o|e(?:trzyn|pno)|rakow)|p(?:o(?:(?:d(?:lasi|hal)|mor(?:ski|z)|lkowic)e|wiat|znan)|r(?:ochowice|zeworsk|uszkow|iv)|i(?:la|sz)|ulawy|c)|b(?:i(?:al(?:owieza|ystok)|e(?:szczady|lawa)|z)|y(?:dgoszcz|tom)|e(?:skidy|dzin)|oleslawiec|abia-gora)|m(?:a(?:l(?:opolska|bork)|z(?:owsze|ury)|il)|i(?:el(?:ec|no)|asta|l)|ed(?:ia)?|ragowo|bone)|w(?:a(?:r(?:szaw|mi)a|lbrzych|w)|o(?:dzislaw|lomin)|locl(?:awek)?|roc(?:law)?|egrow|ielun)|o(?:stro(?:w(?:wlkp|iec)|(?:lek|d)a)|l(?:sztyn|ecko|kusz|awa)|po(?:czno|le)|rg)|l(?:e(?:(?:zajs|bor)k|gnica)|o(?:wicz|mza)|u(?:bin|kow)|imanowa|apy)|t(?:ar(?:nobrzeg|gi)|ur(?:ystyka|ek)|(?:gor|ych)y|(?:ouris)?m|ravel)|n(?:o(?:waruda|m)|ieruchomosci|(?:akl|g)o|ysa|et)|r(?:a(?:wa-maz|dom)|e(?:alestate|l)|zeszow|ybnik)|z(?:a(?:chpomor|kopane|gan|row)|gor(?:zelec|a))|a(?:u(?:gustow|to)|gro|id|rt|tm)|c(?:ze(?:ladz|st)|ieszyn|om?)|j(?:(?:elenia-)?gora|aworzno)|e(?:l(?:blag|k)|du)|i(?:lawa|nfo|rc)|us(?:enet|tka)|dlugoleka|6bone)|a(?:y(?:station)?|ce)|u(?:mbing|s))?|r(?:o(?:\.(?:(?:ac|cp)a|(?:ba|ju)r|eng|law|med)|d(?:uctions)?|pert(?:ies|y)|gressive|tection|mo|f)?|\.(?:i(?:nfo|sla)|n(?:ame|et)|e(?:du|st)|prof?|biz|com|gov|org|ac)|a(?:merica|xi)|u(?:dential)?|ess|ime)?|a(?:\.(?:(?:in|or)g|(?:me|sl)d|n(?:et|om)|a(?:bo|c)|com|edu|gob)|r(?:t(?:(?:ner)?s|y)|i?s)|n(?:asonic|erai)|t(?:agonia|ch)|mperedchef|ssagens|yu?|ge|d)?|h(?:\.(?:n(?:et|go)|com|edu|gov|mil|org|i)|o(?:to(?:graphy|s)?|ne)|armacy|ilips|ysio|d)?|i(?:c(?:t(?:ures|et)|s)|perlime|n[gk]?|oneer|aget|tney|zza|d)|e(?:\.(?:n(?:et|om)|com|edu|gob|mil|org)|rsiangulf|ace|ts?)?|t(?:\.(?:(?:blogspo|in)t|n(?:ome|et)|publ|com|edu|gov|org))?|k(?:\.(?:(?:co|fa)m|go[bknpsv]|info|biz|edu|net|org|web))?|y(?:\.(?:co(?:op|m)|edu|gov|mil|net|org))?|s(?:\.(?:com|edu|gov|net|org|plo|sec))?|w(?:\.(?:[cg]o|belau|ed|ne|or)|c)?|n(?:\.(?:edu|gov|net|org|co)|c)?|o(?:l(?:itie|o)|ker|hl|rn|st)|f(?:\.(?:com|edu|org)|izer)?|g(?:\.[^\.]+)?|ccw|ub|m)|u(?:s(?:\.(?:m(?:a(?:\.(?:k12(?:\.(?:p(?:aroch|vt)|chtr))?|lib|cc))?|d(?:\.(?:k12|lib|cc))?|e(?:\.(?:k12|lib|cc))?|i(?:\.(?:k12|lib|cc))?|n(?:\.(?:k12|lib|cc))?|o(?:\.(?:k12|lib|cc))?|s(?:\.(?:k12|lib|cc))?|t(?:\.(?:k12|lib|cc))?)|n(?:c(?:\.(?:k12|lib|cc))?|d(?:\.(?:k12|lib|cc))?|e(?:\.(?:k12|lib|cc))?|h(?:\.(?:k12|lib|cc))?|j(?:\.(?:k12|lib|cc))?|m(?:\.(?:k12|lib|cc))?|v(?:\.(?:k12|lib|cc))?|y(?:\.(?:k12|lib|cc))?|sn)|a(?:k(?:\.(?:k12|lib|cc))?|l(?:\.(?:k12|lib|cc))?|r(?:\.(?:k12|lib|cc))?|s(?:\.(?:k12|lib|cc))?|z(?:\.(?:k12|lib|cc))?)|i(?:a(?:\.(?:k12|lib|cc))?|d(?:\.(?:k12|lib|cc))?|l(?:\.(?:k12|lib|cc))?|n(?:\.(?:k12|lib|cc))?|s(?:-by|a))|w(?:a(?:\.(?:k12|lib|cc))?|i(?:\.(?:k12|lib|cc))?|v(?:\.(?:k12|lib|cc))?|y(?:\.(?:k12|lib|cc))?)|c(?:a(?:\.(?:k12|lib|cc))?|o(?:\.(?:k12|lib|cc))?|t(?:\.(?:k12|lib|cc))?)|o(?:h(?:\.(?:k12|lib|cc))?|k(?:\.(?:k12|lib|cc))?|r(?:\.(?:k12|lib|cc))?)|v(?:a(?:\.(?:k12|lib|cc))?|i(?:\.(?:k12|lib|cc))?|t(?:\.(?:k12|lib|cc))?)|s(?:c(?:\.(?:k12|lib|cc))?|d(?:\.(?:k12|lib|cc))?|tuff-4-sale)|k(?:s(?:\.(?:k12|lib|cc))?|y(?:\.(?:k12|lib|cc))?|ids)|d(?:c(?:\.(?:k12|lib|cc))?|e(?:\.(?:k12|lib|cc))?|ni)|g(?:a(?:\.(?:k12|lib|cc))?|u(?:\.(?:k12|lib|cc))?)|p(?:a(?:\.(?:k12|lib|cc))?|r(?:\.(?:k12|lib|cc))?)|t(?:n(?:\.(?:k12|lib|cc))?|x(?:\.(?:k12|lib|cc))?)|la(?:\.(?:k12|lib|cc)|nd-4-sale)?|f(?:l(?:\.(?:k12|lib|cc))?|ed)|ri(?:\.(?:k12|lib|cc))?|ut(?:\.(?:k12|lib|cc))?|hi(?:\.(?:lib|cc))?)|a)?|a(?:\.(?:k(?:[ms]|h(?:meln(?:itskiy|ytskyi)|ark[io]v|erson)?|i(?:rovograd|ev)|(?:yi)?v|r(?:ym)?)|c(?:[knv]|her(?:n(?:i(?:(?:go|hi)v|vtsi)|ovtsy)|kas?sy)|r(?:imea)?|om?)|d(?:n(?:[ei]propetrovsk)?|o(?:minic|netsk)|p)|z(?:[pt]|aporizhzh(?:ia|e)|h(?:itomi|ytomy)r)|l(?:[gt]|u(?:gan|t)sk|(?:vi)?v)|v(?:inn(?:ytsi|ic)a|(?:oly)?n)|s(?:[bm]|e[bv]astopol|umy)|i(?:[fn]|vano-frankivsk)|o(?:d(?:es?sa)?|rg)|p(?:[lp]|oltava)|r(?:ivne|ovno|v)|n(?:ikolaev|et)|m(?:ykolaiv|k)|te(?:rnopil)?|uz(?:hgorod)?|yalta|edu|gov))?|k(?:\.(?:(?!bl$)(?!british-library$)(?!jet$)(?!mod$)(?!national-library-scotland$)(?!nel$)(?!nic$)(?!nls$)(?!parliament$)[^\.]+|police(?:\.[^\.]+)?|nhs(?:\.[^\.]+)?|sch(?:\.[^\.]+)?|co\.blogspot))?|n(?:i(?:c(?:o(?:rn|m)|ef)|versity)|o)|y(?:\.(?:com|edu|gub|mil|net|org))?|g(?:\.(?:[as]c|com?|org?|go|ne))?|z(?:\.(?:com?|net|org))?|b(?:ank|s)|ltrabook|connect|mmah|ol|ps)|a(?:e(?:ro(?:\.(?:a(?:ir(?:-(?:traffic-control|surveillance)|(?:craf|por)t|traffic|line)|(?:ccident-(?:investiga|preven)|ssocia)tion|ero(?:batic|drome|club)|m(?:bulance|usement)|gents|uthor)|c(?:o(?:n(?:sult(?:ant|ing)|ference|trol)|uncil)|(?:ertific|ivilavi)ation|ha(?:mpionship|rter)|a(?:tering|rgo|a)|lub|rew)|e(?:n(?:gine(?:er)?|tertainment)|x(?:change|press)|mergency|quipment|ducator)|p(?:a(?:ssenger-association|ra(?:chut|glid)ing)|r(?:oduction|ess)|ilot)|m(?:a(?:(?:intenan|rketpla)c|gazin)e|icrolight|odelling|edia)|s(?:(?:cientis|tuden)t|kydiving|ervices|oftware|afety|how)|r(?:e(?:s(?:earch)?|creation|pbody)|otorcraft)|g(?:rou(?:ndhandling|p)|overnment|liding)|f(?:(?:re|l)ight|ederation|uel)|t(?:ra(?:d(?:ing|er)|iner)|axi)|h(?:anggliding|omebuilt)|b(?:allooning|roker)|l(?:ogistics|easing)|work(?:inggroup|s)|(?:navigat|un)ion|journal(?:ist)?|d(?:esign|gca)|insurance))?|\.(?:gov|mil|net|org|sch|ac|co)|tna|g)?|u(?:\.(?:edu(?:\.(?:n(?:sw|t)|[sw]a|act|qld|tas|vic))?|gov(?:\.(?:(?:ac|n)t|[sw]a|qld|tas|vic))?|co(?:m(?:\.blogspot)?|nf)|n(?:e?t|sw)|a(?:ct|sn)|i(?:nfo|d)|o(?:rg|z)|[sw]a|qld|tas|vic)|t(?:o(?:insurance|s)?|hor)|di(?:ble|o)?|ction|spost)|r(?:\.(?:(?!congresodelalengua3$)(?!educ$)(?!gobiernoelectronico$)(?!mecon$)(?!nacion$)(?!nic$)(?!promocion$)(?!retina$)(?!uba$)[^\.]+|com\.blogspot)|pa\.(?:i(?:n-addr|ris|p6)|ur[in]|e164)|chi(?:tect)?|a(?:mco|b)|te?|my|e)?|l(?:l(?:finanz(?:berat(?:ung|er))?|state|y)|\.(?:com|edu|gov|mil|net|org)|i(?:baba|pay)|s(?:ace|tom)|faromeo|con)?|c(?:\.(?:com|edu|gov|mil|net|org)|c(?:ountants?|enture)|t(?:ive|or)|ademy|er|o)?|n(?:\.(?:com|edu|net|org)|(?:alytic|son)s|t(?:ivirus|hem)|(?:droi)?d|quan|z)?|t(?:\.(?:co(?:\.blogspot)?|(?:pri|g)v|info|biz|ac|or)|h(?:leta|ens)|torney)?|z(?:\.(?:n(?:ame|et)|in(?:fo|t)|p(?:ro|p)|biz|com|edu|gov|mil|org)|ure)?|f(?:\.(?:com|edu|gov|net|org)|rica(?:magic)?|amilycompany|l)?|m(?:e(?:rican(?:express|family)|x)|(?:sterd|f)am|azon|ica|p)?|i(?:\.(?:o(?:ff|rg)|com|net)|r(?:force|bus|tel)|go?)?|g(?:\.(?:n(?:et|om)|com?|org)|akhan|ency)?|b(?:b(?:ott|vie)?|udhabi|ogado|arth|le|c)|s(?:sociates|[di]a|\.gov|trium)?|o(?:\.(?:co|ed|gv|it|og|pb)|l)?|q(?:u(?:arell|itain)e)?|p(?:artments|p(?:le)?)|d(?:\.nom|ult|ac|s)?|v(?:ianca|ery)|w(?:\.com|s)?|x(?:is|a)?|a(?:rp|a)|kdn)|r(?:u(?:\.(?:k(?:a(?:(?:l(?:myki|ug)|mchatk|reli)a|zan)|u(?:r(?:gan|sk)|stanai|zbass|ban)|h(?:a(?:barovsk|kassia)|v)|(?:rasnoyar|-ural)sk|o(?:stroma|enig|mi)|emerovo|irov|chr|ms)|s(?:a(?:khalin|ratov|mara)|(?:imbir|molen)sk|t(?:avropol|v)|urgut|yzran|nz|pb)|m(?:a(?:g(?:nitka|adan)|ri(?:-el|ne)?)|o(?:rdovia|sreg)|(?:urman)?sk|ytis|il)|v(?:o(?:l(?:gograd|ogda)|ronezh)|ladi(?:kavkaz|vostok|mir)|donsk|yatka|rn)|t(?:a(?:tarstan|mbov)|s(?:aritsyn|k)|om(?:sk)?|u[lv]a|yumen|est|ver)|n(?:o(?:v(?:osibirsk)?|rilsk)|a(?:khodka|lchik)|nov|et|kz|sk)|y(?:a(?:(?:roslav|ma)l|kutia)|uzhno-sakhalinsk|ekaterinburg)|c(?:h(?:(?:u(?:vashi|kotk)|it)a|el(?:yabinsk)?)|bg|mw|om)|a(?:mur(?:sk)?|rkhangelsk|strakhan|dygeya|ltai|c)|b(?:a(?:shkiria|ikal)|elgorod|uryatia|ryansk|ir)|p(?:e(?:nza|rm)|yatigorsk|alana|skov|tz|p)|o(?:r(?:(?:enbur)?g|yol)|skol|msk)|i(?:(?:rkut|zhev)sk|vanovo|nt)|j(?:a(?:mal|r)|oshkar-ola)|u(?:dm(?:urtia)?|lan-ude)|r(?:ubtsovsk|yazan|nd)|d(?:agestan|udinka)|e(?:-burg|du)|g(?:rozny|ov)|fareast|lipetsk|zgrad)|gby|hr|n)?|e(?:a(?:l(?:t(?:or|y)|estate)?|d)|\.(?:blogspot|[cn]om|asso)|(?:s(?:tauran)?|tiremen)t|d(?:umbrella|stone|ken)?|p(?:ublican|air|ort)|n(?:t(?:als)?)?|i(?:sen?|t)|liance|views?|cipes|xroth|hab)?|o(?:\.(?:(?:fir|co|t)m|n(?:om|t)|blogspot|store|arts|info|org|rec|www)|c(?:k(?:wool|s)|her)|gers|deo|ma|om)?|w(?:\.(?:(?:in|ne)t|gou?v|com?|edu|mil|ac)|e)?|i(?:[lop]|c(?:h(?:ardli)?|oh)|ghtathome)|s(?:\.(?:edu|gov|org|ac|co|in)|vp)?|a(?:cing|dio|id|m)|yukyu|mit)|s(?:e(?:\.(?:[c-eg-imruw-z]|k(?:om(?:munalforbund|forb|vux))?|b(?:(?:ran)?d|logspot)?|(?:naturbruksgym)?n|p(?:arti|ress|p)?|f(?:h(?:sk|v)?)?|l(?:anbib)?|s(?:shn)?|o(?:rg)?|ac?|tm?)|cur(?:ity|e)|(?:rvice)?s|a(?:rch|t)|lect|ner|ven|xy?|ek|w)?|t(?:\.(?:(?:s(?:aotom|tor)|princip)e|co(?:nsulado|m)?|e(?:mbaixada|du)|gov|mil|net|org)|a(?:t(?:e(?:bank|farm)|oil)|r(?:hub)?|ples|da)|o(?:r(?:ag)?e|ckholm)|r(?:eam|oke)|c(?:group)?|ud(?:io|y)|yle)?|a(?:\.(?:com|edu|gov|med|net|org|pub|sch)|n(?:dvik(?:coromant)?|ofi)|fe(?:(?:wa|t)y)?|ms(?:club|ung)|p(?:phire|o)?|l(?:on|e)|arland|kura|rl|ve|xo|s)?|c(?:[ab]|h(?:o(?:larships|ol)|warz(?:group)?|aeffler|midt|ule)|\.(?:com|edu|gov|net|org)|johnson|o[rt]?|ience)?|h(?:o(?:p(?:yourway|ping)?|w(?:time)?|uji|es)|\.(?:com|gov|mil|net|org)|a(?:ngrila|rp|w)|i(?:ksh)?a|riram|ell)?|o(?:\.(?:com|net|org)|ft(?:bank|ware)|l(?:utions|ar)|c(?:cer|ial)|n[gy]|hu|y)?|u(?:p(?:p(?:l(?:ies|y)|ort)|ersport)|r(?:gery|f)|zuki|cks)?|y(?:\.(?:com|edu|gov|mil|net|org)|mantec|stems|dney)?|k(?:y(?:(?:driv|p)e)?|\.blogspot|olkovo|ate|in?)?|n(?:\.(?:(?:gou|uni)v|perso|art|com|edu|org)|cf)?|g(?:\.(?:(?:blogspo|ne)t|com|edu|gov|org|per))?|d(?:\.(?:(?:go|t)v|info|com|edu|med|net|org))?|p(?:o(?:rts?|t)|readbetting|a(?:ce)?|iegel)|b(?:[is]|\.(?:com|edu|gov|net|org))?|l(?:\.(?:com|edu|gov|net|org)|ing)?|i(?:n(?:gles|a)|lk|te|c)?|w(?:i(?:ftcover|ss)|atch)|z(?:\.(?:org|ac|co))?|v(?:\.[^\.]+|r)?|m(?:art|ile)?|x(?:\.gov)?|r[lt]?|f[or])|x(?:n--(?:m(?:gb(?:a(?:3a(?:4f(?:16|r)a|3ejt)|(?:7c0bbn0|yh7gp)a|a(?:kc7dvf|m7a8h)|b2bd)|qly7c(?:0a67fbc|vafr)|c(?:0a9azcg|a7dzdo)|b(?:9fbpob|h1a71e)|erp4a5d4a(?:87g|r)|t(?:3dhd|f8fl)|i4ecexp|v6cfpo|2ddes|9awbf)|k1bu44c)|f(?:i(?:q(?:228c5h|s8|z9)s|Q64B)|z(?:ys8d69uvgm|c2c9e2c)|ct429k|jq720a|lw351e|pcrj9c|hbei)|c(?:ck(?:wcxetd|2b3b)|lchc0ea0b2g2a9gcd|1(?:yn36f|avg)|zr(?:694b|u2d)|2br7g)|5(?:5q(?:x5d(?:8y0buji4b870u)?|w42g)|(?:su34j936bgs|tzm5)g|4b7fta0cc)|j(?:6w(?:470d71issc|193g)|lq(?:480n2rg|61u9w7b)|1a(?:ef|mh)|vr189m)|k(?:crx7(?:bb75ajk3b|7d1x4a)|p(?:r(?:w13|y57)d|u716f))|h(?:xt(?:035c(?:mppue|zzpff)l|814e)|db9cza1b|2brj9c)|(?:v(?:ermgensberat(?:ung-pw|er-ct)|uq861)|zfr164)b|n(?:gb(?:c5azd|e9e0a|rx)|(?:nx388|yqy26)a|qv7f|ode)|8(?:0a(?:s(?:ehdb|wg)|qecdr1a|dxhks)|y0a063a)|p(?:gb(?:3ceoj|s0dh)|1a(?:cf|i)|bt977c|ssy2u)|w(?:4r(?:85el8fhu5dnra|s40l)|gb(?:h1c|l6a))|t(?:(?:qq33ed31aqi|60b56)a|iq49xqyj|ckwe)|3(?:e0b707e|bst00m|ds443g|0rr7y|pxu8k)|x(?:kc2(?:dl3a5ee0h|al3hye2a)|hq521b)|g(?:(?:2xx48|ecrj9)c|ckr3f0f|k3at1e)|4(?:5(?:brj9|q11)c|2c2d9a|gbrim)|i(?:1b6b1a6a2e|mr513n|o0a7i)|b(?:ck1b9a5dre4c|4w605ferd)|e(?:ckvdtc9d|fvy88h|stv75g)|o(?:gbpf8fl|tu796d|3cw4h)|9(?:krt00a|0a3ac|et52u)|y(?:fro4i67o|gbi2ammx)|6(?:qq986b3xl|frz82g)|d(?:kwm73cwpn|1acj3b)|1(?:1b4c3d|ck2e1b)|q(?:cka1pm|9jyb4)c|r(?:hqv96g|ovu88b)|s(?:9brj9c|es554g)|lgbbat1ad8j)|(?:(?:er|b)o|x)x|i(?:hua)?n|finity|peria|yz)|b(?:r(?:\.(?:c(?:o(?:m(?:\.blogspot)?|op)|n[gt]|im)|e(?:[ms]p|c[no]|t[ci]|du|ng)|a(?:d[mv]|r[qt]|gr|to|m)|f(?:[os]t|log|ar|nd|m)|t(?:axi|eo|mp|rd|ur|v)|m(?:at|ed|il|us)|n(?:o[mt]|et|tr)|p(?:s[ci]|pg|ro)|b(?:log|io|md)?|g(?:12|gf|ov)|i(?:n[df]|mb)|r(?:adio|ec)|v(?:log|et)|j(?:or|us)|o(?:do|rg)|s(?:lg|rv)|le[gl]|wiki|qsl|zlg)|o(?:(?:th|k)er|adway)|idgestone|adesco|ussels)?|a(?:\.(?:un(?:bi|sa)|com?|edu|gov|mil|net|org|rs)|r(?:c(?:lay(?:card|s)|elona)|efoot|gains)?|n(?:[dk]|a(?:narepublic|mex)|que)|s(?:ket|e)ball|uhaus|yern|idu|by)?|o(?:[mtx]|\.(?:(?:in|ne)t|go[bv]|com|edu|mil|org|tv)|o(?:k(?:ing)?|ts)?|s(?:t(?:ik|on)|ch)|a(?:rd|ts)|ehringer|utique|fa|nd)?|i(?:[do]|z(?:\.(?:for-(?:(?:mor|som|th)e|better)|(?:selfi|webho)p|dyndns))?|\.(?:com?|org?|edu)|(?:bl|k)e|ngo?)?|e(?:\.(?:blogspot|ac)|(?:know|rli)n|a(?:uty|ts)|st(?:buy)?|ntley|er|t)?|l(?:o(?:om(?:ingdales|berg)|ckbuster|g)|a(?:ck(?:friday)?|nco)|ue)|b(?:[bct]|\.(?:store|info|biz|com|edu|gov|net|org)|va)?|u(?:i(?:ld(?:ers)?|ck)|dapest|siness|gatti|zz|y)|j(?:\.(?:b(?:logspot|arreau)|asso|gouv))?|g(?:\.[\da-z])?|h(?:\.(?:com|edu|gov|net|org)|arti)?|m(?:[sw]|\.(?:com|edu|gov|net|org))?|z(?:\.(?:com|edu|gov|net|org)|h)?|s(?:\.(?:com|edu|gov|net|org))?|t(?:\.(?:com|edu|gov|net|org))?|y(?:\.(?:com|gov|mil|of))?|n(?:\.[^\.]+|pparibas|l)?|w(?:\.(?:org|co)|ay)?|d(?:\.[^\.]+)?|f(?:\.gov)?|c[gn])|t(?:[fg]|r(?:a(?:vel(?:ers(?:insurance)?|channel|guard)?|ns(?:(?:former|lation)s|union)|d(?:e(?:rshotels)?|ing)|ining)|\.(?:(?!nic$)[^\.]+|nc\.gov)|ust|v)?|n(?:\.(?:(?:agrine|n[ae])t|t(?:ourism|uren)|e(?:dunet|ns)|in(?:fo|tl|d)|rn(?:[su]|rt)|(?:min)?com|defense|perso|fin|gov|org))?|w(?:\.(?:xn--(?:zf0ao64a|czrw28b|uc0atv)|(?:blogspo|ne)t|c(?:lub|om)|e(?:biz|du)|g(?:ame|ov)|idv|mil|org))?|t(?:\.(?:m(?:useum|obi)|co(?:op|m)?|n(?:ame|et)|(?:ae|p)ro|in(?:fo|t)|travel|jobs|biz|edu|gov|org))?|j(?:\.(?:n(?:ame|et|ic)|(?:tes|in)t|com?|gov?|biz|edu|mil|org|web|ac)|(?:max)?x)?|o(?:\.(?:com|edu|gov|mil|net|org)|y(?:ota|s)|[dr]ay|shiba|urs?|kyo|ols|tal|wn|p)?|h(?:e(?:at(?:er|re)|guardian|hartford)|\.(?:[cg]o|net|ac|in|mi|or)|ai|d)?|e(?:l(?:e(?:fonica|city))?|ch(?:nology)?|(?:rr|v)a|masek|nnis|am)|z\.(?:(?:[cg]|inf)o|m(?:obi|il|e)|[as]c|hotel|ne|or|tv)|v(?:\.(?:(?:better|worse)-than|on-the-web|dyndns)|s)?|a(?:t(?:a(?:motors|r)?|too)|ipei|obao|rget|xi?|lk|b)|m(?:\.(?:n(?:et|om)|com?|edu|gov|mil|org)|all)?|i(?:(?:cket|p)s|(?:end|a)a|r(?:es|ol)|ffany)|d(?:\.blogspot|k)?|u(?:nes|shu|be|i)|l(?:\.gov)?|k(?:maxx)?|ci?)|h(?:u(?:\.(?:(?:(?:utaz|lak)a|game|new)s|(?:f(?:oru|il)|rekla)m|s(?:port|z?ex|hop|uli)|(?:eroti[ck]|medi)a|c(?:(?:asin)?o|ity)|(?:konyvel|vide)o|b(?:logspo|ol)t|in(?:gatlan|fo)|t(?:ozsde|m)|jogasz|agrar|hotel|2000|priv|org)|ghes)?|k(?:\.(?:xn--(?:(?:(?:lcvr3|wcvs2)2|55qx5)d|m(?:k0axi|xtq1m)|gmq(?:050i|w5a)|od0a(?:q3b|lg)|uc0a(?:y4a|tv)|io0a7i|zf0avx|ciqpn|tn0ag)|(?:blogspo|ne)t|(?:go|id)v|com|edu|org)|t)?|o(?:me(?:s(?:ense)?|depot|goods)?|t(?:e(?:l(?:e?s)?|is)|mail)?|s(?:t(?:ing)?|pital)|l(?:dings|iday)|n(?:eywell|da)|[ru]se|ckey|w)|t(?:\.(?:a(?:(?:dul|r)t|sso)|p(?:(?:ers|r)o|ol)|co(?:op|m)|firm|gouv|info|shop|edu|med|net|org|rel)|c)?|e(?:a(?:lth(?:care)?|rt)|l(?:sinki|p)|r(?:mes|e)|inz)|n(?:\.(?:com|edu|gob|mil|net|org))?|i(?:samitsu|tachi|lton|phop|v)|r(?:\.(?:(?:fr|c)om|name|iz))?|a(?:mburg|ngout|lal|ir|us)|y(?:undai|att)|dfc(?:bank)?|gtv|sbc|bo|m)|o(?:r(?:g(?:\.(?:is(?:-(?:a-(?:c(?:elticsfan|andidate|hef)|(?:(?:bruin|pat)s|sox)fan|linux-user|knight|geek)|very-(?:(?:goo|ba)d|sweet|evil|nice)|(?:foun|save)d|lost)|a-geek)|d(?:yn(?:dns(?:\.(?:home|go))?|alias)|o(?:(?:es)?ntexist|omdns)|ns(?:alias|dojo)|vrdns)|s(?:e(?:l(?:lsyourhome|fip)|rve(?:game|bbs|ftp))|tuff-4-sale)|ho(?:me(?:(?:linu|uni)x|dns|ftp)|bby-site)|b(?:oldlygoingnowhere|log(?:site|dns))|endof(?:the)?internet|(?:from-m|podzon|a)e|g(?:ame-host|otdns)|(?:kicks-as|u)s|misconfused|readmyblog|webhop|za)|anic)?|i(?:entexpres|gin)s|a(?:cl|ng)e)|m(?:\.(?!mediaphone$)(?!nawrastelecom$)(?!nawras$)(?!omanmobile$)(?!omanpost$)(?!omantel$)(?!rakpetroleum$)(?!siemens$)(?!songfest$)(?!statecouncil$)[^\.]+|ega)?|l(?:ayan(?:group)?|dnavy|ympus|lo)|n(?:(?:yoursid)?e|l(?:ine)?|g)|v(?:erheidnl|h)|(?:kinaw|sak)a|b(?:server|i)|t(?:suka|t)|ff(?:ice)?|pen|oo)|k(?:r(?:\.(?:g(?:yeong(?:buk|nam|gi)|angwon|wangju|o)|c(?:hung(?:buk|nam)|o)|je(?:on(?:buk|nam)|ju)|b(?:logspot|usan)|(?:incheo|ulsa)n|dae(?:jeon|gu)|s(?:eoul|c)|m(?:il|s)|[npr]e|[eh]s|ac|kg|or)|e?d)?|m(?:\.(?:p(?:r(?:esse|d)|harmaciens)|m(?:edecin|il)|no(?:taires|m)|veterinaire|co(?:op|m)|asso?|gou?v|edu|org|tm))?|i(?:[am]|\.(?:info|biz|com|edu|gov|net|org)|nd(?:er|le)|tchen|ehls|ds?|wi)?|e(?:r(?:ry(?:logisitic|propertie|hotel)s|astase)?|\.[^\.]+|tchup)?|y(?:\.(?:com|edu|gov|net|org)|knet|oto)?|p(?:\.(?:com|edu|gov|org|rep|tra)|mg|n)|g(?:\.(?:com|edu|gov|mil|net|org))?|z(?:\.(?:com|edu|gov|mil|net|org))?|o(?:n(?:ami|e)|matsu|sher|eln)|n(?:\.(?:edu|gov|net|org))?|h(?:\.[^\.]+)?|w(?:\.[^\.]+)?|uokgroup|aufen|ddi|fh|sb)|l(?:a(?:\.(?:in(?:fo|t)|c(?:om)?|edu|gov|net|org|per)|n(?:c(?:aster|ome|ia)|d(?:rover)?|xess)|m(?:borghini|er)|t(?:robe|ino)?|w(?:yer)?|dbrokes|caixa|salle)?|i(?:fe(?:(?:insuranc|styl)e)?|v(?:e(?:strong)?|ing)|n(?:coln|de|k)|m(?:ited|o)|(?:ll|ps)y|(?:xi|d)l|ghting|aison|ke)?|k(?:\.(?:g(?:ov|rp)|n(?:et|go)|s(?:ch|oc)|hotel|assn|com|edu|int|ltd|org|web))?|o(?:c(?:ker|us)|(?:rea)?l|tt[eo]|ans?|ndon|ft|ve)|v(?:\.(?:co(?:nf|m)|asn|edu|gov|mil|net|org|id))?|y(?:\.(?:(?:me|i)d|com|edu|gov|net|org|plc|sch))?|e(?:g(?:al|o)|clerc|frak|ase|xus)|c(?:\.(?:com?|edu|gov|net|org))?|u(?:x(?:ury|e)|ndbeck|pin)?|b\.(?:com|edu|gov|net|org)|r\.(?:com|edu|gov|net|org)|s(?:\.(?:org|co))?|p(?:lfinancia)?l|t(?:\.gov|da?)?|l[cp]|gbt|ds)|f(?:r(?:\.(?:c(?:h(?:irurgiens-dentistes|ambagri)|ci|om)|a(?:vo(?:cat|ues)|ss(?:edic|o)|eroport)|(?:huissier-justic|veterinair)e|p(?:r(?:esse|d)|harmacien|ort)|g(?:eometre-expert|reta|ouv)|experts-comptables|no(?:taires|m)|blogspot|medecin|tm)|o(?:nt(?:doo|ie)r|gans)|e(?:senius|e)|l)?|i(?:n(?:a(?:nc(?:ial(?:aid)?|e)|l)|ish)|\.(?:blogspot|aland|iki)|r(?:(?:eston)?|mdal)e|d(?:elity|o)|sh(?:ing)?|t(?:ness)?|at|lm)?|a(?:i(?:rwinds|th|l)|s(?:hion|t)|rm(?:ers)?|m(?:ily)?|ns?|ge)|o(?:o(?:d(?:network)?|tball)?|r(?:sale|ex|um|d)|undation|x)?|l(?:o(?:ri(?:da|st)|wers)|i(?:(?:ck)?r|ghts)|s(?:midth)?|y)|u(?:ji(?:xerox|tsu)|rniture|tbol|nd?)|e(?:rr(?:ari|ero)|edback|dex)|j(?:\.[^\.]+)?|k(?:\.[^\.]+)?|tr|yi|m)|g(?:[fqsw]|r(?:\.(?:(?:blogspo|ne)t|com|edu|gov|org)|a(?:(?:phic|ti)s|inger)|o(?:cery|up)|een?|ipe)?|e(?:[ad]|\.(?:(?:ne|pv)t|com|edu|gov|mil|org)|nt(?:ing)?|company|orge)?|o(?:[pv]|o(?:d(?:hands|year)|g(?:le)?)?|l(?:d(?:point)?|f)|daddy|tv?)|u(?:ardian(?:media|life)?|i(?:tars|de)|\.[^\.]+|cci|ge|ru)?|i(?:\.(?:(?:lt|mo)d|com|edu|gov|org)|v(?:ing|es)|fts?)?|a(?:[py]|l(?:l(?:ery|up|o))?|r(?:nier|den)|mes?)?|t(?:\.(?:com|edu|gob|ind|mil|net|org))?|l(?:ob(?:alx?|o)|a(?:de|ss)|e(?:an)?)?|p(?:\.(?:asso|mobi|com|edu|net|org))?|g(?:\.(?:gov|net|org|sch|co)|ee)?|h(?:\.(?:com|edu|gov|mil|org))?|n\.(?:com|edu|gov|net|org|ac)|y(?:\.(?:com?|net))?|m(?:[cox]|ail|bh)?|biz|dn?|cc)|d(?:e(?:\.(?:le(?:btimnetz|itungsen)|ist(?:eingeek|mein)|fuettertdasnetz|traeumtgerade|blogspot|com)|l(?:(?:mon|oit)te|ivery|ta|l)|(?:utschepos|mocra)t|nt(?:ist|al)|al(?:er|s)?|si(?:gn)?|gree|v)?|o(?:\.(?:(?:ar|ne)t|go[bv]|com|edu|mil|org|sld|web)|c(?:omo|tor|s)|t(?:africa)?|wnload|mains|osan|dge|ha|g)?|i(?:s(?:co(?:unt|ver)|h)|gi(?:key|tal)|rect(?:ory)?|amonds|et|y)|z(?:\.(?:a(?:sso|rt)|com|edu|gov|net|org|pol))?|a(?:[dy]|t(?:[ae]|ing|sun)|bur|nce)|u(?:n(?:lop|s)|pont|rban|bai|ck)|m(?:\.(?:com|edu|gov|net|org))?|k(?:\.blogspot)?|v(?:ag|r)|n[bp]|rive|s?tv|clk|ds|hl|wg|j)|e(?:c(?:\.(?:(?:inf|pr)o|m(?:ed|il)|go[bv]|com|edu|fin|k12|net|org)|om?)?|s(?:\.(?:com(?:\.blogspot)?|edu|gob|nom|org)|t(?:ate)?|urance|q)?|g(?:\.(?:n(?:ame|et)|e(?:du|un)|com|gov|mil|org|sci))?|e(?:\.(?:riik|aip|com|edu|fie|gov|lib|med|org|pri))?|x(?:(?:traspac|chang)e|p(?:osed|ress|ert))|n(?:g(?:ineer(?:ing)?)?|terprises|ergy)|r(?:\.[^\.]+|icsson|ni)?|d(?:u(?:cation)?|eka)|m(?:er(?:son|ck)|ail)|t(?:\.[^\.]+|isalat)?|u(?:rovision|s)?|ve(?:rbank|nts)|p(?:ost|son)|a(?:rth|t)|quipment)|v(?:i(?:s(?:ta(?:print)?|ion|a)|\.(?:com?|k12|net|org)|(?:aje|lla)s|(?:kin)?g|v[ao]|rgin|deo|no?|p)?|e(?:\.(?:e(?:12|du)|com?|info|gov|mil|net|org|web)|r(?:sicherung|isign)|n(?:tures)?|gas|t)?|n(?:\.(?:n(?:ame|et)|in(?:fo|t)|health|biz|com|edu|gov|org|pro|ac))?|o(?:l(?:kswagen|vo)|t(?:[eo]|ing)|yage|dka|ns)|c(?:\.(?:com|edu|gov|mil|net|org))?|a(?:n(?:guard|ish|a)|cations|l)?|u(?:elos)?|laanderen|g)|w(?:e(?:b(?:s(?:ite)?|cam|jet|er)?|ather(?:channel)?|d(?:ding)?|i(?:bo|r))|i(?:n(?:(?:dow|ner)s|e)?|l(?:liamhill|mar)|en|ki)|a(?:l(?:mart|ter|es)|ng(?:gou)?|tch(?:es)?|rman)|s(?:\.(?:(?:dyndn|mypet)s|com|edu|gov|net|org))?|o(?:lterskluwer|r(?:ks?|ld)|odside|w)|hoswho|t[cf]|me|f)|y(?:o(?:(?:koham|g)a|u(?:tube)?|dobashi)|a(?:maxun|chts|ndex|hoo)|e(?:llowpages|\.[^\.]+)?|un|t)|z(?:a(?:\.[^\.]+|ppos|ra)?|m(?:\.[^\.]+)?|w(?:\.[^\.]+)?|u(?:erich|lu)|ip(?:po)?|ero|one)|q(?:a(?:\.(?:n(?:ame|et)|com|edu|gov|mil|org|sch))?|ue(?:bec|st)|pon|tel|vc))/u
       },
       {},
       {}
     ];
$x; }
