package App::Manoc::Netwalker::Poller::BaseTask;
#ABSTRACT: Base role for poller tasks

use Moose::Role;
our $VERSION = '2.99.4'; ##TRIAL VERSION

requires "nwinfo";


has 'schema' => (
    is       => 'ro',
    isa      => 'Object',
    required => 1
);


has 'config' => (
    is       => 'ro',
    isa      => 'App::Manoc::Netwalker::Config',
    required => 1
);


has 'credentials' => (
    is      => 'ro',
    isa     => 'HashRef',
    lazy    => 1,
    builder => '_build_credentials',
);

sub _build_credentials {
    my $self = shift;

    my $credentials =
        $self->nwinfo->credentials ? $self->nwinfo->credentials->get_credentials_hash : {};
    $credentials->{snmp_community} ||= $self->config->snmp_community;
    $credentials->{snmp_version}   ||= $self->config->snmp_version;

    return $credentials;
}


has 'refresh_interval' => (
    is      => 'ro',
    isa     => 'Int',
    lazy    => 1,
    builder => '_build_refresh_interval',
);

sub _build_refresh_interval {
    shift->config->refresh_interval;
}


has 'timestamp' => (
    is      => 'ro',
    isa     => 'Int',
    default => sub { time },
);


sub reschedule_on_failure {
    my $self   = shift;
    my $nwinfo = $self->nwinfo;

    my $backoff;

    if ( $nwinfo->attempt_backoff ) {
        $backoff = $nwinfo->attempt_backoff * 2;
    }
    else {
        $backoff = $self->config->min_backoff_time;
    }
    $nwinfo->attempt_backoff($backoff);

    my $next_attempt = $self->timestamp + $backoff;
    $nwinfo->scheduled_attempt($next_attempt);
}


sub reschedule_on_success {
    my $self   = shift;
    my $nwinfo = $self->nwinfo;

    $nwinfo->attempt_backoff(0);
    $nwinfo->scheduled_attempt( $self->timestamp + $self->refresh_interval );
}

no Moose::Role;
1;

# Local Variables:
# mode: cperl
# indent-tabs-mode: nil
# cperl-indent-level: 4
# cperl-indent-parens-as-block: t
# End:

__END__

=pod

=head1 NAME

App::Manoc::Netwalker::Poller::BaseTask - Base role for poller tasks

=head1 VERSION

version 2.99.4

=head1 ATTRIBUTES

=head2 schema

The Manoc DB schema.

=head2 config

Netwalker config object

=head2 credentials

Authentication credentials hash.

=head2 refresh_interval

The refresh interval used to compute future scheduling.

=head2 timestamp

UNIX timestamp saying the at which the task was started

=head1 METHODS

=head2 reschedule_on_failure

Update C<nwinfo->scheduled_attempt> field after a failed poller attempt.
Implements backoff, doubling interval extent up to C<config->max_backoff>.

=head2 reschedule_on_success

Update C<nwinfo->scheduled_attempt> field after a successful poller attempt.

=head1 AUTHORS

=over 4

=item *

Gabriele Mambrini <gmambro@cpan.org>

=item *

Enrico Liguori

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Gabriele Mambrini.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
