package Bookings::Security::CVSS::v3;

use 5.006;
use strict;
use warnings;

use Math::Round qw(nearest_ceil);

use Carp qw( croak );

=head1 NAME

Bookings::Security::CVSS::v3 - Calculate CVSSv3 values (Common Vulnerability Scoring System)

=head1 DESCRIPTION

CVSSv3 allows you to calculate all three types of score described
under the CVSS system: Base, Temporal and Environmental.

You can modify any parameter via its setter and recalculate
at any time.

The temporal score depends on the base score, and the environmental
score depends on the temporal score. Therefore you must remember
to supply all necessary parameters.

SetVector allows you to parse a CVSSv3 vector as described at:
https://www.first.org/cvss/specification-document#i6

Vector() will return the CVSS vector as a string.

=head1 POSSIBLE VALUES

For meaning of these values see the official CVSS FAQ
at https://www.first.org/cvss/faq/#c7

=head1 SEE ALSO

This module is based on the formulas supplied at:
http://www.first.org/cvss/

=head1 VERSION

Version 0.01

=cut

our $VERSION = '0.01';


=head1 SYNOPSIS

Quick summary of what the module does.

Perhaps a little code snippet.

    use Bookings::Security::CVSS::v3;

    my $foo = Bookings::Security::CVSS::v3->new();
    ...
=cut

our %BASE_PARAMS = (
    AttackVector          => { Params => {'local' => 0.55, 'adjacent-network' => 0.62, 'network' => 0.85, 'physical' => 0.2},
        P2V    => {'local' => 'L',  'adjacent-network' => 'A',   'network' => 'N', 'physical' => 'P'},
        Abbrev => 'AV',
    },

    AttackComplexity      => { Params => {'low' => 0.77, 'high' => 0.44},
        P2V    => {'low' => 'L',  'high' => 'H'},
        Abbrev => 'AC',
    },

    PrivilegesRequired    => { Params => {'high' => 0.27, 'low' => 0.62, 'none' => 0.85},
        P2V    => {'high' => 'H',  'low' => 'L',  'none' => 'N'},
        Abbrev => 'PR',
    },

    UserInteraction       => { Params => {'required' => 0.62, 'none' => 0.85},
        P2V    => {'required' => 'R',  'none' => 'N'},
        Abbrev => 'UI',
    },

    Scope                 => { Params => {'unchanged' => 0,   'changed' => 1},
        P2V    => {'unchanged' => 'U', 'changed' => 'C'},
        Abbrev => 'S',
    },

    ConfidentialityImpact => { Params => {'none' => 0,   'low' => 0.22, 'high' => 0.56},
        P2V    => {'none' => 'N', 'low' => 'L',  'high' => 'H'},
        Abbrev => 'C',
    },

    IntegrityImpact       => { Params => {'none' => 0,   'low' => 0.22, 'high' => 0.56},
        P2V    => {'none' => 'N', 'low' => 'L',  'high' => 'H'},
        Abbrev => 'I',
    },

    AvailabilityImpact    => { Params => {'none' => 0,   'low' => 0.22, 'high' => 0.56},
        P2V    => {'none' => 'N', 'low' => 'L',  'high' => 'H'},
        Abbrev => 'A',
    },
);
_CreateV2P(\%BASE_PARAMS);

our %TEMPORAL_PARAMS = (
    Exploitability   => { Params => {'not-defined' => 1,   'unproven' => 0.91, 'proof-of-concept' => 0.94, 'functional' => 0.97, 'high' => 1},
        P2V    => {'not-defined' => 'X', 'unproven' => 'U',  'proof-of-concept' => 'P',  'functional' => 'F',  'high' => 'H'},
        Abbrev => 'E',
    },

    RemediationLevel => { Params => {'not-defined' => 1,   'official-fix' => 0.95, 'temporary-fix' => 0.96, 'workaround' => 0.97, 'unavailable' => 1},
        P2V    => {'not-defined' => 'X', 'official-fix' => 'O',  'temporary-fix' => 'T',  'workaround' => 'W',  'unavailable' => 'U'},
        Abbrev => 'RL',
    },

    ReportConfidence => { Params => {'not-defined' => 1,   'unknown' => 0.92, 'reasonable' => 0.96, 'confirmed' => 1},
        P2V    => {'not-defined' => 'X', 'unknown' => 'U',  'reasonable' => 'R',  'confirmed' => 'C'},
        Abbrev => 'RC',
    },
);
_CreateV2P(\%TEMPORAL_PARAMS);

our %ENVIRONMENTAL_PARAMS = (
    ConfidentialityRequirement    => { Params => {'not-defined' => 1,   'low' => 0.5, 'medium' => 1,   'high' => 1.5},
        P2V    => {'not-defined' => 'X', 'low' => 'L', 'medium' => 'M', 'high' => 'H'},
        Abbrev => 'CR',
    },

    IntegrityRequirement          => { Params => {'not-defined' => 1,   'low' => 0.5, 'medium' => 1,   'high' => 1.5},
        P2V    => {'not-defined' => 'X', 'low' => 'L', 'medium' => 'M', 'high' => 'H'},
        Abbrev => 'IR',
    },

    AvailabilityRequirement       => { Params => {'not-defined' => 1,   'low' => 0.5, 'medium' => 1,   'high' => 1.5},
        P2V    => {'not-defined' => 'X', 'low' => 'L', 'medium' => 'M', 'high' => 'H'},
        Abbrev => 'AR',
    },

    ModifiedAttackVector          => { Params => {'not-defined' => -1,  'local' => 0.55, 'adjacent-network' => 0.62, 'network' => 0.85, 'physical' => 0.2},
        P2V    => {'not-defined' => 'X', 'local' => 'L',  'adjacent-network' => 'A',   'network' => 'N', 'physical' => 'P'},
        Abbrev => 'MAV',
    },

    ModifiedAttackComplexity      => { Params => {'not-defined' => -1,  'low' => 0.77, 'high' => 0.44},
        P2V    => {'not-defined' => 'X', 'low' => 'L',  'high' => 'H'},
        Abbrev => 'MAC',
    },

    ModifiedPrivilegesRequired    => { Params => {'not-defined' => -1,  'high' => 0.27, 'low' => 0.62, 'none' => 0.85},
        P2V    => {'not-defined' => 'X', 'high' => 'H',  'low' => 'L',  'none' => 'N'},
        Abbrev => 'MPR',
    },

    ModifiedUserInteraction       => { Params => {'not-defined' => -1,  'required' => 0.62, 'none' => 0.85},
        P2V    => {'not-defined' => 'X', 'required' => 'R',  'none' => 'N'},
        Abbrev => 'MUI',
    },

    ModifiedScope                 => { Params => {'not-defined' => -1,  'unchanged' => 0,   'changed' => 1},
        P2V    => {'not-defined' => 'X', 'unchanged' => 'U', 'changed' => 'C'},
        Abbrev => 'MS',
    },

    ModifiedConfidentialityImpact => { Params => {'not-defined' => -1,  'none' => 0,   'low' => 0.22, 'high' => 0.56},
        P2V    => {'not-defined' => 'X', 'none' => 'N', 'low' => 'L',  'high' => 'H'},
        Abbrev => 'MC',
    },

    ModifiedIntegrityImpact       => { Params => {'not-defined' => -1,  'none' => 0,   'low' => 0.22, 'high' => 0.56},
        P2V    => {'not-defined' => 'X', 'none' => 'N', 'low' => 'L',  'high' => 'H'},
        Abbrev => 'MI',
    },

    ModifiedAvailabilityImpact    => { Params => {'not-defined' => -1,  'none' => 0,   'low' => 0.22, 'high' => 0.56},
        P2V    => {'not-defined' => 'X', 'none' => 'N', 'low' => 'L',  'high' => 'H'},
        Abbrev => 'MA',
    },
);
_CreateV2P(\%ENVIRONMENTAL_PARAMS);

our %ALL_PARAMS = (%BASE_PARAMS, %TEMPORAL_PARAMS, %ENVIRONMENTAL_PARAMS);

# Create getters and setters for all parameters
foreach my $key (keys %ALL_PARAMS) {
    no strict 'refs';

    # Long-name getter.  For example, $self->Exploitability might return 'proof-of-concept'.
    *{"Bookings::Security::CVSS::v3::$key"} = sub {
        my $self = shift;
        return ($self->{$key} // 'not-defined');
    };

    # And create getters for the 'short' values used in the vector.
    # For example, $CVSS->MI might return N.
    my $abbrev = $ALL_PARAMS{$key}->{'Abbrev'};
    *{"Bookings::Security::CVSS::v3::$abbrev"} = sub {
        my $self = shift;
        if ($self->{$key}) {
            return $ALL_PARAMS{$key}->{P2V}->{ $self->{$key} };
        } else {
            return undef;
        }
    };

    # setter
    *{"Bookings::Security::CVSS::v3::Set$key"} = sub {
        my $self = shift;
        $self->_ValidateParam($key, @_);
    };
}

# Create the Vector-to-param hash from the P2V hash
sub _CreateV2P {
    my $params = shift;

    foreach my $param (keys %$params) {
        $params->{$param}->{V2P} = { map { $params->{$param}->{P2V}->{$_} => $_ } keys %{$params->{$param}->{P2V}} };
    }
}

sub _ValidateParam {
    my $self  = shift;
    my $param = shift;
    my $value = shift;

    # If vector value - convert to full value
    if (exists($ALL_PARAMS{$param}->{V2P}->{$value})) {
        $value = $ALL_PARAMS{$param}->{V2P}->{$value};
    } else {
        $value = lc($value);
    }

    if (!grep(/^$value$/i, keys %{$ALL_PARAMS{$param}->{Params}})) {
        croak("Invalid value '$value' for $param");
    }

    $self->{$param} = $value;
}

sub UpdateFromHash {
    my ($self, $params) = @_;

    if (ref($params) ne 'HASH') {
        croak 'Parameter must be a hash reference';
    }

    foreach my $param (keys %$params) {
        if (!exists($ALL_PARAMS{$param})) {
            croak "$param is not a valid parameter";
        }

        my $setter_name = "Set$param";
        $self->$setter_name($params->{$param});
    }
}


=head1 EXPORT
  new
  Vector
  SetVector
  OverallScore
  BaseScore
  TemporalScore
  EnvironmentalScore

=head1 SUBROUTINES/METHODS

=head2 new

=cut

sub new {
    my $class  = shift;
    my $params = shift;

    my $self   = bless({}, $class);

    if (defined($params)) {
        $self->UpdateFromHash($params);
    }
    return $self;
}

=head2 SetVector

=cut
sub SetVector {
    my ($self, $vector) = @_;

    if (defined($vector)) {

        if ($vector !~ m/
            # Base vector
                ^CVSS:3\.0\/AV:([NALP])\/AC:([LH])\/PR:([NLH])\/UI:([NR])\/S:([UC])\/C:([NLH])\/I:([NLH])\/A:([NLH])

                    # Temporal vector
                    (\/E:([XUPFH])\/RL:([XOTWU])\/RC:([XURC]))?

                    # Environmental vector
                    (\/CR:([XLMH])\/IR:([XLMH])\/AR:([XLMH])\/MAV:([XNALP])\/MAC:([XLH])\/MPR:([XNLH])\/MUI:([XNR])\/MS:([XUC])\/MC:([XNLH])\/MI:([XNLH])\/MA:([XNLH]))?

            /x) {
            croak("Invalid CVSS vector $vector");
        }

        my %values = (
            AttackVector          => $1,
            AttackComplexity      => $2,
            PrivilegesRequired    => $3,
            UserInteraction       => $4,
            Scope                 => $5,
            ConfidentialityImpact => $6,
            IntegrityImpact       => $7,
            AvailabilityImpact    => $8,
        );

        # optional temporal portion
        if (defined($9)) {
            %values = (
                %values,
                Exploitability   => $10,
                RemediationLevel => $11,
                ReportConfidence => $12,
            );
        }

        # optional environmental portion.
        if (defined($13)) {
            %values = (
                %values,
                ConfidentialityRequirement    => $14,
                IntegrityRequirement          => $15,
                AvailabilityRequirement       => $16,
                ModifiedAttackVector          => $17,
                ModifiedAttackComplexity      => $18,
                ModifiedPrivilegesRequired    => $19,
                ModifiedUserInteraction       => $20,
                ModifiedScope                 => $21,
                ModifiedConfidentialityImpact => $22,
                ModifiedIntegrityImpact       => $23,
                ModifiedAvailabilityImpact    => $24,
            );
        }

        $self->UpdateFromHash(\%values);

    }
    else {
        croak('Must call SetVector() with a $vector!');
    }
}

=head2 Vector

=cut
sub Vector {
    my ($self) = @_;

    # Check all parameters exist
    foreach my $param (keys %BASE_PARAMS) {
        if (!defined($self->{$param})) {
            croak("You must set '$param' to output the CVSS vector");
        }
    }

    my $vectorValue = sub {
        return $ALL_PARAMS{$_[0]}->{P2V}->{$self->{$_[0]}};
    };

    my $vector = sprintf('AV:%s/AC:%s/PR:%s/UI:%s/S:%s/C:%s/I:%s/A:%s',
        &$vectorValue('AttackVector'),
        &$vectorValue('AttackComplexity'),
        &$vectorValue('PrivilegesRequired'),
        &$vectorValue('UserInteraction'),
        &$vectorValue('Scope'),
        &$vectorValue('ConfidentialityImpact'),
        &$vectorValue('IntegrityImpact'),
        &$vectorValue('AvailabilityImpact'));

    my $temporal = 1;
    foreach my $param (keys %TEMPORAL_PARAMS) {
        if (!defined($self->{$param})) {
            $temporal = 0;
            last;
        }
    }

    if ($temporal) {
        $vector .= sprintf('/E:%s/RL:%s/RC:%s',
            &$vectorValue('Exploitability'),
            &$vectorValue('RemediationLevel'),
            &$vectorValue('ReportConfidence'));
    }

    my $environmental = 1;
    foreach my $param (keys %ENVIRONMENTAL_PARAMS) {
        if (!defined($self->{$param})) {
            $environmental = 0;
            last;
        }
    }

    if ($environmental) {
        $vector .= sprintf('/CR:%s/IR:%s/AR:%s/MAV:%s/MAC:%s/MPR:%s/MUI:%s/MS:%s/MC:%s/MI:%s/MA:%s',
            &$vectorValue('ConfidentialityRequirement'),
            &$vectorValue('IntegrityRequirement'),
            &$vectorValue('AvailabilityRequirement'),
            &$vectorValue('ModifiedAttackVector'),
            &$vectorValue('ModifiedAttackComplexity'),
            &$vectorValue('ModifiedPrivilegesRequired'),
            &$vectorValue('ModifiedUserInteraction'),
            &$vectorValue('ModifiedScope'),
            &$vectorValue('ModifiedConfidentialityImpact'),
            &$vectorValue('ModifiedIntegrityImpact'),
            &$vectorValue('ModifiedAvailabilityImpact'));
    }

    return "CVSS:3.0/$vector";
}

=head2 BaseScore

=cut

sub BaseScore {
    my $self = shift;

    # Check all parameters exist
    foreach my $param (keys %BASE_PARAMS) {
        if (!defined($self->{$param})) {
            return "Not Defined";
        }
    }

    my $vectorValue = sub {
        return $ALL_PARAMS{$_[0]}->{Params}->{$self->{$_[0]}};
    };
    my $vectorKey = sub {
        return $ALL_PARAMS{$_[0]}->{P2V}->{$self->{$_[0]}};
    };

    my $sc = &$vectorValue('Scope');
    my $pr = &$vectorValue('PrivilegesRequired');

    if ($sc) {
        my $prk = &$vectorKey('PrivilegesRequired');
        $pr = 0.68 if $prk eq 'L';
        $pr = 0.50 if $prk eq 'H';
    }

    # Calculate the impact subscore: Spec: Impact = 1 - (1 - ConfImpact)*(1 - IntegImpact) * (1 - AvailImpact)
    my $impact = 1 - (1 - &$vectorValue('ConfidentialityImpact')) * (1 - &$vectorValue('IntegrityImpact')) * (1 - &$vectorValue('AvailabilityImpact'));

    # Calculate the exploitability subscore: Spec: Exploitability = 8.22 * AttackVector * AttackComplexity * PrivilegesRequired * UserInteraction
    my $exploitability = 8.22 * &$vectorValue('AttackVector') * &$vectorValue('AttackComplexity') * $pr * &$vectorValue('UserInteraction');
    my $base = 0;

    # Calculations change based on scope.
    if ($sc == 0) { # unchanged
        $impact = 6.42 * $impact;
        $base = $impact + $exploitability;
    } else {
        $impact = 7.52 * ( $impact - 0.029) - 3.25 * ( ($impact - 0.02) ** 15 );
        $base = 1.08 * ($impact + $exploitability);
    }

    if ($impact < 0) {
        $base = 0;
        $impact = 0;
    } else {
        $base = nearest_ceil(0.1, $base + 0.05);  # Always round up.
        if ($base > 10) {
            $base = 10;
        }
    }

    return $base;
}

=head2 TemporalScore

=cut

sub TemporalScore {
    my $self = shift;

    # Check all parameters exist
    foreach my $param (keys %TEMPORAL_PARAMS) {
        if (!defined($self->{$param})) {
            return "Not Defined";
        }
    }

    my $vectorValue = sub {
        return $ALL_PARAMS{$_[0]}->{Params}->{$self->{$_[0]}};
    };

    my $score = $self->BaseScore() * &$vectorValue('Exploitability') * &$vectorValue('RemediationLevel') * &$vectorValue('ReportConfidence');

    # Round to one sig fig
    return nearest_ceil(0.1, $score + 0.05); # Always round up.
}

=head2 EnvironmentalScore

=cut

sub EnvironmentalScore {
    my $self = shift;

    # Check all parameters exist
    foreach my $param (keys %ENVIRONMENTAL_PARAMS) {
        if (!defined($self->{$param})) {
            return "Not Defined";
        }
    }

    my $vectorValue = sub {
        return $ALL_PARAMS{$_[0]}->{Params}->{$self->{$_[0]}};
    };
    my $vectorKey = sub {
        return $ALL_PARAMS{$_[0]}->{P2V}->{$self->{$_[0]}};
    };

    # If the environment specifies a modified version of the base metric, then use the modified version.
    # Otherwise, use the base version.
    my $ci = (&$vectorValue('ModifiedConfidentialityImpact') != -1) ? &$vectorValue('ModifiedConfidentialityImpact') : &$vectorValue('ConfidentialityImpact');
    my $ii = (&$vectorValue('ModifiedIntegrityImpact') != -1) ? &$vectorValue('ModifiedIntegrityImpact') : &$vectorValue('IntegrityImpact');
    my $ai = (&$vectorValue('ModifiedAvailabilityImpact') != -1) ? &$vectorValue('ModifiedAvailabilityImpact') : &$vectorValue('AvailabilityImpact');

    # Modified impact subscore
    my $mimpact = 1.0 - (1.0 - $ci * &$vectorValue('ConfidentialityRequirement')) *
        (1.0 - $ii * &$vectorValue('IntegrityRequirement')) *
        (1.0 - $ai * &$vectorValue('AvailabilityRequirement'));

    $mimpact = 0.915 if ($mimpact > 0.915);

    my $av = (&$vectorValue('ModifiedAttackVector') != -1) ? &$vectorValue('ModifiedAttackVector') : &$vectorValue('AttackVector');
    my $ac = (&$vectorValue('ModifiedAttackComplexity') != -1) ? &$vectorValue('ModifiedAttackComplexity') : &$vectorValue('AttackComplexity');
    my $ui = (&$vectorValue('ModifiedUserInteraction') != -1) ? &$vectorValue('ModifiedUserInteraction') : &$vectorValue('UserInteraction');

    my $ms = (&$vectorValue('ModifiedScope') != -1) ? &$vectorValue('ModifiedScope') : &$vectorValue('Scope');
    my $pr = (&$vectorValue('ModifiedPrivilegesRequired') != -1) ? &$vectorValue('ModifiedPrivilegesRequired') : &$vectorValue('PrivilegesRequired');

    if ($ms != 0) { # changed
        my $prk = (&$vectorKey('ModifiedPrivilegesRequired') ne 'X') ? &$vectorKey('ModifiedPrivilegesRequired') : &$vectorKey('PrivilegesRequired');
        $pr = 0.68 if $prk eq 'L';
        $pr = 0.50 if $prk eq 'H';
    }

    # Exploitability score
    my $mexploitability = 8.22 * $av * $ac * $pr * $ui;

    my $environment = 0;
    if ($ms == 0) {  # Unchanged
        $mimpact = 6.42 * $mimpact;
        $environment = $mimpact + $mexploitability;
    } else { # changed
        $mimpact = 7.52 * ($mimpact - 0.029) - 3.25 * (($mimpact - 0.02) ** 15);
        $environment = 1.08 * ($mimpact + $mexploitability);
    }

    if ($mimpact < 0) {
        $mimpact = 0;
        $environment = 0;
    } else {
        if ($environment > 10) {
            $environment = 10;
        }
        $environment = nearest_ceil(0.1, $environment + 0.05);  # Always round up.
        my $temporal_product = &$vectorValue('Exploitability') * &$vectorValue('RemediationLevel') * &$vectorValue('ReportConfidence');
        $environment = nearest_ceil(0.1, $environment * $temporal_product + 0.05);
    }

    # Round to one sig fig
    return $environment;  # always round up.
}

=head2 OverallScore

=cut

sub OverallScore {
    my $self = shift;

    my $overall_score = "Not Defined";
    my $base_score = $self->BaseScore;

    if ($base_score eq "Not Defined") {
        return "Not Defined";
    }

    my $environmental_score = $self->EnvironmentalScore;
    if ($environmental_score ne "Not Defined") {
        return $environmental_score;
    }

    my $temporal_score = $self->TemporalScore;
    if ($temporal_score ne "Not Defined") {
        return $temporal_score;
    }
    return $base_score;
}

=head1 AUTHORS

Thomas Briggs, C<< <thomas.briggs at booking.com> >>
Daniel Ostermeier, C<< <daniel.ostermeier at booking.com> >>
Farshid Zaker, C<< <farshid.zaker at booking.com> >>


=head1 BUGS

Please report any bugs or feature requests to C<bug-bookings-security-cvss at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Bookings-Security-CVSS>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.




=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Bookings::Security::CVSS::v3


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Bookings-Security-CVSS>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Bookings-Security-CVSS>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Bookings-Security-CVSS>

=item * Search CPAN

L<http://search.cpan.org/dist/Bookings-Security-CVSS/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2017 booking.com

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut

1; # End of Bookings::Security::CVSS::v3
__END__
