package Prty::JQuery::Form::Inheritance;
use base qw/Prty::Hash/;

use strict;
use warnings;
use utf8;

our $VERSION = 1.081;

use Prty::Storable;
use Prty::Html::Widget::Hidden;
use Prty::Html::Table::Simple;
use Prty::Html::Form::Layout;
use Prty::Html::Widget::Button;
use Prty::Html::Widget::CheckBox;
use Prty::Html::Fragment;

# -----------------------------------------------------------------------------

=encoding utf8

=head1 NAME

Prty::JQuery::Form::Inheritance - HTML-Formular für vererbbare Information

=head1 BASE CLASS

L<Prty::Hash>

=head1 DESCRIPTION

Die Klasse generiert ein Dedicom-Vererbungsformular und verwaltet
den Zustand in Abhängkeit von der (anfänglichen) Datenkonstellation.

Vier Datenkonstellationen sind möglich: Child-Daten yes/no,
Parent-Daten yes/no.

    Child|Parent|Initial|Edit|Buttons
    no|no|Child(empty)|Child(empty)|Speichern
    no|yes|Parent(data)|Parent(data)+Child(empty)|Speichern,Kopieren
    yes|no|Child(data)|Child(data)|Speichern,Löschen
    yes|yes|Child(data)|Parent(data)+Child(data)|Speichern,Löschen

Herbei ist;

=over 4

=item Initial

Das anfänglich dargestellte Formular (leer oder mit Daten).

=item Edit

Das/die dargestellte(n) Formular(e) nach Betätigung von
"bearbeiten" (leer oder mit Daten).

=item Buttons

Die Schaltflächen nach Betätigung von "bearbeiten".

=back

Die mit dem Attribut widgets=>\@widgets übergebenen Widgets erhalten
intern die Namen "parent-NAME" für die Parent-Widgets und "data-NAME"
für die Child-Widgets, wobei nur die Child-Widgets beim Speichern
von Daten von Bedeutung sind.

=head1 ATTRIBUTES

=over 4

=item child => \%hash (Default: undef)

Referenz auf den Hash mit den Werten für das Child-Formular.

=item debug => $bool (Default: 0)

Gib auf STDERR Informationen über die Parent- und Child-Daten aus.

=item deleteAction => $url (Default: undef)

URL, über den die Formulardaten gelöscht werden.

=item head => $html (Default: undef)

Inhalt am Anfang der Seite.

=item hidden => \@keyVal (Default: [])

Schlüssel/Wert-Paare, die als Hidden-Widgets gesetzt werden.

=item id => $id (Default: undef)

DOM-Id des Formulars.

=item layout => $html (Default: [])

Der HTML-Code des Layouts. In das Layout wird der HTML-Code der
Widgets eingesetzt.

=item parent => [$name,\%hash] (Default: undef)

Name des Parent-Mandanteb und Referenz auf den Hash mit den Werten
für das Parent-Formular.

=item saveAction => $url (Default: undef)

URL, über den die Formulardaten gespeichert werden.

=item title => $title (Default: undef)

Abschnittstitel des Formulars.

=item widgets => \@widgets (Default: [])

Liste der Widgets, die in das Layout eingesetzt werden.

=back

=head1 METHODS

=head2 Konstruktor

=head3 new() - Instanziiere Objekt

=head4 Synopsis

    $e = $class->new(@attVal);

=head4 Description

Instanziiere ein Formular-Objekt und liefere eine Referenz auf
dieses Objekt zurück.

=cut

# -----------------------------------------------------------------------------

sub new {
    my $class = shift;
    # @_: @attVal

    my $self = $class->SUPER::new(
        child=>undef,
        debug=>0,
        deleteAction=>undef,
        head=>undef,
        hidden=>[],
        id=>undef,
        layout=>[],
        parent=>[],
        saveAction=>undef,
        title=>undef,
        widgets=>[],
    );
    $self->set(@_);

    return $self;
}

# -----------------------------------------------------------------------------

=head2 Objektmethoden

=head3 html() - Generiere HTML

=head4 Synopsis

    $html = $e->html($h);
    $html = $class->html($h,@attVal);

=head4 Description

Generiere den HTML-Code des Formular-Objekts und liefere diesen
zurück. Als Klassenmethode gerufen, wird das Objekt intern erzeugt
und mit den Attributen @attVal instanziiert.

=cut

# -----------------------------------------------------------------------------

sub html {
    my $this = shift;
    my $h = shift;

    my $self = ref $this? $this: $this->new(@_);

    my ($childH,$debug,$deleteAction,$head,$hiddenA,$id,$layoutA,
        $parentA,$saveAction,$title,$widgetA) = $self->get(qw/child
        debug deleteAction head hidden id layout parent saveAction title
        widgets/);
    
    my ($parentName,$parentH) = @$parentA;
    $parentName ||= 'geerbt';
    
    # Zu den Formularzeilen den Zeilentyp (form-widget) hinzufügen

    my @rows;
    my $columns = 0;
    for (@$layoutA) {
        if (ref $_->[0]) {
            push @rows,['form-widget',@$_];
        }
        else {
            push @rows,[@$_];
        }
        my $n = grep {ref} @$_;
        if ($n > $columns) {
            $columns = $n;
        }
    }

    if ($debug) {
        if ($parentH) {
            warn "PARENT\n";
            Data::Printer::p($parentH);
        }
        if ($childH) {
            warn "CHILD\n";
            Data::Printer::p($childH);
        }
    }
    
    # Parent- und Child-Widgets instanziieren (unabhängig davon,
    # ob Parent- und Child-Daten vorliegen)

    my (@parentWidgets,$parentValues,@childWidgets,$childValues);    
    for (@$widgetA) {
        my $name = $_->name;
    
        my $w = Prty::Storable->clone($_);
        $w->set(disabled=>1); # da disabled, wird nicht übertragen
        $w->set(class=>'disabled');
        if ($parentH) {
            my $val = $parentH->{$name};
            if (defined $val && $val ne '') {
                $parentValues++;
            }
            $w->value($val);
        }
        push @parentWidgets,$w;
    
        $w = Prty::Storable->clone($_);
        $w->name('data-'.$name); # Präfix für Erkennung beim Speichern
        $w->set(disabled=>1);
        $w->set(class=>'disabled');
        if ($childH) {
            my $val = $childH->{$name};
            if (defined $val && $val ne '') {
                $childValues++;
            }
            $w->value($childH->{$name});
        }
        push @childWidgets,$w;
    }
    if (!$parentValues) {
        $parentH = undef;
    }
    if (!$childValues) {
        $childH = undef;
    }
    for (my $i = 0; $i < @$hiddenA; $i += 2) {
        push @childWidgets,Prty::Html::Widget::Hidden->new(
            name=>$hiddenA->[$i],
            value=>$hiddenA->[$i+1],
        );
    }
    
    # Parent-Formular. Wird angezeigt, wenn Parent-Daten und
    # keine Child-Daten vorliegen.

    my $layout = Prty::Html::Table::Simple->html($h,
        class=>'parentTable form',
        # style=>'display: none',
        rows=>[
            ['form-section',
                [colspan=>$columns,
                    $title? "$title ($parentName)": $parentName]],
            @rows,
        ],
    );

    my $lay1 = Prty::Html::Form::Layout->new(
        layout=>$layout,
        widgets=>\@parentWidgets,
    );

    # Child-Formular. Wird angezeigt, wenn Child-Daten vorliegen
    # oder keine Parent-Daten vorliegen.

    $layout = Prty::Html::Table::Simple->html($h,
        class=>'childTable form',
        # style=>'display: none',
        rows=>[
            ['form-section',
                [colspan=>$columns,$title? $title: 'lokal']],
            @rows,
        ],
    );
    
    # Platzhalter-Namen an Namen der Child-Widgets anpassen
    $layout =~ s/__([A-Z0-9_]+?)__/__DATA-$1__/g;

    my $lay2 = Prty::Html::Form::Layout->new(
        layout=>$layout,
        widgets=>\@childWidgets,
    );

    # Button-Zeile
    
    my $but = Prty::Html::Form::Layout->new(
        layout=>Prty::Html::Table::Simple->html($h,
            class=>'form',
            rows=>[
                [['__SAVE__ __COPY__ __DELETE__ __EDIT__']],
            ],
        ),
        widgets=>[
            Prty::Html::Widget::Button->new(
                class=>'saveButton',
                name=>'save',
                content=>'Speichern',
            ),
            Prty::Html::Widget::Button->new(
                class=>'copyButton',
                name=>'copy',
                content=>'Kopieren',
            ),
            Prty::Html::Widget::Button->new(
                class=>'deleteButton',
                name=>'delete',
                content=>'Löschen',
            ),
            Prty::Html::Widget::CheckBox->new(
                class=>'editCheckBox enabled',
                name=>'edit',
                option=>1,
                label=>'bearbeiten',
            ),
        ],
    );

    return Prty::Html::Fragment->html($h,
        placeholders=>[
            __ID__=>$id,
            __TITLE__=>$title,
            __PARENT_DATA__=>$parentH? 'true': 'false',
            __CHILD_DATA__=>$childH? 'true': 'false',
            __SAVE_ACTION__=>$saveAction? "'$saveAction'": 'null',
            __DELETE_ACTION__=>$deleteAction? "'$deleteAction'": 'null',
        ],
        html=>$h->cat(
            $head,
            $h->tag('form',
                id=>$id,
                Prty::Html::Table::Simple->html($h,
                    rows=>[
                        [[$lay1->html($h)],[],[$lay2->html($h)]],
                        [[colspan=>3,$but->html($h)]],
                    ],
                ),
            ),
        ),
        javaScript=>q~
            $.widget('prty.inheritForm',{
                options: {
                    parentData: false,
                    childData: false,
                    saveAction: null,
                    deleteAction: null,
                },
                _create: function () {
                    this.element.addClass('inheritForm');

                    this._on(this.element,{
                        'click .editCheckBox': function (event) {
                            this.__render();
                        },
                        'click .saveButton': function (event) {
                            this.__save();
                        },
                        'click .copyButton': function (event) {
                            this.__copy();
                        },
                        'click .deleteButton': function (event) {
                            this.__delete();
                        },
                    });

                    this.__render()
                },
                __save: function () {
                    // Prüfe, ob das Formular Daten enthält

                    var dataFound = 0;
                    $('.childTable :input',this.element).each(function () {
                        var name = this.name;
                        var val = $(this).val();
                        if (val != '') {
                            dataFound++;
                        }
                    });
                    if (!dataFound) {
                        alert('Bitte füllen Sie das Formular aus');
                        return;
                    }
    
                    var instance = this;
                    var url = this.options.saveAction;
                    var data = $(':input',this.element).serialize();

                    $.ajax({
                        url: url,
                        type: 'POST',
                        data: data,
                        async: false,
                        success: function () {
                            instance.options.childData = true;
                            $('.editCheckBox',instance.element)
                                .prop('checked',false);
                            instance.__render();
                        },
                        error: function () {
                            alert('FEHLER: Speichern fehlgeschlagen');
                        },
                    });
                },
                __clear: function () {
                    // Child-Formular Feldinhalte löschen
                    $('.childTable :input',this.element).each(function () {
                        var $this = $(this);
                        if ($this.is(':radio,:checkbox')) {
                            $this.prop('checked',false);
                        }
                        else {
                            $this.val('');
                        }
                    });
                },
                __copy: function () {
                    var $parentInputs = $('.parentTable :input',this.element);
                    var $childInputs = $('.childTable :input',this.element);
                    for (var i = 0; i < $parentInputs.length; i++) {
                        var $parentInput = $($parentInputs.get(i));
                        var $childInput = $($childInputs.get(i));
                        if ($parentInput.is(':radio,:checkbox')) {
                            $childInput.prop('checked',
                                $parentInput.prop('checked'));
                        }
                        else {
                            $childInput.val($parentInput.val());
                        }
                    }
                    this.__enableButton('.deleteButton');
                },
                __delete: function () {
                    if (!this.options.childData) {
                        this.__clear();
                        this.__disableButton('.deleteButton');
                    }
                    else if (confirm('"'+this.options.title+'"'
                            +' des Mandanten wirklich löschen?')) {
                        var instance = this;
                        var url = this.options.deleteAction;
                        var data = $(':input',this.element).serialize();

                        $.ajax({
                            url: url,
                            type: 'POST',
                            data: data,
                            async: false,
                            success: function () {
                                instance.options.childData = false;
                                $('.editCheckBox',instance.element)
                                    .prop('checked',false);
                                instance.__render();
                                instance.__clear();
                            },
                            error: function () {
                                alert('FEHLER: Speichern fehlgeschlagen');
                            },
                        });
                    }
                },
                __enableButton: function (selector) {
                    var $button = $(selector,this.element);
                    $button.prop('disabled',false);
                    $button.removeClass('disabled');
                    $button.addClass('enabled');
                },
                __disableButton: function (selector) {
                    var $button = $(selector,this.element);
                    $button.prop('disabled',true);
                    $button.removeClass('enabled');
                    $button.addClass('disabled');
                },
                __render: function () {
                    var $parentTable = $('.parentTable',this.element);
                    var $childTable = $('.childTable',this.element);
                    var $childWidgets = $(':input',$childTable);
                    var edit = $('.editCheckBox',this.element).is(':checked');
    
                    if (edit) {
                        if (this.options.parentData) {
                            $parentTable.show();
                        }
                        $childTable.show();

                        // Child-Widgets enablen
    
                        $childWidgets.prop('disabled',false);
                        $childWidgets.removeClass('disabled');
                        $childWidgets.addClass('enabled');

                        // Buttons enablen

                        if (this.options.saveAction) {
                            this.__enableButton('.saveButton');
                        }

                        if (!this.options.childData) {
                            if (this.options.parentData) {
                                this.__enableButton('.copyButton');
                            }
                        }
                        else if (this.options.deleteAction) {
                            this.__enableButton('.deleteButton');
                        }
                    }
                    else { // view
                        if (this.options.parentData
                                && !this.options.childData) {
                            $parentTable.show();
                            $childTable.hide();
                        }
                        else {
                            $parentTable.hide();
                            $childTable.show();
                        }

                        // Child-Widgets disablen
    
                        $childWidgets.prop('disabled',true);
                        $childWidgets.removeClass('enabled');
                        $childWidgets.addClass('disabled');

                        // Buttons disablen
    
                        this.__disableButton('.saveButton');
                        this.__disableButton('.copyButton');
                        this.__disableButton('.deleteButton');
                    }
                }
            });

            $('#__ID__').inheritForm({
                title: '__TITLE__',
                parentData: __PARENT_DATA__,
                childData: __CHILD_DATA__,
                saveAction: __SAVE_ACTION__,
                deleteAction: __DELETE_ACTION__,
            });
        ~
    );
}

# -----------------------------------------------------------------------------

=head1 VERSION

1.081

=head1 AUTHOR

Frank Seitz, L<http://fseitz.de/>

=head1 COPYRIGHT

Copyright (C) 2016 Frank Seitz

=head1 LICENSE

This code is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut

# -----------------------------------------------------------------------------

1;

# eof
