package WWW::Shorten::TinyURL;

use 5.006;
use strict;
use warnings;
use Carp ();

use base qw( WWW::Shorten::generic Exporter );
our $_error_message = '';
our @EXPORT         = qw( makeashorterlink makealongerlink );
our $VERSION        = '3.092';
$VERSION = eval $VERSION;

sub makeashorterlink {
    my $url = shift or Carp::croak('No URL passed to makeashorterlink');
    $_error_message = '';
    my $ua      = __PACKAGE__->ua();
    my $tinyurl = 'http://tinyurl.com/api-create.php';
    my $resp
        = $ua->post($tinyurl, [url => $url, source => "PerlAPI-$VERSION",]);
    return undef unless $resp->is_success;
    my $content = $resp->content;
    if ($content =~ /Error/) {

        if ($content =~ /<html/) {
            $_error_message = 'Error is a html page';
        }
        elsif (length($content) > 100) {
            $_error_message = substr($content, 0, 100);
        }
        else {
            $_error_message = $content;
        }
        return undef;
    }
    if ($resp->content =~ m!(\Qhttp://tinyurl.com/\E\w+)!x) {
        return $1;
    }
    return;
}

sub makealongerlink {
    my $tinyurl_url = shift
        or Carp::croak('No TinyURL key / URL passed to makealongerlink');
    $_error_message = '';
    my $ua = __PACKAGE__->ua();

    $tinyurl_url = "http://tinyurl.com/$tinyurl_url"
        unless $tinyurl_url =~ m!^http://!i;

    my $resp = $ua->get($tinyurl_url);

    unless ($resp->is_redirect) {
        my $content = $resp->content;
        if ($content =~ /Error/) {
            if ($content =~ /<html/) {
                $_error_message = 'Error is a html page';
            }
            elsif (length($content) > 100) {
                $_error_message = substr($content, 0, 100);
            }
            else {
                $_error_message = $content;
            }
        }
        else {
            $_error_message = 'Unknown error';
        }

        return undef;
    }
    my $url = $resp->header('Location');
    return $url;
}

1;

=head1 NAME

WWW::Shorten::TinyURL - Perl interface to L<http://tinyurl.com>

=head1 SYNOPSIS

  use strict;
  use warnings;

  use WWW::Shorten::TinyURL;
  use WWW::Shorten 'TinyURL';

  my $short_url = makeashorterlink('http://www.foo.com/some/long/url');
  my $long_url  = makealongerlink($short_url);

=head1 DESCRIPTION

A Perl interface to the web site L<http://tinyurl.com>.  The service simply maintains
a database of long URLs, each of which has a unique identifier.

=head1 Functions

=head2 makeashorterlink

The function C<makeashorterlink> will call the L<http://TinyURL.com> web site passing
it your long URL and will return the shorter version.

=head2 makealongerlink

The function C<makealongerlink> does the reverse. C<makealongerlink>
will accept as an argument either the full URL or just the identifier.

If anything goes wrong, then either function will return C<undef>.

=head2 EXPORT

makeashorterlink, makealongerlink

=head1 SUPPORT, LICENSE, THANKS and SUCH

See the main L<WWW::Shorten> docs.

=head1 AUTHOR

Iain Truskett <spoon@cpan.org>

=head1 SEE ALSO

L<WWW::Shorten>, L<http://tinyurl.com/>

=cut
