package Data::Object::Autobox;

use 5.014;

use strict;
use warnings;

use base 'autobox';

require Data::Object::Autobox::Array;
require Data::Object::Autobox::Code;
require Data::Object::Autobox::Float;
require Data::Object::Autobox::Hash;
require Data::Object::Autobox::Integer;
require Data::Object::Autobox::Number;
require Data::Object::Autobox::Scalar;
require Data::Object::Autobox::String;
require Data::Object::Autobox::Undef;
require Data::Object::Autobox::Any;

our $VERSION = '1.00'; # VERSION

sub import {
  my ($class) = @_;

  $class->SUPER::import(
    ARRAY     => 'Data::Object::Autobox::Array',
    CODE      => 'Data::Object::Autobox::Code',
    FLOAT     => 'Data::Object::Autobox::Float',
    HASH      => 'Data::Object::Autobox::Hash',
    INTEGER   => 'Data::Object::Autobox::Integer',
    NUMBER    => 'Data::Object::Autobox::Number',
    SCALAR    => 'Data::Object::Autobox::Scalar',
    STRING    => 'Data::Object::Autobox::String',
    UNDEF     => 'Data::Object::Autobox::Undef',
    UNIVERSAL => 'Data::Object::Autobox::Any'
  );

  return;
}

1;

=encoding utf8

=head1 NAME

Data::Object::Autobox

=cut

=head1 ABSTRACT

Data-Object Autoboxing

=cut

=head1 SYNOPSIS

  use Data::Object::Autobox;

  my $input  = [1,1,1,1,3,3,2,1,5,6,7,8,9];
  my $output = $input->grep(sub{$_[0] < 5})->unique->sort; # [1,2,3]
  my $object = $output->array;

  $output->join(',')->print; # 1,2,3
  $object->join(',')->print; # 1,2,3

  $object->isa('Data::Object::Array');

=cut

=head1 DESCRIPTION

This package implements autoboxing via L<autobox> to provide
L<boxing|http://en.wikipedia.org/wiki/Object_type_(object-oriented_programming)>
for native Perl 5 data types. This functionality is provided by L<Data::Object>
which provides a collection of object classes for handling SCALAR, ARRAY, HASH,
CODE, INTEGER, FLOAT, NUMBER, STRING, UNDEF, and other data type operations.

Data::Object provides its own boxing strategy in that every method call which
would normally return a native data type will return a data type object, but
this functionality requires an initial data type object. Data::Object::Autobox
makes it so that you do not need to explicitly create the initial data type
object, and once the initial autobox method call is made, the Data::Object
boxing takes over.

=cut
