package Data::Object;

use 5.014;

use strict;
use warnings;

use parent 'Data::Object::Config';

our $VERSION = '1.00'; # VERSION

# METHODS

sub new {
  my ($class, $name) = @_;

  die "Invalid argument" unless ($name || '') =~ /^[a-zA-Z]\w*/;

  require Data::Object::Space;

  return Data::Object::Space->new(join '::', __PACKAGE__, $name);
}

sub any {
  my ($class, $data) = @_;

  return $class->new('Any')->build($data);
}

sub array {
  my ($class, $data) = @_;

  return $class->new('Array')->build($data);
}

sub code {
  my ($class, $data) = @_;

  return $class->new('Code')->build($data);
}

sub exception {
  my ($class, $data) = @_;

  return $class->new('Exception')->build($data);
}

sub float {
  my ($class, $data) = @_;

  return $class->new('Float')->build($data);
}

sub hash {
  my ($class, $data) = @_;

  return $class->new('Hash')->build($data);
}

sub integer {
  my ($class, $data) = @_;

  return $class->new('Integer')->build($data);
}

sub number {
  my ($class, $data) = @_;

  return $class->new('Number')->build($data);
}

sub regexp {
  my ($class, $data) = @_;

  return $class->new('Regexp')->build($data);
}

sub scalar {
  my ($class, $data) = @_;

  return $class->new('Scalar')->build($data);
}

sub string {
  my ($class, $data) = @_;

  return $class->new('String')->build($data);
}

sub undef {
  my ($class, $data) = @_;

  return $class->new('Undef')->build($data);
}

1;

=encoding utf8

=head1 NAME

Data::Object

=cut

=head1 ABSTRACT

Modern Perl Development Framework and Standard Library

=cut

=head1 SYNOPSIS

  package User;

  use Data::Object 'Class';

  extends 'Identity';

  has 'fname';
  has 'lname';

  1;

=cut

=head1 DESCRIPTION

The Data::Object package and framework aims to provide a standardized and
cohesive set of classes, types, objects, functions, patterns, and tools for
jump-starting application development with modern conventions and best
practices.

=cut

=head1 FUNCTIONS

This package implements the following functions.

=cut

=head2 any

  any(Any $arg) : Object

The C<any> constructor function returns a L<Data::Object::Any> object for given
argument.

=over 4

=item any example

  # given \*main

  my $object = Data::Object->any(\*main);

=back

=cut

=head2 array

  array(ArrayRef $arg) : Object

The C<array> constructor function returns a L<Data::Object::Array> object for
given argument.

=over 4

=item array example

  # given [1..4]

  my $object = Data::Object->array([1..4]);

=back

=cut

=head2 code

  code(CodeRef $arg) : Object

The C<code> constructor function returns a L<Data::Object::Code> object for
given argument.

=over 4

=item code example

  # given sub { shift + 1 }

  my $object = Data::Object->code(sub { $_[0] + 1 });

=back

=cut

=head2 exception

  exception(HashRef $arg) : Object

The C<exception> constructor function returns a L<Data::Object::Exception>
object for given argument.

=over 4

=item exception example

  # given { message => 'Oops' }

  my $object = Data::Object->exception({ message => 'Oops' });

=back

=cut

=head2 float

  float(Num $arg) : Object

The C<float> constructor function returns a L<Data::Object::Float> object for given
argument.

=over 4

=item float example

  # given 1.23

  my $object = Data::Object->float(1.23);

=back

=cut

=head2 hash

  hash(HashRef $arg) : Object

The C<hash> constructor function returns a L<Data::Object::Hash> object for given
argument.

=over 4

=item hash example

  # given {1..4}

  my $object = Data::Object->hash({1..4});

=back

=cut

=head2 integer

  integer(Int $arg) : Object

The C<integer> constructor function returns a L<Data::Object::Integer> object for given
argument.

=over 4

=item integer example

  # given -123

  my $object = Data::Object->integer(-123);

=back

=cut

=head2 number

  number(Num $arg) : Object

The C<number> constructor function returns a L<Data::Object::Number> object for given
argument.

=over 4

=item number example

  # given 123

  my $object = Data::Object->number(123);

=back

=cut

=head2 regexp

  regexp(Regexp $arg) : Object

The C<regexp> constructor function returns a L<Data::Object::Regexp> object for given
argument.

=over 4

=item regexp example

  # given qr(\w+)

  my $object = Data::Object->regexp(qr(\w+));

=back

=cut

=head2 scalar

  scalar(Any $arg) : Object

The C<scalar> constructor function returns a L<Data::Object::Scalar> object for given
argument.

=over 4

=item scalar example

  # given \*main

  my $object = Data::Object->scalar(\*main);

=back

=cut

=head2 string

  string(Str $arg) : Object

The C<string> constructor function returns a L<Data::Object::String> object for given
argument.

=over 4

=item string example

  # given 'hello'

  my $object = Data::Object->string('hello');

=back

=cut

=head2 undef

  undef(Undef $arg?) : Object

The C<undef> constructor function returns a L<Data::Object::Undef> object for given
argument.

=over 4

=item undef example

  # given undef

  my $object = Data::Object->undef(undef);

=back

=cut

=head1 METHODS

This package implements the following methods.

=cut

=head2 new

  new(Str $arg) : Object

The new method expects a string representing a class name under the
Data::Object namespace and returns a L<Data::Object::Space> object.

=over 4

=item new example

  # given 'String'

  my $space = Data::Object->new('String');

  my $string = $space->build('hello world');

=back

=cut
