package Test::ListParser;

use Test::Most;
use Test::Moose 'has_attribute_ok';
use base 'Test::Class';

sub class { 'Siebel::Srvrmgr::ListParser' }

sub startup : Tests(startup => 1) {
    my $test = shift;
    use_ok $test->class;
}

sub constructor : Tests(15) {

    my $test  = shift;
    my $class = $test->class;

    can_ok( $class, 'new' );

    #extended method tests
    can_ok( $class,
        qw(get_parsed_tree get_prompt_regex set_prompt_regex get_hello_regex set_hello_regex get_last_command is_cmd_changed set_last_command set_buffer clear_buffer count_parsed clear_parsed_tree set_parsed_tree append_output parse)
    );

    my @data = <Test::ListParser::DATA>;
    close(Test::ListParser::DATA);

    ok( my $parser = $class->new( { is_warn_enabled => 1 } ),
        '... and the constructor should succeed' );

    has_attribute_ok( $parser, 'parsed_tree' );
    has_attribute_ok( $parser, 'has_tree' );
    has_attribute_ok( $parser, 'prompt_regex' );
    has_attribute_ok( $parser, 'hello_regex' );
    has_attribute_ok( $parser, 'last_command' );
    has_attribute_ok( $parser, 'is_cmd_changed' );
    has_attribute_ok( $parser, 'buffer' );

    isa_ok( $parser, $class, '... and the object it returns' );

    ok( $parser->parse( \@data ), 'parse method works' );

    ok( $parser->has_tree, 'the parser has a parsed tree' );

    my $last_cmd = 'list comp def SRProc';

    is( $parser->get_last_command(),
        $last_cmd, "get_last_command method returns $last_cmd" );

    my $total_itens = 7;

    is( $parser->count_parsed(),
        $total_itens, "count_parsed method returns $total_itens" );

}

1;

__DATA__
Siebel Enterprise Applications Siebel Server Manager, Version 8.0.0.7 [20426] LANG_INDEPENDENT 
Copyright (c) 2001 Siebel Systems, Inc.  All rights reserved.

This software is the property of Siebel Systems, Inc., 2207 Bridgepointe Parkway,
San Mateo, CA 94404.

User agrees that any use of this software is governed by: (1) the applicable
user limitations and other terms and conditions of the license agreement which
has been entered into with Siebel Systems or its authorized distributors; and
(2) the proprietary and restricted rights notices included in this software.

WARNING: THIS COMPUTER PROGRAM IS PROTECTED BY U.S. AND INTERNATIONAL LAW.
UNAUTHORIZED REPRODUCTION, DISTRIBUTION OR USE OF THIS PROGRAM, OR ANY PORTION
OF IT, MAY RESULT IN SEVERE CIVIL AND CRIMINAL PENALTIES, AND WILL BE
PROSECUTED TO THE MAXIMUM EXTENT POSSIBLE UNDER THE LAW.

If you have received this software in error, please notify Siebel Systems
immediately at (650) 295-5000.

Type "help" for list of commands, "help <topic>" for detailed help

Connected to 1 server(s) out of a total of 1 server(s) in the enterprise

srvrmgr:SUsrvr> load preferences
File: C:\Siebel\8.0\web client\BIN\.Siebel_svrmgr.pref

srvrmgr> list comp

SV_NAME     CC_ALIAS              CC_NAME                               CT_ALIAS  CG_ALIAS      CC_RUNMODE   CP_DISP_RUN_STATE  CP_NUM_RUN_TASKS  CP_MAX_TASKS  CP_ACTV_MTS_PROCS  CP_MAX_MTS_PROCS  CP_START_TIME        CP_END_TIME          CP_STATUS  CC_INCARN_NO  CC_DESC_TEXT  
----------  --------------------  ------------------------------------  --------  ------------  -----------  -----------------  ----------------  ------------  -----------------  ----------------  -------------------  -------------------  ---------  ------------  ------------  
sieb__crm01  ClientAdmin           Client Administration                           System        Background   Ativado            0                 1                                                  2012-02-18 17:11:56                                                              
sieb__crm01  CommConfigMgr         Communications Configuration Manager            CommMgmt      Batch        Ativado            0                 20            1                  1                 2012-02-18 17:11:56                                                              
sieb__crm01  CommInboundMgr        Communications Inbound Manager                  CommMgmt      Batch        Ativado            0                 20            1                  1                 2012-02-18 17:11:56                                                              
sieb__crm01  CommOutboundMgr       Communications Outbound Manager                 CommMgmt      Batch        Ativado            0                 50            1                  1                 2012-02-18 17:11:56                                                              
sieb__crm01  CommSessionMgr        Communications Session Manager                  CommMgmt      Batch        Ativado            0                 20            1                  1                 2012-02-18 17:11:56                                                              
sieb__crm01  DbXtract              Database Extract                                Remote        Batch        Ativado            0                 10                                                 2012-02-18 17:11:56                                                              
sieb__crm01  EAIObjMgr_ptb         EAI Object Manager (PTB)                        EAI           Interactive  Ativado            0                 20            1                  1                 2012-02-18 17:11:56                                                              
sieb__crm01  MailMgr               Email Manager                                   CommMgmt      Background   Ativado            0                 20                                                 2012-02-18 17:11:56                                                              
sieb__crm01  EIM                   Enterprise Integration Mgr                      EAI           Batch        Ativado            0                 10                                                 2012-02-18 17:11:56                                                              
sieb__crm01  FSMSrvr               File System Manager                             System        Batch        Ativado            0                 20            1                  1                 2012-02-18 17:11:56                                                              
sieb__crm01  GenNewDb              Generate New Database                           Remote        Batch        Ativado            0                 1                                                  2012-02-18 17:11:56                                                              
sieb__crm01  GenTrig               Generate Triggers                               Workflow      Batch        Ativado            0                 1                                                  2012-02-18 17:11:56                                                              
sieb__crm01  PageMgr               Page Manager                                    CommMgmt      Background   Ativado            0                 20                                                 2012-02-18 17:11:56                                                              
sieb__crm01  PDbXtract             Parallel Database Extract                       Remote        Batch        Em execuo        4                 10            1                  1                 2012-02-18 17:11:56                                                              
sieb__crm01  ServerMgr             Server Manager                                  System        Interactive  Em execuo        1                 20                                                 2012-02-18 17:11:56                                                              
sieb__crm01  SRBroker              Server Request Broker                           System        Interactive  Em execuo        10                100           1                  1                 2012-02-18 17:11:56                                                              
sieb__crm01  SRProc                Server Request Processor                        System        Interactive  Em execuo        2                 20            1                  1                 2012-02-18 17:11:56                                                              
sieb__crm01  SynchMgr              Synchronization Manager                         Remote        Interactive  Ativado            0                 100           1                  1                 2012-02-18 17:11:56                                                              
sieb__crm01  TxnMerge              Transaction Merger                              Remote        Background   Em execuo        1                 10                                                 2012-02-18 17:11:56                                                              
sieb__crm01  TxnProc               Transaction Processor                           Remote        Background   Em execuo        1                 1                                                  2012-02-18 17:11:56                                                              
sieb__crm01  TxnRoute              Transaction Router                              Remote        Background   Em execuo        3                 10                                                 2012-02-18 17:11:56                                                              
sieb__crm01  UpgKitBldr            Upgrade Kit Builder                             SiebAnywhere  Batch        Ativado            0                 1             1                  1                 2012-02-18 17:11:56                                                              
sieb__crm01  WorkActn              Workflow Action Agent                           Workflow      Background   Ativado            0                 5                                                  2012-02-18 17:11:56                                                              
sieb__crm01  WorkMon               Workflow Monitor Agent                          Workflow      Background   Em execuo        1                 1                                                  2012-02-18 17:11:56                                                              
sieb__crm01  WfProcBatchMgr        Workflow Process Batch Manager                  Workflow      Batch        Ativado            0                 20            1                  1                 2012-02-18 17:11:56                                                              
sieb__crm01  WfProcMgr             Workflow Process Manager                        Workflow      Batch        Ativado            0                 20            1                  1                 2012-02-18 17:11:56                                                              
sieb__crm01  ePharmaObjMgr_ptb     ePharma Object Manager (PTB)                    LifeSciences  Interactive  Em execuo        3                 60            1                  2                 2012-02-18 17:11:56                                                              

51 rows returned.

srvrmgr:SUsrvr> list comp types

CT_NAME                                                                       CT_RUNMODE                       CT_ALIAS                                                       CT_DESC_TEXT                                                                                                                                                                                                                                                 
----------------------------------------------------------------------------  -------------------------------  -------------------------------------------------------------  --------------------------------------------------------------------------------------------------------------------  
ADM Service                                                                   Batch                            ADM Service                                                    ADM Service Service                                                                                                                                                                                                                                          
ADM Service                                                                   Background                       ADM Service                                                    ADM Service Service                                                                                                                                                                                                                                          
Siebel Administrator Notification Component                                   Batch                            AdminNotify                                                    Component used for administrator notification                                                                                                                                                                                                                
Analytic Adaptor Manager                                                      Batch                            Analytic Adaptor Manager                                       Analytic Adaptor Manager Service                                                                                                                                                                                                                             
Analytic Adaptor Manager                                                      Background                       Analytic Adaptor Manager                                       Analytic Adaptor Manager Service                                                                                                                                                                                                                             
Application Object Manager                                                    Interactive                      AppObjMgr                                                      Object manager component type for thin-client and web-client applications                                                                                                                                                                                    
Appointment Booking Service                                                   Batch                            Appointment Booking Service                                    Appointment Booking Service Service                                                                                                                                                                                                                          
Appointment Booking Service                                                   Background                       Appointment Booking Service                                    Appointment Booking Service Service                                                                                                                                                                                                                          
Batch Assignment                                                              Batch                            AsgnBatch                                                      Batch assigns positions and employees to objects                                                                                                                                                                                                             
Assignment Manager                                                            Batch                            AsgnSrvr                                                       Assigns positions and employees to objects                                                                                                                                                                                                                   
Business Service Manager                                                      Batch                            BusSvcMgr                                                      Business Service Manager component                                                                                                                                                                                                                           
Business Service Manager                                                      Background                       BusSvcMgr                                                      Business Service Manager component                                                                                                                                                                                                                           
Categorization Manager                                                        Batch                            Categorization Manager                                         Categorization Manager Service                                                                                                                                                                                                                               
Categorization Manager                                                        Background                       Categorization Manager                                         Categorization Manager Service                                                                                                                                                                                                                               
CheckDup                                                                      Batch                            CheckDup                                                       CheckDup Service                                                                                                                                                                                                                                             
CheckDup                                                                      Background                       CheckDup                                                       CheckDup Service                                                                                                                                                                                                                                             
Communications Inbound Receiver                                               Batch                            CommInboundRcvr                                                Loads response groups to receive and queue inbound events                                                                                                                                                                                                    
Communications Session Manager                                                Batch                            CommSessionMgr                                                 Interact with end user for utilizing communications channels                                                                                                                                                                                                 
Communication Server Configuration                                            Batch                            Communication Server Configuration                             Communication Server Configuration Service                                                                                                                                                                                                                   
Communication Server Configuration                                            Background                       Communication Server Configuration                             Communication Server Configuration Service                                                                                                                                                                                                                   
Communications Inbound Processor                                              Batch                            Communications Inbound Processor                               Communications Inbound Processor Service                                                                                                                                                                                                                     
Communications Inbound Processor                                              Background                       Communications Inbound Processor                               Communications Inbound Processor Service                                                                                                                                                                                                                     
Complete Activity                                                             Batch                            Complete Activity                                              Complete Activity Service                                                                                                                                                                                                                                    
Complete Activity                                                             Background                       Complete Activity                                              Complete Activity Service                                                                                                                                                                                                                                    
Content Project                                                               Batch                            Content Project                                                Content Project Service                                                                                                                                                                                                                                      
Content Project                                                               Background                       Content Project                                                Content Project Service                                                                                                                                                                                                                                      
Contracts Accumulator Service                                                 Batch                            Contracts Accumulator Service                                  Contracts Accumulator Service Service                                                                                                                                                                                                                        
Contracts Accumulator Service                                                 Background                       Contracts Accumulator Service                                  Contracts Accumulator Service Service                                                                                                                                                                                                                        
Contracts Evaluator Service                                                   Batch                            Contracts Evaluator Service                                    Contracts Evaluator Service Service                                                                                                                                                                                                                          
Contracts Evaluator Service                                                   Background                       Contracts Evaluator Service                                    Contracts Evaluator Service Service                                                                                                                                                                                                                          
Contracts Resolver Service                                                    Batch                            Contracts Resolver Service                                     Contracts Resolver Service Service                                                                                                                                                                                                                           
Contracts Resolver Service                                                    Background                       Contracts Resolver Service                                     Contracts Resolver Service Service                                                                                                                                                                                                                           
Crediting Engine DB Operations                                                Batch                            Crediting Engine DB Operations                                 Crediting Engine DB Operations Service                                                                                                                                                                                                                       
Crediting Engine DB Operations                                                Background                       Crediting Engine DB Operations                                 Crediting Engine DB Operations Service                                                                                                                                                                                                                       
Custom Application Object Manager                                             Interactive                      CustomAppObjMgr                                                Object manager component type for thin-client Custom UI applications                                                                                                                                                                                         
DCommerce Alerts                                                              Background                       DCommerce Alerts                                               Background process that manages DCommerce alerts                                                                                                                                                                                                             
DCommerce Automatic Auction Close                                             Background                       DCommerce Automatic Auction Close                              Background process that detects and closes auctions                                                                                                                                                                                                          
DNB Update                                                                    Batch                            DNB Update                                                     DNB Update Service                                                                                                                                                                                                                                           
DNB Update                                                                    Background                       DNB Update                                                     DNB Update Service                                                                                                                                                                                                                                           
Data Cleansing                                                                Batch                            Data Cleansing                                                 Data Cleansing Service                                                                                                                                                                                                                                       
Data Cleansing                                                                Background                       Data Cleansing                                                 Data Cleansing Service                                                                                                                                                                                                                                       
Database Extract                                                              Batch                            DbXtract                                                       Extracts visible data for a Siebel Remote client                                                                                                                                                                                                             
DeDuplication                                                                 Batch                            DeDuplication                                                  DeDuplication Service                                                                                                                                                                                                                                        
DeDuplication                                                                 Background                       DeDuplication                                                  DeDuplication Service                                                                                                                                                                                                                                        
Document Driver                                                               Batch                            Document Driver                                                Document Driver Service                                                                                                                                                                                                                                      
Document Driver                                                               Background                       Document Driver                                                Document Driver Service                                                                                                                                                                                                                                      
Data Quality Manager                                                          Batch                            Dqmgr                                                          Cleanse data and de-duplicate records                                                                                                                                                                                                                        
DynamicCommerce                                                               Batch                            DynamicCommerce                                                DynamicCommerce Service                                                                                                                                                                                                                                      
DynamicCommerce                                                               Background                       DynamicCommerce                                                DynamicCommerce Service                                                                                                                                                                                                                                      
EAI Business Integration Manager                                              Batch                            EAI Business Integration Manager                               EAI Business Integration Manager Service                                                                                                                                                                                                                     
EAI Business Integration Manager                                              Background                       EAI Business Integration Manager                               EAI Business Integration Manager Service                                                                                                                                                                                                                     
Enterprise Application Integration Connector Receiver                         Background                       EAIDeprecatedRcvr                                              Deprecated receiver for in-bound EAI transactions used by connectors                                                                                                                                                                                         
EAILOVService                                                                 Batch                            EAILOVService                                                  EAILOVService Service                                                                                                                                                                                                                                        
EAILOVService                                                                 Background                       EAILOVService                                                  EAILOVService Service                                                                                                                                                                                                                                        
EAI Object Manager                                                            Interactive                      EAIObjMgr                                                      Object manager component type for thin-client EAI applications                                                                                                                                                                                               
Enterprise Application Integration Receiver                                   Background                       EAIRcvr                                                        Generic receiver for in-bound EAI transactions                                                                                                                                                                                                               
Enterprise Integration Mgr                                                    Batch                            EIM                                                            Integrates enterprise data to and from other systems                                                                                                                                                                                                         
ERM Compensation Planning Service                                             Batch                            ERM Compensation Planning Service                              ERM Compensation Planning Service Service                                                                                                                                                                                                                    
ERM Compensation Planning Service                                             Background                       ERM Compensation Planning Service                              ERM Compensation Planning Service Service                                                                                                                                                                                                                    
Excel Importer Exporter                                                       Batch                            Excel Importer Exporter                                        Excel Importer Exporter Service                                                                                                                                                                                                                              
Excel Importer Exporter                                                       Background                       Excel Importer Exporter                                        Excel Importer Exporter Service                                                                                                                                                                                                                              
FS Cycle Counting                                                             Batch                            FS Cycle Counting                                              FS Cycle Counting Service                                                                                                                                                                                                                                    
FS Cycle Counting                                                             Background                       FS Cycle Counting                                              FS Cycle Counting Service                                                                                                                                                                                                                                    
FS Fulfillment Service                                                        Batch                            FS Fulfillment Service                                         FS Fulfillment Service Service                                                                                                                                                                                                                               
FS Fulfillment Service                                                        Background                       FS Fulfillment Service                                         FS Fulfillment Service Service                                                                                                                                                                                                                               
FS Mobile Inventory Transaction                                               Batch                            FS Mobile Inventory Transaction                                FS Mobile Inventory Transaction Service                                                                                                                                                                                                                      
FS Mobile Inventory Transaction                                               Background                       FS Mobile Inventory Transaction                                FS Mobile Inventory Transaction Service                                                                                                                                                                                                                      
FS Part Locator Service                                                       Batch                            FS Part Locator Service                                        FS Part Locator Service Service                                                                                                                                                                                                                              
FS Part Locator Service                                                       Background                       FS Part Locator Service                                        FS Part Locator Service Service                                                                                                                                                                                                                              
FS Preventive Maintenance                                                     Batch                            FS Preventive Maintenance                                      FS Preventive Maintenance Service                                                                                                                                                                                                                            
FS Preventive Maintenance                                                     Background                       FS Preventive Maintenance                                      FS Preventive Maintenance Service                                                                                                                                                                                                                            
FS Replenish                                                                  Batch                            FS Replenish                                                   FS Replenish Service                                                                                                                                                                                                                                         
FS Replenish                                                                  Background                       FS Replenish                                                   FS Replenish Service                                                                                                                                                                                                                                         
File System Manager                                                           Batch                            FSMSrvr                                                        The file system manager component                                                                                                                                                                                                                            
Forecast 2000 Internal Service                                                Batch                            Forecast 2000 Internal Service                                 Forecast 2000 Internal Service Service                                                                                                                                                                                                                       
Forecast 2000 Internal Service                                                Background                       Forecast 2000 Internal Service                                 Forecast 2000 Internal Service Service                                                                                                                                                                                                                       
Generate New Database                                                         Batch                            GenNewDb                                                       Generates a new Sybase SQL Anywhere database template file for Siebel Remote                                                                                                                                                                                 
Generate Triggers                                                             Batch                            GenTrig                                                        Generates triggers for Workflow Manager and Assignment Manager                                                                                                                                                                                               
HTIM MDF Period Ending Service                                                Batch                            HTIM MDF Period Ending Service                                 HTIM MDF Period Ending Service Service                                                                                                                                                                                                                       
HTIM MDF Period Ending Service                                                Background                       HTIM MDF Period Ending Service                                 HTIM MDF Period Ending Service Service                                                                                                                                                                                                                       
Handheld Batch Synchronization                                                Batch                            Handheld Batch Synchronization                                 Handheld Batch Synchronization Service                                                                                                                                                                                                                       
Handheld Batch Synchronization                                                Background                       Handheld Batch Synchronization                                 Handheld Batch Synchronization Service                                                                                                                                                                                                                       
Handheld Synchronization Agent                                                Batch                            Handheld Synchronization Agent                                 Handheld Synchronization Agent Service                                                                                                                                                                                                                       
Handheld Synchronization Agent                                                Background                       Handheld Synchronization Agent                                 Handheld Synchronization Agent Service                                                                                                                                                                                                                       
Handheld Synchronization                                                      Batch                            Handheld Synchronization                                       Handheld Synchronization Service                                                                                                                                                                                                                             
Handheld Synchronization                                                      Background                       Handheld Synchronization                                       Handheld Synchronization Service                                                                                                                                                                                                                             
IC Quota Import Service                                                       Batch                            IC Quota Import Service                                        IC Quota Import Service Service                                                                                                                                                                                                                              
IC Quota Import Service                                                       Background                       IC Quota Import Service                                        IC Quota Import Service Service                                                                                                                                                                                                                              
IComp Credit Assignment Engine                                                Batch                            ICompCreditAsgn                                                Assigns credit allocation based on crediting rules                                                                                                                                                                                                           
Incentive Compensation Mgr                                                    Batch                            ICompMgr                                                       Calculates Incentive Compensations                                                                                                                                                                                                                           
SRM Tester for key-based routing                                              Batch                            KRSrmTst                                                       Server Request Manager Test Component for key-based routing                                                                                                                                                                                                  
LOY Processing Engine                                                         Batch                            LOY Processing Engine                                          LOY Processing Engine Service                                                                                                                                                                                                                                
LOY Processing Engine                                                         Background                       LOY Processing Engine                                          LOY Processing Engine Service                                                                                                                                                                                                                                
List Import                                                                   Batch                            List Import                                                    List Import Service                                                                                                                                                                                                                                          
List Import                                                                   Background                       List Import                                                    List Import Service                                                                                                                                                                                                                                          
List Manager                                                                  Batch                            ListMgr                                                        Loads outside files for list manager                                                                                                                                                                                                                         
E-mail Manager                                                                Background                       MailMgr                                                        Sends e-mail initiated by Workflow Manager                                                                                                                                                                                                                   
Message Board Maintenance Service                                             Batch                            Message Board Maintenance Service                              Message Board Maintenance Service Service                                                                                                                                                                                                                    
Message Board Maintenance Service                                             Background                       Message Board Maintenance Service                              Message Board Maintenance Service Service                                                                                                                                                                                                                    
OM Regression Test                                                            Batch                            OM Regression Test                                             OM Regression Test Service                                                                                                                                                                                                                                   
OM Regression Test                                                            Background                       OM Regression Test                                             OM Regression Test Service                                                                                                                                                                                                                                   
OM Remote Regression Test                                                     Batch                            OM Remote Regression Test                                      OM Remote Regression Test Service                                                                                                                                                                                                                            
OM Remote Regression Test                                                     Background                       OM Remote Regression Test                                      OM Remote Regression Test Service                                                                                                                                                                                                                            
Optimizer Service                                                             Batch                            Optimizer Service                                              Optimizer Service Service                                                                                                                                                                                                                                    
Optimizer Service                                                             Background                       Optimizer Service                                              Optimizer Service Service                                                                                                                                                                                                                                    
Outbound Communications Manager                                               Batch                            Outbound Communications Manager                                Outbound Communications Manager Service                                                                                                                                                                                                                      
Outbound Communications Manager                                               Background                       Outbound Communications Manager                                Outbound Communications Manager Service                                                                                                                                                                                                                      
PIMSI Engine Service                                                          Batch                            PIMSI Engine Service                                           PIMSI Engine Service Service                                                                                                                                                                                                                                 
PIMSI Engine Service                                                          Background                       PIMSI Engine Service                                           PIMSI Engine Service Service                                                                                                                                                                                                                                 
Page Manager                                                                  Background                       PageMgr                                                        Sends pages initiated by Workflow Manager and Siebel Client                                                                                                                                                                                                  
Perf MQReceive Processor                                                      Batch                            Perf MQReceive Processor                                       Perf MQReceive Processor Service                                                                                                                                                                                                                             
Perf MQReceive Processor                                                      Background                       Perf MQReceive Processor                                       Perf MQReceive Processor Service                                                                                                                                                                                                                             
Quick Fill Service                                                            Batch                            Quick Fill Service                                             Quick Fill Service Service                                                                                                                                                                                                                                   
Quick Fill Service                                                            Background                       Quick Fill Service                                             Quick Fill Service Service                                                                                                                                                                                                                                   
Batch Real-Time Integration                                                   Batch                            RTIBatch                                                       Batch for Real-Time integration with BackOffice (ERP) Systems                                                                                                                                                                                                
RTS Extractor Service                                                         Batch                            RTS Extractor Service                                          RTS Extractor Service Service                                                                                                                                                                                                                                
RTS Extractor Service                                                         Background                       RTS Extractor Service                                          RTS Extractor Service Service                                                                                                                                                                                                                                
RTS Message Apply Service                                                     Batch                            RTS Message Apply Service                                      RTS Message Apply Service Service                                                                                                                                                                                                                            
RTS Message Apply Service                                                     Background                       RTS Message Apply Service                                      RTS Message Apply Service Service                                                                                                                                                                                                                            
RTS Sender Service                                                            Batch                            RTS Sender Service                                             RTS Sender Service Service                                                                                                                                                                                                                                   
RTS Sender Service                                                            Background                       RTS Sender Service                                             RTS Sender Service Service                                                                                                                                                                                                                                   
RTS Subscription Service                                                      Batch                            RTS Subscription Service                                       RTS Subscription Service Service                                                                                                                                                                                                                             
RTS Subscription Service                                                      Background                       RTS Subscription Service                                       RTS Subscription Service Service                                                                                                                                                                                                                             
RTS Transaction Dispatch Service                                              Batch                            RTS Transaction Dispatch Service                               RTS Transaction Dispatch Service Service                                                                                                                                                                                                                     
RTS Transaction Dispatch Service                                              Background                       RTS Transaction Dispatch Service                               RTS Transaction Dispatch Service Service                                                                                                                                                                                                                     
RTS Transaction Dispatcher                                                    Background                       RTSDispatcher                                                  Dispatches the transactions to the RTS channel                                                                                                                                                                                                               
Replication Agent                                                             Background                       RepAgent                                                       Synchronizes a Siebel Remote regional database with HQ                                                                                                                                                                                                       
Report Business Service                                                       Batch                            Report Business Service                                        Report Business Service Service                                                                                                                                                                                                                              
Report Business Service                                                       Background                       Report Business Service                                        Report Business Service Service                                                                                                                                                                                                                              
Server Request Broker                                                         Interactive                      ReqBroker                                                      Route requests and asynchronous notification among clients and components                                                                                                                                                                                    
Row Set Transformation Toolkit                                                Batch                            Row Set Transformation Toolkit                                 Row Set Transformation Toolkit Service                                                                                                                                                                                                                       
Row Set Transformation Toolkit                                                Background                       Row Set Transformation Toolkit                                 Row Set Transformation Toolkit Service                                                                                                                                                                                                                       
Rule Manager Service                                                          Batch                            Rule Manager Service                                           Rule Manager Service Service                                                                                                                                                                                                                                 
Rule Manager Service                                                          Background                       Rule Manager Service                                           Rule Manager Service Service                                                                                                                                                                                                                                 
Rule Runtime Administration                                                   Batch                            Rule Runtime Administration                                    Rule Runtime Administration Service                                                                                                                                                                                                                          
Rule Runtime Administration                                                   Background                       Rule Runtime Administration                                    Rule Runtime Administration Service                                                                                                                                                                                                                          
Siebel Connection Broker                                                      Background                       SCBroker                                                       Route and load balance connections to components                                                                                                                                                                                                             
SCF Message Facility Test Component                                           Batch                            SCFMsgFacTest                                                  Component used for testing the SCF Message Facility                                                                                                                                                                                                          
SCF Message Facility Test Component                                           Background                       SCFMsgFacTest                                                  Component used for testing the SCF Message Facility                                                                                                                                                                                                          
SIS Test Component                                                            Batch                            SISTst                                                         SIS Test Component use for testing shared memory instance                                                                                                                                                                                                    
SIS Test Component                                                            Interactive                      SISTst                                                         SIS Test Component use for testing shared memory instance                                                                                                                                                                                                    
SIS Test Component                                                            Background                       SISTst                                                         SIS Test Component use for testing shared memory instance                                                                                                                                                                                                    
SMQ Message Service                                                           Batch                            SMQ Message Service                                            SMQ Message Service Service                                                                                                                                                                                                                                  
SMQ Message Service                                                           Background                       SMQ Message Service                                            SMQ Message Service Service                                                                                                                                                                                                                                  
SMQ Transport Service                                                         Batch                            SMQ Transport Service                                          SMQ Transport Service Service                                                                                                                                                                                                                                
SMQ Transport Service                                                         Background                       SMQ Transport Service                                          SMQ Transport Service Service                                                                                                                                                                                                                                
Server Request Processor (SRP)                                                Interactive                      SRProc                                                         Store requests/notifications into database and forward them to components/clients                                                                                                                                                                            
Sales Hierarchy Service                                                       Batch                            Sales Hierarchy Service                                        Sales Hierarchy Service Service                                                                                                                                                                                                                              
Sales Hierarchy Service                                                       Background                       Sales Hierarchy Service                                        Sales Hierarchy Service Service                                                                                                                                                                                                                              
Search Data Processor                                                         Batch                            Search Data Processor                                          Search Data Processor Service                                                                                                                                                                                                                                
Search Data Processor                                                         Background                       Search Data Processor                                          Search Data Processor Service                                                                                                                                                                                                                                
Search External Service                                                       Batch                            Search External Service                                        Search External Service Service                                                                                                                                                                                                                              
Search External Service                                                       Background                       Search External Service                                        Search External Service Service                                                                                                                                                                                                                              
Server Manager                                                                Interactive                      ServerMgr                                                      Administration of Siebel Servers within the Enterprise                                                                                                                                                                                                       
Smart Answer Client                                                           Batch                            Smart Answer Client                                            Smart Answer Client Service                                                                                                                                                                                                                                  
Smart Answer Client                                                           Background                       Smart Answer Client                                            Smart Answer Client Service                                                                                                                                                                                                                                  
Smart Script Execution Cleanup                                                Batch                            Smart Script Execution Cleanup                                 Smart Script Execution Cleanup Service                                                                                                                                                                                                                       
Smart Script Execution Cleanup                                                Background                       Smart Script Execution Cleanup                                 Smart Script Execution Cleanup Service                                                                                                                                                                                                                       
Source Code Lookup                                                            Batch                            Source Code Lookup                                             Source Code Lookup Service                                                                                                                                                                                                                                   
Source Code Lookup                                                            Background                       Source Code Lookup                                             Source Code Lookup Service                                                                                                                                                                                                                                   
SrchFileSrvr                                                                  Batch                            SrchFileSrvr                                                   SrchFileSrvr Service                                                                                                                                                                                                                                         
SrchFileSrvr                                                                  Background                       SrchFileSrvr                                                   SrchFileSrvr Service                                                                                                                                                                                                                                         
SRM Tester 1                                                                  Batch                            SrmTst1                                                        Server Request Manager Test Component 1                                                                                                                                                                                                                      
SRM Tester 2                                                                  Batch                            SrmTst2                                                        Server Request Manager Test Component 2                                                                                                                                                                                                                      
SRM Tester 3                                                                  Batch                            SrmTst3                                                        Server Request Manager Test Component 3                                                                                                                                                                                                                      
SRM Tester 4                                                                  Batch                            SrmTst4                                                        Server Request Manager Test Component 4                                                                                                                                                                                                                      
Synchronization Manager                                                       Interactive                      SynchMgr                                                       Services Siebel Remote synchronization clients                                                                                                                                                                                                               
TNT SHM Quote Service                                                         Batch                            TNT SHM Quote Service                                          TNT SHM Quote Service Service                                                                                                                                                                                                                                
TNT SHM Quote Service                                                         Background                       TNT SHM Quote Service                                          TNT SHM Quote Service Service                                                                                                                                                                                                                                
Task Log Cleanup Service                                                      Batch                            Task Log Cleanup Service                                       Task Log Cleanup Service Service                                                                                                                                                                                                                             
Task Log Cleanup Service                                                      Background                       Task Log Cleanup Service                                       Task Log Cleanup Service Service                                                                                                                                                                                                                             
Test Utilities                                                                Batch                            Test Utilities                                                 Test Utilities Service                                                                                                                                                                                                                                       
Test Utilities                                                                Background                       Test Utilities                                                 Test Utilities Service                                                                                                                                                                                                                                       
Test Data Access                                                              Batch                            TestDataAccess                                                 Component used for testing SCF data access layer.                                                                                                                                                                                                            
Test Data Access                                                              Background                       TestDataAccess                                                 Component used for testing SCF data access layer.                                                                                                                                                                                                            
Test SCF Data Access Performance                                              Batch                            TestDataAccessPerf                                             Component used for testing SCF data access layer performance.                                                                                                                                                                                                
Test SCF Event Facility                                                       Batch                            TestEventFacility                                              Component used for testing SCF event facility and its performance.                                                                                                                                                                                           
Request MT Test Server                                                        Batch                            TestMTReq                                                      Component used for testing Multi-threaded Request Mode Server                                                                                                                                                                                                
Multithreaded Test Server                                                     Batch                            TestMTSrvr                                                     Component used for testing Multithreaded Server                                                                                                                                                                                                              
Multithreaded Test Server                                                     Interactive                      TestMTSrvr                                                     Component used for testing Multithreaded Server                                                                                                                                                                                                              
Request Test Server                                                           Batch                            TestReq                                                        Component used for testing Request Mode Server                                                                                                                                                                                                               
Test SCF Facilities                                                           Batch                            TestScfFacilities                                              Component used for testing new SCF facilities.                                                                                                                                                                                                               
Test SCF Facilities                                                           Background                       TestScfFacilities                                              Component used for testing new SCF facilities.                                                                                                                                                                                                               
Session Test Server                                                           Interactive                      TestSess                                                       Component used for testing Session Mode Server                                                                                                                                                                                                               
Test Server                                                                   Batch                            TestSrvr                                                       Component used for testing Client Administration                                                                                                                                                                                                             
Test Server                                                                   Background                       TestSrvr                                                       Component used for testing Client Administration                                                                                                                                                                                                             
Transaction Merger                                                            Background                       TxnMerge                                                       Merges transactions from Siebel Remote clients                                                                                                                                                                                                               
Transaction Processor                                                         Background                       TxnProc                                                        Prepares the transaction log for the Transaction Router                                                                                                                                                                                                      
Transaction Router                                                            Background                       TxnRoute                                                       Routes visible transactions to Siebel Remote clients                                                                                                                                                                                                         
UDA Service                                                                   Batch                            UDA Service                                                    UDA Service Service                                                                                                                                                                                                                                          
UDA Service                                                                   Background                       UDA Service                                                    UDA Service Service                                                                                                                                                                                                                                          
Universal Data Cleansing Service                                              Batch                            Universal Data Cleansing Service                               Universal Data Cleansing Service Service                                                                                                                                                                                                                     
Universal Data Cleansing Service                                              Background                       Universal Data Cleansing Service                               Universal Data Cleansing Service Service                                                                                                                                                                                                                     
UoM Conversion Business Service                                               Batch                            UoM Conversion Business Service                                UoM Conversion Business Service Service                                                                                                                                                                                                                      
UoM Conversion Business Service                                               Background                       UoM Conversion Business Service                                UoM Conversion Business Service Service                                                                                                                                                                                                                      
Update Manager                                                                Batch                            UpdateMgr                                                      Server component that will update DNB and List Management                                                                                                                                                                                                    
Upgrade Kit Wizard OMSV                                                       Batch                            Upgrade Kit Wizard OMSV                                        Upgrade Kit Wizard OMSV Service                                                                                                                                                                                                                              
Upgrade Kit Wizard OMSV                                                       Background                       Upgrade Kit Wizard OMSV                                        Upgrade Kit Wizard OMSV Service                                                                                                                                                                                                                              
Workflow Action Agent                                                         Background                       WorkActn                                                       Executes actions for pre-defined events                                                                                                                                                                                                                      
Workflow Monitor Agent                                                        Background                       WorkMon                                                        Monitors the database for pre-defined events                                                                                                                                                                                                                 
Workflow Process Manager                                                      Batch                            Workflow Process Manager                                       Workflow Process Manager Service                                                                                                                                                                                                                             
Workflow Process Manager                                                      Background                       Workflow Process Manager                                       Workflow Process Manager Service                                                                                                                                                                                                                             
Workflow Recovery Manager                                                     Batch                            Workflow Recovery Manager                                      Workflow Recovery Manager Service                                                                                                                                                                                                                            
Workflow Recovery Manager                                                     Background                       Workflow Recovery Manager                                      Workflow Recovery Manager Service                                                                                                                                                                                                                            
eAuto VDS Accessorization Utility Service                                     Batch                            eAuto VDS Accessorization Utility Service                      eAuto VDS Accessorization Utility Service Service                                                                                                                                                                                                            
eAuto VDS Accessorization Utility Service                                     Background                       eAuto VDS Accessorization Utility Service                      eAuto VDS Accessorization Utility Service Service                                                                                                                                                                                                            

206 rows returned.

srvrmgr:SUsrvr> list params

PA_ALIAS                               PA_VALUE                      PA_DATATYPE  PA_SCOPE   PA_SUBSYSTEM                    PA_SETLEVEL       PA_DISP_SETLEVEL      PA  PA  PA  PA  PA_NAME                                                                 
-------------------------------------  ----------------------------  -----------  ---------  ------------------------------  ----------------  --------------------  --  --  --  --  ----------------------------------------------------------------------  
ActivityId                                                           String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Activity Id                                                             
ActuateReportCastDomain                                              String       Subsystem  Infrastructure Actuate Reports  Never set         Never set             Y   N   N   N   Actuate Report Cast Domain                                              
ActuateReportCastLang                  LANG_INDEPENDENT              String       Subsystem  Infrastructure Actuate Reports  Server level      Server level set      Y   N   N   N   Actuate Server Report Cast Language                                     
AddressList                                                          String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Address List                                                            
AnonymousQuote                         FALSE                         String       Subsystem  Infrastructure Shopping Servic  Default value     Default value         Y   N   N   N   AnonymousQuote                                                          
AppendOrigMsg                                                        String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Append Original Message                                                 
AssetBasedOrderingEnabled              False                         Boolean      Subsystem  Object Manager                  Default value     Default value         Y   N   N   N   Order Management - Enable Asset Based Ordering                          
AttachFileList                                                       String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Attachment File List                                                    
AttachNameList                                                       String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Attachment Name List                                                    
BizRuleUseLIC                          False                         Boolean      Subsystem  Rules Engine                    Default value     Default value         Y   N   N   N   Use LIC for Business Rules                                              
BusObjCacheSize                        0                             Integer      Subsystem  EAI                             Never set         Never set             Y   N   N   N   Business Object Cache Size                                              
BusinessServiceQueryAccessList                                       String       Subsystem  Object Manager                  Never set         Never set             N   N   N   N   Business Service Query Access List                                      
CACertFileName                                                       String       Subsystem  Networking                      Never set         Never set             N   N   Y   N   CA certificate file name                                                
CFGAccessDir                           \sea\access                   String       Subsystem  Infrastructure Objmgr configur  Default value     Default value         Y   N   N   N   Application Access Directory                                            
CFGClientRootDir                       C:\sba80\siebsrvr             String       Subsystem  Infrastructure Objmgr configur  Server level      Server level set      Y   N   N   N   Application Client Rootdir                                              
CFGCorrespODBCDatasource               Siebel Reports: Access        String       Subsystem  Infrastructure Objmgr configur  Default value     Default value         Y   N   N   N   Application Correspondence ODBC datasource                              
CFGDatasource                          ServerDataSrc                 String       Subsystem  Infrastructure Objmgr configur  Default value     Default value         Y   N   N   N   Application Datasource                                                  
CFGDocumentIntegrator                  Microsoft Office              String       Subsystem  Infrastructure Objmgr configur  Default value     Default value         Y   N   N   N   Application Document Integrator                                         
CFGEnableBizRule                       True                          Boolean      Subsystem  Infrastructure Objmgr configur  Default value     Default value         Y   N   N   N   Enable Business Rule modules                                            
CFGEnableMsgBrdcstCache                False                         Boolean      Subsystem  Infrastructure Objmgr configur  Default value     Default value         Y   N   N   N   Application Enable Message Broadcast Cache                              
CFGEnableOLEAutomation                 True                          Boolean      Subsystem  Infrastructure Objmgr configur  Server level      Server level set      Y   N   N   N   Application OLE Automated Flag                                          
CFGEnableScripting                     True                          Boolean      Subsystem  Infrastructure Objmgr configur  Default value     Default value         Y   N   N   N   Application Scripting Enabled                                           
CFGEnableTrainingQueue                 False                         Boolean      Subsystem  Infrastructure Objmgr configur  Default value     Default value         Y   N   N   N   Application Enable Training Queue                                       
CFGJTCHelpURL                                                        String       Subsystem  Infrastructure Objmgr configur  Never set         Never set             Y   N   N   N   Application JTC HELP URL                                                
CFGJseCorbaConnector                                                 String       Subsystem  Infrastructure Objmgr configur  Never set         Never set             Y   N   N   N   Application JSE Corba Connector Dll                                     
CFGMsgBrdcstCacheSize                  100                           Integer      Subsystem  Infrastructure Objmgr configur  Default value     Default value         Y   N   N   N   Application Message Broadcast Cache Size                                
CFGOLEAutomationDLL                    sscfole.dll                   String       Subsystem  Infrastructure Objmgr configur  Server level      Server level set      Y   N   N   N   Application OLE Automation DLL                                          
CFGReportsDir                                                        String       Subsystem  Infrastructure Objmgr configur  Never set         Never set             Y   N   N   N   Application Reports Directory                                           
CFGScriptingDLL                        sscfbas.dll                   String       Subsystem  Infrastructure Objmgr configur  Server level      Server level set      Y   N   N   N   Application Scripting Dll                                               
CFGServerODBCDatasource                                              String       Subsystem  Infrastructure Objmgr configur  Never set         Never set             Y   N   N   N   Application Server ODBC datasource                                      
CFGShowMessageBar                      User Enable                   String       Subsystem  Infrastructure Objmgr configur  Default value     Default value         Y   N   N   N   Application Message Bar Flag                                            
CacheCategoryId                        False                         Boolean      Subsystem  Categorization Manager          Default value     Default value         N   N   N   N   Cache Category ID                                                       
CatMgrType                             Master                        String       Subsystem  Categorization Manager          Default value     Default value         N   N   N   N   Categorization Manager Type                                             
CertFileName                                                         String       Subsystem  Networking                      Never set         Never set             N   N   Y   N   Certificate file name                                                   
Charset                                                              String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Charset                                                                 
CheckIfCandidateActive                 False                         Boolean      Subsystem  Assignment Subsystem            Default value     Default value         Y   N   N   N   Check If CandidateActive                                                
ChildRecipSearchSpec                                                 String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Child Recipient Search Spec                                             
CommProfile                                                          String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Communication Profile                                                   
CommProfileOverride                                                  String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Comm Profile Override                                                   
CommRequestId                                                        String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Comm Request Id                                                         
CommRequestParentId                                                  String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Request Parent Id                                                       
Comments                                                             String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Comm Request Comments                                                   
CompressedFileDownload                 True                          Boolean      Subsystem  Object Manager                  Default value     Default value         Y   N   N   N   Compressed File Download                                                
ConfigFile                             siebel.cfg                    String       Subsystem  Object Manager                  Default value     Default value         N   N   Y   N   OM - Configuration File                                                 
Connect                                Siebel_DSN                    String       Subsystem  Database Access                 Enterprise level  Enterprise level set  Y   N   N   N   ODBC Data Source                                                        
CopyCandSpecData                       No                            String       Subsystem  Assignment Subsystem            Default value     Default value         Y   N   N   N   Copy Candidate Specific Data                                            
CopyPersonSpecData                     No                            String       Subsystem  Assignment Subsystem            Default value     Default value         Y   N   N   N   Copy Person Specific Data                                               
DDDSN                                                                String       Subsystem  Data Dictionary Manager         Never set         Never set             N   N   N   N   ODBC DSN                                                                
DDPWD                                                                String       Subsystem  Data Dictionary Manager         Never set         Never set             N   N   N   N   Database PWD                                                            
DDTABLEID                                                            String       Subsystem  Data Dictionary Manager         Never set         Never set             N   N   N   N   TABLEID                                                                 
DDTABLENAME                                                          String       Subsystem  Data Dictionary Manager         Never set         Never set             N   N   N   N   Database TABLENAME                                                      
DDTABLEOWNER                                                         String       Subsystem  Data Dictionary Manager         Never set         Never set             N   N   N   N   Database TABLEOWNER                                                     
DDUID                                                                String       Subsystem  Data Dictionary Manager         Never set         Never set             N   N   N   N   Database UID                                                            
DSChartImageFormat                                                   String       Subsystem  Datasources Subsystem           Never set         Never set             Y   N   N   N   Chart Image Format                                                      
DSChartServer                                                        String       Subsystem  Datasources Subsystem           Never set         Never set             Y   N   N   N   Chart Server                                                            
DataCleansingType                      Vendor1                       String       Subsystem  Infrastructure Datacleansing s  Enterprise level  Enterprise level set  Y   N   N   N   Data Cleansing Type                                                     
DataSource                             ServerDataSrc                 String       Subsystem  Object Manager                  Default value     Default value         Y   N   N   N   OM - Data Source                                                        
DeDupTypeType                          None                          String       Subsystem  Infrastructure DeDuplication s  Enterprise level  Enterprise level set  Y   N   N   N   DeDuplication Data Type                                                 
DefaultAdminAddress                                                  String       Subsystem  SMTP subsystem                  Never set         Never set             Y   N   N   N   Default Administrator Address                                           
DefaultAnalyticsWebServer              Default Analytics Web Server  String       Subsystem  Marketing Server Subsystem      Default value     Default value         Y   N   N   N   Default Analytics Web Server                                            
DefaultFromAddress                                                   String       Subsystem  SMTP subsystem                  Never set         Never set             Y   N   N   N   Default From Address                                                    
DefinedComponent                                                     String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Defined Component                                                       
DisableNotification                    False                         Boolean      Subsystem  Infrastructure Notification     Default value     Default value         N   N   N   N   Disable Notification                                                    
DynCandParam                                                         String       Subsystem  Assignment Subsystem            Never set         Never set             Y   N   N   N   Dynamic Candidate Parameters                                            
EligibilityDisplayMode                 1                             Integer      Subsystem  PSP Engine                      Default value     Default value         Y   N   N   N   Eligibility Display Mode                                                
EmailDebugLevel                        0                             Integer      Subsystem  Email Client Subsystem          Never set         Never set             Y   N   N   N   Email Client Debug Level                                                
EmailDefaultClient                     Siebel Mail Client            String       Subsystem  Email Client Subsystem          Default value     Default value         Y   N   N   N   Default Email Client                                                    
EmailExtMailClientAttDir                                             String       Subsystem  Email Client Subsystem          Never set         Never set             Y   N   N   N   Email Temporary Attachment Location                                     
EmailLotusForm                                                       String       Subsystem  Email Client Subsystem          Never set         Never set             Y   N   N   N   Siebel/Lotus Form                                                       
EmailOutlookForm                                                     String       Subsystem  Email Client Subsystem          Never set         Never set             Y   N   N   N   Siebel/Outlook Form                                                     
EmailPersonalizationFormat             Default Merge Fields          String       Subsystem  Marketing Server Subsystem      Default value     Default value         Y   N   N   N   Email Personalization Format                                            
EnableAssetBasedOrdering               FALSE                         String       Subsystem  OrderManagement                 Default value     Default value         Y   N   N   N   Enable Asset Based Ordering                                             
EnableEAIMemoryMetrics                 False                         Boolean      Subsystem  EAI                             Default value     Default value         Y   N   N   N   Enable Memory Metrics for EAI                                           
EnablePrePickCompatibility             False                         Boolean      Subsystem  OrderManagement                 Default value     Default value         N   N   N   N   Enable Pre Pick Compatibility                                           
EnableTransferCart                     FALSE                         String       Subsystem  Infrastructure Shopping Servic  Default value     Default value         Y   N   N   N   EnableTransferCart                                                      
EnableVirtualHosts                     False                         Boolean      Subsystem  Client Uses Session Manager     Default value     Default value         N   N   Y   N   Enable internal load balancing                                          
EngineID                                                             String       Subsystem  PIMSI Engine                    Never set         Never set             N   Y   N   Y   Engine Id                                                               
ExtDBODBCDataSource                                                  String       Subsystem  External DB Subsystem           Never set         Never set             Y   N   N   N   External DB ODBC Data Source                                            
ExtDBPassword                          ********                      String       Subsystem  External DB Subsystem           Never set         Never set             Y   N   N   N   External DB Password                                                    
ExtDBTableOwner                                                      String       Subsystem  External DB Subsystem           Never set         Never set             Y   N   N   N   External DB Table Owner                                                 
ExtDBUserName                                                        String       Subsystem  External DB Subsystem           Never set         Never set             Y   N   N   N   External DB User Name                                                   
FDRAppendFile                          False                         Boolean      Subsystem  (FDR) Flight Data Recorder      Default value     Default value         N   N   N   N   FDR Periodic Dump and Append                                            
FDRBufferSize                          5000000                       Integer      Subsystem  (FDR) Flight Data Recorder      Default value     Default value         N   Y   N   Y   FDR Buffer Size                                                         
ForwardFlag                                                          String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Forward Flag                                                            
Host                                   siebel8                       String       Server                                     Server level      Server level set      N   N   Y   N   Host Name                                                               
ISSCtxtNumSignals                      150                           Integer      Subsystem  Object Manager                  Default value     Default value         Y   N   N   N   ISS Context - Maximum number of signal objects cached in memory         
ISSCtxtNumVarMaps                      100                           Integer      Subsystem  Object Manager                  Default value     Default value         Y   N   N   N   ISS Context - Maximum number of variable map objects cached in memory   
ISSCtxtSignalSnapshot                  True                          Boolean      Subsystem  Object Manager                  Default value     Default value         Y   N   N   N   ISS Context - Collect and use snapshots of ISS Context signal metadata  
ISSCtxtVarMapSnapshot                  True                          Boolean      Subsystem  Object Manager                  Default value     Default value         Y   N   N   N   ISS Context - Collect and use snapshots of ISS Context variable maps    
Industries                                                           String       Subsystem  Vertical                        Never set         Never set             Y   N   N   N   OM - Industries                                                         
JVMsubsys                              JAVA                          String       Subsystem  EAI                             Default value     Default value         Y   N   N   N   JVM Subsystem Name                                                      
KBName                                 KB                            String       Subsystem  Categorization Manager          Default value     Default value         N   N   N   N   Knowledge Base Name                                                     
KBWriteInterval                        100                           Integer      Subsystem  Categorization Manager          Default value     Default value         N   N   N   N   KB Write Back Interval                                                  
KeepSuccessMessage                     False                         Boolean      Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Keep Successful Tracking Message                                        
KeyFileName                                                          String       Subsystem  Networking                      Never set         Never set             N   N   Y   N   Private key file name                                                   
KeyFilePassword                        ********                      String       Subsystem  Networking                      Never set         Never set             N   N   Y   N   Private key file password                                               
LOYEngineMemberSync                    Y                             String       Subsystem  LoyEngineBatch                  Default value     Default value         Y   N   N   N   LOY - Engine Member Synchronization                                     
LOYEngineNumberOfRuns                  -1                            Integer      Subsystem  LoyEngineBatch                  Default value     Default value         Y   N   N   N   LOY - Engine Number of Runs                                             
LOYEngineNumberofTasks                 0                             Integer      Subsystem  LoyEngineBatch                  Default value     Default value         Y   N   N   N   LOY - Engine Number of Tasks                                            
LOYEngineQueueObjects                  Transaction:500,Bucket:200    String       Subsystem  LoyEngineBatch                  Default value     Default value         Y   N   N   N   LOY - Engine Queue Objects                                              
LOYEngineSearchSpec                                                  String       Subsystem  LoyEngineBatch                  Never set         Never set             Y   N   N   N   LOY - Engine Search Specification                                       
LOYEngineSleepTime                     5                             Integer      Subsystem  LoyEngineBatch                  Default value     Default value         Y   N   N   N   LOY - Engine Sleep Time (secs.)                                         
Lang                                   enu                           String       Subsystem  Infrastructure Core             Server level      Server level set      Y   N   N   N   Language Code                                                           
LoadAtStart                            True                          Boolean      Subsystem  Categorization Manager          Default value     Default value         N   N   N   N   Load KB at Startup                                                      
LocaleCode                                                           String       Subsystem  Object Manager                  Never set         Never set             N   Y   N   Y   Locale Code                                                             
LogArchiveDir                                                        String       Subsystem  Event Logging                   Never set         Never set             Y   N   N   N   Log Archive Directory                                                   
LogDir                                                               String       Subsystem  Event Logging                   Never set         Never set             N   N   N   N   Log directory                                                           
LogMaxSegments                         0                             Integer      Subsystem  Event Logging                   Default value     Default value         N   N   N   N   Maximum number of log file segments                                     
LogSegmentSize                         0                             Integer      Subsystem  Event Logging                   Default value     Default value         N   N   N   N   Log file segment size in KB                                             
MarketingFileSystem                    C:\sba80\gtwysrvr\fs          String       Subsystem  Marketing Server Subsystem      Enterprise level  Enterprise level set  Y   N   N   N   Marketing File System                                                   
MarketingFileSystemForCommOutboundMgr                                String       Subsystem  Marketing Server Subsystem      Never set         Never set             Y   N   N   N   Marketing File System for Communications Outbound Manager               
MarketingWorkflowProcessManager                                      String       Subsystem  Marketing Server Subsystem      Never set         Never set             Y   N   N   N   Marketing Workflow Process Manager                                      
MarketingWorkflowServer                                              String       Subsystem  Marketing Server Subsystem      Never set         Never set             Y   N   N   N   Marketing Workflow Server                                               
MaxNumCat                              4                             Integer      Subsystem  Categorization Manager          Default value     Default value         N   N   N   N   Maximum Number of Categories                                            
MaxThreads                             20                            Integer      Subsystem  Communications Inbound Process  Default value     Default value         Y   N   N   N   Max Threads                                                             
MaximumPageSize                        100                           Integer      Subsystem  EAI                             Default value     Default value         Y   N   N   N   Maximum Page Size                                                       
MemoryBasedRecycle                     False                         Boolean      Subsystem  Multi-Threading                 Default value     Default value         Y   N   N   N   Memory usage based multithread shell recycling                          
MemoryLimit                            1500                          Integer      Subsystem  Multi-Threading                 Default value     Default value         Y   N   N   N   Process VM usage lower limit                                            
MemoryLimitPercent                     20                            Integer      Subsystem  Multi-Threading                 Default value     Default value         Y   N   N   N   Process VM usage upper limit                                            
MessageObjectId                                                      String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Message Tracking Object Id                                              
MessageTracking                        False                         Boolean      Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Message Tracking                                                        
MessageType                                                          String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Message Tracking Type                                                   
MsgBccList                                                           String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Message Bcc List                                                        
MsgBody                                                              String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Message Body                                                            
MsgCcList                                                            String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Message Cc List                                                         
MsgClientAddInCacheRefreshInterval     4320                          Integer      Subsystem  Messaging Client Subsystem      Default value     Default value         Y   N   N   N   Messaging Client AddIn Cache Refresh Interval                           
MsgClientAddInCommModule               EAI                           String       Subsystem  Messaging Client Subsystem      Default value     Default value         Y   N   N   N   Messaging Client AddIn Communication Module                             
MsgClientAddInEAIUrl                                                 String       Subsystem  Messaging Client Subsystem      Never set         Never set             Y   N   N   N   Messaging Client AddIn EAI Url                                          
MsgClientAddInLinkHistory              30                            Integer      Subsystem  Messaging Client Subsystem      Default value     Default value         Y   N   N   N   Messaging Client AddIn Link History                                     
MsgFrom                                                              String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Message From                                                            
MsgHTMLBody                                                          String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Message HTML Body                                                       
MsgReplyToAddressList                                                String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   MessageReplyAddress  List                                               
MsgSubject                                                           String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Message Subject                                                         
MsgToList                                                            String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Message To List                                                         
NotifyHandler                          AdminEmailAlert               String       Subsystem  Infrastructure Notification     Server level      Server level set      N   N   N   N   Notification Handler                                                    
NotifyTimeOut                          100                           Integer      Subsystem  Infrastructure Notification     Default value     Default value         N   N   N   N   Time to wait for doing notification                                     
NumRecipients                                                        String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Number Of Recipients                                                    
NumRetries                             10000                         Integer      Subsystem  Recovery                        Default value     Default value         Y   N   N   N   Number of Retries                                                       
NumTasks                                                             String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Number Of Tasks                                                         
ORCLBatchSize                          0                             Integer      Subsystem  External DB Subsystem           Never set         Never set             Y   N   N   N   ORCL Batch Size                                                         
ORCLGroupSize                          0                             Integer      Subsystem  External DB Subsystem           Never set         Never set             Y   N   N   N   ORCL Group Size                                                         
ORCLQueueName                                                        String       Subsystem  External DB Subsystem           Never set         Never set             Y   N   N   N   ORCL Queue Name                                                         
OrigMsgFile                                                          String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Original Message File                                                   
PSPCacheMaxItemCntLevel1               10000                         Integer      Subsystem  PSP Engine                      Default value     Default value         Y   N   N   N   PSP Level 1 Cache Max Item Count                                        
PSPCacheMaxItemCntLevel2               10000                         Integer      Subsystem  PSP Engine                      Default value     Default value         Y   N   N   N   PSP Level 2 Cache Max Item Count                                        
PackageNameList                                                      String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Comm Template Name List                                                 
Password                               ********                      String       Subsystem  Database Access                 Enterprise level  Enterprise level set  Y   N   N   N   Password                                                                
PeerAuth                               False                         Boolean      Subsystem  Networking                      Default value     Default value         N   N   Y   N   Peer Authentication                                                     
PeerCertValidation                     False                         Boolean      Subsystem  Networking                      Default value     Default value         N   N   Y   N   Validate peer certificate                                               
PersistentShoppingCart                 FALSE                         String       Subsystem  Infrastructure Shopping Servic  Default value     Default value         Y   N   N   N   PersistentShoppingCart                                                  
PostAddToCartLogic                                                   String       Subsystem  Infrastructure Shopping Servic  Never set         Never set             Y   N   N   N   PostAddToCartLogic                                                      
PriceListCacheLifeTime                 -1                            Integer      Subsystem  Infrastructure Pricingcache su  Default value     Default value         Y   N   N   N   Price List Cache Life Time                                              
PriceListItemCacheLifeTime             -1                            Integer      Subsystem  Infrastructure Pricingcache su  Default value     Default value         Y   N   N   N   Price List Item Cache Life Time                                         
PricerMappingCacheSize                 100                           Integer      Subsystem  Infrastructure Pricingcache su  Default value     Default value         Y   N   N   N   Pricer Mapping Cache Size                                               
PricerPriceItemCacheSize               100                           Integer      Subsystem  Infrastructure Pricingcache su  Default value     Default value         Y   N   N   N   Price Item Cache Size                                                   
PricerPriceListCacheSize               20                            Integer      Subsystem  Infrastructure Pricingcache su  Default value     Default value         Y   N   N   N   Price List Cache Size                                                   
PricerPriceModelCacheSize              50                            Integer      Subsystem  Infrastructure Pricingcache su  Default value     Default value         Y   N   N   N   Price Model Cache Size                                                  
PricerVolDisCacheSize                  50                            Integer      Subsystem  Infrastructure Pricingcache su  Default value     Default value         Y   N   N   N   Price Volume discount Cache Size                                        
ProcessMode                            Remote                        String       Subsystem  Communications Outbound Manage  Default value     Default value         N   N   N   N   Process Mode                                                            
ProductDetailView                      NONE                          String       Subsystem  Infrastructure Shopping Servic  Default value     Default value         Y   N   N   N   ProductDetailView                                                       
ProposalReportTimeoutInSecond          1800                          Integer      Subsystem  eDoc Sub System                 Default value     Default value         N   N   N   N   ProposalReportTimeoutInSecond                                           
RTSSleepTime                           30                            Integer      Subsystem  RTS Sub System                  Default value     Default value         Y   N   N   N   RTS Sleep Time                                                          
RecipSearchSpec                                                      String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Recipient Search Spec                                                   
RecipientBusComp                                                     String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Recipient Business Component                                            
RecipientGroup                                                       String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Recipient Group                                                         
RegularAsgn                            True                          Boolean      Subsystem  Assignment Subsystem            Default value     Default value         Y   N   N   N   Regular Assignment                                                      
Repository                             Siebel Repository             String       Subsystem  Database Access                 Default value     Default value         Y   N   N   N   Siebel Repository                                                       
RequestBodyTemplate                                                  String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Request Body Template                                                   
RequestDefaultMedium                                                 String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Request Default Medium                                                  
RequestLanguageCode                                                  String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Language Code                                                           
RequestLocaleCode                                                    String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Locale Code                                                             
RequestName                                                          String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Request Name                                                            
RequestSendFlag                        N                             String       Subsystem  Communications Outbound Manage  Default value     Default value         N   N   N   N   Request Send Flag                                                       
RequestSubjectTemplate                                               String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Request Subject Template                                                
RequestTimeZone                                                      String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Request Time Zone                                                       
ResourceLanguage                                                     String       Subsystem  Object Manager                  Never set         Never set             Y   N   N   N   OM - Resource Language Code                                             
RestrictPrimaryJoin                    False                         Boolean      Subsystem  Datasources Subsystem           Default value     Default value         Y   N   N   N   Restrict Primary Join                                                   
RetryInterval                          5                             Integer      Subsystem  Recovery                        Default value     Default value         Y   N   N   N   Retry Interval                                                          
RetryUpTime                            600                           Integer      Subsystem  Recovery                        Default value     Default value         Y   N   N   N   Retry Up Time                                                           
RootDir                                C:\sba80\siebsrvr             String       Server                                     Server level      Server level set      N   N   Y   N   Siebel Root Directory                                                   
RptMode                                None                          String       Subsystem  Assignment Subsystem            Default value     Default value         Y   N   N   N   Reporting Mode                                                          
RtdApplicationName                                                   String       Subsystem  Marketing Server Subsystem      Never set         Never set             Y   N   N   N   RTD Application Name                                                    
RtdSessionCookieName                   JSESSIONID                    String       Subsystem  Marketing Server Subsystem      Default value     Default value         Y   N   N   N   RTD Session Cookie Name                                                 
RtdSoapURL                             http://CHANGE_ME/rtis/sdwp    String       Subsystem  Marketing Server Subsystem      Default value     Default value         Y   N   N   N   RTD SOAP URL                                                            
RtdWebServiceTimeout                   3000                          Integer      Subsystem  Marketing Server Subsystem      Default value     Default value         Y   N   N   N   RTD WebService Timeout                                                  
SAPReceiverPrtnrNum                    DFLT_PRTNR                    String       Subsystem  SAP                             Default value     Default value         Y   N   N   N   SAP Receiver Partner Number                                             
SAPRfcConnectString                                                  String       Subsystem  SAP                             Never set         Never set             Y   N   N   N   SAP RFC Connect String                                                  
SAPRfcDestEntry                                                      String       Subsystem  SAP                             Never set         Never set             Y   N   N   N   SAP RFC Destination Entry                                               
SAPRfcPassword                         ********                      String       Subsystem  SAP                             Never set         Never set             Y   N   N   N   SAP RFC Password                                                        
SAPRfcUserName                                                       String       Subsystem  SAP                             Never set         Never set             Y   N   N   N   SAP RFC User Name                                                       
SAPSenderPrtnrNum                                                    String       Subsystem  SAP                             Never set         Never set             Y   N   N   N   SAP Sender Partner Number                                               
SARMLevel                              0                             Integer      Subsystem  (SARM) Response Measurement     Default value     Default value         N   N   N   N   SARM Granularity Level                                                  
SMTPServer                                                           String       Subsystem  SMTP subsystem                  Never set         Never set             Y   N   N   N   SMTP Server Name                                                        
SMTPServerPort                         25                            Integer      Subsystem  SMTP subsystem                  Default value     Default value         Y   N   N   N   SMTP Server Port                                                        
SRB ReqId                              0                             String       Subsystem  Infrastructure Core             Default value     Default value         Y   N   N   N   SRB RequestId                                                           
SearchInstallDir                                                     String       Subsystem  Object Manager                  Never set         Never set             Y   N   N   N   Search - Install Directory                                              
SearchRemoteServer                     FALSE                         String       Subsystem  Object Manager                  Default value     Default value         Y   N   N   N   Search - Remote Server                                                  
SearchRemoteServerPath                                               String       Subsystem  Object Manager                  Never set         Never set             Y   N   N   N   Search - Remote Server Path                                             
SecAdptMode                            DB                            String       Subsystem  Security Manager                Enterprise level  Enterprise level set  N   Y   N   Y   Security Adapter Mode                                                   
SecAdptName                            DBSecAdpt                     String       Subsystem  Security Manager                Enterprise level  Enterprise level set  N   Y   N   Y   Security Adapter Name                                                   
Server                                                               String       Server                                     Never set         Never set             Y   N   N   N   Siebel Server Name                                                      
SmqAlwaysAuthenticate                  False                         Boolean      Subsystem  SMQ Transport Subsystem         Default value     Default value         Y   N   N   N   SMQ Always Authenticate                                                 
SmqCompression                         GZIP                          String       Subsystem  SMQ Transport Subsystem         Default value     Default value         Y   N   N   N   SMQ Compression Algorithm                                               
SmqEncryption                          RC4                           String       Subsystem  SMQ Transport Subsystem         Default value     Default value         Y   N   N   N   SMQ Encryption Algorithm                                                
SourceBusObj                                                         String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Source Business Object                                                  
SourceIdList                                                         String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Source Id List                                                          
TableOwnPass                           ********                      String       Subsystem  Database Access                 Never set         Never set             Y   N   N   N   Table Owner Password                                                    
TableOwner                             SIEBEL                        String       Subsystem  Database Access                 Enterprise level  Enterprise level set  Y   N   N   N   Table Owner                                                             
TaskRecipMin                                                         String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Task Recipient Minimum                                                  
TaskStartDate                                                        String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Task Start Date                                                         
TblSpace                                                             String       Subsystem  Database Access                 Never set         Never set             Y   N   N   N   Tablespace Name                                                         
TestAddress                                                          String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Test Address                                                            
TestExecutableSearchPath                                             String       Subsystem  Testing Subsystem               Never set         Never set             Y   N   N   N   Test Exectuable Search Path                                             
TestExecuteMethodName                                                String       Subsystem  Testing Subsystem               Never set         Never set             Y   N   N   N   Test Execute Method Name                                                
TestExecuteServiceName                                               String       Subsystem  Testing Subsystem               Never set         Never set             Y   N   N   N   Test Execute Service Name                                               
TestInputDataSearchPath                                              String       Subsystem  Testing Subsystem               Never set         Never set             Y   N   N   N   Test Input Data Search Path                                             
TestResultsSearchPath                                                String       Subsystem  Testing Subsystem               Never set         Never set             Y   N   N   N   Test Results Search Path                                                
TestScriptSearchPath                                                 String       Subsystem  Testing Subsystem               Never set         Never set             Y   N   N   N   Test Script Search Path                                                 
TestSuiteFileName                                                    String       Subsystem  Testing Subsystem               Never set         Never set             Y   N   N   N   Test Suite File Name                                                    
TestSuiteTestName                                                    String       Subsystem  Testing Subsystem               Never set         Never set             Y   N   N   N   Test Suite Test Name                                                    
TestSuiteWriteResults                  True                          Boolean      Subsystem  Testing Subsystem               Default value     Default value         Y   N   N   N   Test Suite Write Results                                                
UCMBatchObjectType                     Contact                       String       Subsystem  Universal Customer Master Subs  Default value     Default value         N   N   N   N   UCM Batch Object Type                                                   
UCMBatchSize                           10                            Integer      Subsystem  Universal Customer Master Subs  Default value     Default value         N   N   N   N   UCM Batch Size                                                          
UCMCDMCleanseFlag                      False                         Boolean      Subsystem  Universal Customer Master Subs  Default value     Default value         N   N   N   N   UCM Data Management Cleanse Flag                                        
UCMCDMExactMatchFlag                   False                         Boolean      Subsystem  Universal Customer Master Subs  Default value     Default value         N   N   N   N   UCM Data Management Exact Match Flag                                    
UCMCDMMatchFlag                        False                         Boolean      Subsystem  Universal Customer Master Subs  Default value     Default value         N   N   N   N   UCM Data Management Match Flag                                          
UCMPubSubFlag                          False                         Boolean      Subsystem  Universal Customer Master Subs  Default value     Default value         N   N   N   N   UCM Publish/Subscribe Flag                                              
UCMSearchSpec                                                        String       Subsystem  Universal Customer Master Subs  Never set         Never set             N   N   N   N   UCM Search Specification                                                
UCMSleepTime                           60                            Integer      Subsystem  Universal Customer Master Subs  Default value     Default value         N   N   N   N   UCM Sleep Time                                                          
UCMSortSpec                                                          String       Subsystem  Universal Customer Master Subs  Never set         Never set             N   N   N   N   UCM Sort Specification                                                  
UCMSurvivorshipEngineFlag              False                         Boolean      Subsystem  Universal Customer Master Subs  Default value     Default value         N   N   N   N   UCM Survivorship Engine Flag                                            
UpdateInterval                         1440                          Integer      Subsystem  Categorization Manager          Default value     Default value         N   N   N   N   KB Update Interval                                                      
UpgComponent                           Siebel HQ Server              String       Server                                     Default value     Default value         Y   N   N   N   Upgrade Component                                                       
UseKeyVal                              None                          String       Subsystem  Assignment Subsystem            Default value     Default value         Y   N   N   N   Use Key Value                                                           
Username                               SADMIN                        String       Subsystem  Database Access                 Enterprise level  Enterprise level set  Y   N   N   N   User Name                                                               
Vertical                                                             String       Subsystem  Vertical                        Never set         Never set             Y   N   N   N   OM - Vertical                                                           
VirtualHostsFile                                                     String       Subsystem  Client Uses Session Manager     Never set         Never set             N   N   Y   N   Session manager load balancing configuration file                       
WaveBatchSize                          10000                         Integer      Subsystem  Marketing Server Subsystem      Default value     Default value         Y   N   N   N   Wave Batch Size                                                         
WebCollabEnable                        False                         Boolean      Subsystem  Web Collaboration Subsystem     Default value     Default value         Y   N   N   N   Web Collab Enable                                                       
WebCollabEnableSimulation              False                         Boolean      Subsystem  Web Collaboration Subsystem     Default value     Default value         Y   N   N   N   Web Collab Enable Simulation                                            
WebCollabLangCodeMap                   PIXION_LANGCODE_MAP           String       Subsystem  Web Collaboration Subsystem     Default value     Default value         Y   N   N   N   Web Collab Language Code Map                                            
WebCollabLogFile                       WebCollab.log                 String       Subsystem  Web Collaboration Subsystem     Default value     Default value         Y   N   N   N   Web Collab Log File                                                     
WebCollabLogLevel                      0                             Integer      Subsystem  Web Collaboration Subsystem     Default value     Default value         Y   N   N   N   Web Collab Log Level                                                    
WebCollabServer                        CHANGE_ME                     String       Subsystem  Web Collaboration Subsystem     Default value     Default value         Y   N   N   N   Web Collab Server                                                       
WebCollabType                          Siebel eCollaboration         String       Subsystem  Web Collaboration Subsystem     Default value     Default value         Y   N   N   N   Web Collab Package                                                      
WebCollabUseSiebelSessionId            True                          Boolean      Subsystem  Web Collaboration Subsystem     Default value     Default value         Y   N   N   N   Web Collab Using Siebel Session Id                                      
WebCollabUseSiebelUQ                   True                          Boolean      Subsystem  Web Collaboration Subsystem     Default value     Default value         Y   N   N   N   Web Collab Using Siebel UQ                                              
WebServer                                                            String       Subsystem  Communications Outbound Manage  Never set         Never set             N   N   N   N   Web Server                                                              
XMLPReportDataDir                      /xmlp/data/                   String       Subsystem  Infrastructure XMLP Reports Su  Default value     Default value         Y   N   N   N   XMLP Report Data.cfg Dir                                                
XMLPReportOutputDir                    /xmlp/reports/                String       Subsystem  Infrastructure XMLP Reports Su  Default value     Default value         Y   N   N   N   XMLP Report Output.cfg Dir                                              
XMLPReportWaitTime                     10                            Integer      Subsystem  Infrastructure XMLP Reports Su  Default value     Default value         Y   N   N   N   XMLP Report Wait Time Limit                                             
XMLPReportXdoDir                       /xmlp/templates/              String       Subsystem  Infrastructure XMLP Reports Su  Default value     Default value         Y   N   N   N   XMLP Report xdo.cfg Dir                                                 
eProdCfgKeepAliveTime                  900                           Integer      Subsystem  Object Manager                  Default value     Default value         Y   N   N   N   Product Configurator - Keep Alive Time of Idle Session                  
eProdCfgMaxNumbOfWorkerReuses          10                            Integer      Subsystem  Object Manager                  Default value     Default value         Y   N   N   N   Product Configurator - Number of Reuse for each Worker                  
eProdCfgNumbOfCachedFactories          10                            Integer      Subsystem  Object Manager                  Default value     Default value         Y   N   N   N   Product Configurator - Number of Factories Cached in Memory             
eProdCfgRemote                         False                         Boolean      Subsystem  Object Manager                  Default value     Default value         Y   N   N   N   Product Configurator - Use Remote Service                               
eProdCfgServer                                                       String       Subsystem  Object Manager                  Never set         Never set             Y   N   N   N   Product Configurator - Remote Server Name                               
eProdCfgWorkerMemReleaseFlg            False                         Boolean      Subsystem  Object Manager                  Default value     Default value         Y   N   N   N   Product Configurator - Release Worker Memory at the end of session      

258 rows returned.

srvrmgr:SUsrvr> list params for component SRProc

PA_ALIAS                 PA_VALUE           PA_DATATYPE  PA_SCOPE   PA_SUBSYSTEM                 PA_SETLEVEL       PA_DISP_SETLEVEL                PA  PA  PA  PA  PA_NAME                                         
-----------------------  -----------------  -----------  ---------  ---------------------------  ----------------  ------------------------------  --  --  --  --  ----------------------------------------------  
CACertFileName                              String       Subsystem  Networking                   Never set         Never set                       N   N   Y   N   CA certificate file name                        
CertFileName                                String       Subsystem  Networking                   Never set         Never set                       N   N   Y   N   Certificate file name                           
ConnIdleTime             -1                 Integer      Subsystem  Networking                   Default value     Default value                   Y   N   N   N   SISNAPI connection maximum idle time            
Connect                  Siebel_DSN         String       Subsystem  Database Access              Enterprise level  Enterprise level set            Y   N   N   N   ODBC Data Source                                
DisableNotification      False              Boolean      Subsystem  Infrastructure Notification  Default value     Default value                   N   N   N   N   Disable Notification                            
EnableDbSessCorrelation  False              Boolean      Subsystem  Database Access              Default value     Default value                   Y   N   N   N   Enable Database Session Correlation             
EnableHousekeeping       False              Boolean      Component                               Default value     Default value                   N   N   N   N   Enable Various Housekeeping Tasks               
EnableUsageTracking      False              Boolean      Subsystem  Usage Tracking               Default value     Default value                   N   N   N   N   UsageTracking Enabled                           
FDRAppendFile            False              Boolean      Subsystem  (FDR) Flight Data Recorder   Default value     Default value                   N   N   N   N   FDR Periodic Dump and Append                    
FDRBufferSize            5000000            Integer      Subsystem  (FDR) Flight Data Recorder   Default value     Default value                   N   Y   N   Y   FDR Buffer Size                                 
KeyFileName                                 String       Subsystem  Networking                   Never set         Never set                       N   N   Y   N   Private key file name                           
KeyFilePassword          ********           String       Subsystem  Networking                   Never set         Never set                       N   N   Y   N   Private key file password                       
Lang                     enu                String       Subsystem  Infrastructure Core          Server level      Server level set                Y   N   N   N   Language Code                                   
LogArchiveDir                               String       Subsystem  Event Logging                Never set         Never set                       Y   N   N   N   Log Archive Directory                           
LogDir                                      String       Subsystem  Event Logging                Never set         Never set                       N   N   N   N   Log directory                                   
LogFileDir               c:\temp            String       Subsystem  Usage Tracking               Default value     Default value                   N   N   N   N   UsageTracking LogFile Dir                       
LogFileEncoding          ASCII              String       Subsystem  Usage Tracking               Default value     Default value                   N   N   N   N   UsageTracking LogFile Encoding                  
LogFileFormat            XML                String       Subsystem  Usage Tracking               Default value     Default value                   N   N   N   N   UsageTracking LogFile Format                    
LogFilePeriod            Hourly             String       Subsystem  Usage Tracking               Default value     Default value                   N   N   N   N   UsageTracking LogFile Period                    
LogMaxSegments           100                Integer      Subsystem  Event Logging                Compdef level     Component definition level set  N   N   N   N   Maximum number of log file segments             
LogSegmentSize           100                Integer      Subsystem  Event Logging                Compdef level     Component definition level set  N   N   N   N   Log file segment size in KB                     
MaxMTServers             1                  Integer      Subsystem  Multi-Threading              Compdef level     Component definition level set  N   N   Y   N   Maximum MT Servers                              
MaxTasks                 20                 Integer      Subsystem  Process Management           Compdef level     Component definition level set  N   N   Y   N   Maximum Tasks                                   
MemoryBasedRecycle       False              Boolean      Subsystem  Multi-Threading              Default value     Default value                   Y   N   N   N   Memory usage based multithread shell recycling  
MemoryLimit              1500               Integer      Subsystem  Multi-Threading              Default value     Default value                   Y   N   N   N   Process VM usage lower limit                    
MemoryLimitPercent       20                 Integer      Subsystem  Multi-Threading              Default value     Default value                   Y   N   N   N   Process VM usage upper limit                    
MinMTServers             1                  Integer      Subsystem  Multi-Threading              Compdef level     Component definition level set  N   N   Y   N   Minimum MT Servers                              
NotifyHandler            AdminEmailAlert    String       Subsystem  Infrastructure Notification  Server level      Server level set                N   N   N   N   Notification Handler                            
NotifyTimeOut            100                Integer      Subsystem  Infrastructure Notification  Default value     Default value                   N   N   N   N   Time to wait for doing notification             
NumRetries               10000              Integer      Subsystem  Recovery                     Default value     Default value                   Y   N   N   N   Number of Retries                               
Password                 ********           String       Subsystem  Database Access              Enterprise level  Enterprise level set            Y   N   N   N   Password                                        
PeerAuth                 False              Boolean      Subsystem  Networking                   Default value     Default value                   N   N   Y   N   Peer Authentication                             
PeerCertValidation       False              Boolean      Subsystem  Networking                   Default value     Default value                   N   N   Y   N   Validate peer certificate                       
Repository               Siebel Repository  String       Subsystem  Database Access              Default value     Default value                   Y   N   N   N   Siebel Repository                               
RetryInterval            5                  Integer      Subsystem  Recovery                     Default value     Default value                   Y   N   N   N   Retry Interval                                  
RetryUpTime              600                Integer      Subsystem  Recovery                     Default value     Default value                   Y   N   N   N   Retry Up Time                                   
SARMLevel                0                  Integer      Subsystem  (SARM) Response Measurement  Default value     Default value                   N   N   N   N   SARM Granularity Level                          
SRB ReqId                0                  String       Subsystem  Infrastructure Core          Default value     Default value                   Y   N   N   N   SRB RequestId                                   
SessionCacheSize         100                Integer      Subsystem  Usage Tracking               Default value     Default value                   N   N   N   N   UsageTracking Session Cache Size                
TableOwnPass             ********           String       Subsystem  Database Access              Never set         Never set                       Y   N   N   N   Table Owner Password                            
TableOwner               SIEBEL             String       Subsystem  Database Access              Enterprise level  Enterprise level set            Y   N   N   N   Table Owner                                     
TblSpace                                    String       Subsystem  Database Access              Never set         Never set                       Y   N   N   N   Tablespace Name                                 
UserList                                    String       Subsystem  Event Logging                Never set         Never set                       N   N   N   N   List of users                                   
Username                 SADMIN             String       Subsystem  Database Access              Enterprise level  Enterprise level set            Y   N   N   N   User Name                                       

44 rows returned.

srvrmgr:SUsrvr> list comp def SRProc

CC_NAME                                                                       CT_NAME                                                                       CC_RUNMODE                       CC_ALIAS                         CC_DISP_ENABLE_ST                                              CC_DESC_TEXT                                                                                                                                                                                                                                                 CG_NAME                                                                       CG_ALIAS                         CC_INCARN_NO             
----------------------------------------------------------------------------  ----------------------------------------------------------------------------  -------------------------------  -------------------------------  -------------------------------------------------------------  --------------------------------------------------------------------------------------------------------------------  ----------------------------------------------------------------------------  -------------------------------  -----------------------  
Server Request Processor                                                      Server Request Processor (SRP)                                                Interactive                      SRProc                           Active                                                         Server Request scheduler and request/notification store and forward processor                                                                                                                                                                                Auxiliary System Management                                                   SystemAux                        0                        

1 row returned.

