package SLUB::LZA::Rosetta::TA::Command::count;
use SLUB::LZA::Rosetta::TA -command;
use strict;
use warnings;
use feature qw(say);
use SLUB::LZA::Rosetta::TA::common_sru;
use SLUB::LZA::Rosetta::TA::SRU;


# VERSION

# ABSTRACT: count IEs module for ta-tool


sub abstract {"count IEs in Rosetta based Archival Information System";}

my $description=<<"DESCR";
Searches a Rosetta-based AIS for descriptive oder source metadata on behalf of the Technical Analyst
and return counts of matches.

Examples:

  * Is this dc identifier in Archiv?
    $0 count -d SLUB:LZA:Kitodo:kitodo:422766
  * How many IEs were modified in 2021-05-31?
    $0 count -m 2021-05-31
  * How many IEs have at least one invalid file?
    $0 count --with-invalid-files
  * How many IEs with mkv-files are in archive?
    $0 count --with-format=fmt/569
  * How many IEs have at least one file without virus check?
    $0 count --with-missed-viruscheck
DESCR

sub description {
    "$description"
}
sub opt_spec {
    return(
        ["verbose|v" => "enable verbose output"],
        ["debug" => "enable user agent debug output"],
        ["datemode" => hidden => {one_of => [
            ["creationdate|c=s" => "search based on creationdate string"],
            ["modificationdate|m=s" => "search based on modificationdate string"]
        ] } ],
        ["descriptive|d=s" => "count based on string search in descriptive metadata"],
        ["source|s=s" => "count based on string search in source metadata"],
        ["ie|i=s" => "search a specific IE"],
        ["format" => hidden => {one_of => [
            ["with-format=s" => "with pronom format id"],
            # unsupported by Rosetta <= 7.0
            #["without-format=s" => "without pronom format id"],
        ] } ],
        ["validation" => hidden => {one_of => [
            ["with-valid-files" => "with valid files"],
            ["with-invalid-files" => "with at least one invalid file"],
        ] } ],
        ["viruscheck" => hidden => {one_of => [
            ["with-passed-viruschecks" => "with passed virus checks"],
            ["with-missed-viruschecks" => "with missed virus checks"],
            ["with-failed-viruschecks" => "with failed virus checks"],
        ]}],
    );
}
sub validate_args {
    my ($self, $opt, $args) = @_;
    # no args allowed but options!
    $self->usage_error("No args allowed") if @$args;
    1;
}

sub execute {
    my ($self, $opt, $args) = @_;
    my $maxrecords="0";
    my $startrecord=1;
    my $query = SLUB::LZA::Rosetta::TA::common_sru::prepare_query($opt);
    my $response = SLUB::LZA::Rosetta::TA::SRU::sru_search('ie', $query, $startrecord, $maxrecords, $opt->{verbose}, $opt->{debug});
    $response=~s|.*?<numberOfRecords>(\d+)</numberOfRecords.*|$1|s;
    say $response;
}

1;
