functor Events (Par_select: PAR_SELECT) : EVENTS =
    struct
	local 
	    open Par_select
	in

	    (** Miscellaneous **)

	    exception Sync

	    val wait = Par_select.wait

	    fun iterate_apply f l =
		let fun itap m [] = ()
		      | itap m (x::rest) = (f (x,m);
					    itap (m+1) rest)
		in
		    itap 0 l
		end

	    fun iterate_map f l =
		let fun itmap m [] = []
		      | itmap m (x::rest) = (f (x,m))::(itmap (m+1) rest)
		in
		    itmap 0 l
		end

	    (** Processes **)
		
	    type procid = {pid:int,isDead:bool ref,
			   procLock:mutex,waiters:condition}
	    
	    type proc = (unit -> unit) (* why do we need this? *)

	    val init_procid =
		{pid=0,
		 isDead=ref false,
		 procLock=mutex(),
		 waiters=condition()}

	    val current_procid = var init_procid

	    val procid_counter = ref 0

	    fun getpid () = get(current_procid)

	    fun pid2string ({pid,...}: procid) =
		implode ["[",(makestring pid),"]"]

	    fun process_die () =
		let val {isDead,procLock,waiters,...} = getpid() in
		    (with_mutex procLock (fn () =>
					  (isDead := true;
					   broadcast waiters));
		     exit())
		end

	    fun process f = 
		(let val pid_cntr = (!procid_counter) + 1
		     val cond = condition()
		     val isDead = ref false
		     val lock = mutex()
		     val child = {pid=pid_cntr,
				  isDead=isDead,
				  procLock=lock,
				  waiters=cond}
		 in
		     procid_counter := pid_cntr;
		     fork (fn () => 
			   (set current_procid child;
			    (f() handle exn =>
				(print (pid2string (getpid()));
				 print " uncaught exception ";
				 print (System.exn_name exn);
				 print "\n";
				 process_die()));
				process_die()));
		     child
		 end)
		 

	    val yield = Par_select.yield

	    (** Channels **)
	
	    fun reverse(nil,rl) = rl
	      | reverse (x::rest, rl) = reverse(rest,x::rl)


	    datatype event_state = 
		UNCOMMITTED 
	      | TENT_COMMITTED 
	      | COMMITTED of int

	    datatype eventRec = EVT_REC of {evtLock: mutex ref,
					    evtState: event_state ref,
					    evtCond: condition ref,
					    evtIndex: int}

	    datatype 'a chanq = CHANQ of 
		{front : (eventRec * 'a) list ref,
		 rear  : (eventRec * 'a) list ref }

	    fun newq () = CHANQ{front= ref nil, rear= ref nil}
		
	    fun insert (CHANQ{front,rear}, item) = 
		(case (!front) of
		     nil => (front := reverse(!rear, [item]);
			     rear := nil)
		   | _ => rear := (item :: !rear))

	    exception GetEvent

	    fun getEvent (CHANQ{front,rear}) =
		     let fun getEvt nil = (NONE,nil)
			   | getEvt ((evtRec as 
				      (EVT_REC{evtLock,evtState,...},_))
				     ::rest) =
			     (acquire (!evtLock);
			      case (!evtState) of
				  COMMITTED(_) => (release (!evtLock);
						   getEvt rest)
				| _ => (SOME(evtRec),rest))
		     in
			 case (getEvt (!front)) of
			     (NONE,nil) =>
				 let val (result,rest) = 
				     getEvt(reverse (!rear,nil)) in
				     (front := rest;
				      result)
				 end
			   | (result,r) =>
				 (front := r;
				  result)
		     end
	    datatype 'a chan = CHAN of 
		{chanLock : mutex,
		 senders  : 'a chanq,
		 receivers: ('a option ref) chanq}

	    fun channel() = CHAN{chanLock  = mutex(),
				 senders   = newq(),
				 receivers = newq()}

	    (** Events **)
	    
	    datatype evt_sts = EVT_ANY | EVT_READY | EVT_BLOCK

	    datatype 'a base_evt = BASE_EVT of {pollEvent : (unit -> evt_sts),
						abortEvent: (unit -> unit),
						doEvent   : (unit -> 'a),
						blockEvent: (eventRec -> 'a)}

	    datatype 'a event = EVT of 'a base_evt list


	    local 

		fun extract nil pollResults = pollResults
		  | extract 
		    ((evt as BASE_EVT {pollEvent,...})::rest)
		    ((rdy,n1),(any,n2),(block,n3)) =
		    (case (pollEvent()) of
			 EVT_ANY => 
			     extract rest 
			     ((rdy,n1),((evt::any),n2+1),(block,n3))
		       | EVT_BLOCK =>
			     extract rest 
			     ((rdy,n1),(any,n2),((evt::block),n3+1))
		       | EVT_READY =>
			     extract rest 
			     (((evt::rdy),n1+1),(any,n2),(block,n3)))

		val cnt = ref 0

		fun random i = ((!cnt mod i) before (inc cnt))

		fun doIt (BASE_EVT{doEvent,...}) = 
		    doEvent()

		fun abortIt (BASE_EVT{abortEvent,...}) = 
		    abortEvent()

		fun select (l,n) =
		    let val rand = random n
			val result = doIt(nth (l,rand)) 
		    in
			(iterate_apply (fn (evt,index) =>
				       if (index=rand) then 
					   () 
				       else
					   abortIt(evt)) 
			 l);
			result
		    end

		fun blockEvts evtlist =
		    let val chsLock = ref (mutex())
			val chsState = ref UNCOMMITTED
			val chsCond = ref (condition())
			fun blockIt (BASE_EVT{blockEvent,...},index) =
			    (blockEvent,EVT_REC{evtLock=chsLock,
						evtState=chsState,
						evtCond=chsCond,
						evtIndex=index})
		    in
			par_select (iterate_map blockIt evtlist)
		    end
			
	    in
		
		(* sync : 'a event -> 'a *)
		fun sync (EVT el) = 
		    (case el of
			 nil => exit()
		       | el =>  
			     let val ((rdy,numRdy),(any,numAny),
				      (block,numBlock)) = 
				 extract el ((nil,0),(nil,0),(nil,0))
			     in
				 case (rdy) of
				     nil =>
					 (case (any) of
					      nil => blockEvts el
					    | _ => select (any,numAny))
		                   | _ =>(app abortIt any;
					  select (rdy,numRdy))
			     end)

		fun wrap (EVT el, f) =
		    let fun wrap' (BASE_EVT{pollEvent,
					   abortEvent,
					   doEvent,
					   blockEvent}) =
			let fun doEvent' () = f(doEvent())
			    fun blockEvent' evtRec = f(blockEvent(evtRec))
			in
			    BASE_EVT{pollEvent  = pollEvent,
				     abortEvent = abortEvent,
				     doEvent    = doEvent',
				     blockEvent = blockEvent'}
			end
		    in
			EVT(map wrap' el)
		    end

	    end

	    fun choose evts = 
		let fun choose' (nil,nil,el) = el
		      | choose' ((EVT evt)::rest,nil,el) = 
			choose' (rest,evt,el)
		      | choose' (evts,e::rest,el) = choose'(evts,rest,e::el)
		in
		    EVT (choose'(evts,nil,nil))
		end

	    (* Base events **)
	    
	    val anyevent = 
		EVT[BASE_EVT{pollEvent= (fn () => EVT_ANY),
			     abortEvent=(fn () => ()),
			     doEvent=   (fn () => ()),
			     blockEvent=(fn(evtRec) => ())}]

	    val rdyevent =
		EVT[BASE_EVT{pollEvent= (fn () => EVT_READY),
			     abortEvent=(fn () => ()),
			     doEvent=   (fn () => ()),
			     blockEvent=(fn(evtRec) => ())}]

	    val noevent = EVT[]

	    exception Transmit

	    fun transmit (msg,CHAN{chanLock,receivers,senders}) =

		let val rcvr = ref NONE

		    fun pollEvent () =
		    if (try_acquire chanLock) then
			(case (getEvent receivers) of 
			     NONE => (release chanLock;
				      EVT_BLOCK)
			   | SOME(r as 
				  (EVT_REC{evtLock,evtState,...},_)) =>
				 (case (!evtState) of
				      UNCOMMITTED => 
					  (evtState := TENT_COMMITTED;
					   rcvr := SOME(r);
					   release (!evtLock);
					   EVT_READY)
				    | TENT_COMMITTED =>
					  (release (!evtLock);
					   release chanLock;
					   EVT_BLOCK)
				    | COMMITTED(_) =>
					  (release (!evtLock);
					   release chanLock;
					   raise GetEvent)))
		    else EVT_BLOCK

		    fun abortEvent () =
			case (!rcvr) of
			    NONE => raise Transmit
			  | SOME(EVT_REC{evtLock,evtState,evtCond,...},_) =>
				(with_mutex (!evtLock) 
				 (fn () => 
				  (rcvr := NONE;
				   evtState := UNCOMMITTED;
				   broadcast(!evtCond)));
				 release chanLock)

		    fun doEvent () =
			case (!rcvr) of
			    NONE => raise Transmit
			  | SOME
			    (EVT_REC{evtLock,evtState,evtCond,evtIndex},
			     msgSlot)=>
			    (with_mutex 
			      (!evtLock) 
			      (fn () =>
			       (rcvr := NONE;
				evtState := COMMITTED(evtIndex);
				msgSlot := SOME(msg);
				broadcast(!evtCond)));
			     release chanLock)

		    fun blockEvent (evtRec as 
				    EVT_REC{evtLock,
					    evtState,
					    evtCond,
					    evtIndex}) =

			let fun blockPoll() =
			    case (getEvent receivers) of
				NONE => EVT_BLOCK
			      | SOME(r as 
				     (EVT_REC{evtLock,evtState,...},_)) 
				=>
				    (case (!evtState) of
					 UNCOMMITTED =>
					     (evtState := TENT_COMMITTED;
					      rcvr := SOME(r);
					      release(!evtLock);
					      EVT_READY)
				       | TENT_COMMITTED =>
					     (wait (!evtLock) (!evtCond);
					      release(!evtLock);
					      blockPoll())
				       | COMMITTED(_) =>
					     (release chanLock;
					      release (!evtLock);
					      raise GetEvent))
					     
			    fun stateCheck () = 
				case (!evtState) of
				    COMMITTED(_) => 
					(release (!evtLock);
					 exit())
				  | TENT_COMMITTED =>
					(wait (!evtLock) (!evtCond);
					 stateCheck())
				  | UNCOMMITTED => ()

			in
			    acquire(!evtLock);
			    stateCheck();
			    acquire chanLock;
			    case (blockPoll()) of
				EVT_READY => 
				    (evtState := COMMITTED(evtIndex);
				     broadcast (!evtCond);
				     release(!evtLock);
				     doEvent())
			      | _ =>
				    let val sender = (evtRec,msg)
					fun blockWait () =
					    (wait (!evtLock) (!evtCond);
					     case (!evtState) of
						 COMMITTED(index) =>
						     if (index=evtIndex) then
							 (release (!evtLock);
							  ())
						     else
							 (release (!evtLock);
							  exit())
					       | _ => blockWait())
				    in
					insert(senders,sender);
					release chanLock;
					blockWait()
				    end
			end
		in
		    EVT [BASE_EVT{pollEvent=pollEvent,
				  abortEvent=abortEvent,
				  doEvent=doEvent,
				  blockEvent=blockEvent}]
		end

	    exception Receive

	    fun receive (CHAN{chanLock,receivers,senders}) =

		let val sndr = ref NONE
		     
		    fun pollEvent () =
			if (try_acquire chanLock) then
			    (case (getEvent senders) of
				 NONE => (release chanLock;
					  EVT_BLOCK)
			       | SOME(s as 
				      (EVT_REC{evtLock,evtState,...},_)) =>
				 (case (!evtState) of
				      UNCOMMITTED =>
					  (evtState := TENT_COMMITTED;
					   sndr := SOME(s);
					   release (!evtLock);
					   EVT_READY)
				    | TENT_COMMITTED =>
					  (release (!evtLock);
					   release chanLock;
					   EVT_BLOCK)
				    | COMMITTED(_) =>
					  (release (!evtLock);
					   release chanLock;
					   raise GetEvent)))
		    else EVT_BLOCK

		    fun abortEvent () =
			case (!sndr) of
			    NONE => raise Receive
			  | SOME(EVT_REC{evtLock,evtState,evtCond,...},_) =>
				(with_mutex 
				 (!evtLock) 
				 (fn () =>
				  (sndr := NONE;
				   evtState := UNCOMMITTED;
				   broadcast(!evtCond)));
				 release chanLock)

		    fun doEvent () =
			case (!sndr) of
			    NONE => raise Receive
			  | SOME(EVT_REC{evtLock,evtState,evtCond,evtIndex},
				 msg) =>
				(with_mutex (!evtLock) 
				 (fn () =>
				  (sndr := NONE;
				   evtState := COMMITTED(evtIndex);
				   broadcast(!evtCond)));
				 release chanLock;
				 msg)

		    fun blockEvent (evtRec as 
				    EVT_REC{evtLock,
					    evtState,evtCond,evtIndex}) =

			let fun blockPoll () =
			    case (getEvent senders) of
				NONE => EVT_BLOCK
			      | SOME(s as 
				     (EVT_REC{evtLock,evtState,...},_)) 
				=>
				     (case (!evtState) of
					  UNCOMMITTED =>
					      (evtState := TENT_COMMITTED;
					       sndr := SOME(s);
					       release(!evtLock);
					       EVT_READY)
					| TENT_COMMITTED =>
					      (wait (!evtLock) (!evtCond);
					       release (!evtLock);
					       blockPoll())
					| COMMITTED(_) =>
					      (release chanLock;
					       release (!evtLock);
					       raise GetEvent))

			    fun stateCheck () =
				case (!evtState) of
				    COMMITTED(_) =>
					(release (!evtLock);
					 exit())
				  | TENT_COMMITTED =>
					(wait (!evtLock) (!evtCond);
					 stateCheck())
				  | UNCOMMITTED => ()
			    
			in
			    acquire (!evtLock);
			    stateCheck();
			    acquire chanLock;
			    case (blockPoll()) of
				EVT_READY =>
				    (evtState := COMMITTED(evtIndex);
				     broadcast (!evtCond);
				     release (!evtLock);
				     doEvent())
			      | _ =>
				    let val msgSlot = ref NONE
					val receiver = (evtRec,msgSlot)
					fun blockWait () =
					    (wait (!evtLock) (!evtCond);
					     case (!evtState) of
						 COMMITTED(index) =>
						     if (index=evtIndex) then
							 (release (!evtLock);
							  case (!msgSlot) of
							      NONE => 
								  raise Receive
							    | SOME(msg) =>
								  msg)
						     else
							 (release (!evtLock);
							  exit())
					       | _ => blockWait())
				    in
					insert(receivers,receiver);
					release chanLock;
					blockWait()
				    end
			end
		in
		   EVT [BASE_EVT{pollEvent=pollEvent,
				 abortEvent=abortEvent,
				 doEvent=doEvent,
				 blockEvent=blockEvent}]
		end

	    fun wait' {pid,isDead,procLock,waiters} =
		let fun pollEvent() =
		    if (try_acquire procLock) then  
			(if (!isDead) then EVT_READY else 
			     (release procLock;
			      EVT_BLOCK))
		    else EVT_BLOCK

		    fun abortEvent () = (release procLock)
		    fun doEvent () = (release procLock)

		    fun blockEvent (evtRec as
				    EVT_REC{evtLock,evtState,
					    evtCond,evtIndex}) =
			let fun blockPoll () =
			    (if (!isDead) then EVT_READY else EVT_BLOCK)

			    fun stateCheck () =
				case (!evtState) of
				    COMMITTED(_) =>
					(release (!evtLock);
					 exit())
				  | TENT_COMMITTED =>
					(wait (!evtLock) (!evtCond);
					 stateCheck())
				  | UNCOMMITTED => ()
			in
			    acquire(!evtLock);
			    stateCheck();
			    acquire procLock;
			    case (blockPoll()) of
				EVT_READY => 
				    (evtState := COMMITTED(evtIndex);
				     broadcast (!evtCond);
				     release(!evtLock);
				     doEvent())
			      | _ =>
				    (release (!evtLock);
				     wait (procLock) waiters;
				     release procLock;
				     blockEvent evtRec)
			end
		in
		    EVT[BASE_EVT{pollEvent=pollEvent,
				 abortEvent=abortEvent,
				 doEvent=doEvent,
				 blockEvent=blockEvent}]
		end

	    fun reset () = (Par_select.reset();
			    set current_procid init_procid;
			    procid_counter := 0)

	    fun send (msg,chan) = sync(transmit(msg,chan))

	    fun accept chan = sync(receive(chan))

	end (*local*)

    end (*struct*)


		 
