(* Copyright (c) 1991 by Carnegie Mellon University *)

functor BatchC(val statistics : unit -> unit
	       val generate : CPS.function * ErrorMsg.complainer * string *
		              outstream -> unit
	       structure Print : CPRINT
	       structure Control : CONTROL
	       structure CProcessFile : CPROCESSFILE) : sig end =
struct

open CProcessFile
val pr = outputc std_out

(* set up environment *)

local
  val RESULT {entrypoints,...} =  bootEnv (load (fn i=>i))
in
  val entryPoints = ref entrypoints
end

local
   val env = Env.closeCurrentNewEnv()
in val pervasiveEnv = ref env
   val _ = (Env.resetEnv env; Env.commit())
end

(* uniq: remove duplicate entrypoints from the list of entry points while
   preserving order.  Duplicates can arise from repeated imports of a 
   functor from the same file.  There is no harm in having them, except
   that it is a waste of memory space and cpu time.  It is important to
   preserve order here. *)

val uniq : ''a list -> ''a list =
    let fun f (seen, h :: t) =
	  if exists (fn a => a=h) seen then f(seen,t)
          else h :: f (h :: seen,t)
          | f (_,nil) = nil
    in fn x => f (nil,x)
    end

val intersect : ''a list * ''a list -> ''a list =
    let fun f (nil,_,s) = uniq s
          | f (h::t,r,s) = f(t,r,if exists(fn a=>a=h) r then h::s else s)
    in fn (x,y) => f(x,y,nil)
    end

val repeated : ''a list -> ''a list =
    let fun f (seen, h :: t, s) =
                  f(h :: seen,t,if exists (fn a=>a=h) seen then h::s else s)
          | f (seen, nil, s) = uniq s
    in fn x => f(nil, x, nil)
    end

val printDatalist = fn (dl : string list,f : outstream) =>
	   Print.printDlist ("datalist",outputc f,
		    map (fn s => {entry=s,name="mo/"^s^".mo"})
		        (!entryPoints @ uniq dl))

fun compile rename cc (fname : string,dest : outstream) : result =
   let val _ = Print.startFile(outputc dest)
   in CProcessFile.process{env = SOME (!pervasiveEnv),
			  src = fname,
			  dest= dest,
			  gencode = SOME generate,
			  transform = rename,
			  cc=cc}
   end

val mboot = fn () =>
    let val RESULT{entrypoints,...} =
	CProcessFile.bootEnv (fn s =>
	    let val d = open_out (s^".c")
            in (compile (fn i => i) (fn _ => ()) (s,d)) before close_out d
            end)
         val env = Env.closeCurrentNewEnv()
    in pervasiveEnv := env;
       Env.resetEnv env;
       Env.commit();
       entryPoints := entrypoints
    end

val timestamp : unit -> string =
    let open System.Unsafe.CInterface
	val timeofday : unit -> int* int =
	         (c_function "timeofday") handle CFunNotFound _ =>
		             (fn () => (0,0))
    in fn () => makestring (#1 (timeofday()))
    end
	
val compile_unique =
   let val count = ref 1
       val rename = 
	    fn _ => let val s' = "U" ^ (makestring (!count)) ^ timestamp()
		     in count := (!count) + 1; s'
		     end
   in compile rename
   end

 val cc = ref "cc"
 val cflags = ref ""
 val copt = ref true
 val error = ref false
 val object = ref (NONE : string option)
 val onlyc = ref (NONE : string option)
 val perv = ref false
 val stats = ref false

 val init_args = fn () =>
     (cc := "cc";
      cflags := "";
      error := false;
      copt := true;
      object := NONE;
      onlyc := NONE;
      perv := false;
      stats := false
     )     
  
 val complain = fn s => (pr ("sml2c: error: "^s^"\n"); error := true;
			 flush_out std_out)
		
 val toggles =
	let open Control
        in  [("arithmetic",fn x => arithOpt := x),
	     ("autoincrement",fn x => autoIncrement := x),
             ("debug",fn x => debug := x),
             ("function-integration",fn x => integOpt := x),
             ("local-variables",fn x => regOpt := x),
             ("register-array",fn x => regArray := x),
             ("signals",fn x => (zeroCheck := x;
                                 integOpt := not x)),
	     ("tail-call-elimination",fn x => tailOpt := x),
             ("unsafe-arithmetic",fn x => unsafeArith := x),
             ("zero-check",fn x => zeroCheck := x)]

        end
 
 val ignore = ["-h","-r","-s","-m","-g"]
 val commands = [("-gcc",fn () => (cc := "gcc";
			cflags := ("-fomit-frame-pointer " ^ !cflags))),
		 ("-pervasives", fn () => perv := true),
		 ("-statistics", fn () => stats := true),
                 ("-plain",fn () => copt := false),
                 ("-profiling",fn () =>
                                  System.Control.Profile.profiling := true)
		]
 val commands2 = [("-o", fn a => object := SOME a),
		  ("-c",fn a => onlyc := SOME a),
		  ("-b",fn a => cc := a),
		  ("-cf",fn a => cflags := (!cflags^" "^a))]

 fun exists2 nil s = false
   | exists2 ((a,b) :: r) s = a=s orelse exists2 r s

 exception UnknownFlag

 fun find ((a,b) :: r) s = if s=a then b else find r s
   | find _ _ = raise UnknownFlag

 val process_args : string list -> string list =
   fn args =>
    let val _ = init_args()
        fun f nil = nil
          | f (""::t) = f t
          | f (h::t) =
	    if substring(h,0,1)="-" handle Substring => false then
	     (if substring(h,0,5) = "-fno-" handle Substring => false then
		  (find toggles ((substring(h,5,size h-5)) handle Substring =>
				       raise UnknownFlag) false;
		   f t)
	      else if (substring(h,0,2) = "-f") handle Substring => false then
		  (find toggles ((substring(h,2,size h-2)) handle Substring =>
				     raise UnknownFlag) true;
		   f t)
              else if exists2 commands h then
		((find commands h) (); f t)
	      else if exists (fn a=>h=a) ignore then
	        case t of nil => nil
	           | a :: b => f b
	      else if exists2 commands2 h then
		 case t
		 of nil => (complain(" "^h ^": argument missing"); nil)
	          | a :: b => ((find commands2 h) a; f b)
	      else raise UnknownFlag)
		    handle UnknownFlag => (complain(h^": unknown flag"); f t)
			 | e => raise e
            else h :: f t
   in f args before (if !copt then cflags := "-O " ^ !cflags else ())
   end

 exception Fail

 val process_files = fn (files,onlyc,objectfile) =>
      let val system = fn command => (pr ("["^command^"]\n");
				      flush_out std_out;
				      System.system command)
	  val compile_to_ofile = fn f =>
	       (system(!cc ^ " " ^ !cflags ^ "-c -o " ^ f ^ ".o " ^ f ^ ".c");
                system("rm -f "^f^".c"))
          val cc' = (case onlyc
	                of NONE => compile_to_ofile
			 | SOME _ => (fn _ => ()))
          val compile = compile_unique cc'
          fun scan (nil,e,s) = s
	    | scan (from :: from',e, s) : string list =
	    let val dest = open_out (from^".c")
		val RESULT{entrypoints=ce,imports=cs} = compile(from,dest)
		val ne = e @ ce
		val ns = s @ cs
	    in case from'
	       of nil => printDatalist(ne,dest)
	        | _ =>  ();
	      close_out dest;
	      cc' from;
	      scan(from',ne,ns)
	    end

          (* check that no file name is repeated *)

          val stringlist : string list -> string =
              fn l => fold (fn (x,r) => x^" "^r) l ""

          val _ = case repeated files
                  of (l as _ :: _) =>
                      (complain("repeated source file names: " ^
                                stringlist l);
                       raise Fail)
                   | _ => ()

          val importfiles = map (fn s=>s^".sml") (scan(files,nil,nil))

          val _ = case intersect(files,importfiles)
                  of (l as _ :: _) =>
		       (complain("files are imported and used as top-level\
                                 \ source files: "^stringlist l);
                        raise Fail)
                   | _ => ()

          val srcfiles = (uniq importfiles) @ files

	  val ofiles = stringlist (map (fn a => a^".o") srcfiles)
      in case onlyc
	 of SOME makefile =>
	       let  val command = "\t$(CC) $(CFLAGS) -o $@ $> -lsml2c -lm"
		    val f = open_out makefile
	        in outputc f ("CFLAGS = "^ !cflags ^ "\n" ^
		        "CC = "^ !cc ^ "\n\n" ^ objectfile ^ ":\t" ^
			ofiles ^ "\n" ^ command ^ "\n\n" ^
			fold (fn (a,b) =>(a^".o:\t"^a^".c\n\t$(CC) \
                              \$(CFLAGS) -c -o $@ "^a^".c\n\n"^b)) srcfiles ""^
		        "\n");
		   close_out f
		end
	  | NONE =>
	      let val cfiles = fold (fn (a,b) => a^".c "^b) srcfiles ""
	      in system (!cc ^ " " ^ !cflags ^ " -o "^objectfile ^ " " ^
			 ofiles ^ " -lsml2c -lm");
                 system ("rm -f "^stringlist (map (fn a=>a^".o") files))
 	      end
      end

 fun command_line (h :: args,_) =
      (let val files = process_args args
       in if !perv then ()
	  else (case files
	        of nil => complain("no source files")
		 | _ => ();
	        case !object
	        of NONE => complain ("must specify output file")
		 | SOME i => 
		     if exists (fn a=>a=i) files then
			 complain("-o "^i^" would overwrite a source file")
		     else ());
          if !error then raise Fail
	  else if !perv then (mboot(); ())
	  else process_files(files,!onlyc,case !object of SOME i=>i);
          if !stats then statistics() else ()
      end handle CProcessFile.Stop => ()
               | Fail => ()
	       | (Io f) =>
		     (pr "[Failed on ";
		      pr f; pr "]\n"; flush_out std_out)
	       | e => 
                  (pr "[Failed "; pr " with ";
                   pr(System.exn_name e); pr "]\n"; flush_out std_out))
    | command_line (nil,_) = ()
 
   val _ = exportFn("sml2c-heap",command_line)
(* val _ = command_line (System.argv(),nil) *)
end
