(* Copyright (c) 1991 by Carnegie Mellon University *)

(* Functor which converts cps to a set of instructions for an abstract
   machine based on C.

   Created: 6/20/90 by David Tarditi

   This is derived from cps/generic.sml.  The major changes are:
             (1) the abstract machine functions for creating instructions
	     do not issue the instructions as side-effects.  They actually
	     return the instructions.
	     (2) this is an abstract machine for C, not assembly: functions,
                  data, and switches are explicitly represented.
	     (3) spelling conventions are different.
	     (4) I've added comments to try to explain the more
	         cryptic aspects of register allocation, which occurs
		 at the same time that CPS is being converted to
		 instructions for the abstract machine.
 *)

functor CPStoC(structure Machine: ANSICMACHINE
	       structure Call : CALL
	       structure Control : CONTROL
	       structure FreeMap : CFREEMAP
	       structure CPS : CPS
	       sharing CPS = Call.CPS = Machine.CPS = FreeMap.CPS
	       sharing type CPS.lvar = int)
                    : CPSCODEGEN =
struct

type entrypoint = string
open Machine Machine.CPS System.Tags Access

val DEBUG_C = false
val DEBUG_CG = false

(* split: split a list into lists containing elements for which pred is and
   false, respectively *)

fun split pred nil = (nil,nil)
  | split pred (a::r) = let val (x,y) = split pred r
			 in if pred a then (a::x, y) else (x, a::y)
		        end

type register = int
val allregs = standardClosure :: standardArg :: standardCont ::miscRegs
val numregs = length allregs

local val standardContIndices = [2,1]
in 
  val standardArgIndices = [0,1,2]
  fun standardformals [_,_] = standardContIndices
    | standardformals [_,_,_] = standardArgIndices
    | standardformals _ = ErrorMsg.impossible "standardformals in cpstoc" 
end

val any = 3

fun standardArgs [_,_] = [standardArg,standardCont]
  | standardArgs [_,_,_] = [standardClosure,standardCont,standardArg]
  | standardArgs _ = ErrorMsg.impossible "standardargs in cpstoc"

local val regarr = arrayoflist allregs
in val regt : int -> EA =
      fn x => regarr sub x
end

val knowngen = System.Control.CG.knowngen
val stdgen = System.Control.CG.stdgen

(* getscratch: int * int list -> int
   Choose a register not on the prohibited list of registers and choose
   the preferred register if possible *)

local val okreg = array(length allregs,true)
     fun mark b i = update(okreg,i,b)
in exception Getscratch
    val getscratch : int * int list -> int =
      fn (preferred,prohibited) =>
        let fun f i = if okreg sub i then i else f (i+1)
        in app (mark false) prohibited;
	  (if okreg sub preferred then preferred else f 0)
	   before app (mark true) prohibited
	   handle Subscript => (app (mark true) prohibited;
				 raise Getscratch)
        end
end

fun sublist test =
    let fun subl (a::r) = if test a then a::(subl r) else subl r
          | subl nil = nil
    in subl
    end

val immed1 = immedEA 1
val immed2 = immedEA 2
val immedn1 = immedEA (~1)

fun codegen(funs : ((lvar * lvar list * cexp) * bool) list,err,entry) =
let 
    (* regbindtable: maps an lvar to its register number *)

    exception Regbind and Labbind
    val regbindtable : register Intmap.intmap = Intmap.new(32, Regbind)
    val addbinding = Intmap.add regbindtable
    val rmvbinding = Intmap.rmv regbindtable
    val regmap = Intmap.map regbindtable

    (* labmap: maps an lvar to its label *)

    val labbindtable : EA Intmap.intmap = Intmap.new(32,Labbind)
    val labmap = Intmap.map labbindtable
    val addlabbinding = Intmap.add labbindtable

    val prefer = fn (VAR v) => regmap v | _ => any
      
    (* knowtable: maps lvars for function names to their calling
       conventions.  Argument 1 must be placed in the first register,
       argument 2 in the second, and so on. *)
      
    exception Know
    val knowtable : register list Intmap.intmap = Intmap.new(32, Know)
    val addknow = Intmap.add knowtable
    val know = Intmap.map knowtable

    exception Args
    val argtable : EA list Intmap.intmap = Intmap.new(32,Args)
    val addargs = Intmap.add argtable
    val getargs = Intmap.map argtable

    val notconstant = fn v => (know v; false) handle Know => true

    (* freemaptable: maps lvars which are bound by some cps expression
       to a list of lvars which are free in the scope of those bound
       lvars.

       All bound lvars live in registers, so those registers which are
       used for lvars that are free in a CPS expression cannot be
       used to hold a variable that is newly bound by a CPS expression.
    *)

    exception Freemap
    val freemaptable : lvar list Intmap.intmap = Intmap.new(32, Freemap)
    val freemap = sublist notconstant o Intmap.map freemaptable
    val addfree = Intmap.add freemaptable
    val rmvfree = Intmap.rmv freemaptable

    (* doneCodeGenTable: maps lvars for functions to boolean indicating
       whether we've already generated code for the function *)

    datatype done = DONE of cexp * stmt * bool
	          | TODO of lvar * lvar list * cexp * bool * bool
    exception DoneCodeGen
    val doneCodeGenTable : done Intmap.intmap = Intmap.new(32,DoneCodeGen)
    val done = Intmap.map doneCodeGenTable
    val addDone = Intmap.add doneCodeGenTable

    (* setUpFunction: set up tables for functions
             (1) names every function and binds the lvar for a function
	         to the name
	     (2) if the function escapes, and uses the standard calling
	         sequence, bind the standard calling sequences in the
		 know table.
	     (3) map each lvar bound in cps expressions in the function
		 to the set of lvars free at the time of binding. *)

     fun setUpFunction entryFlag ((info as (f,vl,e),known)) = 
        (addlabbinding(f,if entryFlag then namedFunctionEA entry
                         else functionEA());
	 addDone (f,TODO (f,vl,e,entryFlag,known));
	 if known then ()
	 else (addknow(f,standardformals vl); addargs(f,standardArgs vl)))

    (* set up functions *)

    val _  = case funs
	of h :: t => (setUpFunction true h; app (setUpFunction false) t)
	 | nil => ()

    val funcs = ref (let fun f nil = nil
                           | f ((_,true) :: r) = f r
                           | f (((func,_,_),false) :: r) = func :: f r
                     in f funs
                     end)

    (* decls: ref cell containing list of declarations *)

    val decls = ref (nil : decl list)

    fun addDecl decl = (decls := decl :: (!decls))

    fun regbind (VAR v) = regt (regmap v)
      | regbind (LABEL v) =labmap v
      | regbind (INT i) = immedEA(i+i+1)
      | regbind (REAL s) = let val l = realEA()
	                   in addDecl(mkReal(s,l)); l
			   end
      | regbind (STRING s) = let val l = stringEA()
	                     in addDecl(mkString(s,l)); l
			     end
      | regbind _ = ErrorMsg.impossible "cpstoc: 189"

    fun root(RECORD(_,_,e)) = root e
      | root(SELECT(_,_,_,e)) = root e
      | root(OFFSET(_,_,_,e)) = root e
      | root(SWITCH(_,e::_)) = root e
      | root(PRIMOP(_,_,_,e::_)) = root e
      | root(e as APP _) = e

    (* allocReg: allocates a register to hold lvar v.  Applies continue to
       this register.

       It uses the following heuristics to select a register:
      
       If v is used in the application that has been passed in, then try
       to place v in the register which it must be in for the application.

       If the application is a known function for which we have already
       generated code or is an unknown function, then we try to place it
       in the desired register without using any other registers used
       in the application.

       If this fails or v is not used in the application that has been pased
       in, then we try to place v in the default register.  If this fails, then
       we use any available register.
    *)

    val allocReg = fn APP (f,wl) =>
       let val f = case f
	           of (VAR v) => v
		    | (LABEL v) => v
		    | _ => ErrorMsg.impossible "220"
	   val formals = (know f) handle Know =>
	     ((done f; nil)
 	      handle DoneCodeGen => standardformals wl)
      in fn (v,default,continue) =>
	let val prohibited = map regmap (freemap v)
	    fun get (good,nil) = getscratch (good,prohibited)
	      | get (good,bad) =
		      getscratch(good,bad@prohibited)
		            handle Getscratch => getscratch(default,prohibited)
	    fun find nil = get (default, nil)
              | find fmls =
               let fun delete (z,nil) = nil
	             | delete (z,a::r) = if a=z then r
					   else a :: delete(z,r)
		   fun g(VAR w::wl, r::rl) =
		          if w=v then get(r, delete(r,fmls))
			  else g(wl,rl)
		     | g(_::wl,_::rl) = g(wl,rl)
		     | g(nil,nil) = get (default, fmls)
		     | g _ = ErrorMsg.impossible "cpstoc: find/g"
	       in g(wl,fmls)
	       end
	    val register = if v=f then get(default,standardArgIndices)
				  else find formals
	 in addbinding (v,register);
	   continue (regt register)
         end
     end

    (* shuffle:
          Generate code for calling the function at contents(func),
       where func is an EA.  Args is a list of EAs where arguments 1 ...
       length(args) live, formals is a list of EAs (registers) where
       arguments 1 ... length(args) should end up.

          First, if func is a register that is needed for calling
       the function, we move func to another register.  (The handler there
       appears useless to me).

          We then zip the lists of arguments and formals into a list of pairs
       of EA's.  We then split this list into a list of pairs whose argument
       component is already in a register and a list of those pairs which are
       not in a register.

          We then split the list of pairs whose argument component
       is already in a register into a list of pairs where the arguments
       don't need to be moved and those whose arguments need to be
       moved.

          The function f then moves arguments to formals, being careful
       not to tread on any registers that are being used.
    *)

    (* for each pair (a,b) in the list: b <- a) *)

    fun movelist (pairs : (EA * EA) list) (cont : stmt) : stmt = 
	  fold (fn ((a,b),r) => move(a,b,r)) pairs cont

local	
    fun shuffle1(args : value list,formals : int list) (cont : stmt) =
      let fun classify(VAR v::al, f::fl, match, nomatch, notinreg) :
	     int list * (int * int) list * (EA * EA) list =
	        let val v' = regmap v
		 in if v' = f
			then classify(al,fl,f::match,nomatch,notinreg)
			else classify(al,fl,match,(v',f)::nomatch,notinreg)
		end
	      | classify(a::al,f::fl, m,n,notinreg) = 
	            classify(al,fl,m,n,(regbind a, regt f)::notinreg)
	      | classify(_,_, m,n,nr) = (m,n,nr)
		 
	    val (matched, notmatched, notinreg) =
				classify(args,formals,[],[],[])
          
	  fun f (nil, used) = movelist notinreg cont
	    | f (pairs,used) = 
	    let

		(* u' = list of arguments and used registers *)

		val u' = map #1 pairs @ used

		(* movable: true if the formal is not in the list
		   of arguments and used registers.*)

	        fun movable (a, b) = not (exists (fn z=>z=b) u')

		(* splits pairs into those which are movable and
		   those which aren't movable.

		   If nothing is movable, grab a register and move
		   the contents of an argument register to this
		   register.  This implies that some argument register
		   which is the destination of a formal has been moved
		   out of the way, and thus that something must be
		   movable next time through.
		   
		   If something is movable, then issue instructions
		   to move those arguments to formals, and call ourself
		   again with those formals marked as "used".

		   This algorithm terminates in time proportional to the
		   number of arguments, since at least one argument
		   will be movable after every two iterations.
		 *)

	     in case split movable pairs
	         of (nil,(a,b)::r) => 
			    let val x = getscratch(any,u')
			    in move(regt a,regt x,f((x,b)::r, used))
			    end
	          | (m,m') => movelist (map (fn (a,b) => (regt a,regt b)) m)
		                       (f(m', (map #2 m) @ used))
	    end
     in f(notmatched,matched)
     end

    fun getregs (VAR v :: rest, l) = getregs(rest, regmap v :: l)
      | getregs(_::rest,l) = getregs(rest,l)
      | getregs (nil,l) =l

  in
    fun shuffle (func as VAR f,args : value list, formals : int list) =
	let val x = getscratch(regmap f,getregs(args,nil)@formals)
	              handle Getscratch => getscratch(regmap f,formals)
	in shuffle1(func::args,x::formals) (jmp(regt x :: (getargs f handle Args => standardArgs formals)))
        end
       | shuffle(func as LABEL f,args,formals) =
	   shuffle1(args,formals) (jmp(labmap f :: getargs f handle Args => standardArgs formals))
       | shuffle _ = ErrorMsg.impossible "cpstoc: 347"
  end

    (* allocparams: decide in which registers formal parameters should be
       passed and generate code to move arguments to formal parameters.
       It returns a list of registers to hold the formal parameters and
       a list of code to move the arguments to the formal paramters.

       We use the following algorithm to minimize moves at this call
       point:
       
            If an argument is in a register, choose that register
       to hold one of the formal parameters to which the argument
       is passed.

        We implement this algorithm in the following way:

           If an argument is in a register, mark all but one occurrence of the
       argument in the argument list as needing to be copied.  If an argument
       is not in a register, mark it as needing to be copied.

           Scan the list of arguments and formals at the same time.  If
       an argument need to be copied, choose a register which has not been
       taken and which does not already hold an argument in a register and
       bind the register to the formal parameter.
*)

      fun allocparams(f : lvar, args : value list,formals : lvar list) =
	let
	    val argRegisterIndices = 
		let fun f (VAR v :: args) = regmap v :: f(args)
		      | f (arg :: args) = f args
		      | f nil = nil 	
                in f args
		end

	    (* flag those arguments which will need to be copied to their
	       destination register.*)

	    val copyArgs =
	        let fun scan ((arg as (VAR v)) :: args) =
		     if exists (fn a=>arg=a) args
			   then (arg,true) :: scan args
			   else (arg,false) :: scan args
                      | scan (arg :: args) = (arg,true) :: scan args
		      | scan nil = nil
                in scan args
		end
			
            fun g ((arg,move_flag) :: argRest, formal :: formalRest,
		   taken, stmts) =
		if move_flag then
		      let val z = getscratch(prefer arg,
					     taken @ argRegisterIndices)
		      in addbinding(formal,z);
		         g(argRest,formalRest, z :: taken,
			   fn cont => stmts (move(regbind arg,regt z,cont)))
		      end
                else let val argreg =
			  case arg
			  of VAR v => regmap v
			   | _ => ErrorMsg.impossible "cps/cpstoc.sml: argreg"
		     in addbinding(formal,argreg);
			g(argRest,formalRest,argreg :: taken,stmts)
		      end
	     | g(nil,nil,taken,stmts)=
	          let val args = map regt taken
		      val _ = addargs(f,args)
		  in (rev taken,stmts (jmp (labmap f::args)))
		  end
             | g _ = ErrorMsg.impossible "cpstoc: 415"
      in g (copyArgs,formals,nil,fn i => i) 
      end

   (* stupidargs: just bind the formals to allregs, one register
      after another.  Then call shuffle to generate code.
   *)

    fun stupidargs(f,args' : value list,vl : lvar list) =
	let fun argregs(nil,_) = nil 
	      | argregs(a::rest,i) = (addbinding(a,i);
					  i::argregs(rest,i+1))
	    val formals = argregs(vl,0)
	 in (formals,shuffle(f,args',formals))
	end

(* Compute the maximum amount of allocation done by this function (in
   the number of integers  (i.e. longwords) allocated) *)
    fun sumAlloc exp = let
	  fun sum (RECORD (fields, _, exp'), max) = 
	                  sum (exp', max+(length fields)+1)
	    | sum (SELECT (_, _, _, exp'), max) = sum (exp', max)
	    | sum (OFFSET (_, _, _, exp'), max) = sum (exp', max)
	    | sum (APP _, max) = max
	    | sum (SWITCH (_, lst), max) = max + lstMax(lst, 0)
	    | sum (PRIMOP (P.makeref, _, _, [exp']), max) = sum (exp', max+2)
	    | sum (PRIMOP (P.delay, _, _, [exp']), max) = sum (exp', max+2)
	    | sum (PRIMOP (P.update, _, _, [exp']), max) = sum (exp', max+4)
	    | sum (PRIMOP (P.:=, _, _, [exp']), max) = sum (exp', max+4)
	    | sum (PRIMOP (P.fadd, _, _, [exp']), max) = sum (exp', max+3)
	    | sum (PRIMOP (P.fsub, _, _, [exp']), max) = sum (exp', max+3)
	    | sum (PRIMOP (P.fmul, _, _, [exp']), max) = sum (exp', max+3)
	    | sum (PRIMOP (P.fdiv, _, _, [exp']), max) = sum (exp', max+3)
	    | sum (PRIMOP (_, _, _, [exp']), max) = sum (exp', max)
	    | sum (PRIMOP (_, _, _, lst), max) = max + lstMax(lst, 0)
	  and lstMax (nil, max) = max
	    |lstMax (e::rest, max) = let val m = sum (e, 0)
		in
		  if m > max then lstMax(rest, m) else lstMax(rest, max)
		end
	  in
	    (sum (exp, 0))
	  end

    fun genstmt alloc cexp =
      let fun gen cexp : stmt =
	case cexp
	 of RECORD(vl,w,e) =>
		 alloc(w, any,  fn w' => 
		     let val stmt = 
			   record((immedEA(make_desc(length vl,tag_record)),OFFp 0) ::
				      map (fn(x,p)=>(regbind x, p)) vl,
			           w',gen e)
		     in if DEBUG_C then
			    comment("record",stmt)
			else stmt
		     end)
	  | SELECT(i,v,w,e) =>
	    alloc(w, any,  fn w' => 
		  let val stmt = select(i,regbind v,w',gen e)
	          in if DEBUG_C then
		          comment("select",stmt)
		     else stmt
		  end)
	  | OFFSET(i,v,w,e) =>
	    alloc(w, prefer v, fn w' =>
		   let val stmt = offset(i,regbind v,w',gen e)
		   in if DEBUG_C then
		         comment("offset",stmt)
		      else stmt
		   end)
	  | APP(f,args) =>
	     let val v = case f
		         of VAR v => v
		          | LABEL v => v
			  | _ => ErrorMsg.impossible "cpstoc: 488"	        
                 val stmt =
		 let val formals = know v
		  in shuffle(f,args,formals)
		  end handle Know =>
		       (case (done v)
		       of TODO (_,vl,_,_,_) =>
		          let val (regs,stmt) =
			     if !System.Control.CG.argrep
			       then allocparams(v,args,vl)
			       else stupidargs(f,args,vl)
			   in funcs := v :: (!funcs);
			       addknow(v,regs);
			       stmt
			   end
			  | _ => raise 
			 (ErrorMsg.impossible "cpstoc/done but unknown args"))
		           handle DoneCodeGen =>
		              shuffle (f,args,standardformals args)
	     in if DEBUG_C then
		  comment("app",stmt)
		else stmt
             end
	  | SWITCH(v,l) => 
		let fun f(i, s::r) =
		           (i,genstmt (allocReg (root s)) s) :: f(i+1,r)
		      | f (_, nil) = nil
		    val stmt =  mkCase(regbind v,f(0,l))
		in if DEBUG_C then
		           comment("switch",stmt)
		   else stmt
		end
        | PRIMOP(P.+,[INT v,w],[x],[e]) =>
             alloc(x,any,fn x' =>
		 addti(immedEA(v+v),regbind w,x',gen e))

        | PRIMOP(P.+,[v,INT w],[x],[e]) =>
             alloc(x,any,fn x' =>
		 addti(immedEA(w+w),regbind v,x',gen e))

        | PRIMOP(P.+,[v,w],[x],[e]) =>
             alloc(x,any,fn x' => 
		subi(immed1,regbind v,arithTemp,
		   addti(arithTemp, regbind w, x', gen e)))

       | PRIMOP(P.orb,[v,w],[x],[e]) =>
             alloc(x,prefer w, fn x' => orb(regbind v, regbind w, x',gen e))

       | PRIMOP(P.andb,[v,w],[x],[e]) =>
             alloc(x,prefer w, fn x' => andb(regbind v, regbind w, x',gen e))

       | PRIMOP(P.xorb,[INT v,w],[x],[e]) =>
	     alloc(x,any,fn x' =>
	        xorb(immedEA(v+v), regbind w, x',gen e))

       | PRIMOP(P.xorb,[v,INT w],[x],[e]) =>
             alloc(x,any,fn x' =>
		   xorb(immedEA(w+w),regbind v,x', gen e))

       | PRIMOP(P.xorb,[v,w],[x],[e]) =>
	     alloc(x,any,fn x' =>
		   xorb(regbind v, regbind w,arithTemp,
			addi(immed1, arithTemp, x',gen e)))

       | PRIMOP(P.notb,[v],[x],[e]) =>
          alloc(x,prefer v, fn x' =>
		notb(regbind v, x',orb(immed1, x', x',gen e)))

       | PRIMOP(P.lshift,[INT v,w],[x],[e]) =>
	  alloc(x,any, fn x' =>
             ashr(immed1,regbind w,arithTemp,
	       ashl(arithTemp,immedEA(v+v), x',
		 addi(immed1, x', x',gen e))))

       | PRIMOP(P.lshift,[v,INT w],[x],[e]) =>
	   alloc(x,any,fn x' =>
	     addi(immedn1,regbind v,arithTemp,
	       ashl(immedEA w,arithTemp, x',
		  addi(immed1, x', x',gen e))))

       | PRIMOP(P.lshift,[v,w],[x],[e]) =>
	   alloc(x,any, fn x' =>
	      ashr(immed1, regbind w,arithTemp,
		 addi(immedn1,regbind v,x',
		   ashl(arithTemp, x', x',
		     addi(immed1, x', x',gen e)))))

       | PRIMOP(P.rshift,[v,INT w],[x],[e]) =>
	     alloc(x,prefer v,fn x' =>
		    ashr(immedEA w,regbind v,
			x',orb(immed1,x', x',gen e)))

       | PRIMOP(P.rshift,[v,w],[x],[e]) =>
            alloc (x,prefer v, fn x' =>
	      ashr(immed1, regbind w,arithTemp,
		 ashr(arithTemp, regbind v, x',
		   orb(immed1,x', x',gen e))))

       | PRIMOP(P.-,[INT v,w],[x],[e]) =>
	      alloc(x,prefer w, fn x' => 
		  subti(regbind w, immedEA(v+v+2), x',gen e))

       | PRIMOP(P.-,[v,INT w],[x],[e]) =>
	      alloc(x,prefer v, fn x' => 
		   subti(immedEA(w+w),regbind v,x',gen e))

       | PRIMOP(P.-,[v,w],[x],[e]) =>
	      alloc(x, prefer v,fn x' => 
		 subi(immed1,regbind w,arithTemp,
		     subti(arithTemp, regbind v, x',gen e)))

       | PRIMOP(P.*,[INT v,w],[x],[e]) =>
           alloc(x,any,fn x' =>
	     ashr(immed1, regbind w, arithTemp,
	       multi(immedEA(v+v),arithTemp,x',
		 addi(immed1,x',x',gen e))))

       | PRIMOP(P.*,[v,INT w],[x],[e]) =>
           alloc(x,any,fn x' =>
	     ashr(immed1, regbind v, arithTemp,
	       multi(immedEA(w+w),arithTemp,x',
		 addi(immed1,x',x',gen e))))

       | PRIMOP(P.*,[v,w],[x],[e]) =>
          alloc(x,any,fn x' =>
	     ashr(immed1, regbind v, arithTemp,
	       subi(immed1, regbind w, x',
		 multi(x',arithTemp,x',
		   addi(immed1,x',x',gen e)))))

         (* divide v by w, place result in x *)

       | PRIMOP(P.div,[INT v,w],[x],[e]) =>
	  alloc(x,any,fn x' =>
	     ashr(immed1, regbind w, arithTemp,
		divti(arithTemp,immedEA v,x',
		    addi(x', x', x',
		       addi(immed1, x', x',gen e)))))

       | PRIMOP(P.div,[v,INT w],[x],[e]) =>
	  alloc(x,any,fn x' =>
	    ashr(immed1, regbind v, x',
	      divti(immedEA w, x',x',
		addi(x', x', x',
		   addi(immed1, x', x',gen e)))))

       | PRIMOP(P.div,[v,w],[x],[e]) =>
	  alloc(x,prefer v,fn x' =>
	    ashr(immed1, regbind w, arithTemp,
	      ashr(immed1, regbind v, x',
		divti(arithTemp,x',x',
		  addi(x', x', x',
		    addi(immed1, x', x',gen e))))))

       | PRIMOP(P.!,[v],[w],[e]) => gen(SELECT(0,v,w,e))

       | PRIMOP(P.:=,[v,w],[],[e]) =>
	    let val v' = regbind v
	    in record([(immedEA(make_desc(3,tag_record)),OFFp 0), (v', OFFp 0),
		       (immed1, OFFp 0), (storePtr, OFFp 0)], storePtr,
		   storeIndex(regbind w, v', immed1,gen e))
	    end
       | PRIMOP(P.unboxedassign,[v,w],[],[e]) =>
              storeIndex(regbind w, regbind v, immed1,gen e)
       | PRIMOP(P.~,[v],[w],[e]) =>
	 alloc(w,any,fn w' =>
	      subi(regbind v,immed2,w',gen e))
       | PRIMOP(P.makeref,[v],[w],[e]) =>
	    alloc(w, any, fn w' =>
		record([(immedEA(make_desc(1,tag_array)),OFFp 0),
			 (regbind v, OFFp 0)], w',gen e))
       | PRIMOP(P.delay,[i,v],[w],[e]) =>
	    alloc(w, any, fn w' =>
		record([(regbind i, OFFp 0),(regbind v, OFFp 0)], w',gen e))
       | PRIMOP(P.rangechk, [v,w],[],[d,e])  =>
	    rangeChk(regbind v,regbind w, gen d, gen e)
       | PRIMOP(P.subscript,[v,w],[x],[e]) =>
	    alloc(x, any, fn x' => fetchIndex(regbind v, x', regbind w,gen e))
       | PRIMOP(P.update,[a, i, v], [], [e]) =>
	    let val a' = regbind a and i' = regbind i
	    in record([(immedEA(make_desc(3,tag_record)),OFFp 0), (a',OFFp 0),
		        (i', OFFp 0), (storePtr, OFFp 0)], storePtr,
	          storeIndex(regbind v, a', i',gen e))
	    end
       | PRIMOP(P.unboxedupdate,[a, i, v], [], [e]) =>
	   storeIndex(regbind v, regbind a, regbind i,gen e)
       | PRIMOP(P.alength,[a], [w], [e]) =>
	   alloc(w,  any,  fn w' =>
	      select(~1, regbind a, arithTemp,
	       ashr(immedEA(width_tags-1),arithTemp, arithTemp,
	          orb(immed1, arithTemp, w',gen e))))
       | PRIMOP(P.slength,[a], [w], [e]) =>
	  alloc(w, any, fn w' =>
	    let val cont =ashr(immedEA(width_tags-1), arithTemp, arithTemp,
	          orb(immed1, arithTemp, w', gen e))
	     in case a
		of VAR v => select(~1,regbind a,arithTemp,cont)
		 | _ => move(regbind a,w',select(~1,w',arithTemp,cont))
	    end)
       | PRIMOP(P.store,[s,i,v], [], [e])  =>
	     ashr(immed1, regbind v, arithTemp,
		storeIndexB(arithTemp, regbind s,regbind i,gen e))
       | PRIMOP(P.ordof,[s,i], [v], [e]) =>
	    alloc(v, any, fn v' =>
	       fetchIndexB(regbind s, v', regbind i,
		   addi(v',v',v',
		     addi(immed1, v', v',gen e))))
       | PRIMOP(P.profile,[index,incr],[],[c]) => gen c
       | PRIMOP(P.boxed,[x],[],[a,b]) =>
	     bbs(0,regbind x,
		 genstmt (allocReg (root a)) a,
		 genstmt (allocReg (root b)) b)
       | PRIMOP(P.gethdlr,[],[x],[e]) =>
		  alloc(x, any, fn x' => move(exnPtr,x',gen e))
       | PRIMOP(P.sethdlr,[x],[],[e]) => move(regbind x, exnPtr, gen e)
       | PRIMOP(P.fmul,[x,y], [z], [e]) =>
	    alloc(z,  any, fn z' => mulf(regbind x, regbind y, z', gen e))
       | PRIMOP(P.fdiv,[x,y], [z], [e]) =>
	    alloc(z,  any, fn z' => divf(regbind x, regbind y, z', gen e))
       | PRIMOP(P.fadd,[x,y], [z], [e]) =>
	    alloc(z,  any, fn z' => addf(regbind x, regbind y, z', gen e))
       | PRIMOP(P.fsub,[x,y], [z], [e]) =>
	    alloc(z,  any, fn z' => subf(regbind x, regbind y, z', gen e))
       | PRIMOP (args as (P.ieql ,_,_,_)) => compare(ibranch,NEQ,args)
       | PRIMOP (args as (P.ineq ,_,_,_)) => compare(ibranch,EQL,args)
       | PRIMOP (args as (P.> ,_,_,_)) => compare(ibranch,LEQ,args)
       | PRIMOP (args as (P.>= ,_,_,_)) => compare(ibranch,LT,args)
       | PRIMOP (args as (P.< ,_,_,_)) => compare(ibranch,GEQ,args)
       | PRIMOP (args as (P.<= ,_,_,_)) => compare(ibranch,GT,args)
       | PRIMOP (args as (P.feql,_,_,_)) => compare(gbranch,NEQ,args)
       | PRIMOP (args as (P.fneq,_,_,_)) => compare(gbranch,EQL,args)
       | PRIMOP (args as (P.fgt,_,_,_)) => compare(gbranch,LEQ,args)
       | PRIMOP (args as (P.flt,_,_,_)) => compare(gbranch,GEQ,args)
       | PRIMOP (args as (P.fge,_,_,_)) => compare(gbranch,LT,args)
       | PRIMOP (args as (P.fle,_,_,_)) => compare(gbranch,GT,args)
       | _ => ErrorMsg.impossible "cps/cpstoc: 709"  
    and compare (branch,test,(_,[v,w],[],[d,e])) : stmt =
	    branch(test,regbind v, regbind w,
		  genstmt (allocReg (root e)) e,genstmt (allocReg (root d)) d)
      | compare _ = ErrorMsg.impossible "cpstoc: 721" 

    in if DEBUG_CG then
	(print "starting gen";
	 (gen cexp) before (print "ending gen"))
       else gen cexp
    end

   (* clean: purge the register bindings for lvars bound in the continuation
      expression from the register binding table.  We can throw away the
      register bindings for a function body after generating code for the
      body.*)

   fun clean (cexp,formals) =
       let fun bindings (RECORD(_,w,e),r) = bindings(e,w::r)
             | bindings (SELECT(_,_,w,e),r) = bindings(e,w::r)
             | bindings (OFFSET(_,_,w,e),r) = bindings(e,w::r)
             | bindings (PRIMOP(_,_,l,el),r)  =
                  let fun f (nil,r) = r
                        | f (h::t,r) = f(t,bindings(h,r))
                  in f(el,l @ r)
                  end
             | bindings (_,r) = r
           val bindingList = bindings(cexp,nil)
       in app rmvbinding bindingList;
          app rmvfree bindingList;
          app rmvfree formals
       end

   fun genFunBody cexp = genstmt (allocReg (root cexp)) cexp

   fun regmask formals : int =
       let fun f(i,mask) = Bits.orb(Bits.lshift(1,i),mask)
       in fold f formals 0
       end

   val genFunBodies =
       let fun loop () =
            case !funcs
	    of nil => ()
	     | h :: t =>
		(funcs := t;
		 case done h
		 of DONE _ => ()
		  | TODO (f,vl,e,global,known) =>
                       (FreeMap.freemap addfree e;
                        List2.app2 addbinding(vl,know f);
			addDone (h,DONE(e,genFunBody e,global));
			clean (e,vl));
		 loop ())
        in loop 
        end

    val _ = genFunBodies ()

    val {integrate,includefuns} = 
	 if !Control.integOpt
	     then Call.info funs
	     else {integrate = fn _ => false, includefuns = fn a => nil}
		 
    exception NotUsed
    val genFun = fn f =>
       mkFunction (map (fn f =>
	         case done f
		 of DONE(e,stmt,global) =>
		    {name = labmap f,
		     body = stmt,
		     args = getargs f,
		     alloc = sumAlloc e,
		     global = global,
		    regmask = regmask ((know f) handle Know =>
			       ErrorMsg.impossible ("unbound registers"))
		     }
                  | _ => raise NotUsed)
	             (f :: includefuns f))
         
    val functions = fold (fn (((f,_,_),_),rest) =>
		     if integrate f then rest
		      else (genFun f :: rest) handle NotUsed => rest) funs nil

  in mkProg(!decls,functions)
   end
end
