(* Copyright (c) 1991 by Carnegie Mellon University *)

(* c.sig
 *
 *  The signature for a structure containing the representation for
 *  instructions for a C code machine.
 *
 * Author: David Tarditi
 *         Carnegie Mellon University
 *         tarditi@cs.cmu.edu
 *
 * The semantics of the instructions are described below.
 *
 * Here is a description of the internals semantics of our implementation.
 * 
 *   Types:
 *
 * These are the basic things we have around:
 *	 locations
 *	 registers
 *	 integers, reals, functions, and strings
 *
 *      type loc
 *      type register
 *      type store-values = int | loc | string | real | function
 *      type register-store-values = int | loc | function 
 *
 *  We have two stores, since the set of values storable in the register
 *  store is different from the set of values storable in the memory.
 *  The first store is like a conventional store, and returns values
 *  of type store-values.  The second store is a register store.  It maps
 *  registers to integers, locations, and functions.  It cannot hold strings
 *  and reals.

 *  The first store has type loc -> store-values.  The second store has
 *  type register -> register-store-values.
 *
 *  Since the two stores are disjoint in their sets of locations, we
 *  can overload the update and deref operators.  We use the following
 *  operators:
 *
 *	1) Update --
 *	  <- : register * register-store-values -> nullary
 *          or loc * store-values -> nullary
 *	2) Deref --
 *	   !  : register -> register-store-values
 *	        loc -> store-values
 *
 *   We assume that the conventional store is a large array of bytes which
 *   happen to encode various store-values when properly addressed.  The
 *   variable dataptr points to a location in the heap.  The operation
 *   dataptr[offset] = value writes the value into the heap.   We omit the
 *   details of the garbage collection algorithm here.
 *
 *   Statements may use addressing modes.  Define
 *
 *	type address-mode = int | loc | register
 *
 *   with the following operation:
 *
 *	contents : maps a register to its contents in the register store,
 *		   passes other values through unchanged.
 *
 *   All integers are tagged unless otherwise noted.
 *
 *   Here are the semantics of statement.  If a statement has only
 *   a single continuation statement, execution continues with that
 *   statement.
 *
 *     ALLOC(l,dataptr,s) = Construct a record from the list of access
 *	 paths * EA at contents(dataptr), increment dataptr by the size of
 *       the of the record.  Access paths work as follows:
 *	        access(EA,OFFp 0) = contents(EA)
 *		access(EA,OFFp j) = contents(EA)+j
 *		access(EA,SELp (i,p')) = access(deref(contents(EA)+j),p')
 *     ASSIGN(ea,r,s) = r <- contents(ea). 
 *     CALL(ea1,ea2 list,ea3 list,ea4 list,ea5 option,s) =
 *           call ea1, spilling values in ea2 list, reloading values
 *           in ea3 list, passing argument ea4 list to the function,
 *           placing result value ea5, and continue with s.
 *     CASE(REGISTER r,int * stmt list) = Untag r and branch to the ith
 *                statement.
 *     COND(exp,ea1,ea2,consequent,antecedent):
 *            if exp is nonzero do the consequent.  Otherwise do the
 *	      antecedent
 *     FETCH(ea1,ea2,r,s): 
 *	       r <- !((contents(ea1) + sizeof(int) * untag(contents(ea2)))
 *     FETCHB(ea1,ea2,r,s): Fetches a byte. 
 *		a <- !(contents(ea1) + sizeof(byte) * untag(contents(ea2)))
 *     GOTO l = jump to label l
 *     INT2OP (intBinOp,ea1,ea2,register,s) =
 *           We assume that ea1 and ea2 are untagged integers.  These
 *	     operations work on 32-bit integers.
 *             case intBinOp
 *		 of ashl: register <- shift contents(ea1) left by contents(ea2)
 *		    ashr: shift contents(ea1) right by contents(ea2)
 *		  | orb, andb, xorb: 
 *			register <- contents(ea1) op contents(ea2)
 *		  | notb:
 *		         register <- not contents(ea1)
 *	          | add:
 *			register <- contents(ea1) + contents(ea2)
 *		  | sub:
 *			register <- contents(ea1) - contents(ea2)
 *		  | div:
 *			register <- contents(ea1) / contents(ea2)
 *		  | ptradd:
 *		        register <- (cast ((cast contents(ea1) to int
 *			                pointer) + contents(ea2)) to integer)
 *     INT1OP (intUnaryOp,ea1,register,s) =
 *	      case intUnaryOp
 *		of notb:
 *		register <- not (contents(ea1))
 *     JMP ea = if contents(ea) is a function then goto contents(ea)
 *	         else *undefined*
 *     LABEL(l,s) = label s with l
 *     SET(ea1,ea2,ea3,s): 
 *	   (contents(ea2) + sizeof(int)*untag(contents(ea3))) <- contents(ea1)
 *     SETB(ea1,ea2,ea3,s): Set a byte in memory.  
 *	   (contents(ea2) + sizeof(byte) * untag(contents(ea3)) <-
 *		         contents(ea1) 
 *     SEQ [l1...ln] = A sequence of code blocks.  Executes l1, may branch
 *         to l1, l2, ... or ln.
 *
 * The meanings of expressions are:
 *
 *    VAL ea = contents(ea)
 *    IB (cond,ea1,ea2) = Same as INT2OP, except value is not stored
 *    in a register.
 *    IU (cond,ea1) = Same as INT1OP, except value is not stored in a register
 *    ICOND (cond,ea1,ea2) = 
 *          1 iff cond(contents(ea1),contents(ea2)) holds
 *	    <>1 otherwise
 *    FCOND (cond,e1,e2) =
 *          1 iff cond(!(contents(ea1)),!(contents(ea2)))
 *          <>1 otherwise
 *    CAND (exp1,exp2) =
 *          1 iff exp1 and exp2 hold
 *	    0 otherwise
*)
   
signature C =
   sig
      structure CPS : CPS
      structure Bignum : CBIGNUM

      type register
      type label

      sharing type register = int
      sharing type label = int

      datatype EA = R of register        (* register *)
	          | IMMED of int         (* immediate integer *)
                  | N of label           (* function name *)
                  | S of label           (* string location *)
                  | REAL of label        (* real location *)
                  | LOCAL of register    (* local register *)
                  | EXTERN_REC of string (* external record *)
	          | NAMED of string      (* external function *)
                  | BIGNUM of Bignum.bignum  (* arbitrary precision integer *)
		    
      datatype int2Op = ASHL | ASHR | ORB | ANDB | XORB
		      | MUL | ADD | SUB | DIV | PTRADD
      datatype int1Op = NOTB
      datatype floatOp = FMUL | FADD | FSUB | FDIV
      datatype condition = NEQ | EQL | LEQ | GEQ | LT | GT | ULT (* unsigned *)
      datatype decl = STRING_DECL of string * EA
		    | REAL_DECL of string * EA
      datatype exp = VAL of EA
	           | IB of int2Op * exp * exp
		   | IU of int1Op * exp
		   | ICOND of condition * exp * exp
		   | FCOND of condition * EA * EA
		   | CAND of exp * exp

      datatype stmt = ASSIGN of EA * EA * stmt
	            | ALLOC of (EA * CPS.accesspath) list * EA * stmt
		    | CALL of (EA * EA list * EA list) * EA list *
		               EA option * stmt
		    | CASE of EA * (int * stmt) list 
		    | COND of exp * stmt * stmt
		    | COMMENT of string * stmt
		    | FETCH of EA * EA * EA * stmt
		    | FETCHB of EA * EA * EA * stmt
		    | FLOAT of floatOp * EA * EA * EA * stmt
		    | GOTO of label
		    | INT2OP of int2Op * EA * EA * EA * stmt
		    | INT1OP of int1Op * EA * EA * stmt
		    | JMP of EA list
		    | LABEL of label * stmt
		    | SEQ of stmt list
		    | SET of EA * EA * EA * stmt 
		    | SETB of EA * EA * EA * stmt

      (* FUNC(name,local vars, body, external or static) *)

      datatype function = FUNC of EA * EA list * stmt * bool
      datatype prog = PROG of decl list * function list
  end
