(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                            CAML                                       *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            Inria                                      *)
(*                      Domaine de Voluceau                              *)
(*                      78150  Rocquencourt                              *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* prelude	Basic functions for the CAML system.			 *)
(*		Group effort.						 *)

(* 0. Basic data types *)
(*\
\chapter{Basic values and basic data types}
\index{Concrete data types}
\index{Data types}
\index{Types}
\index{Basic data types}

\begin{bf}
Convention:
\end{bf}

In this manual, the symbol $\clubsuit$ in the left margin of the page,
 will indicate a function (or set of functions) provided by the 
CAML system. These functions are listed in the index, and separately
with their type in the glossary.

\section{Types and values}
\paragraph\ 
Since CAML is strongly typed (i.e. every legal phrase has a type), the 
description of the values of the language is very close to the one
 of the types.

In CAML values are built using very few concepts:
\begin{itemize}
\item {\bf Built-in} values, such as strings or integers.
\item {\bf Definable} data types : these are means to define new values which
are elements of new types, introduced by a {\em type declaration}.
Definable data types fall into two distinct categories:
 \begin{enumerate}
   \item {\bf Sums}, from the mathematical notion of ``disjoint union'':
the disjoint union of two sets A and B is the set C
(classically named A + B) of all the elements belonging
 either to A or to B, marked with a tag indicating whether they come from
A or B.
   \item {\bf Products}, from the mathematical notion of ``cartesian product'':
the cartesian product of two sets A and B is the set C  (classically
named A $\times$ B) of all the pairs whose first
component is an element of A and second component an element of B.
 \end{enumerate}
\end{itemize}

As for values, types may be either

\begin{enumerate}
\item {\bf Built-in} types (such as \verb"string" or
\verb"int").
\item {\bf Defined} types (such as \verb"list").
\end{enumerate}

In short:
\begin{itemize}
\item Values are built using predefined constants or {\bf constructors}
\item Types are similarly built using predefined {\em type} constants or
{\em type} constructors.
\item New type constants and constructors are declared and these declarations
introduce new {\em value} constructors associated with the new type.
\end{itemize}

\subsection{Value Constructors}
\index{Constructors}
\paragraph\ 

Sum type declarations introduce new values, bound to identifiers of a special kind:
the {\em constructors} (the word ``constructor'' being intended to be a shorthand
for ``constructor of values'').

 These constructors may be {\bf constants} or {\bf injections} from a given
type to the type being defined,
and in the latter case they possess a functional type (one calls them 
``functional'' constructors).

Let us give some examples of types predefined in CAML:
\begin{itemize}
 \item The classical type of boolean values contains only two constants 
\verb"true" and \verb"false": thus it may be defined as
\begin{verbatim}
type bool = true | false;;
\end{verbatim}
  This phrase declares the two new constant values \verb"true"
 and \verb"false",
 which are the two constant constructors of the new defined type \verb"bool".
The meaning of the above type declaration is exactly that every value
belonging to the type \verb"bool" must be either \verb"true" or \verb"false".
Thus the symbol \kwd{|} may be read as \verb"or".

In the mathematical sense, we have \verb"bool = {true} + {false}":
 the set \verb"bool" is the disjoint union
of the two singleton sets \verb"{true}" and \verb"{false}".

 \item An enumerated set may be defined with a type declaration, listing all
the elements of the set, and thus introducing these elements in the language
as new constant constructors.

 \item The predefined type \verb"unit" which possesses only one element
denoted by ``\verb"()"'' could be defined as:
\begin{caml}
type unit = ();;
\end{caml}
which defines its single element as a new constant value of the language.

 \item Integer values may be seen as an infinite enumerated set: it is
the infinite sum of the singleton sets
\verb"{0}", \verb"{1}", \verb"{2}", \verb"{3}", \ldots, as suggested by
 \begin{verbatim}
  type int = 0 | 1 | 2 | 3 | .... | -1 | -2 | ....;;
 \end{verbatim}
  This declares an infinite number of constant constructors, i.e.
 exactly all integer values (but, since it is impossible to declare
 infinitely many constants in practice, the type \verb"int" must be predefined).

  \item In the same vein, string values may also be seen as an infinite sum of singleton sets
(all the sets \verb"{s}" where \verb"s" ranges on all the legal strings of
the language (every finite sequence of characters enclosed by doublequotes
\verb|""|)).

  \item Let us write a ``real'' type definition:
\begin{caml}
type variable = Variable of string;;
\end{caml}
In this case we define a new type ``\verb"variable"'', together with
 a new functional constructor ``\verb"Variable"'',
 which injects string values into the new type \verb"variable":
\verb|Variable "x"| belongs to the type \verb"variable".

This is used to distinguish between data types which are semantically
 identical (i.e. contain the same set of values, as the type \verb"variable" 
and the predefined type \verb"string") but must not be identified
 for the sake of clarity (and safety) of the programs manipulating
 values of such isomorphic types.

 \item Basic cartesian product is defined in CAML: elements of
type \verb"t1 * t2" are pairs of values of type \verb"t1" and \verb"t2":
\begin{caml_example}
(1,true);;
\end{caml_example}
\end{itemize}


\subsection{Type constructors}
\index{Type constructors}
\paragraph\ 
 Types are built using constructors as well, which may be either constant 
(as predefined types \verb"string" or \verb"num") or functional.
 For example \verb"list" is such a functional type constructor,
since from type argument \verb"string" it builds the type \verb"string list"
and from type argument \verb"bool" the type \verb"bool list".
Do not be confused by the fact that the constructor \verb"list" is postfixed.
It constructs types in the same way as the constructor \verb"Variable"
from our previous type definition
builds values:
\verb"bool list" is strictly analogous to \verb|Variable "foo"|, except that
a type is built instead of a value.

The predefined type constructor ``\verb"*"'' has two type variables as
arguments, since it builds (the type of) the cartesian product of two types.

Type constructors may have as many arguments as desired.

\section{Sum Types}
\index{Sum Types}

\subsection{Simple examples}
\paragraph\ 

First we define the type of string lists: a list may be either empty
(then we denote it with the constant constructor \verb"Empty") or a pair of
its first element in front of another list (then we denote it with the
functional constructor \verb"Add"):
\begin{caml_example}
type string_list = Empty | Add of (string * string_list);;
\end{caml_example}

\begin{caml_example}
let l = Empty;;
Add ("toto",Add ("tata",l));;
\end{caml_example}

The drawback of the type \verb"string_list" is that we cannot reuse it for 
another kind of list: we ought to define another
type to get list of integers, or list of whatever.
In fact, it is possible to define a {\em polymorphic} type list which will
 provide us with completely general list constructors.

 When a 
functional {\em type} constructor is defined it takes as argument a {\em type}
 variable.
Such a type variable is denoted by an identifier (element of the lexical class
 \syncat{Ident}) preceded by a \kwd{'} symbol to distinguish it from ordinary
constant type names. For instance the empty list will have type \verb"t list"
for every type \verb"t", which will be written \verb"'t list".

\begin{caml_example}
type 't List = Empty | Add of 't * 't List;;
\end{caml_example}

Now we have:
\begin{caml_example}
Empty;;
Add (1,Empty);;
Add ("oui",Empty);;
\end{caml_example}

\subsection{Pattern matching}
\index{Pattern matching}
\paragraph\ 

Constructors have the important property that they allow {\em pattern matching} on
values of their types. Pattern matching is a kind of switch among
 different cases
according to an argument value. For instance using the (implicit) constructor
\verb"1", you may write the following function which will decide whether its
argument is in fact the constructor \verb"1":
\begin{caml_example}
let is_one = function 1 -> true | x -> false;;
\end{caml_example}

Cases of matches are clauses \syncat{pattern} \kwd{->} \syncat{expression}.
\begin{itemize}
\item The value of
\syncat{expression} is returned if the corresponding clause is selected.
\item If
\syncat{pattern} is reduced to a variable (such as \verb"x" in the 
preceding example) then every value of the type matches the corresponding
pattern.
\item Patterns of clauses are tried in the order of presentation. Thus
permutation of clauses generally affects the semantics:
\verb"is_one" is equivalent to
\begin{caml_example}
let is_one x = if x = 1 then true else false;;
\end{caml_example}
but if we change the order of the clauses:
\index{Unused match case}
\begin{caml_example}
let is_one = function x -> false | 1 -> true;;
\end{caml_example}
the function will always return \verb"false" since every value of type
\verb"num" matches the first pattern \verb"x". 
 Thus the second clause will never be tried, hence the system warning.

\item Moreover pattern matching is {\em specific} to constructors and variables. You {\em cannot match with computed values as patterns}.
\end{itemize}
Consider:
\begin{caml_example}
let int_one = 1;;
let is_one = function int_one -> true | x -> false;;
\end{caml_example}
This function will allways return \verb"true", since \verb"int_one" is not a
declared constructor (either explicitly in a type definition, or implicitly
as one of the basic constant values of the language). It is then understood by the
system as a variable, distinct from the global one with the same name
which is bound to the value 1,
and again all values of type \verb"num"
match the first clause.
\par\noindent
\begin{bf}
Beware:
\end{bf}
\par\noindent
\verb"--" To prevent such mistakes it is a good habit to use
capitalized identifiers for constructors and labels, and lower case identifiers
for variables.

In addition to constructors and variables, the
special symbol \kwd{\_} (\verb"wildcard") may appear in patterns: it stands for ``whatever''.
The function \verb"is_one" may then be rewritten with such a wildcard:
\begin{caml_example}
let is_one = function 1 -> true | _ -> false;;
\end{caml_example}

The last purpose of patterns is to effect bindings of values or part of
values, needed in the right hand-side of the clause. For instance to compute
the sum of all the elements of a list:
\begin{caml_example}
let rec sum = function
  Empty -> 0
| Add (x,l) -> x + (sum l);;
\end{caml_example}
There we have bound simultaneously the head and the tail of the list argument to
\verb"x" and \verb"l" in the left hand-side of the clause. Notice that these
bindings are done recursively with the proper values:
\begin{caml_example}
sum (Add (1,Add (2, Add (3,Empty))));;
\end{caml_example}

 This binding effect of pattern matching together with proper use of
``wildcard'' clauses is very powerful and elegant.

\begin{caml_example}
let is_not_empty = function
    Add (_,_) -> true
  | _ -> false;;
\end{caml_example}

 Moreover patterns may overlap (a pattern may be an
{\em instance} of another one) and in such a case ambiguities are solved using
order of presentation: if a given value matches 2 clauses the first one
 is chosen. For instance:

\begin{caml_example}
let rec sum = function
  Empty -> 0
| Add (x,Empty) -> x
| Add (0,l) -> sum l
| Add (x,(Add (0,l))) -> sum (Add (x,l))
| Add (x,l) -> x + (sum l);;
\end{caml_example}
the value \verb"Add(0,Add (0,Empty))" matches the 3 last clauses.

These implicit rules can simplify the number of necessary clauses. Consider the
 following function which appends lists. We can write it as follows:
\begin{caml_example}
let rec append_lists = function
  Empty,Empty -> Empty
| Empty,L -> L
| L,Empty -> L
| Add (x,l),L1 -> Add (x, append_lists (l,L1));;
\end{caml_example}

We can discard the first clause and still get the same function:
\begin{caml_example}
let rec append_lists = function
  Empty,L -> L
| L,Empty -> L
| Add (x,l),L1 -> Add (x, append_lists (l,L1));;
\end{caml_example}

the value \verb"Empty,Empty" matches the 2 first clauses, and the first one
will be applied. In our example this is not very important,
 but you must be careful about the semantics of pattern matching, when
clauses overlap.

\index{Partial match}
\subsection*{Partial matches}
\paragraph\ 
So far the set of clauses we used were always {\em exhaustive}: every value
of the source type matched at least one clause.

In fact pattern matching can be not completely exhaustive as in:
\begin{caml_example}
let is_agreement = function "yes" -> true | "no" -> false;;
\end{caml_example}
Since there are more than 2 constant constructors in the type \verb"string"
the match may fail:
\begin{caml_example}
is_agreement "ok";;
\end{caml_example}

When encountering this kind of pattern matching,
 the compiler adds a safeguard to prevent runtime failures, namely
an extra clause
\begin{verbatim}
| _ -> raise match_failure
\end{verbatim}
 This exception is then 
trapped by the toplevel of the language which reports the above message
(about exceptions see the section \ref{exceptions}).
\par\noindent
\begin{bf}
Beware:
\end{bf}
\par\noindent
\verb"-- " Remember that it is a nasty programming habit to allow partial
 matches into your programs.
 Do your best to avoid them since they often lead to unexpected
errors.

\subsection*{Complex patterns}
\paragraph\ 

A pattern may be as complex as desired including as many constant constructors,
wildcards and variables as necessary.
\index{\verb"as"}
\index{\verb"aliases in patterns"}
\index{\verb"synonyms in patterns"}
Bindings may occur in any sub-pattern of a
pattern using the syntactic construction \verb"as", which names a part of 
a pattern for further use in the right-hand part of the clause.
\begin{caml_example}
let rec remove_but_one = function
  Add(x,(Add (y,l) as L)) -> remove_but_one L
| L -> L;;
\end{caml_example}

\par\noindent
\begin{bf}
Beware:
\end{bf}
\par\noindent
\verb"-- " All patterns in CAML must be \verb"linear", that is to say {\em no
variable can appear more than once in a pattern}:
\begin{caml_example}
let egal = function (x,x) -> true | _ -> false;;
\end{caml_example}

\subsection{Disjoint union}
\paragraph\ 

The general type of disjoint union of two types
 can be defined in CAML as a polymorphic type definition:
\*)
type 't1 + 't2 = Inl of 't1 | Inr of 't2;;

(*\

You may notice that \verb"+", as a type constructor, gets two parameters
(or more exactly a pair of parameters).

With \verb"Inl" you inject a value into the left summand of the type
and similarly for \verb"Inr" into the right summand.
\begin{caml_example}
let x = Inl 1;;
\end{caml_example}
There is no typechecking constraint on the right summand, thus it remains
 polymorphic. If it is not desired,
it is possible to add an explicit type constraint to get an instance of the
 above type for \verb"x":
\begin{caml_example}
let (x':num + bool) = Inl 1;;
let (y:num + bool) = Inr false;;
let boolean_value_of = function
    Inl 1 -> true | Inl _ -> false | Inr b -> b;;
boolean_value_of x';;
boolean_value_of y;;
boolean_value_of x;;
\end{caml_example}

\section{Products}
\index{Products}
\index{records}
\paragraph\ 

Product type declarations are similar to sum type declarations, but one defines
 the format of a record of values with labelled fields. Instead of 
constructors one declares new labels\footnote{which have not to be different from all other record's labels.} to name the fields of the record:
\begin{caml_example}
type car = {Weight:num; Speed:num; Brand:string};;
\end{caml_example}
This defines a new type of records with 3 fields, which must contain
respectively a \verb"num" for the labels \verb"Weight" and \verb"Speed", and a
string for the label \verb"Brand".
 Then we can build values of type \verb"car" by just
filling in the fields of the record with any value compatible with the
declared type of the field:
 
\begin{caml_example}
let R25 = {Weight=1200; Speed=220; Brand="Renault"};;
let BX16 = {Weight=1050; Speed=200; Brand="Citroen"};;
\end{caml_example}

As for sum types, accessing is done via pattern matching:
\begin{caml_example}
let weight_of_car = function
    {Weight = w; Speed = _; Brand = _} -> w;;
weight_of_car R25;;
\end{caml_example}

The semantics of variables and wildcards in pattern matching is 
identical to those for sum types,
but for products there is in addition a special ellipsis
 for non ``interesting'' fields:
an underscore just following a field delimitor \kwd{;}, that is
``\verb"; _"'' means ``whatever may be the other fields and  their contents''
 (more precisely this \verb"_"
stands for {\em 0 or more} fields).
\index{Ellipsis in records}
\index{Wildcard in records}
 The preceding example may be rewritten as:
\begin{caml_example}
let weight_of_car = function {Weight=w; _} -> w;;
weight_of_car R25;;
\end{caml_example}

\index{Dot notation}
\index{\verb"."}
Another way to access the content of a field is provided with the \kwd{.}
notation:
\verb"r.field_name" returns the content of the field \verb"field_name" into
 the record \verb"r".

\begin{caml_example}
R25.Speed > BX16.Speed;;
\end{caml_example}

Finally notice that pattern matching with sum and product types may be mixed arbitrarily.
\begin{caml_example}
type loading = {Weight : num; Goods : string};;
type vehicle = Car of car | Lorry of car*loading;;
let weight_of_vehicle = function
  Car {Weight=w;_} -> w
| Lorry ({Weight=w;_},{Weight=w';_}) -> w+w';;
\end{caml_example}

\subsection*{Cartesian product}
\paragraph\ 
This is the common binary product type: the cartesian product t1 $\times$ t2
 of two types t1 and t2. That is, the type of all the pairs which may be
 built with elements of t1 and t2. It is a polymorphic product type
 definition with two type variables:
\begin{caml_example}
type ('t1,'t2) pair = {Fst:'t1; Snd:'t2};;
{Fst = 1; Snd = "oui"};;
\end{caml_example}

\section{Type abbreviations}
\index{Type abbreviations}
\paragraph\ 

These are mainly used to make type information easier to read and enter.
They are introduced with an \kwd{==} symbol between a type and its
 abbreviated version:
\begin{caml_example}
type single == unit;;
\end{caml_example}
Then you get:

\begin{caml_example}
();;
\end{caml_example}

An abbreviation may be ``active'' or ``inactive'':
\begin{caml_primitive}
echo_abbrev
echo_abbrevs
\end{caml_primitive}
With \verb|echo_abbrev "single" false| you disable the reduction of the
type abbreviation named \verb"single", and you revert to normal reduction
mode with \verb|echo_abbrev "single" true|.
\begin{caml_example}
echo_abbrev "single" false;;
\end{caml_example}

With \verb"echo_abbrevs" you can switch globally (i.e. for all
 abbreviations) the reduction process.

\section{Exception handling}
\index{Exceptions}
\paragraph\ 
\label{exceptions}
Apart from values and types, the basic notions of {\em exceptions}
and exceptional values
 are used to control the way computation runs. One can think of
 exceptional values as signals which are sent, and which may be received by
 functions. Sending is performed by the \kwd{raise} construct and
receiving by the \kwd{try} \ldots \kwd{with} construct.

Exceptions are typechecked, and thus have to be declared:
\begin{caml_example}
exception Failure of string;;
\end{caml_example}
Now the exception \verb"Failure" can be raised:
\begin{caml_example}
raise (Failure "uncaught");;
\end{caml_example}
and trapped:
\begin{caml_example}
try raise (Failure "uncaught")
with Failure "uncaught" -> 1;;
\end{caml_example}
The \kwd{with} part of a \kwd{try} \ldots \kwd{with} is just an ordinary
match on exceptional values.

Now two functions may interact through exception handling:
\begin{caml_example}
let divide_primitive (x,y) =
    if y = 0 then raise (Failure "overflow")
    else x/y;;
let divide_without_overflow (x,y) =
    try divide_primitive (x,y) with
     (Failure "overflow") -> 0
;;
divide_without_overflow (1,0);;
\end{caml_example}

This exception mechanism\footnote{This mechanism is (almost completely) independant from the underlying operating system.} is as fast ordinary procedure call, it even can be 
faster.
When returning from a recursive function with an exceptional value,
 all the recursive calls are discarded at once.

\section{Mutable and lazy values}
\index{Mutable objects}
\index{References}
\paragraph\ 

When declaring a type, some constructors or labels can be qualified as 
\kwd{mutable} or \kwd{lazy}:
\begin{itemize}
\item[-] \kwd{mutable} means that the argument of the constructor (or the
content of the field of the label) may be dynamically (and {\em physically})
updated: the corresponding value is still a constant (i.e. bound to the same
address in the computer memory) but its ``value'' may change (i.e. the
content of the address may be updated).
\item[-] \kwd{lazy} means that  the argument of the constructor (or the
content of the field of the label) will not be computed {\em at once}, but
if and only if this argument is absolutely necessary for further
computation. In this case it will be effectively computed, but {\em only
once}, and the address bound to the argument will be physically updated with
the computed value.
\end{itemize}

Mutable values are used mainly to maintain a global state which changes
dynamically when the program runs. Global variables bound to mutable values
allow as well information passing between as many function as desired:
every function may read (and write) the global variables to get
(and give) information about
the current state of the computation. This may be simulated with extra argument
parameters and results, but at the price of useless effort
 to maintain and read the programs.

Lazy values mainly enable potentially ``infinite'' data
structures. Values of that kind cannot be completely evaluated and hence
they are progressively computed (``expanded'' in a ``lazy'' way) as long as
the computation needs it.

Another advantage of lazy values is the automatic avoidance of useless
computations (not needed for the final result). This implies that so-called
``side-effects'' must be avoided in lazy values, because they may never be
evaluated at all or at least in a surprising order.

\section{Mutable values in sums}

\subsection{References}
\index{\verb"ref"}
\paragraph\ 

We first define the basic type of mutable values:
\*)

type 't ref = mutable ref of 't;;

(*\
Now we may define a counter using a reference:
\begin{caml_example}
let counter = ref 1;;
\end{caml_example}

\subsection{Occurrences}
\index{Occurrences}
\paragraph\ 
We present the basic mechanism to update mutable values. Other ways to update
values will be presented afterwards.

To update mutable values one has to name the precise place where the value
has to be ``changed''.
 This is done by special annotations of patterns using the
keyword \kwd{at}, followed by an identifier naming the occurrence. For instance
\begin{verbatim}
match ... with ref (y at occ) -> ...
\end{verbatim}
designates \verb"y" as being at occurrence named \verb"occ", and then it is
possible to update \verb"y" in the right-hand side of the match.

The physical replacement of a value at a designated occurrence \verb"occ"
has the following syntax:
\begin{verbatim}
at occ <- new_value
\end{verbatim}
and the semantics is that \verb"new_value" is inserted at the place
 designated
by the occurrence \verb"occ".

The construct \verb"at occ <- new_value", may be read
``at occ put new\_value'', or ``occ receives new\_value''.

 The value of the whole construct is the
updated value (i.e. \verb"new_value").

\begin{caml_example}
match counter with ref (_ at occ) ->
at occ <- 2;;
counter;;
\end{caml_example}


\subsection{Primitives for references}
\paragraph\ 
References are a very commmon instance of mutable values. Thus, we provide
specialised functions to update references and to find their contents:
\begin{caml_primitive}
prefix !
prefix :=
\end{caml_primitive}

\kwd{prefix} allows the distinction between an ``infix''
 identifier and its ``prefix'' form. This is explained in section \ref{infixes}.
\begin{itemize}
\item \verb"prefix !" (read as ``prefix'' ``deref'') returns the content of a
reference. It is defined as:
\begin{caml}
let prefix ! (ref x) = x;;
\end{caml}
\item \verb"prefix :=" is used to assign a value to a reference. It may be defined as
\begin{caml}
let x_ref := x =
    match x_ref with ref _ at occ -> at occ <- x;;
\end{caml}

Or more concisely by
\begin{caml}
let (ref _ at occ) := x =
 at occ <- x;;
\end{caml}
\end{itemize}

\subsection{Using references}

\subsubsection*{Symbol generation}
\paragraph\ 
We define a generator of symbols in the spirit of the well-known Lisp
``gensym''\footnote{Except that in Lisp, gensym returns new, unbound {\em identifiers}, not strings.}. We define simultaneously the symbol generator and the function
 which resets it: \verb"reset_gen_sym".

 We use a local reference counter which is shared between
\verb"gen_sym" and \verb"reset_gen_sym", incremented each time \verb"gen_sym"
 is called and reset to 0 when necessary by \verb"reset_gen_sym".
\begin{caml_example}
let gen_sym,reset_gen_sym =
 let sym_counter = ref 0 in
 (function () -> "symbol"^
                 (string_of_num (sym_counter:=!sym_counter+1))),
 (function () -> sym_counter:=0);;
gen_sym();;
gen_sym();;
reset_gen_sym();;
gen_sym();;
\end{caml_example}

\subsubsection*{For loop}
\index{for loop}
\paragraph\ 
Combined with exception handling, references allow us to define a ``for'' loop.
We define first a specific exception to be raised when a next step is
required:
\begin{caml_example}
exception next of num ref;;
\end{caml_example}

Then the iterator itself:
\begin{caml_example}
let for (index,start,stop,step) =
 index:=start;
 let rec loop body =
  if !index > stop then ()
  else
  (try body index
   with next j ->
    (* Test physical identity of the two references index and j *)
    if j==index
    (* This is our index: increment it and continue *)
     then (index:=!index+step;loop body)
    (* This is not our index: exception is propagated *)
     else raise next j)
 in loop;;
\end{caml_example}

Then we can use two interleaved loops, with two indexes \verb"i" \verb"j":
\begin{caml_example}
for (ref 0,1,2,1)
 (fun i ->
   (for (ref 0,1,4,2)
     (fun j ->
       print_string"i = ";print_num !i;print_space();
       print_string"j = ";print_num !j;print_newline();
       raise next j));
   raise next i);;
\end{caml_example}

\subsubsection*{Recursivity and physical modifications}

Using mutables you may easily create cyclic structures, even functional ones:
\begin{caml_example}
(* Creation of a dummy closure for fib *)
let fib_dummy = ref (I : num -> num);;
(* The body of fib refers to fib_dummy *)
let fib_body =
 function 1 -> 1 | 2 -> 1 | n -> !fib_dummy(n-1)+!fib_dummy(n-2);;
(* Which is updated to fib_body to create a cyclic closure *)
fib_dummy:=fib_body;;
(* Now fib is defined to be the cyclic closure *)
let fib = !fib_dummy;;
fib 20;;
\end{caml_example}

Furthermore, we can avoid all these global declarations and simply write
a recursive delcaration with no \verb"let rec":
\begin{caml_example}
let fib = let fib=ref I in fib:=
 function 1 -> 1 | 2 -> 1 | n -> !fib(n-1)+!fib(n-2);;
fib 20;;
\end{caml_example}

\subsection*{Forward and autoload}
\index{Forward declarations}
\index{Autoload declarations}
\index{forward}
\index{autoload}
This kind of dummy declarations is extended to 2 toplevel constructs:
\begin{itemize}
\item \kwd{forward} \syncat{Ident} \kwd{:} \syncat{Type} declares a
 {\em function} of the given type, which has no associated value. The given
 type must be unifiable to a functional one. For instance:  
\begin{caml_example}
(* Declaring a forward function *)
forward foo : num -> num;;
(* Binding this forward to a value, whose type must be
   more general than the declared type *)
let foo x = x;;
foo 1;;
\end{caml_example}
This construct may be used to built cyclic closures as well:
\begin{caml_example}
forward fib : num -> num;;
let fib = function
           1 -> 1 | 2 -> 1 | n -> fib(n-1) + fib(n-2);;
fib 20;;
\end{caml_example}
There is no loss of efficiency with this kind of declarations, compared
with a classical ``\verb"let rec"''\footnote{At least in the present implementation.}.
\item \kwd{autoload} \syncat{Ident} \kwd{:} \syncat{Type} \kwd{from}
\syncat{file\_name} declares a
 {\em function} of the given type, which has no associated value. This value
will be found in the \syncat{file\_name} compiled code file:
at the first call to the function, the file is loaded
and the corresponding value bound to the function.
\end{itemize}
These two constructs are extended to simultaneous declarations, using the
keyword \kwd{and} between successive function names.

\section{Mutable values in products}
\paragraph\ 

We start with a basic example: mutable pairs, and then a full example with
queues.

\subsection{Mutable pairs}
\index{Pairs (mutable)}
\paragraph\ 

Mutable pairs are records of two mutable fields, let us name them \verb"Car"
and \verb"Cdr"\footnote{Which are the acronyms of Content of Address part of the Register and Content of Decrement part of the Register. Pourquoi pas ?}:
\begin{caml_example}
type ('a,'b) mpair = {mutable Car : 'a; mutable Cdr : 'b};;
\end{caml_example}
For mutable records, construction and accessing are as usual:
\begin{caml_example}
let a_pair = {Car = 1; Cdr = "oui"};;
a_pair.Car;;
\end{caml_example}

\subsection{Updating mutable values in products}
\index{Assignment in records}
\paragraph\ 

Update is also done by pattern matching:
\begin{caml_example}
match a_pair with {Car = _ at occ; _} -> at occ <- 2;;
a_pair;;
\end{caml_example}

 We have an extension of the ``dot'' notation to assign fields of
records: r\verb"."label \verb"<-" expression will update the field named
label with the value of expression.
\begin{caml_example}
a_pair.Cdr <- "non";;
a_pair;;
\end{caml_example}

\subsection{Queues}
\index{Queues}
\paragraph\ 
As an example we will define an efficient implementation of ``fifo''
(first-in first out) stacks or queues.
 Queues are similar to lists, except that elements are not added in front
 but at the end of the list. For this purpose we will use mutable list cells
 and modify physically the end of the list when adding an element.

Hence we define the type of mutable list cells. Let us first define recursively
the type of mutable cells and mutable queue bodies (list of cells)
which may be empty or a ``cons'' of some cell:
\begin{caml_example}
type 'a qbody = qnil | qcons of 'a cell
and 'a cell = {mutable Head : 'a ; mutable Tail : 'a qbody};;
\end{caml_example}

Now we can define queues which possess an insert pointer, which is the cell
where a new element
can be added and a ``body'' which contains the items enqueued:
\begin{caml_example}
type 'a queue =
     {mutable Insert_point : 'a qbody;
      mutable Queue_body : 'a qbody};;
\end{caml_example}

Now we present the function which given a cell and a queue will enqueue the cell:
if the queue is empty then its body becomes the cell itself, and we store
the cell in the insert point as well, since the insert point must be a pointer
to the end of the queue. Otherwise we update the insert point with the new cell.
This will automatically update the body of the queue, due to the physical
sharing between the last cell of the queue and the insert point:
\begin{caml_example}
let enqueue cell = function
   {Queue_body=qnil;_} as queue ->
     queue.Insert_point <- queue.Queue_body <- cell

 | {Queue_body=qcons _; Insert_point=qcons insert} as queue ->
     insert.Tail <- cell;
     queue.Insert_point <- cell

 | _ -> failwith "ill built queue";;
\end{caml_example}

The converse function successively gets the enqueued elements. It fails
if the queue is empty, otherwise it returns the head of the queue and updates
the queue body with the tail of the queue.
\begin{caml_example}
let dequeue = function
   {Queue_body=qnil;_} -> failwith "empty queue"
 | {Queue_body=qcons {Head=x;Tail=tl}; _} as queue ->
      queue.Queue_body <- tl;x;;
\end{caml_example}

Now we define the allocator of cells and queues for numbers:
\begin{caml_example}
let new_num_cell (x:num) = qcons {Head=x;Tail=qnil};;
let enqueue_num x q = enqueue (new_num_cell x) q;;
let new_num_queue () =
    {Insert_point=qnil;Queue_body=(qnil:num qbody)};;
let q = new_num_queue ();;
\end{caml_example}
\begin{caml_example}
enqueue_num 1 q;enqueue_num 2 q;enqueue_num 3 q;
enqueue_num 4 q;enqueue_num 5 q;;
(* For advanced users: may be skipped at first reading *)
show q;;
print_num (dequeue q);print_num (dequeue q);
print_num (dequeue q);print_num (dequeue q);
print_num (dequeue q);print_num (dequeue q);;
\end{caml_example}

\section{Restrictions on mutable values and exceptions}
\index{Polymorphic references}
\index{Cannot generalize}
\paragraph\ 
There is a restriction on the type of mutable values {\em built} in a program:
 namely they can be of
any monomorphic type, or must possess a polymorphic type with no free
variables.

For instance, the empty list is polymorphic:
\begin{caml_example}
[];;
\end{caml_example}

Thus we can use it as a list of any type:
\begin{caml_example}
let l = [];;
1::l;;
true::l;;
\end{caml_example}

Hence we cannot define a mutable value including it:

\begin{caml_example}
let x = ref [];;
\end{caml_example}
This is done for sake of safety since, if we could define a polymorphic
reference containing the empty list we may then add to the list any
element of any type, and this is not compatible with the definition of the
type list which states that lists are indeed homogeneous:

Imagine that the types of polymorphic reference were generalized:
\begin{caml}
let l = ref [] in
l:=1::!l;;
l:=true::!l;;
\end{caml}
At this step \verb"l" would contain a number and a boolean!

This should be improved in the future using the so-called notion of ``weak''
type variables.

Notice that it is still possible to use polymorphic mutable values if there
is no need to generalize their types:
\begin{caml_example}
let l = ref [] in
l:=1::!l;;
\end{caml_example}
and that the restriction is only for the {\em construction} 
 of mutable values.

One may freely define a function with polymorphic mutable arguments, as long
as no polymorphic mutable values are {\em built} inside the body of the 
function and returned as part of the result:
\begin{caml_example}
(* This function does not built mutable values *)
fun (ref x) -> x;;
(* This function attempts to build polymorphic mutable values *)
fun x -> ref x;;
\end{caml_example}

Finally, remember that the generalization of types is implicit
at the end of every toplevel phrase, and is performed
 before each ``in part'' of a ``let'' construct in the CAML programs.
\par\noindent
\begin{bf}
Beware:
\end{bf}
\par\noindent
\verb"--" The same restriction exists for exceptions.
\section{Overloading of labels}
\index{overloading}
\paragraph\ 

Labels of records may be shared by several product types: this is a kind
of {\em overloading} since a label may have several declared types.
 Thus the CAML
typechecker uses a strategy to find out to which type a label belongs 
in a particular use of the label.

The main point to keep in mind is that the system allows any overloading,
 as soon as the proposed CAML phrases are not {\em completely} ambiguous.

Consider the types \verb"foo" and \verb"bar" sharing the same labels:
\begin{caml_example}
type foo = {L1 : string ; L2 : string};;
type bar = {L1 : bool; L2 : string};;
\end{caml_example}

Then the following are clearly not ambiguous:
\begin{caml_example}
let x = {L1="poi"; L2="sdcs"};;
let y = {L1 = true; L2="poi"};;
\end{caml_example}

Nor is the following function (although it is a bit harder to prove this fact):
\begin{caml_example}
fun {L1=x;L2=y} -> x=y;;
\end{caml_example}

Nevertheless, the labels of the types \verb"bar" and \verb"foo"
can create unresolvable ambiguities as in:
\begin{caml_example}
function {L1=x; _} -> x;;
\end{caml_example}

\subsection*{Open overloading}
\index{\verb"#open overloading"}
\paragraph\ 

If you heavily use overloading over labels, you may ask the system to
``open'' overloading. That is to say that in case of complete ambiguity it
chooses the last defined type and continues the typechecking (emitting a
warning).
\begin{caml_example}
#open overloading true;;
function{L1=x; _} -> x;;
\end{caml_example}
\par\noindent
\begin{bf}
Beware:
\end{bf}
\par\noindent
\verb"-- " This option should not be exercised in common programming situations, since
this may lead to very complex typechecking problems,
 where more than one type may be acceptable for the same phrase!

Consider the following legal type definition:
\begin{caml_example}
type rec0 = {lab:num;label:rec0}
and  rec1 = {lab:num;label:rec2}
and  rec2 = {lab:num;label:rec1};;
\end{caml_example}

The types \verb"rec0", \verb"rec1" and \verb"rec2" share exactly
the same labels, and in each type, the label \verb"lab" contains values of type
\verb"num".
\begin{caml_example**}
#open compilation true;;
max_print_depth 4;;
limit_print_depth 20;;
\end{caml_example**}

Now consider the following legal phrase:
\begin{caml_example}
let rec x = {lab=1;label={lab=2;label=x}};;
\end{caml_example}
It is completely ambiguous, and the default strategy of the system
 for resolving ambiguities has found \verb"rec1"
for the expression \verb"{lab=1;label={lab=2;label=x}}". Thus
\verb"x" has been declared with type \verb"rec1".

But now, just add a type constraint on \verb"x", constraining \verb"x" to be
of type \verb"rec0":
\begin{caml_example}
let rec (x:rec0) = {lab=1;label={lab=2;label=x}};;
\end{caml_example}
\verb"rec0" is not an erroneous type for \verb"x"!

Moreover constraining \verb"x" to be of type \verb"rec1" or \verb"rec2" is still
 correct!
\begin{caml_example}
let rec (x:rec1) = {lab=1;label={lab=2;label=x}};;
\end{caml_example}
\begin{caml_example}
let rec (x:rec2) = {lab=1;label={lab=2;label=x}};;
\end{caml_example}
\begin{caml_example**}
#open compilation false;;
#open overloading false;;
limit_print_depth 55;;
max_print_depth 35;;
\end{caml_example**}

\section {Lazy values}
\index{Laziness}
\paragraph\ 

\subsection{Freezing values}
\index{Freeze}
\paragraph\ 

We define the type of ``frozen'' values, by using the lazy constructor
 \verb"Freeze":
\begin{caml_example}
type 'a frozen = lazy Freeze of 'a;;
\end{caml_example}

Now we can evaluate:
\begin{caml_example}
let frozen_two = Freeze (succ 1);;
\end{caml_example}
which does not compute the application \verb"succ 1". Instead, the CAML system
has created a ``thaw'' (an unevaluated application),
 which is denoted by the toplevel pretty-printer with a \verb"*"
symbol. Notice that this is useless in the case of immediate constants
 involving
no computation. In practice this is optimized:
\begin{caml_example}
let one = Freeze 1;;
\end{caml_example}

Evaluation of frozen values is done when {\em accessing} them (via pattern
 matching in right-hand side of clauses).
 Thus a general forcing function for elements of type
 \verb"frozen" is:
\begin{caml_example}
let Unfreeze (Freeze x) = x;;
\end{caml_example}
since it accesses \verb"x":
\begin{caml_example}
Unfreeze frozen_two;;
\end{caml_example}

As soon as forced, a frozen value is updated:
\begin{caml_example}
frozen_two;;
\end{caml_example}

Notice that the pattern matching process does not force the evaluation of
 thaws, if there is no need to access them:
\begin{caml_example}
let Id (Freeze x as thaw) = thaw;;
Id (Freeze (succ 1));;
\end{caml_example}
This is not always the case, compare with:
\begin{caml_example}
let is_frozen_two = function
   Freeze 2 -> true
 | _ -> false;;
let y = (Freeze (succ 2));;
is_frozen_two y;;
y;;
\end{caml_example}

\subsection{Lazy pairs}
\paragraph\ 

The type of lazy pairs is a bit more complex:
\begin{caml_example}
type ('a,'b) lpair = {lazy Lfst: 'a; lazy Lsnd:'b};;
\end{caml_example}

Let us define a lazy pair:
\begin{caml_example}
let lazy_pair = {Lfst=succ 1;Lsnd=succ 2};;
\end{caml_example}

and destructors for lazy pairs:
\begin{caml_example}
let lfst {Lfst=x;Lsnd=_} = x
and lsnd {Lfst=_;Lsnd=x} = x;;
\end{caml_example}

Now notice the {\em minimal} forcing done by \verb"lfst" and \verb"lsnd":
\begin{caml_example}
lfst lazy_pair;;
lazy_pair;;
lsnd lazy_pair;;
lazy_pair;;
\end{caml_example}

Using the laziness of this type you may then recursively define a function
 and a constant value using this function. We recursively define a lazy pair
whose components are the function and the constant value which uses it,
 as the lazy pair of 2 expressions. This
builds frozen values which are forced by the toplevel when defining the two
global identifiers:
\begin{caml_example}
let rec {Lfst=fact5 ; Lsnd=fact} =
    {Lfst=fact 5;
     Lsnd=function 1 -> 1 | n -> n * fact (n-1)};;
\end{caml_example}


\subsection{Lazy lists or streams}
\index{Streams}
\paragraph\ 

Now we continue with lazy lists, which will allow us to compute with
 potentially infinite objects:
\begin{caml_example}
type 'a stream = {lazy Hd : 'a ; lazy Tl : 'a stream};;
\end{caml_example}

We need a \verb"map" functional to apply a function on each element of a
 stream:
\begin{caml_example}
let rec map_stream f = function
 {Hd=x;Tl=t} -> {Hd=f x;Tl = map_stream f t};;
\end{caml_example}


\subsection{Potentially infinite lists}
\paragraph\ 

The stream of natural numbers begins with 0 and its tail is just the mapping
of the successor function on itself!
\begin{caml_example}
let rec (Freeze Nat) =
    Freeze {Hd=0;Tl = map_stream succ Nat};;
\end{caml_example}

Notice the use of \verb"Freeze" to be sure that lazy values are built,
which are automatically forced by the toplevel when accessing \verb"Nat"
(while defining it).

Notice that this is not magic:
\begin{caml_example}
let rec (Freeze x) = (Freeze x);;
\end{caml_example}

The system reports ``undefined'' when a thaw needs its own value to be
 completely evaluated.

The {\em n}th element of a stream is computed by:
\begin{caml_example}
let rec nth_stream n {Hd=h;Tl=t} =
 if n<=1 then h
  else nth_stream (n-1) t;;
nth_stream 5 Nat;;
\end{caml_example}

A simple recursive function will display the beginning of a stream up to a
 given integer rank:
\begin{caml_example}
let rec display_num_stream n =
 if n = 0 then (fun _ -> ())
  else
  (function {Hd=x;Tl=t} ->
    display_num x;display_string" ";display_num_stream (n-1) t);;
display_num_stream 10 Nat;;
\end{caml_example}
Notice the automatic update of \verb"Nat":
\begin{caml_example}
Nat;;
\end{caml_example}

To compute the stream of even numbers, we just have to add \verb"Nat" to
\verb"Nat", i.e. to apply the function \verb"add" to each pair of elements
of \verb"Nat" and of another instance of \verb"Nat" in turn.

Thus we need a functional to apply a function with two arguments on two
 streams:
\begin{caml_example}
let rec map_stream2 f = function
  {Hd=x1;Tl=t1} ->
 function {Hd=x2;Tl=t2} ->
  {Hd=f x1 x2;Tl = map_stream2 f t1 t2};;
\end{caml_example}
\begin{caml_example}
nth_stream 20 (map_stream2 add Nat Nat);;
\end{caml_example}

Now we define the stream of Fibonacci numbers: it starts with the value 1
 then 1 again, and afterwards the {\em n}th number is obtained by addition of the 
{\em n}th-1 and {\em n}th-2 numbers of the stream:
\begin{caml_example}
let rec (Freeze Fib) =
    Freeze {Hd=1; Tl={Hd=1; Tl=map_stream2 add Fib Fib.Tl}};;
display_num_stream 10 Fib;;
\end{caml_example}

In the same spirit it is possible to define the stream of prime numbers.

We use the well known sieve of Erathostenes:
\begin{caml_example}
let rec sieve p = sieve_rec where rec sieve_rec = function
 {Hd=x;Tl=t} -> if x mod p = 0 then sieve_rec t
                 else {Hd=x;Tl=sieve_rec t};;
\end{caml_example}

Now the function \verb"primes" which will recursively sift the tail of the
stream of natural numbers:
\begin{caml_example}
let rec primes = function
 {Hd=x;Tl = t} -> {Hd=x;Tl= primes (sieve x t)};;
\end{caml_example}

Now we just need to sift natural numbers starting from 2:
\begin{caml_example}
let rec (Freeze Nat_from_2) =
    Freeze {Hd=2;Tl = map_stream succ Nat_from_2};;
let Primes = primes Nat_from_2;;
\end{caml_example}

We have ``obtained'' the whole set of prime numbers! (it just remains
 to be expanded as needed!)
\begin{caml_example}
display_num_stream 10 Primes;;
\end{caml_example}

\subsection{Formal series}
\index{Formal series}
\paragraph\ 

Our aim is to define the function \verb"sinus" and \verb"cosinus" as the
streams of the coefficients of their serial developments.

First we define the negation of a stream:
\begin{caml_example}
let sf_uminus = map_stream (fun x -> (-x));;
\end{caml_example}

and an integrating function for series:

\begin{caml_example}
(* Formal series integration fonction *)
let integrate l =
    (intrec 0 l)
    where rec intrec n {Hd=x;Tl=l} =
        {Hd=(x/(n+1));Tl=(intrec (n+1) l)}
;;
\end{caml_example}

We define the stream $(-1)^n$:
\begin{caml_example}
let rec (Freeze l) = Freeze {Hd=1;Tl=(sf_uminus l)};;
display_num_stream 10 l;;
\end{caml_example}

and then we integrate it to get:
\begin{caml_example}
let log1 = {Hd=0;Tl=integrate l};;
display_num_stream 10 log1;;
\end{caml_example}

and the recursive definition of \verb"sinus" and \verb"cosinus" as the integrals of each other, just giving the first element of the series:
\begin{caml_example}
let rec (Freeze {Lfst=cosinus;Lsnd=sinus}) =
        Freeze {Lfst={Hd=1;Tl=integrate (sf_uminus sinus)};
                Lsnd={Hd=0;Tl=integrate (cosinus)}};;
\end{caml_example}
To get the first elements of a stream as a list:
\begin{caml_example}
let rec stream_to_list S =
    function 0 -> []
           | n -> S.Hd::stream_to_list S.Tl (n-1);;
\end{caml_example}
and finally the first elements of \verb"sinus" and \verb"cosinus":
\begin{caml_example}
stream_to_list sinus 5;;
stream_to_list cosinus 5;;
sinus;;
\end{caml_example}

\subsection{Forcing lazy values}
\index{Forcing lazy values}
\index{\verb"force"}
\paragraph\ 
A primitive construct is provided to (recursively) force the evaluation of a lazy value:
\begin{caml_example}
let x = Freeze (succ 1);;
force x;;
x;;
\end{caml_example}

A toplevel directive is also available to force automatically the evaluation
when printing lazy values. If you want to get this feature type in
\verb"#eval when print true;;". Be careful with potentially infinite
 data structures!

\section{Redefining types}
\index{Redefinition of types}
\index{Type redefinition}
\paragraph\ 

When you redefine a type (i.e. when you define a type with the same name
as an existing one), all its constructors or labels are removed from the 
current global symbol table: they become ``unbound''.
 In addition, the system emits
 a warning. Identifiers bound to values of the old type (which is now 
``out of scope'') do not disappear. Their type is now written suffixed
by a question mark:
\begin{caml_example}
type barfoo = Barfoo of num;;
let x = Barfoo 1;;
type barfoo = Foobar of string;;
Barfoo;;
x;;
\end{caml_example}

\section{Implementation issues}
\index{Efficiency}

\index{Representation of values}
\paragraph\ 

All values in CAML are implemented using values of the predefined type
\verb"obj" (more on objects in section \ref{Objs}):
\*)

(*exception failure of string;;*)

type obj = obj_int of int | obj_float of float
         | obj_string of string | obj_atom of atom
         | obj_vect of obj vect | obj_cons of obj * obj;;

(*\
There is a function which maps a value on its representation
\begin{caml_primitive}
Repr
\end{caml_primitive}

The representation algorithm for concrete types is fairly sophisticated: it
tends to use as litle memory allocation as possible. In particular it employs
the concept of ``superfluous'' constructors, which are removed from the
internal representation of data. For instance
\begin{caml_example}
Repr (Foobar "oui");;
\end{caml_example}
the constructor \verb"Foobar" is superfluous.

One can inspect the representation of values with the function \verb"info"
which gives information about identifiers:
\begin{caml_primitive}
info
\end{caml_primitive}
\begin{caml_example}
info"barfoo";;
\end{caml_example}
\*)

(*\begin{caml_ignore}*)
type exc = failure of string
         | Identity             (* Sharing Transducers *)
         | system of string * obj
         | break		(* To catch interrupts from keyboard *)
	 | toplevel		(* To reenter the toplevel *)
         | format of string     (* When pretty_printer must fail *)
	 | bottom
         | undefined_forward of string
         | match_failure
;;


(* 7.3. Defining exceptions for CAML system *)


(*|
Exception break defined
Exception toplevel defined
Exception format of string defined
Exception bottom defined
|*)
exception toplevel		(* To reenter the toplevel *)
and       system of string * obj(* System exceptions from le-lisp *)
and       break			(* To catch interrupts from keyboard *)
and       format of string      (* When pretty_printer must fail *)
and	  bottom
and       undefined_forward of string
and       match_failure;;

(* Management of the code associated to a continue statment in a stack *)

exception continue_exception of string;;

let continue_bucket =
 #(if COMPATIBLE then <:Caml:Expr<
   <:obj<((eval (mlexc "continue_exception")) . "bad continue management")>>
                       >>
   else <:Caml:Expr<
   <:obj<((mlexc "continue_exception") . "bad continue management")>>
          >>)
;;

(* Updating undefined values *)

(* To create a dummy value for forward idents *)
let make_forward s _ = raise undefined_forward s;;

(*\end{caml_ignore}*)

(* 0. Booleans *)
(*\
\chapter{Arithmetic and boolean operations}
\paragraph\ 
We now start to describe the primitives operations provided on basic types:
 in this chapter we examine primitives for numbers and booleans.

Since the CAML system is written in CAML (it is {\em bootstrapped}) the set of
primitives available to the user is the one needed in the
 implementation of the language.

\section{Infix identifiers}
\index{Infix identifiers}
\label{infixes}
You may write in CAML \verb"3+2": in this expression ``+'' is written in
``{\em infix form}'' that is to say that
 ``+'' appears {\em between} the arguments.
But the addition is clearly a binary operation: thus its type is
 \verb"(num * num) -> num".
 In other words we should apply the addition on the \begin{em}pair\end{em}
\verb"(3,2)", writing \verb"+ (3,2)": in this expression 
 the  addition appears in ``{\em prefix form}''.
 In fact CAML knows the function
 ``\verb"prefix +"'', whose type is  \verb"(num * num) -> num".
But since it is much more convenient and natural to write \verb"2 + 3",
 CAML provides the user with the infix form as a syntactical commodity: 
``\verb"x + y"'' is expanded as ``\verb"prefix + (x,y)"''.

\begin{caml_example}
+;;
prefix +;;
prefix + (3,2) = 3+2;;
\end{caml_example}

In this chapter many primitives will have an infix form.

\section{Booleans}
\index{Booleans}
\subsection*{Definition}

The type \verb"bool" has two constant constructors, 
\verb"true" and \verb"false".  For efficiency reasons
the type \verb"bool" is a primitive type, but
\verb"bool" could have been declared in CAML in the following way
(but note that this definition would produce a syntax error, since
 \verb"true" and \verb"false" are reserved keywords):
\begin{verbatim}
type bool = true | false;;
\end{verbatim}

\subsection*{Conditional}

The {\em conditional} language construct \verb"if b then e1 else e2",
evaluates the expression \verb"b" and if the result is \verb"true"
 (resp. \verb"false"), the expression \verb"e1"  (resp. \verb"e2") is evaluated
 and its result is the result of the whole expression.
 Remark that the conditional is a lazy construct in the sense that only one
 of the expressions (\verb"e1",\verb"e2") is evaluated.

 The two principal logical connectors (\verb"&" and \verb"or")
are defined in terms of the conditional.
\begin{itemize}
\item ``\verb"e1 & e2"'' is equivalent to ``\verb"if e1 then e2 else false"''.
\item ``\verb"e1 or e2"'' is equivalent to ``\verb"if e1 then true else e2"''.
\end{itemize}

\begin{caml_primitive}
prefix not
\end{caml_primitive}
\index{not}

\begin{itemize}
\item \verb"not e1" is equivalent to \verb"if e1 then false else true".
\end{itemize}


\begin{caml_primitive}
prefix &
prefix or
\end{caml_primitive}
\index{\verb"&"}
\index{\verb"or"}
Not all the infix operators have a corresponding prefix version:
even though the symbol \verb"&" is used in infix position, as in
\verb"(x=1)&(y=2)",
\verb"prefix &" is {\em not equivalent} to the symbol \verb"&",
since it is impossible to define a call-by-value
function corresponding to \verb"&"
 (because \verb"e & e'" evaluates to \verb"false"
 as soon as \verb"e" does, even if the
evaluation of \verb"e'" would fail or loop).

Since \verb"e & e'" is expanded on the fly as the CAML phrase:
\verb"if e then e' else false", clearly if \verb"e" evaluates to false
\verb"e'" is {\em not evaluated}.

Note that it is no help to define:
\begin{caml_example}
let boolean_and e e' = e & e';;
\end{caml_example}
because when the body of \verb"boolean_and" is executed \verb"e" and \verb"e'"
 will have been completely evaluated:
\begin{caml_example}
boolean_and false fail;;
false & fail;;
\end{caml_example}

Thus prefix operator ``\verb"prefix &"'' is a real {\em function},
 equivalent to the previous \verb"boolean_and".

A similar phenomenon exists with \verb"or".

\begin{caml_example}
prefix or (true,true);;
prefix or (true,fail);;
true or fail;;
\end{caml_example}

\par\noindent

It is possible to define functions to implement other logical connectives,
as \verb"nand" or \verb"nor". Even the conditional can be expressed
in terms of a function, but notice that the laziness of the conditional
is not inherited by these functions:

\begin{caml_example}
let nand x y = not (x & y)
and nor x y = not (x or y)
and cond test t f = if test then t else f;;
let careful_divide x y =
 if y = 0 then x else x quo y
and careful_divide2 x y =
 cond (y = 0) x (x quo y);;
careful_divide 1 0;;
careful_divide2 1 0;;
\end{caml_example}

\begin{caml_primitive}
neg
\end{caml_primitive}
\verb"neg" is the functional version of \verb"not". It can be used to negate
a predicate:
\*)

(* neg pred = prefix not o pred *)
(*|
Value neg : (('a -> bool) -> 'a -> bool)
|*)
let neg predicate x = not (predicate x);;

(*\
\par\noindent
\begin{bf}
Beware:
\end{bf}
\par\noindent
\verb"--" \verb"not" has a particular precedence distinct from the one of
application. Thus the above definition is equivalent to:
\begin{caml}
let neg predicate x = not (predicate x);;
\end{caml}
You can not use the $\eta$-rule of $\lambda$-calculus to simplify it!

\section{Equality}
\index{Semantics of equality}
\paragraph\ 
In mathematics, equality is usually defined as identity.
 In a computer the story
is not so simple; since different ``computed objects'' can be created
which are mathematically identical but physically completely different
(i.e. stored in different memory locations). Another problem arises
when objects share common sub-structures: we may have cycles in the 
representation of objects and the equality test should take care of this.

So we can distinguish two
different kinds of equality: identity of computer locations (which evidently
ensures equality of objects), and equality which tests if objects
are isomorphic.

\subsection{Identity}
\index{Identity}\index{\verb"=="}
\begin{caml_primitive}
eq
prefix ==
\end{caml_primitive}
\verb"eq" means equality of adresses where objects are stored
(i.e. physical identity of the locations holding the representations
in the computer memory).
Thus \verb"eq (e1,e2)" (or equivalently \verb"e1 == e2") tests the {\em identity} (internally the same object)
of \verb"e1" and \verb"e2".
This primitive is very efficient since it just performs {\em one} pointer 
comparison.
\begin{caml_example}
let x = ref 1 and y = ref 1;;
eq(x,y);;
eq(!x,!y);;
\end{caml_example}
Be careful with identity of numbers:
\begin{itemize}
\item semantical equality of small integers is identity.
\item semantical equality of floating numbers is
 identity (up to precision problems).
\item semantical quality of other kinds of numbers is not identity.
\end{itemize}
\begin{caml_example}
1 == 2-1;;
1/2 == 3/2 - 1;;
1.0 == 2.0 - 1.0;;
\end{caml_example}

\subsection{Equality}
\index{Equality}
\label{Equality}

\subsection*{Primitives for equality}
\begin{caml_primitive}
prefix =
prefix <>
equal
\end{caml_primitive}
\index{\verb"="}\index{\verb"<>"}

\begin{itemize}
\item \verb"e1 = e2"  compares the two values of the same type and
produces \verb"true" when they are {\em equal}.
\item \verb"e1 <> e2" is equivalent to \verb"not e1=e2".
\item \verb"equal (e1,e2)" tests the {\em equality}
of \verb"e1" and \verb"e2", if they are {\em not functional objects}
(in which case \verb"equal" fails with the string \verb|"equal"|).
\verb"equal" is more sophisticated than \verb"=" since it is able to handle
cyclic data (for which \verb"=" may loop). On the other hand,
 \verb"equal" is less efficient than
\verb"=", since it performs much more computation to prove that data are
isomorphic.
\end{itemize}

All this operators are primitives in CAML.
\par\noindent
\begin{bf}
Beware:
\end{bf}
\par\noindent
\verb"--" All these functions are {\em undefined} when applied to frozen values.
Before the use of one of them with such lazy values you have to force
 the values to ensure their complete evaluation.
\verb"-- prefix =" is in {\bf no way} the same function as \verb"equal"
 (see above).

\subsection{Semantics of equality}

\subsubsection*{The primitive \verb"="}
\index{\verb"="}
\paragraph\ 
The use of the well-known symbol \verb"=" suggests that \verb"prefix =" is
 bound to the
corresponding expected predicate. It is indeed true for values of
non functional types. But note the following points:
\par\noindent
\begin{bf}
Beware:
\end{bf}
\par\noindent
\verb"-- prefix =" {\bf may loop} on cyclic values, because it examines 
recursively all the components of its arguments. This is particularly the
case for recursive functional values. See the primitive
\verb"equal" in the following to test equality in such a case.
\par\noindent
\verb"-- prefix =" is in no way the mathematical extensional equality when
used with functions.
In this case \verb"prefix =" means identity (i.e. identity of representations
in the computer memory).
So, necessarily, semantically distinct functions would not be equal, since
their representation are not physically the same. But different
occurrences of the same textual expression may build functions which
are extensionally  equal but not recognized as such by  \verb"prefix =":
\begin{caml_example}
let f x = x+1;;
let g x = x+1;;
f=g;;
\end{caml_example}

\verb"f" and \verb"g" are the same mathematical function,
 but \verb"prefix =" does not detect
this because their internal representations are really different
objects. Compare with:
\begin{caml_example}
let h = f;;
h=f;;
\end{caml_example}

Different evaluations of the same textual expression may build equal
objects:
\begin{caml_example}
let f = add 1
and g = add 1 ;;
eq (f,g);;
f=g;;
\end{caml_example}
the two distinct calls to the same curried function with the same argument
leads to distinct objects, but \verb"prefix =" is able to prove them 
isomorphic.

\bigskip

For references, equality means equality of contents:
\begin{caml_example}
let x = ref 1;;
let y = ref 1;;
x=y;;
!x=!y;;
eq(x,y);;
\end{caml_example}
\verb"x" and \verb"y" are different locations even though they hold the same value.

\bigskip

In short,
 non functional values are equal iff they are semantically equal,
whereas function equality is {\em unreliable}.

\begin{caml_example}
let f = I;;
f = I;;
let g = f;;
g = f;;
\end{caml_example}

\subsection*{The primitive \verb"equal"}
\index{Equality}

\begin{caml_primitive}
equal
\end{caml_primitive}

This is ``cautious'' equality since
\verb"equal" tests equality without looping on values having cycles.
It fails with \verb|"equal"| if you try
to compare functions which are not identical (``\verb"eq"'')
 and is ``\verb"eq"'' for references.

\begin{caml_example}
let f = I;;
equal(f,I);;
let g = f;;
equal(g,f);;
\end{caml_example}
\*)

(* 1. Numbers *)

(*\
\section{Numbers}
\index{Numbers}
Numbers consist of:
\begin{itemize}
\item small integers: type \verb"int".
\item floating point representations of reals: type \verb"float".
\item symbolic representations of large integers and rationals, internally
represented as structures of type \verb"obj" (see chapter \ref{Objs}).
\end{itemize}

These three sorts of numbers are grouped in one predefined type of numbers:
\verb"num". Most of the time the casual user will manipulate \verb"num"
values without worrying about their representations.

More sophisticated users will take advantage of the definition of the
 \verb"num" type:
\*)

(*
type big_int = Small_int of int | Big_int of big_int * big_int;;

type num = Int of int | Float of float | Big_num of big_int vect;;
*)
type num = Int of int | Float of float | Big_num of obj vect;;

(*\
The elements of the type  \verb"num" are completed by the
special values, {\em infinity} (\verb"1/0" or  \verb"-1/0") and 
{\em undefined} (\verb"0/0"). This is a
predefined type in CAML as well as most of its operators.


\subsection{Reading numbers}

There are special conventions to input the different kinds of numbers.

\subsubsection{Reading num elements}

One may type in such numbers with the following syntax:

\begin{verbatim}
NUM ::=  digit+ {'.' digit+} {'e' {'+' | '-'} digit+}
\end{verbatim}
which stands for:
\begin{itemize}
\item a number is a non null sequence of digits,
\item optionally followed by a dot and a non null sequence of digits,
\item optionally followed by \verb"e" which is followed optionally by 
\verb"+" or \verb"-" and a non null sequence of digits.
\end{itemize}

Rule of thumb: a number always begins and ends by a digit.

\subsubsection*{Reading Rationals and Naturals}

The lexical convention above applies to naturals, for which it is just:
\begin{verbatim}
digit+
\end{verbatim}

The notation for rationals is the common mathematical one, i.e. two integers
separated with a \verb"/" symbol:

\begin{caml_example}
1/2;;
1/-2;;
(* In this case + is parsed as the binary operator *)
+1/2;;
\end{caml_example}
\par\noindent
\begin{bf}
Beware:
\end{bf}
\par\noindent
\verb"--" This notation is {\em not} a lexical convention: rationals do not belong to
a special lexical class.

\subsubsection*{Reading floating point numbers}

Some elements of the type \verb"num" are floating
point numbers, in a range depending on the machine which runs the CAML system
but generally $-1\ 10^{38}\ ;\ +1\ 10^{38}$.

\begin{caml_example}
1.0;;
1e2;;
1e-1;;
1.2e+2;;
1.;;
.1;;
\end{caml_example}


\subsubsection{Reading ``small'' Integers}

Elements of type \verb"int" are ``small integers'' in the range -32767, +32767
(more exactly the ring, since there is no notion of overflow).
Integer values of type \verb"int" are mainly manipulated for the sake of
efficiency, since their representation is only a 16 bit value (moreover
 they are not allocated, hence there is no need to garbage collect them).

The maximum value of a ``small'' integer is:

\begin{caml_primitive}
max_int
\end{caml_primitive}
\*)
let max_int = 32767;;

(*\

CAML is able to read small integers entered directly,
 when you type in a number
 preceded by a sharp character ``\verb"#"'', following the lexical convention:

\begin{verbatim}
INT ::= '#' {'-' | '+'} digit+
\end{verbatim}

\begin{caml_example}
#32767;;
#-1;;
#+1;;
#32768;;
\end{caml_example}

\subsubsection*{Reading numbers does some magic}

When you type in a number (of type \verb"num") the constructors of the type
\verb"num" are automatically added by the lexical analyser :
typing-in \verb"1" is equivalent to type in \verb"Int #1" and analogously for
\verb"1.1" and \verb"Float #1.1". For big numbers (or rationals) the 
corresponding vector of elements of the type \verb"obj" is built and applied
to the constructor \verb"Big_num".

\subsection{Primitive operators for numbers}
\paragraph\ 

Most of the operations on numbers are ``generic'', which means that they are
 able to handle every kind of number, with implicit coercions if 
necessary (e.g. basic arithmetic operations \verb"+", \verb"*", \verb"-" and
\verb"/"). Some others are specialized for one precise kind of numbers
as is the primitive operation \verb"succ_int : int -> int".

Roughly speaking, the user needs not to worry about all these subtleties
about numbers and uses as numbers the elements of the predefined type
\verb"num" which contains implicitly all the other kinds of numbers,
along with all its operations which ``know'' how to handle the necessary
 coercions.

\subsubsection*{Addition}

\begin{caml_primitive}
prefix +
\end{caml_primitive}
\verb"n1+n2" gives the result of the sum of \verb"n1" and \verb"n2".

\par\noindent
Example:
\begin{caml_example}
1/3 + 2/3;;
\end{caml_example}
\par\noindent
\begin{bf}
Beware:
\end{bf}
\par\noindent
\verb"--" The normalisation of rational numbers is {\em not} done by the
addition: while computing with
rationals, the results are {\em not normalised}.

 The normalisation process only takes place {\em when printing}: thus
the runtimes reported by the CAML system, which does not include the
time necessary to print the results, may be much less than the real
evaluation time.

\subsubsection*{Subtraction}

\begin{caml_primitive}
prefix -
minus
\end{caml_primitive}
\verb"prefix -" is the binary subtraction function. \verb"minus" is the unary
function (unary minus) which returns the negation of its single numeric
argument:
 \verb"minus n" or \verb"-n" are
 equivalent to \verb"0-n".

\begin{caml_example}
prefix - (7,3);;
minus 7;;
\end{caml_example}
\*)


(*|
Value minus : (num -> num)
|*)
let minus n = 0-n;;

(*\
\par\noindent
Example:
\begin{caml_example}
52131--5212+-123;;
let x = 5 in -x;;
\end{caml_example}

\subsubsection*{Multiplication and Division}
\begin{caml_primitive}
prefix *
prefix /
\end{caml_primitive}
\verb"n1*n2" gives n1 times n2 as result. 
\verb"n1/n2" gives \verb"n1" divided by \verb"n2", these are total
functions and both are primitives.

\par\noindent
Example:
\begin{caml_example}
255452*222*4545;;
(7/3) / 2;;
5/0;;
\end{caml_example}

\subsubsection*{Absolute value}
\begin{caml_primitive}
abs
\end{caml_primitive}

\verb"abs x" returns the absolute value of the number \verb"x".
\*)
let abs x = if x >= 0 then x else -x;;

(*\
\subsection{Incrementing and decrementing}

\subsubsection*{Incrementing and decrementing numbers}
\begin{caml_primitive}
pred
succ
\end{caml_primitive}
These are bound to the mathematical successor and predecessor functions when
used with integers:
\verb"pred n" (resp  \verb"succ n") is a primitive equivalent to \verb"n-1"
(resp  \verb"n+1").

\begin{caml}
let pred x = x-1
and succ x = x+1;;
\end{caml}

\begin{caml_example}
succ 1;;
succ 1.2e3;;
succ (1/2);;
(* Beware of precedence *)
succ 1/2;;
\end{caml_example}


\subsubsection*{Incrementing and decrementing counters}
\begin{caml_primitive}
incr
decr
\end{caml_primitive}
When one has created a reference containing a number (whose type is
 \verb"num"), it is convenient
to increment and decrement its value, using these two (primitive) functions
semantically equivalent to:

\begin{caml}
let incr i = i:= succ !i and decr i = i:= pred !i;;
\end{caml}

\subsection{Curried operators}
\begin{caml_primitive}
add
sub
mult
div
\end{caml_primitive}

\verb"add", \verb"sub", \verb"mult" and \verb"div" are the curried version of
 \verb"prefix +", \verb"prefix -", \verb"prefix *" 
 and \verb"prefix /":
\*)

(*|
Value add : (num -> num -> num)
Value sub : (num -> num -> num)
Value mult : (num -> num -> num)
Value div : (num -> num -> num)
|*)
let add x y = (x + y) and sub x y = x - y
and mult x y = (x * y) and div x y = x / y;;

(*\
\begin{caml_example}
let add5 x = add 5 x in
 add5 20,add5 10;;
\end{caml_example}

\subsection{Integer results}
Some of the operations on numbers are made to take or produce only
integers.
\begin{caml_primitive}
prefix quo
prefix mod
integer
floor
round
numerator
denominator
\end{caml_primitive}

\begin{itemize}
\item \verb"n1 quo n2" is the integer (part of a) division.
\item \verb"n1 mod n2" gives the remainder of an integer division and
for integers \verb"n1" and \verb"n2"
is equivalent to \verb"n1 - (n1 quo n2)*n2". Both \verb"quo" and \verb"mod",
 fail to produce a result when \verb"n2" is either \verb"0" or \verb"0/0".
\item The function \verb"integer" produces the integer part of a number whereas
 \verb"floor" returns the mathematical integer part
(\verb"floor x" is the greatest integer lower or equal to x). These two
 functions differ with no-positive numbers arguments. Both are useful, since
if you want to compute each digit of a rational you need \verb"integer":
\begin{caml_example}
integer (-1/2);;
floor (-1/2);;
\end{caml_example}
\item The function \verb"round" returns the nearest integer of a number.
\item Finally \verb"numerator" and \verb"denominator" give access to 
 the two components of a rational.
\end{itemize}

\par\noindent
Example:
\begin{caml_example}
(7/3) quo 2;;
(7/3) mod 2;;
0 quo 0;;
1.2 quo 1;;
0 mod 0;;
integer (1/3);;
integer (-0.5);;
integer (-1);;
numerator (30/14);;
denominator(30/14);;
denominator 5;;
denominator 3.14;;
\end{caml_example}
\*)
let floor x = x quo 1;;
let round x = (x quo 1) + (if x mod 1 < 1/2 then 0 else 1);;

(*\

\begin{caml_example}
floor (-0.5);;
floor 1.2;;
round 1.4;;
round 1.5;;
\end{caml_example}

\subsection{Logical operations}
\begin{caml_primitive}
land
lor
lxor
lshift
lnot
\end{caml_primitive}

These are ``logical'' functions which operate on numbers {\em bitwise}.
 These numbers must be ``small'' integers, that is 16 bits words. All these
functions are expanded on the fly into one or a few lines of assembly code.
Their meanings are:
\begin{itemize}
\item \verb"land (n1,n2)": logical ``and'' between \verb"n1" and \verb"n2".
 For instance to find if a number is a power of 2, you may define
\begin{caml_example}
let is_power2 n = n=land(n,-n);;
is_power2 4;;
is_power2 3;;
\end{caml_example}
\item \verb"lor": logical ``or''.
\item \verb"lxor": exclusive ``or''.
\item \verb"lshift (n1,n2)": shifts \verb"n1" of \verb"n2" positions.
 If \verb"n2" is positive the shift is done to the left (towards the
 most significant bits, hence it is a multiplication)
 If \verb"n2" is negative the shift is done to the right (towards the least
significant bits, then it is a division).
Using lshift you may extract a bit field from an integer representation, as in:
\begin{caml_example}
let extract n i = lshift (land (n, lshift (1,i)), -i);;
map (extract 24) (interval 0 15);;
map (extract (-24)) (interval 0 15);;
\end{caml_example}
\item \verb"lnot n": returns the logical complement of \verb"n".
\end{itemize}


\subsection{Floating point operations}

Some of the operations on numbers are made to take or produce only
numbers which are representations of floats. These are the classical:
\begin{caml_primitive}
sin
cos
asin
acos
atan
exp
log
log10
power
sqrt
float
\end{caml_primitive}

\begin{itemize}
\item The function \verb"float" produces the floating point representation
of a number.
\item Other primitives may be applied to numbers which are automatically
coerced to a floating point representation if necessary.
\end{itemize}

As an example one can create a function \verb"tan" using the mathematical
definition:

\begin{caml_example}
let tan x = (sin x) / (cos x);;
tan 1;;
atan it;;
let pi_over_2 = 3.14159265/2;;
(* Computation with floating point numbers approximates *)
tan pi_over_2;;
(* But sometimes consistently ! *)
atan it;;
\end{caml_example}

\subsection{Small integers operations}

 The normal use of arithmetic must be done with elements
of the type \verb"num", and thus very few operations are provided for
the type \verb"int".

 One can create integers from element of type \verb"num" with
explicit coercions:
\begin{caml_primitive}
int_of_num
num_of_int
\end{caml_primitive}
\*)

(*|
Value int_of_num : (num -> int)
Value num_of_int : (int -> num)
|*)
let int_of_num =
    function Int n -> n
           | B -> if is_integer B
                  then
                   (match integer B with
                     Int n -> n
                    | _ -> failwith "int_of_num")
                   else failwith "int_of_num"
and num_of_int = Int;;

(*\
\par\noindent
Example:

\begin{caml_example}
int_of_num 32767;;
int_of_num 32768;;
\end{caml_example}

In addition there is a coercion between strings and integers:
\begin{caml_primitive}
string_of_int
\end{caml_primitive}

\subsubsection*{Basic operations for ``small'' integers}
Most often, small integers are used to create counters, generally initialized
to zero. So the two specialized
primitive operations \verb"succ_int" and \verb"pred_int" are provided.
In addition, specialized versions of arithmetic operations are provided.
\begin{caml_primitive}
succ_int
pred_int
add_int
mult_int
sub_int
quo_int
le
ge
lt
gt
\end{caml_primitive}

\begin{caml_example}
succ_int #0;;
it - 1;;
add_int(#1,#2);;
add_int(#32767,#10);;
mult_int(#32767,#2);;
ge (#1,#2);;
\end{caml_example}

\begin{itemize}
\item \verb"add_int", \verb"mult_int", \verb"sub_int" and \verb"quo_int", are the 
four arithmetic operations on integers.

\item \verb"le", \verb"ge", \verb"lt" and \verb"gt", are arithmetic comparisons
corresponding respectively to \verb"<=", \verb">=", \verb"<" and \verb">".
\end{itemize}

Notice that the specialized versions of the
arithmetic operations are primitives. In a sense they
are not ``cautious'', since they do not check for overflow.

 A careful (but less efficient) version may be defined as in:
\begin{caml_example}
let mult_int_care (x,y) =
 match (Int x) * (Int y)
 with Int result -> result
   |      _      -> failwith "overflow"
;;
mult_int_care(#32767,#2);;
\end{caml_example}

\subsubsection{``int'' Counters}

If a reference containing an integer is built, one can define a general
function to increment such a reference with:
\begin{caml_example}
let incr_int x = x:=succ_int !x
and decr_int x = x:=pred_int !x;;
\end{caml_example}
Recall that equality between small integers\index{equality between integers}
 may be tested with \verb"=", or
equivalently with \verb"eq" (since small integers are uniquely represented).

\subsection{Numbers of type float}

 One can create floats (numbers of type \verb"float") from elements of
type \verb"num" with explicit coercions:
\begin{caml_primitive}
float_of_num
num_of_float
\end{caml_primitive}
\*)

(*|
Value float_of_num : (num -> float)
Value num_of_float : (float -> num)
|*)
let float_of_num n =
  (float_of_num_float (float n)
   where float_of_num_float =
    fun (Float f) -> f | _ -> failwith "float_of_num")
and num_of_float = Float;;


(*\

In addition there is a coercion between strings and floats:
\begin{caml_primitive}
string_of_float
\end{caml_primitive}

\subsubsection*{Basic operations for floats}
\begin{caml_primitive}
add_float
sub_float
mult_float
div_float
\end{caml_primitive}
These are basic operations specialized to elements of type ``\verb"float"''.

\subsection{Comparisons between numbers}
\index{\verb">"}\index{\verb"<"}\index{\verb"<="}\index{\verb">="}
\begin{caml_primitive}
prefix >
prefix <
prefix <=
prefix >=
\end{caml_primitive}
These are the usual predicates for numbers of type \verb"num".
\begin{caml_example}
1.1 < 3/2;;
\end{caml_example}

\begin{caml_primitive}
min
max
\end{caml_primitive}
\verb"min" and \verb"max" are bound to the expected curried functions.
\*)
(*|
Value max : (num -> num -> num)
Value min : (num -> num -> num)
|*)
let max x y = if x<=y then y else x
and min x y = if x<=y then x else y;;

(*\
\subsection{Miscellaneous operations for numbers}

\subsubsection*{Discriminators}
\begin{caml_primitive}
is_float
is_int
is_integer
\end{caml_primitive}
\begin{itemize}
\item the function \verb"is_float" allows you to test if a value of type
\verb"num" is in fact a floating number.
\item one can test if a given number is a small integer using \verb"is_int",
and if a number is an integer value with \verb"is_integer".
\end{itemize}
\*)

(*|
Value is_float : (num -> bool)
|*)
let is_float = function Float _ -> true | _ -> false;;
(*|
Value is_int : (num -> bool)
|*)
let is_int = function Int _ -> true | _ -> false;;

(*\
\begin{caml_example}
is_int 12;;
is_integer 12;;
is_int 1234567890;;
is_integer 1234567890;;
is_int 1.0;;
is_integer 1.0;;
is_int (1 / 2);;
is_integer (1 / 2);;
\end{caml_example}

\subsubsection*{Coercion between objects and numbers}
\begin{caml_primitive}
num_of_obj
int_of_obj
float_of_obj
\end{caml_primitive}

These are the usual coercions between representations of numbers
and numbers.
\*)

(*|
Value int_of_obj : (obj -> int)
Value float_of_obj : (obj -> float)
|*)
let int_of_obj = fun (obj_int i) -> i | _ -> failwith "int_of_obj"
and float_of_obj = fun (obj_float f) -> f | _ -> failwith "float_of_obj"
;;

(*\
\subsubsection*{Coercion between strings and numbers}
\begin{caml_primitive}
string_of_num
num_of_string
\end{caml_primitive}

\begin{itemize}
\item \verb"string_of_num" and \verb"num_of_string" maps element of type
 \verb"num" onto corresponding elements of type \verb"string", such that
the following relation holds for each \verb"n" from type \verb"num":
\begin{verbatim}
 num_of_string (string_of_num n) = n
\end{verbatim}
\end{itemize}

\begin{caml_primitive}
string_of_integer
string_of_floating
string_of_big_num
\end{caml_primitive}
\begin{itemize}
\item \verb"string_of_integer", \verb"string_of_floating" and
 \verb"string_of_big_num" map onto strings the elements of the type \verb"num"
 issued respectively from the set of
``small'' integers, floating point numbers and big integers or rational numbers.
They are used to define \verb"num_of_string".
\end{itemize}
\begin{caml_example}
string_of_float #1.1;;
string_of_floating #1.1;;
\end{caml_example}

The function \verb"string_of_num" is defined in CAML as a simple pattern 
matching on the values of type \verb"num":
\*)

let string_of_num = function
    Int i -> string_of_integer i
  | Float f -> string_of_floating f
  | Big_num b -> string_of_big_num b
;;

(*\

\index{\verb"int_of_string"}
\index{\verb"float_of_string"}

The two functions \verb"int_of_string" and \verb"float_of_string" are not
 provided but are easy to write:
\begin{caml_example}
let int_of_string = function
     "" -> failwith"int_of_string"
    | s ->
      if ascii_code s = ascii_code "#"
       then
        (match num_of_string (sub_string s 1 (length_string s))
         with Int i -> i | _ -> failwith"int_of_string")
       else failwith"int_of_string"
and float_of_string = function
     "" -> failwith"float_of_string"
    | s ->
      if ascii_code s = ascii_code "#"
       then
        (match num_of_string (sub_string s 1 (length_string s))
         with Float f -> f | _ -> failwith"float_of_string")
       else failwith"float_of_string"
;;
\end{caml_example}
\subsection{Random functions}
\begin{caml_primitive}
random
init_random
\end{caml_primitive}

There is a pseudo random number generator in the CAML system. It is based on the
linear congruence method given by D.E. Knuth ("The Art of Computer
Programming"). It provides two functions~: \verb"init_random" and
\verb"random". The first one resets the generator to the given value, the
second one returns a random value in the range between 0 and its argument.

Both use argument of type \verb"int" which must be greater than or equal to 0
for \verb"init_random" and strictly greater than 0 for \verb"random".

\begin{caml_example}
print_int(random #3);print_int(random #3);print_int(random #3);;
\end{caml_example}
\*)

(*\begin{caml_ignore}*)
(*|
Value random : (int -> int)
Value init_random : (int -> int)
{num_of_int}
|*)
let random n = 
    if (num_of_int n) > 0 then system_random (#0, n)
    else failwith "random"

and init_random n =
    if (num_of_int n) >= 0 then system_srandom n
    else failwith "init_random"
;;

(*\end{caml_ignore}*)

(* 3. General Combinators *)
(*\

\chapter{General purpose functions}
\index{General purpose functions}
\begin{caml_eval}
set_any_where true;;
set_with_indent false;;
set_latex_margin 67;;
\end{caml_eval}

We describe some commonly used functions applicable to pairs,
lists, and other CAML values.  All are definable in CAML. For each function we
give, in addition to its type, a CAML declaration equivalent to that defining
it in the current system. In fact, this part of the documentation is 
automatically produced from the prelude of the system, in order to always be
up-to-date.

\section{Operations on pairs}

\subsection{Infix identifiers not bound to functions}
\paragraph\ 
These are essentially syntactical matters.
\subsubsection{Pair construction}
\index{\verb"prefix ,"}\index{\verb","}
\verb"prefix ," is obviously useless since it should be a special case of
identity:
\begin{verbatim}
let prefix , (x,y) = (x,y);;
\end{verbatim}

A curried (see next section) function to build pairs is provided:
\begin{caml_primitive}
pair
\end{caml_primitive}
This is just the curried form of ``\verb","'':
\*)

(*|
Value pair : ('a -> 'b -> 'a * 'b)
|*)
let pair x y = (x,y);;

(*\

\subsection{Destructors for pairs}
\index{Destructors for pairs}
\begin{caml_primitive}
fst
snd
\end{caml_primitive}
\verb"fst" and \verb"snd" are polymorphic predefined operators to
access the components of a pair.
They may be defined in CAML with
\begin{caml}
let fst (x,_) = x
and snd (_,y) = y;;
\end{caml}

\section{General combinators}
\index{General combinators}

\subsection{Curryfication}
\index{Curryfication}
A very important and useful feature of CAML is the possibility to ``curry'' a
function. The idea is mainly to allow the definition of functions with several
arguments, which can receive one argument at a time. For example consider
the function \verb"add" defined by:
\begin{caml_example}
let add x y = x + y;;
\end{caml_example}
This function is not exactly \verb"prefix +",
 since \verb"prefix +" has only one
 argument (which is actually a pair) whereas \verb"add" has two arguments.
Thus the type of \verb"add" is \verb"num -> num -> num", i.e. \verb"add"
 has to receive a number, and then another one before it returns their
 sum. We say that add is the ``curried'' form of \verb"prefix +", since we have
 broken the pair, which is the single argument of \verb"prefix +", into
two components which are the two arguments of \verb"add".
It is a first way to understand the ``currying'' mechanism.

Another way is to get a deeper understanding of \verb"add"'s type:
\verb"num -> num -> num" is nothing but \verb"num -> (num -> num)".
That is to say that \verb"add" is a function of high order: given a number,
\verb"add" leads to another function (the type of which is
 \verb"num -> num").
It could seem very hard to understand at first glance, but in fact it is
something that we commonly do while reasoning: Nobody should be surprised
if you say `` add 2 to your last result '', though this really means:
 Use addition with
the first argument fixed to 2, and apply this function to your last
result, i.e. use the curried version of addition with first argument 2 and
apply it to your last result. In CAML it would be:
\begin{caml}
(add 2) (your_last_result);;
\end{caml}
  
There are many ways to understand currying, and in the case of
 \verb"add", one can think of \verb"add" as a ``generic'' function, to get
 the family of functions (with type \verb"num -> num") which add a constant
 value. For example we can define the function \verb"add_2", which increases
 its argument by 2, with the following:
\begin{caml}
let add_2 = add 2;;
\end{caml}
and if you need the function which increases its argument twofold:
\begin{caml}
let double = mult 2
 where mult x y = x * y;;
\end{caml}

 This process is so useful that a functional ``curry'' is provided, which maps
a function to its curried form. Uncurry reverts this process.
\begin{caml_primitive}
curry
uncurry
\end{caml_primitive}
\*)

(* 3.1. Functional combinators *)

(*|
Value curry : (('a * 'b -> 'c) -> 'a -> 'b -> 'c)
Value uncurry : (('a -> 'b -> 'c) -> 'a * 'b -> 'c)
|*)
let curry f x y = f (x,y) 
and uncurry f (x,y) = f x y;;

(*\
\begin{caml_example}
curry prefix + 2 3;;
\end{caml_example}
In the same way we could have defined the function \verb"double" as:
\begin{verbatim}
let double = mult 2
 where mult = curry ( prefix * );;
\end{verbatim}

\subsection{Composition}
\index{Composition}
\begin{caml_primitive}
prefix o
\end{caml_primitive}
\verb"prefix o (f,g)", or equivalently\index{\verb"o"}  \verb"f o g" , returns the functional
composition\index{\verb"composition"} of \verb"f" and \verb"g".
\*)

(* #infix o;; *)

(*|
Value prefix o : (('a -> 'b) * ('c -> 'a) -> 'c -> 'b)
|*)
let f o g = function x -> f (g x);;

(*\
\par\noindent
Example:
\begin{caml_example}
let square x = x*x;;
let double x = 2*x;;
let quadr_square = square o double
and double_square = double o square;;
double_square 3;;
quadr_square 3;;
\end{caml_example}
\begin{caml_primitive}
pairing
distr_pair
tee
\end{caml_primitive}
\begin{itemize}
\item \verb"pairing (f,g)" combines \verb"f" and \verb"g" to yield
 the function on pairs determined by applying \verb"f" and \verb"g" to
 the corresponding components of the pairs.
\item \verb"distr_pair f" applies the function \verb"f" to each component
 of a pair and returns the pair of results. Notice that the typechecking of
CAML forces the pair argument of \verb"distr_pair" to be homogeneous (due
to the application of \verb"f" to its components).
\item \verb"tee (f,g)" applies the functions \verb"f" and \verb"g" to a
common argument and returns the pair of results.
\end{itemize}
\*)

(*|
Value pairing : (('a -> 'b) * ('c -> 'd) -> 'a * 'c -> 'b * 'd)
Value distr_pair : (('a -> 'b) -> 'a * 'a -> 'b * 'b)
Value tee : (('a -> 'b) * ('a -> 'c) -> 'a -> 'b * 'c)
|*)
let pairing (f,g) = fun (x,y) -> (f x,g y)
and distr_pair f = fun (x,y) -> (f x,f y)
and tee (f,g) = fun x -> (f x,g x);;

(* 3.2 Combinators, as in Curry and Feys *)
(*\
\subsection{Combinators, as in Curry and Feys}
\paragraph\ 
These are commonly used functional to build new functions from already
existing functional values.
\index{Combinators}

\begin{caml_primitive}
I
K
CK
C
W
B
S
\end{caml_primitive}

The predefined combinators are:
\begin{itemize}
\item the polymorphic identity function \verb"I".
\item the ``cancellators'' \verb"K" and \verb"CK" which forget their second
 (resp. first) argument.
\item the ``permutator'' \verb"C" which exchanges the arguments of
 a curried function.
\item the ``duplicator'': \verb"W f x" provides the function
\verb"f" with two copies of \verb"x".
\item \verb"B" is just the curried form of the usual functional composition
\verb"o".
\item \verb"S" is the classical combinator from combinatory logic, which
distributes its third argument to each of its functional arguments.
\end{itemize}

\verb"I" is a primitive operation whose definition is equivalent to:
\begin{caml}
let I x = x;;
\end{caml}
\*)

(*|
Value K : ('a -> 'b -> 'a)
Value CK : ('a -> 'b -> 'b)
Value C : (('a -> 'b -> 'c) -> 'b -> 'a -> 'c)
Value W : (('a -> 'a -> 'b) -> 'a -> 'b)
Value B : (('a -> 'b) -> ('c -> 'a) -> 'c -> 'b)
Value S : (('a -> 'b -> 'c) -> ('a -> 'b) -> 'a -> 'c)
|*)
let K x y = x               (* the cancelators *)
and CK x y = y

and C f x y = f y x	    (*  the permutator  *)
and W f x   = f x x	    (*  the duplicator  *)
and B f g x = f (g x)	    (*  the compositor  *)  (*curried form of o *)
and S f g x = f x (g x);;

(*\
\begin{caml_primitive}
prefix Co
\end{caml_primitive}
Permutation-composition, named so because  \verb"(f Co g) = C (f o g)".
\*)

(* #infix Co;; *)

(*|
Value prefix Co : (('a -> 'b -> 'c) * ('d -> 'a) -> 'b -> 'd -> 'c)
|*)
let f Co g = fun x y -> f (g y) x;;

(*\
\par\noindent
Note that a so-called ``apply'' operator with the following semantics:
\begin{caml_example}
let apply f x = f (x);;
\end{caml_example}
is useless, since this apply is a special case of \verb"I".

\subsection{Tacticals}
\index{Tacticals}
\paragraph\ 
Tacticals are functions that may be applied to other functions to sequence
them or try one after the other, \ldots This terminolgy is issued from
 automatic theorem proving, where tacticals are used to build strategies
to find the proofs.

\begin{caml_primitive}
can
\end{caml_primitive}
Test for failure: \verb"can f x" applies \verb"f" to \verb"x",
 returning true if evaluation succeeds and false otherwise:
\*)

(* 3.3 Tacticals *)
(*|
Value can : (('a -> 'b) -> 'a -> bool)
|*)
let can f x = (f x ; true) ? false;;

(*\
\begin{caml_primitive}
iterate
repeat
prim_rec
\end{caml_primitive}
\begin{itemize}
\item \verb"iterate f" performs self composition of \verb"f" a given number
 of times. It satisfies:
\verb"iterate f n x = (f (f (f ... (f x) ... )))" which is just the expanded 
form of the relation
\verb"iterate f n = (f o f o f ... o f)" with \verb"n" occurrences of \verb"f".
\item \verb"repeat n f x" evaluates sequentially \verb"n" times the
 application of the function \verb"f" to the argument \verb"x" and then
 returns \verb"()": it is mainly used for side-effects when \verb"f" is a
 ``command''.
\item \verb"prim_rec f init n" evaluates the {\em n}th application of \verb"f",
 starting from \verb"init".
\begin{caml_example}
(* The factorial function *)
let fact = prim_rec mult 1;;
(* Another way to define the primitive range *)
let range = prim_rec (C cons) [];;
let letters = prim_rec (fun s n -> (ascii (n+64))^s) "" 26;;
\end{caml_example}
\end{itemize}
\*)

(*|
Value iterate : (('a -> 'a) -> num -> 'a -> 'a)
|*)
let iterate f = iterate_f
    where rec iterate_f n x = if n<=0 then x else iterate_f (pred n) (f x);;

(*|
Value repeat : (num -> ('a -> 'b) -> 'a -> void)
|*)
let repeat n action arg = repeat_action n
    where rec repeat_action n = 
    if n<=0 then () 	else (action arg; repeat_action (pred n));;

let prim_rec f = loop
 where rec loop init = function
   0 -> init
 | n -> loop (f init n) (n-1);;

(*\
\par\noindent
Example:
\begin{caml_example}
let hello () = message "hello world!";;
repeat 3 hello ();;
\end{caml_example}


\index{THEN}\index{ORELSE}
If we wanted to define the LCF tacticals \verb"THEN" and \verb"ORELSE"
in CAML, we could proceed as follows. We recall that:
\begin{itemize}
\item \verb"(f THEN g) x", applies \verb"f" to \verb"x" and then returns
\verb"g(x)".
\item \verb"(f ORELSE g) x", applies \verb"f" to \verb"x" and if it fails
(raising exception \verb"failure") then returns \verb"g(x)".
\end{itemize}

\begin{caml_example}
#infix THEN;;
let a THEN b = function x ->  (a x; () ? ()); b x;;
#infix ORELSE;;
let a ORELSE b = function x -> a x ? b x;;
\end{caml_example}
\*)



(* 3.4 Fixed Point *)

(*\
\subsection{Fixed Point Combinator}
\begin{caml_primitive}
Y
fix_point
\end{caml_primitive}
\begin{itemize}
\item \verb"Y" is a classical fixed point combinator which returns the 
function which is the fixed point of a functional.
\item \verb"fix_point f x" iterates the function \verb"f" starting from
the value \verb"x", to find a fix point for \verb"f" (that is a value
 \verb"f_p" such that \verb"f f_p = f_p").
\end{itemize}
\*)
(*|
Value Y : ((('a -> 'b) -> 'a -> 'b) -> 'a -> 'b)
|*)
let Y (f : (('a -> 'b) -> ('a -> 'b))) = g where rec g x = f g x;;

let rec fix_point f x =
 let y = f x in
  if y = x then x else fix_point f y;;


(* 3.5 Sharing transducers *)
(*\
\subsection{Sharing transducers}
\index{Sharing transducers}
\index{Identity}
\begin{caml_primitive}
share
\end{caml_primitive}
Assuming that a function fails for identity (raising the special exception
 \verb"Identity"), \verb"share f" permits maximum sharing between the result
 of \verb"f" and its argument.
\*)

(*|
Exception Identity defined
|*)
exception Identity;;

(*|
Value share : (('a -> 'a) -> 'a -> 'a)
|*)
let share f x = try f x with Identity -> x;;

(*\

\begin{caml_example}
let rec share_append (l1,l2) = share append_rec l1
 where rec append_rec = function
  [] -> if l2<>[] then l2 else raise Identity
| x1::l1 -> x1 :: append_rec l1;;
let l = [1;2;3] in
eq (l, share_append (l,[]));;
let l = [1;2;3] in
eq (l, share_append ([],l));;
\end{caml_example}

\begin{caml_primitive}
pair_share
share_pair_share
\end{caml_primitive}

\begin{itemize}
\item If \verb"f" is a function designed for sharing (i.e. it raises exception
\verb"Identity" in case of identity between its argument and its result),
 \verb"pair_share f" extends this
 property to pairs. \verb"pair_share" raises \verb"Identity" when the
 argument pair can be shared.
\item \verb"share_pair_share f" is equivalent to
 \verb"share (pair_share f)", thus it shares its argument pair as well.
\end{itemize}
\*)
(*|
Value pair_share : (('a -> 'a) -> 'a * 'a -> 'a * 'a)
{share}
|*)
let pair_share f (x,y) = try (f x,share f y) with Identity -> (x,f y)
;;

let share_pair_share f ((x,y) as pair) =
 try (f x,share f y) with
  Identity -> try (x,f y) with Identity -> pair
;;

(*\

\begin{caml_example}
(* Share Identity (che`re but not expensive):
   instead of returning its argument its raises Identity *)
let Sharing_Identity x = raise Identity;;
(* Let us define some pair *)
let a_pair = ("foo","bar");;
(* If we distribute Sharing_Identity on the elements of a pair,
   the pair itself is shared *)
eq(a_pair, share_pair_share Sharing_Identity a_pair);;
(* But if we distribute the usual identity function I
   on the elements of a pair, the pair itself is not shared *)
eq(a_pair, distr_pair I a_pair);;
(* Share_pair_share is really useful and can not be
   simulated using share and distr_pair *)
eq(a_pair, distr_pair (share Sharing_Identity) a_pair);;
(* If a sharing function is not encapsulated in a ``share''
   call, then Identity may escape to the CAML toplevel
   which reports a message *)
eq(a_pair, pair_share Sharing_Identity a_pair);;
\end{caml_example}

\*)

(* 4. List and set processing functions for ML *)

(*\

\section{List and set processing functions}
\index{List and set processing functions}

\subsection{List definition}
\begin{caml_primitive}
[]
prefix ::
\end{caml_primitive}
The type \verb"list" of homogeneous sequences of elements, is a type with one
type argument, and is defined {\em quite normally} as
\*)

(*|
Type list defined
     [] : 'a list
   | prefix :: : ('a * 'a list -> 'a list)
|*)
type 't list = [] | prefix :: of 't * 't list;;


(*\
It is not advisable to redefine it, since the parser of the CAML
system ``knows'' the abbreviation \verb"[e1;e2; ... ;en]" for
 \verb"e1 :: e2 :: ... :: []". Thus strange things may happen if
you redefine this ``almost built in'' type.

\subsection{List primitives}
\index{List primitives}
\begin{caml_primitive}
cons
uncons
\end{caml_primitive}
\verb"cons" is the curried form of \verb"prefix ::" and \verb"uncons" is the 
converse operation: it maps a non-empty list (which a pair of an element
and a list) onto the pair of its head and its tail.
\*)
let uncons = function prefix :: p -> p | _ -> failwith "uncons";;

(* 4.1. List primitives *)


(*|
Value cons : ('a -> 'a list -> 'a list)
|*)
let cons x y = x::y;;

(*\
\begin{caml_primitive}
hd
tl
null
singleton
append
prefix @
\end{caml_primitive}

\begin{itemize}
\item \verb"hd" returns the first element of a list and \verb"tl" the rest
of the list.
These two functions fail (with respective failure
string \verb|"tl"| and \verb|"hd"|) when called with the empty list.
\item \verb"null" tests whether a list is empty or not.
\item \verb"singleton" builds a list whose unique element is the
argument of \verb"singleton".
\item \verb"l1 @ l2" catenates the two lists \verb"l1" and \verb"l2",
and \verb"append" is its curried version.
\end{itemize}
\*)

(*|
Value null : ('a list -> bool)
Value hd : ('a list -> 'a)
Value tl : ('a list -> 'a list)
Value singleton : ('a -> 'a list)
|*)
let null = fun [] -> true | _ -> false
and hd = fun [] -> failwith "hd" | (x::l) -> x
and tl = fun [] -> failwith "tl" | (x::l) -> l
and singleton x = [x];;

(*\
\begin{caml_primitive}
length
\end{caml_primitive}
\verb"length l" returns the length of the list \verb"l".
\*)
(*|
Value length : ('a list -> num)
|*)
let length l = length_l(0,l) where rec
  length_l = fun (n,[]) -> n | (n,(_::l)) -> length_l(succ n,l);;

(*\
\begin{caml_primitive}
mem
mem_list_list
\end{caml_primitive}
\begin{itemize}
\item \verb"mem x l" returns true if some element of \verb"l" is equal to \verb"x",
otherwise false.
\item \verb"mem_list_list x ll" tests if \verb"x" is a member of one of the
lists of the list of lists \verb"ll".
\end{itemize}
\*)
(*|
Value mem : ('a -> 'a list -> bool)
|*)
let mem x = mem_x
  where rec mem_x = fun []  ->  false | (y::l)  ->  x=y or mem_x l;;

(*\
\verb"mem_list_list" is equivalent to:
\begin{caml}
(* Membership of a list of lists *)
(*|
Value mem_list_list : ('a -> 'a list list -> bool)
 CAML_system{mem}
|*)
let rec mem_list_list s = mem_rec where rec mem_rec =
 function [] -> false
        | l1::L -> (mem s l1) or (mem_rec L)
;;
\end{caml}

\subsection{Indexing}
\index{Indexing}
\begin{caml_primitive}
nth
item
index
\end{caml_primitive}
\begin{itemize}
\item The first function returns the {\em n}th element of a list.
\item \verb"item l n" returns the element of the list \verb"l"
 having the position \verb"n" (starting from 0).
\item Finally \verb"index" gives
the position of the first occurrence of a given value in a list.
\end{itemize}
\*)
(* Indexing *)

(* index "b" ["a";"b";"c";"b"] = 2 *)

(*|
Value index : ('a -> 'a list -> num)
|*)
let index x l = index_x (1,l)
 where rec index_x =
    fun (n,y::l) -> if x=y then n else index_x (succ n,l)
    |   _ -> failwith "index";;


(*|
Value nth : ('a list -> num -> 'a)
|*)
let nth l n = if n<=0 then failwith "nth" else
 (nth_op (l,n) where rec nth_op = 
   fun (hd::_,1) -> hd
   |   (_::tl,n) -> nth_op (tl,pred n)
   |   ([],_) -> failwith "nth");;


(*|
Value item : ('a list -> num -> 'a)
{nth}
|*)
let item l n = nth l (succ n);;

(*\
\subsection{Reversing lists}
\index{Reversing lists}
\begin{caml_primitive}
rev_append
rev
\end{caml_primitive}

\begin{itemize}
\item \verb"rev l" reverses the list \verb"l".
\item \verb"rev_append l1 l2" appends \verb"l2" to the reverse of the list
\verb"l1".

 Thus \verb"rev_append l1 l2 == rev l1 @ l2".
\end{itemize}
\begin{caml_example}
rev_append [5;6;7] [4;3;2;1];;
\end{caml_example}
\*)
(* Reversing *)

(* rev_append [4;3;2;1] [5;6;7] == [1;2;3;4;5;6;7]   *)
(* rev_append l1 l2 == rev l1 @ l2                   *)

(*|
Value rev_append : ('a list -> 'a list -> 'a list)
|*)
let rec rev_append  =
  fun [] -> I | (x::l') l ->  rev_append l' (x::l);;


(*|
Value rev : ('a list -> 'a list)
{rev_append}
|*)
let rev l = rev_append l [];;

(*\

\subsection{Finding items in lists}
\index{Finding items in lists}
The following functions search lists for elements with
various properties; those returning elements fail if no such
element is found, those returning boolean results never fail.
\begin{caml_primitive}
find
try_find
\end{caml_primitive}
\begin{itemize}
\item \verb"find p l" returns the first element of \verb"l" which satisfies
 the predicate \verb"p".
\item \verb"try_find f l" returns the result of applying \verb"f" to the
 first member of \verb"l" for which the application of \verb"f" succeeds.
\end{itemize}
\*)
(*|
Value find : (('a -> bool) -> 'a list -> 'a)
|*)
let find p = find_p
  where rec find_p = 
    fun (x::l) -> if p x then x else find_p l
    | _ -> failwith "find";;


(*|
Value try_find : (('a -> 'b) -> 'a list -> 'b)
|*)
let try_find f = try_find_f where rec try_find_f =
   fun [] -> failwith"try_find"
   |   (h::t) -> f h ? try_find_f t;;

(*\
\subsection{List functionals}
\index{List functionals}
\begin{caml_primitive}
map
end_map
map_i
end_map_i
map_share
share_map_share
map2
\end{caml_primitive}

\begin{itemize}
\item \verb"map f l" returns the list obtained by applying \verb"f" to
the elements of \verb"l" in turn:
\begin{verbatim}
map f [a1;...;an] = [f a1;...;f an]
\end{verbatim}
\item \verb"end_map f default l" returns the list obtained by applying \verb"f" to
the elements of \verb"l" in turn, but in front of the list \verb"default"
(i.e. \verb"map f" = \verb"end_map f []" and
 \verb"end_map f default l = (map f l) @ default").
\item \verb"map_i f start l" returns as well
 the list obtained by applying \verb"f" to
the elements of \verb"l" in turn, but it provides to \verb"f" the rank of the
item processed by \verb"f":
\begin{verbatim}
map_i f start [e1; e2; ... ; ei; ... ; en] =
              [f start e1; f (start+1) e2; ... ;
               f (start+i) ei; ... ;
               f (start+n) en]
\end{verbatim}
\par\noindent
Example:
\begin{caml_example}
map_i pair 0 ["a";"b";"c"];;
map_i
 (fun i e ->
   print_string"The ";print_num i;
   print_string"th element is: ";message e)
 0
 ["a";"b";"c";"d"];;
\end{caml_example}

\item \verb"end_map_i" is the analog of \verb"end_map" for \verb"map_i".

\item \verb"map_share" extends the sharing property to lists.
 \verb"map_share" raises \verb"Identity" when the argument list can be shared.
 \verb"share_map_share f" is equivalent to \verb"share (map_share f)", thus :
\item \verb"share_map_share" tries to share its result with its list argument.
\item \verb"map2 f l1 l2" maps a curried function with two arguments on two
lists. Arguments of \verb"f" are picked up  in \verb"l1" and \verb"l2"
in turn (thus \verb"l1" and \verb"l2" must have the same length).
\end{itemize}
\begin{caml}
let map f = map_f
 where rec map_f = fun [] -> [] | (a::l)  ->  (f a::map_f l);;
\end{caml}

\*)
let end_map f default = map_f
 where rec map_f = fun [] -> default | (a::l)  ->  (f a::map_f l);;


(*\begin{caml_ignore}*)

(* 3.2. List functionals *)

(* map f [a1;...;an] ==> [f a1;...;f an] *)

(*|
Value map : (('a -> 'b) -> 'a list -> 'b list)
|*)
let map f =
 fun [] -> [] | [a] -> [f a] | [a1;a2] -> [f a1;f a2] | l -> map_f l
 where rec map_f = fun [] -> [] | (a::l)  ->  (f a::map_f l);;

(*\end{caml_ignore}*)

(* assuming f fails for identity (raising exception Identity)
   extends this property to lists *)

(*
let map_share f = map_share_f
 where rec map_share_f = function
    [] -> raise Identity
  | x::r -> try (f x)::(share_map_share_f r) with
             Identity -> x::map_share_f r
 and share_map_share_f = share map_share_f
;;
*)

let fun_pair f =
 let rec map_share_f = function
    [] -> raise Identity
  | x::r -> try (f x)::(share_map_share_f r) with
             Identity -> x::map_share_f r
  and share_map_share_f = share map_share_f in
  (map_share_f,share_map_share_f) in

let map_share = fst o fun_pair
and share_map_share = snd o fun_pair;;

(*|
Value map2 : (('a -> 'b -> 'c) -> 'a list -> 'b list -> 'c list)
|*)
let map2 f l1 l2 = map2_f(l1,l2)
 where rec map2_f =
   fun	([],[])	 -> []
   |	(h1::t1,h2::t2) -> (f h1 h2)::(map2_f (t1,t2))
   |	_ -> failwith "map2";;


let map_i f =
 map_i_rec where rec map_i_rec i = function
  [] -> [] | x::l -> (f i x)::map_i_rec (i+1) l
;;

let end_map_i f default =
 map_i_rec where rec map_i_rec i = function
  [] -> default | x::l -> (f i x)::map_i_rec (i+1) l
;;

(*\
\begin{caml_primitive}
map_uncons
do_list_uncons
\end{caml_primitive}
These functions are designed to operate on non-empty lists (that is a pair
of an element and a list), and for such
arguments they correspond respectively to \verb"map" and \verb"do_list".
\*)

(* Should be in prelude *)
(*|
Value map_uncons : (('a -> 'b) -> ('a * 'a list) -> ('b * 'b list))
Value do_list_uncons : (('a -> 'b) -> ('a * 'a list) -> unit)
 CAML_system{do_list,map}
|*)
let map_uncons f (x,L) = (f x, map f L)
and do_list_uncons f (x,L) = f x; do_list f L
;;

(*\
\begin{caml_primitive}
it_list
\end{caml_primitive}
For \verb"f", \verb"b", and \verb"l" arguments, this function performs
 iterated compositions of the curried function \verb"f", using elements of
 \verb"l" as second arguments of \verb"f"; \verb"b" is used for the first
 argument of the first call of \verb"f", and thereafter the result
of each call is passed as the first argument of the next call.
\verb"it_list" is characterised by:
\begin{center}
\verb"it_list f a [b1;...;bn] = (f (...(f a b1)...) bn)"
\end{center}
\*)
(* it_list f a [b1;...;bn] = (f (...(f a b1)...) bn) *)

(*|
Value it_list : (('a -> 'b -> 'a) -> 'a -> 'b list -> 'a)
|*)
let it_list f = it_list_f
    where rec it_list_f a = fun [] -> a | (b::l) -> (it_list_f (f a b) l);;

(*\
\begin{caml_primitive}
sigma
pi
\end{caml_primitive}
In an example using \verb"it_list", these two functions return the sum
 (resp. the product) of all the elements of a list:
\*)
(*|
Value sigma : (num list -> num)
Value pi : (num list -> num)
{add,it_list,mult}
|*)
let sigma = (it_list add 0)
and pi = (it_list mult 1);;

(*\
\begin{caml_primitive}
list_it
\end{caml_primitive}
For \verb"f", \verb"l", and \verb"b" arguments, this function performs
 iterated compositions of the curried function \verb"f", using elements
 of \verb"l" as first arguments of \verb"f"; \verb"b" is used for the
 second argument of the first call of \verb"f", thereafter the result
of each call is passed as the second argument of the next call.
\verb"list_it" is characterised by:
\begin{center}
\begin{verbatim}
list_it f [l1;l2;...;ln] b = f l1 (f l2 (...(f ln b)...))
                           = ((f l1) o (f l2) o...o (f ln)) b
\end{verbatim}
\end{center}

Notice that \verb"it_list" is tail recursive and thus more efficient
 than \verb"list_it" \ldots

As an example we can define the ``old'' \verb"rev_itlist" function,
from ML V6.2, using:
\begin{caml_example}
let rev_itlist f = C (list_it (C f));;
\end{caml_example}
\*)

(* list_it f [a1;...;an] b = f a1 (... (f an b) ...) *)

(*|
Value list_it : (('a -> 'b -> 'b) -> 'a list -> 'b -> 'b)
|*)
let list_it f l b = list_it_f l
    where rec list_it_f = fun [] -> b | (a::l) -> (f a (list_it_f l));;

(*\
\begin{caml_primitive}
end_it_list
\end{caml_primitive}
\verb"end_it_list" is almost the same as \verb"it_list" but for non empty lists
it does not use an extra argument for the first call of \verb"f": the first
element of \verb"l" will be used instead. If the list is empty then
\verb"default" is returned.
\verb"end_it_list" is characterised by:
\begin{center}
\begin{verbatim}
end_it_list f default [x1; ...; xn] =
 (f ... (f x1 x2) ... xn) (for n>1)
 x1                          (for n = 1)
 default                     (for n = 0)
\end{verbatim}
\end{center}

\begin{caml}
let end_it_list f default = function
   [] -> default
 | b::l -> it_list f b l
;;
\end{caml}
\begin{caml_primitive}
fold
fold_share
share_fold_share
\end{caml_primitive}

The functionnal \verb"fold" is used to map a curried function on a list
in a special way, since the function updates a 
``memory'' (presumably an association list viewed as an environment)
 which is passed to each call
in turn. Thus the result of \verb"fold f init_memory l"
is the pair \verb"m,l'" where \verb"m" is the new memory modified
by all the successive calls to \verb"f" and \verb"l'" is the list of results.
\begin{center}
\begin{verbatim}
 fold f a1 [b1; ... ;bn] =
        let a2,c1 = f a1 b1 in
          ...
        let a,cn = f an bn in
          a,[c1; ... ;cn]
\end{verbatim}
\end{center}

\*)
(* fold f a1 [b1; ... ;bn] =
	let a2,c1 = f a1 b1 in
	  ...
	let a,cn = f an bn in
	  a,[c1; ... ;cn]	*)


(*|
Value fold : (('a -> 'b -> 'a * 'c) -> 'a -> 'b list -> 'a * 'c list)
|*)
let fold f = fold_f
  where  rec fold_f a1 =
  fun [] -> a1,[]
  |   (b1::bl) -> let a2,c2 = f a1 b1 in 
    			let a,cl = fold_f a2 bl in a,(c2::cl);;

(*\
As an example if you want to map the \verb"successor" function on a list of 
integers, while counting the number of elements of the list, you may use
as environment the current index of the item processed by \verb"successor" :
\begin{caml_example}
fold (fun i n -> i+1,succ n) 0 [1;2;3];;
\end{caml_example}

In the same vein, you get a definition of the functional \verb"map_i", using
\verb"fold" :
\begin{caml_example}
let map_i f start l =
    snd (fold (fun i x -> (i+1,f i x)) start l);;
map_i (fun i s -> s^(string_of_num i)) 1 ["a";"b";"c"];;
\end{caml_example}
An equivalent but more functional way to define \verb"map_i" is:
\begin{caml_example}
let map_i f start = snd o fold consf start
    where consf i x = (i+1,f i x);;
\end{caml_example}
\*)
(*\

Sharing versions of \verb"fold" are also available:
\*)

(*|
Warning: recursive declaration of non abstraction
Value fold_share
      : (('a * 'b -> 'a * 'b) -> 'a * 'b list -> 'a * 'b list)
{share}
|*)
let fold_share f = fold_share_f
  where rec fold_share_f = function
     _,[] -> raise Identity
   | a1,(b1::bl) ->
     try let a2,c2 = f (a1,b1) in
         let a,cl = share_fold_share_f (a2,bl) in a,(c2::cl)
     with Identity -> let a,cl = fold_share_f (a1,bl) in a,b1::cl
  and share_fold_share_f = share fold_share_f
;;

let share_fold_share f = share_fold_share_f
  where rec fold_share_f = function
     _,[] -> raise Identity
   | a1,(b1::bl) ->
     try let a2,c2 = f (a1,b1) in
         let a,cl = share_fold_share_f (a2,bl) in a,(c2::cl)
     with Identity -> let a,cl = fold_share_f (a1,bl) in a,b1::cl
  and share_fold_share_f = share fold_share_f
;;

(*\
\begin{caml_primitive}
do_list
do_list_i
do_list2
\end{caml_primitive}
\begin{itemize}
\item \verb"do_list f l" applies \verb"f" to every element of \verb"l"
in turn and then returns \verb"()": no list of results is built.
\item \verb"do_list_i f start l" is a specialised version of \verb"do_list"
which provides to its functional argument \verb"f" the rank of the element to
which \verb"f" is applied (starting from \verb"start").
\item \verb"do_list2 f l1 l2" works in the same way for a curried function
with two arguments which are taken from \verb"l1" and \verb"l2" in
parallel.
\end{itemize}
\*)
(* procedural version of map (only for side effects) *)

(*|
Value do_list : (('a -> 'b) -> 'a list -> void)
|*)
let do_list f = do_list_f where rec do_list_f =
    fun	[] -> ()
    |	(x::l) -> f x;do_list_f l;;

let do_list_i f = do_list_f where rec do_list_f i =
    fun	[] -> ()
    |	(x::l) -> f i x;do_list_f (succ i) l;;


(*|
Value do_list2 : (('a -> 'b -> 'c) -> 'a list -> 'b list -> void)
|*)
let do_list2 f l1 l2 = do_list2_f (l1,l2)
 where rec do_list2_f =
  fun ([],[]) -> ()
   | (h1::t1,h2::t2) -> f h1 h2;(do_list2_f (t1,t2))
   |     _    -> failwith "do_list2";;

(*\

\subsection{Various list operators}
\paragraph\
\index{List operators}

List concatenation, either infix or prelix form:
\begin{caml_primitive}
append
prefix @
\end{caml_primitive}
\*)
(* 3.3. Various list operators *)

(*
let append l1 l2 = append_l2 l1                                     
    where rec append_l2 = fun [] -> l2 | (x::l)  -> (x::append_l2 l)
*)

(*|
Value append : ('a list -> 'a list -> 'a list)
{cons,list_it}
|*)
let append l1 l2 = if l2 = [] then l1 else list_it cons l1 l2;;

(*|
Value prefix @ : ('a list * 'a list -> 'a list)
{append}
|*)
let l1 @ l2 = if l2=[] then l1 else append l1 l2;;

(*\
Notice that \verb"append" may be defined as:
\begin{caml_example}
let append = list_it cons;;
\end{caml_example}
but this should copy the first argument of \verb"append" even if the second
is the empty list.

\begin{caml_primitive}
partition
replicate
flat
flat_map
\end{caml_primitive}
\begin{itemize}
\item \verb"partition p l" returns a pair of lists; the first element of this pair
is the list of all
the elements of l which satisfy the predicate p, and the second list of the
pair is the rest of l.
\item \verb"replicate x n" builds a list of length n, all elements of which are equal to x.
\item \verb"flat ll" flattens a list of lists.
\item \verb"flat_map f l" maps the function \verb"f" on the list \verb"l",
and catenates the results which must therefore be lists.
\end{itemize}
\*)
(*
partition p l = (filter_pos p l,filter_neg p l)   
*)

(*|
Value partition : (('a -> bool) -> 'a list -> 'a list * 'a list)
{list_it}
|*)
let partition p l =
    list_it fork l ([],[])
    where fork a (pos,neg) =  if p a then (a::pos),neg else pos,(a::neg);;


(*|
Value replicate : (num -> 'a -> 'a list)
|*)
let replicate n = if n <0 then failwith "replicate" else
 	(fun x -> (replicate_aux ([],n)) where rec
	replicate_aux = 
		fun (l,0) -> l | (l,n) -> replicate_aux (x::l,pred n));;


(*|
Value flat : ('a list list -> 'a list)
{append,list_it}
|*)
let flat ll = list_it append ll [];;

let flat_map f = flat_map_f where rec flat_map_f = function
 [] -> [] | (x::l) -> (f x) @ (flat_map_f l);;


(*\
\begin{caml_primitive}
first_n
last_n
\end{caml_primitive}
\begin{itemize}
\item \verb"first_n n l" returns the first \verb"n" elements of the list
\verb"l".
\item \verb"last_n n l" returns the last \verb"n" elements of the list
\verb"l".
\end{itemize}
\*)

(*|
Value first_n : (num -> 'a list -> 'a list)
Value last_n : (num -> 'a list -> 'a list)
{iterate,length,tl}
|*)
let rec first_n =
  fun 0 -> K []
  |   n -> (fun [] -> failwith "first_n"
	    |   (x::l) -> x::first_n (pred n) l)
and last_n n l = iterate tl (length l - n) l ? failwith "last_n";;


(* 3.4. Set operators *)

(*\
\subsection{Set operations}
\index{Set operations}

\subsubsection{Set operators}
\begin{caml_primitive}
exists
for_all
\end{caml_primitive}
\begin{itemize}
\item \verb"exists p l" applies \verb"p" to the elements of \verb"l"
 in order until one is found which satisfies \verb"p", or until the list is
exhausted, returning true or false accordingly.
More exactly \verb"exists p [x1; ... ;xn]" is false if \verb"p(xi)" is false
 for every \verb"i". Otherwise, it returns \verb"p(xi)" for \verb"i" minimum
 such that \verb"p(xi)" is not false.
\item \verb"for_all p l" applies \verb"p" to every element of
\verb"l" to see if they all verify \verb"p".
More exactly \verb"for_all p [x1; ... ;xn]" is true if \verb"p(xi)" is true
 for every \verb"i". Otherwise, it returns \verb"p(xi)" for \verb"i" minimum
 such that \verb"p(xi)" is not true.
\end{itemize}
\*)

(* for_all p [x1; ... ;xn] is true if p(xi) is true for every i.         *)
(* Otherwise, it returns p(xi) for i minimum such that p(xi) is not true *)
(*|
Value for_all : (('a -> bool) -> 'a list -> bool)
|*)
let for_all p = for_all_p where rec for_all_p = fun
    [] -> true
  | (a::l) -> p a & for_all_p l;;

(* exists p [x1; ... ;xn] is false if p(xi) is false for every i.         *)
(* Otherwise, it returns p(xi) for i minimum such that p(xi) is not false *)
(*|
Value exists : (('a -> bool) -> 'a list -> bool)
|*)
let exists p = exists_p where rec exists_p = fun
    [] -> false
  | (a::l) -> p a or exists_p l;;

(*\
As an example, one can define \verb"mem_list_list" using:
\*)

let mem_list_list x = exists (mem x);;

(*\
\begin{caml_primitive}
for_all2
exists2
for_all_pair
exists_pair
\end{caml_primitive}

\begin{itemize}
\item \verb"for_all2" and \verb"exists2" are the analogues of
\verb"exists" and \verb"for_all" for a curried predicate applied in
turn on the corresponding elements of two lists examined in parallel.
\item \verb"exists_pair" sees if there exists a pair
of elements in a list which verifies a given predicate,
 and \verb"for_all_pair" checks if all the pairs built up from elements of the 
list verify the predicate.
\end{itemize}
The following characterization may help:
\begin{center}
\begin{verbatim}
exists_pair p [x1; ... ; xn] == true
iff there exists a pair xi,xj st. i<j and p xi xj

for_all_pair p [x1; ... ; xn] == true
iff for all pairs xi,xj st. i<j :  p xi xj
\end{verbatim}
\end{center}
As an example we can define the function \verb"is_set" which tests if a
given finite list is in fact a finite set (that is, that there are no two
equal elements in the list):
\begin{caml_example}
let is_set = for_all_pair (curry (prefix <>));;
\end{caml_example}
\*)

(*|
Value for_all2 : (('a -> 'b -> bool) -> 'a list -> 'b list -> bool)
|*)
let for_all2 rel = for_all2_rel where rec for_all2_rel = function
     [] -> (function [] -> true | _ -> failwith "for_all2")
   | x1::l1 -> function x2::l2 -> rel x1 x2 & for_all2_rel l1 l2
                      |    _   -> failwith "for_all2";;

(*|
Value exists2 : (('a -> 'b -> bool) -> 'a list -> 'b list -> bool)
|*)
let exists2 rel = exist2_rel where rec exist2_rel = function
     [] -> (function [] -> false | _ -> failwith "exists2")
   | x1::l1 -> function x2::l2 -> rel x1 x2 or exist2_rel l1 l2
                      |    _   -> failwith "exists2";;

(* exists_pair p [x1; ... ; xn] == true		     *)
(* iff there exists a pair xi,xj st. i<j and p xi xj *)
(*|
Value exists_pair : (('a -> 'a -> bool) -> 'a list -> bool)
{exists}
|*)
let exists_pair p = exists_pair_p where rec exists_pair_p = function
   [] -> false
  |(x::l) -> exists (p x) l or exists_pair_p l;;


(* for_all_pair p [x1; ... ; xn] == true      *)
(* iff for all pairs xi,xj st. i<j :  p xi xj *)
(* Ex:  is_set = for_all_pair (curry prefix <>)   *)

(*|
Value for_all_pair : (('a -> 'a -> bool) -> 'a list -> bool)
{for_all}
|*)
let for_all_pair p = for_all_pair_p where rec for_all_pair_p = function
   [] -> true
  |(x::l) -> for_all (p x) l & for_all_pair_p l;;

(*\
\subsubsection{Filtering lists}
\index{Filtering lists}
\begin{caml_primitive}
filter
filter_pos
filter_neg
filter_succeed
\end{caml_primitive}
\begin{itemize}
\item \verb"filter p l" applies p to every element of \verb"l",
returning the list of those which satisfy p.
\item \verb"filter_pos" is an alias of \verb"filter".
\item \verb"filter_neg" is the converse function which returns
 the list of all the elements
of \verb"l" which do \begin{em}not\end{em} satisfy \verb"p".
\item \verb"filter_succeed f l" which returns
 the list of all the elements
of \verb"l" for which the call of \verb"f" does not fail
 (i.e. does not raise the exception \verb"failure").
\end{itemize}
\begin{caml_example}
filter_neg (fun x -> x = 1) [1;2;3;4];;
filter_pos (fun x -> x = 1) [1;2;3;4];;
\end{caml_example}
\*)

(*let filter_pos p = it_list select []
    	where select res x = if p x then x::res else res
and filter_neg p = it_list select []
    	where select res x = if p x then res else x::res
*)

(*|
Value filter_pos : (('a -> bool) -> 'a list -> 'a list)
Value filter_neg : (('a -> bool) -> 'a list -> 'a list)
Value filter_succeed : (('a -> 'b) -> 'a list -> 'a list)
{share}
|*)
let filter_pos p = (share filter_aux where rec filter_aux =
   fun [] ->  raise Identity
   |   (x::l) -> if p x then  x:: filter_aux l
                 else share filter_aux l)
and filter_neg p = (share filter_aux where rec filter_aux =
   fun [] ->  raise Identity
   |   (x::l) -> if p x then share filter_aux l
                 else  x:: filter_aux l)
and filter_succeed p = (share filter_aux where rec filter_aux =
   fun [] ->  raise Identity
   |   (x::l) -> (p x ; x:: filter_aux l) ? share filter_aux l)
;;

(* Obsolete *)

(*|
Value filter : (('a -> bool) -> 'a list -> 'a list)
{filter_pos}
|*)
let filter = filter_pos;;

(*\
\begin{caml_primitive}
map_succeed
do_list_succeed
\end{caml_primitive}

\begin{itemize}
\item \verb"map_succeed f l" applies f to every element of \verb"l",
 returning the list of results for those elements of \verb"l"
 for which the application of \verb"f" succeeds.
\item \verb"do_list_succeed" is the corresponding functional for functions
used only for side-effects.
%\item \verb"map_filter_succeed" is an alias of \verb"map_succeed", defined
%for sake of compatibility.
\end{itemize}

The function \verb"map_succeed" could be defined simply as:
\begin{caml}
let map_succeed f = map f (filter_succeed f);;
\end{caml}

For obvious efficiency reasons it is defined instead with:
\*)

(*|
Value map_succeed : (('a -> 'b) -> 'a list -> 'b list)
|*)
let map_succeed f = map_f where rec map_f =
 function [] -> []
        (* Note: map_f never raises exception failure *)
 |	h::t -> (f h :: map_f t) ? map_f t;;

(*\begin{caml_ignore}*)
let map_filter_succeed = map_succeed;;
(*\end{caml_ignore}*)

let do_list_succeed f = do_list_f where rec do_list_f =
    fun	[] -> ()
    |	(x::l) -> (f x ; do_list_f l) ? do_list_f l
;;

(*\
Notice that the local functions \verb"map_f" or \verb"do_list_f" never
 raise the exception \verb"failure".

\subsection{Mapping ``in accumulators''}
\paragraph\
\index{Mapping ``in accumulators''}

These functions are used to map functions on lists, accumulating the results
in a list reference.

\begin{caml_primitive}
do_list_replace
do_list_i_replace
do_list_succeed_replace
\end{caml_primitive}

\begin{itemize}
\item \verb"do_list_replace f accu l" applies f to every element of \verb"l",
 adding the results in the list reference \verb"accu".
 Then it returns \verb"()", in the tradition of the \verb"do_list" family.
\item \verb"do_list_succeed_replace" and \verb"do_list_i_replace" are
 the functionals corresponding to \verb"do_list_i" and \verb"do_list_succeed".
\end{itemize}

\*)
let do_list_replace f accu = do_list_f where rec do_list_f =
    fun	[] -> ()
    |	(x::l) -> accu:= (f x) :: !accu ;do_list_f l
;;

let do_list_i_replace f accu = do_list_f where rec do_list_f i =
    fun	[] -> ()
    |	(x::l) -> accu:= (f i x) :: !accu ;do_list_f (succ i) l
;;

let do_list_succeed_replace f accu = do_list_f where rec do_list_f =
    fun	[] -> ()
    |	(x::l) -> (accu:= (f x) :: !accu ;do_list_f l) ? do_list_f l
;;

(*\
\subsubsection{Sets}
\index{Sets}
\begin{caml_primitive}
add_set
make_set
intersect
subtract
union
distinct
\end{caml_primitive}
The first function is the set constructor.
The second function maps a list into a set, deleting duplicate elements,
and the next three provide the obvious functions on sets represented
as lists without repetitions (this may be checked using \verb"distinct").
\*)

let add_set x s = if mem x s then s else x::s;;

(* make_set [1;2;3;1;3;4;2;5] --> [1; 3; 4; 2; 5] *)
(*
let make_set l = 
     list_it (fun a res -> if mem a res then res else a::res) l [] 

  or sharing the end of the list:
*)

(*|
Value make_set : ('a list -> 'a list)
{mem,share}
|*)
let make_set l = share make_aux l where rec make_aux =
    fun [] -> raise Identity
    |   (x::l) -> if mem x l then share make_aux l else x::make_aux l;;

(*|
Value intersect : ('a list -> 'a list -> 'a list)
Value subtract : ('a list -> 'a list -> 'a list)
{C,filter,mem,neg}
|*)
let intersect l1 l2 = filter_pos (C mem l2) l1
and subtract l1 l2 = filter_neg (C mem l2) l1;;


(* union l1 l2 = subtract l1 l2 @ l2  *)
(*|
Value union : ('a list -> 'a list -> 'a list)
{mem}
|*)
let union l1 l2 = urec l1 where rec urec =
    fun []->l2 | (a::l) -> if mem a l2 then urec l else a :: (urec l);;

(* Check if the elements of a list are all distinct *)

(*|
Value distinct : ('a list -> bool)
{mem}
|*)
let rec distinct = fun
	(h::t)  ->  not(mem h t) & distinct t
 |	_  ->  true;;

(*\
\subsection{Association lists}
\index{Association lists}
An ``association list'' is just a list of pairs. It is used to record
some values with some selectors to access the values. To some extent
 it is just a way to represent the graph of a finite function.
\begin{caml_primitive}
equal_fst
equal_snd
assoc
rev_assoc
mem_assoc
except_assoc
except_rev_assoc
pair_assoc
pair_rev_assoc
\end{caml_primitive}
\begin{itemize}
\item \verb"equal_fst x p" (resp. \verb"equal_snd") returns true if
 the first (resp. second) element of the pair \verb"p" is equal to \verb"x".
\item \verb"assoc x l" searches the list \verb"l" of pairs for one whose
first component is equal to \verb"x", returning the second component
 of the first pair found.
\item \verb"rev_assoc y l" similarly
searches for a pair whose second component is equal
to \verb"y", and returns its first component.
\item \verb"mem_assoc x l" returns true or false according to the
existence in the list of pairs \verb"l" of a pair whose first component
is equal to \verb"x".
\item \verb"except_assoc x l" returns the list l without the first pair whose
first component is \verb"x" (resp. second component for
 \verb"except_rev_assoc").
\item \verb"pair_assoc x l" searches the list \verb"l" of pairs for one whose
first component is equal to \verb"x", returning the pair
 found as result.
\item \verb"pair_rev_assoc x l" searches the list \verb"l" of pairs for one whose
second component is equal to \verb"x", returning the pair
 found as result.
\end{itemize}
\*)

(* 4.5. Association lists *)


(*|
Value equal_fst : ('a -> 'a * 'b -> bool)
Value equal_snd : ('a -> 'b * 'a -> bool)
|*)
let equal_fst u (x,y) = (u=x)
and equal_snd u (x,y) = (u=y);;


(*|
Value pair_assoc : ('a -> ('a * 'b) list -> 'a * 'b)
Value pair_rev_assoc : ('a -> ('b * 'a) list -> 'b * 'a)
{equal_fst,equal_snd,find}
|*)
let pair_assoc x = find (equal_fst x)
and pair_rev_assoc x = find (equal_snd x);;

(*|
Value assoc : ('a -> ('a * 'b) list -> 'b)
Value rev_assoc : ('a -> ('b * 'a) list -> 'b)
{o,pair_assoc,pair_rev_assoc}
|*)
(*\
\begin{caml}
let assoc x = snd o (pair_assoc x)
and rev_assoc x = fst o (pair_rev_assoc x);;
let except_assoc e = except_e where rec except_e = function
    [] -> []
 | (x,_ as y)::l -> if x=e then l else y::except_e l
;;
let except_rev_assoc e = except_e where rec except_e = function
    [] -> []
 | (_,x as y)::l -> if x=e then l else y::except_e l
;;
\end{caml}
\*)


(*|
Value mem_assoc : ('a -> ('a * 'b) list -> bool)
{equal_fst,exists,o}
|*)
let mem_assoc = exists o equal_fst;;

(*\
\subsection{Adding and removing items}
\index{Adding and removing items}
\begin{caml_primitive}
except
chop_list
change
update
\end{caml_primitive}
\begin{itemize}
\item \verb"except x l" returns the list \verb"l" without \begin{em}the
first\end{em} occurrence of \verb"x".
\item \verb"chop_list n l" divides \verb"l" into a pair of lists,
 the first element of the pair is
built from the first \verb"n" elements of \verb"l"
 and the second with the remainder of \verb"l".
\verb"chop-list" is characterised by:
\begin{center}
\begin{verbatim}
chop_list n [e1; ...; en; e[n+1]; ... ; e[n+m]] =
            [e1; ...; en] , [e[n+1]; ...; e[n+m]] 
\end{verbatim}
\end{center}
\item \verb"change l n f" returns the list \verb"l'" built with the
elements of \verb"l" except for the {\em n}th, \verb"a", which is replaced by 
its image by \verb"f", \verb"f a".
\item \verb"update l n x" returns \verb"l'" such that \verb"l'" is
\verb"l" except the {\em n}th element of \verb"l" which becomes \verb"x".
\end{itemize}
\*)
(* To remove (the first occurrence of) a given element from a list 
  Example : except 2 [1;2;3;1;2;3] ==> [1;3;1;2;3] *)

(*|
Value except : ('a -> 'a list -> 'a list)
|*)
let except e = except_e
  where rec except_e =
   fun [] -> [] 
     | (elem::l) -> if e = elem then l else elem::except_e l;;


(*|
Value change : ('a list -> num -> ('a -> 'a) -> 'a list)
|*)
let change l n f = change_f l n
    where rec change_f = fun
     [] -> failwith "change"
    | (h::t) n -> if n=1 then f h::t else h::change_f t (pred n);;

(*|
Value update : ('a list -> num -> 'a -> 'a list)
{K,change}
|*)
let update l n x = change l n (K x);;

(* Initial segment of given length and remainder *)

(*|
Value chop_list : (num -> 'a list -> 'a list * 'a list)
{rev}
|*)
let chop_list = (fun n l -> chop_aux n ([],l))
  where rec chop_aux =
   fun 0 (l1,l2) -> rev l1,l2
     | n ->
         fun (_,[]) -> failwith "chop_list"
           | (l1,h::t) -> chop_aux (pred n) (h::l1,t);;


(*\
\begin{caml_example}
except 2 [1;2;3;1;2;3];;
chop_list 2 it;;
update [1;2;3;4] 2 3;;
change it 2 pred;;
\end{caml_example}

\subsubsection*{Handling the last element of lists}
\begin{caml_primitive}
last
sep_last
add_last
except_last
\end{caml_primitive}
These are obvious functions to deal with the last element of a list.
\*)

(* let last = hd o rev;;  *)
(* let rec last = fun [] -> failwith "last" | [x] -> x | (x::l) ->  last l;; *)
(* Or, still better, and using (prefix ::) as coercing non-null lists to pairs: *)

(*|
Value last : ('a list -> 'a)
|*)
let last =
     fun [] -> failwith "last"
     | (prefix :: pair) -> last_aux pair where rec last_aux =
     	fun (x,[]) -> x
	| (_,prefix :: pair) -> last_aux pair;;

(* sep_last [x1;x2; ... ;xn] -> (xn,[x1;...;xn-1]) *)
(*
let sep_last =
     (fun [] -> failwith "sep_last" | (prefix :: x) -> x) o rotate_right 
*)

(*|
Value sep_last : ('a list -> 'a * 'a list)
|*)
let sep_last =  function
   [] -> failwith "sep_last" 
  | prefix :: pair -> sep_last_aux pair 
  where rec sep_last_aux = function
       h,[] as pair -> pair
     | h,prefix :: pair -> let x,l = sep_last_aux pair in x,h :: l;;

(* add_last x [x1;x2; ... ;xn] -> [x1;x2; ... ;xn;x] *)

(*|
Value add_last : ('a -> 'a list -> 'a list)
{@}
|*)
let add_last x l = l @ [x];;

(* except_last = rev o tl o rev *)
(* except_last [1;2;3] --> [1;2] *)

(*|
Value except_last : ('a list -> 'a list)
|*)
let  except_last = function
     [] -> failwith "except_last"
   | prefix :: pair -> except_last_aux pair 
   where rec except_last_aux = function
        _,[] -> []
      | (x,prefix :: pair) -> x :: except_last_aux pair;;


(*\
\subsubsection*{Rotating lists}
\begin{caml_primitive}
rotate_left
rotate_right
\end{caml_primitive}
These two functions work so that the first (resp. the last) element of a list
becomes the last one (resp. the first one).
\*)
(*|
Value rotate_left : ('a list -> 'a list)
Value rotate_right : ('a list -> 'a list)
{@,sep_last}
|*)
let rotate_left = fun [] -> failwith "rotate_left" | (a::l) -> l @ [a]
and rotate_right = 
    fun [] -> failwith "rotate_right" 
    |   l -> (prefix ::) (sep_last l);;


(*\
\subsection{Sorting lists (quick sort)}
\index{Sorting lists}
\begin{caml_primitive}
select
sort_append
sort
\end{caml_primitive}
\begin{itemize}
\item \verb"select p l" returns a pair \verb"(elem,l')", where \verb"elem" is 
the first element of \verb"l" which satisfies the predicate \verb"p", and
\verb"l'" is the rest of the list \verb"l".
\item \verb"sort_append p order l l'" is designed mainly to implement
the function \verb"sort". It appends the list \verb"l'" to the list \verb"l",
but \verb"l" has been
sorted by the binary predicate \verb"order" (using the predicate \verb"p"
 to select
the way the list \verb"l" is cut while sorting).
\item \verb"sort order l" sorts the list \verb"l" according to the
 binary predicate
\verb"sort". The algorithm implemented is ``quick sort''.
\end{itemize}
\*)
(* Sorting of lists (quick sort) *)

(*|
Value select : (('a -> bool) -> 'a list -> 'a * 'a list)
|*)
let select p = select_p where rec select_p =
    fun [] -> failwith "select" 
    |   (prefix :: (x,l as pair)) -> if p x then pair else 
                  let (u,ll) = select_p l in u,(x::ll);;


(*|
Value sort_append
      : (('a -> bool) -> ('a * 'a -> bool) -> 'a list -> 'a list ->
         'a list)
{cons,o,partition,select}
|*)
let sort_append p le = sort_aux where rec sort_aux =
    fun [] -> I
    |   l -> 
       let (y,rest) =  select p l in
       let left,right = partition (fun x -> le(x,y)) rest in
       sort_aux left o cons y o sort_aux right;;


(*|
Value sort : (('a * 'a -> bool) -> 'a list -> 'a list)
{sort_append}
|*)
let sort order list = sort_append (fun _ -> true) order list [];;

(*sort (prefix <) [1; 9; 5; 2; 3; 4; 2; 1; 0] = [0; 1; 1; 2; 2; 3; 4; 5; 9] *)


(*\
\subsection{Lists of pairs and pairs of lists}
\index{Lists of pairs and pairs of lists}
\begin{caml_primitive}
split
combine
\end{caml_primitive}
These two map a list of pairs into the corresponding pair of lists,
 and conversely (with \verb"combine" failing if its argument
 lists are not of the same length).
\*)
(* Lists of pairs and pairs of lists *)

(*|
Value split : (('a * 'b) list -> 'a list * 'b list)
|*)
let rec split = fun
	((x1,x2) :: l)  ->  let l1,l2 = split l in (x1::l1,x2::l2)
 |	[]  ->  ([],[]);;


(*|
Value combine : ('a list * 'b list -> ('a * 'b) list)
|*)
let rec combine = fun
	(h1::t1,h2::t2)  ->  (h1,h2)::combine(t1,t2)
 |	([],[])  ->  []
 |	_  ->  failwith "combine";;

(*\
\begin{caml_primitive}
it_list2
list_it2
\end{caml_primitive}
These are iterators corresponding to \verb"it_list" and \verb"list_it"
for a curried function and two lists.
\*)

(* Applications of it_list, list_it and combine *)
(*|
Value it_list2
      : (('a -> 'b * 'c -> 'a) -> 'a -> 'b list -> 'c list -> 'a)
{combine,it_list}
|*)
let it_list2 f init list1 list2 =
    it_list f init (combine (list1,list2) ? failwith "it_list2");;

(*|
Value list_it2
      : (('a * 'b -> 'c -> 'c) -> 'a list -> 'b list -> 'c -> 'c)
{combine,list_it}
|*)
let list_it2 f list1 list2  =
    list_it f (combine (list1,list2) ? failwith "list_it2");;


(*\
\subsection{Lists of consecutive integers}
\index{Lists of consecutive integers}
\begin{caml_primitive}
interval
range
\end{caml_primitive}
\begin{itemize}
\item \verb"interval n m" returns the list of all integers in increasing
 order, from \verb"n" to \verb"m".
\item \verb"range n" gives the first n integers.
\end{itemize}
\*)

(* Lists of consecutive integers *)

(*|
Value interval : (num -> num -> num list)
|*)
let interval n m = interval_n ([],m) where rec
 interval_n (l,m) = if n>m then l else interval_n (m::l,pred m);;

(*|
Value range : (num -> num list)
{interval}
|*)
let range = interval 1;;

(*\
\subsection{Testing identity of values in lists}
\index{Testing identity of values in lists}
\begin{caml_primitive}
eq_fst
eq_snd
assq
rev_assq
except_assq
pair_assq
pair_rev_assq
mem_assq
memq
exceptq
add_setq
make_setq
intersectq
subtractq
unionq
\end{caml_primitive}
These are functions corresponding to the ``\verb"assoc"'' and ``\verb"mem"''
 families, using the predicate \verb"eq" instead of \verb"=" to test equality.
\*)
(* Testing identity of values *)


(*|
Value pair_assq : ('a -> ('a * 'b) list -> 'a * 'b)
Value pair_rev_assq : ('a -> ('b * 'a) list -> 'b * 'a)
{find}
|*)
let eq_fst x =  (fun (t,_) -> eq(t,x))
and eq_snd x =  (fun (_,t) -> eq(t,x))
;;

let pair_assq x = find (fun (t,_) -> eq(t,x))
and pair_rev_assq x = find (fun (_,t) -> eq(t,x));;

let mem_assq = exists o eq_fst;;

(*|
Value assq : ('a -> ('a * 'b) list -> 'b)
Value rev_assq : ('a -> ('b * 'a) list -> 'b)
Value memq : ('a -> 'a list -> bool)
{o,pair_assq,pair_rev_assq}
|*)
let assq x = snd o (pair_assq x)
and rev_assq x = fst o (pair_rev_assq x)
and memq x = (memq_x
     where rec memq_x = fun []  ->  false
     			   | (y::l)  ->  eq(x,y) or memq_x l);;

(*|
Value except_assq : ('a -> ('a * 'b) list -> ('a * 'b) list)
{rev_append}
|*)
let except_assq e = except_e where rec except_e = function
    [] -> []
 | (x,_ as y)::l -> if x==e then l else y::except_e l
;;

let add_setq x s = if memq x s then s else x::s;;

(*\
It would be very easy to define the analogue of \verb"make_set" for the
\verb"eq" function:
\begin{caml_example}
let rec make_setq = function
  [] -> []
| (x::L) -> if memq x L then make_setq L else x::make_setq L;;
\end{caml_example}
We choose instead a sharing version, for sake of efficiency:
\*)

let make_setq l = share make_aux l where rec make_aux =
    fun [] -> raise Identity
    |   (x::l) -> if memq x l then share make_aux l else x::make_aux l;;


(*|
Value exceptq : ('a -> 'a list -> 'a list)
|*)
let exceptq e = exceptq_e
  where rec exceptq_e =
   fun [] -> [] 
     | (elem::l) -> if eq(e,elem) then l else elem::exceptq_e l;;


(*|
Value intersectq : ('a list -> 'a list -> 'a list)
Value subtractq : ('a list -> 'a list -> 'a list)
{C,filter,memq,neg}
|*)
let intersectq l1 l2 = filter (C memq l2) l1
and subtractq l1 l2 = filter (neg (C memq l2)) l1;;


(* unionq l1 l2 = subtractq l1 l2 @ l2  *)
(*|
Value unionq : ('a list -> 'a list -> 'a list)
{memq}
|*)
let unionq l1 l2 = urec l1 where rec urec =
    fun []->l2 | (a::l) -> if memq a l2 then urec l else a :: (urec l);;



(*\
\section{Segments}
\index{Segments}
 Segments are homogeneous sequences of CAML values, stored in consecutive
memory cells for the sake of efficiency.
 The positions of items in a segment are counted starting from 0.

\subsection{Creating segments}

As for lists, elements of segments cannot be modified. You may either create 
functional segments (using the keyword \verb"segment" preceding a list) or
segments of data:
\begin{itemize}
%\item \verb"segment l" creates a functional segment whose elements are those%
% of the list \verb"l": it returns %
%a function to fetch these elements.%
%\begin{caml_example}%
%segment [1;2;3];;%
%\end{caml_example}%
\item \verb"segment of l" creates a segment whose elements are those of the
list \verb"l".
\item \verb"segment n of e" creates a segment of length n,
 whose elements are all equal to \verb"e".
\end{itemize}

An alternative way to create segments is to enumerate their elements using
the concrete notation for segments:
\begin{caml_example}
[<1;2;3>];;
\end{caml_example}

\subsection{Primitive operations on segments}
\begin{caml_primitive}
seg_item
\end{caml_primitive}
\begin{itemize}
\item \verb"seg_item (seg,n)" returns the item number \verb"n" of the
segment \verb"seg".
\end{itemize}
\begin{caml_primitive}
seg_length
is_null_seg
\end{caml_primitive}
\begin{itemize}
\item \verb"seg_length seg" returns the length of the segment
\verb"seg".
\item \verb"is_null_seg" is bound to the expected predicate and allows
to test if a segment has no items at all.
\end{itemize}
\begin{caml_primitive}
do_seg
map_seg
do_seg_i
map_seg_i
\end{caml_primitive}
\begin{itemize}
\item \verb"do_seg f seg" is the analog of \verb"do_list" for
 segments: it executes \verb"f" on each element of the segment
\verb"seg" and returns \verb"()".
\item \verb"map_seg f seg" is the analog of \verb"map" for
 segments: it returns the list of results of applying \verb"f" on each
 element of the segment.
\item \verb"map_seg_i" and \verb"do_seg_i"
are slightly different versions of the primitive without \verb"_i":
they provide the item rank in the segment to which the function is applied.
\end{itemize}
\*)
(* 5. Segments *)

(*|
Value do_seg : (('a -> 'b) -> 'a seg -> void)
|*)
let do_seg f v = paired_do_seg (f,v);;
let do_seg_i f v = paired_do_seg_i (f,v);;

let map_seg f v =
 let l = pred (seg_length v) in
 maprec 0 where rec 
    maprec n = f (seg_item (v,n)) :: 
    	       (if n = l then [] else maprec (succ n));;

let map_seg_i f v =
 let l = pred (seg_length v) in
 maprec 0 where rec 
    maprec n = f n (seg_item (v,n)) :: 
    	       (if n = l then [] else maprec (succ n));;

(*\
\begin{caml_primitive}
seg_of_obj
is_segment
\end{caml_primitive}
\begin{itemize}
\item \verb"is_segment obj" tests if its argument \verb"obj" is the
representation of a CAML segment.
\item \verb"seg_of_obj obj" given an \verb"obj" which is the representation
of a segment returns this segment as a segment of \verb"objs".
\end{itemize}
\*)

(*\
\section{Vectors}
\index{Vectors}
 Vectors are homogeneous sequences of CAML values, stored in consecutive
memory cells. The positions of items in a vector are counted
starting from 0.

\subsection{Creating vectors}

As for references, vectors cannot be polymorphic. You may create 
vectors of data using the following syntax, analogous to the syntax for segments:
\begin{itemize}
\item \verb"vector of l" creates a vector whose elements are those of the
list \verb"l".
\item \verb"vector n of e" creates a vector of length n,
 whose elements are all equal to \verb"e".
\end{itemize}

An alternative way to create vectors is to enumerate their elements using
the concrete notation for vectors:
\begin{caml_example}
[|1;2;3|];;
\end{caml_example}

\subsection{Primitive operations on vectors}
\paragraph\

Access and physical modification of cells in a vector have concrete
 syntax constructs
 similar to access and replacement in mutable structures:
\begin{itemize}
\item \verb"vector" \kwd{.} \kwd{(} \syncat{expression} \kwd{)}\\
 where \verb"vector" and \verb"expression"
are expressions evaluating respectively to a vector \verb"v"
and a number \verb"n",
returns the content of the cell of index \verb"n" in the vector \verb"v".
\item \verb"vector" \kwd{.} \kwd{(} \syncat{expression1} \kwd{)} \kwd{<-}
\syncat{expression2}\\
 where \verb"vector", \verb"expression1" and \verb"expression2"
are expressions evaluating respectively to a vector \verb"v",
 a number \verb"n" and an expression \verb"e",
physically modifies
 the content of the cell of index \verb"n" in the vector \verb"v", which
becomes the value \verb"e". The value returned by the
whole construct is \verb"e".
\end{itemize}

\begin{caml_example}
let v = [|1;2;3|];;
let w = [| v; [|5|] |];;
(* Access: vect . ( expr ) *)
w.(0);;
w.(0).(0);;
(* Replacement:  vect . ( expr ) <- expr *)
v.(0) <- 2; v;;
w.(0);;
\end{caml_example}

Primitive functions to access and assign vector's cells are available:
\begin{caml_primitive}
vect_assign
vect_item
\end{caml_primitive}
\begin{itemize}
\item \verb"vect_item (vect,n)" returns the item number \verb"n" of the
vector \verb"vect".
\item \verb"vect_assign (vect,n,value)" changes the element of the vector
\verb"vect" whose index is \verb"n" by the CAML value \verb"value".
\end{itemize}
\begin{caml_primitive}
vect_length
is_null_vect
\end{caml_primitive}
\begin{itemize}
\item \verb"vect_length vect" returns the length of the vector
\verb"vect".
\item \verb"is_null_vect" is bound to the expected predicate and allows one
to test if a vector has no items at all.
\end{itemize}
\begin{caml_primitive}
do_vect
modify_vect
map_vect
do_vect_i
modify_vect_i
map_vect_i
list_of_vect
fold_vect
it_vect
vect_it
\end{caml_primitive}
\begin{itemize}
\item \verb"do_vect f vect" is the analog of \verb"do_list" for
 vectors: it executes \verb"f" on each element of the vector
\verb"vect" and returns \verb"()".
\item \verb"modify_vect f vect" maps the function \verb"f" on the
element of the vector \verb"vect" and modifies each item with the result
of the application of \verb"f" to this item.
\item \verb"map_vect f vect" is the analog of \verb"map" for
 vectors: it returns the list of results of applying \verb"f" on each
 element of the vector.
\item \verb"map_vect_i", \verb"do_vect_i" and \verb"modify_vect_i"
are slightly different versions of the primitive without \verb"_i":
they provide the item rank in the vector to which the function is applied.
\item \verb"list_of_vect v" returns the list of the elements of the vector
 \verb"v".
\item \verb"fold_vect", \verb"it_vect", \verb"vect_it" are the 
functionals for vectors corresponding to \verb"fold", \verb"it_list" and
\verb"list_it".
\end{itemize}
\*)
(* 5. Vectors *)

(*|
Value do_vect : (('a -> 'b) -> 'a vect -> void)
|*)
let do_vect f v = paired_do_vect (f,v);;
let do_vect_i f v = paired_do_vect_i (f,v);;


let map_vect f v =
    maprec 0 where rec 
    maprec n =
     if n >= vect_length v then []
     else  f v.(n) ::  maprec (succ n);;

let map_vect_i f v =
 maprec 0 where rec 
    maprec n =
     if n >= vect_length v then []
     else f n v.(n) :: maprec (succ n);;

let list_of_vect = map_vect I;;

let fold_vect f a1 v =
    let l = vect_length v in
    fold_f a1 0
    where rec fold_f a1 n =
      if n >= l then a1,[]
      else let a2,c2 = f a1 v.(n) in
            let a,cl = fold_f a2 (succ n) in a,(c2::cl);;

let it_vect f a v = it_vect_f a 0
    where rec it_vect_f a n =
    if n>= vect_length v then a else it_vect_f (f a v.(n)) (succ n);;

let vect_it f v a = vect_it_f 0
    where rec vect_it_f n =
    if n>= vect_length v then a else (f v.(n) (vect_it_f (succ n)));;
(*\
\begin{caml_primitive}
vect_of_obj
is_vector
\end{caml_primitive}
\begin{itemize}
\item \verb"is_vector obj" tests if its \verb"obj" argument is the
representation of a CAML vector.
\item \verb"vect_of_obj obj" given an obj which is the representation
of a vector returns this vector as a vector of objs.
\end{itemize}
\*)
(*\begin{caml_ignore}*)
(* 6. String operations *)

(* String comparisons *)


(*|
Value ge_string : (string * string -> bool)
Value lt_string : (string * string -> bool)
Value gt_string : (string * string -> bool)
|*)
let ge_string (s1,s2) = le_string (s2,s1)
and lt_string (s1,s2) = not (le_string (s2,s1))
and gt_string s_pair = not (le_string s_pair);;

(* String basic functions *)


(*|
Value concat : (string -> string -> string)
|*)
let concat x y = x^y;;

(*|
Value concat_list : (string list -> string)
{concat,list_it}
|*)
(*** concat_list == implode!
let concat_list l = list_it concat l "";;
 ***)

(* returns the string of length <len> beginning at <pos> in <s> *)

(*|
Value sub_string : (string -> num -> num -> string)
|*)
let sub_string s p l = tripled_sub_string(s,p,l);;

(* returns the position of the beginning of the first occurrence of s2 *)
(* in s1 after position pos. Returns -1 if s2 does not occur in s1     *)

(*|
Value index_string : (string -> string -> num -> num)
|*)
let index_string s1 s2 pos = tripled_index_string (s2,s1,pos);;

(*let pos_string s1 s2 = index_string s1 s2 0;; *)

(*|
Value pos_string : (string -> string -> num)
|*)
let pos_string s1 s2 = tripled_index_string(s2,s1,0);;

(* Returns the first occurrence in <s> of the character <c> *)

(*|
Value pos_ascii : (num -> string -> num)
|*)
let pos_ascii c s = paired_pos_ascii (c,s);;

(* Returns the first occurrence in s1 after pos    *)
(* of a character belonging to s2                  *)

(*|
Value scan_string : (string -> string -> num -> num)
|*)
let scan_string s1 s2 pos = tripled_scan_string (s1,s2,pos);;


(* Returns the first occurrence in s1 after pos  *)
(* of a character NOT belonging to s2            *)

(*|
Value span_string : (string -> string -> num -> num)
|*)
let span_string s1 s2 pos = tripled_span_string (s1,s2,pos);;


(*|
Value ascii : (num -> string)
{o,singleton}
|*)
let ascii = implode_ascii o singleton;;


(*|
Value ascii_code : (string -> num)
|*)
let ascii_code s = nth_ascii (0,s);;

(* make a string of n characters which all have char as ASCII code *)

(*|
Value make_string : (num -> string -> string)
{ascii_code}
|*)
let make_string n char =
    if n<0 then failwith "make_string"
     else paired_make_string (n, ascii_code char);;

(* Fill <s> after <pos> with the character <c> *)

(*|
Value fill_string : (string -> num -> num -> string)
|*)
let fill_string s c pos = tripled_fill_string (s,c,pos);;

(* Replaces in s1 the string beginning in pos by s2. It does not replace *)
(* more characters than length of s2 and never changes the length of s1. *)
(* BEWARE : It physically modifies s1. In case of sharing it can be very *)
(* confusing                                                             *)

(*|
Value replace_string : (string -> string -> num -> string)
|*)
let replace_string s1 s2 pos = tripled_replace_string (s1,s2,pos);;

(* Replaces s1 by the string beginning at pos in s2. It does not replace    *)
(* more characters than (length of s2)-pos and never changes the length     *)
(* of s1.                                                                   *)
(* BEWARE : It physically modifies s1. In case of sharing it can be very    *)
(* confusing                                                                *)

(*|
Value ireplace_string : (string -> string -> num -> string)
|*)
let ireplace_string s1 s2 pos = tripled_ireplace_string (s1,s2,pos);;


(* Returns character number n in a string *)

(*|
Value nth_char : (num -> string -> string)
{sub_string}
|*)
let nth_char n s = sub_string s n 1;;


(*|
Value last_n_string : (num -> string -> string)
Value first_n_string : (num -> string -> string)
{sub_string}
|*)
let last_n_string n s =
    let l = length_string s in
    if l < n then failwith "last_n_string"
    else sub_string s (l - n) n
and first_n_string n s =
    if length_string s < n then failwith "first_n_string"
    else sub_string s 0 n;;

(* Tests presence of a given character string *)

(*|
Value present : (string -> string -> bool)
{scan_string}
|*)
let present char str = not ((scan_string str char 0) = (-1));;

(* To extract a string knowing its (positions) boundaries
  (chars of index from and to are included in the result). *)

(*|
Value extract_string : (string -> num -> num -> string)
{sub_string}
|*)
let extract_string s From To =
 sub_string s From (succ (To - From));;

(* Substitutes all occurences (from pos n) of a character belonging to
   the string char by the string char' in the string s *)
(* e.g.
   substitute_char "ab" "e" "abcdedcba" 0 --> "eecdedcee"
   substitute_char "e" "ab" "abcdedcba" 0 --> "abcdabdcba"
*)
(*|
Value substitute_char : (string -> string -> string -> num -> string)
 CAML_system{extract_string,scan_string}
|*)
let rec substitute_char char char' s = subst_rec where rec subst_rec n =
  match scan_string s char n with
   (-1) -> extract_string s n (length_string s)
  | p -> (extract_string s n (pred p)) ^ char' ^
         (subst_rec (succ p));;

(* Substitutes all occurrences of the string old_str by the string
   new_str into the string s beginning from a given position *)
(* e.g.
   substitute_string "ab" "e" "abcdedcba" 0 --> "ecdedcba"
   substitute_string "e" "ab" "abcdedcba" 0 --> "abcdabdcba"
*)
let rec substitute_string old_str new_str s =
 subst_rec where rec subst_rec n =
  match index_string s old_str n with
   (-1) -> extract_string s n (length_string s)
  | p -> (extract_string s n (pred p)) ^ new_str ^
         (subst_rec (p+(length_string old_str)));;

(* New line , Line feed , Tab , Space *)

(*|
Value return_char : string
Value lf_char : string
Value tab_char : string
Value space_char : string
|*)
let return_char = "\r" (* ascii 13 *)
and line_feed_char = "\n" (* ascii 10 *)
and form_feed_char = "\f" (* ascii 12 *)
and tab_char = "\t" (* ascii 9 *)
and space_char = " " (* ascii 32 *)
and back_space_char = "\b" (* ascii 8 *);;

(* Common words separators *)

(*|
Value space_chars : string
{lf_char,return_char,space_char,tab_char}
|*)
let space_chars =
    space_char^tab_char^line_feed_char^return_char^form_feed_char;;

(* To skip all chars belonging to string skip, at the beginning of string s *)

(*|
Value skip_string : (string -> string -> string)
{span_string,sub_string}
|*)
let skip_string skip s = let pos = (span_string s skip 0) in
if pos = (-1) then "" else sub_string s pos (length_string s);;

(* To skip spaces and tabs *)

(*|
Value skip_space : (string -> string)
{skip_string,space_char,tab_char}
|*)
let skip_space = let sp = space_char^tab_char in skip_string sp;;

(*|
Value skip_space_return : (string -> string)
{skip_string,space_chars}
|*)
let skip_space_return = skip_string space_chars;;

(* Extract the first word of string s (if word is " "*" ") *)

(*|
Value first_word : (string -> string)
{scan_string,skip_space_return,space_chars,sub_string}
|*)
let first_word s =
 let s = skip_space_return s in
 let b = scan_string s space_chars 0 in
 if b <> (-1) then sub_string s 0 b else s;;

(* Split a string in words (result in reverse order) *)

(*|
Value rev_words : (string -> string -> string list)
{scan_string,span_string,sub_string}
|*)
let rev_words sep s = 
 let scan = scan_string s sep
 and span = span_string s sep
 and substring = sub_string s in
 words_aux (span 0) []
 (* pos is the first non-space occurrence
   newpos is the first space occurrence *)
 where rec words_aux pos l =
  if pos < 0 then l else
   (let newpos = scan pos in
    if newpos = (-1) then (substring pos (length_string s))::l
   else words_aux (span newpos) ((substring pos (newpos-pos))::l));;


(* Split a string into words separated by string sep *)

(*|
Value words2 : (string -> string -> string list)
{rev,rev_words}
|*)
let words2 sep = let revwords = rev_words sep in fun s -> rev (revwords s);;

(* If a word is slightly more general than " "*" " *)

(*|
Value words : (string -> string list)
{space_chars,words2}
|*)
let words = words2 space_chars;;


(* break_string "_" "ab_cd__e_" = ["ab"; "cd"; ""; "e"; ""]  *)

(*|
Value break_string : (string -> string -> string list)
{it_list,rev}
|*)
let break_string sep string =
  fst(it_list scan ([],[]) (rev (sep::explode string)))
  where scan (strs,chars) char =
   if char=sep then ((implode chars)::strs,[])
    else (strs,char::chars);;


(* Make a phrase from a list of strings and a separator *)
(* make_phrase " " ["I";"am";"CAML"] --> "I am CAML" *)
(*|
Value make_phrase : (string -> string list -> string)
|*)
let make_phrase sep =
    make_phrase_sep where rec make_phrase_sep =
    	fun [] -> ""
	| [s] -> s
	| (s::l) -> s^sep^(make_phrase_sep l);;

(* chop_string 3 "CAML" --> "CAM","L" *)

(*|
Value chop_string : (num -> string -> string * string)
{sub_string}
|*)
let chop_string n =
 if n<0 then failwith "chop_string"  else
 (fun s -> sub_string s 0 n,sub_string s n (length_string s));;

(* 7. Miscellaneous operations *)

(* 7.1. Bool coercion *)

(*|
Value string_of_bool : (bool -> string)
|*)
let string_of_bool = fun true -> "true" | _ -> "false";;

(*|
Value bool_of_string : (string -> bool)
|*)
let bool_of_string = fun "true" -> true | "false" -> false
                       |  _     -> failwith"bool_of_string";;

let bool_of_obj = fun <:obj<()>> -> false | _ -> true;;


(* 7.2. The Objs *)

(*|
Value cons_obj : (obj -> obj -> obj)
Value is_obj_cons : (obj -> bool)
|*)
let cons_obj x y = <:obj<(^x . ^y)>>
and is_obj_cons = fun (obj_cons c) -> true | _ -> false;;

(*|
Value obj_nil : obj
|*)
let obj_nil = <:obj<()>>;;

(*|
Value obj_left : (obj -> obj)
Value obj_right : (obj -> obj)
|*)
let obj_left = fun <:obj<(^x . ^y)>> -> x | _ -> failwith "obj_left"
and obj_right = fun <:obj<(^x . ^y)>> -> y | _ -> failwith "obj_right";;

let string_of_obj = function 
    (obj_string s) -> s
  | _ -> failwith "string_of_obj"
;;

let string_of_atom a = string_from_obj (obj_atom a)
;;

let rec list_of_obj = function
      obj_atom <:obj:Atom<()>> -> []
   |  obj_cons(first,last) -> first::list_of_obj last
   |  _ -> failwith "list_of_obj: not a list object"
;;

(*|
Value lisp : (string -> obj)
{o}
|*)
let lisp = lisp_eval o obj_of_string;;

(* Warning: the functions
   add mult div sub add_float sub_float mult_float div_float succ_int pred_int
   Co pairing Y sigma pi change update
   are not used in the CAML system *)


(*\end{caml_ignore}*)
