;; Eulisp Module
;; Author: pete broadbery
;; File: syntax-utils.em
;; Date: 15/sep/1991
;;
;; Project: Compiler
;; Description: 
;;   General utils related to ast's
;;  Guesses type of fn. calls.

(defmodule syntx-utl 
  (standard0
   list-fns
   
   syntx-env
   pass
   props

   stop
   )
  ()
  
  (expose syntx-env)

  (defun find-decls (defn)
    (cond ((module-definition-p defn)
	   nil)
	  ((definition-p defn) 
	   (list defn))
	  ((and-decl-p defn)
	   (fold append
		 (mapcar find-decls
			 (and-decl-decls defn))
		 nil))
	  ((rec-decl-p defn)
	   (find-decls (rec-decl-decl defn)))))
  
  (export find-decls)

  (defgeneric get-internal-closed-bindings (obj))

  (defmethod get-internal-closed-bindings ((x syntax-obj))
    (fold append
	  (mapcar get-internal-closed-bindings (subcomponents x))
	  nil))

  (defmethod get-internal-closed-bindings ((x definition))
    (if (binding-closed x) 
	(cons x (call-next-method))
      (call-next-method)))

  (defmethod get-internal-closed-bindings ((x lambda-term))
    ;; XXX Not if its inline...
    nil)

  ;; finds the innermost non-tail posn lambda
  
  (defun get-enclosing-object (fn start)
    (get-enclose-aux fn (enclosing-block start)))

  (defun get-enclose-aux (fn obj)
    (if (fn obj)
	obj
      (get-enclose-aux fn (enclosing-block obj))))

  (defun enclosing-lambda (obj)
    (get-enclosing-object is-real-lambda obj))

  (defgeneric is-real-lambda (obj)
    methods ((((x lambda-term))
	      t)
	     (((x module-block))
	      t)
	     (((x object))
	      nil)))

  (defun enclosing-module (x)
    (get-enclosing-object module-p x))
  
  (export get-internal-closed-bindings enclosing-lambda)

  (defun function-fn (obj)
    (car obj))
  
  (defun function-type (obj)
    (cadr obj))
  
  ;; if we dont know, pretend we do+that it will be sorted out later.
  (defun function-nargs (obj)
    (if (equal (caddr obj) 0)
	(cons () 9999)
      (caddr obj)))

  (defun function-nary-p (obj) 
    nil)

  ;; If at all possible, find the function object referenced by obj. 
  ;; Guessing what a function is.

  (defgeneric find-fn (obj)
    methods ((((x syntax-obj))
	      (list x 'unknown 0))
	     (((x applic-term))
	      (let ((xx (compile-time-value x)))
		(if (null xx)
		    (list x 'unknown 0)
		  (find-fn xx))))
	     (((x lambda-term))
	      ;;(format t "Lambda: nargs: ~a~%" (lambda-nargs x))
	      (list x 'bytefunction (lambda-nargs x)))
	     (((x ident-term))
	      (let ((xx (find-fn (car (ident-defblock x)))))
		(if (and (eq (function-type xx) 'local-defun)
			 (eq (car xx) (enclosing-lambda x)))
		    (cons (car xx)
			  (cons 'lexical
				(cddr xx)))
		  xx)))
	     (((x definition))
	      (if (defn-mutable-p x)
		  (list x 'unknown 0)
		(let ((actual (find-fn (defn-body x))))
		  ;; try to env+call via a jump
		  (if (eq (function-type actual) 'bytefunction)
		      (cons (car actual) 
			    (cons 'lexical ;; was local-defun
				  (cddr actual)))
		    (list x 'unknown 0)))))
	     (((x module-definition))
	      (if (defn-mutable-p x) 
		  (list x 'unknown 0)
		(let ((actual (find-fn (defn-body x))))
		  (if (eq (function-type actual) 'bytefunction)
		      (cons (car actual)
			(cons 'local
			      (cddr actual)))
		    (list x 'unknown 0)))))
	     (((x imported-definition))
	      ;; defined in syntax-utils
	      (list x (import-object-type x) 
		    (import-function-nargs x)))
	     (((x special-term))
	      (list x 'special 0))
	     (((x object))
	      ;;(format t "Find Fn Got Strange: ~a~%" x)
	      (stop x))))
  
  (export function-fn function-type function-nargs function-nary-p find-fn)

  ;; Real analysis: Find the type of a declaration.
  ;; Returns a-list --- keys: mutable, class (bytefunction, bytemacro, internal, 
  ;;                  function --- 'C', object), argtype, 
  (defun classify-decl (defn)
    (let ((lst (std-class-list defn)))
      (if (defn-mutable-p defn)
	  (append (list (list 'mutable t) (cons 'class 'unknown))
		  lst)
	(cons (list 'mutable nil)
	      (append (generic-classify (defn-body defn))
		      lst)))))
  
  (defgeneric generic-classify (body))
  
  (defmethod generic-classify ((lam lambda-term))
    (list (cons 'class 'bytefunction)
	  (list 'argtype (lambda-nargs lam))))

  (defmethod generic-classify ((mlam macro-lambda-term))
    (list (cons 'class 'macro)
	  (list 'argtype (lambda-nargs mlam))))

  (defmethod generic-classify ((x term))
    (list (cons 'class 'unknown)))

  (defmethod generic-classify ((x special-term))
    (cond ((eq (special-term-name x) 'inline-fn)
	   (list (cons 'class 'inline)
		 (cons 'argtype (car (special-term-data x)))
		 (cons 'code (cdr (special-term-data x)))))
	  (t (cons 'class 'unknown))))
    
  (defun decl-class (x)
    (let ((xx (decl-class-uncached x)))
      (if (null xx)
	  (let ((aa (classify-decl x)))
	    ((setter decl-class-uncached) x aa)
	    aa)
	xx)))

  (defun std-class-list (x)
    ;; just the address at the moment...
    (cons (cons 'name (defn-ide x))
	  (cons (list 'address (module-name (enclosing-module x)) (defn-ide x))
		(let ((xx (obj-setter-decl x)))
		  (if (null xx)
		      nil
		    (list (cons 'setter (decl-class xx))))))))
		
  (export decl-class)

  ;; dependencies...
  
  (defun add-dependency (mod defn)
    (if (memq (import-home defn) (module-dependencies mod))
	nil
      ((setter module-dependencies) mod
       (cons (import-home defn) (module-dependencies mod)))))
  
  (export add-dependency)

  
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (defgeneric compute-compile-time-value (x))

  (defmethod compute-compile-time-value ((app applic-term))
    (let ((fn (compile-time-value (applic-fun app)))
	  (args (applic-args app)))
      (format t "Compile time value: ~a~%" app)
      ;;(import-real-name fn)
      ;;(import-home fn)
      (cond ((and (imported-defn-p fn)
		  (eq (import-real-name fn) 'setter)
		  (eq (import-home fn) 'others))
	     (print (find-setter (print (compile-time-value (car args))))))
	    (t nil))))

  (defmethod compute-compile-time-value ((x ident-term))
    (compute-compile-time-value (car (ident-defblock x))))

  (defmethod compute-compile-time-value ((x imported-definition))
    x)

  (defmethod compute-compile-time-value ((x syntax-obj))
    nil)

  ;; Generic --- could look at the property thang...
  ;;
  ;; nasty local-defn case cos of
  ;; (let ((x (if foo a b)))
  ;;   ((setter x) y))
  (defgeneric find-setter (value)
    methods ((((value imported-definition))
	      (import-defn-setter value))
	     (((x local-definition))
	      (if (binding-mutable x) nil
		(obj-setter-decl x)))
	     (((x syntax-obj))
	      nil)
	     (((x (class-of nil)))
	      nil)
	     ))

  (defun compile-time-value (x)
    (let ((xx (cached-compile-time-value x)))
      (cond ((null xx)
	     (let ((val (compute-compile-time-value x)))
	       (when val (format t "Val: ~a->~a~%" x val))
	       ((setter cached-compile-time-value) x 
		(if (null val) 'no-way val))
	       val))
	    ((eq xx 'no-way) nil)
	    (t xx))))

  
  ;; end module
  )
