 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for mlse
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lmlse
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lmlse - compute least-squares solution to a set of linear equations
* 
*       Purpose: Library Routine for mlse - this routine employs the
*                LAPACK DGELS or ZGELS driver to obtain the least-squares
*                solution to a set of linear equations AX=B where
*                B is made up of column vectors, each of which is assumed
*                to be an independent right-hand-side, and X is made up
*                of column vectors, each of which is an independent solution
*                to the corresponding X column vector.
*
*                For an MxN matrix A, if M>=N then the solution is obtained
*                using the QR factorization; if M<N, then the solution is
*                obtained using the LQ factorization.
*
*         Input: kobject A_obj - A matrix
*                kobject B_obj - set of b vectors (col vectors) for RHS
*
*        Output: kobject X_obj - set of x vectors (col vectors for solution
*
*       Returns: TRUE (1) on success, FALSE (0) on failure
*  Restrictions: 
*    Written By: Scott Wilson
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lmlse (
kobject A_obj,
kobject B_obj,
kobject X_obj)
/* -library_def_end */

/* -library_code */
{
        /* SPECIAL NOTE: To anyone who really gets into this kind of stuff...

           There is some slight of hand involved with obtaining the minimum-norm
           solution (meaning that the system is underdetermined). DGELS (and
           ZGELS) both plop the solution vectors into the B array of the
           fortran call. BUT, the number of rows in this array needs to be
           equal to the number of columns in A. On entry the B values are stored
           in the first M rows, and nothing in the rest. The extra rows
           *will* be filled in upon return, containing the solution vector.
           This is all gotten away with in here by using the copy of B_obj data
           that is larger than the data in B_obj itself. The *copy* is fed
           to DGELS and is then picked apart for the solution vectors.  SRW */

        char *lib = "kmatrix", *rtn = "lmlse";
        double *A=NULL,*AT=NULL,*X=NULL,*B=NULL,*BT=NULL,*WORK=NULL;
        kdcomplex *AZ=NULL,*ATZ=NULL,*XZ=NULL,*BZ=NULL,*BTZ=NULL,*WORKZ=NULL;
        int i,j,m,n,num,nplanes,lda,*ipiv,info;
	int typeA,typeB, otype;
	int wc,hc,dc,tc,ec;
	int wb,hb,db,tb,eb;
	int wx,hx,dx,tx,ex;
        int lwork;
        int hbt;
        char trans;
        klist *objlist=NULL;

        /* Make sure we have valid objects */
        if (A_obj == KOBJECT_INVALID || B_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus input object");
            return(FALSE);
          }
        if (X_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus output object");
            return(FALSE);
          }

        /* Reference the input objects to avoid side effects, then add the
           references to the list of goodies to be autmatically free'd on
           error. */
        KCALL((A_obj = kpds_reference_object(A_obj)));
        objlist = klist_add(objlist,A_obj,"KOBJECT");

        /* See if the data type of the input object is real or complex */
        KCALL(kpds_get_attribute(A_obj,KPDS_VALUE_DATA_TYPE,&typeA));
        KCALL(kpds_get_attribute(B_obj,KPDS_VALUE_DATA_TYPE,&typeB));
        if (typeA == KCOMPLEX || typeA == KDCOMPLEX ||
            typeB == KCOMPLEX || typeB == KDCOMPLEX) otype = KDCOMPLEX;
        else otype = KDOUBLE;

        /* Constrain the data to double real or double complex */
        KCALL(kpds_set_attribute(A_obj,KPDS_VALUE_DATA_TYPE, otype));
        KCALL(kpds_set_attribute(B_obj,KPDS_VALUE_DATA_TYPE, otype));

        /* See how big the data is */
        KCALL(kpds_get_attribute(A_obj,KPDS_VALUE_SIZE,&wc,&hc,&dc,&tc,&ec));
	/* Set up to process input by planes */
        KCALL(kpds_set_attribute(A_obj,KPDS_VALUE_REGION_SIZE,wc,hc,1,1,1));
        nplanes = dc*tc*ec;
        KCALL(kpds_get_attribute(B_obj,KPDS_VALUE_SIZE,&wb,&hb,&db,&tb,&eb));
        KCALL(kpds_set_attribute(B_obj,KPDS_VALUE_REGION_SIZE,wb,hb,1,1,1));

        /* Check sizes of things */
        if (hb != hc)
          {
            kinfo(KSTANDARD,"lmlse: system: height of A (%d) must match height of B(%d)\n",hc,hb);
            (void)klist_free(objlist, (kfunc_void)lkcall_free);
            return(FALSE);
          }
        if (dc != db || tc != tb || ec != eb)
          {
            kinfo(KSTANDARD,"lmlse: {D,T,E} of A must match {D,T,E} of B\n");
            (void)klist_free(objlist, (kfunc_void)lkcall_free);
            return(FALSE);
          }

        /* Set the output object size and data type */
        if (!kpds_query_value(X_obj)) KCALL(kpds_create_value(X_obj));
        wx = wb; hx = wc; dx = dc; tx = tc; ex = ec;
        KCALL(kpds_set_attributes(X_obj,
                             KPDS_VALUE_SIZE,wx,hx,dx,tx,ex,
                             KPDS_VALUE_REGION_SIZE,wx,hx,1,1,1,
                             KPDS_VALUE_DATA_TYPE,otype,NULL));

        /* Now ref the output object since it should be all set up */
        KCALL((X_obj = kpds_reference_object(X_obj)));
        objlist = klist_add(objlist,X_obj,"KOBJECT");

        /* Allocate buffers */
        lwork = kmin(wc,hc)+kmax3(wc,hc,wb)*8;

        if (hc < wc) hbt = wc; /* For underdetermined case */
        else hbt = hb; /* For over- or exactly determined case */

        if (otype == KDOUBLE)
          {
            KCALL(!((A=(double *)kmalloc(wc*hc*sizeof(double))) == NULL));
	    objlist = klist_add(objlist,A,"KMALLOC");
            KCALL(!((AT=(double *)kmalloc(wc*hc*sizeof(double))) == NULL));
	    objlist = klist_add(objlist,AT,"KMALLOC");
            KCALL(!((B=(double *)kmalloc(wb*hb*sizeof(double))) == NULL));
	    objlist = klist_add(objlist,B,"KMALLOC");
            KCALL(!((BT=(double *)kmalloc(wb*hbt*sizeof(double))) == NULL));
	    objlist = klist_add(objlist,BT,"KMALLOC");
            KCALL(!((X=(double *)kmalloc(wx*hx*sizeof(double))) == NULL));
	    objlist = klist_add(objlist,X,"KMALLOC");
            KCALL(!((WORK=(double *)kmalloc(lwork*sizeof(double))) == NULL));
	    objlist = klist_add(objlist,WORK,"KMALLOC");
          }
        else
          {
            KCALL(!((AZ=(kdcomplex *)kmalloc(wc*hc*sizeof(kdcomplex)))== NULL));
            objlist = klist_add(objlist,AZ,"KMALLOC");
            KCALL(!((ATZ=(kdcomplex *)kmalloc(wc*hc*sizeof(kdcomplex)))==NULL));
            objlist = klist_add(objlist,ATZ,"KMALLOC");
            KCALL(!((BZ=(kdcomplex *)kmalloc(wb*hb*sizeof(kdcomplex)))== NULL));
            objlist = klist_add(objlist,BZ,"KMALLOC");
            KCALL(!((BTZ=(kdcomplex *)kmalloc(wb*hbt*sizeof(kdcomplex)))==NULL));
            objlist = klist_add(objlist,BTZ,"KMALLOC");
            KCALL(!((XZ=(kdcomplex *)kmalloc(wx*hx*sizeof(kdcomplex)))== NULL));
            objlist = klist_add(objlist,XZ,"KMALLOC");
            KCALL(!((WORKZ=(kdcomplex *)kmalloc(lwork*sizeof(kdcomplex)))== NULL));
	    objlist = klist_add(objlist,WORKZ,"KMALLOC");
          } 

	trans='N';
        for (num=0; num<nplanes; num++)
          {
            /* Get a plane from the input objects */
            if (otype == KDOUBLE)
              {
                A = (double *)kpds_get_data(A_obj,KPDS_VALUE_REGION,(kaddr)A);
                B = (double *)kpds_get_data(B_obj,KPDS_VALUE_REGION,(kaddr)B);
              }
            else
              {
                AZ=(kdcomplex *)kpds_get_data(A_obj,KPDS_VALUE_REGION,(kaddr)AZ);
                BZ=(kdcomplex *)kpds_get_data(B_obj,KPDS_VALUE_REGION,(kaddr)BZ);
              }

            /* Transpose A so that F77 will like it */
            if (otype == KDOUBLE)
              {
                for (i=0; i<hc; i++)
                  for (j=0; j<wc; j++)
                    AT[j*hc+i] = A[i*wc+j];
              }
            else
              {
                for (i=0; i<hc; i++)
                  for (j=0; j<wc; j++)
                    ATZ[j*hc+i] = AZ[i*wc+j];
              }
 
            /* Transpose B so that F77 will like it */
            if (otype == KDOUBLE)
              {
                for (i=0; i<hb; i++)
                  for (j=0; j<wb; j++)
                    BT[j*hbt+i] = B[i*wb+j];
              }
            else
              {
                for (i=0; i<hb; i++)
                  for (j=0; j<wb; j++)
                    BTZ[j*hbt+i] = BZ[i*wb+j];
              }

            /* Obtain solution */
	    if (otype == KDOUBLE)
              DGELS(&trans,&hc,&wc,&wb,AT,&hc,BT,&hbt,WORK,&lwork,&info);
            else
              ZGELS(&trans,&hc,&wc,&wb,ATZ,&hc,BTZ,&hbt,WORKZ,&lwork,&info);

            if (info != 0)
              {
                kinfo(KSTANDARD,"lmlse: LAPACK {dz}gels() returned with INFO=%d\n\
where INFO < 0 -> for INFO = -i, the i'th argument had an illegal value",
                  info);
                (void)klist_free(objlist, (kfunc_void)lkcall_free);
                return(FALSE);
              }

            /* Pull out the solution vectors from BT. Look out for oddness
               in addressing the F77 style BT data */
            if (otype == KDOUBLE)
              {
                for (i=0; i<wb; i++) /* Sol vect # (col) */
                  for (j=0; j<hx; j++) /* Sol comp # (row) */
                    X[j*wx+i] = BT[i*hbt+j];
              }
            else
              {
                for (i=0; i<wb; i++)
                  for (j=0; j<hx; j++)
                    XZ[j*wx+i] = BZ[i*hbt+j];
              }

            /* Write the data to the output objects */
            if (otype == KDOUBLE)
              {
                KCALL(kpds_put_data(X_obj,KPDS_VALUE_REGION,(kaddr)X));
              }
            else
              {
                KCALL(kpds_put_data(X_obj,KPDS_VALUE_REGION,(kaddr)XZ));
              }
          }

        (void)klist_free(objlist, (kfunc_void)lkcall_free);

        return(TRUE);
  }
/* -library_code_end */
