 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for imedian
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	limedian
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
#define IPIXEL(x,y) ((y)*wr + (x))
#define OPIXEL(x,y) ((y)*w + (x))

long limedian_long_median(long *,int);
double limedian_double_median(double *,int);
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: limedian - perform median filtering
* 
*       Purpose: for each WxH window in the image, replace the pixel
*		 at (W/2)x(H/2) (integer arithmetic) with the median of
*		 the values in the window. The median is computed by
*		 sorting the values in the window with quicksort and
*		 extracting the middle value as the median. This is
*		 not the fastest approach for BYTE data where a
*		 histogram updating method could be used. It is, however,
*		 extensible to higher data types including floating
*		 point data.
*
*		 If a map is present, the data is mapped prior to
*		 operation. Mask data is ignored.
*
*		 This implementation processes the image data by
*		 full planes. 
*
*		 Boundaries are handled by zero padding.
*
*         Input: in_obj - input object to be processed
*		 wsize - width of window in pixels
*		 hsize - height of window in pixels
*		 reps -  number of repetitions of filter to perform
*
*        Output: out_obj - output object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Complex data types are not handled at all.
*    Written By: Scott Wilson (stolen from K1.5 vhmed and vqmed, which were
written by Ramiro Jordan and Marcelo Teran)
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int limedian(kobject in_obj, int wsize, int hsize, int reps, kobject out_obj)
/* -library_def_end */

/* -library_code */
{
        char *lib = "kdatamanip", *rtn = "limedian";
        klist *objlist=NULL;
        unsigned int i,j,nd,nt,ne,x,y,rcount;
        unsigned int ylo,yhi,xlo,xhi;
        int itype,otype;
        int map_exists,mask_exists;
        unsigned int w,h,d,t,e; /* Dimensions of input data set */
        unsigned int wr,hr;
        unsigned short *si,*so;
        long *li,*lo;
        double *fi,*fo;
        double *farray,*fptr;
        long *larray,*lptr,cnt;

        /* Make sure we have valid objects */
        if (in_obj == KOBJECT_INVALID || out_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus input or output object");
            return(FALSE);
          }

        /* See if the data type of the input object is real or complex */
        KCALL(kpds_get_attribute(in_obj,KPDS_VALUE_DATA_TYPE,&itype));
        if (itype == KCOMPLEX || itype == KDCOMPLEX) 
          {
            kerror(lib,rtn, "Can't handle complex data types (yet).");
            return(FALSE);
          }

        /* Reference the input object to avoid side effects, then add the
           reference to the list of goodies to be autmatically free'd on
           error. */
        KCALL((in_obj = kpds_reference_object(in_obj)));
        objlist = klist_add(objlist,in_obj,"KOBJECT");

        /* Create the output value segment if it's not there already and
           copy it's size and data type attributes from the input obj.  */
        KCALL(kpds_copy_object(in_obj,out_obj));
 
        mask_exists=kpds_query_mask(in_obj);
        map_exists=kpds_query_map(in_obj);
 
        /* If map exists, push all the data thru the map */
        if (map_exists)
          {
            KCALL(kpds_set_attribute(in_obj,KPDS_MAPPING_MODE,KMAPPED));
            KCALL(kpds_destroy_map(out_obj));
          }

        /* Fix up the data type for presentation */
        if ( itype == KINT || itype == KUINT ||
             itype == KLONG || itype == KULONG )
             otype = KLONG;
        else if (itype == KFLOAT || itype == KDOUBLE) otype = KDOUBLE;
        else otype = KUSHORT;
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_DATA_TYPE,otype));
        KCALL((out_obj = kpds_reference_object(out_obj)));
        objlist = klist_add(objlist,out_obj,"KOBJECT");
        KCALL(kpds_set_attribute(out_obj,KPDS_VALUE_DATA_TYPE,otype));

        /* See how big the data is */
        KCALL(kpds_get_attribute(in_obj,KPDS_VALUE_SIZE,&w,&h,&d,&t,&e));

        /* Do the actual processing differently depending on the type of
           data we have to work with */
        if (otype == KUSHORT)
          {
            wr = w+2*(wsize/2); hr = h+2*(hsize/2); /* INTEGER ARITHMETIC!*/
            /* Allocate space for the input data, sort, and output array */
            KCALL(!((si=(unsigned short *)
                               kmalloc(wr*hr*sizeof(unsigned short)))== NULL));
            objlist = klist_add(objlist,si,"KMALLOC");
            KCALL(!((so=(unsigned short *)
                               kmalloc(w*h*sizeof(unsigned short)))== NULL));
            objlist = klist_add(objlist,so,"KMALLOC");
            KCALL(!((larray=(long *)kmalloc(wsize*hsize*sizeof(long)))== NULL));
            objlist = klist_add(objlist,larray,"KMALLOC");
            cnt = wsize*hsize;

            /* Set up the proper region size for accessing the input data.
               Turn on padding so we don't have to mess with boundary
               processing */
            KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_REGION_SIZE,wr,hr,1,1,1));
            KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_INTERPOLATE,KPAD));

            /* Step thru the input data with the window centered on each pixel.
               Do the median operation, and slap the result into the output
               data object. This is done one plane at a time for efficiency. */
            for (nd=0; nd<d; nd++)
              {
                for (nt=0; nt<t; nt++)
                  {
                    for (ne=0; ne<e; ne++)
                      {
                        /* Process a plane of data */
                        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_POSITION,
                                         -wsize/2,-hsize/2,nd,nt,ne));
                        KCALL(kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)si));
                        for (rcount = 0; rcount < reps; rcount++)
                          {
                            for (j=0; j<h; j++)
                              {
                                for (i=0; i<w; i++)
                                  {
                                    lptr = larray;
                                    ylo = 0; yhi = ylo+hsize;
                                    xlo = 0; xhi = xlo+wsize;
                                    for (y=ylo; y<yhi; y++)
                                      for (x=xlo; x<xhi; x++)
                                        *lptr++ = si[IPIXEL(i+x,j+y)];
                                    so[OPIXEL(i,j)] = 
                                      limedian_long_median(larray,cnt);
                                  }
                              }
                            if (rcount < reps-1) /* Copy back for iteration */
                              for (j=0; j<h; j++)
                                for (i=0; i<w; i++)
                                  si[IPIXEL(i+wsize/2,j+hsize/2)] =
                                    so[OPIXEL(i,j)];
                          }
                        KCALL(kpds_put_data(out_obj,KPDS_VALUE_PLANE,(kaddr)so));
                      }
                  }
              }
          }
        else if (otype == KLONG)
          {
            wr = w+2*(wsize/2); hr = h+2*(hsize/2); /* INTEGER ARITHMETIC!*/
            /* Allocate space for the input data, sort, and output array */
            KCALL(!((li=(long *)kmalloc(wr*hr*sizeof(long)))== NULL));
            objlist = klist_add(objlist,li,"KMALLOC");
            KCALL(!((lo=(long *)kmalloc(w*h*sizeof(long)))== NULL));
            objlist = klist_add(objlist,lo,"KMALLOC");
            KCALL(!((larray=(long *)kmalloc(wsize*hsize*sizeof(long)))== NULL));
            objlist = klist_add(objlist,larray,"KMALLOC");
            cnt = wsize*hsize;

            /* Set up the proper region size for accessing the input data.
               Turn on padding so we don't have to mess with boundary
               processing */
            KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_REGION_SIZE,wr,hr,1,1,1));
            KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_INTERPOLATE,KPAD));

            /* Step thru the input data with the window centered on each pixel.
               Do the median operation, and slap the result into the output
               data object. This is done one plane at a time for efficiency. */
            for (nd=0; nd<d; nd++)
              {
                for (nt=0; nt<t; nt++)
                  {
                    for (ne=0; ne<e; ne++)
                      {
                        /* Process a plane of data */
                        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_POSITION,
                                         -wsize/2,-hsize/2,nd,nt,ne));
                        KCALL(kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)li));
                        for (rcount = 0; rcount < reps; rcount++)
                          {
                            for (j=0; j<h; j++)
                              {
                                for (i=0; i<w; i++)
                                  {
                                    lptr = larray;
                                    ylo = 0; yhi = ylo+hsize;
                                    xlo = 0; xhi = xlo+wsize;
                                    for (y=ylo; y<yhi; y++)
                                      for (x=xlo; x<xhi; x++)
                                        *lptr++ = li[IPIXEL(i+x,j+y)];
                                    lo[OPIXEL(i,j)] =
                                      limedian_long_median(larray,cnt);
                                  }
                              }
                            if (rcount < reps-1) /* Copy back for iteration */
                              for (j=0; j<h; j++)
                                for (i=0; i<w; i++)
                                  li[IPIXEL(i+wsize/2,j+hsize/2)] =
                                    lo[OPIXEL(i,j)];
                          }
                        KCALL(kpds_put_data(out_obj,KPDS_VALUE_PLANE,(kaddr)lo));
                      }
                  }
              }
            /* More data types here */
          }
        else if (otype == KDOUBLE)
          {
            wr = w+2*(wsize/2); hr = h+2*(hsize/2); /* INTEGER ARITHMETIC!*/
            /* Allocate space for the input data, sort, and output array */
            KCALL(!((fi=(double *)kmalloc(wr*hr*sizeof(double)))== NULL));
            objlist = klist_add(objlist,fi,"KMALLOC");
            KCALL(!((fo=(double *)kmalloc(w*h*sizeof(double)))== NULL));
            objlist = klist_add(objlist,fo,"KMALLOC");
            KCALL(!((farray=(double *)kmalloc(wsize*hsize*sizeof(double)))== NULL));
            objlist = klist_add(objlist,farray,"KMALLOC");
            cnt = wsize*hsize;
 
            /* Set up the proper region size for accessing the input data.
               Turn on padding so we don't have to mess with boundary
               processing */
            KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_REGION_SIZE,wr,hr,1,1,1))
;
            KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_INTERPOLATE,KPAD));
 
            /* Step thru the input data with the window centered on each pixel.
               Do the median operation, and slap the result into the output
               data object. This is done one plane at a time for efficiency. */
            for (nd=0; nd<d; nd++)
              {
                for (nt=0; nt<t; nt++)
                  {
                    for (ne=0; ne<e; ne++)
                      {
                        /* Process a plane of data */
                        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_POSITION,
                                         -wsize/2,-hsize/2,nd,nt,ne));
                        KCALL(kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)fi))
;
                        for (rcount = 0; rcount < reps; rcount++)
                          {
                            for (j=0; j<h; j++)
                              {
                                for (i=0; i<w; i++)
                                  {
                                    fptr = farray;
                                    ylo = 0; yhi = ylo+hsize;
                                    xlo = 0; xhi = xlo+wsize;
                                    for (y=ylo; y<yhi; y++)
                                      for (x=xlo; x<xhi; x++)
                                        *fptr++ = fi[IPIXEL(i+x,j+y)];
                                    fo[OPIXEL(i,j)] =
                                      limedian_double_median(farray,cnt);
                                  }
                              }
                            if (rcount < reps-1) /* Copy back for iteration */
                              for (j=0; j<h; j++)
                                for (i=0; i<w; i++)
                                  fi[IPIXEL(i+wsize/2,j+hsize/2)] =
                                    fo[OPIXEL(i,j)];
                          }	
                        KCALL(kpds_put_data(out_obj,KPDS_VALUE_PLANE,(kaddr)fo))
;
                      }
                  }
              }
          }
        else
          {
            kerror(lib, rtn, "Unsupported data type for image data.");
          }

        (void)klist_free(objlist,(kfunc_void)lkcall_free); 
	return TRUE;
}

/* This is a median search based on that found in "Algorithms in C",
   by Robert Sedgewick, Addison-Wesley, 1990, p128. ISBN 0-201-51425-7.
   It's really a modified Quicksort, but it only does a partial sort. */
double limedian_double_median(double *x, int n)
  {
    int i,j;
    int left,right;
    int mid;
    double tmp,val;
    mid = n/2;
    left=0;
    right=n-1;
    while (right>left) /* Run until left and right hit at the mid */
     {
       val = x[right];
       i = left-1; /* Set left and right markers */
       j = right;
       for (;;) /* Stop when markers meet; else keep swapping values */
         {
           while (i < n-1 && x[++i] < val);
           while (j > 0 && x[--j] > val);
           if (i >= j) break;
           tmp = x[i]; x[i] = x[j]; x[j] = tmp;
         }
       tmp = x[i]; x[i] = x[right]; x[right] = tmp; /* Update search value */
       if (i >= mid) right = i-1; /* Bring in left and right bounds */
       if (i <= mid) left = i+1;
     }
    return(x[mid]);
  }

long limedian_long_median(long *x, int n)
  {
    int i,j;
    int left,right;
    int mid;
    long tmp,val;
    mid = n/2;
    left=0;
    right=n-1;
    while (right>left) /* Run until left and right hit at the mid */
     {
       val = x[right];
       i = left-1; /* Set left and right markers */
       j = right;
       for (;;) /* Stop when markers meet; else keep swapping values */
         {
           while (i < n-1 && x[++i] < val);
           while (j > 0 && x[--j] > val);
           if (i >= j) break;
           tmp = x[i]; x[i] = x[j]; x[j] = tmp;
         }
       tmp = x[i]; x[i] = x[right]; x[right] = tmp; /* Update search value */
       if (i >= mid) right = i-1; /* Bring in left and right bounds */
       if (i <= mid) left = i+1;
     }
    return(x[mid]);
  }
/* -library_code_end */
