 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for igauss_func
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	ligauss_func
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
int gauss_gen PROTO((int, int, double, double, double, double, int, int, int, int, int, double **));
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: ligauss_func - generate 2D gaussian function data
* 
*       Purpose: library routine for igauss_func. ligauss_func    
*                creates a 2D gaussian function value data set
*                containing one or many gaussian functions.For each 
*                gaussian function generated, 6 parameters are required:
*                (x,y) coordinate of the peak, variance along x&y direction,
*                correlation coefficient between the two variables of the
*                2D gaussian function and the peak value of the gaussian
*                function. For generating more than one gaussian function this 
*                information must be stored in param_obj along its bands,
*                organised in the order described above. If param_obj is 
*                NULL it is assumed that only one gaussian is to be
*                generated. The parameters of that gaussian are passed
*                in directly. For the multiple gaussian case the amplitude
*                values are ignored if norm is TRUE. The output object, dst_obj
*                has size rows*cols*1 and datatype given by type. Complex data
*                types are not supported for ligauss_func.
*
*         Input: param_obj - object containing multiple gaussina parameters
*                rows      - number of rows in output object
*                cols      - number of columns in output object 
*                xpeak     - x coordinate of function peak value
*                ypeak     - y coordinate of function peak value
*                xvar      - variance along the x direction
*                yvar      - variance along the y  direction
*                coeff     - correlation coefficient between the 2 dimensions
*                val       - peak value of the gaussian function
*                norm      - flag indicating normalisation, if TRUE multiple 
*                            gaussian amplitude is normalised
*                type      - datatype of the output object
*
*        Output: dst_obj   - destination object contain gaussian function 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Ashish Malhotra
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int ligauss_func(kobject param_obj,
                 int rows, int cols,
                 int xpeak, int ypeak,
                 double xvar, double yvar,
                 double coeff, double val,
                 int norm, int type,
                 kobject dst_obj)
/* -library_def_end */

/* -library_code */
{
    char *lib="kimage_proc";
    char *rtn="ligauss_func";
    int  multi=FALSE;
    int  par_w,par_h,par_d,par_t,par_e;
    int  num,num2,start;
    int  num_regs;
    double *data=NULL;
    double *tmp_ptr=NULL;
    double *params=NULL;

    par_w=par_h=par_t=par_e=0;

    if (param_obj != NULL) multi = TRUE;

    if (!kpds_query_value(dst_obj))
    {
       if (!kpds_create_value(dst_obj))
       {
          kerror(lib, rtn, "Unable to create destination value data.");
          return(FALSE);
       }
    }

    if (!kpds_set_attributes(dst_obj,KPDS_VALUE_SIZE,cols,rows,1,1,1,
                             KPDS_VALUE_DATA_TYPE,type,NULL))
    {
       kerror(lib, rtn, "Unable to set destination object attributes.");
       return(FALSE);
    }

    if (multi)
    {
       if ((param_obj = kpds_reference_object(param_obj)) == KOBJECT_INVALID)
       {
          kerror(lib, rtn, "Failed to duplicate parameter object.");
          return (FALSE);
       }

       if (kpds_query_mask(param_obj))
       {
          kerror(lib, rtn, "Mask segment is not allowed in parameter object.");
          (void)kpds_close_object(param_obj);
          return (FALSE);
       }

       if (kpds_query_map(param_obj))
       {
          kpds_set_attribute(param_obj,KPDS_MAPPING_MODE,KMAPPED);
       }

       kpds_get_attribute(param_obj,KPDS_VALUE_SIZE,&par_w,&par_h,&par_d,
                          &par_t,&par_e,NULL);
       if (par_e != 6)
       {
          kerror(lib, rtn, "Paramater object shld. have 6 bands.");
          (void)kpds_close_object(param_obj);
          return (FALSE);
       }

       par_w *= par_h;
       par_w *= par_t;
    }

    if ((dst_obj = kpds_reference_object(dst_obj)) == KOBJECT_INVALID)
    {
       kerror(lib, rtn, "Failed to duplicate output object");
       if (multi) (void) kpds_close_object(param_obj);
       return(FALSE);
    }

    kpds_set_attribute(dst_obj,KPDS_VALUE_DATA_TYPE,KDOUBLE);

    kpds_set_attribute(dst_obj,KPDS_VALUE_SIZE,cols,1,1,1,1);
    kpds_get_attribute(dst_obj,KPDS_VALUE_OPTIMAL_REGION_SIZE,&num,NULL,NULL,
                       NULL,NULL,NULL);
    kpds_set_attributes(dst_obj,KPDS_VALUE_SIZE,cols,rows,1,1,1,
                        KPDS_VALUE_REGION_SIZE,num,1,1,1,1,NULL);

    if (num != cols) 
       start=num; 
    else 
       start=cols;

    data = (double *)kpds_get_data(dst_obj,KPDS_VALUE_REGION,
                     (kaddr)data);

    kpds_get_attribute(dst_obj,KPDS_VALUE_REGION_INFO,NULL,NULL,NULL,NULL,NULL,
      &num_regs);

    /*num_regs = rows*cols/start;*/

    if (!multi)
    {
       for (num=0; num<num_regs; num++)
       {
          tmp_ptr = (data+((start*num)%cols));
          (void) gauss_gen(xpeak,ypeak,xvar,yvar,coeff,val,0,start,
                          (start*num)%cols,num,0,&tmp_ptr);
   
          if (!kpds_put_data(dst_obj,KPDS_VALUE_REGION,data))
          {
             kerror(lib, rtn, "Failed to put data in output object");
             if (multi) (void) kpds_close_object(param_obj);
             kpds_close_object(dst_obj);
             return(FALSE);
          }
       }
    }
    else
    {
       for (num=0; num<num_regs; num++)
       {
          for (num2=0; num2<par_w; num2++)
          {
             tmp_ptr = (data+((start*num)%cols));
             params = (double *)kpds_get_data(param_obj,
                      KPDS_VALUE_VECTOR,(kaddr)params);

             if (params[2] <= 0.0 || params[3] <= 0.0)
             {
                kerror(lib, rtn, "Variance cannot be less than 0");
                if (multi) (void) kpds_close_object(param_obj);
                kpds_close_object(dst_obj);
                return(FALSE);
             }
           
             if (params[4] > 1  || params[4] < -1)
             {
                kerror(lib, rtn, "Correlation coefficient must be between -1 and1");
                if (multi) (void) kpds_close_object(param_obj);
                kpds_close_object(dst_obj);
                return(FALSE);
             }

             (void) gauss_gen((int)params[0],(int)params[1],params[2],params[3],
                              params[4],params[5],norm,start,(start*num)%cols,
                              num,num2,&tmp_ptr);
          } 
          if (!kpds_put_data(dst_obj,KPDS_VALUE_REGION,data))
          {
             kerror(lib, rtn, "Failed to put data in output object");
             if (multi) (void) kpds_close_object(param_obj);
             kpds_close_object(dst_obj);
             return(FALSE);
          }
       }
    }

    if (multi) (void) kpds_close_object(param_obj);
    kpds_close_object(dst_obj);

    return TRUE;
}

/************************************************************
*
*  Routine Name: gauss_gen - gaussian generator for a line of data
*
*       Purpose: Generates a 2D gaussian function for a line in the
*                image.
*
*         Input: xpeak     - x location of peak of gaussian function
*                ypeak     - y location of peak of gaussian function
*		 xvar      - variance along x dimension
*		 yvar      - variance along y dimension
*                corr_coeff- correlation coefficient between 2 dimensions
*                ampl      - amplitude of the gaussian function
*		 norm      - normalization flag, if set normalize
*		 size      - size of the output data
*		 start_xpos- starting position of x coordinate
*		 ypos      - y position of gaussian function
*                flag      - add flag, if set values are added to existing 
*		             values
*
*        Output: data      - array containing gaussian data
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Ashish Malhotra
*          Date: May 2, 1994
*      Verified:
*  Side Effects:
* Modifications: *
*   Declaration: int gauss_gen(
*		 !   int xpeak,
*		 !   int ypeak, 
*		 !   double xvar, 
*		 !   double yvar, 
*		 !   double corr_coeff, 
*		 !   double ampl, 
*		 !   int norm, 
*		 !   int size, 
*		 !   int start_xpos,
*		 !   int ypos, 
*		 !   int flag, 
*		 !   double **data)
*************************************************************/
int gauss_gen(int xpeak,int ypeak, 
              double xvar, double yvar, 
              double corr_coeff, double ampl, 
              int norm, int size, int start_xpos,
              int ypos, int flag, double **data)
{
   double coeff2      ;  /* 1 - corr_coeff**2   */
   double sdev1,sdev2 ;  /* Standard deviations */
   double t1, t2, t3  ;  /* Cross terms         */
   double exp1        ;  /* Exponent multiplier */
   double exp2        ;  /* Full exponent       */
   double amp2        ;  /* Constant in front   */
   int    i,j         ;  /* Loop counter        */

   if ((data != NULL) && (*data == NULL))
   {
      flag = FALSE;
      if ((*data = (double *)kmalloc(size*sizeof(double))) == NULL)
         return FALSE;
   }

   coeff2 = ksqrt(1-corr_coeff*corr_coeff);
   sdev1  = ksqrt(xvar);
   sdev2  = ksqrt(yvar);

   amp2  = 1.0/(K2PI*sdev1*sdev2*coeff2);
   if (!norm) amp2 = ampl;

   for (i=0,j=start_xpos; i<size; i++,j++)
   {
      t1 = (j-xpeak)*(j-xpeak)/xvar;
      t2 = 2.0*corr_coeff*(j-xpeak)*(ypos-ypeak)/(sdev1*sdev2);
      t3 = (ypos-ypeak)*(ypos-ypeak)/yvar;

      exp1 = 1.0/(2.0*coeff2*coeff2);
      exp2 = -exp1*(t1-t2+t3);
      if (flag)
         (*data)[i] += amp2 * kexp(exp2);
      else
         (*data)[i] = amp2 * kexp(exp2);
   }

   return TRUE;
} 
/* -library_code_end */
