
/*********************************************************************

Copyright (C) 1995, Lawrence Berkeley Laboratory.  All Rights
Reserved.  Permission to copy and modify this software and its
documentation (if any) is hereby granted, provided that this notice
is retained thereon and on all copies.  

This software is provided as a professional academic contribution
for joint exchange.   Thus it is experimental and scientific
in nature, undergoing development, and is provided "as is" with
no warranties of any kind whatsoever, no support, promise of
updates or printed documentation.

This work is supported by the U. S. Department of Energy under 
contract number DE-AC03-76SF00098 between the U. S. Department 
of Energy and the University of California.


	Author: Wes Bethel
		Lawrence Berkeley Laboratory
                Berkeley, California 

  "this software is 100% hand-crafted by a human being in the USA"

  This file contains routines for accessing and displaying (within the
  software renderer) triangle strips.

  There are a number of combinations of colors and normals, not all of
  which are supported.

  Vertices    Normals    Colors
1.   n           n         n           - per-vertex everything
2.   n           n        n-2          - per-vertex normals, per-face colors
3.   n          n-2        n           - per-vertex colors, per-face normals
4.   n          n-2       n-2          - per-face colors and normals

     Of these permutations 1, 3, 4 are supported.  2 is presently
     not supported.

*********************************************************************/

#include "rmonster.h"
#include "rmonster_objs.h"
#include "poly.h"
#include "camera.h"

static Poly work_poly;
static PWindow work_window;

extern void (*opaque_shaderfuncs[])();
extern void (*transparent_shaderfuncs[])();
extern void (*init_shaderfuncs[])();
extern int num_init_shaderfuncs;

static void polytri_con_no_color PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void polytri_con_alpha_per_vertex PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void polytri_con_alpha_per_face PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void polytri_con_noalpha_per_vertex PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void polytri_con_noalpha_per_face PROTO((rmonster_prim *,rmonsterobj *,int,float *));
    
void polytri_con(rmonster_prim *p,
		 int xs, int ys,
		 vertex_3d *obj_space_verts,
		 vertex_3d *view_space_verts,
		 vertex_3d *norms,
		 float *unused,
		 int projection,
		 int opaque_status,
		 rmonsterobj *r,
		 int hither,
		 int yon)
{
    float xscale,xd,yscale,yd,zscale;
    float xview,yview,zview;
    float epsilon = 0.001;
    float xscreen,yscreen,zscreen;
    int use_global_obj_color;
    float obj_rgba[4],color_v1[4],color_v2[4];
    int i,j;
    Poly lwork_poly;
    int px1,py1,px2,py2;
    float x;
    float *color_ptr;
    surface_colors_rgba *tc,blend;
    void (**shaderfuncs)();
    int in_use_shader;
    int has_normals;
    vertex_3d p1,p2,pt,wnormal;
    double dmag;
    int has_alpha;
    int offset;
    int ncolors;
    int status;
    float zmin;
    int color_layout_type,normal_layout_type,index;
    void (*colorfunc)();
    surface_colors_rgba *work_colors,*work_colors2;
    int work_nverts;
    vertex_3d *work_verts,*work_normals;

    get_zmin(&zmin);
    has_alpha = r->has_alpha;

    if (norms == NULL)
	has_normals = 0;
    else
    {
	has_normals = 1;
	if (p->nverts == p->nthings)
	    normal_layout_type = KPER_VERTEX;
	else
	    normal_layout_type = KPER_FACE;
    }
    
    in_use_shader = r->shader;
    
    if (opaque_status == TRUE)
	shaderfuncs = opaque_shaderfuncs;
    else
	shaderfuncs = transparent_shaderfuncs;

    work_window.x0 = work_window.y0 = 0;
    work_window.x1 = xs-1;
    work_window.y1 = ys-1;
    
    /** see Jim Blinn's Corner, IEEE CG&A, mmm/yyy **/
    xscale = (xs - epsilon)/2.0;
    xd = xscale;
    yscale = (ys - epsilon)/2.0;  /* the aspect ratio term goes here. */
    yd = yscale;
    zscale = 1.;

    lwork_poly.n = 3;
    lwork_poly.mask = POLY_MASK(sx) | POLY_MASK(sy) | POLY_MASK(sz);

    if (in_use_shader == KSHADER_GOUROUD)
    {
	if (opaque_status == TRUE)
	    lwork_poly.mask |= POLY_MASK(r) | POLY_MASK(g) | POLY_MASK(b);
	else
	    lwork_poly.mask |= POLY_MASK(r) | POLY_MASK(g) | POLY_MASK(b) | POLY_MASK(a);
    }

    if (p->ncolors == 0)
	colorfunc = polytri_con_no_color;
    else if (has_alpha)
	colorfunc = polytri_con_alpha_per_vertex;
    else
	colorfunc = polytri_con_noalpha_per_vertex;

    /* the following chunk of code the computes local layout type
       needs to be cleaned up. */
    
    if ((p->ncolors == 0) || (p->ncolors == p->nverts))
    {
	if (p->ncolors != p->nverts)
	{
	    /* number of normals isn't the same as number of verts. */
	    color_layout_type = KPER_FACE;
	    ncolors = p->nverts-2;
	}
	else
	{
	    color_layout_type = KPER_VERTEX;
	    ncolors = p->nverts;
	}
    }
    else
    {
	color_layout_type = KPER_FACE;
	ncolors = p->nverts - 2;
    }
    
    /**
      * deal with shading the vertices.
    **/
    
    work_colors = (surface_colors_rgba *)kmalloc(sizeof(surface_colors_rgba)*
						 ncolors);
    work_colors2 = NULL;

    /* set up the work colors array */
    
    for (i=0;i<ncolors;i++)
	(*colorfunc)(p,r,i,work_colors+i);

    /**
      * now, choose the shader, either per-vertex or per-facet,
      * based upon the presence
    **/
    /* compute the per-vertex shade */
    if (color_layout_type == KPER_VERTEX)
    {
	if (normal_layout_type == KPER_VERTEX)
	    /* per vertex colors and normals */
	    vertex_shader(obj_space_verts,norms,ncolors,
			  work_colors,&work_colors2,&(r->attribs),
			  &(r->colors.a),
			  r->shader);
	
	else			/* normal_layout_type == KPER_FACE */
	    /* not ready for april 95 release.  at present, these are
	     converted to disjoint triangles.  */
	    
	    /* per vertex colors, per face normals */
	    vertex_per_face_normal_shader(obj_space_verts,norms,ncolors,
					  work_colors,&work_colors2,
					  &(r->attribs),
					  &(r->colors.a),
					  r->shader);
    }
    else
    {
	/* per face colors and normals */
	tstrip_facet_shader(obj_space_verts,norms,p->nverts,
			    work_colors,&work_colors2,&(r->attribs),
			    &(r->colors.a),
			    r->shader);
    }
    
    for (i=0;i<p->nverts;i++)
    {
	xview = view_space_verts[i].v[0];
	yview = view_space_verts[i].v[1];
	zview = view_space_verts[i].v[2];

	if (projection == KPROJECTION_PERSPECTIVE)
	{
	    xview = xview/zview;
	    yview = yview/zview;
	}
	    
	xscreen = xview*xscale + xd;
	yscreen = yview*yscale + yd;
	zscreen = zview;

	if ((i == 0) || (i == 1))
        {
	    lwork_poly.vert[i].sx = xscreen;
	    lwork_poly.vert[i].sy = yscreen;
	    lwork_poly.vert[i].sz = zscreen;
#if 0
	    lwork_poly.vert[i].x = obj_space_verts[i].v[0];
	    lwork_poly.vert[i].y = obj_space_verts[i].v[1];
	    lwork_poly.vert[i].z = obj_space_verts[i].v[2];
#endif
	    lwork_poly.vert[i].x = xview;
	    lwork_poly.vert[i].y = yview;
	    lwork_poly.vert[i].z = zview;
	    
	    /* when per-face normals are present, the number of colors
	     changes */
	    
	    if (normal_layout_type == KPER_VERTEX)
	    {
		if (color_layout_type == KPER_VERTEX)
		    index = i;
		else
		    index = 0;

		lwork_poly.vert[i].r = work_colors2[index].r;
		lwork_poly.vert[i].g = work_colors2[index].g;
		lwork_poly.vert[i].b = work_colors2[index].b;
		lwork_poly.vert[i].a = work_colors2[index].a;
	    }
	    else		/* normal_layout_type == KPER_FACE */
	    {
		/* assume per-vertex colors.  the order of stuff in the
		 colors array is determined by a routine in shader.c.  we
		 make assumptions about what this order is here.  */

		lwork_poly.vert[i].r = work_colors2[i].r;
		lwork_poly.vert[i].g = work_colors2[i].g;
		lwork_poly.vert[i].b = work_colors2[i].b;
		lwork_poly.vert[i].a = work_colors2[i].a;

	    }
	}
	else
	{
	    lwork_poly.vert[2].sx = xscreen;
	    lwork_poly.vert[2].sy = yscreen;
	    lwork_poly.vert[2].sz = zscreen;
#if 0
	    lwork_poly.vert[2].x = obj_space_verts[i].v[0];
	    lwork_poly.vert[2].y = obj_space_verts[i].v[1];
	    lwork_poly.vert[2].z = obj_space_verts[i].v[2];
#endif
	    lwork_poly.vert[2].x = xview;
	    lwork_poly.vert[2].y = yview;
	    lwork_poly.vert[2].z = zview;
	    
	    if (normal_layout_type == KPER_FACE)
	    {
		int index2;

		index2 = (i-2) > 0 ? i-2 : 0; /* which face? */
		index2 *= 3;	/* expand to per-vertex colors */

		/* have to load up all three verts worth of color */
		for (j=0;j<3;j++)
		{
		    lwork_poly.vert[j].r = work_colors2[index2+j].r;
		    lwork_poly.vert[j].g = work_colors2[index2+j].g;
		    lwork_poly.vert[j].b = work_colors2[index2+j].b;
		    lwork_poly.vert[j].a = work_colors2[index2+j].a;
		}
	    }
	    else		/* normals layout KPER_VERTEX */
	    {
		if (color_layout_type == KPER_VERTEX)
		{
		    index = i;
		    lwork_poly.vert[2].r = work_colors2[index].r;
		    lwork_poly.vert[2].g = work_colors2[index].g;
		    lwork_poly.vert[2].b = work_colors2[index].b;
		    lwork_poly.vert[2].a = work_colors2[index].a;
		}
		else
		{
		    index = i-2;
		    for (j=0;j<3;j++)
		    {
			lwork_poly.vert[j].r = work_colors2[index].r;
			lwork_poly.vert[j].g = work_colors2[index].g;
			lwork_poly.vert[j].b = work_colors2[index].b;
			lwork_poly.vert[j].a = work_colors2[index].a;
		    }
		}
	    }
	}
	if (has_normals)   /* deal with normals */	
        {
	    if (normal_layout_type == KPER_VERTEX)
	    {
		index = i;
		if ((i==0) || (i==1))
		{
		    lwork_poly.vert[i].nx = norms[index].v[0];
		    lwork_poly.vert[i].ny = norms[index].v[1];
		    lwork_poly.vert[i].nz = norms[index].v[2];
		}
		else
		{
		    lwork_poly.vert[2].nx = norms[index].v[0];
		    lwork_poly.vert[2].ny = norms[index].v[1];
		    lwork_poly.vert[2].nz = norms[index].v[2];
		}
	    }
	    else
	    {
		index = (i-2) > 0 ? (i-2) : 0; 
		for (j=0;j<3;j++)
		{
		    lwork_poly.vert[j].nx = norms[index].v[0];
		    lwork_poly.vert[j].ny = norms[index].v[1];
		    lwork_poly.vert[j].nz = norms[index].v[2];
		}
	    }

	}
	/* else, you're hosed because these should have been
	 computed when the object was read in, and not deferred til now. */
	
	if (i >= 2)  /* do final stuff, then scan convert the triangle. */
	{
	    if (!has_normals)
	    {
		kerror("","RMONSTER"," we should not be computing normals on the fly.  Fix this!!");
		/* this stuff should be moved to object reading time */
		p1.v[0] = lwork_poly.vert[1].x - lwork_poly.vert[0].x;
		p1.v[1] = lwork_poly.vert[1].y - lwork_poly.vert[0].y;
		p1.v[2] = lwork_poly.vert[1].z - lwork_poly.vert[0].z;
		
		p2.v[0] = lwork_poly.vert[2].x - lwork_poly.vert[1].x;
		p2.v[1] = lwork_poly.vert[2].y - lwork_poly.vert[1].y;
		p2.v[2] = lwork_poly.vert[2].z - lwork_poly.vert[1].z;

		vertex_cross(&p1,&p2,&wnormal);
		vertex_unit(&wnormal,&dmag);

		if (i & 1) /* flip every other normal */
		{
		    for (j=0;j<3;j++)
			wnormal.v[j] *= -1.;
		}

		for (j=0;j<3;j++)
		{
		    lwork_poly.vert[j].nx = wnormal.v[0];
		    lwork_poly.vert[j].ny = wnormal.v[1];
		    lwork_poly.vert[j].nz = wnormal.v[2];
		}
	    }
	    
	    (*init_shaderfuncs[in_use_shader])(&(r->attribs),&(r->colors),
					       NULL,NULL,&lwork_poly);

	    status = (*p->clipfunc)(&lwork_poly,projection,&zmin,hither,yon);
	    if (status == CHILL)
		poly_scan(&lwork_poly,&work_window,shaderfuncs[in_use_shader]);

	    memcpy((char *)&(lwork_poly.vert[0]),(char *)&(lwork_poly.vert[1]),sizeof(Poly_vert));
	    memcpy((char *)&(lwork_poly.vert[1]),(char *)&(lwork_poly.vert[2]),sizeof(Poly_vert));
	}
    }
}

/**
  * routines for handling color in connected polytri prims.
**/
static void
polytri_con_no_color (rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    *rgba++ = r->colors.r;
    *rgba++ = r->colors.g;
    *rgba++ = r->colors.b;
    *rgba = r->colors.a;
}

static void
polytri_con_alpha_per_vertex(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    register float *f;
    f = (float *)&(p->rgba_color_list[n]);
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba = *f;
}

static void
polytri_con_alpha_per_face(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    register float *f;
    int index;

    if (n >= 2)
	index = n-2;
    else
	index = 0;
    f = (float *)(p->rgba_color_list+index);
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba = *f;
}

static void
polytri_con_noalpha_per_vertex(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    register float *f;
    f = (float *)(p->rgb_color_list+n);
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f;
    *rgba = 1.;
}

static void
polytri_con_noalpha_per_face(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    register float *f;
    int index;

    if (n >= 2)
	index = n-2;
    else
	index = n;
    f = (float *)(p->rgb_color_list+index);
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f;
    *rgba = 1.;
}





