 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>           Routines for Reading Images / Maps
   >>>>
   >>>>  Private:
   >>>>         spc_set_image_and_maps
   >>>>   Static:
   >>>>         read_normal_map
   >>>>         create_count_column
   >>>>         create_default_codebook
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "spectrum.h"

static int  read_codebook           PROTO((void));
static int  create_default_codebook PROTO((void));
static int *create_count_column     PROTO((void));


/*-----------------------------------------------------------
|
|  Routine Name: spc_set_image_and_maps
|       Purpose: sets the global image (spc_image)
|                and the global map data (spc_maps)
|         Input: filename - name of the image file
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: May 12, 1993
| Modifications:
|
---------------------------------------------------------------*/

int spc_set_image_and_maps(
    char *filename)
{
	int i, ok = FALSE;
	int maps_present;
	
	/*
	 * if there was an image up before, need to free associated memory
	if (spc_image != NULL)
	{
            kpds_close_object(spc_image);
	    karray_free(spc_maps, spc_map_rownum, NULL);
	    kfree(spc_maps);
	    kfree(spc_count);
	    kfree(spc_class);
	}
	 */

	spc_image = kpds_open_object(filename, KOBJ_READ);
	if (spc_image == NULL)
	{
	    kerror(NULL, "spc_set_image_and_maps",
		   "Can't get data from input image '%s'", filename);
	    kinfo(KHOSTILE, "Blame PDS or yourself, not me");
	    return(FALSE);
	}
	kpds_get_attribute(spc_image, KPDS_VALUE_DATA_TYPE, &spc_image_type);

	/*
	 *  if it has codebook data, read in the codebook data.
         *  otherwise, create default ramp maps for minimum functionality.
	 */
	maps_present = kpds_query_map(spc_image);

	/*
	 *  read in codebook info, or create default codebook info
         *  if there is none provided with the input image.
	 */
	if (maps_present)
	    ok = read_codebook();

	if ((!ok) && (spc_image != NULL))
	    create_default_codebook();

	if (spc_maps == NULL)
	    exit(1);

/*
kprintf("--- Count Column ---\n");
for (i = 0; i < spc_map_rownum; i++)
{
    if (spc_count[i] > 0)
        kprintf("%d: %d\n", i, spc_count[i]);
}
kprintf("\n\n");
*/

	/*
	 * copy red map column
	 */
	kfree(spc_display->red_col);
	spc_display->red_col = (double *) 
				kmalloc(spc_map_colsize * sizeof(double));

	for (i = 0; i < spc_map_colsize; i++)
	    spc_display->red_col[i] = spc_maps[i][0];

	/*
	 * copy green map column
	 */
	kfree(spc_display->green_col);
	spc_display->green_col = (double *) 
				  kmalloc(spc_map_colsize * sizeof(double));
	for (i = 0; i < spc_map_colsize; i++)
            spc_display->green_col[i] = spc_maps[i][1];

	/*
         * copy blue map column
         */
	kfree(spc_display->blue_col);
	spc_display->blue_col = (double *) 
				kmalloc(spc_map_colsize * sizeof(double));
	for (i = 0; i < spc_map_colsize; i++)
	    spc_display->blue_col[i] = spc_maps[i][2];

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: read_codebook
|       Purpose: sets the global image (spc_image)
|                and the global map data (spc_maps)
|         Input: 
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: May 12, 1993
| Modifications:
|
---------------------------------------------------------------*/

static int read_codebook (void)
{
	int      i, j, num;
	int      rownum, colnum, map_location;
        kaddr    mapdata;
	unsigned int ispare1, ispare2;
	double   *covar_data = NULL;

	/* 
	 * read in the double map data 
	 */
	kpds_set_attribute(spc_image, KPDS_MAP_DATA_TYPE, KDOUBLE);

	/* 
	 * get number of rows in maps 
	 */
	kpds_get_attribute(spc_image, KPDS_MAP_SIZE, 
				      &colnum, &rownum, NULL, NULL, NULL);
	spc_map_rownum = (int) rownum;
	spc_map_colsize= (int) rownum;

	/*
	 * each column will be of the same size
	 */
	kpds_set_attribute(spc_image, KPDS_MAP_REGION_SIZE, 
				      1, spc_map_colsize, 1, 1, 1);
	
	/* 
	 * K1 VIFF has map contents = ispare1, column number = ispare2 
         * K2 VIFF has map contents = "colorMapContents" attribute,
         *            column number = "colorMapColnum" attribute
	 */
	if (!(kpds_get_attribute(spc_image, "colorMapContents", 
                                 &spc_map_contents)))
	{
	    if (kpds_get_attribute(spc_image, "ispare1", &ispare1))
               spc_map_contents = (int) ispare1;
	    else spc_map_contents = 0;

	}
	if (!(kpds_get_attribute(spc_image, "colorMapColnum", 
	  		         &spc_map_colnum)))
	{
		
	    if (kpds_get_attribute(spc_image, "ispare2", &ispare2)) 
	    {
                spc_map_colnum   = (int) ispare2;
                spc_map_rowsize  = (int) ispare2;
	    }
	    else spc_map_colnum = 0;
	}

	/*
	 *  this was probably an image with forced maps;  it was
         *  not created with or for spectrum. get column number 
         *  directly from the map size obtained earlier.
	 */
	if ((spc_map_contents == 0) && (spc_map_colnum == 0))
	{
	    spc_map_colnum  = (int) colnum;
	    spc_map_rowsize = (int) colnum;
	
	    /* no map columns => have to create a default codebook */
	    if (spc_map_colnum == 0) return(FALSE);
	}
	
	/*
	 *  get the map columns of feature data & store in the 2D spc_maps 
	 *  array, so that it can be referenced as spc_maps[cluster][map_col]
	 */
	kpds_set_attribute(spc_image, KPDS_MAP_POSITION, 0, 0, 0, 0, 0);
	for (i = 0, num = 0; i < spc_map_rownum; i++)
	{
	   mapdata = kpds_get_data(spc_image, KPDS_MAP_LINE, NULL);
	   spc_maps = (double **) karray_add((char **) spc_maps, mapdata, num++);
	}

/*
kprintf("\n\n");
kprintf("--- Codebook data ----\n");
for (i = 0; i < spc_map_rownum; i++)
{
    for (j = 0; j < spc_map_colnum; j++)
	kprintf(" %10g", spc_maps[i][j]);
    kprintf("\n");
}
kprintf("\n\n");
*/

	/*
	 *  following feature data, read in count column (if there is one)
	 */
	kpds_set_attribute(spc_image, KPDS_MAP_POSITION, 
				      spc_map_colnum, 0, 0, 0, 0);
	map_location = spc_map_colnum;

	if (spc_map_contents & SPC_COUNT)
	{
	    map_location++;
	    kpds_set_attribute(spc_image, KPDS_MAP_DATA_TYPE, KINT);
	    kpds_set_attribute(spc_image, KPDS_MAP_INTERPOLATE, KNONE);
	    spc_count =  kpds_get_data(spc_image, KPDS_MAP_REGION, NULL);
	    if (spc_count == NULL)
	    {
		kwarn(NULL, "read_codebook",
		      "The spc_map_contents attribute for your input image (if it's a  VIFF image) or the ispare1 value (if it's an xvimage) contains a mask which indicates the presence of a count column.  However, there is no count column stored in the maps of the image.  Somehow, your count column seems to have been lost from your map data.  Creating a count column for the image from scratch at this time.");
		spc_count = create_count_column();
	    }
		
	}
	else spc_count = create_count_column();
	
	/*
	 *  next, read in class array (if there is one)
	 */
	if (spc_map_contents & SPC_CLASS)
	{
	    map_location++;
	    /* class array is integer, following count column */
	    kpds_set_attribute(spc_image, KPDS_MAP_DATA_TYPE, KINT);
	    kpds_set_attribute(spc_image, KPDS_MAP_INTERPOLATE, KNONE);
	    spc_classes =  kpds_get_data(spc_image, KPDS_MAP_REGION, NULL);
	    if (spc_classes == NULL)
	    {
		kwarn(NULL, "read_codebook",
		      "The spc_map_contents attribute for your input image (if it's a  VIFF image) or the ispare1 value (if it's an xvimage) contains a mask which indicates the presence of a class column.  However, there is no class column stored in the maps of the image.  Somehow, your class column seems to have been lost from your map data.  This means that any classes previously associated with the image have been lost. Starting up as though there were never any classes associated with the image.");

	        spc_classes = (int *) kmalloc(spc_map_rownum * sizeof(int));
	        for (i = 0; i < spc_map_rownum; i++)
	            spc_classes[i] = -1;
	    }
	}
	else 
	{
	    spc_classes = (int *) kmalloc(spc_map_rownum * sizeof(int));
	    for (i = 0; i < spc_map_rownum; i++)
	        spc_classes[i] = -1;
	}

/*
if (spc_classes != NULL)
{
    kprintf("--- Class Column ---\n");
    for (i = 0; i < spc_map_rownum; i++)
    if (spc_classes[i] != -1)
        kprintf("spc_classes[%d] = %d\n", i, spc_classes[i]);
    kprintf("\n\n");
}
*/

	/*
	 *  either a covariance matrix or a covariance diagonal may follow;
	 *  these are mutually exclusive
	 */
	spc_matrix_size = 0;
	spc_diag_size   = 0;
	if (spc_map_contents & SPC_COVAR_MATRIX & SPC_COVAR_DIAG)
	{
	    kerror(NULL, "read_normal_map",
	           "Input Images are allowed to have a covariance matrix OR a codiagonal, NOT BOTH.");
	    kinfo(KHOSTILE, "You obviously don't know how to use this application");
	    return(FALSE);
	}
	
	/*
	 *  read in covariance matrix (if applicable)
	 */
	else if (spc_map_contents & SPC_COVAR_MATRIX)
	{
	    kpds_set_attribute(spc_image, KPDS_MAP_INTERPOLATE, KNONE);
	    kpds_set_attribute(spc_image, KPDS_MAP_DATA_TYPE, KDOUBLE);
	    spc_matrix_size = ((spc_map_colnum * (spc_map_colnum+1)) / 2) ;
	    kpds_set_attribute(spc_image, KPDS_MAP_REGION_SIZE, 
				spc_matrix_size, 1, 1, 1, 1);

	    spc_covar_matrix = (double ***) kmalloc (spc_map_rownum *
	                                             sizeof(double **));
	    if ( spc_covar_matrix == NULL )
	    {
	    	kerror(NULL, "read_normal_map",
		       "Unable to allocate storage for covariance matrix");
	        kinfo(KHOSTILE, "Buy a new memory board.  Then i'll tell you where to put it.");
		return(FALSE);
	    }
	
	    for ( i = 0; i < spc_map_rownum; i++ )
	    {
	
		kpds_set_attribute(spc_image, KPDS_MAP_POSITION, 
				   map_location, i, 0, 0, 0);
		covar_data = (double *) kpds_get_data(spc_image, 
					      KPDS_MAP_REGION, covar_data);
		if (covar_data == NULL)
		{
		    kwarn(NULL, "read_codebook",
		         "The spc_map_contents attribute for your input image (if it's a  VIFF image) or the ispare1 value (if it's an xvimage) contains a mask which indicates the presence of a covariance matrix.  However, there is no covariance matrix stored in the maps of the image.  Somehow, your covariance matrix seems to have been lost from your map data.  This means that any output images created using spectrum will NOT have a covariance matrix output.");
		   spc_map_contents = spc_map_contents & (~SPC_COVAR_MATRIX);
		   break;
		}
	        spc_covar_matrix[i] = SM_get_matrix (spc_map_colnum);
                for(j = 0; j < spc_matrix_size; j++) 
                    spc_covar_matrix[i][0][j] = covar_data [j];
	    }
	    kfree(covar_data);

/*
if (spc_covar_matrix != NULL)
{
    kfprintf(kstderr, "\n\n");
    kfprintf(kstderr, "--- Codebook Covariance matrices (as vectors) ----\n");
    kfprintf(kstderr, "--- matrix is %dx%d\n", spc_map_colnum, spc_map_colnum );
    for (i = 0; i < spc_map_rownum; i++)
    {
      for(j = 0; j < spc_map_colnum; j++ ) 
      {
        int k;
        for(k = 0; k < spc_map_colnum; k++ ) 
        {
          if (k >= j)
            kfprintf(kstderr, "%9.5f ",spc_covar_matrix[i][j][k]);
          else kfprintf(kstderr, "          ");
        }
        kfprintf(kstderr, "\n");
      }
      kfprintf(kstderr, "------\n");
    }
    kfprintf(kstderr, "\n\n");
}
*/

	}
	
	/*
	 *  read in covariance diagonal (if applicable)
	 */
	else if (spc_map_contents & SPC_COVAR_DIAG)
	{
	    kpds_set_attribute(spc_image, KPDS_MAP_INTERPOLATE, KNONE);
	    kpds_set_attribute(spc_image, KPDS_MAP_DATA_TYPE, KDOUBLE);
	    spc_diag_size = spc_map_colnum;
	    spc_covar_diag = (double **) kmalloc (spc_diag_size * 
						 sizeof(double *));
	    if ( spc_covar_diag == NULL )
	    {
		kerror(NULL, "read_normal_map",
		       "Unable to allocate storage for covariance diagonal");
	        kinfo(KHOSTILE, "Buy a new memory board.  Then i'll tell you where to put it.");
	        return(FALSE);
	    }
	    for (i = 0; i < spc_diag_size; i++)
	    {
		spc_covar_diag[i] = kpds_get_data(spc_image, KPDS_MAP_REGION,
						  NULL);
		if (spc_covar_diag[i] == NULL)
		{
		    kwarn(NULL, "read_codebook",
		         "The spc_map_contents attribute for your input image (if it's a  VIFF image) or the ispare1 value (if it's an xvimage) contains a mask which indicates the presence of a covariance diagonal.  However, there is no covariance diagonal stored in the maps of the image.  Somehow, your covariance diagonal seems to have been lost from your map data.  This means that any output images created using spectrum will NOT have a covariance diagonal output.");
		   spc_map_contents = spc_map_contents & (~SPC_COVAR_DIAG);
		}
	    }
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: create_count_column
|       Purpose: Creates a count column for the image if there was
|                not one provided.  This is done by counting the occurances
|                of each cluster in the image data, and entering that number
|                at the location indexed by the cluster number.
|
|         Input: none
|        Output: none
|       Returns: The count column for the image.
|    Written By: Danielle Argiro & Mark Young
|          Date: June 23, 1994
| Modifications:
|
-----------------------------------------------------------*/
static int *create_count_column(void)
{
	int   *count_col;
	int   *data = NULL;
	int   i, j, w, h, d, t, e, type, num_regions, num_points;
	
	
	kpds_get_attributes(spc_image, 
	    KPDS_VALUE_DATA_TYPE, &type,
	    KPDS_VALUE_OPTIMAL_REGION_SIZE,  &w, &h, &d, &t, &e, &num_regions,
	    NULL);

	count_col = (int *) kcalloc(1, spc_map_rownum*sizeof(int));

	/*
	 * for the purposes of the count column, we need integers.
	 * however, we might be getting in a float, double, or complex
         * image with lord-knows-what values.  therefore, normalize temporarily,
         * just long enough to create the count column.
	 */
        if ((type == KFLOAT)   || (type == KDOUBLE) ||
            (type == KCOMPLEX) || (type == KDCOMPLEX))
        {
           kpds_set_attributes(spc_image,
                KPDS_VALUE_SCALING,         KNORMALIZE,
                KPDS_VALUE_NORM_MIN,        0.0,
                KPDS_VALUE_NORM_MAX,        (double) (spc_map_rownum-1),
                KPDS_VALUE_COMPLEX_CONVERT, KMAGNITUDE,
                NULL);
        }
	kpds_set_attributes(spc_image, 
		KPDS_VALUE_DATA_TYPE, KINT,
		KPDS_VALUE_REGION_SIZE, w, h, d, t, e,
                NULL);

	num_points = w*h*d*t*e;
	for (i = 0; i < num_regions; i++)
	{
	    data = kpds_get_data(spc_image, KPDS_VALUE_REGION, data);
	    for (j = 0; j < num_points; j++)
		count_col[data[j]] += 1;
	}

	kpds_set_attributes(spc_image, 
			    KPDS_VALUE_DATA_TYPE, type,
			    KPDS_VALUE_SCALING,   KNONE,
			    NULL);
	kfree(data);
	    
	return(count_col);
}

/*-----------------------------------------------------------
|
|  Routine Name: create_default_codebook
|       Purpose: given an image with no map columns,
|                creates map data containing three map columns 
|                with incremental ramp values so that spectrum 
|                features can work with their minimum functionality.
|
|         Input: none
|        Output: none
|       Returns: The ramp map on success, FALSE on failure.
|    Written By: Danielle Argiro
|          Date: May 12, 1993
| Modifications:
|
---------------------------------------------------------------*/
#define SPC_RAMP_ROWNUM 256
#define SPC_RAMP_COLNUM 3

static int create_default_codebook(void)
{
        int i, j;

        spc_maps = (double **) kmalloc (SPC_RAMP_ROWNUM * sizeof(double *));
	if (spc_maps == NULL)
	{
	    kerror(NULL, "create_default_codebook",
		   "Unable to allocate room for ramp maps");
	    kinfo(KHOSTILE, "Buy a new memory board.  Then i'll tell you where to put it.");
	    return(FALSE);
	}

        for (i = 0; i < SPC_RAMP_ROWNUM; i++)
        {
            spc_maps[i] =  (double *) kmalloc (SPC_RAMP_COLNUM * 
					       sizeof(double));
            for (j = 0; j < SPC_RAMP_COLNUM; j++)
                spc_maps[i][j] = i;
        }
        spc_map_rownum = spc_map_colsize = SPC_RAMP_ROWNUM;
        spc_map_colnum = spc_map_rowsize = SPC_RAMP_COLNUM;

	kpds_set_attribute(spc_image, KPDS_MAP_SIZE,
			   spc_map_colnum, spc_map_rownum, 1, 1, 1);

        spc_count = create_count_column();

	spc_classes = (int *) kmalloc(spc_map_rownum * sizeof(int));
	for (i = 0; i < spc_map_rownum; i++)
	    spc_classes[i] = -1;

/*
kprintf("\n\n");
kprintf("--- Codebook data ----\n");
for (i = 0; i < spc_map_rownum; i++)
{
    for (j = 0; j < spc_map_colnum; j++)
	kprintf(" %10g", spc_maps[i][j]);
    kprintf("\n");
}
kprintf("\n\n");
*/

        return(TRUE);
}
