 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */


/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>           Display Creation Routines
   >>>>
   >>>>  Private:
   >>>>           spc_create_legend_display()
   >>>>           spc_create_entry_display()
   >>>>           spc_create_zoom_display()
   >>>>           spc_create_curve_display()
   >>>>           spc_create_scatter_display()
   >>>>           spc_create_contents_display()
   >>>>           spc_create_stats_display()
   >>>>           spc_create_mapcolstats_label()
   >>>>           spc_create_classcontents_label()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/

#include "spectrum.h"

/*-----------------------------------------------------------
|
|  Routine Name: spc_create_legend_display
|
|       Purpose: creates the legend workspace, on which the legend
|                info for each cluster will be listed
|
|         Input: None
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: May 12, 1993
| Modifications:
|
------------------------------------------------------------*/
xvobject viewport;

void spc_create_legend_display(void)
{
	xvobject parent;      /* parent of the whole mess                   */
        xvobject label_obj;   /* label for legend display                   */

	parent = gui_info->Legend->legend->legend_workspace;
	xvw_set_attribute(parent, XVW_TACK_EDGE,
                          KMANAGER_TACK_BOTTOM | KMANAGER_TACK_RIGHT);
        xvw_set_attribute(xvw_parent(parent), XVW_TACK_EDGE,
                          KMANAGER_TACK_BOTTOM | KMANAGER_TACK_RIGHT);
	xvw_set_attributes(xvw_parent(xvw_parent(parent)),
                           XVW_MINIMUM_WIDTH,  455,
                           XVW_MINIMUM_HEIGHT, 270,
                           NULL);
	/*
         * create the label object for legend display as a whole
         */
	label_obj = xvw_create_label(parent, "legend_label");
	xvw_set_attributes(label_obj,
		XVW_BORDER_WIDTH,  0,
		XVW_LABEL,         "Current Legend",
	        XVW_LEFT_OF,       NULL,
		XVW_RIGHT_OF,      NULL,
		XVW_BELOW,         NULL,
		NULL);
	/*
	 * create the pseudocolor display w/ RGB scrollbars so that they
         * can set the color for each class
	 */
	legend_pseudo = xvw_create_pseudo(parent, "legend_sbs");
	xvw_set_attributes(legend_pseudo,
		XVW_CHAR_WIDTH,          45.0,   /* width of 45 characters   */
		XVW_CHAR_HEIGHT,         3.0,    /* height of 3 characters   */
                XVW_BORDER_WIDTH,        0,      /* no border necessary      */
		XVW_BELOW,               label_obj, /* below the label       */
		XVW_LEFT_OF,             NULL,   /* centered                 */
		XVW_RIGHT_OF,            NULL,   /* centered                 */
		XVW_PSEUDO_SHOW_PALETTE, FALSE,  /* don't need pallette      */
		XVW_TACK_EDGE,           KMANAGER_TACK_HORIZ,/* grow horiz.   */
		NULL);
	xvw_add_callback(legend_pseudo, XVW_PSEUDO_CALLBACK, 
			 spc_update_legend_color_cb, NULL);

	
        /*
         * create the viewport widget which will
         * provide backplane with scroll bar
         */
	viewport = xvw_create_viewport(parent, "legend_vp");
	xvw_set_attributes(viewport, 
		   XVW_VP_ALLOW_VERT,   TRUE,
		   XVW_VP_FORCE_VERT,   TRUE,
		   XVW_TACK_EDGE,	KMANAGER_TACK_ALL,
		   XVW_BELOW,           legend_pseudo,
		   NULL);

	/*
	 *  associate pseudocolor bars with image object
	 */
	if (spc_image != NULL)
	    xvw_set_attribute(legend_pseudo, XVW_COLOR_COLOROBJ, spc_image);

}

/*-----------------------------------------------------------
|
|  Routine Name: spc_create_entry_display
|       Purpose: creates a single widget set for an entry
|                in the legend, displaying info about the
|                associated class
|
|         Input: count        - id of this entry
|                legend_ptr   - ptr to info about the current entry
|                offset       - xvobject above this one
|
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: May 12, 1993
| Modifications:
|
------------------------------------------------------------*/

void spc_create_entry_display(
    int          count,
    LegendEntry  *legend_ptr,
    xvobject     offset)
{
	char     temp[KLENGTH];
	xvobject classid;

	/*
	 *  create backplane for the legend entry
	 */
        ksprintf(temp, "back%d", count);
	legend_ptr->back = xvw_create_manager(viewport, temp);
	xvw_set_attributes(legend_ptr->back,
			   XVW_CHAR_WIDTH,   45.0, 
			   XVW_CHAR_HEIGHT,  2.0,
			   XVW_BORDER_WIDTH, 1,
                           XVW_BELOW,        offset,
			   NULL);
	/*
         *  create the box which will display current color
         */

	ksprintf(temp, "colorbox_%d", count);
	legend_ptr->colorbox = xvw_create_colorcell(legend_ptr->back, temp);
	xvw_set_attributes(legend_ptr->colorbox,
		           XVW_BELOW,                      NULL,
		           XVW_RIGHT_OF,                   NULL,
                           XVW_WIDTH,                      30,
                           XVW_HEIGHT,                     30,
                           XVW_BORDER_WIDTH,               2,
                           XVW_COLORCELL_SHOWINDEX,        FALSE,
                           XVW_COLORCELL_UPDATE_ONADD,     TRUE,
                           XVW_COLORCELL_RESTORE_ONDELETE, TRUE,
	                   XVW_BACKGROUND,                 white,
			   NULL);

	if (spc_image != NULL)
	    xvw_set_attribute(legend_ptr->colorbox, 
			      XVW_COLOR_COLOROBJ, spc_image);

        /*
         * create the label displaying class ID
         */
	classid = xvw_create_label(legend_ptr->back, temp);
	ksprintf(temp, " Class ID: %d  ", legend_ptr->class);
	xvw_set_attributes(classid, 
			   XVW_BELOW,        NULL,
                           XVW_RIGHT_OF,     legend_ptr->colorbox,
                           XVW_LABEL,        temp,
                           XVW_BORDER_WIDTH, 0,
                           NULL);
        /*
         * create the label displaying RGB values
         */
	ksprintf(temp, "rgb_%d", count);
	legend_ptr->RGBobj = xvw_create_label(legend_ptr->back, temp);
	ksprintf(temp, " RGB values:  0 0 0");

	xvw_set_attributes(legend_ptr->RGBobj,
			   XVW_BELOW,        NULL,
			   XVW_RIGHT_OF,     classid,
			   XVW_LABEL,        temp,
			   XVW_BORDER_WIDTH, 0,
			   NULL);

	/*
         *  create text object to show/set name of category
         */
	ksprintf(temp, "textbutton_%d", count);
        legend_ptr->classnamebutton = xvw_create_button(legend_ptr->back, temp);
	xvw_set_attributes(legend_ptr->classnamebutton,
			   XVW_BELOW,       legend_ptr->RGBobj,
			   XVW_RIGHT_OF,    legend_ptr->colorbox,
		           XVW_LABEL,       legend_ptr->classname,
			   XVW_CHAR_WIDTH,  40.0,
			   XVW_CHAR_HEIGHT, 1.0,
			   NULL);

	if ((spc_image != NULL) && (legend_ptr->clusternum > 0))
	    xvw_set_attributes(legend_ptr->colorbox,
			    XVW_COLOR_COLOROBJ,  spc_image,
			    XVW_COLORCELL_INDEX, legend_ptr->clusters[0],
			    NULL);
        /*
         *  install event handler for color box, 
         *  install callbacks for text widget, list button
         */
        xvw_add_event(legend_ptr->colorbox, ButtonPressMask, 
		      spc_colorbox_cb, legend_ptr);

        xvw_add_callback(legend_ptr->classnamebutton, XVW_BUTTON_SELECT, 
			 spc_update_classname_cb, legend_ptr);

}


/*-----------------------------------------------------------
|
|  Routine Name: spc_create_zoom_display
|
|       Purpose: creates the zoom workspace
|
|         Input: None
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: May 12, 1993
| Modifications:
|
------------------------------------------------------------*/

void spc_create_zoom_display(void)
{
	xvobject parent;

        /*
         * create the zoom window
         */
        parent = gui_info->Zoom->zoom->zoom_workspace;

	xvw_set_attribute(parent, XVW_TACK_EDGE, 
			  KMANAGER_TACK_BOTTOM | KMANAGER_TACK_RIGHT);
	xvw_set_attribute(xvw_parent(parent), XVW_TACK_EDGE, 
                          KMANAGER_TACK_BOTTOM | KMANAGER_TACK_RIGHT);
        xvw_set_attributes(xvw_parent(xvw_parent(parent)),
                           XVW_MINIMUM_WIDTH,  285,
                           XVW_MINIMUM_HEIGHT, 230,
                           NULL);

        zoom_workspace = xvw_create_zoom(parent, "zoom_window");

        xvw_set_attributes(zoom_workspace,
		   XVW_WIDTH,           200,
		   XVW_HEIGHT,          200,
		   XVW_ZOOM_FACTOR,     2.0,
		   XVW_ZOOM_UPDATEMODE, gui_info->Zoom->zoom->zoommode,
		   XVW_TACK_EDGE,       KMANAGER_TACK_ALL,
		   XVW_IMAGE_ROI_SHAPE, KIMAGE_ROI_POLYLINE,
                   NULL);

}


/*-----------------------------------------------------------
|
|  Routine Name: spc_create_curve_display
|
|       Purpose: creates the curve workspace, on which the 
|                plot of column values for each cluster will be displayed
|
|         Input: None
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: May 12, 1993
| Modifications:
|
------------------------------------------------------------*/
void spc_create_curve_display(void)
{
	xvobject button;

	/*
         * allocate curve structure
	 */
	kfree(spc_curve);
	spc_curve = (SpectralCurveStruct *) 
			 kcalloc(1, sizeof(SpectralCurveStruct));

	spc_curve->parent = gui_info->Curve->curve->curve_workspace;

	xvw_set_attribute(spc_curve->parent, XVW_TACK_EDGE,     
			  KMANAGER_TACK_BOTTOM | KMANAGER_TACK_RIGHT);
        xvw_set_attribute(xvw_parent(spc_curve->parent), XVW_TACK_EDGE,
                          KMANAGER_TACK_BOTTOM | KMANAGER_TACK_RIGHT);
	xvw_set_attributes(xvw_parent(xvw_parent(spc_curve->parent)),
			   XVW_MINIMUM_WIDTH,  525,
			   XVW_MINIMUM_HEIGHT, 260,
			   NULL);

	/*
	 * create the color box
	 */
	spc_curve->colorbox = xvw_create_colorcell(spc_curve->parent, 
					           "colorbox");
	xvw_set_attributes(spc_curve->colorbox, 
			   XVW_WIDTH,               30,
                           XVW_HEIGHT,              30,
                           XVW_BORDER_WIDTH,        2,
                           XVW_COLORCELL_SHOWINDEX, FALSE,
                           XVW_SELECTABLE,          FALSE,
                           NULL);

	/*
	 * create the label for listing cluster number and class membership
	 */
	spc_curve->label = xvw_create_label(spc_curve->parent, "label");
	xvw_set_attributes(spc_curve->label, 
                           XVW_CHAR_WIDTH, 30.0,
			   XVW_LABEL,      "    Cluster Number ",
			   XVW_RIGHT_OF,   spc_curve->colorbox,
			   XVW_ABOVE,      spc_curve->colorbox,
			   XVW_BELOW,      spc_curve->colorbox,
			   XVW_SELECTABLE, FALSE,
			   NULL);

	/*
         * create the plot area
	 */
	spc_curve->area = xvw_create_area(spc_curve->parent, "area");
        xvw_set_attributes(spc_curve->area,
                           XVW_BELOW,                   spc_curve->colorbox,
                           XVW_AREA_DISPLAY_TITLE,      FALSE,
                           XVW_AREA_DISPLAY_DATE,       FALSE,
                           XVW_GRAPHICS_VIEWPORT_MIN_X, 0.08,
                           XVW_GRAPHICS_VIEWPORT_MIN_Y, 0.25,
                           XVW_GRAPHICS_VIEWPORT_MAX_X, 0.95,
                           XVW_GRAPHICS_VIEWPORT_MAX_Y, 0.9,
		           XVW_TACK_EDGE,               KMANAGER_TACK_ALL,
                           NULL);

	/*
	 * create the axis
	 */
	spc_curve->axis = xvw_create_axis2d(spc_curve->area, "axis");
        xvw_set_attributes(spc_curve->axis,
                           XVW_AREA_ATTACH,             spc_curve->area, 
                           NULL);

	xvw_get_attributes(spc_curve->axis, 
			   XVW_AXIS2D_AXIS_X, &spc_curve->xaxis,
			   XVW_AXIS2D_AXIS_Y, &spc_curve->yaxis,
			   NULL);

	xvw_set_attributes(spc_curve->xaxis, 
			   XVW_AXIS_LABEL,           "Band",
			   XVW_AXIS_SHOW_BOX,        FALSE,
			   NULL);
	xvw_set_attributes(spc_curve->yaxis, 
			   XVW_AXIS_SHOW_AXIS_LABEL, FALSE,
			   XVW_AXIS_SHOW_BOX,    FALSE,
			   NULL);
	/*
	 * create the plot
	 */
	spc_curve->plot = xvw_create_plot2d(spc_curve->area, "plot");
        xvw_set_attributes(spc_curve->plot,
                           XVW_AREA_ATTACH,          spc_curve->area, 
                           XVW_PLOT2D_PLOTTYPE,      KPLOT2D_LINEPLOT,
                           NULL);

	/*
	 *  unmap the "Clear" button, which by default is unneeded because
	 *  we are in single plot, continuous mode.
	 */
	button = xvf_get_xvobject(gui_info->Curve->curve->clear_struct, 
			          XVF_BUTTON_OBJ, TRUE);
        xvw_unmap(button);

}


/*-----------------------------------------------------------
|
|  Routine Name: spc_create_scatter_display
|
|       Purpose: creates the scatterplot workspace, on which the
|                scatterplot of cluster values will be displayed
|
|         Input: None
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Oct 27, 1993
| Modifications:
|
------------------------------------------------------------*/
void spc_create_scatter_display(void)
{
	char temp[KLENGTH];

	/*
	 * allocate structure associated w/ scatter plot
	 */
	spc_scatter = (ScatterPlotStruct *) 
			 kcalloc(1, sizeof(ScatterPlotStruct));

	spc_init_plot_info();
	spc_scatter->parent = gui_info->Scatter->scatter->plot_workspace;
	xvw_set_attribute(spc_scatter->parent, XVW_TACK_EDGE,
                          KMANAGER_TACK_BOTTOM | KMANAGER_TACK_RIGHT);
        xvw_set_attribute(xvw_parent(spc_scatter->parent), XVW_TACK_EDGE,
                          KMANAGER_TACK_BOTTOM | KMANAGER_TACK_RIGHT);
	xvw_set_attributes(xvw_parent(xvw_parent(spc_scatter->parent)),
                           XVW_MINIMUM_WIDTH,  455,
                           XVW_MINIMUM_HEIGHT, 540,
                           NULL);

	/*
	 *  create the label that will display the cluster number
	 */
        spc_scatter->cluster_label = xvw_create_labelstr(spc_scatter->parent,
                                                      "cluster_label");
        ksprintf(temp, "Cluster Number %d, not member of class", 0.0);
        xvw_set_attributes(spc_scatter->cluster_label,
                           XVW_ABOVE,           NULL,
                           XVW_RIGHT_OF,        NULL,
                           XVW_LEFT_OF,         NULL,
                           XVW_LABEL,           temp,
                           NULL);
	/*
	 *  create the label that will display the plot position
	 */
	spc_scatter->position_label = xvw_create_labelstr(spc_scatter->parent, 
						       "position_label");
	ksprintf(temp, "Plot position = (%3g x %3g) ", 0.0, 0.0);
	xvw_set_attributes(spc_scatter->position_label,
			   XVW_ABOVE,           spc_scatter->cluster_label,
			   XVW_RIGHT_OF,        NULL,
			   XVW_LEFT_OF,         NULL,
			   XVW_LABEL,           temp,
			   NULL);

	/*
         * create the plot area
	 */
	spc_scatter->area = xvw_create_area(spc_scatter->parent, "area");
        xvw_set_attributes(spc_scatter->area,
                      XVW_AREA_DISPLAY_TITLE,      FALSE,
                      XVW_AREA_DISPLAY_DATE,       FALSE,
		      XVW_ABOVE,                   spc_scatter->position_label,
                      XVW_GRAPHICS_VIEWPORT_MIN_X, 0.2,
                      XVW_GRAPHICS_VIEWPORT_MIN_Y, 0.2,
                      XVW_GRAPHICS_VIEWPORT_MAX_X, 0.9,
                      XVW_GRAPHICS_VIEWPORT_MAX_Y, 0.9,
		      XVW_TACK_EDGE,               KMANAGER_TACK_ALL,
                           NULL);

	/*
	 * create the axis
	 */
	spc_scatter->axis = xvw_create_axis2d(spc_scatter->area, "axis");
        xvw_set_attributes(spc_scatter->axis,
                           XVW_AREA_ATTACH,             spc_scatter->area, 
                           XVW_GRAPHICS_WCMIN_Y,        spc_scatter->wcymin,
                           XVW_GRAPHICS_WCMAX_Y,        spc_scatter->wcymax,
                           XVW_GRAPHICS_WCMIN_X,        spc_scatter->wcxmin,
                           XVW_GRAPHICS_WCMAX_X,        spc_scatter->wcxmax,
                           NULL);

	xvw_get_attributes(spc_scatter->axis,  
			   XVW_AXIS2D_AXIS_X,    &spc_scatter->xaxis,
			   XVW_AXIS2D_AXIS_Y,    &spc_scatter->yaxis,
			   NULL);

	xvw_set_attributes(spc_scatter->xaxis, 
			   XVW_AXIS_LABEL,       "M0",
			   /* XVW_AXIS_NICE_LABELS, TRUE,  */
			   NULL);
	xvw_set_attributes(spc_scatter->yaxis, 
			   XVW_AXIS_LABEL,       "M1",
			   /* XVW_AXIS_NICE_LABELS, TRUE,  */
			   NULL);

	/*
	 * create a colorcell, but deliberately cause plot area to obscure it.
	 */
	spc_scatter->colorcell = xvw_create_colorcell(spc_scatter->parent, 
						      "colorcell");
        xvw_set_attributes(spc_scatter->colorcell,
                           XVW_BELOW,                      spc_scatter->area,
                           XVW_RIGHT_OF,                   NULL,
                           XVW_ABOVE,                      NULL,
                           XVW_WIDTH,                      30,
                           XVW_HEIGHT,                     30,
                           NULL);

	/*
	 * create the plot
	 */
	spc_scatter->plot = xvw_create_plot2d(spc_scatter->area, "plot");
        xvw_set_attributes(spc_scatter->plot,
                           XVW_PLOT2D_PLOTTYPE,     KPLOT2D_COLORMARKER,
                           XVW_GRAPHICS_MARKERTYPE, KMARKER_DIAMOND,
                           XVW_AREA_ATTACH,         spc_scatter->area, 
                           NULL);
        /*
         * create the marker
         */
        spc_scatter->plot_marker = xvw_create_marker(spc_scatter->area, 
						     "marker");
        xvw_set_attributes(spc_scatter->plot_marker,
			   XVW_MARKER_XPLACEMENT,   spc_scatter->wcymin,
                           XVW_MARKER_YPLACEMENT,   spc_scatter->wcxmin,
                           XVW_GRAPHICS_MARKERTYPE, KMARKER_CROSS,
                           XVW_FOREGROUND_COLOR,    "green",
                           XVW_AREA_ATTACH,         spc_scatter->area,
                           NULL);

	/*
	 *  if we are starting with an image, 
         *  associate plotdata w/ the plot object
	 */
	spc_set_plotdata();

	/*
	 *  add event handlers
         *      1 - to update spectral curve,
         *      2 - to update individual cluster statistics,
         *      3 - to update scatterplot marker
         *      4 - to update image position/cluster number labels under image
         *      5 - to support "painting" addition/deletion of clusters in plot
	 */
	xvw_add_event(spc_scatter->area, PointerMotionMask,
                      spc_update_spectral_curve, NULL);
	xvw_add_event(spc_scatter->area, PointerMotionMask,
                      spc_update_cluster_stats, NULL);
	xvw_add_event(spc_scatter->area, PointerMotionMask,
		      spc_update_plot_position, NULL);
	xvw_add_event(spc_scatter->area, PointerMotionMask,
                      spc_update_image_position, NULL);
	xvw_add_event(spc_scatter->area, ButtonMotionMask,
		      spc_paint_from_plot, NULL);
}


/*-----------------------------------------------------------
|
|  Routine Name: spc_create_stats_display
|
|       Purpose: creates the workspace on the statistics subform, 
|                on which the cluster statistics will be displayed
|         Input: None
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: July 17, 1994
| Modifications:
|
------------------------------------------------------------*/
void spc_create_stats_display(void)
{
	xvobject parent, labelstr;

	parent = gui_info->Stats->stats->stats_wksp;

	labelstr = xvw_create_labelstr(parent, "labelstr1");
	xvw_set_attributes(labelstr, 
			   XVW_LABEL,           "Individual Cluster Statistics",
			   XVW_BELOW,           NULL,
			   XVW_RIGHT_OF,        NULL,
			   XVW_LEFT_OF,         NULL,
			   NULL);

	/* labelstr string to display cluster number */
	spc_info->clusternum_obj = 
		xvw_create_labelstr(parent, "clusternum");
	xvw_set_attributes(spc_info->clusternum_obj, 
			   XVW_BELOW,           labelstr,
			   XVW_RIGHT_OF,        NULL,
			   XVW_LABEL,           "Cluster:     ",
			   XVW_CHAR_WIDTH,      15.0,
			   XVW_FORCE_REDISPLAY, TRUE,
		           NULL);
						       
	/* labelstr string to display assigned class */
	spc_info->assignedclass_obj = 
		xvw_create_labelstr(parent, "assignedclass");
	xvw_set_attributes(spc_info->assignedclass_obj, 
			   XVW_BELOW,           spc_info->clusternum_obj,
			   XVW_RIGHT_OF,        NULL,
			   XVW_LABEL,           "Class:",
			   XVW_CHAR_WIDTH,      15.0,
			   XVW_FORCE_REDISPLAY, TRUE,
		           NULL);

        /* labelstr string to display cluster count */
        spc_info->count_obj =
                xvw_create_labelstr(parent, "count");
        xvw_set_attributes(spc_info->count_obj, 
                           XVW_RIGHT_OF,        NULL,
			   XVW_BELOW,           spc_info->assignedclass_obj,
                           XVW_LABEL,           "Count:",
			   XVW_CHAR_WIDTH,      15.0,
			   XVW_FORCE_REDISPLAY, TRUE,
                           NULL);

        /* labelstr string to display variance 
        spc_info->variance_obj =
                xvw_create_labelstr(parent, "variance");
        xvw_set_attributes(spc_info->variance_obj, 
                           XVW_RIGHT_OF,        spc_info->count_obj,
			   XVW_BELOW,           labelstr,
                           XVW_LABEL,           " ",
			   XVW_CHAR_WIDTH,      15.0,
			   XVW_FORCE_REDISPLAY, TRUE,
                           NULL);
        */

}


/*-----------------------------------------------------------
|
|  Routine Name: spc_create_contents_display
|
|       Purpose: creates the viewport on the class contents workspace, 
|                on which the class contents will be displayed
|         Input: None
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: July 17, 1994
| Modifications:
|
------------------------------------------------------------*/
void spc_create_contents_display(void)
{
	xvobject parent, labelstr;

        /*
         * allocate structure associated w/ info subform
         */
        spc_info = (InfoStruct *) kcalloc(1, sizeof(InfoStruct));

        /*
         * create the viewport widget which will provide backplane with
         * scroll bar for listing the contents of each class
         */

        parent = gui_info->Contents->contents->contents_wksp;
	xvw_set_attribute(parent, XVW_TACK_EDGE,
                          KMANAGER_TACK_BOTTOM | KMANAGER_TACK_RIGHT);
        xvw_set_attribute(xvw_parent(parent), XVW_TACK_EDGE,
                          KMANAGER_TACK_BOTTOM | KMANAGER_TACK_RIGHT);
        xvw_set_attributes(xvw_parent(xvw_parent(parent)),
                           XVW_MINIMUM_WIDTH,  475,
                           XVW_MINIMUM_HEIGHT, 170,
                           NULL);

        labelstr = xvw_create_labelstr(parent, "labelstr2");
        xvw_set_attributes(labelstr,
                           XVW_LABEL,        "Clusters Assigned to Classes",
                           XVW_BELOW,           NULL,
                           XVW_RIGHT_OF,        NULL,
                           XVW_LEFT_OF,         NULL,
                           XVW_FORCE_REDISPLAY, TRUE,
                           NULL);

        spc_info->contents_viewport = xvw_create_viewport(parent, "stats_vp");
        xvw_set_attributes(spc_info->contents_viewport,
                   XVW_VP_ALLOW_VERT,   TRUE,
                   XVW_VP_FORCE_VERT,   TRUE,
                   XVW_TACK_EDGE,       KMANAGER_TACK_ALL,
                   XVW_BELOW,           labelstr,
                   XVW_CHAR_WIDTH,      60.0,
                   XVW_CHAR_HEIGHT,     30.0,
                   NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_create_mapcolstats_label
|
|       Purpose: creates the label string objects in the statistics viewport
|                in which the map column values for a particular cluster
|                will be displayed.
|
|         Input: mapcol - number of map column for which object is being created
|                offset - mapcol stats label above this object (if any)
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: July 17, 1994
| Modifications:
|
------------------------------------------------------------*/
xvobject spc_create_mapcolstats_label(
      int      mapcol,
      xvobject offset)
{
	char temp[KLENGTH];
	xvobject parent = gui_info->Stats->stats->stats_wksp;

	spc_info->mapcols_obj = (xvobject *) krealloc(spc_info->mapcols_obj,
					(mapcol+1)*sizeof(xvobject));
	
	sprintf(temp, "mapcol_statslabelstr%d", mapcol);
	spc_info->mapcols_obj[mapcol] = 
		xvw_create_labelstr(parent, temp);
	xvw_set_attributes(spc_info->mapcols_obj[mapcol],
                           XVW_RIGHT_OF,        NULL,
			   XVW_BELOW,           offset,
			   XVW_LABEL,           spc_mapcol_names[mapcol],
			   XVW_FORCE_REDISPLAY, TRUE,
			   XVW_HORIZ_DIST,      6,
			   NULL);
	return(spc_info->mapcols_obj[mapcol]);
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_destroy_mapcolstats_labels
|
|       Purpose: destroys the label string objects in the statistics viewport
|                in which the map column values for a particular cluster
|                were previously displayed, in preparation for new input.
|
|         Input: None
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Aug 26, 1994
| Modifications:
|
------------------------------------------------------------*/
int spc_destroy_mapcolstats_labels(void)
{
	int i;

	/* nothing to destroy; statistics viewport not created once yet */
	if (spc_info->mapcols_obj == NULL) return(TRUE);

	for (i = spc_map_colnum-1; i >= 0; i--)
	    xvw_destroy(spc_info->mapcols_obj[i]);
	kfree (spc_info->mapcols_obj);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_create_classcontents_label
|
|       Purpose: creates the label string objects in the class contents viewport
|                in which the clusters assigned to a particular class
|                will be displayed.
|
|         Input: legend_entry - legend entry representing class 
|                               for which object is being created
|                offset       - classcontents label above this object (if any)
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: July 17, 1994
| Modifications:
|
------------------------------------------------------------*/
xvobject spc_create_classcontents_label(
      LegendEntry *legend_ptr,
      xvobject    offset)
{
        int     id;
        char    temp[KLENGTH];
	

	id = legend_ptr->class;

        sprintf(temp, "classcontents_labelstr%d", id);
        legend_ptr->contents_obj =
                xvw_create_labelstr(spc_info->contents_viewport, temp);

        xvw_set_attributes(legend_ptr->contents_obj,
                           XVW_RIGHT_OF,        NULL,
                           XVW_BELOW,           offset,
                           XVW_FORCE_REDISPLAY, TRUE,
                           XVW_HORIZ_DIST,      10,
                           XVW_FORCE_REDISPLAY, TRUE,
                           NULL);

	return(legend_ptr->contents_obj);
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_set_zoom_buttonpress
|
|       Purpose: This is the "extra call" callback that is added
|		 to the Zoom subform button so that the zoom window
|                is set to an update mode of button press the first
|                time the Zoom subform is displayed.  Don't want it
|                with an update mode of button press before that,
|                since the user wouldn't be able to select clusters
|                from the main image workspace then.  Don't want to
|                set the update mode to button press after that,
|                as you don't want to change it if the user left
|                the zoom window in Continuous mode (that would be
|                confusing).
|
|         Input: client_data - not used
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: Aug 26, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void spc_set_zoom_buttonpress(
    kaddr client_data)
{
	static int count = 0;
	if (count == 0)
	{
            if (zoom_workspace != NULL)
	    {
                xvw_set_attribute(zoom_workspace, XVW_ZOOM_FACTOR, 
				  KZOOM_UM_BUTTON_PRESS);
		xvf_set_attribute(gui_info->Zoom->zoom->zoommode_struct,
				  XVF_LOGIC_VAL, KZOOM_UM_BUTTON_PRESS);
	    }
	}
	count++;
}
