#include <xvinclude.h>


/*
 *  Building on the technique given in the 03.plot2d_area/ example,
 *  this program demonstrates how multiple plots may be easily created
 *  within a single area.
 * 
 *  Initially, an area object with a plot and a 2d axis inside is created.
 *  The "Add A Plot" button underneath invokes a callback which
 *  creates a new plot object within the area object (the same axis object
 *  works for all the plots together).
 */

static char *colornames[] =
{ "LimeGreen",  "Yellow",  "Cyan", "MediumOrchid", 
  "Magenta",    "Orange",  "Wheat", "Turquoise",
  "Purple",     "Blue",    "Brown", "Red",
  "Violet",     "Grey",    "Tan",   "Pink",
};

#define POINTNUM 512
#define MAXPLOTS 16
xvobject plot[MAXPLOTS];
xvobject area, axis;

int  count = 0;
char *filename = "plot2d:cosine";


static void add_another PROTO((xvobject, kaddr, kaddr));
static void create_plot PROTO((void));

void main(
   int  argc,
   char *argv[],
   char *envp[])
{
	int      i, degrees;
	xvobject button, parent;
	
        /* initialize Khoros program */
        khoros_initialize(argc, argv, envp, "ENVISION");

	/* initialize the xvwidgets lib */
        if (!xvw_initialize(XVW_MENUS_XVFORMS))
        {
	   kerror(NULL, "main", "Cannot open display");
	   kexit(KEXIT_FAILURE);
        }

	/*
	 * create manager to be parent of area 
	 */

	parent = xvw_create_manager(NULL, "parent");

	/* 
	 * create the area object
	 */
        area = xvw_create_area(parent, "area");

	/*
	 * create the axis object
	 */
	axis = xvw_create_axis2d(area, "axis");
        xvw_set_attributes(axis,
                     XVW_AREA_ATTACH, axis,
                     XVW_GRAPHICS_VIEWPORT_MIN_X, 0.2,
                     XVW_GRAPHICS_VIEWPORT_MIN_Y, 0.2,
                     XVW_GRAPHICS_VIEWPORT_MAX_X, 0.9,
                     XVW_GRAPHICS_VIEWPORT_MAX_Y, 0.9,
                     NULL);

	/* create button so the user can create a new plot */
	button = xvw_create_button(parent, "button");
	xvw_set_attributes(button, 
			   XVW_LABEL,   "Add A Plot",
			   XVW_BELOW,    area,
			   XVW_RIGHT_OF, NULL,
			   XVW_LEFT_OF,  NULL,
			   NULL);

	/* add the "add_another" callback to the button */
	xvw_add_callback(button, XVW_BUTTON_SELECT, add_another, area);

	/* display and run */
	xvf_run_form();
}

/*
 *  This is the callback that creates a new area with a plot inside 
 *  when the user clicks on the "Add Another Plot" button.
 */
static void add_another(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
        char     string[KLENGTH];
	xvobject area = (xvobject) client_data;

        if ( count == MAXPLOTS )
            kinfo(KSTANDARD, "No more than %d plots allowed", MAXPLOTS );

        else create_plot();
}

/*
 *  this subroutine creates an area object with a plot inside
 */
static void create_plot(void)
{
	int    i, degrees, ampl;
	char   name[KLENGTH];
	Coord  curve[POINTNUM];

	/* 
	 * create the data points for the plot; each new plot has
	 * a larger amplitude, just to make them differ from each other
	 */
	ampl = count+1;
	for (i = 0, degrees = 9; i < POINTNUM; i++, degrees++)
        {
           curve[i].x = i;
           curve[i].y = (count+1) * ksin((double) kdegrees_radians(degrees));
	   if (count >= MAXPLOTS/2)
	      curve[i].y = -curve[i].y;
        }

	/*
	 * create the plot object; set plot data points, number
         * of data points (mandatory!) and color; also do an area attach
         * to the axis to keep the axis values in line with the plot world
         * coordinate system.
	 */
	ksprintf(name, "Plot %d", count);
        plot[count] = xvw_create_plot2d(area, name);
        xvw_set_attributes(plot[count],
                     XVW_AREA_ATTACH,      axis,
                     XVW_PLOT2D_POINTS,    curve,
                     XVW_PLOT2D_PLOTSIZE,  POINTNUM,
                     XVW_FOREGROUND_COLOR, colornames[count],
                     NULL);
        count++;
}
