#include <xvinclude.h>


/*
 *  Building on the technique given in the 03.plot2d_area/ example,
 *  this program demonstrates how multiple plot areas may be easily created
 *  within a single display.
 * 
 *  First, a layout object is created;  using a layout object as the parent
 *  for the area objects inside will cause each of the area objects to be
 *  laid out neatly without explicit location specification.
 *
 *  Next, an area object with a plot inside is created inside the layout.
 *  Since there is only a single area object initially created, it takes up
 *  the entire layout object.
 *
 *  An "Add Another Plot" button underneath invokes a callback which
 *  creates a new area object within the layout object.
 */

static char *colornames[] =
{ "LimeGreen",  "Yellow",  "Cyan", "MediumOrchid", 
  "Magenta",    "Orange",  "Wheat", "Turquoise"
};

#define MAXPLOTS 8
xvobject area[MAXPLOTS]; 
xvobject plot[MAXPLOTS];
xvobject axis[MAXPLOTS];

int  count = 0;
char *filename = "plot2d:cosine";

static void add_another PROTO((xvobject, kaddr, kaddr));
static void create_area PROTO((xvobject));

void main(
   int  argc,
   char *argv[],
   char *envp[])
{
	int      i, degrees;
	xvobject parent, layout, button;
	
        /* initialize Khoros program */
        khoros_initialize(argc, argv, envp, "ENVISION");

	/* initialize the xvwidgets lib */
        if (!xvw_initialize(XVW_MENUS_XVFORMS))
        {
	   kerror(NULL, "main", "Cannot open display");
	   kexit(KEXIT_FAILURE);
        }

	/*
	 * create manager to be parent of layout & button
	 */
	parent = xvw_create_manager(NULL, "parent");

        /*
         * create the layout object
         */
        layout = xvw_create_layout(parent, "Layout"); 
	xvw_set_attributes(layout,
		    XVW_MINIMUM_WIDTH,             600,
		    XVW_MINIMUM_HEIGHT,	           600,
                    XVW_LAYOUT_BUFFER_SIZE,        10,
                    XVW_LAYOUT_NUMBER_ACROSS,      2,
                    XVW_LAYOUT_AREA_JUSTIFICATION, KLAYOUT_AREA_CENTER,
		    NULL);

	/* call subroutine to create first area object & first plot object */
	create_area(layout);

	/* create button so the user can create a new area/plot pair */
	button = xvw_create_button(parent, "button");
	xvw_set_attributes(button, 
			   XVW_LABEL,   "Add Another Plot",
			   XVW_BELOW,    layout,
			   XVW_RIGHT_OF, NULL,
			   XVW_LEFT_OF,  NULL,
			   NULL);

	/* add the "add_another" callback to the button */
	xvw_add_callback(button, XVW_BUTTON_SELECT, add_another, layout);

	/* display and run */
	xvf_run_form();
}

/*
 *  This is the callback that creates a new area with a plot inside 
 *  when the user clicks on the "Add Another Plot" button.
 */
static void add_another(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
        char     string[KLENGTH];
	xvobject layout = (xvobject) client_data;

        if ( count == MAXPLOTS )
            kinfo(KSTANDARD, "No more than %d plots allowed", MAXPLOTS );

        else create_area(layout);
}

/*
 *  this subroutine creates an area object with a plot inside
 */
static void create_area(
     xvobject layout)
{
	char     name[KLENGTH];
	xvobject xaxis, yaxis;

	/* create the area object; don't want title or date */
	ksprintf(name, "Area %d", count);
        area[count] = xvw_create_area(layout, name);
	xvw_set_attributes(area[count],
		     XVW_AREA_DISPLAY_TITLE, FALSE,
		     XVW_AREA_DISPLAY_DATE,  FALSE,
                     XVW_GRAPHICS_VIEWPORT_MIN_X, 0.2,
                     XVW_GRAPHICS_VIEWPORT_MIN_Y, 0.2,
                     XVW_GRAPHICS_VIEWPORT_MAX_X, 0.9,
                     XVW_GRAPHICS_VIEWPORT_MAX_Y, 0.9,
		     NULL);

	ksprintf(name, "Axis %d", count);
	axis[count] = xvw_create_axis2d(area[count], name);
        xvw_set_attribute(axis[count], XVW_AREA_ATTACH, area[count]);
	xvw_get_attribute(axis[count], XVW_AXIS2D_AXIS_X, &xaxis);
	xvw_get_attribute(axis[count], XVW_AXIS2D_AXIS_Y, &yaxis);
	xvw_set_attribute(xaxis, XVW_AXIS_LABEL, "- x -");
	xvw_set_attribute(yaxis, XVW_AXIS_LABEL, "- y -");

	/* create the plot object; set plot file and color */
	ksprintf(name, "Plot %d", count);
        plot[count] = xvw_create_plot2d(area[count], name);
        xvw_set_attributes(plot[count],
                     XVW_AREA_ATTACH,      area[count], 
                     XVW_PLOT2D_PLOTFILE,  filename,
                     XVW_FOREGROUND_COLOR, colornames[count],
                     NULL);
        count++;
}
