#include <xvinclude.h>


static void print_location_info     PROTO((xvobject, XEvent *, kaddr, int *));
static void imagefile_cb            PROTO((xvobject, kaddr, kaddr));
static void locationfile_cb         PROTO((xvobject, kaddr, kaddr));
static void quit_cb                 PROTO((xvobject, kaddr, kaddr));
static void clear_cb                PROTO((xvobject, kaddr, kaddr));
static int  get_rectangle_locations PROTO((char *, xvobject));

/* 
 * This example displays an image, with 2 inputfile objects underneath,
 * one with which you may enter a new image, and one with which 
 * you may enter a location file containing coordinates where rectangle
 * annotations are to be drawn.  You start it like this:
 *
 * % example {image file}
 *
 * By default, it will display the moon image.
 *
 * When you input a location file, one rectangle is created for each 
 * location specified.  An event handler is installed on the rectangle so 
 * that when you click on the rectangle, the location of the rectangle
 * is displayed.
 * The location file is an ascii file of (x, y) DEVICE coordinates, where the 
 * device coordinates correspond to the locations of the pixels in the image.  
 * It is very simpleminded, and has to be like this:
 *
 * x y 
 * x y
 * x y
 *
 * When you enter a new file, any rectangles currently displayed will
 * be transferred to the new image.  A "Clear" button is provided
 * so that you can get rid of the rectangles if desired. An array of the 
 * rectangle objects is saved so that they can all be destroyed when 
 * the you click on the clear button.
 *
 */

xvobject *rectangles = NULL;
int      rectangle_count = 0;

void main(
   int  argc,
   char *argv[],
   char *envp[])
{
	xvobject parent, imageobj, imagefileobj;
        xvobject locationfileobj, quitobj, clearobj;
	char *filename = "image:moon";

        /* initialize Khoros program */
        khoros_initialize(argc, argv, envp, "ENVISION");

	/*  initialize the xvwidgets lib */
	if (!xvw_initialize(XVW_MENUS_XVFORMS))
	{
	   kerror(NULL, "main", "unable to open display");
	   kexit(KEXIT_FAILURE);
	}
	
	/* let them specify alternate input file if desired */
	if (argc > 1)
	   filename = argv[1];

	/* 
	 * create manager object to be the backplane.
         */
	parent = xvw_create_manager(NULL, "image_annotate");

	/*
	 * put in quit button.  doing it first since i want it on
         * the top.  making it below NULL and left of NULL puts it
         * in the far right corner.  add callback to quit program.
	 */
	quitobj = xvw_create_button(parent, "quit");
	xvw_set_attributes(quitobj, 
			   XVW_BELOW,        NULL,
			   XVW_LEFT_OF,      NULL,
			   XVW_LABEL,       "Quit",
			   NULL);
	xvw_add_callback(quitobj, XVW_BUTTON_SELECT, quit_cb, parent);

	/*
         * create the clear button so the user can make rectangles go away.
         * want this on the top left (right of NULL).  add callback to 
         * clear rectangles.
         */
        clearobj = xvw_create_button(parent, "clear");
        xvw_set_attributes(clearobj,
                           XVW_BELOW,        NULL,
                           XVW_RIGHT_OF,     NULL,
                           XVW_LABEL,       "Clear",
                           NULL);
        xvw_add_callback(clearobj, XVW_BUTTON_SELECT, clear_cb, NULL);

	/* 
         * create the image object under clear button;
         * set image file to be displayed 
         */
	imageobj = xvw_create_image(parent, "image");
	xvw_set_attributes(imageobj, 
			   XVW_IMAGE_IMAGEFILE, filename,
			   XVW_BELOW,           clearobj,
			   NULL);

	/* 
         * create an inputfile object so they can enter a new image.
         * give it a label, and set the contents to the input file.
         * want it to appear below the image;  setting it to be left of
         * NULL and right of NULL makes it centered.  add a callback
         * to input a new filename.
         */
	imagefileobj = xvw_create_inputfile(parent, "inputfile");
        xvw_set_attributes(imagefileobj,
                           XVW_INPUTFILE_LABEL,     "Image File",
                           XVW_INPUTFILE_FILENAME,  filename,
                           XVW_CHAR_WIDTH,          55.0,
                           XVW_BELOW,               imageobj,
                           XVW_RIGHT_OF,            NULL,
                           XVW_LEFT_OF,             NULL,
                           NULL);
	xvw_add_callback(imagefileobj, XVW_INPUTFILE_CALLBACK, 
			 imagefile_cb, imageobj);


	/* 
         * create an inputfile object so they can enter a location file.
         * want it to appear below the image inputfile object.  no
         * filename by default. add callback to input a location file.
         */
	locationfileobj = xvw_create_inputfile(parent, "locationfile");
        xvw_set_attributes(locationfileobj,
                           XVW_INPUTFILE_LABEL,     "Location File",
                           XVW_CHAR_WIDTH,          55.0,
                           XVW_BELOW,               imagefileobj,
                           XVW_RIGHT_OF,            NULL,
                           XVW_LEFT_OF,             NULL,
                           NULL);
	xvw_add_callback(locationfileobj, XVW_INPUTFILE_CALLBACK, 
			 locationfile_cb, imageobj);

	/* display & run; there is no way to exit the program but ^C */
	xvf_run_form();
}

/*
 *  the callback for the imagefileobj input file object gets the filename 
 *  from the inputfile object, and sets it on the image object.
 */
static void imagefile_cb(
    xvobject object,
    kaddr client_data,
    kaddr call_data)
{
        char *filename;
	xvobject imageobj = (xvobject) client_data; 

	xvw_get_attribute(object, XVW_INPUTFILE_FILENAME, &filename);
        xvw_set_attribute(imageobj, XVW_IMAGE_IMAGEFILE, filename);
}

/*
 *  the callback for the locationfileobj input file object reads in the 
 *  new location file, and creates the rectangles.  rectangle objects are
 *  kept in a global array so that we can destroy them in the clear button's
 *  callback.
 */
static void locationfile_cb(
    xvobject object,
    kaddr client_data,
    kaddr call_data)
{
        char *filename;
	xvobject imageobj = (xvobject) client_data; 

	xvw_get_attribute(object, XVW_INPUTFILE_FILENAME, &filename);

	get_rectangle_locations(filename, imageobj);
}

/*
 * the callback for the quit button simply quits application.
 */
static void quit_cb(
    xvobject object,
    kaddr client_data,
    kaddr call_data)
{
	xvobject parent = client_data;

	/* cleanup: destroying the manager will also destroy all the children */
	xvw_destroy(parent);

	/* always use kexit to exit */
	kexit(KEXIT_SUCCESS);
}

/*
 * the callback for the clear button destroys all the rectangles.
 */
static void clear_cb(
    xvobject object,
    kaddr client_data,
    kaddr call_data)
{
	int i;
	
	for (i = 0; i < rectangle_count; i++)
	   xvw_destroy(rectangles[i]);

	rectangle_count = 0;

}

/*
 *  this helper routine opens the location file, reads in the (x,y) device
 *  coordinates, and creates the rectangle objects.  the file reading is 
 *  not robust, it expects x y locations in the first column of the file,
 *  separated by a single space, no extraneous characters. no errorchecking.
 */
static int get_rectangle_locations(
   char     *filename,
   xvobject imageobj)
{
	kfile    *file;
	int      status, x, y;
	char     temp[KLENGTH];

	/* open the file */
	if ((file = kfopen(filename, "r")) == NULL)
        {
           kerror(NULL, "get_rectangle_locations",
                  "Unable to open '%s' to get rectangle locations.", filename);
           return(FALSE);
        }

	while (1)
        {
	   /* on eof, the kgets will fail and we break from routine */
	   if (kfgets(temp, KLENGTH, file) == NULL)
                return(TRUE);

	   /* scan for 2 ints separated by a single space */
	   status = ksscanf(temp, "%d %d", &x, &y);
	   if (status != 2) 
	   {
	      kerror(NULL, "get_rectangle_locations", 
                     "bogus line in location file");
	      return(FALSE);
	   }

	   /* ok, got the device coordinates */
	   rectangle_count++;

	   /* 
	    * create a rectangle object at that location. store it in array
            * so we can destroy it later. Note the XVW_XPOSITION and 
            * XVW_YPOSITION are used because we are dealing in device 
	    * coordinates, NOT world coordinates (that would have been 
	    * XVW_RECTANGLE_XCORNER, XVW_RECTANGLE_YCORNER, 
	    * XVW_RECTANGLE_XDIAGONAL, and XVW_RECTANGLE_YDIAGONAL)
            * you can make them any way you want, i chose 5 x 5 in red.
	    * add event handler on ButtonPress which will print information.
	    */
	   rectangles = (xvobject *) krealloc(rectangles, rectangle_count *
					      (sizeof(xvobject)));
	   rectangles[rectangle_count-1] = 
			xvw_create_rectangle(imageobj, "rectangle");
           xvw_set_attributes(rectangles[rectangle_count-1],
                           XVW_XPOSITION,           x,
                           XVW_YPOSITION,           y,
                           XVW_WIDTH,               5,
                           XVW_HEIGHT,              5,
			   XVW_FOREGROUND_COLOR, "red",
                           NULL);
	   xvw_add_event(rectangles[rectangle_count-1], ButtonPress, 
			 print_location_info, rectangles[rectangle_count-1]);
        }
}

/*
 *  this is the event handler which is fired when the user clicks on one
 *  of the rectangle objects.  i just display the (x, y) location.
 */
void print_location_info(
    xvobject object,
    XEvent   *event,
    kaddr    client_data,
    int      *dispatch)
{
	int x, y;
	xvw_get_attributes(object, 
			   XVW_XPOSITION, &x,
                           XVW_YPOSITION, &y,
			   NULL);
	kinfo(KSTANDARD, "The rectangle is located at %d, %d", x, y);
}
