 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>         GUI Item Creation Utility Routines 
   >>>>
   >>>>  Private:
   >>>>            create_new_selection
   >>>>            add_item_to_submenu
   >>>>            find_pane_num
   >>>>            compose_gb_name
   >>>>            find_subform_num
   >>>>            compose_sfb_name
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "guise.h"


/*-----------------------------------------------------------
|
|  Routine Name: create_new_selection
|
|       Purpose: creates a new selection based on the UIS line passed in.
|         Input: display_control - master, guidepane, or pane to create 
|                                  selection on.
|                line            - UIS line describing selection
|                name            - name for GUI item
|        Output: 
|       Returns: 
|    Written By: Danielle Argiro
|          Date: Nov 12, 1993
| Modifications:
|
------------------------------------------------------------*/
kselection *create_new_selection(
    kcontrol *display_control,
    char     *line)
{
	kselection *selection;

	/*
	 *  create the new selection in the form tree
	 */
	selection = kvf_create_new_selection(display_control, line);

        /*
         * create the new selection on the GUI
         */
	if ((selection->type != KUIS_MUTEXCL) &&
	    (selection->type != KUIS_MUTINCL) &&
	    (selection->type != KUIS_GROUP))
            xvf_create_selection(selection, display_control->back,
                                 display_control->type);

	/*
	 *  update the Save button to say, 'Save (Needed)'
	 */
	set_save_needed(gui_info->options->create->save_struct, NULL, NULL,
                        gui_info->options->create->save_struct);

	/*
	 *  make sure changes to selection will change 
         *  the Save button to say 'Save (Needed)'
	 */
	xvf_add_gui_callback(selection->back_kformstruct, "xvf_all_attributes",
                             set_save_needed, 
			     (kaddr) gui_info->options->create->save_struct);

	/*
	 *  go ahead and put the selection in "selected" mode
	 */
	if (selection->type != KUIS_STARTSUBMENU)
	{
	    xvw_set_attributes(display_control->back, 
				XVW_SELECT_REPLACE, selection->back,
				XVW_EDIT_MODE_ON,   TRUE,
				NULL);
	    
	}

	return(selection);
}

/*-----------------------------------------------------------
|
|  Routine Name: add_item_to_submenu
|
|       Purpose: Adds a GUI item into the submenu list.
|         Input: submenu     - pointer to submenu selection
|                kformstruct - kformstruct of selection, subform button,
|                              or guide button to be linked in
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Mon Jan 3, 1994
| Modifications:
|
------------------------------------------------------------*/
int add_item_to_submenu(
   kselection   *submenu,
   kform_struct *kformstruct)
{

	kform_struct *tmpstruct;

        if (submenu->submenu_next == NULL)
        {
	    kformstruct->next = NULL;
            submenu->submenu_next = kformstruct;
        }
        else
        {
            tmpstruct = submenu->submenu_next;
            while (tmpstruct->next != NULL)
                tmpstruct = tmpstruct->next;
	    kformstruct->next = NULL;
            tmpstruct->next = kformstruct;
        }
	/* Note: DON'T link the kformstruct - because you don't want
	   it's "next" pointer to be used to point to the global list
	   of kformstructs;  the "next" pointer has to be used to end
	   the list! - it won't get freed, but oh well. */

	switch (kformstruct->type)
	{
	    case KSELECTION:
                 kformstruct->Selptr->back_submenu = submenu;
                 kformstruct->Selptr->back_kformstruct = kformstruct;
		 break;

	    case KSUBFORM:
                 kformstruct->Subformptr->back_submenu = submenu;
                 kformstruct->Subformptr->back_kformstruct = kformstruct;
		 break;

	    case KGUIDE:
                 kformstruct->Guideptr->back_submenu = submenu;
                 kformstruct->Guideptr->back_kformstruct = kformstruct;
		 break;
	}
        return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: find_pane_num
|
|       Purpose: Finds the number of panes on a subform, plus one;
|                for use with numbering newly created guide/pane pairs.
|         Input: subform - subform to which guide button is to be added
|        Output: None
|       Returns: The number of the guide/pane pair to be added
|    Written By: Danielle Argiro
|          Date: Aug 31, 1993
| Modifications:
|
------------------------------------------------------------*/
int find_pane_num(
    ksubform *subform)
{
	int    panenum;
	kguide *guide;

        panenum = 1;
        if (subform->guidepane != NULL)
        {
            guide = subform->guidepane->guide_list;
            while (guide->next != NULL)
            {
                panenum++;
                guide = guide->next;
            }
        }
        panenum++;

	return(panenum);
}



/*-----------------------------------------------------------
|
|  Routine Name: compose_gb_name
|
|       Purpose: Finds an appropriate name for the guide button/pane pair
|         Input: subform - subform to which guide button is to be added
|                panenum - the number of the guide/pane pair to be added
|        Output: None
|       Returns: An allocated variable name for the guide/pane pair
|    Written By: Danielle Argiro
|          Date: Aug 31, 1993
| Modifications:
|
------------------------------------------------------------*/

char *compose_gb_name(
    ksubform *subform,
    int      *panenum)
{
	char      *gb_name, *variable;
	kguide    *guide;
	char      name[KLENGTH];
	int       trial_num; 
	int       found = FALSE;
	int       repeated = FALSE;

	/*
	 *  If this is only the second pane (ie, no guidepane has 
         *  been created yet) then it should be easy - name it "pane2".
         *  If the first pane is named "pane2", then name it "pane1".
	 */
	if (subform->guidepane == NULL)
	{
	    variable = ktoken_to_string(subform->guide->var_token);
	    if (kstrcmp(variable, "pane2") != 0)
		ksprintf(name, "pane2");
	    else ksprintf(name, "pane1");
	}

	/*
	 *  There are already multiple guide/pane pairs
	 */
        else
        {
            /*
	     * The easy choice is just pane%d, where %d is the number
             * of the guide/pane pair.  See if this number is already used.
             */
            ksprintf(name, "pane%d", *panenum);
            guide = subform->guidepane->guide_list;
            while (guide != NULL)
            {
	        variable = ktoken_to_string(guide->pane->var_token);
                if (kstrcmp(variable, name) == 0) 
		{
		    repeated = TRUE;
		    break;
		}
                guide = guide->next;
            }

	    /*
	     * The obvious choice is already taken.  Do it the long way -
             * start with 1 and move forward until there is an unused slot.
	     */
	    if (repeated)
	    {
		trial_num = 0;
		while (!found)
		{
		    repeated = FALSE;
		    trial_num++;
		    ksprintf(name, "pane%d", trial_num);
		    guide = subform->guidepane->guide_list;
            	    while (guide != NULL)
            	    {
	                variable = ktoken_to_string(guide->pane->var_token);
                        if (kstrcmp(variable, name) == 0)
                            repeated = TRUE;
                        guide = guide->next;
                    }
		    if (!repeated) found = TRUE;
		}
	        ksprintf(name, "pane%d", trial_num);
		*panenum = trial_num;
	    }
        }
	gb_name = kstring_copy(name, NULL);
	return(gb_name);
}
    

/*-----------------------------------------------------------
|
|  Routine Name: find_subform_num
|
|       Purpose: Finds the number of subforms on a form, plus one;
|                for use with numbering newly created subformbuton/subform
|                pairs.
|         Input: form - form to which subform is to be added
|        Output: None
|       Returns: The number of the sbf/subform pair to be added
|    Written By: Danielle Argiro
|          Date: Aug 31, 1993
| Modifications:
|
------------------------------------------------------------*/
int find_subform_num(
    kform *form)
{
        int      subformnum;
        ksubform *subform;

        subformnum = 1;
        if (form->master != NULL)
        {
            subform = form->master->subform_list;
            while (subform->next != NULL)
            {
                subformnum++;
                subform = subform->next;
            }
        }
        subformnum++;

        return(subformnum);
}


/*-----------------------------------------------------------
|
|  Routine Name: compose_sfb_name
|
|       Purpose: Finds an appropriate name for the subformbutton/subform pair
|         Input: form       - form to which subform button is to be added
|                subformnum - the number of the sfb/subform pair to be added
|        Output: None
|       Returns: An allocated variable name for the sfb/subform pair
|    Written By: Danielle Argiro
|          Date: Aug 31, 1993
| Modifications:
|
------------------------------------------------------------*/

char *compose_sfb_name(
    kform *form,
    int   *subformnum)
{
        char      *variable, *sfb_name;
        ksubform  *subform;
        char      name[KLENGTH];
        int       trial_num;
        int       found = FALSE;
        int       repeated = FALSE;

        /*
         *  If this is only the second subform (ie, no master has
         *  been created yet) then it should be easy - name it "subform2".
         *  If the first subform is named "subform2", then name it "subform1".
         */
        if (form->master == NULL)
        {
	    variable = ktoken_to_string(form->subform->var_token);
            if (kstrcmp(variable, "subform2") != 0)
                ksprintf(name, "subform2");
            else ksprintf(name, "subform1");
        }

        /*
         *  There are already multiple subformbutton/subform pairs
         */
        else
        {
            /*
             * The easy choice is just subform%d, where %d is the number
             * of the sfb/subform pair.  See if this number is already used.
             */
            ksprintf(name, "subform%d", *subformnum);
            subform = form->master->subform_list;
            while (subform != NULL)
            {
	        variable = ktoken_to_string(subform->var_token);
                if (kstrcmp(variable, name) == 0)
                {
                    repeated = TRUE;
                    break;
                }
                subform = subform->next;
            }

            /*
             * The obvious choice is already taken.  Do it the long way -
             * start with 1 and move forward until there is an unused slot.
             */
            if (repeated)
            {
                trial_num = 0;
                while (!found)
                {
                    repeated = FALSE;
                    trial_num++;
                    ksprintf(name, "subform%d", trial_num);
                    subform = form->master->subform_list;
                    while (subform != NULL)
                    {
	                variable = ktoken_to_string(subform->var_token);
                        if (kstrcmp(variable, name) == 0)
                            repeated = TRUE;
                        subform = subform->next;
                    }
                    if (!repeated) found = TRUE;
                }
                ksprintf(name, "subform%d", trial_num);
                *subformnum = trial_num;
            }
        }
        sfb_name = kstring_copy(name, NULL);
        return(sfb_name);
}

