/* 
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/* 
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Image Display Utility Routines
   >>>>
   >>>>  Private:
   >>>>         delete_object()
   >>>>         edit_object()
   >>>>         select_object()
   >>>>         update_object_list()
   >>>>         create_object()
   >>>>
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "craftsman.h"


static int update_object_attributes PROTO((kobject));

extern AttributeSelection object_attributes[NUM_OBJ_ATTRS];

InfoFile info_files[] =
{
   {KCMS_CMOBJ_TODO,
    FALSE, NULL, NULL, NULL, "todo",
    KCMS_FOBJ_SUBTYPE_TODO, KCMS_TEMPLATE_TODO},

   {KCMS_CMOBJ_BUGS,
    FALSE, NULL, NULL, NULL, "bugs",
    KCMS_FOBJ_SUBTYPE_BUGS, KCMS_TEMPLATE_BUGS},

   {KCMS_CMOBJ_DONE,
    FALSE, NULL, NULL, NULL, "done",
    KCMS_FOBJ_SUBTYPE_DONE, KCMS_TEMPLATE_DONE},

   {KCMS_CMOBJ_CHANGELOG,
    FALSE, NULL, NULL, NULL, "changelog",
    KCMS_FOBJ_SUBTYPE_CHANGELOG, KCMS_TEMPLATE_CHANGELOG}
};

static int type_selections[] =
{
   (KCMS_KROUTINE | KCMS_XVROUTINE | KCMS_LIBRARY | KCMS_PANE | KCMS_SCRIPT),
   KCMS_KROUTINE,
   KCMS_XVROUTINE,
   KCMS_PANE,
   KCMS_LIBRARY,
   KCMS_SCRIPT
};

/*-----------------------------------------------------------
| Routine Name:	delete_object - Delete the current object.
|
| Purpose:	This routine is called to delete the object. 
|		This routine is called when the object action
|		toggle is set to "Delete" and an object is
|		selected from the list of existing objects.
|
| Written By:	Becky Bishop and Neil Bowers
| Date:		23-feb-94
------------------------------------------------------------*/
void
delete_object(void)
{
   kobject  cmobj;
   int      otype;
   kstring  typestring;


   if ((cmobj = kcms_open_cmobj(cmsToolbox, strObject)) == NULL)
   {
      kerror(NULL, "delete_object()",
	     "Failed to open object `%s' in toolbox `%s'.",
	     strObject, strToolbox);
      return;
   }

   if (!kprompt(KFORCE, "Yes", "No", FALSE,
		"Operation: delete software object from toolbox\n\n"
		"\tToolbox : %s\n"
		"\tObject  : %s\n\n"
		"Do you want to continue with removal of object?",
		strToolbox, strObject))
      return;

   if (kcms_get_attribute(cmobj, KCMS_CMOBJ_PROGTYPE, &otype))
      typestring = kcms_attr_int2string(KCMS_CMOBJ_TYPE, otype);
   else
      typestring = "(unknown)";
   kcms_set_bit(cmobj, KCMS_CMOBJ_FLAGS, KCMS_BIT_CMOBJ_FORCE);

   xvw_set_attribute(xvoNotifier, XVW_NOTIFYWINDOW_VISIBLE, TRUE);
   xvw_format_attribute(xvoNotifier, XVW_NOTIFYWINDOW_MESSAGE,
			"Removing %s `%s' from toolbox `%s'",
			typestring, strObject, strToolbox);

   /*-- attempt to destroy the object. --*/
   if (kcms_destroy(cmobj))
   {
      update_categories();

      if (otype == KCMS_LIBRARY)
	 change_lib_list();
      update_object_list();
      select_object(NULL);
   }

   xvw_set_attribute(xvoNotifier, XVW_NOTIFYWINDOW_VISIBLE, FALSE);
}

/*-----------------------------------------------------------
| Routine Name:	edit_object - invoke Composer on currently selected object
|
| Purpose:	This function forks Composer on the currently selected
|		software object.
|
| Written By:	Neil Bowers
| Date:		11-nov-93
------------------------------------------------------------*/
void
edit_object(void)
{
   char  command[KLENGTH];


   (void)ksprintf(command, "composer -tb %s -oname %s",
		  strToolbox, strObject);
   if (kspawn(command) == -1)
      kerror(NULL, "edit_object()", "Failed to spawn composer :-(");
}

/*-----------------------------------------------------------
| Routine Name:	select_object() - set `current' software object
|
| Purpose:	This routine is called to set or unset the current software
|		object in the current toolbox.  This activates and
|		de-activates appropriate parts of the user interface.
|
| Input:	oname	-	name of the selected toolbox
|
| Returns:	TRUE (1) if the specified object was successfully selected,
|		FALSE (0) otherwise.
| Written By:	Neil Bowers
| Date:		Sep 19, 1993
------------------------------------------------------------*/
int
select_object(
   kstring oname)
{
   kobject  object;
   kobject  pane;
   int      active;
   kstring  typestring;
   kstring  pane_path = NULL;
   int      i;
   int      otype;

   active = (oname != NULL);

   /*-- activate action buttons and suform buttons --*/
   xvf_set_attribute(gui_info->open_struct, XVF_ACTIVATE, active);
   xvf_set_attribute(gui_info->delete_struct, XVF_ACTIVATE, active);
   xvf_set_attribute(gui_info->obj_attributes_struct, XVF_ACTIVATE, active);
   xvf_set_attribute(gui_info->copyobject_struct, XVF_ACTIVATE, active);
   xvf_set_attribute(gui_info->mvobject_struct, XVF_ACTIVATE, active);
   xvf_set_attribute(gui_info->objinfo_struct, XVF_ACTIVATE, active);
   xvf_set_attribute(gui_info->obj_attributes_struct, XVF_ACTIVATE, active);
   xvf_set_attribute(gui_info->rename_obj_struct, XVF_ACTIVATE, active);
   xvf_set_attribute(gui_info->klintobj_struct, XVF_ACTIVATE, active);

   if (oname != NULL)
   {
      /*-- highlight selected toolbox in list --*/
      for (i = 0; i < countObjects; i++)
	 if (!kstrcasecmp(oname, object_names[i]))
	 {
	    xvf_set_attribute(gui_info->objlist_struct, XVF_LIST_INDEX, i);
	 }

      if ((object = kcms_open_cmobj(cmsToolbox, oname)) == NULL)
      {
	 kerror(NULL, "Select Object",
		"Unable to open object `%s' in toolbox `%s'.",
		oname, strToolbox);

	 /*-- unselect/unhighlight the object clicked on --------------*/
	 xvf_set_attribute(gui_info->objlist_struct, XVF_LIST_INDEX, -1);
	 return FALSE;
      }
      update_object_attributes(object);
      xvf_set_attribute(paneObjAttrs->oname_struct, XVF_TITLE, oname);
      kcms_get_attribute(object, KCMS_CMOBJ_PROGTYPE, &otype);
      typestring = kcms_attr_int2string(KCMS_CMOBJ_TYPE, otype);
      xvf_set_attribute(paneObjAttrs->otype_struct, XVF_TITLE, typestring);

      /*-- install-in-cantata only applicable if .pane present --------*/
      if (!kcms_get_attribute(object, KCMS_CMOBJ_UIS_PANE, &pane))
	 return FALSE;
      if (pane != NULL && !kcms_get_attribute(pane, KCMS_PATH, &pane_path))
	 return FALSE;

      if (tbstatus == KCMS_STATUS_DEVELOPMENT
	  || tbstatus == KCMS_STATUS_PRODUCTION)
      {
	 if (pane_path == NULL)
	    xvf_unmap_kformstruct(paneObjAttrs->install_in_cantata_struct);
	 else
	    xvf_map_kformstruct(paneObjAttrs->install_in_cantata_struct);
      }

      update_info_files(object);
      update_keywords(object, NULL);

      /*-- generate executable toggle only applies to panes -----------*/
      if (otype == KCMS_PANE)
	 xvf_set_attribute(gui_info->obj_attributes->paneattrs_struct,
			   XVF_ACTIVATE, TRUE);
      else
	 xvf_set_attribute(gui_info->obj_attributes->paneattrs_struct,
			   XVF_ACTIVATE, FALSE);

      kcms_close(object);
   }

   /*-- set the labels on the "Copy Object" subform -------------------*/
   xvf_set_attribute(gui_info->copyobject->copy->oname_struct,
		     XVF_TITLE, oname);
   xvf_set_attribute(gui_info->copyobject->copy->tbname_struct,
		     XVF_TITLE, strToolbox);

   /*-- set the labels on the "Move Object" subform -------------------*/
   xvf_set_attribute(gui_info->mvobject->move->oname_struct, XVF_TITLE, oname);
   xvf_set_attribute(gui_info->mvobject->move->tbname_struct,
		     XVF_TITLE, strToolbox);

   /*-- set the labels on the "Move Object" subform -------------------*/
   xvf_set_attribute(gui_info->rename_obj->rename->oname_struct,
		     XVF_TITLE, oname);
   xvf_set_attribute(gui_info->rename_obj->rename->tbname_struct,
		     XVF_TITLE, strToolbox);

   /*-- set the labels on the "Object Info Files" subform -------------*/
   xvf_set_attribute(gui_info->objinfo->infopane->oname_struct,
		     XVF_TITLE, oname);
   xvf_set_attribute(gui_info->objinfo->infopane->tbname_struct,
		     XVF_TITLE, strToolbox);

   if (strObject != NULL)
      kfree(strObject);
   strObject = kstrdup(oname);
   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	update_object_list() - update list of objects
|
| Purpose:	This function updates the widget list of objects on
|		the master form.
|
| Written By:	Neil Bowers
| Date:		19-sep-93
------------------------------------------------------------*/
void
update_object_list(void)
{
   char     label[KLENGTH];
   klist   *object_list;
   int      selector;
   kstring  typestring;


   if (cmsToolbox == NULL)
   {
      countObjects = 0;
      object_names = NULL;

      xvf_set_attribute(gui_info->objlist_struct,
			XVF_TITLE, "(no toolbox selected)");
   }
   else
   {
      selector = gui_info->preferences->types->objtypes_num - 1;
      typestring = (selector == 0
		    ? "all"
		    : kcms_attr_int2string(KCMS_CMOBJ_TYPE,
					   type_selections[selector]));
      object_list = kcms_get_objects(cmsToolbox, type_selections[selector]);
      object_names = kcms_list_get_attribute(object_list,
					   KCMS_NAME, &countObjects);
      object_names = karray_sort(object_names, countObjects, FALSE);

      ksprintf(label, "%s (%s: %d)", strToolbox, typestring, countObjects);
      xvf_set_attribute(gui_info->objlist_struct, XVF_TITLE, label);
   }

   xvf_set_attributes(gui_info->objlist_struct,
		      XVF_LIST_SIZE, countObjects,
		      XVF_LIST_CONTENTS, object_names,
		      NULL);

   return;
}

/*-----------------------------------------------------------
| Routine Name:	create_object() - Prepare to create a obj
|
| Purpose:	This routine pulls together the various parameters for
|		a new software object, and calls kcms_generate_object().
|
| Input:	otype	-	The type of object to create.
|		bname	-	The binary name for the object.
|		langtype  -	Language type of new object.
|		incantata -	A boolean flag, which specifies whether
|		the new object should be visible in cantata.
|
| Written By:	Neil Bowers
| Date:		13-dec-93
------------------------------------------------------------*/
void
create_object(
   int     otype,
   char *  bname,
   int     langtype,
   int     incantata)
{
   kstring  routine       = "create_object()";
   kobject  new_object;
   kstring  oname         = gui_info->template->obj_name;
   kstring  comp_lib      = NULL;
   kstring  typestring    = NULL;
   int      create_pane   = FALSE;
   int      create_form   = FALSE;
   kstring  author        = NULL;
   kstring  email         = NULL;


   if (cmsToolbox == NULL)
   {
      kerror(NULL, routine, "You have not selected a toolbox.");
      return;
   }

   if (!kcms_legal_identifier(oname, KOBJ_CMSOBJ))
   {
      kerror(NULL, routine, "Invalid object name -- %s%s",
	     "object names must start with a letter, followed ",
	     "by letters, digits or _");
      return;
   }

   create_pane = (otype == KCMS_PANE || otype == KCMS_KROUTINE ||
		  otype == KCMS_XVROUTINE
		  || (otype == KCMS_SCRIPT &&
		      (gui_info->template->script->wantpane_num - 1)));

   /*-- object-type-specific preparation ------------------------------*/
   switch (otype)
   {
      case KCMS_KROUTINE:
	 if (paneKroutine->lib_routine_val == 2)
	 {
	    comp_lib = paneKroutine->lib_list;
	    if (!verify_lib_list(comp_lib))
	       return;
	 }
	 break;

      case KCMS_XVROUTINE:
	 create_form = gui_info->template->xvroutine->useform_val - 1;
	 break;
   }

   if ((author = gui_info->template->author) == NULL)
      kcms_get_attribute(cmsToolbox, KCMS_TB_AUTHOR, &author);
   if ((email = gui_info->template->email) == NULL)
      kcms_get_attribute(cmsToolbox, KCMS_TB_AUTHOR_EMAIL, &email);

   /*--------------------------------------------------------+
   | OK, Now we're about to create the object
   +--------------------------------------------------------*/

   typestring = kcms_attr_int2string(KCMS_CMOBJ_TYPE, otype);

   xvw_format_attribute(xvoNotifier, XVW_NOTIFYWINDOW_MESSAGE,
			"Creating %s `%s' in toolbox `%s'",
			typestring, oname, strToolbox);
   xvw_set_attribute(xvoNotifier, XVW_NOTIFYWINDOW_VISIBLE, TRUE);

   new_object = kcms_generate_cmobj(cmsToolbox,
				    oname, otype, bname,
				    gui_info->template->category,
				    gui_info->template->subcategory,
				    gui_info->template->description,
				    comp_lib, langtype,
				    create_form, create_pane,
				    incantata, NULL);

   if (new_object != NULL)
   {
      kcms_set_attributes(new_object,
			  KCMS_CMOBJ_ICON_NAME,
			     gui_info->template->icon_name,
			  KCMS_CMOBJ_AUTHOR,       author,
			  KCMS_CMOBJ_AUTHOR_EMAIL, email,
			  KCMS_END);

      if (otype == KCMS_PANE)
      {
	 kcms_set_attributes(new_object,
			     KCMS_CMOBJ_GENERATE_EXECUTABLE,
			     gui_info->template->pane->genexec_val - 1,
			     KCMS_CMOBJ_ASSOCIATED_OBJECT,
			     gui_info->template->pane->tbname,
			     gui_info->template->pane->oname,
			     gui_info->template->pane->args,
			     KCMS_END);
      }

      /*-- call kcodegen routine to generate code in object -----------*/
      xvw_set_attribute(xvoNotifier, XVW_NOTIFYWINDOW_LABEL,
			"Generating code.");
      kgen_generate_object(new_object);

      xvw_set_attributes(xvoNotifier,
			 XVW_NOTIFYWINDOW_LABEL, "Please Wait",
			 XVW_NOTIFYWINDOW_MESSAGE, "Saving object",
			 NULL);

      kcms_close(new_object);

      /*-- need to update the library string list for kroutines --*/
      if (otype == KCMS_LIBRARY)
	 change_lib_list();

      update_object_list();
      merge_category_strings(gui_info->template->category,
			     gui_info->template->subcategory);
      update_categories();
      select_object(oname);
   }

   xvw_set_attribute(xvoNotifier, XVW_NOTIFYWINDOW_VISIBLE, FALSE);
}

/*-----------------------------------------------------------
| Routine Name:	update_object_attributes - load object attributes into
|		selections
|
| Purpose:	This function take a KCMS Software Object and updates
|		the object attributes subform using the object.
|
| Input:	object		- The Software Object to fill in the
|		attributes pane from.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		5-dec-93
------------------------------------------------------------*/
static int
update_object_attributes(
   kobject object)
{
   int     i;
   int     installed;
   int     otype;
   int     gen;
   char *  attrval;
   kstring a_tb;
   kstring a_oname;
   kstring a_args;


   for (i = 0; i < NUM_OBJ_ATTRS; i++)
   {
      if (!kcms_get_attribute(object, object_attributes[i].attribute,
			      &attrval))
	 return FALSE;

      xvf_set_attribute(*object_attributes[i].selection,
			XVF_STRING_VAL, attrval);
      kfree(*object_attributes[i].selvalue);
      *object_attributes[i].selvalue = kstrdup(attrval);
   }

   if (!kcms_get_attribute(object, KCMS_CMOBJ_CI_INSTALLED, &installed))
      return FALSE;

   xvf_set_attribute(paneObjAttrs->install_in_cantata_struct,
		     XVF_TOGGLE_NUM, installed ? 2 : 1);

   if (kcms_get_attribute(object, KCMS_CMOBJ_PROGTYPE, &otype)
       && otype == KCMS_PANE)
   {
      if (kcms_get_attribute(object, KCMS_CMOBJ_GENERATE_EXECUTABLE, &gen))
	 xvf_set_attribute(gui_info->obj_attributes->paneattrs->genexec_struct,
			   XVF_TOGGLE_NUM, gen + 1);
      if (kcms_get_attribute(object, KCMS_CMOBJ_ASSOCIATED_OBJECT,
			     &a_tb, &a_oname, &a_args))
      {
	 xvf_set_attribute(gui_info->obj_attributes->paneattrs->tbname_struct,
			   XVF_STRING_VAL, a_tb);
         kfree(gui_info->obj_attributes->paneattrs->tbname);
         gui_info->obj_attributes->paneattrs->tbname = kstrdup(a_tb);

	 xvf_set_attribute(gui_info->obj_attributes->paneattrs->oname_struct,
			   XVF_STRING_VAL, a_oname);
         kfree(gui_info->obj_attributes->paneattrs->oname);
	 gui_info->obj_attributes->paneattrs->oname = kstrdup(a_oname);

	 xvf_set_attribute(gui_info->obj_attributes->paneattrs->args_struct,
			   XVF_STRING_VAL, a_args);
         kfree(gui_info->obj_attributes->paneattrs->args);
         gui_info->obj_attributes->paneattrs->args = kstrdup(a_args);
      }
   }

   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	update_info_files - update info files subform
|
| Purpose:	This function take a KCMS Software Object and updates
|		the object attributes subform using the object.
|
| Input:	object	-	The Software Object to fill in the
|				info files subform from.
|
| Written By:	Neil Bowers
| Date:		7-may-94
------------------------------------------------------------*/
void
update_info_files(
   kobject  object)
{
   int      i;
   kobject  file_object;
   kstring  tmp;


   /*-- put gui structure hooks into info file table --*/
   info_files[0].button     = gui_info->objinfo->infopane->todo_struct;
   info_files[0].cr_button = gui_info->objinfo->infopane->crtodo_struct;

   info_files[1].button     = gui_info->objinfo->infopane->bugs_struct;
   info_files[1].cr_button = gui_info->objinfo->infopane->crbugs_struct;

   info_files[2].button     = gui_info->objinfo->infopane->done_struct;
   info_files[2].cr_button = gui_info->objinfo->infopane->crdone_struct;

   info_files[3].button     = gui_info->objinfo->infopane->changelog_struct;
   info_files[3].cr_button = gui_info->objinfo->infopane->crchangelog_struct;

   for (i = 0; i < knumber(info_files); i++)
   {
      if (info_files[i].path != NULL)
	 kfree(info_files[i].path);
      if (!kcms_get_attribute(object, info_files[i].attribute, &file_object))
	 continue;
      info_files[i].present = (file_object != NULL);
      if (file_object != NULL)
      {
	 kcms_get_attribute(file_object, KCMS_PATH, &tmp);
	 if (tmp != NULL)
	    info_files[i].path = kstrdup(tmp);
      }
      xvf_set_attribute(info_files[i].cr_button,
			XVF_TITLE, (info_files[i].present
				    ? "Delete"
				    : "Create"));
      xvf_set_attribute(info_files[i].button,
			XVF_ACTIVATE, info_files[i].present);
      xvw_sync(FALSE);
   }
}
