 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>             file name: signal.c                       <<<<
   >>>>                                                       <<<<
   >>>>		       Dispatch Routines                      <<<<
   >>>>                                                       <<<<
   >>>>       These routines are used to handle the execution <<<<
   >>>>	 of running processes.				      <<<<
   >>>>                                                       <<<<
   >>>>			xvw_sleep()			      <<<<
   >>>>			xvw_waitpid()			      <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"

static  xvw_signal  *signal_list = NULL;

/*-----------------------------------------------------------
|
|  Routine Name: remove_entry - 
|
|       Purpose: This should be a complete description that anyone
|                could understand;  it should have acceptable grammar
|                and correct spelling.
|
|         Input: entry - 
|    Written By: Mark Young
|          Date: Jan 21, 1995
|
------------------------------------------------------------*/
/*ARGSUSED*/
static void remove_entry(
   xvw_signal *entry)
{
	if (entry->inputid != NONE)
	   XtRemoveInput(entry->inputid);

	if (entry->next == NULL && entry->prev == NULL)
	{
	   signal_list =  NULL;
	}
	else if (entry->prev == NULL)
	{
	   entry->next->prev = NULL;
	   signal_list = entry->next;
	}
	else if (entry->next == NULL)
	{
	   entry->prev->next = NULL;
	}
	else
	{
	   entry->prev->next = entry->next;
	   entry->next->prev = entry->prev;
	}
	kfree(entry->error);
	kfree(entry);
}

/*-----------------------------------------------------------
|
|  Routine Name: dispatch_entry -
|
|       Purpose: This should be a complete description that anyone
|                could understand;  it should have acceptable grammar
|                and correct spelling.
|
|         Input: entry - 
|    Written By: Mark Young
|          Date: Jan 21, 1995
|
------------------------------------------------------------*/
/*ARGSUSED*/
static void dispatch_entry(
   xvw_signal *entry)
{
	kaddr	    data;
	char	    *error;
	kfunc_void  routine;
	int	    pid, status;

	/*
	 *  Need to call the callback routine notifying that the
	 *  process is done.   First we copy the information into
	 *  local variables and remove the entry.  We do this just
	 *  in case the user does something stupid like delete the handler
	 *  while in the callback.
	 */
	data    = entry->data;
	routine = entry->routine;
	pid     = entry->pid;
	status  = kpclose(entry->file);
	error   = entry->error, entry->error = NULL;

	remove_entry(entry);
	if (routine != NULL)
	   routine(pid, status, data, error);
	else if (error != NULL)
	   kfile_writedata(2, error, 0);

	kfree(error);
}

/*-----------------------------------------------------------
|
|  Routine Name: error_callback - 
|
|       Purpose: This should be a complete description that anyone
|                could understand;  it should have acceptable grammar
|                and correct spelling.
|
|         Input: data -
|                fid  -
|                id   -
|    Written By: Mark Young
|          Date: Jan 21, 1995
|
------------------------------------------------------------*/
/*ARGSUSED*/
static void error_callback(
   XtPointer data,
   int       *fid,
   XtInputId *id)
{
	xvw_signal *entry = (xvw_signal *) data;
	int         nbytes;
	char        *tmp, buffer[512];


	if ((nbytes = read(*fid, buffer, 511)) > 0)
	{
	   buffer[nbytes] = '\0'; tmp = entry->error;
	   entry->error = kstring_cat(tmp, buffer, NULL); kfree(tmp);
	}
	else
	   dispatch_entry(entry);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvw_add_handler - 
|
|       Purpose: This should be a complete description that anyone
|                could understand;  it should have acceptable grammar
|                and correct spelling.
|
|    Written By: Mark Young
|          Date: Jan 21, 1995
|
------------------------------------------------------------*/
/*ARGSUSED*/
xvw_signal *xvw_add_handler(
   kfunc_void routine,
   kaddr      client_data,
   kfile      *file)
{
	int	   fid;
	xvw_signal *entry;


	/*
	 *  create a new entry in which to record the signal information.
	 *  This is where a process records it's pid and status so that when
	 *  the process is done we can call the routine back or display any
	 *  errors that may occur.
	 */
	if ((entry = (xvw_signal *) kcalloc(1,sizeof(xvw_signal))) == NULL)
	{
	    kerror("xvforms", "xvw_add_handler", 
		   "Unable to allocate form signal entry"); 
	    return(NULL);
	}
	entry->next   = signal_list;
	entry->prev   = NULL;

	if (signal_list != NULL)
	   signal_list->prev = entry;

	signal_list    = entry;
	entry->routine = routine;
	entry->data    = client_data;
	entry->file    = file;
	entry->pid     = kpinfo(file);

	kgetdescriptors(file, &fid, NULL);
	if (fid != -1)
	{
	   entry->inputid = XtAppAddInput(xvw_appcontext(NULL), fid, 
			(XtPointer) XtInputReadMask, error_callback,
			(XtPointer) entry);
	}
	return(entry);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvw_remove_handler - library routine for vadd
|
|       Purpose: This should be a complete description that anyone
|                could understand;  it should have acceptable grammar
|                and correct spelling.
|
|         Input: pid - 
|    Written By: Mark Young
|          Date: Jan 21, 1995
|
------------------------------------------------------------*/
/*ARGSUSED*/
void xvw_remove_handler(
   int pid)
{
	xvw_signal  *entry;

	if (signal_list == NULL)
	   return;

	entry = signal_list;
	while (entry->pid != pid && entry->next != NULL)
	{
	   entry = entry->next;
	}

	/*
	 *  If we found the process then delete it from the linked list,
	 *  and free memory.
	 */
	if (entry->pid == pid)
	{
	   kpclose(entry->file);
	   remove_entry(entry);
	}
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_check_process
|
|       Purpose: check to see if a process exists.  In order
|                to be portable we will use kill instead of
|		 waitpid or wait4.  We send a 0 to the process,
|		 if the kill fails and errno is set to ESRCH we
|		 know that process is finished.
|
|         Input:  pid -  integer corresponding to the process
|			 we wish to check.
|       Returns: Return TRUE if process is exists.  Otherwise, return FALSE.
|    Written By: Mark Young
|          Date: Nov 15, 1992
|
-------------------------------------------------------------*/
/*ARGSUSED*/
int  xvw_check_process(
   int pid)
{
	if (waitpid(pid, NULL, WNOHANG) == -1)
	{
	   /*
	    *  Failed to send a 0 to the process.  If errno
	    *  is set to EINVAL (note: this should never happen)
	    *
	    *  If errno is set to ESRCH then the process is done,
	    *  otherwise assume the process to be alive.
	    */
	   if (errno == ECHILD || errno == EINVAL)
	      return(FALSE);
	   else
	      return(TRUE);
	}
	else
	   return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvw_sleep
|
|       Purpose: The purpose of this routine is to sleep
|                for the desired number of seconds or until
|                there is input pending from the X server
|		 ie) XtPending().
|
|         Input: stime -  integer corresponding to the process
|    Written By: Mark Young
|          Date: Nov 15, 1992
|
------------------------------------------------------------*/
/*ARGSUSED*/
int xvw_sleep(
   double stime)
{
#if 0
	Boolean ignoreInputs = FALSE, 
	        ignoreEvents = FALSE, 
		ignoreTimers = FALSE,
		block = TRUE;
	unsigned long howlong;

	howlong = stime*1000;
	(void) _XtwaitForSomething(ignoreTimers, ignoreInputs, ignoreEvents,
                        block, &howlong, xvw_appcontext(NULL));
#endif
	return(TRUE);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvw_waitpid -
|
|      Purpose:  The purpose of this routine is to wait until
|		 a certain process is done while still processing
|		 certain X Events.
|
|        Input:  pid -  integer corresponding to the process
|			we wish to check.
|
|		 stop_button -  The object in which we should
|			abort this process.  If the user button
|			presses the stop button then we return
|		        -1 so that the user can kill the process
|			and cleanup.  The "stop_button" can be
|			passed as NULL which will prevent the user
|			from stop the process until done.
|
|       Output:  returns 0 upon suceesful completion of process
|		 pid, or -1 if the user aborted the process.
|   Written By:   Mark Young
|
-----------------------------------------------------------*/
/*ARGSUSED*/
int xvw_waitpid(
   int      pid,
   xvobject stop_button)
{
	xvw_signal *entry;

	/*
	 *  Check to see if the process is done.  If not then sleep for
	 *  1 second, then check to see if the user depressed the stop
	 *  button.  If not then repeat the process until the process
	 *  is done.
	 */
	do
	{
	    xvw_sleep(1.0);
	    if (xvw_clean_event_list(stop_button) == FALSE)
	       return(-1);
	} while (xvw_check_process(pid));

	/*
	 *  Race thru the entry list looking for our pid, if it's in the
	 *  list then return the status, otherwise return NULL.
	 */
	entry = signal_list;
	while (entry != NULL)
	{
	   if (entry->pid == pid)
	   {
	       dispatch_entry(entry);
	       return(0);
	   }
	   entry = entry->next;
	}
	return(-1);
}
