 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Dispatch Event Utilities
   >>>>
   >>>>   Static:
   >>>>		    process_event()
   >>>>  Private:
   >>>>		    xvw_sync()
   >>>>		    xvw_process_event()
   >>>>		    xvw_clean_event_list()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/*--------------------------------------------------------------
|
|  Routine Name: process_event
|
|       Purpose: Reads an event and then dispatches it.  If
|                no events remain and block is set to FALSE, then
|                FALSE is returned, indicating that no more events remain.
|                In addition, will record the event the event if 
|                a journal recording is going on.
|
|         Input: block - whether we should block ("wait") until
|                        an event arrives.
|
|        Output: none
|	Returns: Returns TRUE if more events are pending; otherwise,
|                FALSE is returned.  Note: if block is set to TRUE
|                then we will always return TRUE.
|
|    Written By: Mark Young
|          Date: Apr 07, 1995
| Modifications:
|
------------------------------------------------------------------*/


static int process_event(
   int block)
{
        XtInputMask  mask;
	XtAppContext context = xvw_appcontext(NULL);


        /*
         *  Get the next event.
         */
        if ((mask = XtAppPending(xvw_appcontext(NULL))) & XtIMXEvent)
	   XtAppProcessEvent(context, XtIMXEvent);
	else if (mask != 0)
	   XtAppProcessEvent(context, mask);
	else if (block == TRUE)
	   XtAppProcessEvent(context, XtIMAll);

        return((mask & XtIMXEvent) ? TRUE : FALSE);
}

/*--------------------------------------------------------------
|
|  Routine Name: xvw_sync
|
|       Purpose: Gathers and dispatches XEvents until none are
|                left.  The purpose is to remove all pending events 
|                by either processing them or discarding them from 
|                the queue.  This is done by calling xvw_clean_event_list() 
|                if all pending events are to be discarded, or by 
|                repeatedly calling process_event() with block set to FALSE.
|
|         Input: discard - whether the pending events should be
|                          processed or simply removed from the
|                          queue.
|        Output: none
|	Returns: none
|
|    Written By: Mark Young 
|          Date: Jul 08, 1992
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------------*/

int xvw_sync(
   int discard)
{
        int     events_processed = FALSE;


        if (discard == TRUE)
           (void) xvw_clean_event_list(NULL);
        else
           while (process_event(FALSE))
	      events_processed = TRUE;

	return(events_processed);
}

/*--------------------------------------------------------------
|
|  Routine Name: xvw_process_event
|
|       Purpose: Gathers and dispatches XEvents.
|
|         Input: none
|        Output: none
|	Returns: none
|
|    Written By: Mark Young
|          Date: Jul 16, 1992   
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------------*/

void xvw_process_event(void)
{
        (void) process_event(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: xvw_clean_event_list()
|
|       Purpose: Cleans out the event list, discarding all events
|                that we are not interested in, including:
|                KeyPress, KeyRelease, ButtonPress, ButtonRelease,
|                and MotionNotify events.  The user is allowed to
|                initiate these events, but they will not be dispatched.
|
|                An exception object may be passed in; if the exception
|                object is not ButtonPress events within that object only
|                will be dispatched.  Using this mechanism, the user may be
|                allowed to do a ButtonPress on a special object to abort
|                the current process (or to perform some other action).
|
|         Input: exception - one object that is an exception to the rule
|                            (if applicable) and will still get ButtonPress
|                            events.
|
|        Output: none
|	Returns: Returns (0) if the user wants to abort the current process
|                Returns (1) otherwise
|    Written By: Mark Young
|          Date: Jul 08, 1992
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------------*/

int xvw_clean_event_list(
   xvobject exception)
{
        XtInputMask mask;
        XEvent      event;
        Window      window;


        XSync(xvw_display(NULL), FALSE);

        if (exception != NULL)
           window = xvw_window(exception);
        else
           window = NONE;

        do
        {
           mask = XtAppPending(xvw_appcontext(NULL));
           if (mask & XtIMXEvent)
           {
              XtAppNextEvent(xvw_appcontext(NULL), &event);
              if (event.xany.window == window)
              {
                 if (event.type == ButtonPress)
                    return(FALSE);
                 else
                    XtDispatchEvent(&event);
              }
              else if (event.type >= EnterNotify)
              {
                 XtDispatchEvent(&event);
              }
           }
           else if (mask != 0)
           {
              XtAppProcessEvent(xvw_appcontext(NULL), mask);
           }
        }
        while (mask != 0);

        return(TRUE);
}
