 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>       Conversion routines for xvwidgets
   >>>>
   >>>>  Private:
   >>>>			cvtFontToString()
   >>>>			cvtStringToFont()
   >>>>			cvtFontStructToString()
   >>>>			cvtStringToFontStruct()
   >>>>
   >>>>			cvtPixelToString()
   >>>>			cvtStringToPixel()
   >>>>
   >>>>			cvtFilenameToPixmap()
   >>>>			cvtPixmapToFilename()
   >>>>			cvtFilenameToPixmapMask()
   >>>>			cvtPixmapMaskToFilename()
   >>>>
   >>>>			cvtPixmapToXImage()
   >>>>			cvtXImageToPixmap()
   >>>>
   >>>>			cvtFilenameToXImage()
   >>>>			cvtXImageToFilename()
   >>>>
   >>>>			cvtStringToCursor()
   >>>>
   >>>>			cvtBooleanToInt()
   >>>>			cvtBooleanToShort
   >>>>			cvtBooleanToString
   >>>>			cvtShortToString
   >>>>			cvtIntToString
   >>>>			cvtShortToInt
   >>>>			cvtIntToShort
   >>>>			cvtDoubleToFloat
   >>>>			cvtFloatToDouble
   >>>>			cvtObjectToWidget()
   >>>>			cvtWidgetToObject()
   >>>>
   >>>>			xvw_init_converters()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <X11/Xatom.h>
#include <kdatafmt/xpm.h>

#define cvtDone(type, value, toVal) \
{                                                       \
    if (toVal->addr != NULL) {                          \
	if (toVal->size < sizeof(type)) {               \
	    toVal->size = sizeof(type);                 \
	    return(FALSE);                              \
	}                                               \
	*((type*)(toVal->addr)) = (type) (value);       \
    }                                                   \
    else {                                              \
	static type static_val;                         \
	static_val = (value);                           \
	toVal->addr = (XtPointer)&static_val;           \
    }                                                   \
    toVal->size = sizeof(type);                         \
}


/*-------------------------------------------------------------------*
|
|   Font Conversion
|
--------------------------------------------------------------------*/

static klist *fontlist = NULL;

/*-----------------------------------------------------------
|  Routine Name: cvtFontToString
|       Purpose: Convert a font to a string.
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtFontToString(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	klist	    *temp;
	Atom	    name, value;
	XFontStruct *fontstruct;
	char	    *fontname = NULL;
	Font *font = (Font *) fromVal->addr;


	if ((temp = klist_locate(fontlist, (kaddr) *font)) != NULL)
	   fontname = ktoken_to_string((int) temp->client_data);
	else if ((fontstruct = XQueryFont(display, *font)) != NULL)
	{
	   if ((name = XInternAtom(display, "FONTNAME", FALSE)) != None &&
	        XGetFontProperty(fontstruct, name, &value) == TRUE)
	           fontname = XGetAtomName(display, value);
	}

	if (fontname == NULL)
	{
	   cvtDone(String, "Unknown font", toVal);
	   return(FALSE);
	}
	cvtDone(String, fontname, toVal);
	return(TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: cvtStringToFont
|       Purpose: Convert a string to a font.
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtStringToFont(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	int	    token;
	klist	    *list;
	XFontStruct *fontstruct = NULL;
	char	    fontname[KLENGTH], *string = (String) fromVal->addr;


        if (!kstring_cleanup(string, fontname))
           return(FALSE);

	/*
	 *  Go ahead and see if the fontname is already cached in our
	 *  static fontlist.
	 */
	if ((token = kstring_to_token(fontname)) == 0)
	   return(FALSE);

	if ((list = klist_locate_clientdata(fontlist, (kaddr) token)) != NULL)
	{
	   fontstruct = (XFontStruct *) klist_identifier(list);
	   cvtDone(Font, fontstruct->fid, toVal);
	   return(TRUE);
	}

        if ((fontstruct = XLoadQueryFont(display, fontname)) == NULL)
	{
	   XrmName xrm_name[2];
	   XrmClass xrm_class[2];
	   XrmRepresentation rep_type;
	   XrmValue value;

	   /*
	    *  Error on default font...
	    */
	   if (kstrcmp(fontname,"XtDefaultFont") != 0)
	   {
	      kerror("xvwidgets", "cvtStringToFont", "Error! Cannot load \
font '%s'.  Loading default font 'xtDefaultFont'", fontname);
	   }

	   /*
	    *  Try to find default font
	    */
	   kstrcpy(fontname, "XtDefaultFont");
	   xrm_name[0] = XrmStringToName ("xtDefaultFont");
	   xrm_name[1] = (XrmClass) NULL;
	   xrm_class[0] = XrmStringToClass ("XtDefaultFont");
	   xrm_class[1] = (XrmClass) NULL;
	   if (XrmQGetResource(XtDatabase(display), xrm_name, xrm_class,
                            &rep_type, &value))
	   {
              if (rep_type == XtQstring && (fontstruct = XLoadQueryFont(display,
			(char *) value.addr)) == None)
		 kinfo(KSTANDARD,"Unable to load default font 'XtDefaultFont'");
	      else if (rep_type == XtQfont)
		fontstruct = XQueryFont(display, *((Font *) value.addr));
	      else if (rep_type == XtQfontStruct)
		fontstruct = ((XFontStruct*)value.addr);
           }
	   else
	      kinfo(KXVLIB, "cvtStringToFont: couldn't find 'xtDefaultFont' \
to be loaded.");

	   kstrcpy(fontname, "-*-*-*-R-*-*-*-120-*-*-*-*-ISO8859-1");
	   if (fontstruct == NULL && (fontstruct = XLoadQueryFont(display,
		fontname)) == NULL)
	   {
	      kinfo(KSTANDARD, "cvtStringToFont: could not find \
'XtDefaultFont' to be loaded.");
	   }
	   fontlist = klist_add(fontlist, (kaddr) fontstruct, (kaddr) token);
	   token = kstring_to_token(fontname);
        }

	if (fontstruct != NULL)
	   fontlist = klist_add(fontlist, (kaddr) fontstruct, (kaddr) token);
	else
	{
	   kerror("xvwidgets", "cvtStringToFont", "Error!  Unable to \
find any reasonable font to be loaded.  Unpredictable behavior may occur!");
	   return(FALSE);
	}
	cvtDone(Font, fontstruct->fid, toVal);
	return(TRUE);
}


/*-------------------------------------------------------------------*
|
|   FontStruct Conversion
|
--------------------------------------------------------------------*/

/*-----------------------------------------------------------
|  Routine Name: cvtFontStructToString
|       Purpose: Convert a font structure into a string.
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtFontStructToString(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	klist	    *temp;
	Atom	    name, value;
	char	    *fontname = NULL;
	XFontStruct *fontstruct = *((XFontStruct **) fromVal->addr);


	if (fontstruct == NULL)
	   return(FALSE);

	if ((temp = klist_locate(fontlist, (kaddr) fontstruct)) != NULL)
	   fontname = ktoken_to_string((int) klist_clientdata(temp));
	else if ((name = XInternAtom(display, "FONTNAME", FALSE)) != None &&
	        XGetFontProperty(fontstruct, name, &value) == TRUE)
	   fontname = XGetAtomName(display, value);

	if (fontname == NULL)
	{
	   cvtDone(String, "Unknown font", toVal);
	   return(FALSE);
	}
	cvtDone(String, fontname, toVal);
	return(TRUE);
}



/*-----------------------------------------------------------
|  Routine Name: cvtStringToFontStruct - 
|       Purpose: Convert a string into a font structure.  
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtStringToFontStruct(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	int	    token;
	klist	    *list;
	XFontStruct *fontstruct = NULL;
	char	    fontname[KLENGTH], *string = (String) fromVal->addr;


        if (!kstring_cleanup(string, fontname))
           return(FALSE);

	/*
	 *  Go ahead and see if the fontname is already cached in our
	 *  static fontlist.
	 */
	if ((token = kstring_to_token(fontname)) == 0)
	   return(FALSE);

	if ((list = klist_locate_clientdata(fontlist, (kaddr) token)) != NULL)
	{
	   cvtDone(XFontStruct *, klist_identifier(list), toVal);
	   return(TRUE);
	}

	if ((fontstruct = XLoadQueryFont(display, fontname)) == NULL)
	{
	   XrmName xrm_name[2];
	   XrmClass xrm_class[2];
	   XrmRepresentation rep_type;
	   XrmValue value;

	   /*
	    *  Error on default font...
	    */
	   if (kstrcmp(fontname, "XtDefaultFont"))
	   {
	      kerror("xvwidgets", "cvtStringToFontStruct", "Error! \
Cannot load font '%s'.  Loading default font 'xtDefaultFont'", fontname);
	   }

	   /*
	    *  Try to find default font
	    */
	   kstrcpy(fontname, "xtDefaultFont");
	   xrm_name[0] = XrmStringToName ("xtDefaultFont");
	   xrm_name[1] = (XrmClass) NULL;
	   xrm_class[0] = XrmStringToClass ("XtDefaultFont");
	   xrm_class[1] = (XrmClass) NULL;
	   if (XrmQGetResource(XtDatabase(display), xrm_name, xrm_class,
                            &rep_type, &value))
	   {
              if (rep_type == XtQstring && (fontstruct = XLoadQueryFont(display,
			(char *) value.addr)) == NULL)
		 kinfo(KSTANDARD,"Unable to load default font 'xtDefaultFont'");
	      else if (rep_type == XtQfont)
                fontstruct = XQueryFont(display, *((Font *) value.addr));
              else if (rep_type == XtQfontStruct)
                fontstruct = ((XFontStruct*)value.addr);
           }
	   else
	      kinfo(KXVLIB, "cvtStringToFontStruct: could not find \
'XtDefaultFont' to be loaded.");

	   kstrcpy(fontname, "-*-*-*-R-*-*-*-120-*-*-*-*-ISO8859-1");
	   if (fontstruct == NULL && (fontstruct = XLoadQueryFont(display,
		fontname)) == NULL)
	   {
	      kinfo(KSTANDARD, "cvtStringToFontStruct: could not find \
'XtDefaultFont' to be loaded.");
	   }
	   fontlist = klist_add(fontlist, (kaddr) fontstruct, (kaddr) token);
	   token = kstring_to_token(fontname);
        }

	if (fontstruct != NULL)
	   fontlist = klist_add(fontlist, (kaddr) fontstruct, (kaddr) token);
	else
	{
	   kerror("xvwidgets", "cvtStringToFontStruct", "Error!  Unable \
to find any reasonable font to be loaded.  Unpredictable behavior may occur!");
	   return(FALSE);
	}
	cvtDone(XFontStruct *, fontstruct, toVal);
	return(TRUE);
}


/*-------------------------------------------------------------------*
|
|   Color conversion
|
--------------------------------------------------------------------*/
 
static klist *colorlist = NULL;


/*-----------------------------------------------------------
|  Routine Name: cvtPixelToString
|       Purpose: Convert a pixel to a string. 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/

static XtConvertArgRec ptosConvertArgs[] = {
    {XtWidgetBaseOffset, (XtPointer)XtOffsetOf(WidgetRec, core.screen),
     sizeof(Screen *)},
    {XtWidgetBaseOffset, (XtPointer)XtOffsetOf(WidgetRec, core.colormap),
     sizeof(Colormap)}
};

/* ARGSUSED */
static Boolean cvtPixelToString(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	klist	 *temp;
	XColor   xcolor;
	Screen   *screen;
	Colormap colormap;
	int	 red, green, blue;
	char	 *string, color[KLENGTH];


	if (*num_args != 2)
	{
	   kinfo(KSTANDARD, 
		 "Pixel to kstring conversion needs the screen and colormap \
arguments");
	   return(FALSE);
	}
	screen = *((Screen **) args[0].addr);
	colormap = *((Colormap *) args[1].addr);
	xcolor.pixel = (*((Pixel *) fromVal->addr));

	if ((temp = klist_locate(colorlist, (kaddr) xcolor.pixel)) != NULL)
	   string = ktoken_to_string((int) temp->client_data);
	else if (xcolor.pixel == WhitePixelOfScreen(screen))
	   string = XtDefaultBackground;
	else if (xcolor.pixel == BlackPixelOfScreen(screen))
	   string = XtDefaultForeground;
	else
	{
	   XQueryColor(DisplayOfScreen(screen), colormap, &xcolor);

	   red   = xcolor.red >> 8;
	   green = xcolor.green >> 8;
	   blue  = xcolor.blue >> 8;
	   (void) ksprintf(color,"#%02x%02x%02x", red, green, blue);
	   string = kstrdup(color);
	}
	cvtDone(String, string, toVal);
	return(TRUE);
}



/*-----------------------------------------------------------
|  Routine Name: cvtStringToPixel
|       Purpose: Convert a string to a pixel.
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtStringToPixel(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	int	 token;
	klist    *list;
	Screen   *screen;
	XColor   xcolor;
	Colormap colormap;
	char	 colorname[KLENGTH], *string = (String) fromVal->addr;



	if (*num_args != 2)
	{
	   kinfo(KSTANDARD, 
		 "Pixel to kstring conversion needs the screen and colormap \
arguments");
	   return(FALSE);
	}
	screen    = *((Screen **) args[0].addr);
	colormap  = *((Colormap *) args[1].addr);

	if (!kstring_cleanup(string, colorname))
	   return(FALSE);

	/*
	 *  Go ahead and see if the colorname is already cached in our
	 *  static colorlist.
	 */
	if ((token = kstring_to_token(colorname)) == 0)
	   return(FALSE);

	if ((list = klist_locate_clientdata(colorlist, (kaddr) token)) != NULL)
	{
	   xcolor.pixel = (Pixel) klist_identifier(list);
	   cvtDone(Pixel, xcolor.pixel, toVal);
	   return(TRUE);
	}

	if (kstrcmp(colorname, XtDefaultBackground) == 0)
	{
	   xcolor.pixel = WhitePixelOfScreen(screen);
	   cvtDone(Pixel, xcolor.pixel, toVal);
	   return(TRUE);
	}
	else if (kstrcmp(colorname, XtDefaultForeground) == 0)
	{
	   xcolor.pixel = BlackPixelOfScreen(screen);
	   cvtDone(Pixel, xcolor.pixel, toVal);
	   return(TRUE);
	}

	if (colorname[0] == '#')
	{
           if (!XParseColor(display, colormap, colorname, &xcolor))
	   {
	      kinfo(KSTANDARD, 
		 "RGB color specification \"%s\" has invalid format",colorname);
              return(FALSE);
	   }
           else if (!XAllocColor(display, colormap, &xcolor))
	   {
              kinfo(KSTANDARD, "Out of Colors for current colormap.  Unable to \
allocate the color %s.", colorname);
              return(FALSE);
	   }
	}
	else
	{
	   xcolor.flags = DoRed | DoGreen | DoBlue;
           if (!XLookupColor(display, colormap, colorname, &xcolor, &xcolor))
	   {
	      kinfo(KSTANDARD, 
		 "RGB color specification \"%s\" has invalid format",colorname);
              return(FALSE);
	   }
           else if (!XAllocColor(display, colormap, &xcolor))
	   {
              kinfo(KSTANDARD, "Out of Colors for current colormap.  Unable to \
allocate the color %s.", colorname);
              return(FALSE);
	   }
	}
	colorlist = klist_add(colorlist, (kaddr) xcolor.pixel, (kaddr) token);
	cvtDone(Pixel, xcolor.pixel, toVal);
	return(TRUE);
}


/*-------------------------------------------------------------------*
|
|   Pixmap and Mask conversion
|
--------------------------------------------------------------------*/
 
static klist *pixmaplist = NULL;
static klist *masklist = NULL;


/*-----------------------------------------------------------
|  Routine Name: cvtPixmapToFilename
|       Purpose: Convert a pixmap to a string. 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/

static XtConvertArgRec pixtosConvertArgs[] = {
    {XtWidgetBaseOffset, (XtPointer)XtOffsetOf(WidgetRec, core.screen),
     sizeof(Screen *)},
    {XtWidgetBaseOffset, (XtPointer)XtOffsetOf(WidgetRec, core.colormap),
     sizeof(Colormap)},
    {XtWidgetBaseOffset, (XtPointer)XtOffsetOf(WidgetRec, core.self),
     sizeof(Widget)}
};

/* ARGSUSED */
static Boolean cvtPixmapToFilename(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	klist	 *temp;
	Pixmap   pixmap;
	char	 *string;

	if (*num_args != 3)
	{
	   kinfo(KSTANDARD, 
		 "Pixmap to kstring conversion needs the screen and colormap \
arguments");
	   cvtDone(String, NULL,  toVal);
	   return(FALSE);
	}
	pixmap   = (*((Pixmap *) fromVal->addr));

	if ((temp = klist_locate(pixmaplist, (kaddr) pixmap)) != NULL)
	   string = ktoken_to_string((int) temp->client_data);
	else if (pixmap == None)
	   string = "None";
	else if (pixmap == ParentRelative)
	   string = "ParentRelative";
	else
	{
/*
	   kerror("xvwidgets", "cvtPixmapToFilename", "Error!  Unable to \
find the filename that the pixmap was created from.");
 */
	   cvtDone(String, NULL,  toVal);
	   return(FALSE);
	}
	cvtDone(String, string,  toVal);
	return(TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: cvtFilenameToPixmap
|       Purpose: Convert a string to a pixmap.
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtFilenameToPixmap(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	klist    *list;
	Widget   widget;
	Screen   *screen;
	Window   rootwindow;
	unsigned char *data;
	Pixmap   bitmap, pixmap, mask = None;
	int      token, i, x, y, screennum;
	unsigned int width, height, border_width, depth;
	char	 pixmapfile[KLENGTH], *string = (String) fromVal->addr;

	Pixel	 fg, bg;
	XColor	 xcolor;
	xpm	 *image;
	Visual   *visual;
	GC	 gc, gc_mask;
	XImage   *ximage, *xmask;


	if (*num_args != 3)
	{
	   kinfo(KSTANDARD, 
		 "String to Pixmap conversion needs the screen and colormap \
arguments");
	   cvtDone(Pixmap, None, toVal);
	   return(FALSE);
	}
	screen     = *((Screen **) args[0].addr);
	widget     = *((Widget *) args[2].addr);
	screennum  = XScreenNumberOfScreen(screen);
	fg	   = BlackPixelOfScreen(screen);
	bg	   = WhitePixelOfScreen(screen);
	visual     = DefaultVisual(display, screennum);
	rootwindow = RootWindow(display, screennum);

	if (!kfullpath(string, NULL, pixmapfile))
	{
	   cvtDone(Pixmap, None, toVal);
	   return(FALSE);
	}

	token = kstring_to_token(pixmapfile);
	if ((list = klist_locate_clientdata(pixmaplist, (kaddr) token)) != NULL)
	{
	   pixmap = (Pixmap)  klist_identifier(list);
	   cvtDone(Pixmap, pixmap, toVal);
	   return(TRUE);
	}
	else if (kstrcmp(pixmapfile, "None") == 0)
	{
	   cvtDone(Pixmap, None, toVal);
	   return(TRUE);
	}
	else if (kstrcmp(pixmapfile, "Unspecified") == 0)
	{
	   cvtDone(Pixmap, XtUnspecifiedPixmap, toVal);
	   return(TRUE);
	}
	else if (kstrcmp(pixmapfile, "ParentRelative") == 0)
	{
	   cvtDone(Pixmap, ParentRelative, toVal);
	   return(TRUE);
	}

	bg = widget->core.background_pixel;
	if ((image = xpm_read(pixmapfile)) != NULL)
	{
	   data  = (unsigned char *) image->data;
	   width = image->width; height = image->height;

	   pixmap = XCreatePixmap(display, rootwindow, width, height,
			widget->core.depth);
	   ximage = XCreateImage(display, visual, widget->core.depth, ZPixmap,
			0, NULL, width, height, 8, 0);
	   ximage->data = kmalloc(ximage->bytes_per_line * height);

	   /*
	    *  Mask stuff
	    */
	   mask  = XCreatePixmap(display, rootwindow, width, height, 1);
	   xmask = XCreateImage(display, visual, 1, XYPixmap, 0, NULL,
				width, height, 8, 0);
	   xmask->data  = kcalloc(1, xmask->bytes_per_line * height);

	   gc      = XCreateGC(display, pixmap, 0, NULL);  
	   gc_mask = XCreateGC(display, mask, 0, NULL);  
	   for (i = 0; i < width*height; i++, data++)
	   {
	      x = i % width; y = i / width;
	      if (image->maps[*data].flags == 0xff)
	      {
		 image->maps[*data].flags = 1;
		 image->maps[*data].pixel = bg;
	      }
	      else if (!image->maps[*data].flags)
	      {
		 xcolor.red   = image->maps[*data].red;
		 xcolor.green = image->maps[*data].green;
		 xcolor.blue  = image->maps[*data].blue;
		 xcolor.flags = DoRed | DoGreen | DoBlue;
		 XAllocColor(display, widget->core.colormap, &xcolor);
		 image->maps[*data].flags = (int) xcolor.flags;
		 image->maps[*data].pixel = xcolor.pixel;
	      }
	
	      XPutPixel(ximage, x, y, image->maps[*data].pixel);
	      if (image->maps[*data].flags != 1)
	         XPutPixel(xmask, x, y, 1);
	   }
	   XPutImage(display, pixmap, gc, ximage, 0, 0, 0, 0, width, height);
	   XPutImage(display, mask, gc_mask, xmask,  0, 0, 0, 0, width, height);
	   XDestroyImage(ximage); XDestroyImage(xmask);
	   XFreeGC(display, gc); XFreeGC(display, gc_mask);
	   xpm_free(image);
	}
	else if ((bitmap = XmuLocateBitmapFile(screen, pixmapfile, NULL, 0,
			NULL, NULL, NULL, NULL)) != None)
	{
	   XGetGeometry(display, bitmap, &rootwindow, &x, &y, 
		     &width, &height, &border_width, &depth);
	   pixmap = XmuCreatePixmapFromBitmap (display,
			rootwindow, bitmap, width, height, 
			widget->core.depth, fg, bg);
	}
	else
	{
	   kerror("xvwidgets", "cvtFilenameToPixmap", "Error!  Unable to \
locate the pixmap file '%s'.", pixmapfile);
	   cvtDone(Pixmap, None, toVal);
	   return(FALSE);
	}
	pixmaplist = klist_add(pixmaplist, (kaddr) pixmap, (kaddr) token);
	masklist   = klist_add(masklist,   (kaddr) mask, (kaddr) token);
	cvtDone(Pixmap, pixmap, toVal);
	return(TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: cvtPixmapMaskToFilename
|       Purpose: Convert a pixmap mask to a filename. 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Jul 19, 1994
------------------------------------------------------------*/

/* ARGSUSED */
static Boolean cvtPixmapMaskToFilename(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	klist	 *temp;
	Pixmap   pixmap;
	char	 *string;


	pixmap   = (*((Pixmap *) fromVal->addr));

	if ((temp = klist_locate(masklist, (kaddr) pixmap)) != NULL)
	   string = ktoken_to_string((int) temp->client_data);
	else if (pixmap == None)
	   string = "None";
	else if (pixmap == ParentRelative)
	   string = "ParentRelative";
	else
	{
	   kerror("xvwidgets", "cvtPixmapMaskToFilename", "Error!  Unable to \
find the filename that the pixmap was created from.");
	   cvtDone(String, NULL,  toVal);
	   return(FALSE);
	}
	cvtDone(String, string,  toVal);
	return(TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: cvtFilenameToPixmapMask
|       Purpose: Convert a filename to a pixmap mask.
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Jul 19, 1994
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtFilenameToPixmapMask(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	Pixmap   mask;
	int      token;
	klist    *list;
	char	 pixmapfile[KLENGTH], *string = (String) fromVal->addr;


	if (!kfullpath(string, NULL, pixmapfile))
	{
	   cvtDone(Pixmap, None, toVal);
	   return(FALSE);
	}

	token = kstring_to_token(pixmapfile);
	if ((list = klist_locate_clientdata(masklist, (kaddr) token)) != NULL)
	{
	   mask = (Pixmap)  klist_identifier(list);
	   cvtDone(Pixmap, mask, toVal);
	   return(TRUE);
	}
	else if (kstrcmp(pixmapfile, "None") == 0)
	{
	   cvtDone(Pixmap, None, toVal);
	   return(TRUE);
	}
	else if (kstrcmp(pixmapfile, "ParentRelative") == 0)
	{
	   cvtDone(Pixmap, ParentRelative, toVal);
	   return(TRUE);
	}
	else
	{
	   cvtDone(Pixmap, None, toVal);
	   return(FALSE);
	}
}

/*-------------------------------------------------------------------*
|
|   XImage Pixmap conversion
|
--------------------------------------------------------------------*/
 
static klist *ximage_pixmaplist = NULL;

/*-----------------------------------------------------------
|  Routine Name: cvtXImageToPixmap
|       Purpose: Convert a XImage to a pixmap. 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Sep 02, 1994
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtXImageToPixmap(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	klist	 *temp;
	XImage   *ximage;
	Pixmap   pixmap;
	ximage   = (*((XImage **) fromVal->addr));

	if ((temp = klist_locate(ximage_pixmaplist, (kaddr) ximage)) != NULL)
	   pixmap = (Pixmap) ktoken_to_string((int) temp->client_data);
	else
	{
	   kerror("xvwidgets", "cvtXImageToPixmap", "Error!  Unable to \
find the pixmap that the ximage was created from.");
	   cvtDone(Pixmap, None, toVal);
	   return(FALSE);
	}
	cvtDone(Pixmap, pixmap, toVal);
	return(TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: cvtPixmapToXImage
|       Purpose: Convert a pixmap to a XImage.
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Sep 02, 1994
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtPixmapToXImage(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	klist    *list;
	XImage   *ximage;

	int      x, y;
	Window   rootwindow;
	unsigned int width, height, bw, depth;
	Pixmap   pixmap = *((Pixmap *) fromVal->addr);


	list = klist_locate_clientdata(ximage_pixmaplist, (kaddr) pixmap);
	if (list != NULL)
	{
	   ximage = (XImage *) klist_identifier(list);
	   cvtDone(XImage *, ximage, toVal);
	   return(TRUE);
	}

	if (!XGetGeometry(display, pixmap, &rootwindow, &x, &y, &width,
			&height, &bw, &depth))
	{
	   kerror("xvwidgets", "cvtPixmapToXImage", "Error!  Unable to \
stat the pixmap in order to create the final XImage structure.");
	   cvtDone(XImage *, NULL, toVal);
	   return(FALSE);
	}
	else if ((ximage = XGetImage(display, pixmap, 0, 0, width, height,
				AllPlanes, ZPixmap)) == NULL)
	{
	   kerror("xvwidgets", "cvtPixmapToXImage", "Error!  Unable to \
convert the pixmap to an XImage structure.");
	   cvtDone(XImage *, NULL, toVal);
	   return(FALSE);
	}
	ximage_pixmaplist = klist_add(ximage_pixmaplist, (kaddr) ximage,
					(kaddr) pixmap);
	cvtDone(XImage *, ximage, toVal);
	return(TRUE);
}

/*-------------------------------------------------------------------*
|
|   XImage Filename conversion
|
--------------------------------------------------------------------*/
 
static klist *ximage_filelist = NULL;

static XtConvertArgRec ximtosConvertArgs[] = {
    {XtWidgetBaseOffset, (XtPointer)XtOffsetOf(WidgetRec, core.screen),
     sizeof(Screen *)},
    {XtWidgetBaseOffset, (XtPointer)XtOffsetOf(WidgetRec, core.colormap),
     sizeof(Colormap)},
    {XtWidgetBaseOffset, (XtPointer)XtOffsetOf(WidgetRec, core.self),
     sizeof(Widget)}
};

/*-----------------------------------------------------------
|  Routine Name: cvtXImageToFilename
|       Purpose: Convert a XImage to a string. 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtXImageToFilename(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	klist	 *temp;
	XImage   *ximage;
	char	 *string;

	if (*num_args != 3)
	{
	   kinfo(KSTANDARD, 
		 "XImage to Filename conversion needs the screen, colormap, \
and widget arguments");
	   return(FALSE);
	}
	ximage   = (*((XImage **) fromVal->addr));

	if ((temp = klist_locate(ximage_filelist, (kaddr) ximage)) != NULL)
	   string = ktoken_to_string((int) temp->client_data);
	else if (ximage == NULL)
	   string = "None";
	else
	{
	   kerror("xvwidgets", "cvtXImageToFilename", "Error!  Unable to \
find the filename that the ximage was created from.");
	   cvtDone(String, NULL, toVal);
	   return(FALSE);
	}
	cvtDone(String, string, toVal);
	return(TRUE);
}


/*-----------------------------------------------------------
|  Routine Name: cvtFilenameToXImage
|       Purpose: Convert a string to a XImage.
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtFilenameToXImage(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	klist    *list;
	Widget   widget;
	Pixmap   pixmap;
	XColor   xcolor;
	xpm	 *image;
	XImage   *ximage;
	Screen   *screen;
	Visual   *visual;
	Colormap colormap;
	Window   rootwindow;

	Pixel	  fg, bg;
	unsigned  char *data;
	int       i, x, y, token, screennum;
	unsigned  int width, height, bw, depth;
	char	  pixmapfile[KLENGTH], *string = (String) fromVal->addr;


	if (*num_args != 3)
	{
	   kinfo(KSTANDARD, 
		 "Filename to XImage conversion needs the screen, colormap, \
and widget arguments");
	   return(FALSE);
	}
	screen     = *((Screen **) args[0].addr);
	colormap   = *((Colormap *) args[1].addr);
	widget     = *((Widget *) args[2].addr);
	fg	   = BlackPixelOfScreen(screen);
	bg	   = WhitePixelOfScreen(screen);
	screennum  = XScreenNumberOfScreen(screen);
	visual     = DefaultVisual(display, screennum);
	rootwindow = RootWindow(display, screennum);

	if (!kfullpath(string, NULL, pixmapfile))
	   return(FALSE);

	token = kstring_to_token(pixmapfile);
	list = klist_locate_clientdata(ximage_filelist, (kaddr) token);
	if (list != NULL)
	{
	   ximage = (XImage *) klist_identifier(list);
	   cvtDone(XImage *, ximage, toVal);
	   return(TRUE);
	}
	else if (kstrcmp(pixmapfile, "None") == 0)
	{
	   cvtDone(XImage *, NULL, toVal);
	   return(TRUE);
	}
	else if (kstrcmp(pixmapfile, "ParentRelative") == 0)
	{
	   cvtDone(XImage *, NULL, toVal);
	   return(TRUE);
	}

	bg = widget->core.background_pixel;
	if ((image = xpm_read(pixmapfile)) != NULL)
	{
	   data  = (unsigned char *) image->data;
	   width = image->width; height = image->height;

	   ximage = XCreateImage(display, visual, widget->core.depth, ZPixmap,
			0, NULL, width, height, 8, 0);
	   ximage->data = kmalloc(ximage->bytes_per_line * height);
	   for (i = 0; i < width*height; i++, data++)
	   {
	      x = i % width; y = i / width;
	      if (image->maps[*data].flags == 0xff)
	      {
		 image->maps[*data].flags = DoRed | DoGreen | DoBlue;
		 image->maps[*data].pixel = bg;
	      }
	      else if (!image->maps[*data].flags)
	      {
		 xcolor.red   = image->maps[*data].red;
		 xcolor.green = image->maps[*data].green;
		 xcolor.blue  = image->maps[*data].blue;
		 xcolor.flags = DoRed | DoGreen | DoBlue;
		 XAllocColor(display, widget->core.colormap, &xcolor);
		 image->maps[*data].flags = (int) xcolor.flags;
		 image->maps[*data].pixel = xcolor.pixel;
	      }
	      XPutPixel(ximage, x, y, image->maps[*data].pixel);
	   }
	   xpm_free(image);
	}
	else if ((pixmap = XmuLocateBitmapFile(screen, pixmapfile, NULL, 0,
			NULL, NULL, NULL, NULL)) != None)
	{
	   if (!XGetGeometry(display, pixmap, &rootwindow, &x, &y, &width,
			&height, &bw, &depth))
	   {
	      kerror("xvwidgets", "cvtFilenameToXImage", "Error!  Unable to \
stat the pixmap loaded from file '%s' in order to create the final XImage \
structure.", pixmapfile);
	      cvtDone(XImage *, NULL, toVal);
	      return(FALSE);
	   }
	   else if ((ximage = XGetImage(display, pixmap, 0, 0, width, height,
				1, XYPixmap)) == NULL)
	   {
	      kerror("xvwidgets", "cvtFilenameToXImage", "Error!  Unable to \
convert the pixmap loaded from file '%s' into an XImage structure.",pixmapfile);
	      cvtDone(XImage *, NULL, toVal);
	      return(FALSE);
	   }
	   ximage->format = XYBitmap;
	   pixmaplist = klist_add(pixmaplist, (kaddr) pixmap, (kaddr) token);
	}
	else
	{
	   kerror("xvwidgets", "cvtFilenameToXImage", "Error!  Unable to \
locate the file '%s'.", pixmapfile);
	   cvtDone(XImage *, NULL, toVal);
	   return(FALSE);
	}
	ximage_filelist = klist_add(ximage_filelist, (kaddr) ximage,
				(kaddr) token);
	cvtDone(XImage *, ximage, toVal);
	return(TRUE);
}

/*-------------------------------------------------------------------*
|
|   Cursor conversion
|
--------------------------------------------------------------------*/

/*-----------------------------------------------------------
|
|  Routine Name: cvtStringToCursor
|
|       Purpose: Convert a kstring to a Cursor. 
|
|         Input:
|
|        Output:
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Jul 18, 1993
| Modifications:
|
------------------------------------------------------------*/

static XtConvertArgRec cursorConvertArg[] = {
    {XtWidgetBaseOffset, (XtPointer) XtOffsetOf(WidgetRec, core.screen),
     sizeof(Screen *)}
};

/* ARGSUSED */
static Boolean cvtStringToCursor(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	Cursor   cursor;
        char     filename[KLENGTH];
	String	 string = (String) fromVal->addr;
	XrmValue ret_val;


	if (kfullpath((char *) string, NULL, filename))
	   fromVal->addr = (XtPointer) filename;

	XmuCvtStringToCursor(args, num_args, fromVal, &ret_val);

	cursor = *((Cursor *) ret_val.addr);
	fromVal->addr = (XtPointer) string;
	cvtDone(Cursor, cursor, toVal);
	return(TRUE);
}


/*-------------------------------------------------------------------*
|
|   Boolean/Short/Int/Long/String conversion
|
--------------------------------------------------------------------*/


/*-----------------------------------------------------------
|  Routine Name: cvtBooleanToInt
|       Purpose: Convert a boolean to an int.
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtBooleanToInt(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	cvtDone(int, (*(Boolean *)fromVal->addr) != 0, toVal);
	return(TRUE);
}



/*-----------------------------------------------------------
|  Routine Name: cvtBooleanToShort
|       Purpose: Convert a boolean to a short. 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtBooleanToShort(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	cvtDone(short, ((*(Boolean *) fromVal->addr) != 0), toVal);
	return(TRUE);
}



/*-----------------------------------------------------------
|  Routine Name: cvtBooleanToString
|       Purpose: Convert a boolean to a string.
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtBooleanToString(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	char	*temp;

	if ((*(Boolean *) fromVal->addr) != 0)
	   temp = "1";
	else
	   temp = "0";

	cvtDone(String, kstrdup(temp), toVal);
	return(TRUE);
}



/*-----------------------------------------------------------
|  Routine Name: cvtShortToString
|       Purpose: Convert a short to a string. 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtShortToString(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	char	temp[KLENGTH];

	(void) ksprintf(temp, "%d", (*(short *) fromVal->addr));
	cvtDone(String, kstrdup(temp), toVal);
	return(TRUE);
}



/*-----------------------------------------------------------
|  Routine Name: cvtIntToString
|       Purpose: Convert an int to a string.
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtIntToString(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	char	temp[KLENGTH];

	(void) ksprintf(temp, "%d", (*(int *) fromVal->addr));
	cvtDone(String, kstrdup(temp), toVal);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: cvtFloatToString
|
|       Purpose: Convert a float to a string.
|
|         Input:
|
|        Output:
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Oct 24, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtFloatToString(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	char	temp[KLENGTH];

	(void) ksprintf(temp, "%g", (*(float *) fromVal->addr));
	cvtDone(String, kstrdup(temp), toVal);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: cvtDoubleToString
|
|       Purpose: Convert a double to a string.
|
|         Input:
|
|        Output:
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Oct 24, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtDoubleToString(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	char	temp[KLENGTH];

	(void) ksprintf(temp, "%g", (*(double *) fromVal->addr));
	cvtDone(String, kstrdup(temp), toVal);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: cvtStringToDouble
|
|       Purpose: Convert a string to double.
|
|         Input:
|
|        Output:
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Oct 24, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtStringToDouble(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	double num;

	num = strtod(fromVal->addr, NULL);
	cvtDone(double, num, toVal);
	return(TRUE);
}


/*-----------------------------------------------------------
|  Routine Name: cvtIntToShort
|       Purpose: Convert a int to short
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtIntToShort(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	cvtDone(short, (*(int *) fromVal->addr), toVal);
	return(TRUE);
}


/*-----------------------------------------------------------
|  Routine Name: cvtShortToInt
|       Purpose: Convert a int to a short. 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtShortToInt(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	cvtDone(int, (*(short *) fromVal->addr), toVal);
	return(TRUE);
}


/*-----------------------------------------------------------
|  Routine Name: cvtIntToUShort
|       Purpose: Convert a int to short
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtIntToUShort(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	cvtDone(unsigned short, (*(int *) fromVal->addr), toVal);
	return(TRUE);
}


/*-----------------------------------------------------------
|  Routine Name: cvtUShortToInt
|       Purpose: Convert a int to a short. 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtUShortToInt(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	cvtDone(int, (*(unsigned short *) fromVal->addr), toVal);
	return(TRUE);
}


/*-----------------------------------------------------------
|  Routine Name: cvtDoubleToFloat
|       Purpose: Convert a double to a float. 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtDoubleToFloat(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	cvtDone(float, (*(double *) fromVal->addr), toVal);
	return(TRUE);
}


/*-----------------------------------------------------------
|  Routine Name: cvtFloatToDouble
|       Purpose: Convert a float to a double. 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtFloatToDouble(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	cvtDone(double, (*(float *) fromVal->addr), toVal);
	return(TRUE);
}


/*-----------------------------------------------------------
|  Routine Name: cvtObjectToWidget
|       Purpose: Convert an xvobject to a widget
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young and John Salas
|          Date: Aug 18, 1992
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtObjectToWidget(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	cvtDone(Widget, xvw_widget(*(xvobject *) fromVal->addr), toVal);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: cvtWidgetToObject
|
|       Purpose: Convert a widget to an xvobject.
|
|		 CANT TELL WHAT THE INPUT/OUTPUT ARGUMENTS ARE  -SK
|         Input: display     -
|                args        -
|                num_args    -
|                fromVal     -
|                toVal       -
|
|        Output:
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Feb 16, 1993 14:32
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean cvtWidgetToObject(
   Display     *display,
   XrmValuePtr args,
   Cardinal    *num_args,
   XrmValuePtr fromVal,
   XrmValuePtr toVal,
   XtPointer   *closure_ret)
{
	cvtDone(xvobject, xvw_object(*(Widget *) fromVal->addr), toVal);
	return(TRUE);
}

/*------------------------------------------------------------
|  Routine Name: xvw_init_converters
|       Purpose: Initialize all the type converters.  This routine
|		 is called from xvw_initialize routine.
|          Date: Aug 15, 1992
|    Written By: Mark Young and John Salas
------------------------------------------------------------*/

void xvw_init_converters(void)
{
	XtAppContext  app_context;

	XtQstring     = XrmStringToQuark(XtRString);
	XtQfont       = XrmStringToQuark(XtRFont);
	XtQfontStruct = XrmStringToQuark(XtRFontStruct);
	XtQpixmap     = XrmStringToQuark(XtRPixmap);
	XtQdouble     = XrmStringToQuark(XtRDouble);
	XtQfloat      = XrmStringToQuark(XtRFloat);
	XtQfilename   = XrmStringToQuark(XtRFilename);

	app_context   = xvw_appcontext(NULL);
	XtAppSetTypeConverter(app_context, XtRPixel, XtRString,
		cvtPixelToString, ptosConvertArgs, knumber(ptosConvertArgs),
		XtCacheByDisplay, NULL);
	XtAppSetTypeConverter(app_context, XtRString, XtRPixel,
		cvtStringToPixel, ptosConvertArgs, knumber(ptosConvertArgs),
		XtCacheByDisplay, NULL);

	/*
	 *  Bitmap/Pixmap (to & from) filename
	 */
	XtAppSetTypeConverter(app_context, XtRPixmap, XtRFilename,
		cvtPixmapToFilename, pixtosConvertArgs,
		knumber(pixtosConvertArgs), XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRFilename, XtRPixmap,
		cvtFilenameToPixmap, pixtosConvertArgs,
		knumber(pixtosConvertArgs), XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRString, XtRPixmap,
		cvtFilenameToPixmap, pixtosConvertArgs,
		knumber(pixtosConvertArgs), XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRBitmap, XtRFilename,
		cvtPixmapToFilename, pixtosConvertArgs,
		knumber(pixtosConvertArgs), XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRFilename, XtRBitmap,
		cvtFilenameToPixmap, pixtosConvertArgs,
		knumber(pixtosConvertArgs), XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRString, XtRBitmap,
		cvtFilenameToPixmap, pixtosConvertArgs,
		knumber(pixtosConvertArgs), XtCacheNone, NULL);

	/*
	 *  Pixmap Mask (to & from) filename
	 */
	XtAppSetTypeConverter(app_context, XtRPixmapMask, XtRFilename,
		cvtPixmapMaskToFilename, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRFilename, XtRPixmapMask,
		cvtFilenameToPixmapMask, NULL, 0, XtCacheNone, NULL);

	/*
	 *  XImage (to & from) Pixmap
	 */
	XtAppSetTypeConverter(app_context, XtRPixmap, XtRXImage,
		cvtPixmapToXImage, NULL, 0, XtCacheByDisplay, NULL);
	XtAppSetTypeConverter(app_context, XtRXImage, XtRPixmap,
		cvtXImageToPixmap, NULL, 0, XtCacheByDisplay, NULL);
	XtAppSetTypeConverter(app_context, XtRBitmap, XtRXImage,
		cvtPixmapToXImage, NULL, 0, XtCacheByDisplay, NULL);
	XtAppSetTypeConverter(app_context, XtRXImage, XtRBitmap,
		cvtXImageToPixmap, NULL, 0, XtCacheByDisplay, NULL);

	/*
	 *  XImage (to & from) filename
	 */
        XtAppSetTypeConverter(app_context, XtRFilename, XtRXImage,
                cvtFilenameToXImage, ximtosConvertArgs,
                knumber(ximtosConvertArgs), XtCacheByDisplay, NULL);
        XtAppSetTypeConverter(app_context, XtRXImage, XtRFilename,
                cvtXImageToFilename, ximtosConvertArgs,
                knumber(ximtosConvertArgs), XtCacheByDisplay, NULL);

	/*
	 *  Font/Fontstruct (to & from) font name
	 */
	XtAppSetTypeConverter(app_context, XtRFont, XtRString,
		cvtFontToString, NULL, 0, XtCacheByDisplay, NULL);
	XtAppSetTypeConverter(app_context, XtRString, XtRFont,
		cvtStringToFont, NULL, 0, XtCacheByDisplay, NULL);
	XtAppSetTypeConverter(app_context, XtRFontStruct, XtRString,
		cvtFontStructToString, NULL, 0, XtCacheByDisplay, NULL);
	XtAppSetTypeConverter(app_context, XtRString, XtRFontStruct,
		cvtStringToFontStruct, NULL, 0, XtCacheByDisplay, NULL);

	/*
	 *  Int/Boolean/Short/String/Float/Double
	 */
	XtAppSetTypeConverter(app_context, XtRBoolean, XtRInt,
		cvtBooleanToInt, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRBoolean, XtRShort,
		cvtBooleanToShort, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRBoolean, XtRString,
		cvtBooleanToString, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRBoolean, XtRString,
		cvtBooleanToString, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRShort, XtRString,
		cvtShortToString, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRInt, XtRString,
		cvtIntToString, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRFloat, XtRString,
		cvtFloatToString, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRDouble, XtRString,
		cvtDoubleToString, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRShort, XtRInt,
		cvtShortToInt, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRDimension, XtRInt,
		cvtUShortToInt, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRPosition, XtRInt,
		cvtShortToInt, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRInt, XtRShort,
		cvtIntToShort, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRInt, XtRDimension,
		cvtIntToUShort, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRInt, XtRPosition,
		cvtIntToShort, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRDouble, XtRFloat,
		cvtDoubleToFloat, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRFloat, XtRDouble,
		cvtFloatToDouble, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRObject, XtRWidget,
		cvtObjectToWidget, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRWidget, XtRObject,
		cvtWidgetToObject, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRObjectList, XtRWidgetList,
		cvtObjectToWidget, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRWidgetList, XtRObjectList,
		cvtWidgetToObject, NULL, 0, XtCacheNone, NULL);
	XtAppSetTypeConverter(app_context, XtRString, XtRDouble,
		cvtStringToDouble, NULL, 0, XtCacheNone, NULL);

	/*
	 *  Cursor (to & from) string
	 */
	XtAppSetTypeConverter(app_context, XtRString, XtRCursor,
		cvtStringToCursor, cursorConvertArg, knumber(cursorConvertArg),
	        XtCacheByDisplay, NULL);
}
