 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>		    Gadget/Widget Action Handler Routines
   >>>>
   >>>>   These routines are used to add an action handler
   >>>>   on a object or gadget.  Since Xt cannot support
   >>>>   action handling on gadgets we do this by placing
   >>>>   the handler on the parent and then direct the
   >>>>   dispatching of the action ourselves.
   >>>>
   >>>>   Static:
   >>>>		  action_handler()
   >>>>  Private:
   >>>>
   >>>>   Public:  
   >>>>		  xvw_add_action()
   >>>>		  xvw_remove_action()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	


/*
 *  We need to keep track of all the gadget action handlers.  We would rather
 *  simply add it to the gadget's translation table, but since gadgets don't
 *  have one we are forced to save this information outself.
 */
typedef struct {
	xvobject    object;
	int	    action;
        kaddr       client_data;
        kfunc_void  routine;
} HandlerAction;

/*-----------------------------------------------------------
|
|  Routine Name: action_handler - intermediary action handler
|
|       Purpose: This is the intermediary Action Procedure that is
|		 initiated when the users invokes some specified action.
|
|         Input: widget - the widget that recieved the action
|		 event  - the event that triggered the action handler
|		 params - the action parameters
|		 num_params - the number of passed parameters
|        Output: 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By:
|          Date: 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void action_handler(
   Widget   widget,
   XEvent   *event,
   kstring  *params,
   Cardinal *num_params)
{
	klist	      *list;
	HandlerAction *entry;
	char	      *action;
	kaddr	      identifier;
	xvobject      object = xvw_object(widget);


	if (*num_params != 1)
	{
	   kinfo(KSTANDARD, "action_handler: Internal error.  Incorrect \
number of parameters found for internal action handler. ");
	   return;
	}

	/*
	 *  We have stored a linked list of action handler structures within the 
	 *  object.  The token'ized action string is used as the identifier for
	 *  each action handler structure, and the action handler information itself
	 *  is stored in the clientdata.  
  	 *
	 *  xvw_add_action () overrode the translation table with one that
	 *  will pass the identifier for the action handler structure of this action
	 *  as params[0].  Thus, all we have to do is get it back into integer
	 *  form and search the list for the action handler information structure
	 *  that corresponds to this action, and then fire the action handler
	 *  routine with the client_data stored inside. 
	 */
	identifier = (kaddr) atoi(params[0]);
	if ((list = klist_locate(object->actions, identifier)) == NULL)
	{
	   action = ktoken_to_string((int) identifier);
	   if (action == NULL) action = "(NULL)";
	   kinfo(KSTANDARD, "action_handler: Internal error.  Action \
registered for '%s', but action handler for '%s' could not be found.",
		action, action);
	   return;
	}
	entry = (HandlerAction *) klist_clientdata(list);
	entry->routine(object, entry->client_data, event);
}


/************************************************************
*
*  Routine Name: xvw_add_action - add an action handler to an object
*
*       Purpose: Adds an action handler to a GUI or visual object.
*		 When the specified action occurs, the action handler
*                will be called. 
*
*		 Since the X Toolkit cannot support action handling on 
*                gadgets, we support them by placing the action 
*                handler on the parent of the gadget and then directing the 
*                dispatch of the action handler directly.
*
*                The action handler \fImust\fP be associated with an object; 
*                only when the specified action(s) occur in the specified object
*                will the action handler be called (ie, the same action in 
*                another object will be ignored).
*
*                An action handler must be declared in the following form:\f(CW
*                !void action_handler(
*                !   xvobject object,
*                !   kaddr  client_data,
*                !   XEvent *event)\fP
*
*                !\fIobject -\fP
*                !    The object for which the action handler was invoked.
*                !    It will not be NULL.
*
*                !\fIclient_data -\fP
*                !    The pointer to the client data, used to pass parameters
*                !    from the application to the action handler.
*
*                !\fIevent -\fP
*                !    This is a pointer to the XEvent union which caused the
*                !    action handler to be invoked.  For details on the XEvent
*                !    union, see Chapter 8 of the \fIXlib Programming Manual\fP,
*                !    by Adrian Nye;  the definition of the XEvent union is on
*                !    page 232.
*
*         Input: object      - object on which to add action handler 
*                              (note that you may not pass NULL)
*                action      - the action which will invoke the action handler
*                routine     - the action handler to install
*                client_data - private data to be used by action handler
*                override    - TRUE if the action specified should override 
*                              any previously installed action handlers,
*                              FALSE if the action specified should augment
*                              any previously installed action handlers
*        Output: 
*       Returns:
*  Restrictions:
*    Written By: Mark Young & Danielle Argiro
*          Date: Apr 27, 1993 12:33
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

void xvw_add_action(
   xvobject   object,
   char       *action,
   kfunc_void routine,
   kaddr      client_data,
   int        override)
{
	char temp[KLENGTH];
	HandlerAction *handler;
	XtTranslations translation;
	static int initialized = FALSE;
	static XtActionsRec actiontable[] =
	{
	   { "action_handler",  action_handler },
	};

	/*
	 * if this is the first time calling xvw_add_action(), need to
	 * call XtAppAddActions() to get our intermediate action_handler()
	 * routine to be fired by any actions that occur.
	 */
	if (!initialized)
	{
	   initialized = TRUE;
	   XtAppAddActions(xvw_appcontext(NULL), actiontable, knumber(actiontable));
	}

	/*
	 *  Right now, we are not supporting installation of 
	 *  action handlers on gadgets...
	 */
	if (!XtIsWidget(xvw_widget(object)))
	   return;

	/*
	 *  Create the temporary action handler structure that will contain
	 *  all the information needed to fire the action handler: 
         *  1 - the tokenized string representing the action (eg, "<Key>D")
         *  2 - the object on which the action handler is installed
	 *  3 - the action handler routine to be fired
	 *  4 - the client data to be passed to action handler routine
         *
         *  Add the action handler structure to the linked list of such 
	 *  structure stored in the object.  A linked list is used because
	 *  a single object may have multiple action handlers installed.
	 *  The identifier to look up the structure from the linked list
	 *  is the tokenized string representing the action.
	 */
	handler = (HandlerAction *) kmalloc(sizeof(HandlerAction));
	handler->action      = kstring_to_token(action);
	handler->object      = object;
	handler->routine     = routine;
	handler->client_data = client_data;
	object->actions = (kaddr) klist_add(object->actions, 
				    (kaddr) handler->action, handler);

	/*
	 *  Xt Action handlers can only pass string parameters, not clientdata 
	 *  (as is used in callbacks and event handlers).  To force K2 action
	 *  handlers to support client data, we pass the identifier of the
	 *  action handler structure (created above) thru the translation table.  
	 *  When the intermediate action_handler() routine is fired by Xt,
	 *  we can pull out the identifier from the "params" argument that is
	 *  passed to action_handler() by Xt, look up our action handler 
	 *  information from the linked list of action handler structures 
	 *  stored with the object, and fire the specified action handler routine.
	 *
	 */
	ksprintf(temp, "%s: action_handler(%d)", action, handler->action);
	translation = XtParseTranslationTable(temp);

	/*
	 *  Use Xt to over-ride the translation table for this action with
	 *  our translation table that passes the action of interest,
	 *  and the identifier of the action handler information as params[0].
	 */
	if (override)
	   XtOverrideTranslations(xvw_widget(object), translation);
	else
	   XtAugmentTranslations(xvw_widget(object), translation);
}

/************************************************************
*
*  Routine Name: xvw_remove_action - remove an action handler from an object
*
*       Purpose: Removes an action handler from a GUI or visual object.
*
*         Input: object      - object from which to remove action handler 
*                action      - the action which was invoking the action handler
*                routine     - the action handler to de-install
*                client_data - pointer to private application data that
*                              was being passed to action handler
*        Output:
*       Returns:
*  Restrictions:
*    Written By: Mark Young & Danielle Argiro
*          Date: Apr 27, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

/* ARGSUSED */
void xvw_remove_action(
   xvobject   object,
   char       *action,
   kfunc_void routine,
   kaddr      client_data)
{
	klist  *list;
        Widget widget = xvw_widget(object);
	HandlerAction *entry = NULL;

        /*
         *  If the object is a gadget, we would have installed the action
	 *  handler on its parent, since the X Toolkit will allow a gadget 
         *  to have an action handler installed on it, but the X Toolkit
         *  will then simply ignore it.  Therefore, if the object is a gadget,
         *  we need to remove the action handler from the parent; if the object
         *  is a widget, simply remove the action handler from the widget.
         */
	if (!XtIsWidget(widget))
           widget = XtParent(widget);

        /*
         *  Go through our action handler list, find the action handler,
         *  and remove it from the list.
         */
        list = object->actions;
        while (list != NULL)
        {
           int token = (int) klist_identifier(list);

	   if (token == kstring_to_token(action))
           {
	      /*
	       *  not that this is a "feature"; calling xvw_remove_action()
	       *  on an object will uninstall ALL action handlers on the
	       *  object, because that's what XtUninstallTranslations() does.
	       */
              XtUninstallTranslations(xvw_widget(object));
	      entry = (HandlerAction *) klist_clientdata(list);
           }
           list = klist_next(list);
        }

	
	/*
	 *  delete action handler info structure from object's list
	 */
	if (entry != NULL)
            object->actions = klist_delete(object->actions, entry);
}
