 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		       Warn Display			      <<<<
   >>>>                                                       <<<<
   >>>>  Private:      xvu_warn()			      <<<<              
   >>>>                                                       <<<<
   >>>>   Static:      xvu_warn_ok()		              <<<<              
   >>>>                                                       <<<<
   >>>>   Public:      xvu_warn_wait()			      <<<<              
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	

static int  xvu_warn_done;

/*-----------------------------------------------------------
|
|  Routine Name: xvu_warn_ok()
|
|       Purpose: Destroys the warn object when the user clicks 
|                on the acknowledgement button.
|
|         Input: object      - the acknowledgement button
|                client_data - toplevel of warn object
|		 call_data   - unused
|
|        Output:
|    Written By: Danielle Argiro & Mark Young
|          Date: Jan 4, 1994
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void xvu_warn_ok(
   xvobject object,
   kaddr  client_data,
   kaddr  call_data)
{
	xvu_warn_done = TRUE;
}

/*-----------------------------------------------------------------
|
|  Routine Name: xvu_warn
|
|       Purpose: xvu_warn produces standardized warn messages for
|                X-based Khoros library routines and X based Khoros 
|                applications.  It will be set up as the warn
|		 handler for kwarn in all X-based Khoros routines by
|                xvf_initialize().
|
|         Input: toolbox   - name of toolbox (if applicable)
|                program   - name of program (if applicable, NULL otherwise)
|                library   - name of library (if applicable, NULL otherwise)
|                routine   - name of routine
|                message   - grammatically correct, clear explanation of
|                            the warn that occurred
|        Output:
|    Written By: Danielle Argiro & Mark Young
|          Date: Jan 4, 1994
| Modifications:
|
--------------------------------------------------------------*/

int xvu_warn(
   char *toolbox,
   char *program,
   char *library,
   char *routine,
   char *message)
{
	knotify handler;
	char buffer[25*KLENGTH], title[KLENGTH];

	buffer[0]  = '\0';
	title[0] = '\0';

	handler = kset_warnhandler(NULL);
	if (toolbox != NULL) 
	{
	    kstrcat(buffer, "Toolbox: ");
	    kstrcat(buffer, toolbox);
	}
	if (program != NULL) 
	{
	    kstrcat(buffer, "\nProgram: ");
	    kstrcat(buffer, program);
	}

	if (library != NULL) 
	{
	    kstrcat(buffer, "\nLibrary: ");
	    kstrcat(buffer, library);
	}

	if (routine != NULL)
	{
	    kstrcat(buffer, "\nRoutine: ");
	    kstrcat(buffer, routine);
	}

	kstrcat(buffer, "\n");

	if (message != NULL)
	{
	    kstrcat(buffer, "\n");
	    kstrcat(buffer, message);

	    kstrcat(title, "Warning: ");
	}
	else
	{
	    kstrcat(buffer, "\n");
	    kstrcat(title, "Warning:");
	}
	xvu_warn_wait(buffer, title, NULL);
	(void) kset_warnhandler(handler);
	return(TRUE);
}

/**************************************************************
*
*  Routine Name: xvu_warn_wait - pop up warn object (1 button); wait for
*				 acknowledgement
*
*       Purpose: Creates a pop-up warn object which must be acknowledged 
*                by the user before control is returned to the application 
*                program.  
*
*                IMPORTANT NOTE: to be consistent with the standards
*                of the Khoros Software development system, you should really
*                be calling kwarn(), not xvu_warn_wait().  
*
*         Input: warn_mesg   - string describing warn message
*                warn_label  - short label for top of warn object; passing 
*                               NULL will result in default of "Warning".
*		 button_label - label for acknowledgment button; passing
*                               NULL will result in default of "Ok".
*        Output:
*	Returns: Returns FALSE if it failed to create the warn object
*		 otherwise waits for user to acknowledge warn message,
*    Written By: Danielle Argiro & Mark Young
*          Date: Jan 4, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
**************************************************************/

int xvu_warn_wait(
   char *warn_mesg,
   char *warn_label,
   char *button_label)
{
	xvobject warn;
	knotify  handler;
	char     name[KLENGTH];


        /*
         *  xvu_warn_wait supports non-X displays
         */
	xvu_warn_done = FALSE;
	if (xvw_display(NULL) == NULL)
	{
	    kinfo(KSTANDARD, "%s\n", warn_mesg);
	    return(TRUE);
	}

	/*
	 *  xvw_create_warn creates the warn object pop-up
	 */
	xvw_busy(NULL, TRUE);
	handler = kset_warnhandler(NULL);

	(void) ksprintf(name, "Warning for %s", kprog_get_program());
	if ((warn = xvw_create_warn(NULL, name)) == NULL)
	{
	   (void) kset_warnhandler(handler);
	    xvw_busy(NULL, FALSE);
	   return(FALSE);
	}
	xvw_set_attributes(warn,
		XVW_WARN_MESSAGE, warn_mesg,
		XVW_WARN_LABEL,   warn_label ? warn_label : "Warning",
		XVW_WARN_BUTTON_LABEL, button_label ? button_label : "Ok",
		NULL);
	xvw_add_callback(warn, XVW_WARN_CALLBACK, xvu_warn_ok, NULL);

	(void) kset_warnhandler(handler);
	while (!xvu_warn_done)
	    xvw_process_event();  

	xvw_busy(NULL, FALSE);
	return(TRUE);
}
