 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		  Pop-Up Prompt xvobject Utilities	      <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                xvu_prompt()                           <<<<
   >>>>                xvu_create_prompt()		      <<<<              
   >>>>   Static:                                             <<<<
   >>>>                xvu_prompt_cont()		      <<<<              
   >>>>                xvu_prompt_stop()		      <<<<              
   >>>>   Public:                                             <<<<
   >>>>                xvu_prompt_wait()		      <<<<              
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	

static int  xvu_prompt_count = 0;
static int  xvu_prompt_done  = FALSE;
static int  xvu_prompt_sig;

static void xvu_prompt_cont PROTO((xvobject, kaddr));
static void xvu_prompt_stop PROTO((xvobject, kaddr));

/*-----------------------------------------------------------
|
|  Routine Name: xvu_prompt
|
|       Purpose: xvu_prompt produces standardized prompting messages for
|                X-based Khoros library routines and X based Khoros 
|                applications.  It should be set up as the prompt
|		 handler for all X-based promptings.
|
|         Input: yes_response  - prompt to put in the yes button
|		 no_response   - prompt to put in the no button
|		 default_val   - default value
|                message       - grammatically correct, clear explanation of
|                                the error that occurred
|        Output: none
|    Written By: Danielle Argiro and Steven Jorgensen
|          Date: Jul 29, 1992 18:39
|      Verified:
|  Side Effects:
| Modifications:
|
------------------------------------------------------------*/
int xvu_prompt(
   char *yes_response,
   char *no_response,
   int  default_val,
   char *message)
{
	int	status;

	if (message == NULL)
	{
	    errno = KNULL_PARAMETER;
	    kerror("xvutils", "xvu_prompt", 
		   "Passing NULL message to xvu_prompt");
	    return(default_val);
	}
	status = xvu_prompt_wait(message, "Please Respond", yes_response,
			       no_response);
	return(status);
}

/************************************************************
*
*  Routine Name: xvu_prompt_wait - pop up prompting object (2 choices); wait for response
*
*       Purpose: Creates and maps a pop-up prompting object which has 
*                two buttons representing two choices. The user must 
*                click on one of the buttons;  a status representing 
*                which of the two buttons was chosen will be returned 
*                to the application.  The user must click on one of the 
*                two buttons before control will be returned to the 
*                application program.
*
*         Input: prompting     - prompting string 
*                label       - short label for top of prompting object;
*			       NULL will produce default of "WARNING".
*		 cont_label  - label for button representing value of 1 (TRUE);
*                              NULL will produce default of "Cont".
*                stop_label  - label for second representing value of 0 (FALSE);
*                              NULL will produce default of "Stop".
*        Output:
*	Returns:  1 (TRUE)  if user selected button with cont_label (Yes) 
*                 0 (FALSE) if user selected button with stop_label (No) 
*
*  Restrictions: 
*    Written By: Danielle Argiro & Stephanie Hallett
*          Date: Jul 9, 1992 13:51
*      Verified:
*  Side Effects:
* Modifications: Converted from Khoros 1.0 (DA)
*
*************************************************************/

int xvu_prompt_wait(
   char *prompting,
   char *label,
   char *cont_label,
   char *stop_label)
{
        knotify handler;
	xvu_prompt_done = FALSE;

	handler = kset_prompthandler(NULL);
	if (!(xvu_create_prompt(prompting, label, cont_label, stop_label)))
	    return(FALSE);

	(void) kset_prompthandler(handler);
	while (!xvu_prompt_done)
	    xvw_process_event(); 

	return(xvu_prompt_sig);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvu_create_prompt()
|
|       Purpose: Creates the prompting object. 
|
|         Input: prompt_string  - the prompting message 
|                prompt_label   - short label for top of object
|                               NULL will give default of "Warning"
|		 cont_label   - label for acknowledgement button returning 1
|                               NULL will give default of "Cont" 
|		 stop_label   - label for acknowledgement button returning 0
|                               NULL will give default of "Stop"
|
|        Output: Returns TRUE on success, FALSE on failure
|    Written By: Danielle Argiro & Stephanie Hallett
|          Date: Jul 9, 1992
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/

#define TEXT_WIDTH  35
#define TEXT_HEIGHT 7

xvobject cont_obj; 	/* button to select to continue (1) */
xvobject stop_wid; 	/* button to select to stop action (0) */

int xvu_create_prompt(
   char *prompt_string,
   char *prompt_label,
   char *cont_label,
   char *stop_label)
{
	xvobject toplevel;	/* toplevel object for message */
	xvobject back;		/* backplane for message */
	xvobject label;	        /* label describing type of message */
	xvobject mesg;	        /* text object containing prompting message */
	float    width, height, cont_width, stop_width, label_width; 
	char     name[KLENGTH];
	char     *use_prompt, *use_cont, *use_stop;

	/*
	 * create identifying object name
	 */
	ksprintf(name, "%d_prompt", xvu_prompt_count++);

	if (cont_label == NULL)
	    use_cont = kstring_copy("Cont", NULL);
	else use_cont = kstring_copy(cont_label, NULL);
	cont_width = (float) (kstrlen(use_cont)+1);

	if (stop_label == NULL)
	    use_stop = kstring_copy("Stop", NULL);
	else use_stop = kstring_copy(stop_label, NULL);
	stop_width = (float) (kstrlen(use_stop)+1);
   
	if (prompt_label == NULL)
	    use_prompt  = kstring_copy("Warning:", NULL);
	else use_prompt = kstring_copy(prompt_label, NULL);
	label_width = (float) (kstrlen(use_prompt)+1);

	width =  (float) kmin(kmax3(TEXT_WIDTH, xvu_figure_width(prompt_string),
                  (int) (cont_width + stop_width + label_width)), 60);

	/*
	 * create the prompting box's toplevel object and add it
	 * to the list of toplevels that is used with journal playback.
	 */
        toplevel = xvw_create_transient_shell(name, NULL, NULL);

	/* 
	 * create the backplane object 
	 */
	back = xvw_create_manager(toplevel, "back");

	/* 
	 * create the label object 
	 */
	label = xvw_create_label(back, "label");
	xvw_set_attributes(label,
		XVW_LABEL,            use_prompt,           /* label         */
		XVW_BELOW,            NULL,  	          /* top line      */
		XVW_RIGHT_OF,         NULL,  	          /* to the Left   */
		XVW_CHAR_WIDTH,       label_width,        /* set width     */
                XVW_BORDER_WIDTH,     0,                  /* no border     */
                XVW_MAP_WHEN_MANAGED, TRUE,               /* mapped        */
		XVW_LABEL_JUSTIFY,    KLABEL_JUSTIFY_CENTER, /* centered      */
                NULL);
	/* 
	 * create "Stop" acknowledgment button in upper righthand corner
	 */
	stop_wid = xvw_create_button(back, "Stop");
        xvw_set_attributes(stop_wid,
		XVW_LABEL,            use_stop,    /* button label   */
                XVW_CHAR_WIDTH,       stop_width,  /* set width      */
		XVW_BELOW,            NULL,  	   /* top line       */
		XVW_LEFT_OF,          NULL,        /* in R corner    */
                NULL);
	xvw_add_callback(stop_wid, XVW_BUTTON_SELECT, 
			 xvu_prompt_stop, toplevel);

	/* 
	 * create "Cont" acknowledgment button 
	 */
	cont_obj = xvw_create_button(back, "Cont");
        xvw_set_attributes(cont_obj,
		XVW_LABEL,        use_cont,       /* button label    */
		XVW_CHAR_WIDTH,   cont_width,     /* set width       */
		XVW_BELOW,        NULL,  	  /* top line        */
		XVW_LEFT_OF,      stop_wid,       /* left of stop    */
		NULL);
	xvw_add_callback(cont_obj, XVW_BUTTON_SELECT, 
		         xvu_prompt_cont, toplevel);


	/* 
	 * create text object w/ prompting mesg
	 */
	height = (float) xvu_figure_height(prompt_string, (int) width);
	if (height < TEXT_HEIGHT) height = (float) TEXT_HEIGHT;
        mesg = xvw_create_text(back, "prompt_mesg");
        xvw_set_attributes(mesg,
		XVW_TEXT_MULTILINE,   TRUE,	              /* multi-line   */
		XVW_TEXT_STRING,      kstrdup(prompt_string), /* text         */
                XVW_CHAR_WIDTH,       width,                  /* set width    */
                XVW_CHAR_HEIGHT,      height,                 /* set height   */
                XVW_MAP_WHEN_MANAGED, TRUE,                   /* mapped       */
                XVW_BELOW,            cont_obj,               /* below label  */
                XVW_TEXT_EDIT_TYPE,   KTEXT_READ,             /* read-only    */
                XVW_TEXT_WRAP,        KTEXT_WRAP_WORD,        /* wrap on word */
                NULL);

	/*
         *  map object & return
         */
	xvw_add_protocol(toplevel, "WM_DELETE_WINDOW", xvu_prompt_cont,
			 toplevel);

	xvw_place(toplevel,stop_wid);
	kfree(use_prompt);
	kfree(use_stop);
	kfree(use_cont);
	return(TRUE);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvu_prompt_cont()
|
|       Purpose: Destroys prompting object when the user clicks on
|                the "Cont" button; returns status of 1.
|
|         Input: object      - the "Cont" acknowledgement button
|                client_data - stores the toplevel object
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: Jul 9, 1992
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/
/* ARGSUSED */
static void xvu_prompt_cont(
   xvobject object,
   kaddr    client_data)
{
	xvobject toplevel = (xvobject) client_data;

	xvw_remove_callback(cont_obj, XVW_BUTTON_SELECT, 
		            xvu_prompt_cont, client_data);
	xvw_remove_callback(stop_wid, XVW_BUTTON_SELECT, 
		            xvu_prompt_stop, client_data);

	xvw_remove_protocol(toplevel, "WM_DELETE_WINDOW", 
			    xvu_prompt_cont, toplevel);
	xvw_unmap(toplevel);
	xvw_destroy(toplevel);
	xvu_prompt_done = TRUE;
	xvu_prompt_sig = 1;

}

/*-------------------------------------------------------------
|
|  Routine Name: xvu_prompt_stop()
|
|       Purpose: Destroys prompting object when the user clicks on
|                the "Stop" button; returns status of 0.
|
|         Input: object      - the "Stop" acknowledgement button
|                client_data - stores the toplevel object
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: Jul 9, 1992
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/
/* ARGSUSED */
static void xvu_prompt_stop(
   xvobject object,
   kaddr    client_data)
{
	xvobject toplevel  = (xvobject) client_data;

	xvw_remove_callback(cont_obj, XVW_BUTTON_SELECT, 
		            xvu_prompt_cont, client_data);
	xvw_remove_callback(stop_wid, XVW_BUTTON_SELECT, 
		            xvu_prompt_stop, client_data);
	xvw_unmap(toplevel);
	xvw_destroy(toplevel);
	xvu_prompt_done = TRUE;
	xvu_prompt_sig = 0;

}

