 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		       List xvobjects                         <<<<
   >>>>                                                       <<<<
   >>>>  Private:					      <<<<
   >>>>   Static:					      <<<<
   >>>>                create_multsel_list_object()	      <<<<              
   >>>>                list_multi_select()	       	      <<<<              
   >>>>                add_user_defined_item()  	      <<<<              
   >>>>                use_list()   	      	              <<<<              
   >>>>                cancel_list()   	      	              <<<<
   >>>>   Public:					      <<<<
   >>>>                xvu_run_list_multsel_wait()	      <<<<              
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	

#define MaxNumLinesInViewport 35

static int xvu_list_size     = 0; 
static int xvu_list_count    = 0;
static int xvu_list_done     = FALSE;
static int xvu_use_list      = FALSE;
static int xvu_duplicates_ok = FALSE;

static void list_multi_select        PROTO((xvobject, kaddr, kaddr));
static void add_user_defined_item    PROTO((xvobject, kaddr, kaddr));
static void use_list                 PROTO((xvobject, kaddr, kaddr));
static void cancel_list              PROTO((xvobject, kaddr, kaddr));

static xvobject create_multsel_list_object   PROTO((char **, int, char *, 
					  	    char *, int, int));

xvu_list_entry  *list_head, *list_tail;
static char    **user_list, **display_list;

/************************************************************
*
*  Routine Name: xvu_run_list_multsel_wait - display list, wait for multiple 
*                                            choices and acknowledgement
*
*       Purpose: Takes an array of strings, and uses them to create a pop-up
*                list object.  The user is allowed to select as many different
*                strings as desired from the list; those strings are then 
*                returned to the caller with their indices in an array of 
*                xvu_list_struct's.  If the user clicks on the "Cancel" button,
*                NULL is returned.
*
*                Control is not returned to the application program until
*                the user chooses an item from the list, or clicks on "cancel".
*
*                If the "duplicates_ok" flag is passed as FALSE, the user
*                will only be allowed to select any given item from the list
*                once.  Selected items will be marked with a star.  For example,
*                the item "depth" would be displayed as "depth" if it had never
*                been selected, but would be displayed as "* depth" after it
*                was selected.  A second click on the entry would cause it to 
*                be de-selected, and again it would be displayed as "depth".  
*                The array of xvu_list_struct's returned will have all unique
*                elements;  elements will appear according to the order in
*                which selections were made by the user.
*
*                If the "duplicates_ok" flag is passed as TRUE, the user will
*                be allowed to select any item from the list as many times as
*                desired.  Each selection from the list will cause that item 
*                to have the number in front of it incremented.  For example, 
*                the item "depth" would be displayed as "depth" if it had never
*                been selected, but "(1) depth" after the first time it was 
*                selected, and "(2) depth" after the second time it was 
*                selected.  The array of xvu_list_struct's returned will have 
*                identical elements for each item that is selected multiple 
*                times.  The elements of the xvu_list_struct array will appear
*                in the order in which the items were selected by the user.
*                For example, if the user chose "depth" twice, and then "width",
*                and then "depth" again before clicking on "Use", the array
*                of xvu_list_structs returned would appear in the following
*                order (identified by list_struct->string): depth, depth,
*                width, depth.
*
*                If the "user_defined" flag is passed as TRUE, a string object
*                will be created at the bottom of the list object where the
*                user will be allowed to enter their own new value for the list.
*                When the user hits <cr> following their newly defined list
*                item, the new value will be added to the list, and may 
*                subsequently be selected, either once (if duplicates are
*                not allowed), or many times (if duplicates are allowed).
*
*         Input: list         - array of strings to be displayed in the list
*		 size         - size of the 'list' array
*		 prompt       - prompt to display the list object
*		 label        - label to display at top of list object
*		 user_defined - flag indicating if a string object should be
*			        included to allow the user to enter a string
*				that is not part of the list array.
*		 duplicates_ok - pass TRUE if the user is allowed to
*				 select the same item twice, FALSE otherwise
*
*        Output: num - returns the number of items selected from the list.
*
*	Returns: An array of xvw_list_structs containing the strings and the 
*                indices of those strings describing the selections made from 
*                the list by the user.  Returns NULL if the user clicks on 
*                "Cancel".
*
*
*  Restrictions:
*    Written By: Danielle Argiro & Mark Young
*          Date: May 19, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xvw_list_struct
**xvu_run_list_multsel_wait(
   char *list[],
   int  size,
   char *prompt,
   char *label,
   int  user_defined,
   int  duplicates_ok,
   int  *num)
{
	char   buffer[KLENGTH];
	int    i, num_selections;
	xvobject toplevel;
	xvu_list_entry  *selection, *temp;
	xvw_list_struct **selections;

	xvu_list_done     = FALSE;
	xvu_use_list      = FALSE;
	xvu_duplicates_ok = duplicates_ok;

	xvu_list_size  = size;
	user_list      = (char **) kmalloc(sizeof(char *) * size);
	display_list = (char **) kmalloc(sizeof(char *) * size);

	for (i = 0; i < size; i++)
	{
	    if (xvu_duplicates_ok)
	       ksprintf(buffer,"      %s", list[i]);
	    else ksprintf(buffer,"   %s", list[i]);

	    display_list[i] = kstrdup(buffer);
	    user_list[i] = kstrdup(list[i]);
	}

	list_head = list_tail = NULL;
	toplevel = create_multsel_list_object(display_list, size, prompt,
					      label, user_defined,
					      duplicates_ok);

	if (toplevel == NULL) return(NULL);

	while (!xvu_list_done)
	   xvw_process_event();      

       /*
        * delete the list object's toplevel from the list 
        * used by journal playback before destroying it
        */
	xvw_unmap(toplevel);
	xvw_destroy(toplevel); 

	if ((list_head == NULL) && (xvu_use_list == TRUE))
	{
	   *num = 0;
	   return(NULL);
	}
	else if (list_head == NULL)
	{
	   *num = -1;
	   return(NULL);
        }

	/*
	 *  Get the number of entries in the selection list.  This will be used
	 *  in malloc'ing the proper amount of memory for returning the list
	 *  to the user.
	 */
	selection = list_head;
	num_selections = 0;
	while (selection != NULL)
	{
	   num_selections++;
	   selection = selection->next;
	}

	/*
	 *  malloc the selections to be returned and then copy the entries.
	 *  While copying the entries we free the xvu_list_entry structures.
	 */
	selections = (xvw_list_struct **) kmalloc(num_selections *
		      sizeof(xvw_list_struct *));

	selection = list_head;
	for (i = 0; i < num_selections; i++)
	{
	   selections[i] = (xvw_list_struct *) selection->entry;
	   temp = selection;
	   selection = selection->next;
	   if (temp != NULL)
	      kfree(temp);
	}

	/*
	 *  return the number of elements and the selections
	 */
	*num = num_selections;
	return(selections);
}

/*------------------------------------------------------------
|
|   Routine Name: create_multsel_list_object()
|
|        Purpose: Creates the list object.  It takes an array of strings, 
|                 and uses each string as an item in the list. The prompt 
|                 and label strings are used to complete the list object.
|
|          Input: list         - the array of strings to be displayed as 
|                                elements of the list.
|		  size         - the size of the 'list' array
|		  prompt       - prompt to display at top of the list object
|		  label        - the label of the list object
|		  user_defined - flag indicating if a string object should be
|			        included to allow the user to enter a string
|				that is not part of the list array.
|
|        Output: Returns the toplevel of the list object
|
|    Written By: Danielle Argiro & Mark Young
|          Date: May 19, 1994
| Modifications: 
|
|
--------------------------------------------------------------------*/

static xvobject create_multsel_list_object(
   char **list,
   int  size,
   char *prompt,
   char *label,
   int  user_defined,
   int  duplicates_ok)
{
	xvobject toplevel;	  /* toplevel object */
	xvobject back;		  /* backplane for list object */
	xvobject label_object;	  /* label object */
	xvobject prompt_object;	  /* prompt object */
	xvobject list_object;	  /* compound list object */
	xvobject actual_list;	  /* list component of compound list object */
	xvobject cancel_object;	  /* button to cancel list object*/
	xvobject string_wid;      /* button to enter user-specified string */
	xvobject use_object;	  /* button to register choices */
	xvobject offset = NULL;   /* offset object */
	char	 *spec_buffer;
	char name[KLENGTH];
	int    i, max_strlen; 
	float  width, height; 

	if (list == NULL)
	{  
	    errno = KNULL_PARAMETER;
 	    kerror("xvutils", "create_multsel_list_object","'list' cannot be null");
	    return(NULL);
	}

	spec_buffer = (char *) kcalloc(1, sizeof(char)*KLENGTH);
	kmemset(spec_buffer, 0, KLENGTH);

	if (label == NULL)
	    label = kstring_copy(" ", NULL);

	if (prompt == NULL)
	    prompt = kstring_copy("Choose from:", NULL);

	/*
	 * create identifying object name
	 */
	ksprintf(name, "%d_list", xvu_list_count++);

	/* 
	 * create the list's toplevel object and add it to the
	 * list used by journal playback.
	 */
	toplevel = xvw_create_transient_shell(name, NULL, NULL);
	xvw_add_protocol(toplevel, "WM_DELETE_WINDOW", 
			 cancel_list, toplevel);

	/*
	 * Compute the longest selection length. Then set the width & 
         * height of the manager backplane accordingly.
	 */
	max_strlen = kmax(kstrlen(prompt), kstrlen(label) + kstrlen("cancel"));
	for (i = 0; i < size; i++)
	{
	    if (kstrlen(list[i]) > max_strlen) 
		max_strlen = kstrlen(list[i]);
	}
	width  = (float) max_strlen;
	if (size > MaxNumLinesInViewport)
	    height = (float) MaxNumLinesInViewport;
	else  height = (float) size;

	/* 
	 * create backplane object 
	 */ 
	back = xvw_create_manager(toplevel, "back");
	xvw_set_attribute(back, XVW_CHAR_WIDTH, width);

	/* 
	 * need a "Use" button in upper left corner
	 */
	use_object = xvw_create_button(back, "use");
        xvw_set_attributes(use_object,
		XVW_LABEL,	 "Use",  /* label                           */
		XVW_RIGHT_OF,     NULL,  /* to far L                        */
		XVW_BELOW,        NULL,  /* upper far R                     */
		NULL);
	offset = use_object;

	/* 
	 * create cancel button to upper right corner
	 */
	width = 7.0;
	cancel_object = xvw_create_button(back, "cancel");
        xvw_set_attributes(cancel_object,
		XVW_LABEL,         "Cancel",       /* button label           */
		XVW_LEFT_OF,       NULL,           /* to far R               */
		XVW_BELOW,         NULL,           /* upper far R            */
		NULL);
	xvw_add_callback(cancel_object, XVW_BUTTON_SELECT, 
			 cancel_list, toplevel);

	/* 
	 * create the label object 
	 */
        label_object = xvw_create_label(back, "label_object");
        xvw_set_attributes(label_object,
		XVW_LABEL,        label,       /* label                       */
		XVW_BORDER_WIDTH, 0,           /* no border                   */
		XVW_RIGHT_OF,     offset,      /* R of use (multsel) or NULL  */
		XVW_BELOW,        NULL,        /* at the top                  */
		XVW_LEFT_OF,      cancel_object, /* L of cancel               */
		NULL);

	/*
	 * create the prompt object
	 */
	prompt_object = xvw_create_label(back, "prompt");
        xvw_set_attributes(prompt_object,
		XVW_LABEL,        prompt,         /* label                   */
		XVW_BORDER_WIDTH, 0,              /* no border               */
		XVW_BELOW,        cancel_object,  /* under "cancel", "use"   */
	        XVW_LEFT_OF,      NULL,           /* centered                */
	        XVW_RIGHT_OF,     NULL,           /* centered                */
		NULL);

	/*
	 *  create the compound list object.
	 */
	list_object = xvw_create_list(back, "list");
	xvw_set_attributes(list_object, 
		   XVW_BELOW,        prompt_object,  /* below prompt      */
		   XVW_TACK_EDGE,    KMANAGER_TACK_HORIZ,/* centered      */
	           XVW_CHAR_HEIGHT,  height + 1.0,  
		   NULL);

	/*
	 *  need the actual list from the list object in order to
         *  set list contents and install the callback for list selection
	 */
	actual_list = xvw_retrieve_list(list_object);
	xvw_change_list(actual_list, list, size, FALSE);
	xvw_add_callback(actual_list, XVW_LIST_ITEM_SELECT, 
			 list_multi_select, (kaddr) duplicates_ok);

	/* 
	 *  if specified, create text object in which user 
	 *  may specify their own string response 
	 */
	if (user_defined)
	{
	    string_wid = xvw_create_text(back, "user_defined");
	    xvw_set_attributes(string_wid,
		XVW_TEXT_STRING, spec_buffer,       /* text              */
		XVW_TACK_EDGE,	 KMANAGER_TACK_HORIZ,/* tack horizontal   */
		XVW_BELOW,       list_object,       /* under list object */
		NULL);

	    xvw_add_action(string_wid, "<Key>Return", 
			   add_user_defined_item, list_object, TRUE);
	}

	xvw_add_callback(use_object, XVW_BUTTON_SELECT, use_list, NULL);

	xvw_place(toplevel, NULL);
	kfree(spec_buffer);
	return(toplevel);
}

/*------------------------------------------------------------
|
|  Routine Name: list_multi_select()
|
|       Purpose: Adds the selected item to the list of 
|		 selections from the list object;  marks the
|                selection in the list object with a '*' (if
|                duplicated entries are not allowed) or with
|                a number (if duplicated entries are allowed).
|
|         Input: object      - the actual list component of compound list object
|                client_data - contents of the text object
|                call_data   - returns the item that was chosen
|
|        Output: none
|    Written By: Danielle Argiro & Mark Young
|          Date: May 19, 1994
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void list_multi_select(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	int	count = 1;
	char	buffer[10];
	xvu_list_entry *list, *new;

	/* cast the pointer to the Callback data */
	xvw_list_struct *latest = (xvw_list_struct *) call_data;
	int duplicates_ok = (int) client_data;

	list = list_head;
	while (list != NULL)
	{
	   if (list->entry->list_index == latest->list_index)
	   {
	      if (duplicates_ok == FALSE)
	      {
		 display_list[latest->list_index][1] = ' ';
	         xvw_change_list(object, (String *) display_list, 
				 xvu_list_size, FALSE);

		 xvu_delete_list_entry(list);
	         return;
	      }
	      else
	      {
		 list->count++;
		 count = list->count;
		 break;
	      }
	   }
	   else list = list->next;
	}

	new = xvu_add_list_entry();
	new->entry->list_index = latest->list_index;
	new->entry->string = kstring_copy(user_list[latest->list_index], NULL);

	if (duplicates_ok == FALSE)
	   display_list[latest->list_index][1] = '*';
	else
	{
	   (void) ksprintf(buffer,"(%1d)", count);
	   (void) kstrncpy(display_list[latest->list_index],
			   buffer, kstrlen(buffer));
	}

	xvw_change_list(object, (String *) display_list, 
			xvu_list_size, FALSE);
}

/*------------------------------------------------------------
|
|  Routine Name: add_user_defined_item()
|
|       Purpose: Adds the user-defined item to the list of 
|		 selections from the list object.
|
|         Input: object      - the ascii text object
|                client_data - not used
|                call_data   - unused
|
|        Output: none
|    Written By: Danielle Argiro & Mark Young
|          Date: May 19, 1994
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void add_user_defined_item(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	char           buffer[KLENGTH];
	char           *string_return;
	xvobject       list_object = (xvobject) client_data;
	xvobject       actual_list;

	xvw_get_attribute(object, XVW_TEXT_STRING, &string_return);

	if (xvu_duplicates_ok)
	    ksprintf(buffer,"      %s", string_return);
	else ksprintf(buffer,"   %s", string_return);

	xvu_list_size++;
	display_list = (char **) krealloc(display_list,
					    sizeof(char *) * xvu_list_size);
	display_list[xvu_list_size-1] = kstrdup(buffer);

	user_list = (char **) krealloc(user_list,
				       sizeof(char *) * xvu_list_size);
	user_list[xvu_list_size-1] = kstrdup(string_return);

	actual_list = xvw_retrieve_list(list_object);
	xvw_change_list(actual_list, (String *) display_list, 
			xvu_list_size, FALSE);
}

/*------------------------------------------------------------
|
|  Routine Name: use_list()
|
|       Purpose: Registers user's selection from the list object
|
|         Input: object      - the "Use" button
|                client_data - unused
|                call_data   - unused
|
|        Output: none
|    Written By: Danielle Argiro & Mark Young
|          Date: May 19, 1994
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void use_list(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	xvu_list_done = TRUE;
	xvu_use_list = TRUE;
}

/*------------------------------------------------------------
|
|  Routine Name: cancel_list()
|
|       Purpose: Cancels list object
|
|         Input: object      - the "Cancel" button
|                client_data - unused
|                call_data   - unused
|
|        Output: none
|    Written By: Danielle Argiro & Mark Young
|          Date: May 19, 1994
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void cancel_list(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	if (list_head != NULL)
	   xvu_delete_list_entry(list_head);
	 
	list_head = NULL;
	xvu_list_done = TRUE;
}
