 /*
  * Khoros: $Id$
  */

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                 Pixmap Gadget Routines
   >>>>
   >>>>	Private:
   >>>>	 Static:
   >>>>			ClassInitialize()
   >>>>			Initialize()
   >>>>			Realize()
   >>>>			Redisplay()
   >>>>			SetValues()
   >>>>			RecomputeSize()
   >>>>			PickGadget()
   >>>>			PixmapCallback()
   >>>>  Public:
   >>>>			xvw_create_pixmap()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <xvobjects/ManagerP.h>
#include <xvobjects/PixmapP.h>

static void ClassInitialize	 PROTO((void));
static void Initialize           PROTO((Widget, Widget, ArgList, Cardinal *));
static void Realize              PROTO((Widget));
static void Redisplay		 PROTO((Widget, XEvent *, Region));
static Boolean SetValues	 PROTO((Widget, Widget, Widget, ArgList,
					Cardinal *));
static void RecomputeSize	 PROTO((Widget));
static int  PickGadget           PROTO((Widget, int, int));
static void PixmapCallback	 PROTO((xvobject, kaddr, XEvent *, int *));


/*-------------------------------------------------------
|
|   Full class attributes
|
--------------------------------------------------------*/

static xvattribute attributes[] = {
{XVW_PIXMAP,          NULL,        XtRPixmap,     NULL},
{XVW_PIXMAP_FILENAME, XVW_PIXMAP,  XtRFilename,   XtRPixmap},
{XVW_PIXMAP_CALLBACK, NULL,	   XtRCallback,   NULL},
{XVW_PIXMAP_MASK,     NULL,        XtRPixmapMask, NULL},
{XVW_PIXMAP_MASKNAME, XVW_PIXMAP_MASK,  XtRFilename, XtRPixmapMask},
};

/*-------------------------------------------------------
|
|   Full class record constant
|
--------------------------------------------------------*/

#define offset(field) XtOffsetOf(XvwPixmapGadgetRec, pixmap.field)

static XtResource resources[] = { 
{XVW_PIXMAP, XVW_PIXMAP_FILENAME, XtRPixmap, sizeof(Pixmap), 
      offset(pixmap), XtRImmediate, (XtPointer) None},
{XVW_PIXMAP_MASK, XVW_PIXMAP_MASKNAME, XtRPixmap, sizeof(Pixmap), 
      offset(mask), XtRImmediate, (XtPointer) None},
{XVW_PIXMAP_CALLBACK, NULL, XtRCallback, sizeof(XtPointer),
      offset(pixmap_callback), XtRCallback, (XtPointer) NULL},
};
#undef offset


#define SUPERCLASS (&xvwManagerGadgetClassRec)

XvwPixmapGadgetClassRec xvwPixmapGadgetClassRec =
{
  {
    (WidgetClass) SUPERCLASS,		/* superclass		  */	
    "Pixmap",				/* class_name		  */
    sizeof(XvwPixmapGadgetRec),		/* size			  */
    ClassInitialize,			/* class_initialize	  */
    NULL,				/* class_part_initialize  */
    FALSE,				/* class_inited		  */
    Initialize,				/* initialize		  */
    NULL,				/* initialize_hook	  */
    (XtProc) Realize,			/* realize		  */
    NULL,				/* actions		  */
    0,					/* num_actions		  */
    resources,				/* resources		  */
    XtNumber(resources),		/* resource_count	  */
    NULLQUARK,				/* xrm_class		  */
    FALSE,				/* compress_motion	  */
    XtExposeCompressMaximal,		/* compress_exposure	  */
    FALSE,				/* compress_enterleave    */
    FALSE,				/* visible_interest	  */
    NULL,				/* destroy		  */
    XtInheritResize,			/* resize		  */
    Redisplay,				/* expose		  */
    SetValues,				/* set_values		  */
    NULL,				/* set_values_hook	  */
    XtInheritSetValuesAlmost,		/* set_values_almost	  */
    NULL,				/* get_values_hook	  */
    NULL,				/* accept_focus		  */
    XtVersion,				/* version		  */
    NULL,				/* callback_private	  */
    NULL,				/* tm_table		  */
    NULL,				/* query_geometry	  */
    NULL,				/* display_accelerator	  */
    NULL				/* extension		  */
  },  /* RectObjClass fields initialization */
  {
    PickGadget,				/* pick object proc       */
    XtInheritEraseSel,                  /* erase selected proc    */
    XtInheritRefreshSel,	        /* refresh selection proc */
    XtInheritChangeGeometry,	        /* change geometry proc   */
  },  /* XvwManagerGadgetClass fields initialization */
  {
    NULL,				/* extension - not used   */
  },  /* XvwPixmapGadgetClass fields initialization */
};
#undef SUPERCLASS

/*
 * xvwPixmapGadgetClass for public consumption
 */
WidgetClass xvwPixmapGadgetClass = (WidgetClass) &xvwPixmapGadgetClassRec;


/*-----------------------------------------------------------
|
|  Routine Name: ClassInitialize
|
|       Purpose: This method is called the first time an  
|                instance of pixmap object class has been created.
|                It will initialize all the class attributes. 
|
|         Input: None 
|        Output: None
|
|    Written By: Mark Young
|          Date: Mar 03, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ClassInitialize(void)
{
	xvw_init_attributes(xvwPixmapGadgetClass, attributes,
		XtNumber(attributes), NULL, 0, 
		"$DESIGN/objects/library/xvobjects/uis/Pixmap.pane");
	xvw_load_resources("$DESIGN/objects/library/xvobjects/app-defaults/Pixmap");
}

/*-----------------------------------------------------------
|
|  Routine Name: Initialize
|
|       Purpose: This will initialize the pixmap object.
|
|         Input: request - not used
|                new     - widget instance after initialization
|        Output:
|       Returns:
|
|    Written By: Mark Young
|          Date: Mar 03, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Initialize(
   Widget request,
   Widget new,
   ArgList args,
   Cardinal *num_args)
{
	XvwPixmapGadget xobj = (XvwPixmapGadget) new;

	xobj->pixmap.xorigin =
	xobj->pixmap.yorigin = -1;
	xobj->pixmap.gc      = NULL;
	xvw_add_event(xvw_object(new), ButtonPressMask, PixmapCallback, NULL);
	if (xobj->pixmap.pixmap != None)
	{
	   char *name;
	   xvobject object = xvw_object(new);

	   xvw_get_attribute(object, XVW_PIXMAP_FILENAME, &name);
	   xvw_set_attribute(object, XVW_PIXMAP_MASKNAME, name);
	}
	RecomputeSize(new);
}

/*-----------------------------------------------------------
|
|  Routine Name: Realize
|
|       Purpose: This method will redisplay the pixmap if force redisplay
|		 is set.
|
|         Input: widget - the pixmap object to be realized
|        Output:
|
|    Written By: Mark Young
|          Date: Apr 10, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Realize(
   Widget   widget)
{
	XvwPixmapGadget xobj = (XvwPixmapGadget) widget;


	/*
	 *  Refresh the pixmap if force_redisplay is set.
	 */
	if (xobj->manager.force_redisplay)
	{
	   Redisplay(widget, NULL, NULL);
	   XFlush(XtDisplay(widget));
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: Redisplay
|
|       Purpose: This routine will redraw the pixmap object in response
|                to an expose event. 
|
|         Input: widget - the pixmap object that was exposed 
|                event  - the event that caused the redraw
|                region - the region that was exposed 
|
|        Output: None
|
|    Written By: Mark Young
|          Date: Mar 03, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Redisplay(
   Widget widget,
   XEvent *event,
   Region region)
{
	XvwPixmapGadget xobj = (XvwPixmapGadget) widget;

	Window window;
	int    x, y, w, h;
	XGCValues values;
	unsigned long mask;


	/*
	 *  If there is no pixmap then 
	 */
	if (!xobj->pixmap.pixmap)
	   return;

	x = widget->core.x + widget->core.border_width;
	y = widget->core.y + widget->core.border_width;
	w = widget->core.width;
	h = widget->core.height;
	window = XtWindowOfObject(widget);

	if (x != xobj->pixmap.xorigin || y != xobj->pixmap.yorigin ||
	    xobj->pixmap.gc == NULL)
	{
	   xobj->pixmap.xorigin = x; xobj->pixmap.yorigin = y;

	   if (xobj->pixmap.gc != NULL)
	      XtReleaseGC(widget, xobj->pixmap.gc);

           /*
            *  Create the gc to be used to refresh the pixmap
            */
           values.clip_x_origin = x;
           values.clip_y_origin = y;
           values.clip_mask  = xobj->pixmap.mask;
           values.foreground = xobj->manager.foreground;
           values.background = xobj->manager.background;
           mask = GCForeground | GCBackground | GCClipXOrigin | GCClipYOrigin |
		  GCClipMask;
	   xobj->pixmap.gc = XtGetGC(widget, mask, &values);
	}
	XCopyArea(XtDisplay(widget), xobj->pixmap.pixmap, window,
			xobj->pixmap.gc, 0, 0, w, h, x, y);
}

/*-----------------------------------------------------------
|
|  Routine Name: RecomputeSize
|
|       Purpose: recomputes the size of the pixmap object, based on the
|		 pixmap being displayed.
|
|         Input: widget - the pixmap object in which we will recompute our size
|        Output:
|    Written By: Mark Young
|          Date: Mar 03, 1994
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void RecomputeSize(
   Widget  widget)
{
	XvwPixmapGadget xobj = (XvwPixmapGadget) widget;

	int	 x, y;
	Window   rootwindow;
	unsigned int width, height, bw, depth;

	if (xobj->pixmap.pixmap)
	{
	   XGetGeometry(XtDisplay(widget), xobj->pixmap.pixmap, &rootwindow,
			&x, &y, &width, &height, &bw, &depth);
	}
	else
	   width = height = 0;

	widget->core.width  = width;
	widget->core.height = height;
	if (xobj->pixmap.gc != NULL)
	{
	   XtReleaseGC(widget, xobj->pixmap.gc);
	   xobj->pixmap.gc = NULL;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: SetValues
|
|       Purpose: If the the size or postion of the pixmap has changed, 
|		 recalculate the size and position of the pixmap.
|
|         Input: current - the object containing current settings 
|                request - the object containing requested settings
|                new     - the object processed through all set values methods 
|        Output: None directly 
|
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Mar 03, 1994
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean SetValues(
   Widget   current,
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
	XvwPixmapGadget cobj = (XvwPixmapGadget) current;
	XvwPixmapGadget nobj = (XvwPixmapGadget) new;

	char *name = NULL;
	Boolean  redisplay = FALSE;

	if (cobj->pixmap.pixmap != nobj->pixmap.pixmap)
	{
	   xvw_get_attribute(xvw_object(new), XVW_PIXMAP_FILENAME, &name);
	   xvw_set_attribute(xvw_object(new), XVW_PIXMAP_MASKNAME, name);

	   RecomputeSize(new);
	   redisplay = TRUE;
	}

	if (redisplay && nobj->manager.force_redisplay)
	   ManagerObjForceRedisplay(current, new, FALSE, &redisplay);

	return(redisplay);
}

/*-----------------------------------------------------------
|
|  Routine Name: PickGadget
|
|       Purpose: Given a connection object and an attempted picking location,
|                see if the pixmap has been picked.
|
|         Input: widget - the connection object we are checking
|                x      - the x device coordinate that the pick attempt was at
|                y      - the y device coordinate that the pick attempt was at
|
|        Output: None
|
|       Returns: TRUE (1) if object was successfully picked, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Jun 29, 1992 10:35
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int PickGadget(
   Widget widget,
   int    x,
   int    y)
{
	XvwPixmapGadget xobj = (XvwPixmapGadget) widget;
	int xpos, ypos, width, height;

	xpos = xobj->rectobj.x;      ypos = xobj->rectobj.y;
	width = xobj->rectobj.width; height = xobj->rectobj.height;
	if (xpos <= x && x <= xpos+width && ypos <= y && y <= ypos+height)
	   return(TRUE);
	else
	   return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: PixmapCallback
|
|       Purpose: Pixmap callback, which is used to call any pixmap callbacks

|         Input: object - the selected object
|                client_data  - (not used)
|                event  - the event
|        Output: dispatch - whether to continue dispatching this event
|    Written By: Mark Young
|          Date: Dec 6, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
static void PixmapCallback(
   xvobject object,
   kaddr    client_data,
   XEvent   *event,
   int	    *dispatch)
{
	XvwPixmapGadget xobj = (XvwPixmapGadget) xvw_widget(object);

	XtCallCallbacks((Widget) xobj, XVW_PIXMAP_CALLBACK, NULL);
}


/************************************************************
*
*  Routine Name: xvw_create_pixmap - create a pixmap object
*
*       Purpose: A pixmap object simply supports display of a 
*		 bitmap or pixmap, as defined by an xbm or xpm file.
*
*         Input: parent - the parent object; NULL will cause a
*                         default toplevel to be created automatically
*                name   - the name with which to reference pixmap object 
*
*        Output:
*       Returns: The pixmap object on success, NULL on failure
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Mar 03, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xvobject xvw_create_pixmap(
   xvobject parent,
   char   *name)
{
	xvobject  object;


	/*
	 *  Create the line object
	 */
	object = xvw_create(parent, FALSE, TRUE, name, PixmapGadgetClass);
	if (object != NULL)
	{
	   xvw_set_attributes(object,
                XVW_MENUABLE,     TRUE,     /* menuable       */
                XVW_RESIZABLE,    FALSE,    /* resizable      */
                XVW_SELECTABLE,   TRUE,     /* selectable     */
		NULL);
	}
	return(object);
}
