 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */



/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>	        Manager Widget Utility Routines
   >>>>
   >>>>   Static:
   >>>>			CheckSelection()
   >>>>			LocateSelection()
   >>>>			CheckIfSelection()
   >>>>			RefreshSelection()
   >>>>			EraseSelection()
   >>>>			AddSelection()
   >>>>			DeleteSelection()
   >>>>			ExtentSelections()
   >>>>			AddExtentSelections()
   >>>>			DeleteAllSelections()
   >>>>			AddAllSelections()
   >>>>			RaiseInputWindow()
   >>>>			ConfigureWidget()
   >>>>			WidgetHandler()
   >>>>			WidgetInfo()
   >>>>  Private:
   >>>>			ManagerResizeObject()
   >>>>			ManagerRealizeObject()
   >>>>			ManagerCheckSelection()
   >>>>			ManagerCheckIfSelection()
   >>>>			ManagerChangeSelection()
   >>>>			ManagerRefreshSelection()
   >>>>			ManagerEraseSelection()
   >>>>			ManagerCreateGrid()
   >>>>			ManagerWidgetHandler()
   >>>>			ManagerSetSelect()
   >>>>			ManagerGetSelect()
   >>>>			ManagerSetGroup()
   >>>>			ManagerPixelGeometry()
   >>>>			ManagerActionHandler()
   >>>>			ManagerSetWMColormap()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <xvobjects/ManagerP.h>
#include <xvobjects/ManagerObjP.h>

#define MENUABLE     (1L << 0)
#define SELECTABLE   (1L << 1)
#define RESIZABLE    (1L << 2)

#define NO_EDGE     0
#define LEFT_EDGE   (1 << 0)
#define RIGHT_EDGE  (1 << 1)
#define TOP_EDGE    (1 << 2)
#define BOTTOM_EDGE (1 << 3)

#define SELECT_CURSOR	"hand1"
#define EDIT_CURSOR	"target"


static void RefreshSelection        PROTO((XvwManagerWidget, Widget));
static void EraseSelection          PROTO((XvwManagerWidget, Widget));
static void AddSelection            PROTO((XvwManagerWidget, Widget));
static void DeleteSelection         PROTO((XvwManagerWidget, Widget));
static void ExtentSelections        PROTO((XvwManagerWidget, int, int));
static void DeleteAllSelections     PROTO((XvwManagerWidget));
static void AddAllSelections	    PROTO((XvwManagerWidget, int));
static Boolean CheckSelection       PROTO((Widget, int, int));
static Boolean CheckIfSelection     PROTO((XvwManagerWidget, Widget));
static void    RaiseInputWindow	    PROTO((xvobject, kaddr,  XEvent *));
static void    WidgetHandler	    PROTO((XvwManagerWidget, XEvent *));
static void    WidgetInfo	    PROTO((xvobject));
static void    ConfigureWidget	    PROTO((Widget, Position, Position,
				       Dimension, Dimension, Dimension));
static Widget  LocateSelection      PROTO((XvwManagerWidget, int, int, 
				       unsigned long));
static void AddExtentSelections     PROTO((XvwManagerWidget, int, int, int, int,
				       unsigned long));


/*-------------------------------------------------------------------*
|
|   Miscelleanous defines for Class and Constraint Declarations
|
--------------------------------------------------------------------*/

#define kwidget(widget)      (XvwManagerWidget) (widget)
#define kwidgetclass(widget) (XvwManagerWidgetClass) (widget)->core.widget_class
#define kconstraint(widget)  (XvwManagerWidgetConstraints) (widget)->core.constraints



/*-----------------------------------------------------------
|
|  Routine Name: CheckSelection 
|
|       Purpose: Check to see if the widget in question is the selection
|		 within the xpos, ypos.
|
|         Input: child - the child to check against the xpos, ypos
|                xpos  -  the x position to look at
|                ypos  -  the y position to look at
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean CheckSelection(
   Widget   child,
   int      xpos,
   int      ypos)
{
	Position  tempx, tempy;
	Dimension tempw, temph;
	int       x, y, width, height, tolerance;


	/*
	 *  If the child is outside the bounding box then don't bother
	 *  checking to see if it's been picked.
	 */
	tolerance = kmax(child->core.border_width, PICK_TOLERANCE);
	ManagerGeometry(child, &tempx, &tempy, &tempw, &temph, NULL);

	x = tempx; y = tempy; width = tempw; height = temph;
	if (!((xpos >= (x-tolerance) && xpos <= (x+width+tolerance)) &&
	      (ypos >= (y-tolerance) && ypos <= (y+height+tolerance))))
	   return(FALSE);

	/*
	 *  If the child's picking routine is not NULL
	 */
	if (XtIsSubclass(child, ManagerGadgetClass))
	{
	   XvwManagerGadget xobj = (XvwManagerGadget) child;
	   XvwManagerGadgetClass mobj_class = (XvwManagerGadgetClass)
					XtClass(child);

	   /*
	    *  If the object is not mapped, then always return failure.
	    */
	   if (!xobj->manager.mapped)
	      return(FALSE);

	   if (mobj_class->manager_class.pick_selection != NULL &&
               !(mobj_class->manager_class.pick_selection) (child, xpos, ypos))
	   {
	      return(FALSE);
	   }
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: LocateSelection 
|
|       Purpose:
|
|         Input: xwid -  
|                xpos -
|                ypos -
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/

/* ARGSUSED */
static Widget LocateSelection(
   XvwManagerWidget xwid,
   int              xpos,
   int              ypos,
   unsigned long    mask)
{
	int       i;
	Widget   child;
	XvwManagerWidgetConstraints kman;

	/*
	 *  Print DEBUG message
	 */
	kinfo(KDEBUG,"Inside LocateSelection routine for widget '%s'\n\n",
			XtName((Widget) xwid));

	/*
	 *  Sanity check...
	 */
	if (!xwid) return(NULL);

	for (i = xwid->composite.num_children-1; i >= 0; i--)
	{
	   /*
	    *  If the child is not editable or is not managed don't bother
	    *  checking.
	    */
	   child = xwid->composite.children[i];
	   kman  = kconstraint(child);

	   if ((mask == NONE)					     ||
	      ((mask & MENUABLE) && kman->manager.allow_menuing)     ||
	      ((mask & SELECTABLE) && kman->manager.allow_selecting) ||
	      ((mask & RESIZABLE) && kman->manager.allow_resizing))
	   {
	       if (xvw_check_mapped(xvw_object(child)) &&
		   CheckSelection(child, xpos, ypos))
	       {
	          /*
	           *  Print DEBUG message
	           */
	          kinfo(KDEBUG,"Inside LocateSelection located widget '%s'\n\n",
			 XtName((Widget) child));
	          return(child);
	       }
	   }
	}

	if (xvw_check_mapped(xvw_object((Widget) xwid)) && ((mask == NONE) ||
              ((mask & MENUABLE) && xwid->manager.allow_menuing)     ||
              ((mask & SELECTABLE) && xwid->manager.allow_selecting) ||
              ((mask & RESIZABLE) && xwid->manager.allow_resizing)))
	{
	   /*
	    *  Print DEBUG message
	    */
	   kinfo(KDEBUG,"Inside LocateSelection located parent widget \
'%s'\n\n", XtName((Widget) xwid));
	   return((Widget) xwid);
	}

	/*
	 *  Print DEBUG message
	 */
	kinfo(KDEBUG,"Inside LocateSelection unable to locate widget\n\n");
	return(NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: AddExtentSelections
|
|       Purpose: Add all selections within the bounding box to the manager's
|		 selection list.
|
|         Input: xwid  - the manager widget
|                xpos1 - startx position
|                ypos1 - starty position
|                xpos2 - endx position
|                ypos2 - endy position
|                mask  - mask for which objects to add
|    Written By: Mark Young
|          Date: Jul 30, 1994
|
------------------------------------------------------------*/

/* ARGSUSED */
static void AddExtentSelections(
   XvwManagerWidget xwid,
   int              xpos1,
   int              ypos1,
   int              xpos2,
   int              ypos2,
   unsigned long    mask)
{
	int       i;
	Widget   child;
	XvwManagerWidgetConstraints kman;

        int       x, y, w, h;
        Position  tempx, tempy;
        Dimension tempw, temph;

	/*
	 *  Print DEBUG message
	 */
	kinfo(KDEBUG,"Inside AddExtentSelections routine for widget '%s'\n\n",
			XtName((Widget) xwid));

	/*
	 *  Sanity check...
	 */
	if (!xwid) return;

	for (i = xwid->composite.num_children-1; i >= 0; i--)
	{
	   /*
	    *  If the child is not editable or is not managed don't bother
	    *  checking.
	    */
	   child = xwid->composite.children[i];
	   kman  = kconstraint(child);

	   if ((mask == NONE)					     ||
	      ((mask & MENUABLE) && kman->manager.allow_menuing)     ||
	      ((mask & SELECTABLE) && kman->manager.allow_selecting) ||
	      ((mask & RESIZABLE) && kman->manager.allow_resizing))
	   {
	       if (!xvw_check_mapped(xvw_object(child)))
		  continue;

	       ManagerGeometry(child, &tempx, &tempy, &tempw, &temph, NULL);
	       x = tempx; y = tempy; w = tempw; h = temph;
	       if (x >= xpos1 && y >= ypos1 && (x+w) <= xpos2 && (y+h) <= ypos2)
	       {
	          /*
	           *  Print DEBUG message
	           */
	          kinfo(KDEBUG,"Inside AddExtentSelections adding widget '%s'",
			 XtName((Widget) child));
	          AddSelection(xwid, child);
	       }
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: CheckIfSelection 
|
|       Purpose:
|
|         Input: xwid      -  
|                selection -
|	Returns: TRUE (1) if selection is on the list, FALSE (0) otherwise 
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean CheckIfSelection(
   XvwManagerWidget xwid,
   Widget         selection)
{
	/*
	 *  Sanity check...
	 */
	if (!xwid) return(FALSE);

	/*
	 *  Search the list to see if the selection is on the list.
	 */
	if (karray_locate((char **) xwid->manager.widgetsels,
		(kaddr) selection, xwid->manager.numsels) != -1)
	   return(TRUE);
	else
	   return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: RefreshSelection
|
|       Purpose:
|
|         Input: xwid      -  
|                selection -
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static void RefreshSelection(
   XvwManagerWidget xwid,
   Widget         selection)
{
	/*
	 *  Sanity check...
	 */
	if (!xwid) return;

	/*
	 *  If the child's picking routine is not NULL
	 */
	if (XtIsSubclass(selection, ManagerGadgetClass))
	{
           XvwManagerGadgetClass mobj_class = (XvwManagerGadgetClass)
					XtClass(selection);

	   if (mobj_class->manager_class.refresh_selection != NULL)
	   {
	      (mobj_class->manager_class.refresh_selection)(selection);
	      return;
	   }
	}
	else if (XtIsSubclass(selection, ManagerWidgetClass))
	{
           XvwManagerWidgetClass mclass = (XvwManagerWidgetClass)
					XtClass(selection);

	   if (mclass->manager_class.refresh_selection != NULL)
	   {
	      (mclass->manager_class.refresh_selection)(selection);
	      return;
	   }
	}
	else
	   DefaultRefreshSelection(selection);
}

/*-----------------------------------------------------------
|
|  Routine Name: EraseSelection 
|
|       Purpose:
|
|         Input: xwid      -  
|                selection -  
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static void EraseSelection(
   XvwManagerWidget xwid,
   Widget         selection)
{
	/*
	 *  Sanity check...
	 */
	if (!xwid) return;

	/*
	 *  If the child's picking routine is not NULL
	 */
	if (XtIsSubclass(selection, ManagerGadgetClass))
	{
           XvwManagerGadgetClass mobj_class = (XvwManagerGadgetClass)
					XtClass(selection);

	   if (mobj_class->manager_class.erase_selection != NULL)
	   {
	      (mobj_class->manager_class.erase_selection)(selection);
	      return;
	   }
	}
	else if (XtIsSubclass(selection, ManagerWidgetClass))
	{
           XvwManagerWidgetClass mclass = (XvwManagerWidgetClass)
					XtClass(selection);

	   if (mclass->manager_class.erase_selection != NULL)
	   {
	      (mclass->manager_class.erase_selection)(selection);
	      return;
	   }
	}
	else
	   DefaultEraseSelection(selection);
}

/*-----------------------------------------------------------
|
|  Routine Name: AddSelection 
|
|       Purpose:
|
|         Input: xwid      -
|                selection -
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static void AddSelection(
   XvwManagerWidget xwid,
   Widget         selection)
{
	XvwManagerWidgetConstraints kman = kconstraint(selection);

	int i;

	/*
	 *  Santity check...  Make sure the selection is not already
	 *  on the list
	 */
	if (!xwid || CheckIfSelection(xwid, selection) == TRUE)
	   return;

	/*
	 *  Add the selection to the list and then refresh the selection to
	 *  visually show that it is selected.
	 */
	xwid->manager.objectsels = (xvobject *) karray_add((char **)
		xwid->manager.objectsels, (kaddr) xvw_object(selection),
		xwid->manager.numsels);
	xwid->manager.widgetsels = (Widget *) karray_add((char **)
		xwid->manager.widgetsels, (kaddr) selection,
		xwid->manager.numsels++);

	for (i = 0; i < kman->manager.group_size; i++)
	   AddSelection(xwid, xvw_widget(kman->manager.group[i]));

	RefreshSelection(xwid, selection);

	/*
	 *  If this is a ManagerWidgetClass then we need to see if the class
	 *  has an change selection mechanism to call.
	 */
	if (selection->core.being_destroyed == FALSE &&
	    XtIsSubclass(selection, ManagerWidgetClass))
	{
           XvwManagerWidgetClass mclass = (XvwManagerWidgetClass)
					XtClass(selection);

	   if (mclass->manager_class.change_selection != NULL)
	   {
	      (mclass->manager_class.change_selection)(selection, TRUE);
	      return;
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: DeleteSelection 
|
|       Purpose:
|
|         Input: xwid      -
|                selection -
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static void DeleteSelection(
   XvwManagerWidget xwid,
   Widget         selection)
{
	XvwManagerWidgetConstraints kman = kconstraint(selection);

	/*
	 *  Santity check...  Make sure the selection is not already
	 *  on the list
	 */
	if (!xwid || !CheckIfSelection(xwid, selection))
	   return;

	/*
	 *  Add the selection to the list and then refresh the selection to
	 *  visually show that it is selected.
	 */
	xwid->manager.objectsels = (xvobject *) karray_delete((char **)
                xwid->manager.objectsels, (kaddr) xvw_object(selection),
                xwid->manager.numsels);
	xwid->manager.widgetsels = (Widget *) karray_delete((char **)
                xwid->manager.widgetsels, (kaddr) selection,
                xwid->manager.numsels--);

#if 0
        for (i = 0; i < kman->manager.group_size; i++)
           DeleteSelection(xwid, xvw_widget(kman->manager.group[i]));
#endif


	/*
	 *  If this is a ManagerWidgetClass then we need to see if the class
	 *  has an change selection mechanism to call.
	 */
	if (selection->core.being_destroyed == FALSE &&
	    XtIsSubclass(selection, ManagerWidgetClass))
	{
           XvwManagerWidgetClass mclass = (XvwManagerWidgetClass)
					XtClass(selection);

	   if (mclass->manager_class.change_selection != NULL)
	   {
	      (mclass->manager_class.change_selection)(selection, FALSE);
	      return;
	   }
	}
	EraseSelection(xwid, selection);
}

/*-----------------------------------------------------------
|
|  Routine Name: ExtentSelections 
|
|       Purpose:
|
|         Input: xwid -
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ExtentSelections(
   XvwManagerWidget xwid,
   int		    xpos,
   int		    ypos)
{
	Widget  widget;
	XvwManagerWidgetConstraints kman;
	int i, xpos1, xpos2, ypos1, ypos2, w, h, x, y, tolerance;


	/*
	 *  Sanity check...
	 */
	if (!xwid) return;

	if (xwid->manager.numsels == 0)
	  return;

	/*
	 *  Initialize the extent of the object
	 */
	xpos1 = xwid->manager.widgetsels[0]->core.x;
	ypos1 = xwid->manager.widgetsels[0]->core.y;
	xpos2 = xpos1 + xwid->manager.widgetsels[0]->core.width;
	ypos2 = ypos1 + xwid->manager.widgetsels[0]->core.height;

	/*
	 *  Clear the selections and reset the number of selected items to
	 *  zero.
	 */
	for (i = 0; i < xwid->manager.numsels; i++)
	{
	   widget = xwid->manager.widgetsels[i];
	   kman   = kconstraint(widget);

	   x = widget->core.x;
	   y = widget->core.y;
	   w = widget->core.width;
	   h = widget->core.height;

	   if (xpos1 > x)
	      xpos1 = x;
	   else if (ypos1 > y)
	      ypos1 = y;

	   if (xpos2 < (x + w))
	      xpos2 = x + w;
	   else if (ypos2 < (y + h))
	      ypos2 = y + h;

	   kman->manager.flags = NO_EDGE;
	   kman->manager.xoffset = x - xpos;
	   kman->manager.yoffset = y - ypos;

	   if (kman->manager.allow_resizing)
	   {
	      tolerance = krange(3, w - 4*PICK_TOLERANCE, PICK_TOLERANCE);
	      if ((y - PICK_TOLERANCE) < ypos && ypos < (y+h + PICK_TOLERANCE))
	      {
	         if (kabs(xpos - x) <= tolerance)
	            kman->manager.flags |= LEFT_EDGE;
	         else if (kabs(xpos - (x+w)) <= tolerance)
		 {
	            kman->manager.flags |= RIGHT_EDGE;
		    kman->manager.xoffset = (x+w) - xpos;
		 }
	      }

	      tolerance = krange(3, h - 4*PICK_TOLERANCE, PICK_TOLERANCE);
	      if ((x - PICK_TOLERANCE) < xpos && xpos < (x+w + PICK_TOLERANCE))
	      {
	         if (kabs(ypos - y) <= tolerance)
	            kman->manager.flags |= TOP_EDGE;
	         else if (kabs(ypos - (y+h)) <= tolerance)
		 {
	            kman->manager.flags |= BOTTOM_EDGE;
		    kman->manager.yoffset = (y+h) - ypos;
		 }
	      }
	   }
	}
	xwid->manager.xpos = xpos1;
	xwid->manager.ypos = ypos1;
	xwid->manager.width  = xpos2 - xpos1;
	xwid->manager.height = ypos2 - ypos1;
}

/*-----------------------------------------------------------
|
|  Routine Name: DeleteAllSelections
|
|       Purpose:
|
|         Input: xwid -
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static void DeleteAllSelections(
   XvwManagerWidget xwid)
{
	int i;

	/*
	 *  Sanity check...
	 */
	if (!xwid) return;

	/*
	 *  Clear the selections and reset the number of selected items to
	 *  zero.
	 */
	for (i = xwid->manager.numsels-1; i >= 0; i--)
	   DeleteSelection(xwid, xwid->manager.widgetsels[i]);
}

/*-----------------------------------------------------------
|
|  Routine Name: AddAllSelections
|
|       Purpose:
|
|         Input: xwid -
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static void AddAllSelections(
   XvwManagerWidget xwid,
   int mask)
{
	int i;
	Widget   child;
	XvwManagerWidgetConstraints kman;


	/*
	 *  Sanity check...
	 */
	if (!xwid) return;

	/*
	 *  Add the children into the selection list
	 */
	for (i = xwid->composite.num_children-1; i >= 0; i--)
	{
	   child = xwid->composite.children[i];
	   kman = kconstraint(child);
	   if ((mask == NONE)                                        ||
	      ((mask & MENUABLE) && kman->manager.allow_menuing)     ||
	      ((mask & SELECTABLE) && kman->manager.allow_selecting) ||
	      ((mask & RESIZABLE) && kman->manager.allow_resizing))
	   {
	      AddSelection(xwid, child);
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: RaiseInputWindow - make sure the input only window is
|				    always on the top
|
|       Purpose: Makes sure that if new windows or someone raises an existing
|		 window that we are always on top...
|
|         Input: object     -
|                client_data -
|                event      -
|    Written By: Mark Young
|          Date: Nov 21, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
static void RaiseInputWindow(
   xvobject object,
   kaddr    client_data,
   XEvent   *event)
{
	Widget   widget = xvw_widget(object);
	XvwManagerWidget xwid = kwidget(widget);


	if (event->type == CreateNotify && xwid->manager.editing == TRUE &&
	    event->xany.window != xvw_window(xwid->manager.inputonly))
	{
	   xvw_raise(xwid->manager.inputonly);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: ConfigureWidget
|
|       Purpose: Similar to XtConfigureWidget() but supports the change
|		 geometry method for the ManagerGadget Class.
|         Input: widget - the widget we want to re-configure
|                x      - the widget's new x position
|                y      - the widget's new y position
|                width  - the widget's new width dimension
|                height - the widget's new height dimension
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ConfigureWidget(
   Widget    widget,
   Position  x,
   Position  y,
   Dimension width,
   Dimension height,
   Dimension border_width)
{
	Widget parent = XtParent(widget);
	XvwManagerWidgetConstraints kman = kconstraint(widget);

	/*
	 *  This routine exists since there is a bug in X11R4
	 *  XtConfigureWidget() that will not properly configure
	 *  gadgets.
	 */
	if (width  == 0) width = 1;
	if (height == 0) height = 1;
	kinfo(KDEBUG,"ConfigureWidget: %s is a child of %s \n,\
changing geometry to (x,y,width,height) = (%d,%d,%d,%d)", XtName(widget),
			XtName(XtParent(widget)), x, y, width, height);

	if (!XtIsWidget(widget))
	{
	   XtConfigureWidget(widget, x, y, width, height, border_width);
	   if (XtIsSubclass(widget, ManagerGadgetClass) == TRUE)
	   {
	      XvwManagerGadget xobj = (XvwManagerGadget) widget;
	      XvwManagerGadgetClass mobj_class = (XvwManagerGadgetClass)
                                        XtClass(widget);
              /*
               *  If the object does not have a change position method
	       *  then ignore it.
               */
	      if (mobj_class->manager_class.change_geometry != NULL)
		 (mobj_class->manager_class.change_geometry) (widget);
	   }
	}
	else
	   XtConfigureWidget(widget, x, y, width, height, border_width);
}

/*-----------------------------------------------------------
|
|  Routine Name: WidgetHandler
|
|       Purpose:
|
|         Input: xwid     -
|                event      -
|    Written By: Mark Young
|          Date: Jul 06, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
static void WidgetHandler(
   XvwManagerWidget xwid,
   XEvent	    *event)
{
	XtWidgetGeometry temp;
	Widget *widgetsels, widget = (Widget) xwid;

	Window root;
	double snap;
	double xsnap = xwid->manager.xsnap;
	double ysnap = xwid->manager.ysnap;
	int    buffer_dist = xwid->manager.buffer_dist;
	register int i, x, y, w, h, bw, xpos, ypos;


	/*
	 *  If the user button releases then go ahead and remove the event
	 *  handler.
	 */
	if (!XtIsSubclass(widget, ManagerWidgetClass) ||
	    event->type != MotionNotify)
	{
	   return;
	}

	/*
	 *  See if we need to query the position...
	 */
	if (event->xmotion.is_hint == TRUE)
	{
	   int tmpx, tmpy, dummy;

	   XQueryPointer(XtDisplay(widget), XtWindow(widget), &root, &root,
			&dummy, &dummy, &tmpx, &tmpy, (unsigned int *) &dummy);
	   xpos = tmpx; ypos = tmpy;
	}
	else if (XtWindow(widget) == event->xany.window)
	{
	   xpos = event->xmotion.x;
	   ypos = event->xmotion.y;
	}
	else
	{
	   Position tmpx, tmpy;

	   XtTranslateCoords(widget, 0, 0, &tmpx, &tmpy);
	   xpos = event->xmotion.x_root - tmpx;
	   ypos = event->xmotion.y_root - tmpy;
	}

	/*
	 *  Move the widgets on the selection list
	 */
	temp.request_mode = CWWidth | CWHeight | CWX | CWY;
	for (i = 0, widgetsels = xwid->manager.widgetsels;
	     i < xwid->manager.numsels; i++, widgetsels++)
	{
	   XvwManagerWidgetConstraints kman = kconstraint(*widgetsels);

	   /*
	    *  Get the current geometry so that we can perform the
	    *  translation or resizing of the selection.
	    */
	   ManagerGeometry(*widgetsels, &temp.x, &temp.y, &temp.width,
			&temp.height, &temp.border_width);

	   bw = 2*temp.border_width;
	   x = temp.x;           y = temp.y;
	   w = temp.width + bw;  h = temp.height + bw;

	   if (kman->manager.flags  & LEFT_EDGE  ||
	       kman->manager.flags  & RIGHT_EDGE ||
	      !(kman->manager.flags & TOP_EDGE) &&
              !(kman->manager.flags & BOTTOM_EDGE))
	   {
	      if (kman->manager.flags & RIGHT_EDGE)
	      {
		 w = xpos + kman->manager.xoffset - x;
		 if (xsnap > 1.0)
		 {
		    snap = fmod((double) (x-buffer_dist)+w, xsnap);
		    w += (int) ((snap > (xsnap/2)) ? xsnap - snap : -snap);
		 }
	      }
	      else
	      {
	         x = xpos + kman->manager.xoffset;
	         if (xsnap > 1.0)
	         {
	            snap = fmod((double) x-buffer_dist, xsnap);
	            x += (int) ((snap > (xsnap/2)) ? xsnap - snap : -snap);
	         }
		 if (kman->manager.flags & LEFT_EDGE)
	            w += temp.x - x;
	      }
	      if (x < buffer_dist) x = buffer_dist;
	      if (w < bw) w = bw;
	   }

	   if (kman->manager.flags  & TOP_EDGE  ||
	       kman->manager.flags  & BOTTOM_EDGE ||
	      !(kman->manager.flags & LEFT_EDGE) &&
              !(kman->manager.flags & RIGHT_EDGE))
	   {
	      if (kman->manager.flags & BOTTOM_EDGE)
	      {
		 h = ypos + kman->manager.yoffset - y;
		 if (ysnap > 1.0)
		 {
		    snap = fmod((double) (y-buffer_dist)+h, ysnap);
		    h += (int) ((snap > (ysnap/2)) ? ysnap - snap : -snap);
		 }
	      }
	      else
	      {
	         y = ypos + kman->manager.yoffset;
	         if (ysnap > 1.0)
	         {
	            snap = fmod((double) y-buffer_dist, ysnap);
	            y += (int) ((snap > (ysnap/2)) ? ysnap - snap : -snap);
	         }
	         if (kman->manager.flags & TOP_EDGE)
	            h += temp.y - y;
	      }
	      if (y < buffer_dist) y = buffer_dist;
	      if (h < bw) h = bw;
	   }

           if (w < ((int) kman->manager.min_width))
	      w = kman->manager.min_width;
	   else if (w > ((int) kman->manager.max_width))
	      w = kman->manager.max_width;

	   if (h < ((int) kman->manager.min_height))
	      h = kman->manager.min_height;
	   else if (h > ((int) kman->manager.max_height))
	      h = kman->manager.max_height;

	   w -= bw; h -= bw;
	   if (temp.x == x && temp.y == y && temp.width == w && temp.height==h)
	      continue;

	   kman->manager.preferred_width  = w;
	   kman->manager.preferred_height = h;
	   if (XtIsSubclass(*widgetsels, ManagerWidgetClass))
	   {
	      XvwManagerWidget xsel = kwidget(*widgetsels);

	      xsel->manager.preferred_width  = w;
	      xsel->manager.preferred_height = h;
	   }

	   /*
	    *  Erase the selection indicators so that we can move the
	    *  widget without leaving mouse droppings.
	    */
	   EraseSelection(xwid, *widgetsels);
	   temp.x = x; temp.y = y; temp.width = w; temp.height = h;
	   ManagerGeometryManager(*widgetsels, &temp, NULL);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: WidgetInfo
|
|       Purpose:
|
|         Input: object     -
|    Written By: Mark Young
|          Date: Jul 06, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
static void WidgetInfo(
   xvobject object)
{
	Widget info = xvw_widget(object);
	XvwManagerWidgetConstraints kman = kconstraint(info);

	Widget   par, left, right, below, above;
	int      dist, vert_offset, horiz_offset;
	int      tacking, menuing, selecting, resizing;
	int      minwidth, minheight, maxwidth, maxheight;
	float	 char_xpos, char_ypos, char_width, char_height,
		 char_minwidth, char_minheight, char_maxwidth, char_maxheight;
	
	par = XtParent(info);
	if (XtIsSubclass(par, ManagerWidgetClass) == FALSE)
	{
	   left = right = above = below = (Widget) KMANAGER_UNDEFINED;
	   menuing = selecting = resizing = FALSE;
	   char_minwidth = char_minheight = char_maxwidth =
	   char_maxheight = char_xpos = char_ypos = char_width =
	   char_height = -1.0; tacking = NONE;
	   dist = vert_offset = horiz_offset = 0;
	   minwidth = minheight = maxwidth = maxheight = 0;
	}
	else
	{
	   tacking     = kman->manager.tacking;
	   menuing     = kman->manager.allow_menuing;
	   selecting   = kman->manager.allow_selecting;
	   resizing    = kman->manager.allow_resizing;
	   left        = kman->manager.left_of;
	   right       = kman->manager.right_of;
	   above       = kman->manager.above;
	   below       = kman->manager.below;
	   if (!widgetdefined(left) && (tacking & KMANAGER_TACK_LEFT))
	      left = NULL;
	   if (!widgetdefined(above) && (tacking & KMANAGER_TACK_TOP))
	      above = NULL;
	   if (!widgetdefined(right) && (tacking & KMANAGER_TACK_RIGHT))
	      right = NULL;
	   if (!widgetdefined(below) && (tacking & KMANAGER_TACK_BOTTOM))
	      below=NULL;

	   left  = (!left)  ? par : left;
	   right = (!right) ? par : right;
	   above = (!above) ? par : above;
	   below = (!below) ? par : below;
	   xvw_get_attributes(xvw_object(info), 
			XVW_CHAR_WIDTH,  &char_width,
			XVW_CHAR_HEIGHT, &char_height,
			XVW_CHAR_XPOS,   &char_xpos,
			XVW_CHAR_YPOS,   &char_ypos,
			XVW_CHAR_MIN_WIDTH, &char_minwidth,
			XVW_CHAR_MIN_HEIGHT, &char_minheight,
			XVW_CHAR_MAX_WIDTH, &char_maxwidth,
			XVW_CHAR_MAX_HEIGHT, &char_maxheight,
			XVW_MINIMUM_WIDTH, &minwidth,
			XVW_MINIMUM_HEIGHT, &minheight,
			XVW_MAXIMUM_WIDTH, &maxwidth,
			XVW_MAXIMUM_HEIGHT, &maxheight,
			NULL);

	   xvw_get_attributes(xvw_object(par),
			XVW_BUFFER_DIST,    &dist,
			XVW_DEF_VERT_DIST,  &vert_offset,
			XVW_DEF_HORIZ_DIST, &horiz_offset,
			NULL);
	   if (kman->manager.horiz_offset != -1)
	      horiz_offset = kman->manager.horiz_offset;
	   if (kman->manager.vert_offset != -1)
	      vert_offset = kman->manager.vert_offset;
	}
	kinfo(KFORCE, "Information for the %s: '%s'\n\n\
 WidgetClass: %s\n\n\
 Position: Pixels (%d, %d)\n\
           Chars  (%.1f,%.1f)\n\
 Geometry: Pixels (%d,%d)\n\
           Chars  (%.1f,%.1f)\n\
       Min Pixels (%d,%d)\n\
       Max Pixels (%d,%d)\n\n\
       Min Chars  (%.1f,%.1f)\n\
       Max Chars  (%.1f,%.1f)\n\n\
 BdrWidth: %d\n\n\
 Menuing  : %s\n\
 Selecting: %s\n\
 Resizing : %s\n\n\
 Left  of    %s\n\
 Right of    %s\n\
 Above       %s\n\
 Below       %s\n\
 Tack Left   %s\n\
 Tack Right  %s\n\
 Tack Top    %s\n\
 Tack Bottom %s\n\n\
 Vertical   Dist %d\n\
 Horizontal Dist %d\n\
 Buffer     Dist %d\n",
	(XtIsWidget(info) ? "widget" : "gadget"), XtName(info),
	info->core.widget_class->core_class.class_name,
	info->core.x, info->core.y, char_xpos, char_ypos,
	info->core.width, info->core.height, char_width, char_height,
	minwidth, minheight, maxwidth, maxheight,
	char_minwidth, char_minheight, char_maxwidth, char_maxheight,
	info->core.border_width,
	(!menuing ? "False" : "True"), (!selecting ? "False" : "True"),
	(!resizing ? "False" : "True"),
	(!widgetdefined(left) ? "Undefined" : XtName(left)),
	(!widgetdefined(right) ? "Undefined" : XtName(right)),
	(!widgetdefined(above) ? "Undefined" : XtName(above)),
	(!widgetdefined(below) ? "Undefined" : XtName(below)),
	(tacking & KMANAGER_TACK_LEFT ? "True" : "False"),
	(tacking & KMANAGER_TACK_RIGHT ? "True" : "False"),
	(tacking & KMANAGER_TACK_TOP ? "True" : "False"),
	(tacking & KMANAGER_TACK_BOTTOM ? "True" : "False"),
	vert_offset, horiz_offset, dist);
}


/*-----------------------------------------------------------
|
|  Routine Name: ManagerCheckSelection
|
|       Purpose: 
|
|         Input: widget - the widget we want to check
|                x      - 
|                y      - 
|                width  - 
|                height - 
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
int ManagerCheckSelection(
   xvobject object,
   int	    xpos,
   int	    ypos)
{
	return(CheckSelection(xvw_widget(object), xpos, ypos));
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerCheckIfSelection
|
|       Purpose: see if the selection is current selected within the
|		 manager.
|
|         Input: widget    - the manager widget
|                selection - the selection we want to check for
|    Written By: Mark Young
|          Date: Mar 26, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */
int ManagerCheckIfSelection(
   Widget   widget,
   Widget   selection)
{
	return(CheckIfSelection((XvwManagerWidget) widget, selection));
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerRefreshSelection
|
|       Purpose: refresh the selection within the manager.
|
|         Input: widget    - the manager widget
|                selection - the selection we want to refresh for
|    Written By: Mark Young
|          Date: Mar 26, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */
void ManagerRefreshSelection(
   Widget   widget,
   Widget   selection)
{
	RefreshSelection((XvwManagerWidget) widget, selection);
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerEraseSelection
|
|       Purpose: erase the selection within the manager.
|
|         Input: widget    - the manager widget
|                selection - the selection we want to refresh for
|    Written By: Mark Young
|          Date: Mar 26, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */
void ManagerEraseSelection(
   Widget   widget,
   Widget   selection)
{
	EraseSelection((XvwManagerWidget) widget, selection);
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerDeleteSelection 
|
|       Purpose:
|
|         Input: xwid      -
|                selection -
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
void ManagerDeleteSelection(
   Widget	widget,
   Widget	selection)
{
	DeleteSelection((XvwManagerWidget) widget, selection);
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerAddSelection 
|
|       Purpose:
|
|         Input: xwid      -
|                selection -
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
void ManagerAddSelection(
   Widget	widget,
   Widget	selection)
{
	AddSelection((XvwManagerWidget) widget, selection);
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerAddExtentSelection 
|
|       Purpose: Add all selections within the bounding box to the manager's
|		 selection list.
|
|         Input: widget - the manager widget
|                xpos1 - startx position
|                ypos1 - starty position
|                xpos2 - endx position
|                ypos2 - endy position
|    Written By: Mark Young
|          Date: Dec 06, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */
void ManagerAddExtentSelections(
   Widget widget,
   int	  xpos1,
   int	  ypos1,
   int	  xpos2,
   int	  ypos2)
{
	AddExtentSelections((XvwManagerWidget) widget, xpos1, ypos1,
			xpos2, ypos2, SELECTABLE);
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerPixelGeometry
|
|       Purpose: This routine will attempt to reconfigure a given
|		 widget to conform to a requested geometry.  
|
|         Input: widget  - the widget that is being configured
|                request - the requested geometry for the widget 
|                reply   - the size that the routine will agree to 
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
void ManagerPixelGeometry(
   Widget           widget,
   XtWidgetGeometry *request,
   XtWidgetGeometry *reply)
{
	XvwManagerWidgetConstraints kman = kconstraint(widget);

	Position  x, y;
	Dimension width, height, border_width;


	/*
	 *  Get the widget's current geometry values
	 */
	ManagerGeometry(widget, &x, &y, &width, &height, &border_width);

	/*
	 *  See if a new configuration is being requested
	 */
	if (request != NULL)
	{
	   if (request->request_mode & CWX)
	      x = request->x;
	   if (request->request_mode & CWY)
	      y = request->y;
	   if (request->request_mode & CWWidth)
	      width = request->width;
	   if (request->request_mode & CWHeight)
	      height = request->height;
	   if (request->request_mode & CWBorderWidth)
	      border_width = request->border_width;
	}

	/*
	 *  If this is not a QueryOnly then go ahead and resize the child
	 */
	if (request == NULL || !(request->request_mode & XtCWQueryOnly))
	{
	   XvwManagerWidget xwid = kwidget(XtParent(widget));

	   if (CheckIfSelection(xwid, widget)) EraseSelection(xwid, widget);
	   ConfigureWidget(widget, x, y, width, height, border_width);
	}

	/*
	 *  See if a reply configuration is desired.
	 */
	if (reply != NULL)
	{
	   reply->request_mode = request->request_mode;

	   ManagerGeometry(widget, &x, &y, &width, &height, &border_width);
	   reply->x = x; reply->y = y; reply->width = width;
	   reply->height = height; reply->border_width = border_width;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerResizeObject 
|
|       Purpose:
|
|         Input: wclass -
|                widget -
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
void ManagerResizeObject(
   WidgetClass wclass,
   Widget      widget)
{
	if (wclass != objectClass && wclass->core_class.superclass)
	   ManagerResizeObject(wclass->core_class.superclass, widget);

	if (wclass->core_class.resize != NULL)
	   (*(wclass->core_class.resize))(widget);
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerRealizeObject 
|
|       Purpose:
|
|         Input: wclass -
|                widget -
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
void ManagerRealizeObject(
   WidgetClass wclass,
   Widget      widget)
{
	XvwManagerGadget xobj;

	if (wclass != objectClass && wclass->core_class.superclass)
	   ManagerRealizeObject(wclass->core_class.superclass, widget);
	else
	{
	   xobj = (XvwManagerGadget) widget;
	   xobj->manager.realized = TRUE;
	}

	if (wclass->core_class.realize != NULL)
	   (*(wclass->core_class.realize))(widget, NULL, NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerGeometry
|
|       Purpose: Returns the geometry associated with the object.
|
|         Input: object - object to return the display
|        Output: x            - the x position of the object
|		 y            - the y position of the object
|	   	 width        - the width of the object
|		 height       - the heigth of the object
|		 border_width - the border width of the object
|       Returns: TRUE (1) on success, FALSE (0) on failure
|    Written By: Mark Young & John Salas
|          Date: Feb 15, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
Boolean ManagerGeometry(
   Widget    widget,
   Position  *x,
   Position  *y,
   Dimension *width,
   Dimension *height,
   Dimension *border_width)
{
	RectObj	    robj;
	RectObjPart *rect;
	CorePart    *core;


	/*
	 *  Make sure we have an object to retrieve the geometry
	 */
	if (widget == NULL)
	   return(FALSE);

	if (XtIsWidget(widget))
	{
	   core = (CorePart *) &widget->core;

	   if (x != NULL) *x = core->x;
	   if (y != NULL) *y = core->y;
	   if (width  != NULL) *width  = core->width;
	   if (height != NULL) *height = core->height;
	   if (border_width != NULL) *border_width = core->border_width;
	}
	else if (XtIsRectObj(widget))
	{
	   robj = (RectObj) widget;
	   rect = (RectObjPart *) &robj->rectangle;

	   if (x != NULL) *x = rect->x;
	   if (y != NULL) *y = rect->y;
	   if (width  != NULL) *width  = rect->width;
	   if (height != NULL) *height = rect->height;
	   if (border_width != NULL) *border_width = rect->border_width;
	}
	else return(FALSE);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerActionHandler
|
|       Purpose:
|
|         Input: 
|    Written By: Mark Young
|          Date: Jun 30, 1993
|
------------------------------------------------------------*/

void ManagerActionHandler(
   xvobject object,
   kaddr    client_data,
   XEvent   *event)
{
	Widget widget = xvw_widget(object);
	XvwManagerWidget xwid = (XvwManagerWidget) client_data;

	Position x, y;
	int      xpos, ypos;

	/*
	 *  If the object is a Manager then put it into select mode
	 */
	if (XtIsSubclass(widget, ManagerWidgetClass))
	{
	   /*
	    *  Need to make sure that selecting is allowed on the object
	    *  first.
	    */
	   if (xwid->manager.allow_selecting == TRUE)
	      ManagerSelect(xwid, TRUE);
	}
	else
	{
	   Widget selection;

	   XtTranslateCoords((Widget) xwid, 0, 0, &x, &y);
	   xpos = event->xbutton.x_root - x;
	   ypos = event->xbutton.y_root - y;

	   selection = LocateSelection(xwid, xpos, ypos, SELECTABLE);
	   if (selection == (Widget) xwid)
	      ManagerSelect(xwid, FALSE);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerSetWMColormap
|
|       Purpose: If the widget's colormap is different than the toplevel then
|		 we need to update the colormap list. 
|         Input: xwid 
|    Written By: Mark Young
|          Date: Oct 23, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */
void ManagerSetWMColormap(
   XvwManagerWidget xwid)
{
	xvobject object   = xvw_object((Widget) xwid);
	xvobject toplevel = xvw_toplevel(object);
	Widget   topwid   = xvw_widget(toplevel);

	int    i, num = 0;
	Window *windows = NULL;

	if (xwid->core.colormap == topwid->core.colormap ||
	    xvw_check_realized(object)   == FALSE 	 ||
	    xvw_check_realized(toplevel) == FALSE)
	{
	   return;
	}

	XGetWMColormapWindows(xvw_display(toplevel), xvw_window(toplevel),
			      &windows, &num);
	for (i = 0; i < num; i++)
	{
	   if (windows[i] == xwid->core.window)
	      return;
	}
	if (num == 0) num = 1;

	windows = (Window *) krealloc(windows, sizeof(Window) * (num+1));
	windows[num-1] = xwid->core.window;
	windows[num]   = topwid->core.window;
	XSetWMColormapWindows(xvw_display(toplevel), xvw_window(toplevel),
			windows, num+1);
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerSetSelect - select an item
|
|       Purpose: Resource attribute for selecting an item.  The
|		 attribute XVW_SELECT_REPLACE with the xvobject
|		 to be selected.  If NULL then it clears all currently
|		 selected items.  Either way, the previously selected
|		 items are cleared.
|
|         Input: xvobject - the parent object in which we are to select the item
|                attribute - the XVW_SELECT_REPLACE
|		 calldata - the item to be selected
|    Written By: Mark Young and Daniele Argiro
|          Date: Jun 18, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
int ManagerSetSelect(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	xvobject selobj = *(xvobject *) calldata;
	Widget   widget = xvw_widget(object);
	XvwManagerWidget xwid = kwidget(widget);


	if (kstrcmp(attribute, XVW_SELECT_ADD_ALL) == 0)
	{
	   AddAllSelections(xwid, SELECTABLE);
	}
	else if (kstrcmp(attribute, XVW_SELECT_DELETE_ALL) == 0)
	{
	   DeleteAllSelections(xwid);
	}
	else if (karray_locate((char **) xwid->composite.children,
		xvw_widget(selobj), xwid->composite.num_children) != -1)
	{
	   if (kstrcmp(attribute, XVW_SELECT_REPLACE) == 0)
	   {
	      DeleteAllSelections(xwid);
	      AddSelection(xwid, xvw_widget(selobj));
	   }
	   else if (kstrcmp(attribute, XVW_SELECT_ADD) == 0)
	   {
	      AddSelection(xwid, xvw_widget(selobj));
	   }
	   else if (kstrcmp(attribute, XVW_SELECT_DELETE) == 0)
	   {
	      DeleteSelection(xwid, xvw_widget(selobj));
	   }
	}
	else if (selobj)
	{
	   kinfo(KSTANDARD,"ManagerSetSelect:  Object '%s' is not a child of \
Manager '%s'", xvw_name(selobj), xvw_name(object));
	   return(FALSE);
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerSetGroup - group an item
|
|       Purpose: Resource attribute for grouping an item.
|
|         Input: xvobject - the child object in which we are to group the item
|                attribute - the XVW_GROUP_ADD, XVW_GROUP_DELETE, etc
|		 calldata - the item to be grouped
|    Written By: Mark Young
|          Date: Dec 06, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */
int ManagerSetGroup(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	xvobject sibling = *(xvobject *) calldata;
	Widget   widget = xvw_widget(object);
	XvwManagerWidget xwid = kwidget(XtParent(widget));
        XvwManagerWidgetConstraints kman = kconstraint(widget);

	int  i;

	if (kstrcmp(attribute, XVW_GROUP_DELETE_ALL) == 0)
	{
	   for (i = 0; i < kman->manager.group_size; i++)
	      DeleteSelection(xwid, xvw_widget(kman->manager.group[i]));

	   kfree(kman->manager.group);
	   kman->manager.group_size = 0;
	}
	else if (kstrcmp(attribute, XVW_GROUP_ADD) == 0)
	{
	   if (karray_locate((char **) kman->manager.group,
                (kaddr) sibling, kman->manager.group_size) != -1)
	      return(TRUE);

	   kman->manager.group = (xvobject *) karray_add((char **)
		kman->manager.group, (kaddr) sibling,
		kman->manager.group_size++);

	   if (CheckIfSelection(xwid, xvw_widget(sibling)))
	      AddSelection(xwid, xvw_widget(sibling));
	}
	else if (kstrcmp(attribute, XVW_SELECT_DELETE) == 0)
	{
	   if (karray_locate((char **) kman->manager.group,
                (kaddr) sibling, kman->manager.group_size) == -1)
	      return(TRUE);

	   kman->manager.group = (xvobject *) karray_delete((char **)
		kman->manager.group, (kaddr) sibling,
		kman->manager.group_size--);

	   if (CheckIfSelection(xwid, xvw_widget(sibling)))
	      DeleteSelection(xwid, xvw_widget(sibling));
	}
	else
	{
	   kerror(XVOBJECTS,"ManagerSetGroup","invalid attribute %s",attribute);
	   return(FALSE);
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerGetSelect - return the selected item
|
|       Purpose: Resource attribute for returning the selected item.  The
|		 attribute XVW_SELECT_REPLACE with the xvobject
|		 to be selected.  If NULL then it clears all currently
|		 selected items.  Either way, the previously selected
|		 items are cleared.
|
|         Input: xvobject - the parent object in which we are to return the item
|                attribute - the XVW_SELECT_REPLACE
|		 calldata - the item to be selected
|    Written By: Mark Young and Daniele Argiro
|          Date: Jun 18, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
int ManagerGetSelect(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	xvobject *selobj = (xvobject *) calldata;

	Widget   widget = xvw_widget(object);
        XvwManagerWidget xwid = kwidget(widget);


	if (!XtIsSubclass(widget, ManagerWidgetClass))
	   return(FALSE);

	/*
	 *  Return the first selection or NULL if non selected.
	 */
	if (xwid->manager.numsels == 0)
	   *selobj = NULL;
	else
	   *selobj = xvw_object(xwid->manager.widgetsels[0]);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerCreateGrid - creates and sets the background grid
|
|       Purpose: Creates and sets the background pixmap associated with
|		 the manager widget.
|
|         Input: widget - the widget to set the grid background for
|    Written By: Mark Young
|          Date: May 22, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
void ManagerCreateGrid(
   Widget widget)
{
        XvwManagerWidget xwid = kwidget(widget);
	xvobject object = xvw_object(widget);

	Pixmap    grid;
	XSegment  seg[2];
	XGCValues values;
	unsigned  long mask;
	GC        fill, gc1, gc2;
	int       width, height;
	int	  buffer_dist = xwid->manager.buffer_dist;

	static int  length   = 2;
	static char dotted[] = { 1, 1 };

	/*
	 *  Since we need the window in order to create the GC and Pixmap
	 *  we better make sure the widget has been realized or else we'll
	 *  end up giving the ole X protocol dump
	 */
	if (!xvw_check_realized(object))
	   return;

	/*
	 *  Set the GC to use LineOnOffDash.
	 */
	values.line_width = 1;
	values.join_style = JoinMiter;
	values.cap_style  = CapButt;
	values.line_style = LineDoubleDash;
	values.background = xwid->core.background_pixel;
	mask = GCLineWidth | GCForeground | GCBackground;

	values.foreground = xwid->core.background_pixel;
	fill = XCreateGC(XtDisplay(widget), XtWindow(widget), mask, &values);

	values.foreground = xwid->manager.foreground_pixel;
	gc1 = XCreateGC(XtDisplay(widget), XtWindow(widget), mask, &values);

	values.foreground = xwid->manager.grid_pixel;
	gc2 = XCreateGC(XtDisplay(widget), XtWindow(widget), mask, &values);

	/*
	 *  Set the width and height
	 */
	if (xwid->manager.gridsize > 0)
	{
	   width = height = xwid->manager.gridsize;
	}
	else
	{
	   if ((width  = (int) xwid->manager.xsnap) <= 0) width = 10;
	   if ((height = (int) xwid->manager.ysnap) <= 0) height = 10;
	}

	/*
	 *  Create the background pixmap...  If the create pixmap fails
	 *  normally we are sure to die a violent death, but then life
	 *  is a risk.  We also know that the xvwidget's default Xlib
	 *  error handler is installed it will save us, but if the
	 *  programmer overrides then there isn't squat we can do...
	 *
	 *    well ok we could temporary install our own handler,
	 *    but i'm just not in the mood.   
	 */
	if ((grid = XCreatePixmap(XtDisplay(widget), XtWindow(widget),
			width, height, widget->core.depth)) == None)
	{
	   return;
	}

	/*
	 *  The grid will be accomplished by drawing a line from (0,0) to
	 *  (width,0)  and (0,0) to (0,height).  This pattern will be set to
	 *  the window's background.  Thus, the background will be tiled
	 *  with the pixmap giving us the effect of a grid.  The real
	 *  advantage is that having a tiled background means the server
	 *  will take care of maintaining it.
	 */
	XFillRectangle(XtDisplay(widget), grid, fill, 0, 0, width, height);
	if (width >= 20 && height >= 20)
	{
	   seg[0].x1 = buffer_dist-1;	seg[0].y1 = 0;
	   seg[0].x2 = buffer_dist-1;	seg[0].y2 = height-1;
	   seg[1].x1 = 0;		seg[1].y1 = buffer_dist-1;
	   seg[1].x2 = width-1; 	seg[1].y2 = buffer_dist-1;
	   XDrawSegments(XtDisplay(widget), grid, gc2, seg, 2);
	}
	else
	{
	   XSetDashes(XtDisplay(widget), gc1, 0, dotted, length);
	   XSetLineAttributes(XtDisplay(widget), gc1, values.line_width,
		values.line_style, values.cap_style, values.join_style);
	}
	seg[0].x1 = buffer_dist;	seg[0].y1 = 0;
	seg[0].x2 = buffer_dist;	seg[0].y2 = height-1;
	seg[1].x1 = 0;			seg[1].y1 = buffer_dist;
	seg[1].x2 = width-1; 		seg[1].y2 = buffer_dist;
	XDrawSegments(XtDisplay(widget), grid, gc1, seg, 2);

	/*
	 *  Go ahead and display the grid by setting the background grid
	 *  on ourselves if they want a grid.  If not then go ahead and
	 *  set it to be unspecified.
	 */
	if (xwid->manager.gridtype == KMANAGER_GRID_OFF)
	{
	   xvw_set_attribute(object,XVW_BACKGROUND_PIXMAP, XtUnspecifiedPixmap);
	}
	else if (xwid->manager.gridtype == KMANAGER_GRID_ON &&
		 (xwid->core.background_pixmap == xwid->manager.grid ||
		  xwid->core.background_pixmap == XtUnspecifiedPixmap))
	{
	   xvw_set_attribute(object, XVW_BACKGROUND_PIXMAP, grid);
	}

	/*
	 *  Cleanup time...  
	 */
	XFreeGC(XtDisplay(widget), fill);
	XFreeGC(XtDisplay(widget), gc1);
	XFreeGC(XtDisplay(widget), gc2);
	if (xwid->manager.grid != None)
	   XFreePixmap(XtDisplay(widget), xwid->manager.grid);

	xwid->manager.grid = grid;
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerWidgetHandler - handles the widget management
|
|       Purpose: This routine is the widget handler.
|
|         Input: object - the inputonly object in which we installed the handler
|                client_data - the selection being handled entry
|                event  - the type of event that invoked us
|    Written By: Mark Young
|          Date: Jul 06, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
void ManagerWidgetHandler(
   xvobject object,
   kaddr    client_data,
   XEvent   *event)
{
	Widget selection;
	Widget widget = (Widget) client_data;
	XvwManagerWidget xwid = kwidget(widget);
	xvobject parent = xvw_object(widget);

	Position x, y;
        Widget   *widgetsels;
	unsigned int wtmp, htmp;
	int      i, xpos, ypos, xtmp, ytmp;


	kinfo(KDEBUG,"ManagerWidgetHandler: %s", XtName(widget));
	if (event->type == ButtonPress)
	{
	   XtTranslateCoords((Widget) xwid, 0, 0, &x, &y);
	   xpos = event->xbutton.x_root - x;
	   ypos = event->xbutton.y_root - y;

	   if ((selection = LocateSelection(xwid, xpos, ypos,
			SELECTABLE)) == NULL)
	   {
	      return;
	   }

	   if (selection != (Widget) xwid)
	   {
	      if (!(event->xbutton.state & ShiftMask) &&
	          !CheckIfSelection(xwid, selection))
	      {
	         DeleteAllSelections(xwid);
	      }
	      AddSelection(xwid, selection);
	   }
	   else
	   {
	      if (!(event->xbutton.state & ShiftMask))
	         DeleteAllSelections(xwid);

	      if ((object == parent || object == xwid->manager.inputonly) &&
	          xvw_getextent(object, &xtmp, &ytmp, &wtmp, &htmp) == TRUE)
	      {
	         AddExtentSelections(xwid, xtmp, ytmp, xtmp+wtmp, ytmp+htmp,
				SELECTABLE);
	         return;
	      }
	   }

	   if (event->xbutton.button == Button1)
	   {
	      xwid->manager.active = TRUE;
	      ExtentSelections(xwid, xpos, ypos);
	      if (xwid->manager.gridtype == KMANAGER_GRID_EDIT)
	      {
		 xvw_set_attribute(parent, XVW_BACKGROUND_PIXMAP,
			xwid->manager.grid);
	      }

	      if (selection != (Widget) xwid)
	         xvw_set_attribute(object, XVW_CURSORNAME, EDIT_CURSOR);

#if 0
	      if (XtIsSubclass(selection, ManagerWidgetClass))
	         xvw_busy(xvw_object(selection), TRUE);
#endif
	   }
	   else if (event->xbutton.button == Button2)
	   {
	      selection = LocateSelection(xwid, xpos, ypos, MENUABLE);
	      if (selection != NULL)
		 xvw_activate_menu(xvw_object(selection));
	   }
	   else if (event->xbutton.button == Button3)
	   {
	      WidgetInfo(xvw_object(selection));
	   }
	}
	else if (event->type == MotionNotify)
	{
	   if (xwid->manager.active == TRUE &&
	       event->xmotion.state & Button1Mask &&
	       xwid->manager.numsels > 0)
	   {
	      WidgetHandler(xwid, event);
	   }
	}
	else if (event->type == ButtonRelease &&
		 event->xbutton.button == Button1)
	{
	   xwid->manager.active = FALSE;
	   xvw_set_attribute(object, XVW_CURSORNAME, SELECT_CURSOR);
	   if (xwid->manager.gridtype == KMANAGER_GRID_EDIT)
	   {
	      xvw_set_attribute(parent, XVW_BACKGROUND_PIXMAP,
			XtUnspecifiedPixmap);
	   }
 
           for (i = 0, widgetsels = xwid->manager.widgetsels;
                i < xwid->manager.numsels; i++, widgetsels++)
           {
              object = xvw_object(*widgetsels);
              XtCallCallbacks(widget, XVW_GEOMETRY_CALLBACK, &object);
           }

#if 0
	   if (XtIsSubclass(selection, ManagerWidgetClass))
	      xvw_busy(xvw_object(selection), FALSE);
#endif
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: ManagerSelect
|
|       Purpose:
|
|         Input: 
|    Written By: Mark Young
|          Date: Jun 30, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
void ManagerSelect(
   XvwManagerWidget xwid,
   int		    select)
{
	xvobject inputonly = xwid->manager.inputonly;
	xvobject manager = xvw_object((Widget) xwid);

	Widget *child;
	int    i, busy;


	if (!inputonly)
	{
	   inputonly = xvw_create_inputonly(manager, "inputonly");
	   xvw_set_attributes(inputonly,
			XVW_CURSORNAME,  SELECT_CURSOR,
			XVW_TACK_EDGE,   KMANAGER_TACK_ALL,
			XVW_VERT_DIST,   0,
			XVW_HORIZ_DIST,  0,
			XVW_SELECTABLE,  FALSE,
			XVW_MENUABLE,    FALSE,
			XVW_RESIZABLE,   FALSE,
			NULL);

           xvw_add_event(manager,SubstructureNotifyMask,RaiseInputWindow,NULL);
	   xvw_add_event(inputonly, ButtonPressMask | ButtonMotionMask |
		        ButtonReleaseMask, ManagerWidgetHandler, (kaddr) xwid);

	   xvw_add_action(inputonly,"Meta<Btn1Down>", 
                      ManagerActionHandler, (kaddr) xwid, TRUE);
	   xvw_add_action(inputonly,"Shift<Btn1Down>", 
		      ManagerActionHandler, (kaddr) xwid, TRUE);
	   xwid->manager.inputonly = inputonly;
	}

	/*
	 *  If the child is a Manager then put it into select mode (edit)
	 *  by activating it's busy window.  Next is to clear all it's
	 *  selections.
	 */
	if (select)
	{
	   /*
	    *  See if the window is currently busy.  If it isn't then make
	    *  the input only window appear and change the background so
	    *  that the grid appears.
	    */
	   xvw_get_attribute(inputonly, XVW_BUSY, &busy);
	   if (!busy)
	   {
	      if (xwid->manager.gridtype == KMANAGER_GRID_SELECT &&
		  xwid->core.background_pixmap == XtUnspecifiedPixmap)
	      {
	         xvw_set_attribute(manager, XVW_BACKGROUND_PIXMAP,
			   xwid->manager.grid);
	      }
	      busy = xwid->manager.editing = TRUE;
	      xvw_set_attribute(inputonly, XVW_BUSY, busy);
	      XtCallCallbacks((Widget) xwid, XVW_SELECT_CALLBACK, &busy);
	   }

	   child  = xwid->composite.children;
	   for (i = 0; i < xwid->composite.num_children; i++, child++)
	   {
	       if (XtIsSubclass(*child, ManagerWidgetClass))
		  ManagerSelect((XvwManagerWidget) *child, FALSE);
	   }
	}
	else
	{
	   /*
	    *  See if the window is currently busy.  If it is then make
	    *  the input only window appear and change the background so
	    *  that the grid appears.
	    */
	   xvw_get_attribute(inputonly, XVW_BUSY, &busy);
	   if (busy)
	   {
	      if (xwid->manager.gridtype == KMANAGER_GRID_SELECT &&
	          xwid->core.background_pixmap == xwid->manager.grid)
	      {
		 xvw_set_attribute(manager, XVW_BACKGROUND_PIXMAP,
			XtUnspecifiedPixmap);
	      }

	      /*
	       *  Clear all currently selected objects and make our busy
	       *  window go away.
	       */
	      DeleteAllSelections(xwid);
	      busy = xwid->manager.editing = FALSE;
	      xvw_set_attribute(inputonly, XVW_BUSY, busy);
	      XtCallCallbacks((Widget) xwid, XVW_SELECT_CALLBACK, &busy);
	   }

	   child  = xwid->composite.children;
	   for (i = 0; i < xwid->composite.num_children; i++, child++)
	   {
	       if (XtIsSubclass(*child, ManagerWidgetClass))
		  ManagerSelect((XvwManagerWidget) *child, FALSE);
	   }
	}
}
