 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                 Line Gadget Routines
   >>>>
   >>>>	Private:
   >>>>	 Static:
   >>>>			ClassInitialize()
   >>>>			Redisplay()
   >>>>			SetValues()
   >>>>			RecomputePosition()
   >>>>			ChangeGeometry()
   >>>>			PickGadget()
   >>>>			EraseSelection()
   >>>>			RefreshSelection()
   >>>>  Public:
   >>>>			xvw_create_line()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <xvisual/LineP.h>

static void ClassInitialize	 PROTO((void));
static void Redisplay		 PROTO((Widget, XEvent *, Region));
static Boolean SetValues	 PROTO((Widget, Widget, Widget, ArgList,
					Cardinal *));
static void RecomputePosition	 PROTO((Widget));
static void ChangeGeometry	 PROTO((Widget));
static int  PickGadget		 PROTO((Widget, int, int));
static void EraseSelection	 PROTO((Widget));
static void RefreshSelection	 PROTO((Widget));


/*-------------------------------------------------------
|
|   Full class attributes
|
--------------------------------------------------------*/

static xvattribute attributes[] = {
{XVW_LINE_XBEGIN,    NULL,    XtRDouble,    NULL},
{XVW_LINE_YBEGIN,    NULL,    XtRDouble,    NULL},
{XVW_LINE_XEND,      NULL,    XtRDouble,    NULL},
{XVW_LINE_YEND,      NULL,    XtRDouble,    NULL},
};

/*-------------------------------------------------------
|
|   Full class record constant
|
--------------------------------------------------------*/

#define offset(field) XtOffsetOf(XvwLineGadgetRec, line.field)

static XtResource resources[] = { 
{XVW_LINE_XBEGIN, NULL, XtRDouble, sizeof(double), 
      offset(begin.x), XtRString, (XtPointer) "0.0"},
{XVW_LINE_YBEGIN, NULL, XtRDouble, sizeof(double), 
      offset(begin.y), XtRString, (XtPointer) "0.0"},
{XVW_LINE_XEND, NULL, XtRDouble, sizeof(double), 
      offset(end.x), XtRString, (XtPointer) "1.0"},
{XVW_LINE_YEND, NULL, XtRDouble, sizeof(double), 
      offset(end.y), XtRString, (XtPointer) "1.0"},
};
#undef offset


#define SUPERCLASS (&xvwGraphicsGadgetClassRec)

XvwLineGadgetClassRec xvwLineGadgetClassRec =
{
  {
    (WidgetClass) SUPERCLASS,		/* superclass		  */	
    "Line",				/* class_name		  */
    sizeof(XvwLineGadgetRec),		/* size			  */
    ClassInitialize,			/* class_initialize	  */
    NULL,				/* class_part_initialize  */
    FALSE,				/* class_inited		  */
    NULL,				/* initialize		  */
    NULL,				/* initialize_hook	  */
    NULL,				/* realize		  */
    NULL,				/* actions		  */
    0,					/* num_actions		  */
    resources,				/* resources		  */
    knumber(resources),		/* resource_count	  */
    NULLQUARK,				/* xrm_class		  */
    FALSE,				/* compress_motion	  */
    FALSE,				/* compress_exposure	  */
    FALSE,				/* compress_enterleave    */
    FALSE,				/* visible_interest	  */
    NULL,				/* destroy		  */
    XtInheritResize,			/* resize		  */
    Redisplay,				/* expose		  */
    SetValues,				/* set_values		  */
    NULL,				/* set_values_hook	  */
    XtInheritSetValuesAlmost,		/* set_values_almost	  */
    NULL,				/* get_values_hook	  */
    NULL,				/* accept_focus		  */
    XtVersion,				/* version		  */
    NULL,				/* callback_private	  */
    NULL,				/* tm_table		  */
    NULL,				/* query_geometry	  */
    NULL,				/* display_accelerator	  */
    NULL				/* extension		  */
  },  /* RectObjClass fields initialization */
  {
    PickGadget,				/* pick object proc       */
    EraseSelection,                     /* erase selected proc    */
    RefreshSelection,		        /* refresh selection proc */
    ChangeGeometry,			/* change geometry proc   */
  },  /* XvwManagerGadgetClass fields initialization */
  {
    RecomputePosition,			/* recompute position proc    */
  },  /* XvwGraphicsGadgetClass fields initialization */
  {
    NULL,				/* extension - not used   */
  },  /* XvwLineGadgetClass fields initialization */
};
#undef SUPERCLASS

/*
 * xvwLineGadgetClass for public consumption
 */
WidgetClass xvwLineGadgetClass = (WidgetClass) &xvwLineGadgetClassRec;


/*-----------------------------------------------------------
|
|  Routine Name: ClassInitialize
|
|       Purpose: This method is called the first time an  
|                instance of line object class has been created.
|                It will initialize all the class attributes. 
|
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
------------------------------------------------------------*/

static void ClassInitialize(void)
{
	xvw_init_attributes(xvwLineGadgetClass, attributes,
		knumber(attributes), NULL, 0, 
		"$DESIGN/objects/library/xvisual/uis/Line.pane");
	xvw_load_resources("$DESIGN/objects/library/xvisual/app-defaults/Line");
}

/*-----------------------------------------------------------
|
|  Routine Name: Redisplay
|
|       Purpose: This routine will redraw the line object in response
|                to an expose event. 
|
|         Input: widget - the line object that was exposed 
|                event  - the event that caused the redraw
|                region - the region that was exposed 
|
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Redisplay(
   Widget widget,
   XEvent *event,
   Region region)
{
	XvwLineGadget xobj = (XvwLineGadget) widget;


	X2D_draw_line(xobj->graphics.id, xobj->line.begin, xobj->line.end,
			&xobj->graphics.fg);
}

/*-----------------------------------------------------------
|
|  Routine Name: SetValues
|
|       Purpose: If the the size or postion of the line has changed, 
|		 recalculate the size and position of the line.
|
|         Input: current - the object containing current settings 
|                request - the object containing requested settings
|                new     - the object processed through all set values methods 
|
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean SetValues(
   Widget   current,
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
	XvwLineGadget cobj = (XvwLineGadget) current;
	XvwLineGadget nobj = (XvwLineGadget) new;


	if (cobj->line.begin.x != nobj->line.begin.x ||
	    cobj->line.begin.y != nobj->line.begin.y ||
	    cobj->line.end.x   != nobj->line.end.x   ||
	    cobj->line.end.y   != nobj->line.end.y)
	{
	   RecomputePosition(new);
	}
	return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: RecomputePosition
|
|       Purpose: Given a new world coordinate line begin 
|		 and end point, recompute the new device 
|		 coordinate position of the line. 
|
|         Input: widget - the line object 
|
|        Output: None directly 
|
|    Written By: Mark Young
|          Date: Jun 29, 1992 10:35
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void RecomputePosition(
   Widget widget)
{
	XvwLineGadget xobj = (XvwLineGadget) widget;

	int id = xobj->graphics.id;
	Real xpos1, xpos2, ypos1, ypos2;
	XtWidgetGeometry temp;
	int bw = widget->core.border_width << 1;


	X2D_convert_point_wc_to_dc(id, xobj->line.begin, &xpos1, &ypos1);
	X2D_convert_point_wc_to_dc(id, xobj->line.end, &xpos2, &ypos2);

	temp.request_mode = CWWidth | CWHeight | CWX | CWY;

	temp.x = (Position) kmin(xpos1, xpos2);
	temp.y = (Position) kmin(ypos1, ypos2);

	if (kabs(xpos2 - xpos1) < bw)
	    temp.width = 0;
	else temp.width  = (Dimension) kabs(xpos2 - xpos1) - bw;

	if (kabs(ypos2 - ypos1) < bw)
	    temp.height = 0;
	else temp.height = (Dimension) kabs(ypos2 - ypos1) - bw;

	(void) XtMakeGeometryRequest(widget, &temp, &temp);
}

/*-----------------------------------------------------------
|
|  Routine Name: ChangeGeometry
|
|       Purpose: Given a new device coordinate description of the
|                line, recompute the world coordinate begin and end 
|                point.
|
|         Input: widget - the line object
|
|        Output: None directly 
|
|    Written By: Mark Young
|          Date: Jun 29, 1992 10:35
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ChangeGeometry(
   Widget widget)
{
	XvwLineGadget xobj = (XvwLineGadget) widget;

	int id = xobj->graphics.id;
	Real xpos1, xpos2, ypos1, ypos2;
	int bw = widget->core.border_width << 1;


	if (xobj->manager.realized == FALSE)
	   return;

	if (xobj->line.begin.x < xobj->line.end.x)
	{
	   xpos1 = (Real) widget->core.x;
	   xpos2 = (Real) ((int) widget->core.x + (int) widget->core.width +bw);
	}
	else
	{
	   xpos1 = (Real) ((int) widget->core.x + (int) widget->core.width +bw);
	   xpos2 = (Real) widget->core.x;
	}

	if (xobj->line.end.y < xobj->line.begin.y)
	{
	   ypos1 = (Real) widget->core.y;
	   ypos2 = (Real) ((int) widget->core.y + (int) widget->core.height+bw);
	}
	else
	{
	   ypos1 = (Real) ((int) widget->core.y + (int) widget->core.height+bw);
	   ypos2 = (Real) widget->core.y;
	}
	X2D_convert_point_dc_to_wc(id, xpos1, ypos1, &xobj->line.begin);
	X2D_convert_point_dc_to_wc(id, xpos2, ypos2, &xobj->line.end);
}

/*-----------------------------------------------------------
|
|  Routine Name: PickGadget
|
|       Purpose: Given a line object and an attempted picking location,
|                see if the line has been picked. 
|
|         Input: widget - the line object we are checking 
|                x      - the x device coordinate that the pick attempt was at 
|                y      - the y device coordinate that the pick attempt was at
|
|        Output: None 
|
|       Returns: TRUE (1) if object was successfully picked, FALSE (0) otherwise|
|    Written By: Mark Young
|          Date: Jun 29, 1992 10:35
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static int PickGadget(
   Widget widget,
   int    x,
   int    y)
{
	XvwLineGadget xobj = (XvwLineGadget) widget;

	Position boundy, deltax, deltay, rise, run, tolerance,
		 xpos1, xpos2, ypos1, ypos2;

	if (xobj->line.begin.x < xobj->line.end.x)
	{
	   xpos1 = widget->core.x;
	   xpos2 = widget->core.x  + widget->core.width;
	}
	else
	{
	   xpos1 = widget->core.x  + widget->core.width;
	   xpos2 = widget->core.x;
	}

	if (xobj->line.end.y < xobj->line.begin.y)
	{
	   ypos1 = widget->core.y;
	   ypos2 = widget->core.y + widget->core.height;
	}
	else
	{
	   ypos1 = widget->core.y + widget->core.height;
	   ypos2 = widget->core.y;
	}

	rise = ypos2 - ypos1;
	run = xpos2 - xpos1;

	deltay = y - ypos1;
	deltax = x - xpos1;

	tolerance = kmax(PICK_TOLERANCE, xobj->graphics.line_width);

	if (kabs(run) <= tolerance)
	{
	   if (kabs(deltax) <= tolerance)
	      return(TRUE);
	}
	else
	{
	   boundy = (rise * deltax) / run;
	   if (kabs(deltay - boundy) <= tolerance)
	      return(TRUE);
	}
	return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: EraseSelection
|
|       Purpose: Erase the boxes which had indicated that 
|                the line had been selected 
|
|         Input: widget - the line object whose selection we are erasing 
|
|        Output: None 
|
|    Written By: Mark Young
|          Date: Jun 29, 1992 10:35
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void EraseSelection(
   Widget widget)
{
	XvwLineGadget xobj = (XvwLineGadget) widget;

        Window  window;
        Display *display;

        Position tolerance, x, y, xpos1, xpos2, ypos1, ypos2;
        Dimension width, height;

        display = XtDisplay(widget);
        window  = XtWindowOfObject(widget);
        ManagerGeometry(widget, &x, &y, &width, &height, NULL);

        if (xobj->line.begin.x < xobj->line.end.x)
        {
           xpos1 = x;
           xpos2 = x + width;
        }
        else
        {
           xpos1 = x + width;
           xpos2 = x;
        }

        if (xobj->line.end.y < xobj->line.begin.y)
        {
           ypos1 = y;
           ypos2 = y + height;
        }
        else
        {
           ypos1 = y + height;
           ypos2 = y;
        }

	tolerance = kmax(PICK_TOLERANCE, xobj->graphics.line_width);

        XClearArea(display, window,
                       (int) (xpos1 - tolerance/2),
                       (int) (ypos1 - tolerance/2),
                       (unsigned int) tolerance,
                       (unsigned int) tolerance,
                       TRUE);
        XClearArea(display, window,
                       (int) (xpos2 - tolerance/2),
                       (int) (ypos2 - tolerance/2),
                       (unsigned int) tolerance,
                       (unsigned int) tolerance,
                       TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: RefreshSelection
|
|       Purpose: Draw little boxes at the ends of the line to
|                indicate that the line had been selected 
|
|         Input: widget - the line object whose selection we are refreshing
|
|        Output: None 
|
|    Written By: Mark Young
|          Date: Jun 29, 1992 10:35
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */ 
static void RefreshSelection(
   Widget widget)
{
	XvwLineGadget xobj = (XvwLineGadget) widget;

        Window  window;
        Display *display;

        Position tolerance, x, y, xpos1, xpos2, ypos1, ypos2;
        Dimension width, height;

        display = XtDisplay(widget);
        window  = XtWindowOfObject(widget);
        ManagerGeometry(widget, &x, &y, &width, &height, NULL);

        if (xobj->line.begin.x < xobj->line.end.x)
        {
           xpos1 = x;
           xpos2 = x + width;
        }
        else
        {
           xpos1 = x + width;
           xpos2 = x;
        }

        if (xobj->line.end.y < xobj->line.begin.y)
        {
           ypos1 = y;
           ypos2 = y + height;
        }
        else
        {
           ypos1 = y + height;
           ypos2 = y;
        }

	tolerance = kmax(PICK_TOLERANCE, xobj->graphics.line_width);

        XFillRectangle(display, window, xobj->manager.gc,
                       (int) (xpos1 - tolerance/2),
                       (int) (ypos1 - tolerance/2),
                       (unsigned int) tolerance,
                       (unsigned int) tolerance);
        XFillRectangle(display, window, xobj->manager.gc,
                       (int) (xpos2 - tolerance/2),
                       (int) (ypos2 - tolerance/2),
                       (unsigned int) tolerance,
                       (unsigned int) tolerance);
}


/************************************************************
*
*  Routine Name: xvw_create_line - create a line object.
*
*       Purpose: A line visual object supports the display of a line.
*                The (x, y) location of the line end points
*                are specified in world coordinates.  The world
*                coordinates are specified by the parent.  By default, these
*                world coordinates are from 0 to 1 unless otherwise specified.
*                So, for example, if the line is created as the child of
*                a manager object, its world coordinates are from 0 to 1.
*                In contrast, if the line is created as the child of an image,
*                the world coordinates will be dictated by the size of the
*                image; if it created as the child of a plot object, the world
*                coordinates will be dictated by the plot.
*
*         Input: parent - the parent object; NULL will cause a
*                         default toplevel to be created automatically
*                name   - the name with which to reference the object 
*
*        Output: None 
*
*	Returns: The line object on success, NULL on failure
*
*  Restrictions:
*    Written By: Mark Young 
*          Date: Jun 29, 1992 10:31
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xvobject xvw_create_line(
   xvobject parent,
   char   *name)
{
	xvobject  object;


	/*
	 *  Create the line object
	 */
	object = xvw_create(parent, FALSE, TRUE, name, LineGadgetClass);
	return(object);
}
