/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                Image Widget Routines
   >>>>
   >>>>	Private:
   >>>>	 Static:
   >>>>			ClassInitialize()
   >>>>			Initialize()
   >>>>			Realize()
   >>>>			Resize()
   >>>>			Redisplay()
   >>>>			Destroy()
   >>>>			SetValues()
   >>>>
   >>>>			ReloadCallback()
   >>>>			ColorCallback()
   >>>>			ImageCallback()
   >>>>			ResizeImage()
   >>>>			RedisplayImage()
   >>>>  Public:
   >>>>			xvw_create_image()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#if KOPSYS_LOCAL == KOPSYS_OSF
#undef _POSIX_SOURCE
#endif

#include "internals.h"
#include <xvisual/ImageP.h>


static void ClassInitialize PROTO((void));
static void Initialize	    PROTO((Widget, Widget, ArgList, Cardinal *));
static void Resize	    PROTO((Widget));
static void Redisplay	    PROTO((Widget, XEvent *, Region));
static void Destroy	    PROTO((Widget));
static Boolean SetValues    PROTO((Widget, Widget, Widget, ArgList,Cardinal *));
static void Realize	    PROTO((Widget, XtValueMask *,
				   XSetWindowAttributes *));
static void ReloadCallback  PROTO((kobject, char *, Widget, kdms_callback *));
static void ColorCallback   PROTO((xvobject, kaddr));
static void ImageCallback   PROTO((kobject, char *, Widget, kdms_callback *));
static void ResizeImage     PROTO((Widget));
static void RedisplayImage  PROTO((Widget, int, int, int, int, int));


#undef kwidget
#undef kwidgetclass

#define kwidget(widget)      (XvwImageWidget) (widget)
#define kwidgetclass(widget) (XvwImageWidgetClass) (widget)->core.widget_class
#define ImageBacking(xwid)   (xwid->image.backing /*&& xwid->composite.num_children > 0*/)

/*#undef KMITSHM_DEF*/

/*-------------------------------------------------------
|
|   Full class attributes
|
--------------------------------------------------------*/

static xvattribute attributes[] = {
{XVW_IMAGE_XOFFSET,	     NULL,    XtRInt,	   NULL},
{XVW_IMAGE_YOFFSET,	     NULL,    XtRInt,	   NULL},
{XVW_IMAGE_BANDNUM,	     NULL,    XtRInt,	   NULL},
{XVW_IMAGE_IMAGEOBJ,	     NULL,    XtRPointer,  NULL},
{XVW_IMAGE_CLIPOBJ,	     NULL,    XtRPointer,  NULL},
{XVW_IMAGE_XPOSITION,	     NULL,    XtRInt,	   NULL},
{XVW_IMAGE_YPOSITION,	     NULL,    XtRInt,	   NULL},
{XVW_IMAGE_PIXEL,	     NULL,    XtRPixel,	   NULL},
{XVW_IMAGE_VALUE,	     NULL,    XtRDouble,   NULL},
{XVW_IMAGE_BACKING,	     NULL,    XtRInt,	   XtRBoolean},
{XVW_IMAGE_ROI, 	     NULL,    XtRPointer,  NULL},
{XVW_IMAGE_ROI_SHAPE, 	     NULL,    XtRInt,	   NULL},
{XVW_IMAGE_ROI_POLICY, 	     NULL,    XtRInt,	   NULL},
{XVW_IMAGE_ROI_PRESENTATION, NULL,    XtRInt,	   NULL},
{XVW_IMAGE_ROI_MULTIBAND,    NULL,    XtRInt,	   XtRBoolean},
{XVW_IMAGE_ROI_MULTIPLE,     NULL,    XtRInt,	   XtRBoolean},
{XVW_IMAGE_RELOAD, 	     NULL,    XtRInt,	   XtRBoolean},
{XVW_IMAGE_REDISPLAY, 	     NULL,    XtRInt,	   XtRBoolean},
{XVW_IMAGE_COMPLEX_CONVERT,  NULL,    XtRInt,	   NULL},
{XVW_IMAGE_BAND_MAXNUM,	     NULL,    XtRInt,	   NULL},
{XVW_IMAGE_BAND_DIMENSIONS,  NULL,    XtRInt,	   NULL},
{XVW_IMAGE_DATA, 	     NULL,    XtRPointer,  NULL},
};

/*-------------------------------------------------------
|
|   Full class record constant
|
--------------------------------------------------------*/

static double default_value = 0.0;

#define offset(field) XtOffsetOf(XvwImageWidgetRec, image.field)

static XtResource resources[] = {

{XVW_IMAGE_XOFFSET, NULL, XtRInt, sizeof(int),
      offset(xoffset), XtRImmediate, (XtPointer) 0},
{XVW_IMAGE_YOFFSET, NULL, XtRInt, sizeof(int),
      offset(yoffset), XtRImmediate, (XtPointer) 0},
{XVW_IMAGE_BANDNUM, NULL, XtRInt, sizeof(int),
      offset(band_number), XtRImmediate, (XtPointer) 0},
{XVW_IMAGE_IMAGEOBJ, NULL, XtRPointer, sizeof(kobject),
      offset(iobject), XtRImmediate, (XtPointer) NULL},
{XVW_IMAGE_CLIPOBJ, NULL, XtRPointer, sizeof(kobject),
      offset(cobject), XtRImmediate, (XtPointer) NULL},
{XVW_IMAGE_XPOSITION, NULL, XtRInt, sizeof(int),
      offset(xposition), XtRImmediate, (XtPointer) 0},
{XVW_IMAGE_YPOSITION, NULL, XtRInt, sizeof(int),
      offset(yposition), XtRImmediate, (XtPointer) 0},
{XVW_IMAGE_PIXEL, NULL, XtRPixel, sizeof(Pixel),
      offset(pixel), XtRImmediate, (XtPointer) 0},
{XVW_IMAGE_VALUE, NULL, XtRDouble, sizeof(double),
      offset(value), XtRDouble, (XtPointer) &default_value},
{XVW_IMAGE_BACKING, NULL, XtRBoolean, sizeof(Boolean),
      offset(backing), XtRImmediate, (XtPointer) TRUE},
{XVW_IMAGE_ROI_SHAPE, NULL, XtRInt, sizeof(int),
      offset(roi_shape), XtRImmediate, (XtPointer) KIMAGE_ROI_RECTANGLE},
{XVW_IMAGE_ROI_POLICY, NULL, XtRInt, sizeof(int),
      offset(roi_policy), XtRImmediate, (XtPointer) KIMAGE_ROI_INSIDE},
{XVW_IMAGE_ROI_PRESENTATION, NULL, XtRInt, sizeof(int),
      offset(roi_presentation), XtRImmediate, (XtPointer) KIMAGE_ROI_IMAGE},
{XVW_IMAGE_ROI_MULTIBAND, NULL, XtRBoolean, sizeof(Boolean),
      offset(roi_multiband), XtRImmediate, (XtPointer) FALSE},
{XVW_IMAGE_ROI_MULTIPLE, NULL, XtRBoolean, sizeof(Boolean),
      offset(multiple_roi), XtRImmediate, (XtPointer) FALSE},
{XVW_IMAGE_RELOAD, NULL, XtRBoolean, sizeof(Boolean),
      offset(reload), XtRImmediate, (XtPointer) FALSE},
{XVW_IMAGE_REDISPLAY, NULL, XtRBoolean, sizeof(Boolean),
      offset(redisplay), XtRImmediate, (XtPointer) FALSE},
{XVW_IMAGE_COMPLEX_CONVERT, NULL, XtRInt, sizeof(int),
      offset(complex_convert), XtRImmediate, (XtPointer) KLOGMAGP1},
{XVW_IMAGE_BAND_DIMENSIONS, NULL, XtRInt, sizeof(int),
      offset(band_dimensions), XtRImmediate,
      (XtPointer) (KIMAGE_ELEMENTS | KIMAGE_DEPTH | KIMAGE_TIME) },
{XVW_IMAGE_BAND_MAXNUM, NULL, XtRInt, sizeof(int),
      offset(band_maxnum), XtRImmediate, (XtPointer) 0},
{XVW_IMAGE_DATA, NULL, XtRPointer, sizeof(XtPointer),
      offset(idata), XtRImmediate, (XtPointer) NULL},
{NULL, NULL, XtRInt, sizeof(int), offset(reload_type),
      XtRImmediate, (XtPointer) IMAGE_OBJECT_RELOAD},

{XVW_CURSOR, NULL, XtRCursor, sizeof(Cursor),
      XtOffsetOf(XvwImageWidgetRec, manager.cursor), XtRString,
      (XtPointer) "$DESIGN/objects/library/xvisual/misc/cursors/cross"},
{XVW_BACKGROUND, NULL, XtRPixel, sizeof(Pixel),
      XtOffsetOf(XvwImageWidgetRec, core.background_pixel), XtRString,
      (XtPointer) "black"},
};
#undef offset


#define SUPERCLASS (&xvwColorWidgetClassRec)

XvwImageWidgetClassRec xvwImageWidgetClassRec =
{
  {
    (WidgetClass) SUPERCLASS,		/* superclass		  */	
    "Image",				/* class_name		  */
    sizeof(XvwImageWidgetRec),		/* size			  */
    ClassInitialize,			/* class_initialize	  */
    NULL,				/* class_part_initialize  */
    FALSE,				/* class_inited		  */
    Initialize,				/* initialize		  */
    NULL,				/* initialize_hook	  */
    Realize,				/* realize		  */
    NULL,				/* actions		  */
    0,					/* num_actions		  */
    resources,				/* resources		  */
    knumber(resources),			/* resource_count	  */
    NULLQUARK,				/* xrm_class		  */
    TRUE,				/* compress_motion	  */
    XtExposeCompressMaximal,		/* compress_exposure	  */
    TRUE,				/* compress_enterleave    */
    FALSE,				/* visible_interest	  */
    Destroy,				/* destroy		  */
    NULL,				/* resize		  */
    Redisplay,				/* expose		  */
    SetValues,				/* set_values		  */
    NULL,				/* set_values_hook	  */
    XtInheritSetValuesAlmost,		/* set_values_almost	  */
    NULL,				/* get_values_hook	  */
    NULL,				/* accept_focus		  */
    XtVersion,				/* version		  */
    NULL,				/* callback_private	  */
    XtInheritTranslations,		/* tm_table		  */
    NULL,				/* query_geometry	  */
    XtInheritDisplayAccelerator,	/* display_accelerator	  */
    NULL				/* extension		  */
  },  /* CoreClass fields initialization */
  {
    NULL,                    		/* geometry_manager	  */
    XtInheritChangeManaged,		/* change_managed	  */
    XtInheritInsertChild,		/* insert_child	  	  */
    XtInheritDeleteChild,		/* delete_child	  	  */
    NULL,				/* extension	 	  */
  },  /* CompositeClass fields initialization */
  {
    NULL,	            		/* subresources       */
    0,					/* subresources_count */
    sizeof(XvwManagerWidgetConstraintsRec),	/* constraint_size    */
    NULL,				/* initialize         */
    NULL,                               /* destroy            */
    NULL,				/* set_values         */
    NULL,                               /* extension          */
  },  /* ConstraintClass fields initialization */
  {
    XtInheritLayout,			/* child layout routine  */
    XtInheritChangeSel,		        /* change selected proc   */
    XtInheritEraseSel,			/* erase selected proc    */
    XtInheritRefreshSel,		/* refresh selection proc */
    Resize,				/* resize		  */
    XtInheritGeometryManager,  		/* geometry_manager	  */
  }, /* XvwManagerWidgetClass fields initialization */
  {
    NULL,                                     /* field not used    */
  },  /* XvwGraphicsWidgetClass fields initialization */
  {
    ColorCallback,                      /* reload color proc    */
  },  /* XvwColorWidgetClass fields initialization */
  {
    NULL,                               /* field not used    */
  },  /* XvwImageWidgetClass fields initialization */
};
#undef SUPERCLASS

/*
 * xvwImageWidgetClass for public consumption
 */
WidgetClass xvwImageWidgetClass = (WidgetClass) &xvwImageWidgetClassRec;


/*-----------------------------------------------------------
|
|  Routine Name: ClassInitialize
|
|       Purpose: This method is called the first time an  
|                instance of image widget class has been created.
|                It will initialize all the class attributes.
|
|         Input: none 
|
|        Output: none
|
|    Written By: Mark Young 
|          Date: Aug 15, 1992
| Modifications:
|
------------------------------------------------------------*/

static void ClassInitialize(void)
{
        xvw_init_attributes(xvwImageWidgetClass, attributes,
		knumber(attributes), NULL, 0, 
		"$DESIGN/objects/library/xvisual/uis/Image.pane");
	xvw_load_resources("$DESIGN/objects/library/xvisual/app-defaults/Image");

	xvw_define_attributes(xvwImageWidgetClass,
	  XVW_IMAGE_ROI, XtRPointer, ImageSetRoi, ImageGetRoi,
	  XVW_IMAGE_IMAGEFILE, XtRString, ImageSetName, ImageGetName,
	  XVW_IMAGE_CLIPFILE, XtRString, ImageSetName, ImageGetName,
	  XVW_IMAGE_SAVEIMAGE, XtRString, ImageSetName, NULL,
	  NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: Initialize
|
|       Purpose: This method will set up the initial image
|		 for an image widget instance. 
|
|         Input: request - not used
|		 new	 - widget instance after initialization, with 
|		           initial image
|
|        Output: None
|
|    Written By: Mark Young
|          Date: Jun 29, 1992
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Initialize(
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
	XvwImageWidget xwid = kwidget(new);


	xwid->image.resize  = FALSE;
	xwid->image.recolor = FALSE;
	xwid->image.band_maxnum = 0;
	xwid->image.band_number = 0;
	xwid->image.region_num    = 1;
	xwid->image.region_width  = 0;
	xwid->image.region_height = 0;

	xwid->image.idata =
	xwid->image.cdata =
	xwid->image.mdata =
	xwid->image.alpha =
	xwid->image.ldata = NULL;

	xwid->image.shminfo = NULL;
	xwid->image.ximage  =
	xwid->image.xclip   =
	xwid->image.xcolormap =
	xwid->image.xoverlay  = NULL;
	xwid->image.pixmap    = None;

	xvw_add_event(xvw_object(new), PointerMotionMask | ButtonPressMask,
		      ImageUpdatePosition, NULL);
	xvw_set_attributes(xvw_object(new),
		XVW_MAXIMUM_WIDTH,  800,
		XVW_MAXIMUM_HEIGHT, 750,
		NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: Realize
|
|       Purpose: This method will set up the window attributes 
|                and create a window for the image widget.
|
|         Input: widget	    - the widget to realize 
|		 valuemask  - determines the window attributes being passed in
|		 attributes - the list of window attributes to be set 
|
|        Output: None 
|
|    Written By: Mark Young
|          Date: Jun 29, 1992
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Realize(
   Widget               widget,
   XtValueMask          *valuemask,
   XSetWindowAttributes *winattrib)
{
	XvwImageWidget xwid = kwidget(widget);


	(*xvwImageWidgetClass->core_class.superclass->core_class.realize)
                (widget, valuemask, winattrib);
	xwid->image.reload = TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: Resize
|
|       Purpose: This method is used to set the image size
|		 to a new size if the image is resized, and
|		 to redraw the image at its new size. 
|
|         Input: widget - the widget containing the new position 
|                         and size information
|
|        Output: None
|
|    Written By: Mark Young
|          Date: Jun 29, 1992
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Resize(
   Widget widget)
{
	XvwImageWidget xwid = kwidget(widget);

	xwid->image.resize = TRUE;
	xwid->image.reload = TRUE;
	(*xvwImageWidgetClass->core_class.superclass->core_class.resize)
		(widget);
}

/*-----------------------------------------------------------
|
|  Routine Name: Redisplay
|
|       Purpose: This routine will redraw the image widget in response
|                to an expose event.  We actually call RedisplayImage()
|		 to do the actual work of refreshing the Image.  Note:
|		 the "refresh" parameter to RedisplayImage() is suppose
|		 to be FALSE, since typically the image is made to be
|		 the background pixmap and therefore should not be
|		 manually refreshed (backing store does it for us).
|
|         Input: widget - widget data structure for image
|		 event	- the event that caused the redraw
|		 region	- the region that was exposed
|        Output: None
|
|    Written By: Mark Young
|          Date: Jun 29, 1992
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Redisplay(
   Widget widget,
   XEvent *event,
   Region region)
{
	int x, y, w, h;

	/*
	 *  Make sure that the event is an Expose event
	 */
	if (event->type != Expose)
	   return;

	x = event->xexpose.x; y = event->xexpose.y;
	w = event->xexpose.width; h = event->xexpose.height;
	RedisplayImage(widget, FALSE, x, y, w, h);
	(*xvwImageWidgetClass->core_class.superclass->core_class.expose)
                (widget, event, region);
}

/*-----------------------------------------------------------
|
|  Routine Name: Destroy
|
|       Purpose: Close the image, shape, colormap, and overlay
|		 objects before the image widget is destroyed.
|
|         Input: widget - the image widget being destroyed 
|
|        Output: None
|
|    Written By: Mark Young
|          Date: Nov 21, 1992 12:52
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Destroy(
   Widget widget)
{
	XvwImageWidget xwid = kwidget(widget);


	/*
	 *  Close all objects...
	 */
	kpds_close_object(xwid->image.iobject);
	kpds_close_object(xwid->image.cobject);
	kfree(xwid->image.idata);
	kfree(xwid->image.cdata);
	kfree(xwid->image.mdata);
	kfree(xwid->image.alpha);
 
        /*xwid->image.shminfo = NULL;*/
	if (xwid->image.ximage) XDestroyImage(xwid->image.ximage);
	if (xwid->image.xclip)  XDestroyImage(xwid->image.xclip);
	if (xwid->image.xcolormap) XDestroyImage(xwid->image.xcolormap);
	if (xwid->image.xoverlay)  XDestroyImage(xwid->image.xoverlay);

	if (xwid->image.pixmap != None)
           XFreePixmap(XtDisplay(widget), xwid->image.pixmap);

#ifdef KMITSHM_DEF
	if (xwid->image.shminfo)
	{
	   XShmSegmentInfo *shminfo = (XShmSegmentInfo *) xwid->image.shminfo;

	   XShmDetach(XtDisplay(widget), shminfo);
	   shmdt(shminfo->shmaddr);
	   shmctl(shminfo->shmid, IPC_RMID, 0);
           kfree(xwid->image.shminfo);
	}
#endif /* KMITSHM_DEF */
}

/*-----------------------------------------------------------
|
|  Routine Name: SetValues
|
|       Purpose: Determine what the widget should do by
|		 comparing the current values verse the
|		 new settings.
|
|         Input: current - the widget containing current settings 
|		 request - the widget containing requested settings
|		 new	 - the widget processed through all set values methods 
|
|        Output: None directly
|
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Nov 21, 1992 12:52
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean SetValues(
   Widget   current,
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
	XvwImageWidget cxwid = kwidget(current);
	XvwImageWidget nxwid = kwidget(new);

	Boolean redisplay = FALSE;
	int     size, width, height;


	if (cxwid->image.iobject != nxwid->image.iobject)
	{
	   nxwid->image.iobject = kpds_reference_object(nxwid->image.iobject);
	   kpds_close_object(cxwid->image.iobject);
	   nxwid->image.reload = FALSE;
	   xvw_set_attribute(xvw_object(new), XVW_COLOR_COLOROBJ,
                        nxwid->image.iobject);

	   ImageInit(new);
	   kpds_add_value_callback(nxwid->image.iobject,
		     KPDS_CALLBACK_ACCESS, ImageCallback, (kaddr) new);

	   /*
	    *  Add a Change Callback for the value segment and mask
	    *  segment (if one exists).
	    */
	   kpds_add_value_callback(nxwid->image.iobject,
		     KPDS_CALLBACK_CHANGE, ReloadCallback, (kaddr) new);
	   if (kpds_query_mask(nxwid->image.iobject) == TRUE)
	   {
	      kpds_add_mask_callback(nxwid->image.iobject,
		     KPDS_CALLBACK_CHANGE, ReloadCallback, (kaddr) new);
	   }

	   new->core.width  = nxwid->image.wsize;
	   new->core.height = nxwid->image.hsize;
	   if (new->core.width  != current->core.width ||
	       new->core.height != current->core.height)
	   {
	      nxwid->image.resize = TRUE;
	      xvw_set_attributes(xvw_object(new),
		     XVW_PREFERRED_WIDTH,  nxwid->core.width,
		     XVW_PREFERRED_HEIGHT, nxwid->core.height,
		     NULL);

	      /*
	       *  Destroy the old data...
	       */
	      kfree(nxwid->image.idata);
	      kfree(nxwid->image.mdata);
	      kfree(nxwid->image.alpha);
	      kfree(nxwid->image.cdata);

	      width = kmin(nxwid->core.width, nxwid->manager.max_width);
	      if (nxwid->image.xoffset+width > nxwid->image.wsize)
	         nxwid->image.xoffset = nxwid->image.wsize - width;

	      height = kmin(nxwid->core.height, nxwid->manager.max_height);
	      if (nxwid->image.yoffset+height > nxwid->image.hsize)
	         nxwid->image.yoffset = nxwid->image.hsize - height;
	   }
	   else if (cxwid->image.image_depth != nxwid->image.image_depth)
	   {
	      /*
	       *  Destroy the old data...
	       */
	      kfree(nxwid->image.idata);
	      kfree(nxwid->image.mdata);
	      kfree(nxwid->image.alpha);
	      kfree(nxwid->image.cdata);
	   }
	   nxwid->image.reload = TRUE;
	   redisplay = TRUE;

	   if (ImageBacking(nxwid) == TRUE)
	      XtVaSetValues(new, XtNbackgroundPixmap, XtUnspecifiedPixmap,NULL);
	}

	if (cxwid->image.cobject != nxwid->image.cobject)
	{
	   /*
	    *  Close the object and then set it to NULL so that we don't
	    *  every try messing with it.
	    */
	   nxwid->image.cobject = kpds_reference_object(nxwid->image.cobject);
	   if (cxwid->image.cobject)
	   {
	      kpds_close_object(cxwid->image.cobject);
	      cxwid->image.cobject = NULL;
	   }
	   nxwid->image.reload = TRUE;
	   kfree(nxwid->image.cdata);
	   redisplay = TRUE;
	}

	if (cxwid->image.band_dimensions  != nxwid->image.band_dimensions ||
	    cxwid->image.complex_convert  != nxwid->image.complex_convert)
	{
	   ImageInit(new);
	   nxwid->image.reload = TRUE;
	   redisplay = TRUE;
	}

	if (cxwid->image.band_number != nxwid->image.band_number)
	{
           if (nxwid->image.band_number > nxwid->image.band_maxnum)
              nxwid->image.band_number = nxwid->image.band_maxnum;
           else if (nxwid->image.band_number < 0)
              nxwid->image.band_number = 0;

	   nxwid->image.reload = TRUE;
	   redisplay = TRUE;
	}

	if (cxwid->image.xoffset != nxwid->image.xoffset ||
	    cxwid->image.yoffset != nxwid->image.yoffset)
	{
	   int width = new->core.width, height = new->core.height;

	   if (nxwid->image.xoffset+width > nxwid->image.wsize)
	      nxwid->image.xoffset = nxwid->image.wsize - width;
	   if (nxwid->image.yoffset+height > nxwid->image.hsize)
	      nxwid->image.yoffset = nxwid->image.hsize - height;

	   if (cxwid->image.xoffset != nxwid->image.xoffset ||
               cxwid->image.yoffset != nxwid->image.yoffset)
	   {
	      nxwid->image.reload = TRUE;
              redisplay = TRUE;
	   }
	}

	if (cxwid->image.xposition != nxwid->image.xposition  ||
            cxwid->image.yposition != nxwid->image.yposition)
	{
	   ImageSetPosition(new, nxwid->image.xposition,
		       nxwid->image.yposition, IMAGE_POSITION_UPDATE);
	}

	if (cxwid->image.redisplay != nxwid->image.redisplay)
	{
	   if (nxwid->color.info != NULL) kfree(nxwid->color.info->histogram);
	   nxwid->image.redisplay = FALSE;
	   redisplay = TRUE;
	}

	if ((redisplay || nxwid->image.reload) && !ImageBacking(nxwid))
	{
	   RedisplayImage(new, TRUE, 0, 0, (int) new->core.width,
			(int) new->core.height);
	   redisplay = FALSE;
	}
	return(redisplay);
}

/*-----------------------------------------------------------
|
|  Routine Name: ReloadCallback
|
|       Purpose: Reload the ximage data structure from the image
|		 data object.
|
|         Input: xwid - the image widget to load the image data
|
|        Output: none
|
|    Written By: Mark Young
|          Date: May 18, 1993 14:44
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ReloadCallback(
   kobject object,
   char    *segment,
   Widget  widget,
   kdms_callback *cb)
{
	XvwImageWidget xwid = (XvwImageWidget) kwidget(widget);
	int x, y, w, h;


	w = (cb->end.w - cb->begin.w) + 1;
	h = (cb->end.h - cb->begin.h) + 1;
	x = cb->begin.w - xwid->image.xoffset;
	y = cb->begin.h - xwid->image.yoffset;
	xwid->image.reload = TRUE;
	if (xwid->color.info && kstrcmp(cb->segment, KDMS_SEGMENT_VALUE) == 0)
	   kfree(xwid->color.info->histogram);

	ImageReload(widget, cb);
	RedisplayImage(widget, TRUE, x, y, w, h);
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorCallback
|
|       Purpose: Reload the ximage data structure from the image
|		 data object and refresh the image.  This is done
|		 when our ColorClass has had a change to it's map
|		 data, and we are required to change the colors in the
|		 image.
|
|         Input: object - the object to be refreshed (the Image, Animate, etc)
|		 call_data   - (not used)
|        Output:
|    Written By: Mark Young & Danielle Argiro
|          Date: May 24, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ColorCallback(
   xvobject object,
   kaddr    call_data)
{
	XvwImageWidget xwid = (XvwImageWidget) xvw_widget(object);
        Widget widget = xvw_widget(object);
	XvwImageWidgetClass iclass;

        XEvent event;
        Region region;
 
	xwid->image.recolor = TRUE;
	iclass = (XvwImageWidgetClass) widget->core.widget_class;
	if (iclass->core_class.expose != NULL)
	{
           event.type = Expose;
           event.xexpose.x = event.xexpose.y = 0;
           event.xexpose.width  = widget->core.width;
           event.xexpose.height = widget->core.height;
	   region = XCreateRegion();
           XtAddExposureToRegion(&event, region);
           iclass->core_class.expose(widget, &event, region);
           XDestroyRegion(region);
	}
	else
	{
	   RedisplayImage((Widget) xwid, TRUE, 0, 0, (int) xwid->core.width,
                            (int) xwid->core.height);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: ImageCallback
|
|       Purpose: This updates the position of the object being tracked
|                 
|         Input: 
|        Output:
|
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Nov 17, 1992 10:24
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ImageCallback(
   kobject object,
   char    *segment,
   Widget  widget,
   kdms_callback *cb)
{
	int      w, h, reload, update;

	/*
	 *  Make sure we aren't here because of ourself
	 */
	if (object == cb->object)
	   return;

	xvw_get_type(object, &reload, &update);
	if (update != PANICON_POSITION_UPDATE)
	   return;

	w = (cb->end.w - cb->begin.w)/2 + cb->begin.w;
	h = (cb->end.h - cb->begin.h)/2 + cb->begin.h;

	xvw_set_attributes(xvw_object(widget),
		XVW_IMAGE_XOFFSET, w,
		XVW_IMAGE_YOFFSET, h,
		NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: ResizeImage
|
|       Purpose: Resize the ximage data structure from the widget's
|		 current size.
|
|         Input: widget - the image widget to resize the ximage data
|
|        Output: none
|
|    Written By: Mark Young
|          Date: Nov 21, 1992 12:52
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ResizeImage(
   Widget widget)
{
	XvwImageWidget xwid = (XvwImageWidget) widget;
	unsigned int width, height, depth;
	XImage *ximage = xwid->image.ximage;
#ifdef KMITSHM_DEF
	int major, minor;
	Bool  sharedMaps;
#endif /* KMITSHM_DEF */


	/*
	 *  If the current ximage structure is the same then simply return
	 */
	if (ximage && ximage->width  == xwid->core.width &&
		      ximage->height == xwid->core.height &&
		      ximage->depth  == xwid->core.depth)
	{
	   return;
	}

	/*
	 *  Destroy the old data...
	 */
	kfree(xwid->image.idata);
	kfree(xwid->image.mdata);
	kfree(xwid->image.alpha);
	kfree(xwid->image.cdata);

	/*
	 *  Destroy the old ximage...
	 */
	if (xwid->image.ximage)
	{
	   XDestroyImage(xwid->image.ximage);
#ifdef KMITSHM_DEF
	   if (xwid->image.shminfo)
	   {
	      XShmSegmentInfo *shminfo= (XShmSegmentInfo *) xwid->image.shminfo;

	      XShmDetach(XtDisplay(widget), shminfo);
	      shmdt(shminfo->shmaddr);
	      shmctl(shminfo->shmid, IPC_RMID, 0);
	      kfree(xwid->image.shminfo);
	   }
#endif /* KMITSHM_DEF */
	   ximage = xwid->image.ximage = NULL;
	}

	/*
	 *  Create the new Shared Memory XImage.
	 */
#ifdef KMITSHM_DEF
	xwid->image.shminfo = (kaddr) kcalloc(1, sizeof(XShmSegmentInfo));
	if (xvw_check_localhost(xvw_object(widget)) &&
	    XShmQueryVersion(XtDisplay(widget), &major, &minor, &sharedMaps) &&
	    (ximage = XShmCreateImage(XtDisplay(widget), xwid->manager.visual,
		     	xwid->core.depth, ZPixmap, NULL, (XShmSegmentInfo *)
		     	xwid->image.shminfo, xwid->core.width,
			xwid->core.height)) != NULL)
	{
	   XShmSegmentInfo *shminfo = (XShmSegmentInfo *) xwid->image.shminfo;
	   struct shmid_ds buf;


	   if ((shminfo->shmid = shmget(IPC_PRIVATE, ximage->bytes_per_line *
		ximage->height, IPC_CREAT|0777)) != -1)
	   {
	      shminfo->readOnly = FALSE;
	      shminfo->shmaddr = ximage->data =
				(kaddr) shmat(shminfo->shmid, 0, 0);
	      if (shminfo->shmaddr != ((char *) -1) ||
                  shmctl(shminfo->shmid, IPC_STAT, &buf) == -1 ||
	          XShmAttach(XtDisplay(widget), shminfo) == FALSE)
	      {
		 XDestroyImage(ximage); ximage = NULL;
		 shmctl(shminfo->shmid, IPC_RMID, 0); 
	      }
	   }
	   else
	   {
	      XDestroyImage(ximage); ximage = NULL;
	   }
	}

	if (!ximage)
	   kfree(xwid->image.shminfo);
#endif /* KMITSHM_DEF */

	/*
	 *  If not a Shared Memory then create a regular XImage
	 */
	if (!ximage && (ximage = XCreateImage(XtDisplay(widget),
		xwid->manager.visual, xwid->core.depth, ZPixmap, 0,
		NULL, xwid->core.width, xwid->core.height, 8, 0)) != NULL)
	{
	   /*
	    *  Allocate the image data
	    */
	   if ((ximage->data = kcalloc(1, ximage->bytes_per_line *
			xwid->core.height)) == NULL)
	   {
	      XDestroyImage(xwid->image.ximage);
	      return;
	   }
	}

	if (!ximage)
	   return;
	else
	   xwid->image.ximage = ximage;

	/*
	 *  So we got our XImage, but there maybe a problem.  Put in a check
	 *  to make sure that the bytes per line returned in the XImage
	 *  structure is equal to the requested width.  With the MIT XShm
         *  extension if the requested XImage width is not a multiple of 4,
	 *  it rounds up the width request.  Hence the image would appear to be
         *  skewed.
	 */
	ximage->width = ximage->bytes_per_line/(ximage->bits_per_pixel >> 3);
	
	/*
	 *  Destroy the old pixmap...
	 */
	if (xwid->image.pixmap != None)
	   XFreePixmap(XtDisplay(widget), xwid->image.pixmap);

	/*
	 *  Create the new Pixmap.
	 */
	depth  = xwid->core.depth;
	height = xwid->core.height;
	width  = xwid->core.width + xwid->core.width % 8;
	if ((xwid->image.pixmap = XCreatePixmap(XtDisplay(widget),
		XtWindow(widget), width, height, depth)) == None)
	{
	   return;
	}
	xwid->image.resize = FALSE;
}

/*-----------------------------------------------------------
|
|  Routine Name: RedisplayImage
|
|       Purpose: This routine will redraw the image widget in response
|                to a rectangle specified by x, y, w, h. Note:
|                the "refresh" parameter to RedisplayImage() can be
|                FALSE, since typically the image is made to be
|                the background pixmap and therefore should not be
|                manually refreshed (backing store does it for us).
|		 It is set to TRUE when we need the contents of
|		 image to be updated.
|
|         Input: widget - widget data structure for image
|		 refresh - force a refresh (Redisplay() calls this with FALSE)
|		 x, y, w, h - the rectangle specifing the region to be refreshed
|        Output: None
|    Written By: Mark Young
|          Date: May 18, 1993 15:49
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void RedisplayImage(
   Widget widget,
   int    refresh,
   int    x,
   int    y,
   int    w,
   int    h)
{
	XvwImageWidget xwid = kwidget(widget);

	Boolean resize  = xwid->image.resize;
	Boolean recolor = xwid->image.recolor;
	Boolean reload  = xwid->image.reload;

	/*
	 *  Check to see if we need to reload and/or resize the image
	 */
	if (!XtWindow(widget) || !xvw_check_visible(xvw_object(widget)))
	   return;

	/*
	 *  For interactive resize by user, change in image size, or the
	 *  programmer changes the widget width or height.
	 */  
	if (resize == TRUE)
	   ResizeImage(widget);

	/*
	 *  Reload the Image, because the value or map data has changed.
	 */
	if (reload == TRUE || recolor == TRUE)
	{
	   ImageReload(widget, NULL);
	   x = y = 0; w = xwid->core.width; h = xwid->core.height;
	}

	/*
	 *  Make sure that a valid ximage setructure exists.  Otherwise
	 *  return.
	 */
	if (!xwid->image.ximage || !xwid->image.pixmap)
	   return;

	/*
	 *   If reload or refresh is set, then we must update the contents
	 *   of our offscreen image pixmap before refreshing the Image window.
	 */
	if (reload || recolor || refresh)
	{
/*
 *  If MIT Shared Memory extension is defined then use the XShmPutImage()
 *  call to display the image.  Otherwise, use the plain ole XPutImage().
 */
#ifdef KMITSHM_DEF
	   if (xwid->image.shminfo)
	      XShmPutImage(XtDisplay(widget), xwid->image.pixmap,
			xwid->graphics.gc, xwid->image.ximage,
			x, y, x, y, w, h, TRUE);
	   else
#endif /* KMITSHM_DEF */
	      XPutImage(XtDisplay(widget), xwid->image.pixmap,
			xwid->graphics.gc, xwid->image.ximage,
			x, y, x, y, w, h);

	   /*
	    *  If we are using backing store then make this our background
	    *  pixmap.  This helps reduce the techno-flash when children
	    *  of the image do a XClearArea().  However, this is a fairly
	    *  expensive operation.  So we allow the programmer to turn this
	    *  off using the XVW_IMAGE_BACKING attribute.  If set to false,
	    *  then use XCopyArea to refresh the contents of our Image window.
	    */
	   if (ImageBacking(xwid) == TRUE)
	   {
	      XtVaSetValues(widget,XtNbackgroundPixmap,xwid->image.pixmap,NULL);
	      XClearArea(XtDisplay(widget), XtWindow(widget), 0, 0, 0, 0,FALSE);
	   }
	   else
	   {
	      XCopyArea(XtDisplay(widget), xwid->image.pixmap, XtWindow(widget),
                  xwid->graphics.gc, x, y, w, h, x, y);
	   }
	}

	/*
	 *  If we have had an expose, but we don't need to update the contents
	 *  of our offscreen pixmap, refresh the contents of our Image window,
	 *  unless image backing store is being done.
	 */
	else if (ImageBacking(xwid) == FALSE)
	{
	   XCopyArea(XtDisplay(widget), xwid->image.pixmap, XtWindow(widget),
                  xwid->graphics.gc, x, y, w, h, x, y);
	}
}


/************************************************************
*
*  Routine Name: xvw_create_image - create an image object
*
*       Purpose: The image visual object provides a mechanism with 
*                which to easily display an image. If the input file
*                has an associated colormap, that colormap will be
*                used to display the image.  
*
*                8 bit and 24 bit displays are fully supported by the 
*                image object;  monochrome and 4 bit displays may be 
*                used but are not recommended.
*
*                Bit images, greyscale images, and RGB images are also 
*                supported.  The image object is completely flexible as 
*                to the data type of the image data; all data types, 
*                including signed and unsigned bit, byte, short, int, float, 
*                double, and complex data types are supported.
*
*                The image object will react to changes in the input image;
*                that is, if the contents of the input image change, the
*                image object will immediately update to display the new
*                image.
*
*                The image object supports the reading/writing of a variety 
*                of file formats, including: 
*                ! xvimage (Khoros 1.0.5 format) 
*                ! VIFF (Khoros 2.0 format) 
*                ! Portable Any Map (PNM) format (including PPM, PGM, etc) 
*                ! ASCII format
*                ! X Bitmap (XBM) format
*                ! X Pixel Map (XPM) format
*                ! X Window Dump (XWD) format
*                ! Application Visualization System (AVS) format
*
*                In addition to those listed above, the following formats
*                are supported for write only:
*                ! Encapsulated PostScript (eps) format
*
*                In addition to those listed above, the following formats
*                are supported for read only:
*                ! Raw (headerless) format
* 
*                In the future, we also hope to support a number of additional
*                formats, including:
*                ! Tagged Image File Format (TIFF)
*                ! Graphics Interchange Format (GIF)
*                
*         Input: parent - the parent object; NULL will cause a
*                         default toplevel to be created automatically
*                name   - the name with which to reference the object 
*
*        Output: None
*
*	Returns: The image object on success, NULL on failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 21, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xvobject xvw_create_image(
   xvobject parent,
   char     *name)
{
	xvobject object;


	/*
	 *  Call the xvw_create() routine to do the actual creation.
	 */
	object = xvw_create(parent, FALSE, TRUE, name, ImageWidgetClass);
	return(object);
}
