/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>              3D Matrix  Utilities
   >>>>
   >>>>  Private:
   >>>>              _X3D_clear_matrix()
   >>>>              _X3D_matrix_set_identity()
   >>>>              _X3D_matrix_set_viewing()
   >>>>              _X3D_matrix_set_inviewing()
   >>>>              _X3D_matrix_set_parallel()
   >>>>              _X3D_matrix_set_perspective()
   >>>>              _X3D_matrix_mult()
   >>>>              _X3D_matrix_set_ndc()
   >>>>              _X3D_write_matrix()
   >>>>              _X3D_matrix_set_dc()
   >>>>              X3D_matrix_set_translate()
   >>>>              X3D_matrix_set_scale()
   >>>>              X3D_matrix_set_rotate()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "graphics.h"


/*-----------------------------------------------------------
|
|  Routine Name: _X3D_clear_matrix
|
|       Purpose: Set a 4x4 matrix to ALL zeros.
|
|         Input: matrix -- a 4x4 Real array.
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X3D_clear_matrix (
   Matrix matrix)
{
	int i,j;

	for (i = 0; i < 4; i++)
	{
	    for (j = 0; j < 4; j++)
	    {
		matrix[i][j] = 0.0;
	    }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_matrix_set_identity
|
|       Purpose: Set a 4x4 matrix to identity matrix.
|
|         Input: matrix -- a 4x4 Real array.
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X3D_matrix_set_identity (
   Matrix matrix)
{
	_X3D_clear_matrix(matrix);
	matrix[0][0] = matrix[1][1] = 
	matrix[2][2] = matrix[3][3] = 1.0;
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_matrix_set_oldviewing
|
|       Purpose: Set a viewing matrix system.
|
|         Input: 1) theta -- a Real representing the x-y axis offset.
|                2) alpha -- a Real representing the z-y axis offset.
|                3) eye_distance -- a Real representing the WC distance from
|                       eye to the origin.
|        Output: matrix - resulting viewing matrix
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void _X3D_matrix_set_oldviewing(
   Real    theta,
   Real    alpha,
   Real    eye_distance,
   Matrix  matrix)
{
        theta = kdegrees_radians(theta);
        alpha = kdegrees_radians(alpha);
 
        _X3D_clear_matrix(matrix);
        matrix[0][0] =  kcos(theta);
        matrix[0][1] = -(kcos(alpha) * ksin(theta));
        matrix[0][2] = -(ksin(alpha) * kcos(theta));
        matrix[1][0] =  ksin(theta);
        matrix[1][1] =  kcos(alpha) * kcos(theta);
        matrix[1][2] = -(ksin(alpha) * kcos(theta));
        matrix[2][1] =  ksin(alpha);
        matrix[2][2] = -(kcos(alpha));
        matrix[3][2] =  eye_distance;
        matrix[3][3] =  1.0;
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_matrix_set_viewing
|
|       Purpose: Set a viewing matrix system.
|
|         Input: camera	- camera position in WC
|		 focus	- focus position in WC
|		 up	- represents the camera rotation
|
|        Output: matrix	- resulting viewing matrix
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X3D_matrix_set_viewing(
   Coord  camera,
   Coord  focus,
   Coord  up,
   Matrix matrix)
{
	register Real  dot;
	Coord	 U, V, N;
	Matrix	 matrix1, matrix2;


	/*
	 *  compute vect n which is the vector from the camera to focus point
	 */
	N.x = focus.x - camera.x;
	N.y = focus.y - camera.y;
	N.z = focus.z - camera.z;
	X3D_normalize(&N);

	/*
	 *  compute vect v which is the vector from the camera to up point
	 *  make vector orthogonal to N and then normalize
	 */
	V.x = up.x - camera.x; V.y = up.y - camera.y; V.z = up.z - camera.z;
	dot = X3D_dot_product(&V, &N);

	V.x -= dot * N.x;
	V.y -= dot * N.y;
	V.z -= dot * N.z;
	X3D_normalize(&V);
	/*X3D_cross_product(&V, &N, &U);*/
	X3D_cross_product(&N, &V, &U);

	_X3D_clear_matrix(matrix2);
	matrix2[0][0] =  U.x;
	matrix2[1][0] =  U.y;
	matrix2[2][0] =  U.z;

	matrix2[0][1] =  V.x;
	matrix2[1][1] =  V.y;
	matrix2[2][1] =  V.z;

	matrix2[0][2] =  N.x;
	matrix2[1][2] =  N.y;
	matrix2[2][2] =  N.z;
/*
 */
	matrix2[3][0] =  - U.x * camera.x - U.y * camera.y - U.z * camera.z;
	matrix2[3][1] =  - V.x * camera.x - V.y * camera.y - V.z * camera.z;
	matrix2[3][2] =  - N.x * camera.x - N.y * camera.y - N.z * camera.z;
/*
 */
	matrix2[3][3] = 1.0;

	X3D_matrix_set_translate(-camera.x, -camera.y, -camera.z, matrix1);
	_X3D_matrix_mult(matrix1, matrix2, matrix);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_matrix_set_viewing2
|
|       Purpose: Set a viewing matrix (4x4) system
|
|         Input: camera	- camera position in WC
|		 focus	- focus position in WC
|		 up	- represents the camera rotation
|
|        Output: matrix	- resulting viewing matrix
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X3D_matrix_set_viewing2(
   Coord  camera,
   Coord  focus,
   Real   crotation,
   Matrix matrix)
{
	Matrix matrix1, matrix2;
	Real   a, b, c, det, e, f, g;


	a = focus.x;
	b = focus.y;
	c = focus.z;

	det = (1 - c*c);
	if (det < 0)
	   det = 0.01;
	else
	   det = sqrt((double) det);

	_X3D_clear_matrix(matrix1);
	matrix1[0][0] = (b)/det;
	matrix1[0][1] = (-a*c)/det;
	matrix1[0][2] = (a);

	matrix1[1][0] = (-a)/det;
	matrix1[1][1] = (-b*c)/det;
	matrix1[1][2] = (b);

	matrix1[2][0] = 0.0;
	matrix1[2][1] = det;
	matrix1[2][2] = c;

	matrix1[3][0] = 0.0;
	matrix1[3][1] = 0.0;
	matrix1[3][2] = 0.0;
	matrix1[3][3] = 1.0;

	e = camera.x;
	f = camera.y;
	g = camera.z;

	X3D_matrix_set_translate(-e, -f, -g, matrix2);
	_X3D_matrix_mult(matrix1, matrix2, matrix);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_matrix_set_inviewing
|
|       Purpose: Set the inverse viewing matrix system
|
|         Input: camera	- camera position in WC
|		 focus	- focus position in WC
|		 up	- represents the camera rotation
|
|        Output: matrix	- resulting viewing matrix
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X3D_matrix_set_inviewing (
   Coord  camera,
   Coord  focus,
   Coord  up,
   Matrix matrix)
{
	Coord	 U, V, N;
	register Real  dot, det;
	Matrix   matrix1, matrix2;


	/*
	 *  compute vect n which is the vector from the camera to focus point
	 */
	N.x = focus.x - camera.x;
	N.y = focus.y - camera.y;
	N.z = focus.z - camera.z;
	X3D_normalize(&N);

	/*
	 *  compute vect v which is the vector from the camera to up point
	 *  make vector orthogonal to N and then normalize
	 */
	V.x = up.x - camera.x; V.y = up.y - camera.y; V.z = up.z - camera.z;
	dot = X3D_dot_product(&V, &N);

	V.x -= N.x * dot;
	V.y -= N.y * dot;
	V.z -= N.z * dot;
	X3D_normalize(&V);
	/*X3D_cross_product(&V, &N, &U);*/
	X3D_cross_product(&N, &V, &U);

	det = (-U.x*V.y*N.z + U.x*V.z*N.y + U.y*V.x*N.z - U.y*V.z*N.x
			- U.z*V.x*N.y + U.z*V.y*N.x);
	if (det == 0.0)
	   det = 1.0;

	_X3D_clear_matrix(matrix1);
	matrix1[0][0] =  -(V.y*N.z - V.z*N.y)/det;
	matrix1[0][1] =  (V.x*N.z - V.z*N.x)/det;
	matrix1[0][2] =  -(V.x*N.y - V.y*N.x)/det;

	matrix1[1][0] =  (U.y*N.z - U.z*N.y)/det;
	matrix1[1][1] =  -(U.x*N.z - U.z*N.x)/det;
	matrix1[1][2] =  (U.x*N.y - U.y*N.x)/det;

	matrix1[2][0] =  -(U.y*V.z - U.z*V.y)/det;
	matrix1[2][1] =  (U.x*V.z - U.z*V.x)/det;
	matrix1[2][2] =  -(U.x*V.y - U.y*V.x)/det;

	matrix1[3][0] =  0.0;
	matrix1[3][1] =  0.0;
	matrix1[3][2] =  0.0;
	matrix1[3][3] = 1.0;

	X3D_matrix_set_translate(camera.x, camera.y, camera.z, matrix2);
	_X3D_matrix_mult(matrix1, matrix2, matrix);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_matrix_set_parallel
|
|       Purpose: Sets up a parallel transformation matrix.
|
|         Input: projection - Type of parallel view
|
|        Output: matrix	    - resulting matrix
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X3D_matrix_set_parallel (
   int  projection,
   Matrix matrix)
{
	Real  length, beta;

	/*  The z-axis is projected to a 2D space.  The axis
	     extends at a specified angle, between the x & y axes
	     (a left handed system).  */

	switch (projection)
	{
	    default:
	    case KGRAPHICS_ORTHOGRAPHIC:	
		 length = 0.0; 
		 beta = 90.0;
		 break;

	    case KGRAPHICS_CAVALIER:	
		 length = 1.0; 
		 beta = 45.0;
		 break;

	    case KGRAPHICS_CABINET:	
		 length = 0.5;
		 beta = 63.4;
		 break;
	}

	_X3D_matrix_set_identity(matrix);

	beta = kdegrees_radians(beta);
	matrix[2][0] = length*cos(beta); matrix[2][1] = length*sin(beta);
	matrix[2][2] = 0.0;
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_matrix_set_perspective
|
|       Purpose: Set a 4x4 matrix to a perspective viewing matrix.
|
|         Input: view_distance - distance from the eye to the viewport
|
|        Output: matrix	       - resulting viewing matrix
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X3D_matrix_set_perspective(
   Real   view_distance,
   Matrix matrix)
{
	Real  distance;


	if (view_distance == 0.0)
	   distance = 1.0;
	else
	   distance = 1.0/view_distance;

	_X3D_matrix_set_identity(matrix);

	matrix[2][2] = 0.0;
	matrix[2][3] = distance;
/*
	matrix[2][2] =  1.0/(1+view_distance);
	matrix[2][3] = -1.0;
	matrix[3][2] = -view_distance/(1+view_distance);
	matrix[3][3] = 0.0;
 */
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_matrix_mult
|
|       Purpose: Multiplies 2 4x4 matrices.
|
|         Input: matrix1 -
|                matrix2 -
|
|        Output: outmatrix - resulting matrix
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void _X3D_matrix_mult (
   Matrix matrix1,
   Matrix matrix2,
   Matrix outmatrix)
{
        int   i, j;
        Matrix tmp_matrix;

        /*
         *  Multiply the two matrices together and return the result in
         *  the out matrix, "outmatrix".
         */
        for (i = 0; i < 4; i++)
        {
            for (j = 0; j < 4; j++)
            {
                tmp_matrix[i][j]  = matrix1[i][0] * matrix2[0][j] +
                                    matrix1[i][1] * matrix2[1][j] +
                                    matrix1[i][2] * matrix2[2][j] +
                                    matrix1[i][3] * matrix2[3][j];
            }
        }

        /*
         *  Copy the temporary matrix to the final matrix (outmatrix).  We
         *  could have put this directly into the final matrix but this enables
         *  the user to use the output matrix as either matrix1 or matrix2.
         *  ie)
         *      matrix1 = matrix1 * matrix2
         */
        kmemcpy( (char *) outmatrix, (char *) tmp_matrix, sizeof(Matrix));
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_matrix_set_ndc
|
|       Purpose: Sets a transform matrix which will map world
|		 coordinates to normalized device coordinates.
|
|		 Three Dimensional Transform Matrix:
|
|		 ----            ---
|		 | Sx   0    0   0 |
|		 |  0   Sy   0   0 |
|		 |  0   0    Sz  0 |
|		 |  Ax  Ay   Az  1 |
|		 ----            ---
|
|         Input: scale  -
|		 wc_min -
|		 wc_max -
|
|        Output: matrix - resulting 4x4 transformation matrix
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X3D_matrix_set_ndc (
   Coord  scale,
   Coord  wc_min,
   Coord  wc_max,
   Matrix matrix)
{
	Real	scale_x, scale_y, scale_z,
		offset_x, offset_y, offset_z;
	Coord   smin, smax;

        if (!(_X3D_scale_wc_coord(scale, wc_min, &smin)))
           smin = wc_min;

        if (!(_X3D_scale_wc_coord(scale, wc_max, &smax)))
           smax = wc_max;

	scale_x = 1/(smax.x - smin.x);
	scale_y = 1/(smax.y - smin.y);
	if(smax.z == smin.z)
	    scale_z = 0.0;
	else
	   scale_z = 1/(smax.z - smin.z);

	offset_x = 0.0 - (smin.x * scale_x);
	offset_y = 0.0 - (smin.y * scale_y);
	offset_z = 0.0 - (smin.z * scale_z);

	/*
	 *  Now set the 3D ndc matrix  
	 */
	_X3D_matrix_set_identity(matrix);

	matrix[0][0] = scale_x;
	matrix[1][1] = scale_y;
	matrix[2][2] = scale_z;

	matrix[3][0] = offset_x;
	matrix[3][1] = offset_y;
	matrix[3][2] = offset_z;
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_matrix_set_invndc
|
|       Purpose: Sets a transform matrix which will map normalized
|		 device coordinates to three dimensional
|		 world coordinates.
|
|		 Three Dimensional Inverse Transform Matrix:
|
|		 ----                    ---
|		 | 1/Sx      0      0    0 |
|		 |  0       1/Sy    0    0 |
|		 |  0        0     1/Sz  0 |
|		 | -Ax/Sx -Ay/Sy -Az/Sz  1 |
|		 ----                    ---
|
|         Input: scale  -
|		 wc_min -
|		 wc_max -
|
|        Output: matrix - resulting 4x4 inverse transformation matrix
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X3D_matrix_set_invndc(
   Coord  scale,
   Coord  wc_min,
   Coord  wc_max,
   Matrix matrix)
{
	Coord	smin, smax;
	Real	scale_x, scale_y, scale_z,
		offset_x, offset_y, offset_z;

        if (!(_X3D_scale_wc_coord(scale, wc_min, &smin)))
           smin = wc_min;

        if (!(_X3D_scale_wc_coord(scale, wc_max, &smax)))
           smax = wc_max;

	scale_x = 1/(smax.x - smin.x);
	scale_y = 1/(smax.y - smin.y);
	if(smax.z == smin.z)
	    scale_z = 0.0;
	else
	   scale_z = 1/(smax.z - smin.z);

	offset_x = 0.0 - (smin.x * scale_x);
	offset_y = 0.0 - (smin.y * scale_y);
	offset_z = 0.0 - (smin.z * scale_z);

	/*
	 *  Now set the 3D ndc matrix  
	 */
	_X3D_matrix_set_identity(matrix);

	matrix[0][0] = 1/scale_x;
	matrix[1][1] = 1/scale_y;
	matrix[2][2] = 1/scale_z;

	matrix[3][0] = -offset_x/scale_x;
	matrix[3][1] = -offset_y/scale_y;
	matrix[3][2] = -offset_z/scale_z;
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_write_matrix
|
|       Purpose: Prints out a Matrix.  This is mostly used by
|		 for debugging purposes.
|
|         Input: matrix - The matrix to be printed.
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X3D_write_matrix (
   Matrix matrix)
{
     int i,j;

     printf ("\n");
     for (i = 0; i < 4; i++)
     {
         for (j = 0; j < 4; j++)
	 {
	     printf (" %7.3f ",matrix[i][j]);
	 }
	 printf ("\n");
     }
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_matrix_set_dc
|
|       Purpose: Sets a transform matrix which will map world
|		 coordinates to device coordinates.
|
|         Input: graphics - X3D graphics structure
|
|        Output: matrix	  - resulting 4x4 matrix
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X3D_matrix_set_dc (
   X3DGraphics *graphics,
   Matrix      matrix)
{
	/*
	 *  Now set the 3D dc matrix  
	 */
	_X3D_matrix_set_identity(matrix);

	if (graphics->device == X11)
	{
	   matrix[0][0] = graphics->X11_xmax - graphics->X11_xmin;
	   matrix[1][1] = graphics->X11_ymin - graphics->X11_ymax;
	   matrix[3][0] = graphics->X11_xmin;
	   matrix[3][1] = graphics->X11_ymax;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_matrix_set_translate
|
|       Purpose: Sets a translation matrix in order to translate world
|                coordinates to new world coordinates.
|
|                Three Dimensional Translation Matrix:
|
|                ----            ---
|                | 1    0    0   0 |
|                | 0    1    0   0 |
|                | 0    0    1   0 |
|                | Tx   Ty   Tz  1 |
|                ----            ---
|
|         Input: xtrans - a Real containing the x distance to be translated
|                ytrans - a Real containing the y distance to be translated
|                ztrans - a Real containing the z distance to be translated
|
|        Output: matrix - a three dimensional (4x4) Real array
|                         containing the scaling matrix.
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X3D_matrix_set_translate (
   Real   xtrans,
   Real   ytrans,
   Real   ztrans,
   Matrix matrix)
{
        /*
         *  Set up the translation matrix --
         *  these entries will not change
         */
        _X3D_matrix_set_identity(matrix);

        /*
         *  Values in the active 4th column of the
         *  translation matrix have been provided by the user.
         */
        matrix[3][0] = xtrans; matrix[3][1] = ytrans; matrix[3][2] = ztrans;
}


/*-----------------------------------------------------------
|
|  Routine Name: X3D_matrix_set_scale
|
|       Purpose: Sets a scale matrix in order to scale world
|		 coordinates to new world coordinates.
|
|		 Three Dimensional Scaling Matrix
|
|		 ----            ---
|		 | Sx   0    0   0 |
|		 |  0   Sy   0   0 |
|		 |  0   0    Sz  0 |
|		 |  0   0    0   1 |
|		 ----            ---
|
|         Input: xscale - a Real containing the x distance to be translated
|		 yscale - a Real containing the y distance to be translated
|		 zscale - a Real containing the z distance to be translated
|
|        Output: matrix - a three dimensional (4x4) float array
|			  containing the scaling matrix.
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_matrix_set_scale (
   Real   xscale,
   Real   yscale,
   Real   zscale,
   Matrix matrix)
{


        /*
         *  set identity matrix
         */
	_X3D_matrix_set_identity(matrix);

        /*
         *  the active entries are on the diagonal, providing the
         *  scaling factor requested by the user
         */
        matrix[0][0] = xscale; matrix[1][1] = yscale; matrix[2][2] = zscale;

}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_matrix_set_rotate
|
|       Purpose: Sets a rotation matrix in order to rotate world
|		 coordinates to new world coordinates.
|
|		 Three Dimensional Rotation Matrix:
|
|		 ----            ---
|		 | Sx   0    0   0 |
|		 |  0   Sy   0   0 |
|		 |  0   0    0   0 |
|		 | Ax   Ay   0   1 |
|		 ----            ---
|
|         Input: xrotate -
|		 yrotate -
|		 zrotate -
|
|        Output: matrix - a three dimensional (4x4) Real array
|			   containing the rotation matrix.
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_matrix_set_rotate (
   Real   xrotate,
   Real   yrotate,
   Real   zrotate,
   Matrix matrix)
{
        Real  rad;
	Matrix  xmatrix, ymatrix, zmatrix;


        /*
         *  initialize the matrix to the identity matrix, for
         *  the sake of simplicity -- some values will be over-
         *  written.
         */
	_X3D_matrix_set_identity(xmatrix);
	_X3D_matrix_set_identity(ymatrix);
	_X3D_matrix_set_identity(zmatrix);

        /*
         *  set up the rotation matrix as described above --
         *  with cos@ and sin@ in the proper places depending
         *  on which axis we are rotating about
         */

	/* x rotation matrix */
        rad = kdegrees_radians(xrotate);
        xmatrix[1][1] = xmatrix[2][2] = cos(rad);
        xmatrix[1][2] = sin(rad); xmatrix[2][1] = -sin(rad);

	/* y rotation matrix */
        rad = kdegrees_radians(yrotate);
        ymatrix[0][0] = ymatrix[2][2] = cos(rad);
        ymatrix[2][0] = sin(rad); ymatrix[0][2] = -sin(rad);

	/* z rotation matrix */
        rad = kdegrees_radians(zrotate);
        zmatrix[0][0] = zmatrix[1][1] = cos(rad);
        zmatrix[0][1] = sin(rad); zmatrix[1][0] = -sin(rad);

	/*
	 *  Get the final rotation matrix in terms of the
	 *  x, y, & z rotation.
	 */
	_X3D_matrix_mult (xmatrix, ymatrix, matrix);
	_X3D_matrix_mult (matrix,  zmatrix, matrix);
}
