/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Image Display Utility Routines
   >>>>
   >>>>  Private:
   >>>>			X3D_set_X11()
   >>>>			X11_DrawLine()
   >>>>			X11_DrawLines()
   >>>>			X11_DrawSegments()
   >>>>			X11_DrawRectangle()
   >>>>			X11_DrawArc()
   >>>>			X11_FillPolygon()
   >>>>			X11_close_device()
   >>>>			X11_set_line_width()
   >>>>			X11_set_line_type()
   >>>>			X11_set_draw()
   >>>>			X11_set_fill()
   >>>>			X11_DrawPolygon()
   >>>>			X11_DrawText()
   >>>>			X11_FillArc()
   >>>>			X11_FillRectangle()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "graphics.h"


/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_X11 - sets up the X11 display, window and GC
|
|       Purpose: This routine sets the display, the window, and
|                the graphics contexts for future calls to the
|                xvgraphics library using this id.
|
|         Input: id	  - xvgraphics id
|		 display  - the display to which all graphics will go
|		 drawable - the window that all graphics will appear in
|		 widget   - the widget that all graphics will appear in
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By:  Mike Lang & Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

int X3D_set_X11(
   int      id,
   Display  *display,
   Drawable drawable,
   Widget   widget)
{
	Window		root;
	int		x, y;
        unsigned	int width, height, border_width, depth;
	X3DGraphics     *graphics;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_set_X11:");
	   kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}

	graphics->device    = X11;
	graphics->widget    = widget;
	graphics->display   = display;

	if (drawable != None)
	   graphics->workspace = drawable;
	else if (widget != NULL)
	   graphics->workspace = XtWindow(widget);
	else
	   return(TRUE);

	if (widget == NULL && graphics->workspace == 0)
	   return(FALSE);

	if (widget != NULL)
	{
	   if (!(xvw_geometry(xvw_object(widget), NULL, NULL, &width,
			&height, NULL)))
	   {
              kfprintf (kstderr,"\nX3D_set_X11: Object does not exist.\n");
              return(FALSE);
	   }
	}
	else if (!XGetGeometry(graphics->display, graphics->workspace, &root,
			&x, &y, &width, &height, &border_width, &depth))
        {
           kfprintf (kstderr,"\nX3D_set_X11: Window does not exist.\n");
           return(FALSE);
        }
	X3D_set_window(id, 0, 0, width, height);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X11_DrawLine - displays a set line to an X11 device
|
|       Purpose: Draw a single line for an X11 device between
|                (px1,py1) to (px2,py2)
|
|         Input: graphics - graphics structure
|                p1x      - x coordinate for the begin point
|                p1y      - y coordinate for the begin point
|                p2x      - x coordinate for the end point
|                p2y      - y coordinate for the end point
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mike Lang & Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X11_DrawLine (
   X3DGraphics *graphics,
   int         p1x,
   int         p1y,
   int         p2x,
   int         p2y)
{
        XDrawLine(graphics->display, graphics->workspace, graphics->gc_draw,
		  p1x, p1y, p2x, p2y);
}

/*-----------------------------------------------------------
|
|  Routine Name: X11_DrawLines - displays a set of lines to an X11 device
|
|       Purpose: Draw a series of lines for an X11 device.
|
|         Input: graphics  - graphics structure
|                points    - series of points that defines the lines
|                            to be drawn
|                size      - number of points
|                coordmode - the output mode
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mike Lang & Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X11_DrawLines (
   X3DGraphics *graphics,
   XPoint      *points,
   int         size,
   int         coordmode)
{
	int		i, num;

	i = 0;
	while (i < size)
	{
	   num = kmin(size - i, 10000);
	   XDrawLines(graphics->display, graphics->workspace, graphics->gc_draw,
		   &points[i], num, coordmode);
	   i += num;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: X11_DrawSegments - displays a set of segments
|                                   to an X11 device
|
|       Purpose: Draw a series of disjoint segments for an X11
|                device.
|
|         Input: graphics - graphics structure
|                segments - segments to be drawn
|                seg_num  - number of segments to draw
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mike Lang & Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X11_DrawSegments (
   X3DGraphics *graphics,
   XSegment    *segments,
   int         seg_num)
{
	XDrawSegments(graphics->display, graphics->workspace, graphics->gc_draw,
		      segments, seg_num);
}

/*-----------------------------------------------------------
|
|  Routine Name: X11_DrawRectangle - draws a rectangle
|                                    to an X11 device
|
|       Purpose: Draw a rectangle to an X11 device using the
|                specified location (x,y) and with given width
|                and height (w,h).
|
|         Input: graphics - graphics structure
|                x        - upper left x coordinate
|                y        - upper left y coordinate
|                w        - rectangle's width
|                h        - rectangle's height
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mike Lang & Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X11_DrawRectangle (
   X3DGraphics  *graphics,
   int          x,
   int          y,
   unsigned int w,
   unsigned int h)
{
	XDrawRectangle(graphics->display, graphics->workspace,graphics->gc_draw,
		       x, y, w, h);
}

/*-----------------------------------------------------------
|
|  Routine Name: X11_DrawArc - draws an arc to an X11 device
|
|       Purpose: Draw an arc (circle) for an X11 device (x,y) are
|		 the upper left corner of the bounding box of the
|		 circle and ang1 & ang2 are the start and finish angles
|		 specified in 64th's of a degree.  The center of the
|		 circle is (x - w/2, y - h/2).
|
|         Input: graphics - graphics structure
|                x        - upper left x coordinate of the circle
|                y        - upper left y coordinate of the circle
|                w        - circle's width
|                h        - circle's height
|                ang1     - starting angle
|                ang2     - finishing angle
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mike Lang & Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X11_DrawArc (
   X3DGraphics  *graphics,
   int          x,
   int          y,
   unsigned int w,
   unsigned int h,
   int          ang1,
   int          ang2)
{
	ang1 *= 64;
	ang2 *= 64;
	x = x - ((float) w)/2.0 + 0.5;
	y = y - ((float) h)/2.0 + 0.5;

	XDrawArc(graphics->display, graphics->workspace, graphics->gc_draw,
		 x, y, w, h, ang1, ang2);
}

/*-----------------------------------------------------------
|
|  Routine Name: X11_FillPolygon - draws a filled polygon
|                                  to an X11 device
|
|       Purpose: Draws a filled polygon (not really) for an X11
|                device.
|
|         Input: graphics  - graphics structure
|                points    - series of points that defines the polygon
|                size      - number of points
|                shape     - Complex, Convex, or Nonconvex
|                coordmode - the output mode
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mike Lang & Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X11_FillPolygon (
   X3DGraphics *graphics,
   XPoint      *points,
   int         size,
   int         shape,
   int         coordmode)
{
	XFillPolygon(graphics->display, graphics->workspace, graphics->gc_fill,
		     points, size, shape, coordmode);
}

/*-----------------------------------------------------------
|
|  Routine Name: X11_close_device - closes the X11 device
|
|       Purpose: Writes out the command to close an X11 device.
|
|         Input: graphics - graphics structure
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mike Lang & Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X11_close_device(
   X3DGraphics *graphics)
{
	/* just return */
}

/*-----------------------------------------------------------
|
|  Routine Name: X11_set_line_width - sets the line width for
|                                     an X11 device
|
|       Purpose: Sets the line width for a given graphical context for
|                an X11 device.
|
|         Input: graphics   - pointer to graphics context
|                line_width - line width as defined in xvgraphics.h
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mike Lang & Tom Sauer
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X11_set_line_width(
   X3DGraphics *graphics,
   int         line_width)
{
	X11_set_draw(graphics, NULL, FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X11_set_line_type - sets the line type
|                                    for an X11 device
|
|       Purpose: Sets the line type for a given graphics context for
|                an X11 device.
|
|         Input: graphics  - pointer to graphics context
|                line_type - line type as defined in xvgraphics.h
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Tom Sauer
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X11_set_line_type(
   X3DGraphics *graphics,
   int         line_type)
{
	X11_set_draw(graphics, NULL, FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X11_set_draw - sets the foreground draw color
|                               for an X11 device
|
|       Purpose: Set the foreground draw color for an X11 device.
|
|         Input: graphics - graphics context
|                fg       - foreground color
|                override - whether the foreground color can
|                           be overriden.
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X11_set_draw(
   X3DGraphics *graphics,
   XColor      *fg,
   int         override)
{
	XGCValues  values;
	Pixel	   pixel;
	int	   line_width;
	unsigned long mask = GCForeground | GCLineStyle;


	if (fg == NULL)
	   pixel = graphics->fg;
	else if (fg->pixel == graphics->fg)
	   return;
	else
	   pixel = fg->pixel;

	values.foreground = pixel;

	/*
	 *  Get the line width
	 */
	if ((line_width = graphics->line_width) != KLINE_NONE)
	{
	   line_width--;
	   mask |= GCLineWidth;
	   values.line_width = line_widths[line_width][graphics->device];
	}

	if (graphics->line_type == KLINE_SOLID)
	   values.line_style = LineSolid;
	else
	   values.line_style = LineOnOffDash;

	/*
	 * Need to check to see if graphics->widget is not set.
	 */
	if (graphics->widget == NULL)
	{
	   kerror("xvgraphics", "X11_set_draw", "Need to call X3D_set_X11() to initialize graphics structure");
	   return;
	}

	graphics->gc_draw = XtGetGC(graphics->widget, mask, &values);
	if (graphics->line_type != KLINE_SOLID)
	{
           XSetDashes(graphics->display, graphics->gc_draw, 0,
				dash_type[graphics->line_type],
				dash_len[graphics->line_type]);
	}
	graphics->fg = pixel;
}

/*-----------------------------------------------------------
|
|  Routine Name: X11_set_fill - set the background fill color for an
|                               X11 device.
|
|       Purpose: Set the background fill color for an X11 device.
|
|         Input: graphics - graphics context
|                bg       - background color
|                override - whether the background color can
|                           be overriden.
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X11_set_fill(
   X3DGraphics *graphics,
   XColor      *bg,
   int         override)
{
	XGCValues  _values;


	if (graphics->bg != bg->pixel)
	{
	   _values.foreground = bg->pixel;
	   graphics->gc_fill = XtGetGC(graphics->widget, GCForeground,
				&_values);
	   graphics->bg = bg->pixel;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: X11_DrawPolygon - draws a non-filled polygon to
|                                  an X11 device
|
|       Purpose: Draws a non-filled polygon to an X11 device
|                using the point specified in points.
|
|         Input: graphics  - graphics structure
|                points    - series of points that defines the polygon
|                size      - number of points
|
|        Output: none
|
|       Returns: nothing
|
|    Written By:
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X11_DrawPolygon(
   X3DGraphics *graphics,
   XPoint      *points,
   int         size)
{
	X11_DrawLines(graphics, points, size, CoordModeOrigin);
        XDrawLine(graphics->display, graphics->workspace, graphics->gc_draw,
		  points[size-1].x, points[size-1].y, points[0].x, points[0].y);
}

/*-----------------------------------------------------------
|
|  Routine Name: X11_DrawText - draws text to an X11 device
|
|       Purpose: Draws text string passed in to an X11 device at
|                the given (x,y) position.
|
|         Input: graphics - graphics structure
|                x        - x position
|                y        - y position
|                font     - font to be used
|                text     - text to be printed
|                size     - size of text array
|
|        Output: none
|
|       Returns: nothing
|
|    Written By:
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X11_DrawText(
   X3DGraphics *graphics,
   int         x,
   int         y,
   char        *font,
   char        *text,
   int         size)
{
	XDrawString(graphics->display, graphics->workspace, graphics->gc_draw, 
		    x, y, text, size);
}

/*-----------------------------------------------------------
|
|  Routine Name: Postscript_FillArc - draws a filled arc to an postscript device
|
|       Purpose: Draw a filled arc (circle) for an X11 device (x,y) are
|                the upper left corner of the bounding box of the
|                circle and ang1 & ang2 are the start and finish angles
|                specified in 64th's of a degree.  The center of the
|                circle is (x - w/2, y - h/2).
|
|         Input: graphics - graphics structure
|                x        - of upper left corner
|                y        - of upper left corner
|                w        - width of Arc
|                h        - height of Arc
|                arc1     - start angle
|                arc2     - end angle
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young & Mike Lang
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X11_FillArc(
   X3DGraphics  *graphics,
   int          x,
   int          y,
   unsigned int w,
   unsigned int h,
   int          ang1,
   int          ang2)
{
	ang1 *= 64;
	ang2 *= 64;
	x = x - ((float) w)/2.0 + 0.5;
	y = y - ((float) h)/2.0 + 0.5;

	XFillArc(graphics->display, graphics->workspace, graphics->gc_fill,
		 x, y, w, h, ang1, ang2);
}

/*-----------------------------------------------------------
|
|  Routine Name: X11_FillRectangle - draws a filled rectangle
|                                    to an X11 device
|
|       Purpose: Draw a filled rectangle to an X11 device using the
|                specified location (x,y) and with given width
|                and height (w,h).
|
|         Input: graphics - graphics structure
|                x        - upper left x coordinate
|                y        - upper left y coordinate
|                w        - rectangle's width
|                h        - rectangle's height
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young & Mike Lang
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X11_FillRectangle(
   X3DGraphics *graphics,
   int         x,
   int         y,
   unsigned int w,
   unsigned int h)
{
	XFillRectangle(graphics->display, graphics->workspace,
		       graphics->gc_fill, x, y, w, h);
}
