/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>                Form  Manipulation Routines            <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                 xvf_delete_subform()                  <<<<
   >>>>                 xvf_delete_guide()  	              <<<<
   >>>>                 xvf_delete_selection()                <<<<
   >>>>                 xvf_create_new_guide()                <<<<
   >>>>                 xvf_create_new_subform()              <<<<
   >>>>   Static:                                             <<<<
   >>>>                 delete_submenu()        	      <<<<
   >>>>                 create_transferred_sels()             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"


static int   delete_submenu PROTO((kselection *));
int create_transferred_sels PROTO((kselection *, xvobject));



/*-----------------------------------------------------------------
|
|  Routine Name: xvf_delete_subform
|
|	Purpose: Deletes a subform from the form tree provided. 
|
|	  Input: form - pointer to the form tree
|		 old_subform - subform branch to be deleted 		
|
|        Output: Returns TRUE on success, FALSE on failure
|
|    Written By: Danielle Argiro
|          Date: Jul 16, 1992
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------------*/

int xvf_delete_subform(
   kform     *form,
   ksubform  *old_subform)
{
	ksubform   *subformptr;

	/*
         * make sure this is not the only subform on the form
         */
        if (form->subform == old_subform)
        {
            kerror("xvforms", "xvf_delete_subform",
                 "Illegal request to delete the only subform on the form");
            return(FALSE);
        }

	/*
         *  no subforms in form tree to delete (situation shouldn't occur)
         */
        if ((form->subform == NULL) && (form->master == NULL))
	{
            kerror("xvforms", "xvf_delete_subform",
                "Illegal request to delete subform from form w/ no subforms");
            return(FALSE);
	}

	/*
	 *  if the internal menuform for the subform is up, pop it down
	 */
	if (xvw_activated_menu(old_subform->button))
	    xvw_inactivate_menu(old_subform->button);

	/*
	 *  if there was an associated subform button, destroy it.
	 */
	if (old_subform->button != NULL)
	{
	    xvw_destroy(old_subform->button);
	    old_subform->button = NULL;
	}

	/*
	 * deleting a subform from the list of subforms on the master.
	 */
	if (form->master->subform_list != NULL)
	{
	    /*
	     * deleting the last subform from the master; delete master
	     */
            subformptr = form->master->subform_list;
            if (subformptr == old_subform)
            {
		if (form->master->subform_list->next == NULL)
		{
		    /*
		     *  destroy the master or form toplevel
		     */
		    if (form->master->back != NULL)
		    {
		        xvw_destroy(form->master->back);
		        form->master->back = NULL;

		        xvw_destroy(form->toplevel);
		        form->toplevel = NULL;
		    }
		}
		/*
	         *  deleting the 1st (of several) subforms from the master 
	         */
		else xvf_destroy_subform(old_subform);
            }

           /*
            * delete nth subform from list on master
            */
            else
            {
                while ((subformptr != NULL) &&
                       (subformptr != old_subform))
                    subformptr = subformptr->next;

                if (subformptr == old_subform)
                    xvf_destroy_subform(old_subform);

                else return(FALSE);
            }
	}

	/*
	 *  now that the widgets have all been destroyed,
	 *  destroy the actual subform branch of the form tree
	 */
	kvf_delete_subform(form, old_subform);

	
	/*
	 * if the deletion procedure has left us with a single subform,
	 * create the single subform if it's never been created,
         * or simply map it if it has been created already.
  	 */
	if (form->subform != NULL) 
	{
	    if (form->subform->back == NULL)
	        xvf_create_subform(form->subform, -1, -1, TRUE);
	    else 
	    {
		if (form->subform->guide != NULL)
		   create_transferred_sels(form->subform->guide->pane->sel_list,
		   		           form->subform->guide->pane->back);
		else create_transferred_sels(form->subform->guidepane->sel_list,
                                           form->subform->guidepane->back);
		xvf_map_subform(form->subform);
	    }
	}
	return(TRUE);
}


/*-----------------------------------------------------------------
|
|  Routine Name: xvf_delete_guide
|
|       Purpose: Deletes a guide button & pane from the subform tree provided.
|
|         Input: guide - pointer to the guide
|        Output: Returns TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: May 26, 1993
| Modifications: 
|
------------------------------------------------------------------*/

int xvf_delete_guide(
   kguide *guide)
{
        kguide     *guideptr;
        ksubform   *subform;
	kselection *selection;
	float      width, height;

	subform = guide->back_subform;

        /*
         * make sure this is not the only guide/pane pair on the subform
         */
        if (subform->guide == guide)
        {
            kerror("xvforms", "xvf_delete_guide",
                   "Illegal request to delete the only pane from a subform");
            return(FALSE);
        }

	/*
	 *  if the internal menuform for the pane is up, pop it down
	 */
	if (xvw_activated_menu(guide->pane->back))
            xvw_inactivate_menu(guide->pane->back);

	/*
	 *  if the internal menuform for the guidebutton is up, pop it down
	 */
	if (guide->button != NULL)
	{
	    if (xvw_activated_menu(guide->button))
                xvw_inactivate_menu(guide->button);
	}

	/*
	 *  if there was an associated guide button, destroy it.
	 */
        if (guide->button != NULL)
	{
            xvw_destroy(guide->button);
	    guide->button = NULL;
	}

	/*
	 * if this guide button was selected, select another one
	 */
	if (guide->selected)
	{
	    guideptr = subform->guidepane->guide_list;
	
	    /* find a guide button other than this one */
	    while (guideptr != NULL) 
	    {
	         if (guideptr != guide) break;
	         guideptr = guideptr->next;
	    }
	 
	    /* 
	     * calling the guide button callback is the easiest way
	     *  to change the guide button that is currently selected
	     */
	    if (guideptr != NULL) 
		xvf_guide_cb(guideptr->button, 
			     guideptr->back_kformstruct, NULL);
	}


        /*
         * deleting one of 2 or more guide buttons from the
         * list of guide buttons on the guidepane.
         */
        if (subform->guidepane->guide_list != NULL)
        {
            /*
             * delete 1st guide from list on guidepane
             */
            guideptr = subform->guidepane->guide_list;
            if (guideptr == guide)
	    {
		xvw_destroy(guideptr->pane->back);
                xvf_destroy_control(guideptr->pane);

	    }

            else
            {
                /*
                 * delete nth guide from list on guidepane
                 */
                while ((guideptr != NULL) && (guideptr != guide))
                    guideptr = guideptr->next;

                if (guideptr == guide)
		{
		    xvw_destroy(guideptr->pane->back);
                    xvf_destroy_control(guide->pane);
		}

                else /* situation shouldn't happen */ 
		{
		    kerror("xvforms", "xvf_delete_guide",
			   "Couldn't find specified guide in guide list");
		    return(FALSE);
		}
            }

	    /*
	     *  there are only two guide buttons in the guidepane guide list,
	     *  and we are deleting one of them.  this means we have to destroy
             *  the other guide button, and destroy selections on the guide pane
	     */
	    guideptr = subform->guidepane->guide_list;
	
	    /* nasty check to see if there's 2 gb's */
	    if (((guideptr == guide) &&
		 (guideptr->next != NULL) &&
	         (guideptr->next->next == NULL)) ||
		((guideptr->next == guide) &&
		 (guide->next == NULL)) )
	    {
		/* delete selections on guidepane (if any) */
		selection = subform->guidepane->sel_list;
		while (selection != NULL)
		{
		    xvf_destroy_selection(selection);
		    selection = selection->next;
		}

		/* delete selections on guidepane (if any) */
		if (guideptr != guide)
		    xvw_destroy(guideptr->button);
		else xvw_destroy(guideptr->next->button);

		if (subform->guidepane->label_object != NULL)
		{
		    xvw_destroy(subform->guidepane->label_object);
		    subform->guidepane->label_object = NULL;
		}
	    }
	}

	/*
	 *  delete the guide;  if this means deleting the guidepane (ie,
	 *  deleting the second-to-last guide), then any selections on the 
	 *  guide pane will be transferred onto the remaining pane. 
	 */
	if (!(kvf_delete_guide(guide)))
	    return(FALSE);

	/*
	 *  deletion of guide just made this a single pane on single subform
	 *  various things that still need to be done
	 */
        if (subform->guide != NULL)
        {
	   /*
	    * create the single pane if it's never been created,
            * or simply map it if it has been created already.
  	    */
	    if (subform->guide->pane->back == NULL)
	        xvf_create_pane(subform->guide->pane, TRUE);
	    else create_transferred_sels(subform->guide->pane->sel_list,
					 subform->guide->pane->back);

	   /*
	    * scoot the pane up and to the left; no need for offset
            * to accomodate guide buttons any more
 	    */
            xvf_set_attributes(subform->guide->pane->back_kformstruct,
                               XVF_X, 0.0,
                               XVF_Y, 0.0,
                               NULL);
            /* 
	     * resize subform to match the single pane,  put it in edit mode
             */
            xvw_get_attributes(subform->guide->pane->back,
                               XVW_CHAR_WIDTH,  &width,
                               XVW_CHAR_HEIGHT, &height,
                               NULL);
            xvw_set_attributes(subform->back,
                               XVW_EDIT_MODE_ON, FALSE,
                               XVW_CHAR_WIDTH,   width,
                               XVW_CHAR_HEIGHT,  height,
                               NULL);
        }
        return(TRUE);
}


/*-----------------------------------------------------------------
|
|  Routine Name: xvf_delete_selection
|
|	Purpose: Deletes a selection. 
|
|	  Input: selection   - selection to be deleted 		
|        Output: 
|       Returns: Returns TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Dec 06, 1992
| Modifications: 
|
------------------------------------------------------------------*/

int xvf_delete_selection(
   kselection *selection)
{
	kselection *last_member;
	char       *toggle_var;
	
	/*
	 *  if the selection is a submenu, have to delete the 
         *  submenu button and maybe the contents of the submenu
	 */
	if (selection->type == KUIS_STARTSUBMENU)
	{
	    /* unmap the submenu button */
	    xvw_unmap(selection->label_object);

	    /* pop down the menuform if it's up */
	    if (xvw_activated_menu(selection->label_object))
	        xvw_inactivate_menu(selection->label_object);
	
	    /* delete the submenu & maybe the contents */
	    delete_submenu(selection);

	    /* destroy the submenu button */
	    xvw_destroy(selection->label_object);
	    selection->label_object = NULL;
	}

	/*
	 *  selection not a submenu
	 */
	else 
	{
	    /* unmap the selection's backplane */
	    xvw_unmap(selection->back);

	    /* put down the menuform for the selection if it's up */
	    if (xvw_activated_menu(selection->back))
	        xvw_inactivate_menu(selection->back);

	    /* 
	     * deleting last member of toggle: delete toggle 
             */
	    if ((selection->back_toggle != NULL) &&
	        (selection->back_toggle->toggle_next == NULL))
	    {
		xvf_destroy_selection(selection);
		xvf_delete_selection(selection->back_toggle);
	    }

	    /* 
             * deleting 2nd-to-last member of toggle: delete toggle,
             * move remaining member to outer selection list 
             */
	    else if ((selection->back_toggle != NULL) &&
                     (selection->back_toggle->toggle_next->next == NULL))
	    {
		/* find last member of toggle */
		if (selection->back_toggle->toggle_next == selection)
		    last_member = selection->back_toggle->toggle_next->next;
	        else last_member = selection->back_toggle->toggle_next;

		/* add last member of toggle to outer selection list */
		kvf_add_selection(selection->back_control->sel_list,
				  last_member);

		/* destroy the selection */
		xvf_destroy_selection(selection);

		/* delete the toggle it belonged to */
		toggle_var = ktoken_to_string(selection->back_toggle->var_token);
		xvf_delete_selection(selection->back_toggle);
	

		/* re-create selection, it was destroyed w/ toggle */
		last_member->back = NULL;
		xvf_create_selection(last_member, selection->back_control->back,
				     selection->back_control->type);
		xvf_set_attribute(last_member->back_kformstruct, 
				  XVF_VARIABLE, toggle_var);
	    }

	    /* 
             * deleting last member of group: delete group 
             */
	    else if ((selection->back_group != NULL) &&
                     (selection->back_group->group_next == NULL))
	    {
		xvf_destroy_selection(selection);
		xvf_delete_selection(selection->back_group);
	    }

	    /* 
             * deleting 2nd-to-last member of group: delete group,
             * move remaining member to outer selection list 
             */
	    else if ((selection->back_group != NULL) &&
                     (selection->back_group->group_next->next == NULL))
	    {
		/* find last member of group */
		if (selection->back_group->group_next == selection)
		    last_member = selection->back_group->group_next->next;
	        else last_member = selection->back_group->group_next;

		/* add it to the outer group or outer selection list  */
		if (selection->back_group->back_group != NULL)
		    kvf_add_selection(selection->back_group->back_group->group_next,
				  last_member);
		else kvf_add_selection(selection->back_control->sel_list,
				       last_member);

		/* delete the selection */
		xvf_destroy_selection(selection);

		/* delete the group it belonged to */
		xvf_delete_selection(selection->back_group);

		/* add callback onto last member again, it got deleted when
	           the group was destroyed */
	        if ((last_member != NULL) && (last_member->opt_object != NULL))
		{
		    xvw_insert_callback(last_member->opt_object, 
				     XVW_BUTTON_SELECT, FALSE, xvf_gen_opt_cb, 
				     (kaddr) last_member->back_kformstruct);
		}
	    }

	    /* deleting a "plain" selection */
	    else
	    {
	        xvf_destroy_selection(selection);
	    }

	    /*
	     *  finally, have kforms actually delete the selection
	     *  from the form tree.
	     */
	     kvf_delete_selection(selection);
	}

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_new_guide
|
|       Purpose: Creates a new guide branch of the form tree
|                (no widgets) from the specified *.pane file.
|
|         Input: filename - path to the *.pane file
|        Output: none
|       Returns: The new guide on success, NULL on failure
|    Written By: Danielle Argiro
|          Date: Aug 20, 1993
| Modifications:
|
------------------------------------------------------------*/
kguide *xvf_create_new_guide(
   char *filename,
   char *guide_line)
{
        char      *guide_var;
        kform     *tmp_form;
        kguide    *guide;
        Line_Info line_info;

	tmp_form = kvf_create_form(filename, NONE, NULL, NULL);
        if (tmp_form == NULL) return(NULL);
        kvf_delete_entry(tmp_form);

        /*
         *  lots 'o' error checking
         */
        if (tmp_form->master != NULL)
        {
            xvf_destroy_form(tmp_form);
            kerror("xvforms", "xvf_create_new_guide",
                   "*.pane file may not contain a master form definition");
            return(NULL);
        }
        if (tmp_form->subform == NULL)
        {
            xvf_destroy_form(tmp_form);
	    errno = KINTERNAL;
            kerror("xvforms", "xvf_create_new_guide",
                   "Subform from %s created incorrectly", filename);
            return(NULL);
        }
        if (tmp_form->subform->guidepane != NULL)
        {
            xvf_destroy_form(tmp_form);
            kerror("xvforms", "xvf_create_new_guide",
                   "*.pane file may not contain a guidepane definition");
            return(NULL);
        }
        if (tmp_form->subform->guide == NULL)
        {
            xvf_destroy_form(tmp_form);
	    errno = KINTERNAL;
            kerror("xvforms", "xvf_create_new_guide",
                   "Guide from %s created incorrectly", filename);
            return(NULL);
        }
        if (tmp_form->subform->guide->pane == NULL)
        {
            xvf_destroy_form(tmp_form);
	    errno = KINTERNAL;
            kerror("xvforms", "xvf_create_new_guide",
                   "Pane from %s created incorrectly", filename);
            return(NULL);
        }

        /*
         *  the guide branch of the form tree is all we really want
         */
        guide = tmp_form->subform->guide;
	guide->line = kstrdup(guide_line);

	/*
	 *  need to ensure that the variable name on the -g line
         *  matches the one on the -P line
	 */
	kvf_clear_line_info(&line_info);
	kvf_gen_parse(guide->line, &line_info);
	guide_var = kstrdup(line_info.variable);
        kvf_free_line_info_strings(&line_info);
	kvf_gen_parse(guide->pane->control_line, &line_info);
	kfree(line_info.variable);
	line_info.variable = kstrdup(guide_var);
	kvf_gen_deparse(&line_info, &guide->pane->control_line);
        kvf_free_line_info_strings(&line_info);

        /*
         *  cleanup
         */
        kvf_free_line_info_strings(&line_info);
        kfree(tmp_form->subform->line);
        kfree(tmp_form->subform->comment);
        kfree(tmp_form->subform->control_line);
        kfree(tmp_form->subform->control_comment);
        kfree(tmp_form->subform);
        kfree(tmp_form->control_line);
        kfree(tmp_form->control_comment);
        kfree(tmp_form);
        kfree(guide_var);

        return(guide);

}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_new_subform
|
|       Purpose: Creates a new subform branch of the form tree
|                (no widgets) from the specified *.form file.
|
|         Input: filename - path to the *.form file
|        Output: none
|       Returns: The new subform on success, NULL on failure
|    Written By: Danielle Argiro
|          Date: Aug 24, 1993
| Modifications:
|
------------------------------------------------------------*/
ksubform *xvf_create_new_subform(
   char *filename,
   char *subform_line)
{
        char      *subform_var;
        kform     *tmp_form;
        ksubform  *subform;
        Line_Info line_info;
	int       selected;

        /*
         *  build the form tree w/o creating any widgets
         */
        tmp_form = kvf_create_form(filename, NONE, NULL, NULL);
        if (tmp_form == NULL) return(NULL);
        kvf_delete_entry(tmp_form);

        /*
         *  error checking
         */
        if (tmp_form->master != NULL)
        {
            xvf_destroy_form(tmp_form);
            kerror("xvforms", "xvf_create_new_guide",
                   "*.form file may not contain a master form definition");
            return(NULL);
        }
        if (tmp_form->subform == NULL)
        {
            xvf_destroy_form(tmp_form);
	    errno = KINTERNAL;
            kerror("xvforms", "xvf_create_new_guide",
                   "Subform from %s created incorrectly", filename);
            return(NULL);
        }

        /*
         *  the subform branch of the form tree is all we really want
         */
        subform = tmp_form->subform;
	subform->line = kstrdup(subform_line);

	/*
	 *  need to ensure that the variable name on the -d line
         *  matches the one on the -M line
	 */
	kvf_clear_line_info(&line_info);
	kvf_gen_parse(subform->line, &line_info);
	subform_var = kstrdup(line_info.variable);
	selected    = line_info.selected;
        kvf_free_line_info_strings(&line_info);
	kvf_gen_parse(tmp_form->subform->control_line, &line_info);
	kfree(line_info.variable);
	line_info.variable = kstrdup(subform_var);
	line_info.selected = selected;
	kvf_gen_deparse(&line_info, &tmp_form->subform->control_line);
        kvf_free_line_info_strings(&line_info);

        /*
         *  cleanup
         */
        kvf_free_line_info_strings(&line_info);
        kfree(tmp_form->control_line);
        kfree(tmp_form->control_comment);
        kfree(tmp_form);
        kfree(subform_var);

        return(subform);

}

/*-----------------------------------------------------------------
|
|  Routine Name: create_transferred_sels
|
|       Purpose: When selections have been transferred by 
|                kvf_transfer_selections() from a master
|                to a pane (because the master was deleted), or from a 
|                guidepane to a pane (because the guidepane was deleted),
|                *and* the pane in question was already mapped, you
|                have a situation where the new selections on the pane
|                exist in the selection list, but have not had their
|                associated widgets created on the pane backplane.
|                This routine just goes through and creates the widgets
|                for any selections that were just transferred (ie,
|                their backplanes are still NULL).
|
|         Input: control - pointer to the pane
|        Output: Returns TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: July 8, 1994
| Modifications:
|
------------------------------------------------------------------*/

int create_transferred_sels(
    kselection *sel_list,
    xvobject   back)
{
	kselection *selection;
	kcontrol   *control;

	/* 
	 * no selections to transfer 
         */
	if (sel_list == NULL)
	    return(TRUE);

	/* 
	 *  if the control backplane is not NULL, we know we need to 
	 *  create widgets assoc with the selections just transferred 
         *  (ie, the ones with NULL backplanes).
         *
         *  if the control backplane is still NULL, don't need to do anything;
         *  the transferred selections will be created along with all the 
         *  others when the control is mapped for the first time.
	 */
	control = sel_list->back_control;
	if (back != NULL)
	{
	    selection = sel_list;
	    while (selection != NULL)
            {
		/*
	         * if the selection has not actually been transferred (ie,
		 * selection back is not NULL), it still might be a quit, or
	         * help button that had applied to it the "trick" used in 
	         * kvf_transfer_selections() of moving the UIS line from the 
	         * old selection to this one.  If so, this might mean that 
	         * labels, geometry, etc. have changed.  The easiest way
	         * to handle this is simply to recreate it; even if nothing has
	         * changed, this won't hurt anything.
	         */
	        if (selection->back != NULL)
		{
		    xvw_destroy(selection->back);
		    selection->back = NULL;
		}

		/* 
		 * create the widgets that represent the selection, whether
	         * this one is brand new, or an old one that just had its
	         * widgets destroyed (above) so they can be updated.
	         */
                xvf_create_selection(selection, back, control->type);

		/*
	         *  need to recurse for transferred toggles
	         */
		if (selection->toggle_next != NULL)
		{
		    create_transferred_sels(selection->toggle_next,
					    selection->back);
		    xvf_highlight_toggle(selection);
		}

		/*
	         *  need to recurse for transferred groups
	         */
		if (selection->group_next != NULL)
		    create_transferred_sels(selection->group_next, back);

                selection = selection->next;
            }
	}

	return(TRUE);
}

/*-----------------------------------------------------------------
|
|  Routine Name: delete_submenu
|
|       Purpose: Deletes a submenu button.  Prompts the user whether
|                they want to delete only the submenu button, or the
|                entire contents of the submenu.
|
|         Input: submenu - submenu to be deleted
|        Output: 
|       Returns: Returns TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Oct 19, 1993
| Modifications:
|
------------------------------------------------------------------*/

static int delete_submenu(
    kselection *submenu)
{
	int          status;
	xvobject     back;
	char         *name;
	ksubform     *subform;
	kguide       *guide;
	kform_struct *kformstruct;
	kselection   *selection;

	if (submenu->submenu_next == NULL)
	    return(TRUE);
	

	status = kprompt(KSTANDARD, "Contents", "Button Only", 0, 
			 "Be *careful* now: Do you want to delete all the selections within the submenu, or only the submenu button (selections in submenu will be moved to the backplane in their original positions)?");

	/*
	 *  delete only the submenu button
	 */
	if (status == 0)
	{
	    back = submenu->back_control->back;
	    kformstruct = submenu->submenu_next;
	    while (kformstruct != NULL)
	    {
		switch(kformstruct->flag)
		{
		 case KUIS_SUBFORMBUTTON:
	              subform = kformstruct->Subformptr;
		      subform->back_submenu = NULL;
		      subform->button = xvf_create_subform_sel(subform, back);
		      break;

                 case KUIS_GUIDEBUTTON:
	              guide = kformstruct->Guideptr;
		      guide->back_submenu = NULL;
		      guide->button = xvf_create_guide_sel(guide, back);
		      break;

	         case KUIS_MASTERACTION:
		      selection = kformstruct->Selptr;
		      selection->back_submenu = NULL;
                      selection->back = 
		  	 	xvf_create_master_action_sel(selection, back);
		      break;

	         case KUIS_SUBFORMACTION:
		      selection = kformstruct->Selptr;
		      selection->back_submenu = NULL;
                      selection->back = 
		  	 	xvf_create_subform_action_sel(selection, back);
		      break;
			  
	         case KUIS_PANEACTION:
		      selection = kformstruct->Selptr;
		      selection->back_submenu = NULL;
                      selection->back = 
		  	 	xvf_create_pane_action_sel(selection, back);
		      break;

	         case KUIS_BLANK:
		      selection = kformstruct->Selptr;
		      selection->back_submenu = NULL;
                      selection->back = xvf_create_blank_sel(selection, back);
		      break;

	         case KUIS_HELP:
		      selection = kformstruct->Selptr;
		      selection->back_submenu = NULL;
                      selection->back = xvf_create_help_sel(selection, back);
		      break;

	         case KUIS_QUIT:
		      selection = kformstruct->Selptr;
		      selection->back_submenu = NULL;
		      if (selection->back_control->type == KMASTER)
                          selection->back = 
				xvf_create_quit_sel(selection, back, KMASTER);
		      else if (selection->back_control->type == KMASTER)
                          selection->back = 
                                xvf_create_quit_sel(selection, back, KGUIDEPANE);
		      else if (selection->back_control->type == KPANE)
                          selection->back = 
                                xvf_create_quit_sel(selection, back, KPANE);
		      break;

	         case KUIS_ROUTINE:
		      selection = kformstruct->Selptr;
		      selection->back_submenu = NULL;
                      selection->back = xvf_create_routine_sel(selection, back);
		      break;
		}
		kformstruct = kformstruct->next;
	    }
	}

	/*
	 * delete the submenu contents
	 */
	else
	{
	    kformstruct = submenu->submenu_next;
	    while (kformstruct != NULL)
	    {
		switch(kformstruct->flag)
		{
	         case KUIS_SUBFORMBUTTON:
		      subform = kformstruct->Subformptr;
	              name = ktoken_to_string(subform->var_token);
		    
		      status = kprompt(KSTANDARD, "Delete", "Save", 0, "About to delete the subform button for subform %s, as well as the entire %s subform.  Are you SURE you want to delete this subformbutton & subform?", name, name);
		      if (!status)
		      {
		         subform->back_submenu = NULL;
	                 if (submenu->back_control != NULL)
		             subform->button = xvf_create_subform_sel(subform, 
						  submenu->back_control->back);
		      }
		      else
		      {
            	         if (!(xvf_delete_subform(subform->back_form, subform)))
                             return(FALSE);
                      }
		      break;

                 case KUIS_GUIDEBUTTON:
		      guide = kformstruct->Guideptr;
		      name = ktoken_to_string(guide->var_token);
		      status = kprompt(KSTANDARD, "Delete", "Save", 0, "About to delete the guide button for pane %s, as well as the entire %s pane.  Are you SURE you want to delete this guidebutton & pane?", name, name);
                      if (!status)
                      {
                         guide->back_submenu = NULL;
	                 if (submenu->back_control != NULL)
                             guide->button = xvf_create_guide_sel(guide, 
					      submenu->back_control->back);
                      }
                      else
                      {
                          if (!(xvf_delete_guide(guide)))
                             return(FALSE);
                      }
		      break;

	         case KUIS_MASTERACTION:
	         case KUIS_SUBFORMACTION:
	         case KUIS_PANEACTION:
	         case KUIS_BLANK:
	         case KUIS_HELP:
	         case KUIS_QUIT:
	         case KUIS_ROUTINE:
	              if (!(xvf_delete_selection(kformstruct->Selptr)))
                         return(FALSE);
		      break;

		}
		kformstruct = kformstruct->next;
	    }
	}

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: destroy_selection
|
|       Purpose: This static routine destroys a selection.
|                It differs from xvf_destroy_selection in that
|                it calls xvw_destroy(selection->back). The
|                xvf_destroy_selection() routine *can't* destroy
|                the selection backplane;  if it does, and the
|                toplevel object is then destroyed, there will
|                be a memory leak.
|
|         Input: selection - the selection to be destroyed
|        Output: none
|          Date: May 7, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
void destroy_selection(
   kselection *selection)
{
        kselection   *toggle, *group;

        toggle = selection->toggle_next;
        while (toggle != NULL)
        {
            xvf_destroy_selection(toggle);
            toggle = toggle->next;
        }

        group = selection->group_next;
        while (group != NULL)
        {
            xvf_destroy_selection(group);
            group = group->next;
        }

        if (selection->back != NULL)
        {
            if (xvw_activated_menu(selection->back))
                xvw_inactivate_menu(selection->back);
        }
	xvw_destroy(selection->back);
        selection->back = NULL;
}

