/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>                                                       <<<<
   >>>>		       Callback Routines                      <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                xvf_gen_opt_cb()                       <<<<
   >>>>                xvf_quit_master_cb()                   <<<<
   >>>>                xvf_quit_subform_cb()                  <<<<
   >>>>                xvf_subform_cb()                       <<<<
   >>>>                xvf_guide_cb()                         <<<<
   >>>>                xvf_logic_val_cb()                     <<<<
   >>>>                xvf_cycle_val_cb()                     <<<<
   >>>>                xvf_list_val_cb()                      <<<<
   >>>>                xvf_displaylist_val_cb()               <<<<
   >>>>                xvf_int_scroll_cb()                    <<<<
   >>>>                xvf_float_scroll_cb()                  <<<<
   >>>>                xvf_double_scroll_cb()                 <<<<
   >>>>                xvf_pseudo_label_cb()                  <<<<
   >>>>                xvf_action_cb()                        <<<<
   >>>>                xvf_help_button_cb()                   <<<<
   >>>>                xvf_routine_cb()                       <<<<
   >>>>                                                       <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"


/*-----------------------------------------------------------
|
|  Routine Name: xvf_gen_opt_cb
| 
|        Purpose: This is the callback for the general case of a
|                 selections' optional button.  For use with
|                 -I, -O, -i, -f, -s, -l selections
| 
|          Input: object      - the optional button
|                 client_data - pointer used to pass kform_struct
|                 call_data   - not used
|         Output: none
|           Date: Apr 27, 1992 
|     Written By: Danielle Argiro
|  Modifications: 
| 
-----------------------------------------------------------*/

/* ARGSUSED */
void xvf_gen_opt_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	kselection    *selection; 
	kform_struct  *kformstruct;
	int           optional, live = FALSE, optsel;

	/* 
	 * get selection from generic kformstruct in clientdata
	 */
	kformstruct = (kform_struct *) client_data;
	selection  = kformstruct->Selptr;

	kvf_get_attributes(kformstruct, 
			   KVF_OPTIONAL, &optional,
			   KVF_OPTSEL,   &optsel, NULL);
	/* 
	 * doing a toggle 
	 */
	if (selection->back_toggle != NULL)
	{
	    if (selection->opt_selected == TRUE)
            	return;
	    xvf_set_attributes(selection->back_toggle->back_kformstruct, 
                               XVF_TOGGLE_NUM, selection->toggle_num,
			       XVF_SELECTED,   TRUE,
			       NULL);
	}

	/* 
	 * selection is part of ME/MI group
	 */
	else if (selection->back_group != NULL)
	{
	    xvf_call_do_group(selection);
	}

	/* 
	 * general case : setting optional button of optional selection
         */
	else
	{
	    xvf_set_attribute(kformstruct, XVF_OPTSEL, !optsel);

	    /*
	     * optional button of toggle: sensitize or desensitize members
	     */
	    if (selection->type == KUIS_TOGGLE)
            	xvf_set_optional_sensitivity(selection, !optsel);

	}

	if (selection->back_toggle != NULL)
	    kvf_get_attribute(selection->back_toggle->back_kformstruct, 
			      KVF_LIVE, &live);
	else if ((selection->type != KUIS_STDIN) && 
		 (selection->type != KUIS_STDOUT))
		kvf_get_attribute(kformstruct, KVF_LIVE, &live);

	/*
         * book-keeping
	if (selection->type != KUIS_LOGICAL)
	    selection->selected = TRUE;
         */
	selection->modified = TRUE;

	if (selection->type == KUIS_FLAG)
            xvf_set_attribute(kformstruct, XVF_SELECTED, TRUE);

	if (live)
	{
	    xvf_update_form_tree(selection);
	    if (xvf_collect_check_data(selection->back_form))
	    {
	        xvf_current_form        = selection->back_form;
	        xvf_current_subform     = selection->back_subform;
	        xvf_run_done(TRUE);
	    }
	}
}

/*----------------------------------------------------------
|
|  Routine Name: xvf_quit_master_cb
|
|       Purpose: This is the callback for the "Quit" button on
|                a master form; closes down the entire GUI.
|
|         Input: object      - the quit button
|                client_data - pointer used to pass kform_struct
|                call_data   - not used
|        Output: none
|          Date: Apr 20, 1992 
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

/* ARGSUSED */
void xvf_quit_master_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	kform_struct     *kformstruct;
	kselection  *selection;
	ksubform   *subform;
	kform       *form;

	/*
         * get selection from generic kformstruct in clientdata
         */
	kformstruct = (kform_struct *) client_data;
	selection = kformstruct->Selptr;
        form      = selection->back_form;

	/* call any additional routines registered XVF_CALL_SUBSTITUTE */
	if (xvf_call_extra_calls(selection->extra_call_list, 
				 XVF_CALL_SUBSTITUTE))
	    return;

	/* call any additional routines registered XVF_CALL_FIRST */
	(void) xvf_call_extra_calls(selection->extra_call_list, XVF_CALL_FIRST);
	
	/* change Quit line to show that it has been selected */
	xvf_set_attribute(kformstruct, XVF_SELECTED, TRUE);

	/* unmap all currently mapped independent subforms */
	subform = form->master->subform_list;
	while (subform != NULL)
	{
	    if ((subform->toplevel != NULL) &&
		(subform->selected == TRUE))
		xvw_unmap(subform->toplevel);
	    subform = subform->next;
	}

	/* unmap the master form */
	xvw_unmap(form->toplevel);  

	/* ready to return control to application */
	form->quit      = TRUE;
	xvf_current_form    = form;
	xvf_current_subform = NULL;

	/* call any additional routines registered XVF_CALL_LAST */
	(void) xvf_call_extra_calls(selection->extra_call_list, XVF_CALL_LAST);

	xvf_run_done(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_quit_subform_cb
|
|       Purpose: This is the callback for the "Quit" button on
|                a subform; closes down the entire GUI if it's a
|                single subform, or unmaps the subform if there is
|                more than one.
|
|         Input: object      - the quit button
|                client_data - pointer used to pass kform_struct
|                call_data   - not used
|        Output: none
|          Date: Apr 20, 1992 
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

/* ARGSUSED */
void xvf_quit_subform_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	kform_struct *kformstruct;
	Line_Info    line_info;
	kselection   *selection;
	kform        *form;
	ksubform     *subform;
	kguide       *guide;

	/* cast pointers using generic form tree node struct & backlinks */
	kformstruct = (kform_struct *) client_data;
	selection = kformstruct->Selptr;
	subform   = selection->back_subform;
	form      = selection->back_form;

	/* call any additional routines registered XVF_CALL_SUBSTITUTE */
	if (xvf_call_extra_calls(selection->extra_call_list, 
			         XVF_CALL_SUBSTITUTE))
	    return;

	/* call any additional routines registered XVF_CALL_FIRST */
	(void) xvf_call_extra_calls(selection->extra_call_list, XVF_CALL_FIRST);

	/* change the subform's Quit line to show that it has been selected */
	kvf_clear_line_info(&line_info);
	xvf_set_attribute(kformstruct, XVF_SELECTED, TRUE);

	/* collect & check data from forms */
	xvf_collect_check_data(form);
	subform->quit = TRUE;

	/* 
         * if there is more than one subform,
	 * change the subform's (-M) KUIS_STARTSUBFORM line 
         * to show that it is no longer selected 
	 */
	if (form->subform == NULL)
	{
	    kvf_parse_startsubform_line(subform->control_line, &line_info);
	    line_info.selected = FALSE;
	    kvf_deparse_startsubform_line(&line_info, &subform->control_line);
	    kvf_free_line_info_strings(&line_info);
	    subform->selected = FALSE;
	}

	/* 
	 * if there is a TRUE master, change the master form's 
         * (-d) subformbutton line to show that it is no longer selected 
         */
	if (subform->line != NULL) 
	{
	    kvf_parse_subformbutton_line(subform->line, &line_info);
	    line_info.selected = FALSE;
	    kvf_deparse_subformbutton_line(&line_info, &subform->line);
	    kvf_free_line_info_strings(&line_info);

	    /* un-highlight the subform button if it not on a submenu */
	    if ((subform->back_submenu == NULL) && (subform->button != NULL))
	        xvw_reverse_colors(subform->button, FALSE);
	}

	/* unmap subform */
	xvw_unmap(subform->toplevel);
	XFlush(xvw_display(subform->toplevel));

	/* get ready to return control to application */
	if (form->toplevel == NULL) form->quit = TRUE;
	xvf_current_form    = form;
	xvf_current_subform = subform;

	/* call any additional routines registered XVF_CALL_LAST */
	(void) xvf_call_extra_calls(selection->extra_call_list, XVF_CALL_LAST);

	/* return control to application */
	xvf_run_done(TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: xvf_quit_pane_cb
|
|       Purpose: This is the callback for the "Quit" button on
|                a pane; quits the pane and also the subform.
|
|         Input: object      - the quit button
|                client_data - pointer used to pass kform_struct
|                call_data   - not used
|        Output: none
|          Date: Apr 20, 1992 
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

/* ARGSUSED */
void xvf_quit_pane_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	kform_struct  *kformstruct;
	Line_Info     line_info;
	kform	      *form;
	ksubform      *subform;
	kguide        *guide;
	kselection    *selection;

	/* cast pointers using generic form tree node struct & backlinks */
	kformstruct = (kform_struct *) client_data;
	selection = kformstruct->Selptr;
	form      = selection->back_form;
	subform   = selection->back_subform;
	guide     = selection->back_guide;

	/* call any additional routines registered XVF_CALL_SUBSTITUTE */
	if (xvf_call_extra_calls(selection->extra_call_list, 
			         XVF_CALL_SUBSTITUTE))
	    return;

	/* call any additional routines registered XVF_CALL_FIRST */
	(void) xvf_call_extra_calls(selection->extra_call_list, XVF_CALL_FIRST);

	selection->modified = TRUE;
	selection->selected = TRUE;

	/* change the Quit line to show that it has been selected */
	kvf_clear_line_info(&line_info);
	kvf_parse_quit_line(selection->line, &line_info);
	line_info.selected = TRUE;
	kvf_deparse_quit_line(&line_info, &selection->line);
	kvf_free_line_info_strings(&line_info);

	/* collect & check data from forms */
	xvf_collect_check_data(form);

	/* 
	 * change the subform's (-M) KUIS_STARTSUBFORM line to show that 
         * the pane is no longer selected 
	 */

	/* don't want to do this if there is a single pane on single subform */
	if ((form->master == NULL) && (subform->guidepane == NULL))
	    subform->selected = TRUE;
	else 
	{
	    kvf_parse_startsubform_line(subform->control_line, &line_info);
	    line_info.selected = FALSE;
	    kvf_deparse_startsubform_line(&line_info, &subform->control_line);
	    kvf_free_line_info_strings(&line_info);
	    subform->selected = FALSE;
	}
	subform->quit = TRUE;

	/* 
	 * if there is a guide pane, change the subform's 
         * (-g) subformbutton line to show that it is no longer selected 
	 */
	if (subform->line != NULL)
	{
	    kvf_parse_subformbutton_line(subform->line, &line_info);
	    line_info.selected = FALSE;
	    kvf_deparse_subformbutton_line(&line_info, &subform->line);
	    kvf_free_line_info_strings(&line_info);

	    /* un-highlight the subform button if it not on a submenu */
	    if ((subform->back_submenu == NULL) && (subform->button != NULL))
		xvw_reverse_colors(subform->button, FALSE);
	}

	/* unmap subform */
	xvw_unmap(subform->toplevel);
	XFlush(xvw_display(subform->toplevel));

	/* ready to return control to application */
	if (form->toplevel == NULL) form->quit = TRUE;
	xvf_current_subform = subform;
	xvf_current_form = form;

	/* call any additional routines registered XVF_CALL_LAST */
	(void) xvf_call_extra_calls(selection->extra_call_list, XVF_CALL_LAST);

	/* return control to application */
	xvf_run_done(TRUE);
}




/*-----------------------------------------------------------
|
|  Routine Name: xvf_subform_cb
|
|       Purpose: This is the callback for a subform button
|                on the master form; it brings up a subform, and
|                if the form is mutually exclusive, it will unmap
|                any previously-mapped subform.
|
|         Input: object      - the subform button
|                client_data - pointer used to pass kform_struct
|                call_data   - not used
|        Output: none
|          Date: Apr 20, 1992 
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
-----------------------------------------------------------*/

/* ARGSUSED */
void xvf_subform_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	kform_struct *kformstruct;
	ksubform     *subform;

	/* cast pointers using generic form tree node struct & backlinks */
	kformstruct = (kform_struct *) client_data;
	subform   = kformstruct->Subformptr;

	/* call any additional routines registered XVF_CALL_SUBSTITUTE */
	if (xvf_call_extra_calls(subform->extra_call_list, 
			         XVF_CALL_SUBSTITUTE))
	    return;

	/* call any additional routines registered XVF_CALL_FIRST */
	(void) xvf_call_extra_calls(subform->extra_call_list, XVF_CALL_FIRST);

	/* go ahead & create subform now if it wasn't already created */
	if (subform->back == NULL)
	    xvf_create_subform(subform, -1, -1, TRUE);

	/* subform already selected: simply raise it */
	if (subform->selected)
	{
	    xvw_raise(subform->toplevel);
	    return;
	}

	/* set the new subform to be displayed */
	if (!(xvf_set_attribute(kformstruct, XVF_DISPLAY_SUBFORM, 1)))
	    return;

	xvf_current_form    = subform->back_form;
	xvf_current_subform = subform;

	/* call any additional routines registered XVF_CALL_LAST */
	(void) xvf_call_extra_calls(subform->extra_call_list, XVF_CALL_LAST);

   
} /* end xvf_subform_cb */


/*-----------------------------------------------------------
|
|  Routine Name: xvf_guide_cb
|
|       Purpose: This is the callback for a guidebutton;
|                it unmaps the old pane, and brings up the new pane.
|
|         Input: object      - the guide button
|                client_data - pointer used to pass kform_struct
|                call_data   - not used
|        Output: none
|          Date: Apr 20, 1992 
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
-----------------------------------------------------------*/

/* ARGSUSED */
void xvf_guide_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	kform_struct *kformstruct;
	kform        *form;
	ksubform     *subform;
	kguide       *guide; 

	/* cast pointers using generic form tree node struct & backlinks */
	kformstruct = (kform_struct *) client_data;
	guide     = kformstruct->Guideptr;
	form      = guide->back_form;
	subform   = guide->back_subform;

	/* call any additional routines registered XVF_CALL_SUBSTITUTE */
	if (xvf_call_extra_calls(guide->extra_call_list, 
			         XVF_CALL_SUBSTITUTE))
	    return;

	/* call any additional routines registered XVF_CALL_FIRST */
	(void) xvf_call_extra_calls(guide->extra_call_list, XVF_CALL_FIRST);

	/* if this guide button is already selected, can just return */
	if (guide->selected)
	    return;

	if (guide->pane->back == NULL)
	    xvf_create_pane(guide->pane, TRUE);

	/* set the new subform to be displayed */
	if (!(xvf_set_attribute(kformstruct, XVF_DISPLAY_PANE, 1)))
	    return;

	xvf_current_form    = form;
	xvf_current_subform = subform;
   
	/* call any additional routines registered XVF_CALL_LAST */
	(void) xvf_call_extra_calls(guide->extra_call_list, XVF_CALL_LAST);

} /* end xvf_guide_cb */


/*-----------------------------------------------------------
|
|  Routine Name: xvf_logic_val_cb
|
|       Purpose: This is the callback for the logical selection's value
|                button. It switches between displaying the label specified
|                for 1 and displaying the label specified for 0.
|
|         Input: object      - the logical value button
|                client_data - pointer used to pass kform_struct
|                call_data   - not used
|        Output: none
|          Date: Apr 20, 1992 
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
-----------------------------------------------------------*/

/* ARGSUSED */
void xvf_logic_val_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	int           logic_val, live;
	kselection    *selection;
	kform_struct  *kformstruct;

	/* cast pointers using generic form tree node struct & backlinks */
	kformstruct = (kform_struct *) client_data;
	selection = kformstruct->Selptr;

	/* 
	 *  mark logical as selected
	 */
	xvf_set_attribute(kformstruct, XVF_SELECTED, TRUE);

	/*
	 *  invert logical value
	 */
	kvf_get_attribute(kformstruct, KVF_LOGIC_VAL, &logic_val);
	logic_val = !logic_val;
	xvf_set_attribute(kformstruct, XVF_LOGIC_VAL, logic_val);

	/*
	 *  special case for when menuform is up - need to update 
         *  the default on the menuform according to logical value.
	 */
	if (xvw_activated_menu(selection->back))
	    xvf_set_attribute(kformstruct, XVF_LOGIC_DEF, logic_val);

	/*
	 *  book-keeping
	 */
	selection->modified = TRUE;
	selection->selected = TRUE;

	/* 
	 * if the selection is live, prepare to return control to applic. 
	 */
	kvf_get_attribute(kformstruct, KVF_LIVE, &live);
	if (live)
	{
	    xvf_update_form_tree(selection);
	    if (xvf_collect_check_data(selection->back_form))
	    {
	        xvf_current_form        = selection->back_form;
	        xvf_current_subform     = selection->back_subform;
	        xvf_run_done(TRUE);
	    }
	}

}


/*------------------------------------------------------------
|
|  Routine Name: xvf_cycle_val_cb
|
|       Purpose: This is the callback for the cycle options' value
|                button. It cycles through the string values specified,
|                updating the integer value of the cycle according to
|                the string value specified for that value.
|
|         Input: object      - the cycle value button
|                client_data - pointer used to pass kform_struct
|                call_data   - not used
|        Output: none
|          Date: Apr 20, 1992 
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
-----------------------------------------------------------*/
   
/* ARGSUSED */
void xvf_cycle_val_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	kform_struct    *kformstruct;
	kselection *selection;
	int           live, list_val, list_start, list_num;

	/* 
	 * get selection from generic kformstruct in clientdata
	 */
	kformstruct = (kform_struct *) client_data;
	selection = kformstruct->Selptr;

	/* 
	 *  cycle now selected
	 */
	xvf_set_attribute(kformstruct, XVF_SELECTED, TRUE);

	/*
	 *  increment cycle value by one (treated just like a list)
	 */
	kvf_get_attributes(kformstruct, 
			   KVF_LIST_VAL,   &list_val,
			   KVF_LIST_SIZE,  &list_num,
			   KVF_LIST_START, &list_start,
			   KVF_LIVE,       &live, 
			   NULL);
	list_val++;
	if (list_val > (list_start + list_num - 1))
            list_val = list_start;
	xvf_set_attribute(kformstruct, XVF_LIST_VAL, list_val);

	/*
	 *  book-keeping
	 */
	selection->selected = TRUE;
	selection->modified = TRUE;

	/* 
	 * if the selection is live, prepare to return control to applic. 
	 */
	if (live)
	{
	    xvf_update_form_tree(selection);
	    if (xvf_collect_check_data(selection->back_form))
	    {
	        xvf_current_form        = selection->back_form;
	        xvf_current_subform     = selection->back_subform;
		xvf_run_done(TRUE);
	    }
	}

}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_list_val_cb
|
|       Purpose: This is the callback for the list selection's
|                button. It puts up a list with the strings specified,
|                returns the number of the list item selected.
|
|         Input: object      - the list value button
|                client_data - pointer used to pass kform_struct
|                call_data   - not used
|        Output: none
|          Date: Nov 24, 1992 
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
-----------------------------------------------------------*/
   
/* ARGSUSED */
void xvf_list_val_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	int          i, found;
	kform_struct *kformstruct;
	kselection   *selection;
	char         *label = NULL;
	Line_Info    line_info;

	/* cast the pointer to the generic form tree node struct */
	kformstruct = (kform_struct *) client_data;
	selection = kformstruct->Selptr;
	kvf_gen_parse_kformstruct(kformstruct, &line_info);

	line_info.selected = TRUE;
	xvw_get_attribute(object, XVW_LABEL, &label);

	i = 0;
	found = FALSE;
	while ((i < line_info.list_num) & (!found))
	{
	    if ((kstrcmp(line_info.val_labels[i], label)) == 0)
	    {
		line_info.list_val = i + line_info.int_val;
		found = TRUE;
	    }
	    i++;
	}

	/* deliberately redundant w/ deparse - to update menuform */
	xvf_set_attribute(kformstruct, XVF_LIST_VAL, line_info.list_val);
	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	/*
	 * book-keeping
	 */
	selection->modified = TRUE;
	selection->selected = TRUE;

	/* 
	 * if the selection is live, prepare to return control to applic.
	 */
	if (line_info.live)
	{
	    xvf_update_form_tree(selection);
	    if (xvf_collect_check_data(selection->back_form))
	    {
	        xvf_current_form        = selection->back_form;
	        xvf_current_subform     = selection->back_subform;
	        xvf_run_done(TRUE);
	    }
	}


}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_displaylist_val_cb
|
|       Purpose: This is the callback for the dispalylist selection's
|                displayed list items. It registers the item selected.
|
|         Input: object      - the list value button
|                client_data - pointer used to pass kform_struct
|                call_data   - not used
|        Output: none
|          Date: Jan 18, 1994 
|    Written By: Danielle Argiro 
| Modifications: 
|
-----------------------------------------------------------*/
   
/* ARGSUSED */
void xvf_displaylist_val_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	kform_struct    *kformstruct;
	kselection      *selection;
	Line_Info       line_info;
	xvw_list_struct *entry = (xvw_list_struct *) call_data;

	/* cast the pointer to the generic form tree node struct */
	kformstruct = (kform_struct *) client_data;
	selection = kformstruct->Selptr;
	kvf_gen_parse_kformstruct(kformstruct, &line_info);

	xvf_set_attribute(kformstruct, XVF_SELECTED, TRUE);

	/* deliberately redundant w/ deparse - to update menuform */
	xvf_set_attribute(kformstruct, XVF_LIST_VAL, 
			  entry->list_index + line_info.int_val);

	/*
	 * book-keeping
	 */
	selection->prev_selected = FALSE;
	selection->modified = TRUE;
	selection->selected = TRUE;

	/* 
	 * if the selection is live, prepare to return control to applic.
	 */
	if (line_info.live)
	{
	    xvf_update_form_tree(selection);
	    if (xvf_collect_check_data(selection->back_form))
	    {
	        xvf_current_form        = selection->back_form;
	        xvf_current_subform     = selection->back_subform;
	        xvf_run_done(TRUE);
	    }
	}


}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_displaylist_val_doubleclick_cb
|
|       Purpose: This is the callback for the displaylist selection's
|                doubleclick. It registers the item selected, and
|                marks the fact that doubleclick was used.
|
|         Input: object      - the list value button
|                client_data - pointer used to pass kform_struct
|                call_data   - not used
|        Output: none
|          Date: Jan 18, 1994
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/
  
/* ARGSUSED */
void xvf_displaylist_val_doubleclick_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	kform_struct *kformstruct;

	xvf_displaylist_val_cb(object, client_data, call_data);

	/* cast the pointer to the generic form tree node struct */
        kformstruct = (kform_struct *) client_data;

	/* this is to store the fact that doubleclick happened */
	kformstruct->Selptr->prev_selected = TRUE;

}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_stringlist_val_cb
|
|       Purpose: This is the callback for the StringList selection's
|                button. It puts up a list with the strings specified,
|                returns the number of the list item selected, and
|                updates the contents of the ascii object with the
|                string selected.
|
|         Input: object      - the list value button
|                client_data - pointer used to pass kform_struct
|                call_data   - not used
|        Output: none
|          Date: Nov 24, 1992 
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
-----------------------------------------------------------*/
   
/* ARGSUSED */
void xvf_stringlist_val_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	int             i, found;
	kform_struct      *kformstruct;
	Line_Info       line_info;
	kselection   *selection;
	char            *label = NULL;

	/* cast the pointer to the generic form tree node struct */
	kformstruct = (kform_struct *) client_data;
	selection = kformstruct->Selptr;

	kvf_gen_parse_kformstruct(kformstruct, &line_info);
	line_info.selected = TRUE;

	xvw_get_attribute(object, XVW_LABEL, &label);
   
	i = 0;
	found = FALSE;
	while ((i < line_info.list_num) & (!found))
	{
	    if ((kstrcmp(line_info.val_labels[i], label)) == 0)
	    {
		line_info.list_val = i+1;
		found = TRUE;
	    }
	    i++;
	}

	/* deliberately redundant w/ deparse - to update menuform & string */
	xvf_set_attributes(kformstruct, 
		   XVF_LIST_VAL,   line_info.list_val,
		   XVF_STRING_VAL, line_info.val_labels[line_info.list_val-1],
		   NULL);

	kvf_gen_deparse_kformstruct(kformstruct, &line_info);
	XFlush(xvw_display(selection->back));

	/*
	 * book-keeping
	 */
	selection->modified = TRUE;
	selection->selected = TRUE;

	/*
         * if the selection is live, prepare to return control to applic.
         */
	if (line_info.live)
	{
	    xvf_update_form_tree(selection);
	    if (xvf_collect_check_data(selection->back_form))
	    {
	        xvf_current_form        = selection->back_form;
	        xvf_current_subform     = selection->back_subform;
	        xvf_run_done(TRUE);
	    }
	}
	else
	{
	    xvf_collect_value(selection);
	}


}

/*------------------------------------------------------------
|
|  Routine Name: xvf_int_scroll_cb
|
|       Purpose: This is the callback for the integer selections'
|                scroll bar
|
|         Input: object      - the integer selection's scroll bar
|                client_data - pointer used to pass selection
|		 call_data   - passes movement of the scroll bar
|
|        Output: none
|          Date: Apr 20, 1992 
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/


/* ARGSUSED */
void xvf_int_scroll_cb(
   xvobject object,
   kaddr  client_data,
   kaddr  call_data)
{
	kselection  *selection;
	char        temp[KLENGTH];
	double      tmp_double;
	xvobject    scroll_object = object;
	int         live, scroll_value;

	selection   = (kselection *) client_data;

	/*
         *  get current status of scrollbar, update text widget
         */
	xvw_get_attribute(scroll_object, XVW_SCROLL_VALUE, &tmp_double);
	scroll_value = (int) tmp_double;
	ksprintf(temp,"%d", scroll_value);
	xvw_set_attribute(selection->value_object, XVW_TEXT_STRING, temp);
	xvf_set_attribute(selection->back_kformstruct, XVF_LITERAL, temp); 

	/*
         * when menuform is up, update default (default => int val)
         */
         if (xvw_activated_menu(selection->back))
             xvf_set_attributes(selection->back_kformstruct,
                                XVF_SELECTED, TRUE,
                                XVF_INT_DEF,  scroll_value, 
				NULL);
	selection->modified = TRUE;

	/*
         * only if the selection is live need we update selection
         */
	kvf_get_attribute(selection->back_kformstruct, KVF_LIVE, &live);
	if (live)  
	{
            /*
             *  update integer value.
             */
            xvf_set_attributes(selection->back_kformstruct,
                               XVF_SELECTED, TRUE,
                               XVF_INT_VAL,  scroll_value, 
			       NULL);
	    /*
             *  prepare to return control to application
             */
	    xvf_update_form_tree(selection);
	    if (xvf_collect_check_data(selection->back_form))
	    {
		xvf_current_form        = selection->back_form;
		xvf_current_subform     = selection->back_subform;
	        xvf_run_done(FALSE);
	    }
	}

}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_float_scroll_cb
|
|       Purpose: This is the callback for the float selections'
|                scroll bar
|
|         Input: object      - the float selection's  scroll bar
|                client_data - pointer used to pass selection
|		 call_data   - passes movement of the scroll bar
|
|        Output: none
|          Date: Apr 20, 1992 
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

/* ARGSUSED */
void xvf_float_scroll_cb(
   xvobject object,
   kaddr  client_data,
   kaddr  call_data)
{
	int         live, precision;
	char        temp[KLENGTH], scale[KLENGTH];
	double      tmp_double;
	float       scroll_value;
	kselection  *selection;
	xvobject    scroll_object = object;

	selection   = (kselection *) client_data;

	/*
         *  get current status of scrollbar
         */
        xvw_get_attribute(scroll_object, XVW_SCROLL_VALUE, &tmp_double);
	scroll_value = (float) tmp_double;
	kvf_get_attribute(selection->back_kformstruct, 
			  KVF_FLOAT_PREC, &precision);
	if (precision == 0)
            ksprintf(scale, "%%g");
        else if (precision == -1)
            ksprintf(scale, "%%f");
        else ksprintf(scale, "%%.%df", precision);
            ksprintf(temp, scale, scroll_value);

	ksprintf(temp, "%g", scroll_value);
	xvw_set_attribute(selection->value_object, XVW_TEXT_STRING, temp); 
	xvf_set_attribute(selection->back_kformstruct, XVF_LITERAL, temp); 

	/*
         * when menuform is up, update default (default => float val)
         */
         if (xvw_activated_menu(selection->back))
             xvf_set_attributes(selection->back_kformstruct,
                                XVF_SELECTED,  TRUE,
                                XVF_FLOAT_DEF, scroll_value, 
				NULL);

	selection->modified = TRUE;

	/*
         * only if the selection is live need we update selection
         */
        kvf_get_attribute(selection->back_kformstruct, KVF_LIVE, &live);
	if (live)  
	{
            /*
             *  update float value.
             */
            xvf_set_attributes(selection->back_kformstruct,
                               XVF_SELECTED,  TRUE,
                               XVF_FLOAT_VAL, scroll_value, 
			       NULL);

	    xvf_update_form_tree(selection);
	    if (xvf_collect_check_data(selection->back_form))
	    {
		xvf_current_form        = selection->back_form;
		xvf_current_subform     = selection->back_subform;
	        xvf_run_done(FALSE);
	    }
	}

}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_double_scroll_cb
|
|       Purpose: This is the callback for the double selections'
|                scroll bar
|
|         Input: object      - the double selection's  scroll bar
|                client_data - pointer used to pass selection
|		 call_data   - passes movement of the scroll bar
|
|        Output: none
|          Date: Apr 20, 1992 
|    Written By: Danielle Argiro 
| Modifications: 
|
------------------------------------------------------------*/

/* ARGSUSED */
void xvf_double_scroll_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	int         live, precision;
	char        temp[KLENGTH], scale[KLENGTH];
	double      scroll_value;
	kselection  *selection;
	xvobject    scroll_object = object;

	selection   = (kselection *) client_data;

	/*
         *  get current status of scrollbar
         */
        xvw_get_attribute(scroll_object, XVW_SCROLL_VALUE, &scroll_value);

	kvf_get_attribute(selection->back_kformstruct, 
			  KVF_DOUBLE_PREC, &precision);
	if (precision == 0)
            ksprintf(scale, "%%g");
        else if (precision == -1)
            ksprintf(scale, "%%f");
        else ksprintf(scale, "%%.%df", precision);
            ksprintf(temp, scale, scroll_value);

	ksprintf(temp, "%g", scroll_value);
	xvw_set_attribute(selection->value_object, XVW_TEXT_STRING, temp); 
	xvf_set_attribute(selection->back_kformstruct, XVF_LITERAL, temp); 


	/*
         * when menuform is up, update default (default => double val)
         */
         if (xvw_activated_menu(selection->back))
             xvf_set_attributes(selection->back_kformstruct,
                                XVF_SELECTED,   TRUE,
                                XVF_DOUBLE_DEF, scroll_value, 
				NULL);

	selection->modified = TRUE;

	/*
         * only if the selection is live need we update selection
         */
        kvf_get_attribute(selection->back_kformstruct, KVF_LIVE, &live);
	if (live)  
	{
            /*
             *  update double value.
             */
            xvf_set_attributes(selection->back_kformstruct,
                               XVF_SELECTED,   TRUE,
                               XVF_DOUBLE_VAL, scroll_value, 
			       NULL);

	    xvf_update_form_tree(selection);
	    if (xvf_collect_check_data(selection->back_form))
	    {
		xvf_current_form        = selection->back_form;
		xvf_current_subform     = selection->back_subform;
	        xvf_run_done(FALSE);
	    }
	}

}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_pseudo_label_cb
|
|       Purpose: This is the callback for a pseudo-label button on
|                a -I, -O, -A, or -Z selection. When clicked on, the
|                pseudo-label object pops up a file browser
|                and lets the user select a file.  If valid, this filename
|                then put into the selection's associated text object.
|
|         Input: object      - the pseudo label button
|                client_data - pointer used to pass kform_struct
|		 call_data   - passes movement of the scroll bar
|
|        Output: none
|          Date: Apr 20, 1992 
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

/* ARGSUSED */
void xvf_pseudo_label_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	kform_struct *kformstruct;
	kselection   *selection;
	char         *filename = NULL;
	char         *new_dir, *tmp_dir;
	int          live;

	/* cast the pointer to the generic form tree node struct */
	kformstruct = (kform_struct *) client_data;
	selection = kformstruct->Selptr;

	if (selection->type == KUIS_INPUTFILE) 
	   xvf_collect_value(selection);

	else if (selection->type == KUIS_OUTPUTFILE) 
	   xvf_collect_value(selection);

	kvf_get_attributes(kformstruct, 
			   KVF_FILE_NAME, &filename,
			   KVF_LIVE,      &live,     
			   NULL);
	new_dir = "./";
	if (filename != NULL)
	{
	    /* strip off directory name */
	    tmp_dir = kdirname(filename, NULL);
	    if (tmp_dir != NULL) new_dir = tmp_dir;
	}
	filename = xvu_run_browser(new_dir);

	if (filename != NULL)
	{
	    selection->modified = TRUE;

	    xvf_set_attributes(kformstruct,
			       XVF_SELECTED, TRUE,
	    		       XVF_LITERAL,  filename,
			       NULL);

	    if (live)  
	    {
		xvf_update_form_tree(selection);
		if (xvf_collect_check_data(selection->back_form))
	        {
   		    xvf_current_form    = selection->back_form;
		    xvf_current_subform = selection->back_subform;
		    xvf_run_done(TRUE);
	        }
	    }
	}

}

/*------------------------------------------------------------
|
|  Routine Name: xvf_action_cb
|
|       Purpose: This is the callback for the action button that resides
|                in a master, guidepane, or pane.  When this button is
|                selected, the corresponding "selected" field in the UIS
|                line is to be set *1 and control is returned to the application.
|
|         Input: object      - the action button
|                client_data - pointer used to pass kform_struct
|                call_data   - not used
|        Output: none
|          Date: Apr 20, 1992 
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/
   
/* ARGSUSED */
void xvf_action_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	kform_struct *kformstruct;
	kselection   *selection;
	char         *title;

	/* cast the pointer to the generic form tree node struct */
	kformstruct = (kform_struct *) client_data;
	selection = kformstruct->Selptr;

	/* set the -a line to be selected */
	kvf_set_attribute(selection->back_kformstruct, KVF_SELECTED, TRUE);

	xvf_update_form_tree(selection);

	selection->modified = TRUE;
	if (xvf_collect_check_data(selection->back_form))
	{
   	    xvf_current_form    = selection->back_form;
	    xvf_current_subform = selection->back_subform;
	    xvf_run_done(TRUE);
	}
	else
	{
	    kvf_get_attribute(kformstruct, KVF_TITLE, &title);
	    kerror("xvforms", "xvf_action_cb", "Cancelling action %s", title);
	}


} /* end xvf_action_cb */



/*-----------------------------------------------------------
|
|  Routine Name: xvf_help_button_cb
|
|       Purpose: This is the callback for a help button.  When this button
|                is selected, the specified help page is displayed.
|
|         Input: object      - the help button
|                client_data - pointer used to pass kform_struct
|                call_data   - not used
|        Output: none
|          Date: Apr 20, 1992 
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/
   
/* ARGSUSED */
void xvf_help_button_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	kform_struct  *kformstruct;
	Line_Info     line_info;
	kselection    *selection;
	xvobject      help;

	/* cast the pointer to the generic form tree node struct */
	xvw_busy(NULL, TRUE);
	kformstruct = (kform_struct *) client_data;
	selection = kformstruct->Selptr;

	/* call any additional routines registered XVF_CALL_SUBSTITUTE */
	if (xvf_call_extra_calls(selection->extra_call_list, 
			         XVF_CALL_SUBSTITUTE))
	    return;

	/* call any additional routines registered XVF_CALL_FIRST */
	(void) xvf_call_extra_calls(selection->extra_call_list, XVF_CALL_FIRST);

	/* get the help file name from [-H] UIS line & display help page */
	kvf_clear_line_info(&line_info);
	kvf_parse_help_line(selection->line, &line_info);
	help = xvw_create_help(NULL, "Online Help");
	xvw_set_attributes(help,
		XVW_HELP_TITLE,  "Online Help",
		XVW_HELP_FILENAME, line_info.help_file,
		NULL);
	kvf_free_line_info_strings(&line_info);

	/* call any additional routines registered XVF_CALL_LAST */
	(void) xvf_call_extra_calls(selection->extra_call_list, XVF_CALL_LAST);
	xvw_busy(NULL, FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_routine_cb
|
|       Purpose: This is the callback for the routine button.
|                When this button is selected, the specified command
|                is forked.
|
|         Input: object      - the routine button
|                client_data - pointer used to pass kform_struct
|                call_data   - not used
|        Output: none
|          Date: Apr 20, 1992 
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

/* ARGSUSED */
void xvf_routine_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	kform_struct  *kformstruct;
	char          *command;
	kselection    *selection;
	char          *routine;

	/* cast the pointer to the generic form tree node struct */
	kformstruct = (kform_struct *) client_data;
	selection  = kformstruct->Selptr;

	/* call any additional routines registered XVF_CALL_SUBSTITUTE */
        if (xvf_call_extra_calls(selection->extra_call_list,
                                 XVF_CALL_SUBSTITUTE))
            return;

        /* call any additional routines registered XVF_CALL_FIRST */
        (void) xvf_call_extra_calls(selection->extra_call_list, XVF_CALL_FIRST);


	if (!(xvf_collect_check_data(selection->back_form)))
	{
	    kvf_get_attribute(kformstruct, KVF_ROUTINE, &routine);
	    kerror("xvforms", "xvf_routine_cb", 
		   "Cancelling execution of routine %s", routine);
	    return;
	}

	/* form the command line */
	command = kvf_form_to_cmd(selection->back_form,
                                 selection->back_subform, NULL, TRUE);
	if (command == NULL)
	{
	    kerror("xvforms", "xvf_routine_cb", "Unable to build command line");
	    return;
	}

	/* 
	 *  Invert the routines button in order to show that we are executing
	 *  the command.  Then call xvw_system() to system off the routine.
	 *  The xvw_system() will wait until the process is completed before
	 *  returning.  We then uninvert the routines button when we are done.
         */
	xvw_reverse_colors (object, TRUE);
	xvw_system(command); kfree(command);
	xvw_reverse_colors(object, FALSE);

        /* call any additional routines registered XVF_CALL_LAST */
        (void) xvf_call_extra_calls(selection->extra_call_list, XVF_CALL_LAST);
}
